/* packet-dvb-eit.c
 * Routines for DVB (ETSI EN 300 468) Event Information Table (EIT) dissection
 * Copyright 2012, Guy Martin <gmsoft@tuxicoman.be>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "config.h"

#include <epan/packet.h>
#include <epan/tfs.h>
#include "packet-mpeg-sect.h"
#include "packet-mpeg-descriptor.h"

void proto_register_dvb_eit(void);
void proto_reg_handoff_dvb_eit(void);

static dissector_handle_t dvb_eit_handle;

static int proto_dvb_eit;
static int hf_dvb_eit_service_id;
static int hf_dvb_eit_reserved;
static int hf_dvb_eit_version_number;
static int hf_dvb_eit_current_next_indicator;
static int hf_dvb_eit_section_number;
static int hf_dvb_eit_last_section_number;

static int hf_dvb_eit_transport_stream_id;
static int hf_dvb_eit_original_network_id;
static int hf_dvb_eit_segment_last_section_number;
static int hf_dvb_eit_last_table_id;

static int hf_dvb_eit_event_id;
static int hf_dvb_eit_start_time;
static int hf_dvb_eit_duration;
static int hf_dvb_eit_running_status;
static int hf_dvb_eit_free_ca_mode;
static int hf_dvb_eit_descriptors_loop_length;

static int ett_dvb_eit;
static int ett_dvb_eit_event;


#define DVB_EIT_RESERVED_MASK                     0xC0
#define DVB_EIT_VERSION_NUMBER_MASK               0x3E
#define DVB_EIT_CURRENT_NEXT_INDICATOR_MASK       0x01

#define DVB_EIT_RUNNING_STATUS_MASK             0xE000
#define DVB_EIT_FREE_CA_MODE_MASK               0x1000
#define DVB_EIT_DESCRIPTORS_LOOP_LENGTH_MASK    0x0FFF

static const value_string dvb_eit_running_status_vals[] = {
    { 0, "Undefined" },
    { 1, "Not Running" },
    { 2, "Starts in a few seconds" },
    { 3, "Pausing" },
    { 4, "Running" },
    { 5, "Service off-air" },

    { 0, NULL }
};

static const value_string dvb_eit_free_ca_mode_vals[] = {
    { 0, "Not Scrambled" },
    { 1, "One or more component scrambled" },

    { 0, NULL }
};

static int
dissect_dvb_eit(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{

    unsigned    offset = 0, length = 0;
    unsigned    descriptor_len;
    uint16_t    evt_id;

    proto_item *ti;
    proto_tree *dvb_eit_tree;
    proto_tree *dvb_eit_event_tree;
    proto_item *duration_item;

    nstime_t    start_time;

    col_set_str(pinfo->cinfo, COL_INFO, "Event Information Table (EIT)");

    ti = proto_tree_add_item(tree, proto_dvb_eit, tvb, offset, -1, ENC_NA);
    dvb_eit_tree = proto_item_add_subtree(ti, ett_dvb_eit);

    offset += packet_mpeg_sect_header(tvb, offset, dvb_eit_tree, &length, NULL);
    length -= 4;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_service_id,                  tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_reserved,                    tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_version_number,              tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_current_next_indicator,      tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_section_number,              tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_last_section_number,         tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_transport_stream_id,         tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_original_network_id,         tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_segment_last_section_number, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(dvb_eit_tree, hf_dvb_eit_last_table_id, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    if (offset >= length) {
        packet_mpeg_sect_crc(tvb, pinfo, dvb_eit_tree, 0, offset);

        return offset;
    }

    /* Parse all the events */
    while (offset < length) {

        evt_id = tvb_get_ntohs(tvb, offset);
        dvb_eit_event_tree = proto_tree_add_subtree_format(dvb_eit_tree, tvb, offset, 12, ett_dvb_eit_event, NULL, "Event 0x%04hx", evt_id);

        proto_tree_add_item(dvb_eit_event_tree, hf_dvb_eit_event_id, tvb, offset, 2, ENC_BIG_ENDIAN);
        offset += 2;

        if (tvb_memeql(tvb, offset, (const uint8_t*)"\xFF\xFF\xFF\xFF\xFF", 5)) {
            if (packet_mpeg_sect_mjd_to_utc_time(tvb, offset, &start_time) < 0) {
                proto_tree_add_time_format(dvb_eit_event_tree, hf_dvb_eit_start_time, tvb, offset, 5,
                                    &start_time, "Unparseable time");
            } else {
                proto_tree_add_time(dvb_eit_event_tree, hf_dvb_eit_start_time, tvb, offset,
                    5, &start_time);
            }
        } else {
            start_time.secs = 0xFFFFFFFF;
            start_time.nsecs = 0xFFFFFFFF;
            proto_tree_add_time_format_value(tree, hf_dvb_eit_start_time, tvb, offset, 5, &start_time, "Undefined (0xFFFFFFFFFF)");
        }
        offset += 5;

        duration_item = proto_tree_add_item(dvb_eit_event_tree, hf_dvb_eit_duration, tvb, offset, 3, ENC_BIG_ENDIAN);
        proto_item_append_text(duration_item, " (%02u:%02u:%02u)",
            MPEG_SECT_BCD44_TO_DEC(tvb_get_uint8(tvb, offset)),
            MPEG_SECT_BCD44_TO_DEC(tvb_get_uint8(tvb, offset + 1)),
            MPEG_SECT_BCD44_TO_DEC(tvb_get_uint8(tvb, offset + 2)));
        offset += 3;

        proto_tree_add_item(dvb_eit_event_tree, hf_dvb_eit_running_status,          tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(dvb_eit_event_tree, hf_dvb_eit_free_ca_mode,            tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(dvb_eit_event_tree, hf_dvb_eit_descriptors_loop_length, tvb, offset, 2, ENC_BIG_ENDIAN);
        descriptor_len = tvb_get_ntohs(tvb, offset) & DVB_EIT_DESCRIPTORS_LOOP_LENGTH_MASK;
        offset += 2;

        offset += proto_mpeg_descriptor_loop_dissect(tvb, pinfo, offset, descriptor_len, dvb_eit_event_tree);
    }

    offset += packet_mpeg_sect_crc(tvb, pinfo, dvb_eit_tree, 0, offset);
    proto_item_set_len(ti, offset);
    return tvb_captured_length(tvb);
}


void
proto_register_dvb_eit(void)
{

    static hf_register_info hf[] = {

        { &hf_dvb_eit_service_id, {
            "Service ID", "dvb_eit.sid",
            FT_UINT16, BASE_HEX, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_reserved, {
            "Reserved", "dvb_eit.reserved",
            FT_UINT8, BASE_HEX, NULL, DVB_EIT_RESERVED_MASK, NULL, HFILL
        } },

        { &hf_dvb_eit_version_number, {
            "Version Number", "dvb_eit.version",
            FT_UINT8, BASE_HEX, NULL, DVB_EIT_VERSION_NUMBER_MASK, NULL, HFILL
        } },

        { &hf_dvb_eit_current_next_indicator, {
            "Current/Next Indicator", "dvb_eit.cur_next_ind",
            FT_BOOLEAN, 8, TFS(&tfs_current_not_yet), DVB_EIT_CURRENT_NEXT_INDICATOR_MASK, NULL, HFILL
        } },

        { &hf_dvb_eit_section_number, {
            "Section Number", "dvb_eit.sect_num",
            FT_UINT8, BASE_DEC, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_last_section_number, {
            "Last Section Number", "dvb_eit.last_sect_num",
            FT_UINT8, BASE_DEC, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_transport_stream_id, {
            "Transport Stream ID", "dvb_eit.tsid",
            FT_UINT16, BASE_HEX, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_original_network_id, {
            "Original Network ID", "dvb_eit.original_nid",
            FT_UINT16, BASE_HEX, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_segment_last_section_number, {
            "Segment Last Section Number", "dvb_eit.segment_last_sect_num",
            FT_UINT8, BASE_DEC, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_last_table_id, {
            "Last Table ID", "dvb_eit.last_tid",
            FT_UINT8, BASE_HEX, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_event_id, {
            "Event ID", "dvb_eit.evt.id",
            FT_UINT16, BASE_HEX, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_start_time, {
            "UTC Start Time", "dvb_eit.evt.start_time",
            FT_ABSOLUTE_TIME, ABSOLUTE_TIME_UTC, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_duration, {
            "Duration", "dvb_eit.evt.duration",
            FT_UINT24, BASE_HEX, NULL, 0, NULL, HFILL
        } },

        { &hf_dvb_eit_running_status, {
            "Running Status", "dvb_eit.evt.running_status",
            FT_UINT16, BASE_HEX, VALS(dvb_eit_running_status_vals), DVB_EIT_RUNNING_STATUS_MASK, NULL, HFILL
        } },

        { &hf_dvb_eit_free_ca_mode, {
            "Free CA Mode", "dvb_eit.evt.free_ca_mode",
            FT_UINT16, BASE_HEX, VALS(dvb_eit_free_ca_mode_vals), DVB_EIT_FREE_CA_MODE_MASK, NULL, HFILL
        } },

        { &hf_dvb_eit_descriptors_loop_length, {
            "Descriptors Loop Length", "dvb_eit.evt.descr_loop_len",
            FT_UINT16, BASE_DEC, NULL, DVB_EIT_DESCRIPTORS_LOOP_LENGTH_MASK, NULL, HFILL
        } }
    };

    static int *ett[] = {
        &ett_dvb_eit,
        &ett_dvb_eit_event
    };

    proto_dvb_eit = proto_register_protocol("DVB Event Information Table", "DVB EIT", "dvb_eit");

    proto_register_field_array(proto_dvb_eit, hf, array_length(hf));
    proto_register_subtree_array(ett, array_length(ett));

    dvb_eit_handle = register_dissector("dvb_eit", dissect_dvb_eit, proto_dvb_eit);
}


void proto_reg_handoff_dvb_eit(void)
{
    int tid;

    for (tid = DVB_EIT_TID_MIN; tid <= DVB_EIT_TID_MAX; tid++)
        dissector_add_uint("mpeg_sect.tid", tid, dvb_eit_handle);
}


/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
