/* packet-ieee80211.c
 * Routines for Wireless LAN (IEEE 802.11) dissection
 * Copyright 2000, Axis Communications AB
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Credits:
 *
 * The following people helped me by pointing out bugs etc. Thank you!
 *
 * Marco Molteni
 * Lena-Marie Nilsson
 * Magnus Hultman-Persson
 */

/*
 * 10/24/2005 - Add dissection for 802.11e
 * Zhu Yi <yi.zhu@intel.com>
 *
 * 04/21/2008 - Added dissection for 802.11p
 * Arada Systems <http://www.aradasystems.com>
 *
 * 04/20/2013 - Added dissection of 802.11ad according to the 9th draft of the standard.
 * Extended as a project in the Laboratory of Computer Communication & Networking (LCCN), Computer Science Department, Technion, Israel.
 * Majd Omari    <majd.omari@outlook.com>
 * Jalil Moraney <moraney.jalil@outlook.com>
 */

/*
 * Reference :
 * The 802.11 standard is "free", 6 month after the publication.
 *
 * IEEE Std 802.11-2020: Revision of IEEE Std 802.11-2016
 * include 5 amendments (802.11ai,ah,aj,ak,aq) 802.11-2016
 * include 5 amendments (802.11ae,aa,ad,ac,af) 802.11-2012
 * include 10 amendments (802.11k,r,y,w,n,p,z,v,u,s) 802.11-2007
 * include 8 amendments (802.11a,b,d,e,g,h,i,j) 802.11-1999
 * https://ieeexplore.ieee.org/document/9502043
 *
 * WAPI (IE 68)
 * http://isotc.iso.org/livelink/livelink/fetch/-8913189/8913214/8913250/8913253/JTC001-N-9880.pdf?nodeid=8500308&vernum=-2
 */

#include "config.h"

#include <math.h>

#include <epan/packet.h>
#include <epan/capture_dissectors.h>
#include <epan/exceptions.h>
#include <wsutil/pint.h>
#include <wsutil/ws_roundup.h>
#include <epan/addr_resolv.h>
#include <epan/address_types.h>
#include <epan/strutil.h>
#include <epan/prefs.h>
#include <epan/reassemble.h>
#include "packet-eapol.h"
#include "packet-ieee80211.h"
#include <epan/etypes.h>
#include <epan/oui.h>
#include <epan/crc32-tvb.h>
#include <epan/crypt/wep-wpadefs.h>
#include <epan/expert.h>
#include <epan/conversation_table.h>
#include <epan/uat.h>
#include <epan/proto_data.h>
#include <epan/tfs.h>
#include <epan/unit_strings.h>
#include <wsutil/array.h>
#include <wsutil/bits_ctz.h>

#include "packet-wps.h"
#include "packet-e212.h"
#include "packet-sflow.h"
#include "packet-gre.h"

#include <epan/crypt/dot11decrypt_ws.h>

void proto_register_ieee80211(void);
void proto_reg_handoff_ieee80211(void);
void proto_register_wlan_rsna_eapol(void);

static dissector_handle_t centrino_handle;

typedef struct {
  DOT11DECRYPT_KEY_ITEM used_key;
  unsigned keydata_len;
  uint8_t *keydata;
} proto_keydata_t;

extern value_string_ext eap_type_vals_ext; /* from packet-eap.c */

/* TUs are used a lot in 802.11 ... */
static const unit_name_string units_tu_tus = { "TU", "TUs" };

/* DIs are also used */
static const unit_name_string units_di_dis = { "DI", "DIs" };

/* az min/max time units */
static const unit_name_string units_100_us = { " times 100 microseconds", NULL };
static const unit_name_string units_10_ms = { " times 10 milliseconds", NULL };

#define IS_2_4_GHZ(freq) (freq >= 2400 && freq <= 2500)
#define IS_5PLUS GHZ(freq) (freq >= 5000)
#define IS_6_GHZ(freq) (freq >= 5955)

/*
 * We keep STA properties here, like whether they are S1G STAs or DMG STAs.
 * This is based on looking at BEACONs, or perhaps from the radiotap header
 * if we get one.
 */
static wmem_map_t *sta_prop_hash;

/*
 * Not sure that they can be both, so are bit values wanted?
 */
#define STA_IS_S1G 0x00000001
#define STA_IS_DMG 0x00000002

/*
 * Add the top three bytes of the STA address to the bottom three bytes
 */
static unsigned
sta_prop_hash_fn(const void *k)
{
  return wmem_strong_hash((const uint8_t *)k, 6);
}

static gboolean
sta_prop_equal_fn(const void *v, const void *w)
{
  const uint8_t *k1 = (const uint8_t *)v;
  const uint8_t *k2 = (const uint8_t *)w;

  return memcmp(k1, k2, 6) == 0; /* Compare each address for equality */
}

/* bitmask for bits [l..h]
 * taken from kernel's include/linux/bitops.h
 */
#define GENMASK(h, l)  (((1U << ((h) - (l) + 1)) - 1) << (l))
#define GENMASK64(h, l)  (((UINT64_C(1) << ((h) - (l) + 1)) - 1) << (l))

/* Defragment fragmented 802.11 datagrams */
static bool wlan_defragment = true;

/* call subdissector for retransmitted frames */
static bool wlan_subdissector = true;

/* Check for the presence of the 802.11 FCS */
static bool wlan_check_fcs;

/* Check the FCS checksum */
static bool wlan_check_checksum;

/* Ignore vendor-specific HT elements */
static bool wlan_ignore_draft_ht;

/* Ignore the Protection bit; assume packet is decrypted */
#define WLAN_IGNORE_PROT_NO     0
#define WLAN_IGNORE_PROT_WO_IV  1
#define WLAN_IGNORE_PROT_W_IV   2
static int wlan_ignore_prot = WLAN_IGNORE_PROT_NO;

/* The Key MIC len has been set by the user */
static bool wlan_key_mic_len_enable;
static unsigned wlan_key_mic_len;

/* Counter incremented on each (re)association
 * This value will be assigned to each packet's pinfo->srcport/pinfo->destport
 * as a way to uniquely make a one to one mapping between conversations and
 * associations
 */
static uint32_t association_counter;

/* Treat all Wi-Fi frames as being S1G frames where it is important */
static bool treat_as_s1g;

/* Table for reassembly of fragments. */
static reassembly_table wlan_reassembly_table;

/* Statistical data */
static struct _wlan_stats wlan_stats;

/*-------------------------------------
 * UAT for WEP decoder
 *-------------------------------------
 */
static uat_wep_key_record_t *uat_wep_key_records;
static uat_t                *wep_uat;
static unsigned              num_wepkeys_uat;

static void *
uat_wep_key_record_copy_cb(void* n, const void* o, size_t siz _U_)
{
  uat_wep_key_record_t* new_key = (uat_wep_key_record_t *)n;
  const uat_wep_key_record_t* old_key = (const uat_wep_key_record_t *)o;

  new_key->string = g_strdup(old_key->string);

  return new_key;
}

static bool
uat_wep_key_record_update_cb(void* r, char** err)
{
  uat_wep_key_record_t* rec = (uat_wep_key_record_t *)r;
  decryption_key_t* dk;
  unsigned dk_type;

  if (rec->string == NULL) {
    *err = g_strdup("Key can't be blank");
    return false;
  }

  *err = NULL;
  g_strstrip(rec->string);
  dk = parse_key_string(rec->string, rec->key, err);

  if (dk != NULL) {
    dk_type = dk->type;
    free_key_string(dk);
    switch (dk_type) {
      case DOT11DECRYPT_KEY_TYPE_WEP:
      case DOT11DECRYPT_KEY_TYPE_WEP_40:
      case DOT11DECRYPT_KEY_TYPE_WEP_104:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_WEP) {
          *err = g_strdup("Invalid WEP key format");
          return false;
        }
        break;
      case DOT11DECRYPT_KEY_TYPE_WPA_PWD:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_WPA_PWD) {
          *err = g_strdup("Invalid WPA_PWD key format");
          return false;
        }
        break;
      case DOT11DECRYPT_KEY_TYPE_WPA_PSK:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_WPA_PSK) {
          *err = g_strdup("Invalid WPA_PSK key format");
          return false;
        }
        break;
      case DOT11DECRYPT_KEY_TYPE_TK:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_TK) {
          *err = g_strdup("Invalid TK key format");
          return false;
        }
        break;
      case DOT11DECRYPT_KEY_TYPE_MSK:
        if (rec->key != DOT11DECRYPT_KEY_TYPE_MSK) {
          *err = g_strdup("Invalid MSK key format");
          return false;
        }
        break;
      default:
        *err = g_strdup("Invalid key format");
        return false;
    }
  } else {
    if (*err == NULL) {
        *err = g_strdup("Invalid key format");
    }
    return false;
  }
  return true;
}

static void
uat_wep_key_record_free_cb(void*r)
{
  uat_wep_key_record_t* key = (uat_wep_key_record_t *)r;
  g_free(key->string);
}

UAT_VS_DEF(uat_wep_key_records, key, uat_wep_key_record_t, uint8_t, 0, STRING_KEY_TYPE_WEP)
UAT_CSTRING_CB_DEF(uat_wep_key_records, string, uat_wep_key_record_t)

/* Stuff for the WEP/WPA/WPA2 decoder */
static bool enable_decryption = true;

static void
ieee_80211_add_tagged_parameters(tvbuff_t *tvb, int offset, packet_info *pinfo,
                                  proto_tree *tree, int tagged_parameters_len, int ftype,
                                  association_sanity_check_t *association_sanity_check);


static void
save_proto_data(tvbuff_t *tvb, packet_info *pinfo, int offset, size_t size, int key);

static void
save_proto_data_value(packet_info *pinfo, unsigned value, int key);

static void try_scan_tdls_keys(tvbuff_t *tvb, packet_info *pinfo, int offset);

static void try_scan_ft_assoc_keys(packet_info *pinfo, const wlan_hdr_t *whdr);

static tvbuff_t *
try_decrypt(tvbuff_t *tvb, packet_info *pinfo, unsigned offset, unsigned len,
            uint8_t *algorithm, uint32_t *sec_trailer,
            PDOT11DECRYPT_KEY_ITEM used_key);

static int weak_iv(unsigned char *iv);

typedef struct mimo_control
{
  uint8_t nc;
  uint8_t nr;
  bool chan_width;
  uint8_t grouping;
  uint8_t coefficient_size;
  uint8_t codebook_info;
  uint8_t remaining_matrix_segment;
} mimo_control_t;

/* ************************************************************************* */
/*                          Miscellaneous Constants                          */
/* ************************************************************************* */
#define SHORT_STR 256

typedef enum {
  IS_DMG_KEY = 1,
  IS_AP_KEY,
  IS_CTRL_GRANT_OR_GRANT_ACK_KEY,
  IS_S1G_KEY,
  DECRYPTED_EAPOL_KEY,
  DECRYPTED_GTK_KEY,
  PACKET_DATA_KEY,
  ASSOC_COUNTER_KEY,
  STA_KEY,
  BSSID_KEY,
  NONCE_KEY,
  GROUP_CIPHER_KEY,
  CIPHER_KEY,
  AKM_KEY,
  MIC_KEY,
  MIC_LEN_KEY,
  KEY_VERSION_KEY,
  KEY_LEN_KEY,
  KEY_IV_KEY,
  KEY_DATA_KEY,
  KEY_DATA_LEN_KEY,
  GTK_KEY,
  GTK_LEN_KEY,
  MDID_KEY,
  FTE_R0KH_ID_KEY,
  FTE_R0KH_ID_LEN_KEY,
  FTE_R1KH_ID_KEY,
  FTE_R1KH_ID_LEN_KEY,
  FTE_ANONCE_KEY,
  FTE_SNONCE_KEY,
  FTE_MIC_KEY,
  FTE_MIC_LEN_KEY,
  FTE_TAG_KEY,
  MDE_TAG_KEY,
  RSNE_TAG_KEY,
  RSNXE_TAG_KEY,
  RDE_TAG_KEY,
  GTK_SUBELEM_KEY_LEN_KEY,
  PASN_DATA_KEY,
  HE_CHANNEL_WIDTH_KEY,
  FRAME_TYPE_KEY,
  MLD_MAC_KEY,
  MLO_LINK_COUNT,
  MLO_LINK_ID_FIRST,
  MLO_LINK_ID_LAST = MLO_LINK_ID_FIRST + DOT11DECRYPT_MAX_MLO_LINKS - 1,
  MLO_LINK_MAC_FIRST,
  MLO_LINK_MAC_LAST = MLO_LINK_MAC_FIRST + DOT11DECRYPT_MAX_MLO_LINKS - 1,
  MLO_GTK_COUNT,
  MLO_GTK_LINK_ID_FIRST,
  MLO_GTK_LINK_ID_LAST = MLO_GTK_LINK_ID_FIRST + DOT11DECRYPT_MAX_MLO_LINKS - 1,
  MLO_GTK_KEY_FIRST,
  MLO_GTK_KEY_LAST = MLO_GTK_KEY_FIRST + DOT11DECRYPT_MAX_MLO_LINKS - 1,
  MLO_GTK_KEY_LEN_FIRST,
  MLO_GTK_KEY_LEN_LAST = MLO_GTK_KEY_LEN_FIRST + DOT11DECRYPT_MAX_MLO_LINKS - 1,
} wlan_proto_key_t;

/* ************************************************************************* */
/*  Define some very useful macros that are used to analyze frame types etc. */
/* ************************************************************************* */

/*
 * Fetch the frame control field and swap it if needed.  "fcf" and "tvb"
 * must be valid variables.
 */
#define FETCH_FCF(off) ((option_flags & IEEE80211_COMMON_OPT_BROKEN_FC) ? \
  GUINT16_SWAP_LE_BE(tvb_get_letohs(tvb, off)) : \
  tvb_get_letohs(tvb, off))

/*
 * Extract the fragment number and sequence number from the sequence
 * control field.
 */
#define SEQCTL_FRAGMENT_NUMBER(x) ((x) & 0x000F)
#define SEQCTL_SEQUENCE_NUMBER(x) (((x) & 0xFFF0) >> 4)

/*
 * Extract subfields from the QoS control field.
 */
#define QOS_TID(x)            ((x) & 0x000F)
#define QOS_PRIORITY(x)       ((x) & 0x0007)
#define QOS_EOSP(x)           (((x) & 0x0010) >> 4) /* end of service period */
#define QOS_ACK_POLICY(x)     (((x) & 0x0060) >> 5)
#define QOS_AMSDU_PRESENT(x)  (((x) & 0x0080) >> 6)
#define QOS_FIELD_CONTENT(x)  (((x) & 0xFF00) >> 8)
#define QOS_SCALING_FACTOR(x)  (((x) & 0xD0) >> 6)
#define QOS_UNSCALED_VALUE(x)  ((x) & 0x3F)
#define QOS_MESH_CONTROL_PRESENT(x) (((x) & 0x0100) >> 8)

#define QOS_FLAG_EOSP    0x0010

/*
 * Extract subfields from the result of QOS_FIELD_CONTENT().
 */
#define QOS_PS_BUF_STATE_INDICATED(x)  (((x) & 0x02) >> 1)
#define QOS_PS_HIGHEST_PRI_BUF_AC(x)   (((x) & 0x0C) >> 2)
#define QOS_PS_QAP_BUF_LOAD(x)         (((x) & 0xF0) >> 4)

/*
 * Bits from the HT Control field.
 * 802.11-2016 9.2.4.6, and 802.11ax draft, 32 bits.
 */
#define HTC_VHT              0x00000001
#define HTC_HE               0x00000002
#define HTC_MRQ              0x00000004
#define HTC_UNSOLICITED_MFB  0x20000000

/*
 * Extract subfields from the HT Control field.
 */
#define HTC_LAC(htc)           ((htc) & 0xFE)
#define HTC_LAC_MAI(htc)       (((htc) >> 2) & 0xF)
#define HTC_IS_ASELI(htc)      (HTC_LAC_MAI(htc) == 0xE)
#define HTC_LAC_MAI_MRQ(htc)   ((HTC_LAC_MAI(htc))  & 0x1)
#define HTC_LAC_MAI_MSI(htc)   ((HTC_LAC_MAI(htc) >> 1) & 0x7)
#define HTC_LAC_MFSI(htc)      (((htc) >> 4) & 0x7)
#define HTC_LAC_ASEL_CMD(htc)  (((htc) >> 9) & 0x7)
#define HTC_LAC_ASEL_DATA(htc) (((htc) >> 12) & 0xF)
#define HTC_LAC_MFB(htc)       (((htc) >> 9) & 0x7F)
#define HTC_CAL_POS(htc)       (((htc) >> 16) & 0x3)
#define HTC_CAL_SEQ(htc)       (((htc) >> 18) & 0x3)
#define HTC_CSI_STEERING(htc)  (((htc) >> 22) & 0x3)
#define HTC_NDP_ANN(htc)       (((htc) >> 24) & 0x1)
#define HTC_AC_CONSTRAINT(htc) (((htc) >> 30) & 0x1)
#define HTC_RDG_MORE_PPDU(htc) (((htc) >> 31) & 0x1)

#define HTC_MFB(htc)           (((htc) >> 9) & 0x7FFF)

/* VHT-MCS = 15, NUM_STS = 7 */
#define HTC_NO_FEEDBACK_PRESENT(mfb) (((mfb) & 0x7F) == 0x7F)

/*
 * Extract subfields from the key octet in WEP-encrypted frames.
 */
#define KEY_OCTET_WEP_KEY(x)   (((x) & 0xC0) >> 6)

#define KEY_EXTIV    0x20
#define EXTIV_LEN    8

/*
 * Bits from the Mesh Flags field
 */
#define MESH_FLAGS_ADDRESS_EXTENSION  0x3

/* ************************************************************************* */
/*              Constants used to identify cooked frame types                */
/* ************************************************************************* */
#define MGT_FRAME            0x00  /* Frame type is management */
#define CONTROL_FRAME        0x01  /* Frame type is control */
#define DATA_FRAME           0x02  /* Frame type is Data */
#define EXTENSION_FRAME      0x03  /* Frame type is Extension */

#define DATA_SHORT_HDR_LEN     24
#define DATA_LONG_HDR_LEN      30
#define MGT_FRAME_HDR_LEN      24  /* Length of Management frame-headers */

/* ************************************************************************* */
/*        Logical field codes (IEEE 802.11 encoding of tags)                 */
/* ************************************************************************* */
const value_string ie_tag_num_vals[] = {
  { TAG_SSID,                                 "SSID parameter set" },
  { TAG_SUPP_RATES,                           "Supported Rates" },
  { TAG_FH_PARAMETER,                         "FH Parameter set" },
  { TAG_DS_PARAMETER,                         "DS Parameter set" },
  { TAG_CF_PARAMETER,                         "CF Parameter set" },
  { TAG_TIM,                                  "Traffic Indication Map (TIM)" },
  { TAG_IBSS_PARAMETER,                       "IBSS Parameter set" },
  { TAG_COUNTRY_INFO,                         "Country Information" },
  { TAG_FH_HOPPING_PARAMETER,                 "Hopping Pattern Parameters" },
  { TAG_FH_HOPPING_TABLE,                     "Hopping Pattern Table" },
  { TAG_REQUEST,                              "Request" },
  { TAG_QBSS_LOAD,                            "QBSS Load Element" },
  { TAG_EDCA_PARAM_SET,                       "EDCA Parameter Set" },
  { TAG_TSPEC,                                "Traffic Specification" },
  { TAG_TCLAS,                                "Traffic Classification" },
  { TAG_SCHEDULE,                             "Schedule" },
  { TAG_CHALLENGE_TEXT,                       "Challenge text" },
  { TAG_POWER_CONSTRAINT,                     "Power Constraint" },
  { TAG_POWER_CAPABILITY,                     "Power Capability" },
  { TAG_TPC_REQUEST,                          "TPC Request" },
  { TAG_TPC_REPORT,                           "TPC Report" },
  { TAG_SUPPORTED_CHANNELS,                   "Supported Channels" },
  { TAG_CHANNEL_SWITCH_ANN,                   "Channel Switch Announcement" },
  { TAG_MEASURE_REQ,                          "Measurement Request" },
  { TAG_MEASURE_REP,                          "Measurement Report" },
  { TAG_QUIET,                                "Quiet" },
  { TAG_IBSS_DFS,                             "IBSS DFS" },
  { TAG_ERP_INFO,                             "ERP Information" },
  { TAG_TS_DELAY,                             "TS Delay" },
  { TAG_TCLAS_PROCESS,                        "TCLAS Processing" },
  { TAG_HT_CAPABILITY,                        "HT Capabilities" },
  { TAG_QOS_CAPABILITY,                       "QoS Capability" },
  { TAG_ERP_INFO_OLD,                         "ERP Information" }, /* Reserved... */
  { TAG_RSN_IE,                               "RSN Information" },
  { TAG_EXT_SUPP_RATES,                       "Extended Supported Rates" },
  { TAG_AP_CHANNEL_REPORT,                    "AP Channel Report" },
  { TAG_NEIGHBOR_REPORT,                      "Neighbor Report" },
  { TAG_RCPI,                                 "RCPI" },
  { TAG_MOBILITY_DOMAIN,                      "Mobility Domain" },
  { TAG_FAST_BSS_TRANSITION,                  "Fast BSS Transition" },
  { TAG_TIMEOUT_INTERVAL,                     "Timeout Interval" },
  { TAG_RIC_DATA,                             "RIC Data" },
  { TAG_DSE_REG_LOCATION,                     "DSE Registered Location" },
  { TAG_SUPPORTED_OPERATING_CLASSES,          "Supported Operating Classes" },
  { TAG_EXTENDED_CHANNEL_SWITCH_ANNOUNCEMENT, "Extended Channel Switch Announcement" },
  { TAG_HT_OPERATION,                         "HT Operation" },
  { TAG_SECONDARY_CHANNEL_OFFSET,             "Secondary Channel Offset (802.11n D1.10)" },
  { TAG_BSS_AVG_ACCESS_DELAY,                 "BSS Average Access Delay" },
  { TAG_ANTENNA,                              "Antenna" },
  { TAG_RSNI,                                 "RSNI" },
  { TAG_MEASURE_PILOT_TRANS,                  "Measurement Pilot Transmission" },
  { TAG_BSS_AVB_ADM_CAPACITY,                 "BSS Available Admission Capacity" },
  { TAG_IE_68_CONFLICT,                       "BSS AC Access Delay/WAPI Parameter Set" },
  { TAG_TIME_ADV,                             "Time Advertisement" },
  { TAG_RM_ENABLED_CAPABILITY,                "RM Enabled Capabilities" },
  { TAG_MULTIPLE_BSSID,                       "Multiple BSSID" },
  { TAG_20_40_BSS_CO_EX,                      "20/40 BSS Coexistence" },
  { TAG_20_40_BSS_INTOL_CH_REP,               "20/40 BSS Intolerant Channel Report" },   /* IEEE P802.11n/D6.0 */
  { TAG_OVERLAP_BSS_SCAN_PAR,                 "Overlapping BSS Scan Parameters" },       /* IEEE P802.11n/D6.0 */
  { TAG_RIC_DESCRIPTOR,                       "RIC Descriptor" },
  { TAG_MMIE,                                 "Management MIC" },
  { TAG_EVENT_REQUEST,                        "Event Request" },
  { TAG_EVENT_REPORT,                         "Event Report" },
  { TAG_DIAGNOSTIC_REQUEST,                   "Diagnostic Request" },
  { TAG_DIAGNOSTIC_REPORT,                    "Diagnostic Report" },
  { TAG_LOCATION_PARAMETERS,                  "Location Parameters" },
  { TAG_NO_BSSID_CAPABILITY,                  "Non Transmitted BSSID Capability" },
  { TAG_SSID_LIST,                            "SSID List" },
  { TAG_MULTIPLE_BSSID_INDEX,                 "Multiple BSSID Index" },
  { TAG_FMS_DESCRIPTOR,                       "FMS Descriptor" },
  { TAG_FMS_REQUEST,                          "FMS Request" },
  { TAG_FMS_RESPONSE,                         "FMS Response" },
  { TAG_QOS_TRAFFIC_CAPABILITY,               "QoS Traffic Capability" },
  { TAG_BSS_MAX_IDLE_PERIOD,                  "BSS Max Idle Period" },
  { TAG_TFS_REQUEST,                          "TFS Request" },
  { TAG_TFS_RESPONSE,                         "TFS Response" },
  { TAG_WNM_SLEEP_MODE,                       "WNM-Sleep Mode" },
  { TAG_TIM_BROADCAST_REQUEST,                "TIM Broadcast Request" },
  { TAG_TIM_BROADCAST_RESPONSE,               "TIM Broadcast Response" },
  { TAG_COLLOCATED_INTER_REPORT,              "Collocated Interference Report" },
  { TAG_CHANNEL_USAGE,                        "Channel Usage" },
  { TAG_TIME_ZONE,                            "Time Zone" },
  { TAG_DMS_REQUEST,                          "DMS Request" },
  { TAG_DMS_RESPONSE,                         "DMS Response" },
  { TAG_LINK_IDENTIFIER,                      "Link Identifier" },
  { TAG_WAKEUP_SCHEDULE,                      "Wakeup Schedule" },
  { TAG_CHANNEL_SWITCH_TIMING,                "Channel Switch Timing" },
  { TAG_PTI_CONTROL,                          "PTI Control" },
  { TAG_PU_BUFFER_STATUS,                     "PU Buffer Status" },
  { TAG_INTERWORKING,                         "Interworking" },
  { TAG_ADVERTISEMENT_PROTOCOL,               "Advertisement Protocol"},
  { TAG_EXPIDITED_BANDWIDTH_REQ,              "Expedited Bandwidth Request" },
  { TAG_QOS_MAP_SET,                          "QoS Map Set" },
  { TAG_ROAMING_CONSORTIUM,                   "Roaming Consortium" },
  { TAG_EMERGENCY_ALERT_ID,                   "Emergency Alert Identifier" },
  { TAG_MESH_CONFIGURATION,                   "Mesh Configuration" },
  { TAG_MESH_ID,                              "Mesh ID" },
  { TAG_MESH_LINK_METRIC_REPORT,              "Mesh Link Metric Report" },
  { TAG_CONGESTION_NOTIFICATION,              "Congestion Notification" },
  { TAG_MESH_PEERING_MGMT,                    "Mesh Peering Management" },
  { TAG_MESH_CHANNEL_SWITCH,                  "Mesh Channel Switch Parameters" },
  { TAG_MESH_AWAKE_WINDOW,                    "Mesh Awake Window" },
  { TAG_BEACON_TIMING,                        "Beacon Timing" },
  { TAG_MCCAOP_SETUP_REQUEST,                 "MCCAOP Setup Request" },
  { TAG_MCCAOP_SETUP_REPLY,                   "MCCAOP SETUP Reply" },
  { TAG_MCCAOP_ADVERTISEMENT,                 "MCCAOP Advertisement" },
  { TAG_MCCAOP_TEARDOWN,                      "MCCAOP Teardown" },
  { TAG_GANN,                                 "Gate Announcement" },
  { TAG_RANN,                                 "Root Announcement" },
  { TAG_EXTENDED_CAPABILITIES,                "Extended Capabilities" },
  { TAG_AGERE_PROPRIETARY,                    "Agere Proprietary" },
  { TAG_MESH_PREQ,                            "Path Request" },
  { TAG_MESH_PREP,                            "Path Reply" },
  { TAG_MESH_PERR,                            "Path Error" },
  { TAG_CISCO_CCX1_CKIP,                      "Cisco CCX1 CKIP + Device Name" },
  { TAG_CISCO_CCX2,                           "Cisco CCX2" },
  { TAG_PXU,                                  "Proxy Update" },
  { TAG_PXUC,                                 "Proxy Update Confirmation"},
  { TAG_AUTH_MESH_PEERING_EXCH,               "Authenticated Mesh Peering Exchange" },
  { TAG_MIC,                                  "MIC (Message Integrity Code)" },
  { TAG_DESTINATION_URI,                      "Destination URI" },
  { TAG_U_APSD_COEX,                          "U-APSD Coexistence" },
  { TAG_WAKEUP_SCHEDULE_AD,                   "Wakeup Schedule 802.11ad" },
  { TAG_EXTENDED_SCHEDULE,                    "Extended Schedule" },
  { TAG_STA_AVAILABILITY,                     "STA Availability" },
  { TAG_DMG_TSPEC,                            "DMG TSPEC" },
  { TAG_NEXT_DMG_ATI,                         "Next DMG ATI" },
  { TAG_DMG_CAPABILITIES,                     "DMG Capabilities" },
  { TAG_CISCO_CCX3,                           "Cisco Unknown 95" },
  { TAG_CISCO_VENDOR_SPECIFIC,                "Vendor Specific" },
  { TAG_DMG_OPERATION,                        "DMG Operating" },
  { TAG_DMG_BSS_PARAMETER_CHANGE,             "DMG BSS Parameter Change" },
  { TAG_DMG_BEAM_REFINEMENT,                  "DMG Beam Refinement" },
  { TAG_CHANNEL_MEASURMENT_FB,                "Channel Measurement Feedback" },
  { TAG_AWAKE_WINDOW,                         "Awake Window" },
  { TAG_MULTI_BAND,                           "Multi Band" },
  { TAG_ADDBA_EXT,                            "ADDBA Extension" },
  { TAG_NEXTPCP_LIST,                         "NEXTPCP List" },
  { TAG_PCP_HANDOVER,                         "PCP Handover" },
  { TAG_DMG_LINK_MARGIN,                      "DMG Link Margin" },
  { TAG_SWITCHING_STREAM,                     "Switching Stream" },
  { TAG_SESSION_TRANSMISSION,                 "Session Transmission" },
  { TAG_DYN_TONE_PAIR_REP,                    "Dynamic Tone Pairing Report" },
  { TAG_CLUSTER_REP,                          "Cluster Report" },
  { TAG_RELAY_CAPABILITIES,                   "Relay Capabilities" },
  { TAG_RELAY_TRANSFER_PARAM,                 "Relay Transfer Parameter" },
  { TAG_BEAMLINK_MAINTENANCE,                 "Beamlink Maintenance" },
  { TAG_MULTIPLE_MAC_SUBLAYERS,               "Multiple MAC Sublayers" },
  { TAG_U_PID,                                "U-PID" },
  { TAG_DMG_LINK_ADAPTION_ACK,                "DMG Link Adaption Acknowledgment" },
  { TAG_SYMBOL_PROPRIETARY,                   "Symbol Proprietary" },
  { TAG_MCCAOP_ADVERTISEMENT_OV,              "MCCAOP Advertisement Overview" },
  { TAG_QUIET_PERIOD_REQ,                     "Quiet Period Request" },
  { TAG_QUIET_PERIOD_RES,                     "Quiet Period Response" },
  { TAG_ECAPC_POLICY,                         "ECAPC Policy" },
  { TAG_CLUSTER_TIME_OFFSET,                  "Cluster Time Offset" },
  { TAG_INTRA_ACCESS_CAT_PRIO,                "Intra-Access Category Priority" },
  { TAG_SCS_DESCRIPTOR,                       "SCS Descriptor" },
  { TAG_ANTENNA_SECTOR_ID,                    "Antenna Sector ID" },
  { TAG_VHT_CAPABILITY,                       "VHT Capabilities" },
  { TAG_VHT_OPERATION,                        "VHT Operation" },
  { TAG_EXT_BSS_LOAD,                         "Extended BSS Load" },
  { TAG_WIDE_BW_CHANNEL_SWITCH,               "Wide Bandwidth Channel Switch" },
  { TAG_TX_PWR_ENVELOPE,                      "Tx Power Envelope" },
  { TAG_CHANNEL_SWITCH_WRAPPER,               "Channel Switch Wrapper" },
  { TAG_OPERATING_MODE_NOTIFICATION,          "Operating Mode Notification" },
  { TAG_REDUCED_NEIGHBOR_REPORT,              "Reduced Neighbor Report" },
  { TAG_FINE_TIME_MEASUREMENT_PARAM,          "Fine Time Measurement Params" },
  { TAG_S1G_OPEN_LOOP_LINK_MARGIN_INDEX,      "S1G Open-Loop Link Margin Index" },
  { TAG_RPS,                                  "RPS" },
  { TAG_PAGE_SLICE,                           "Page Slice" },
  { TAG_AID_REQUEST,                          "AID Request" },
  { TAG_AID_RESPONSE,                         "AID Response" },
  { TAG_S1G_SECTOR_OPERATION,                 "Sector Operation" },
  { TAG_S1G_BEACON_COMPATIBILITY,             "S1G Beacon Compatibility" },
  { TAG_SHORT_BEACON_INTERVAL,                "Short Beacon Interval" },
  { TAG_CHANGE_SEQUENCE,                      "Change Sequence" },
  { TAG_TWT,                                  "Target Wake Time" },
  { TAG_S1G_CAPABILITIES,                     "S1G Capabilities" },
  { TAG_SUBCHANNEL_SELECTIVE_TRANSMISSION,    "Subchannel Selective Transmission" },
  { TAG_VENDOR_SPECIFIC_IE,                   "Vendor Specific" },
  { TAG_AUTHENTICATION_CONTROL,               "Authentication Control" },
  { TAG_TSF_TIMER_ACCURACY,                   "TSF Timer Accuracy" },
  { TAG_S1G_RELAY,                            "S1G Relay" },
  { TAG_REACHABLE_ADDRESS,                    "Reachable Address" },
  { TAG_S1G_RELAY_DISCOVERY,                  "S1G Relay Discovery" },
  { TAG_AID_ANNOUNCEMENT,                     "AID Announcement" },
  { TAG_PV1_PROBE_RESPONSE_OPTION,            "PV1 Probe Response Option" },
  { TAG_EL_OPERATION,                         "EL Operation" },
  { TAG_SECTORIZED_GROUP_ID_LIST,             "Sectorized Group ID List" },
  { TAG_S1G_OPERATION,                        "S1G Operation" },
  { TAG_HEADER_COMPRESSION,                   "Header Compression" },
  { TAG_SST_OPERATION,                        "SST Operation" },
  { TAG_MAD,                                  "MAD" },
  { TAG_S1G_RELAY_ACTIVATION,                 "S1G Relay Activation" },
  { TAG_CAG_NUMBER,                           "CAG Number"},
  { TAG_AP_CSN,                               "AP-CSN"},
  { TAG_FILS_INDICATION,                      "FILS Indication"},
  { TAG_DIFF_INITIAL_LINK_SETUP,              "Differential Initial Link Setup"},
  { TAG_FRAGMENT,                             "Fragment"},
  { TAG_RSNX,                                 "RSN eXtension"},
  { TAG_ELEMENT_ID_EXTENSION,                 "Element ID Extension" },
  { 0, NULL }
};

static value_string_ext tag_num_vals_ext = VALUE_STRING_EXT_INIT(ie_tag_num_vals);

#define ETAG_ASSOC_DELAY_INFO          1
#define ETAG_FILS_REQ_PARAMS           2
#define ETAG_FILS_KEY_CONFIRM          3
#define ETAG_FILS_SESSION              4
#define ETAG_FILS_HLP_CONTAINER        5
#define ETAG_FILS_IP_ADDRESS_ASSIGN    6
#define ETAG_KEY_DELIVERY              7
#define ETAG_FILS_WRAPPED_DATA         8
#define ETAG_FTM_SYNC_INFO             9
#define ETAG_EXTENDED_REQUEST          10
#define ETAG_ESTIMATED_SERVICE_PARAM   11
#define ETAG_FILS_PUBLIC_KEY           12
#define ETAG_FILS_NONCE                13
#define ETAG_FUTURE_CHANNEL_GUIDANCE   14

/* RFC 8110 */
#define ETAG_OWE_DH_PARAMETER          32

#define ETAG_PASSWORD_IDENTIFIER       33

/* 802.11AX defined tags */
#define ETAG_HE_CAPABILITIES                   35
#define ETAG_HE_OPERATION                      36
#define ETAG_UORA_PARAMETER_SET                37
#define ETAG_MU_EDCA_PARAMETER_SET             38
#define ETAG_SPATIAL_REUSE_PARAMETER_SET       39
#define ETAG_NDP_FEEDBACK_REPORT_PARAMETER_SET 41
#define ETAG_BSS_COLOR_CHANGE_ANNOUNCEMENT     42
#define ETAG_QUIET_TIME_PERIOD_SETUP           43
#define ETAG_ESS_REPORT                        45
#define ETAG_OPS                               46
#define ETAG_HE_BSS_LOAD                       47
#define ETAG_MAX_CHANNEL_SWITCH_TIME           52
#define ETAG_OCI                               54
#define ETAG_MULTIPLE_BSSID_CONFIGURATION      55
#define ETAG_NON_INHERITANCE                   56
#define ETAG_KNOWN_BSSID                       57
#define ETAG_SHORT_SSID                        58
#define ETAG_HE_6GHZ_BAND_CAPABILITIES         59
#define ETAG_UL_MU_POWER_CAPABILITIES          60
#define ETAG_MSCS_DESCRIPTOR_ELEMENT           88
#define ETAG_TCLAS_MASK                        89
#define ETAG_REJECTED_GROUPS                   92
#define ETAG_ANTI_CLOGGING_TOKEN               93

/* 802.11az */
#define ETAG_SECURE_LTF_PARAMETERS                      94
#define ETAG_ISTA_PASSIVE_TB_RANGING_MEASUREMENT_REPORT 95
#define ETAG_RSTA_PASSIVE_TB_RANGING_MEASUREMENT_REPORT 96
#define ETAG_PASSIVE_TB_RANGING_LCI_TABLE_ELEMENT       97
#define ETAG_ISTA_AVAILABILITY_WINDOW                   98
#define ETAG_RSTA_AVAILABILITY_WINDOW                   99
#define ETAG_PASN_PARAMETERS                            100
#define ETAG_RANGING_PARAMETERS                         101
#define ETAG_DIRECTION_MEASUREMENT_RESULTS              102
#define ETAG_MULTIPLE_AOD_FEEDBACK                      103
#define ETAG_MULTIPLE_BEST_AWV_ID                       104
#define ETAG_LOS_LIKELIHOOD                             105
#define ETAG_EHT_OPERATION                              106
#define ETAG_MULTI_LINK                                 107
#define ETAG_EHT_CAPABILITIES                           108
#define ETAG_TID_TO_LINK_MAPPING                        109
#define ETAG_MULTI_LINK_TRAFFIC                         110
#define ETAG_QOS_CHARACTERISTICS                        113
#define ETAG_AKM_SUITE_SELECTOR                         114
#define ETAG_MLO_LINK_INFORMATION                       133
#define ETAG_AID_BITMAP                                 134
#define ETAG_BANDWIDTH_INDICATION                       135
#define ETAG_NONAP_STA_REGULATORY_CONNECT               137


static const value_string tag_num_vals_eid_ext[] = {
  { ETAG_ASSOC_DELAY_INFO,                    "Association Delay Info" },
  { ETAG_FILS_REQ_PARAMS,                     "FILS Request Parameters" },
  { ETAG_FILS_KEY_CONFIRM,                    "FILS Key Confirmation" },
  { ETAG_FILS_SESSION,                        "FILS Session" },
  { ETAG_FILS_HLP_CONTAINER,                  "FILS HLP Container" },
  { ETAG_FILS_IP_ADDRESS_ASSIGN,              "FILS IP Address Assignment" },
  { ETAG_KEY_DELIVERY,                        "Key Delivery" },
  { ETAG_FILS_WRAPPED_DATA,                   "Wrapped Data" },
  { ETAG_FTM_SYNC_INFO,                       "FTM Synchronization Information" },
  { ETAG_EXTENDED_REQUEST,                    "Extended Request" },
  { ETAG_ESTIMATED_SERVICE_PARAM,             "Estimated Service Parameters" },
  { ETAG_FILS_PUBLIC_KEY,                     "FILS Public Key" },
  { ETAG_FILS_NONCE,                          "FILS Nonce" },
  { ETAG_FUTURE_CHANNEL_GUIDANCE,             "Future Channel Guidance" },
  { ETAG_OWE_DH_PARAMETER,                    "OWE Diffie-Hellman Parameter" },
  { ETAG_PASSWORD_IDENTIFIER,                 "Password Identifier" },
  { ETAG_HE_CAPABILITIES,                     "HE Capabilities" },
  { ETAG_HE_OPERATION,                        "HE Operation" },
  { ETAG_UORA_PARAMETER_SET,                  "UORA Parameter Set" },
  { ETAG_MU_EDCA_PARAMETER_SET,               "MU EDCA Parameter Set" },
  { ETAG_SPATIAL_REUSE_PARAMETER_SET,         "Spatial Reuse Parameter Set" },
  { ETAG_NDP_FEEDBACK_REPORT_PARAMETER_SET,   "NDP Feedback Report Parameter Set" },
  { ETAG_BSS_COLOR_CHANGE_ANNOUNCEMENT,       "BSS Color Change Announcement" },
  { ETAG_QUIET_TIME_PERIOD_SETUP,             "Quiet Time Period Setup" },
  { ETAG_ESS_REPORT,                          "ESS Report" },
  { ETAG_OPS,                                 "Opportunistic Power Save" },
  { ETAG_HE_BSS_LOAD,                         "HE BSS Load" },
  { ETAG_MAX_CHANNEL_SWITCH_TIME,             "Max Channel Switch Time" },
  { ETAG_OCI,                                 "Operating Channel Information" },
  { ETAG_MULTIPLE_BSSID_CONFIGURATION,        "Multiple BSSID Configuration" },
  { ETAG_NON_INHERITANCE,                     "Non-Inheritance"},
  { ETAG_KNOWN_BSSID,                         "Known BSSID" },
  { ETAG_SHORT_SSID,                          "Short SSID" },
  { ETAG_HE_6GHZ_BAND_CAPABILITIES,           "HE 6 GHz Band Capabilities" },
  { ETAG_UL_MU_POWER_CAPABILITIES,            "UL MU Power Capabilities" },
  { ETAG_MSCS_DESCRIPTOR_ELEMENT,             "MSCS Descriptor Element" },
  { ETAG_TCLAS_MASK,                          "TCLAS Mask" },
  { ETAG_REJECTED_GROUPS,                     "Rejected Groups" },
  { ETAG_ANTI_CLOGGING_TOKEN,                 "Anti-Clogging Token Container" },
  /* 802.11az */
  { ETAG_SECURE_LTF_PARAMETERS,               "Secure LTF Parameters" },
  { ETAG_ISTA_PASSIVE_TB_RANGING_MEASUREMENT_REPORT, "ISTA Passive TB Ranging Measurement Report" },
  { ETAG_RSTA_PASSIVE_TB_RANGING_MEASUREMENT_REPORT, "RSTA Passive TB Ranging Measurement Report" },
  { ETAG_PASSIVE_TB_RANGING_LCI_TABLE_ELEMENT, "Passive TB Ranging LCI Table element" },
  { ETAG_ISTA_AVAILABILITY_WINDOW,            "ISTA Availability Window" },
  { ETAG_RSTA_AVAILABILITY_WINDOW,            "RSTA Availability Window" },
  { ETAG_PASN_PARAMETERS,                     "PASN Parameters" },
  { ETAG_RANGING_PARAMETERS,                  "Ranging Parameters" },
  { ETAG_DIRECTION_MEASUREMENT_RESULTS,       "Direction Measurement Results" },
  { ETAG_MULTIPLE_AOD_FEEDBACK,               "Multiple AOD Feedback" },
  { ETAG_MULTIPLE_BEST_AWV_ID,                "Multiple Best AWV ID" },
  { ETAG_LOS_LIKELIHOOD,                      "LOS Likelihood" },
  { ETAG_EHT_OPERATION,                       "EHT Operation" },
  { ETAG_MULTI_LINK,                          "Multi-Link" },
  { ETAG_EHT_CAPABILITIES,                    "EHT Capabilities" },
  { ETAG_TID_TO_LINK_MAPPING,                 "TID-To-Link Mapping" },
  { ETAG_MULTI_LINK_TRAFFIC,                  "Multi-Link Traffic Indication" },
  { ETAG_QOS_CHARACTERISTICS,                 "QoS Characteristics" },
  { ETAG_AKM_SUITE_SELECTOR,                  "AKM Suite Selector" },
  { ETAG_MLO_LINK_INFORMATION,                "MLO Link Information" },
  { ETAG_AID_BITMAP,                          "AID Bitmap" },
  { ETAG_BANDWIDTH_INDICATION,                "Bandwidth Indication" },
  { ETAG_NONAP_STA_REGULATORY_CONNECT,        "Non-AP STA Regulatory Connectivity" },
  { 0, NULL }
};
static value_string_ext tag_num_vals_eid_ext_ext = VALUE_STRING_EXT_INIT(tag_num_vals_eid_ext);

static const value_string wfa_qos_subtype_vals[] = {
  { 0, "DSCP Policy Query" },
  { 1, "DSCP Policy Request" },
  { 2, "DSCP Policy Response" },
  { 0, NULL }
};

static const value_string wfa_action_subtype_vals[] = {
  { WFA_SUBTYPE_ACTION_QOS_MGMT, "QoS Management" },
  { WFA_SUBTYPE_ACTION_CAPABILITIES, "Wi-Fi Alliance Capabilities" },
  { 0, NULL }
};

const value_string wfa_subtype_vals[] = {
  { WFA_SUBTYPE_SUBSCRIPTION_REMEDIATION, "Subscription Remediation" },
  { WFA_SUBTYPE_DEAUTHENTICATION_IMMINENT, "Deauthentication Imminent" },
  { WFA_WNM_SUBTYPE_NON_PREF_CHAN_REPORT, "Non-preferred Channel Report" },
  { WFA_WNM_SUBTYPE_CELL_DATA_CAPABILITIES, "Cellular Data Capabilities" },
  { WFA_SUBTYPE_P2P, "P2P" },
  { WFA_SUBTYPE_HS20_INDICATION, "Hotspot 2.0 Indication" },
  { WFA_SUBTYPE_NAN_IE, "NAN" },
  { WFA_SUBTYPE_OSEN, "OSU Server-only l2 Encryption Network" },
  { WFA_SUBTYPE_MBO_OCE, "Multi Band Operation - Optimized Connectivity Experience"},
  { WFA_SUBTYPE_WIFI_60G, "60GHz Information Element" },
  { WFA_SUBTYPE_NAN_ACTION, "NAN Action" },
  { WFA_SUBTYPE_DPP, "Device Provisioning Protocol" },
  { WFA_SUBTYPE_IEEE1905_MULTI_AP, "IEEE1905 Multi-AP" },
  { WFA_SUBTYPE_OWE_TRANSITION_MODE, "OWE Transition Mode" },
  { WFA_SUBTYPE_TRANSITION_DISABLE_KDE, "Transition Disable KDE" },
  { WFA_SUBTYPE_QOS_MGMT, "QoS Management" },
  { WFA_SUBTYPE_WFA_CAPA, "Wi-Fi Alliance Capabilities" },
  { WFA_SUBTYPE_RSN_OVERRIDE, "RSN Element Override" },
  { WFA_SUBTYPE_RSN_OVERRIDE_2, "RSN Element Override 2" },
  { WFA_SUBTYPE_RSNX_OVERRIDE, "RSN Extension Element Override" },
  { WFA_SUBTYPE_RSN_SELECTION, "RSN Selection" },
  { WFA_SUBTYPE_RSN_OVERRIDE_LINK_KDE, "RSN Override Link KDE" },
  { 0, NULL }
};

static const value_string wfa_anqp_subtype_vals[] = {
  { WFA_ANQP_SUBTYPE_HS20, "Hotspot 2.0 ANQP" },
  { WFA_ANQP_SUBTYPE_MBO, "Multi Band Operation ANQP" },
  { 0, NULL }
};

#define DPP_CONFIGURATION_PROTOCOL 0x01

static const value_string dpp_subtype_vals[] = {
  { DPP_CONFIGURATION_PROTOCOL, "DPP Configuration Protocol" },
  { 0, NULL }
};

/* ************************************************************************* */
/*              Supported Rates (7.3.2.2)                                    */
/* ************************************************************************* */

static const value_string ieee80211_supported_rates_vals[] = {
  { 0x02, "1" },
  { 0x03, "1.5" },
  { 0x04, "2" },
  { 0x05, "2.5" },
  { 0x06, "3" },
  { 0x09, "4.5" },
  { 0x0B, "5.5" },
  { 0x0C, "6" },
  { 0x12, "9" },
  { 0x16, "11" },
  { 0x18, "12" },
  { 0x1B, "13.5" },
  { 0x24, "18" },
  { 0x2C, "22" },
  { 0x30, "24" },
  { 0x36, "27" },
  { 0x42, "33" },
  { 0x48, "36" },
  { 0x60, "48" },
  { 0x6C, "54" },
  { 0x82, "1(B)" },
  { 0x83, "1.5(B)" },
  { 0x84, "2(B)" },
  { 0x85, "2.5(B)" },
  { 0x86, "3(B)" },
  { 0x89, "4.5(B)" },
  { 0x8B, "5.5(B)" },
  { 0x8C, "6(B)" },
  { 0x92, "9(B)" },
  { 0x96, "11(B)" },
  { 0x98, "12(B)" },
  { 0x9B, "13.5(B)" },
  { 0xA4, "18(B)" },
  { 0xAC, "22(B)" },
  { 0xB0, "24(B)" },
  { 0xB6, "27(B)" },
  { 0xC2, "33(B)" },
  { 0xC8, "36(B)" },
  { 0xE0, "48(B)" },
  { 0xEC, "54(B)" },
  /* BSS membership selector */
  { 0xFA, "HE PHY" },
  { 0xFB, "SAE Hash to Element Only" },
  { 0xFC, "EPD" }, /* 802.11ak */
  { 0xFD, "GLK" }, /* 802.11ak */
  { 0xFE, "VHT PHY" },
  { 0xFF, "HT PHY" },
  { 0,    NULL}
};
value_string_ext ieee80211_supported_rates_vals_ext = VALUE_STRING_EXT_INIT(ieee80211_supported_rates_vals);

/* ************************************************************************* */
/*                         8.4.1.7 Reason Code field                         */
/* ************************************************************************* */
static const value_string ieee80211_reason_code[] = {
  {  1, "Unspecified reason" },
  {  2, "Previous authentication no longer valid" },
  {  3, "Deauthenticated because sending STA is leaving (or has left) the BSS" },
  {  4, "Disassociated due to inactivity" },
  {  5, "Disassociated because AP is unable to handle all currently associated STAs" },
  {  6, "Class 2 frame received from nonauthenticated STA" },
  {  7, "Class 3 frame received from nonassociated STA" },
  {  8, "Disassociated because sending STA is leaving (or has left) BSS" },
  {  9, "STA requesting (re)association is not authenticated with responding STA" },
  { 10, "Disassociated because the information in the Power Capability element is unacceptable" },
  { 11, "Disassociated because the information in the Supported Channels element is unacceptable" },
  { 12, "Disassociated due to BSS transition management" },
  { 13, "Invalid information element, i.e., an information element defined in this standard for which the content does not meet the specifications in Clause 9" },
  { 14, "Message integrity code (MIC) failure" },
  { 15, "4-way handshake timeout" },
  { 16, "Group key handshake timeout" },
  { 17, "Element in 4-way handshake different from (Re)Association Request/Probe Response/Beacon frame" },
  { 18, "Invalid group cipher" },
  { 19, "Invalid pairwise cipher" },
  { 20, "Invalid AKMP" },
  { 21, "Unsupported RSNE version" },
  { 22, "Invalid RSNE capabilities" },
  { 23, "IEEE 802.1X authentication failed" },
  { 24, "Cipher suite rejected because of the security policy" },
  { 25, "TDLS direct-link teardown due to TDLS peer STA unreachable via the TDLS direct link" },
  { 26, "TDLS direct-link teardown for unspecified reason" },
  { 27, "Disassociated because session terminated by SSP request" },
  { 28, "Disassociated because of lack of SSP roaming agreement" },
  { 29, "Requested service rejected because of SSP cipher suite or AKM requirement " },
  { 30, "Requested service not authorized in this location" },
  { 31, "TS deleted because QoS AP lacks sufficient bandwidth for this QoS STA due to a change in BSS service characteristics or operational mode" },
  { 32, "Disassociated for unspecified, QoS-related reason" },
  { 33, "Disassociated because QoS AP lacks sufficient bandwidth for this QoS STA" },
  { 34, "Disassociated because excessive number of frames need to be acknowledged, but are not acknowledged due to AP transmissions and/or poor channel conditions" },
  { 35, "Disassociated because STA is transmitting outside the limits of its TXOPs" },
  { 36, "Requested from peer STA as the STA is leaving the BSS (or resetting)" },
  { 37, "Requesting STA is no longer using the stream or session" },
  { 38, "Requesting STA received frames using a mechanism for which a setup has not been completed" },
  { 39, "Requested from peer STA due to timeout" },
  { 46, "Disassociated because authorized access limit reached" },
  { 47, "Disassociated due to external service requirements" },
  { 48, "Invalid FT Action frame count" },
  { 49, "Invalid pairwise master key identifier (PMKID)" },
  { 50, "Invalid MDE" },
  { 51, "Invalid FTE" },
  { 52, "Mesh peering canceled for unknown reasons" },
  { 53, "The mesh STA has reached the supported maximum number of peer mesh STAs" },
  { 54, "The received information violates the Mesh Configuration policy configured in the mesh STA profile" },
  { 55, "The mesh STA has received a Mesh Peering Close message requesting to close the mesh peering" },
  { 56, "The mesh STA has re-sent dot11MeshMaxRetries Mesh Peering Open messages, without receiving a Mesh Peering Confirm message" },
  { 57, "The confirmTimer for the mesh peering instance times out" },
  { 58, "The mesh STA fails to unwrap the GTK or the values in the wrapped contents do not match" },
  { 59, "The mesh STA receives inconsistent information about the mesh parameters between Mesh Peering Management frames" },
  { 60, "The mesh STA fails the authenticated mesh peering exchange because due to failure in selecting either the pairwise ciphersuite or group ciphersuite" },
  { 61, "The mesh STA does not have proxy information for this external destination" },
  { 62, "The mesh STA does not have forwarding information for this destination" },
  { 63, "The mesh STA determines that the link to the next hop of an active path in its forwarding information is no longer usable" },
  { 64, "The Deauthentication frame was sent because the MAC address of the STA already exists in the mesh BSS. See 11.3.3 (Additional mechanisms for an AP collocated with a mesh STA)" },
  { 65, "The mesh STA performs channel switch to meet regulatory requirements" },
  { 66, "The mesh STA performs channel switch with unspecified reason" },
  { 67, "Transmission link establishment in alternative channel failed" },
  { 68, "The alternative channel is occupied" },
  { 71, "Disassociated due to poor RSSI" },
  { 0,  NULL}
};
value_string_ext ieee80211_reason_code_ext = VALUE_STRING_EXT_INIT(ieee80211_reason_code);

/* ************************************************************************* */
/*                         8.4.1.9 Status Code field                         */
/* ************************************************************************* */
static const value_string ieee80211_status_code[] = {
  {  0, "Successful" },
  {  1, "Unspecified failure" },
  {  2, "TDLS wakeup schedule rejected but alternative schedule provided" },
  {  3, "TDLS wakeup schedule rejected" },
  {  4, "Reserved" },
  {  5, "Security disabled" },
  {  6, "Unacceptable lifetime" },
  {  7, "Not in same BSS" },
  {  8, "Reserved" },
  {  9, "Reserved" },
  { 10, "Cannot support all requested capabilities in the Capability Information field" },
  { 11, "Reassociation denied due to inability to confirm that association exists" },
  { 12, "Association denied due to reason outside the scope of this standard" },
  { 13, "Responding STA does not support the specified authentication algorithm" },
  { 14, "Received an Authentication frame with authentication transaction sequence number out of expected sequence" },
  { 15, "Authentication rejected because of challenge failure" },
  { 16, "Authentication rejected due to timeout waiting for next frame in sequence" },
  { 17, "Association denied because AP is unable to handle additional associated STAs" },
  { 18, "Association denied due to requesting STA not supporting all of the data rates in the BSSBasicRateSet parameter, the Basic HT-MCS Set field of the HT Operation parameter, or the Basic VHT-MCS and NSS Set field in the VHT Operation parameter" },
  { 19, "Association denied due to requesting STA not supporting the short preamble option" },
  { 20, "Reserved" },
  { 21, "Reserved" },
  { 22, "Association request rejected because spectrum management capability is required" },
  { 23, "Association request rejected because the information in the Power Capability element is unacceptable" },
  { 24, "Association request rejected because the information in the Supported Channels element is unacceptable" },
  { 25, "Association denied due to requesting STA not supporting short slot time" },
  { 26, "Reserved" },
  { 27, "Association denied because the requesting STA does not support HT features" },
  { 28, "R0KH unreachable" },
  { 29, "Reserved"},
  { 30, "Association request rejected temporarily; try again later" },
  { 31, "Robust management frame policy violation" },
  { 32, "Unspecified, QoS-related failure" },
  { 33, "Association denied because QoS AP or PCP has insufficient bandwidth to handle another QoS STA" },
  { 34, "Association denied due to excessive frame loss rates and/ or poor conditions on current operating channel" },
  { 35, "Association (with QoS BSS) denied because the requesting STA does not support the QoS facility" },
  { 36, "Reserved" },
  { 37, "The request has been declined" },
  { 38, "The request has not been successful as one or more parameters have invalid values" },
  { 39, "The allocation or TS has not been created because the request cannot be honored; however, a suggested TSPEC/DMG TSPEC is provided so that the initiating STA can attempt to set another allocation or TS with the suggested changes to the TSPEC/DMG TSPEC" },
  { 40, "Invalid element, i.e., an element defined in this standard for which the content does not meet the specifications in Clause 9 (Frame formats)" },
  { 41, "Invalid group cipher" },
  { 42, "Invalid pairwise cipher" },
  { 43, "Invalid AKMP" },
  { 44, "Unsupported RSNE version" },
  { 45, "Invalid RSNE capabilities" },
  { 46, "Cipher suite rejected because of security policy" },
  { 47, "The TS or allocation has not been created; however, the HC or PCP might be capable of creating a TS or allocation, in response to a request, after the time indicated in the TS Delay element" },
  { 48, "Reserved" },
  { 49, "The Destination STA is not present within this BSS" },
  { 50, "The Destination STA is not a QoS STA" },
  { 51, "Association denied because the listen interval is too large" },
  { 52, "Invalid FT Action frame count" },
  { 53, "Invalid pairwise master key identifier (PMKID)" },
  { 54, "Invalid MDE" },
  { 55, "Invalid FTE" },
  { 56, "Requested TCLAS processing is not supported by the AP or PCP" },
  { 57, "The AP or PCP has insufficient TCLAS processing resources to satisfy the request" },
  { 58, "The TS has not been created because the request cannot be honored; however, the HC or PCP suggests that the STA transition to a different BSS to set up the TS" },
  { 59, "GAS advertisement protocol not supported" },
  { 60, "No outstanding GAS request" },
  { 61, "GAS response not received from the advertisement server" },
  { 62, "STA timed out waiting for GAS query response" },
  { 63, "GAS response is larger than query response length limit" },
  { 64, "Request refused because home network does not support request" },
  { 65, "Advertisement server in the network is not currently reachable" },
  { 66, "Reserved" },
  { 67, "Request refused due to permissions received via SSPN interface" },
  { 68, "Request refused because the AP or PCP does not support unauthenticated access" },
  { 69, "Reserved" },
  { 70, "Reserved" },
  { 71, "Reserved" },
  { 72, "Invalid contents of RSNE, other than unsupported RSNE version or invalid RSNE capabilities, AKMP or pairwise cipher" },
  { 73, "U-APSD coexistence is not supported" },
  { 74, "Requested U-APSD coexistence mode is not supported" },
  { 75, "Requested interval/duration value cannot be supported with U-APSD coexistence" },
  { 76, "Authentication is rejected because an anti-clogging token is required" },
  { 77, "Authentication is rejected because the offered finite cyclic group is not supported" },
  { 78, "The TBTT adjustment request has not been successful because the STA could not find an alternative TBTT" },
  { 79, "Transmission failure" },
  { 80, "Requested TCLAS not supported" },
  { 81, "TCLAS resources exhausted" },
  { 82, "Rejected with suggested BSS transition" },
  { 83, "Reject with recommended schedule" },
  { 84, "Reject because no wakeup schedule specified" },
  { 85, "Success, the destination STA is in power save mode" },
  { 86, "FST pending, in process of admitting FST session" },
  { 87, "Performing FST now" },
  { 88, "FST pending, gap(s) in block ack window" },
  { 89, "Reject because of U-PID setting" },
  { 90, "Reserved" },
  { 91, "Reserved" },
  { 92, "(Re)Association refused for some external reason" },
  { 93, "(Re)Association refused because of memory limits at the AP" },
  { 94, "(Re)Association refused because emergency services are not supported at the AP" },
  { 95, "GAS query response not yet received" },
  { 96, "Reject since the request is for transition to a frequency band subject to DSE procedures and FST Initiator is a dependent STA" },
  { 97, "Requested TCLAS processing has been terminated by the AP" },
  { 98, "The TS schedule conflicts with an existing schedule; an alternative schedule is provided" },
  { 99, "The association has been denied; however, one or more Multi-band elements are included that can be used by the receiving STA to join the BSS" },
  { 100, "The request failed due to a reservation conflict" },
  { 101, "The request failed due to exceeded MAF limit" },
  { 102, "The request failed due to exceeded MCCA track limit" },
  { 103, "Association denied because the information in the Spectrum Management field is unacceptable" },
  { 104, "Association denied because the requesting STA does not support VHT features" },
  { 105, "Enablement denied" },
  { 106, "Enablement denied due to restriction from an authorized GDB" },
  { 107, "Authorization deenabled" },
  { 108, "Re(association) refused or disassociated because energy limited operation is not supported at the AP" },
  { 109, "BlockAck negotiation refused because, due to buffer constraints and other unspecified reasons, the recipient prefers to generate only NDP BlockAck frames" },
  { 110, "Association denied/disassociated because the suggested value for max away duration is unacceptable" },
  { 111, "Re(association) refused or disassociated because flow control operation is not supported by the non-AP STA" },
  { 112, "Authentication rejected due to FILS authentication failure" },
  { 113, "Authentication rejected due to unknown Authentication Server" },
  { 114, "Reserved" },
  { 115, "Reserved" },
  { 116, "Request denied because the allocation of notification period is failed" },
  { 117, "Request denied because the request of channel splitting is failed" },
  { 118, "Request denied because the allocation request is failed" },
  { 119, "Association denied because the requesting STA does not support CMMG features" },
  { 120, "The requested GAS fragment is not available" },
  { 121, "Success, the CAG Version provided by the requesting STA is the same as the latest CAG Version provided by the relevant server" },
  { 122, "The STA is not authorized to use GLK per local policy" },
  { 123, "Authentication rejected because the password identifier is unknown" },
  { 124, "Reserved"},
  { 125, "Request denied because source address of request is inconsistent with local MAC address policy" },
  { 126, "SAE authentication uses direct hashing, instead of looping, to obtain the PWE" },
  { 127, "Reserved"},
  { 128, "Requested TCLAS processing has been terminated by the AP due to insufficient QoS capacity" },
  { 129, "Requested TCLAS processing has been terminated by the AP due to conflict with higher layer QoS policies" },
  { 130, "Association denied because the requesting STA is affiliated with a non-AP MLD that is associated with the AP MLD"},
  { 131, "EPCS priority access denied because the non-AP MLD is not authorized to use the service"},
  { 132, "EPCS priority access denied due to reason outside the scope of this standard"},
  { 133, "Request denied because the requested TID-to-link mapping is unacceptable"},
  { 134, "Preferred TID-to-link mapping suggested"},
  { 135, "Association denied because the requesting STA does not support EHT features"},
  { 139, "Link not accepted because the link on which the (Re)Association Request frame is transmitted is not accepted"},
  { 140, "EPCS priority access is temporarily denied because the receiving AP MLD is unable to verify that the non-AP MLD is authorized for an unspecified reason"},
  { 141, "Operation parameter update denied because the requested operation parameters or capabilities are not acceptable"},
  { 142, "The non-AP STA MAC address is used by an existing associated non-AP STA"},
  {   0, NULL}
};
value_string_ext ieee80211_status_code_ext = VALUE_STRING_EXT_INIT(ieee80211_status_code);

static const value_string ieee80211_transition_reasons[] = {
  { 0, "Unspecified" },
  { 1, "Excessive frame loss rates and/or poor conditions" },
  { 2, "Excessive delay for current traffic streams" },
  { 3, "Insufficient QoS capacity for current traffic streams (TSPEC rejected)" },
  { 4, "First association to ESS (the association initiated by an Association Request frame instead of a Reassociation Request frame)" },
  { 5, "Load balancing" },
  { 6, "Better AP found" },
  { 7, "Deauthenticated or Disassociated from the previous AP" },
  { 8, "AP failed IEEE 802.1X EAP Authentication" },
  { 9, "AP failed 4-way handshake" },
  { 10, "Received too many replay counter failures" },
  { 11, "Received too many data MIC failures" },
  { 12, "Exceeded maximum number of retransmissions" },
  { 13, "Received too many broadcast disassociations" },
  { 14, "Received too many broadcast deauthentications" },
  { 15, "Previous transition failed" },
  { 16, "Low RSSI" },
  { 17, "Roam from a non-IEEE-802.11 system" },
  { 18, "Transition due to received BSS Transition Request frame" },
  { 19, "Preferred BSS transition candidate list included" },
  { 20, "Leaving ESS" },
  { 0,    NULL}
};
/* ************************************************************************* */
/*                         Frame types, and their names                      */
/* ************************************************************************* */
static const value_string frame_type_subtype_vals[] = {
  {MGT_ASSOC_REQ,             "Association Request"},
  {MGT_ASSOC_RESP,            "Association Response"},
  {MGT_REASSOC_REQ,           "Reassociation Request"},
  {MGT_REASSOC_RESP,          "Reassociation Response"},
  {MGT_PROBE_REQ,             "Probe Request"},
  {MGT_PROBE_RESP,            "Probe Response"},
  {MGT_MEASUREMENT_PILOT,     "Measurement Pilot"},
  {MGT_BEACON,                "Beacon frame"},
  {MGT_ATIM,                  "ATIM"},
  {MGT_DISASS,                "Disassociate"},
  {MGT_AUTHENTICATION,        "Authentication"},
  {MGT_DEAUTHENTICATION,      "Deauthentication"},
  {MGT_ACTION,                "Action"},
  {MGT_ACTION_NO_ACK,         "Action No Ack"},
  {MGT_ARUBA_WLAN,            "Aruba Management"},

  {CTRL_TRIGGER,              "Trigger"},
  {CTRL_TACK,                 "TWT Ack"},
  {CTRL_BEAMFORM_RPT_POLL,    "Beamforming Report Poll"},
  {CTRL_VHT_NDP_ANNC,         "VHT/HE/EHT/RANGING NDP Announcement"},
  {CTRL_CONTROL_WRAPPER,      "Control Wrapper"},
  {CTRL_BLOCK_ACK_REQ,        "802.11 Block Ack Req"},
  {CTRL_BLOCK_ACK,            "802.11 Block Ack"},
  {CTRL_PS_POLL,              "Power-Save poll"},
  {CTRL_RTS,                  "Request-to-send"},
  {CTRL_CTS,                  "Clear-to-send"},
  {CTRL_ACKNOWLEDGEMENT,      "Acknowledgement"},
  {CTRL_CFP_END,              "CF-End (Control-frame)"},
  {CTRL_CFP_ENDACK,           "CF-End + CF-Ack (Control-frame)"},

  {DATA,                      "Data"},
  {DATA_CF_ACK,               "Data + CF-Ack"},
  {DATA_CF_POLL,              "Data + CF-Poll"},
  {DATA_CF_ACK_POLL,          "Data + CF-Ack + CF-Poll"},
  {DATA_NULL_FUNCTION,        "Null function (No data)"},
  {DATA_CF_ACK_NOD,           "Acknowledgement (No data)"},
  {DATA_CF_POLL_NOD,          "CF-Poll (No data)"},
  {DATA_CF_ACK_POLL_NOD,      "CF-Ack/Poll (No data)"},
  {DATA_QOS_DATA,             "QoS Data"},
  {DATA_QOS_DATA_CF_ACK,      "QoS Data + CF-Acknowledgment"},
  {DATA_QOS_DATA_CF_POLL,     "QoS Data + CF-Poll"},
  {DATA_QOS_DATA_CF_ACK_POLL, "QoS Data + CF-Ack + CF-Poll"},
  {DATA_QOS_NULL,             "QoS Null function (No data)"},
  {DATA_QOS_CF_POLL_NOD,      "QoS CF-Poll (No Data)"},
  {DATA_QOS_CF_ACK_POLL_NOD,  "QoS CF-Ack + CF-Poll (No data)"},

  {EXTENSION_DMG_BEACON,      "DMG Beacon"},
  {EXTENSION_S1G_BEACON,      "S1G Beacon"},

  {CTRL_POLL,                 "Poll"},
  {CTRL_SPR,                  "Service Period Request"},
  {CTRL_GRANT,                "Grant"},
  {CTRL_DMG_CTS,              "DMG Clear-to-send"},
  {CTRL_DMG_DTS,              "DMG Denial-to-send"},
  {CTRL_GRANT_ACK,            "Grant Acknowledgment"},
  {CTRL_SSW,                  "Sector Sweep"},
  {CTRL_SSW_FEEDBACK,         "Sector Sweep Feedback"},
  {CTRL_SSW_ACK,              "Sector Sweep Acknowledgment"},

  {0,                         NULL}
};
static value_string_ext frame_type_subtype_vals_ext = VALUE_STRING_EXT_INIT(frame_type_subtype_vals);

/* ************************************************************************* */
/*                 802.1D Tag Name (by WME Access Category Names)            */
/* ************************************************************************* */
static const value_string ieee80211_qos_tags_acs[] = {
  { 0, "Best Effort (Best Effort)" },
  { 1, "Background (Background)" },
  { 2, "Spare (Background)" },
  { 3, "Excellent Effort (Best Effort)" },
  { 4, "Controlled Load (Video)" },
  { 5, "Video (Video)" },
  { 6, "Voice (Voice)" },
  { 7, "Network Control (Voice)" },
  { 0, NULL }
};

/* ************************************************************************* */
/*                   WME Access Category Names (by WME ACI)                  */
/* ************************************************************************* */
static const value_string wme_acs[] = {
  { 0, "Best Effort" },
  { 1, "Background" },
  { 2, "Video" },
  { 3, "Voice" },
  { 0, NULL }
};

/* ************************************************************************* */
/*                  Aruba Management Type                                    */
/* ************************************************************************* */
static const value_string aruba_mgt_typevals[] = {
  { 0x0001,       "Hello" },
  { 0x0002,       "Probe" },
  { 0x0003,       "MTU" },
  { 0x0004,       "Ageout" },
  { 0x0005,       "Heartbeat" },
  { 0x0006,       "Deauth" },
  { 0x0007,       "Disassoc" },
  { 0x0008,       "Probe response" },
  { 0x0009,       "Tunnel update" },
  { 0x000A,       "Laser beam active" },
  { 0x000B,       "Client IP" },
  { 0x000C,       "Laser beam active v2" },
  { 0x000D,       "AP statistics" },
  { 0,            NULL }
};
static value_string_ext aruba_mgt_typevals_ext = VALUE_STRING_EXT_INIT(aruba_mgt_typevals);

/*** Begin: Action Fixed Parameter ***/
#define CAT_SPECTRUM_MGMT          0
#define CAT_QOS                    1
#define CAT_DLS                    2
#define CAT_BLOCK_ACK              3
#define CAT_PUBLIC                 4
#define CAT_RADIO_MEASUREMENT      5
#define CAT_FAST_BSS_TRANSITION    6
#define CAT_HT                     7
#define CAT_SA_QUERY               8
#define CAT_PUBLIC_PROTECTED       9
#define CAT_WNM                   10
#define CAT_UNPROTECTED_WNM       11
#define CAT_TDLS                  12
#define CAT_MESH                  13
#define CAT_MULTIHOP              14
#define CAT_SELF_PROTECTED        15
#define CAT_DMG                   16
#define CAT_MGMT_NOTIFICATION     17
#define CAT_FAST_SESSION_TRANSFER 18
#define CAT_ROBUST_AV_STREAMING   19
#define CAT_UNPROTECTED_DMG       20
#define CAT_VHT                   21
#define CAT_S1G                   22
#define CAT_PROTECTED_S1G         23
#define CAT_FLOW_CONTROL          24
#define CAT_CONTROL_RESPONSE_MCS_NEG 25
#define CAT_FILS                  26
/* aj */
#define CAT_CDMG                  27
#define CAT_CMMG                  28
/* ak */
#define CAT_GLK                   29

#define CAT_HE                    30
#define CAT_PROTECTED_HE          31
#define CAT_PROTECTED_FTM         34
#define CAT_EHT                   36
#define CAT_PROTECTED_EHT         37
#define CAT_VENDOR_SPECIFIC_PROTECTED 126
#define CAT_VENDOR_SPECIFIC     127

#define CAT_MESH_LINK_METRIC               31
#define CAT_MESH_PATH_SELECTION            32
#define CAT_MESH_INTERWORKING              33
#define CAT_MESH_RESOURCE_COORDINATION     34
#define CAT_MESH_SECURITY_ARCHITECTURE     35

#define SM_ACTION_MEASUREMENT_REQUEST   0
#define SM_ACTION_MEASUREMENT_REPORT    1
#define SM_ACTION_TPC_REQUEST           2
#define SM_ACTION_TPC_REPORT            3
#define SM_ACTION_CHAN_SWITCH_ANNC      4
#define SM_ACTION_EXT_CHAN_SWITCH_ANNC  5

#define QOS_ACTION_ADDTS_REQUEST     0
#define QOS_ACTION_ADDTS_RESPONSE    1
#define QOS_ACTION_DELTS             2
#define QOS_ACTION_SCHEDULE      3
#define QOS_ACTION_MAP_CONFIGURE 4

#define DLS_ACTION_REQUEST       0
#define DLS_ACTION_RESPONSE      1
#define DLS_ACTION_TEARDOWN      2

#define BA_ADD_BLOCK_ACK_REQUEST      0
#define BA_ADD_BLOCK_ACK_RESPONSE     1
#define BA_DELETE_BLOCK_ACK           2

#define BA_NDP_ADD_BLOCK_ACK_REQUEST  128
#define BA_NDP_ADD_BLOCK_ACK_RESPONSE 129
#define BA_NDP_DELETE_BLOCK_ACK       130
#define BA_BAT_ADD_BLOCK_ACK_REQUEST  132
#define BA_BAT_ADD_BLOCK_ACK_RESPONSE 133
#define BA_BAT_DELETE_BLOCK_ACK       134

#define BA_ADD_BLOCK_ACK_REQUEST    0
#define BA_ADD_BLOCK_ACK_RESPONSE   1
#define BA_DELETE_BLOCK_ACK         2

/* Keep in sync with PPA_* defines */
#define PA_20_40_BSS_COEXISTENCE_MANAGEMENT 0
#define PA_DSE_ENABLEMENT                   1
#define PA_DSE_DEENABLEMENT                 2
#define PA_DSE_REG_LOC_ANNOUNCEMENT         3
#define PA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT  4
#define PA_DSE_MEASUREMENT_REQUEST          5
#define PA_DSE_MEASUREMENT_REPORT           6
#define PA_MEASUREMENT_PILOT                7
#define PA_DSE_POWER_CONSTRAINT             8
#define PA_VENDOR_SPECIFIC                  9
#define PA_GAS_INITIAL_REQUEST             10
#define PA_GAS_INITIAL_RESPONSE            11
#define PA_GAS_COMEBACK_REQUEST            12
#define PA_GAS_COMEBACK_RESPONSE           13
#define PA_TDLS_DISCOVERY_RESPONSE         14
#define PA_LOCATION_TRACK_NOTIFICATION     15
#define PA_QAB_REQUEST                     16
#define PA_QAB_RESPONSE                    17
#define PA_QMF_POLICY                      18
#define PA_QMF_POLICY_CHANGE               19
#define PA_QLOAD_REQUEST                   20
#define PA_QLOAD_REPORT                    21
#define PA_HCCA_TXOP_ADVERTISEMENT         22
#define PA_HCCA_TXOP_RESPONSE              23
#define PA_PUBLIC_KEY                      24
#define PA_CHANNEL_AVAILABILITY_QUERY      25
#define PA_CHANNEL_SCHEDULE_MANAGEMENT     26
#define PA_CONTACT_VERIFICATION_SIGNAL     27
#define PA_GDD_ENABLEMENT_REQUEST          28
#define PA_GDD_ENABLEMENT_RESPONSE         29
#define PA_NETWORK_CHANNEL_CONTROL         30
#define PA_WHITE_SPACE_MAP_ANNOUNCEMENT    31
#define PA_FTM_REQUEST                     32
#define PA_FTM                             33
#define PA_FILS_DISCOVERY                  34
/* 802.11aj */
#define PA_DCS_MEASUREMENT_REQUEST               35
#define PA_DCS_MEASUREMENT_REPORT                36
#define PA_DCS_REQUEST                           37
#define PA_DCS_RESPONSE                          38
#define PA_EXTENDED_NOTIFICATION_PERIOD_REQUEST  39
#define PA_EXTENDED_NOTIFICATION_PERIOD_RESPONSE 40
#define PA_EXTENDED_CHANNEL_SPLITTING_REQUEST    41
#define PA_EXTENDED_CHANNEL_SPLITTING_RESPONSE   42
/* 802.11aq */
#define PA_GROUP_ADDRESSED_GAS_REQUEST     43
#define PA_GROUP_ADDRESSED_GAS_RESPONSE    44
/* Unknown */
#define PA_ON_CHANNEL_TUNNEL_REQUEST       45
#define PA_VALUE_46                        46
/* 802.11az */
#define PA_LOCATION_MEASUREMENT_REPORT                                      47
#define PA_ISTA_PASSIVE_TB_RANGING_MEASUREMENT_REPORT                       48
#define PA_PRIMARY_RSTA_BROADCAST_PASSIVE_TB_RANGING_MEASUREMENT_REPORT     49
#define PA_SECONDARY_RSTA_BROADCAST_PASSIVE_TB_RANGING_MEASUREMENT_REPORT   50

/* Keep in sync with PA_* defines */
#define PPA_DSE_ENABLEMENT                   1
#define PPA_DSE_DEENABLEMENT                 2
#define PPA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT  4
#define PPA_DSE_MEASUREMENT_REQUEST          5
#define PPA_DSE_MEASUREMENT_REPORT           6
#define PPA_DSE_POWER_CONSTRAINT             8
#define PPA_VENDOR_SPECIFIC                  9
#define PPA_GAS_INITIAL_REQUEST             10
#define PPA_GAS_INITIAL_RESPONSE            11
#define PPA_GAS_COMEBACK_REQUEST            12
#define PPA_GAS_COMEBACK_RESPONSE           13
#define PPA_QAB_REQUEST                     16
#define PPA_QAB_RESPONSE                    17

#define HT_ACTION_NOTIFY_CHAN_WIDTH           0
#define HT_ACTION_SM_PWR_SAVE                 1
#define HT_ACTION_PSMP_ACTION                 2
#define HT_ACTION_SET_PCO_PHASE               3
#define HT_ACTION_MIMO_CSI                    4
#define HT_ACTION_MIMO_BEAMFORMING            5
#define HT_ACTION_MIMO_COMPRESSED_BEAMFORMING 6
#define HT_ACTION_ANT_SEL_FEEDBACK            7
#define HT_ACTION_HT_INFO_EXCHANGE            8

#define DMG_ACTION_PWR_SAVE_CONFIG_REQ           0
#define DMG_ACTION_PWR_SAVE_CONFIG_RES           1
#define DMG_ACTION_INFO_REQ                      2
#define DMG_ACTION_INFO_RES                      3
#define DMG_ACTION_HANDOVER_REQ                  4
#define DMG_ACTION_HANDOVER_RES                  5
#define DMG_ACTION_DTP_REQ                       6
#define DMG_ACTION_DTP_RES                       7
#define DMG_ACTION_RELAY_SEARCH_REQ              8
#define DMG_ACTION_RELAY_SEARCH_RES              9
#define DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_REQ 10
#define DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_RES 11
#define DMG_ACTION_RLS_REQ                       12
#define DMG_ACTION_RLS_RES                       13
#define DMG_ACTION_RLS_ANNOUNCE                  14
#define DMG_ACTION_RLS_TEARDOWN                  15
#define DMG_ACTION_RELAY_ACK_REQ                 16
#define DMG_ACTION_RELAY_ACK_RES                 17
#define DMG_ACTION_TPA_REQ                       18
#define DMG_ACTION_TPA_RES                       19
#define DMG_ACTION_TPA_REP                       20
#define DMG_ACTION_ROC_REQ                       21
#define DMG_ACTION_ROC_RES                       22

#define UNPROTECTED_DMG_ANNOUNCE                 0
#define UNPROTECTED_DMG_BRP                      1

#define FST_SETUP_REQUEST                        0
#define FST_SETUP_RESPONSE                       1
#define FST_TEAR_DOWN                            2
#define FST_ACK_REQUEST                          3
#define FST_ACK_RESPONSE                         4
#define FST_ON_CHANNEL_TUNNEL_REQUEST            5

#define ROBUST_AV_STREAMING_SCS_REQUEST          0
#define ROBUST_AV_STREAMING_SCS_RESPONSE         1
#define ROBUST_AV_STREAMING_GROUP_MEMBERSHIP_REQ 2
#define ROBUST_AV_STREAMING_GROUP_MEMBERSHIP_RSP 3
#define ROBUST_AV_STREAMING_MSCS_REQUEST         4
#define ROBUST_AV_STREAMING_MSCS_RESPONSE        5


/* IEEE Std 802.11r-2008, 7.4.8, Table 7-57g */
#define FT_ACTION_REQUEST               1
#define FT_ACTION_RESPONSE              2
#define FT_ACTION_CONFIRM               3
#define FT_ACTION_ACK                   4

/* SA Query Action frame codes (IEEE 802.11w-2009, 7.4.9) */
#define SA_QUERY_REQUEST                0
#define SA_QUERY_RESPONSE               1

/* IEEE Std 802.11z-2010, 7.4.11, Table 7-57v1 */
#define TDLS_SETUP_REQUEST              0
#define TDLS_SETUP_RESPONSE             1
#define TDLS_SETUP_CONFIRM              2
#define TDLS_TEARDOWN                   3
#define TDLS_PEER_TRAFFIC_INDICATION    4
#define TDLS_CHANNEL_SWITCH_REQUEST     5
#define TDLS_CHANNEL_SWITCH_RESPONSE    6
#define TDLS_PEER_PSM_REQUEST           7
#define TDLS_PEER_PSM_RESPONSE          8
#define TDLS_PEER_TRAFFIC_RESPONSE      9
#define TDLS_DISCOVERY_REQUEST          10

/* IEEE Std 802.11-2012, 8.5.7.1, Table 8-206 */
#define RM_ACTION_RADIO_MEASUREMENT_REQUEST         0
#define RM_ACTION_RADIO_MEASUREMENT_REPORT          1
#define RM_ACTION_LINK_MEASUREMENT_REQUEST          2
#define RM_ACTION_LINK_MEASUREMENT_REPORT           3
#define RM_ACTION_NEIGHBOR_REPORT_REQUEST           4
#define RM_ACTION_NEIGHBOR_REPORT_RESPONSE          5

/* 11s draft 12.0, table 7-57v30 */
#define MESH_ACTION_LINK_METRIC_REPORT              0
#define MESH_ACTION_HWMP                            1
#define MESH_ACTION_GATE_ANNOUNCE                   2
#define MESH_ACTION_CONGESTION_CTL                  3
#define MESH_ACTION_MCCA_SETUP_REQUEST              4
#define MESH_ACTION_MCCA_SETUP_REPLY                5
#define MESH_ACTION_MCCA_ADV_REQUEST                6
#define MESH_ACTION_MCCA_ADV                        7
#define MESH_ACTION_MCCA_TEARDOWN                   8
#define MESH_ACTION_TBTT_ADJ_REQUEST                9
#define MESH_ACTION_TBTT_ADJ_RESPONSE              10

/* 11s draft 12.0, table 7-57v42: Multihop Action field values */
#define MULTIHOP_ACTION_PROXY_UPDATE                0
#define MULTIHOP_ACTION_PROXY_UPDATE_CONF           1

/* 11s draft 12.0, table 7-57v24: Self-protected Action field values */
#define SELFPROT_ACTION_MESH_PEERING_OPEN           1
#define SELFPROT_ACTION_MESH_PEERING_CONFIRM        2
#define SELFPROT_ACTION_MESH_PEERING_CLOSE          3
#define SELFPROT_ACTION_MESH_GROUP_KEY_INFORM       4
#define SELFPROT_ACTION_MESH_GROUP_KEY_ACK          5

/* 11s draft 12.0, table 7-43bj6: Mesh Peering Protocol Identifier field values */
#define MESH_PEERING_PROTO_MGMT                     0
#define MESH_PEERING_PROTO_AMPE                     1
#define MESH_PEERING_PROTO_VENDOR                 255

/* Vendor actions */
/* MARVELL */
#define MRVL_ACTION_MESH_MANAGEMENT     1

#define MRVL_MESH_MGMT_ACTION_RREQ      0
#define MRVL_MESH_MGMT_ACTION_RREP      1
#define MRVL_MESH_MGMT_ACTION_RERR      2
#define MRVL_MESH_MGMT_ACTION_PLDM      3

#define ANQP_INFO_ANQP_QUERY_LIST                256
#define ANQP_INFO_ANQP_CAPAB_LIST                257
#define ANQP_INFO_VENUE_NAME_INFO                258
#define ANQP_INFO_EMERGENCY_CALL_NUMBER_INFO     259
#define ANQP_INFO_NETWORK_AUTH_TYPE_INFO         260
#define ANQP_INFO_ROAMING_CONSORTIUM_LIST        261
#define ANQP_INFO_IP_ADDR_TYPE_AVAILABILITY_INFO 262
#define ANQP_INFO_NAI_REALM_LIST                 263
#define ANQP_INFO_3GPP_CELLULAR_NETWORK_INFO     264
#define ANQP_INFO_AP_GEOSPATIAL_LOCATION         265
#define ANQP_INFO_AP_CIVIC_LOCATION              266
#define ANQP_INFO_AP_LOCATION_PUBLIC_ID_URI      267
#define ANQP_INFO_DOMAIN_NAME_LIST               268
#define ANQP_INFO_EMERGENCY_ALERT_ID_URI         269
#define ANQP_INFO_TDLS_CAPAB_INFO                270
#define ANQP_INFO_EMERGENCY_NAI                  271
#define ANQP_INFO_NEIGHBOR_REPORT                272
#define ANQP_INFO_QUERY_AP_LIST                  273
#define ANQP_INFO_AP_LIST_RESPONSE               274
#define ANQP_INFO_FILS_REALM_INFO                275
#define ANQP_INFO_CAG                            276
#define ANQP_INFO_VENUE_URL                      277
#define ANQP_INFO_ADVICE_OF_CHARGE               278
#define ANQP_INFO_LOCAL_CONTENT                  279
#define ANQP_INFO_NETWORK_AUTH_TYPE_TIMESTAMP    280
#define ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST    56797

/* ANQP information ID - IEEE Std 802.11u-2011 - Table 7-43bk */
static const value_string anqp_info_id_vals[] = {
  {ANQP_INFO_ANQP_QUERY_LIST, "ANQP Query list"},
  {ANQP_INFO_ANQP_CAPAB_LIST, "ANQP Capability list"},
  {ANQP_INFO_VENUE_NAME_INFO, "Venue Name information"},
  {ANQP_INFO_EMERGENCY_CALL_NUMBER_INFO, "Emergency Call Number information"},
  {ANQP_INFO_NETWORK_AUTH_TYPE_INFO,
   "Network Authentication Type information"},
  {ANQP_INFO_ROAMING_CONSORTIUM_LIST, "Roaming Consortium list"},
  {ANQP_INFO_IP_ADDR_TYPE_AVAILABILITY_INFO,
   "IP Address Type Availability information"},
  {ANQP_INFO_NAI_REALM_LIST, "NAI Realm list"},
  {ANQP_INFO_3GPP_CELLULAR_NETWORK_INFO, "3GPP Cellular Network information"},
  {ANQP_INFO_AP_GEOSPATIAL_LOCATION, "AP Geospatial Location"},
  {ANQP_INFO_AP_CIVIC_LOCATION, "AP Civic Location"},
  {ANQP_INFO_AP_LOCATION_PUBLIC_ID_URI, "AP Location Public Identifier URI"},
  {ANQP_INFO_DOMAIN_NAME_LIST, "Domain Name list"},
  {ANQP_INFO_EMERGENCY_ALERT_ID_URI, "Emergency Alert Identifier URI"},
  {ANQP_INFO_TDLS_CAPAB_INFO, "TDLS Capability information"},
  {ANQP_INFO_EMERGENCY_NAI, "Emergency NAI"},
  {ANQP_INFO_NEIGHBOR_REPORT, "Neighbor Report"},
  {ANQP_INFO_QUERY_AP_LIST, "Query AP List"},
  {ANQP_INFO_AP_LIST_RESPONSE, "AP List Response"},
  {ANQP_INFO_FILS_REALM_INFO, "FILS Realm Info"},
  {ANQP_INFO_CAG, "CAG"},
  {ANQP_INFO_VENUE_URL, "Venue URL"},
  {ANQP_INFO_ADVICE_OF_CHARGE, "Advice of Charge"},
  {ANQP_INFO_LOCAL_CONTENT, "Local Content"},
  {ANQP_INFO_NETWORK_AUTH_TYPE_TIMESTAMP,
   "Network Authentication Type with Timestamp"},
  {ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST, "ANQP vendor-specific list"},
  {0, NULL}
};
static value_string_ext anqp_info_id_vals_ext = VALUE_STRING_EXT_INIT(anqp_info_id_vals);

/* IEEE 802.11v - WNM Action field values */
enum wnm_action {
  WNM_EVENT_REQ = 0,
  WNM_EVENT_REPORT = 1,
  WNM_DIAGNOSTIC_REQ = 2,
  WNM_DIAGNOSTIC_REPORT = 3,
  WNM_LOCATION_CFG_REQ = 4,
  WNM_LOCATION_CFG_RESP = 5,
  WNM_BSS_TRANS_MGMT_QUERY = 6,
  WNM_BSS_TRANS_MGMT_REQ = 7,
  WNM_BSS_TRANS_MGMT_RESP = 8,
  WNM_FMS_REQ = 9,
  WNM_FMS_RESP = 10,
  WNM_COLLOCATED_INTERFERENCE_REQ = 11,
  WNM_COLLOCATED_INTERFERENCE_REPORT = 12,
  WNM_TFS_REQ = 13,
  WNM_TFS_RESP = 14,
  WNM_TFS_NOTIFY = 15,
  WNM_SLEEP_MODE_REQ = 16,
  WNM_SLEEP_MODE_RESP = 17,
  WNM_TIM_BROADCAST_REQ = 18,
  WNM_TIM_BROADCAST_RESP = 19,
  WNM_QOS_TRAFFIC_CAPAB_UPDATE = 20,
  WNM_CHANNEL_USAGE_REQ = 21,
  WNM_CHANNEL_USAGE_RESP = 22,
  WNM_DMS_REQ = 23,
  WNM_DMS_RESP = 24,
  WNM_TIMING_MEASUREMENT_REQ = 25,
  WNM_NOTIFICATION_REQ = 26,
  WNM_NOTIFICATION_RESP = 27
};

static const value_string wnm_action_codes[] = {
  { WNM_EVENT_REQ, "Event Request" },
  { WNM_EVENT_REPORT, "Event Report" },
  { WNM_DIAGNOSTIC_REQ, "Diagnostic Request" },
  { WNM_DIAGNOSTIC_REPORT, "Diagnostic Report" },
  { WNM_LOCATION_CFG_REQ, "Location Configuration Request" },
  { WNM_LOCATION_CFG_RESP, "Location Configuration Response" },
  { WNM_BSS_TRANS_MGMT_QUERY, "BSS Transition Management Query" },
  { WNM_BSS_TRANS_MGMT_REQ, "BSS Transition Management Request" },
  { WNM_BSS_TRANS_MGMT_RESP, "BSS Transition Management Response" },
  { WNM_FMS_REQ, "FMS Request" },
  { WNM_FMS_RESP, "FMS Response" },
  { WNM_COLLOCATED_INTERFERENCE_REQ, "Collocated Interference Request" },
  { WNM_COLLOCATED_INTERFERENCE_REPORT, "Collocated Interference Report" },
  { WNM_TFS_REQ, "TFS Request" },
  { WNM_TFS_RESP, "TFS Response" },
  { WNM_TFS_NOTIFY, "TFS Notify" },
  { WNM_SLEEP_MODE_REQ, "WNM-Sleep Mode Request" },
  { WNM_SLEEP_MODE_RESP, "WNM-Sleep Mode Response" },
  { WNM_TIM_BROADCAST_REQ, "TIM Broadcast Request" },
  { WNM_TIM_BROADCAST_RESP, "TIM Broadcast Response" },
  { WNM_QOS_TRAFFIC_CAPAB_UPDATE, "QoS Traffic Capability Update" },
  { WNM_CHANNEL_USAGE_REQ, "Channel Usage Request" },
  { WNM_CHANNEL_USAGE_RESP, "Channel Usage Response" },
  { WNM_DMS_REQ, "DMS Request" },
  { WNM_DMS_RESP, "DMS Response" },
  { WNM_TIMING_MEASUREMENT_REQ, "Timing Measurement Request" },
  { WNM_NOTIFICATION_REQ, "WNM-Notification Request" },
  { WNM_NOTIFICATION_RESP, "WNM-Notification Response" },
  { 0, NULL }
};
static value_string_ext wnm_action_codes_ext = VALUE_STRING_EXT_INIT(wnm_action_codes);

enum unprotected_wnm_action {
  UNPROTECTED_WNM_TIM = 0,
  UNPROTECTED_WNM_TIMING_MEASUREMENT = 1
};

static const value_string unprotected_wnm_action_codes[] = {
  { UNPROTECTED_WNM_TIM, "TIM" },
  { UNPROTECTED_WNM_TIMING_MEASUREMENT, "Timing Measurement" },
  { 0, NULL }
};
static value_string_ext unprotected_wnm_action_codes_ext = VALUE_STRING_EXT_INIT(unprotected_wnm_action_codes);

static const value_string wnm_notification_types[] = {
  { 0, "Firmware Update Notification" },
  { 1, "Reserved for use by WFA" },
  { 221, "Vendor Specific" },
  { 0, NULL }
};

static value_string_ext wnm_notification_types_ext =
  VALUE_STRING_EXT_INIT(wnm_notification_types);

static const range_string wnm_notification_response_code[] = {
  { 0, 0, "Notification Acknowledged" },
  { 1, 255, "Reserved" },
  { 0, 0, NULL }
};

/*** End: Action Fixed Parameter ***/

static const value_string ieee80211_tag_measure_request_type_flags[] = {
  {0x00, "Basic Request"},
  {0x01, "Clear Channel Assessment (CCA) Request"},
  {0x02, "Receive Power Indication (RPI) Histogram Request"},
  {0x03, "Channel Load Request"},
  {0x04, "Noise Histogram Request"},
  {0x05, "Beacon Request"},
  {0x06, "Frame Request"},
  {0x07, "STA Statistics Request"},
  {0x08, "Location Configuration Indication (LCI) Request"},
  {0x09, "Transmit Stream Measurement Request"},
  {0x0a, "Multicast Diagnostics Request"},
  {0x0b, "Location Civic Request"},
  {0x0c, "Location Identifier Request"},
  {0x0d, "Directional Channel Quality Request"},
  {0x0e, "Directional Measurement Request"},
  {0x0f, "Directional Statistics Request"},
  {0x10, "Fine Timing Measurement Range Request"},
  {0xFF, "Measurement Pause Request"},
  {0x00, NULL}
};
static value_string_ext ieee80211_tag_measure_request_type_flags_ext =
  VALUE_STRING_EXT_INIT(ieee80211_tag_measure_request_type_flags);

static const value_string ieee80211_tag_measure_report_type_flags[] = {
  { 0x00, "Basic Report" },
  { 0x01, "Clear Channel Assessment (CCA) Report" },
  { 0x02, "Receive Power Indication (RPI) Histogram Report" },
  { 0x03, "Channel Load Report" },
  { 0x04, "Noise Histogram Report" },
  { 0x05, "Beacon Report" },
  { 0x06, "Frame Report" },
  { 0x07, "STA Statistics Report" },
  { 0x08, "Location Configuration Information (LCI) Report" },
  { 0x09, "Transmit Stream Measurement Report" },
  { 0x0a, "Multicast Diagnostics Report"},
  { 0x0b, "Location Civic Report"},
  { 0x0c, "Location Identifier Report"},
  { 0x0d, "Directional Channel Quality Report"},
  { 0x0e, "Directional Measurement Report"},
  { 0x0f, "Directional Statistics Report"},
  { 0x10, "Fine Timing Measurement Range Report"},
  { 0x00, NULL }
};
static value_string_ext ieee80211_tag_measure_report_type_flags_ext =
  VALUE_STRING_EXT_INIT(ieee80211_tag_measure_report_type_flags);

static const true_false_string ieee80211_tag_measure_report_frame_info_frame_type_flag = {
  "Measurement Pilot Frame",
  "Beacon/Probe Response Frame"
};

static const true_false_string ig_tfs = {
  "Group address (multicast/broadcast)",
  "Individual address (unicast)"
};

static const true_false_string lg_tfs = {
  "Locally administered address (this is NOT the factory default)",
  "Globally unique address (factory default)"
};

static const true_false_string ieee80211_tag_measure_map_field_bss_flag = {
  "At least one valid MPDU was received by another BSS or IBSS during the measurement period.",
  "No valid MPDUs were received from another BSS or IBSS during the measurement period."
};

static const value_string ieee80211_tag_measure_request_measurement_mode_flags[] = {
  { 0x00, "Passive" },
  { 0x01, "Active" },
  { 0x02, "Beacon Table" },
  { 0x00, NULL }
};

#define MEASURE_REQ_BEACON_SUB_SSID 0
#define MEASURE_REQ_BEACON_SUB_BRI 1
#define MEASURE_REQ_BEACON_SUB_RD 2
#define MEASURE_REQ_BEACON_SUB_REQUEST 10
#define MEASURE_REQ_BEACON_SUB_REQUEST_EXT 11
#define MEASURE_REQ_BEACON_SUB_APCP 51
#define MEASURE_REQ_BEACON_SUB_WIDE_BW_CHANNEL_SWITCH 163
#define MEASURE_REQ_BEACON_SUB_LAST_REPORT_REQ 164
#define MEASURE_REQ_BEACON_SUB_VS 221

static const value_string ieee80211_tag_measure_request_beacon_sub_id_flags[] = {
  { MEASURE_REQ_BEACON_SUB_SSID, "SSID" },
  { MEASURE_REQ_BEACON_SUB_BRI, "Beacon Reporting Information" },
  { MEASURE_REQ_BEACON_SUB_RD, "Reporting Detail" },
  { MEASURE_REQ_BEACON_SUB_REQUEST, "Request" },
  { MEASURE_REQ_BEACON_SUB_REQUEST_EXT, "Extended Request" },
  { MEASURE_REQ_BEACON_SUB_APCP, "AP Channel Report" },
  { MEASURE_REQ_BEACON_SUB_WIDE_BW_CHANNEL_SWITCH, "Wide Channel Bandwidth Switch" },
  { MEASURE_REQ_BEACON_SUB_LAST_REPORT_REQ, "Last Beacon Report Indication Request" },
  { MEASURE_REQ_BEACON_SUB_VS, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string ieee80211_tag_measure_request_beacon_sub_bri_reporting_condition_flags[] = {
  { 0x00, "Report to be issued after each measurement." },
  { 0x01, "The measured RCPI level is greater than an absolute threshold." },
  { 0x02, "The measured RCPI level is less than an absolute threshold." },
  { 0x03, "The measured RSNI level is greater than an absolute threshold." },
  { 0x04, "The measured RSNI level is less than an absolute threshold." },
  { 0x05, "The measured RCPI level is greater than a threshold defined by an offset from the serving AP's reference RCPI." },
  { 0x06, "The measured RCPI level is less than a threshold defined by an offset from the serving AP's reference RCPI." },
  { 0x07, "The measured RSNI level is greater than a threshold defined by an offset from the serving AP's reference RSNI." },
  { 0x08, "The measured RSNI level is less than a threshold defined by an offset from the serving AP's reference RSNI." },
  { 0x09, "The measured RCPI level is in a range bound by the serving AP's reference RCPI and an offset from the serving AP's reference RCPI." },
  { 0x0a, "The measured RSNI level is in a range bound by the serving AP's reference RSNI and an offset from the serving AP's reference RSNI." },
  { 0xfe, "Report not required to be issued" },
  { 0x00, NULL }
};

static const value_string ieee80211_tag_measure_request_beacon_sub_reporting_detail_flags[] = {
  { 0, "No fixed length fields or elements" },
  { 1, "All fixed length fields and any requested elements in the Request information element if present" },
  { 2, "All fixed length fields and elements (default, used when Reporting Detail subelement is not included in Beacon Request" },
  { 0x00, NULL }
};

static const value_string ieee80211_tag_measure_request_group_id_flags[] = {
  { 0x00, "STA Counters from dot11CountersTable" },
  { 0x01, "STA Counters from dot11MacStatistics group" },
  { 0x02, "QoS STA Counters for UP0 from dot11QosCountersTable" },
  { 0x03, "QoS STA Counters for UP1 from dot11QosCountersTable" },
  { 0x04, "QoS STA Counters for UP2 from dot11QosCountersTable" },
  { 0x05, "QoS STA Counters for UP3 from dot11QosCountersTable" },
  { 0x06, "QoS STA Counters for UP4 from dot11QosCountersTable" },
  { 0x07, "QoS STA Counters for UP5 from dot11QosCountersTable" },
  { 0x08, "QoS STA Counters for UP6 from dot11QosCountersTable" },
  { 0x09, "QoS STA Counters for UP7 from dot11QosCountersTable" },
  { 0x0a, "BSS Average Access Delays" },
  { 0x0b, "STA Counters from dot11A-MSDU Group" },
  { 0x0c, "STA Counters from dot11A-MPDU Group" },
  { 0x0d, "STA Counters from dot11 BAR, Channel Width, PSMP Group" },
  { 0x0e, "STA Counters from dot11Protection Group" },
  { 0x0f, "STBC Group" },
  { 0x00, NULL }
};
static value_string_ext ieee80211_tag_measure_request_group_id_flags_ext =
  VALUE_STRING_EXT_INIT(ieee80211_tag_measure_request_group_id_flags);

static const value_string ieee80211_tag_measure_request_location_subject[] = {
  { 0, "Local" },
  { 1, "Remote" },
  { 2, "Third party" },
  { 0x00, NULL }
};

static const value_string ieee80211_tag_measure_request_civic_location_type[] = {
  { 0, "IETF RFC 4776" },
  { 1, "Vendor Specific" },
  { 0x00, NULL }
};

static const value_string ieee80211_tclas_process_flag[] = {
  {0x00, "Incoming MSDU's higher layer parameters have to match to the parameters in all associated TCLAS elements."},
  {0x01, "Incoming MSDU's higher layer parameters have to match to at least one of the associated TCLAS elements."},
  {0x02, "Incoming MSDU's that do not belong to any other TS are classified to the TS for which this TCLAS Processing element is used. In this case, there will not be any associated TCLAS elements."},
  {0, NULL}
};


#define MEASURE_REQ_CHANNEL_LOAD_SUB_REPORTING_INFO 1

static const value_string ieee80211_tag_measure_request_channel_load_sub_id_vals[] = {
  { MEASURE_REQ_CHANNEL_LOAD_SUB_REPORTING_INFO, "Channel Load Reporting Information" },
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string ieee80211_tag_measure_request_channel_load_sub_reporting_condition_vals[] = {
  { 0x00, "Report to be issued after each measurement (default, used when Channel Load Reporting Information subelement is not included in Channel Load Request)." },
  { 0x01, "Report to be issued when measured Channel Load is equal to or greater than the reference value." },
  { 0x02, "Report to be issued when measured Channel Load is equal to or less than the reference value." },
  { 0x00, NULL}
};

#define MEASURE_REQ_NOISE_HISTOGRAM_SUB_REPORTING_INFO 1

static const value_string ieee80211_tag_measure_request_noise_histogram_sub_id_vals[] = {
  { MEASURE_REQ_NOISE_HISTOGRAM_SUB_REPORTING_INFO, "Noise Histogram Reporting Information" },
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string ieee80211_tag_measure_request_noise_histogram_sub_reporting_condition_vals[] = {
  { 0x00, "Report to be issued after each measurement (default, used when Noise Histogram Reporting Information subelement is not included in Noise Histogram Request)." },
  { 0x01, "Noise Histogram Report to be issued when measured ANPI is equal to or greater than the reference value." },
  { 0x02, "Noise Histogram Report to be issued when measured ANPI is equal to or less than the reference value." },
  { 0x00, NULL}
};

#define MEASURE_REP_BEACON_SUB_REPORTED_FRAME_BODY 1
#define MEASURE_REP_BEACON_SUB_REPORTED_FRAME_BODY_FRAG_ID 2
#define MEASURE_REP_BEACON_SUB_WIDE_BW_CHANNEL_SWITCH 163
#define MEASURE_REP_BEACON_SUB_LAST_REPORT_INDICATION 164

static const value_string ieee80211_tag_measure_report_beacon_sub_id_vals[] = {
  { MEASURE_REP_BEACON_SUB_REPORTED_FRAME_BODY, "Reported Frame Body" },
  { MEASURE_REP_BEACON_SUB_REPORTED_FRAME_BODY_FRAG_ID, "Reported Frame Body Fragment ID" },
  { MEASURE_REP_BEACON_SUB_WIDE_BW_CHANNEL_SWITCH, "Wide Bandwidth Channel Switch"},
  { MEASURE_REP_BEACON_SUB_LAST_REPORT_INDICATION, "Last Beacon Report Indication"},
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

#define MEASURE_REP_LCI_SUB_REPORTED_LCI 0
#define MEASURE_REP_LCI_SUB_REPORTED_AZIMUTH_REPORT 1
#define MEASURE_REP_LCI_SUB_REPORTED_OR_STA 2
#define MEASURE_REP_LCI_SUB_REPORTED_T_MAC 3
#define MEASURE_REP_LCI_SUB_REPORTED_Z 4
#define MEASURE_REP_LCI_SUB_REPORTED_RLE 5
#define MEASURE_REP_LCI_SUB_REPORTED_URP 6
#define MEASURE_REP_LCI_SUB_REPORTED_CO_BSSID 7

static const value_string ieee80211_tag_measure_report_lci_sub_id_vals[] = {
  { MEASURE_REP_LCI_SUB_REPORTED_LCI, "LCI" },
  { MEASURE_REP_LCI_SUB_REPORTED_AZIMUTH_REPORT, "Azimuth Report" },
  { MEASURE_REP_LCI_SUB_REPORTED_OR_STA, "Originator Requesting STA MAC Address" },
  { MEASURE_REP_LCI_SUB_REPORTED_T_MAC, "Target MAC Address" },
  { MEASURE_REP_LCI_SUB_REPORTED_Z, "Z" },
  { MEASURE_REP_LCI_SUB_REPORTED_RLE, "Relative Location Error" },
  { MEASURE_REP_LCI_SUB_REPORTED_URP, "Usage Rules/Policy" },
  { MEASURE_REP_LCI_SUB_REPORTED_CO_BSSID, "Co-Located BSSID List" },
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

#define MEASURE_REP_CIVIC_SUB_REPORTED_LOCATION_CIVIC 0
#define MEASURE_REP_CIVIC_SUB_REPORTED_OR_STA 1
#define MEASURE_REP_CIVIC_SUB_REPORTED_T_MAC 2
#define MEASURE_REP_CIVIC_SUB_REPORTED_LOCATION_REFERENCE 3
#define MEASURE_REP_CIVIC_SUB_REPORTED_LOCATION_SHAPE 4
#define MEASURE_REP_CIVIC_SUB_REPORTED_MAP_IMAGE 5
#define MEASURE_REP_CIVIC_SUB_REPORTED_RESERVED 6
#define MEASURE_REP_CIVIC_SUB_REPORTED_CO_BSSID 7

static const value_string ieee80211_tag_measure_report_civic_sub_id_vals[] = {
  { MEASURE_REP_CIVIC_SUB_REPORTED_LOCATION_CIVIC, "Location Civic" },
  { MEASURE_REP_CIVIC_SUB_REPORTED_OR_STA, "Originator Requesting STA MAC Address" },
  { MEASURE_REP_CIVIC_SUB_REPORTED_T_MAC, "Target MAC Address" },
  { MEASURE_REP_CIVIC_SUB_REPORTED_LOCATION_REFERENCE, "Location Reference" },
  { MEASURE_REP_CIVIC_SUB_REPORTED_LOCATION_SHAPE, "Location Shape" },
  { MEASURE_REP_CIVIC_SUB_REPORTED_MAP_IMAGE, "Map Image" },
  { MEASURE_REP_CIVIC_SUB_REPORTED_RESERVED, "Reserved" },
  { MEASURE_REP_CIVIC_SUB_REPORTED_CO_BSSID, "Co-Located BSSID List" },
  { 221, "Vendor Specific" },
  { 0x00, NULL}
};

static const value_string frame_type[] = {
  {MGT_FRAME,       "Management frame"},
  {CONTROL_FRAME,   "Control frame"},
  {DATA_FRAME,      "Data frame"},
  {EXTENSION_FRAME, "Extension frame"},
  {0,               NULL}
};

static const value_string tofrom_ds[] = {
  {0,                     "Not leaving DS or network is operating "
    "in AD-HOC mode (To DS: 0 From DS: 0)"},
  {FLAG_TO_DS,            "Frame from STA to DS via an AP (To DS: 1 "
    "From DS: 0)"},
  {FLAG_FROM_DS,          "Frame from DS to a STA via AP(To DS: 0 "
    "From DS: 1)"},
  {FLAG_TO_DS|FLAG_FROM_DS, "WDS (AP to AP) or Mesh (MP to MP) Frame "
    "(To DS: 1 From DS: 1)"},
  {0, NULL}
};

static const true_false_string tods_flag = {
  "Frame is entering DS",
  "Frame is not entering DS"
};

static const true_false_string fromds_flag = {
  "Frame is exiting DS",
  "Frame is not exiting DS"
};

static const true_false_string more_fragments = {
  "More fragments follow",
  "This is the last fragment"
};

static const true_false_string retry_flags = {
  "Frame is being retransmitted",
  "Frame is not being retransmitted"
};

static const true_false_string s1g_dynamic_indication_flags = {
  "Dynamic",
  "Static"
};

static const true_false_string pm_flags = {
  "STA will go to sleep",
  "STA will stay up"
};

static const true_false_string md_flags = {
  "Data is buffered for STA at AP",
  "No data buffered"
};

static const true_false_string protected_flags = {
  "Data is protected",
  "Data is not protected"
};

static const true_false_string order_flags = {
  "Strictly ordered",
  "Not strictly ordered"
};

static const true_false_string cf_ess_flags = {
  "Transmitter is an AP",
  "Transmitter is a STA"
};


static const true_false_string cf_privacy_flags = {
  "Data confidentiality required",
  "Data confidentiality not required"
};

static const true_false_string cf_ibss_flags = {
  "Transmitter belongs to an IBSS",
  "Transmitter belongs to a BSS"
};

static const true_false_string eosp_flag = {
  "End of service period",
  "Service period"
};

static const true_false_string bit4_flag = {
  "Bits 8-15 of QoS Control field are Queue Size",
  "Bits 8-15 of QoS Control field are TXOP Duration Requested"
};

static const true_false_string ieee80211_qos_amsdu_present_flag = {
  "A-MSDU",
  "MSDU"
};

static const true_false_string csa_txrestrict_flags = {
  "Tx Restrict",
  "No Tx Restrict"
};

static const true_false_string csa_initiator_flags = {
  "Initiator",
  "Non Initiator"
};

static const true_false_string mesh_config_cap_power_save_level_flags = {
   "At least one of the peer-specific mesh power modes is deep sleep mode",
   "None of the peer-specific mesh power modes is deep sleep mode"
};

static const true_false_string ieee80211_qos_mesh_ps = {
  "deep sleep mode",
  "light sleep mode"
};

#define AUTH_ALG_OPEN                   0
#define AUTH_ALG_SHARED                 1
#define AUTH_ALG_FAST_BSS_TRANS         2
#define AUTH_ALG_SAE                    3
#define AUTH_ALG_FILS_SK_WITHOUT_PFS    4
#define AUTH_ALG_FILS_SK_WITH_PFS       5
#define AUTH_ALG_FILS_PK                6
#define AUTH_ALG_PASN                   7
#define AUTH_ALG_NETWORK_EAP         0x80

static const value_string auth_alg[] = {
  {AUTH_ALG_OPEN,                "Open System"},
  {AUTH_ALG_SHARED,              "Shared key"},
  {AUTH_ALG_FAST_BSS_TRANS,      "Fast BSS Transition"},
  {AUTH_ALG_SAE,                 "Simultaneous Authentication of Equals (SAE)"},
  {AUTH_ALG_FILS_SK_WITHOUT_PFS, "FILS Shared Key authentication without PFS"},
  {AUTH_ALG_FILS_SK_WITH_PFS,    "FILS Shared Key authentication with PFS"},
  {AUTH_ALG_FILS_PK,             "FILS Public Key authentication"},
  {AUTH_ALG_PASN,                "Pre-Association Security Negotiation (PASN)"},
  {AUTH_ALG_NETWORK_EAP,         "Network EAP"},  /* Cisco proprietary? */
  {0, NULL}
};

/* IANA - Internet Key Exchange (IKE) Attributes - Group Description (https://www.iana.org/assignments/ipsec-registry/ipsec-registry.xhtml) */
static const value_string ff_finite_cyclic_group_vals[] = {
  { 1, "default 768-bit MODP group"},
  { 2, "alternate 1024-bit MODP group"},
  { 3, "EC2N group on GP[2^155]"},
  { 4, "EC2N group on GP[2^185]"},
  { 5, "1536-bit MODP group"},
  {14, "2048-bit MODP group"},
  {15, "3072-bit MODP group"},
  {16, "4096-bit MODP group"},
  {17, "6144-bit MODP group"},
  {18, "8192-bit MODP group"},
  {19, "256-bit random ECP group"},
  {20, "384-bit random ECP group"},
  {21, "521-bit random ECP group"},
  {22, "1024-bit MODP Group with 160-bit Prime Order Subgroup"},
  {23, "2048-bit MODP Group with 224-bit Prime Order Subgroup"},
  {24, "2048-bit MODP Group with 256-bit Prime Order Subgroup"},
  {25, "192-bit Random ECP Group"},
  {26, "224-bit Random ECP Group"},
  {27, "224-bit Brainpool ECP group"},
  {28, "256-bit Brainpool ECP group"},
  {29, "384-bit Brainpool ECP group"},
  {30, "512-bit Brainpool ECP group"},
  {0, NULL}
};

static const true_false_string ff_block_ack_params_amsdu_permitted_flag = {
  "Permitted in QoS Data MPDUs",
  "Not Permitted"
};

static const true_false_string ff_block_ack_params_policy_flag = {
  "Immediate Block Ack",
  "Delayed Block Ack"
};

static const value_string  ff_channel_width_vals[] = {
  {0x00, "20 MHz channel width only"},
  {0x01, "Any channel width in the STA's Supported Channel Width Set"},
  {0, NULL}
};

static const true_false_string ff_qos_info_ap_q_ack_flag = {
  "APs MIB attribute dot11QAckOptionImplemented is true",
  "APs MIB attribute dot11QAckOptionImplemented is false"
};

static const true_false_string ff_qos_info_ap_queue_req_flag = {
  "AP can process a nonzero Queue Size subfield in the QoS Control field in QoS data frames",
  "AP cannot process a nonzero Queue Size subfield in the QoS Control field in QoS data frames"
};

static const true_false_string ff_qos_info_ap_txop_request_flag = {
  "AP can process a nonzero TXOP Duration Requested subfield in the QoS Control field in QoS data frames",
  "AP cannot process a nonzero TXOP Duration Requested subfield in the QoS Control field in QoS data frames"
};

static const true_false_string ff_qos_info_sta_ac_flag = {
  "Trigger-enabled and Delivery-enabled",
  "Neither Trigger-enabled nor Delivery-enabled"
};

static const true_false_string ff_qos_info_sta_q_ack_flag = {
  "STAs MIB attribute dot11QAckOptionImplemented is true",
  "STAs MIB attribute dot11QAckOptionImplemented is false"
};

static const value_string ff_qos_info_sta_max_sp_len_flags[] = {
  {0x00, "AP may deliver all buffered MSDUs, A-MSDUs and MMPDUs"},
  {0x01, "AP may deliver a maximum of two MSDUs, A-MSDUs and MMPDUs per SP"},
  {0x02, "AP may deliver a maximum of four MSDUs, A-MSDUs and MMPDUs per SP"},
  {0x03, "AP may deliver a maximum of six MSDUs, A-MSDUs and MMPDUs per SP"},
  {0, NULL}
};

static const true_false_string ff_qos_info_sta_more_data_ack_flag = {
  "STA can process ACK frames with the More Data bit in the Frame Control field set to 1 and will remain in the wake state",
  "STA cannot process ACK frames with the More Data bit in the Frame Control field set to 1"
};

static const true_false_string ff_sm_pwr_save_sm_mode_flag = {
  "Dynamic SM Power Save mode",
  "Static SM Power Save mode"
};

static const true_false_string ff_pco_phase_cntrl_flag = {
  "40 MHz Phase",
  "20 MHz Phase"
};

static const true_false_string ff_psmp_param_set_more_psmp_flag = {
  "More PSMP Sequences Follow",
  "No PSMP Sequences Follow"
};

static const value_string ff_mimo_cntrl_nc_index_flags[] = {
  {0x00, "1 Column"},
  {0x01, "2 Columns"},
  {0x02, "3 Columns"},
  {0x03, "4 Columns"},
  {0, NULL}
};

static const value_string ff_mimo_cntrl_nr_index_flags[] = {
  {0x00, "1 Row"},
  {0x01, "2 Rows"},
  {0x02, "3 Rows"},
  {0x03, "4 Rows"},
  {0, NULL}
};

static const true_false_string ff_mimo_cntrl_channel_width_flag = {
  "40 MHz",
  "20 MHz"
};

static const true_false_string ff_ht_info_information_request_flag = {
  "Requesting HT Information Exchange management action frame",
  "Should not send an HT Information Exchange management action frame"
};

static const true_false_string ff_ht_info_40_mhz_intolerant_flag = {
  "Transmitting station is intolerant of 40 MHz operation",
  "Transmitting station permits 40 MHz operation"
};

static const true_false_string ff_ht_info_sta_chan_width_flag = {
  "40 MHz",
  "20 MHz"
};

static const value_string ff_ht_action_flags[] = {
  {HT_ACTION_NOTIFY_CHAN_WIDTH,           "Notify Channel Width"},
  {HT_ACTION_SM_PWR_SAVE,                 "Spatial Multiplexing (SM) Power Save"},
  {HT_ACTION_PSMP_ACTION,                 "Power Save Multi-Poll (PSMP) action frame"},
  {HT_ACTION_SET_PCO_PHASE,               "Set PCO Phase"},
  {HT_ACTION_MIMO_CSI,                    "MIMO CSI Matrices"},
  {HT_ACTION_MIMO_BEAMFORMING,            "MIMO Non-compressed Beamforming"},
  {HT_ACTION_MIMO_COMPRESSED_BEAMFORMING, "MIMO Compressed Beamforming"},
  {HT_ACTION_ANT_SEL_FEEDBACK,            "Antenna Selection Indices Feedback"},
  {HT_ACTION_HT_INFO_EXCHANGE,            "HT Information Exchange"},
  {0x00, NULL}
};

static const value_string ff_fst_action_flags[] = {
  {FST_SETUP_REQUEST,             "FST Setup Request"},
  {FST_SETUP_RESPONSE,            "FST Setup Response"},
  {FST_TEAR_DOWN,                 "FST Tear Down"},
  {FST_ACK_REQUEST,               "FST Ack Request"},
  {FST_ACK_RESPONSE,              "FST Ack Response"},
  {FST_ON_CHANNEL_TUNNEL_REQUEST, "FST On-channel Tunnel Request"},
  {0x00, NULL}
};

static const value_string ff_robust_av_streaming_action_flags[] = {
  {ROBUST_AV_STREAMING_SCS_REQUEST,          "SCS Request"},
  {ROBUST_AV_STREAMING_SCS_RESPONSE,         "SCS Response"},
  {ROBUST_AV_STREAMING_GROUP_MEMBERSHIP_REQ, "Group Membership Request"},
  {ROBUST_AV_STREAMING_GROUP_MEMBERSHIP_RSP, "Group Membership Response"},
  {ROBUST_AV_STREAMING_MSCS_REQUEST,         "MSCS Request"},
  {ROBUST_AV_STREAMING_MSCS_RESPONSE,        "MSCS Response"},
  {0x00, NULL}
};

static const value_string ff_dmg_action_flags[] = {
  {DMG_ACTION_PWR_SAVE_CONFIG_REQ,           "Power Save Configuration Request"},
  {DMG_ACTION_PWR_SAVE_CONFIG_RES,           "Power Save Configuration Response"},
  {DMG_ACTION_INFO_REQ,                      "Information Request"},
  {DMG_ACTION_INFO_RES,                      "Information Response"},
  {DMG_ACTION_HANDOVER_REQ,                  "Handover Request"},
  {DMG_ACTION_HANDOVER_RES,                  "Handover Response"},
  {DMG_ACTION_DTP_REQ,                       "DTP Request"},
  {DMG_ACTION_DTP_RES,                       "DTP Response"},
  {DMG_ACTION_RELAY_SEARCH_REQ,              "Relay Search Request"},
  {DMG_ACTION_RELAY_SEARCH_RES,              "Relay Search Response"},
  {DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_REQ, "Multi Relay Channel Measurement Request"},
  {DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_RES, "Multi Relay Channel Measurement Response"},
  {DMG_ACTION_RLS_REQ,                       "RLS Request"},
  {DMG_ACTION_RLS_RES,                       "RLS Response"},
  {DMG_ACTION_RLS_ANNOUNCE,                  "RLS Announcement"},
  {DMG_ACTION_RLS_TEARDOWN,                  "RLS Teardown"},
  {DMG_ACTION_RELAY_ACK_REQ,                 "Relay ACK Request"},
  {DMG_ACTION_RELAY_ACK_RES,                 "Relay ACK Response"},
  {DMG_ACTION_TPA_REQ,                       "TPA Request"},
  {DMG_ACTION_TPA_RES,                       "TPA Response"},
  {DMG_ACTION_TPA_REP,                       "TPA Report"},
  {DMG_ACTION_ROC_REQ,                       "ROC Request"},
  {DMG_ACTION_ROC_RES,                       "ROC Response"},
  {0x00, NULL}
};

static const value_string ff_unprotected_dmg_action_flags[] = {
  {UNPROTECTED_DMG_ANNOUNCE,      "Announce"},
  {UNPROTECTED_DMG_BRP,           "BRP"},
  {0x00, NULL}
};
static const value_string ff_mimo_cntrl_grouping_flags[] = {
  {0x00, "No Grouping"},
  {0x01, "Carrier Groups of 2"},
  {0x02, "Carrier Groups of 4"},
  {0x03, "Reserved"},
  {0, NULL}
};

static const value_string ff_mimo_cntrl_coefficient_size_flags[] = {
  {0x00, "4 Bits"},
  {0x01, "5 Bits"},
  {0x02, "6 Bits"},
  {0x03, "8 Bits"},
  {0, NULL}
};

static const value_string ff_mimo_cntrl_codebook_info_flags[] = {
  {0x00, "1 bit for 'Capital Psi', 3 bits for 'Small Psi'"},
  {0x01, "2 bit for 'Capital Psi', 4 bits for 'Small Psi'"},
  {0x02, "3 bit for 'Capital Psi', 5 bits for 'Small Psi'"},
  {0x03, "4 bit for 'Capital Psi', 6 bits for 'Small Psi'"},
  {0, NULL}
};

static const value_string ff_ppa_action_codes[] = {
  {PPA_DSE_ENABLEMENT,                  "Protected DSE enablement"},
  {PPA_DSE_DEENABLEMENT,                "Protected DSE deenablement"},
  {PPA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT, "Protected Extended Channel Switch Announcement"},
  {PPA_DSE_MEASUREMENT_REQUEST,         "Protected DSE measurement request"},
  {PPA_DSE_MEASUREMENT_REPORT,          "Protected DSE measurement report"},
  {PPA_DSE_POWER_CONSTRAINT,            "Protected DSE power constraint"},
  {PPA_VENDOR_SPECIFIC,                 "Protected Vendor Specific"},
  {PPA_GAS_INITIAL_REQUEST,             "Protected GAS Initial Request"},
  {PPA_GAS_INITIAL_RESPONSE,            "Protected GAS Initial Response"},
  {PPA_GAS_COMEBACK_REQUEST,            "Protected GAS Comeback Request"},
  {PPA_GAS_COMEBACK_RESPONSE,           "Protected GAS Comeback Response"},
  {PPA_QAB_REQUEST,                     "Protected QAB Request"},
  {PPA_QAB_RESPONSE,                    "Protected QAB Response"},
  {0x00, NULL}
};
static value_string_ext ff_ppa_action_codes_ext = VALUE_STRING_EXT_INIT(ff_ppa_action_codes);

static const value_string ff_pa_action_codes[] = {
  {PA_20_40_BSS_COEXISTENCE_MANAGEMENT, "20/40 BSS Coexistence Management"},
  {PA_DSE_ENABLEMENT,                  "DSE enablement"},
  {PA_DSE_DEENABLEMENT,                "DSE deenablement"},
  {PA_DSE_REG_LOC_ANNOUNCEMENT,        "DSE Registered Location Announcement"},
  {PA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT, "Extended Channel Switch Announcement"},
  {PA_DSE_MEASUREMENT_REQUEST,         "DSE measurement request"},
  {PA_DSE_MEASUREMENT_REPORT,          "DSE measurement report"},
  {PA_MEASUREMENT_PILOT,               "Measurement Pilot"},
  {PA_DSE_POWER_CONSTRAINT,            "DSE power constraint"},
  {PA_VENDOR_SPECIFIC,                 "Vendor Specific"},
  {PA_GAS_INITIAL_REQUEST,             "GAS Initial Request"},
  {PA_GAS_INITIAL_RESPONSE,            "GAS Initial Response"},
  {PA_GAS_COMEBACK_REQUEST,            "GAS Comeback Request"},
  {PA_GAS_COMEBACK_RESPONSE,           "GAS Comeback Response"},
  {PA_TDLS_DISCOVERY_RESPONSE,         "TDLS Discovery Response"},
  {PA_LOCATION_TRACK_NOTIFICATION,     "Location Track Notification"},
  {PA_QAB_REQUEST,                     "QAB Request"},
  {PA_QAB_RESPONSE,                    "QAB Response"},
  {PA_QMF_POLICY,                      "QMF Policy"},
  {PA_QMF_POLICY_CHANGE,               "QMF Policy Change"},
  {PA_QLOAD_REQUEST,                   "QLoad Request"},
  {PA_QLOAD_REPORT,                    "QLoad Report"},
  {PA_HCCA_TXOP_ADVERTISEMENT,         "HCCA TXOP Advertisement"},
  {PA_HCCA_TXOP_RESPONSE,              "HCCA TXOP Response"},
  {PA_PUBLIC_KEY,                      "Public Key"},
  {PA_CHANNEL_AVAILABILITY_QUERY,      "Channel Availability Query"},
  {PA_CHANNEL_SCHEDULE_MANAGEMENT,     "Channel Schedule Management"},
  {PA_CONTACT_VERIFICATION_SIGNAL,     "Contact Verification Signal"},
  {PA_GDD_ENABLEMENT_REQUEST,          "GDD Enablement Request"},
  {PA_GDD_ENABLEMENT_RESPONSE,         "GDD Enablement Response"},
  {PA_NETWORK_CHANNEL_CONTROL,         "Network Channel Control"},
  {PA_WHITE_SPACE_MAP_ANNOUNCEMENT,    "White Space Map Announcement"},
  {PA_FTM_REQUEST,                     "FTM Request"},
  {PA_FTM,                             "FTM"},
  {PA_FILS_DISCOVERY,                  "FILS Discovery"},
  {PA_LOCATION_MEASUREMENT_REPORT,                                      "Location Measurement Report"},
  {PA_ISTA_PASSIVE_TB_RANGING_MEASUREMENT_REPORT,                       "ISTA Passive TB Ranging Measurement Report"},
  {PA_PRIMARY_RSTA_BROADCAST_PASSIVE_TB_RANGING_MEASUREMENT_REPORT,     "Primary RSTA Broadcast Passive TB Ranging Measurement Report"},
  {PA_SECONDARY_RSTA_BROADCAST_PASSIVE_TB_RANGING_MEASUREMENT_REPORT,   "Secondary RSTA Broadcast Passive TB Ranging Measurement Report"},
  {0x00, NULL}
};
value_string_ext ff_pa_action_codes_ext = VALUE_STRING_EXT_INIT(ff_pa_action_codes);

static const value_string protected_ftm_action_vals[] = {
  {0, "Reserved"},
  {1, "Protected FTM Request"},
  {2, "Protected FTM"},
  {3, "Protected Location Measurement Report"},
  {0, NULL}
};

static const value_string ftm_trigger_vals[] = {
  {0, "Stop sending FTM frames"},
  {1, "Start or continue sending FTM frames"},
  /* all other values reserved */
  {0, NULL}
};

static const value_string category_codes[] = {
  {CAT_SPECTRUM_MGMT,                    "Spectrum Management (SM)"},
  {CAT_QOS,                              "Quality of Service (QoS)"},
  {CAT_DLS,                              "Direct-Link Setup (DLS)"},
  {CAT_BLOCK_ACK,                        "Block Ack"},
  {CAT_PUBLIC,                           "Public Action"},
  {CAT_RADIO_MEASUREMENT,                "Radio Measurement"},
  {CAT_FAST_BSS_TRANSITION,              "Fast BSS Transition"},
  {CAT_HT,                               "High Throughput"},
  {CAT_SA_QUERY,                         "SA Query"},
  {CAT_PUBLIC_PROTECTED,                 "Protected Dual of Public Action"},
  {CAT_WNM,                              "WNM"},
  {CAT_UNPROTECTED_WNM,                  "Unprotected WNM"},
  {CAT_TDLS,                             "TDLS"},
  {CAT_MESH,                             "MESH"},
  {CAT_MULTIHOP,                         "Multihop"},
  {CAT_SELF_PROTECTED,                   "Self-protected"},
  {CAT_DMG,                              "DMG"},
  {CAT_MGMT_NOTIFICATION,                "Management Notification"},
  {CAT_FAST_SESSION_TRANSFER,            "Fast Session Transfer"},
  {CAT_ROBUST_AV_STREAMING,              "Robust AV Streaming"},
  {CAT_UNPROTECTED_DMG,                  "Unprotected DMG"},
  {CAT_VHT,                              "VHT"},
  {CAT_S1G,                              "S1G"},
  {CAT_PROTECTED_S1G,                    "Protected S1G"},
  {CAT_FLOW_CONTROL,                     "Flow Control"},
  {CAT_CONTROL_RESPONSE_MCS_NEG,         "Control Response MCS Negotiation"},
  {CAT_FILS,                             "FILS"},
  {CAT_HE,                               "HE"},
  {CAT_PROTECTED_HE,                     "Protected HE"},
  {CAT_PROTECTED_FTM,                    "Protected FTM"},
  {CAT_EHT,                              "EHT"},
  {CAT_PROTECTED_EHT,                    "Protected EHT"},
  {CAT_VENDOR_SPECIFIC_PROTECTED,        "Vendor-specific Protected"},
  {CAT_VENDOR_SPECIFIC,                  "Vendor Specific"},

  {0x80 | CAT_SPECTRUM_MGMT,             "Spectrum Management (SM) (error)"},
  {0x80 | CAT_QOS,                       "Quality of Service (QoS (error))"},
  {0x80 | CAT_DLS,                       "Direct-Link Setup (DLS) (error)"},
  {0x80 | CAT_BLOCK_ACK,                 "Block Ack (error)"},
  {0x80 | CAT_PUBLIC,                    "Public Action (error)"},
  {0x80 | CAT_RADIO_MEASUREMENT,         "Radio Measurement (error)"},
  {0x80 | CAT_FAST_BSS_TRANSITION,       "Fast BSS Transition (error)"},
  {0x80 | CAT_HT,                        "High Throughput (error)"},
  {0x80 | CAT_SA_QUERY,                  "SA Query (error)"},
  {0x80 | CAT_PUBLIC_PROTECTED,          "Protected Dual of Public Action (error)"},
  {0x80 | CAT_WNM,                       "WNM (error)"},
  {0x80 | CAT_UNPROTECTED_WNM,           "Unprotected WNM (error)"},
  {0x80 | CAT_TDLS,                      "TDLS (error)"},
  {0x80 | CAT_MESH,                      "Mesh (error)"},
  {0x80 | CAT_MULTIHOP,                  "Multihop (error)"},
  {0x80 | CAT_SELF_PROTECTED,            "Self-protected (error)"},
  {0x80 | CAT_DMG,                       "DMG (error)"},
  {0x80 | CAT_MGMT_NOTIFICATION,         "Management Notification (error)"},
  {0x80 | CAT_FAST_SESSION_TRANSFER,     "Fast Session Transfer (error)"},
  {0x80 | CAT_ROBUST_AV_STREAMING,       "Robust AV Streaming (error)"},
  {0x80 | CAT_UNPROTECTED_DMG,           "Unprotected DMG (error)"},
  {0x80 | CAT_VHT,                       "VHT (error)"},
  {0x80 | CAT_HE,                        "HE (error)"},
  {0x80 | CAT_PROTECTED_HE,              "Protected HE (error)"},
  {0x80 | CAT_PROTECTED_FTM,             "Protected FTM (error)"},
  {0x80 | CAT_EHT,                       "EHT (error)"},
  {0x80 | CAT_PROTECTED_EHT,             "Protected EHT (error)"},
  {0x80 | CAT_VENDOR_SPECIFIC_PROTECTED, "Vendor-specific Protected (error)"},
  {0x80 | CAT_VENDOR_SPECIFIC,           "Vendor Specific (error)"},
  {0, NULL}
};
static value_string_ext category_codes_ext = VALUE_STRING_EXT_INIT(category_codes);

#define NR_SUB_ID_TSF_INFO                 1
#define NR_SUB_ID_CON_COU_STR              2
#define NR_SUB_ID_BSS_TRN_CAN_PREF         3
#define NR_SUB_ID_BSS_TER_DUR              4
#define NR_SUB_ID_BEARING                  5
#define NR_SUB_ID_WIDE_BW_CHANNEL          6

#define NR_SUB_ID_MEASUREMENT_REPORT       39
#define NR_SUB_ID_HT_CAPABILITIES          45
#define NR_SUB_ID_HT_OPERATION             61
#define NR_SUB_ID_SEC_CHANNEL_OFFSET       62
#define NR_SUB_ID_MEASUREMENT_PILOT_INFO   66
#define NR_SUB_ID_RM_ENABLE_CAP            70
#define NR_SUB_ID_HT_MULTIPLE_BSSID        71
#define NR_SUB_ID_VHT_CAPABILITIES         191
#define NR_SUB_ID_VHT_OPERATION            192
#define NR_SUB_ID_HE_CAPABILITIES          193
#define NR_SUB_ID_HE_OPERATION             194
#define NR_SUB_ID_EHT_CAPABILITIES         199
#define NR_SUB_ID_EHT_OPERATION            200
#define NR_SUB_ID_BASIC_MULTI_LINK         201

#define NR_SUB_ID_VENDOR_SPECIFIC          221

static const value_string ieee80211_neighbor_report_subelement_id_vals[] = {
  {NR_SUB_ID_TSF_INFO, "TSF Information"},
  {NR_SUB_ID_CON_COU_STR, "Condensed Country String"},
  {NR_SUB_ID_BSS_TRN_CAN_PREF, "BSS Transition Candidate Preference"},
  {NR_SUB_ID_BSS_TER_DUR, "BSS Termination Duration"},
  {NR_SUB_ID_BEARING, "Bearing"},
  {NR_SUB_ID_WIDE_BW_CHANNEL, "Wide Bandwidth Channel"},
  {NR_SUB_ID_MEASUREMENT_REPORT, "Measurement Report"},
  {NR_SUB_ID_HT_CAPABILITIES, "HT Capabilities"},
  {NR_SUB_ID_HT_OPERATION, "HT Operation"},
  {NR_SUB_ID_SEC_CHANNEL_OFFSET, "Secondary Channel Offset"},
  {NR_SUB_ID_MEASUREMENT_PILOT_INFO, "Measurement Pilot Transmission"},
  {NR_SUB_ID_RM_ENABLE_CAP, "RM Enabled Capabilities"},
  {NR_SUB_ID_HT_MULTIPLE_BSSID, "Multiple BSSID"},
  {NR_SUB_ID_VHT_CAPABILITIES, "VHT Capabilities"},
  {NR_SUB_ID_VHT_OPERATION, "VHT Operation"},
  {NR_SUB_ID_HE_CAPABILITIES, "HE Capabilities"},
  {NR_SUB_ID_HE_OPERATION, "HE Operation"},
  {NR_SUB_ID_EHT_CAPABILITIES, "EHT Capabilities"},
  {NR_SUB_ID_EHT_OPERATION, "EHT Operation"},
  {NR_SUB_ID_BASIC_MULTI_LINK, "Basic Multi-Link"},
  {NR_SUB_ID_VENDOR_SPECIFIC, "Vendor Specific"},
  {0, NULL}
};

static const value_string ieee80211_neighbor_report_bssid_info_reachability_vals[] = {
  {0, "Reserved"},
  {1, "Not Reachable"},
  {2, "Unknown"},
  {3, "Reachable"},
  {0, NULL}
};

static const value_string action_codes[] = {
  {SM_ACTION_MEASUREMENT_REQUEST, "Measurement Request"},
  {SM_ACTION_MEASUREMENT_REPORT,  "Measurement Report"},
  {SM_ACTION_TPC_REQUEST,         "TPC Request"},
  {SM_ACTION_TPC_REPORT,          "TPC Report"},
  {SM_ACTION_CHAN_SWITCH_ANNC,    "Channel Switch Announcement"},
  {0, NULL}
};

static const value_string vendor_action_types_mrvl[] = {
  {MRVL_ACTION_MESH_MANAGEMENT, "Mesh Management"},
  {0, NULL}
};

static const value_string mesh_mgt_action_codes_mrvl[] = {
  {MRVL_MESH_MGMT_ACTION_RREQ, "Route Request"},
  {MRVL_MESH_MGMT_ACTION_RREP, "Route Response"},
  {MRVL_MESH_MGMT_ACTION_RERR, "Route Error"},
  {MRVL_MESH_MGMT_ACTION_PLDM, "Peer Link Down"},
  {0, NULL}
};

static const value_string mesh_path_selection_codes[] = {
  {0x0, "Hybrid Wireless Mesh Protocol"},
  {0, NULL}
};

static const value_string mesh_metric_codes[] = {
  {0x0, "Airtime Link Metric"},
  {0, NULL}
};

static const value_string wme_action_codes[] = {
  {0x00, "Setup request"},
  {0x01, "Setup response"},
  {0x02, "Teardown"},
  {0x00, NULL}
};

static const value_string wme_status_codes[] = {
  {0x00, "Admission accepted"},
  {0x01, "Invalid parameters"},
  {0x03, "Refused"},
  {0x00, NULL}
};

static const value_string mesh_action[] = {
  {MESH_ACTION_LINK_METRIC_REPORT, "Mesh Link Metric Report"},
  {MESH_ACTION_HWMP,               "HWMP Mesh Path Selection"},
  {MESH_ACTION_GATE_ANNOUNCE,      "Gate Announcement"},
  {MESH_ACTION_CONGESTION_CTL,     "Congestion Control Notification"},
  {MESH_ACTION_MCCA_SETUP_REQUEST, "MCCA Setup Request"},
  {MESH_ACTION_MCCA_SETUP_REPLY,   "MCCA Setup Reply"},
  {MESH_ACTION_MCCA_ADV_REQUEST,   "MCCA Advertisement Request"},
  {MESH_ACTION_MCCA_ADV,           "MCCA Advertisement"},
  {MESH_ACTION_MCCA_TEARDOWN,      "MCCA Teardown"},
  {MESH_ACTION_TBTT_ADJ_REQUEST,   "TBTT Adjustment Request"},
  {MESH_ACTION_TBTT_ADJ_RESPONSE,  "TBTT Adjustment Response"},
  {0, NULL}
};
static value_string_ext mesh_action_ext = VALUE_STRING_EXT_INIT(mesh_action);

static const value_string multihop_action[] = {
  {MULTIHOP_ACTION_PROXY_UPDATE,      "Proxy Update"},
  {MULTIHOP_ACTION_PROXY_UPDATE_CONF, "Proxy Update Confirmation"},
  {0, NULL}
};

static const value_string selfprot_action[] = {
  {SELFPROT_ACTION_MESH_PEERING_OPEN,     "Mesh Peering Open"},
  {SELFPROT_ACTION_MESH_PEERING_CONFIRM,  "Mesh Peering Confirm"},
  {SELFPROT_ACTION_MESH_PEERING_CLOSE,    "Mesh Peering Close"},
  {SELFPROT_ACTION_MESH_GROUP_KEY_INFORM, "Mesh Group Key Inform"},
  {SELFPROT_ACTION_MESH_GROUP_KEY_ACK,    "Mesh Group Key Ack"},
  {0, NULL}
};

static const value_string mesh_peering_proto_ids[] = {
  {MESH_PEERING_PROTO_MGMT,   "Mesh peering management protocol"},
  {MESH_PEERING_PROTO_AMPE,   "Authenticated mesh peering exchange protocol"},
  {MESH_PEERING_PROTO_VENDOR, "Vendor specific"},
  {0, NULL}
};

static const true_false_string hwmp_targ_usn_flags = {
  "[USN = 1] Target Sequence Number Unknown at Originator",
  "[USN = 0] Target Sequence Number Known at Originator"
};

static const true_false_string hwmp_targ_to_flags = {
  "[TO = 1] Only Target Will Respond",
  "[TO = 0] Intermediate Nodes May Respond"
};

static const value_string ack_policy[] = {
  {0x00, "Normal Ack"},
  {0x01, "No Ack"},
  {0x02, "No explicit acknowledgment"},
  {0x03, "Block Ack"},
  {0x00, NULL}
};

static const value_string qos_action_codes[] = {
  {QOS_ACTION_ADDTS_REQUEST,  "ADDTS Request"},
  {QOS_ACTION_ADDTS_RESPONSE, "ADDTS Response"},
  {QOS_ACTION_DELTS,          "DELTS"},
  {QOS_ACTION_SCHEDULE,   "Schedule"},
  {QOS_ACTION_MAP_CONFIGURE, "QoS Map Configure"},
  {0, NULL}
};

static const value_string ba_action_codes[] = {
  {BA_ADD_BLOCK_ACK_REQUEST,      "Add Block Ack Request"},
  {BA_ADD_BLOCK_ACK_RESPONSE,     "Add Block Ack Response"},
  {BA_DELETE_BLOCK_ACK,           "Delete Block Ack"},
  {BA_NDP_ADD_BLOCK_ACK_REQUEST,  "NDP ADDBA Request"},
  {BA_NDP_ADD_BLOCK_ACK_RESPONSE, "NDP ADDBA Response"},
  {BA_NDP_DELETE_BLOCK_ACK,       "NDP DELBA"},
  {BA_BAT_ADD_BLOCK_ACK_REQUEST,  "BAT ADDBA Request"},
  {BA_BAT_ADD_BLOCK_ACK_RESPONSE, "BAT ADDBA Response"},
  {BA_BAT_DELETE_BLOCK_ACK,       "BAT DELBA"},
  {0x00, NULL}
};

static const value_string dls_action_codes[] = {
  {DLS_ACTION_REQUEST,  "DLS Request"},
  {DLS_ACTION_RESPONSE, "DLS Response"},
  {DLS_ACTION_TEARDOWN, "DLS Teardown"},
  {0, NULL}
};

static const value_string tsinfo_type[] = {
  {0x0, "Aperiodic or unspecified Traffic"},
  {0x1, "Periodic Traffic"},
  {0, NULL}
};

static const value_string tsinfo_direction[] = {
  {0x00, "Uplink"},
  {0x01, "Downlink"},
  {0x02, "Direct link"},
  {0x03, "Bidirectional link"},
  {0, NULL}
};

static const value_string tsinfo_access[] = {
  {0x00, "Reserved"},
  {0x01, "EDCA"},
  {0x02, "HCCA"},
  {0x03, "HEMM"},
  {0, NULL}
};

static const value_string qos_up[] = {
  {0x00, "Best Effort"},
  {0x01, "Background"},
  {0x02, "Spare"},
  {0x03, "Excellent Effort"},
  {0x04, "Controlled Load"},
  {0x05, "Video"},
  {0x06, "Voice"},
  {0x07, "Network Control"},
  {0, NULL}
};

static const range_string classifier_type[] = {
  {0x00, 0x00, "Ethernet parameters"},
  {0x01, 0x01, "TCP/UDP IP parameters"},
  {0x02, 0x02, "IEEE 802.1D/Q parameters"},
  {0x03, 0x03, "Filter Offset parameters"},
  {0x04, 0x04, "IP and higher layer parameters"},
  {0x05, 0x05, "IEEE 802.1D/Q parameters"},
  {0x06, 0x06, "IEEE 802.11 MAC header parameters"},
  {0x07, 0x07, "IEEE Std 802.11 downlink PV1 MPDU MAC header parameters"},
  {0x08, 0x08, "IEEE Std 802.11 non-downlink PV1 MPDU MAC header parameters"},
  {0x09, 0x09, "IEEE Std 802.11 PV1 MPDU Full Address MAC header parameters"},
  {0x0A, 0x0A, "IP extensions and higher layer parameters"},
  {0x0B, 0xFF, "Reserved" },
  {0, 0, NULL}
};

static const true_false_string ieee80211_block_ack_control_ack_policy_flag = {
    "Sender Does Not Require Immediate Acknowledgement",
    "Immediate Acknowledgement Required"
};

static const value_string ft_action_codes[] = {
  {FT_ACTION_REQUEST, "FT Request"},
  {FT_ACTION_RESPONSE, "FT Response"},
  {FT_ACTION_CONFIRM, "FT Confirm"},
  {FT_ACTION_ACK, "FT Ack"},
  {0, NULL}
};

static const value_string sa_query_action_codes[] = {
  {SA_QUERY_REQUEST, "SA Query Request"},
  {SA_QUERY_RESPONSE, "SA Query Response"},
  {0, NULL}
};

static const value_string ieee80211_data_encap_payload_types[] = {
  {1, "Remote Request/Response"},
  {2, "TDLS"},
  {0, NULL}
};

static const true_false_string rsn_preauth_flags = {
  "Transmitter supports pre-authentication",
  "Transmitter does not support pre-authentication"
};

static const true_false_string rsn_no_pairwise_flags = {
  "Transmitter cannot support WEP default key 0 simultaneously with "
  "Pairwise key",
  "Transmitter can support WEP default key 0 simultaneously with "
  "Pairwise key"
};

static const value_string rsn_cap_replay_counter[] = {
  {0x00, "1 replay counter per PTKSA/GTKSA/STAKeySA"},
  {0x01, "2 replay counters per PTKSA/GTKSA/STAKeySA"},
  {0x02, "4 replay counters per PTKSA/GTKSA/STAKeySA"},
  {0x03, "16 replay counters per PTKSA/GTKSA/STAKeySA"},
  {0, NULL}
};

static const true_false_string ht_ldpc_coding_flag = {
  "Transmitter supports receiving LDPC coded packets",
  "Transmitter does not support receiving LDPC coded packets"
};

static const true_false_string ht_chan_width_flag = {
  "Transmitter supports 20MHz and 40MHz operation",
  "Transmitter only supports 20MHz operation"
};

static const value_string ht_sm_pwsave_flag[] = {
  {0x00, "Static SM Power Save mode"},
  {0x01, "Dynamic SM Power Save mode"},
  {0x02, "Reserved"},
  {0x03, "SM Power Save disabled"},
  {0x00, NULL}
};

static const true_false_string ht_green_flag = {
  "Transmitter is able to receive PPDUs with Green Field (GF) preamble",
  "Transmitter is not able to receive PPDUs with Green Field (GF) preamble"
};

static const value_string ht_rx_stbc_flag[] = {
  {0x00, "No Rx STBC support"},
  {0x01, "Rx support of one spatial stream"},
  {0x02, "Rx support of one and two spatial streams"},
  {0x03, "Rx support of one, two, and three spatial streams"},
  {0x00, NULL}
};

  /* IEEE Stc 802.11ac-2013 8.4.2.29 Extended Capabilities element*/
static const value_string vht_max_mpdu_in_amsdu[] = {
    {0x00, "No limit"},
    {0x01, "32"},
    {0x02, "16"},
    {0x03, "8"},
    {0, NULL}
};

  /* IEEE Stc 802.11ac/D3.1 */

static const value_string vht_max_mpdu_length_flag[] = {
  {0x00, "3 895"},
  {0x01, "7 991"},
  {0x02, "11 454"},
  {0x03, "Reserved"},
  {0x00, NULL}
};

static const value_string vht_supported_chan_width_set_flag[] = {
  {0x00, "Neither 160MHz nor 80+80 supported"},
  {0x01, "160MHz supported"},
  {0x02, "160MHz and 80+80 Supported"},
  {0x03, "Reserved"},
  {0x00, NULL}
};

static const value_string vht_rx_stbc_flag[] = {
  {0x00, "None"},
  {0x01, "1 Spatial Stream Supported"},
  {0x02, "1 to 2 Spatial Stream Supported"},
  {0x03, "1 to 3 Spatial Stream Supported"},
  {0x04, "1 to 4 Spatial Stream Supported"},
  {0x05, "Reserved"},
  {0x06, "Reserved"},
  {0x07, "Reserved"},
  {0x00, NULL}
};

static const value_string num_plus_one_3bit_flag[] = {
  {0x00, "1"},
  {0x01, "2"},
  {0x02, "3"},
  {0x03, "4"},
  {0x04, "5"},
  {0x05, "6"},
  {0x06, "7"},
  {0x07, "8"},
  {0x00, NULL}
};

static const value_string vht_max_ampdu_flag[] = {
  {0x00, "8 191"},
  {0x01, "16 383"},
  {0x02, "32 767"},
  {0x03, "65 535"},
  {0x04, "131 071"},
  {0x05, "262 143"},
  {0x06, "524 287"},
  {0x07, "1 048 575"},
  {0x00, NULL}
};

static const value_string vht_link_adapt_flag[] = {
  {0x00, "No Feedback"},
  {0x01, "Reserved (logically only solicited feedback)"},
  {0x02, "Unsolicited feedback only"},
  {0x03, "Both (can provide unsolicited feedback and respond to VHT MRQ)"},
  {0x00, NULL}
};

static const value_string vht_supported_mcs_flag[] = {
  {0x00, "MCS 0-7"},
  {0x01, "MCS 0-8"},
  {0x02, "MCS 0-9"},
  {0x03, "Not Supported"},
  {0x00, NULL}
};

static const value_string vht_tpe_pwr_units[] = {
  { 0, "Local EIRP" },
  { 1, "Local EIRP PSD" },
  { 2, "Regulatory client EIRP" },
  { 3, "Regulatory client EIRP PSD" },
  { 4, "Additional regulatory client EIRP" },
  { 5, "Additional regulatory client EIRP PSD" },
  { 6, "Reserved" },
  { 7, "Reserved" },
  {0x00, NULL}
};

static const true_false_string vht_ndp_annc_sta_info_feedback_type = {
  "MU feedback requested",
  "SU feedback requested"
};

static const true_false_string ht_max_amsdu_flag = {
  "7935 bytes",
  "3839 bytes"
};

static const true_false_string ht_dss_cck_40_flag = {
  "Will/Can use DSSS/CCK in 40 MHz",
  "Won't/Can't use of DSSS/CCK in 40 MHz"
};

static const true_false_string ht_40_mhz_intolerant_flag = {
  "Use of 40 MHz transmissions restricted/disallowed",
  "Use of 40 MHz transmissions unrestricted/allowed"
};

static const value_string ampduparam_mpdu_start_spacing_flags[] = {
  {0x00, "No restriction"},
  {0x01, "1/4 [usec]"},
  {0x02, "1/2 [usec]"},
  {0x03, "1 [usec]"},
  {0x04, "2 [usec]"},
  {0x05, "4 [usec]"},
  {0x06, "8 [usec]"},
  {0x07, "16 [usec]"},
  {0x00, NULL}
};

static const true_false_string mcsset_tx_rx_mcs_set_not_equal_flag = {
  "Not Equal",
  "Equal",
};

static const value_string mcsset_tx_max_spatial_streams_flags[] = {
  {0x00, "1 spatial stream"},
  {0x01, "2 spatial streams"},
  {0x02, "3 spatial streams"},
  {0x03, "4 spatial streams"},
  {0x04, "TX MCS Set Not Defined"},
  {0x00, NULL}
};

static const value_string htex_mcs_flags[] = {
  {0x00, "STA does not provide MCS feedback"},
  {0x01, "Reserved"},
  {0x02, "STA provides only unsolicited MCS feedback"},
  {0x03, "STA can provide MCS feedback in response to MRQ as well as unsolicited MCS feedback"},
  {0x00, NULL}
};

static const value_string txbf_calib_flag[] = {
  {0x00, "Not supported"},
  {0x01, "Can respond to a calibration request but cannot initiate calibration"},
  {0x02, "Reserved"},
  {0x03, "Can both initiate and respond to a calibration request"},
  {0x00, NULL}
};

static const value_string txbf_feedback_flags[] = {
  {0x00, "Not supported"},
  {0x01, "Delayed feedback capable"},
  {0x02, "Immediate feedback capable"},
  {0x03, "Delayed and immediate feedback capable"},
  {0x00, NULL}
};

static const value_string txbf_antenna_flags[] = {
  {0x00, "1 TX antenna sounding"},
  {0x01, "2 TX antenna sounding"},
  {0x02, "3 TX antenna sounding"},
  {0x03, "4 TX antenna sounding"},
  {0x00, NULL}
};

static const value_string txbf_csi_max_rows_bf_flags[] = {
  {0x00, "1 row of CSI"},
  {0x01, "2 rows of CSI"},
  {0x02, "3 rows of CSI"},
  {0x03, "4 rows of CSI"},
  {0x00, NULL}
};

static const value_string txbf_chan_est_flags[] = {
  {0x00, "1 space time stream"},
  {0x01, "2 space time streams"},
  {0x02, "3 space time streams"},
  {0x03, "4 space time streams"},
  {0x00, NULL}
};

static const value_string txbf_min_group_flags[] = {
  {0x00, "No grouping supported"},
  {0x01, "Groups of 1, 2 supported"},
  {0x02, "Groups of 1, 4 supported"},
  {0x03, "Groups of 1, 2, 4 supported"},
  {0x00, NULL}
};

static const value_string hta_ext_chan_offset_flag[] = {
  {0x00, "No Extension Channel"},
  {0x01, "Extension Channel above control channel"},
  {0x02, "Undefined"},
  {0x03, "Extension Channel below control channel"},
  {0x00, NULL}
};

static const true_false_string hta_rec_tx_width_flag = {
  "Any channel width enabled",
  "Use 20MHz channel (control)"
};

static const true_false_string hta_rifs_mode_flag = {
  "Use of RIFS permitted",
  "Use of RIFS prohibited"
};

static const true_false_string hta_controlled_access_flag = {
  "Not only PSMP",
  "PSMP only"
};

static const value_string hta_service_interval_flag[] = {
  {0x00, "5ms"},
  {0x01, "10ms"},
  {0x02, "15ms"},
  {0x03, "20ms"},
  {0x04, "25ms"},
  {0x05, "30ms"},
  {0x06, "35ms"},
  {0x07, "40ms"},
  {0x00, NULL}
};

static const value_string hta_operating_mode_flag[] = {
  {0x00, "Pure HT, no protection"},
  {0x01, "There may be non-HT devices (control & ext channel)"},
  {0x02, "No non-HT is associated, but at least 1 20MHz is. protect on"},
  {0x03, "Mixed: no non-HT is associated, protect on"},
  {0x00, NULL}
};

static const true_false_string hta_non_gf_devices_flag = {
  "All HT devices associated are GF capable",
  "One or More HT devices are not GF capable"
};

static const true_false_string hta_dual_stbc_protection_flag = {
  "Dual CTS protections is used",
  "Regular use of RTS/CTS"
};

static const true_false_string hta_secondary_beacon_flag = {
  "Secondary Beacon",
  "Primary Beacon"
};

static const true_false_string hta_lsig_txop_protection_flag = {
  "Full Support",
  "Not full support"
};

static const true_false_string hta_pco_active_flag = {
  "PCO is activated in the BSS",
  "PCO is not activated in the BSS"
};

static const true_false_string hta_pco_phase_flag = {
  "Switch to 20MHz phase/keep 20MHz",
  "Switch to 40MHz phase/keep 40MHz"
};

static const value_string ht_info_secondary_channel_offset_flags[] = {
  {0x00, "No secondary channel"},
  {0x01, "Secondary channel is above the primary channel"},
  {0x02, "Reserved"},
  {0x03, "Secondary channel is below the primary channel"},
  {0x00, NULL}
};

static const true_false_string ht_info_channel_sta_width_flag = {
  "Channel of any width supported",
  "20 MHz channel width only"
};

static const true_false_string ht_info_rifs_mode_flag = {
  "Permitted",
  "Prohibited"
};

static const value_string ht_info_operating_protection_mode_flags[] = {
  {0x00, "No protection mode"},
  {0x01, "HT non-member protection mode"},
  {0x02, "20 MHz protection mode"},
  {0x03, "non-HT mixed mode"},
  {0x00, NULL}
};

static const true_false_string ht_info_non_greenfield_sta_present_flag = {
  "One or more associated STAs are not greenfield capable",
  "All associated STAs are greenfield capable"
};

static const true_false_string ht_info_obss_non_ht_stas_present_flag = {
  "Use of protection for non-HT STAs by overlapping BSSs is needed",
  "Use of protection for non-HT STAs by overlapping BSSs is not needed"
};

static const true_false_string ht_info_dual_beacon_flag = {
  "AP transmits a secondary beacon",
  "No second beacon is transmitted"
};

static const true_false_string ht_info_stbc_beacon_flag = {
  "STBC beacon",
  "Primary beacon"
};

static const true_false_string htc_lac_trq_flag = {
  "Want sounding PPDU",
  "Don't want sounding PPDU"
};

static const true_false_string htc_lac_mai_mrq_flag = {
  "MCS feedback requested",
  "No MCS feedback requested"
};

static const value_string ieee80211_htc_lac_asel_command_flags[] = {
  {0x00, "Transmit Antenna Selection Sounding Indication (TXASSI)"},
  {0x01, "Transmit Antenna Selection Sounding Request (TXASSR)"},
  {0x02, "Receive Antenna Selection Sounding Indication (RXASSI)"},
  {0x03, "Receive Antenna Selection Sounding Request (RXASSR)"},
  {0x04, "Sounding Label"},
  {0x05, "No feedback, ASEL training failure"},
  {0x06, "Transmit Antenna Selection Sounding Indication (TXASSI) requesting feedback of explicit CSI"},
  {0x07, "Reserved"},
  {0x00, NULL}
};

static const value_string ieee80211_htc_cal_pos_flags[] = {
  {0x00, "Not a calibration frame"},
  {0x01, "Calibration Start"},
  {0x02, "Sounding Response"},
  {0x03, "Sounding Complete"},
  {0x00, NULL}
};

static const value_string ieee80211_htc_csi_steering_flags[] = {
  {0x00, "No feedback required"},
  {0x01, "CSI"},
  {0x02, "Non-compressed Beamforming Feedback Matrix"},
  {0x03, "Compressed Beamforming Feedback Matrix"},
  {0x00, NULL}
};

static const true_false_string ieee80211_htc_ndp_announcement_flag = {
  "NDP will follow",
  "No NDP will follow"
};

static const value_string ieee80211_htc_bw_recommended_vht_mcs_vals[] = {
  {0, "20 MHz"},
  {1, "40 MHz"},
  {2, "80 MHz"},
  {3, "160 MHz and 80+80 MHz"},
  {0, NULL}
};

static const value_string ieee80211_htc_s1g_bw_recommended_vht_mcs_vals[] = {
  {0, "1 MHz"},
  {1, "2 MHz"},
  {2, "4 MHz"},
  {3, "8 MHz"},
  {4, "16 MHz"},
  {5, "Reserved"},
  {6, "Reserved"},
  {7, "Reserved"},
  {0, NULL}
};

static const value_string ieee80211_htc_coding_type_vals[] = {
  {0, "BCC"},
  {1, "LDPC"},
  {0, NULL}
};

static const value_string ieee80211_htc_fb_tx_type_vals[] = {
  {0, "Not beamformed"},
  {1, "Beamformed"},
  {0, NULL}
};

static const value_string ieee80211_tag_secondary_channel_offset_flags[] = {
  {0x00, "No Secondary Channel"},
  {0x01, "Above Primary Channel"},
  {0x02, "Reserved"},
  {0x03, "Below Primary Channel"},
  {0x00, NULL}
};

#define BSS_BITMASK_UP0   0x0001
#define BSS_BITMASK_UP1   0x0002
#define BSS_BITMASK_UP2   0x0004
#define BSS_BITMASK_UP3   0x0008
#define BSS_BITMASK_UP4   0x0010
#define BSS_BITMASK_UP5   0x0020
#define BSS_BITMASK_UP6   0x0040
#define BSS_BITMASK_UP7   0x0080
#define BSS_BITMASK_AC0   0x0100
#define BSS_BITMASK_AC1   0x0200
#define BSS_BITMASK_AC2   0x0400
#define BSS_BITMASK_AC3   0x0800
#define BSS_BITMASK_RSV   0xF000

static const value_string ieee80211_tag_ext_channel_switch_announcement_switch_mode_flags[] = {
  {0x00, "Frames may be transmitted before the channel switch has been completed"},
  {0x01, "No more frames are to be transmitted until the channel switch has been completed"},
  {0x00, NULL}
};

static const value_string service_interval_granularity_vals[] = {
  { 0, "5 ms" },
  { 1, "10 ms" },
  { 2, "15 ms" },
  { 3, "20 ms" },
  { 4, "25 ms" },
  { 5, "30 ms" },
  { 6, "35 ms" },
  { 7, "40 ms" },
  { 0x00, NULL }
};

static const value_string wep_type_vals[] = {
  { DOT11DECRYPT_KEY_TYPE_WEP, STRING_KEY_TYPE_WEP },
  { DOT11DECRYPT_KEY_TYPE_WPA_PWD, STRING_KEY_TYPE_WPA_PWD },
  { DOT11DECRYPT_KEY_TYPE_WPA_PSK, STRING_KEY_TYPE_WPA_PSK },
  { DOT11DECRYPT_KEY_TYPE_TK, STRING_KEY_TYPE_TK },
  { DOT11DECRYPT_KEY_TYPE_MSK, STRING_KEY_TYPE_MSK },
  { 0x00, NULL }
};

static const value_string ieee80211_ht_pren_type_vals[] = {
  { 51,  "HT Capabilities (802.11n D1.10)" },
  { 52,  "HT Additional Capabilities (802.11n D1.00)" },
  { 0, NULL }
};

static const true_false_string ieee80211_cf_ssw_direction = {
  "Transmitted by the beamforming responder",
  "Transmitted by the beamforming initiator"
};

static const value_string bss_type[] = {
  {0x0, "Reserved"},
  {0x1, "IBSS"},
  {0x2, "PBSS"},
  {0x3, "Infrastructure BSS"},
  {0,   NULL}
};

static const value_string band_id[] = {
  {0x0, "TV white spaces"},
  {0x1, "Sub-1 GHz (excluding TV white spaces)"},
  {0x2, "2.4 GHz"},
  {0x3, "3.6 GHz"},
  {0x4, "4.9 and 5 GHz"},
  {0x5, "60 GHz"},
  {0x6, "45 GHz"},
  {0x7, "6 GHz"},
  {0,   NULL}
};

static const value_string extended_sc_mcs[] = {
  {0x0, "None"},
  {0x1, "MCS 9.1"},
  {0x2, "MCS 12.1"},
  {0x3, "MCS 12.2"},
  {0x4, "MCS 12.3"},
  {0x5, "MCS 12.4"},
  {0x6, "MCS 12.5"},
  {0x7, "MCS 12.6"},
  {0, NULL}
};

static const range_string max_basic_sf_amsdu[] = {
  {0, 0, "No Limit"},
  {1, 1, "4 Basic subframes"},
  {2, 2, "8 Basic subframes"},
  {3, 3, "16 Basic subframes"},
  {4, 4, "32 Basic subframes"},
  {5, 5, "64 Basic subframes"},
  {6, 6, "128 Basic subframes"},
  {7, 7, "256 Basic subframes"},
  {8, 255, "reserved"},
  {0, 0, NULL}
};

static const range_string max_short_sf_amsdu[] = {
  {0, 0, "No Limit"},
  {1, 1, "32 Short subframes"},
  {2, 2, "64 Short subframes"},
  {3, 3, "128 Short subframes"},
  {4, 4, "256 Short subframes"},
  {5, 5, "512 Short subframes"},
  {6, 6, "1024 Short subframes"},
  {7, 255, "reserved"},
  {0, 0, NULL}
};

static const value_string allocation_type[] = {
  {0x0, "SP Allocation"},
  {0x1, "CBAP allocation"},
  {0,   NULL}
};

static const value_string vht_operation_info_channel_width[] = {
  {0x00, "20MHz or 40MHz"},
  {0x01, "80MHZ"},
  {0x10, "160MHz"},
  {0x11, "80MHz+80MHz"},
  {0, NULL}
};
static const value_string operating_mode_field_channel_width[] = {
  {0x00, "20MHz"},
  {0x01, "40MHZ"},
  {0x02, "80MHz"},
  {0x03, "160MHz or 80MHz+80MHz"},
  {0, NULL}
};

static const value_string operat_mode_field_rxnss[] = {
  {0x0, "1Nss"},
  {0x1, "2Nss"},
  {0x2, "3Nss"},
  {0x3, "4Nss"},
  {0x4, "5Nss"},
  {0x5, "6Nss"},
  {0x6, "7Nss"},
  {0x7, "8Nss"},
  {0, NULL}
};

#define VHT_ACT_VHT_COMPRESSED_BEAMFORMING  0
#define VHT_ACT_GROUP_ID_MANAGEMENT         1
#define VHT_ACT_OPERATION_MODE_NOTIFICATION 2

static const value_string vht_action_vals[] = {
  {VHT_ACT_VHT_COMPRESSED_BEAMFORMING, "VHT Compressed Beamforming"},
  {VHT_ACT_GROUP_ID_MANAGEMENT, "Group ID Management"},
  {VHT_ACT_OPERATION_MODE_NOTIFICATION, "Operating Mode Notification"},
  {0,   NULL}
};

static const value_string ff_vht_mimo_cntrl_nc_index_vals[] = {
  {0x00, "1 Column"},
  {0x01, "2 Columns"},
  {0x02, "3 Columns"},
  {0x03, "4 Columns"},
  {0x04, "5 Columns"},
  {0x05, "6 Columns"},
  {0x06, "7 Columns"},
  {0x07, "8 Columns"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_nr_index_vals[] = {
  {0x00, "1 Row"},
  {0x01, "2 Rows"},
  {0x02, "3 Rows"},
  {0x03, "4 Rows"},
  {0x04, "5 Rows"},
  {0x05, "6 Rows"},
  {0x06, "7 Rows"},
  {0x07, "8 Rows"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_channel_width_vals[] = {
  {0x00, "20 MHz"},
  {0x01, "40 MHz"},
  {0x02, "80 MHz"},
  {0x03, "160 MHz / 80+80 MHz"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_grouping_vals[] = {
  {0x00, "1 (No Grouping)"},
  {0x01, "2"},
  {0x02, "4"},
  {0x03, "Reserved"},
  {0, NULL}
};

static const value_string ff_vht_mimo_cntrl_feedback_vals[] = {
  {0x00, "SU"},
  {0x01, "MU"},
  {0, NULL}
};

#define S1G_ACT_AID_SWITCH_REQUEST   0
#define S1G_ACT_AID_SWITCH_RESPONSE  1
#define S1G_ACT_SYNC_CONTROL         2
#define S1G_ACT_STA_INFO_ANNOUNCE    3
#define S1G_ACT_EDCA_PARAM_SET       4
#define S1G_ACT_EL_OPERATION         5
#define S1G_ACT_TWT_SETUP            6
#define S1G_ACT_TWT_TEARDOWN         7
#define S1G_ACT_SECT_GROUP_ID_LIST   8
#define S1G_ACT_SECT_ID_FEEDBACK     9
#define S1G_ACT_RESERVED             10
#define S1G_ACT_TWT_INFORMATION      11

static const value_string s1g_action_vals[] = {
  {S1G_ACT_AID_SWITCH_REQUEST, "AID Switch Request"},
  {S1G_ACT_AID_SWITCH_RESPONSE, "AID Switch Response"},
  {S1G_ACT_SYNC_CONTROL, "Sync Control"},
  {S1G_ACT_STA_INFO_ANNOUNCE, "STA Information Announcement"},
  {S1G_ACT_EDCA_PARAM_SET, "EDCA Parameter Set"},
  {S1G_ACT_EL_OPERATION, "EL Operation"},
  {S1G_ACT_TWT_SETUP, "TWT Setup"},
  {S1G_ACT_TWT_TEARDOWN, "TWT Teardown"},
  {S1G_ACT_SECT_GROUP_ID_LIST, "Sectorized Group ID List"},
  {S1G_ACT_SECT_ID_FEEDBACK, "Sector ID Feedback"},
  {S1G_ACT_RESERVED, "Reserved"},
  {S1G_ACT_TWT_INFORMATION, "TWT Information"},
  {0,   NULL},
};

#define PROT_S1G_ACT_REACH_ADDR_UPDATE    0
#define PROT_S1G_ACT_RELAY_ACTIVATE_REQ   1
#define PROT_S1G_ACT_RELAY_ACTIVATE_RESP  2
#define PROT_S1G_ACT_HEADER_COMPRESSION   3
#define PROT_S1G_ACT_TWT_SETUP            4
#define PROT_S1G_ACT_TWT_TEARDOWN         5
#define PROT_S1G_ACT_TWT_INFORMATION      6
#define PROT_S1G_ACT_AID_SWITCH_REQUEST   7
#define PROT_S1G_ACT_AID_SWITCH_RESPONSE  8
#define PROT_S1G_ACT_SYNC_CONTROL         9
#define PROT_S1G_ACT_STA_INFO_ANNOUNCE    10
#define PROT_S1G_ACT_EDCA_PARAM_SET       11
#define PROT_S1G_ACT_EL_OPERATION         12
#define PROT_S1G_ACT_SECT_GROUP_ID_LIST   13
#define PROT_S1G_ACT_SECT_ID_FEEDBACK     14

static const value_string prot_s1g_action_vals[] = {
  {PROT_S1G_ACT_REACH_ADDR_UPDATE, "Reachable Address Update"},
  {PROT_S1G_ACT_RELAY_ACTIVATE_REQ, "Relay Activation Request"},
  {PROT_S1G_ACT_RELAY_ACTIVATE_RESP, "Relay Activation Response"},
  {PROT_S1G_ACT_HEADER_COMPRESSION, "Header Compression"},
  {PROT_S1G_ACT_TWT_SETUP, "Protected TWT Setup"},
  {PROT_S1G_ACT_TWT_TEARDOWN, "Protected TWT Teardown"},
  {PROT_S1G_ACT_TWT_INFORMATION, "Protected TWT Information"},
  {PROT_S1G_ACT_AID_SWITCH_REQUEST, "Protected AID Switch Request"},
  {PROT_S1G_ACT_AID_SWITCH_RESPONSE, "Protected AID Switch Response"},
  {PROT_S1G_ACT_SYNC_CONTROL, "Protected Sync Control"},
  {PROT_S1G_ACT_STA_INFO_ANNOUNCE, "Protected STA Information Announcement"},
  {PROT_S1G_ACT_EDCA_PARAM_SET, "Protected EDCA Parameter Set"},
  {PROT_S1G_ACT_EL_OPERATION, "Protected EL Operation"},
  {PROT_S1G_ACT_SECT_GROUP_ID_LIST, "Protected Sectorized Group ID List"},
  {PROT_S1G_ACT_SECT_ID_FEEDBACK, "Protected Sector ID Feedback"},
  {0,   NULL},
};

static const value_string twt_neg_type_vals[] = {
  {0x0, "Individual TWT"},
  {0x1, "Wake TBTT"},
  {0x2, "Broadcast TWT schedule, TWT element in broadcast management frame"},
  {0x3, "Broadcast TWT schedule, TWT element in unicast management frame"},
  {0,   NULL},
};

static const true_false_string twt_responder_pm_mode_tfs = {
  "AP may doze outside the TWT",
  "AP is always awake",
};

static const true_false_string twt_requester_tfs = {
  "This STA is a TWT Requesting STA",
  "This STA is a TWT Responding STA or a TWT scheduling AP",
};

static const true_false_string twt_info_frame_disabled_tfs = {
  "Reception of TWT Information frames is disabled",
  "Reception of TWT Information frames is enabled",
};

static const true_false_string twt_wake_duration_unit_tfs = {
  "1 TU",
  "256 microseconds",
};

#define REQUEST_TWT   0
#define SUGGEST_TWT   1
#define DEMAND_TWT    2
#define TWT_GROUPING  3
#define ACCEPT_TWT    4
#define ALTERNATE_TWT 5
#define DICTATE_TWT   6
#define REJECT_TWT    7

static const value_string twt_setup_cmd[] = {
  {REQUEST_TWT,   "Request TWT"},
  {SUGGEST_TWT,   "Suggest TWT"},
  {DEMAND_TWT,    "Demand TWT"},
  {TWT_GROUPING,  "TWT Grouping"},
  {ACCEPT_TWT,    "Accept TWT"},
  {ALTERNATE_TWT, "Alternate TWT"},
  {DICTATE_TWT,   "Dictate TWT"},
  {REJECT_TWT,    "Reject TWT"},
  {0,   NULL},
};

static const true_false_string twt_trigger = {
  "TWT SP includes trigger frames",
  "TWT SP does not include trigger frames",
};

static const true_false_string twt_implicit = {
  "TWT is implicit",
  "TWT is explicit",
};

static const true_false_string twt_flow_type = {
  "TWT is unannounced, the TWT responding STA can send frames at any time",
  "TWT is announced, the TWT Requesting STA will send trigger frames",
};

static const value_string he_phy_device_class_vals[] = {
  { 0, "Class B Device" },
  { 1, "Class A Device" },
  { 0, NULL }
};

static const value_string he_phy_midamble_tx_rx_max_nsts_vals[] = {
  { 0, "1 Space-Time Stream" },
  { 1, "2 Space-Time Streams" },
  { 2, "3 Space-Time Streams" },
  { 3, "4 Space-Time Streams" },
  { 0, NULL }
};

static const value_string he_phy_dcm_max_constellation_vals[] = {
  { 0, "DCM is not supported" },
  { 1, "BPSK" },
  { 2, "QPSK" },
  { 3, "16-QAM" },
  { 0, NULL }
};

static const value_string he_phy_dcm_max_nss_vals[] = {
  { 0, "1 Spatial Stream" },
  { 1, "2 Spatial Streams" },
  { 0, NULL }
};

static const value_string he_phy_dcm_max_ru_vals[] = {
  { 0, "242-tone RU" },
  { 1, "484-tone RU" },
  { 2, "996-tone RU" },
  { 3, "2x996-tone RU"},
  { 0, NULL }
};

static const value_string he_phy_nominal_packet_padding_vals[] = {
  { 0, "0 µs for all constellations, Nss and RU allocations" },
  { 1, "8 µs for all constellations, Nss and RU allocations" },
  { 2, "16 µs for all constellations, Nss and RU allocations" },
  { 3, "Reserved" },
  { 0, NULL }
};

// MBO-OCE attributes
#define MBO_AP_CAPABILITY_INDICATION    1
#define MBO_NON_PREF_CHANNEL_REPORT     2
#define MBO_CELLULAR_DATA_CAPABILITIES  3
#define MBO_ASSOCIATION_DISALLOWED      4
#define MBO_CELLULAR_DATA_PREFERENCE    5
#define MBO_TRANSITION_REASON           6
#define MBO_TRANSITION_REJECTION_REASON 7
#define MBO_ASSOCIATION_RETRY_DELAY     8
#define OCE_CAPABILITY_INDICATION       101
#define OCE_RSSI_ASSOCIATION_REJECTION  102
#define OCE_REDUCED_WAN_METRICS         103
#define OCE_RNR_COMPLETENESS            104
#define OCE_PROBE_SUPPR_BSSID           105
#define OCE_PROBE_SUPPR_SSID            106

static const value_string wfa_mbo_oce_attr_id_vals[] = {
  { MBO_AP_CAPABILITY_INDICATION, "MBO AP Capability Indication"},
  { MBO_NON_PREF_CHANNEL_REPORT, "Non-preferred Channel Report"},
  { MBO_CELLULAR_DATA_CAPABILITIES, "Cellular Data Capabilities"},
  { MBO_ASSOCIATION_DISALLOWED, "Association Disallowed"},
  { MBO_CELLULAR_DATA_PREFERENCE, "Cellular Data Connection Preference"},
  { MBO_TRANSITION_REASON, "Transition Reason Code BTM Request"},
  { MBO_TRANSITION_REJECTION_REASON, "Transition Rejection Reason Code"},
  { MBO_ASSOCIATION_RETRY_DELAY, "Association Retry Delay"},
  { OCE_CAPABILITY_INDICATION, "OCE Capability Indication" },
  { OCE_RSSI_ASSOCIATION_REJECTION, "RSSI-based (Re-)Association Rejection" },
  { OCE_REDUCED_WAN_METRICS, "Reduced WAN Metrics" },
  { OCE_RNR_COMPLETENESS, "RNR Completeness" },
  { OCE_PROBE_SUPPR_BSSID, "Probe Suppression BSSIDs" },
  { OCE_PROBE_SUPPR_SSID, "Probe Suppression SSIDs" },
  { 0, NULL}
};

static const value_string wfa_mbo_non_pref_chan_pref_vals[] = {
  { 0, "non-operable band/channel for the STA"},
  { 1, "band/channel the STA prefers not to operate in"},
  { 255, "band/channel the STA prefers to operate in"},
  { 0, NULL }
};

static const value_string wfa_mbo_non_pref_chan_reason_vals[] = {
  { 0, "Unspecified reason"},
  { 1, "An unacceptable level of interference is being experienced by STA in this channel"},
  { 2, "The STA has another active connection in this channel, "
    "or near enough to this channel to cause operating interference"},
  { 0, NULL }
};

static const value_string wfa_mbo_cellular_cap_vals[] = {
  { 1, "Cellular data connection available"},
  { 2, "Cellular data connection not available"},
  { 3, "Not Cellular data capable"},
  { 0, NULL }
};

static const value_string wfa_mbo_assoc_disallow_reason_vals[] = {
  { 1, "Unspecified reason"},
  { 2, "Maximum number of associated STAs reached"},
  { 3, "Air interface is overloaded"},
  { 4, "Authentication server overloaded"},
  { 5, "Insufficient RSSI"},
  { 0, NULL }
};

static const value_string wfa_mbo_cellular_pref_vals[] = {
  { 0, "Excluded. The AP does not want STA to use the cellular data connection"},
  { 1, "The AP prefers the STA should not use cellular data connection"},
  { 255, "The AP prefers the STA should use cellular data connection"},
  { 0, NULL }
};

static const value_string wfa_mbo_transition_reason_vals[] = {
  { 0, "Unspecified"},
  { 1, "Excessive frame loss rate"},
  { 2, "Excessive delay for current traffic stream"},
  { 3, "Insufficient bandwidth for current traffic stream"},
  { 4, "Load balancing"},
  { 5, "Low RSSI"},
  { 6, "Received excessive number of retransmissions"},
  { 7, "High interference"},
  { 8, "Gray zone"},
  { 9, "Transitioning to a premium AP"},
  { 0, NULL }
};

static const value_string wfa_mbo_transition_rej_reason_vals[] = {
  { 0, "Unspecified"},
  { 1, "Excessive frame loss rate expected by the STA if it transitions"},
  { 2, "Excessive delay for current traffic stream would be incurred by BSS transition at this time"},
  { 3, "Insufficient QoS capacity for current traffic stream expected by the STA if it transitions"},
  { 4, "Low RSSI in frames being received by the STA from to the suggested candidate BSS(s)"},
  { 5, "High interference expected by STA if it transitions"},
  { 6, "Service Availability – the STA expects that services it needs "
    "which are available at its serving AP will not be available if it transitions"},
  { 0, NULL }
};

/* 802.11ai FILS Discovery */
#define PA_FILS_FC_SSID_LENGTH  0x001F
#define PA_FILS_FC_CAPABILITY   0x0020
#define PA_FILS_FC_SHORT_SSID   0x0040
#define PA_FILS_FC_AP_CSN       0x0080
#define PA_FILS_FC_ANO          0x0100
#define PA_FILS_FC_CCFS1        0x0200
#define PA_FILS_FC_PC           0x0400
#define PA_FILS_FC_RSN_INFO     0x0800
#define PA_FILS_FC_LENGTH       0x1000
#define PA_FILS_FC_MD           0x2000
#define PA_FILS_FC_RESERVED     0xC000

static const value_string fils_discovery_capability_bss_operating_channel_width[] = {
  {0, "20MHz (or 22Mhz) / TVHT_W"},
  {1, "40MHZ / TVHT_W+W"},
  {2, "80MHz / TVHT_2W"},
  {3, "160MHz or 80MHz+80MHz / TVHT_4W or TVHT_2W+2W"},
  {0, NULL}
};

static const value_string fils_discovery_capability_max_number_of_spatial_streams[] = {
  {0, "1 spatial stream"},
  {1, "2 spatial streams"},
  {2, "3 spatial streams"},
  {3, "4 spatial streams"},
  {4, "5-8 spatial streams"},
  {0, NULL}
};

static const value_string fils_discovery_capability_phy_index[] = {
  {0, "HR/DSSS"},
  {1, "ERP-OFDM"},
  {2, "HT"},
  {3, "VHT or TVHT"},
  {4, "HE"},
  {0x00, NULL}
};

static const value_string fils_discovery_capability_fils_minimum_rate_dsss[] = {
  {0, "1 Mbps"},
  {1, "2 Mbps"},
  {2, "5.5 Mbps"},
  {3, "11 Mbps"},
  {0x00, NULL}
};

static const value_string fils_discovery_capability_fils_minimum_rate_ofdm[] = {
  {0, "6 Mbps"},
  {1, "9 Mbps"},
  {2, "12 Mbps"},
  {3, "18 Mbps"},
  {4, "24 Mbps"},
  {0x00, NULL}
};

static const value_string fils_discovery_capability_fils_minimum_rate_ht_vht_tvht[] = {
  {0, "MCS 0"},
  {1, "MCS 1"},
  {2, "MCS 2"},
  {3, "MCS 3"},
  {4, "MCS 4"},
  {0x00, NULL}
};

static const value_string fils_discovery_capability_fils_minimum_rate_he[] = {
  {0, "HE-MCS 0"},
  {1, "HE-MCS 1"},
  {2, "HE-MCS 2"},
  {3, "HE-MCS 3"},
  {4, "HE-MCS 4"},
  {0x00, NULL}
};

static const value_string wfa_rsne_variant_vals[] = {
  { 0, "RSNE" },
  { 1, "RSNE Override" },
  { 2, "RSNE Override 2" },
  { 0, NULL }
};

static int proto_wlan;
static int proto_centrino;
static int proto_aggregate;
static bool ieee80211_tvb_invalid;
static int proto_eapol; /* Externally retrieved */

/* ************************************************************************* */
/*                Header field info values for FC-field                      */
/* ************************************************************************* */
static int hf_ieee80211_fc_field;
static int hf_ieee80211_fc_proto_version;
static int hf_ieee80211_fc_frame_type;
static int hf_ieee80211_fc_frame_subtype;
static int hf_ieee80211_fc_frame_extension;
static int hf_ieee80211_fc_frame_type_subtype;

static int hf_ieee80211_fc_flags;
static int hf_ieee80211_fc_flags_str;
static int hf_ieee80211_fc_to_ds;
static int hf_ieee80211_fc_from_ds;
static int hf_ieee80211_fc_data_ds;

static int hf_ieee80211_fc_more_frag;
static int hf_ieee80211_fc_retry;
static int hf_ieee80211_fc_pwr_mgt;
static int hf_ieee80211_fc_more_data;
static int hf_ieee80211_fc_protected;
static int hf_ieee80211_fc_order;

/* S1G Flags */
static int hf_ieee80211_fc_s1g_next_tbtt_present;
static int hf_ieee80211_fc_s1g_compressed_ssid_present;
static int hf_ieee80211_fc_s1g_ano_present;
static int hf_ieee80211_fc_s1g_bss_bw;
static int hf_ieee80211_fc_s1g_security;
static int hf_ieee80211_fc_s1g_ap_pm;

/* S1G PV0 fields */
static int hf_ieee80211_fc_s1g_bw_indication;
static int hf_ieee80211_fc_s1g_dynamic_indication;

/* PV1 fields */
static int hf_ieee80211_fc_pv1_proto_version;
static int hf_ieee80211_fc_pv1_type;
static int hf_ieee80211_fc_pv1_ptid;
static int hf_ieee80211_fc_pv1_mgmt_subtype;
static int hf_ieee80211_fc_pv1_cntl_subtype;
static int hf_ieee80211_fc_pv1_unk_field;
static int hf_ieee80211_fc_pv1_bw_indication;
static int hf_ieee80211_fc_pv1_dynamic_indication;
static int hf_ieee80211_fc_pv1_cntl_power_mgmt;
static int hf_ieee80211_fc_pv1_cntl_more_data;
static int hf_ieee80211_fc_pv1_cntl_flow_control;
static int hf_ieee80211_fc_pv1_cntl_next_twt_info;
static int hf_ieee80211_fc_pv1_mgmt_pr_next_tbt;
static int hf_ieee80211_fc_pv1_mgmt_pr_full_ssid;
static int hf_ieee80211_fc_pv1_mgmt_pr_ano;
static int hf_ieee80211_fc_pv1_mgmt_pr_bss_bw;
static int hf_ieee80211_fc_pv1_mgmt_pr_security;
static int hf_ieee80211_fc_pv1_mgmt_pr_1mhz_pc;
static int hf_ieee80211_fc_pv1_mgmt_pr_slot_assign;
static int hf_ieee80211_fc_pv1_mgmt_pr_more_frag;
static int hf_ieee80211_fc_pv1_mgmt_pr_pwr_mgmt;
static int hf_ieee80211_fc_pv1_mgmt_pr_grp_indic;
static int hf_ieee80211_fc_pv1_mgmt_pr_protected;
static int hf_ieee80211_fc_pv1_mgmt_pr_end_of_svc;
static int hf_ieee80211_fc_pv1_mgmt_pr_relayed_frm;
static int hf_ieee80211_fc_pv1_mgmt_pr_ack_policy;
static int hf_ieee80211_fc_pv1_from_ds;
static int hf_ieee80211_fc_pv1_more_fragments;
static int hf_ieee80211_fc_pv1_power_mgmt;
static int hf_ieee80211_fc_pv1_more_data;
static int hf_ieee80211_fc_pv1_protected_frame;
static int hf_ieee80211_fc_pv1_end_service_per;
static int hf_ieee80211_fc_pv1_relayed_frame;
static int hf_ieee80211_fc_pv1_ack_policy;
static int hf_ieee80211_pv1_sid;
static int hf_ieee80211_pv1_sid_association_id;
static int hf_ieee80211_pv1_sid_a3_present;
static int hf_ieee80211_pv1_sid_a4_present;
static int hf_ieee80211_pv1_sid_a_msdu;

static int hf_ieee80211_pv1_cnt_stack_tetra_timest;
static int hf_ieee80211_pv1_cnt_bat_beacon_seq;
static int hf_ieee80211_pv1_cnt_bat_penta_timest;
static int hf_ieee80211_pv1_cnt_bat_next_twt_info;
static int hf_ieee80211_pv1_cnt_bat_stating_seq_cntl;
static int hf_ieee80211_pv1_cnt_bat_bitmap;

static int hf_ieee80211_pv1_mgmt_reserved;
static int hf_ieee80211_pv1_cntl_reserved;

typedef struct retransmit_key {
  uint8_t bssid[6];
  uint8_t src[6];
  uint16_t seq_control;
  unsigned   fnum;
} retransmit_key;

static GHashTable *fc_analyse_retransmit_table;
static GHashTable *fc_first_frame_table;

static int hf_ieee80211_fc_analysis_retransmission;
static int hf_ieee80211_fc_analysis_retransmission_frame;

/* ************************************************************************* */
/*                   Header values for Duration/ID field                     */
/* ************************************************************************* */
static int hf_ieee80211_did_duration;
static int hf_ieee80211_assoc_id;

/* ************************************************************************* */
/*         Header values for different address-fields (all 4 of them)        */
/* ************************************************************************* */
static int hf_ieee80211_addr_da;  /* Destination address subfield */
static int hf_ieee80211_addr_da_resolved;  /* Dst addr subfield resolved*/
static int hf_ieee80211_addr_da_oui;  /* Destination address subfield */
static int hf_ieee80211_addr_da_oui_resolved;  /* Dst addr subfield resolved*/
static int hf_ieee80211_addr_da_ig;  /* Destination address IG bit */
static int hf_ieee80211_addr_da_lg;  /* Destination address LG bit */

static int hf_ieee80211_addr_sa;  /* Source address subfield */
static int hf_ieee80211_addr_sa_resolved;  /* Src addr subfield resolved*/
static int hf_ieee80211_addr_sa_oui;  /* Source address subfield */
static int hf_ieee80211_addr_sa_oui_resolved;  /* Src addr subfield resolved*/
static int hf_ieee80211_addr_sa_ig;  /* Source address IG bit */
static int hf_ieee80211_addr_sa_lg;  /* Source address LG bit */

static int hf_ieee80211_addr_ra;  /* Receiver address subfield */
static int hf_ieee80211_addr_ra_resolved;  /* Rcv addr subfield resolved*/
static int hf_ieee80211_addr_ra_oui;  /* Receiver address subfield */
static int hf_ieee80211_addr_ra_oui_resolved;  /* Rcv addr subfield resolved*/
static int hf_ieee80211_addr_ra_ig;  /* Receiver address IG bit */
static int hf_ieee80211_addr_ra_lg;  /* Receiver address LG bit */

static int hf_ieee80211_addr_ta;  /* Transmitter address subfield */
static int hf_ieee80211_addr_ta_resolved;  /* Txm addr subfield resolved*/
static int hf_ieee80211_addr_ta_oui;  /* Transmitter address subfield */
static int hf_ieee80211_addr_ta_oui_resolved;  /* Txm addr subfield resolved*/
static int hf_ieee80211_addr_ta_ig;  /* Transmitter address IG bit */
static int hf_ieee80211_addr_ta_lg;  /* Transmitter address LG bit */

static int hf_ieee80211_addr_bssid;  /* address is bssid */
static int hf_ieee80211_addr_bssid_resolved;  /* bssid resolved*/
static int hf_ieee80211_addr_bssid_oui;  /* address is bssid */
static int hf_ieee80211_addr_bssid_oui_resolved;  /* bssid resolved*/
static int hf_ieee80211_addr_bssid_ig;  /* bssid address IG bit */
static int hf_ieee80211_addr_bssid_lg;  /* bssid address LG bit */

static int hf_ieee80211_addr_staa;  /* address is station address */
static int hf_ieee80211_addr_staa_resolved;  /* station address resolved*/
static int hf_ieee80211_addr_staa_oui;  /* address is station address */
static int hf_ieee80211_addr_staa_oui_resolved;  /* station address resolved*/
static int hf_ieee80211_addr_staa_ig;  /* station address IG bit */
static int hf_ieee80211_addr_staa_lg;  /* station address LG bit */

static int hf_ieee80211_addr;  /* Source or destination address subfield */
static int hf_ieee80211_addr_resolved;/*Src/dst address subfield resolved*/
static int hf_ieee80211_addr_oui;  /* Source or destination address subfield */
static int hf_ieee80211_addr_oui_resolved;/*Src/dst address subfield resolved*/
static int hf_ieee80211_addr_ig;  /* Src/dst address IG bit */
static int hf_ieee80211_addr_lg;  /* Src/dst address LG bit */

static int hf_ieee80211_mgt;

/* ************************************************************************* */
/*                Header values for QoS control field                        */
/* ************************************************************************* */
static int hf_ieee80211_qos;
static int hf_ieee80211_qos_tid;
static int hf_ieee80211_qos_priority;
static int hf_ieee80211_qos_ack_policy;
static int hf_ieee80211_qos_amsdu_present;
static int hf_ieee80211_qos_eosp;
static int hf_ieee80211_qos_bit4;
static int hf_ieee80211_qos_txop_limit;
static int hf_ieee80211_qos_ps_buf_state;
static int hf_ieee80211_qos_buf_state_indicated;
static int hf_ieee80211_qos_highest_pri_buf_ac;
static int hf_ieee80211_qos_qap_buf_load;
static int hf_ieee80211_qos_txop_dur_req;
static int hf_ieee80211_qos_queue_size;
static int hf_ieee80211_qos_mesh_ctl_present;
static int hf_ieee80211_qos_mesh_ps_rsvd;
static int hf_ieee80211_qos_mesh_ps_unicast;
static int hf_ieee80211_qos_mesh_ps_multicast;
static int hf_ieee80211_qos_mesh_rspi;

/* ************************************************************************* */
/*                Header values for HT control field (+HTC) and HE control   */
/* ************************************************************************* */
/* 802.11-2012 and 802.11ac-2013 8.2.4.6 */
static int hf_ieee80211_htc;
static int hf_ieee80211_htc_vht;
static int hf_ieee80211_htc_he;
static int hf_ieee80211_htc_he_ctrl_id;
static int hf_ieee80211_he_a_control_padding;
static int hf_ieee80211_he_a_control_ones;
static int hf_ieee80211_he_trs_he_tb_ppdu_len;
static int hf_ieee80211_he_trs_ru_allocation;
static int hf_ieee80211_he_dl_tx_power;
static int hf_ieee80211_he_ul_target_rssi;
static int hf_ieee80211_he_ul_mcs;
static int hf_ieee80211_he_ul_reserved;
static int hf_ieee80211_he_om_rx_nss;
static int hf_ieee80211_he_om_channel_width;
static int hf_ieee80211_he_om_ul_mu_disable;
static int hf_ieee80211_he_om_tx_nsts;
static int hf_ieee80211_he_om_er_su_disable;
static int hf_ieee80211_he_om_dl_mu_mimo_resound;
static int hf_ieee80211_he_om_ul_mu_data_disable;
static int hf_ieee80211_he_hla_unsolicited_mfb;
static int hf_ieee80211_he_hla_mrq;
static int hf_ieee80211_he_hla_nss;
static int hf_ieee80211_he_hla_he_mcs;
static int hf_ieee80211_he_hla_dcm;
static int hf_ieee80211_he_hla_ru;
static int hf_ieee80211_he_hla_bw;
static int hf_ieee80211_he_hla_msi_ppdu_type;
static int hf_ieee80211_he_hla_tx_bf;
static int hf_ieee80211_he_hla_reserved;
static int hf_ieee80211_he_bsr_aci_bitmap;
static int hf_ieee80211_he_bsr_delta_tid;
static int hf_ieee80211_he_bsr_aci_high;
static int hf_ieee80211_he_bsr_scaling_factor;
static int hf_ieee80211_he_bsr_queue_size_high;
static int hf_ieee80211_he_bsr_queue_size_all;
static int hf_ieee80211_he_uph_ul_power_headroom;
static int hf_ieee80211_he_uph_ul_min_transmit_power_flag;
static int hf_ieee80211_he_uph_reserved;
static int hf_ieee80211_he_cci_ac_constraint;
static int hf_ieee80211_he_cci_rdg_more_ppdu;
static int hf_ieee80211_he_cci_sr_ppdu_indic;
static int hf_ieee80211_he_cci_reserved;
static int hf_ieee80211_he_eht_om_rx_nss_ext;
static int hf_ieee80211_he_eht_om_chan_w_ext;
static int hf_ieee80211_he_eht_om_tx_nsts_ext;
static int hf_ieee80211_he_eht_om_reserved;
static int hf_ieee80211_he_srs_ppdu_resp_dur;
static int hf_ieee80211_he_srs_reserved;
static int hf_ieee80211_he_aar_assisted_ap_bitmap;
static int hf_ieee80211_he_aar_reserved;
static int hf_ieee80211_he_btc_avail_chan;
static int hf_ieee80211_he_btc_reserved;
static int hf_ieee80211_he_trigger_common_info;
static int hf_ieee80211_he_trigger_type;
static int hf_ieee80211_he_trigger_ul_length;
static int hf_ieee80211_he_trigger_more_tf;
static int hf_ieee80211_he_trigger_cs_required;
static int hf_ieee80211_he_trigger_ul_bw;
static int hf_ieee80211_he_trigger_gi_and_ltf_type;
static int hf_ieee80211_he_trigger_mu_mimo_ltf_mode;
static int hf_ieee80211_he_trigger_num_he_ltf_syms_etc;
static int hf_ieee80211_he_trigger_ul_stbc;
static int hf_ieee80211_he_trigger_ldpc_extra_sym_seg;
static int hf_ieee80211_he_trigger_ap_tx_power;
static int hf_ieee80211_he_trigger_pre_fec_padding_factor;
static int hf_ieee80211_he_trigger_pe_disambiguity;
static int hf_ieee80211_he_trigger_ul_spatial_reuse;
static int hf_ieee80211_he_trigger_doppler;
static int hf_ieee80211_he_trigger_ul_he_sig_a_reserved;
static int hf_ieee80211_he_trigger_reserved;
static int hf_ieee80211_he_trigger_user_info;
static int hf_ieee80211_he_trigger_user_info_padding_start;
static int hf_ieee80211_he_trigger_padding;
static int hf_ieee80211_he_trigger_bar_ctrl;
static int hf_ieee80211_he_trigger_bar_ctrl_ba_ack_policy;
static int hf_ieee80211_he_trigger_bar_ctrl_ba_type;
static int hf_ieee80211_he_trigger_bar_ctrl_reserved;
static int hf_ieee80211_he_trigger_bar_ctrl_tid_info;
static int hf_ieee80211_he_trigger_bar_info;
static int hf_ieee80211_he_trigger_bar_info_blk_ack_seq_ctrl;
static int hf_ieee80211_he_trigger_mpdu_mu_spacing;
static int hf_ieee80211_he_trigger_tid_aggregation_limit;
static int hf_ieee80211_he_trigger_dependent_reserved1;
static int hf_ieee80211_he_trigger_preferred_ac;
static int hf_ieee80211_he_trigger_starting_aid;
static int hf_ieee80211_he_trigger_dependent_reserved2;
static int hf_ieee80211_he_trigger_feedback_type;
static int hf_ieee80211_he_trigger_dependent_reserved3;
static int hf_ieee80211_he_trigger_nfrp_target_rssi;
static int hf_ieee80211_he_trigger_multiplexing_flag;
static int hf_ieee80211_he_trigger_dep_nfrp_user_info;
static int hf_ieee80211_he_trigger_feedback_seg_retrans_bm;
static int hf_ieee80211_he_trigger_aid12;
static int hf_ieee80211_he_trigger_ru_allocation;
static int hf_ieee80211_he_trigger_ru_allocation_region;
static int hf_ieee80211_he_trigger_ru_starting_spatial_stream;
static int hf_ieee80211_he_trigger_ru_number_spatial_streams;
static int hf_ieee80211_he_trigger_ru_number_ra_ru;
static int hf_ieee80211_he_trigger_ru_no_more_ra_ru;
static int hf_ieee80211_he_trigger_ul_fec_coding_type;
static int hf_ieee80211_he_trigger_ul_mcs;
static int hf_ieee80211_he_trigger_ul_dcm;
static int hf_ieee80211_he_trigger_ul_target_rssi;
static int hf_ieee80211_he_trigger_user_reserved;
static int hf_ieee80211_he_trigger_dep_basic_user_info;
static int hf_ieee80211_he_trigger_ranging_common_info_1;
static int hf_ieee80211_he_trigger_ranging_common_info_2;
static int hf_ieee80211_eht_trigger_common_info;
static int hf_ieee80211_eht_trigger_type;
static int hf_ieee80211_eht_trigger_ul_length;
static int hf_ieee80211_eht_trigger_more_tf;
static int hf_ieee80211_eht_trigger_cs_required;
static int hf_ieee80211_eht_trigger_ul_bw;
static int hf_ieee80211_eht_trigger_gi_and_eht_ltf_type;
static int hf_ieee80211_eht_trigger_num_he_eht_ltf_syms_etc;
static int hf_ieee80211_eht_trigger_reserved2;
static int hf_ieee80211_eht_trigger_ldpc_extra_sym_seg;
static int hf_ieee80211_eht_trigger_ap_tx_power;
static int hf_ieee80211_eht_trigger_pre_fec_padding_factor;
static int hf_ieee80211_eht_trigger_pe_disambiguity;
static int hf_ieee80211_eht_trigger_ul_spatial_reuse;
static int hf_ieee80211_eht_trigger_reserved3;
static int hf_ieee80211_eht_trigger_he_eht_p160;
static int hf_ieee80211_eht_trigger_special_user_info_flag;
static int hf_ieee80211_eht_trigger_eht_reserved;
static int hf_ieee80211_eht_trigger_reserved4;
static int hf_ieee80211_eht_trigger_reserved;
static int hf_ieee80211_eht_trigger_aid12;
static int hf_ieee80211_eht_trigger_special_user_info;
static int hf_ieee80211_eht_trigger_phy_version_identifier;
static int hf_ieee80211_eht_trigger_ul_bw_extenstion;
static int hf_ieee80211_eht_trigger_eht_spatial_reuse_1;
static int hf_ieee80211_eht_trigger_eht_spatial_reuse_2;
static int hf_ieee80211_eht_trigger_disregard_u_sig_1;
static int hf_ieee80211_eht_trigger_validate_u_sig_2;
static int hf_ieee80211_eht_trigger_disregard_u_sig_2_4lsb;
static int hf_ieee80211_eht_trigger_disregard_u_sig_2_msb;
static int hf_ieee80211_eht_trigger_special_reserved;
static int hf_ieee80211_eht_trigger_user_info;
static int hf_ieee80211_eht_trigger_ru_allocation_region;
static int hf_ieee80211_eht_trigger_ru_allocation;
static int hf_ieee80211_eht_trigger_ul_fec_coding_type;
static int hf_ieee80211_eht_trigger_ul_eht_mcs;
static int hf_ieee80211_eht_trigger_ru_starting_spatial_stream;
static int hf_ieee80211_eht_trigger_ru_number_spatial_streams;
static int hf_ieee80211_eht_trigger_ul_target_recv_power;
static int hf_ieee80211_eht_trigger_ps160;
static int hf_ieee80211_eht_trigger_user_info_reserved;
static int hf_ieee80211_uhr_trigger_common_info;
static int hf_ieee80211_uhr_trigger_type;
static int hf_ieee80211_uhr_trigger_ul_length;
static int hf_ieee80211_uhr_trigger_more_tf;
static int hf_ieee80211_uhr_trigger_cs_required;
static int hf_ieee80211_uhr_trigger_ul_bw;
static int hf_ieee80211_uhr_trigger_gi_and_he_uhr_ltf_type;
static int hf_ieee80211_uhr_trigger_num_he_uhr_ltf_syms_etc;
static int hf_ieee80211_uhr_trigger_reserved2;
static int hf_ieee80211_uhr_trigger_ldpc_extra_sym_seg;
static int hf_ieee80211_uhr_trigger_ap_tx_power;
static int hf_ieee80211_uhr_trigger_pre_fec_padding_factor;
static int hf_ieee80211_uhr_trigger_pe_disambiguity;
static int hf_ieee80211_uhr_trigger_ul_spatial_reuse;
static int hf_ieee80211_uhr_trigger_reserved3;
static int hf_ieee80211_uhr_trigger_he_uhr_p160;
static int hf_ieee80211_uhr_trigger_special_user_info_flag;
static int hf_ieee80211_uhr_trigger_dru_rru;
static int hf_ieee80211_uhr_trigger_ifcs;
static int hf_ieee80211_uhr_trigger_uhr_reserved;
static int hf_ieee80211_uhr_trigger_reserved4;
static int hf_ieee80211_uhr_trigger_reserved;
static int hf_ieee80211_uhr_trigger_user_info;
static int hf_ieee80211_uhr_trigger_aid12;
static int hf_ieee80211_uhr_trigger_ru_allocation_region;
static int hf_ieee80211_uhr_trigger_ru_allocation;
static int hf_ieee80211_uhr_trigger_ul_fec_coding_type;
static int hf_ieee80211_uhr_trigger_ul_uhr_mcs;
static int hf_ieee80211_uhr_trigger_2xldpc;
static int hf_ieee80211_uhr_trigger_ru_starting_spatial_stream;
static int hf_ieee80211_uhr_trigger_ru_number_spatial_streams;
static int hf_ieee80211_uhr_trigger_dru_distribution_bw;
static int hf_ieee80211_uhr_trigger_dru_reserved;
static int hf_ieee80211_uhr_trigger_ul_target_recv_power;
static int hf_ieee80211_uhr_trigger_ps160;


static int hf_ieee80211_ranging_trigger_subtype1;
static int hf_ieee80211_ranging_trigger_reserved1;
static int hf_ieee80211_ranging_trigger_token;
static int hf_ieee80211_ranging_trigger_subtype2;
static int hf_ieee80211_ranging_trigger_reserved2;
static int hf_ieee80211_ranging_trigger_sounding_dialog_token;
static int hf_ieee80211_he_trigger_ranging_trigger_poll_rpt;
static int hf_ieee80211_ranging_pol_rpt_aid12_rsid12;
static int hf_ieee80211_ranging_pol_rpt_ru_alloc_region;
static int hf_ieee80211_ranging_pol_rpt_ru_alloc;
static int hf_ieee80211_ranging_pol_rpt_ul_fec_coding_type;
static int hf_ieee80211_ranging_pol_rpt_ulmcs;
static int hf_ieee80211_ranging_pol_rpt_uldcm;
static int hf_ieee80211_ranging_pol_rpt_starting_spatial_stream;
static int hf_ieee80211_ranging_pol_rpt_number_spatial_streams;
static int hf_ieee80211_ranging_pol_rpt_ul_target_rssi;
static int hf_ieee80211_ranging_pol_rpt_reserved;
static int hf_ieee80211_he_trigger_ranging_trigger_sounding;
static int hf_ieee80211_ranging_sounding_aid12_rsid12;
static int hf_ieee80211_ranging_sounding_reserved1;
static int hf_ieee80211_ranging_sounding_i2r_rep;
static int hf_ieee80211_ranging_sounding_reserved2;
static int hf_ieee80211_ranging_sounding_starting_spatial_stream;
static int hf_ieee80211_ranging_sounding_number_spatial_streams;
static int hf_ieee80211_ranging_sounding_ul_target_rssi;
static int hf_ieee80211_ranging_sounding_reserved3;
static int hf_ieee80211_he_trigger_ranging_trigger_sec_sound;
static int hf_ieee80211_ranging_sec_sound_aid12_rsid12;
static int hf_ieee80211_ranging_sec_sound_reserved1;
static int hf_ieee80211_ranging_sec_sound_i2r_rep;
static int hf_ieee80211_ranging_sec_sound_reserved2;
static int hf_ieee80211_ranging_sec_sound_starting_spatial_stream;
static int hf_ieee80211_ranging_sec_sound_number_spatial_streams;
static int hf_ieee80211_ranging_sec_sound_ul_target_rssi;
static int hf_ieee80211_ranging_sec_sound_reserved3;
static int hf_ieee80211_he_trigger_ranging_user_info_sac;
static int hf_ieee80211_he_ndp_annc_sta;
static int hf_ieee80211_he_ndp_annc_aid11;
static int hf_ieee80211_he_ndp_annc_ru_start;
static int hf_ieee80211_he_ndp_annc_ru_end;
static int hf_ieee80211_he_ndp_annc_feedback_type_and_ng;
static int hf_ieee80211_he_ndp_annc_disambiguation;
static int hf_ieee80211_he_ndp_annc_codebook_size;
static int hf_ieee80211_he_ndp_annc_nc;
static int hf_ieee80211_he_ndp_annc_disallowed_bitmap;
static int hf_ieee80211_he_ndp_annc_reserved1;
static int hf_ieee80211_he_ndp_annc_reserved2;
static int hf_ieee80211_he_qtp_control;
static int hf_ieee80211_he_qtp_setup_quiet_period_duration;
static int hf_ieee80211_he_qtp_setup_srv_specific_identif;
static int hf_ieee80211_he_qtp_request_dialog_token;
static int hf_ieee80211_he_qtp_request_quiet_period_offset;
static int hf_ieee80211_he_qtp_request_quiet_period_duration;
static int hf_ieee80211_he_qtp_request_quiet_period_interval;
static int hf_ieee80211_he_qtp_request_repetition_count;
static int hf_ieee80211_he_qtp_request_srv_specific_identif;
static int hf_ieee80211_he_qtp_response_dialog_token;
static int hf_ieee80211_he_qtp_response_status_code;
static int hf_ieee80211_he_qtp_response_quiet_period_offset;
static int hf_ieee80211_he_qtp_response_quiet_period_duration;
static int hf_ieee80211_he_qtp_response_quiet_period_interval;
static int hf_ieee80211_he_qtp_response_repetition_count;
static int hf_ieee80211_he_qtp_response_srv_specific_identif;
static int hf_ieee80211_htc_ht_lac;
static int hf_ieee80211_htc_lac_trq;
static int hf_ieee80211_htc_lac_mai_aseli;
static int hf_ieee80211_htc_lac_mai_mrq;
static int hf_ieee80211_htc_lac_mai_msi;
static int hf_ieee80211_htc_lac_mai_reserved;
static int hf_ieee80211_htc_lac_mfsi;
static int hf_ieee80211_htc_lac_mfb;
static int hf_ieee80211_htc_lac_asel_command;
static int hf_ieee80211_htc_lac_asel_data;
static int hf_ieee80211_htc_cal_pos;
static int hf_ieee80211_htc_cal_seq;
static int hf_ieee80211_htc_reserved1;
static int hf_ieee80211_htc_csi_steering;
static int hf_ieee80211_htc_ndp_announcement;
static int hf_ieee80211_htc_reserved2;
static int hf_ieee80211_htc_mrq;
static int hf_ieee80211_htc_msi;
static int hf_ieee80211_htc_msi_stbc_reserved;
static int hf_ieee80211_htc_compressed_msi;
static int hf_ieee80211_htc_ppdu_stbc_encoded;
static int hf_ieee80211_htc_mfsi;
static int hf_ieee80211_htc_gid_l;
static int hf_ieee80211_htc_mfb;
static int hf_ieee80211_htc_num_sts;
static int hf_ieee80211_htc_vht_mcs;
static int hf_ieee80211_htc_bw;
static int hf_ieee80211_htc_s1g_num_sts;
static int hf_ieee80211_htc_s1g_vht_mcs;
static int hf_ieee80211_htc_s1g_bw;
static int hf_ieee80211_htc_snr;
static int hf_ieee80211_htc_reserved3;
static int hf_ieee80211_htc_gid_h;
static int hf_ieee80211_htc_coding_type;
static int hf_ieee80211_htc_fb_tx_type;
static int hf_ieee80211_htc_unsolicited_mfb;
static int hf_ieee80211_htc_ac_constraint;
static int hf_ieee80211_htc_rdg_more_ppdu;

/* ************************************************************************* */
/*                Header values for sequence number field                    */
/* ************************************************************************* */
static int hf_ieee80211_frag_number;
static int hf_ieee80211_seq_number;

/* ************************************************************************* */
/*                   Header values for Frame Check field                     */
/* ************************************************************************* */
static int hf_ieee80211_fcs;
static int hf_ieee80211_fcs_status;

/* ************************************************************************* */
/*                   Header values for reassembly                            */
/* ************************************************************************* */
static int hf_ieee80211_fragments;
static int hf_ieee80211_fragment;
static int hf_ieee80211_fragment_overlap;
static int hf_ieee80211_fragment_overlap_conflict;
static int hf_ieee80211_fragment_multiple_tails;
static int hf_ieee80211_fragment_too_long_fragment;
static int hf_ieee80211_fragment_error;
static int hf_ieee80211_fragment_count;
static int hf_ieee80211_reassembled_in;
static int hf_ieee80211_reassembled_length;

static int proto_wlan_ext;

/* ************************************************************************* */
/*                      Fixed fields found in mgt frames                     */
/* ************************************************************************* */
static int hf_ieee80211_fixed_parameters;  /* Protocol payload for management frames */

static int hf_ieee80211_ff_auth_alg;            /* Authentication algorithm field            */
static int hf_ieee80211_ff_auth_seq;            /* Authentication transaction sequence       */
static int hf_ieee80211_ff_current_ap;          /* Current AP MAC address                    */
static int hf_ieee80211_ff_listen_ival;         /* Listen interval fixed field               */
static int hf_ieee80211_ff_timestamp;           /* 64 bit timestamp                          */
static int hf_ieee80211_ff_beacon_interval;     /* 16 bit Beacon interval                    */
static int hf_ieee80211_ff_assoc_id;            /* 16 bit AID field                          */
static int hf_ieee80211_ff_reason;              /* 16 bit reason code                        */
static int hf_ieee80211_ff_status_code;         /* Status code                               */
static int hf_ieee80211_ff_category_code;       /* 8 bit Category code */
static int hf_ieee80211_ff_action_code;         /* 8 bit Action code */
static int hf_ieee80211_ff_dialog_token;        /* 8 bit Dialog token */
static int hf_ieee80211_ff_trigger;
static int hf_ieee80211_ff_ftm_tod;
static int hf_ieee80211_ff_ftm_toa;
static int hf_ieee80211_ff_ftm_tod_err;
static int hf_ieee80211_ff_ftm_toa_err;
static int hf_ieee80211_ff_followup_dialog_token;
static int hf_ieee80211_ff_wme_action_code;     /* Management notification action code */
static int hf_ieee80211_ff_wme_status_code;     /* Management notification setup response status code */
static int hf_ieee80211_ff_qos_action_code;
static int hf_ieee80211_ff_dls_action_code;
static int hf_ieee80211_ff_dst_mac_addr;        /* DLS destination MAC address */
static int hf_ieee80211_ff_src_mac_addr;        /* DLS source MAC address */
static int hf_ieee80211_ff_req_ap_addr;
static int hf_ieee80211_ff_res_ap_addr;
static int hf_ieee80211_ff_check_beacon;
static int hf_ieee80211_ff_dls_timeout;         /* DLS timeout value */
static int hf_ieee80211_ff_ft_action_code; /* 8 bit FT Action code */
static int hf_ieee80211_ff_sta_address;
static int hf_ieee80211_ff_target_ap_address;
static int hf_ieee80211_ff_gas_comeback_delay;
static int hf_ieee80211_ff_gas_fragment_id;
static int hf_ieee80211_ff_more_gas_fragments;
static int hf_ieee80211_ff_query_request_length;
static int hf_ieee80211_ff_query_request;
static int hf_ieee80211_ff_query_response_length;
static int hf_ieee80211_ff_query_response;
static int hf_ieee80211_ff_anqp_info_id;
static int hf_ieee80211_ff_anqp_info_length;
static int hf_ieee80211_ff_anqp_info;
static int hf_ieee80211_ff_anqp_query_id;
static int hf_ieee80211_ff_anqp_capability;
static int hf_ieee80211_ff_anqp_capability_vlen;
static int hf_ieee80211_ff_anqp_capability_vendor;
static int hf_ieee80211_ff_venue_info_group;
static int hf_ieee80211_ff_venue_info_type;
static int hf_ieee80211_ff_anqp_venue_length;
static int hf_ieee80211_ff_anqp_venue_language;
static int hf_ieee80211_ff_anqp_venue_name;
static int hf_ieee80211_ff_anqp_nw_auth_type_indicator;
static int hf_ieee80211_ff_anqp_nw_auth_type_url_len;
static int hf_ieee80211_ff_anqp_nw_auth_type_url;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_indicator;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_url_len;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_url;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_year;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_mon;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_day;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_hr;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_min;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_sec;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_msec;
static int hf_ieee80211_ff_anqp_nw_auth_type_ts_rsvd;
static int hf_ieee80211_ff_anqp_roaming_consortium_oi_len;
static int hf_ieee80211_ff_anqp_roaming_consortium_oi;
static int hf_ieee80211_ff_anqp_ip_addr_avail_ipv6;
static int hf_ieee80211_ff_anqp_ip_addr_avail_ipv4;
static int hf_ieee80211_ff_anqp_nai_realm_count;
static int hf_ieee80211_ff_anqp_nai_field_len;
static int hf_ieee80211_ff_anqp_nai_realm_encoding;
static int hf_ieee80211_ff_anqp_nai_realm_length;
static int hf_ieee80211_ff_anqp_nai_realm;
static int hf_ieee80211_ff_anqp_nai_realm_eap_count;
static int hf_ieee80211_ff_anqp_nai_realm_eap_len;
static int hf_ieee80211_ff_anqp_nai_realm_eap_method;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_count;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_id;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_len;
static int hf_ieee80211_ff_anqp_nai_realm_auth_param_value;
static int hf_ieee80211_3gpp_gc_gud;
static int hf_ieee80211_3gpp_gc_udhl;
static int hf_ieee80211_3gpp_gc_iei;
static int hf_ieee80211_3gpp_gc_num_plmns;
static int hf_ieee80211_3gpp_gc_plmn;
static int hf_ieee80211_3gpp_gc_plmn_len;
static int hf_ieee80211_ff_anqp_domain_name_len;
static int hf_ieee80211_ff_anqp_domain_name;
static int hf_ieee80211_ff_tdls_action_code;
static int hf_ieee80211_ff_target_channel;
static int hf_ieee80211_ff_operating_class;
static int hf_ieee80211_ff_channel;
static int hf_ieee80211_ff_wnm_action_code;
static int hf_ieee80211_ff_unprotected_wnm_action_code;
static int hf_ieee80211_ff_key_data_length;
static int hf_ieee80211_ff_key_data;
static int hf_ieee80211_ff_wnm_notification_type;
static int hf_ieee80211_ff_wnm_notification_response_status;
static int hf_ieee80211_ff_rm_action_code;
static int hf_ieee80211_ff_rm_dialog_token;
static int hf_ieee80211_ff_rm_repetitions;
static int hf_ieee80211_ff_rm_tx_power;
static int hf_ieee80211_ff_rm_max_tx_power;
static int hf_ieee80211_ff_tpc;
static int hf_ieee80211_ff_tpc_element_id;
static int hf_ieee80211_ff_tpc_length;
static int hf_ieee80211_ff_tpc_tx_power;
static int hf_ieee80211_ff_tpc_link_margin;
static int hf_ieee80211_ff_rm_rx_antenna_id;
static int hf_ieee80211_ff_rm_tx_antenna_id;
static int hf_ieee80211_ff_rm_rcpi;
static int hf_ieee80211_ff_rm_rsni;
static int hf_ieee80211_ff_request_mode_pref_cand;
static int hf_ieee80211_ff_request_mode_abridged;
static int hf_ieee80211_ff_request_mode_disassoc_imminent;
static int hf_ieee80211_ff_request_mode_bss_term_included;
static int hf_ieee80211_ff_request_mode_ess_disassoc_imminent;
static int hf_ieee80211_ff_request_mode_link_removal_imminent;
static int hf_ieee80211_ff_request_mode_reserved;
static int hf_ieee80211_ff_disassoc_timer;
static int hf_ieee80211_ff_validity_interval;
static int hf_ieee80211_ff_url_len;
static int hf_ieee80211_ff_url;
static int hf_ieee80211_ff_target_bss;
static int hf_ieee80211_ff_bss_transition_query_reason;
static int hf_ieee80211_ff_bss_transition_status_code;
static int hf_ieee80211_ff_bss_termination_delay;
static int hf_ieee80211_ff_bss_transition_candidate_list_entries;

static int hf_ieee80211_ff_sa_query_action_code;
static int hf_ieee80211_ff_transaction_id;

static int hf_ieee80211_ff_send_confirm;
static int hf_ieee80211_ff_scalar;
static int hf_ieee80211_ff_finite_field_element;
static int hf_ieee80211_ff_confirm;
static int hf_ieee80211_ff_finite_cyclic_group;
static int hf_ieee80211_ff_sae_message_type;
static int hf_ieee80211_ff_sae_anti_clogging_token;


/* Vendor specific */
static int hf_ieee80211_ff_marvell_action_type;
static int hf_ieee80211_ff_marvell_mesh_mgt_action_code;
static int hf_ieee80211_ff_marvell_mesh_mgt_length;     /* Mesh Management length */
static int hf_ieee80211_ff_marvell_mesh_mgt_mode;       /* Mesh Management mode */
static int hf_ieee80211_ff_marvell_mesh_mgt_ttl;        /* Mesh Management TTL */
static int hf_ieee80211_ff_marvell_mesh_mgt_dstcount;   /* Mesh Management dst count */
static int hf_ieee80211_ff_marvell_mesh_mgt_hopcount;   /* Mesh Management hop count */
static int hf_ieee80211_ff_marvell_mesh_mgt_rreqid;     /* Mesh Management RREQ ID */
static int hf_ieee80211_ff_marvell_mesh_mgt_sa;         /* Mesh Management src addr */
static int hf_ieee80211_ff_marvell_mesh_mgt_ssn;        /* Mesh Management src sequence number */
static int hf_ieee80211_ff_marvell_mesh_mgt_metric;     /* Mesh Management metric */
static int hf_ieee80211_ff_marvell_mesh_mgt_flags;      /* Mesh Management RREQ flags */
static int hf_ieee80211_ff_marvell_mesh_mgt_da;         /* Mesh Management dst addr */
static int hf_ieee80211_ff_marvell_mesh_mgt_dsn;        /* Mesh Management dst sequence number */
static int hf_ieee80211_ff_marvell_mesh_mgt_lifetime;   /* Mesh Management lifetime */


static int hf_ieee80211_ff_ba_action;

static int hf_ieee80211_ff_block_ack_params;
static int hf_ieee80211_ff_block_ack_params_amsdu_permitted;
static int hf_ieee80211_ff_block_ack_params_policy;
static int hf_ieee80211_ff_block_ack_params_tid;
static int hf_ieee80211_ff_block_ack_params_buffer_size;

static int * const ieee80211_ff_block_ack_params_fields[] = {
  &hf_ieee80211_ff_block_ack_params_amsdu_permitted,
  &hf_ieee80211_ff_block_ack_params_policy,
  &hf_ieee80211_ff_block_ack_params_tid,
  &hf_ieee80211_ff_block_ack_params_buffer_size,
  NULL
};

static int hf_ieee80211_ff_block_ack_timeout;

static int hf_ieee80211_ff_block_ack_ssc;
static int hf_ieee80211_ff_block_ack_ssc_fragment;
static int hf_ieee80211_ff_block_ack_ssc_sequence;

static int * const ieee80211_ff_block_ack_ssc_fields[] = {
  &hf_ieee80211_ff_block_ack_ssc_fragment,
  &hf_ieee80211_ff_block_ack_ssc_sequence,
  NULL
};

static int hf_ieee80211_ff_delba_param;
static int hf_ieee80211_ff_delba_param_reserved;
static int hf_ieee80211_ff_delba_param_init;
static int hf_ieee80211_ff_delba_param_tid;

static int * const ieee80211_ff_delba_param_fields[] = {
  &hf_ieee80211_ff_delba_param_reserved,
  &hf_ieee80211_ff_delba_param_init,
  &hf_ieee80211_ff_delba_param_tid,
  NULL
};

static int hf_ieee80211_ff_max_reg_pwr;
static int hf_ieee80211_ff_measurement_pilot_int;
static int hf_ieee80211_ff_country_str;
static int hf_ieee80211_ff_max_tx_pwr;
static int hf_ieee80211_ff_tx_pwr_used;
static int hf_ieee80211_ff_transceiver_noise_floor;
static int hf_ieee80211_ff_channel_width;

static int hf_ieee80211_ff_qos_info_ap;
static int hf_ieee80211_ff_qos_info_ap_edca_param_set_counter;
static int hf_ieee80211_ff_qos_info_ap_q_ack;
static int hf_ieee80211_ff_qos_info_ap_queue_req;
static int hf_ieee80211_ff_qos_info_ap_txop_request;
static int hf_ieee80211_ff_qos_info_ap_more_data_ack;

static int * const ieee80211_ff_qos_info_ap_fields[] = {
  &hf_ieee80211_ff_qos_info_ap_edca_param_set_counter,
  &hf_ieee80211_ff_qos_info_ap_q_ack,
  &hf_ieee80211_ff_qos_info_ap_queue_req,
  &hf_ieee80211_ff_qos_info_ap_txop_request,
  &hf_ieee80211_ff_qos_info_ap_more_data_ack,
  NULL
};

static int hf_ieee80211_ff_qos_info_sta;
static int hf_ieee80211_ff_qos_info_sta_ac_vo;
static int hf_ieee80211_ff_qos_info_sta_ac_vi;
static int hf_ieee80211_ff_qos_info_sta_ac_bk;
static int hf_ieee80211_ff_qos_info_sta_ac_be;
static int hf_ieee80211_ff_qos_info_sta_q_ack;
static int hf_ieee80211_ff_qos_info_sta_max_sp_length;
static int hf_ieee80211_ff_qos_info_sta_more_data_ack;

static int * const ieee80211_ff_qos_info_sta_fields[] = {
  &hf_ieee80211_ff_qos_info_sta_ac_vo,
  &hf_ieee80211_ff_qos_info_sta_ac_vi,
  &hf_ieee80211_ff_qos_info_sta_ac_bk,
  &hf_ieee80211_ff_qos_info_sta_ac_be,
  &hf_ieee80211_ff_qos_info_sta_q_ack,
  &hf_ieee80211_ff_qos_info_sta_max_sp_length,
  &hf_ieee80211_ff_qos_info_sta_more_data_ack,
  NULL
};

static int hf_ieee80211_ff_sm_pwr_save;
static int hf_ieee80211_ff_sm_pwr_save_enabled;
static int hf_ieee80211_ff_sm_pwr_save_sm_mode;
static int hf_ieee80211_ff_sm_pwr_save_reserved;

static int * const ieee80211_ff_sw_pwr_save_fields[] = {
  &hf_ieee80211_ff_sm_pwr_save_enabled,
  &hf_ieee80211_ff_sm_pwr_save_sm_mode,
  &hf_ieee80211_ff_sm_pwr_save_reserved,
  NULL
};

static int hf_ieee80211_ff_pco_phase_cntrl;

static int hf_ieee80211_ff_psmp_param_set;
static int hf_ieee80211_ff_psmp_param_set_n_sta;
static int hf_ieee80211_ff_psmp_param_set_more_psmp;
static int hf_ieee80211_ff_psmp_param_set_psmp_sequence_duration;

static int * const ieee80211_ff_psmp_param_set_fields[] = {
  &hf_ieee80211_ff_psmp_param_set_n_sta,
  &hf_ieee80211_ff_psmp_param_set_more_psmp,
  &hf_ieee80211_ff_psmp_param_set_psmp_sequence_duration,
  NULL
};

static int hf_ieee80211_ff_mimo_cntrl;
static int hf_ieee80211_ff_mimo_cntrl_nc_index;
static int hf_ieee80211_ff_mimo_cntrl_nr_index;
static int hf_ieee80211_ff_mimo_cntrl_channel_width;
static int hf_ieee80211_ff_mimo_cntrl_grouping;
static int hf_ieee80211_ff_mimo_cntrl_coefficient_size;
static int hf_ieee80211_ff_mimo_cntrl_codebook_info;
static int hf_ieee80211_ff_mimo_cntrl_remaining_matrix_segment;
static int hf_ieee80211_ff_mimo_cntrl_reserved;
static int hf_ieee80211_ff_mimo_cntrl_sounding_timestamp;

static int hf_ieee80211_ff_ftm_param_delim1;
static int hf_ieee80211_ff_ftm_param_status_indication;
static int hf_ieee80211_ff_ftm_param_value;
static int hf_ieee80211_ff_ftm_param_reserved1;
static int hf_ieee80211_ff_ftm_param_burst_exponent;
static int hf_ieee80211_ff_ftm_param_burst_duration;

static int hf_ieee80211_ff_ftm_param_delim2;
static int hf_ieee80211_ff_ftm_param_min_delta_ftm;
static int hf_ieee80211_ff_ftm_param_partial_tsf_timer;
static int hf_ieee80211_ff_ftm_param_partial_tsf_no_pref;
static int hf_ieee80211_ff_ftm_param_asap_capable;
static int hf_ieee80211_ff_ftm_param_asap;
static int hf_ieee80211_ff_ftm_param_ftm_per_burst;

static int hf_ieee80211_ff_ftm_param_delim3;
static int hf_ieee80211_ff_ftm_param_reserved2;
static int hf_ieee80211_ff_ftm_param_format_and_bw;
static int hf_ieee80211_ff_ftm_param_burst_period;

/* az D3.0 introduces a 1-octet TOD Error field; use a different name to avoid
 * conflicting with the existing hf_ieee80211_ff_ftm_tod_err (which is 2
 * octets).
 */
static int hf_ieee80211_ff_ftm_tod_err1;
static int hf_ieee80211_ff_ftm_max_tod_error_exponent;
static int hf_ieee80211_ff_ftm_tod_err_reserved;
static int hf_ieee80211_ff_ftm_tod_not_continuous;

/* Same situation with ...toa_err1 as ...tod_err1 */
static int hf_ieee80211_ff_ftm_toa_err1;
static int hf_ieee80211_ff_ftm_max_toa_error_exponent;
static int hf_ieee80211_ff_ftm_toa_err_reserved;
static int hf_ieee80211_ff_ftm_invalid_measurement;
static int hf_ieee80211_ff_ftm_toa_type;

static int hf_ieee80211_ff_ftm_cfo;
static int hf_ieee80211_ff_ftm_r2i_ndp_tx_power;
static int hf_ieee80211_ff_ftm_i2r_ndp_target_rssi;

/* az: FTM Ranging Parameters Element */
static int hf_ieee80211_tag_ranging_parameters;
static int hf_ieee80211_tag_ranging_subelt_tag;
static int hf_ieee80211_tag_ranging_subelt_len;
static int hf_ieee80211_tag_ranging_status_indication;
static int hf_ieee80211_tag_ranging_value;
static int hf_ieee80211_tag_ranging_i2r_lmr_feedback;
static int hf_ieee80211_tag_ranging_secure_ltf_required;
static int hf_ieee80211_tag_ranging_secure_ltf_support;
static int hf_ieee80211_tag_ranging_ranging_priority;
static int hf_ieee80211_tag_ranging_r2i_toa_type;
static int hf_ieee80211_tag_ranging_i2r_toa_type;
static int hf_ieee80211_tag_ranging_r2i_aoa_requested;
static int hf_ieee80211_tag_ranging_i2r_aoa_requested;
static int hf_ieee80211_tag_ranging_format_and_bandwidth;
static int hf_ieee80211_tag_ranging_immediate_r2i_feedback;
static int hf_ieee80211_tag_ranging_immediate_i2r_feedback;
static int hf_ieee80211_tag_ranging_max_i2r_repetition;
static int hf_ieee80211_tag_ranging_max_r2i_repetition;
static int hf_ieee80211_tag_ranging_reserved1;
static int hf_ieee80211_tag_ranging_reserved2;
static int hf_ieee80211_tag_ranging_max_r2i_sts_le_80_mhz;
static int hf_ieee80211_tag_ranging_max_r2i_sts_gt_80_mhz;
static int hf_ieee80211_tag_ranging_max_r2i_ltf_total;
static int hf_ieee80211_tag_ranging_max_i2r_ltf_total;
static int hf_ieee80211_tag_ranging_max_i2r_sts_le_80_mhz;
static int hf_ieee80211_tag_ranging_max_i2r_sts_gt_80_mhz;
static int hf_ieee80211_tag_ranging_bss_color_info;

/* az: FTM Ranging Parameters NTB-specific subelement */
static int hf_ieee80211_tag_ranging_ntb;
static int hf_ieee80211_tag_ranging_ntb_reserved1;
static int hf_ieee80211_tag_ranging_ntb_min_time_msmts;
static int hf_ieee80211_tag_ranging_ntb_max_time_msmts;
static int hf_ieee80211_tag_ranging_ntb_r2i_tx_power;
static int hf_ieee80211_tag_ranging_ntb_i2r_tx_power;
static int hf_ieee80211_tag_ranging_ntb_reserved2;

/* az: FTM Ranging Specific TB subelement */
static int hf_ieee80211_tag_ranging_aid_rsid;
static int hf_ieee80211_tag_ranging_device_class;
static int hf_ieee80211_tag_ranging_full_bw_ul_mu_mimo;
static int hf_ieee80211_tag_ranging_trigger_frame_paddur;
static int hf_ieee80211_tag_ranging_max_sess_exp;
static int hf_ieee80211_tag_ranging_passive_tb_ranging;
static int hf_ieee80211_tag_ranging_tb_specific_reserved;

/* az: FTM Ranging Secure HE-LTF subelement */
static int hf_ieee80211_tag_ranging_secure_he_ltf;
static int hf_ieee80211_tag_ranging_secure_he_ltf_version;
static int hf_ieee80211_tag_ranging_secure_he_ltf_req;
static int hf_ieee80211_tag_ranging_secure_he_ltf_r2i_tx_window;
static int hf_ieee80211_tag_ranging_secure_he_ltf_i2r_tx_window;
static int hf_ieee80211_tag_ranging_secure_he_ltf_reserved;

/* az: PASN subelements etc. */
static int hf_ieee80211_tag_pasn_parameters_control;
static int hf_ieee80211_tag_pasn_params_comeback_info_present;
static int hf_ieee80211_tag_pasn_params_group_and_key_present;
static int hf_ieee80211_tag_pasn_parameters_reserved;
static int hf_ieee80211_tag_pasn_parameters_wrapped_fmt;
static int hf_ieee80211_tag_pasn_comeback_after;
static int hf_ieee80211_tag_pasn_cookie_length;
static int hf_ieee80211_tag_pasn_cookie;
static int hf_ieee80211_tag_pasn_finite_cyclic_group_id;
static int hf_ieee80211_tag_pasn_ephemeral_public_key_len;
static int hf_ieee80211_tag_pasn_ephemeral_public_key;
static int hf_ieee80211_pasn_auth1_frame_len;
static int hf_ieee80211_pasn_auth2_frame_len;

/* az: Secure LTF Parameters */
static int hf_ieee80211_tag_secure_ltf_params_counter;
static int hf_ieee80211_tag_secure_ltf_generation_sac;
static int hf_ieee80211_tag_secure_ltf_management_sac;
static int hf_ieee80211_tag_secure_ltf_result_ltf_ofs;

/* az: ista and rsta availability details */
static int hf_ieee80211_ftm_ista_availability_count;
static int hf_ieee80211_ftm_ista_availability_reserved;
static int hf_ieee80211_ftm_ista_avail_bits;
static int hf_ieee80211_ftm_ista_avail_pad;

static int hf_ieee80211_ftm_rsta_header;
static int hf_ieee80211_ftm_rsta_count;
static int hf_ieee80211_ftm_rsta_avail_window_bcast_fmt;
static int hf_ieee80211_ftm_rsta_partial_tsf_timer1;
static int hf_ieee80211_ftm_rsta_duration1;
static int hf_ieee80211_ftm_rsta_passive_tb_ranging_reserved1;
static int hf_ieee80211_ftm_rsta_periodicity1;
static int hf_ieee80211_ftm_rsta_partial_tsf_timer;
static int hf_ieee80211_ftm_rsta_duration;
static int hf_ieee80211_ftm_rsta_passive_tb_ranging_reserved;
static int hf_ieee80211_ftm_rsta_periodicity;
static int hf_ieee80211_ftm_rsta_format_and_bandwidth;
static int hf_ieee80211_ftm_rsta_reserved;
static int hf_ieee80211_ftm_rsta_avail_subfield_short;
static int hf_ieee80211_ftm_rsta_avail_subfield_long;

/* be: Multi-link elements and other fields */
static int hf_ieee80211_eht_multi_link_control;
static int hf_ieee80211_eht_multi_link_control_type;
static int hf_ieee80211_eht_multi_link_control_reserved;
static int hf_ieee80211_eht_multi_link_control_link_id_present;
static int hf_ieee80211_eht_multi_link_control_bss_parms_ch_count;
static int hf_ieee80211_eht_multi_link_control_medium_sync_delay;
static int hf_ieee80211_eht_multi_link_control_eml_capa;
static int hf_ieee80211_eht_multi_link_control_mld_capa;
static int hf_ieee80211_eht_multi_link_control_basic_mld_id_present;
static int hf_ieee80211_eht_multi_link_control_ext_mld_capa;
static int hf_ieee80211_eht_multi_link_control_bitmap_reserved;
static int hf_ieee80211_eht_multi_link_control_probe_mld_id_present;
static int hf_ieee80211_eht_multi_link_control_probe_mld_mac_addr_present;
static int hf_ieee80211_eht_multi_link_control_probe_reserved;
static int hf_ieee80211_eht_multi_link_control_reconfig_mld_mac;
static int hf_ieee80211_eht_multi_link_control_reconfig_eml_capa;
static int hf_ieee80211_eht_multi_link_control_reconfig_mld_capa_oper;
static int hf_ieee80211_eht_multi_link_control_reconfig_ext_mld_capa_oper;
static int hf_ieee80211_eht_multi_link_control_reconfig_reserved;
static int hf_ieee80211_eht_multi_link_control_tdls_reserved;
static int hf_ieee80211_eht_multi_link_control_prio_access_reserved;
static int hf_ieee80211_eht_common_field_length;
static int hf_ieee80211_eht_common_field_mld_mac;
static int hf_ieee80211_eht_common_field_link_id_field;
static int hf_ieee80211_eht_common_info_link_id;
static int hf_ieee80211_eht_common_info_link_id_reserved;
static int hf_ieee80211_eht_common_field_bss_param_change_count;
static int hf_ieee80211_eht_common_field_medium_sync_field;
static int hf_ieee80211_eht_common_info_medium_sync_duration;
static int hf_ieee80211_eht_common_info_medium_sync_threshold;
static int hf_ieee80211_eht_common_info_medium_sync_max_txops;
static int hf_ieee80211_eht_common_field_eml_capabilities;
static int hf_ieee80211_eht_common_info_eml_capa_emlsr_support;
static int hf_ieee80211_eht_common_info_eml_capa_emlsr_emlmr_padding_delay;
static int hf_ieee80211_eht_common_info_eml_capa_emlsr_emlmr_transition_delay;
static int hf_ieee80211_eht_common_info_eml_capa_emlmr_support;
static int hf_ieee80211_eht_common_info_eml_capa_reserved1;
static int hf_ieee80211_eht_common_info_eml_capa_transition_timeout;
static int hf_ieee80211_eht_common_info_eml_capa_reserved2;
static int hf_ieee80211_eht_common_field_mld_capabilities;
static int hf_ieee80211_eht_common_info_mld_max_simul_links;
static int hf_ieee80211_eht_common_info_mld_srs_support;
static int hf_ieee80211_eht_common_info_mld_tid_to_link_map_neg;
static int hf_ieee80211_eht_common_info_mld_freq_sep_for_str;
static int hf_ieee80211_eht_common_info_mld_aar_support;
static int hf_ieee80211_eht_common_info_mld_link_reconf_op_support;
static int hf_ieee80211_eht_common_info_mld_aligned_twt_support;
static int hf_ieee80211_eht_common_info_mld_reserved;
static int hf_ieee80211_eht_common_field_mld_id;
static int hf_ieee80211_eht_common_field_ap_mld_mac;
static int hf_ieee80211_eht_common_field_ext_mld_capabilities;
static int hf_ieee80211_eht_common_info_ext_mld_op_update_support;
static int hf_ieee80211_eht_common_info_ext_mld_max_simul_links;
static int hf_ieee80211_eht_common_info_ext_mld_nstr_status_support;
static int hf_ieee80211_eht_common_info_ext_mld_emlsr_enable_one_link_support;
static int hf_ieee80211_eht_common_info_ext_mld_btm_mld_recom_aps_support;
static int hf_ieee80211_eht_common_info_ext_mld_reserved;
static int hf_ieee80211_eht_multi_link_subelt_tag;
static int hf_ieee80211_eht_multi_link_subelt_len;
static int hf_ieee80211_eht_multi_link_type_0_link_count;
static int hf_ieee80211_eht_multi_link_type_1_link_count;
static int hf_ieee80211_eht_multi_link_type_2_link_count;
static int hf_ieee80211_eht_multi_link_type_3_link_count;
static int hf_ieee80211_eht_multi_link_type_4_link_count;
static int hf_ieee80211_eht_multi_link_link_id_list;
static int hf_ieee80211_eht_profile_sta_control;
static int hf_ieee80211_eht_profile_link_id;
static int hf_ieee80211_eht_profile_complete_profile;
static int hf_ieee80211_eht_profile_mac_address_present;
static int hf_ieee80211_eht_profile_beacon_interval_present;
static int hf_ieee80211_eht_profile_tsf_offset_present;
static int hf_ieee80211_eht_profile_dtim_info_present;
static int hf_ieee80211_eht_profile_nstr_link_pair_present;
static int hf_ieee80211_eht_profile_nstr_bitmap_size;
static int hf_ieee80211_eht_profile_bss_params_change_count_present;
static int hf_ieee80211_eht_profile_reserved;
static int hf_ieee80211_eht_profile_probe_reserved;
static int hf_ieee80211_eht_profile_removal_timer_present;
static int hf_ieee80211_eht_profile_reconfig_operation_type;
static int hf_ieee80211_eht_profile_operation_para_present;
static int hf_ieee80211_eht_profile_reconfig_nstr_bitmap_size;
static int hf_ieee80211_eht_profile_reconfig_nstr_bitmap_present;
static int hf_ieee80211_eht_profile_reconfig_reserved;
static int hf_ieee80211_eht_profile_prio_acc_reserved;
static int hf_ieee80211_eht_sta_profile_info_len;
static int hf_ieee80211_eht_sta_profile_info_mac;
static int hf_ieee80211_eht_sta_profile_info_beacon;
static int hf_ieee80211_eht_sta_profile_info_tsf_offset;
static int hf_ieee80211_eht_sta_profile_info_dtim_count;
static int hf_ieee80211_eht_sta_profile_info_dtim_period;
static int hf_ieee80211_eht_sta_profile_info_bitmap;
static int hf_ieee80211_eht_sta_profile_bss_params_change_count;
static int hf_ieee80211_eht_sta_profile_removal_timer;
static int hf_ieee80211_eht_sta_profile_presence_indi;
static int hf_ieee80211_eht_sta_profile_presence_indi_max_mpdu_length_present;
static int hf_ieee80211_eht_sta_profile_presence_indi_max_amsdu_length_present;
static int hf_ieee80211_eht_sta_profile_presence_indi_reserved;
static int hf_ieee80211_eht_sta_profile_operation_para_info;
static int hf_ieee80211_eht_sta_profile_operation_para_info_max_mpdu_length;
static int hf_ieee80211_eht_sta_profile_operation_para_info_amsdu_length;
static int hf_ieee80211_eht_sta_profile_operation_para_info_pad;
static int hf_ieee80211_eht_operation_parameters;
static int hf_ieee80211_eht_basic_eht_mcs_nss_set;
static int hf_ieee80211_eht_operation_control_chan_width;
static int hf_ieee80211_eht_operation_control_reserved;
static int hf_ieee80211_eht_operation_info_present;
static int hf_ieee80211_eht_operation_subchannel_bitmap_present;
static int hf_ieee80211_eht_operation_default_pe_duration;
static int hf_ieee80211_eht_operation_group_addressed_bu_indication_limit;
static int hf_ieee80211_eht_operation_group_addressed_bu_indication_exp;
static int hf_ieee80211_eht_operation_mcs15_disable;
static int hf_ieee80211_eht_operation_reserved;
static int hf_ieee80211_eht_operation_control;
static int hf_ieee80211_eht_operation_ccfs0;
static int hf_ieee80211_eht_operation_ccfs1;
static int hf_ieee80211_eht_operation_disabled_bitmap;
static int hf_ieee80211_eht_mac_capabilities;
static int hf_ieee80211_eht_mac_capa_epcs_prio_access_support;
static int hf_ieee80211_eht_mac_capa_eht_om_control_support;
static int hf_ieee80211_eht_mac_capa_trig_txop_sharing_mode_1_support;
static int hf_ieee80211_eht_mac_capa_trig_txop_sharing_mode_2_support;
static int hf_ieee80211_eht_mac_capa_restricted_twt_support;
static int hf_ieee80211_eht_mac_capa_scs_traffic_description_support;
static int hf_ieee80211_eht_mac_capa_maximum_mpdu_length;
static int hf_ieee80211_eht_mac_capa_maximum_ampdu_length_exp_ext;
static int hf_ieee80211_eht_mac_capa_eht_trs_support;
static int hf_ieee80211_eht_mac_capa_txop_return_support_txop_sha_mode;
static int hf_ieee80211_eht_mac_capa_two_bqrs_support;
static int hf_ieee80211_eht_mac_capa_eht_link_adaptation_support;
static int hf_ieee80211_eht_mac_capa_unsolicited_epcs_update;
static int hf_ieee80211_eht_mac_capa_reserved;
static int hf_ieee80211_eht_phy_bits_0_15;
static int hf_ieee80211_eht_phy_bits_0_15_reserved;
static int hf_ieee80211_eht_phy_bits_0_15_320_mhz_in_6ghz;
static int hf_ieee80211_eht_phy_bits_0_15_242_tone_ru_bw_wider_20mhz;
static int hf_ieee80211_eht_phy_bits_0_15_ndp_and_3_2_us_gi;
static int hf_ieee80211_eht_phy_bits_0_15_partial_bw_ul_mu_mimo;
static int hf_ieee80211_eht_phy_bits_0_15_su_beamformer;
static int hf_ieee80211_eht_phy_bits_0_15_su_beamformee;
static int hf_ieee80211_eht_phy_bits_0_15_su_beamformee_le_80mhz;
static int hf_ieee80211_eht_phy_bits_0_15_su_beamformee_eq_160mhz;
static int hf_ieee80211_eht_phy_bits_0_15_su_beamformee_eq_320mhz;
static int hf_ieee80211_eht_phy_bits_16_31;
static int hf_ieee80211_eht_phy_bits_16_31_num_sounding_dims_lt_80mhz;
static int hf_ieee80211_eht_phy_bits_16_31_num_sounding_dims_eq_160mhz;
static int hf_ieee80211_eht_phy_bits_16_31_num_sounding_dims_eq_320mhz;
static int hf_ieee80211_eht_phy_bits_16_31_num_ng_eq_16_su_feedback;
static int hf_ieee80211_eht_phy_bits_16_31_num_ng_eq_16_mu_feedback;
static int hf_ieee80211_eht_phy_bits_16_31_codebook_size_4_2_su_fbck;
static int hf_ieee80211_eht_phy_bits_16_31_codebook_size_7_5_mu_fbck;
static int hf_ieee80211_eht_phy_bits_16_31_triggered_su_beemform_fbck;
static int hf_ieee80211_eht_phy_bits_16_31_triggered_mu_beemform_p_bw_fbck;
static int hf_ieee80211_eht_phy_bits_16_31_triggered_cqi_feedback;
static int hf_ieee80211_eht_phy_bits_32_39;
static int hf_ieee80211_eht_phy_bits_32_39_partial_bw_dl_mu_mimo;
static int hf_ieee80211_eht_phy_bits_32_39_eht_psr_based_sr_support;
static int hf_ieee80211_eht_phy_bits_32_39_power_boost_factor_support;
static int hf_ieee80211_eht_phy_bits_32_39_eht_mu_ppdu_w_4x_eht_ltf_08_gi;
static int hf_ieee80211_eht_phy_bits_32_39_max_nc;
static int hf_ieee80211_eht_phy_bits_40_63;
static int hf_ieee80211_eht_phy_bits_40_63_non_triggered_cqi_fbck;
static int hf_ieee80211_eht_phy_bits_40_63_tx_1024_4096_qam_lt_242_ru_support;
static int hf_ieee80211_eht_phy_bits_40_63_rx_1024_4096_qam_lt_242_ru_support;
static int hf_ieee80211_eht_phy_bits_40_63_ppe_thresholds_present;
static int hf_ieee80211_eht_phy_bits_40_63_common_nominal_packet_padding;
static int hf_ieee80211_eht_phy_bits_40_63_max_num_supported_eht_ltfs;
static int hf_ieee80211_eht_phy_bits_40_63_support_of_mcx_15;
static int hf_ieee80211_eht_phy_bits_40_63_support_of_eht_dup_in_6_ghz;
static int hf_ieee80211_eht_phy_bits_40_63_support_20_mhz_sta_ndp_wide_bw;
static int hf_ieee80211_eht_phy_bits_40_63_non_ofdma_ul_mu_bw_le_80_mhz;
static int hf_ieee80211_eht_phy_bits_40_63_non_ofdma_ul_mu_bw_eq_160_mhz;
static int hf_ieee80211_eht_phy_bits_40_63_non_ofdma_ul_mu_bw_eq_320_mhz;
static int hf_ieee80211_eht_phy_bits_40_63_mu_beamformer_bw_le_80_mhz;
static int hf_ieee80211_eht_phy_bits_40_63_mu_beamformer_bw_eq_160_mhz;
static int hf_ieee80211_eht_phy_bits_40_63_mu_beamformer_bw_eq_320_mhz;
static int hf_ieee80211_eht_phy_bits_40_63_tb_sounding_feedback_rate_limit;
static int hf_ieee80211_eht_phy_bits_64_71;
static int hf_ieee80211_eht_phy_bits_64_71_rx_1024_qam_wid_bw_dl_ofdma_sup;
static int hf_ieee80211_eht_phy_bits_64_71_rx_4096_qam_wid_bw_dl_ofdma_sup;
static int hf_ieee80211_eht_phy_bits_64_71_20m_limit_capa_support;
static int hf_ieee80211_eht_phy_bits_64_71_20m_mu_beam_feedback_dl_mu_mimo;
static int hf_ieee80211_eht_phy_bits_64_71_20m_mru_support;
static int hf_ieee80211_eht_phy_bits_64_71_reserved;
static int hf_ieee80211_eht_supported_mcs_nss_bytes;
static int hf_ieee80211_eht_mcs_and_nss_non_ap;
static int hf_ieee80211_eht_rx_max_nss_20mhz_0_7;
static int hf_ieee80211_eht_tx_max_nss_20mhz_0_7;
static int hf_ieee80211_eht_rx_max_nss_20mhz_8_9;
static int hf_ieee80211_eht_tx_max_nss_20mhz_8_9;
static int hf_ieee80211_eht_rx_max_nss_20mhz_10_11;
static int hf_ieee80211_eht_tx_max_nss_20mhz_10_11;
static int hf_ieee80211_eht_rx_max_nss_20mhz_12_13;
static int hf_ieee80211_eht_tx_max_nss_20mhz_12_13;
static int hf_ieee80211_eht_le_80_rx_max_nss_0_9;
static int hf_ieee80211_eht_le_80_tx_max_nss_0_9;
static int hf_ieee80211_eht_le_80_rx_max_nss_10_11;
static int hf_ieee80211_eht_le_80_tx_max_nss_10_11;
static int hf_ieee80211_eht_le_80_rx_max_nss_12_13;
static int hf_ieee80211_eht_le_80_tx_max_nss_12_13;
static int hf_ieee80211_eht_160_rx_max_nss_0_9;
static int hf_ieee80211_eht_160_tx_max_nss_0_9;
static int hf_ieee80211_eht_160_rx_max_nss_10_11;
static int hf_ieee80211_eht_160_tx_max_nss_10_11;
static int hf_ieee80211_eht_160_rx_max_nss_12_13;
static int hf_ieee80211_eht_160_tx_max_nss_12_13;
static int hf_ieee80211_eht_320_rx_max_nss_0_9;
static int hf_ieee80211_eht_320_tx_max_nss_0_9;
static int hf_ieee80211_eht_320_rx_max_nss_10_11;
static int hf_ieee80211_eht_320_tx_max_nss_10_11;
static int hf_ieee80211_eht_320_rx_max_nss_12_13;
static int hf_ieee80211_eht_320_tx_max_nss_12_13;
static int hf_ieee80211_eht_mcs_and_nss_le_80mhz;
static int hf_ieee80211_eht_mcs_and_nss_eq_160mhz;
static int hf_ieee80211_eht_mcs_and_nss_eq_320mhz;
static int hf_ieee80211_eht_ppe_thresholds;
static int hf_ieee80211_eht_ttl_mapping_control;
static int hf_ieee80211_eht_ttl_mapping_direction;
static int hf_ieee80211_eht_ttl_default_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_switch_time_pres;
static int hf_ieee80211_eht_ttl_expected_dura_pres;
static int hf_ieee80211_eht_ttl_link_mapping_size;
static int hf_ieee80211_eht_ttl_mapping_reserved;
static int hf_ieee80211_eht_ttl_mapping_presence;
static int hf_ieee80211_eht_ttl_mapping_switch_time;
static int hf_ieee80211_eht_ttl_mapping_expected_duration;
static int hf_ieee80211_eht_ttl_mapping_tid_0_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_tid_1_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_tid_2_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_tid_3_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_tid_4_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_tid_5_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_tid_6_link_mapping;
static int hf_ieee80211_eht_ttl_mapping_tid_7_link_mapping;
static int hf_ieee80211_eht_multi_link_traffic_control;
static int hf_ieee80211_eht_multi_link_tc_bitmap_size;
static int hf_ieee80211_eht_multi_link_tc_aid_offset;
static int hf_ieee80211_eht_multi_link_tc_reserved;
static int hf_ieee80211_eht_multi_link_traffic_indication;
static int hf_ieee80211_eht_qos_chars_dirn;
static int hf_ieee80211_eht_qos_chars_tid;
static int hf_ieee80211_eht_qos_chars_user_prio;
static int hf_ieee80211_eht_qos_chars_bitmap;
static int hf_ieee80211_eht_qos_chars_linkid;
static int hf_ieee80211_eht_qos_chars_resrvd;
static int hf_ieee80211_eht_qos_chars_min_svc_interval;
static int hf_ieee80211_eht_qos_chars_max_svc_interval;
static int hf_ieee80211_eht_qos_chars_min_data_rate;
static int hf_ieee80211_eht_qos_chars_delay_bound;
static int hf_ieee80211_eht_qos_chars_max_msdu_size;
static int hf_ieee80211_eht_qos_chars_service_start_time;
static int hf_ieee80211_eht_qos_chars_service_start_time_linkid;
static int hf_ieee80211_eht_qos_chars_mean_data_rate;
static int hf_ieee80211_eht_qos_chars_burst_size;
static int hf_ieee80211_eht_qos_chars_msdu_lifetime;
static int hf_ieee80211_eht_qos_chars_msdu_delivery_ratio;
static int hf_ieee80211_eht_qos_chars_msdu_count_exponent;
static int hf_ieee80211_eht_qos_chars_medium_time;
static int hf_ieee80211_eht_link_id_bitmap;
static int hf_ieee80211_eht_aid_bitmap_length;
static int hf_ieee80211_eht_aid_bitmap_control;
static int hf_ieee80211_eht_aid_bitmap_control_reserved;
static int hf_ieee80211_eht_aid_bitmap_control_offset;
static int hf_ieee80211_eht_aid_bitmap_partial_aid_bitmap;
static int hf_ieee80211_eht_aid_bitmap_aid;
static int hf_ieee80211_eht_bw_indi_param;
static int hf_ieee80211_eht_bw_indi_param_reserved;
static int hf_ieee80211_eht_bw_indi_param_disabled_subchan_bitmap;
static int hf_ieee80211_eht_bw_indi_param_reserved1;
static int hf_ieee80211_eht_bw_indi_diabled_bitmap;
static int hf_ieee80211_eht_mimo_ctrl_field;
static int hf_ieee80211_eht_mimo_ctrl_nc_index;
static int hf_ieee80211_eht_mimo_ctrl_nr_index;
static int hf_ieee80211_eht_mimo_ctrl_bw;
static int hf_ieee80211_eht_mimo_ctrl_grouping;
static int hf_ieee80211_eht_mimo_ctrl_feedback_type;
static int hf_ieee80211_eht_mimo_ctrl_reserved1;
static int hf_ieee80211_eht_mimo_ctrl_remaining_feedback_segments;
static int hf_ieee80211_eht_mimo_ctrl_first_feedback_segment;
static int hf_ieee80211_eht_mimo_ctrl_partial_bw_info;
static int hf_ieee80211_eht_mimo_ctrl_sounding_dialog_token_number;
static int hf_ieee80211_eht_mimo_ctrl_codebook_info;
static int hf_ieee80211_eht_mimo_ctrl_reserved2;

/* be: Compressed beamforming report etc */
static int hf_ieee80211_eht_compressed_beamforming_report_snr;
static int hf_ieee80211_eht_compressed_beamform_scidx;

/* be: MU Exclusive beamforming report */
static int hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr;

static int hf_ieee80211_eht_eml_control_field;
static int hf_ieee80211_eht_eml_control_emlsr_mode;
static int hf_ieee80211_eht_eml_control_emlmr_mode;
static int hf_ieee80211_eht_eml_control_emlsr_para_update_control;
static int hf_ieee80211_eht_eml_control_device_coexist_activities;
static int hf_ieee80211_eht_eml_control_reserved;
static int hf_ieee80211_eht_eml_control_link_bitmap;
static int hf_ieee80211_eht_eml_control_link_enable_id;
static int hf_ieee80211_eht_eml_control_mcs_map_count;
static int hf_ieee80211_eht_eml_control_mcs_map_count_bw;
static int hf_ieee80211_eht_eml_control_mcs_map_count_reserved;
static int hf_ieee80211_eht_emlsr_para_update;
static int hf_ieee80211_eht_emlsr_para_update_padding_delay;
static int hf_ieee80211_eht_emlsr_para_update_tran_delay;
static int hf_ieee80211_eht_emlsr_para_update_reserved;

static int hf_ieee80211_eht_group_key_data_length;
static int hf_ieee80211_eht_reconfig_link_id_info;
static int hf_ieee80211_eht_reconfig_link_id;
static int hf_ieee80211_eht_reconfig_link_id_reserved;
static int hf_ieee80211_eht_reconfig_status_code;

static int hf_ieee80211_ff_ant_selection;
static int hf_ieee80211_ff_ant_selection_0;
static int hf_ieee80211_ff_ant_selection_1;
static int hf_ieee80211_ff_ant_selection_2;
static int hf_ieee80211_ff_ant_selection_3;
static int hf_ieee80211_ff_ant_selection_4;
static int hf_ieee80211_ff_ant_selection_5;
static int hf_ieee80211_ff_ant_selection_6;
static int hf_ieee80211_ff_ant_selection_7;

static int * const ieee80211_ff_ant_selection_fields[] = {
  &hf_ieee80211_ff_ant_selection_0,
  &hf_ieee80211_ff_ant_selection_1,
  &hf_ieee80211_ff_ant_selection_2,
  &hf_ieee80211_ff_ant_selection_3,
  &hf_ieee80211_ff_ant_selection_4,
  &hf_ieee80211_ff_ant_selection_5,
  &hf_ieee80211_ff_ant_selection_6,
  &hf_ieee80211_ff_ant_selection_7,
  NULL
};

static int hf_ieee80211_ff_ext_channel_switch_announcement;
static int hf_ieee80211_ff_ext_channel_switch_announcement_switch_mode;
static int hf_ieee80211_ff_ext_channel_switch_announcement_new_ope_class;
static int hf_ieee80211_ff_ext_channel_switch_announcement_new_chan_number;
static int hf_ieee80211_ff_ext_channel_switch_announcement_switch_count;

static int * const ieee80211_ff_ext_channel_switch_announcement_fields[] = {
  &hf_ieee80211_ff_ext_channel_switch_announcement_switch_mode,
  &hf_ieee80211_ff_ext_channel_switch_announcement_new_ope_class,
  &hf_ieee80211_ff_ext_channel_switch_announcement_new_chan_number,
  &hf_ieee80211_ff_ext_channel_switch_announcement_switch_count,
  NULL
};

static int hf_ieee80211_ff_ht_info;
static int hf_ieee80211_ff_ht_info_information_request;
static int hf_ieee80211_ff_ht_info_40_mhz_intolerant;
static int hf_ieee80211_ff_ht_info_sta_chan_width;
static int hf_ieee80211_ff_ht_info_reserved;

static int * const ieee80211_ff_ht_info_fields[] = {
  &hf_ieee80211_ff_ht_info_information_request,
  &hf_ieee80211_ff_ht_info_40_mhz_intolerant,
  &hf_ieee80211_ff_ht_info_sta_chan_width,
  &hf_ieee80211_ff_ht_info_reserved,
  NULL
};

static int hf_ieee80211_ff_ht_action;

static int hf_ieee80211_ff_psmp_sta_info;
static int hf_ieee80211_ff_psmp_sta_info_type;
static int hf_ieee80211_ff_psmp_sta_info_dtt_start_offset;
static int hf_ieee80211_ff_psmp_sta_info_dtt_duration;
static int hf_ieee80211_ff_psmp_sta_info_sta_id;
static int hf_ieee80211_ff_psmp_sta_info_utt_start_offset;
static int hf_ieee80211_ff_psmp_sta_info_utt_duration;
static int hf_ieee80211_ff_psmp_sta_info_reserved_small;
static int hf_ieee80211_ff_psmp_sta_info_reserved_large;
static int hf_ieee80211_ff_psmp_sta_info_psmp_multicast_id;

static int hf_ieee80211_ff_mimo_csi_snr;
static int hf_ieee80211_ff_mimo_csi_matrices;
static int hf_ieee80211_ff_mimo_csi_bf_matrices;
static int hf_ieee80211_ff_mimo_csi_cbf_matrices;

/*** Begin: 802.11s additions ***/
static int hf_ieee80211_mesh_control_field;

static int hf_ieee80211_ff_mesh_action;
static int hf_ieee80211_ff_multihop_action;
static int hf_ieee80211_ff_mesh_flags;
static int hf_ieee80211_ff_mesh_ttl;
static int hf_ieee80211_ff_mesh_sequence;
static int hf_ieee80211_ff_mesh_addr4;
static int hf_ieee80211_ff_mesh_addr5;
static int hf_ieee80211_ff_mesh_addr6;
static int hf_ieee80211_ff_selfprot_action;

static int hf_ieee80211_mesh_peering_proto;
static int hf_ieee80211_mesh_peering_local_link_id;
static int hf_ieee80211_mesh_peering_peer_link_id;

static int hf_ieee80211_ff_hwmp_flags;
static int hf_ieee80211_ff_hwmp_hopcount;
static int hf_ieee80211_ff_hwmp_ttl;
static int hf_ieee80211_ff_hwmp_pdid;
static int hf_ieee80211_ff_hwmp_orig_sta;
static int hf_ieee80211_ff_hwmp_orig_sn;
static int hf_ieee80211_ff_hwmp_orig_ext;
static int hf_ieee80211_ff_hwmp_lifetime;
static int hf_ieee80211_ff_hwmp_metric;
static int hf_ieee80211_ff_hwmp_targ_count;
static int hf_ieee80211_ff_hwmp_targ_flags;
static int hf_ieee80211_ff_hwmp_targ_to_flags;
static int hf_ieee80211_ff_hwmp_targ_usn_flags;
static int hf_ieee80211_ff_hwmp_targ_sta;
static int hf_ieee80211_ff_hwmp_targ_sn;
static int hf_ieee80211_ff_hwmp_targ_ext;
static int hf_ieee80211_rann_flags;
static int hf_ieee80211_rann_root_sta;
static int hf_ieee80211_rann_sn;
static int hf_ieee80211_rann_interval;

static int hf_ieee80211_mesh_channel_switch_ttl;
static int hf_ieee80211_mesh_channel_switch_flag;
static int hf_ieee80211_mesh_channel_switch_reason_code;
static int hf_ieee80211_mesh_channel_switch_precedence_value;
static int hf_ieee80211_mesh_chswitch_flag_txrestrict;
static int hf_ieee80211_mesh_chswitch_flag_initiator;

static int hf_ieee80211_mesh_config_path_sel_protocol;
static int hf_ieee80211_mesh_config_path_sel_metric;
static int hf_ieee80211_mesh_config_congestion_control;
static int hf_ieee80211_mesh_config_sync_method;
static int hf_ieee80211_mesh_config_auth_protocol;
static int hf_ieee80211_mesh_config_formation_info;
static int hf_ieee80211_mesh_config_capability;
static int hf_ieee80211_mesh_id;
static int hf_ieee80211_mesh_config_cap_accepting;
static int hf_ieee80211_mesh_config_cap_mcca_support;
static int hf_ieee80211_mesh_config_cap_mcca_enabled;
static int hf_ieee80211_mesh_config_cap_forwarding;
static int hf_ieee80211_mesh_config_cap_mbca_enabled;
static int hf_ieee80211_mesh_config_cap_tbtt_adjusting;
static int hf_ieee80211_mesh_config_cap_power_save_level;
static int hf_ieee80211_mesh_config_cap_reserved;
static int hf_ieee80211_mesh_form_info_conn_to_mesh_gate;
static int hf_ieee80211_mesh_form_info_num_of_peerings;
static int hf_ieee80211_mesh_form_info_conn_to_as;
static int hf_ieee80211_mesh_awake_window;
static int hf_ieee80211_mesh_mic;
static int hf_ieee80211_mesh_ampe_encrypted_data;

static int hf_ieee80211_bcn_timing_rctrl;
static int hf_ieee80211_bcn_timing_rctrl_more;
static int hf_ieee80211_bcn_timing_rctrl_element_num;
static int hf_ieee80211_bcn_timing_rctrl_status_num;
static int hf_ieee80211_bcn_timing_info;
static int hf_ieee80211_bcn_timing_info_nsta_id;
static int hf_ieee80211_bcn_timing_info_nsta_tbtt;
static int hf_ieee80211_bcn_timing_info_nsta_bi;

static int hf_ieee80211_gann_flags;
static int hf_ieee80211_gann_flags_reserved;
static int hf_ieee80211_gann_hop_count;
static int hf_ieee80211_gann_elem_ttl;
static int hf_ieee80211_gann_mesh_gate_addr;
static int hf_ieee80211_gann_seq_num;
static int hf_ieee80211_gann_interval;

static int hf_ieee80211_pxu_pxu_id;
static int hf_ieee80211_pxu_pxu_origin_mac;
static int hf_ieee80211_pxu_no_proxy_info;
static int hf_ieee80211_pxu_proxy_info;
static int hf_ieee80211_pxu_proxy_info_flags;
static int hf_ieee80211_pxu_proxy_info_flags_delete;
static int hf_ieee80211_pxu_proxy_info_flags_orig_is_proxy;
static int hf_ieee80211_pxu_proxy_info_flags_lifetime;
static int hf_ieee80211_pxu_proxy_info_flags_reserved;
static int hf_ieee80211_pxu_proxy_info_ext_mac;
static int hf_ieee80211_pxu_proxy_info_seq_num;
static int hf_ieee80211_pxu_proxy_info_proxy_mac;
static int hf_ieee80211_pxu_proxy_info_lifetime;

static int hf_ieee80211_pxuc_pxu_id;
static int hf_ieee80211_pxuc_pxu_recip_mac;

static int hf_ieee80211_ff_public_action;
static int hf_ieee80211_ff_protected_public_action;
static int hf_ieee80211_ff_tod;
static int hf_ieee80211_ff_toa;
static int hf_ieee80211_ff_max_tod_err;
static int hf_ieee80211_ff_max_toa_err;

/* ************************************************************************* */
/*            Flags found in the capability field (fixed field)              */
/* ************************************************************************* */
static int hf_ieee80211_ff_capture;
static int hf_ieee80211_ff_cf_ess;
static int hf_ieee80211_ff_cf_ibss;
static int hf_ieee80211_ff_cf_reserved1;
static int hf_ieee80211_ff_cf_reserved2;
static int hf_ieee80211_ff_cf_privacy;
static int hf_ieee80211_ff_cf_preamble;
static int hf_ieee80211_ff_cf_critical_update_flag;
static int hf_ieee80211_ff_cf_nontran_bss_critical_update_flag;
static int hf_ieee80211_ff_cf_spec_man;
static int hf_ieee80211_ff_cf_qos;
static int hf_ieee80211_ff_cf_short_slot_time;
static int hf_ieee80211_ff_cf_apsd;
static int hf_ieee80211_ff_cf_radio_measurement;
static int hf_ieee80211_ff_cf_epd;
static int hf_ieee80211_ff_cf_reserved5;
static int hf_ieee80211_ff_cf_reserved6;

/* ************************************************************************* */
/*                       A-MSDU fields                                       */
/* ************************************************************************* */
static int hf_ieee80211_amsdu_subframe;
static int hf_ieee80211_amsdu_length;
static int hf_ieee80211_amsdu_padding;

/* ************************************************************************* */
/*                       Tagged value format fields                          */
/* ************************************************************************* */
static int hf_ieee80211_tagged_parameters;  /* Tagged payload item */
static int hf_ieee80211_tag;
static int hf_ieee80211_tag_number;
static int hf_ieee80211_tag_length;
static int hf_ieee80211_tag_data;
static int hf_ieee80211_tag_oui;
static int hf_ieee80211_tag_oui_wfa_subtype;
static int hf_ieee80211_tag_oui_wfa_action_type;
static int hf_ieee80211_tag_ssid;
static int hf_ieee80211_tag_supp_rates;
static int hf_ieee80211_tag_fh_dwell_time;
static int hf_ieee80211_tag_fh_hop_set;
static int hf_ieee80211_tag_fh_hop_pattern;
static int hf_ieee80211_tag_fh_hop_index;
static int hf_ieee80211_tag_ds_param_channel;
static int hf_ieee80211_tag_cfp_count;
static int hf_ieee80211_tag_cfp_period;
static int hf_ieee80211_tag_cfp_max_duration;
static int hf_ieee80211_tag_cfp_dur_remaining;
static int hf_ieee80211_tim_dtim_count;
static int hf_ieee80211_tim_dtim_period;
static int hf_ieee80211_tim_bmapctl;
static int hf_ieee80211_tim_bmapctl_mcast;
static int hf_ieee80211_tim_bmapctl_offset;
static int hf_ieee80211_tim_partial_virtual_bitmap;
static int hf_ieee80211_tim_aid;
static int hf_ieee80211_tag_ibss_atim_window;
static int hf_ieee80211_tag_country_info_code;
static int hf_ieee80211_tag_country_info_env;
static int hf_ieee80211_tag_country_info_pad;
static int hf_ieee80211_tag_country_info_fnm;
static int hf_ieee80211_tag_country_info_fnm_fcn;
static int hf_ieee80211_tag_country_info_fnm_nc;
static int hf_ieee80211_tag_country_info_fnm_mtpl;
static int hf_ieee80211_tag_country_info_rrc;
static int hf_ieee80211_tag_country_info_rrc_oei;
static int hf_ieee80211_tag_country_info_rrc_oc;
static int hf_ieee80211_tag_country_info_rrc_cc;
static int hf_ieee80211_tag_fh_hopping_parameter_prime_radix;
static int hf_ieee80211_tag_fh_hopping_parameter_nb_channels;
static int hf_ieee80211_tag_fh_hopping_table_flag;
static int hf_ieee80211_tag_fh_hopping_table_number_of_sets;
static int hf_ieee80211_tag_fh_hopping_table_modulus;
static int hf_ieee80211_tag_fh_hopping_table_offset;
static int hf_ieee80211_tag_fh_hopping_random_table;
static int hf_ieee80211_tag_request;
static int hf_ieee80211_tag_extended_request_id;
static int hf_ieee80211_tag_extended_request_extension;
static int hf_ieee80211_tag_challenge_text;

static int hf_ieee80211_oui_qos_subtype;
static int hf_ieee80211_oui_qos_mgmt_dialog_token;
static int hf_ieee80211_oui_qos_mgmt_resp_control;
static int hf_ieee80211_oui_qos_mgmt_rsp_ctrl_more;
static int hf_ieee80211_oui_qos_mgmt_rsp_ctrl_reset;
static int hf_ieee80211_oui_qos_mgmt_rsp_reserved;
static int hf_ieee80211_oui_qos_mgmt_rqst_control;
static int hf_ieee80211_oui_qos_mgmt_rq_ctrl_more;
static int hf_ieee80211_oui_qos_mgmt_rq_ctrl_reset;
static int hf_ieee80211_oui_qos_mgmt_rq_reserved;
static int hf_ieee80211_oui_qos_mgmt_count;
static int hf_ieee80211_dscp_policy_id;
static int hf_ieee80211_dscp_policy_status;
static int hf_ieee80211_dscp_policy_scs_sts_list;

static int hf_ieee80211_tag_he_6ghz_cap_inf;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b0_b2;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b3_b5;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b6_b7;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b8;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b9_b10;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b11;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b12;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b13;
static int hf_ieee80211_tag_he_6ghz_cap_inf_b14_b15;

static int hf_ieee80211_tag_ftm_tsf_sync_info;

static int * const ieee80211_tag_he_6ghz_cap_inf[] = {
  &hf_ieee80211_tag_he_6ghz_cap_inf_b0_b2,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b3_b5,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b6_b7,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b8,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b9_b10,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b11,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b12,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b13,
  &hf_ieee80211_tag_he_6ghz_cap_inf_b14_b15,
  NULL
};


static int hf_ieee80211_wep_iv;
static int hf_ieee80211_wep_iv_weak;
static int hf_ieee80211_tkip_extiv;
static int hf_ieee80211_ccmp_extiv;
static int hf_ieee80211_wep_key;
static int hf_ieee80211_wep_icv;
static int hf_ieee80211_fc_analysis_pmk;
static int hf_ieee80211_fc_analysis_kck;
static int hf_ieee80211_fc_analysis_kek;
static int hf_ieee80211_fc_analysis_tk;
static int hf_ieee80211_fc_analysis_gtk;

static int hf_ieee80211_block_ack_control;
static int hf_ieee80211_block_ack_control_ack_policy;
static int hf_ieee80211_block_ack_control_type;
static int hf_ieee80211_block_ack_control_reserved;
static int hf_ieee80211_block_ack_control_tid_info;

static int hf_ieee80211_block_ack_multi_tid_reserved;
static int hf_ieee80211_block_ack_multi_tid_value;
static int hf_ieee80211_block_ack_bitmap;
static int hf_ieee80211_block_ack_bitmap_missing_frame;
static int hf_ieee80211_block_ack_bitmap_last_ack_frame;
static int hf_ieee80211_block_ack_gcr_addr;

static int hf_ieee80211_block_ack_multi_sta_aid11;
static int hf_ieee80211_block_ack_multi_sta_ack_type;
static int hf_ieee80211_block_ack_multi_sta_tid;
static int hf_ieee80211_block_ack_multi_sta_aid_tid;
static int hf_ieee80211_block_ack_multi_sta_reserved;
static int hf_ieee80211_block_ack_multi_sta_ra;

static int hf_ieee80211_tag_measure_request_measurement_mode;
static int hf_ieee80211_tag_measure_request_bssid;

static int hf_ieee80211_tag_measure_request_subelement_length;
static int hf_ieee80211_tag_measure_request_beacon_sub_id;
static int hf_ieee80211_tag_measure_request_beacon_sub_ssid;
static int hf_ieee80211_tag_measure_request_beacon_sub_bri_reporting_condition;
static int hf_ieee80211_tag_measure_request_beacon_sub_bri_threshold_offset;
static int hf_ieee80211_tag_measure_request_beacon_sub_reporting_detail;
static int hf_ieee80211_tag_measure_request_beacon_sub_last_report_indication_request;
static int hf_ieee80211_tag_measure_request_beacon_unknown;

static int hf_ieee80211_tag_measure_request_channel_load_sub_id;
static int hf_ieee80211_tag_measure_request_channel_load_sub_reporting_condition;
static int hf_ieee80211_tag_measure_request_channel_load_sub_reporting_ref;

static int hf_ieee80211_tag_measure_request_noise_histogram_sub_id;
static int hf_ieee80211_tag_measure_request_noise_histogram_sub_reporting_condition;
static int hf_ieee80211_tag_measure_request_noise_histogram_sub_reporting_anpi_ref;

static int hf_ieee80211_tag_measure_request_frame_request_type;
static int hf_ieee80211_tag_measure_request_mac_address;
static int hf_ieee80211_tag_measure_request_peer_mac_address;
static int hf_ieee80211_tag_measure_request_group_id;

static int hf_ieee80211_tag_measure_request_location_subject;
static int hf_ieee80211_tag_measure_request_civic_location_type;
static int hf_ieee80211_tag_measure_request_location_service_interval_units;
static int hf_ieee80211_tag_measure_request_location_service_interval;

static int hf_ieee80211_tag_measure_request_unknown;

static int hf_ieee80211_ht_pren_type;
static int hf_ieee80211_ht_pren_unknown;

static int hf_ieee80211_ht_cap;
static int hf_ieee80211_ht_vs_cap;
static int hf_ieee80211_ht_ldpc_coding;
static int hf_ieee80211_ht_chan_width;
static int hf_ieee80211_ht_sm_pwsave;
static int hf_ieee80211_ht_green;
static int hf_ieee80211_ht_short20;
static int hf_ieee80211_ht_short40;
static int hf_ieee80211_ht_tx_stbc;
static int hf_ieee80211_ht_rx_stbc;
static int hf_ieee80211_ht_reserved_b10;
static int hf_ieee80211_ht_max_amsdu;
static int hf_ieee80211_ht_dss_cck_40;
static int hf_ieee80211_ht_reserved_b13;
static int hf_ieee80211_ht_40_mhz_intolerant;
static int hf_ieee80211_ht_reserved_b15;

static int hf_ieee80211_ext_bss_mu_mimo_capable_sta_count;
static int hf_ieee80211_ext_bss_ss_underutilization;
static int hf_ieee80211_ext_bss_observable_sec_20mhz_utilization;
static int hf_ieee80211_ext_bss_observable_sec_40mhz_utilization;
static int hf_ieee80211_ext_bss_observable_sec_80mhz_utilization;
static int hf_ieee80211_wide_bw_new_channel_width;
static int hf_ieee80211_wide_bw_new_channel_center_freq_segment0;
static int hf_ieee80211_wide_bw_new_channel_center_freq_segment1;

static int hf_ieee80211_operat_notification_mode;
static int hf_ieee80211_operat_mode_field_channel_width;
static int hf_ieee80211_operat_mode_field_160_80plus80_bw;
static int hf_ieee80211_operat_mode_field_no_ldpc;
static int hf_ieee80211_operat_mode_field_rxnss;
static int hf_ieee80211_operat_mode_field_rxnsstype;

static int hf_ieee80211_tbtt_info;
static int hf_ieee80211_tbtt_filtered_nap;
static int hf_ieee80211_tbtt_info_count;
static int hf_ieee80211_tbtt_info_length;

static int hf_ieee80211_tbtt_operating_class;
static int hf_ieee80211_tbtt_channel_number;

static int hf_ieee80211_tbtt_offset;
static int hf_ieee80211_tbtt_bssid;
static int hf_ieee80211_tbtt_short_ssid;
static int hf_ieee80211_rnr_bss_params;
static int hf_ieee80211_rnr_oct_recommended;
static int hf_ieee80211_rnr_same_ssid;
static int hf_ieee80211_rnr_multiple_bssid;
static int hf_ieee80211_rnr_transmitted_bssid;
static int hf_ieee80211_rnr_ess_with_colocated_ap;
static int hf_ieee80211_rnr_unsolicited_probe_responses;
static int hf_ieee80211_rnr_same_colocated_ap;
static int hf_ieee80211_rnr_same_reserved;
static int hf_ieee80211_rnr_20mhz_psd_subfield;
static int hf_ieee80211_rnr_reserved_data;
static int hf_ieee80211_rnr_mld_params;
static int hf_ieee80211_rnr_mld_id;
static int hf_ieee80211_rnr_mld_link_id;
static int hf_ieee80211_rnr_mld_bss_params_change_count;
static int hf_ieee80211_rnr_mld_all_updates_included;
static int hf_ieee80211_rnr_mld_disabled_link_indication;
static int hf_ieee80211_rnr_mld_reserved;

static int hf_ieee80211_ampduparam;
static int hf_ieee80211_ampduparam_vs;
static int hf_ieee80211_ampduparam_mpdu;
static int hf_ieee80211_ampduparam_mpdu_start_spacing;
static int hf_ieee80211_ampduparam_reserved;

static int hf_ieee80211_beacon_sequence;
static int hf_ieee80211_pentapartial_timestamp;
static int hf_ieee80211_tack_next_twt_info;
static int hf_ieee80211_tack_next_twt;
static int hf_ieee80211_tack_flow_identifier;

static int hf_ieee80211_ff_s1g_action;
static int hf_ieee80211_ff_prot_s1g_action;
static int hf_ieee80211_ff_s1g_timestamp;
static int hf_ieee80211_ff_s1g_change_sequence;
static int hf_ieee80211_ff_s1g_next_tbtt;
static int hf_ieee80211_ff_s1g_compressed_ssid;
static int hf_ieee80211_ff_s1g_access_network_options;

static int hf_ieee80211_s1g_sync_control;
static int hf_ieee80211_s1g_sync_control_uplink_sync_request;
static int hf_ieee80211_s1g_sync_control_time_slot_protection_request;
static int hf_ieee80211_s1g_sync_control_reserved;

static int hf_ieee80211_s1g_sector_id_index;
static int hf_ieee80211_s1g_sector_id_preferred_sector_id;
static int hf_ieee80211_s1g_sector_id_snr;
static int hf_ieee80211_s1g_sector_id_receive_sector_bitmap;

static int hf_ieee80211_s1g_twt_information_control;
static int hf_ieee80211_s1g_twt_flow_identifier;
static int hf_ieee80211_s1g_twt_response_required;
static int hf_ieee80211_s1g_twt_next_twt_request;
static int hf_ieee80211_s1g_twt_next_twt_subfield_size;
static int hf_ieee80211_s1g_twt_reserved;
static int hf_ieee80211_s1g_twt_next_twt_32;
static int hf_ieee80211_s1g_twt_next_twt_48;
static int hf_ieee80211_s1g_twt_next_twt_64;

static int hf_ieee80211_s1g_update_edca_info;
static int hf_ieee80211_s1g_update_edca_override;
static int hf_ieee80211_s1g_update_edca_ps_poll_aci;
static int hf_ieee80211_s1g_update_edca_raw_aci;
static int hf_ieee80211_s1g_update_edca_sta_type;
static int hf_ieee80211_s1g_update_edca_reserved;

static int hf_ieee80211_s1g_cap_byte1;
static int hf_ieee80211_s1g_cap_byte2;
static int hf_ieee80211_s1g_cap_byte3;
static int hf_ieee80211_s1g_cap_byte4;
static int hf_ieee80211_s1g_cap_byte5;
static int hf_ieee80211_s1g_cap_byte6;
static int hf_ieee80211_s1g_cap_byte7;
static int hf_ieee80211_s1g_cap_byte8;
static int hf_ieee80211_s1g_cap_byte9;
static int hf_ieee80211_s1g_cap_byte10;
static int hf_ieee80211_s1g_cap_s1g_long_support;
static int hf_ieee80211_s1g_cap_short_gi_for_1_mhz;
static int hf_ieee80211_s1g_cap_short_gi_for_2_mhz;
static int hf_ieee80211_s1g_cap_short_gi_for_4_mhz;
static int hf_ieee80211_s1g_cap_short_gi_for_8_mhz;
static int hf_ieee80211_s1g_cap_short_gi_for_16_mhz;
static int hf_ieee80211_s1g_cap_supported_channel_width;
static int hf_ieee80211_s1g_cap_rx_ldpc;
static int hf_ieee80211_s1g_cap_tx_stbc;
static int hf_ieee80211_s1g_cap_rx_stbc;
static int hf_ieee80211_s1g_cap_su_beamformer_capable;
static int hf_ieee80211_s1g_cap_su_beamformee_capable;
static int hf_ieee80211_s1g_cap_beamformee_sts_capability;
static int hf_ieee80211_s1g_cap_number_sounding_dimensions;
static int hf_ieee80211_s1g_cap_mu_beamformer_capable;
static int hf_ieee80211_s1g_cap_mu_beamformee_capable;
static int hf_ieee80211_s1g_cap_htc_vht_capable;
static int hf_ieee80211_s1g_cap_travelling_pilot_support;
static int hf_ieee80211_s1g_cap_rd_responder;
static int hf_ieee80211_s1g_cap_ht_delayed_block_ack;
static int hf_ieee80211_s1g_cap_maximum_mpdu_length;
static int hf_ieee80211_s1g_cap_maximum_a_mpdu_length_exp;
static int hf_ieee80211_s1g_cap_minimum_mpdu_start_spacing;
static int hf_ieee80211_s1g_cap_uplink_sync_capable;
static int hf_ieee80211_s1g_cap_dynamic_aid;
static int hf_ieee80211_s1g_cap_bat_support;
static int hf_ieee80211_s1g_cap_tim_ade_support;
static int hf_ieee80211_s1g_cap_non_tim_support;
static int hf_ieee80211_s1g_cap_group_aid_support;
static int hf_ieee80211_s1g_cap_sta_type_support;
static int hf_ieee80211_s1g_cap_centralized_authentication_control;
static int hf_ieee80211_s1g_cap_distributed_authentication_control;
static int hf_ieee80211_s1g_cap_a_msdu_support;
static int hf_ieee80211_s1g_cap_a_mpdu_support;
static int hf_ieee80211_s1g_cap_asymmetic_block_ack_support;
static int hf_ieee80211_s1g_cap_flow_control_support;
static int hf_ieee80211_s1g_cap_sectorized_beam_capable;
static int hf_ieee80211_s1g_cap_obss_mitigation_support;
static int hf_ieee80211_s1g_cap_fragment_ba_support;
static int hf_ieee80211_s1g_cap_ndp_ps_poll_supported;
static int hf_ieee80211_s1g_cap_raw_operation_support;
static int hf_ieee80211_s1g_cap_page_slicing_support;
static int hf_ieee80211_s1g_cap_txop_sharing_implicit_ack_support;
static int hf_ieee80211_s1g_cap_vht_link_adaptation_capable;
static int hf_ieee80211_s1g_cap_tack_support_as_ps_poll_response;
static int hf_ieee80211_s1g_cap_duplicate_1_mhz_support;
static int hf_ieee80211_s1g_cap_mcs_negotiation_support;
static int hf_ieee80211_s1g_cap_1_mhz_control_response_preamble_support;
static int hf_ieee80211_s1g_cap_ndp_beamforming_report_poll_support;
static int hf_ieee80211_s1g_cap_unsolicited_dynamic_aid;
static int hf_ieee80211_s1g_cap_sector_training_operation_supported;
static int hf_ieee80211_s1g_cap_temporary_ps_mode_switch;
static int hf_ieee80211_s1g_cap_twt_grouping_support;
static int hf_ieee80211_s1g_cap_bdt_capable;
static int hf_ieee80211_s1g_cap_color;
static int hf_ieee80211_s1g_cap_twt_requester_support;
static int hf_ieee80211_s1g_cap_twt_responder_support;
static int hf_ieee80211_s1g_cap_pv1_frame_support;
static int hf_ieee80211_s1g_cap_link_adaptation_per_normal_control_response_capable;
static int hf_ieee80211_s1g_cap_reserved;
static int hf_ieee80211_s1g_mcs_and_nss_set;
static int hf_ieee80211_s1g_rx_s1g_mcs_map;
static int hf_ieee80211_s1g_rx_highest_supported_long_gi_data_rate;
static int hf_ieee80211_s1g_tx_s1g_mcs_map;
static int hf_ieee80211_s1g_tx_highest_supported_long_gi_data_rate;
static int hf_ieee80211_s1g_rx_single_spatial_stream_map_for_1_mhz;
static int hf_ieee80211_s1g_tx_single_spatial_stream_map_for_1_mhz;
static int hf_ieee80211_s1g_mcs_and_nss_reserved;
static int hf_ieee80211_s1g_subchannel_selective_transmission;
static int hf_ieee80211_s1g_sst_sounding_option;
static int hf_ieee80211_s1g_channel_activity_bitmap;
static int hf_ieee80211_s1g_ul_activity;
static int hf_ieee80211_s1g_dl_activity;
static int hf_ieee80211_s1g_max_trans_width;
static int hf_ieee80211_s1g_activity_start_time;
static int hf_ieee80211_s1g_sst_sounding_option1;
static int hf_ieee80211_s1g_channel_activity_bitmap1;
static int hf_ieee80211_s1g_sounding_start_time_present;
static int hf_ieee80211_s1g_channel_activity_reserved;
static int hf_ieee80211_s1g_max_trans_width1;
static int hf_ieee80211_s1g_sounding_start_time;
static int hf_ieee80211_s1g_open_loop_link_margin;
static int hf_ieee80211_s1g_raw_control;
static int hf_ieee80211_s1g_raw_type;
static int hf_ieee80211_s1g_raw_type_options;
static int hf_ieee80211_s1g_raw_start_time_indication;
static int hf_ieee80211_s1g_raw_raw_group_indication;
static int hf_ieee80211_s1g_raw_channel_indication_preference;
static int hf_ieee80211_s1g_raw_periodic_raw_indication;
static int hf_ieee80211_s1g_raw_slot_def;
static int hf_ieee80211_s1g_slot_def_format_indication;
static int hf_ieee80211_s1g_slot_def_cross_slot_boundary;
static int hf_ieee80211_s1g_slot_def_slot_duration_count8;
static int hf_ieee80211_s1g_slot_def_num_slots6;
static int hf_ieee80211_s1g_slot_def_slot_duration_count11;
static int hf_ieee80211_s1g_slot_def_num_slots3;
static int hf_ieee80211_s1g_raw_start_time;
static int hf_ieee80211_s1g_raw_group_subfield;
static int hf_ieee80211_s1g_raw_group_page_index;
static int hf_ieee80211_s1g_raw_group_start_aid;
static int hf_ieee80211_s1g_raw_group_end_aid;
static int hf_ieee80211_s1g_raw_channel_indication;
static int hf_ieee80211_s1g_raw_ci_channel_activity_bitmap;
static int hf_ieee80211_s1g_raw_ci_max_trans_width;
static int hf_ieee80211_s1g_raw_ci_ul_activity;
static int hf_ieee80211_s1g_raw_ci_dl_activity;
static int hf_ieee80211_s1g_raw_ci_reserved;
static int hf_ieee80211_s1g_raw_praw_periodicity;
static int hf_ieee80211_s1g_raw_praw_validity;
static int hf_ieee80211_s1g_raw_praw_start_offset;
static int hf_ieee80211_s1g_page_slice_page_period;
static int hf_ieee80211_s1g_page_slice_control;
static int hf_ieee80211_s1g_page_slice_page_index;
static int hf_ieee80211_s1g_page_slice_page_slice_length;
static int hf_ieee80211_s1g_page_slice_page_slice_count;
static int hf_ieee80211_s1g_page_slice_block_offset;
static int hf_ieee80211_s1g_page_slice_tim_offset;
static int hf_ieee80211_s1g_page_slice_reserved;
static int hf_ieee80211_s1g_page_slice_page_bitmap;
static int hf_ieee80211_s1g_aid_request_mode;
static int hf_ieee80211_s1g_aid_request_interval_present;
static int hf_ieee80211_s1g_aid_request_per_sta_address_present;
static int hf_ieee80211_s1g_aid_request_service_characteristic_present;
static int hf_ieee80211_s1g_aid_request_non_tim_mode_switch;
static int hf_ieee80211_s1g_aid_request_tim_mode_switch;
static int hf_ieee80211_s1g_aid_request_group_address_present;
static int hf_ieee80211_s1g_aid_request_reserved;
static int hf_ieee80211_s1g_aid_request_interval;
static int hf_ieee80211_s1g_aid_request_characteristic_sensor;
static int hf_ieee80211_s1g_aid_request_characteristic_offload;
static int hf_ieee80211_s1g_aid_request_characteristic_official_service;
static int hf_ieee80211_s1g_aid_request_characteristic_reserved;
static int hf_ieee80211_s1g_aid_req_peer_sta_addr;
static int hf_ieee80211_s1g_aid_request_characteristic;
static int hf_ieee80211_s1g_aid_req_group_addr;
static int hf_ieee80211_s1g_aid_rsp_aid_group_aid;
static int hf_ieee80211_s1g_aid_rsp_aid_switch_count;
static int hf_ieee80211_s1g_aid_rsp_aid_response_interval;
static int hf_ieee80211_s1g_sector_op_control_16b;
static int hf_ieee80211_s1g_sector_op_sectorization_type_b16;
static int hf_ieee80211_s1g_sector_op_periodic_training_indicator;
static int hf_ieee80211_s1g_sector_op_training_period;
static int hf_ieee80211_s1g_sector_op_remaining_beacon_interval;
static int hf_ieee80211_s1g_sector_op_reserved_b16;
static int hf_ieee80211_s1g_sector_op_control;
static int hf_ieee80211_s1g_sector_op_sectorization_type;
static int hf_ieee80211_s1g_sector_op_period;
static int hf_ieee80211_s1g_sector_op_omni;
static int hf_ieee80211_s1g_sector_op_group_info;
static int hf_ieee80211_s1g_short_beacon_interval;
static int hf_ieee80211_s1g_change_sequence;
static int hf_ieee80211_s1g_auth_control_control;
static int hf_ieee80211_s1g_auth_control_deferral;
static int hf_ieee80211_s1g_auth_control_reserved;
static int hf_ieee80211_s1g_auth_control_thresh;
static int hf_ieee80211_s1g_auth_control_thresh_tus;
static int hf_ieee80211_s1g_auth_slot_duration;
static int hf_ieee80211_s1g_auth_max_trans_int;
static int hf_ieee80211_s1g_auth_min_trans_int;
static int hf_ieee80211_s1g_tsf_timer_accuracy;
static int hf_ieee80211_s1g_relay_control;
static int hf_ieee80211_s1g_relay_control_rootap_bssid;
static int hf_ieee80211_s1g_relay_function_activation_mode;
static int hf_ieee80211_s1g_relay_function_direction;
static int hf_ieee80211_s1g_relay_function_enable_relay_function;
static int hf_ieee80211_s1g_relay_function_stas_present_indic;
static int hf_ieee80211_s1g_relay_function_reserved;
static int hf_ieee80211_s1g_number_of_stas;
static int hf_ieee80211_s1g_initiator_mac_address;
static int hf_ieee80211_s1g_address_count;
static int hf_ieee80211_s1g_reachable_add_remove;
static int hf_ieee80211_s1g_reachable_relay_capable;
static int hf_ieee80211_s1g_reachable_reserved;
static int hf_ieee80211_s1g_reachable_mac_address;
static int hf_ieee80211_s1g_relay_discovery_control;
static int hf_ieee80211_s1g_min_data_rate_included;
static int hf_ieee80211_s1g_mean_data_rate_included;
static int hf_ieee80211_s1g_max_data_rate_included;
static int hf_ieee80211_s1g_delay_and_min_phy_rate;
static int hf_ieee80211_s1g_information_not_available;
static int hf_ieee80211_s1g_relay_discovery_reserved;
static int hf_ieee80211_s1g_relay_control_ul_min;
static int hf_ieee80211_s1g_relay_control_ul_mean;
static int hf_ieee80211_s1g_relay_control_ul_max;
static int hf_ieee80211_s1g_relay_control_dl_min;
static int hf_ieee80211_s1g_relay_control_dl_mean;
static int hf_ieee80211_s1g_relay_control_dl_max;
static int hf_ieee80211_s1g_relay_hierarchy_identifier;
static int hf_ieee80211_s1g_relay_no_more_relay_flag;
static int hf_ieee80211_s1g_aid_entry_mac_addr;
static int hf_ieee80211_s1g_aid_entry_assoc_id;
static int hf_ieee80211_s1g_beacon_compatibility_info;
static int hf_ieee80211_s1g_beacon_interval;
static int hf_ieee80211_s1g_tsf_completion;
static int hf_ieee80211_s1g_channel_width;
static int hf_ieee80211_s1g_primary_channel_width;
static int hf_ieee80211_s1g_bss_operating_channel_width;
static int hf_ieee80211_s1g_primary_channel_location;
static int hf_ieee80211_s1g_reserved_b6;
static int hf_ieee80211_s1g_mcs10_use;
static int hf_ieee80211_s1g_operating_class;
static int hf_ieee80211_s1g_primary_channel_number;
static int hf_ieee80211_s1g_channel_center_frequency;
static int hf_ieee80211_s1g_basic_mcs_and_nss_set;
static int hf_ieee80211_s1g_sst_enabled_channel_bitmap;
static int hf_ieee80211_s1g_sst_primary_channel_offset;
static int hf_ieee80211_s1g_sst_channel_unit;
static int hf_ieee80211_s1g_sst_reserved;
static int hf_ieee80211_s1g_max_away_duration;
static int hf_ieee80211_s1g_tim_bmapctrl;
static int hf_ieee80211_s1g_tim_bmapctl_traffic_indicator;
static int hf_ieee80211_s1g_tim_page_slice_number;
static int hf_ieee80211_s1g_tim_page_index;
static int hf_ieee80211_s1g_pvb_block_control_byte;
static int hf_ieee80211_s1g_pvb_encoding_mode;
static int hf_ieee80211_s1g_pvb_inverse_bitmap;
static int hf_ieee80211_s1g_pvb_block_offset;
static int hf_ieee80211_s1g_block_bitmap;
static int hf_ieee80211_s1g_block_bitmap_sta_aid13;
static int hf_ieee80211_s1g_block_bitmap_single_aid;
static int hf_ieee80211_s1g_block_bitmap_olb_length;
static int hf_ieee80211_s1g_block_bitmap_ade;
static int hf_ieee80211_s1g_block_bitmap_ewl;
static int hf_ieee80211_s1g_block_bitmap_len;
static int hf_ieee80211_s1g_block_bitmap_ade_bytes;
static int hf_ieee80211_s1g_probe_response_group_bitmap;
static int hf_ieee80211_s1g_probe_resp_subfield_0;
static int hf_ieee80211_pv1_probe_response_req_full_ssid;
static int hf_ieee80211_pv1_probe_response_req_next_tbtt;
static int hf_ieee80211_pv1_probe_response_req_access_network_option;
static int hf_ieee80211_pv1_probe_response_req_s1g_beacon_compatibility;
static int hf_ieee80211_pv1_probe_response_req_supported_rates;
static int hf_ieee80211_pv1_probe_response_req_s1g_capability;
static int hf_ieee80211_pv1_probe_response_req_s1g_operation;
static int hf_ieee80211_pv1_probe_response_req_rsn;
static int hf_ieee80211_s1g_el_op_max_awake_duration;
static int hf_ieee80211_s1g_el_op_recovery_time_duration;
static int hf_ieee80211_s1g_sectorized_group_id_list;
static int hf_ieee80211_s1g_header_comp_control;
static int hf_ieee80211_s1g_header_comp_req_resp;
static int hf_ieee80211_s1g_header_comp_store_a3;
static int hf_ieee80211_s1g_header_comp_store_a4;
static int hf_ieee80211_s1g_header_comp_ccmp_update_present;
static int hf_ieee80211_s1g_header_comp_pv1_data_type_3_supported;
static int hf_ieee80211_s1g_header_comp_reserved;
static int hf_ieee80211_s1g_header_comp_a3;
static int hf_ieee80211_s1g_header_comp_a4;
static int hf_ieee80211_s1g_header_comp_ccmp_update;

static int hf_ieee80211_mcsset;
static int hf_ieee80211_mcsset_vs;
static int hf_ieee80211_mcsset_rx_bitmask;
static int hf_ieee80211_mcsset_rx_bitmask_0to7;
static int hf_ieee80211_mcsset_rx_bitmask_8to15;
static int hf_ieee80211_mcsset_rx_bitmask_16to23;
static int hf_ieee80211_mcsset_rx_bitmask_24to31;
static int hf_ieee80211_mcsset_rx_bitmask_32;
static int hf_ieee80211_mcsset_rx_bitmask_33to38;
static int hf_ieee80211_mcsset_rx_bitmask_39to52;
static int hf_ieee80211_mcsset_rx_bitmask_53to76;
static int hf_ieee80211_mcsset_highest_data_rate;
static int hf_ieee80211_mcsset_tx_mcs_set_defined;
static int hf_ieee80211_mcsset_tx_rx_mcs_set_not_equal;
static int hf_ieee80211_mcsset_tx_max_spatial_streams;
static int hf_ieee80211_mcsset_tx_unequal_modulation;

static int hf_ieee80211_htex_cap;
static int hf_ieee80211_htex_vs_cap;
static int hf_ieee80211_htex_reserved_b0_b7;
static int hf_ieee80211_htex_mcs;
static int hf_ieee80211_htex_htc_support;
static int hf_ieee80211_htex_rd_responder;
static int hf_ieee80211_htex_reserved_b12_b15;

static int hf_ieee80211_txbf;
static int hf_ieee80211_txbf_vs;
static int hf_ieee80211_txbf_cap;
static int hf_ieee80211_txbf_rcv_ssc;
static int hf_ieee80211_txbf_tx_ssc;
static int hf_ieee80211_txbf_rcv_ndp;
static int hf_ieee80211_txbf_tx_ndp;
static int hf_ieee80211_txbf_impl_txbf;
static int hf_ieee80211_txbf_calib;
static int hf_ieee80211_txbf_expl_csi;
static int hf_ieee80211_txbf_expl_uncomp_fm;
static int hf_ieee80211_txbf_expl_comp_fm;
static int hf_ieee80211_txbf_expl_bf_csi;
static int hf_ieee80211_txbf_expl_uncomp_fm_feed;
static int hf_ieee80211_txbf_expl_comp_fm_feed;
static int hf_ieee80211_txbf_csi_num_bf_ant;
static int hf_ieee80211_txbf_min_group;
static int hf_ieee80211_txbf_uncomp_sm_bf_ant;
static int hf_ieee80211_txbf_comp_sm_bf_ant;
static int hf_ieee80211_txbf_csi_max_rows_bf;
static int hf_ieee80211_txbf_chan_est;
static int hf_ieee80211_txbf_resrv;

/*** Begin: 802.11n - HT Operation IE  ***/
static int hf_ieee80211_ht_operation_primary_channel;

static int hf_ieee80211_ht_operation_info_delimiter1;
static int hf_ieee80211_ht_operation_info_secondary_channel_offset;
static int hf_ieee80211_ht_operation_info_sta_channel_width;
static int hf_ieee80211_ht_operation_info_rifs_mode;
static int hf_ieee80211_ht_operation_info_reserved_b4_b7;

static int hf_ieee80211_ht_operation_info_delimiter2;
static int hf_ieee80211_ht_operation_info_protection;
static int hf_ieee80211_ht_operation_info_non_greenfield_sta_present;
static int hf_ieee80211_ht_operation_info_reserved_b11;
static int hf_ieee80211_ht_operation_info_obss_non_ht_stas_present;
static int hf_ieee80211_ht_operation_info_channel_center_freq_seg_2;
static int hf_ieee80211_ht_operation_info_reserved_b21_b23;

static int hf_ieee80211_ht_operation_info_delimiter3;
static int hf_ieee80211_ht_operation_info_reserved_b24_b29;
static int hf_ieee80211_ht_operation_info_dual_beacon;
static int hf_ieee80211_ht_operation_info_dual_cts_protection;
static int hf_ieee80211_ht_operation_info_stbc_beacon;
static int hf_ieee80211_ht_operation_info_reserved_b33_b39;

static int hf_ieee80211_ht_operation_mcsset_reserved;
/*** End: 802.11n - HT Operation IE  ***/

static int hf_ieee80211_tag_ap_channel_report_operating_class;
static int hf_ieee80211_tag_ap_channel_report_channel_list;

static int hf_ieee80211_tag_secondary_channel_offset;

static int hf_ieee80211_tag_bss_ap_avg_access_delay;

static int hf_ieee80211_tag_antenna_id;

static int hf_ieee80211_tag_rsni;

static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up0;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up2;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up3;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up4;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up5;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up6;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_up7;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac0;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac1;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac2;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_ac3;
static int hf_ieee80211_tag_bss_avb_adm_cap_bitmask_rsv;
static int hf_ieee80211_tag_bss_avb_adm_cap_up0;
static int hf_ieee80211_tag_bss_avb_adm_cap_up1;
static int hf_ieee80211_tag_bss_avb_adm_cap_up2;
static int hf_ieee80211_tag_bss_avb_adm_cap_up3;
static int hf_ieee80211_tag_bss_avb_adm_cap_up4;
static int hf_ieee80211_tag_bss_avb_adm_cap_up5;
static int hf_ieee80211_tag_bss_avb_adm_cap_up6;
static int hf_ieee80211_tag_bss_avb_adm_cap_up7;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac0;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac1;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac2;
static int hf_ieee80211_tag_bss_avb_adm_cap_ac3;

static int hf_ieee80211_tag_bss_avg_ac_access_delay_be;
static int hf_ieee80211_tag_bss_avg_ac_access_delay_bk;
static int hf_ieee80211_tag_bss_avg_ac_access_delay_vi;
static int hf_ieee80211_tag_bss_avg_ac_access_delay_vo;

static int hf_ieee80211_tag_rm_enabled_capabilities;
static int hf_ieee80211_tag_rm_enabled_capabilities_b0;
static int hf_ieee80211_tag_rm_enabled_capabilities_b1;
static int hf_ieee80211_tag_rm_enabled_capabilities_b2;
static int hf_ieee80211_tag_rm_enabled_capabilities_b3;
static int hf_ieee80211_tag_rm_enabled_capabilities_b4;
static int hf_ieee80211_tag_rm_enabled_capabilities_b5;
static int hf_ieee80211_tag_rm_enabled_capabilities_b6;
static int hf_ieee80211_tag_rm_enabled_capabilities_b7;
static int hf_ieee80211_tag_rm_enabled_capabilities_b8;
static int hf_ieee80211_tag_rm_enabled_capabilities_b9;
static int hf_ieee80211_tag_rm_enabled_capabilities_b10;
static int hf_ieee80211_tag_rm_enabled_capabilities_b11;
static int hf_ieee80211_tag_rm_enabled_capabilities_b12;
static int hf_ieee80211_tag_rm_enabled_capabilities_b13;
static int hf_ieee80211_tag_rm_enabled_capabilities_b14;
static int hf_ieee80211_tag_rm_enabled_capabilities_b15;
static int hf_ieee80211_tag_rm_enabled_capabilities_b16;
static int hf_ieee80211_tag_rm_enabled_capabilities_b17;
static int hf_ieee80211_tag_rm_enabled_capabilities_b18to20;
static int hf_ieee80211_tag_rm_enabled_capabilities_b21to23;
static int hf_ieee80211_tag_rm_enabled_capabilities_b24to26;
static int hf_ieee80211_tag_rm_enabled_capabilities_b27;
static int hf_ieee80211_tag_rm_enabled_capabilities_b28;
static int hf_ieee80211_tag_rm_enabled_capabilities_b29;
static int hf_ieee80211_tag_rm_enabled_capabilities_b30;
static int hf_ieee80211_tag_rm_enabled_capabilities_b31;
static int hf_ieee80211_tag_rm_enabled_capabilities_b32;
static int hf_ieee80211_tag_rm_enabled_capabilities_b33;
static int hf_ieee80211_tag_rm_enabled_capabilities_b34;
static int hf_ieee80211_tag_rm_enabled_capabilities_b35;
static int hf_ieee80211_tag_rm_enabled_capabilities_o5;

static int hf_ieee80211_tag_rcpi;
static int hf_ieee80211_tag_multiple_bssid;
static int hf_ieee80211_tag_multiple_bssid_subelem_id;
static int hf_ieee80211_tag_multiple_bssid_subelem_len;
static int hf_ieee80211_tag_multiple_bssid_subelem_reserved;
static int hf_ieee80211_tag_multiple_bssid_subelem_nontrans_profile;

static int hf_ieee80211_tag_20_40_bc;
static int hf_ieee80211_tag_20_40_bc_information_request;
static int hf_ieee80211_tag_20_40_bc_forty_mhz_intolerant;
static int hf_ieee80211_tag_20_40_bc_20_mhz_bss_width_request;
static int hf_ieee80211_tag_20_40_bc_obss_scanning_exemption_request;
static int hf_ieee80211_tag_20_40_bc_obss_scanning_exemption_grant;
static int hf_ieee80211_tag_20_40_bc_reserved;

static int hf_ieee80211_tag_intolerant_operating_class;
static int hf_ieee80211_tag_intolerant_channel_list;
static int hf_ieee80211_tag_intolerant_channel;

static int hf_ieee80211_tag_power_constraint_local;

static int hf_ieee80211_tag_power_capability_min;
static int hf_ieee80211_tag_power_capability_max;

static int hf_ieee80211_tag_tpc_report_trsmt_pow;
static int hf_ieee80211_tag_tpc_report_link_mrg;
static int hf_ieee80211_tag_tpc_report_reserved;

static int hf_ieee80211_tag_supported_channels;
static int hf_ieee80211_tag_supported_channels_first;
static int hf_ieee80211_tag_supported_channels_range;

static int hf_ieee80211_csa_channel_switch_mode;
static int hf_ieee80211_csa_new_channel_number;
static int hf_ieee80211_csa_channel_switch_count;

static int hf_ieee80211_tag_measure_request_token;
static int hf_ieee80211_tag_measure_request_mode;
static int hf_ieee80211_tag_measure_request_mode_parallel;
static int hf_ieee80211_tag_measure_request_mode_enable;
static int hf_ieee80211_tag_measure_request_mode_request;
static int hf_ieee80211_tag_measure_request_mode_report;
static int hf_ieee80211_tag_measure_request_mode_duration_mandatory;
static int hf_ieee80211_tag_measure_request_mode_reserved;
static int hf_ieee80211_tag_measure_request_type;

static int hf_ieee80211_tag_measure_request_channel_number;
static int hf_ieee80211_tag_measure_request_start_time;
static int hf_ieee80211_tag_measure_request_duration;

static int hf_ieee80211_tag_measure_request_operating_class;
static int hf_ieee80211_tag_measure_request_randomization_interval;

static int hf_ieee80211_tag_measure_report_measurement_token;
static int hf_ieee80211_tag_measure_report_mode;
static int hf_ieee80211_tag_measure_report_mode_late;
static int hf_ieee80211_tag_measure_report_mode_incapable;
static int hf_ieee80211_tag_measure_report_mode_refused;
static int hf_ieee80211_tag_measure_report_mode_reserved;
static int hf_ieee80211_tag_measure_report_type;
static int hf_ieee80211_tag_measure_report_channel_number;
static int hf_ieee80211_tag_measure_report_start_time;
static int hf_ieee80211_tag_measure_report_duration;

static int hf_ieee80211_tag_measure_basic_map_field;
static int hf_ieee80211_tag_measure_map_field_bss;
static int hf_ieee80211_tag_measure_map_field_ofdm;
static int hf_ieee80211_tag_measure_map_field_unident_signal;
static int hf_ieee80211_tag_measure_map_field_radar;
static int hf_ieee80211_tag_measure_map_field_unmeasured;
static int hf_ieee80211_tag_measure_map_field_reserved;

static int hf_ieee80211_tag_measure_cca_busy_fraction;

static int hf_ieee80211_tag_measure_rpi_histogram_report;
static int hf_ieee80211_tag_measure_rpi_histogram_report_0;
static int hf_ieee80211_tag_measure_rpi_histogram_report_1;
static int hf_ieee80211_tag_measure_rpi_histogram_report_2;
static int hf_ieee80211_tag_measure_rpi_histogram_report_3;
static int hf_ieee80211_tag_measure_rpi_histogram_report_4;
static int hf_ieee80211_tag_measure_rpi_histogram_report_5;
static int hf_ieee80211_tag_measure_rpi_histogram_report_6;
static int hf_ieee80211_tag_measure_rpi_histogram_report_7;

static int hf_ieee80211_tag_measure_report_operating_class;
static int hf_ieee80211_tag_measure_report_channel_load;
static int hf_ieee80211_tag_measure_report_frame_info;
static int hf_ieee80211_tag_measure_report_frame_info_phy_type;
static int hf_ieee80211_tag_measure_report_frame_info_frame_type;
static int hf_ieee80211_tag_measure_report_rcpi;
static int hf_ieee80211_tag_measure_report_rsni;
static int hf_ieee80211_tag_measure_report_bssid;
static int hf_ieee80211_tag_measure_report_ant_id;
static int hf_ieee80211_tag_measure_report_anpi;
static int hf_ieee80211_tag_measure_report_ipi_density_0;
static int hf_ieee80211_tag_measure_report_ipi_density_1;
static int hf_ieee80211_tag_measure_report_ipi_density_2;
static int hf_ieee80211_tag_measure_report_ipi_density_3;
static int hf_ieee80211_tag_measure_report_ipi_density_4;
static int hf_ieee80211_tag_measure_report_ipi_density_5;
static int hf_ieee80211_tag_measure_report_ipi_density_6;
static int hf_ieee80211_tag_measure_report_ipi_density_7;
static int hf_ieee80211_tag_measure_report_ipi_density_8;
static int hf_ieee80211_tag_measure_report_ipi_density_9;
static int hf_ieee80211_tag_measure_report_ipi_density_10;
static int hf_ieee80211_tag_measure_report_parent_tsf;

static int hf_ieee80211_tag_measure_report_subelement_length;
static int hf_ieee80211_tag_measure_report_beacon_sub_id;
static int hf_ieee80211_tag_measure_report_beacon_unknown;
static int hf_ieee80211_tag_measure_report_beacon_sub_last_report_indication;
static int hf_ieee80211_tag_measure_reported_frame_frag_id;
static int hf_ieee80211_tag_measure_reported_frame_frag_rep_id;
static int hf_ieee80211_tag_measure_reported_frame_frag_number;
static int hf_ieee80211_tag_measure_reported_frame_frag_more;

static int hf_ieee80211_tag_measure_report_lci_sub_id;
static int hf_ieee80211_tag_measure_report_lci_lci;
static int hf_ieee80211_tag_measure_report_lci_z_sta_floor_info;
static int hf_ieee80211_tag_measure_report_lci_z_sta_floor_info_expected_to_move;
static int hf_ieee80211_tag_measure_report_lci_z_sta_floor_info_sta_floor_number;
static int hf_ieee80211_tag_measure_report_lci_z_sta_height_above_floor;
static int hf_ieee80211_tag_measure_report_lci_z_sta_height_above_floor_uncertainty;
static int hf_ieee80211_tag_measure_report_lci_urp;
static int hf_ieee80211_tag_measure_report_lci_urp_retransmission_allowed;
static int hf_ieee80211_tag_measure_report_lci_urp_retention_expires_relative_present;
static int hf_ieee80211_tag_measure_report_lci_urp_sta_location_policy;
static int hf_ieee80211_tag_measure_report_lci_urp_reserved;
static int hf_ieee80211_tag_measure_report_lci_urp_retention_expires_relative;
static int hf_ieee80211_tag_measure_report_lci_unknown;

static int hf_ieee80211_tag_measure_report_civic_location_type;
static int hf_ieee80211_tag_measure_report_civic_sub_id;
static int hf_ieee80211_tag_measure_report_location_civic_country;
static int hf_ieee80211_tag_measure_report_location_civic_type;
static int hf_ieee80211_tag_measure_report_location_civic_length;
static int hf_ieee80211_tag_measure_report_location_civic;

static int hf_ieee80211_tag_measure_report_unknown;

static int hf_ieee80211_tag_quiet_count;
static int hf_ieee80211_tag_quiet_period;
static int hf_ieee80211_tag_quiet_duration;
static int hf_ieee80211_tag_quiet_offset;

static int hf_ieee80211_tag_dfs_owner;
static int hf_ieee80211_tag_dfs_recovery_interval;
static int hf_ieee80211_tag_dfs_channel_map;
static int hf_ieee80211_tag_dfs_channel_number;
static int hf_ieee80211_tag_dfs_map;

static int hf_ieee80211_tag_erp_info;
static int hf_ieee80211_tag_erp_info_erp_present;
static int hf_ieee80211_tag_erp_info_use_protection;
static int hf_ieee80211_tag_erp_info_barker_preamble_mode;
static int hf_ieee80211_tag_erp_info_reserved;

static int hf_ieee80211_tag_extended_capabilities;
static int hf_ieee80211_tag_extended_capabilities_b0;
static int hf_ieee80211_tag_extended_capabilities_b1;
static int hf_ieee80211_tag_extended_capabilities_b2;
static int hf_ieee80211_tag_extended_capabilities_b3;
static int hf_ieee80211_tag_extended_capabilities_b4;
static int hf_ieee80211_tag_extended_capabilities_b5;
static int hf_ieee80211_tag_extended_capabilities_b6;
static int hf_ieee80211_tag_extended_capabilities_b7;
static int hf_ieee80211_tag_extended_capabilities_b8;
static int hf_ieee80211_tag_extended_capabilities_b9;
static int hf_ieee80211_tag_extended_capabilities_b10;
static int hf_ieee80211_tag_extended_capabilities_b11;
static int hf_ieee80211_tag_extended_capabilities_b12;
static int hf_ieee80211_tag_extended_capabilities_b13;
static int hf_ieee80211_tag_extended_capabilities_b14;
static int hf_ieee80211_tag_extended_capabilities_b15;
static int hf_ieee80211_tag_extended_capabilities_b16;
static int hf_ieee80211_tag_extended_capabilities_b17;
static int hf_ieee80211_tag_extended_capabilities_b18;
static int hf_ieee80211_tag_extended_capabilities_b19;
static int hf_ieee80211_tag_extended_capabilities_b20;
static int hf_ieee80211_tag_extended_capabilities_b21;
static int hf_ieee80211_tag_extended_capabilities_b22;
static int hf_ieee80211_tag_extended_capabilities_b23;
static int hf_ieee80211_tag_extended_capabilities_b24;
static int hf_ieee80211_tag_extended_capabilities_b25;
static int hf_ieee80211_tag_extended_capabilities_b26;
static int hf_ieee80211_tag_extended_capabilities_b27;
static int hf_ieee80211_tag_extended_capabilities_b28;
static int hf_ieee80211_tag_extended_capabilities_b29;
static int hf_ieee80211_tag_extended_capabilities_b30;
static int hf_ieee80211_tag_extended_capabilities_b31;
static int hf_ieee80211_tag_extended_capabilities_b32;
static int hf_ieee80211_tag_extended_capabilities_b33;
static int hf_ieee80211_tag_extended_capabilities_b34;
static int hf_ieee80211_tag_extended_capabilities_b35;
static int hf_ieee80211_tag_extended_capabilities_b36;
static int hf_ieee80211_tag_extended_capabilities_b37;
static int hf_ieee80211_tag_extended_capabilities_b38;
static int hf_ieee80211_tag_extended_capabilities_b39;
static int hf_ieee80211_tag_extended_capabilities_b40;
static int hf_ieee80211_tag_extended_capabilities_serv_int_granularity;
static int hf_ieee80211_tag_extended_capabilities_b44;
static int hf_ieee80211_tag_extended_capabilities_b45;
static int hf_ieee80211_tag_extended_capabilities_b46;
static int hf_ieee80211_tag_extended_capabilities_b47;
static int hf_ieee80211_tag_extended_capabilities_b48;
static int hf_ieee80211_tag_extended_capabilities_b49;
static int hf_ieee80211_tag_extended_capabilities_b50;
static int hf_ieee80211_tag_extended_capabilities_b51;
static int hf_ieee80211_tag_extended_capabilities_b52;
static int hf_ieee80211_tag_extended_capabilities_b53;
static int hf_ieee80211_tag_extended_capabilities_b54;
static int hf_ieee80211_tag_extended_capabilities_b55;
static int hf_ieee80211_tag_extended_capabilities_b56;
static int hf_ieee80211_tag_extended_capabilities_b57;
static int hf_ieee80211_tag_extended_capabilities_b58;
static int hf_ieee80211_tag_extended_capabilities_b59;
static int hf_ieee80211_tag_extended_capabilities_b60;
static int hf_ieee80211_tag_extended_capabilities_b61;
static int hf_ieee80211_tag_extended_capabilities_b62;
static int hf_ieee80211_tag_extended_capabilities_b63;
/* Used for the two-byte ext-cap field when present */
static int hf_ieee80211_tag_extended_capabilities_2;
static int hf_ieee80211_tag_extended_capabilities_b56_2;
static int hf_ieee80211_tag_extended_capabilities_b57_2;
static int hf_ieee80211_tag_extended_capabilities_b58_2;
static int hf_ieee80211_tag_extended_capabilities_b59_2;
static int hf_ieee80211_tag_extended_capabilities_b60_2;
static int hf_ieee80211_tag_extended_capabilities_b61_2;
static int hf_ieee80211_tag_extended_capabilities_b62_2;
static int hf_ieee80211_tag_extended_capabilities_max_num_msdus;
static int hf_ieee80211_tag_extended_capabilities_b65_2;
static int hf_ieee80211_tag_extended_capabilities_b66_2;
static int hf_ieee80211_tag_extended_capabilities_b67_2;
static int hf_ieee80211_tag_extended_capabilities_b68_2;
static int hf_ieee80211_tag_extended_capabilities_b69_2;
static int hf_ieee80211_tag_extended_capabilities_b70_2;
static int hf_ieee80211_tag_extended_capabilities_b71_2;

static int hf_ieee80211_tag_extended_capabilities_b72;
static int hf_ieee80211_tag_extended_capabilities_b73;
static int hf_ieee80211_tag_extended_capabilities_b74;
static int hf_ieee80211_tag_extended_capabilities_b75;
static int hf_ieee80211_tag_extended_capabilities_b76;
static int hf_ieee80211_tag_extended_capabilities_b77;
static int hf_ieee80211_tag_extended_capabilities_b78;
static int hf_ieee80211_tag_extended_capabilities_b79;
static int hf_ieee80211_tag_extended_capabilities_b80;
static int hf_ieee80211_tag_extended_capabilities_b81;
static int hf_ieee80211_tag_extended_capabilities_b82;
static int hf_ieee80211_tag_extended_capabilities_b83;
static int hf_ieee80211_tag_extended_capabilities_b84;
static int hf_ieee80211_tag_extended_capabilities_b85;
static int hf_ieee80211_tag_extended_capabilities_b86;
static int hf_ieee80211_tag_extended_capabilities_b87;

static int hf_ieee80211_tag_extended_capabilities_b88;
static int hf_ieee80211_tag_extended_capabilities_b89;
static int hf_ieee80211_tag_extended_capabilities_b90;
static int hf_ieee80211_tag_extended_capabilities_b91;
static int hf_ieee80211_tag_extended_capabilities_b92;
static int hf_ieee80211_tag_extended_capabilities_b93;
static int hf_ieee80211_tag_extended_capabilities_b94;
static int hf_ieee80211_tag_extended_capabilities_b95;

static int hf_ieee80211_tag_extended_capabilities_b96;
static int hf_ieee80211_tag_extended_capabilities_b97;
static int hf_ieee80211_tag_extended_capabilities_b98;
static int hf_ieee80211_tag_extended_capabilities_b99;
static int hf_ieee80211_tag_extended_capabilities_b100;
static int hf_ieee80211_tag_extended_capabilities_b101;
static int hf_ieee80211_tag_extended_capabilities_b102;
static int hf_ieee80211_tag_extended_capabilities_b103;
static int hf_ieee80211_tag_extended_capabilities_b104;
static int hf_ieee80211_tag_extended_capabilities_b105;
static int hf_ieee80211_tag_extended_capabilities_reserved2;

static int hf_ieee80211_tag_cisco_ccx1_unknown;
static int hf_ieee80211_tag_cisco_ccx1_name;
static int hf_ieee80211_tag_cisco_ccx1_clients;
static int hf_ieee80211_tag_cisco_ccx1_unknown2;

static int hf_ieee80211_vs_cisco_ap_name_v2;

static int hf_ieee80211_vht_cap;
static int hf_ieee80211_vht_max_mpdu_length;
static int hf_ieee80211_vht_supported_chan_width_set;
static int hf_ieee80211_vht_rx_ldpc;
static int hf_ieee80211_vht_short_gi_for_80;
static int hf_ieee80211_vht_short_gi_for_160;
static int hf_ieee80211_vht_tx_stbc;
static int hf_ieee80211_vht_rx_stbc;
static int hf_ieee80211_vht_su_beamformer_cap;
static int hf_ieee80211_vht_su_beamformee_cap;
static int hf_ieee80211_vht_beamformer_antennas;
static int hf_ieee80211_vht_sounding_dimensions;
static int hf_ieee80211_vht_mu_beamformer_cap;
static int hf_ieee80211_vht_mu_beamformee_cap;
static int hf_ieee80211_vht_txop_ps;
static int hf_ieee80211_vht_var_htc_field;
static int hf_ieee80211_vht_max_ampdu;
static int hf_ieee80211_vht_link_adaptation_cap;
static int hf_ieee80211_vht_rx_pattern;
static int hf_ieee80211_vht_tx_pattern;
static int hf_ieee80211_vht_ext_nss_bw_support;

static int hf_ieee80211_vht_mcsset;

static int hf_ieee80211_vht_mcsset_rx_mcs_map;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_1_ss;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_2_ss;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_3_ss;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_4_ss;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_5_ss;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_6_ss;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_7_ss;
static int hf_ieee80211_vht_mcsset_rx_max_mcs_for_8_ss;

static int hf_ieee80211_vht_mcsset_max_nsts_total;
static int hf_ieee80211_vht_mcsset_rx_highest_long_gi;
static int hf_ieee80211_vht_mcsset_extended_nss_bw_capable;
static int hf_ieee80211_vht_mcsset_reserved;

static int hf_ieee80211_vht_mcsset_tx_mcs_map;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_1_ss;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_2_ss;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_3_ss;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_4_ss;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_5_ss;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_6_ss;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_7_ss;
static int hf_ieee80211_vht_mcsset_tx_max_mcs_for_8_ss;

static int hf_ieee80211_vht_op;
static int hf_ieee80211_vht_op_channel_width;
static int hf_ieee80211_vht_op_channel_center0;
static int hf_ieee80211_vht_op_channel_center1;

static int hf_ieee80211_vht_op_basic_mcs_map;
static int hf_ieee80211_vht_op_max_basic_mcs_for_1_ss;
static int hf_ieee80211_vht_op_max_basic_mcs_for_2_ss;
static int hf_ieee80211_vht_op_max_basic_mcs_for_3_ss;
static int hf_ieee80211_vht_op_max_basic_mcs_for_4_ss;
static int hf_ieee80211_vht_op_max_basic_mcs_for_5_ss;
static int hf_ieee80211_vht_op_max_basic_mcs_for_6_ss;
static int hf_ieee80211_vht_op_max_basic_mcs_for_7_ss;
static int hf_ieee80211_vht_op_max_basic_mcs_for_8_ss;
static int hf_ieee80211_vht_mcsset_tx_highest_long_gi;

static int hf_ieee80211_vht_tpe_pwr_info;
static int hf_ieee80211_vht_tpe_pwr_info_count;
static int hf_ieee80211_vht_tpe_pwr_info_unit;
static int hf_ieee80211_vht_tpe_pwr_info_category;
static int hf_ieee80211_vht_tpe_pwr_constr_20;
static int hf_ieee80211_vht_tpe_pwr_constr_40;
static int hf_ieee80211_vht_tpe_pwr_constr_80;
static int hf_ieee80211_vht_tpe_pwr_constr_160;
static int hf_ieee80211_vht_tpe_pwr_constr_320;
static int hf_ieee80211_vht_tpe_any_bw_psd;
static int hf_ieee80211_vht_tpe_psd;
static int hf_ieee80211_vht_tpe_ext_count;
static int hf_ieee80211_vht_tpe_ext_reserved;

static int hf_ieee80211_beamform_feedback_seg_retrans_bitmap;

static int hf_ieee80211_ndp_annc_token;
static int hf_ieee80211_ndp_annc_variant;
static int hf_ieee80211_ndp_annc_token_number;
static int hf_ieee80211_vht_ndp_annc_sta_info_aid12;
static int hf_ieee80211_vht_ndp_annc_sta_info_feedback_type;
static int hf_ieee80211_vht_ndp_annc_sta_info_nc_index;
static int hf_ieee80211_vht_ndp_annc_sta_info_reserved;

static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_aid11;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_ltf_offset;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_r2i_n_sts;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_r2i_rep;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_i2r_n_sts;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_reserved1;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_disambiguation;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_i2r_rep;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2008_reserved2;

static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2043;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2043_aid11;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2043_sac;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2043_disambiguation;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2043_reserved;

static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2044;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2044_aid11;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2044_partial_tsf;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2044_disambiguation;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2044_reserved;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2044_token;

static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2045;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2045_aid11;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2045_i2r_ndp_tx_power;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2045_r2i_ndp_target_rssi;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2045_disambiguation;
static int hf_ieee80211_vht_ndp_annc_sta_info_ranging_2045_reserved;


static int hf_ieee80211_ndp_eht_annc_sta_info;
static int hf_ieee80211_ndp_eht_annc_aid11;
static int hf_ieee80211_ndp_eht_annc_resolution;
static int hf_ieee80211_ndp_eht_annc_feedback_map;
static int hf_ieee80211_ndp_eht_annc_reserved1;
static int hf_ieee80211_ndp_eht_annc_nc_index;
static int hf_ieee80211_ndp_eht_annc_feedback_type;
static int hf_ieee80211_ndp_eht_annc_disambiguation;
static int hf_ieee80211_ndp_eht_annc_codebook_size;
static int hf_ieee80211_ndp_eht_annc_reserved2;

static int hf_ieee80211_ff_vht_action;
static int hf_ieee80211_ff_vht_mimo_cntrl;
static int hf_ieee80211_ff_vht_mimo_cntrl_nc_index;
static int hf_ieee80211_ff_vht_mimo_cntrl_nr_index;
static int hf_ieee80211_ff_vht_mimo_cntrl_channel_width;
static int hf_ieee80211_ff_vht_mimo_cntrl_grouping;
static int hf_ieee80211_ff_vht_mimo_cntrl_codebook_info;
static int hf_ieee80211_ff_vht_mimo_cntrl_feedback_type;
static int hf_ieee80211_ff_vht_mimo_cntrl_remaining_feedback_seg;
static int hf_ieee80211_ff_vht_mimo_cntrl_first_feedback_seg;
static int hf_ieee80211_ff_vht_mimo_cntrl_reserved;
static int hf_ieee80211_ff_vht_mimo_cntrl_sounding_dialog_token_number;

static int * const hf_ieee80211_ff_vht_mimo_cntrl_fields[] = {
  &hf_ieee80211_ff_vht_mimo_cntrl_nc_index,
  &hf_ieee80211_ff_vht_mimo_cntrl_nr_index,
  &hf_ieee80211_ff_vht_mimo_cntrl_channel_width,
  &hf_ieee80211_ff_vht_mimo_cntrl_grouping,
  &hf_ieee80211_ff_vht_mimo_cntrl_codebook_info,
  &hf_ieee80211_ff_vht_mimo_cntrl_feedback_type,
  &hf_ieee80211_ff_vht_mimo_cntrl_remaining_feedback_seg,
  &hf_ieee80211_ff_vht_mimo_cntrl_first_feedback_seg,
  &hf_ieee80211_ff_vht_mimo_cntrl_reserved,
  &hf_ieee80211_ff_vht_mimo_cntrl_sounding_dialog_token_number,
  NULL,
};

static int hf_ieee80211_vht_compressed_beamforming_report;
static int hf_ieee80211_vht_compressed_beamforming_report_snr;
static int hf_ieee80211_vht_compressed_beamform_scidx;
static int hf_ieee80211_vht_group_id_management;
static int hf_ieee80211_vht_membership_status_array;
static int hf_ieee80211_vht_user_position_array;
static int hf_ieee80211_vht_operation_mode_notification;
static int hf_ieee80211_vht_membership_status_field;
static int hf_ieee80211_vht_user_position_field;
static int hf_ieee80211_vht_mu_exclusive_beamforming_report;
static int hf_ieee80211_vht_mu_exclusive_beamforming_delta_snr;

static int hf_ieee80211_ff_he_action;
static int hf_ieee80211_ff_protected_he_action;
static int hf_ieee80211_ff_protected_ftm_action;
static int hf_ieee80211_ff_eht_action;
static int hf_ieee80211_ff_protected_eht_action;
static int hf_ieee80211_he_mimo_control_nc_index;
static int hf_ieee80211_he_mimo_control_nr_index;
static int hf_ieee80211_he_mimo_control_bw;
static int hf_ieee80211_he_mimo_control_grouping;
static int hf_ieee80211_he_mimo_control_codebook_info;
static int hf_ieee80211_he_mimo_control_feedback_type;
static int hf_ieee80211_he_mimo_control_remaining_feedback_segs;
static int hf_ieee80211_he_mimo_control_first_feedback_seg;
static int hf_ieee80211_he_mimo_control_ru_start_index;
static int hf_ieee80211_he_mimo_control_ru_end_index;
static int hf_ieee80211_he_mimo_control_sounding_dialog_token_num;
static int hf_ieee80211_he_mimo_control_reserved;
static int hf_ieee80211_he_mimo_control_field;
static int hf_ieee80211_he_compressed_beamforming_report_snr;
static int hf_ieee80211_he_compressed_beamform_scidx;
static int hf_ieee80211_he_beamforming_report_len;

static int hf_ieee80211_tag_neighbor_report_bssid;
static int hf_ieee80211_tag_neighbor_report_bssid_info;
static int hf_ieee80211_tag_neighbor_report_bssid_info_reachability;
static int hf_ieee80211_tag_neighbor_report_bssid_info_security;
static int hf_ieee80211_tag_neighbor_report_bssid_info_key_scope;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_spec_mng;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_qos;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_apsd;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_radio_msnt;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_reserved_b4;
static int hf_ieee80211_tag_neighbor_report_bssid_info_capability_reserved_b5;
static int hf_ieee80211_tag_neighbor_report_bssid_info_mobility_domain;
static int hf_ieee80211_tag_neighbor_report_bssid_info_high_throughput;
static int hf_ieee80211_tag_neighbor_report_bssid_info_very_high_throughput;
static int hf_ieee80211_tag_neighbor_report_bssid_info_ftm;
static int hf_ieee80211_tag_neighbor_report_bssid_info_high_efficiency;
static int hf_ieee80211_tag_neighbor_report_bssid_info_er_bss;
static int hf_ieee80211_tag_neighbor_report_bssid_info_colocated_ap;
static int hf_ieee80211_tag_neighbor_report_bssid_info_unsolicited_probe_responses_active;
static int hf_ieee80211_tag_neighbor_report_bssid_info_ess_with_colocated_ap;
static int hf_ieee80211_tag_neighbor_report_bssid_info_oct_supported_with_reporting_ap;
static int hf_ieee80211_tag_neighbor_report_bssid_info_colocated_6ghz_ap;
static int hf_ieee80211_tag_neighbor_report_bssid_info_eht;
static int hf_ieee80211_tag_neighbor_report_bssid_info_dmg_positioning;
static int hf_ieee80211_tag_neighbor_report_bssid_info_reserved;
static int hf_ieee80211_tag_neighbor_report_ope_class;
static int hf_ieee80211_tag_neighbor_report_channel_number;
static int hf_ieee80211_tag_neighbor_report_phy_type;
static int hf_ieee80211_tag_neighbor_report_subelement_id;
static int hf_ieee80211_tag_neighbor_report_subelement_length;
static int hf_ieee80211_tag_neighbor_report_subelement_data;
static int hf_ieee80211_tag_neighbor_report_subelement_bss_trn_can_pref;
static int hf_ieee80211_tag_neighbor_report_subelement_bss_ter_tsf;
static int hf_ieee80211_tag_neighbor_report_subelement_bss_dur;
static int hf_ieee80211_tag_neighbor_report_subelement_tsf_offset;
static int hf_ieee80211_tag_neighbor_report_subelement_beacon_interval;
static int hf_ieee80211_tag_neighbor_report_subelement_country_code;
static int hf_ieee80211_tag_supported_ope_classes_current;
static int hf_ieee80211_tag_supported_ope_classes_alternate;

static int hf_ieee80211_tag_dirn_meas_results_aoa_results;
static int hf_ieee80211_tag_ftm_aoa_results_aoa_azimuth;
static int hf_ieee80211_tag_ftm_aoa_results_aoa_elevation;
static int hf_ieee80211_tag_ftm_aoa_results_aoa_azimuth_accuracy;
static int hf_ieee80211_tag_ftm_aoa_results_aoa_elevation_accuracy;
static int hf_ieee80211_tag_ftm_aoa_results_best_awv_id;
static int hf_ieee80211_tag_ftm_aoa_results_aoa_reference;
static int hf_ieee80211_tag_ftm_aoa_results_reserved;

/* IEEE Std 802.11r-2008 7.3.2.47 */
static int hf_ieee80211_tag_mobility_domain_mdid;
static int hf_ieee80211_tag_mobility_domain_ft_capab;
static int hf_ieee80211_tag_mobility_domain_ft_capab_ft_over_ds;
static int hf_ieee80211_tag_mobility_domain_ft_capab_resource_req;
static int hf_ieee80211_tag_mobility_domain_ft_capab_reserved;
static int * const ieee80211_tag_mobility_domain_ft_capab_fields[] = {
  &hf_ieee80211_tag_mobility_domain_ft_capab_ft_over_ds,
  &hf_ieee80211_tag_mobility_domain_ft_capab_resource_req,
  &hf_ieee80211_tag_mobility_domain_ft_capab_reserved,
  NULL
};

/* IEEE Std 802.11r-2008 7.3.2.48 */
static int hf_ieee80211_tag_ft_mic_control;
static int hf_ieee80211_tag_ft_mic_control_rsnxe_used;
static int hf_ieee80211_tag_ft_mic_control_mic_length;
static int hf_ieee80211_tag_ft_mic_control_reserved;
static int hf_ieee80211_tag_ft_mic_control_element_count;
static int * const ieee80211_tag_ft_mic_control_fields[] = {
  &hf_ieee80211_tag_ft_mic_control_rsnxe_used,
  &hf_ieee80211_tag_ft_mic_control_mic_length,
  &hf_ieee80211_tag_ft_mic_control_reserved,
  &hf_ieee80211_tag_ft_mic_control_element_count,
  NULL
};
static int hf_ieee80211_tag_ft_mic;
static int hf_ieee80211_tag_ft_anonce;
static int hf_ieee80211_tag_ft_snonce;
static int hf_ieee80211_tag_ft_subelem_id;
static int hf_ieee80211_tag_ft_subelem_len;
static int hf_ieee80211_tag_ft_subelem_data;
static int hf_ieee80211_tag_ft_subelem_r1kh_id;
static int hf_ieee80211_tag_ft_subelem_gtk_key_info;
static int hf_ieee80211_tag_ft_subelem_gtk_key_id;
static int hf_ieee80211_tag_ft_subelem_gtk_key_length;
static int hf_ieee80211_tag_ft_subelem_gtk_rsc;
static int hf_ieee80211_tag_ft_subelem_gtk_key;
static int hf_ieee80211_tag_ft_subelem_gtk_key_encrypted;
static int hf_ieee80211_tag_ft_subelem_r0kh_id;
static int hf_ieee80211_tag_ft_subelem_igtk_key_id;
static int hf_ieee80211_tag_ft_subelem_igtk_ipn;
static int hf_ieee80211_tag_ft_subelem_igtk_key_length;
static int hf_ieee80211_tag_ft_subelem_igtk_key;
static int hf_ieee80211_tag_ft_subelem_oci_op_class;
static int hf_ieee80211_tag_ft_subelem_oci_prim_chan_num;
static int hf_ieee80211_tag_ft_subelem_oci_freq_seg_1;
static int hf_ieee80211_tag_ft_subelem_oci_oct_op_class;
static int hf_ieee80211_tag_ft_subelem_oci_oct_prim_chan_num;
static int hf_ieee80211_tag_ft_subelem_oci_oct_freq_seg_1;
static int hf_ieee80211_tag_ft_subelem_bigtk_key_id;
static int hf_ieee80211_tag_ft_subelem_bigtk_bipn;
static int hf_ieee80211_tag_ft_subelem_bigtk_key_length;
static int hf_ieee80211_tag_ft_subelem_bigtk_key;
static int hf_ieee80211_tag_ft_subelem_mlo_gtk_key_info;
static int hf_ieee80211_tag_ft_subelem_mlo_gtk_key_id;
static int hf_ieee80211_tag_ft_subelem_mlo_gtk_link_id_info;
static int hf_ieee80211_tag_ft_subelem_mlo_gtk_link_id;
static int hf_ieee80211_tag_ft_subelem_mlo_gtk_key_length;
static int hf_ieee80211_tag_ft_subelem_mlo_gtk_rsc;
static int hf_ieee80211_tag_ft_subelem_mlo_gtk_key;
static int hf_ieee80211_tag_ft_subelem_mlo_igtk_key_id;
static int hf_ieee80211_tag_ft_subelem_mlo_igtk_ipn;
static int hf_ieee80211_tag_ft_subelem_mlo_igtk_link_id_info;
static int hf_ieee80211_tag_ft_subelem_mlo_igtk_link_id;
static int hf_ieee80211_tag_ft_subelem_mlo_igtk_key_length;
static int hf_ieee80211_tag_ft_subelem_mlo_igtk_key;
static int hf_ieee80211_tag_ft_subelem_mlo_bigtk_key_id;
static int hf_ieee80211_tag_ft_subelem_mlo_bigtk_bipn;
static int hf_ieee80211_tag_ft_subelem_mlo_bigtk_link_id_info;
static int hf_ieee80211_tag_ft_subelem_mlo_bigtk_link_id;
static int hf_ieee80211_tag_ft_subelem_mlo_bigtk_key_length;
static int hf_ieee80211_tag_ft_subelem_mlo_bigtk_key;

/* IEEE Std 802.11-2012: 11r 8.4.2.52 */
static int hf_ieee80211_tag_ric_data_id;
static int hf_ieee80211_tag_ric_data_desc_cnt;
static int hf_ieee80211_tag_ric_data_status_code;

/* IEEE Std 802.11-2012: 11r 8.4.2.53 */
static int hf_ieee80211_tag_ric_desc_rsrc_type;
static int hf_ieee80211_tag_ric_desc_var_params;

/* IEEE Std 802.11w-2009 7.3.2.55 */
static int hf_ieee80211_tag_mmie_keyid;
static int hf_ieee80211_tag_mmie_ipn;
static int hf_ieee80211_tag_mmie_mic;

/* IEEE Std 802.11-2016: 9.4.2.72 */
static int hf_ieee80211_tag_no_bssid_capability_dmg_bss_control;
static int hf_ieee80211_tag_no_bssid_capability_dmg_bss_control_type;
static int hf_ieee80211_tag_no_bssid_capability_dmg_bss_control_reserved;

/* IEEE Std 802.11-2016: 9.4.2.74 */
static int hf_ieee80211_tag_multiple_bssid_index_bssid_index;
static int hf_ieee80211_tag_multiple_bssid_index_dtim_period;
static int hf_ieee80211_tag_multiple_bssid_index_dtim_count;

/* IEEE Std 802.11-2012: 8.4.2.61 */
static int hf_ieee80211_tag_obss_spd;
static int hf_ieee80211_tag_obss_sad;
static int hf_ieee80211_tag_obss_cwtsi;
static int hf_ieee80211_tag_obss_sptpc;
static int hf_ieee80211_tag_obss_satpc;
static int hf_ieee80211_tag_obss_wctdf;
static int hf_ieee80211_tag_obss_sat;

/* IEEE Std 802.11-2012: 8.4.2.25.1 */
static int hf_ieee80211_group_data_cipher_suite_oui;
static int hf_ieee80211_group_data_cipher_suite_type;
static int hf_ieee80211_osen_pairwise_cipher_suite_oui;
static int hf_ieee80211_osen_pairwise_cipher_suite_type;
static int hf_ieee80211_osen_pcs_count;
static int hf_ieee80211_osen_akm_count;
static int hf_ieee80211_osen_akm_cipher_suite_oui;
static int hf_ieee80211_osen_akm_cipher_suite_type;
static int hf_ieee80211_osen_rsn_cap_preauth;
static int hf_ieee80211_osen_rsn_cap_no_pairwise;
static int hf_ieee80211_osen_rsn_cap_ptksa_replay_counter;
static int hf_ieee80211_osen_rsn_cap_gtksa_replay_counter;
static int hf_ieee80211_osen_rsn_cap_mfpr;
static int hf_ieee80211_osen_rsn_cap_mfpc;
static int hf_ieee80211_osen_rsn_cap_jmr;
static int hf_ieee80211_osen_rsn_cap_peerkey;
static int hf_ieee80211_osen_rsn_spp_a_msdu_capable;
static int hf_ieee80211_osen_rsn_spp_a_msdu_required;
static int hf_ieee80211_osen_rsn_pbac;
static int hf_ieee80211_osen_extended_key_id_iaf;
static int hf_ieee80211_osen_reserved;
static int hf_ieee80211_osen_rsn_cap_flags;
static int hf_ieee80211_osen_pmkid_count;
static int hf_ieee80211_osen_pmkid;
static int hf_ieee80211_osen_group_management_cipher_suite_oui;
static int hf_ieee80211_osen_group_management_cipher_suite_type;

/*WAPI-Specification 7.3.2.25 : WAPI Parameter Set*/
static int hf_ieee80211_tag_wapi_param_set_version;

static int hf_ieee80211_tag_wapi_param_set_akm_suite_count;
static int hf_ieee80211_tag_wapi_param_set_akm_suite_oui;
static int hf_ieee80211_tag_wapi_param_set_akm_suite_type;

static int hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_count;
static int hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_oui;
static int hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_type;

static int hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_oui;
static int hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_type;

static int hf_ieee80211_tag_wapi_param_set_capab;
static int hf_ieee80211_tag_wapi_param_set_capab_preauth;
static int hf_ieee80211_tag_wapi_param_set_capab_rsvd;
static int hf_ieee80211_tag_wapi_param_set_bkid_count;
static int hf_ieee80211_tag_wapi_param_set_bkid_list;

/* IEEE Std 802.11v-2011 7.3.2.61 */
static int hf_ieee80211_tag_time_adv_timing_capab;
static int hf_ieee80211_tag_time_adv_time_value;
static int hf_ieee80211_tag_time_adv_time_value_year;
static int hf_ieee80211_tag_time_adv_time_value_month;
static int hf_ieee80211_tag_time_adv_time_value_day;
static int hf_ieee80211_tag_time_adv_time_value_hours;
static int hf_ieee80211_tag_time_adv_time_value_minutes;
static int hf_ieee80211_tag_time_adv_time_value_seconds;
static int hf_ieee80211_tag_time_adv_time_value_milliseconds;
static int hf_ieee80211_tag_time_adv_time_value_reserved;
static int hf_ieee80211_tag_time_adv_time_error;
static int hf_ieee80211_tag_time_adv_time_update_counter;

/* IEEE Std 802.11-2012 8.4.2.81 */
static int hf_ieee80211_tag_bss_max_idle_period;
static int hf_ieee80211_tag_bss_max_idle_options;
static int hf_ieee80211_tag_bss_max_idle_options_protected;
static int hf_ieee80211_tag_bss_idle_options_reserved;

/* IEEE Std 802.11-2012 8.4.2.82 */
static int hf_ieee80211_tag_tfs_request_id;
static int hf_ieee80211_tag_tfs_request_ac_delete_after_match;
static int hf_ieee80211_tag_tfs_request_ac_notify;
static int hf_ieee80211_tag_tfs_request_subelem_id;
static int hf_ieee80211_tag_tfs_request_subelem_len;
static int hf_ieee80211_tag_tfs_request_subelem;

/* IEEE Std 802.11-2012 8.4.2.83 */
static int hf_ieee80211_tag_tfs_response_subelem_id;
static int hf_ieee80211_tag_tfs_response_subelem_len;
static int hf_ieee80211_tag_tfs_response_subelem;
static int hf_ieee80211_tag_tfs_response_status;
static int hf_ieee80211_tag_tfs_response_id;

/* IEEE Std 802.11-2012 8.4.2.84 */
static int hf_ieee80211_tag_wnm_sleep_mode_action_type;
static int hf_ieee80211_tag_wnm_sleep_mode_response_status;
static int hf_ieee80211_tag_wnm_sleep_mode_interval;

static int hf_ieee80211_wnm_sub_elt_id;
static int hf_ieee80211_wnm_sub_elt_len;

/* IEEE Std 802.11v-2011 7.3.2.87 */
static int hf_ieee80211_tag_time_zone;

/* IEEE Std 802.11u-2011 7.3.2.92 */
static int hf_ieee80211_tag_interworking_access_network_type;
static int hf_ieee80211_tag_interworking_internet;
static int hf_ieee80211_tag_interworking_asra;
static int hf_ieee80211_tag_interworking_esr;
static int hf_ieee80211_tag_interworking_uesa;
static int hf_ieee80211_tag_interworking_hessid;

/* IEEE Std 802.11-2012, 8.4.2.97 */
static int hf_ieee80211_tag_qos_map_set_dscp_exc;
static int hf_ieee80211_tag_qos_map_set_dscp_exc_val;
static int hf_ieee80211_tag_qos_map_set_dscp_exc_up;
static int hf_ieee80211_tag_qos_map_set_range;
static int hf_ieee80211_tag_qos_map_set_low;
static int hf_ieee80211_tag_qos_map_set_high;

/* IEEE Std 802.11u-2011 7.3.2.93 */
static int hf_ieee80211_tag_adv_proto_resp_len_limit;
static int hf_ieee80211_tag_adv_proto_pame_bi;
static int hf_ieee80211_tag_adv_proto_id;
static int hf_ieee80211_tag_adv_vs_len;
/* static int hf_ieee80211_tag_adv_proto_vs_info; */

/* IEEE Std 802.11u-2011 7.3.2.96 */
static int hf_ieee80211_tag_roaming_consortium_num_anqp_oi;
static int hf_ieee80211_tag_roaming_consortium_oi1_len;
static int hf_ieee80211_tag_roaming_consortium_oi2_len;
static int hf_ieee80211_tag_roaming_consortium_oi1;
static int hf_ieee80211_tag_roaming_consortium_oi2;
static int hf_ieee80211_tag_roaming_consortium_oi3;

/* 802.11n 7.3.2.48 */
static int hf_ieee80211_hta_cc;
static int hf_ieee80211_hta_cap1;
static int hf_ieee80211_hta_cap2;
static int hf_ieee80211_hta_ext_chan_offset;
static int hf_ieee80211_hta_rec_tx_width;
static int hf_ieee80211_hta_rifs_mode;
static int hf_ieee80211_hta_controlled_access;
static int hf_ieee80211_hta_service_interval;
static int hf_ieee80211_hta_operating_mode;
static int hf_ieee80211_hta_non_gf_devices;
static int hf_ieee80211_hta_basic_stbc_mcs;
static int hf_ieee80211_hta_dual_stbc_protection;
static int hf_ieee80211_hta_secondary_beacon;
static int hf_ieee80211_hta_lsig_txop_protection;
static int hf_ieee80211_hta_pco_active;
static int hf_ieee80211_hta_pco_phase;

static int hf_ieee80211_antsel;
static int hf_ieee80211_antsel_vs;
static int hf_ieee80211_antsel_b0;
static int hf_ieee80211_antsel_b1;
static int hf_ieee80211_antsel_b2;
static int hf_ieee80211_antsel_b3;
static int hf_ieee80211_antsel_b4;
static int hf_ieee80211_antsel_b5;
static int hf_ieee80211_antsel_b6;
static int hf_ieee80211_antsel_b7;

static int hf_ieee80211_rsn_version;
static int hf_ieee80211_rsn_gcs;
static int hf_ieee80211_rsn_gcs_oui;
static int hf_ieee80211_rsn_gcs_type;
static int hf_ieee80211_rsn_gcs_80211_type;
static int hf_ieee80211_rsn_pcs_count;
static int hf_ieee80211_rsn_pcs_list;
static int hf_ieee80211_rsn_pcs;
static int hf_ieee80211_rsn_pcs_oui;
static int hf_ieee80211_rsn_pcs_80211_type;
static int hf_ieee80211_rsn_pcs_type;
static int hf_ieee80211_rsn_akms_count;
static int hf_ieee80211_rsn_akms_list;
static int hf_ieee80211_rsn_akms;
static int hf_ieee80211_rsn_akms_oui;
static int hf_ieee80211_rsn_akms_80211_type;
static int hf_ieee80211_rsn_akms_type;
static int hf_ieee80211_rsn_cap;
static int hf_ieee80211_rsn_cap_preauth;
static int hf_ieee80211_rsn_cap_no_pairwise;
static int hf_ieee80211_rsn_cap_ptksa_replay_counter;
static int hf_ieee80211_rsn_cap_gtksa_replay_counter;
static int hf_ieee80211_rsn_cap_mfpr;
static int hf_ieee80211_rsn_cap_mfpc;
static int hf_ieee80211_rsn_cap_jmr;
static int hf_ieee80211_rsn_cap_peerkey;
static int hf_ieee80211_rsn_cap_spp_amsdu_cap;
static int hf_ieee80211_rsn_cap_spp_amsdu_req;
static int hf_ieee80211_rsn_cap_pbac;
static int hf_ieee80211_rsn_cap_extended_key_id_iaf;
static int hf_ieee80211_rsn_cap_ocvc;
static int hf_ieee80211_rsn_pmkid_count;
static int hf_ieee80211_rsn_pmkid_list;
static int hf_ieee80211_rsn_pmkid;
static int hf_ieee80211_rsn_gmcs;
static int hf_ieee80211_rsn_gmcs_oui;
static int hf_ieee80211_rsn_gmcs_type;
static int hf_ieee80211_rsn_gmcs_80211_type;

static int hf_ieee80211_wfa_ie_type;
static int hf_ieee80211_wfa_ie_wpa_version;
static int hf_ieee80211_wfa_ie_wpa_mcs;
static int hf_ieee80211_wfa_ie_wpa_mcs_oui;
static int hf_ieee80211_wfa_ie_wpa_mcs_type;
static int hf_ieee80211_wfa_ie_wpa_mcs_wfa_type;
static int hf_ieee80211_wfa_ie_wpa_ucs_count;
static int hf_ieee80211_wfa_ie_wpa_ucs_list;
static int hf_ieee80211_wfa_ie_wpa_ucs;
static int hf_ieee80211_wfa_ie_wpa_ucs_oui;
static int hf_ieee80211_wfa_ie_wpa_ucs_wfa_type;
static int hf_ieee80211_wfa_ie_wpa_ucs_type;
static int hf_ieee80211_wfa_ie_wpa_akms_count;
static int hf_ieee80211_wfa_ie_wpa_akms_list;
static int hf_ieee80211_wfa_ie_wpa_akms;
static int hf_ieee80211_wfa_ie_wpa_akms_oui;
static int hf_ieee80211_wfa_ie_wpa_akms_wfa_type;
static int hf_ieee80211_wfa_ie_wpa_akms_type;
static int hf_ieee80211_wfa_ie_wme_subtype;
static int hf_ieee80211_wfa_ie_wme_version;
static int hf_ieee80211_wfa_ie_wme_qos_info;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_max_sp_length;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_be;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_bk;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vi;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vo;
static int hf_ieee80211_wfa_ie_wme_qos_info_sta_reserved;
static int hf_ieee80211_wfa_ie_wme_qos_info_ap_u_apsd;
static int hf_ieee80211_wfa_ie_wme_qos_info_ap_parameter_set_count;
static int hf_ieee80211_wfa_ie_wme_qos_info_ap_reserved;
static int hf_ieee80211_wfa_ie_wme_reserved;
static int hf_ieee80211_wfa_ie_wme_ac_parameters;
static int hf_ieee80211_wfa_ie_wme_acp_aci_aifsn;
static int hf_ieee80211_wfa_ie_wme_acp_aci_be;
static int hf_ieee80211_wfa_ie_wme_acp_acm_be;
static int hf_ieee80211_wfa_ie_wme_acp_aifsn_be;
static int hf_ieee80211_wfa_ie_wme_acp_reserved_be;
static int hf_ieee80211_wfa_ie_wme_acp_aci_bk;
static int hf_ieee80211_wfa_ie_wme_acp_acm_bk;
static int hf_ieee80211_wfa_ie_wme_acp_aifsn_bk;
static int hf_ieee80211_wfa_ie_wme_acp_reserved_bk;
static int hf_ieee80211_wfa_ie_wme_acp_aci_vi;
static int hf_ieee80211_wfa_ie_wme_acp_acm_vi;
static int hf_ieee80211_wfa_ie_wme_acp_aifsn_vi;
static int hf_ieee80211_wfa_ie_wme_acp_reserved_vi;
static int hf_ieee80211_wfa_ie_wme_acp_aci_vo;
static int hf_ieee80211_wfa_ie_wme_acp_acm_vo;
static int hf_ieee80211_wfa_ie_wme_acp_aifsn_vo;
static int hf_ieee80211_wfa_ie_wme_acp_reserved_vo;
static int hf_ieee80211_wfa_ie_wme_acp_ecw;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_max_be;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_max_bk;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_max_vo;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_max_vi;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_min_be;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_min_bk;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_min_vo;
static int hf_ieee80211_wfa_ie_wme_acp_ecw_min_vi;
static int hf_ieee80211_wfa_ie_wme_acp_txop_limit_be;
static int hf_ieee80211_wfa_ie_wme_acp_txop_limit_bk;
static int hf_ieee80211_wfa_ie_wme_acp_txop_limit_vo;
static int hf_ieee80211_wfa_ie_wme_acp_txop_limit_vi;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_tid;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_direction;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_psb;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_up;
static int hf_ieee80211_wfa_ie_wme_tspec_tsinfo_reserved;
static int hf_ieee80211_wfa_ie_wme_tspec_nor_msdu;
static int hf_ieee80211_wfa_ie_wme_tspec_max_msdu;
static int hf_ieee80211_wfa_ie_wme_tspec_min_srv;
static int hf_ieee80211_wfa_ie_wme_tspec_max_srv;
static int hf_ieee80211_wfa_ie_wme_tspec_inact_int;
static int hf_ieee80211_wfa_ie_wme_tspec_susp_int;
static int hf_ieee80211_wfa_ie_wme_tspec_srv_start;
static int hf_ieee80211_wfa_ie_wme_tspec_min_data;
static int hf_ieee80211_wfa_ie_wme_tspec_mean_data;
static int hf_ieee80211_wfa_ie_wme_tspec_peak_data;
static int hf_ieee80211_wfa_ie_wme_tspec_burst_size;
static int hf_ieee80211_wfa_ie_wme_tspec_delay_bound;
static int hf_ieee80211_wfa_ie_wme_tspec_min_phy;
static int hf_ieee80211_wfa_ie_wme_tspec_surplus;
static int hf_ieee80211_wfa_ie_wme_tspec_medium;
static int hf_ieee80211_wfa_ie_nc_cost_level;
static int hf_ieee80211_wfa_ie_nc_reserved;
static int hf_ieee80211_wfa_ie_nc_cost_flags;
static int hf_ieee80211_wfa_ie_tethering_type;
static int hf_ieee80211_wfa_ie_tethering_mac_length;
static int hf_ieee80211_wfa_ie_tethering_mac;
static int hf_ieee80211_wfa_ie_owe_bssid;
static int hf_ieee80211_wfa_ie_owe_ssid_length;
static int hf_ieee80211_wfa_ie_owe_ssid;
static int hf_ieee80211_wfa_ie_owe_band_info;
static int hf_ieee80211_wfa_ie_owe_channel_info;
static int hf_ieee80211_wfa_ie_mbo_oce_attr;
static int hf_ieee80211_wfa_ie_mbo_oce_attr_id;
static int hf_ieee80211_wfa_ie_mbo_oce_attr_len;
static int hf_ieee80211_wfa_ie_mbo_ap_cap;
static int hf_ieee80211_wfa_ie_mbo_ap_cap_cell;
static int hf_ieee80211_wfa_ie_mbo_ap_cap_reserved;
static int hf_ieee80211_wfa_ie_mbo_non_pref_chan_op_class;
static int hf_ieee80211_wfa_ie_mbo_non_pref_chan_chan;
static int hf_ieee80211_wfa_ie_mbo_non_pref_chan_pref;
static int hf_ieee80211_wfa_ie_mbo_non_pref_chan_reason;
static int hf_ieee80211_wfa_ie_mbo_cellular_cap;
static int hf_ieee80211_wfa_ie_mbo_assoc_disallow_reason;
static int hf_ieee80211_wfa_ie_mbo_cellular_pref;
static int hf_ieee80211_wfa_ie_mbo_transition_reason;
static int hf_ieee80211_wfa_ie_mbo_transition_rej_reason;
static int hf_ieee80211_wfa_ie_mbo_assoc_retry_delay;
static int hf_ieee80211_wfa_ie_oce_cap_ctrl;
static int hf_ieee80211_wfa_ie_oce_cap_release;
static int hf_ieee80211_wfa_ie_oce_cap_sta_cfon;
static int hf_ieee80211_wfa_ie_oce_cap_11b_only_ap;
static int hf_ieee80211_wfa_ie_oce_cap_hlp;
static int hf_ieee80211_wfa_ie_oce_cap_non_oce_ap;
static int hf_ieee80211_wfa_ie_oce_cap_reserved;
static int hf_ieee80211_wfa_ie_oce_rssi_assoc_rej_delta;
static int hf_ieee80211_wfa_ie_oce_rssi_assoc_rej_delay;
static int hf_ieee80211_wfa_ie_oce_wan_metrics_avail_cap;
static int hf_ieee80211_wfa_ie_oce_wan_metrics_avail_cap_downlink;
static int hf_ieee80211_wfa_ie_oce_wan_metrics_avail_cap_uplink;
static int hf_ieee80211_wfa_ie_oce_rnr_completeness_short_ssid;
static int hf_ieee80211_wfa_ie_oce_probe_suppr_bssid;
static int hf_ieee80211_wfa_ie_oce_probe_suppr_ssid;
static int hf_ieee80211_wfa_anqp_mbo_subtype;
static int hf_ieee80211_wfa_anqp_mbo_query;
static int hf_ieee80211_wfa_anqp_mbo_cellular_pref;
static int hf_ieee80211_wfa_ie_transition_disable_bitmap;
static int hf_ieee80211_wfa_ie_transition_disable_wpa3_personal;
static int hf_ieee80211_wfa_ie_transition_disable_sae_pk;
static int hf_ieee80211_wfa_ie_transition_disable_wpa3_enterprise;
static int hf_ieee80211_wfa_ie_transition_disable_enhanced_open;
static int hf_ieee80211_wfa_ie_transition_disable_reserved_b4thru7;
static int hf_ieee80211_wfa_ie_transition_disable_reserved;

static int hf_ieee80211_aironet_ie_type;
static int hf_ieee80211_aironet_ie_dtpc;
static int hf_ieee80211_aironet_ie_dtpc_unknown;
static int hf_ieee80211_aironet_ie_version;
static int hf_ieee80211_aironet_ie_data;
static int hf_ieee80211_aironet_ie_qos_reserved;
static int hf_ieee80211_aironet_ie_qos_paramset;
static int hf_ieee80211_aironet_ie_qos_val;
static int hf_ieee80211_aironet_ie_clientmfp;

static int hf_ieee80211_vs_sgdsn_tag;
static int hf_ieee80211_vs_sgdsn_type;
static int hf_ieee80211_vs_sgdsn_length;
static int hf_ieee80211_vs_sgdsn_version;
static int hf_ieee80211_vs_sgdsn_icaomfrcode;
static int hf_ieee80211_vs_sgdsn_manufacturer;
static int hf_ieee80211_vs_sgdsn_model;
static int hf_ieee80211_vs_sgdsn_serialnumber;
static int hf_ieee80211_vs_sgdsn_serialnumber_len;
static int hf_ieee80211_vs_sgdsn_gpscoord;
static int hf_ieee80211_vs_sgdsn_altitude;
static int hf_ieee80211_vs_sgdsn_speed;
static int hf_ieee80211_vs_sgdsn_heading;

static int hf_ieee80211_vs_nintendo_type;
static int hf_ieee80211_vs_nintendo_length;
static int hf_ieee80211_vs_nintendo_servicelist;
static int hf_ieee80211_vs_nintendo_service;
static int hf_ieee80211_vs_nintendo_consoleid;
static int hf_ieee80211_vs_nintendo_unknown;

static int hf_ieee80211_vs_aruba_subtype;
static int hf_ieee80211_vs_aruba_apname;
static int hf_ieee80211_vs_aruba_data;
static int hf_ieee80211_vs_aruba_gps_length;
static int hf_ieee80211_vs_aruba_gps_subversion;
static int hf_ieee80211_vs_aruba_gps_hop;
static int hf_ieee80211_vs_aruba_gps_latitude;
static int hf_ieee80211_vs_aruba_gps_longitude;
static int hf_ieee80211_vs_aruba_gps_major_axis;
static int hf_ieee80211_vs_aruba_gps_minor_axis;
static int hf_ieee80211_vs_aruba_gps_orientation;
static int hf_ieee80211_vs_aruba_gps_distance;

static int hf_ieee80211_vs_routerboard_unknown;
static int hf_ieee80211_vs_routerboard_subitem;
static int hf_ieee80211_vs_routerboard_subtype;
static int hf_ieee80211_vs_routerboard_sublength;
static int hf_ieee80211_vs_routerboard_subdata;
static int hf_ieee80211_vs_routerboard_subtype1_prefix;
static int hf_ieee80211_vs_routerboard_subtype1_data;

static int hf_ieee80211_vs_meru_subitem;
static int hf_ieee80211_vs_meru_subtype;
static int hf_ieee80211_vs_meru_sublength;
static int hf_ieee80211_vs_meru_subdata;

static int hf_ieee80211_vs_extreme_subtype;
static int hf_ieee80211_vs_extreme_subdata;
static int hf_ieee80211_vs_extreme_unknown;
static int hf_ieee80211_vs_extreme_ap_length;
static int hf_ieee80211_vs_extreme_ap_name;

static int hf_ieee80211_vs_aerohive_version;
static int hf_ieee80211_vs_aerohive_subtype;
static int hf_ieee80211_vs_aerohive_hostname_length;
static int hf_ieee80211_vs_aerohive_hostname;
static int hf_ieee80211_vs_aerohive_data;

static int hf_ieee80211_vs_mist_ap_name;
static int hf_ieee80211_vs_mist_data;

static int hf_ieee80211_vs_ruckus_ap_name;
static int hf_ieee80211_vs_ruckus_data;

static int hf_ieee80211_vs_alcatel_ap_name;
static int hf_ieee80211_vs_alcatel_data;

static int hf_ieee80211_vs_fortinet_subtype;
static int hf_ieee80211_vs_fortinet_system_type;
static int hf_ieee80211_vs_fortinet_system_length;
static int hf_ieee80211_vs_fortinet_system_apname;
static int hf_ieee80211_vs_fortinet_system_apmodel;
static int hf_ieee80211_vs_fortinet_system_apserial;
static int hf_ieee80211_vs_fortinet_data;

static int hf_ieee80211_vs_arista_subtype;
static int hf_ieee80211_vs_arista_apname;
static int hf_ieee80211_vs_arista_data;

static int hf_ieee80211_vs_wisun_type;
static int hf_ieee80211_vs_wisun_ptkid;
static int hf_ieee80211_vs_wisun_gtkl;
static int hf_ieee80211_vs_wisun_gtkl_gtk0;
static int hf_ieee80211_vs_wisun_gtkl_gtk1;
static int hf_ieee80211_vs_wisun_gtkl_gtk2;
static int hf_ieee80211_vs_wisun_gtkl_gtk3;
static int hf_ieee80211_vs_wisun_nr;
static int hf_ieee80211_vs_wisun_lgtkl;
static int hf_ieee80211_vs_wisun_lgtkl_lgtk0;
static int hf_ieee80211_vs_wisun_lgtkl_lgtk1;
static int hf_ieee80211_vs_wisun_lgtkl_lgtk2;
static int hf_ieee80211_vs_wisun_lgtk_key_id;
static int hf_ieee80211_vs_wisun_lgtk_lgtk;
static int hf_ieee80211_vs_wisun_data;

static int hf_ieee80211_vs_apple_type;
static int hf_ieee80211_vs_apple_subtype;
static int hf_ieee80211_vs_apple_length;
static int hf_ieee80211_vs_apple_data;

static int hf_ieee80211_vs_ubiquiti_type;
static int hf_ieee80211_vs_ubiquiti_ap_name;
static int hf_ieee80211_vs_ubiquiti_data;

static int hf_ieee80211_vs_meter_type;
static int hf_ieee80211_vs_meter_ap_name;
static int hf_ieee80211_vs_meter_data;

static int hf_ieee80211_rsn_ie_ptk_keyid;

static int hf_ieee80211_rsn_ie_gtk_kde_data_type;
static int hf_ieee80211_rsn_ie_gtk_kde_key_id;
static int hf_ieee80211_rsn_ie_gtk_kde_tx;
static int hf_ieee80211_rsn_ie_gtk_kde_reserved1;
static int hf_ieee80211_rsn_ie_gtk_kde_reserved2;
static int hf_ieee80211_rsn_ie_gtk_kde_gtk;

static int hf_ieee80211_rsn_ie_mac_address_kde_mac;

static int hf_ieee80211_rsn_ie_pmkid;

static int hf_ieee80211_rsn_ie_unknown;

static int hf_ieee80211_rsn_ie_gtk_kde_nonce;
static int hf_ieee80211_rsn_ie_gtk_kde_lifetime;
static int hf_ieee80211_rsn_ie_error_kde_res;
static int hf_ieee80211_rsn_ie_error_kde_error_type;
static int hf_ieee80211_rsn_ie_igtk_kde_keyid;
static int hf_ieee80211_rsn_ie_igtk_kde_ipn;
static int hf_ieee80211_rsn_ie_igtk_kde_igtk;
static int hf_ieee80211_rsn_ie_oci_operating_class;
static int hf_ieee80211_rsn_ie_oci_primary_channel_number;
static int hf_ieee80211_rsn_ie_oci_frequency_segment_1;
static int hf_ieee80211_rsn_ie_bigtk_key_id;
static int hf_ieee80211_rsn_ie_bigtk_bipn;
static int hf_ieee80211_rsn_ie_bigtk_bigtk;
static int hf_ieee80211_rsn_ie_mlo_link_info;
static int hf_ieee80211_rsn_ie_mlo_linkid;
static int hf_ieee80211_rsn_ie_mlo_rnse_present;
static int hf_ieee80211_rsn_ie_mlo_rnsxe_present;
static int hf_ieee80211_rsn_ie_mlo_reserved;
static int hf_ieee80211_rsn_ie_mlo_mac_addr;
static int hf_ieee80211_rsn_ie_mlo_gtk_kde_key_id;
static int hf_ieee80211_rsn_ie_mlo_gtk_kde_tx;
static int hf_ieee80211_rsn_ie_mlo_gtk_kde_reserved;
static int hf_ieee80211_rsn_ie_mlo_gtk_kde_linkid;
static int hf_ieee80211_rsn_ie_mlo_gtk_kde_pn;
static int hf_ieee80211_rsn_ie_mlo_gtk_kde_gtk;

static int hf_ieee80211_rsn_ie_mlo_igtk_kde_key_id;
static int hf_ieee80211_rsn_ie_mlo_igtk_kde_ipn;
static int hf_ieee80211_rsn_ie_mlo_igtk_kde_reserved;
static int hf_ieee80211_rsn_ie_mlo_igtk_kde_linkid;
static int hf_ieee80211_rsn_ie_mlo_igtk_kde_igtk;

static int hf_ieee80211_rsn_ie_mlo_bigtk_kde_key_id;
static int hf_ieee80211_rsn_ie_mlo_bigtk_kde_ipn;
static int hf_ieee80211_rsn_ie_mlo_bigtk_kde_reserved;
static int hf_ieee80211_rsn_ie_mlo_bigtk_kde_linkid;
static int hf_ieee80211_rsn_ie_mlo_bigtk_kde_bigtk;

static int hf_ieee80211_marvell_ie_type;
static int hf_ieee80211_marvell_ie_mesh_subtype;
static int hf_ieee80211_marvell_ie_mesh_version;
static int hf_ieee80211_marvell_ie_mesh_active_proto_id;
static int hf_ieee80211_marvell_ie_mesh_active_metric_id;
static int hf_ieee80211_marvell_ie_mesh_cap;
static int hf_ieee80211_marvell_ie_data;

static int hf_ieee80211_extreme_mesh_ie_type;
static int hf_ieee80211_extreme_mesh_ie_services;
static int hf_ieee80211_extreme_mesh_ie_hello_f_root;
static int hf_ieee80211_extreme_mesh_ie_hello_f_proxy;
static int hf_ieee80211_extreme_mesh_ie_hello_f_geo;
static int hf_ieee80211_extreme_mesh_ie_hello_f_path_pref;
static int hf_ieee80211_extreme_mesh_ie_hello_f_mobile;
static int hf_ieee80211_extreme_mesh_ie_htr;
static int hf_ieee80211_extreme_mesh_ie_mtr;
static int hf_ieee80211_extreme_mesh_ie_root;
static int hf_ieee80211_extreme_mesh_ie_nh;
static int hf_ieee80211_extreme_mesh_ie_mesh_id;
static int hf_ieee80211_extreme_mesh_ie_mp_id;

static int hf_ieee80211_atheros_ie_type;
static int hf_ieee80211_atheros_ie_subtype;
static int hf_ieee80211_atheros_ie_version;
static int hf_ieee80211_atheros_ie_cap_f_turbop;
static int hf_ieee80211_atheros_ie_cap_f_comp;
static int hf_ieee80211_atheros_ie_cap_f_ff;
static int hf_ieee80211_atheros_ie_cap_f_xr;
static int hf_ieee80211_atheros_ie_cap_f_ar;
static int hf_ieee80211_atheros_ie_cap_f_burst;
static int hf_ieee80211_atheros_ie_cap_f_wme;
static int hf_ieee80211_atheros_ie_cap_f_boost;
static int hf_ieee80211_atheros_ie_advcap_cap;
static int hf_ieee80211_atheros_ie_advcap_defkey;
static int hf_ieee80211_atheros_ie_xr_info;
static int hf_ieee80211_atheros_ie_xr_base_bssid;
static int hf_ieee80211_atheros_ie_xr_xr_bssid;
static int hf_ieee80211_atheros_ie_xr_xr_beacon;
static int hf_ieee80211_atheros_ie_xr_base_cap;
static int hf_ieee80211_atheros_ie_xr_xr_cap;
static int hf_ieee80211_atheros_ie_data;

/*QBSS - Version 1,2,802.11e*/

static int hf_ieee80211_qbss2_cal;
static int hf_ieee80211_qbss2_gl;
static int hf_ieee80211_qbss_cu;
static int hf_ieee80211_qbss2_cu;
static int hf_ieee80211_qbss_scount;
static int hf_ieee80211_qbss2_scount;
static int hf_ieee80211_qbss_version;
static int hf_ieee80211_qbss_adc;

static int hf_ieee80211_tsinfo;
static int hf_ieee80211_tsinfo_type;
static int hf_ieee80211_tsinfo_tsid;
static int hf_ieee80211_tsinfo_dir;
static int hf_ieee80211_tsinfo_access;
static int hf_ieee80211_tsinfo_agg;
static int hf_ieee80211_tsinfo_apsd;
static int hf_ieee80211_tsinfo_up;
static int hf_ieee80211_tsinfo_ack;
static int hf_ieee80211_tsinfo_sched;
static int hf_ieee80211_tsinfo_rsv;

static int * const ieee80211_tsinfo_fields[] = {
  &hf_ieee80211_tsinfo_type,
  &hf_ieee80211_tsinfo_tsid,
  &hf_ieee80211_tsinfo_dir,
  &hf_ieee80211_tsinfo_access,
  &hf_ieee80211_tsinfo_agg,
  &hf_ieee80211_tsinfo_apsd,
  &hf_ieee80211_tsinfo_up,
  &hf_ieee80211_tsinfo_ack,
  &hf_ieee80211_tsinfo_sched,
  &hf_ieee80211_tsinfo_rsv,
  NULL
};

static int hf_ieee80211_tspec_nor_msdu;
static int hf_ieee80211_tspec_max_msdu;
static int hf_ieee80211_tspec_min_srv;
static int hf_ieee80211_tspec_max_srv;
static int hf_ieee80211_tspec_inact_int;
static int hf_ieee80211_tspec_susp_int;
static int hf_ieee80211_tspec_srv_start;
static int hf_ieee80211_tspec_min_data;
static int hf_ieee80211_tspec_mean_data;
static int hf_ieee80211_tspec_peak_data;
static int hf_ieee80211_tspec_burst_size;
static int hf_ieee80211_tspec_delay_bound;
static int hf_ieee80211_tspec_min_phy;
static int hf_ieee80211_tspec_surplus;
static int hf_ieee80211_tspec_medium;
static int hf_ieee80211_tspec_dmg;
static int hf_ieee80211_ts_delay;
static int hf_ieee80211_tclas_process;
static int hf_ieee80211_tag_ext_supp_rates;
static int hf_ieee80211_sched_info;
static int hf_ieee80211_sched_info_agg;
static int hf_ieee80211_sched_info_tsid;
static int hf_ieee80211_sched_info_dir;
static int hf_ieee80211_sched_srv_start;
static int hf_ieee80211_sched_srv_int;
static int hf_ieee80211_sched_spec_int;
static int hf_ieee80211_tclas_up;
static int hf_ieee80211_tclas_class_type;
static int hf_ieee80211_tclas_class_mask;
static int hf_ieee80211_tclas_mask_reserved;
static int hf_ieee80211_tclas_class_mask0_src_addr;
static int hf_ieee80211_tclas_class_mask0_dst_addr;
static int hf_ieee80211_tclas_class_mask0_type;
static int hf_ieee80211_tclas_class_mask1_ver;
static int hf_ieee80211_tclas_class_mask1_src_ip;
static int hf_ieee80211_tclas_class_mask1_dst_ip;
static int hf_ieee80211_tclas_class_mask1_src_port;
static int hf_ieee80211_tclas_class_mask1_dst_port;
static int hf_ieee80211_tclas_class_mask1_ipv4_dscp;
static int hf_ieee80211_tclas_class_mask1_ipv4_proto;
static int hf_ieee80211_tclas_class_mask1_reserved;
static int hf_ieee80211_tclas_class_mask1_ipv6_flow;
static int hf_ieee80211_tclas_class_mask2_tci;
static int hf_ieee80211_tclas_src_mac_addr;
static int hf_ieee80211_tclas_dst_mac_addr;
static int hf_ieee80211_tclas_ether_type;
static int hf_ieee80211_tclas_version;
static int hf_ieee80211_tclas_ipv4_src;
static int hf_ieee80211_tclas_ipv4_dst;
static int hf_ieee80211_tclas_src_port;
static int hf_ieee80211_tclas_dst_port;
static int hf_ieee80211_tclas_dscp;
static int hf_ieee80211_tclas_protocol;
static int hf_ieee80211_tclas_ipv6_src;
static int hf_ieee80211_tclas_ipv6_dst;
static int hf_ieee80211_tclas_flow;
static int hf_ieee80211_tclas_tag_type;
static int hf_ieee80211_tclas_filter_offset;
static int hf_ieee80211_tclas_filter_value;
static int hf_ieee80211_tclas_filter_mask;
static int hf_ieee80211_tclas4_version;
static int hf_ieee80211_tclas_class_mask4_ver;
static int hf_ieee80211_tclas_class_mask4_4_src_ip;
static int hf_ieee80211_tclas_class_mask4_4_dst_ip;
static int hf_ieee80211_tclas_class_mask4_src_port;
static int hf_ieee80211_tclas_class_mask4_dst_port;
static int hf_ieee80211_tclas_class_mask4_dscp;
static int hf_ieee80211_tclas_class_mask4_ipv4_proto;
static int hf_ieee80211_tclas_class_mask4_reserved;
static int hf_ieee80211_tclas_class_mask4_6_src_ip;
static int hf_ieee80211_tclas_class_mask4_6_dst_ip;
static int hf_ieee80211_tclas_reserved_bytes;
static int hf_ieee80211_tclas_class_mask4_next_hdr;
static int hf_ieee80211_tclas_class_mask4_flow_label;
static int hf_ieee80211_tclas4_ipv4_src;
static int hf_ieee80211_tclas4_ipv4_dst;
static int hf_ieee80211_tclas4_src_port;
static int hf_ieee80211_tclas4_dst_port;
static int hf_ieee80211_tclas4_dscp;
static int hf_ieee80211_tclas4_protocol;
static int hf_ieee80211_tclas4_reserved;
static int hf_ieee80211_tclas4_ipv6_src;
static int hf_ieee80211_tclas4_ipv6_dst;
static int hf_ieee80211_tclas4_next_hdr;
static int hf_ieee80211_tclas4_flow;
static int hf_ieee80211_tclas_tclas_8021d_up_pcp;
static int hf_ieee80211_tclas_8021q_dei;
static int hf_ieee80211_tclas_8021q_vid;

static int hf_ieee80211_tclas_class_mask5_up_prio;
static int hf_ieee80211_tclas_class_mask5_dei;
static int hf_ieee80211_tclas_class_mask5_vid;
static int hf_ieee80211_tclas_class_mask5_reserved;

static int hf_ieee80211_tclas_class_mask6_a_above;
static int hf_ieee80211_tclas_class_mask6_frame_control_match_spec;
static int hf_ieee80211_tclas_class_mask6_duration_id_match_spec;
static int hf_ieee80211_tclas_class_mask6_address_1_match_spec;
static int hf_ieee80211_tclas_class_mask6_address_2_match_spec;
static int hf_ieee80211_tclas_class_mask6_address_3_match_spec;
static int hf_ieee80211_tclas_class_mask6_sequence_control_spec;
static int hf_ieee80211_tclas_class_mask6_address_4_match_spec;
static int hf_ieee80211_tclas_class_mask6_qos_control_spec;
static int hf_ieee80211_tclas_class_mask6_ht_control_spec;
static int hf_ieee80211_tclas_class_mask6_reserved;
static int hf_ieee80211_tclas6_frame_control_spec;
static int hf_ieee80211_tclas6_frame_control_mask;
static int hf_ieee80211_tclas6_duration_spec;
static int hf_ieee80211_tclas6_duration_mask;
static int hf_ieee80211_tclas6_address_1_spec;
static int hf_ieee80211_tclas6_address_1_mask;
static int hf_ieee80211_tclas6_address_2_spec;
static int hf_ieee80211_tclas6_address_2_mask;
static int hf_ieee80211_tclas6_address_3_spec;
static int hf_ieee80211_tclas6_address_3_mask;
static int hf_ieee80211_tclas6_sequence_control_spec;
static int hf_ieee80211_tclas6_sequence_control_mask;
static int hf_ieee80211_tclas6_address_4_spec;
static int hf_ieee80211_tclas6_address_4_mask;
static int hf_ieee80211_tclas6_qos_control_spec;
static int hf_ieee80211_tclas6_qos_control_mask;
static int hf_ieee80211_tclas6_ht_control_spec;
static int hf_ieee80211_tclas6_ht_control_mask;

static int hf_ieee80211_tclas_class_mask7_frame_control_match_spec;
static int hf_ieee80211_tclas_class_mask7_address_1_sid_match_spec;
static int hf_ieee80211_tclas_class_mask7_address_2_match_spec;
static int hf_ieee80211_tclas_class_mask7_sequence_control_spec;
static int hf_ieee80211_tclas_class_mask7_address_3_match_spec;
static int hf_ieee80211_tclas_class_mask7_address_4_match_spec;
static int hf_ieee80211_tclas_class_mask7_reserved;
static int hf_ieee80211_tclas7_frame_control_spec;
static int hf_ieee80211_tclas7_frame_control_mask;
static int hf_ieee80211_tclas7_address_1_sid_spec;
static int hf_ieee80211_tclas7_address_1_sid_mask;
static int hf_ieee80211_tclas7_address_2_spec;
static int hf_ieee80211_tclas7_address_2_mask;
static int hf_ieee80211_tclas7_sequence_control_spec;
static int hf_ieee80211_tclas7_sequence_control_mask;
static int hf_ieee80211_tclas7_address_3_spec;
static int hf_ieee80211_tclas7_address_3_mask;
static int hf_ieee80211_tclas7_address_4_spec;
static int hf_ieee80211_tclas7_address_4_mask;

static int hf_ieee80211_tclas_class_mask8_frame_control_match_spec;
static int hf_ieee80211_tclas_class_mask8_address_1_bssid_match_spec;
static int hf_ieee80211_tclas_class_mask8_address_2_sid_match_spec;
static int hf_ieee80211_tclas_class_mask8_sequence_control_spec;
static int hf_ieee80211_tclas_class_mask8_address_3_match_spec;
static int hf_ieee80211_tclas_class_mask8_address_4_match_spec;
static int hf_ieee80211_tclas_class_mask8_reserved;
static int hf_ieee80211_tclas8_frame_control_spec;
static int hf_ieee80211_tclas8_frame_control_mask;
static int hf_ieee80211_tclas8_address_1_bssid_spec;
static int hf_ieee80211_tclas8_address_1_bssid_mask;
static int hf_ieee80211_tclas8_address_2_sid_spec;
static int hf_ieee80211_tclas8_address_2_sid_mask;
static int hf_ieee80211_tclas8_sequence_control_spec;
static int hf_ieee80211_tclas8_sequence_control_mask;
static int hf_ieee80211_tclas8_address_3_spec;
static int hf_ieee80211_tclas8_address_3_mask;
static int hf_ieee80211_tclas8_address_4_spec;
static int hf_ieee80211_tclas8_address_4_mask;

static int hf_ieee80211_tclas_class_mask9_frame_control_match_spec;
static int hf_ieee80211_tclas_class_mask9_address_1_match_spec;
static int hf_ieee80211_tclas_class_mask9_address_2_match_spec;
static int hf_ieee80211_tclas_class_mask9_sequence_control_spec;
static int hf_ieee80211_tclas_class_mask9_reserved;
static int hf_ieee80211_tclas9_frame_control_spec;
static int hf_ieee80211_tclas9_frame_control_mask;
static int hf_ieee80211_tclas9_address_1_spec;
static int hf_ieee80211_tclas9_address_1_mask;
static int hf_ieee80211_tclas9_address_2_spec;
static int hf_ieee80211_tclas9_address_2_mask;
static int hf_ieee80211_tclas9_sequence_control_spec;
static int hf_ieee80211_tclas9_sequence_control_mask;

static int hf_ieee80211_tclas10_protocol_instance;
static int hf_ieee80211_tclas10_protocol_num_next_hdr;

static int hf_ieee80211_aruba;
static int hf_ieee80211_aruba_hb_seq;
static int hf_ieee80211_aruba_mtu;

static int hf_ieee80211_tag_vendor_oui_type;
static int hf_ieee80211_tag_vendor_data;

static int hf_ieee80211_symbp_extreme_assoc_clients;
static int hf_ieee80211_symbp_extreme_load_kbps;
static int hf_ieee80211_symbp_extreme_load_pps;
static int hf_ieee80211_symbp_extreme_client_tx_power;
static int hf_ieee80211_symbp_extreme_timestamp;

static int hf_ieee80211_tag_symbol_proprietary_oui;
static int hf_ieee80211_tag_symbol_proprietary_data;

/* IEEE Std 802.11z-2010 7.3.2.62 */
static int hf_ieee80211_tag_link_id_bssid;
static int hf_ieee80211_tag_link_id_init_sta;
static int hf_ieee80211_tag_link_id_resp_sta;

/* IEEE Std 802.11z-2010 7.3.2.63 */
static int hf_ieee80211_tag_wakeup_schedule_offset;
static int hf_ieee80211_tag_wakeup_schedule_interval;
static int hf_ieee80211_tag_wakeup_schedule_awake_window_slots;
static int hf_ieee80211_tag_wakeup_schedule_max_awake_dur;
static int hf_ieee80211_tag_wakeup_schedule_idle_count;

/* IEEE Std 802.11z-2010 7.3.2.64 */
static int hf_ieee80211_tag_channel_switch_timing_switch_time;
static int hf_ieee80211_tag_channel_switch_timing_switch_timeout;

/* IEEE Std 802.11z-2010 7.3.2.65 */
static int hf_ieee80211_tag_pti_control_tid;
static int hf_ieee80211_tag_pti_control_sequence_control;

/* IEEE Std 802.11z-2010 7.3.2.66 */
static int hf_ieee80211_tag_pu_buffer_status_ac_bk;
static int hf_ieee80211_tag_pu_buffer_status_ac_be;
static int hf_ieee80211_tag_pu_buffer_status_ac_vi;
static int hf_ieee80211_tag_pu_buffer_status_ac_vo;

/* IEEE Std 802.11r-2008 7.3.2.49 */
static int hf_ieee80211_tag_timeout_int_type;
static int hf_ieee80211_tag_timeout_int_value;

/* Ethertype 89-0d */
static int hf_ieee80211_data_encap_payload_type;

static int hf_ieee80211_anqp_wfa_subtype;

static int hf_ieee80211_dpp_subtype;

/* Hotspot 2.0 */
static int hf_ieee80211_hs20_indication_dgaf_disabled;
static int hf_ieee80211_hs20_indication_pps_mo_id_present;
static int hf_ieee80211_hs20_indication_anqp_domain_id_present;
static int hf_ieee80211_hs20_reserved;
static int hf_ieee80211_hs20_indication_version_number;
static int hf_ieee80211_hs20_indication_pps_mo_id;
static int hf_ieee80211_hs20_indication_anqp_domain_id;

static int hf_ieee80211_hs20_anqp_subtype;
static int hf_ieee80211_hs20_anqp_reserved;
static int hf_ieee80211_hs20_anqp_payload;
static int hf_ieee80211_hs20_anqp_hs_query_list;
static int hf_ieee80211_hs20_anqp_hs_capability_list;
static int hf_ieee80211_hs20_anqp_ofn_length;
static int hf_ieee80211_hs20_anqp_ofn_language;
static int hf_ieee80211_hs20_anqp_ofn_name;
static int hf_ieee80211_hs20_anqp_wan_metrics_link_status;
static int hf_ieee80211_hs20_anqp_wan_metrics_symmetric_link;
static int hf_ieee80211_hs20_anqp_wan_metrics_at_capacity;
static int hf_ieee80211_hs20_anqp_wan_metrics_reserved;
static int hf_ieee80211_hs20_anqp_wan_metrics_downlink_speed;
static int hf_ieee80211_hs20_anqp_wan_metrics_uplink_speed;
static int hf_ieee80211_hs20_anqp_wan_metrics_downlink_load;
static int hf_ieee80211_hs20_anqp_wan_metrics_uplink_load;
static int hf_ieee80211_hs20_anqp_wan_metrics_lmd;
static int hf_ieee80211_hs20_anqp_cc_proto_ip_proto;
static int hf_ieee80211_hs20_anqp_cc_proto_port_num;
static int hf_ieee80211_hs20_anqp_cc_proto_status;
static int hf_ieee80211_hs20_anqp_nai_hrq_count;
static int hf_ieee80211_hs20_anqp_nai_hrq_encoding_type;
static int hf_ieee80211_hs20_anqp_nai_hrq_length;
static int hf_ieee80211_hs20_anqp_nai_hrq_realm_name;
static int hf_ieee80211_hs20_anqp_oper_class_indic;
static int hf_ieee80211_hs20_osu_friendly_names_len;
static int hf_ieee80211_hs20_osu_friendly_name_length;
static int hf_ieee80211_hs20_osu_friendly_name_language;
static int hf_ieee80211_hs20_osu_friendly_name_name;
static int hf_ieee80211_hs20_osu_server_uri_len;
static int hf_ieee80211_hs20_osu_server_uri;
static int hf_ieee80211_hs20_osu_method_list_len;
static int hf_ieee80211_hs20_osu_method_val;
static int hf_ieee80211_hs20_icons_avail_len;
static int hf_ieee80211_hs20_osu_providers_list_ssid_len;
static int hf_ieee80211_hs20_osu_providers_ssid;
static int hf_ieee80211_hs20_osu_providers_count;
static int hf_ieee80211_hs20_osu_prov_length;
static int hf_ieee80211_hs20_icon_request_filename;
static int hf_ieee80211_hs20_icon_binary_file_status;
static int hf_ieee80211_hs20_icon_type_length;
static int hf_ieee80211_hs20_icon_type;
static int hf_ieee80211_hs20_icon_binary_data_len;
static int hf_ieee80211_hs20_icon_binary_data;
static int hf_ieee80211_osu_icon_avail_width;
static int hf_ieee80211_osu_icon_avail_height;
static int hf_ieee80211_osu_icon_avail_lang_code;
static int hf_ieee80211_osu_icon_avail_icon_type_len;
static int hf_ieee80211_osu_icon_avail_icon_type;
static int hf_ieee80211_osu_icon_avail_filename_len;
static int hf_ieee80211_osu_icon_avail_filename;
static int hf_ieee80211_hs20_osu_nai_len;
static int hf_ieee80211_hs20_osu_nai;
static int hf_ieee80211_hs20_osu_service_desc_len;
static int hf_ieee80211_hs20_osu_service_desc_duple_len;
static int hf_ieee80211_hs20_osu_service_desc_lang;
static int hf_ieee80211_hs20_osu_service_desc;
static int hf_ieee80211_hs20_anqp_venue_url_length;
static int hf_ieee80211_hs20_anqp_venue_number;
static int hf_ieee80211_hs20_anqp_venue_url;
static int hf_ieee80211_hs20_anqp_advice_of_charge_length;
static int hf_ieee80211_hs20_anqp_advice_of_charge_type;
static int hf_ieee80211_hs20_anqp_aoc_nai_realm_encoding;
static int hf_ieee80211_hs20_anqp_aoc_nai_realm_len;
static int hf_ieee80211_hs20_anqp_aoc_nai_realm;
static int hf_ieee80211_hs20_anqp_aoc_plan_len;
static int hf_ieee80211_hs20_anqp_aoc_plan_lang;
static int hf_ieee80211_hs20_anqp_aoc_plan_curcy;
static int hf_ieee80211_hs20_anqp_aoc_plan_information;

static int hf_ieee80211_hs20_subscription_remediation_url_len;
static int hf_ieee80211_hs20_subscription_remediation_server_url;
static int hf_ieee80211_hs20_subscription_remediation_server_method;
static int hf_ieee80211_hs20_deauth_reason_code;
static int hf_ieee80211_hs20_reauth_delay;
static int hf_ieee80211_hs20_deauth_reason_url_len;
static int hf_ieee80211_hs20_deauth_imminent_reason_url;

/* IEEE Std 802.11ai : FILS Discovery */
static int hf_ieee80211_ff_fils_discovery_frame_control;
static int hf_ieee80211_ff_fils_discovery_frame_control_ssid_length;
static int hf_ieee80211_ff_fils_discovery_frame_control_capability;
static int hf_ieee80211_ff_fils_discovery_frame_control_short_ssid;
static int hf_ieee80211_ff_fils_discovery_frame_control_ap_csn;
static int hf_ieee80211_ff_fils_discovery_frame_control_ano;
static int hf_ieee80211_ff_fils_discovery_frame_control_channel_center_frequency;
static int hf_ieee80211_ff_fils_discovery_frame_control_primary_channel;
static int hf_ieee80211_ff_fils_discovery_frame_control_rsn_info;
static int hf_ieee80211_ff_fils_discovery_frame_control_length;
static int hf_ieee80211_ff_fils_discovery_frame_control_md;
static int hf_ieee80211_ff_fils_discovery_frame_control_reserved;
static int hf_ieee80211_ff_fils_discovery_ssid;
static int hf_ieee80211_ff_fils_discovery_capability;
static int hf_ieee80211_ff_fils_discovery_capability_ess;
static int hf_ieee80211_ff_fils_discovery_capability_privacy;
static int hf_ieee80211_ff_fils_discovery_capability_bss_operating_channel_width;
static int hf_ieee80211_ff_fils_discovery_capability_max_number_of_spatial_streams;
static int hf_ieee80211_ff_fils_discovery_capability_reserved;
static int hf_ieee80211_ff_fils_discovery_capability_multiple_bssid;
static int hf_ieee80211_ff_fils_discovery_capability_phy_index;
static int hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_dsss;
static int hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_ofdm;
static int hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_ht_vht_tvht;
static int hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_he;
static int hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate;

static int hf_ieee80211_ff_fils_discovery_short_ssid;
static int hf_ieee80211_ff_fils_discovery_ap_csn;
static int hf_ieee80211_ff_fils_discovery_ano;
static int hf_ieee80211_ff_fils_discovery_ccfs1;
static int hf_ieee80211_ff_fils_discovery_operating_class;
static int hf_ieee80211_ff_fils_discovery_primary_channel;
static int hf_ieee80211_ff_fils_discovery_rsn_info;
static int hf_ieee80211_ff_fils_discovery_length;
static int hf_ieee80211_ff_fils_discovery_md;

/* IEEE Std 802.11ad */
static int hf_ieee80211_block_ack_RBUFCAP;
static int hf_ieee80211_cf_response_offset;
static int hf_ieee80211_grant_ack_reserved;
static int hf_ieee80211_ff_dynamic_allocation;
static int hf_ieee80211_ff_TID;
static int hf_ieee80211_ff_alloc_type;
static int hf_ieee80211_ff_src_aid;
static int hf_ieee80211_ff_dest_aid;
static int hf_ieee80211_ff_alloc_duration;
static int hf_ieee80211_ff_b39;
static int hf_ieee80211_ff_ssw;
static int hf_ieee80211_ff_ssw_direction;
static int hf_ieee80211_ff_ssw_cdown;
static int hf_ieee80211_ff_ssw_sector_id;
static int hf_ieee80211_ff_ssw_dmg_ant_id;
static int hf_ieee80211_ff_ssw_rxss_len;
static int hf_ieee80211_ff_bf;
static int hf_ieee80211_ff_bf_train;
static int hf_ieee80211_ff_bf_is_init;
static int hf_ieee80211_ff_bf_is_resp;
static int hf_ieee80211_ff_bf_num_sectors;
static int hf_ieee80211_ff_bf_num_rx_dmg_ants;
static int hf_ieee80211_ff_bf_b12b15;
static int hf_ieee80211_ff_bf_rxss_len;
static int hf_ieee80211_ff_bf_rxss_rate;
static int hf_ieee80211_ff_bf_b10b15;
static int hf_ieee80211_addr_nav_da;
static int hf_ieee80211_addr_nav_sa;
static int hf_ieee80211_ff_sswf;
static int hf_ieee80211_ff_sswf_num_rx_dmg_ants;
static int hf_ieee80211_ff_sswf_poll_required;
static int hf_ieee80211_ff_sswf_total_sectors;
static int hf_ieee80211_ff_sswf_reserved1;
static int hf_ieee80211_ff_sswf_reserved2;
static int hf_ieee80211_ff_sswf_sector_select;
static int hf_ieee80211_ff_sswf_dmg_antenna_select;
static int hf_ieee80211_ff_sswf_snr_report;
static int hf_ieee80211_ff_brp;
static int hf_ieee80211_ff_brp_L_RX;
static int hf_ieee80211_ff_brp_TX_TRN_REQ;
static int hf_ieee80211_ff_brp_MID_REQ;
static int hf_ieee80211_ff_brp_BC_REQ;
static int hf_ieee80211_ff_brp_MID_GRANT;
static int hf_ieee80211_ff_brp_BC_GRANT;
static int hf_ieee80211_ff_brp_chan_FBCK_CAP;
static int hf_ieee80211_ff_brp_tx_sector;
static int hf_ieee80211_ff_brp_other_aid;
static int hf_ieee80211_ff_brp_tx_antenna;
static int hf_ieee80211_ff_brp_reserved;
static int hf_ieee80211_ff_blm;
static int hf_ieee80211_ff_blm_unit_index;
static int hf_ieee80211_ff_blm_maint_value;
static int hf_ieee80211_ff_blm_is_master;
static int hf_ieee80211_ff_bic;
static int hf_ieee80211_ff_bic_cc_present;
static int hf_ieee80211_ff_bic_discovery_mode;
static int hf_ieee80211_ff_bic_next_beacon;
static int hf_ieee80211_ff_bic_ati_present;
static int hf_ieee80211_ff_bic_abft_len;
static int hf_ieee80211_ff_bic_fss;
static int hf_ieee80211_ff_bic_is_resp;
static int hf_ieee80211_ff_bic_next_abft;
static int hf_ieee80211_ff_bic_frag_txss;
static int hf_ieee80211_ff_bic_txss_span;
static int hf_ieee80211_ff_bic_NBI_abft;
static int hf_ieee80211_ff_bic_abft_count;
static int hf_ieee80211_ff_bic_nabft;
static int hf_ieee80211_ff_bic_pcp;
static int hf_ieee80211_ff_bic_reserved;
static int * const ieee80211_ff_bic_fields[] = {
  &hf_ieee80211_ff_bic_cc_present,
  &hf_ieee80211_ff_bic_discovery_mode,
  &hf_ieee80211_ff_bic_next_beacon,
  &hf_ieee80211_ff_bic_ati_present,
  &hf_ieee80211_ff_bic_abft_len,
  &hf_ieee80211_ff_bic_fss,
  &hf_ieee80211_ff_bic_is_resp,
  &hf_ieee80211_ff_bic_next_abft,
  &hf_ieee80211_ff_bic_frag_txss,
  &hf_ieee80211_ff_bic_txss_span,
  &hf_ieee80211_ff_bic_NBI_abft,
  &hf_ieee80211_ff_bic_abft_count,
  &hf_ieee80211_ff_bic_nabft,
  &hf_ieee80211_ff_bic_pcp,
  &hf_ieee80211_ff_bic_reserved,
  NULL
};
static int hf_ieee80211_ff_dmg_params;
static int hf_ieee80211_ff_dmg_params_bss;
static int hf_ieee80211_ff_dmg_params_cbap_only;
static int hf_ieee80211_ff_dmg_params_cbap_src;
static int hf_ieee80211_ff_dmg_params_privacy;
static int hf_ieee80211_ff_dmg_params_policy;
static int hf_ieee80211_ff_dmg_params_spec_mgmt;
static int hf_ieee80211_ff_dmg_params_radio_measure;
static int hf_ieee80211_ff_cc;
static int hf_ieee80211_ff_cc_abft_resp_addr;
static int hf_ieee80211_ff_cc_sp_duration;
static int hf_ieee80211_ff_cc_cluster_id;
static int hf_ieee80211_ff_cc_role;
static int hf_ieee80211_ff_cc_max_mem;
static int hf_ieee80211_ff_dmg_action_code;
static int hf_ieee80211_ff_dmg_pwr_mgmt;
static int hf_ieee80211_ff_subject_address;
static int hf_ieee80211_ff_handover_reason;
static int hf_ieee80211_ff_handover_remaining_bi;
static int hf_ieee80211_ff_handover_result;
static int hf_ieee80211_ff_handover_reject_reason;
static int hf_ieee80211_ff_destination_reds_aid;
static int hf_ieee80211_ff_destination_aid;
static int hf_ieee80211_ff_relay_aid;
static int hf_ieee80211_ff_source_aid;
static int hf_ieee80211_ff_timing_offset;
static int hf_ieee80211_ff_sampling_frequency_offset;
static int hf_ieee80211_ff_relay_operation_type;
static int hf_ieee80211_ff_peer_sta_aid;
static int hf_ieee80211_ff_snr;
static int hf_ieee80211_ff_internal_angle;
static int hf_ieee80211_ff_recommend;
static int hf_ieee80211_ff_unprotected_dmg_action_code;
static int hf_ieee80211_ff_fst_action_code;
static int hf_ieee80211_ff_robust_av_streaming_action_code;
static int hf_ieee80211_ff_llt;
static int hf_ieee80211_ff_fsts_id;
static int hf_ieee80211_ff_mmpdu_len;
static int hf_ieee80211_ff_mmpdu_ctrl;
static int hf_ieee80211_ff_oct_mmpdu;
static int hf_ieee80211_ff_scs_scsid;
static int hf_ieee80211_ff_scs_status;
static int hf_ieee80211_ff_scs_response_count;

#if 0
static int hf_ieee80211_ff_rcsi;
static int hf_ieee80211_ff_rcsi_aid;
#endif
static int hf_ieee80211_ff_band_id;
static int hf_ieee80211_tag_relay_support;
static int hf_ieee80211_tag_relay_use;
static int hf_ieee80211_tag_relay_permission;
static int hf_ieee80211_tag_AC_power;
static int hf_ieee80211_tag_relay_prefer;
static int hf_ieee80211_tag_duplex;
static int hf_ieee80211_tag_cooperation;
static int hf_ieee80211_tag_move;
static int hf_ieee80211_tag_size;
static int hf_ieee80211_tag_tbtt_offset;
static int hf_ieee80211_tag_bi_duration;
static int hf_ieee80211_tag_dmg_capa_sta_addr;
static int hf_ieee80211_tag_dmg_capa_aid;
static int hf_ieee80211_tag_reverse_direction;
static int hf_ieee80211_tag_hlts;
static int hf_ieee80211_tag_tpc;
static int hf_ieee80211_tag_spsh;
static int hf_ieee80211_tag_rx_antenna;
static int hf_ieee80211_tag_fast_link;
static int hf_ieee80211_tag_num_sectors;
static int hf_ieee80211_tag_rxss_length;
static int hf_ieee80211_tag_reciprocity;
static int hf_ieee80211_tag_max_ampdu_exp;
static int hf_ieee80211_tag_min_mpdu_spacing;
static int hf_ieee80211_tag_ba_flow_control;
static int hf_ieee80211_tag_max_sc_rx_mcs;
static int hf_ieee80211_tag_max_ofdm_rx_mcs;
static int hf_ieee80211_tag_max_sc_tx_mcs;
static int hf_ieee80211_tag_max_ofdm_tx_mcs;
static int hf_ieee80211_tag_low_power_supported;
static int hf_ieee80211_tag_code_rate;
static int hf_ieee80211_tag_dtp;
static int hf_ieee80211_tag_appdu_supp;
static int hf_ieee80211_tag_heartbeat;
static int hf_ieee80211_tag_other_aid;
static int hf_ieee80211_tag_pattern_recip;
static int hf_ieee80211_tag_heartbeat_elapsed;
static int hf_ieee80211_tag_grant_ack_supp;
static int hf_ieee80211_tag_RXSSTxRate_supp;
static int hf_ieee80211_tag_pcp_tddti;
static int hf_ieee80211_tag_pcp_PSA;
static int hf_ieee80211_tag_pcp_handover;
static int hf_ieee80211_tag_pcp_max_assoc;
static int hf_ieee80211_tag_pcp_power_src;
static int hf_ieee80211_tag_pcp_decenter;
static int hf_ieee80211_tag_pcp_forwarding;
static int hf_ieee80211_tag_pcp_center;
static int hf_ieee80211_tag_sta_beam_track;
static int hf_ieee80211_tag_ext_sc_mcs_max_tx;
static int hf_ieee80211_tag_ext_sc_mcs_tx_code_7_8;
static int hf_ieee80211_tag_ext_sc_mcs_max_rx;
static int hf_ieee80211_tag_ext_sc_mcs_rx_code_7_8;
static int hf_ieee80211_tag_max_basic_sf_amsdu;
static int hf_ieee80211_tag_max_short_sf_amsdu;
static int hf_ieee80211_tag_PSRSI;
static int hf_ieee80211_tag_min_BHI_duration;
static int hf_ieee80211_tag_brdct_sta_info_dur;
static int hf_ieee80211_tag_assoc_resp_confirm_time;
static int hf_ieee80211_tag_min_pp_duration;
static int hf_ieee80211_tag_SP_idle_timeout;
static int hf_ieee80211_tag_max_lost_beacons;
static int hf_ieee80211_tag_type;
static int hf_ieee80211_tag_tap1;
static int hf_ieee80211_tag_state1;
static int hf_ieee80211_tag_tap2;
static int hf_ieee80211_tag_state2;
static int hf_ieee80211_tag_allocation_id;
static int hf_ieee80211_tag_allocation_type;
static int hf_ieee80211_tag_pseudo_static;
static int hf_ieee80211_tag_truncatable;
static int hf_ieee80211_tag_extendable;
static int hf_ieee80211_tag_pcp_active;
static int hf_ieee80211_tag_lp_sc_used;
static int hf_ieee80211_tag_src_aid;
static int hf_ieee80211_tag_dest_aid;
static int hf_ieee80211_tag_alloc_start;
static int hf_ieee80211_tag_alloc_block_duration;
static int hf_ieee80211_tag_num_blocks;
static int hf_ieee80211_tag_alloc_block_period;
static int hf_ieee80211_tag_aid;
static int hf_ieee80211_tag_cbap;
static int hf_ieee80211_tag_pp_avail;
static int hf_ieee80211_tag_next_ati_start_time;
static int hf_ieee80211_tag_next_ati_duration;
static int hf_ieee80211_tag_old_bssid;
static int hf_ieee80211_tag_new_pcp_addr;
static int hf_ieee80211_tag_bssid;
static int hf_ieee80211_tag_duplex_relay;
static int hf_ieee80211_tag_cooperation_relay;
static int hf_ieee80211_tag_tx_mode;
static int hf_ieee80211_tag_link_change_interval;
static int hf_ieee80211_tag_data_sensing_time;
static int hf_ieee80211_tag_first_period;
static int hf_ieee80211_tag_second_period;
static int hf_ieee80211_tag_initiator;
static int hf_ieee80211_tag_tx_train_res;
static int hf_ieee80211_tag_rx_train_res;
static int hf_ieee80211_tag_tx_trn_ok;
static int hf_ieee80211_tag_txss_fbck_req;
static int hf_ieee80211_tag_bs_fbck;
static int hf_ieee80211_tag_bs_fbck_antenna_id;
static int hf_ieee80211_tag_snr_requested;
static int hf_ieee80211_tag_channel_measurement_requested;
static int hf_ieee80211_tag_number_of_taps_requested;
static int hf_ieee80211_tag_sector_id_order_req;
static int hf_ieee80211_tag_snr_present;
static int hf_ieee80211_tag_channel_measurement_present;
static int hf_ieee80211_tag_tap_delay_present;
static int hf_ieee80211_tag_number_of_taps_present;
static int hf_ieee80211_tag_number_of_measurement;
static int hf_ieee80211_tag_sector_id_order_present;
static int hf_ieee80211_tag_number_of_beams;
static int hf_ieee80211_tag_mid_extension;
static int hf_ieee80211_tag_capability_request;
static int hf_ieee80211_tag_beam_refine_reserved;
static int hf_ieee80211_tag_nextpcp_list;
static int hf_ieee80211_tag_nextpcp_token;
static int hf_ieee80211_tag_remaining_BI;
static int hf_ieee80211_tag_request_token;
static int hf_ieee80211_tag_bi_start_time;
static int hf_ieee80211_tag_sleep_cycle;
static int hf_ieee80211_tag_num_awake_bis;
static int hf_ieee80211_tag_tspec_allocation_id;
static int hf_ieee80211_tag_tspec_allocation_type;
static int hf_ieee80211_tag_tspec_allocation_format;
static int hf_ieee80211_tag_tspec_pseudo_static;
static int hf_ieee80211_tag_tspec_truncatable;
static int hf_ieee80211_tag_tspec_extendable;
static int hf_ieee80211_tag_tspec_lp_sc_used;
static int hf_ieee80211_tag_tspec_up;
static int hf_ieee80211_tag_tspec_dest_aid;
static int hf_ieee80211_tag_tspec_allocation_period;
static int hf_ieee80211_tag_tspec_min_allocation;
static int hf_ieee80211_tag_tspec_max_allocation;
static int hf_ieee80211_tag_tspec_min_duration;
static int hf_ieee80211_tag_tspec_num_of_constraints;
static int hf_ieee80211_tag_tspec_tsconst_start_time;
static int hf_ieee80211_tag_tspec_tsconst_duration;
static int hf_ieee80211_tag_tspec_tsconst_period;
static int hf_ieee80211_tag_tspec_tsconst_interferer_mac;
static int hf_ieee80211_tag_channel_measurement_feedback_relative_I;
static int hf_ieee80211_tag_channel_measurement_feedback_relative_Q;
static int hf_ieee80211_tag_channel_measurement_feedback_tap_delay;
static int hf_ieee80211_tag_channel_measurement_feedback_sector_id;
static int hf_ieee80211_tag_channel_measurement_feedback_antenna_id;
static int hf_ieee80211_tag_awake_window;
static int hf_ieee80211_tag_addba_ext_no_frag;
static int hf_ieee80211_tag_addba_ext_he_fragmentation_operation;
static int hf_ieee80211_tag_addba_ext_reserved;
static int hf_ieee80211_tag_addba_ext_buffer_size;
static int hf_ieee80211_tag_multi_band_ctrl_sta_role;
static int hf_ieee80211_tag_multi_band_ctrl_addr_present;
static int hf_ieee80211_tag_multi_band_ctrl_cipher_present;
static int hf_ieee80211_tag_multi_band_oper_class;
static int hf_ieee80211_tag_multi_band_channel_number;
static int hf_ieee80211_tag_multi_band_tsf_offset;
static int hf_ieee80211_tag_multi_band_conn_ap;
static int hf_ieee80211_tag_multi_band_conn_pcp;
static int hf_ieee80211_tag_multi_band_conn_dls;
static int hf_ieee80211_tag_multi_band_conn_tdls;
static int hf_ieee80211_tag_multi_band_conn_ibss;
static int hf_ieee80211_tag_multi_band_fst_timeout;
static int hf_ieee80211_tag_multi_band_sta_mac;
static int hf_ieee80211_tag_activity;
static int hf_ieee80211_tag_dmg_link_adapt_mcs;
static int hf_ieee80211_tag_dmg_link_adapt_link_margin;
static int hf_ieee80211_tag_ref_timestamp;
static int hf_ieee80211_tag_switching_stream_non_qos;
static int hf_ieee80211_tag_switching_stream_param_num;
static int hf_ieee80211_tag_switching_stream_old_tid;
static int hf_ieee80211_tag_switching_stream_old_direction;
static int hf_ieee80211_tag_switching_stream_new_tid;
static int hf_ieee80211_tag_switching_stream_new_direction;
static int hf_ieee80211_tag_switching_stream_new_valid_id;
static int hf_ieee80211_tag_switching_stream_llt_type;

static int hf_ieee80211_mysterious_extra_stuff;

static int hf_ieee80211_mscs_descriptor_type;
static int hf_ieee80211_mscs_user_prio_control_reserved;
static int hf_ieee80211_user_prio_bitmap;
static int hf_ieee80211_user_prio_bitmap_bit0;
static int hf_ieee80211_user_prio_bitmap_bit1;
static int hf_ieee80211_user_prio_bitmap_bit2;
static int hf_ieee80211_user_prio_bitmap_bit3;
static int hf_ieee80211_user_prio_bitmap_bit4;
static int hf_ieee80211_user_prio_bitmap_bit5;
static int hf_ieee80211_user_prio_bitmap_bit6;
static int hf_ieee80211_user_prio_bitmap_bit7;
static int hf_ieee80211_user_prio_limit;
static int hf_ieee80211_user_prio_reserved;
static int hf_ieee80211_stream_timeout_reserved;
static int hf_ieee80211_stream_timeout;
static int hf_ieee80211_mscs_subelement_id;
static int hf_ieee80211_mscs_subelement_len;
static int hf_ieee80211_mscs_subelement_data;

static int hf_ieee80211_intra_access_prio;
static int hf_ieee80211_intra_access_prio_user_prio;
static int hf_ieee80211_intra_access_prio_alt_queue;
static int hf_ieee80211_intra_access_prio_drop_elig;
static int hf_ieee80211_intra_access_prio_reserved;

static int hf_ieee80211_scs_descriptor_scsid;
static int hf_ieee80211_scs_descriptor_type;

static int hf_ieee80211_esp_access_category;
static int hf_ieee80211_esp_reserved;
static int hf_ieee80211_esp_data_format;
static int hf_ieee80211_esp_ba_windows_size;
static int hf_ieee80211_esp_est_air_time_frac;
static int hf_ieee80211_esp_data_ppdu_duration_target;
static int hf_ieee80211_estimated_service_params;

static int hf_ieee80211_fcg_new_channel_number;
static int hf_ieee80211_fcg_extra_info;
static int hf_ieee80211_sae_password_identifier;

static int hf_ieee80211_sae_anti_clogging_token;

static int hf_ieee80211_tag_fils_indication_info_nr_pk;
static int hf_ieee80211_tag_fils_indication_info_nr_realm;
static int hf_ieee80211_tag_fils_indication_info_ip_config;
static int hf_ieee80211_tag_fils_indication_info_cache_id_included;
static int hf_ieee80211_tag_fils_indication_info_hessid_included;
static int hf_ieee80211_tag_fils_indication_info_ska_without_pfs;
static int hf_ieee80211_tag_fils_indication_info_ska_with_pfs;
static int hf_ieee80211_tag_fils_indication_info_pka;
static int hf_ieee80211_tag_fils_indication_info_reserved;
static int hf_ieee80211_tag_fils_indication_cache_identifier;
static int hf_ieee80211_tag_fils_indication_hessid;
static int hf_ieee80211_tag_fils_indication_realm_list;
static int hf_ieee80211_tag_fils_indication_realm_identifier;
static int hf_ieee80211_tag_fils_indication_public_key_list;
static int hf_ieee80211_tag_fils_indication_public_key_type;
static int hf_ieee80211_tag_fils_indication_public_key_length;
static int hf_ieee80211_tag_fils_indication_public_key_indicator;

static int hf_ieee80211_qos_mgmt_attribute_id;
static int hf_ieee80211_qos_mgmt_attribute_len;
static int hf_ieee80211_qos_mgmt_start_port_range;
static int hf_ieee80211_qos_mgmt_end_port_range;
static int hf_ieee80211_qos_mgmt_dscp_pol_id;
static int hf_ieee80211_qos_mgmt_dscp_pol_req_type;
static int hf_ieee80211_qos_mgmt_dscp_pol_dscp;
static int hf_ieee80211_qos_mgmt_domain_name;
static int hf_ieee80211_qos_mgmt_unknown_attr;

static int hf_ieee80211_wfa_capa_tag_data;
static int hf_ieee80211_wfa_capa_tag_len;
static int hf_ieee80211_wfa_capa_tag_capabilities;
static int hf_ieee80211_wfa_capa_mgmt_dscp_policy;
static int hf_ieee80211_wfa_capa_mgmt_unsol_dscp_policy;
static int hf_ieee80211_wfa_capa_mgmt_scs_traffic_desc;
static int hf_ieee80211_wfa_capa_5g_wifi_qos_mapping;
static int hf_ieee80211_wfa_capa_data_plane_stat_report;
static int hf_ieee80211_wfa_capa_radio_counters_stat_support;
static int hf_ieee80211_wfa_capa_control_plane_stat_report;
static int hf_ieee80211_wfa_capa_unsolicited_report_support;
static int hf_ieee80211_wfa_capa_attr_id;
static int hf_ieee80211_wfa_capa_attr_len;
static int hf_ieee80211_wfa_capa_attr_field;
static int hf_ieee80211_wfa_capa_attr_supp_gene_length;
static int hf_ieee80211_wfa_capa_attr_supp_generations;
static int hf_ieee80211_wfa_capa_attr_supp_generations_b0;
static int hf_ieee80211_wfa_capa_attr_supp_generations_b1;
static int hf_ieee80211_wfa_capa_attr_supp_generations_b2;
static int hf_ieee80211_wfa_capa_attr_supp_generations_b3;
static int hf_ieee80211_wfa_capa_attr_supp_generations_reserved;
static int hf_ieee80211_wfa_capa_attr_cert_gene_length;
static int hf_ieee80211_wfa_capa_attr_cert_generations;
static int hf_ieee80211_wfa_capa_attr_cert_generations_b0;
static int hf_ieee80211_wfa_capa_attr_cert_generations_b1;
static int hf_ieee80211_wfa_capa_attr_cert_generations_b2;
static int hf_ieee80211_wfa_capa_attr_cert_generations_b3;
static int hf_ieee80211_wfa_capa_attr_cert_generations_reserved;

static int hf_ieee80211_ext_tag;
static int hf_ieee80211_ext_tag_number;
static int hf_ieee80211_ext_tag_length;
static int hf_ieee80211_ext_tag_data;

static int hf_ieee80211_fils_req_params_parameter_control_bitmap;
static int hf_ieee80211_fils_req_params_fils_criteria_present;
static int hf_ieee80211_fils_req_params_max_delay_limit_present;
static int hf_ieee80211_fils_req_params_minimum_data_rate_present;
static int hf_ieee80211_fils_req_params_rcpi_limit_present;
static int hf_ieee80211_fils_req_params_oui_response_criteria_present;
static int hf_ieee80211_fils_req_params_reserved;
static int hf_ieee80211_fils_req_params_max_channel_time;
static int hf_ieee80211_fils_req_params_fils_criteria;
static int hf_ieee80211_fils_req_params_fils_criteria_bss_delay;
static int hf_ieee80211_fils_req_params_fils_criteria_phy_support;
static int hf_ieee80211_fils_req_params_fils_criteria_reserved;
static int hf_ieee80211_fils_req_params_max_delay_limit;
static int hf_ieee80211_fils_req_params_minimum_data_rate;
static int hf_ieee80211_fils_req_params_rcpi_limit;
static int hf_ieee80211_fils_req_params_oui_response_criteria;

static int hf_ieee80211_fils_session;
static int hf_ieee80211_fils_encrypted_data;
static int hf_ieee80211_fils_nonce;

/* wfa 60g ie tree */
static int hf_ieee80211_wfa_60g_attr;
static int hf_ieee80211_wfa_60g_attr_id;
static int hf_ieee80211_wfa_60g_attr_len;

static int hf_ieee80211_wfa_60g_attr_cap_sta_mac_addr;
static int hf_ieee80211_wfa_60g_attr_cap_recv_amsdu_frames;
static int hf_ieee80211_wfa_60g_attr_cap_reserved;

/* ************************************************************************* */
/*                              802.11AX fields                              */
/* ************************************************************************* */
static int hf_ieee80211_he_mac_capabilities;
static int hf_ieee80211_he_htc_he_support;
static int hf_ieee80211_he_twt_requester_support;
static int hf_ieee80211_he_twt_responder_support;
static int hf_ieee80211_he_dynamic_fragmentation_support;
static int hf_ieee80211_he_max_number_fragmented_msdus;
static int hf_ieee80211_he_min_fragment_size;
static int hf_ieee80211_he_trigger_frame_mac_padding_dur;
static int hf_ieee80211_he_multi_tid_aggregation_rx_support;
static int hf_ieee80211_he_he_link_adaptation_support;
static int hf_ieee80211_he_all_ack_support;
static int hf_ieee80211_he_trs_support;
static int hf_ieee80211_he_bsr_support;
static int hf_ieee80211_he_broadcast_twt_support;
static int hf_ieee80211_he_32_bit_ba_bitmap_support;
static int hf_ieee80211_he_mu_cascading_support;
static int hf_ieee80211_he_ack_enabled_aggregation_support;
static int hf_ieee80211_he_reserved_b24;
static int hf_ieee80211_he_om_control_support;
static int hf_ieee80211_he_ofdma_ra_support;
static int hf_ieee80211_he_max_a_mpdu_length_exponent_ext;
static int hf_ieee80211_he_a_msdu_fragmentation_support;
static int hf_ieee80211_he_flexible_twt_schedule_support;
static int hf_ieee80211_he_rx_control_frame_to_multibss;
static int hf_ieee80211_he_bsrp_bqrp_a_mpdu_aggregation;
static int hf_ieee80211_he_qtp_support;
static int hf_ieee80211_he_bqr_support;
static int hf_ieee80211_he_psr_responder;
static int hf_ieee80211_he_ndp_feedback_report_support;
static int hf_ieee80211_he_ops_support;
static int hf_ieee80211_he_a_msdu_in_a_mpdu_support;
static int hf_ieee80211_he_multi_tid_aggregation_tx_support;
static int hf_ieee80211_he_subchannel_selective_trans_support;
static int hf_ieee80211_he_2_996_tone_ru_support;
static int hf_ieee80211_he_om_control_ul_mu_data_disable_rx_support;
static int hf_ieee80211_he_dynamic_sm_power_save;
static int hf_ieee80211_he_punctured_sounding_support;
static int hf_ieee80211_he_ht_and_vht_trigger_frame_rx_support;
static int hf_ieee80211_he_reserved_bit_18;
static int hf_ieee80211_he_reserved_bit_19;
static int hf_ieee80211_he_reserved_bit_25;
static int hf_ieee80211_he_reserved_bit_29;
static int hf_ieee80211_he_reserved_bit_34;
static int hf_ieee80211_he_reserved_bits_5_7;
static int hf_ieee80211_he_reserved_bits_8_9;
static int hf_ieee80211_he_reserved_bits_15_16;
static int hf_ieee80211_he_phy_reserved_b0;
static int hf_ieee80211_he_phy_cap_reserved_b0;
static int hf_ieee80211_he_phy_chan_width_set;
static int hf_ieee80211_he_40mhz_channel_2_4ghz;
static int hf_ieee80211_he_40_and_80_mhz_5ghz;
static int hf_ieee80211_he_160_mhz_5ghz;
static int hf_ieee80211_he_160_80_plus_80_mhz_5ghz;
static int hf_ieee80211_he_242_tone_rus_in_2_4ghz;
static int hf_ieee80211_he_242_tone_rus_in_5ghz;
static int hf_ieee80211_he_5ghz_b0_reserved;
static int hf_ieee80211_he_5ghz_b4_reserved;
static int hf_ieee80211_he_24ghz_b1_reserved;
static int hf_ieee80211_he_24ghz_b2_reserved;
static int hf_ieee80211_he_24ghz_b3_reserved;
static int hf_ieee80211_he_24ghz_b5_reserved;
static int hf_ieee80211_he_chan_width_reserved;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_1_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_2_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_3_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_4_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_5_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_6_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_7_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_rx_8_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_1_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_2_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_3_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_4_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_5_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_6_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_7_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80_tx_8_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_1_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_2_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_3_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_4_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_5_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_6_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_7_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_rx_8_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_1_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_2_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_3_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_4_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_5_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_6_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_7_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_80p80_tx_8_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_1_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_2_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_3_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_4_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_5_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_6_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_7_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_rx_8_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_1_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_2_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_3_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_4_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_5_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_6_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_7_ss;
static int hf_ieee80211_he_mcs_max_he_mcs_160_tx_8_ss;
static int hf_ieee80211_he_rx_he_mcs_map_lte_80;
static int hf_ieee80211_he_tx_he_mcs_map_lte_80;
static int hf_ieee80211_he_rx_he_mcs_map_160;
static int hf_ieee80211_he_tx_he_mcs_map_160;
static int hf_ieee80211_he_rx_he_mcs_map_80_80;
static int hf_ieee80211_he_tx_he_mcs_map_80_80;
static int hf_ieee80211_he_ppe_thresholds_nss;
static int hf_ieee80211_he_ppe_thresholds_ru_index_bitmask;
static int hf_ieee80211_he_ppe_ppet16;
static int hf_ieee80211_he_ppe_ppet8;
static int hf_ieee80211_he_phy_b8_to_b23;
static int hf_ieee80211_he_phy_cap_punctured_preamble_rx;
static int hf_ieee80211_he_phy_cap_device_class;
static int hf_ieee80211_he_phy_cap_ldpc_coding_in_payload;
static int hf_ieee80211_he_phy_cap_he_su_ppdu_1x_he_ltf_08us;
static int hf_ieee80211_he_phy_cap_midamble_tx_rx_max_nsts;
static int hf_ieee80211_he_phy_cap_ndp_with_4x_he_ltf_32us;
static int hf_ieee80211_he_phy_cap_stbc_tx_lt_80mhz;
static int hf_ieee80211_he_phy_cap_stbc_rx_lt_80mhz;
static int hf_ieee80211_he_phy_cap_doppler_tx;
static int hf_ieee80211_he_phy_cap_doppler_rx;
static int hf_ieee80211_he_phy_cap_full_bw_ul_mu_mimo;
static int hf_ieee80211_he_phy_cap_partial_bw_ul_mu_mimo;
static int hf_ieee80211_he_phy_b24_to_b39;
static int hf_ieee80211_he_phy_cap_dcm_max_constellation_tx;
static int hf_ieee80211_he_phy_cap_dcm_max_nss_tx;
static int hf_ieee80211_he_phy_cap_dcm_max_constellation_rx;
static int hf_ieee80211_he_phy_cap_dcm_max_nss_rx;
static int hf_ieee80211_he_phy_cap_rx_partial_bw_su_20mhz_he_mu_ppdu;
static int hf_ieee80211_he_phy_cap_su_beamformer;
static int hf_ieee80211_he_phy_cap_su_beamformee;
static int hf_ieee80211_he_phy_cap_mu_beamformer;
static int hf_ieee80211_he_phy_cap_beamformee_sts_lte_80mhz;
static int hf_ieee80211_he_phy_cap_beamformee_sts_gt_80mhz;
static int hf_ieee80211_he_phy_b40_to_b55;
static int hf_ieee80211_he_phy_cap_number_of_sounding_dims_lte_80;
static int hf_ieee80211_he_phy_cap_number_of_sounding_dims_gt_80;
static int hf_ieee80211_he_phy_cap_ng_eq_16_su_fb;
static int hf_ieee80211_he_phy_cap_ng_eq_16_mu_fb;
static int hf_ieee80211_he_phy_cap_codebook_size_eq_4_2_fb;
static int hf_ieee80211_he_phy_cap_codebook_size_eq_7_5_fb;
static int hf_ieee80211_he_phy_cap_triggered_su_beamforming_fb;
static int hf_ieee80211_he_phy_cap_triggered_mu_beamforming_fb;
static int hf_ieee80211_he_phy_cap_triggered_cqi_fb;
static int hf_ieee80211_he_phy_cap_partial_bw_extended_range;
static int hf_ieee80211_he_phy_cap_partial_bw_dl_mu_mimo;
static int hf_ieee80211_he_phy_cap_ppe_threshold_present;
static int hf_ieee80211_he_phy_b56_to_b71;
static int hf_ieee80211_he_phy_cap_psr_based_sr_support;
static int hf_ieee80211_he_phy_cap_power_boost_factor_ar_support;
static int hf_ieee80211_he_phy_cap_he_su_ppdu_etc_gi;
static int hf_ieee80211_he_phy_cap_max_nc;
static int hf_ieee80211_he_phy_cap_stbc_tx_gt_80_mhz;
static int hf_ieee80211_he_phy_cap_stbc_rx_gt_80_mhz;
static int hf_ieee80211_he_phy_cap_he_er_su_ppdu_4xxx_gi;
static int hf_ieee80211_he_phy_cap_20mhz_in_40mhz_24ghz_band;
static int hf_ieee80211_he_phy_cap_20mhz_in_160_80p80_ppdu;
static int hf_ieee80211_he_phy_cap_80mgz_in_160_80p80_ppdu;
static int hf_ieee80211_he_phy_cap_he_er_su_ppdu_1xxx_gi;
static int hf_ieee80211_he_phy_cap_midamble_tx_rx_2x_xxx_ltf;
static int hf_ieee80211_he_phy_b72_to_b87;
static int hf_ieee80211_he_phy_cap_dcm_max_ru;
static int hf_ieee80211_he_phy_cap_longer_than_16_he_sigb_ofdm_symbol_support;
static int hf_ieee80211_he_phy_cap_non_triggered_cqi_feedback;
static int hf_ieee80211_he_phy_cap_tx_1024_qam_242_tone_ru_support;
static int hf_ieee80211_he_phy_cap_rx_1024_qam_242_tone_ru_support;
static int hf_ieee80211_he_phy_cap_rx_full_bw_su_using_he_muppdu_w_compressed_sigb;
static int hf_ieee80211_he_phy_cap_rx_full_bw_su_using_he_muppdu_w_non_compressed_sigb;
static int hf_ieee80211_he_phy_cap_nominal_packet_padding;
static int hf_ieee80211_he_phy_cap_he_mu_ppdu_ru_rx_max;
static int hf_ieee80211_he_phy_cap_b81_b87_reserved;
static int hf_ieee80211_he_operation_parameter;
static int hf_ieee80211_he_operation_default_pe_duration;
static int hf_ieee80211_he_operation_twt_required;
static int hf_ieee80211_he_operation_txop_duration_rts_threshold;
static int hf_ieee80211_he_operation_vht_operation_information_present;
static int hf_ieee80211_he_operation_co_hosted_bss;
static int hf_ieee80211_he_operation_er_su_disable;
static int hf_ieee80211_he_operation_6ghz_operation_information_present;
static int hf_ieee80211_he_operation_reserved_b16_b23;
static int hf_ieee80211_he_bss_color_information;
static int hf_ieee80211_he_bss_color_info_bss_color;
static int hf_ieee80211_he_bss_color_partial_bss_color;
static int hf_ieee80211_he_bss_color_bss_color_disabled;
static int hf_ieee80211_he_operation_basic_mcs;
static int hf_ieee80211_he_oper_max_he_mcs_for_1_ss;
static int hf_ieee80211_he_oper_max_he_mcs_for_2_ss;
static int hf_ieee80211_he_oper_max_he_mcs_for_3_ss;
static int hf_ieee80211_he_oper_max_he_mcs_for_4_ss;
static int hf_ieee80211_he_oper_max_he_mcs_for_5_ss;
static int hf_ieee80211_he_oper_max_he_mcs_for_6_ss;
static int hf_ieee80211_he_oper_max_he_mcs_for_7_ss;
static int hf_ieee80211_he_oper_max_he_mcs_for_8_ss;
static int hf_ieee80211_he_operation_channel_width;
static int hf_ieee80211_he_operation_channel_center_freq_0;
static int hf_ieee80211_he_operation_channel_center_freq_1;
static int hf_ieee80211_he_operation_max_co_hosted_bssid_indicator;
static int hf_ieee80211_he_operation_6ghz_primary_channel;
static int hf_ieee80211_he_operation_6ghz_control;
static int hf_ieee80211_he_operation_6ghz_control_channel_width;
static int hf_ieee80211_he_operation_6ghz_control_duplicate_beacon;
static int hf_ieee80211_he_operation_6ghz_control_regulatory_info;
static int hf_ieee80211_he_operation_6ghz_control_reserved;
static int hf_ieee80211_he_operation_6ghz_channel_center_freq_0;
static int hf_ieee80211_he_operation_6ghz_channel_center_freq_1;
static int hf_ieee80211_he_operation_6ghz_minimum_rate;
static int hf_ieee80211_he_muac_aci_aifsn;
static int hf_ieee80211_he_muac_aifsn;
static int hf_ieee80211_he_muac_acm;
static int hf_ieee80211_he_muac_aci;
static int hf_ieee80211_he_muac_reserved;
static int hf_ieee80211_he_mu_edca_timer;
static int hf_ieee80211_he_muac_ecwmin_ecwmax;
static int hf_ieee80211_he_srp_disallowed;
static int hf_ieee80211_he_non_srg_obss_pd_sr_disallowed;
static int hf_ieee80211_he_non_srg_offset_present;
static int hf_ieee80211_he_srg_information_present;
static int hf_ieee80211_he_hesiga_spatial_reuse_value15_allowed;
static int hf_ieee80211_he_sr_control_reserved;
static int hf_ieee80211_he_spatial_reuse_sr_control;
static int hf_ieee80211_he_spatial_non_srg_obss_pd_max_offset;
static int hf_ieee80211_he_spatial_srg_obss_pd_min_offset;
static int hf_ieee80211_he_spatial_srg_obss_pd_max_offset;
static int hf_ieee80211_he_spatial_srg_bss_color_bitmap;
static int hf_ieee80211_he_spatial_srg_partial_bssid_bitmap;
static int hf_ieee80211_he_ess_report_planned_ess;
static int hf_ieee80211_he_ess_report_edge_of_ess;
static int hf_ieee80211_he_resource_request_buffer_thresh;
static int hf_ieee80211_he_bss_color_change_new_color_info;
static int hf_ieee80211_he_new_bss_color_info_color;
static int hf_ieee80211_he_new_bss_color_info_reserved;
static int hf_ieee80211_he_bss_color_change_switch_countdown;
static int hf_ieee80211_he_ess_report_info_field;
static int hf_ieee80211_he_ess_report_recommend_transition_thresh;
static int hf_ieee80211_he_ops_duration;
static int hf_ieee80211_he_uora_field;
static int hf_ieee80211_he_uora_eocwmin;
static int hf_ieee80211_he_uora_owcwmax;
static int hf_ieee80211_he_uora_reserved;

static int hf_ieee80211_max_channel_switch_time;

static int hf_ieee80211_oci_operating_class;
static int hf_ieee80211_oci_primary_channel_number;
static int hf_ieee80211_oci_frequency_segment_1;
static int hf_ieee80211_oci_oct_operating_class;
static int hf_ieee80211_oci_oct_primary_channel_number;
static int hf_ieee80211_oci_oct_frequency_segment_1;

static int hf_ieee80211_multiple_bssid_configuration_bssid_count;
static int hf_ieee80211_multiple_bssid_configuration_full_set_rx_periodicity;
static int hf_ieee80211_known_bssid_bitmap;
static int hf_ieee80211_short_ssid;

static int hf_ieee80211_non_inheritance_element_id_list_length;
static int hf_ieee80211_non_inheritance_element_id_list_element_id;
static int hf_ieee80211_non_inheritance_element_id_ext_list_length;
static int hf_ieee80211_non_inheritance_element_id_ext_list_element_id_ext;

static int hf_ieee80211_rejected_groups_group;

static int hf_ieee80211_twt_bcast_flow;
static int hf_ieee80211_twt_individual_flow;
static int hf_ieee80211_twt_individual_flow_id;
static int hf_ieee80211_twt_individual_reserved;
static int hf_ieee80211_twt_bcast_id;
static int hf_ieee80211_twt_neg_type;
static int hf_ieee80211_twt_neg_type2_reserved1;
static int hf_ieee80211_twt_neg_type2_reserved2;
static int hf_ieee80211_twt_bcast_teardown_all;
static int hf_ieee80211_twt_bcast_teardown_all_twt;
static int hf_ieee80211_twt_bcast_twt_id_reserved;
static int hf_ieee80211_twt_bcast_neg_type_reserved;

static int hf_ieee80211_tag_twt_control_field;
static int hf_ieee80211_tag_twt_ndp_paging_indicator;
static int hf_ieee80211_tag_twt_responder_pm_mode;
static int hf_ieee80211_tag_twt_neg_type;
static int hf_ieee80211_tag_twt_info_frame_disabled;
static int hf_ieee80211_tag_twt_wake_duration_unit;
static int hf_ieee80211_tag_twt_link_id_bitmap_present;
static int hf_ieee80211_tag_twt_aligned_twt;

static int hf_ieee80211_tag_twt_req_type_field;
static int hf_ieee80211_tag_twt_req_type_req;
static int hf_ieee80211_tag_twt_req_type_setup_cmd;
static int hf_ieee80211_tag_twt_req_type_trigger;
static int hf_ieee80211_tag_twt_req_type_implicit;
static int hf_ieee80211_tag_twt_req_type_flow_type;
static int hf_ieee80211_tag_twt_req_type_flow_id;
static int hf_ieee80211_tag_twt_req_type_wake_int_exp;
static int hf_ieee80211_tag_twt_req_type_prot;
static int hf_ieee80211_tag_twt_req_type_last_bcst_parm_set;
static int hf_ieee80211_tag_twt_req_type_bcst_twt_recom;
static int hf_ieee80211_tag_twt_req_type_aligned;

static int hf_ieee80211_tag_twt_ndp_paging_field;
static int hf_ieee80211_tag_twt_ndp_paging_p_id;
static int hf_ieee80211_tag_twt_ndp_max_ndp_paging_period;
static int hf_ieee80211_tag_twt_ndp_partial_tsf_offset;
static int hf_ieee80211_tag_twt_ndp_action;
static int hf_ieee80211_tag_twt_ndp_min_sleep_duration;
static int hf_ieee80211_tag_twt_ndp_reserved;
static int hf_ieee80211_tag_twt_link_id_bitmap;
static int hf_ieee80211_tag_twt_aligned_twt_link_bitmap;
static int hf_ieee80211_tag_twt_broadcast_info;
static int hf_ieee80211_tag_twt_bcast_info_persistence;
static int hf_ieee80211_tag_twt_bcast_info_id;
static int hf_ieee80211_tag_twt_bcast_info_rtwt_traffic_present;
static int hf_ieee80211_tag_twt_bcast_info_rtwt_sche_info;
static int hf_ieee80211_tag_twt_traffic_info_control;
static int hf_ieee80211_tag_twt_traffic_info_dl_bitmap_valid;
static int hf_ieee80211_tag_twt_traffic_info_ul_bitmap_valid;
static int hf_ieee80211_tag_twt_traffic_info_reserved;
static int hf_ieee80211_tag_twt_traffic_info_rtwt_dl_bitmap;
static int hf_ieee80211_tag_twt_traffic_info_rtwt_ul_bitmap;

static int hf_ieee80211_tag_twt_target_wake_time;
static int hf_ieee80211_tag_twt_target_wake_time_short;
static int hf_ieee80211_tag_twt_nom_min_twt_wake_dur;
static int hf_ieee80211_tag_twt_wake_interval_mantissa;
static int hf_ieee80211_tag_twt_channel;

static int hf_ieee80211_tag_rsnx;
/* octet 1 */
static int hf_ieee80211_tag_rsnx_length;
static int hf_ieee80211_tag_rsnx_protected_twt_operations_support;
static int hf_ieee80211_tag_rsnx_sae_hash_to_element;
static int hf_ieee80211_tag_rsnx_sae_pk;
static int hf_ieee80211_tag_rsnx_protected_wur_frame_support;
/* octet 2 */
static int hf_ieee80211_tag_rsnx_secure_ltf_support;
static int hf_ieee80211_tag_rsnx_secure_rtt_supported;
static int hf_ieee80211_tag_rsnx_urnm_mfpr_x20;
static int hf_ieee80211_tag_rsnx_protected_announce_support;
static int hf_ieee80211_tag_rsnx_pbac;
static int hf_ieee80211_tag_rsnx_extended_s1g_action_protection;
static int hf_ieee80211_tag_rsnx_spp_amsdu_capable;
static int hf_ieee80211_tag_rsnx_urnm_mfpr;
static int hf_ieee80211_tag_rsnx_reserved;

static int hf_ieee80211_wfa_rsn_selection;
static int hf_ieee80211_wfa_rsn_or_link_kde_link_id;

static int hf_ieee80211_nonap_sta_regulatory_conn;
static int hf_ieee80211_nonap_sta_regu_conn_indoor_ap_valid;
static int hf_ieee80211_nonap_sta_regu_conn_indoor_ap;
static int hf_ieee80211_nonap_sta_regu_conn_sp_ap_valid;
static int hf_ieee80211_nonap_sta_regu_conn_sp_ap;
static int hf_ieee80211_nonap_sta_regu_conn_reserved;

static int hf_ieee80211_tag_channel_usage_mode;

static int hf_ieee80211_ff_count;

static int hf_ieee80211_tag_dms_id;
static int hf_ieee80211_tag_dms_length;
static int hf_ieee80211_tag_dms_req_type;
static int hf_ieee80211_tag_dms_resp_type;
static int hf_ieee80211_tag_dms_last_seq_control;

/* ************************************************************************* */
/*                              RFC 8110 fields                              */
/* ************************************************************************* */
static int hf_ieee80211_owe_dh_parameter_group;
static int hf_ieee80211_owe_dh_parameter_public_key;

/* ************************************************************************* */
/*                               Protocol trees                              */
/* ************************************************************************* */
static int ett_80211;
static int ett_proto_flags;
static int ett_cap_tree;
static int ett_fc_tree;
static int ett_cntrl_wrapper_fc;
static int ett_cntrl_wrapper_payload;
static int ett_fragments;
static int ett_fragment;
static int ett_block_ack;
static int ett_block_ack_tid;
static int ett_block_ack_request_control;
static int ett_block_ack_bitmap;
static int ett_block_ack_request_multi_sta_aid_tid;
static int ett_multi_sta_block_ack;
static int ett_ath_cap_tree;
static int ett_extreme_mesh_services_tree;
static int ett_addr;

static int ett_80211_mgt;
static int ett_fixed_parameters;
static int ett_tagged_parameters;
static int ett_tag_bmapctl_tree;
static int ett_s1g_pvb_tree;
static int ett_s1g_pvb_eb_tree;
static int ett_s1g_pvb_block_control_byte;
static int ett_s1g_pvb_block_bitmap_tree;
static int ett_s1g_pvb_subblock_tree;
static int ett_s1g_pvb_olb_tree;
static int ett_s1g_pvb_olb_subblock;
static int ett_s1g_pvb_ade_tree;
static int ett_s1g_pvb_ade_control;
static int ett_tag_country_fnm_tree;
static int ett_tag_country_rcc_tree;
static int ett_qos_parameters;
static int ett_qos_ps_buf_state;
static int ett_wep_parameters;
static int ett_msh_control;
static int ett_hwmp_targ_flags_tree;
static int ett_mesh_chswitch_flag_tree;
static int ett_mesh_config_cap_tree;
static int ett_mesh_formation_info_tree;
static int ett_bcn_timing_rctrl_tree;
static int ett_bcn_timing_info_tree;
static int ett_gann_flags_tree;
static int ett_pxu_proxy_info_tree;
static int ett_pxu_proxy_info_flags_tree;

static int ett_rsn_gcs_tree;
static int ett_rsn_pcs_tree;
static int ett_rsn_sub_pcs_tree;
static int ett_rsn_akms_tree;
static int ett_rsn_sub_akms_tree;
static int ett_rsn_cap_tree;
static int ett_rsn_pmkid_tree;
static int ett_rsn_gmcs_tree;

static int ett_kde_mlo_link_info;

static int ett_wpa_mcs_tree;
static int ett_wpa_ucs_tree;
static int ett_wpa_sub_ucs_tree;
static int ett_wpa_akms_tree;
static int ett_wpa_sub_akms_tree;
static int ett_wme_ac;
static int ett_wme_aci_aifsn;
static int ett_wme_ecw;
static int ett_wme_qos_info;

static int ett_update_edca_info;

static int ett_ht_cap_tree;
static int ett_ampduparam_tree;
static int ett_mcsset_tree;
static int ett_mcsbit_tree;
static int ett_htex_cap_tree;
static int ett_txbf_tree;
static int ett_antsel_tree;
static int ett_hta_cap_tree;
static int ett_hta_cap1_tree;
static int ett_hta_cap2_tree;

static int ett_s1g_ndp;
static int ett_s1g_ndp_ack;
static int ett_s1g_ndp_cts;
static int ett_s1g_ndp_cf_end;
static int ett_s1g_ndp_ps_poll;
static int ett_s1g_ndp_ps_poll_ack;
static int ett_s1g_ndp_block_ack;
static int ett_s1g_ndp_beamforming_report_poll;
static int ett_s1g_ndp_paging;
static int ett_s1g_ndp_probe;
static int ett_pv1_sid;
static int ett_pv1_sid_field;
static int ett_pv1_seq_control;
static int ett_ieee80211_s1g_capabilities_info;
static int ett_ieee80211_s1g_capabilities;
static int ett_s1g_cap_byte1;
static int ett_s1g_cap_byte2;
static int ett_s1g_cap_byte3;
static int ett_s1g_cap_byte4;
static int ett_s1g_cap_byte5;
static int ett_s1g_cap_byte6;
static int ett_s1g_cap_byte7;
static int ett_s1g_cap_byte8;
static int ett_s1g_cap_byte9;
static int ett_s1g_cap_byte10;
static int ett_ieee80211_s1g_sup_mcs_and_nss_set;
static int ett_s1g_mcs_and_mcs_set;
static int ett_s1g_operation_info;
static int ett_s1g_channel_width;
static int ett_s1g_subchannel_selective_transmission;
static int ett_s1g_raw_assignment;
static int ett_s1g_raw_assn_tree;
static int ett_s1g_raw_control;
static int ett_s1g_raw_slot_def;
static int ett_s1g_raw_group_subfield;
static int ett_s1g_raw_channel_indication;
static int ett_s1g_page_slice_control;
static int ett_s1g_aid_request_mode;
static int ett_s1g_aid_characteristic;
static int ett_s1g_sector_operation;
static int ett_tack_info;
static int ett_ieee80211_s1g_auth_control;
static int ett_s1g_relay_control;
static int ett_s1g_relay_function;
static int ett_ieee80211_s1g_addr_list;
static int ett_ieee80211_s1g_reach_addr;
static int ett_s1g_relay_discovery_control;
static int ett_ieee80211_s1g_aid_entry;
static int ett_s1g_probe_resp_subfield_0;
static int ett_s1g_header_comp_control;
static int ett_pv1_mgmt_action;
static int ett_pv1_mgmt_action_no_ack;
static int ett_pv1_cntl_stack;
static int ett_pv1_cntl_bat;

static int ett_htc_tree;
static int ett_htc_he_a_control;
static int ett_mfb_subtree;
static int ett_lac_subtree;
static int ett_ieee80211_buffer_status_report;
static int ett_ieee80211_a_control_padding;
static int ett_ieee80211_a_control_ones;
static int ett_ieee80211_triggered_response_schedule;
static int ett_ieee80211_control_om;
static int ett_ieee80211_hla_control;
static int ett_ieee80211_control_uph;
static int ett_ieee80211_buffer_control_bqr;
static int ett_ieee80211_control_cci;
static int ett_ieee80211_control_eht_om;
static int ett_ieee80211_control_srs;
static int ett_ieee80211_control_aar;

static int ett_vht_cap_tree;
static int ett_vht_mcsset_tree;
static int ett_vht_rx_mcsbit_tree;
static int ett_vht_tx_mcsbit_tree;
static int ett_vht_basic_mcsbit_tree;
static int ett_vht_op_tree;
static int ett_vht_tpe_info_tree;
static int ett_tpe_psd;

static int ett_vht_ranging_annc;
static int ett_ndp_ranging_annc_sta_info;

static int ett_ff_he_action;
static int ett_ff_protected_he_action;
static int ett_ff_protected_eht_action;
static int ett_ff_he_mimo_control;
static int ett_ff_he_mimo_beamforming_report_snr;
static int ett_ff_he_mimo_feedback_matrices;

static int ett_ff_vhtmimo_cntrl;
static int ett_ff_vhtmimo_beamforming_report;
static int ett_ff_vhtmimo_beamforming_report_snr;
static int ett_ff_vhtmimo_beamforming_angle;
static int ett_ff_vhtmimo_beamforming_report_feedback_matrices;
static int ett_ff_vhtmu_exclusive_beamforming_report_matrices;

static int ett_vht_grpidmgmt;
static int ett_vht_msa;
static int ett_vht_upa;

static int ett_ht_operation_info_delimiter1_tree;
static int ett_ht_operation_info_delimiter2_tree;
static int ett_ht_operation_info_delimiter3_tree;

static int ett_ff_ftm_param_delim1;
static int ett_ff_ftm_param_delim2;
static int ett_ff_ftm_param_delim3;
static int ett_ff_ftm_tod_err1;
static int ett_ff_ftm_toa_err1;
static int ett_tag_ranging;
static int ett_tag_ranging_ntb;
static int ett_tag_ranging_secure_he_ltf;

static int ett_ranging_subelement_tree;

static int ett_tag_direct_meas_results;

static int ett_rsta_avail_header;
static int ett_rsta_avail_tree;
static int ett_rsta_avail_subfield;

static int ett_pasn_parameters;
static int ett_pasn_comeback_tree;

static int ett_pasn_auth_frame;

/* 802.11be trees */
static int ett_eht_multi_link_control;
static int ett_eht_multi_link_common_info;
static int ett_eht_multi_link_common_info_link_id;
static int ett_eht_multi_link_common_info_medium_sync;
static int ett_eht_multi_link_common_info_eml_capa;
static int ett_eht_multi_link_common_info_mld_capa;
static int ett_eht_multi_link_common_info_ext_mld_capa;
static int ett_eht_multi_link_per_sta;
static int ett_eht_multi_link_subelt;
static int ett_eht_multi_link_sta_control;
static int ett_eht_multi_link_per_sta_info;
static int ett_eht_multi_link_sta_dtim;
static int ett_eht_multi_link_reconf_oper_param;
static int ett_eht_multi_link_reconfig_presence_indi;
static int ett_eht_multi_link_reconfig_operation_para_info;
static int ett_eht_operation_params;
static int ett_eht_operation_control;
static int ett_eht_mac_capa;
static int ett_eht_phy_capa;
static int ett_eht_phy_bits_0_15;
static int ett_eht_phy_bits_16_31;
static int ett_eht_phy_bits_32_39;
static int ett_eht_phy_bits_40_63;
static int ett_eht_phy_bits_64_71;
static int ett_eht_phy_mcs_nss;
static int ett_eht_phy_mcs_nss_set;
static int ett_eht_ttl_mapping;
static int ett_eht_ttl_mapping_link_mapping;
static int ett_eht_eht_multi_link_tc;
static int ett_eht_qos_characteristics;
static int ett_eht_aid_bmapctl_tree;
static int ett_eht_bw_indication_param;
static int ett_eht_eml_control;
static int ett_eht_eml_control_link_map;
static int ett_eht_eml_control_mcs_map_count;
static int ett_eht_emlsr_para_update;
static int ett_eht_mimo_ctrl;
static int ett_eht_beamforming_rpt_ru_index;
static int ett_eht_beamforming_feedback_tree;
static int ett_ff_eht_mimo_beamforming_report_snr;
static int ett_ff_eht_mimo_mu_exclusive_report;
static int ett_eht_mu_exclusive_beamforming_rpt_ru_index;
static int ett_eht_reconfig_status_list;
static int ett_eht_group_key_data;

static int ett_tag_measure_request_mode_tree;
static int ett_tag_measure_request_type_tree;
static int ett_tag_measure_request_sub_element_tree;
static int ett_tag_measure_report_mode_tree;
static int ett_tag_measure_report_type_tree;
static int ett_tag_measure_report_basic_map_tree;
static int ett_tag_measure_report_rpi_tree;
static int ett_tag_measure_report_frame_tree;
static int ett_tag_measure_report_sub_element_tree;
static int ett_tag_measure_reported_frame_tree;
static int ett_tag_measure_reported_frame_frag_id_tree;
static int ett_tag_measure_reported_lci_z_tree;
static int ett_tag_measure_reported_lci_urp_tree;
static int ett_tag_bss_bitmask_tree;
static int ett_tag_dfs_map_tree;
static int ett_tag_dfs_map_flags_tree;
static int ett_tag_erp_info_tree;
static int ett_tag_ex_cap1;
static int ett_tag_ex_cap2;
static int ett_tag_ex_cap3;
static int ett_tag_ex_cap4;
static int ett_tag_ex_cap5;
static int ett_tag_ex_cap6;
static int ett_tag_ex_cap7;
static int ett_tag_ex_cap8;
static int ett_tag_ex_cap89;
static int ett_tag_ex_cap10;
static int ett_tag_ex_cap11;
static int ett_tag_ex_cap12;
static int ett_tag_ex_cap13;
static int ett_tag_ex_cap14;

static int ett_tag_rm_cap1;
static int ett_tag_rm_cap2;
static int ett_tag_rm_cap3;
static int ett_tag_rm_cap4;
static int ett_tag_rm_cap5;

static int ett_tag_rsnx_octet1;
static int ett_tag_rsnx_octet2;

static int ett_tag_multiple_bssid_subelem_tree;
static int ett_tag_20_40_bc;

static int ett_tag_intolerant_tree;

static int ett_tag_tclas_mask_tree;

static int ett_tag_supported_channels;

static int ett_tag_neighbor_report_bssid_info_tree;
static int ett_tag_neighbor_report_bssid_info_capability_tree;
static int ett_tag_neighbor_report_subelement_tree;
static int ett_tag_neighbor_report_sub_tag_tree;

static int ett_tag_wapi_param_set_akm_tree;
static int ett_tag_wapi_param_set_ucast_tree;
static int ett_tag_wapi_param_set_mcast_tree;
static int ett_tag_wapi_param_set_preauth_tree;

static int ett_max_idle_period_options;

static int ett_tag_time_adv_tree;

static int ett_tag_he_6ghz_cap_inf_tree;

static int ett_ff_ba_param_tree;
static int ett_ff_ba_ssc_tree;
static int ett_ff_delba_param_tree;
static int ett_ff_qos_info;
static int ett_ff_sm_pwr_save;
static int ett_ff_psmp_param_set;
static int ett_ff_mimo_cntrl;
static int ett_ff_ant_sel;
static int ett_mimo_report;
static int ett_ff_chan_switch_announce;
static int ett_ff_ht_info;
static int ett_ff_psmp_sta_info;

static int ett_tpc;

static int ett_msdu_aggregation_parent_tree;
static int ett_msdu_aggregation_subframe_tree;

static int ett_80211_mgt_ie;
static int ett_tsinfo_tree;
static int ett_sched_tree;

static int ett_fcs;

static int ett_hs20_osu_providers_list;
static int ett_hs20_osu_provider_tree;
static int ett_hs20_friendly_names_list;
static int ett_hs20_friendly_name_tree;
static int ett_hs20_osu_provider_method_list;
static int ett_osu_icons_avail_list;
static int ett_hs20_osu_icon_tree;
static int ett_hs20_osu_service_desc_list;
static int ett_hs20_osu_service_desc_tree;
static int ett_hs20_venue_url;
static int ett_hs20_advice_of_charge;
static int ett_hs20_aoc_plan;

static int ett_hs20_ofn_tree;

static int ett_adv_proto;
static int ett_adv_proto_tuple;
static int ett_gas_query;
static int ett_gas_anqp;
static int ett_nai_realm;
static int ett_nai_realm_eap;
static int ett_tag_ric_data_desc_ie;
static int ett_anqp_vendor_capab;

static int ett_osen_group_data_cipher_suite;
static int ett_osen_pairwise_cipher_suites;
static int ett_osen_pairwise_cipher_suite;
static int ett_osen_akm_cipher_suites;
static int ett_osen_akm_cipher_suite;
static int ett_osen_rsn_cap_tree;
static int ett_osen_pmkid_list;
static int ett_osen_pmkid_tree;
static int ett_osen_group_management_cipher_suite;

static int ett_hs20_cc_proto_port_tuple;

static int ett_tag_no_bssid_capability_dmg_bss_control_tree;
static int ett_ssid_list;

static int ett_sgdsn;
static int ett_nintendo;

static int ett_routerboard;

static int ett_meru;

static int ett_wisun_gtkl;
static int ett_wisun_lgtkl;

static int ett_qos_map_set_exception;
static int ett_qos_map_set_range;

static int ett_wnm_notif_subelt;

static int ett_ieee80211_3gpp_plmn;

static int ett_mbo_oce_attr;
static int ett_mbo_ap_cap;
static int ett_oce_cap;
static int ett_oce_metrics_cap;

static int ett_tag_mobility_domain_ft_capab_tree;

static int ett_tag_ft_mic_control_tree;
static int ett_tag_ft_subelem_tree;

static int ett_qos_mgmt_pol_capa;
static int ett_qos_mgmt_attributes;
static int ett_qos_mgmt_dscp_policy_capabilities;
static int ett_qos_mgmt_dscp_policy;
static int ett_qos_mgmt_tclas;
static int ett_qos_mgmt_domain_name;
static int ett_qos_mgmt_unknown_attribute;
static int ett_dscp_policy_status_list;
static int ett_pol_rqst_cont_tree;
static int ett_pol_resp_cont_tree;

static int ett_wfa_capa;
static int ett_wfa_capa_attributes;
static int ett_wfa_capa_supp_gene;
static int ett_wfa_capa_cert_gene;

static expert_field ei_ieee80211_bad_length;
static expert_field ei_ieee80211_inv_val;
static expert_field ei_ieee80211_vht_tpe_pwr_info_count;
static expert_field ei_ieee80211_vht_tpe_pwr_info_unit;
static expert_field ei_ieee80211_ff_query_response_length;
static expert_field ei_ieee80211_ff_anqp_nai_realm_eap_len;
static expert_field ei_hs20_anqp_nai_hrq_length;
static expert_field ei_ieee80211_extra_data;
static expert_field ei_ieee80211_tag_data;
static expert_field ei_ieee80211_tdls_setup_confirm_malformed;
static expert_field ei_ieee80211_ff_anqp_nai_field_len;
static expert_field ei_ieee80211_rsn_pcs_count;
static expert_field ei_ieee80211_tag_measure_request_unknown;
static expert_field ei_ieee80211_tag_measure_request_beacon_unknown;
static expert_field ei_ieee80211_tag_measure_report_unknown;
static expert_field ei_ieee80211_tag_measure_report_beacon_unknown;
static expert_field ei_ieee80211_tag_measure_report_lci_unknown;
static expert_field ei_ieee80211_tag_number;
static expert_field ei_ieee80211_ff_anqp_info_length;
static expert_field ei_hs20_anqp_ofn_length;
static expert_field ei_ieee80211_tdls_setup_response_malformed;
static expert_field ei_ieee80211_ff_anqp_capability;
static expert_field ei_ieee80211_not_enough_room_for_anqp_header;
static expert_field ei_ieee80211_ff_query_request_length;
static expert_field ei_ieee80211_wfa_ie_wme_qos_info_bad_ftype;
static expert_field ei_ieee80211_qos_info_bad_ftype;
static expert_field ei_ieee80211_qos_bad_aifsn;
static expert_field ei_ieee80211_pmkid_count_too_large;
static expert_field ei_ieee80211_ff_anqp_venue_length;
static expert_field ei_ieee80211_ff_anqp_roaming_consortium_oi_len;
static expert_field ei_ieee80211_tag_length;
static expert_field ei_ieee80211_missing_data;
static expert_field ei_ieee80211_rsn_pmkid_count;
static expert_field ei_ieee80211_fc_retry;
static expert_field ei_ieee80211_tag_wnm_sleep_mode_no_key_data;
static expert_field ei_ieee80211_dmg_subtype;
static expert_field ei_ieee80211_vht_action;
static expert_field ei_ieee80211_mesh_peering_unexpected;
static expert_field ei_ieee80211_fcs;
static expert_field ei_ieee80211_mismatched_akm_suite;
static expert_field ei_ieee80211_vs_routerboard_unexpected_len;
static expert_field ei_ieee80211_vs_sgdsn_serialnumber_invalid_len_val;
static expert_field ei_ieee80211_vs_sgdsn_serialnumber_unexpected_len_val;
static expert_field ei_ieee80211_twt_tear_down_bad_neg_type;
static expert_field ei_ieee80211_twt_setup_bad_command;
static expert_field ei_ieee80211_twt_bcast_info_no_term;
static expert_field ei_ieee80211_invalid_control_word;
static expert_field ei_ieee80211_invalid_control_id;
static expert_field ei_ieee80211_invalid_control_length;
static expert_field ei_ieee80211_wfa_60g_attr_len_invalid;
static expert_field ei_ieee80211_wfa_60g_unknown_attribute;
static expert_field ei_ieee80211_htc_in_dmg_packet;
static expert_field ei_ieee80211_eht_invalid_subelement;
static expert_field ei_ieee80211_eht_invalid_action;
static expert_field ei_ieee80211_eht_invalid_multi_link;
static expert_field ei_ieee80211_eht_invalid_nc_nr;


/* 802.11ad trees */
static int ett_dynamic_alloc_tree;
static int ett_ssw_tree;
static int ett_bf_tree;
static int ett_sswf_tree;
static int ett_brp_tree;
static int ett_blm_tree;
static int ett_bic_tree;
static int ett_dmg_params_tree;
static int ett_cc_tree;
static int ett_rcsi_tree;
static int ett_80211_ext;
static int ett_allocation_tree;
static int ett_sta_info;

static int ett_ieee80211_esp;

static int ett_ieee80211_wfa_60g_attr;
static int ett_ieee80211_wfa_transition_disable_tree;

/* 802.11ah trees */
static int ett_s1g_sync_control_tree;
static int ett_s1g_sector_id_index;
static int ett_s1g_twt_information_control;
static int ett_twt_tear_down_tree;
static int ett_twt_control_field_tree;
static int ett_twt_req_type_tree;
static int ett_twt_ndp_paging_field_tree;
static int ett_twt_broadcast_info_tree;
static int ett_twt_traffic_info_tree;
static int ett_twt_traffic_info_control_tree;

/* 802.11ax trees */
static int ett_he_mac_capabilities;
static int ett_he_phy_capabilities;
static int ett_he_phy_cap_first_byte;
static int ett_he_phy_cap_chan_width_set;
static int ett_he_phy_cap_b8_to_b23;
static int ett_he_phy_cap_b24_to_b39;
static int ett_he_phy_cap_b40_to_b55;
static int ett_he_phy_cap_b56_to_b71;
static int ett_he_phy_cap_b72_to_b87;
static int ett_he_mcs_and_nss_set;
static int ett_he_rx_tx_he_mcs_map_lte_80;
static int ett_he_rx_mcs_map_lte_80;
static int ett_he_tx_mcs_map_lte_80;
static int ett_he_rx_tx_he_mcs_map_160;
static int ett_he_rx_mcs_map_160;
static int ett_he_tx_mcs_map_160;
static int ett_he_rx_tx_he_mcs_map_80_80;
static int ett_he_rx_mcs_map_80_80;
static int ett_he_tx_mcs_map_80_80;
static int ett_he_ppe_threshold;
static int ett_he_ppe_nss;
static int ett_he_ppe_ru_alloc;
static int ett_he_uora_tree;
static int ett_he_aic_aifsn;
static int ett_he_spatial_reuse_control;
static int ett_he_bss_new_color_info;
static int ett_he_ess_report_info_field;
static int ett_he_operation_params;
static int ett_he_bss_color_information;
static int ett_he_oper_basic_mcs;
static int ett_he_operation_vht_op_info;
static int ett_mscs_user_prio;
static int ett_ieee80211_user_prio_bitmap;
static int ett_ieee80211_intra_access_prio;
static int ett_he_operation_6ghz;
static int ett_he_operation_6ghz_control;
static int ett_he_mu_edca_param;
static int ett_he_trigger_common_info;
static int ett_he_trigger_ranging;
static int ett_he_trigger_ranging_poll;
static int ett_he_trigger_packet_extension;
static int ett_he_trigger_base_common_info;
static int ett_he_trigger_bar_ctrl;
static int ett_he_trigger_bar_info;
static int ett_he_trigger_user_info;
static int ett_he_trigger_base_user_info;
static int ett_he_trigger_dep_basic_user_info;
static int ett_he_trigger_dep_nfrp_user_info;
static int ett_ndp_annc;
static int ett_ndp_vht_annc_sta_list;
static int ett_ndp_vht_annc_sta_info_tree;
static int ett_ndp_he_annc_sta_list;
static int ett_ndp_he_annc_sta_item;
static int ett_ndp_he_annc_sta_info;
static int ett_ndp_ranging_annc_sta_list;
static int ett_ndp_eht_annc_sta_list;
static int ett_ndp_eht_annc_sta_info;
static int ett_non_inheritance_element_id_list;
static int ett_non_inheritance_element_id_ext_list;

/* 802.11ai trees */
static int ett_fils_indication_realm_list;
static int ett_fils_indication_public_key_list;
static int ett_ff_fils_discovery_frame_control;
static int ett_ff_fils_discovery_capability;
static int ett_neighbor_ap_info;
static int ett_tbtt_infos;
static int ett_rnr_bss_params_tree;
static int ett_rnr_mld_params_tree;

static int ett_ff_fils_req_params;
static int ett_ff_fils_req_params_fils_criteria;

static int ett_nonap_sta_regulatory_conn;

static int ett_chan_usage;

/* Generic address HF list for proto_tree_add_mac48_detail() */
static const mac_hf_list_t mac_addr = {
  &hf_ieee80211_addr,
  &hf_ieee80211_addr_resolved,
  &hf_ieee80211_addr_oui,
  &hf_ieee80211_addr_oui_resolved,
  &hf_ieee80211_addr_lg,
  &hf_ieee80211_addr_ig,
};

/* Generic address HF list for proto_tree_add_mac48_detail() --
 * no LG/IG bits */
static const mac_hf_list_t mac_addr_hidden = {
  &hf_ieee80211_addr,
  &hf_ieee80211_addr_resolved,
  &hf_ieee80211_addr_oui,
  &hf_ieee80211_addr_oui_resolved,
  NULL,
  NULL,
};

/* Destination address HF list for proto_tree_add_mac48_detail() */
static const mac_hf_list_t mac_da = {
  &hf_ieee80211_addr_da,
  &hf_ieee80211_addr_da_resolved,
  &hf_ieee80211_addr_da_oui,
  &hf_ieee80211_addr_da_oui_resolved,
  &hf_ieee80211_addr_da_lg,
  &hf_ieee80211_addr_da_ig,
};

/* Source address HF list for proto_tree_add_mac48_detail() */
static const mac_hf_list_t mac_sa = {
  &hf_ieee80211_addr_sa,
  &hf_ieee80211_addr_sa_resolved,
  &hf_ieee80211_addr_sa_oui,
  &hf_ieee80211_addr_sa_oui_resolved,
  &hf_ieee80211_addr_sa_lg,
  &hf_ieee80211_addr_sa_ig,
};

/* Receiver address HF list for proto_tree_add_mac48_detail() */
static const mac_hf_list_t mac_ra = {
  &hf_ieee80211_addr_ra,
  &hf_ieee80211_addr_ra_resolved,
  &hf_ieee80211_addr_ra_oui,
  &hf_ieee80211_addr_ra_oui_resolved,
  &hf_ieee80211_addr_ra_lg,
  &hf_ieee80211_addr_ra_ig,
};

/* Transmitter address HF list for proto_tree_add_mac48_detail() */
static const mac_hf_list_t mac_ta = {
  &hf_ieee80211_addr_ta,
  &hf_ieee80211_addr_ta_resolved,
  &hf_ieee80211_addr_ta_oui,
  &hf_ieee80211_addr_ta_oui_resolved,
  &hf_ieee80211_addr_ta_lg,
  &hf_ieee80211_addr_ta_ig,
};

/* BSSID address HF list for proto_tree_add_mac48_detail() */
static const mac_hf_list_t mac_bssid = {
  &hf_ieee80211_addr_bssid,
  &hf_ieee80211_addr_bssid_resolved,
  &hf_ieee80211_addr_bssid_oui,
  &hf_ieee80211_addr_bssid_oui_resolved,
  &hf_ieee80211_addr_bssid_lg,
  &hf_ieee80211_addr_bssid_ig,
};

/* Station address HF list for proto_tree_add_mac48_detail() */
static const mac_hf_list_t mac_staa = {
  &hf_ieee80211_addr_staa,
  &hf_ieee80211_addr_staa_resolved,
  &hf_ieee80211_addr_staa_oui,
  &hf_ieee80211_addr_staa_oui_resolved,
  &hf_ieee80211_addr_staa_lg,
  &hf_ieee80211_addr_staa_ig,
};

static const fragment_items frag_items = {
  &ett_fragment,
  &ett_fragments,
  &hf_ieee80211_fragments,
  &hf_ieee80211_fragment,
  &hf_ieee80211_fragment_overlap,
  &hf_ieee80211_fragment_overlap_conflict,
  &hf_ieee80211_fragment_multiple_tails,
  &hf_ieee80211_fragment_too_long_fragment,
  &hf_ieee80211_fragment_error,
  &hf_ieee80211_fragment_count,
  &hf_ieee80211_reassembled_in,
  &hf_ieee80211_reassembled_length,
  /* Reassembled data field */
  NULL,
  "fragments"
};

static const enum_val_t wlan_ignore_prot_options[] = {
  { "no",         "No",               WLAN_IGNORE_PROT_NO    },
  { "without_iv", "Yes - without IV", WLAN_IGNORE_PROT_WO_IV },
  { "with_iv",    "Yes - with IV",    WLAN_IGNORE_PROT_W_IV  },
  { NULL,         NULL,               0                     }
};

static int wlan_address_type = -1;
static int wlan_bssid_address_type = -1;
static int wlan_ra_ta_address_type = -1;
static int wlan_aid_address_type = -1;

static int beacon_padding; /* beacon padding bug */

/*
 * Check if we have an S1G STA
 */
static bool
sta_is_s1g(packet_info *pinfo)
{
  void * data_p;

  if (treat_as_s1g)
    return true;

  data_p = p_get_proto_data(wmem_file_scope(), pinfo, proto_wlan, IS_S1G_KEY);
  return GPOINTER_TO_INT(data_p);
}

static heur_dissector_list_t heur_subdissector_list;

static dissector_handle_t ieee80211_handle;
static dissector_handle_t wlan_withoutfcs_handle;
static dissector_handle_t llc_handle;
static dissector_handle_t epd_llc_handle;
static dissector_handle_t ipx_handle;
static dissector_handle_t eth_withoutfcs_handle;

static capture_dissector_handle_t llc_cap_handle;
static capture_dissector_handle_t ipx_cap_handle;

static dissector_table_t ethertype_subdissector_table;
static dissector_table_t tagged_field_table;
static dissector_table_t vendor_specific_action_table;
static dissector_table_t wifi_alliance_action_subtype_table;
static dissector_table_t vendor_specific_anqp_info_table;
static dissector_table_t wifi_alliance_anqp_info_table;
static dissector_table_t wifi_alliance_ie_table;
static dissector_table_t wifi_alliance_public_action_table;

static int wlan_tap;

static const value_string access_network_type_vals[] = {
  {  0, "Private network" },
  {  1, "Private network with guest access" },
  {  2, "Chargeable public network" },
  {  3, "Free public network" },
  {  4, "Personal device network" },
  {  5, "Emergency services only network" },
  { 14, "Test or experimental" },
  { 15, "Wildcard" },
  { 0, NULL }
};

static const value_string adv_proto_id_vals[] = {
  {  0, "Access Network Query Protocol"},
  {  1, "MIH Information Service"},
  {  2, "MIH Command and Event Services Capability Discovery"},
  {  3, "Emergency Alert System (EAS)"},
  {  4, "Location-to-Service Translation Protocol"},
  {221, "Vendor Specific"},
  {0, NULL}
};

static const value_string timeout_int_types[] = {
  {1, "Reassociation deadline interval (TUs)"},
  {2, "Key lifetime interval (seconds)"},
  {3, "Association Comeback time (TUs)"},
  {4, "Time to start (TUs)"},
  {0, NULL}
};

static const value_string tdls_action_codes[] = {
  {TDLS_SETUP_REQUEST,           "TDLS Setup Request"},
  {TDLS_SETUP_RESPONSE,          "TDLS Setup Response"},
  {TDLS_SETUP_CONFIRM,           "TDLS Setup Confirm"},
  {TDLS_TEARDOWN,                "TDLS Teardown"},
  {TDLS_PEER_TRAFFIC_INDICATION, "TDLS Peer Traffic Indication"},
  {TDLS_CHANNEL_SWITCH_REQUEST,  "TDLS Channel Switch Request"},
  {TDLS_CHANNEL_SWITCH_RESPONSE, "TDLS Channel Switch Response"},
  {TDLS_PEER_PSM_REQUEST,        "TDLS Peer PSM Request"},
  {TDLS_PEER_PSM_RESPONSE,       "TDLS Peer PSM Response"},
  {TDLS_PEER_TRAFFIC_RESPONSE,   "TDLS Peer Traffic Response"},
  {TDLS_DISCOVERY_REQUEST,       "TDLS Discovery Request"},
  {0, NULL}
};
static value_string_ext tdls_action_codes_ext = VALUE_STRING_EXT_INIT(tdls_action_codes);

static const value_string rm_action_codes[] = {
  {RM_ACTION_RADIO_MEASUREMENT_REQUEST,   "Radio Measurement Request"},
  {RM_ACTION_RADIO_MEASUREMENT_REPORT,    "Radio Measurement Report"},
  {RM_ACTION_LINK_MEASUREMENT_REQUEST,    "Link Measurement Request"},
  {RM_ACTION_LINK_MEASUREMENT_REPORT,     "Link Measurement Report"},
  {RM_ACTION_NEIGHBOR_REPORT_REQUEST,     "Neighbor Report Request"},
  {RM_ACTION_NEIGHBOR_REPORT_RESPONSE,    "Neighbor Report Response"},
  {0, NULL}
};
static value_string_ext rm_action_codes_ext = VALUE_STRING_EXT_INIT(rm_action_codes);

static const val64_string number_of_taps_values[] = {
  {0x0, "1 tap"},
  {0x1, "5 taps"},
  {0x2, "15 taps"},
  {0x3, "63 taps"},
  {0, NULL}
};

#define PSMP_STA_INFO_BROADCAST 0
#define PSMP_STA_INFO_MULTICAST 1
#define PSMP_STA_INFO_INDIVIDUALLY_ADDRESSED 2

#define PSMP_STA_INFO_FLAG_TYPE         0x00000003
#define PSMP_STA_INFO_FLAG_DTT_START    0x00001FFC
#define PSMP_STA_INFO_FLAG_DTT_DURATION 0x001FE000

#define PSMP_STA_INFO_FLAG_STA_ID       0x001FFFE0

#define PSMP_STA_INFO_FLAG_UTT_START    0x0000FFE0
#define PSMP_STA_INFO_FLAG_UTT_DURATION 0x03FF0000

#define PSMP_STA_INFO_FLAG_IA_RESERVED  0xFC000000

static const value_string ff_psmp_sta_info_flags[] = {
  { PSMP_STA_INFO_BROADCAST,              "Broadcast"},
  { PSMP_STA_INFO_MULTICAST,              "Multicast"},
  { PSMP_STA_INFO_INDIVIDUALLY_ADDRESSED, "Individually Addressed"},
  {0, NULL}
};

static const char*
wlan_conv_get_filter_type(conv_item_t* conv, conv_filter_type_e filter)
{
    if ((filter == CONV_FT_SRC_ADDRESS) && (conv->src_address.type == wlan_address_type))
        return "wlan.sa";

    if ((filter == CONV_FT_DST_ADDRESS) && (conv->dst_address.type == wlan_address_type))
        return "wlan.da";

    if ((filter == CONV_FT_ANY_ADDRESS) && (conv->src_address.type == wlan_address_type))
        return "wlan.addr";

    return CONV_FILTER_INVALID;
}

static ct_dissector_info_t wlan_ct_dissector_info = {&wlan_conv_get_filter_type};

static tap_packet_status
wlan_conversation_packet(void *pct, packet_info *pinfo, epan_dissect_t *edt _U_, const void *vip, tap_flags_t flags)
{
  conv_hash_t *hash = (conv_hash_t*) pct;
  hash->flags = flags;
  const wlan_hdr_t *whdr=(const wlan_hdr_t *)vip;
  tap_packet_status status = TAP_PACKET_DONT_REDRAW;

  if ((whdr->src.type != AT_NONE) && (whdr->dst.type != AT_NONE)) {
    add_conversation_table_data(hash, &whdr->src, &whdr->dst, 0, 0, 1, pinfo->fd->pkt_len, &pinfo->rel_ts, &pinfo->abs_ts, &wlan_ct_dissector_info, CONVERSATION_NONE);

    status = TAP_PACKET_REDRAW;
  }

  return status;
}

static const char*
wlan_endpoint_get_filter_type(endpoint_item_t* endpoint, conv_filter_type_e filter)
{
  if ((filter == CONV_FT_ANY_ADDRESS) && (endpoint->myaddress.type == wlan_address_type))
    return "wlan.addr";

  return CONV_FILTER_INVALID;
}

static et_dissector_info_t wlan_endpoint_dissector_info = {&wlan_endpoint_get_filter_type};

static tap_packet_status
wlan_endpoint_packet(void *pit, packet_info *pinfo, epan_dissect_t *edt _U_, const void *vip, tap_flags_t flags)
{
  conv_hash_t *hash = (conv_hash_t*) pit;
  hash->flags = flags;
  const wlan_hdr_t *whdr=(const wlan_hdr_t *)vip;
  tap_packet_status status = TAP_PACKET_DONT_REDRAW;

  /* Take two "add" passes per packet, adding for each direction, ensures that all
  packets are counted properly (even if address is sending to itself)
  XXX - this could probably be done more efficiently inside endpoint_table */
  if (whdr->src.type != AT_NONE) {
    add_endpoint_table_data(hash, &whdr->src, 0, true, 1, pinfo->fd->pkt_len, &wlan_endpoint_dissector_info, ENDPOINT_NONE);
    status = TAP_PACKET_REDRAW;
  }

  if (whdr->dst.type != AT_NONE) {
    add_endpoint_table_data(hash, &whdr->dst, 0, false, 1, pinfo->fd->pkt_len, &wlan_endpoint_dissector_info, ENDPOINT_NONE);
    status = TAP_PACKET_REDRAW;
  }

  return status;
}

static const char*
wlan_col_filter_str(const address* addr _U_, bool is_src)
{
  if (is_src)
    return "wlan.sa";

  return "wlan.da";
}

static const char*
wlan_bssid_col_filter_str(const address* addr _U_, bool is_src _U_)
{
  return "wlan.bssid";
}

static const char*
wlan_ra_ta_col_filter_str(const address* addr _U_, bool is_src)
{
  if (is_src)
    return "wlan.ta";

  return "wlan.ra";
}

static const char*
wlan_aid_col_filter_str(const address* addr _U_, bool is_src _U_)
{
  return "wlan.fc.sid.association_id";
}

static void
beacon_interval_base_custom(char *result, uint32_t beacon_interval)
{
  double temp_double;

  temp_double = (double)beacon_interval;
  snprintf(result, ITEM_LABEL_LENGTH, "%f [Seconds]", (temp_double * 1024 / 1000000));
}

static void
allocation_duration_base_custom(char *result, uint32_t allocation_duration)
{
  double temp_double;

  temp_double = (double)allocation_duration;
  snprintf(result, ITEM_LABEL_LENGTH, "%f [Seconds]", (temp_double / 1000000));
}

static void
extra_one_base_custom(char *result, uint32_t value)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%d", value+1);
}

static void
extra_one_mul_two_base_custom(char *result, uint32_t value)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%d", (value+1)*2);
}

static void
rcpi_and_power_level_custom(char *result, uint8_t value)
{
  /* 802.11-2016 section 9.4.2.38
     RCPI = |2 x (P + 110)| in steps of 0.5 dB */

  if (value == 0)
    snprintf(result, ITEM_LABEL_LENGTH, "%d (P < -109.5 dBm)", value);
  else if (value == 220)
    snprintf(result, ITEM_LABEL_LENGTH, "%d (P >= 0 dBm)", value);
  else if (value < 220)
    snprintf(result, ITEM_LABEL_LENGTH, "%d (P = %.1f dBm)", value, ((double)value) / 2 - 110);
  else if (value < 255)
    snprintf(result, ITEM_LABEL_LENGTH, "%d (Reserved)", value);
  else
    snprintf(result, ITEM_LABEL_LENGTH, "%d (Measurement not available)", value);
}

static void
sts_custom(char *result, uint32_t value)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%d STS", value + 1);
}

static void
rep_custom(char *result, uint32_t value)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%u repetition%s (%u)", value + 1, plurality(value + 1, "", "s"), value);
}

static void
hundred_us_base_custom(char *result, uint32_t value)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%0.1f ms (%u)", ((double)value * 100 / 1000), value);
}

static void
partial_tsf_custom(char *result, uint32_t value)
{
  uint32_t shifted = value << 10;
  snprintf(result, ITEM_LABEL_LENGTH, "%u %s (%u)", shifted,
             unit_name_string_get_value(shifted, &units_microseconds), value);
}

/*
 * We use this is displaying the ru allocation region.
 */
static uint8_t global_he_trigger_bw;

static void
he_ru_allocation_base_custom(char *result, uint32_t ru_allocation)
{
  uint32_t tones = 0;

  switch (global_he_trigger_bw) {
  case 0:
    if (ru_allocation <= 8) {
      tones = 26;
      break;
    }
    if (ru_allocation >= 37 && ru_allocation <= 40) {
      tones = 52;
      break;
    }
    if (ru_allocation >= 53 && ru_allocation <= 54) {
      tones = 106;
      break;
    }
    if (ru_allocation == 61) {
      tones = 242;
      break;
    }
    // error
    break;
  case 1:
    if (ru_allocation <= 17) {
      tones = 26;
      break;
    }
    if (ru_allocation >= 37 && ru_allocation <= 44) {
      tones = 52;
      break;
    }
    if (ru_allocation >= 53 && ru_allocation <= 56) {
      tones = 106;
      break;
    }
    if (ru_allocation >= 61 && ru_allocation <= 62) {
      tones = 242;
      break;
    }
    if (ru_allocation == 65) {
      tones = 484;
      break;
    }
    // error
    break;
  case 2:
    /* fall-through */
  case 3:
    if (ru_allocation <= 36) {
      tones = 26;
      break;
    }
    if (ru_allocation >= 37 && ru_allocation <= 52) {
      tones = 52;
      break;
    }
    if (ru_allocation >= 53 && ru_allocation <= 60) {
      tones = 106;
      break;
    }
    if (ru_allocation >= 61 && ru_allocation <= 64) {
      tones = 242;
      break;
    }
    if (ru_allocation >= 65 && ru_allocation <= 66) {
      tones = 484;
      break;
    }
    if (ru_allocation == 67) {
      tones = 996;
      break;
    }
    if (ru_allocation == 68 && global_he_trigger_bw == 3) {
      tones = 2*996;
      break;
    }
    break;
  default:
    break;
  }

  if (tones)
    snprintf(result, ITEM_LABEL_LENGTH, "%d (%d tones)", ru_allocation, tones);
  else
    snprintf(result, ITEM_LABEL_LENGTH, "%d (bogus number of tones)", ru_allocation);
}

static void
eht_ru_allocation_base_custom(char *result, uint32_t ru_allocation)
{
  char *ru_str;

  /* TODO: check with PS160, B0 of RU allocation, BW */
  if (ru_allocation <= 36) {
    ru_str = "RU size: 26";
  } else if (ru_allocation >= 37 && ru_allocation <= 52) {
    ru_str = "RU size: 52";
  } else if (ru_allocation >= 53 && ru_allocation <= 60) {
    ru_str = "RU size: 106";
  } else if (ru_allocation >= 61 && ru_allocation <= 64) {
    ru_str = "RU size: 242";
  } else if (ru_allocation == 65 || ru_allocation == 66) {
    ru_str = "RU size: 484";
  } else if (ru_allocation == 67) {
    ru_str = "RU size: 996";
  } else if (ru_allocation == 68) {
    ru_str = "RU size: 2x996";
  } else if (ru_allocation == 69) {
    ru_str = "RU size: 4x996";
  } else if (ru_allocation >= 70 && ru_allocation <= 81) {
    ru_str = "MRU size: 52+26";
  } else if (ru_allocation >= 82 && ru_allocation <= 89) {
    ru_str = "MRU size: 106+26";
  } else if (ru_allocation >= 90 && ru_allocation <= 93) {
    ru_str = "MRU size: 484+242";
  } else if (ru_allocation >= 94 && ru_allocation <= 95) {
    ru_str = "MRU size: 996+484";
  } else if (ru_allocation >= 96 && ru_allocation <= 99) {
    ru_str = "MRU size: 996+484+242";
  } else if (ru_allocation >= 100 && ru_allocation <= 103) {
    ru_str = "MRU size: 2x996+484";
  } else if (ru_allocation == 104) {
    ru_str = "MRU size: 3x996";
  } else if (ru_allocation == 105 || ru_allocation == 106) {
    ru_str = "MRU size: 3x996+484";
  } else {
    ru_str = "Reserved";
  }

  snprintf(result, ITEM_LABEL_LENGTH, "%d (%s)", ru_allocation, ru_str);
}

/*
 * We use this to display the ANSI/CTA-2063 Serial number length
 */
static void
vs_sgdsn_serialnumber_len_custom(char *result, uint32_t val)
{
  if (val >= 0x30 && val <= 0x39) {
    snprintf(result, ITEM_LABEL_LENGTH, "%d byte(s)", val-0x30);
  } else if (val >= 0x41 && val <= 0x46) {
    snprintf(result, ITEM_LABEL_LENGTH, "%d byte(s)", val-0x37);
  } else {
    snprintf(result, ITEM_LABEL_LENGTH, "Invalid length: %u", val);
  }
}

/* ************************************************************************* */
/* Mesh Control field helper functions
 *
 * Per IEEE 802.11s Draft 12.0 section 7.2.2.1:
 *
 * The frame body consists of either:
 * The MSDU (or a fragment thereof), the Mesh Control field (if and only if the
 * frame is transmitted by a mesh STA and the Mesh Control Present subfield of
 * the QoS Control field is 1)...
 *
 * 8.2.4.5.1 "QoS Control field structure", table 8-4, in 802.11-2012,
 * seems to indicate that the bit that means "Mesh Control Present" in
 * frames sent by mesh STAs in a mesh BSS is part of the TXOP Limit field,
 * the AP PS Buffer State field, the TXOP Duration Requested field, or the
 * Queue Size field in some data frames in non-mesh BSSes.
 *
 * We need a statefull sniffer for that.  For now, use heuristics.
 *
 * Notably, only mesh data frames contain the Mesh Control field in the header.
 * Other frames that contain mesh control (i.e., multihop action frames) have
 * it deeper in the frame body where it can be definitively identified.
 * Further, mesh data frames always have to-ds and from-ds either 11 or 01.  We
 * use these facts to make our heuristics more reliable.
 * ************************************************************************* */
static int
has_mesh_control(uint16_t fcf, uint16_t qos_ctl, uint8_t mesh_flags)
{
  /* assume mesh control present if the QOS field's Mesh Control Present bit is
   * set, all reserved bits in the mesh_flags field are zero, and the address
   * extension mode is not a reserved value.
   */
  return (((FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T4) || (FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T2)) &&
          (QOS_MESH_CONTROL_PRESENT(qos_ctl)) &&
          ((mesh_flags & ~MESH_FLAGS_ADDRESS_EXTENSION) == 0) &&
          ((mesh_flags & MESH_FLAGS_ADDRESS_EXTENSION) != MESH_FLAGS_ADDRESS_EXTENSION));
}

/****************************************************************************** */
/*
 * locally originated mesh frames will have a mesh control field, but no QoS header
 * detect the presence of mesh control field by checking if mesh flags are legal
 * and confirming that the next header is an 802.2 LLC header
 *
 ****************************************************************************** */
static int
has_mesh_control_local(uint16_t fcf, uint8_t mesh_flags, uint16_t next_header)
{
  return (((FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T4) || (FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T2)) &&
          ((mesh_flags & ~MESH_FLAGS_ADDRESS_EXTENSION) == 0) &&
          ((mesh_flags & MESH_FLAGS_ADDRESS_EXTENSION) != MESH_FLAGS_ADDRESS_EXTENSION)) &&
          next_header == 0xaaaa;
}

static int
find_mesh_control_length(uint8_t mesh_flags)
{
  return 6 + 6*(mesh_flags & MESH_FLAGS_ADDRESS_EXTENSION);
}

static mimo_control_t
get_mimo_control(tvbuff_t *tvb, int offset)
{
  uint16_t       mimo;
  mimo_control_t output;

  mimo = tvb_get_letohs(tvb, offset);

  output.nc = (mimo & 0x0003) + 1;
  output.nr = ((mimo & 0x000C) >> 2) + 1;
  output.chan_width = (mimo & 0x0010) >> 4;
  output.coefficient_size = 4; /* XXX - Is this a good default? */

  switch ((mimo & 0x0060) >> 5)
    {
      case 0:
        output.grouping = 1;
        break;

      case 1:
        output.grouping = 2;
        break;

      case 2:
        output.grouping = 4;
        break;

      default:
        output.grouping = 1;
        break;
    }

  switch ((mimo & 0x0180) >> 7)
    {
      case 0:
        output.coefficient_size = 4;
        break;

      case 1:
        output.coefficient_size = 5;
        break;

      case 2:
        output.coefficient_size = 6;
        break;

      case 3:
        output.coefficient_size = 8;
        break;
    }

  output.codebook_info = (mimo & 0x0600) >> 9;
  output.remaining_matrix_segment = (mimo & 0x3800) >> 11;

  return output;
}

static int
get_mimo_na(uint8_t nr, uint8_t nc)
{
  if ((nr == 2) && (nc == 1)) {
    return 2;
  } else if ((nr == 2) && (nc == 2)) {
    return 2;
  } else if ((nr == 3) && (nc == 1)) {
    return 4;
  } else if ((nr == 3) && (nc == 2)) {
    return 6;
  } else if ((nr == 3) && (nc == 3)) {
    return 6;
  } else if ((nr == 4) && (nc == 1)) {
    return 6;
  } else if ((nr == 4) && (nc == 2)) {
    return 10;
  } else if ((nr == 4) && (nc == 3)) {
    return 12;
  } else if ((nr == 4) && (nc == 4)) {
    return 12;
  } else{
    return 0;
  }
}

static int
get_mimo_ns(bool chan_width, uint8_t output_grouping)
{
  int ns = 0;

  if (chan_width)
  {
    switch (output_grouping)
      {
        case 1:
          ns = 114;
          break;

          case 2:
            ns = 58;
            break;

          case 4:
            ns = 30;
            break;

          default:
            ns = 0;
      }
  } else {
    switch (output_grouping)
      {
        case 1:
          ns = 56;
          break;

        case 2:
          ns = 30;
          break;

        case 4:
          ns = 16;
          break;

        default:
          ns = 0;
      }
  }

  return ns;
}

static int
add_mimo_csi_matrices_report(proto_tree *tree, tvbuff_t *tvb, int offset, mimo_control_t mimo_cntrl)
{
  proto_tree *snr_tree;
  int         csi_matrix_size, start_offset;
  int         ns, i;

  start_offset = offset;
  snr_tree = proto_tree_add_subtree(tree, tvb, offset, mimo_cntrl.nc,
                        ett_mimo_report, NULL, "Signal to Noise Ratio");

  for (i = 1; i <= mimo_cntrl.nr; i++)
  {
    uint8_t snr;

    snr = tvb_get_uint8(tvb, offset);
    proto_tree_add_uint_format(snr_tree, hf_ieee80211_ff_mimo_csi_snr, tvb, offset, 1,
                               snr, "Channel %d - Signal to Noise Ratio: 0x%02X", i, snr);
    offset += 1;
  }

  ns = get_mimo_ns(mimo_cntrl.chan_width, mimo_cntrl.grouping);
  csi_matrix_size = ns*(3+(2*mimo_cntrl.nc*mimo_cntrl.nr*mimo_cntrl.coefficient_size));
  csi_matrix_size = WS_ROUNDUP_8(csi_matrix_size) / 8;
  proto_tree_add_item(snr_tree, hf_ieee80211_ff_mimo_csi_matrices, tvb, offset, csi_matrix_size, ENC_NA);
  offset += csi_matrix_size;
  return offset - start_offset;
}

static int
add_mimo_beamforming_feedback_report(proto_tree *tree, tvbuff_t *tvb, int offset, mimo_control_t mimo_cntrl)
{
  proto_tree *snr_tree;
  int         csi_matrix_size, start_offset;
  int         ns, i;

  start_offset = offset;
  snr_tree = proto_tree_add_subtree(tree, tvb, offset, mimo_cntrl.nc, ett_mimo_report, NULL, "Signal to Noise Ratio");

  for (i = 1; i <= mimo_cntrl.nc; i++)
  {
    uint8_t snr;

    snr = tvb_get_uint8(tvb, offset);
    proto_tree_add_uint_format(snr_tree, hf_ieee80211_ff_mimo_csi_snr, tvb, offset, 1,
                               snr, "Stream %d - Signal to Noise Ratio: 0x%02X", i, snr);
    offset += 1;
  }

  ns = get_mimo_ns(mimo_cntrl.chan_width, mimo_cntrl.grouping);
  csi_matrix_size = ns*(2*mimo_cntrl.nc*mimo_cntrl.nr*mimo_cntrl.coefficient_size);
  csi_matrix_size = WS_ROUNDUP_8(csi_matrix_size) / 8;
  proto_tree_add_item(snr_tree, hf_ieee80211_ff_mimo_csi_bf_matrices, tvb, offset, csi_matrix_size, ENC_NA);
  offset += csi_matrix_size;
  return offset - start_offset;
}

static int
add_mimo_compressed_beamforming_feedback_report(proto_tree *tree, tvbuff_t *tvb, int offset, mimo_control_t mimo_cntrl)
{
  proto_tree *snr_tree;
  int         csi_matrix_size, start_offset;
  int         ns, na, i;

  start_offset = offset;
  snr_tree = proto_tree_add_subtree(tree, tvb, offset, mimo_cntrl.nc,
                        ett_mimo_report, NULL, "Signal to Noise Ratio");

  for (i = 1; i <= mimo_cntrl.nc; i++)
  {
    int8_t snr;
    char edge_sign;

    snr = tvb_get_int8(tvb, offset);

    switch(snr) {
      case -128:
        edge_sign = '<';
        break;
      case 127:
        edge_sign = '>';
        break;
      default:
        edge_sign = ' ';
        break;
    }
    proto_tree_add_uint_format(snr_tree, hf_ieee80211_ff_mimo_csi_snr, tvb, offset, 1,
                               snr, "Stream %d - Signal to Noise Ratio: %c%3.2fdB", i, edge_sign,snr/4.0+22.0);
    offset += 1;
  }

  na = get_mimo_na(mimo_cntrl.nr, mimo_cntrl.nc);
  ns = get_mimo_ns(mimo_cntrl.chan_width, mimo_cntrl.grouping);
  csi_matrix_size = ns*(na*((mimo_cntrl.codebook_info+1)*2 + 2)/2);
  csi_matrix_size = WS_ROUNDUP_8(csi_matrix_size) / 8;
  proto_tree_add_item(snr_tree, hf_ieee80211_ff_mimo_csi_cbf_matrices, tvb, offset, csi_matrix_size, ENC_NA);
  offset += csi_matrix_size;
  return offset - start_offset;
}

static void
mesh_active_window_base_custom(char *result, uint32_t mesh_active_window)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%f [Seconds]", (mesh_active_window * 1024.0 / 1000000));
}

/* ************************************************************************* */
/*          This is the capture function used to update packet counts        */
/* ************************************************************************* */
static bool
capture_ieee80211_common(const unsigned char * pd, int offset, int len,
                          capture_packet_info_t *cpinfo, const union wtap_pseudo_header *pseudo_header _U_, bool datapad)
{
  uint16_t fcf, hdr_length;

  if (!BYTES_ARE_IN_FRAME(offset, len, 2))
    return false;

  fcf = pletohu16(&pd[offset]);

  if (IS_PROTECTED(FCF_FLAGS(fcf)) && (wlan_ignore_prot == WLAN_IGNORE_PROT_NO))
    return false;

  switch (COMPOSE_FRAME_TYPE (fcf)) {

    case DATA:
    case DATA_CF_ACK:
    case DATA_CF_POLL:
    case DATA_CF_ACK_POLL:
    case DATA_QOS_DATA:
    case DATA_QOS_DATA_CF_ACK:
    case DATA_QOS_DATA_CF_POLL:
    case DATA_QOS_DATA_CF_ACK_POLL:
    {
      /* These are data frames that actually contain *data*. */
      hdr_length = (FCF_ADDR_SELECTOR(fcf) == DATA_ADDR_T4) ? DATA_LONG_HDR_LEN : DATA_SHORT_HDR_LEN;

      if (DATA_FRAME_IS_QOS(COMPOSE_FRAME_TYPE(fcf))) {
        /* QoS frame, so the header includes a QoS field */
        uint16_t qosoff;  /* Offset of the 2-byte QoS field */
        uint8_t mesh_flags;

        qosoff = hdr_length;
        hdr_length += 2; /* Include the QoS field in the header length */

        if (HAS_HT_CONTROL(FCF_FLAGS(fcf))) {
          /* Frame has a 4-byte HT Control field */
          hdr_length += 4;
        }

        if (datapad) {
          /*
           * Include the padding between the 802.11 header and the body,
           * as "helpfully" provided by some Atheros adapters.
           *
           * In the Atheros mesh capture sample we have, the padding
           * is before the mesh header, possibly because it doesn't
           * recognize the mesh header.
           */
          hdr_length = WS_ROUNDUP_4(hdr_length);
        }

        /*
         * Does it look as if we have a mesh header?
         * Look at the Mesh Control subfield of the QoS field and at the
         * purported mesh flag fields.
         */
        if (!BYTES_ARE_IN_FRAME(offset, hdr_length, 1))
          return false;

        mesh_flags = pd[hdr_length];
        if (has_mesh_control(fcf, pletohu16(&pd[qosoff]), mesh_flags)) {
          /* Yes, add the length of that in as well. */
          hdr_length += find_mesh_control_length(mesh_flags);
        }
      }
      /* I guess some bridges take Netware Ethernet_802_3 frames,
         which are 802.3 frames (with a length field rather than
         a type field, but with no 802.2 header in the payload),
         and just stick the payload into an 802.11 frame.  I've seen
         captures that show frames of that sort.

         We also handle some odd form of encapsulation in which a
         complete Ethernet frame is encapsulated within an 802.11
         data frame, with no 802.2 header.  This has been seen
         from some hardware.

         On top of that, at least at some point it appeared that
         the OLPC XO sent out frames with two bytes of 0 between
         the "end" of the 802.11 header and the beginning of
         the payload.

         So, if the packet doesn't start with 0xaa 0xaa:

           we first use the same scheme that linux-wlan-ng does to detect
           those encapsulated Ethernet frames, namely looking to see whether
           the frame either starts with 6 octets that match the destination
           address from the 802.11 header or has 6 octets that match the
           source address from the 802.11 header following the first 6 octets,
           and, if so, treat it as an encapsulated Ethernet frame;

           otherwise, we use the same scheme that we use in the Ethernet
           dissector to recognize Netware 802.3 frames, namely checking
           whether the packet starts with 0xff 0xff and, if so, treat it
           as an encapsulated IPX frame, and then check whether the
           packet starts with 0x00 0x00 and, if so, treat it as an OLPC
           frame. */
      if (!BYTES_ARE_IN_FRAME(offset+hdr_length, len, 2))
        return false;

      if ((pd[offset+hdr_length] != 0xaa) && (pd[offset+hdr_length+1] != 0xaa)) {
#if 0
        /* XXX - this requires us to parse the header to find the source
           and destination addresses. */
        if (BYTES_ARE_IN_FRAME(offset+hdr_length, len, 12)) {
          /* We have two MAC addresses after the header. */
          if ((memcmp(&pd[offset+hdr_length+6], pinfo->dl_src.data, 6) == 0) ||
              (memcmp(&pd[offset+hdr_length+6], pinfo->dl_dst.data, 6) == 0)) {
            return capture_eth (pd, offset + hdr_length, len, cpinfo, pseudo_header);
          }
        }
#endif
        if ((pd[offset+hdr_length] == 0xff) && (pd[offset+hdr_length+1] == 0xff))
          return call_capture_dissector (ipx_cap_handle, pd, offset+hdr_length, len, cpinfo, pseudo_header);
        else if ((pd[offset+hdr_length] == 0x00) && (pd[offset+hdr_length+1] == 0x00))
          return call_capture_dissector (llc_cap_handle, pd, offset + hdr_length + 2, len, cpinfo, pseudo_header);
      }
      else {
        return call_capture_dissector (llc_cap_handle, pd, offset + hdr_length, len, cpinfo, pseudo_header);
      }
      break;
    }
  }

  return false;
}

/*
 * Handle 802.11 with a variable-length link-layer header.
 */
static bool
capture_ieee80211(const unsigned char * pd, int offset, int len, capture_packet_info_t *cpinfo, const union wtap_pseudo_header *pseudo_header _U_)
{
  return capture_ieee80211_common (pd, offset, len, cpinfo, pseudo_header, false);
}

/*
 * Handle 802.11 with a variable-length link-layer header and data padding.
 */
static bool
capture_ieee80211_datapad(const unsigned char * pd, int offset, int len,
                           capture_packet_info_t *cpinfo, const union wtap_pseudo_header *pseudo_header _U_)
{
  return capture_ieee80211_common (pd, offset, len, cpinfo, pseudo_header, true);
}


/* ************************************************************************* */
/*          Add the subtree used to store the fixed parameters               */
/* ************************************************************************* */
static proto_tree *
get_fixed_parameter_tree(proto_tree * tree, tvbuff_t *tvb, int start, int size,
                         bool no_append)
{
  proto_item *fixed_fields;

  fixed_fields = proto_tree_add_item(tree, hf_ieee80211_fixed_parameters, tvb, start, size, ENC_NA);
  if (!no_append)
    proto_item_append_text(fixed_fields, " (%d bytes)", size);

  return proto_item_add_subtree(fixed_fields, ett_fixed_parameters);
}


/* ************************************************************************* */
/*            Add the subtree used to store tagged parameters                */
/* ************************************************************************* */
static proto_tree *
get_tagged_parameter_tree(proto_tree * tree, tvbuff_t *tvb, int start, int size)
{
  proto_item *tagged_fields;

  tagged_fields = proto_tree_add_item(tree, hf_ieee80211_tagged_parameters, tvb, start, -1, ENC_NA);
  proto_item_append_text(tagged_fields, " (%d bytes)", size);

  return proto_item_add_subtree(tagged_fields, ett_tagged_parameters);
}

static void
add_ptk_analysis(tvbuff_t *tvb, proto_tree *tree, DOT11DECRYPT_KEY_ITEM *used_key)
{
  if (!used_key) {
    return;
  }
  const uint8_t *key = NULL;
  proto_item *ti;
  char buf[SHORT_STR];
  int len = Dot11DecryptGetKCK(used_key, &key);
  bytes_to_hexstr(buf, key, len);
  buf[2 * len] = '\0';
  ti = proto_tree_add_string(tree, hf_ieee80211_fc_analysis_kck, tvb, 0, 0, buf);
  proto_item_set_generated(ti);

  len = Dot11DecryptGetKEK(used_key, &key);
  bytes_to_hexstr(buf, key, len);
  buf[2 * len] = '\0';
  ti = proto_tree_add_string(tree, hf_ieee80211_fc_analysis_kek, tvb, 0, 0, buf);
  proto_item_set_generated(ti);
}

static int
dissect_vendor_action_marvell(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void *data _U_)
{
  uint8_t octet;
  int offset = 0;

  octet = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_marvell_action_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  switch (octet)
    {
      case MRVL_ACTION_MESH_MANAGEMENT:
        octet = tvb_get_uint8(tvb, offset);
        proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_action_code, tvb, offset, 1, ENC_LITTLE_ENDIAN);
        offset += 1;
        switch (octet)
          {
            case MRVL_MESH_MGMT_ACTION_RREQ:
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_length, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_mode, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_hopcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_rreqid, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_sa, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ssn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_lifetime, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_metric, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dstcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_flags, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_da, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dsn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              break;
            case MRVL_MESH_MGMT_ACTION_RREP:
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_length, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_mode, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_hopcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ttl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_da, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dsn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_lifetime, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_metric, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_sa, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_ssn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              break;
            case MRVL_MESH_MGMT_ACTION_RERR:
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_length, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_mode, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dstcount, tvb, offset, 1, ENC_LITTLE_ENDIAN);
              offset += 1;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_da, tvb, offset, 6, ENC_NA);
              offset += 6;
              proto_tree_add_item(tree, hf_ieee80211_ff_marvell_mesh_mgt_dsn, tvb, offset, 4, ENC_LITTLE_ENDIAN);
              offset += 4;
              break;
            default:
              break;
          }
        break;
      default:
        break;
    }

  return offset;
}

static int
dissect_dscp_policy_query(tvbuff_t *tvb, packet_info *pinfo _U_,
                          proto_tree *tree, int offset)
{
  int start_offset = offset;

  while (tvb_captured_length_remaining(tvb, offset)) {
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
  }

  return offset - start_offset;
}

static int * const rqst_control_fields[] = {
  &hf_ieee80211_oui_qos_mgmt_rq_ctrl_more,
  &hf_ieee80211_oui_qos_mgmt_rq_ctrl_reset,
  &hf_ieee80211_oui_qos_mgmt_rq_reserved,
  NULL
};

static int
dissect_dscp_policy_request(tvbuff_t *tvb, packet_info *pinfo _U_,
                            proto_tree *tree, int offset)
{
  int start_offset = offset;

  proto_tree_add_bitmask(tree, tvb, offset,
                         hf_ieee80211_oui_qos_mgmt_rqst_control,
                         ett_pol_rqst_cont_tree, rqst_control_fields,
                         ENC_NA);
  offset += 1;

  while (tvb_reported_length_remaining(tvb, offset)) {
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
  }

  return offset - start_offset;
}

static int * const resp_control_fields[] = {
  &hf_ieee80211_oui_qos_mgmt_rsp_ctrl_more,
  &hf_ieee80211_oui_qos_mgmt_rsp_ctrl_reset,
  &hf_ieee80211_oui_qos_mgmt_rsp_reserved,
  NULL
};

static int
dissect_dscp_policy_response(tvbuff_t *tvb, packet_info *pinfo,
                             proto_tree *tree, int offset)
{
  int start_offset = offset;
  uint8_t count;
  wmem_strbuf_t *status_buf = wmem_strbuf_create(pinfo->pool);
  int i;

  proto_tree_add_bitmask(tree, tvb, offset,
                         hf_ieee80211_oui_qos_mgmt_resp_control,
                         ett_pol_resp_cont_tree, resp_control_fields,
                         ENC_NA);
  offset += 1;

  count = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_oui_qos_mgmt_count, tvb, offset,
                      1, ENC_NA);
  offset += 1;

  for (i = 0; i < count; i++) {
    proto_tree *status_tree = NULL;
    uint8_t scsid, status;

    scsid = tvb_get_uint8(tvb, offset);
    status_tree = proto_tree_add_subtree_format(tree, tvb, offset, 2,
                                                ett_dscp_policy_status_list,
                                                NULL, "Status list item %d",
                                                i);
    proto_tree_add_item(status_tree, hf_ieee80211_dscp_policy_id, tvb,
                        offset, 1, ENC_NA);
    if (wmem_strbuf_get_len(status_buf) == 0) {
      wmem_strbuf_append_printf(status_buf, "%u:", scsid);
    } else {
      wmem_strbuf_append_printf(status_buf, " %u:", scsid);
    }
    offset += 1;

    status = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(status_tree, hf_ieee80211_dscp_policy_status, tvb,
                        offset, 1, ENC_NA);
    wmem_strbuf_append_printf(status_buf, "%u", status);
    offset += 1;
  }
  proto_tree_add_string(tree, hf_ieee80211_dscp_policy_scs_sts_list, tvb, 0, 1,
                        wmem_strbuf_finalize(status_buf));

  return offset - start_offset;
}

static int
dissect_vendor_action_wfa_qos_mgmt(tvbuff_t *tvb, packet_info *pinfo,
                                   proto_tree *tree, void *data _U_)
{
  int offset = 0;
  uint8_t subtype = tvb_get_uint8(tvb, offset);

  proto_tree_add_item(tree, hf_ieee80211_oui_qos_subtype, tvb, offset, 1,
                      ENC_NA);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_oui_qos_mgmt_dialog_token, tvb, offset,
                      1, ENC_NA);
  offset += 1;

  switch (subtype) {
  case 0:
    offset += dissect_dscp_policy_query(tvb, pinfo, tree, offset);
    break;
  case 1:
    offset += dissect_dscp_policy_request(tvb, pinfo, tree, offset);
    break;
  case 2:
    offset += dissect_dscp_policy_response(tvb, pinfo, tree, offset);
    break;
  }

  return offset;
}

static int
dissect_vendor_action_wifi_alliance(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data _U_)
{
  uint8_t subtype;
  int offset = 0;
  int dissected;
  tvbuff_t *subtvb;

  subtype = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_oui_wfa_action_type, tvb, offset,
                      1, ENC_NA);
  offset += 1;

  subtvb = tvb_new_subset_remaining(tvb, offset);
  dissected = dissector_try_uint_with_data(wifi_alliance_action_subtype_table, subtype, subtvb, pinfo, tree, false, NULL);
  if (dissected <= 0)
  {
      call_data_dissector(subtvb, pinfo, tree);
      dissected = tvb_reported_length(subtvb);
  }

  offset += dissected;

  return offset;
}

/*
 * This function is called from two different places. In one case it is called
 * without the tag and length. In other cases, it is called with those and
 * is asked to return the type and subtype. We know the difference because
 * type and subtype will be NULL in the first case.
 */
static unsigned
dissect_advertisement_protocol_common(packet_info *pinfo, proto_tree *tree,
                               tvbuff_t *tvb, int offset, unsigned *type,
                               unsigned *subtype)
{
  uint8_t     tag_no = 0, tag_len, left;
  proto_item *item = NULL, *adv_item;
  proto_tree *adv_tree, *adv_tuple_tree;

  if (type)
    *type = 0xff; // Last reserved value
  if (subtype)
    *subtype = 0xff;
  tag_no = tvb_get_uint8(tvb, offset);
  if (type)
    item = proto_tree_add_item(tree, hf_ieee80211_tag_number, tvb, offset, 1, ENC_LITTLE_ENDIAN);

  /*
   * If we have the tag and len, use the len in the tvb, otherwise ask
   * for the length of the tvb.
   */
  if (type)
    tag_len = tvb_get_uint8(tvb, offset + 1);
  else
    tag_len = tvb_reported_length_remaining(tvb, 0);

  if (type && tag_no != TAG_ADVERTISEMENT_PROTOCOL) {
    expert_add_info_format(pinfo, item, &ei_ieee80211_tag_number,
                           "Unexpected IE %d (expected Advertisement "
                           "Protocol)", tag_no);
    return 2 + tag_len;
  }
  if (type)
    item = proto_tree_add_uint(tree, hf_ieee80211_tag_length, tvb, offset + 1, 1, tag_len);
  if (tag_len < 2) {
    if (!type)
      item = proto_tree_add_uint(tree, hf_ieee80211_tag_length, tvb, offset + 1, 1, tag_len);
    expert_add_info_format(pinfo, item, &ei_ieee80211_tag_length,
                           "Advertisement Protocol: IE must be at least 2 "
                           "octets long");
    return 2 + tag_len;
  }

  left = tag_len;
  if (type) /* Skip past the header if there ... */
    offset += 2;
  adv_tree = proto_tree_add_subtree(tree, tvb, offset, left,
                                 ett_adv_proto, &adv_item, "Advertisement Protocol element");

  while (left >= 2) {
    uint8_t id;

    id = tvb_get_uint8(tvb, offset + 1);
    if (id == 0)
      proto_item_append_text(adv_item, ": ANQP");
    adv_tuple_tree = proto_tree_add_subtree_format(adv_tree, tvb, offset, 2, ett_adv_proto_tuple, &item,
                               "Advertisement Protocol Tuple: %s",
                               val_to_str(pinfo->pool, id, adv_proto_id_vals,
                                          "Unknown (%d)"));

    proto_tree_add_item(adv_tuple_tree,
                        hf_ieee80211_tag_adv_proto_resp_len_limit, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(adv_tuple_tree,
                        hf_ieee80211_tag_adv_proto_pame_bi, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    left--;
    proto_tree_add_item(adv_tuple_tree, hf_ieee80211_tag_adv_proto_id, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    left--;

    if ((id == 0) && type)
      *type = ADV_PROTO_ID_ANQP;

    if (id == 221) {
      /* Vendor specific */
      uint8_t len = tvb_get_uint8(tvb, offset);
      unsigned oui;
      uint8_t wfa_subtype;
      proto_tree_add_item(adv_tuple_tree, hf_ieee80211_tag_adv_vs_len, tvb, offset, 1, ENC_NA);
      offset += 1;
      left   -= 1;
      if (type)
        *type = ADV_PROTO_ID_VS;
      if (len > left) {
        expert_add_info_format(pinfo, item, &ei_ieee80211_tag_length,
                               "Vendor specific info length error");
        return 2 + tag_len;
      }
      proto_tree_add_item_ret_uint(adv_tuple_tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
      offset += 3;
      left   -= 3;
      wfa_subtype = tvb_get_uint8(tvb, offset);
      proto_tree_add_item(adv_tuple_tree, hf_ieee80211_anqp_wfa_subtype, tvb,
                        offset, 1, ENC_NA);
      offset += 1;
      left   -= 1;
      if (oui == OUI_WFA) {
        proto_tree_add_item(adv_tuple_tree, hf_ieee80211_dpp_subtype, tvb, offset, 1, ENC_NA);
        if (subtype && wfa_subtype == WFA_SUBTYPE_DPP) {
          *subtype = WFA_SUBTYPE_DPP;
          *subtype |= (tvb_get_uint8(tvb, offset) << 8);
        }
        offset++;
        left--;
      }
    }
  }

  if (left) {
    expert_add_info(pinfo, item, &ei_ieee80211_extra_data);
  }

  return 2 + tag_len;
}

static int
dissect_advertisement_protocol(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  return dissect_advertisement_protocol_common(pinfo, tree, tvb, 0, NULL, NULL);
}

static void
dissect_anqp_query_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  while (offset + 2 <= end) {
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_query_id,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }
  if (offset != end) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_ff_anqp_info_length,
                           "Unexpected ANQP Query list format");
  }
}

static void
dissect_hs20_anqp_hs_capability_list(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset < end) {
    proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_hs_capability_list,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static void
dissect_anqp_capab_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  uint16_t    id, len;
  proto_item *item;
  proto_tree *vtree;
  uint32_t    oui;
  uint8_t     subtype;

  while (offset + 2 <= end) {
    id = tvb_get_letohs(tvb, offset);
    item = proto_tree_add_item(tree, hf_ieee80211_ff_anqp_capability,
                               tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (id == ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST) {
      vtree = proto_item_add_subtree(item, ett_anqp_vendor_capab);
      len = tvb_get_letohs(tvb, offset);
      proto_tree_add_item(vtree, hf_ieee80211_ff_anqp_capability_vlen,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if ((len < 3) || ((offset + len) > end)) {
        expert_add_info(pinfo, tree, &ei_ieee80211_ff_anqp_capability);
        return;
      }
      proto_tree_add_item_ret_uint(vtree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
      offset += 3;
      len    -= 3;

      switch (oui) {
      case OUI_WFA:
        if (len == 0)
          break;
        subtype = tvb_get_uint8(tvb, offset);
        proto_item_append_text(vtree, " - WFA - %s",
                               val_to_str(pinfo->pool, subtype, wfa_anqp_subtype_vals,
                                          "Unknown (%u)"));
        proto_tree_add_item(vtree, hf_ieee80211_anqp_wfa_subtype,
                            tvb, offset, 1, ENC_NA);
        offset++;
        len--;
        switch (subtype) {
        case WFA_ANQP_SUBTYPE_HS20:
          proto_tree_add_item(vtree, hf_ieee80211_hs20_anqp_subtype, tvb, offset, 1, ENC_NA);
          proto_tree_add_item(vtree, hf_ieee80211_hs20_anqp_reserved, tvb, offset + 1, 1, ENC_NA);
          offset += 2;
          len -= 2;
          dissect_hs20_anqp_hs_capability_list(vtree, tvb, offset, end);
          break;
        default:
          proto_tree_add_item(vtree, hf_ieee80211_ff_anqp_capability_vendor,
                              tvb, offset, len, ENC_NA);
          break;
        }
        break;
      default:
        proto_tree_add_item(vtree, hf_ieee80211_ff_anqp_capability_vendor,
                            tvb, offset, len, ENC_NA);
        break;
      }

      offset += len;
    }
  }
  if (offset != end) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_ff_anqp_info_length,
                           "Unexpected ANQP Capability list format");
  }
}

static const value_string venue_group_vals[] = {
  {  0, "Unspecified" },
  {  1, "Assembly" },
  {  2, "Business" },
  {  3, "Educational" },
  {  4, "Factory and Industrial" },
  {  5, "Institutional" },
  {  6, "Mercantile" },
  {  7, "Residential" },
  {  8, "Storage" },
  {  9, "Utility and Miscellaneous" },
  { 10, "Vehicular" },
  { 11, "Outdoor" },
  { 0, NULL }
};
static value_string_ext venue_group_vals_ext = VALUE_STRING_EXT_INIT(venue_group_vals);

static void
dissect_venue_info(proto_tree *tree, tvbuff_t *tvb, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_venue_info_group,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_ff_venue_info_type,
                      tvb, offset + 1, 1, ENC_LITTLE_ENDIAN);
}

static void
dissect_venue_name_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  proto_item *item;

  dissect_venue_info(tree, tvb, offset);
  offset += 2;
  while (offset + 4 <= end) {
    uint8_t vlen = tvb_get_uint8(tvb, offset);
    item = proto_tree_add_item(tree, hf_ieee80211_ff_anqp_venue_length,
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    if ((vlen > (end - offset)) || (vlen < 3)) {
      expert_add_info(pinfo, item, &ei_ieee80211_ff_anqp_venue_length);
      break;
    }
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_venue_language,
                        tvb, offset, 3, ENC_ASCII);
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_venue_name,
                        tvb, offset + 3, vlen - 3, ENC_UTF_8);
    offset += vlen;
  }
}

static const value_string nw_auth_type_vals[] = {
  { 0, "Acceptance of terms and conditions" },
  { 1, "On-line enrollment supported" },
  { 2, "http/https redirection" },
  { 3, "DNS redirection" },
  { 0, NULL }
};

static void
dissect_network_auth_type(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset + 3 <= end) {
    uint16_t len;
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_indicator,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    len = tvb_get_letohs(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_url_len,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (len)
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_url,
                          tvb, offset, len, ENC_ASCII);
    offset += len;
  }
}

static void
dissect_anqp_network_auth_type_timestamp(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset + 2 <= end) {
    uint8_t len;
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_indicator,
                        tvb, offset, 1, ENC_NA);
    len = tvb_get_uint8(tvb, offset + 1);
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_url_len,
                        tvb, offset, 1, ENC_NA);
    offset += 2;
    if(len)
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_url,
                          tvb, offset, len, ENC_ASCII);
    offset += len;
    /* Optional Time Value - Either 0 or 10 octets */
    /* Format: Octet 0-1: Year (0-65534)
               Octet 2: Month (0-12)
               Octet 3: Day of month (0-31)
               Octet 4: Hours (0-23)
               Octet 5: Minutes (0-59)
               Octet 6: Seconds (0-59)
               Octet 7-8: Milliseconds (0-999)
               Octet 9: Reserved */
    if ((offset + 10) < end) {
      /* Enough bytes to dissect a timestamp */
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_year,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_mon,
                          tvb, offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_day,
                          tvb, offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_hr,
                          tvb, offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_min,
                          tvb, offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_sec,
                          tvb, offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_msec,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nw_auth_type_ts_rsvd,
                          tvb, offset, 1, ENC_NA);
      offset += 10;
    } else {
      /* Not enough bytes to dissect a timestamp */
    }
  }
}

static void
add_manuf(proto_item *item, tvbuff_t *tvb, int offset)
{
  const char *manuf_name;

  manuf_name = tvb_get_manuf_name_if_known(tvb, offset);
  if (manuf_name == NULL)
    return;
  proto_item_append_text(item, " - %s", manuf_name);
}

static void
dissect_roaming_consortium_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                                int end)
{
  proto_item *item;
  uint8_t     len;

  while (offset < end) {
    len = tvb_get_uint8(tvb, offset);
    item = proto_tree_add_item(tree,
                               hf_ieee80211_ff_anqp_roaming_consortium_oi_len,
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    if ((len > (end - offset)) || (len < 3)) {
      expert_add_info(pinfo, item, &ei_ieee80211_ff_anqp_roaming_consortium_oi_len);
      break;
    }
    item = proto_tree_add_item(tree,
                               hf_ieee80211_ff_anqp_roaming_consortium_oi,
                               tvb, offset, len, ENC_NA);
    add_manuf(item, tvb, offset);
    offset += len;
  }
}

static const value_string ip_addr_avail_ipv6_vals[] = {
  { 0, "Address type not available" },
  { 1, "Address type available" },
  { 2, "Availability of the address type not known" },
  { 0, NULL }
};

static const value_string ip_addr_avail_ipv4_vals[] = {
  { 0, "Address type not available" },
  { 1, "Public IPv4 address available" },
  { 2, "Port-restricted IPv4 address available" },
  { 3, "Single NATed private IPv4 address available" },
  { 4, "Double NATed private IPv4 address available" },
  { 5, "Port-restricted IPv4 address and single NATed IPv4 address available" },
  { 6, "Port-restricted IPv4 address and double NATed IPv4 address available" },
  { 7, "Availability of the address type is not known" },
  { 0, NULL }
};

static void
dissect_ip_addr_type_availability_info(proto_tree *tree, tvbuff_t *tvb,
                                       int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_anqp_ip_addr_avail_ipv6,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_ff_anqp_ip_addr_avail_ipv4,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
}

static const value_string nai_realm_encoding_vals[] = {
  { 0, "Formatted in accordance with RFC 4282" },
  { 1, "UTF-8 formatted that is not formatted in accordance with RFC 4282" },
  { 0, NULL }
};

static const range_string oper_class_rvals[] = {
  {   0,   0, "Unknown" }, /* 0 should not be used */
  {   1,  65, "Reserved" },
  {  66,  66, "0.863 GHz, 1 MHz Spacing" },
  {  67,  67, "0.863 GHz, 2 MHz Spacing" },
  {  68,  68, "0.902 GHz, 1 MHz Spacing" },
  {  69,  69, "0.902 GHz, 2 MHz Spacing" },
  {  70,  70, "0.902 GHz, 4 MHz Spacing" },
  {  71,  71, "0.902 GHz, 8 MHz Spacing" },
  {  72,  72, "0.902 GHz, 16 MHz Spacing" },
  {  73,  73, "0.9165 GHz, 1 MHz Spacing" },
  {  74,  74, "0.9175 GHz, 1 MHz Spacing" },
  {  75,  75, "0.9175 GHz, 2 MHz Spacing" },
  {  76,  76, "0.9175 GHz, 4 Spacing" },
  {  77,  77, "0.9014 GHz, 1 MHz Spacing" },
  {  78,  80, "Reserved" },
  {  81,  81, "2.407 GHz, Channels 1-13, 25 MHz Spacing" },
  {  82,  82, "2.414 GHz, Channel 14, 25 MHz Spacing" },
  {  83,  83, "2.407 GHz, Channels 1-9, 40 MHz Spacing" },
  {  84,  84, "2.407 GHz, Channels 5-13, 40 MHz Spacing" },
  {  85,  93, "Reserved" },
  {  94,  94, "3.0 GHz, Channels 133 and 137, 20 MHz Spacing" },
  {  95,  95, "3.0 GHz, Channels 132, 134, 136, and 138, 10 MHz Spacing" },
  {  96,  96, "3.0025 GHz, Channels 131-138, 5 MHz Spacing" },
  {  97, 100, "Reserved" },
  { 101, 101, "4.85 GHz, Channels 21 and 25, 20 MHz Spacing" },
  { 102, 102, "4.89 GHz, Channels 11, 13, 15, 17, and 19, 10 MHz Spacing" },
  { 103, 103, "4.9375 GHz, Channels 1-10, 5 MHz Spacing" },
  { 104, 104, "4.0 GHz, Channels 184 and 192, 40 MHz Spacing" },
  { 105, 105, "4.0 GHz, Channels 188 and 196, 40 MHz Spacing" },
  { 106, 106, "4.0 GHz, Channels 191 and 195, 20 MHz Spacing" },
  { 107, 107, "4.0 GHz, Channels 189, 191, 193, 195, and 197, 10 MHz Spacing" },
  { 108, 108, "4.0025 GHz, Channels 188-197, 5 MHz Spacing" },
  { 109, 109, "4.0 GHz, Channels 184, 188, 192, and 196, 20 MHz Spacing" },
  { 110, 110, "4.0 GHz, Channels 183-189, 10 MHz Spacing" },
  { 111, 111, "4.0025 GHz, Channels 182-189, 5 MHz Spacing" },
  { 112, 114, "Reserved" },
  { 115, 115, "5.0 GHz, Channels 36, 40, 44, and 48, 20 MHz Spacing" },
  { 116, 116, "5.0 GHz, Channels 36 and 44, 40 MHz Spacing" },
  { 117, 117, "5.0 GHz, Channels 40 and 48, 40 MHz Spacing" },
  { 118, 118, "5.0 GHz, Channels 52, 56, 60, and 64, 20 MHz Spacing" },
  { 119, 119, "5.0 GHz, Channels 52 and 60, 40 MHz Spacing" },
  { 120, 120, "5.0 GHz, Channels 56 and 64, 40 MHz Spacing" },
  { 121, 121, "5.0 GHz, Channels 100, 104, 108, 112, 116, 120, 124, 128, 132, 136, and 140, 20 MHz Spacing" },
  { 122, 122, "5.0 GHz, Channels 100, 108, 116, 124, and 132, 40 MHz Spacing" },
  { 123, 123, "5.0 GHz, Channels 104, 112, 120, 128, and 136, 40 MHz Spacing" },
  { 124, 124, "5.0 GHz, Channels 149, 153, 157, and 161, 20 MHz Spacing" },
  { 125, 125, "5.0 GHz, Channels 149, 153, 157, 161, 165, and 169, 20 MHz Spacing" },
  { 126, 126, "5.0 GHz, Channels 149 and 157, 40 MHz Spacing" },
  { 127, 127, "5.0 GHz, Channels 153 and 161, 40 MHz Spacing" },
  { 128, 128, "5.0 GHz, Channel center frequency index 42, 58, 106, 122, 138 and 155, 80 MHz Spacing" },
  { 129, 129, "5.0 GHz, Channel center frequency index 50 and 114, 160 MHz Spacing" },
  { 130, 130, "5.0 GHz, Channel center frequency index 42, 58, 106, 122, 138 and 155, 80 MHz Spacing, 80+" },
  { 131, 131, "5.950 GHz, Channels 1, 5, 9, 13, 17, 21, 25, 29, 33, 37, 41, 45, 49, 53, 57, 61, 65, 69, 73, 77, 81, 85, 89, 93, 97, 101, 105, 109, 113, 117, 121, 125, 129, 133, 137, 141, 145, 149, 153, 157, 161, 165, 169, 173, 177, 181, 185, 189, 193, 197, 201, 205, 209, 213, 217, 221, 225, 229, and 233, 20 MHz Spacing" },
  { 132, 132, "5.950 GHz, Channel center frequency index 3, 11, 19, 27, 35, 43, 51, 59, 67, 75, 83, 91, 99, 107, 115, 123, 131, 139, 147, 155, 163, 171, 179, 187, 195, 203, 211, 219, and 227, 40 MHz Spacing" },
  { 133, 133, "5.950 GHz, Channel center frequency index 7, 23, 39, 55, 71, 87, 103, 119, 135, 151, 167, 183, 199, and 215, 80 MHz Spacing" },
  { 134, 134, "5.950 GHz, Channel center frequency index 15, 47, 79, 111, 143, 175, and 207, 160 MHz Spacing" },
  { 135, 135, "5.950 GHz, Channel center frequency index 7, 23, 39, 55, 71, 87, 103, 119, 135, 151, 167, 183, 199, and 215, 80 MHz Spacing, 80+" },
  { 136, 136, "5.925 GHz, Channel center frequency index 2, 20 MHz Spacing" },
  { 137, 137, "5.925 GHz, Channel center frequency index 31, 63, 95, 127, 159, and 191, 320 MHz Spacing" },
  { 138, 179, "Reserved" },
  { 180, 180, "56.16 GHz, Channels 1, 2, 3, 4, 5, and 6, 2160 MHz Spacing" },
  { 181, 181, "56.16 GHz, Channels 2 and 3, 2160 MHz Spacing" },
  { 182, 182, "56.70 GHz, Channels 35, 36, 37, and 38, 1080 MHz Spacing" },
  { 183, 183, "42.66 GHz, Channels 1, 2, 3, 4, 5, 6, 7, and 8, 540 MHz Spacing" },
  { 184, 184, "47.52 GHz, Channels 9 and 10, 540 MHz Spacing" },
  { 185, 185, "42.93 GHz, Channels 11, 12, 13, and 14, 1080 MHz Spacing" },
  { 186, 186, "47.79 GHz, Channel 15, 1080 MHz Spacing" },
  { 187, 191, "Reserved" },
  { 192, 254, "Vendor-Specific" },

  { 255, 255, "Reserved" },
  {   0,   0, NULL }
};

static const value_string nai_realm_auth_param_id_vals[] = {
  {   1, "Expanded EAP Method" },
  {   2, "Non-EAP Inner Authentication Type" },
  {   3, "Inner Authentication EAP Method Type" },
  {   4, "Expanded Inner EAP Method" },
  {   5, "Credential Type" },
  {   6, "Tunneled EAP Method Credential Type" },
  { 221, "Vendor Specific" },
  { 0, NULL }
};

static void
dissect_nai_realm_list(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int end)
{
  uint16_t      count, len;
  proto_item   *item, *r_item;
  int           f_end, eap_end;
  uint8_t       nai_len, eap_count, eap_len, auth_param_count, auth_param_len;
  uint8_t       auth_param_id;
  proto_tree   *realm_tree, *eap_tree;
  const uint8_t *realm;

  count = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_anqp_nai_realm_count,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  while (count > 0) {
    len = tvb_get_letohs(tvb, offset);
    realm_tree = proto_tree_add_subtree(tree, tvb, offset, 2 + len, ett_nai_realm, &r_item, "NAI Realm Data");

    item = proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_field_len,
                               tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (offset + len > end) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_ff_anqp_nai_field_len,
                             "Invalid NAI Realm List");
      break;
    }
    f_end = offset + len;
    proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_realm_encoding,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    nai_len = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_realm_length,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    if (offset + nai_len > f_end) {
      expert_add_info_format(pinfo, r_item, &ei_ieee80211_ff_anqp_nai_field_len,
                             "Invalid NAI Realm Data");
      break;
    }
    proto_tree_add_item_ret_string(realm_tree, hf_ieee80211_ff_anqp_nai_realm,
                        tvb, offset, nai_len, ENC_ASCII|ENC_NA, pinfo->pool, &realm);
    if (realm) {
      proto_item_append_text(r_item, " (%s)", realm);
    }
    offset += nai_len;
    eap_count = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(realm_tree, hf_ieee80211_ff_anqp_nai_realm_eap_count,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    while (eap_count > 0) {
      eap_len = tvb_get_uint8(tvb, offset);
      eap_end = offset + 1 + eap_len;
      eap_tree = proto_tree_add_subtree(realm_tree, tvb, offset, 1 + eap_len,
                                 ett_nai_realm_eap, NULL, "EAP Method");

      item = proto_tree_add_item(eap_tree,
                                 hf_ieee80211_ff_anqp_nai_realm_eap_len,
                                 tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset + eap_len > f_end) {
        expert_add_info(pinfo, item, &ei_ieee80211_ff_anqp_nai_realm_eap_len);
        break;
      }

      proto_item_append_text(eap_tree, ": %s",
                             val_to_str_ext(pinfo->pool, tvb_get_uint8(tvb, offset),
                                            &eap_type_vals_ext, "Unknown (%d)"));
      proto_tree_add_item(eap_tree, hf_ieee80211_ff_anqp_nai_realm_eap_method,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      auth_param_count = tvb_get_uint8(tvb, offset);
      proto_tree_add_item(eap_tree,
                          hf_ieee80211_ff_anqp_nai_realm_auth_param_count,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;

      while (auth_param_count > 0) {
        auth_param_id = tvb_get_uint8(tvb, offset);
        proto_tree_add_item(eap_tree,
                            hf_ieee80211_ff_anqp_nai_realm_auth_param_id,
                            tvb, offset, 1, ENC_LITTLE_ENDIAN);
        offset += 1;
        auth_param_len = tvb_get_uint8(tvb, offset);
        proto_tree_add_item(eap_tree,
                            hf_ieee80211_ff_anqp_nai_realm_auth_param_len,
                            tvb, offset, 1, ENC_LITTLE_ENDIAN);
        offset += 1;
        item = proto_tree_add_item(
          eap_tree, hf_ieee80211_ff_anqp_nai_realm_auth_param_value,
          tvb, offset, auth_param_len, ENC_NA);
        if ((auth_param_id == 3) && (auth_param_len == 1)) {
          uint8_t inner_method = tvb_get_uint8(tvb, offset);
          const char *str;
          str = val_to_str_ext(pinfo->pool, inner_method, &eap_type_vals_ext, "Unknown (%d)");

          proto_item_append_text(eap_tree, " / %s", str);
          proto_item_append_text(item, " - %s", str);
        }
        offset += auth_param_len;

        auth_param_count--;
      }

      offset = eap_end;
      eap_count--;
    }

    offset = f_end;
    count--;
  }
}

static void
dissect_3gpp_cellular_network_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t     iei, num, plmn_idx = 0;
  proto_item *item;

  /* See Annex A of 3GPP TS 24.234 v8.1.0 for description */
  proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_gud, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_udhl, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  item = proto_tree_add_item_ret_uint8(tree, hf_ieee80211_3gpp_gc_iei, tvb, offset, 1, ENC_LITTLE_ENDIAN, &iei);
  if (iei == 0)
    proto_item_append_text(item, " (PLMN List)");
  else
    return;
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_3gpp_gc_plmn_len, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item_ret_uint8(tree, hf_ieee80211_3gpp_gc_num_plmns, tvb, offset, 1, ENC_LITTLE_ENDIAN, &num);
  offset += 1;
  while (num > 0) {
    proto_item *plmn_item = NULL;
    proto_tree *plmn_tree = NULL;
    unsigned plmn_val = 0;

    if (tvb_reported_length_remaining(tvb, offset) < 3)
      break;
    plmn_val = tvb_get_letoh24(tvb, offset);
    plmn_item = proto_tree_add_uint_format(tree, hf_ieee80211_3gpp_gc_plmn,
                                tvb, offset, 3, plmn_val, "PLMN %d (0x%x)",
                                plmn_idx++, plmn_val);
    plmn_tree = proto_item_add_subtree(plmn_item, ett_ieee80211_3gpp_plmn);
    dissect_e212_mcc_mnc_wmem_packet_str(tvb, pinfo, plmn_tree, offset, E212_NONE, true);
    num--;
    offset += 3;
  }
}

static void
dissect_domain_name_list(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  uint8_t len;

  while (offset < end) {
    len = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_domain_name_len,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_domain_name,
                        tvb, offset, len, ENC_ASCII);
    offset += len;
  }
}

static int
dissect_hs20_subscription_remediation(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, void *data _U_)
{
  int offset = 0;
  uint8_t url_len = tvb_get_uint8(tvb, offset);
  proto_item *pi = NULL;

  proto_tree_add_item(tree, hf_ieee80211_hs20_subscription_remediation_url_len, tvb, offset,
                        1, ENC_NA);
  offset++;
  if (tvb_reported_length_remaining(tvb, offset) >= url_len) {
    pi = proto_tree_add_item(tree, hf_ieee80211_hs20_subscription_remediation_server_url,
                        tvb, offset, url_len, ENC_ASCII);
    offset += url_len;
    proto_item_set_url(pi);
    proto_tree_add_item(tree, hf_ieee80211_hs20_subscription_remediation_server_method,
                        tvb, offset, 1, ENC_NA);
    offset++;
  }

  return offset;
}

static int
dissect_hs20_deauthentication_imminent(tvbuff_t *tvb, packet_info *pinfo _U_,
  proto_tree *tree, void *data _U_)
{
  int offset = 0;
  uint8_t url_len = 0;
  proto_item *pi = NULL;

  proto_tree_add_item(tree, hf_ieee80211_hs20_deauth_reason_code, tvb, offset, 1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_hs20_reauth_delay, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  url_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_hs20_deauth_reason_url_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  if (tvb_reported_length_remaining(tvb, offset) >= url_len) {
    pi = proto_tree_add_item(tree, hf_ieee80211_hs20_deauth_imminent_reason_url,
                        tvb, offset, url_len, ENC_ASCII);
    offset += url_len;
    proto_item_set_url(pi);
  }
  return offset;
}

#define HS20_ANQP_HS_QUERY_LIST              1
#define HS20_ANQP_HS_CAPABILITY_LIST         2
#define HS20_ANQP_OPERATOR_FRIENDLY_NAME     3
#define HS20_ANQP_WAN_METRICS                4
#define HS20_ANQP_CONNECTION_CAPABILITY      5
#define HS20_ANQP_NAI_HOME_REALM_QUERY       6
#define HS20_ANQP_OPERATING_CLASS_INDICATION 7
#define HS20_ANQP_OSU_PROVIDERS_LIST         8
/* 9 is reserved */
#define HS20_ANQP_ICON_REQUEST               10
#define HS20_ANQP_ICON_BINARY_FILE           11
#define HS20_ANQP_OPERATOR_ICON_METADATA     12
#define HS20_ANQP_ADVICE_OF_CHARGE           13

static const value_string hs20_anqp_subtype_vals[] = {
  { HS20_ANQP_HS_QUERY_LIST, "HS Query list" },
  { HS20_ANQP_HS_CAPABILITY_LIST, "HS Capability List" },
  { HS20_ANQP_OPERATOR_FRIENDLY_NAME, "Operator Friendly Name" },
  { HS20_ANQP_WAN_METRICS, "WAN Metrics" },
  { HS20_ANQP_CONNECTION_CAPABILITY, "Connection Capability" },
  { HS20_ANQP_NAI_HOME_REALM_QUERY, "NAI Home Realm Query" },
  { HS20_ANQP_OPERATING_CLASS_INDICATION, "Operating Class Indication" },
  { HS20_ANQP_OSU_PROVIDERS_LIST, "OSU Providers List" },
  { 9, "Reserved" },
  { HS20_ANQP_ICON_REQUEST, "Icon Request" },
  { HS20_ANQP_ICON_BINARY_FILE, "Icon Binary File" },
  { HS20_ANQP_OPERATOR_ICON_METADATA, "Operator Icon Metadata" },
  { HS20_ANQP_ADVICE_OF_CHARGE, "Advice of Charge" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_hs_query_list(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset < end) {
    proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_hs_query_list,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static void
dissect_hs20_anqp_operator_friendly_name(proto_tree *tree, tvbuff_t *tvb,
                                         packet_info *pinfo, int offset,
                                         int end, int hf_array[],
                                         int ett_val)
{
  int ofn_index = 0;

  while (offset + 4 <= end) {
    uint8_t vlen = tvb_get_uint8(tvb, offset);
    proto_tree *ofn_tree = NULL;
    proto_item *item = NULL, *pi = NULL;
    int start_offset = offset;

    ofn_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1, ett_val,
                                        &pi, "Friendly Name [%d]", ofn_index + 1);

    item = proto_tree_add_item(ofn_tree, hf_array[0],
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (vlen > end - offset || vlen < 3) {
      expert_add_info(pinfo, item, &ei_hs20_anqp_ofn_length);
      break;
    }
    proto_tree_add_item(ofn_tree, hf_array[1],
                        tvb, offset, 3, ENC_ASCII|ENC_NA);
    proto_tree_add_item(ofn_tree, hf_array[2],
                        tvb, offset + 3, vlen - 3, ENC_UTF_8);
    offset += vlen;

    proto_item_set_len(pi, offset - start_offset);
    ofn_index++;
  }
}

static const value_string hs20_wm_link_status_vals[] = {
  { 0, "Reserved" },
  { 1, "Link up" },
  { 2, "Link down" },
  { 3, "Link in test state" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_wan_metrics(proto_tree *tree, tvbuff_t *tvb, int offset, bool request)
{
  if (request)
    return;

  if(tvb_reported_length_remaining(tvb, offset) < 13)
    return;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_link_status,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_symmetric_link,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_at_capacity,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_reserved,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_downlink_speed,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_uplink_speed,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_downlink_load,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_uplink_load,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_wan_metrics_lmd,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
}

static const value_string hs20_cc_proto_vals[] = {
  {  1, "ICMP" },
  {  6, "TCP" },
  { 17, "UDP" },
  { 50, "ESP" },
  {  0, NULL }
};

static const value_string hs20_cc_port_vals[] = {
  {    0, "[Supported]" }, /* Used to indicate ICMP, ESP for IPSec VPN, or IKEv2 for IPSec VPN */
  {   20, "FTP" },
  {   22, "SSH" },
  {   80, "HTTP" },
  {  443, "HTTPS" },
  {  500, "IKEv2 for IPSec VPN" },
  { 1723, "PPTP for IPSec VPN" },
  { 4500, "[Optional] IKEv2 for IPSec VPN" },
  { 5060, "VOIP" },
  {    0, NULL },
};

static const value_string hs20_cc_status_vals[] = {
  { 0, "Closed" },
  { 1, "Open" },
  { 2, "Unknown" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_connection_capability(proto_tree *tree, packet_info* pinfo, tvbuff_t *tvb,
                                        int offset, int end)
{
  proto_tree *tuple;
  while (offset + 4 <= end) {
    uint8_t ip_proto, status;
    uint16_t port_num;

    ip_proto = tvb_get_uint8(tvb, offset);
    port_num = tvb_get_letohs(tvb, offset + 1);
    status = tvb_get_uint8(tvb, offset + 3);

    tuple = proto_tree_add_subtree_format(tree, tvb, offset, 4, ett_hs20_cc_proto_port_tuple, NULL,
                               "ProtoPort Tuple - ip_proto=%s port_num=%s status=%s",
                               val_to_str(pinfo->pool, ip_proto, hs20_cc_proto_vals, "Unknown (%u)"),
                               val_to_str(pinfo->pool, port_num, hs20_cc_port_vals, "Unknown (%u)"),
                               val_to_str(pinfo->pool, status, hs20_cc_status_vals, "Reserved (%u)"));
    proto_tree_add_item(tuple, hf_ieee80211_hs20_anqp_cc_proto_ip_proto,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    proto_tree_add_item(tuple, hf_ieee80211_hs20_anqp_cc_proto_port_num,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tuple, hf_ieee80211_hs20_anqp_cc_proto_status,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static void
dissect_hs20_anqp_nai_home_realm_query(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                                       int offset, int end)
{
  uint8_t len;
  proto_item *item;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_nai_hrq_count,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;

  while (offset + 2 <= end) {
    proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_nai_hrq_encoding_type,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    len = tvb_get_uint8(tvb, offset);
    item = proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_nai_hrq_length,
                               tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (offset + len > end) {
      expert_add_info(pinfo, item, &ei_hs20_anqp_nai_hrq_length);
      break;
    }
    proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_nai_hrq_realm_name,
                        tvb, offset, len, ENC_ASCII);
    offset += len;
  }
}

static void
dissect_hs20_anqp_oper_class_indic(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  while (offset < end) {
    proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_oper_class_indic,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
  }
}

static int
dissect_hs20_osu_friendly_names(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo, int offset, int end _U_)
{
  int osu_fn_hf_array[3] = {hf_ieee80211_hs20_osu_friendly_name_length,
                            hf_ieee80211_hs20_osu_friendly_name_language,
                            hf_ieee80211_hs20_osu_friendly_name_name };
  uint16_t osu_fn_count = tvb_get_letohs(tvb, offset);
  proto_tree *fn_tree = NULL;

  proto_tree_add_item(tree, hf_ieee80211_hs20_osu_friendly_names_len, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  fn_tree = proto_tree_add_subtree(tree, tvb, offset, osu_fn_count,
                        ett_hs20_friendly_names_list, NULL,
                        "Friendly Names List");

  dissect_hs20_anqp_operator_friendly_name(fn_tree, tvb, pinfo, offset,
                        offset + osu_fn_count,
                        osu_fn_hf_array, ett_hs20_friendly_name_tree);

  return offset + osu_fn_count;
}

static int
dissect_hs20_osu_icon_available(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo _U_, int offset, int end _U_, uint16_t icon_index)
{
  proto_tree *icon_avail = NULL;
  proto_item *pi = NULL;
  int start_offset = offset;
  uint8_t icon_type_len = 0, icon_filename_len = 0;

  icon_avail = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_hs20_osu_icon_tree, &pi,
                        "Icon Available %d", icon_index);

  proto_tree_add_item(icon_avail, hf_ieee80211_osu_icon_avail_width, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(icon_avail, hf_ieee80211_osu_icon_avail_height, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(icon_avail, hf_ieee80211_osu_icon_avail_lang_code, tvb, offset, 3,
                        ENC_ASCII);
  offset += 3;

  icon_type_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(icon_avail, hf_ieee80211_osu_icon_avail_icon_type_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(icon_avail, hf_ieee80211_osu_icon_avail_icon_type, tvb, offset,
                        icon_type_len, ENC_ASCII);
  offset += icon_type_len;

  icon_filename_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(icon_avail, hf_ieee80211_osu_icon_avail_filename_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(icon_avail, hf_ieee80211_osu_icon_avail_filename, tvb, offset,
                        icon_filename_len, ENC_ASCII);
  offset += icon_filename_len;

  proto_item_set_len(pi, offset - start_offset);

  return offset;
}

static const value_string osu_method_vals[] = {
  { 0, "OMA DM" },
  { 1, "SOAP XML SPP" },
  { 0, NULL },
};

static int
dissect_hs20_osu_provider(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo, int offset, int end, uint8_t provider_index)
{
  proto_tree *prov_tree = NULL;
  proto_item *osupi = NULL, *uri_pi = NULL;
  int start_offset = offset;
  uint8_t osu_server_uri_len = 0;
  uint8_t osu_method_list_len = 0;
  uint16_t icons_avail = 0, icons_index = 0;
  uint8_t osu_nai_len = 0;
  uint16_t osu_service_desc_len = 0;

  prov_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_hs20_osu_provider_tree, &osupi,
                        "OSU Provider %d", provider_index);

  proto_tree_add_item(prov_tree, hf_ieee80211_hs20_osu_prov_length, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  offset = dissect_hs20_osu_friendly_names(prov_tree, tvb, pinfo, offset, end);

  proto_item_set_len(osupi, offset - start_offset);

  osu_server_uri_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(prov_tree, hf_ieee80211_hs20_osu_server_uri_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  uri_pi = proto_tree_add_item(prov_tree, hf_ieee80211_hs20_osu_server_uri, tvb, offset,
                        osu_server_uri_len, ENC_ASCII);
  offset += osu_server_uri_len;
  proto_item_set_url(uri_pi);

  osu_method_list_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(prov_tree, hf_ieee80211_hs20_osu_method_list_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  if (osu_method_list_len > 0) {
    proto_tree *osu_method_list = NULL;
    uint8_t osu_method_list_index = 0;

    osu_method_list = proto_tree_add_subtree(prov_tree, tvb, offset,
                                osu_method_list_len,
                                ett_hs20_osu_provider_method_list,
                                NULL, "OSU Method List");
    while (osu_method_list_len > osu_method_list_index) {
      proto_item *pi = NULL;
      uint8_t method = tvb_get_uint8(tvb, offset);

      pi = proto_tree_add_item(osu_method_list, hf_ieee80211_hs20_osu_method_val, tvb,
                        offset, 1, ENC_NA);
      proto_item_append_text(pi, ": %s",
                                val_to_str_const(method, osu_method_vals,
                                        "Reserved"));
      offset++;
      osu_method_list_index++;
    }
  }

  icons_avail = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(prov_tree, hf_ieee80211_hs20_icons_avail_len, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  if (icons_avail > 0) {
    proto_tree *icon_list = NULL;
    proto_item *pi = NULL;

    start_offset = offset;

    icon_list = proto_tree_add_subtree(prov_tree, tvb, offset, -1,
                                ett_osu_icons_avail_list, &pi,
                                "Icons Available");

    while ((offset - start_offset) < icons_avail) {
      offset = dissect_hs20_osu_icon_available(icon_list, tvb, pinfo, offset,
                                end, icons_index);
      icons_index++;
    }

    proto_item_set_len(pi, offset - start_offset);
  }

  proto_tree_add_item_ret_uint8(prov_tree, hf_ieee80211_hs20_osu_nai_len, tvb, offset, 1, ENC_NA, &osu_nai_len);
  offset++;

  if (osu_nai_len > 0) {
    proto_tree_add_item(prov_tree, hf_ieee80211_hs20_osu_nai, tvb, offset,
                        osu_nai_len, ENC_ASCII);
    offset += osu_nai_len;
  }

  osu_service_desc_len = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(prov_tree, hf_ieee80211_hs20_osu_service_desc_len, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  if (osu_service_desc_len > 0) {
    proto_tree *desc_tree = NULL;
    proto_item *pi = NULL;
    uint8_t service_desc_index = 0;

    start_offset = offset;
    desc_tree = proto_tree_add_subtree(prov_tree, tvb, offset, -1,
                                ett_hs20_osu_service_desc_list, &pi,
                                "OSU Service Description List");

    while ((offset - start_offset) < osu_service_desc_len) {
      proto_tree *desc_duple = NULL;
      uint8_t serv_desc_len = tvb_get_uint8(tvb, offset);

      desc_duple = proto_tree_add_subtree_format(desc_tree, tvb, offset,
                                1 + serv_desc_len,
                                ett_hs20_osu_service_desc_tree, NULL,
                                "OSU Service Description Duple %d",
                                service_desc_index);

      proto_tree_add_item(desc_duple, hf_ieee80211_hs20_osu_service_desc_duple_len, tvb,
                                offset, 1, ENC_NA);
      offset++;

      proto_tree_add_item(desc_duple, hf_ieee80211_hs20_osu_service_desc_lang, tvb, offset,
                                3, ENC_ASCII);
      offset += 3;

      proto_tree_add_item(desc_duple, hf_ieee80211_hs20_osu_service_desc, tvb, offset,
                                serv_desc_len - 3, ENC_ASCII);
      offset += serv_desc_len - 3;

      service_desc_index++;
    }

    proto_item_set_len(pi, offset - start_offset);
  }

  return offset;
}

static void
dissect_hs20_anqp_osu_providers_list(proto_tree *tree, tvbuff_t *tvb,
  packet_info *pinfo, int offset, int end)
{
  uint8_t ssid_len = tvb_get_uint8(tvb, offset);
  uint8_t osu_prov_count = 0, osu_prov_index = 0;

  proto_tree_add_item(tree, hf_ieee80211_hs20_osu_providers_list_ssid_len, tvb, offset, 1,
                        ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_hs20_osu_providers_ssid, tvb, offset, ssid_len,
                        ENC_UTF_8);
  offset += ssid_len;

  osu_prov_count = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_hs20_osu_providers_count, tvb, offset, 1,
                        ENC_NA);
  offset++;

  if (osu_prov_count > 0) {
    int start_offset = offset;
    proto_item *pi = NULL;
    proto_tree *osu_prov_list = NULL;

    osu_prov_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_hs20_osu_providers_list, &pi,
                        "OSU Providers List");
    while (offset < end && osu_prov_count > osu_prov_index) {
      offset = dissect_hs20_osu_provider(osu_prov_list, tvb, pinfo, offset, end,
                        osu_prov_index);
      osu_prov_index++;
    }

    proto_item_set_len(pi, offset - start_offset);
  }
}

static void
dissect_hs20_anqp_icon_request(proto_tree *tree, tvbuff_t *tvb, int offset,
  int end)
{
  proto_tree_add_item(tree, hf_ieee80211_hs20_icon_request_filename, tvb, offset,
                        end - offset, ENC_UTF_8);
}

static const value_string hs20_icon_download_status_vals[] = {
  { 0, "Success" },
  { 1, "File not found" },
  { 2, "Unspecified file error" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_icon_binary_file(proto_tree *tree, packet_info* pinfo, tvbuff_t *tvb, int offset,
  int end _U_)
{
  uint8_t icon_download_status = tvb_get_uint8(tvb, offset);
  proto_item *pi = NULL;
  uint8_t icon_type_len = 0;
  uint16_t icon_binary_data_len = 0;

  pi = proto_tree_add_item(tree, hf_ieee80211_hs20_icon_binary_file_status, tvb, offset, 1,
                        ENC_NA);
  offset++;
  proto_item_append_text(pi, ": %s",
                         val_to_str(pinfo->pool, icon_download_status,
                                    hs20_icon_download_status_vals,
                                    "Reserved (%u)"));

  icon_type_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_hs20_icon_type_length, tvb, offset, 1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_hs20_icon_type, tvb, offset, icon_type_len,
                        ENC_UTF_8);
  offset += icon_type_len;

  icon_binary_data_len = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_hs20_icon_binary_data_len, tvb, offset, 2,
                        ENC_BIG_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_hs20_icon_binary_data, tvb, offset,
                        icon_binary_data_len, ENC_NA);
}

static void
dissect_hs20_anqp_operator_icon_metadata(proto_tree *tree, tvbuff_t *tvb,
  int offset, int end _U_)
{
  proto_item *pi = NULL;
  int start_offset = offset;
  uint8_t icon_type_len = 0, icon_filename_len = 0;

  proto_tree_add_item(tree, hf_ieee80211_osu_icon_avail_width, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_osu_icon_avail_height, tvb, offset, 2,
                        ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_osu_icon_avail_lang_code, tvb, offset, 3,
                        ENC_ASCII);
  offset += 3;

  icon_type_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_osu_icon_avail_icon_type_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_osu_icon_avail_icon_type, tvb, offset,
                        icon_type_len, ENC_ASCII);
  offset += icon_type_len;

  icon_filename_len = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_osu_icon_avail_filename_len, tvb, offset,
                        1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_osu_icon_avail_filename, tvb, offset,
                        icon_filename_len, ENC_ASCII);
  offset += icon_filename_len;

  proto_item_set_len(pi, offset - start_offset);
}

static void
dissect_anqp_venue_url(proto_tree *tree, tvbuff_t *tvb, int offset, int end)
{
  uint16_t url_duple_index = 0;

  while (offset < end) {
    proto_tree *venue_url = NULL;
    proto_item *url_pi = NULL;
    uint8_t url_duple_len = tvb_get_uint8(tvb, offset);

    venue_url = proto_tree_add_subtree_format(tree, tvb, offset,
                        url_duple_len + 1, ett_hs20_venue_url, NULL,
                        "Venue URL Duple %d", url_duple_index);

    proto_tree_add_item(venue_url, hf_ieee80211_hs20_anqp_venue_url_length, tvb, offset,
                        1, ENC_NA);
    offset++;

    proto_tree_add_item(venue_url, hf_ieee80211_hs20_anqp_venue_number, tvb, offset, 1,
                        ENC_NA);
    offset++;

    url_pi = proto_tree_add_item(venue_url, hf_ieee80211_hs20_anqp_venue_url, tvb, offset,
                        url_duple_len -1, ENC_ASCII);
    proto_item_set_url(url_pi);

    offset += (url_duple_len - 1);

    url_duple_index++;
  }
}

static const value_string advice_of_charge_type_vals[] = {
  { 0, "Time-based" },
  { 1, "Data-volume-based" },
  { 2, "Time-and-data-volume-based" },
  { 3, "Unlimited" },
  { 0, NULL }
};

static void
dissect_hs20_anqp_advice_of_charge(proto_tree *tree, packet_info* pinfo, tvbuff_t *tvb, int offset,
  int end _U_)
{
  uint16_t toc_index = 0;

  while (offset < end) {
    uint16_t adv_charge_len = tvb_get_letohs(tvb, offset);
    proto_tree *aoc_tree = NULL;
    proto_tree *plan_info_tree = NULL;
    proto_item *pi = NULL, *tpi = NULL;
    int start_offset = offset;
    uint8_t aoc_type = 0, nai_realm_len = 0;
    uint8_t plan_index = 0;
    uint16_t plan_tot_len = 0;
    int plan_offset = 0;

    aoc_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_hs20_advice_of_charge, &pi,
                        "Advice of Charge Duple %d", toc_index);
    proto_tree_add_item(aoc_tree, hf_ieee80211_hs20_anqp_advice_of_charge_length, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    aoc_type = tvb_get_uint8(tvb, offset);
    tpi = proto_tree_add_item(aoc_tree, hf_ieee80211_hs20_anqp_advice_of_charge_type, tvb,
                        offset, 1, ENC_NA);
    offset++;
    proto_item_append_text(tpi, ": %s",
                                val_to_str(pinfo->pool, aoc_type,
                                        advice_of_charge_type_vals,
                                        "Reserved (%u)"));

    proto_tree_add_item(aoc_tree, hf_ieee80211_hs20_anqp_aoc_nai_realm_encoding, tvb,
                        offset, 1, ENC_NA);
    offset++;

    nai_realm_len = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(aoc_tree, hf_ieee80211_hs20_anqp_aoc_nai_realm_len, tvb, offset,
                        1, ENC_NA);
    offset++;

    proto_tree_add_item(aoc_tree, hf_ieee80211_hs20_anqp_aoc_nai_realm, tvb, offset,
                        nai_realm_len, ENC_UTF_8);
    offset += nai_realm_len;

    plan_tot_len = adv_charge_len - 3 - nai_realm_len;
    plan_offset = offset;

    while (offset < (plan_offset + plan_tot_len)) {
        uint16_t plan_len = tvb_get_letohs(tvb, offset);

        plan_info_tree = proto_tree_add_subtree_format(aoc_tree, tvb, offset,
                                plan_len + 2, ett_hs20_aoc_plan, NULL,
                                "Plan #%u", plan_index);

        proto_tree_add_item(plan_info_tree, hf_ieee80211_hs20_anqp_aoc_plan_len, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
        offset += 2;

        proto_tree_add_item(plan_info_tree, hf_ieee80211_hs20_anqp_aoc_plan_lang, tvb,
                        offset, 3, ENC_ASCII);
        offset += 3;

        proto_tree_add_item(plan_info_tree, hf_ieee80211_hs20_anqp_aoc_plan_curcy, tvb,
                        offset, 3, ENC_ASCII);
        offset += 3;

        proto_tree_add_item(plan_info_tree, hf_ieee80211_hs20_anqp_aoc_plan_information,
                        tvb, offset, plan_len - 6, ENC_UTF_8);
        offset += plan_len - 6;

        plan_index++;
    }

    proto_item_set_len(pi, offset - start_offset);

    toc_index++;
  }
}

static int
dissect_hs20_anqp(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data)
{
  uint8_t subtype;
  int ofn_hf_array[3] = {hf_ieee80211_hs20_anqp_ofn_length,
                         hf_ieee80211_hs20_anqp_ofn_language,
                         hf_ieee80211_hs20_anqp_ofn_name };

  int end = tvb_reported_length(tvb);
  int offset = 0;
  anqp_info_dissector_data_t* anqp_data = (anqp_info_dissector_data_t*)data;

  DISSECTOR_ASSERT(anqp_data);

  subtype = tvb_get_uint8(tvb, offset);
  proto_item_append_text(tree, " - HS 2.0 %s",
                         val_to_str(pinfo->pool, subtype, hs20_anqp_subtype_vals,
                                    "Reserved (%u)"));
  if (anqp_data->idx == 0) {
    col_append_fstr(pinfo->cinfo, COL_INFO, " HS 2.0 %s",
                    val_to_str(pinfo->pool, subtype, hs20_anqp_subtype_vals,
                               "Reserved (%u)"));
  } else if (anqp_data->idx == 1) {
    col_append_str(pinfo->cinfo, COL_INFO, ", ..");
  }
  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_subtype, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_reserved, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset++;

  switch (subtype) {
  case HS20_ANQP_HS_QUERY_LIST:
    dissect_hs20_anqp_hs_query_list(tree, tvb, offset, end);
    break;
  case HS20_ANQP_HS_CAPABILITY_LIST:
    dissect_hs20_anqp_hs_capability_list(tree, tvb, offset, end);
    break;
  case HS20_ANQP_OPERATOR_FRIENDLY_NAME:
    dissect_hs20_anqp_operator_friendly_name(tree, tvb, pinfo, offset, end,
                                ofn_hf_array, ett_hs20_ofn_tree);
    break;
  case HS20_ANQP_WAN_METRICS:
    dissect_hs20_anqp_wan_metrics(tree, tvb, offset, anqp_data->request);
    break;
  case HS20_ANQP_CONNECTION_CAPABILITY:
    dissect_hs20_anqp_connection_capability(tree, pinfo, tvb, offset, end);
    break;
  case HS20_ANQP_NAI_HOME_REALM_QUERY:
    dissect_hs20_anqp_nai_home_realm_query(tree, tvb, pinfo, offset, end);
    break;
  case HS20_ANQP_OPERATING_CLASS_INDICATION:
    dissect_hs20_anqp_oper_class_indic(tree, tvb, offset, end);
    break;
  case HS20_ANQP_OSU_PROVIDERS_LIST:
    dissect_hs20_anqp_osu_providers_list(tree, tvb, pinfo, offset, end);
    break;
  case HS20_ANQP_ICON_REQUEST:
    dissect_hs20_anqp_icon_request(tree, tvb, offset, end);
    break;
  case HS20_ANQP_ICON_BINARY_FILE:
    dissect_hs20_anqp_icon_binary_file(tree, pinfo, tvb, offset, end);
    break;
  case HS20_ANQP_OPERATOR_ICON_METADATA:
    dissect_hs20_anqp_operator_icon_metadata(tree, tvb, offset, end);
    break;
  case HS20_ANQP_ADVICE_OF_CHARGE:
    dissect_hs20_anqp_advice_of_charge(tree, pinfo, tvb, offset, end);
    break;
  default:
    if (offset == end)
      break;
    proto_tree_add_item(tree, hf_ieee80211_hs20_anqp_payload, tvb, offset,
                        end - offset, ENC_NA);
    break;
  }

  return tvb_captured_length(tvb);
}

// MBO ANQP element subtypes
#define MBO_ANQP_QUERY_LIST               1
#define MBO_ANQP_CELLULAR_DATA_PREFERENCE 2

static const value_string mbo_anqp_subtype_vals[] = {
  { MBO_ANQP_QUERY_LIST, "MBO Query List" },
  { MBO_ANQP_CELLULAR_DATA_PREFERENCE, "Cellular Data Connection Preference" },
  { 0, NULL }
};

static int
dissect_mbo_anqp(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data _U_)
{
  uint8_t subtype;
  int len = tvb_reported_length(tvb);
  int offset = 0;

  if (len < 2) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
    return offset;
  }

  proto_tree_add_item_ret_uint8(tree, hf_ieee80211_wfa_anqp_mbo_subtype, tvb, offset, 1, ENC_NA, &subtype);
  offset++;
  len--;

  switch (subtype) {
  case MBO_ANQP_QUERY_LIST:
    while (len > 0) {
      proto_tree_add_item(tree, hf_ieee80211_wfa_anqp_mbo_query, tvb, offset, 1, ENC_NA);
      offset++;
      len--;
    }
    break;
  case MBO_ANQP_CELLULAR_DATA_PREFERENCE:
    proto_tree_add_item(tree, hf_ieee80211_wfa_anqp_mbo_cellular_pref, tvb, offset,
                        1, ENC_NA);
    offset++;
    len--;
    break;
  default:
    break;
  }

  return offset;
}

static int
dissect_vendor_wifi_alliance_anqp(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data)
{
  uint8_t subtype;
  int offset = 0;
  tvbuff_t *subtvb;

  proto_tree_add_item_ret_uint8(tree, hf_ieee80211_anqp_wfa_subtype, tvb, offset, 1, ENC_NA, &subtype);
  offset += 1;

  subtvb = tvb_new_subset_remaining(tvb, offset);
  if (!dissector_try_uint_with_data(wifi_alliance_anqp_info_table, subtype, subtvb, pinfo, tree, false, data))
      call_data_dissector(subtvb, pinfo, tree);

  return tvb_captured_length(tvb);
}


static int
dissect_neighbor_report(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data);

static int
dissect_anqp_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                  bool request, int idx)
{
  uint16_t    id, len;
  uint32_t    oui;
  proto_item *item;
  tvbuff_t *vendor_tvb;
  anqp_info_dissector_data_t anqp_info;

  item = proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info_id,
                             tvb, offset, 2, ENC_LITTLE_ENDIAN);
  id = tvb_get_letohs(tvb, offset);
  if (id != ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST) {
    if (idx == 0) {
      proto_item_append_text(tree, " - %s",
                             val_to_str_ext(pinfo->pool, id, &anqp_info_id_vals_ext, "Unknown (%u)"));
      col_append_fstr(pinfo->cinfo, COL_INFO, " %s",
                      val_to_str_ext(pinfo->pool, id, &anqp_info_id_vals_ext, "Unknown (%u)"));
    } else if (idx == 1) {
      proto_item_append_text(tree, ", ..");
      col_append_str(pinfo->cinfo, COL_INFO, ", ..");
    }
  }
  tree = proto_item_add_subtree(item, ett_gas_anqp);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  len = tvb_get_letohs(tvb, offset);
  offset += 2;
  if (tvb_reported_length_remaining(tvb, offset) < len) {
    expert_add_info(pinfo, tree, &ei_ieee80211_ff_anqp_info_length);
    return 4 + len;
  }
  switch (id)
  {
  case ANQP_INFO_ANQP_QUERY_LIST:
    dissect_anqp_query_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_ANQP_CAPAB_LIST:
    dissect_anqp_capab_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_VENUE_NAME_INFO:
    dissect_venue_name_info(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_NETWORK_AUTH_TYPE_INFO:
    dissect_network_auth_type(tree, tvb, offset, offset + len);
    break;
  case ANQP_INFO_ROAMING_CONSORTIUM_LIST:
    dissect_roaming_consortium_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_IP_ADDR_TYPE_AVAILABILITY_INFO:
    dissect_ip_addr_type_availability_info(tree, tvb, offset);
    break;
  case ANQP_INFO_NAI_REALM_LIST:
    dissect_nai_realm_list(tree, tvb, pinfo, offset, offset + len);
    break;
  case ANQP_INFO_3GPP_CELLULAR_NETWORK_INFO:
    dissect_3gpp_cellular_network_info(tree, tvb, pinfo, offset);
    break;
  case ANQP_INFO_DOMAIN_NAME_LIST:
    dissect_domain_name_list(tree, tvb, offset, offset + len);
    break;
  case ANQP_INFO_NEIGHBOR_REPORT:
    {
      tvbuff_t *report_tvb;
      static const uint8_t ids[] = { TAG_NEIGHBOR_REPORT };

      report_tvb = tvb_new_subset_length(tvb, offset, len);
      int report_offset = 0;
      /* Cf. IEEE 802.11-2020 9.4.5.19 Neighbor Report ANQP-element with
       * IEEE 802.11-2016 9.4.5.19 and IEEE 802.11-2012 8.4.4.19.
       * The line "The Element ID and the Length fields of the Neighbor Report
       * element... are not included" was removed as it made it impossible
       * to include more than one Neighbor Element (using the overall ANQP
       * Info length could not distinguish between extra length due to a second
       * Neighbor Report and extra length due to optional subelements.)
       * The headerless version was apparently never deployed so use the new
       * specification (though we could test to see if the next octet is
       * TAG_NEIGHBOR_REPORT and dissect with the older method if not.)
       */
      while (tvb_reported_length_remaining(report_tvb, report_offset)) {
        report_offset += add_tagged_field(pinfo, tree, report_tvb, report_offset, 0,
          ids, G_N_ELEMENTS(ids), NULL);
      }
    }
    break;
  case ANQP_INFO_ANQP_VENDOR_SPECIFIC_LIST:
    proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
    offset += 3;
    vendor_tvb = tvb_new_subset_length(tvb, offset, len - 3);

    anqp_info.request = request;
    anqp_info.idx = idx;
    if (!dissector_try_uint_with_data(vendor_specific_anqp_info_table, oui, vendor_tvb, pinfo, tree, false, &anqp_info))
    {
      proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info, tvb, offset, len, ENC_NA);
    }
    break;
  case ANQP_INFO_VENUE_URL:
    dissect_anqp_venue_url(tree, tvb, offset, offset + len);
    break;
  case ANQP_INFO_ADVICE_OF_CHARGE:
    dissect_hs20_anqp_advice_of_charge(tree, pinfo, tvb, offset, offset + len);
    break;
  case ANQP_INFO_NETWORK_AUTH_TYPE_TIMESTAMP:
    dissect_anqp_network_auth_type_timestamp(tree, tvb, offset, offset + len);
    break;
  default:
    proto_tree_add_item(tree, hf_ieee80211_ff_anqp_info,
                        tvb, offset, len, ENC_NA);
    break;
  }

  return 4 + len;
}

static void
dissect_anqp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, bool request)
{
  int idx = 0;

  proto_item_append_text(tree, ": ANQP ");
  proto_item_append_text(tree, request ? "Request" : "Response");
  if (tvb_reported_length_remaining(tvb, offset) < 4) {
    expert_add_info(pinfo, tree, &ei_ieee80211_not_enough_room_for_anqp_header);
    return;
  }
  col_append_fstr(pinfo->cinfo, COL_INFO, ", ANQP %s",
                  request ? "Req" : "Resp");
  while (tvb_reported_length_remaining(tvb, offset) > 0) {
    offset += dissect_anqp_info(tree, tvb, pinfo, offset, request, idx);
    idx += 1;
  }
}

static unsigned
dissect_gas_initial_request(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                            unsigned type, unsigned subtype)
{
  unsigned    req_len;
  int         start = offset;
  proto_item *item;
  proto_tree *query;

  /* Query Request Length (2 octets) */
  req_len = tvb_get_letohs(tvb, offset);

  query = proto_tree_add_subtree(tree, tvb, offset, 2 + req_len, ett_gas_query, &item, "Query Request");
  if (tvb_reported_length_remaining(tvb, offset) < 2 + req_len) {
    expert_add_info(pinfo, item, &ei_ieee80211_ff_query_request_length);
    return tvb_reported_length_remaining(tvb, offset);
  }

  proto_tree_add_item(query, hf_ieee80211_ff_query_request_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  /*
   * Query Request (GAS query; formatted per protocol specified in the
   * Advertisement Protocol IE)
   */
  switch (type) {
  case ADV_PROTO_ID_ANQP:
    dissect_anqp(query, tvb, pinfo, offset, true);
    break;
  case ADV_PROTO_ID_VS:
    if (subtype == ((DPP_CONFIGURATION_PROTOCOL << 8) | WFA_SUBTYPE_DPP)) {
       col_append_fstr(pinfo->cinfo, COL_INFO, ", DPP - %s",
                       val_to_str(pinfo->pool, subtype >> 8, dpp_subtype_vals, "Unknown (%u)"));
      dissect_wifi_dpp_config_proto(pinfo, query, tvb, offset);
    }
    break;
  default:
    proto_tree_add_item(query, hf_ieee80211_ff_query_request,
                        tvb, offset, req_len, ENC_NA);
  }
  offset += req_len;

  return offset - start;
}

static unsigned
dissect_gas_initial_response(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                             unsigned type, unsigned subtype)
{
  unsigned    resp_len;
  int         start = offset;
  proto_item *item;
  proto_tree *query;

  /* Query Response Length (2 octets) */
  resp_len = tvb_get_letohs(tvb, offset);

  query = proto_tree_add_subtree(tree, tvb, offset, 2 + resp_len,
                             ett_gas_query, &item, "Query Response");
  if (tvb_reported_length_remaining(tvb, offset) < 2 + resp_len) {
    expert_add_info(pinfo, item, &ei_ieee80211_ff_query_response_length);
    return tvb_reported_length_remaining(tvb, offset);
  }

  proto_tree_add_item(query, hf_ieee80211_ff_query_response_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  /* Query Response (optional) */
  if (resp_len) {
    switch (type) {
    case ADV_PROTO_ID_ANQP:
      dissect_anqp(query, tvb, pinfo, offset, false);
      break;
    case ADV_PROTO_ID_VS:
      if (subtype == ((DPP_CONFIGURATION_PROTOCOL << 8) | WFA_SUBTYPE_DPP)) {
         col_append_fstr(pinfo->cinfo, COL_INFO, ", DPP - %s",
                         val_to_str(pinfo->pool, subtype >> 8, dpp_subtype_vals, "Unknown (%u)"));
        dissect_wifi_dpp_config_proto(pinfo, query, tvb, offset);
      }
      break;
    default:
      proto_tree_add_item(query, hf_ieee80211_ff_query_response,
                          tvb, offset, resp_len, ENC_NA);
    }
    offset += resp_len;
  }

  return offset - start;
}

static reassembly_table gas_reassembly_table;

static int ett_gas_resp_fragment;
static int ett_gas_resp_fragments;

static int hf_ieee80211_gas_resp_fragments;
static int hf_ieee80211_gas_resp_fragment;
static int hf_ieee80211_gas_resp_fragment_overlap;
static int hf_ieee80211_gas_resp_fragment_overlap_conflict;
static int hf_ieee80211_gas_resp_fragment_multiple_tails;
static int hf_ieee80211_gas_resp_fragment_too_long_fragment;
static int hf_ieee80211_gas_resp_fragment_error;
static int hf_ieee80211_gas_resp_fragment_count;
static int hf_ieee80211_gas_resp_reassembled_in;
static int hf_ieee80211_gas_resp_reassembled_length;

static const fragment_items gas_resp_frag_items = {
  &ett_gas_resp_fragment,
  &ett_gas_resp_fragments,
  &hf_ieee80211_gas_resp_fragments,
  &hf_ieee80211_gas_resp_fragment,
  &hf_ieee80211_gas_resp_fragment_overlap,
  &hf_ieee80211_gas_resp_fragment_overlap_conflict,
  &hf_ieee80211_gas_resp_fragment_multiple_tails,
  &hf_ieee80211_gas_resp_fragment_too_long_fragment,
  &hf_ieee80211_gas_resp_fragment_error,
  &hf_ieee80211_gas_resp_fragment_count,
  &hf_ieee80211_gas_resp_reassembled_in,
  &hf_ieee80211_gas_resp_reassembled_length,
  /* Reassembled data field */
  NULL,
  "GAS Response fragments"
};

static unsigned
dissect_gas_comeback_response(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                              unsigned type, unsigned subtype _U_, uint8_t frag, bool more,
                              uint8_t dialog_token)
{
  unsigned    resp_len;
  int         start = offset;
  proto_item *item;
  proto_tree *query;

  /* Query Response Length (2 octets) */
  resp_len = tvb_get_letohs(tvb, offset);

  query = proto_tree_add_subtree(tree, tvb, offset, 2 + resp_len,
                             ett_gas_query, &item, "Query Response");
  if (tvb_reported_length_remaining(tvb, offset) < 2 + resp_len) {
    expert_add_info(pinfo, item, &ei_ieee80211_ff_query_response_length);
    return tvb_reported_length_remaining(tvb, offset);
  }

  proto_tree_add_item(query, hf_ieee80211_ff_query_response_length,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  /* Query Response (optional) */
  if (resp_len) {
    if (type == ADV_PROTO_ID_ANQP && (frag == 0) && !more)
      dissect_anqp(query, tvb, pinfo, offset, false);
    else {
      fragment_head *frag_msg;
      bool save_fragmented;
      tvbuff_t *new_tvb;

      save_fragmented = pinfo->fragmented;
      pinfo->fragmented = true;
      frag_msg = fragment_add_seq_check(&gas_reassembly_table, tvb, offset,
                                        pinfo, dialog_token, NULL,
                                        frag, resp_len, more);
      new_tvb = process_reassembled_data(tvb, offset, pinfo,
                                         "Reassembled GAS Query Response",
                                         frag_msg, &gas_resp_frag_items,
                                         NULL, tree);
      if (new_tvb) {
        switch (type) {
        case ADV_PROTO_ID_ANQP:
          dissect_anqp(query, new_tvb, pinfo, 0, false);
          break;
        case ADV_PROTO_ID_VS:
          if (subtype == ((DPP_CONFIGURATION_PROTOCOL << 8) |
                           WFA_SUBTYPE_DPP)) {
            col_append_fstr(pinfo->cinfo, COL_INFO, ", DPP - %s",
                            val_to_str(pinfo->pool, subtype >> 8, dpp_subtype_vals,
                                       "Unknown (%u)"));
            dissect_wifi_dpp_config_proto(pinfo, query, new_tvb, 0);
          } else {
            proto_tree_add_item(query, hf_ieee80211_ff_query_response,
                                new_tvb, 0,
                                tvb_reported_length_remaining(new_tvb, 0),
                                ENC_NA);
          }
          break;
        default:
          proto_tree_add_item(query, hf_ieee80211_ff_query_response,
                              new_tvb, 0,
                              tvb_reported_length_remaining(new_tvb, 0),
                              ENC_NA);
        }
      }

      /* The old tvb cannot be used anymore */
      ieee80211_tvb_invalid = true;

      pinfo->fragmented = save_fragmented;
    }
    offset += resp_len;
  }

  return offset - start;
}

/* ************************************************************************* */
/*              Dissect and add fixed mgmt fields to protocol tree           */
/* ************************************************************************* */

static uint64_t last_timestamp;

static unsigned
add_ff_timestamp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  last_timestamp = tvb_get_letoh64(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_timestamp, tvb, offset, 8,
                      ENC_LITTLE_ENDIAN);
  return 8;
}

static unsigned
add_ff_beacon_interval(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_beacon_interval, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  col_append_fstr(pinfo->cinfo, COL_INFO, ", BI=%d",
                  tvb_get_letohs(tvb, offset));
  return 2;
}

static unsigned
add_ff_dmg_params(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset);

static unsigned
add_ff_cap_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_cap_info_fields[] = {
    &hf_ieee80211_ff_cf_ess,
    &hf_ieee80211_ff_cf_ibss,
    &hf_ieee80211_ff_cf_reserved1,
    &hf_ieee80211_ff_cf_reserved2,
    &hf_ieee80211_ff_cf_privacy,
    &hf_ieee80211_ff_cf_preamble,
    &hf_ieee80211_ff_cf_critical_update_flag,
    &hf_ieee80211_ff_cf_nontran_bss_critical_update_flag,
    &hf_ieee80211_ff_cf_spec_man,
    &hf_ieee80211_ff_cf_qos,
    &hf_ieee80211_ff_cf_short_slot_time,
    &hf_ieee80211_ff_cf_apsd,
    &hf_ieee80211_ff_cf_radio_measurement,
    &hf_ieee80211_ff_cf_epd,
    &hf_ieee80211_ff_cf_reserved5,
    &hf_ieee80211_ff_cf_reserved6,
    NULL
  };

  /* The capability information includes DMG parameters whenever it is transmitted by
     a DMG STA/AP (802.11ad-2012, 8.4.1.4) */

  bool isDMG = GPOINTER_TO_INT(p_get_proto_data(wmem_file_scope(), pinfo, proto_wlan, IS_DMG_KEY));

  if (isDMG) {
    proto_item *cap_item;
    proto_tree *cap_tree;
    cap_item = proto_tree_add_item(tree, hf_ieee80211_ff_capture, tvb, offset, 2,
                                   ENC_LITTLE_ENDIAN);
    cap_tree = proto_item_add_subtree(cap_item, ett_cap_tree);
    add_ff_dmg_params(cap_tree, tvb, pinfo, offset);
  } else {
    /*
     * We can only interpret the ESS and IBSS fields to be an AP if the
     * frame is a BEACON or PROBE_RESPONSE
     */
    uint32_t l_frame_type = GPOINTER_TO_UINT(p_get_proto_data(wmem_file_scope(),
                                                         pinfo, proto_wlan,
                                                         FRAME_TYPE_KEY));
    if (((tvb_get_letohs(tvb, offset) & 0x0003) == 0x1 &&
        (l_frame_type == MGT_BEACON || l_frame_type == MGT_PROBE_RESP)) ||
        l_frame_type == MGT_ASSOC_RESP || l_frame_type == MGT_REASSOC_RESP) {
      p_add_proto_data(wmem_file_scope(), pinfo, proto_wlan, IS_AP_KEY, GINT_TO_POINTER(true));
    }

    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                      hf_ieee80211_ff_capture,
                                      ett_cap_tree, ieee80211_cap_info_fields,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  return 2;
}

static unsigned
add_ff_auth_alg(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_auth_alg, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_auth_trans_seq(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_auth_seq, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_current_ap_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_current_ap, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static unsigned
add_ff_listen_ival(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_listen_ival, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_reason_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_reason, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_assoc_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_assoc_id, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_status_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_status_code, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_category_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_category_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_trigger(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  uint8_t trigger = tvb_get_uint8(tvb, offset);
  col_append_fstr(pinfo->cinfo, COL_INFO, ", Trigger=%d (%s)", trigger,
                  val_to_str_const(trigger, ftm_trigger_vals, "Unknown"));
  proto_tree_add_item(tree, hf_ieee80211_ff_trigger, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_ftm_tod(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ftm_tod, tvb, offset, 6,
                      ENC_LITTLE_ENDIAN);
  return 6;
}

static unsigned
add_ff_ftm_toa(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ftm_toa, tvb, offset, 6,
                      ENC_LITTLE_ENDIAN);
  return 6;
}

static unsigned
add_ff_ftm_tod_err(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ftm_tod_err, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_ftm_toa_err(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ftm_toa_err, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_ftm_tod_err1(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const fields[] = {
    &hf_ieee80211_ff_ftm_max_tod_error_exponent,
    &hf_ieee80211_ff_ftm_tod_err_reserved,
    &hf_ieee80211_ff_ftm_tod_not_continuous,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_ftm_tod_err1,
                                    ett_ff_ftm_tod_err1, fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static unsigned
add_ff_ftm_toa_err1(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const fields[] = {
    &hf_ieee80211_ff_ftm_max_toa_error_exponent,
    &hf_ieee80211_ff_ftm_toa_err_reserved,
    &hf_ieee80211_ff_ftm_invalid_measurement,
    &hf_ieee80211_ff_ftm_toa_type,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_ftm_toa_err1,
                                    ett_ff_ftm_toa_err1, fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static unsigned
add_ff_ftm_cfo_parameter(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ftm_cfo, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_ftm_r2i_ndp_tx_power(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ftm_r2i_ndp_tx_power, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_ftm_i2r_ndp_target_rssi(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ftm_i2r_ndp_target_rssi, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static int
dissect_ftm_params(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  int len = 0;
  proto_tree *ftm_param_tree = tree;
  static int * const ieee80211_ftm_params_fields1[] = {
    &hf_ieee80211_ff_ftm_param_status_indication,
    &hf_ieee80211_ff_ftm_param_value,
    &hf_ieee80211_ff_ftm_param_reserved1,
    &hf_ieee80211_ff_ftm_param_burst_exponent,
    &hf_ieee80211_ff_ftm_param_burst_duration,
    NULL};

  static int * const ieee80211_ftm_params_fields2[] = {
    &hf_ieee80211_ff_ftm_param_min_delta_ftm,
    &hf_ieee80211_ff_ftm_param_partial_tsf_timer,
    &hf_ieee80211_ff_ftm_param_partial_tsf_no_pref,
    &hf_ieee80211_ff_ftm_param_asap_capable,
    &hf_ieee80211_ff_ftm_param_asap,
    &hf_ieee80211_ff_ftm_param_ftm_per_burst,
    NULL};

  static int * const ieee80211_ftm_params_fields3[] = {
    &hf_ieee80211_ff_ftm_param_reserved2,
    &hf_ieee80211_ff_ftm_param_format_and_bw,
    &hf_ieee80211_ff_ftm_param_burst_period,
    NULL};

  len = tvb_captured_length(tvb);
  if (len != 9)
      return 0;

  proto_tree_add_bitmask_with_flags(ftm_param_tree, tvb, offset, hf_ieee80211_ff_ftm_param_delim1,
                                    ett_ff_ftm_param_delim1, ieee80211_ftm_params_fields1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  proto_tree_add_bitmask_with_flags(ftm_param_tree, tvb, offset, hf_ieee80211_ff_ftm_param_delim2,
                                    ett_ff_ftm_param_delim2, ieee80211_ftm_params_fields2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 4;
  proto_tree_add_bitmask_with_flags(ftm_param_tree, tvb, offset, hf_ieee80211_ff_ftm_param_delim3,
                                    ett_ff_ftm_param_delim3, ieee80211_ftm_params_fields3,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 3;

  return offset;
}

static unsigned
add_ff_dialog_token(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  uint8_t value = tvb_get_uint8(tvb, offset);
  col_append_fstr(pinfo->cinfo, COL_INFO, ", Dialog Token=%d", value);
  proto_tree_add_item(tree, hf_ieee80211_ff_dialog_token, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_followup_dialog_token(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_followup_dialog_token, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_wme_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_wme_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_wme_status_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_wme_status_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_qos_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_qos_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_block_ack_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ba_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_block_ack_param(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_block_ack_params,
                         ett_ff_ba_param_tree,
                         ieee80211_ff_block_ack_params_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_block_ack_timeout(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_block_ack_timeout, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_block_ack_ssc(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_block_ack_ssc,
                         ett_ff_ba_ssc_tree, ieee80211_ff_block_ack_ssc_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_qos_ts_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_tsinfo,
                         ett_tsinfo_tree, ieee80211_tsinfo_fields,
                         ENC_LITTLE_ENDIAN);
  return 3;
}

static unsigned
add_ff_mesh_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_multihop_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_multihop_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_mesh_control(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  int    start = offset;
  uint8_t flags;

  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_flags, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  flags = tvb_get_uint8(tvb, offset);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_ttl, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ff_mesh_sequence, tvb, offset, 4,
                      ENC_LITTLE_ENDIAN);
  offset += 4;

  switch (flags & 0x03) {
  case 1:
    proto_tree_add_item(tree, hf_ieee80211_ff_mesh_addr4, tvb, offset, 6,
                        ENC_NA);
    offset += 6;
    break;
  case 2:
    proto_tree_add_item(tree, hf_ieee80211_ff_mesh_addr5, tvb, offset, 6,
                        ENC_NA);
    offset += 6;
    proto_tree_add_item(tree, hf_ieee80211_ff_mesh_addr6, tvb, offset, 6,
                        ENC_NA);
    offset += 6;
    break;
  case 3:
    proto_item_append_text(tree, " Unknown Address Extension Mode");
    break;
  default:
    /* no default action */
    break;
  }

  return offset - start;
}

static unsigned
add_ff_selfprot_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_selfprot_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_dls_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dls_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_dst_mac_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dst_mac_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static unsigned
add_ff_src_mac_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_src_mac_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static unsigned
add_ff_req_ap_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_req_ap_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static unsigned
add_ff_res_ap_addr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_res_ap_addr, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static unsigned
add_ff_check_beacon(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_check_beacon, tvb, offset, 1,
                      ENC_NA);
  return 1;
}

static unsigned
add_ff_tod(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_tod, tvb, offset, 4,
                      ENC_LITTLE_ENDIAN);
  return 4;
}

static unsigned
add_ff_toa(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_toa, tvb, offset, 4,
                      ENC_LITTLE_ENDIAN);
  return 4;
}

static unsigned
add_ff_max_tod_err(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_tod_err, tvb, offset, 1,
                      ENC_NA);
  return 1;
}

static unsigned
add_ff_max_toa_err(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_toa_err, tvb, offset, 1,
                      ENC_NA);
  return 1;
}

static unsigned
add_ff_dls_timeout(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dls_timeout, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_delba_param_set(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_delba_param,
                         ett_ff_ba_param_tree, ieee80211_ff_delba_param_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_max_reg_pwr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_reg_pwr, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_measurement_pilot_int(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_measurement_pilot_int, tvb, offset,
                      1, ENC_NA);
  return 1;
}

static unsigned
add_ff_country_str(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_country_str, tvb, offset, 2,
                      ENC_ASCII);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_country_info_env,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  return 3;
}

static unsigned
add_ff_max_tx_pwr(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_max_tx_pwr, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_tx_pwr_used(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_tx_pwr_used, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_transceiver_noise_floor(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_transceiver_noise_floor, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_channel_width(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_channel_width, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

/* QoS Info:  802.11-2012 8.4.1.17 */
static unsigned
add_ff_qos_info_ap(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* From AP so decode as AP: Figure 8-51-QoS Info field when sent by a AP */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_qos_info_ap,
                                    ett_ff_qos_info, ieee80211_ff_qos_info_ap_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}


/* QoS Info:  802.11-2012 8.4.1.17 */
static unsigned
add_ff_qos_info_sta(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* To AP so decode as STA: Figure 8-52-QoS Info field when set by a non-AP STA */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_qos_info_sta,
                                    ett_ff_qos_info, ieee80211_ff_qos_info_sta_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static unsigned
add_ff_sm_pwr_cntrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_sm_pwr_save,
                         ett_ff_sm_pwr_save, ieee80211_ff_sw_pwr_save_fields,
                         ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_pco_phase_cntrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_pco_phase_cntrl, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_psmp_param_set(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_psmp_param_set,
                         ett_ff_psmp_param_set,
                         ieee80211_ff_psmp_param_set_fields,
                         ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_mimo_cntrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_item *mimo_item;
  proto_tree *mimo_tree;
  static int * const ieee80211_mimo_fields[] = {
    &hf_ieee80211_ff_mimo_cntrl_nc_index,
    &hf_ieee80211_ff_mimo_cntrl_nr_index,
    &hf_ieee80211_ff_mimo_cntrl_channel_width,
    &hf_ieee80211_ff_mimo_cntrl_grouping,
    &hf_ieee80211_ff_mimo_cntrl_coefficient_size,
    &hf_ieee80211_ff_mimo_cntrl_codebook_info,
    &hf_ieee80211_ff_mimo_cntrl_remaining_matrix_segment,
    &hf_ieee80211_ff_mimo_cntrl_reserved,
    NULL
  };

  mimo_item = proto_tree_add_item(tree, hf_ieee80211_ff_mimo_cntrl, tvb,
                                  offset, 6, ENC_NA);
  mimo_tree = proto_item_add_subtree(mimo_item, ett_ff_mimo_cntrl);

  proto_tree_add_bitmask_list(mimo_tree, tvb, offset, 2, ieee80211_mimo_fields, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(mimo_tree, hf_ieee80211_ff_mimo_cntrl_sounding_timestamp,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);

  return 6;
}

static unsigned
add_ff_ant_selection(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_ant_selection,
                         ett_ff_ant_sel, ieee80211_ff_ant_selection_fields,
                         ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_extended_channel_switch_announcement(proto_tree *tree, tvbuff_t *tvb,
                                            packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset,
                         hf_ieee80211_ff_ext_channel_switch_announcement,
                         ett_ff_chan_switch_announce,
                         ieee80211_ff_ext_channel_switch_announcement_fields,
                         ENC_LITTLE_ENDIAN);
  return 4;
}

static unsigned
add_ff_ht_information(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_ht_info,
                         ett_ff_ht_info, ieee80211_ff_ht_info_fields,
                         ENC_LITTLE_ENDIAN);
  return 1;
}
static unsigned
add_ff_lmr_report(proto_tree *tree, tvbuff_t *tvb,
                  packet_info *pinfo _U_, int offset)
{
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_tod(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_toa(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_tod_err1(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_toa_err1(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_cfo_parameter(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_r2i_ndp_tx_power(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_i2r_ndp_target_rssi(tree, tvb, pinfo, offset);
  /* Secure LTF parameters (optional) */
  /* AOA feedback (optional) */
  return offset;
}

static unsigned
add_ff_ftm_request(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  return add_ff_trigger(tree, tvb, pinfo, offset);
}

static unsigned
add_ff_ftm(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  unsigned start = offset;
  uint8_t dialog_token = tvb_get_uint8(tvb, offset);
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  if (dialog_token == 0) {
    col_append_str(pinfo->cinfo, COL_INFO, " (Termination)");
  }
  offset += add_ff_followup_dialog_token(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_tod(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_toa(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_tod_err(tree, tvb, pinfo, offset);
  offset += add_ff_ftm_toa_err(tree, tvb, pinfo, offset);
  return offset - start;
}

static unsigned
add_ff_ht_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ht_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_dmg_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dmg_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_dmg_pwr_mgmt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_dmg_pwr_mgmt, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_psmp_sta_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_item *psmp_item;
  proto_tree *psmp_tree;

  psmp_item = proto_tree_add_item(tree, hf_ieee80211_ff_psmp_sta_info, tvb,
                                  offset, 8, ENC_LITTLE_ENDIAN);
  psmp_tree = proto_item_add_subtree(psmp_item, ett_ff_psmp_sta_info);

  proto_tree_add_item(psmp_item, hf_ieee80211_ff_psmp_sta_info_type, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);

  switch (tvb_get_letohl(tvb, offset) & PSMP_STA_INFO_FLAG_TYPE) {
  case PSMP_STA_INFO_BROADCAST:
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_dtt_start_offset, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_dtt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    /* Missing 64 bit bitmask... */
    proto_tree_add_uint64(psmp_tree,
                          hf_ieee80211_ff_psmp_sta_info_reserved_large,
                          tvb, offset, 8,
                          (tvb_get_letoh64(tvb, offset) &
                           UINT64_C(0xFFFFFFFFFFE00000)) >> 21);
    break;
  case PSMP_STA_INFO_MULTICAST:
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_dtt_start_offset, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_dtt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    /* Missing 64 bit bitmask... */
    proto_tree_add_uint64(psmp_tree,
                          hf_ieee80211_ff_psmp_sta_info_psmp_multicast_id,
                          tvb, offset, 6,
                          (tvb_get_letoh64(tvb, offset) &
                           UINT64_C(0xFFFFFFFFFFE00000)) >> 21);
    break;
  case PSMP_STA_INFO_INDIVIDUALLY_ADDRESSED:
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_dtt_start_offset, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_dtt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_sta_id, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    offset += 2;

    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_utt_start_offset,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree, hf_ieee80211_ff_psmp_sta_info_utt_duration,
                        tvb, offset, 4, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(psmp_tree,
                        hf_ieee80211_ff_psmp_sta_info_reserved_small, tvb,
                        offset, 4, ENC_LITTLE_ENDIAN);
    break;
  }

  return 8;
}

static unsigned
add_ff_schedule_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_schedule_info_fields1[] = {
    &hf_ieee80211_sched_info_agg,
    NULL
  };
  static int * const ieee80211_schedule_info_fields2[] = {
    &hf_ieee80211_sched_info_agg,
    &hf_ieee80211_sched_info_tsid,
    &hf_ieee80211_sched_info_dir,
    NULL
  };

  if (tvb_get_letohs(tvb, offset) & 0x0001) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_sched_info,
                                    ett_sched_tree, ieee80211_schedule_info_fields2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  } else {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_sched_info,
                                    ett_sched_tree, ieee80211_schedule_info_fields1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }

  return 2;
}

static unsigned
add_ff_pa_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_public_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_ppa_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_protected_public_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_ft_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_ft_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_sta_address(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_sta_address, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static unsigned
add_ff_target_ap_address(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_target_ap_address, tvb, offset, 6,
                      ENC_NA);
  return 6;
}

static unsigned
add_ff_gas_comeback_delay(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_gas_comeback_delay, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_gas_fragment_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_gas_fragment_id, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_ff_more_gas_fragments, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_sa_query_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_sa_query_action_code, tvb, offset,
                      1, ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_transaction_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_transaction_id, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_tdls_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  code = tvb_get_uint8(tvb, offset);
  col_set_str(pinfo->cinfo, COL_INFO,
              val_to_str_ext_const(code, &tdls_action_codes_ext,
                                   "Unknown TDLS Action"));
  proto_tree_add_item(tree, hf_ieee80211_ff_tdls_action_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_target_channel(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_target_channel, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_operating_class(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_operating_class, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_channel(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_channel, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_wnm_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;

  code = tvb_get_uint8(tvb, offset);
  col_set_str(pinfo->cinfo, COL_INFO,
              val_to_str_ext_const(code, &wnm_action_codes_ext, "Unknown WNM Action"));
  proto_tree_add_item(tree, hf_ieee80211_ff_wnm_action_code, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_unprotected_wnm_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;

  code = tvb_get_uint8(tvb, offset);
  col_set_str(pinfo->cinfo, COL_INFO,
              val_to_str_ext_const(code, &unprotected_wnm_action_codes_ext, "Unknown Unprotected WNM Action"));
  proto_tree_add_item(tree, hf_ieee80211_ff_unprotected_wnm_action_code, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_unprotected_dmg_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_unprotected_dmg_action_code, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_key_data_length(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_key_data_length, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_wnm_notification_type(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_wnm_notification_type,
                      tvb, offset, 1, ENC_NA);
  return 1;
}

/* Action frame: Radio Measurement actions */
static unsigned
add_ff_rm_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_action_code, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_rm_dialog_token(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_dialog_token, tvb, offset, 1, ENC_NA);
  return 1;
}

/* Radio Measurement Request frame, Action fields */
static unsigned
add_ff_rm_repetitions(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* Note: 65535 means repeated until cancelled or superseded */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_repetitions, tvb, offset, 2,
                      ENC_BIG_ENDIAN);
  return 2;
}

/* Link Measurement Request frame, Action fields*/
static unsigned
add_ff_rm_tx_power(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* In dBm, see 8.4.1.20 */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_tx_power, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_rm_max_tx_power(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* In dBm, see 8.4.1.19 */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_max_tx_power, tvb, offset, 1, ENC_NA);
  return 1;
}

/* Link Measurement Report frame, Action fields */
static unsigned
add_ff_rm_tpc_report(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree *tpc_tree;
  proto_item *tpc_item;

  /* 8.4.2.19 TPC Report element */
  /* XXX - The TPC Report element is exactly the same as that dissected
   * by ieee80211_tag_tpc_report(), so some of these fixed fields duplicate
   * tagged fields.
   */
  tpc_item = proto_tree_add_item(tree, hf_ieee80211_ff_tpc, tvb, offset, 4, ENC_NA);
  tpc_tree = proto_item_add_subtree(tpc_item, ett_tpc);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_element_id, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_length, tvb, offset + 1, 1, ENC_NA);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_tx_power, tvb, offset + 2, 1, ENC_NA);
  proto_tree_add_item(tpc_tree, hf_ieee80211_ff_tpc_link_margin, tvb, offset + 3, 1, ENC_NA);
  return 4;
}

static unsigned
add_ff_rm_rx_antenna_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.42: 0 means unknown, 1-254 is valid, 255 means multiple antennas */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_rx_antenna_id, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_rm_tx_antenna_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.42: 0 means unknown, 1-254 is valid, 255 means multiple antennas */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_tx_antenna_id, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_rm_rcpi(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.40: RCPI as specified for certain PHYs */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_rcpi, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_rm_rsni(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  /* 8.4.2.43: RSNI in steps of 0.5 dB, calculated as:
   * RSNI = (10 * log10((RCPI_{power} - ANPI_{power}) / ANPI_{power}) + 20)*2 */
  proto_tree_add_item(tree, hf_ieee80211_ff_rm_rsni, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_bss_transition_status_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_status_code, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_bss_termination_delay(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_bss_termination_delay, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_fils_discovery(proto_tree *tree, tvbuff_t *tvb,
                                            packet_info *pinfo _U_, int offset)
{

  uint16_t fc, ssid_length;
  static int * const ieee80211_ff_fils_discovery_frame_control[] = {
    &hf_ieee80211_ff_fils_discovery_frame_control_ssid_length,
    &hf_ieee80211_ff_fils_discovery_frame_control_capability,
    &hf_ieee80211_ff_fils_discovery_frame_control_short_ssid,
    &hf_ieee80211_ff_fils_discovery_frame_control_ap_csn,
    &hf_ieee80211_ff_fils_discovery_frame_control_ano,
    &hf_ieee80211_ff_fils_discovery_frame_control_channel_center_frequency,
    &hf_ieee80211_ff_fils_discovery_frame_control_primary_channel,
    &hf_ieee80211_ff_fils_discovery_frame_control_rsn_info,
    &hf_ieee80211_ff_fils_discovery_frame_control_length,
    &hf_ieee80211_ff_fils_discovery_frame_control_md,
    &hf_ieee80211_ff_fils_discovery_frame_control_reserved,
    NULL
  };

  proto_tree_add_bitmask(tree, tvb, offset,
                         hf_ieee80211_ff_fils_discovery_frame_control,
                         ett_ff_fils_discovery_frame_control,
                         ieee80211_ff_fils_discovery_frame_control,
                         ENC_LITTLE_ENDIAN);
  fc = tvb_get_letohs(tvb, offset);
  offset += 2;

  offset += add_ff_timestamp(tree, tvb, pinfo, offset);

  offset += add_ff_beacon_interval(tree, tvb, pinfo, offset);

  if(fc & PA_FILS_FC_SHORT_SSID) {
     /* Always 4 bytes for Short SSID */
     /* TODO add check of SSID Length */
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_short_ssid, tvb, offset, 4, ENC_LITTLE_ENDIAN);
    offset += 4;
  } else {
    ssid_length = (fc & PA_FILS_FC_SSID_LENGTH) + 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_ssid, tvb, offset, ssid_length, ENC_ASCII);
    offset += ssid_length;
  }

  if(fc & PA_FILS_FC_LENGTH){
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_length, tvb, offset, 1, ENC_NA);
    offset += 1;
  }

  if(fc & PA_FILS_FC_CAPABILITY) {
    proto_tree *fdc_tree;
    proto_item *fdc_item;
    uint32_t fdc;
    fdc_item = proto_tree_add_item_ret_uint(tree, hf_ieee80211_ff_fils_discovery_capability, tvb, offset, 2, ENC_LITTLE_ENDIAN, &fdc);
    fdc_tree = proto_item_add_subtree(fdc_item, ett_ff_fils_discovery_capability);
    proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_ess, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_privacy, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_bss_operating_channel_width, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_max_number_of_spatial_streams, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_reserved, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_multiple_bssid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_phy_index, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    switch((fdc & 0x1C00) >> 10){
      case 0:
        proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_dsss, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      break;
      case 1:
        proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_ofdm, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      break;
      case 2:
      case 3:
        proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_ht_vht_tvht, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      break;
      case 4:
        proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate_he, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      break;
      default:
        proto_tree_add_item(fdc_tree, hf_ieee80211_ff_fils_discovery_capability_fils_minimum_rate, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      break;
    }
    offset += 2;
  }

  if(fc & PA_FILS_FC_PC) {
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_operating_class, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_primary_channel, tvb, offset, 1, ENC_NA);
    offset += 1;
  }

  if(fc & PA_FILS_FC_AP_CSN) {
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_ap_csn, tvb, offset, 1, ENC_NA);
    offset += 1;
  }

  if(fc & PA_FILS_FC_ANO) {
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_ano, tvb, offset, 1, ENC_NA);
    offset += 1;
  }

  if(fc & PA_FILS_FC_RSN_INFO) {
    /*TODO Dissect RSN info */
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_rsn_info, tvb, offset, 5, ENC_NA);
    offset += 5;
  }

  if(fc & PA_FILS_FC_CCFS1){
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_ccfs1, tvb, offset, 1, ENC_NA);
    offset += 1;
  }

  if(fc & PA_FILS_FC_MD) {
    /*TODO Dissect Mobility Domain */
    proto_tree_add_item(tree, hf_ieee80211_ff_fils_discovery_md, tvb, offset, 3, ENC_NA);
    offset += 3;
  }

  return offset;
}


static unsigned
add_ff_action_spectrum_mgmt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
    switch (tvb_get_uint8(tvb, offset + 1)) {
    case SM_ACTION_MEASUREMENT_REQUEST:
    case SM_ACTION_MEASUREMENT_REPORT:
    case SM_ACTION_TPC_REQUEST:
    case SM_ACTION_TPC_REPORT:
      add_ff_category_code(tree, tvb, pinfo, offset);
      add_ff_action_code(tree, tvb, pinfo, offset + 1);
      add_ff_dialog_token(tree, tvb, pinfo, offset + 2);
      return 3;
    case SM_ACTION_CHAN_SWITCH_ANNC:
    case SM_ACTION_EXT_CHAN_SWITCH_ANNC:
      add_ff_category_code(tree, tvb, pinfo, offset);
      add_ff_action_code(tree, tvb, pinfo, offset + 1);
      return 2;
    default:
      add_ff_category_code(tree, tvb, pinfo, offset);
      add_ff_action_code(tree, tvb, pinfo, offset + 1);
      return 2;
    }
}

static unsigned
add_ff_action_qos(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  switch (tvb_get_uint8(tvb, offset + 1)) {
  case QOS_ACTION_ADDTS_REQUEST:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    add_ff_dialog_token(tree, tvb, pinfo, offset + 2);
    return 3;
  case QOS_ACTION_ADDTS_RESPONSE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    add_ff_dialog_token(tree, tvb, pinfo, offset + 2);
    add_ff_status_code(tree, tvb, pinfo, offset + 3);
    return 5;
  case QOS_ACTION_DELTS:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    add_ff_qos_ts_info(tree, tvb, pinfo, offset + 2);
    add_ff_reason_code(tree, tvb, pinfo, offset + 5);
    return 7;
  case QOS_ACTION_SCHEDULE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    return 2;
  case QOS_ACTION_MAP_CONFIGURE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_qos_action_code(tree, tvb, pinfo, offset + 1);
    return 2;
  default:
    add_ff_category_code(tree, tvb, pinfo, offset);
    return 2;
  }
}

static unsigned
add_ff_action_dls(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  switch (tvb_get_uint8(tvb, offset + 1)) {
  case DLS_ACTION_REQUEST:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_dls_action_code(tree, tvb, pinfo, offset +  1);
    add_ff_dst_mac_addr(tree, tvb, pinfo, offset +  2);
    add_ff_src_mac_addr(tree, tvb, pinfo, offset +  8);
    add_ff_cap_info(tree, tvb, pinfo, offset + 14);
    add_ff_dls_timeout(tree, tvb, pinfo, offset + 16);
    return 18;
  case DLS_ACTION_RESPONSE:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_dls_action_code(tree, tvb, pinfo, offset +  1);
    add_ff_status_code(tree, tvb, pinfo, offset +  2);
    add_ff_dst_mac_addr(tree, tvb, pinfo, offset +  4);
    add_ff_src_mac_addr(tree, tvb, pinfo, offset + 10);
    if (!hf_ieee80211_ff_status_code) {
      add_ff_cap_info(tree, tvb, pinfo, offset + 16);
    }
    return 16;
  case DLS_ACTION_TEARDOWN:
    add_ff_category_code(tree, tvb, pinfo, offset);
    add_ff_dls_action_code(tree, tvb, pinfo, offset +  1);
    add_ff_dst_mac_addr(tree, tvb, pinfo, offset +  2);
    add_ff_src_mac_addr(tree, tvb, pinfo, offset +  8);
    add_ff_reason_code(tree, tvb, pinfo, offset + 14);
    return 16;
  default:
    add_ff_category_code(tree, tvb, pinfo, offset);
    return 2;
  }
}

static unsigned
add_ff_action_block_ack(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;

  switch (tvb_get_uint8(tvb, offset + 1)) {
  case BA_ADD_BLOCK_ACK_REQUEST:
  case BA_NDP_ADD_BLOCK_ACK_REQUEST:
  case BA_BAT_ADD_BLOCK_ACK_REQUEST:
    offset += add_ff_category_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_action_code(tree, tvb, pinfo, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_param(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_timeout(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_ssc(tree, tvb, pinfo, offset);
    break;
  case BA_ADD_BLOCK_ACK_RESPONSE:
  case BA_NDP_ADD_BLOCK_ACK_RESPONSE:
  case BA_BAT_ADD_BLOCK_ACK_RESPONSE:
    offset += add_ff_category_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_action_code(tree, tvb, pinfo, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_param(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_timeout(tree, tvb, pinfo, offset);
    break;
  case BA_DELETE_BLOCK_ACK:
  case BA_NDP_DELETE_BLOCK_ACK:
  case BA_BAT_DELETE_BLOCK_ACK:
    offset += add_ff_category_code(tree, tvb, pinfo, offset);
    offset += add_ff_block_ack_action_code(tree, tvb, pinfo, offset);
    offset += add_ff_delba_param_set(tree, tvb, pinfo, offset);
    offset += add_ff_reason_code(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

unsigned
add_ff_action_public_fields(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, uint8_t code)
{
  uint32_t oui;
  unsigned type;
  unsigned subtype;
  uint8_t  dialog_token;
  uint8_t  frag;
  bool more;
  tvbuff_t *vendor_tvb;
  int dissected;

  unsigned start = offset;

  switch (code) {
  case PA_EXT_CHANNEL_SWITCH_ANNOUNCEMENT:
    offset += add_ff_extended_channel_switch_announcement(tree, tvb, pinfo, offset);
    break;
  case PA_VENDOR_SPECIFIC:
    proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
    offset += 3;
    switch (oui) {
    case OUI_WFA:
      proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui_wfa_subtype, tvb, offset, 1, ENC_NA, &subtype);
      offset += 1;
      vendor_tvb = tvb_new_subset_remaining(tvb, offset);
      dissected = dissector_try_uint_with_data(wifi_alliance_public_action_table, subtype, vendor_tvb, pinfo, tree, false, NULL);
      offset += dissected;
      break;
    default:
      /* Don't know how to handle this vendor */
      break;
    }
    break;
  case PA_GAS_INITIAL_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += dissect_advertisement_protocol_common(pinfo, tree, tvb, offset,
                                             &type, &subtype);
    offset += dissect_gas_initial_request(tree, tvb, pinfo, offset, type,
                                          subtype);
    break;
  case PA_GAS_INITIAL_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    offset += add_ff_gas_comeback_delay(tree, tvb, pinfo, offset);
    offset += dissect_advertisement_protocol_common(pinfo, tree, tvb, offset,
                                             &type, &subtype);
    offset += dissect_gas_initial_response(tree, tvb, pinfo, offset, type,
                                           subtype);
    break;
  case PA_GAS_COMEBACK_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case PA_GAS_COMEBACK_RESPONSE:
    dialog_token = tvb_get_uint8(tvb, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    frag = tvb_get_uint8(tvb, offset) & 0x7f;
    more = (tvb_get_uint8(tvb, offset) & 0x80) != 0;
    offset += add_ff_gas_fragment_id(tree, tvb, pinfo, offset);
    offset += add_ff_gas_comeback_delay(tree, tvb, pinfo, offset);
    offset += dissect_advertisement_protocol_common(pinfo, tree, tvb, offset,
                                             &type, &subtype);
    offset += dissect_gas_comeback_response(tree, tvb, pinfo, offset, type,
                                            subtype, frag, more, dialog_token);
    break;
  case PA_TDLS_DISCOVERY_RESPONSE:
    col_set_str(pinfo->cinfo, COL_PROTOCOL, "TDLS");
    col_set_str(pinfo->cinfo, COL_INFO, "TDLS Discovery Response");
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    break;
  case PA_QAB_REQUEST:
  case PA_QAB_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_req_ap_addr(tree, tvb, pinfo, offset);
    offset += add_ff_res_ap_addr(tree, tvb, pinfo, offset);
    break;
  case PA_FTM_REQUEST:
    col_set_str(pinfo->cinfo, COL_INFO, "FTM Request");
    offset += add_ff_ftm_request(tree, tvb, pinfo, offset);
    break;
  case PA_FTM:
    col_set_str(pinfo->cinfo, COL_INFO, "FTM");
    offset += add_ff_ftm(tree, tvb, pinfo, offset);
    break;
  case PA_FILS_DISCOVERY:
    col_set_str(pinfo->cinfo, COL_INFO, "FILS Discovery");
    offset = add_ff_fils_discovery(tree, tvb, pinfo, offset);
    break;
  case PA_LOCATION_MEASUREMENT_REPORT:
    col_set_str(pinfo->cinfo, COL_INFO, "Location Measurement Report");
    offset = add_ff_lmr_report(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_public(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  unsigned start = offset;
  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_pa_action_code(tree, tvb, pinfo, offset);
  offset += add_ff_action_public_fields(tree, tvb, pinfo, offset, code);
  return offset - start;
}

static unsigned
add_ff_action_protected_public(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  unsigned start = offset;
  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_ppa_action_code(tree, tvb, pinfo, offset);
  offset += add_ff_action_public_fields(tree, tvb, pinfo, offset, code);
  return offset - start;
}

static unsigned
add_ff_action_radio_measurement(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned  start = offset;
  uint8_t code;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code = tvb_get_uint8(tvb, offset);
  offset += add_ff_rm_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case RM_ACTION_RADIO_MEASUREMENT_REQUEST:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_rm_repetitions(tree, tvb, pinfo, offset);
    /* Followed by Measurement Request Elements */
    break;
  case RM_ACTION_RADIO_MEASUREMENT_REPORT:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    /* Followed by Measurement Report Elements */
    break;
  case RM_ACTION_LINK_MEASUREMENT_REQUEST:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_rm_tx_power(tree, tvb, pinfo, offset);
    offset += add_ff_rm_max_tx_power(tree, tvb, pinfo, offset);
    /* Followed by Optional Subelements */
    break;
  case RM_ACTION_LINK_MEASUREMENT_REPORT:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_rm_tpc_report(tree, tvb, pinfo, offset);
    offset += add_ff_rm_rx_antenna_id(tree, tvb, pinfo, offset);
    offset += add_ff_rm_tx_antenna_id(tree, tvb, pinfo, offset);
    offset += add_ff_rm_rcpi(tree, tvb, pinfo, offset);
    offset += add_ff_rm_rsni(tree, tvb, pinfo, offset);
    /* Followed by Optional Subelements */
    break;
  case RM_ACTION_NEIGHBOR_REPORT_REQUEST:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    /* Followed by Optional Subelements */
    break;
  case RM_ACTION_NEIGHBOR_REPORT_RESPONSE:
    offset += add_ff_rm_dialog_token(tree, tvb, pinfo, offset);
    /* Followed by Neighbor Report Elements */
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_fast_bss_transition(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned  start = offset;
  uint8_t code;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_ft_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case FT_ACTION_REQUEST:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    /* Followed by FT Request frame body (IEs) */
    break;
  case FT_ACTION_RESPONSE:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    /* Followed by FT Response frame body (IEs) */
    break;
  case FT_ACTION_CONFIRM:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    /* Followed by FT Confirm frame body (IEs) */
    break;
  case FT_ACTION_ACK:
    offset += add_ff_sta_address(tree, tvb, pinfo, offset);
    offset += add_ff_target_ap_address(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    /* Followed by FT Ack frame body (IEs) */
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_sa_query(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned  start = offset;
  uint8_t code;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_sa_query_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case SA_QUERY_REQUEST:
    offset += add_ff_transaction_id(tree, tvb, pinfo, offset);
    break;
  case SA_QUERY_RESPONSE:
    offset += add_ff_transaction_id(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_mesh(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned length;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_mesh_action(tree, tvb, pinfo, offset);
  /* The only fixed fields are the category and mesh action.  The rest are IEs.
   */
  length = 2;
  if (tvb_get_uint8(tvb, 1) == MESH_ACTION_TBTT_ADJ_RESPONSE) {
    /* ..except for the TBTT Adjustment Response, which has a status code field
     */
    length += add_ff_status_code(tree, tvb, pinfo, offset);
  }
  return length;
}

static unsigned
add_ff_action_multihop(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_multihop_action(tree, tvb, pinfo, offset);
  offset += add_ff_mesh_control(tree, tvb, pinfo, offset);
  return offset - start;
}

static unsigned
add_ff_action_self_protected(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
                             association_sanity_check_t *association_sanity_check)
{
  unsigned start = offset;
  uint8_t self_protected_action = tvb_get_uint8(tvb, start + 1);

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_selfprot_action(tree, tvb, pinfo, offset);

  switch (self_protected_action) {
  case SELFPROT_ACTION_MESH_PEERING_OPEN:
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    if (association_sanity_check)
      association_sanity_check->ampe_frame = self_protected_action;
    break;
  case SELFPROT_ACTION_MESH_PEERING_CONFIRM:
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    offset += add_ff_assoc_id(tree, tvb, pinfo, offset);
    if (association_sanity_check)
      association_sanity_check->ampe_frame = self_protected_action;
    break;
  case SELFPROT_ACTION_MESH_PEERING_CLOSE:
    if (association_sanity_check)
      association_sanity_check->ampe_frame = self_protected_action;
    break;
  }

  return offset - start;
}

static unsigned
add_ff_vht_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_vht_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_s1g_timestamp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_s1g_timestamp, tvb, offset,
                      4, ENC_LITTLE_ENDIAN);
  return 4;
}

static unsigned
add_ff_change_sequence(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_s1g_change_sequence, tvb, offset,
                      1, ENC_NA);
  return 1;
}

static unsigned
add_ff_next_tbtt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_s1g_next_tbtt, tvb, offset,
                      3, ENC_LITTLE_ENDIAN);
  return 3;
}

static unsigned
add_ff_compressed_ssid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_s1g_compressed_ssid, tvb, offset,
                      4, ENC_LITTLE_ENDIAN);
  return 4;
}

/* This should not be S1G specific because 802.11-2016 defines it as well. */
static unsigned
add_ff_access_network_options(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_s1g_access_network_options, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_s1g_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_s1g_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_prot_s1g_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_prot_s1g_action, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned get_group_element_len(unsigned group) {
  switch (group) {
    /* Diffie-Hellman groups */
    case 1:
      return 96;
    case 2:
    case 22:
      return 128;
    case 5:
      return 192;
    case 14:
    case 23:
    case 24:
      return 256;
    case 15:
      return 384;
    case 16:
      return 512;
    case 17:
      return 768;
    case 18:
      return 1024;
    /* ECC groups */
    case 19:
    case 28:
      return 64;
    case 20:
    case 29:
      return 96;
    case 21:
      return 132;
    case 25:
      return 48;
    case 26:
      return 56;
    case 30:
      return 128;
    default:
      return 0;
  }
}

static unsigned get_scalar_len(unsigned group) {
  switch (group) {
    /* Diffie-Hellman groups */
    case 1:
      return 96;
    case 2:
      return 128;
    case 5:
      return 192;
    case 14:
      return 256;
    case 15:
      return 384;
    case 16:
      return 512;
    case 17:
      return 768;
    case 18:
      return 1024;
    case 22:
      return 20;
    case 23:
      return 28;
    case 24:
      return 32;
    /* ECC groups */
    case 19:
    case 28:
      return 32;
    case 20:
    case 29:
      return 48;
    case 21:
      return 66;
    case 25:
      return 24;
    case 26:
      return 28;
    case 30:
      return 64;
    default:
      return 0;
  }
}

static unsigned
find_fixed_field_len(tvbuff_t *tvb, unsigned offset)
{
  unsigned start_offset = offset;
  unsigned len = tvb_reported_length(tvb);

  if (offset >= len) {
    return 0;
  }

  while (offset < len) {
    if (tvb_get_uint8(tvb, offset) == 0xFF) {
      /*
       * Check if we have a len followed by either ETAG_REJECTED_GROUPS
       * or ETAG_PASSWORD_IDENTIFIER or ETAG_ANTI_CLOGGING_TOKEN
       */
      /* The length of SAE Confirm or Scalar Fixed parameter >= 32 */
      if ((offset < len - 3) && (offset - start_offset >= 32)) {
        uint8_t etag_len = tvb_get_uint8(tvb, offset + 1);
        uint8_t check_byte = tvb_get_uint8(tvb, offset + 2);
        if (check_byte == ETAG_REJECTED_GROUPS ||
            check_byte == ETAG_PASSWORD_IDENTIFIER ||
            check_byte == ETAG_ANTI_CLOGGING_TOKEN ||
            check_byte == ETAG_MULTI_LINK ||
            check_byte == ETAG_AKM_SUITE_SELECTOR) {
          /* Add length check to avoid false detection */
          if (offset + etag_len + 2 <= len) {
            break;
          }
        }
      }
    }
    offset++;
  }

  return offset - start_offset;
}

static const value_string ff_sae_message_type_vals[] = {
  {1, "Commit" },
  {2, "Confirm" },
  {0, NULL }
};

/*
 * We have to deal with the issue that an anti-clogging token may be in this
 * thing.
 */
static unsigned
add_ff_auth_sae(proto_tree *tree, tvbuff_t *tvb,
                         packet_info *pinfo _U_, unsigned offset)
{
  unsigned alg, seq, status_code, len;
  alg = tvb_get_letohs(tvb, 0);

  if (alg != AUTH_ALG_SAE)
    return offset;

  seq = tvb_get_letohs(tvb, 2);
  status_code = tvb_get_letohs(tvb, 4);

  proto_tree_add_uint(tree, hf_ieee80211_ff_sae_message_type, tvb, 2, 2, seq);

  if (seq == 1)
  {
    uint16_t group;
    unsigned sc_len, elt_len;

    /*
     * Order is: Status code,
     *           Finite Cyclic Group,
     *           Anti-Clogging Token in some cases
     *           Send-Confirm in some cases
     *           Scalar in some cases
     *           FFE in some cases
     *           Confirm in some cases
     *           Challenge Text in some cases
     *           RSNE in some cases.
     *           MDE in some cases.
     *           Fast BSS TRansition ... in some cases.
     */

    /* 76: Authentication is rejected because an Anti-Clogging Token is required (cf ieee80211_status_code) */
      /* These are present if status code is 0, 76, 77 or 126 */
    if (status_code == 0 || status_code == 76 || status_code == 77 ||
        status_code == 126)
    {
      group = tvb_get_letohs(tvb, offset);
      proto_tree_add_item(tree, hf_ieee80211_ff_finite_cyclic_group, tvb,
                          offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;


      /*
       * Now, get the fixed field length remaining. It will be divided up into
       * Anti-Clogging token, Scalar, FFE and some IEs.
       */
      len = find_fixed_field_len(tvb, offset);
      sc_len = get_scalar_len(group);
      elt_len = get_group_element_len(group);

      /*
       * The first conditional captures the case where we have an error and
       * an anti-clogging token with Scalar Field and FFE.
       * The second handles the case where we have an error with only an
       * anti-clogging token.
       * The third conditional below is a way to avoid keeping state about
       * what was in a previous response!
       */
      if (((status_code == 76 || status_code == 126) &&
           ((len > (sc_len + elt_len)))) ||
           ((status_code == 76) && (len > 0) && (len < (sc_len + elt_len))) ||
          ((status_code == 0) && (len > (sc_len + elt_len)))) {
        unsigned anti_clogging_len;
        /*
         * Handle the anti-clogging field. There is an anti-clogging token
         * before the other two.
         */
        if (len > (sc_len + elt_len))
          anti_clogging_len = len - (sc_len + elt_len);
        else
          anti_clogging_len = len;

        proto_tree_add_item(tree, hf_ieee80211_ff_sae_anti_clogging_token, tvb,
                            offset, anti_clogging_len, ENC_NA);
        offset += anti_clogging_len;
      }

      if (sc_len == 0) {
        /* assume no anti-clogging token */
        if (!(len % 3))
        {
          sc_len = len / 3;
        }
        else
        {
          sc_len = len / 2;
        }
        elt_len = len - sc_len;
      }

      /* Only present if status = 0 or 126 */
      if (status_code == 0 || status_code == 126) {
        proto_tree_add_item(tree, hf_ieee80211_ff_scalar, tvb, offset,
                            sc_len, ENC_NA);
        offset += sc_len;

        proto_tree_add_item(tree, hf_ieee80211_ff_finite_field_element, tvb,
                            offset, elt_len, ENC_NA);
        offset += elt_len;
      }
    }
  }
  else if ((seq == 2) && (status_code == 0))
  {
    proto_tree_add_item(tree, hf_ieee80211_ff_send_confirm, tvb, 6, 2,
                        ENC_LITTLE_ENDIAN);
    offset += 2;

    /* Check if there are additional elements */
    len = find_fixed_field_len(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_ff_confirm, tvb, offset, len,
                        ENC_NA);
    offset += len;
  }

  return offset;
}

static unsigned
add_ff_auth_fils(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
                 unsigned offset)
{
  unsigned alg, seq, status_code;
  alg = tvb_get_letohs(tvb, 0);

  if ((alg != AUTH_ALG_FILS_SK_WITH_PFS) && (alg != AUTH_ALG_FILS_PK))
    return offset;

  seq = tvb_get_letohs(tvb, 2);
  status_code = tvb_get_letohs(tvb, 4);

  if ((seq == 1) || (seq == 2 && status_code == 0)) {
    unsigned group = tvb_get_letohs(tvb, 6);
    unsigned elt_len;
    proto_tree_add_item(tree, hf_ieee80211_ff_finite_cyclic_group, tvb, 6, 2,
                        ENC_LITTLE_ENDIAN);
    offset = 8;
    elt_len = get_group_element_len(group);

    proto_tree_add_item(tree, hf_ieee80211_ff_finite_field_element, tvb, offset,
                        elt_len, ENC_NA);

    offset += elt_len;
  }

  /* What about the other FILS case? */

  return offset;
}

/*
 * We handle different elements depending on whether the sequence number is
 * 1, 2 or 3.
 */
typedef struct ieee80211_pasn_data {
  unsigned pasn_seq;
  unsigned pasn_status_code;
} ieee80211_pasn_data_t;

static ieee80211_pasn_data_t*
create_pasn_data(packet_info *pinfo, unsigned seq, unsigned status)
{
  ieee80211_pasn_data_t *pasn_data = NULL;

  pasn_data = wmem_new(pinfo->pool, ieee80211_pasn_data_t);

  if(pasn_data) {
    p_add_proto_data(pinfo->pool, pinfo, proto_wlan, PASN_DATA_KEY, pasn_data);
    pasn_data->pasn_seq = seq;
    pasn_data->pasn_status_code = status;
  }

  return pasn_data;
}

static unsigned
add_ff_auth_pasn(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
                 unsigned offset)
{
  unsigned seq, status_code;
  ieee80211_pasn_data_t *pasn_data = NULL;

  seq = tvb_get_letohs(tvb, 2);
  status_code = tvb_get_letohs(tvb, 4);

  pasn_data = create_pasn_data(pinfo, seq, status_code);
  if (!pasn_data) {
    /* Leave it undissected if we cannot get memory. */
    return offset + tvb_captured_length_remaining(tvb, offset);
  }

  if (seq == 1) {
    /*
     * Contains RSN Info,
     * PASN field,
     * Wrapped Data may be present if the PASN element says so,
     * RSNXE may be present
     * Timeout Interval element may be present
     * Fragment element may be present if it was fragmented.
     */
    while (tvb_captured_length_remaining(tvb, offset)) {
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    }
  } else if (seq == 2) {
    /* This test might not be needed */
    if (status_code != 0) {
      offset += tvb_captured_length_remaining(tvb, offset);
      return offset;
    }
    /*
     * RSN element is present.
     * PASN element is present if status == 0.
     * Wrapped Data element present if the PASN element says so.
     * ...
     */
    while (tvb_captured_length_remaining(tvb, offset)) {
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    }
  } else if (seq == 3) {
    /*
     * Contains PASN element if status == 0
     * Contains Wrapped Data element if PASN element says so and status is 0.
     * Contains MC element
     * May contain fragment elements.
     */
    while (tvb_captured_length_remaining(tvb, offset)) {
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    }
  }

  offset += tvb_captured_length_remaining(tvb, offset);

  return offset;
}

/*
 * Handle an Auth Frame. We need to be able to call this from several places.
 *
 * We should also handle the different auth types more correctly.
 */
static int
dissect_auth_frame(proto_tree *tree, packet_info *pinfo, tvbuff_t *tvb)
{
  int offset = 0;
  uint16_t auth_algorithm = tvb_get_letohs(tvb, offset);

  add_ff_auth_alg(tree, tvb, pinfo, offset);
  offset += 2;

  add_ff_auth_trans_seq(tree, tvb, pinfo, offset);
  offset += 2;

  add_ff_status_code(tree, tvb, pinfo, offset);
  offset += 2;

  switch (auth_algorithm) {
  case AUTH_ALG_SAE:
    offset = add_ff_auth_sae(tree, tvb, pinfo, offset);
    break;
  case AUTH_ALG_FILS_PK:
  case  AUTH_ALG_FILS_SK_WITH_PFS:
    offset = add_ff_auth_fils(tree, tvb, pinfo, offset);
    break;
  case AUTH_ALG_PASN:
    offset = add_ff_auth_pasn(tree, tvb, pinfo, offset);
    break;
  }

  return offset;
}

/*
 * If it is PASN wrapped data, handle it correctly, else defer to fils
 * wrapped data.
 */
static void
dissect_wrapped_data(proto_tree *tree, packet_info *pinfo, tvbuff_t *tvb,
                     int offset, uint8_t ext_tag_len _U_)
{
  ieee80211_pasn_data_t *pasn_data =
    (ieee80211_pasn_data_t*)p_get_proto_data(pinfo->pool, pinfo, proto_wlan,
                                             PASN_DATA_KEY);

  if (pasn_data) {
    proto_tree *auth_tree = NULL;
    proto_item *ai = NULL;
    uint16_t frame_len = 0;
    tvbuff_t *new_tvb = NULL;

    switch (pasn_data->pasn_seq) {
    case 1:
      auth_tree = proto_tree_add_subtree(tree, tvb, offset, -1,
                                         ett_pasn_auth_frame, &ai,
                                         "Authentication Frame");
      new_tvb = tvb_new_subset_remaining(tvb, offset);
      offset = dissect_auth_frame(auth_tree, pinfo, new_tvb);
      proto_item_set_len(ai, offset); /* This is correct */
      break;

    case 2:
      /* This has two auth frames in it. */
      frame_len = tvb_get_letohs(tvb, offset);
      proto_tree_add_item(tree, hf_ieee80211_pasn_auth1_frame_len, tvb, offset,
                          2, ENC_LITTLE_ENDIAN);
      offset += 2;
      auth_tree = proto_tree_add_subtree(tree, tvb, offset, frame_len,
                                         ett_pasn_auth_frame, NULL,
                                         "Authentication Frame 1");
      new_tvb = tvb_new_subset_length(tvb, offset, frame_len);
      offset += dissect_auth_frame(auth_tree, pinfo, new_tvb);

      /* Second frame */
      frame_len = tvb_get_letohs(tvb, offset);
      proto_tree_add_item(tree, hf_ieee80211_pasn_auth2_frame_len, tvb, offset,
                          2, ENC_LITTLE_ENDIAN);
      offset += 2;
      auth_tree = proto_tree_add_subtree(tree, tvb, offset, frame_len,
                                         ett_pasn_auth_frame, NULL,
                                         "Authentication Frame 2");
      new_tvb = tvb_new_subset_length(tvb, offset, frame_len);
      dissect_auth_frame(auth_tree, pinfo, new_tvb);
      break;

    case 3:
      auth_tree = proto_tree_add_subtree(tree, tvb, offset, -1,
                                         ett_pasn_auth_frame, &ai,
                                         "Authentication Frame");
      new_tvb = tvb_new_subset_remaining(tvb, offset);
      offset = dissect_auth_frame(auth_tree, pinfo, new_tvb);
      proto_item_set_len(ai, offset);
      break;
    }
  }
}

#define FILS_REQ_PARAMS_FILS_CRITERIA           0x01
#define FILS_REQ_PARAMS_MAX_DELAY_LIMIT         0x02
#define FILS_REQ_PARAMS_MINIMUM_DATA_RATE       0x04
#define FILS_REQ_PARAMS_RCPI_LIMIT              0x08
#define FILS_REQ_PARAMS_OUI_RESPONSE_CRITERIA   0x10
#define FILS_REQ_PARAMS_RESERVED                0xE0

#define FILS_REQ_PARAMS_FILS_CRITERIA_BSS_DELAY   0x07
#define FILS_REQ_PARAMS_FILS_CRITERIA_PHY_SUPPORT 0x38
#define FILS_REQ_PARAMS_FILS_CRITERIA_RESERVED    0xC0

static void
dissect_fils_req_params(proto_tree *tree, packet_info *pinfo _U_, tvbuff_t *tvb,
                     int offset, uint8_t ext_tag_len _U_)
{
  uint8_t bitmap;

  static int * const ieee80211_fils_req_params_paramter_control_bitmap[] = {
    &hf_ieee80211_fils_req_params_fils_criteria_present,
    &hf_ieee80211_fils_req_params_max_delay_limit_present,
    &hf_ieee80211_fils_req_params_minimum_data_rate_present,
    &hf_ieee80211_fils_req_params_rcpi_limit_present,
    &hf_ieee80211_fils_req_params_oui_response_criteria_present,
    &hf_ieee80211_fils_req_params_reserved,
    NULL
  };

  static int * const ieee80211_fils_req_params_fils_criteria[] = {
    &hf_ieee80211_fils_req_params_fils_criteria_bss_delay,
    &hf_ieee80211_fils_req_params_fils_criteria_phy_support,
    &hf_ieee80211_fils_req_params_fils_criteria_reserved,
    NULL
  };

  bitmap = tvb_get_uint8(tvb, offset);
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_fils_req_params_parameter_control_bitmap,
                         ett_ff_fils_req_params, ieee80211_fils_req_params_paramter_control_bitmap, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_fils_req_params_max_channel_time,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset +=1;

  if(bitmap & FILS_REQ_PARAMS_FILS_CRITERIA) {
    proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_fils_req_params_fils_criteria,
                         ett_ff_fils_req_params_fils_criteria, ieee80211_fils_req_params_fils_criteria, ENC_LITTLE_ENDIAN);
    offset += 1;
  }

  if(bitmap & FILS_REQ_PARAMS_MAX_DELAY_LIMIT) {
    proto_tree_add_item(tree, hf_ieee80211_fils_req_params_max_delay_limit,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
  }

  if(bitmap & FILS_REQ_PARAMS_MINIMUM_DATA_RATE) {
    proto_tree_add_item(tree, hf_ieee80211_fils_req_params_minimum_data_rate,
                        tvb, offset, 3, ENC_LITTLE_ENDIAN);
    offset += 3;
  }

  if(bitmap & FILS_REQ_PARAMS_RCPI_LIMIT) {
    proto_tree_add_item(tree, hf_ieee80211_fils_req_params_rcpi_limit,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
  }

  if(bitmap & FILS_REQ_PARAMS_OUI_RESPONSE_CRITERIA) {
    proto_tree_add_item(tree, hf_ieee80211_fils_req_params_oui_response_criteria,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    //offset += 2;
  }

}

static unsigned
wnm_bss_trans_mgmt_query(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int    start = offset;
  int    left;
  int tmp_sublen;
  const uint8_t ids[] = { TAG_NEIGHBOR_REPORT };


  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

  proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_query_reason, tvb, offset, 1,
                      ENC_NA);
  offset += 1;

  left = tvb_reported_length_remaining(tvb, offset);
  if (left > 0) {
    proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_candidate_list_entries,
                        tvb, offset, left, ENC_NA);

    while (left > 0){
      tmp_sublen = tvb_get_uint8(tvb, offset + 1);
      if(add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
        break;
      }
      left -= (tmp_sublen + 2);
      offset += (tmp_sublen + 2);
    }
  }

  return offset - start;
}


static unsigned
wnm_bss_trans_mgmt_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int    start = offset;
  uint8_t mode;
  int    left = tvb_reported_length_remaining(tvb, offset);
  int tmp_sublen;
  const uint8_t ids[] = { TAG_NEIGHBOR_REPORT, TAG_VENDOR_SPECIFIC_IE};

  static int * const ieee80211_ff_request_flags[] = {
    &hf_ieee80211_ff_request_mode_pref_cand,
    &hf_ieee80211_ff_request_mode_abridged,
    &hf_ieee80211_ff_request_mode_disassoc_imminent,
    &hf_ieee80211_ff_request_mode_bss_term_included,
    &hf_ieee80211_ff_request_mode_ess_disassoc_imminent,
    &hf_ieee80211_ff_request_mode_link_removal_imminent,
    &hf_ieee80211_ff_request_mode_reserved,
    NULL
  };

  if (left < 5) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
    return offset - start;
  }

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

  mode = tvb_get_uint8(tvb, offset);
  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_ff_request_flags, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_ff_disassoc_timer, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_ff_validity_interval, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  offset += 1;
  left -= 5;

  if (mode & 0x08) {
    proto_item *item;
    proto_tree  *sub_tree;
    uint8_t sub_id, sub_len;

    // BSS termination Duration sub element is the same as the neighbor report sub element
    if (left < 12) {
      expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
      return offset - start;
    }

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_tag_neighbor_report_subelement_tree,
                                      NULL, "BSS Termination Duration");

    sub_id = tvb_get_uint8(tvb, offset);
    item = proto_tree_add_item(sub_tree, hf_ieee80211_tag_neighbor_report_subelement_id,
                               tvb, offset, 1, ENC_NA);
    offset += 1;
    if (sub_id != NR_SUB_ID_BSS_TER_DUR) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_inv_val,
                             "Incorrect BSS Termination Duration subelement ID");
    }

    sub_len = tvb_get_uint8(tvb, offset);
    item = proto_tree_add_item(sub_tree, hf_ieee80211_tag_neighbor_report_subelement_length,
                               tvb, offset, 1, ENC_NA);
    offset += 1;
    if (sub_len != 10) {
       expert_add_info_format(pinfo, item, &ei_ieee80211_inv_val,
                              "Incorrect BSS Termination Duration subelement length");
    }

    proto_tree_add_item(sub_tree, hf_ieee80211_tag_neighbor_report_subelement_bss_ter_tsf,
                        tvb, offset, 8, ENC_LITTLE_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_ieee80211_tag_neighbor_report_subelement_bss_dur,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    left -= 12;
  }

  if (mode & 0x10) {
    uint8_t url_len;

    url_len = tvb_get_uint8(tvb, offset);
    if (left < url_len) {
      expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
      return offset - start;
    }

    proto_tree_add_item(tree, hf_ieee80211_ff_url_len, tvb, offset, 1,
                        ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_ff_url, tvb, offset, url_len,
                        ENC_ASCII);
    offset += url_len;
    left -= url_len + 1;
  }

  if (left > 0) {
    proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_candidate_list_entries,
                        tvb, offset, left, ENC_NA);

    while (left > 0){
      tmp_sublen = tvb_get_uint8(tvb, offset + 1);
      if(add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
        break;
      }
      left -= (tmp_sublen + 2);
      offset += (tmp_sublen + 2);
    }
  }

  return offset - start;
}


static unsigned
wnm_bss_trans_mgmt_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int    start = offset;
  uint8_t code;
  int    left;
  int tmp_sublen;
  const uint8_t ids[] = { TAG_NEIGHBOR_REPORT, TAG_VENDOR_SPECIFIC_IE };

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  code = tvb_get_uint8(tvb, offset);
  offset += add_ff_bss_transition_status_code(tree, tvb, pinfo, offset);
  offset += add_ff_bss_termination_delay(tree, tvb, pinfo, offset);
  if (!code) {
    proto_tree_add_item(tree, hf_ieee80211_ff_target_bss,
                        tvb, offset, 6, ENC_NA);
    offset += 6;
  }
  left = tvb_reported_length_remaining(tvb, offset);
  if (left > 0) {
    proto_tree_add_item(tree, hf_ieee80211_ff_bss_transition_candidate_list_entries,
                        tvb, offset, left, ENC_NA);
    while (left > 0){
      tmp_sublen = tvb_get_uint8(tvb, offset + 1);
      if(add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL) == 0){
        break;
      }
      left -= (tmp_sublen + 2);
      offset += (tmp_sublen + 2);
    }
  }

  return offset - start;
}

static unsigned
wnm_sleep_mode_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  return offset - start;
}

static unsigned
wnm_sleep_mode_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  uint16_t key_data_len;
  int left;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  key_data_len = tvb_get_letohs(tvb, offset);
  offset += add_ff_key_data_length(tree, tvb, pinfo, offset);
  left = tvb_reported_length_remaining(tvb, offset);
  if (left < key_data_len) {
    expert_add_info(pinfo, tree, &ei_ieee80211_tag_wnm_sleep_mode_no_key_data);
    return offset - start;
  }
  proto_tree_add_item(tree, hf_ieee80211_ff_key_data, tvb, offset,
                      key_data_len, ENC_NA);
  offset += key_data_len;
  return offset - start;
}

static unsigned
wnm_tfs_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  return offset - start;
}

static unsigned
wnm_tfs_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  return offset - start;
}

#define AP_DESCRIPTOR            0
#define FIRMWARE_VERSION_CURRENT 1
#define FIRMWARE_VERSION_NEW     2

static unsigned
dissect_wnm_subelements(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
  int offset) {
  unsigned sub_elt_id = tvb_get_uint8(tvb, offset);
  unsigned sub_elt_len = tvb_get_uint8(tvb, offset + 1);

  proto_tree_add_item(tree, hf_ieee80211_wnm_sub_elt_id, tvb, offset, 1, ENC_NA);
  offset++;

  proto_tree_add_item(tree, hf_ieee80211_wnm_sub_elt_len, tvb, offset, 1, ENC_NA);
  offset++;

  switch (sub_elt_id) {
  case AP_DESCRIPTOR:

    break;

  case FIRMWARE_VERSION_CURRENT:

    break;

  case FIRMWARE_VERSION_NEW:

    break;
  }

  offset += sub_elt_len;
  return offset;
}

static unsigned
wnm_channel_usage_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

  return offset - start;
}

static unsigned
wnm_channel_usage_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset, i;
  uint8_t id, len;
  proto_tree *subtree;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

  /* Parse multiple chan usage elements */
  id = tvb_get_uint8(tvb, offset);
  while (id == TAG_CHANNEL_USAGE){
    len = tvb_get_uint8(tvb, offset + 1);
    subtree = proto_tree_add_subtree(tree, tvb, offset, len + 2,
                                     ett_chan_usage,
                                     NULL, "Channel Usage");
    proto_tree_add_item(subtree, hf_ieee80211_tag_number, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(subtree, hf_ieee80211_tag_length, tvb, offset, 1, ENC_NA);
    offset += 1;
    proto_tree_add_item(subtree, hf_ieee80211_tag_channel_usage_mode, tvb, offset, 1, ENC_NA);
    offset += 1;
    len -= 1;

    for (i = 0; i < (len / 2); i++) {
      offset += add_ff_operating_class(subtree, tvb, pinfo, offset);
      offset += add_ff_channel(subtree, tvb, pinfo, offset);
    }
    id = tvb_get_uint8(tvb, offset);
  }
  offset += add_ff_country_str(tree, tvb, pinfo, offset);

  return offset - start;
}

static unsigned
wnm_notification_req(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;
  int len = 0;
  uint8_t wnm_type = 0;
  uint8_t wnm_sub_elt = 0;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  wnm_type = tvb_get_uint8(tvb, offset);
  offset += add_ff_wnm_notification_type(tree, tvb, pinfo, offset);
  len = tvb_reported_length_remaining(tvb, offset);

  if (wnm_type == 0) {
    int offset_end = offset + len;
    while (offset < offset_end) {
      int start_offset = offset;
      proto_tree *wnm_list = NULL;
      proto_item *wnm_item = NULL;
      wnm_list = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                        ett_wnm_notif_subelt,
                        &wnm_item, "WNM Subelement %d", wnm_sub_elt);
      offset = dissect_wnm_subelements(wnm_list, tvb, pinfo, offset);
      proto_item_set_len(wnm_item, offset - start_offset);
    }
  }
  return offset - start;
}

static unsigned
wnm_notification_resp(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int start = offset;

  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_wnm_notification_response_status,
                      tvb, offset, 1, ENC_NA);
  offset += 1;

  return offset - start;
}

static unsigned
add_ff_action_wnm(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  unsigned  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_wnm_action_code(tree, tvb, pinfo, offset);
  switch (code) {
  case WNM_EVENT_REQ:
  case WNM_EVENT_REPORT:
  case WNM_DIAGNOSTIC_REQ:
  case WNM_DIAGNOSTIC_REPORT:
  case WNM_LOCATION_CFG_REQ:
  case WNM_LOCATION_CFG_RESP:
  case WNM_FMS_REQ:
  case WNM_FMS_RESP:
  case WNM_DMS_REQ:
  case WNM_DMS_RESP:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case WNM_BSS_TRANS_MGMT_QUERY:
    offset += wnm_bss_trans_mgmt_query(tree, tvb, pinfo, offset);
    break;
  case WNM_BSS_TRANS_MGMT_REQ:
    offset += wnm_bss_trans_mgmt_req(tree, tvb, pinfo, offset);
    break;
  case WNM_BSS_TRANS_MGMT_RESP:
    offset += wnm_bss_trans_mgmt_resp(tree, tvb, pinfo, offset);
    break;
  case WNM_TFS_REQ:
    offset += wnm_tfs_req(tree, tvb, pinfo, offset);
    break;
  case WNM_TFS_RESP:
    offset += wnm_tfs_resp(tree, tvb, pinfo, offset);
    break;
  case WNM_SLEEP_MODE_REQ:
    offset += wnm_sleep_mode_req(tree, tvb, pinfo, offset);
    break;
  case WNM_SLEEP_MODE_RESP:
    offset += wnm_sleep_mode_resp(tree, tvb, pinfo, offset);
    break;
  case WNM_CHANNEL_USAGE_REQ:
    offset += wnm_channel_usage_req(tree, tvb, pinfo, offset);
    break;
  case WNM_CHANNEL_USAGE_RESP:
    offset += wnm_channel_usage_resp(tree, tvb, pinfo, offset);
    break;
  case WNM_NOTIFICATION_REQ:
    offset += wnm_notification_req(tree, tvb, pinfo, offset);
    break;
  case WNM_NOTIFICATION_RESP:
    offset += wnm_notification_resp(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_unprotected_wnm(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  unsigned  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_unprotected_wnm_action_code(tree, tvb, pinfo, offset);

  switch (code) {
  case UNPROTECTED_WNM_TIM:
    offset += add_ff_check_beacon(tree, tvb, pinfo, offset);
    offset += add_ff_timestamp(tree, tvb, pinfo, offset);
    offset += add_ff_tod(tree, tvb, pinfo, offset);
    offset += add_ff_toa(tree, tvb, pinfo, offset);
    offset += add_ff_max_tod_err(tree, tvb, pinfo, offset);
    offset += add_ff_max_toa_err(tree, tvb, pinfo, offset);
    break;
  case UNPROTECTED_WNM_TIMING_MEASUREMENT:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_followup_dialog_token(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_tdls(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  uint16_t status;
  unsigned   start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code = tvb_get_uint8(tvb, offset);

  /* Extract keys for dot11decrypt engine */
  try_scan_tdls_keys(tvb, pinfo, offset);

  offset += add_ff_tdls_action_code(tree, tvb, pinfo, offset);
  switch (code) {
  case TDLS_SETUP_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    break;
  case TDLS_SETUP_RESPONSE:
    status = tvb_get_letohs(tvb, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    if (tvb_reported_length_remaining(tvb, offset) < 2) {
      if (status == 0) {
        expert_add_info(pinfo, tree, &ei_ieee80211_tdls_setup_response_malformed);
      }
      break;
    }
    offset += add_ff_cap_info(tree, tvb, pinfo, offset);
    break;
  case TDLS_SETUP_CONFIRM:
    status = tvb_get_letohs(tvb, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    if (tvb_reported_length_remaining(tvb, offset) < 1) {
      if (status == 0) {
        expert_add_info(pinfo, tree, &ei_ieee80211_tdls_setup_confirm_malformed);
      }
      break;
    }
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_TEARDOWN:
    offset += add_ff_reason_code(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_TRAFFIC_INDICATION:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_CHANNEL_SWITCH_REQUEST:
    offset += add_ff_target_channel(tree, tvb, pinfo, offset);
    offset += add_ff_operating_class(tree, tvb, pinfo, offset);
    break;
  case TDLS_CHANNEL_SWITCH_RESPONSE:
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_PSM_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_PSM_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    break;
  case TDLS_PEER_TRAFFIC_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  case TDLS_DISCOVERY_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_mgmt_notification(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_wme_action_code(tree, tvb, pinfo, offset);
  offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
  offset += add_ff_wme_status_code(tree, tvb, pinfo, offset);

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_vendor_specific(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned   start = offset;
  uint32_t oui;
  tvbuff_t *vendor_tvb;
  int dissected;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  proto_tree_add_item_ret_uint(tree, hf_ieee80211_tag_oui, tvb, offset, 3, ENC_BIG_ENDIAN, &oui);
  offset += 3;

  vendor_tvb = tvb_new_subset_remaining(tvb, offset);
  dissected = dissector_try_uint_with_data(vendor_specific_action_table, oui, vendor_tvb, pinfo, tree, false, NULL);
  if (dissected <= 0)
  {
      call_data_dissector(vendor_tvb, pinfo, tree);
      /* Skip the whole TVB because we don't know its format */
      dissected = tvb_reported_length_remaining(vendor_tvb, 0);
  }

  offset += dissected;

  return offset - start;  /* Size of fixed fields */
}

static unsigned
add_ff_action_ht(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned  start = offset;
  uint8_t n_sta, i;
  mimo_control_t mimo_cntrl;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  offset += add_ff_ht_action_code(tree, tvb, pinfo, offset);

  switch (tvb_get_uint8(tvb, offset - 1)) {
  case HT_ACTION_NOTIFY_CHAN_WIDTH:
    offset += add_ff_channel_width(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_SM_PWR_SAVE:
    offset += add_ff_sm_pwr_cntrl(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_PSMP_ACTION:
    n_sta = tvb_get_uint8(tvb, offset);
    offset += add_ff_psmp_param_set(tree, tvb, pinfo, offset);
    for (i = 0; i < (n_sta & 0x0F); i++) {
      offset += add_ff_psmp_sta_info(tree, tvb, pinfo, offset);
    }
    break;
  case HT_ACTION_SET_PCO_PHASE:
    offset += add_ff_pco_phase_cntrl(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_MIMO_CSI:
    mimo_cntrl = get_mimo_control(tvb, offset);
    offset += add_ff_mimo_cntrl(tree, tvb, pinfo, offset);
    offset += add_mimo_csi_matrices_report(tree, tvb, offset, mimo_cntrl);
    break;
  case HT_ACTION_MIMO_BEAMFORMING:
    mimo_cntrl = get_mimo_control(tvb, offset);
    offset += add_ff_mimo_cntrl(tree, tvb, pinfo, offset);
    offset += add_mimo_beamforming_feedback_report(tree, tvb, offset,
                                                   mimo_cntrl);
    break;
  case HT_ACTION_MIMO_COMPRESSED_BEAMFORMING:
    mimo_cntrl = get_mimo_control(tvb, offset);
    offset += add_ff_mimo_cntrl(tree, tvb, pinfo, offset);
    offset += add_mimo_compressed_beamforming_feedback_report(tree, tvb,
                                                              offset,
                                                              mimo_cntrl);
    break;
  case HT_ACTION_ANT_SEL_FEEDBACK:
    offset += add_ff_ant_selection(tree, tvb, pinfo, offset);
    break;
  case HT_ACTION_HT_INFO_EXCHANGE:
    offset += add_ff_ht_information(tree, tvb, pinfo, offset);
    break;
  }

  return offset - start;
}

static unsigned
add_ff_beacon_interval_ctrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_bic,
                                    ett_bic_tree, ieee80211_ff_bic_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  return 6;
}

static unsigned
add_ff_beamforming_ctrl(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset, bool isGrant)
{
  uint16_t bf_field = tvb_get_letohs(tvb, offset);
  bool isInit = (bf_field & 0x2) >> 1;
  bool isResp = (bf_field & 0x4) >> 2;
  static int * const ieee80211_ff_beamforming_ctrl[] = {
    &hf_ieee80211_ff_bf_train,
    &hf_ieee80211_ff_bf_is_init,
    &hf_ieee80211_ff_bf_is_resp,
    &hf_ieee80211_ff_bf_rxss_len,
    &hf_ieee80211_ff_bf_rxss_rate,
    &hf_ieee80211_ff_bf_b10b15,
    NULL
  };

  static int * const ieee80211_ff_beamforming_ctrl_grant[] = {
    &hf_ieee80211_ff_bf_train,
    &hf_ieee80211_ff_bf_is_init,
    &hf_ieee80211_ff_bf_is_resp,
    &hf_ieee80211_ff_bf_num_sectors,
    &hf_ieee80211_ff_bf_num_rx_dmg_ants,
    &hf_ieee80211_ff_bf_b12b15,
    NULL
  };

  if((isInit==true) && (isResp==true) && isGrant) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_bf,
                                    ett_bf_tree, ieee80211_ff_beamforming_ctrl_grant,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  } else {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_bf,
                                    ett_bf_tree, ieee80211_ff_beamforming_ctrl,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  }
  return 2;
}

static unsigned
add_ff_dynamic_allocation(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_ff_dynamic_allocation[] = {
    &hf_ieee80211_ff_TID,
    &hf_ieee80211_ff_alloc_type,
    &hf_ieee80211_ff_src_aid,
    &hf_ieee80211_ff_dest_aid,
    &hf_ieee80211_ff_alloc_duration,
    &hf_ieee80211_ff_b39,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_dynamic_allocation,
                                    ett_dynamic_alloc_tree, ieee80211_ff_dynamic_allocation,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  return 5;
}

static unsigned
add_ff_beamformed_link(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_ff_beamformed_link[] = {
    &hf_ieee80211_ff_blm_unit_index,
    &hf_ieee80211_ff_blm_maint_value,
    &hf_ieee80211_ff_blm_is_master,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_blm,
                                    ett_blm_tree, ieee80211_ff_beamformed_link,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static unsigned
add_ff_BRP_request(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_ff_BRP_request[] = {
    &hf_ieee80211_ff_brp_L_RX,
    &hf_ieee80211_ff_brp_TX_TRN_REQ,
    &hf_ieee80211_ff_brp_MID_REQ,
    &hf_ieee80211_ff_brp_BC_REQ,
    &hf_ieee80211_ff_brp_MID_GRANT,
    &hf_ieee80211_ff_brp_BC_GRANT,
    &hf_ieee80211_ff_brp_chan_FBCK_CAP,
    &hf_ieee80211_ff_brp_tx_sector,
    &hf_ieee80211_ff_brp_other_aid,
    &hf_ieee80211_ff_brp_tx_antenna,
    &hf_ieee80211_ff_brp_reserved,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_brp,
                                    ett_brp_tree, ieee80211_ff_BRP_request,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 4;
}

static unsigned
add_ff_sector_sweep_feedback_from_iss(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_ff_sector_sweep_feedback_from_iss[] = {
    &hf_ieee80211_ff_sswf_total_sectors,
    &hf_ieee80211_ff_sswf_num_rx_dmg_ants,
    &hf_ieee80211_ff_sswf_reserved1,
    &hf_ieee80211_ff_sswf_poll_required,
    &hf_ieee80211_ff_sswf_reserved2,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_sswf,
                                    ett_sswf_tree, ieee80211_ff_sector_sweep_feedback_from_iss,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 3;
}

static unsigned
add_ff_sector_sweep_feedback_to_iss(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_ff_sector_sweep_feedback_to_iss[] = {
    &hf_ieee80211_ff_sswf_sector_select,
    &hf_ieee80211_ff_sswf_dmg_antenna_select,
    &hf_ieee80211_ff_sswf_snr_report,
    &hf_ieee80211_ff_sswf_poll_required,
    &hf_ieee80211_ff_sswf_reserved2,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_sswf,
                                    ett_sswf_tree, ieee80211_ff_sector_sweep_feedback_to_iss,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 3;
}

static unsigned
add_ff_sector_sweep(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_ff_sector_sweep[] = {
    &hf_ieee80211_ff_ssw_direction,
    &hf_ieee80211_ff_ssw_cdown,
    &hf_ieee80211_ff_ssw_sector_id,
    &hf_ieee80211_ff_ssw_dmg_ant_id,
    &hf_ieee80211_ff_ssw_rxss_len,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_ssw,
                                    ett_ssw_tree, ieee80211_ff_sector_sweep,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 3;
}

static unsigned
add_ff_dmg_params(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  static int * const ieee80211_ff_dmg_params[] = {
    &hf_ieee80211_ff_dmg_params_bss,
    &hf_ieee80211_ff_dmg_params_cbap_only,
    &hf_ieee80211_ff_dmg_params_cbap_src,
    &hf_ieee80211_ff_dmg_params_privacy,
    &hf_ieee80211_ff_dmg_params_policy,
    &hf_ieee80211_ff_dmg_params_spec_mgmt,
    &hf_ieee80211_ff_dmg_params_radio_measure,
    NULL
  };

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ff_dmg_params,
                                    ett_dmg_params_tree, ieee80211_ff_dmg_params,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return 1;
}

static unsigned
add_ff_cc_field(proto_tree *tree, tvbuff_t *tvb, int offset, bool dis)
{
  proto_item *cc_item = proto_tree_add_item(tree, hf_ieee80211_ff_cc, tvb, offset, 8, ENC_LITTLE_ENDIAN);
  proto_tree *cc_tree = proto_item_add_subtree(cc_item, ett_cc_tree);
  uint64_t cc_field;
  if(dis) {
    proto_tree_add_item(cc_tree, hf_ieee80211_ff_cc_abft_resp_addr, tvb, offset, 6, ENC_NA);
  } else {
    cc_field = tvb_get_letoh64(tvb, offset);
    /*TODO : Add support of bitmask for FT_(U)INT64 */
    proto_tree_add_uint(cc_tree, hf_ieee80211_ff_cc_sp_duration, tvb, offset, 1, (uint32_t)(cc_field & 0xff));
    proto_tree_add_uint64(cc_tree, hf_ieee80211_ff_cc_cluster_id, tvb, offset+1, 6, (uint64_t)((cc_field & UINT64_C(0x00ffffffffffff00)) >> 8));
    proto_tree_add_uint(cc_tree, hf_ieee80211_ff_cc_role, tvb, offset+7, 1, (uint32_t)((cc_field & UINT64_C(0x0300000000000000)) >> 56));
    proto_tree_add_uint(cc_tree, hf_ieee80211_ff_cc_max_mem, tvb, offset+7, 1, (uint32_t)((cc_field & UINT64_C(0x7c00000000000000)) >> 58));
  }
  return 8;
}


static unsigned
add_ff_band_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_band_id, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_subject_address(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_subject_address, tvb, offset, 6, ENC_NA);
  return 6;
}

static unsigned
add_ff_handover_reason(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_reason, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_handover_remaining_bi(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_remaining_bi, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_handover_result(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_result, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_handover_reject_reason(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_handover_reject_reason, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_destination_reds_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_destination_reds_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_destination_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_destination_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_relay_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_relay_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_source_aid(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_source_aid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_timing_offset(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_timing_offset, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_sampling_frequency_offset(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_sampling_frequency_offset, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return 2;
}

static unsigned
add_ff_relay_operation_type(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_relay_operation_type, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_fst_action_code(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_fst_action_code, tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_robust_av_streaming_action_code(proto_tree *tree, tvbuff_t *tvb,
                                       packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_robust_av_streaming_action_code,
                      tvb, offset, 1, ENC_NA);
  return 1;
}

static unsigned
add_ff_llt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_llt, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  return 4;
}

static unsigned
add_ff_fsts_id(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_fsts_id, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  return 4;
}

static unsigned
add_ff_oct_mmpdu(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  unsigned start = offset;
  unsigned len = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_ff_mmpdu_len, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_ff_mmpdu_ctrl, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_ff_oct_mmpdu, tvb, offset, len, ENC_NA);
  offset += len;
  return offset - start;
}

static int * const eht_eml_control_field_mode_headers[] = {
  &hf_ieee80211_eht_eml_control_emlsr_mode,
  &hf_ieee80211_eht_eml_control_emlmr_mode,
  &hf_ieee80211_eht_eml_control_emlsr_para_update_control,
  &hf_ieee80211_eht_eml_control_device_coexist_activities,
  &hf_ieee80211_eht_eml_control_reserved,
  NULL
};

static int * const eht_emlsr_para_update_headers[] = {
  &hf_ieee80211_eht_emlsr_para_update_padding_delay,
  &hf_ieee80211_eht_emlsr_para_update_tran_delay,
  &hf_ieee80211_eht_emlsr_para_update_reserved,
  NULL
};

static int * const eht_eml_control_mcs_map_count_headers[] = {
  &hf_ieee80211_eht_eml_control_mcs_map_count_bw,
  &hf_ieee80211_eht_eml_control_mcs_map_count_reserved,
  NULL
};

static const value_string eht_eml_control_mcs_map_count_bw[] = {
  {0x00, "80 MHz"},
  {0x01, "160 MHz"},
  {0x02, "320 MHz"},
  {0, NULL}
};

static int * const eht_le_80_mcs_map_hdrs[] = {
  &hf_ieee80211_eht_le_80_rx_max_nss_0_9,
  &hf_ieee80211_eht_le_80_tx_max_nss_0_9,
  &hf_ieee80211_eht_le_80_rx_max_nss_10_11,
  &hf_ieee80211_eht_le_80_tx_max_nss_10_11,
  &hf_ieee80211_eht_le_80_rx_max_nss_12_13,
  &hf_ieee80211_eht_le_80_tx_max_nss_12_13,
  NULL
};

static int * const eht_160_mcs_map_hdrs[] = {
  &hf_ieee80211_eht_160_rx_max_nss_0_9,
  &hf_ieee80211_eht_160_tx_max_nss_0_9,
  &hf_ieee80211_eht_160_rx_max_nss_10_11,
  &hf_ieee80211_eht_160_tx_max_nss_10_11,
  &hf_ieee80211_eht_160_rx_max_nss_12_13,
  &hf_ieee80211_eht_160_tx_max_nss_12_13,
  NULL
};

static int * const eht_320_mcs_map_hdrs[] = {
  &hf_ieee80211_eht_320_rx_max_nss_0_9,
  &hf_ieee80211_eht_320_tx_max_nss_0_9,
  &hf_ieee80211_eht_320_rx_max_nss_10_11,
  &hf_ieee80211_eht_320_tx_max_nss_10_11,
  &hf_ieee80211_eht_320_rx_max_nss_12_13,
  &hf_ieee80211_eht_320_tx_max_nss_12_13,
  NULL
};

static unsigned
dissect_eht_eml_control_field(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  unsigned start = offset;
  unsigned len = tvb_captured_length_remaining(tvb, offset);
  uint8_t mode = tvb_get_uint8(tvb, offset);
  uint8_t mcs_map_count;
  proto_item *link_map_item;
  proto_tree *link_map_tree;
  uint16_t bmap, f;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_eht_eml_control_field,
                                    ett_eht_eml_control,
                                    eht_eml_control_field_mode_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  if (mode & 0x03) { /* EMLSR or EMLMR*/
    if (len < 3) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_eht_invalid_action, "EMLSR or EMLMR length %u is wrong", len);
      return len;
    }
    link_map_item = proto_tree_add_item(tree,
                      hf_ieee80211_eht_eml_control_link_bitmap,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
    link_map_tree = proto_item_add_subtree(link_map_item,
                     ett_eht_eml_control_link_map);
    bmap = tvb_get_letohs(tvb, offset);
    for (f = 0; f < 16; f++) {
      if (bmap & (1 << f)) {
        proto_tree_add_uint_format_value(link_map_tree,
          hf_ieee80211_eht_eml_control_link_enable_id,
          tvb, offset + (f/8), 1, f, "%u", f);
      }
    }
    offset += 2;
  }
  if (mode & 0x02) { /* EMLMR */
    if (len < 7) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_eht_invalid_action, "EMLMR length %u is wrong", len);
      return len;
    }
    mcs_map_count = tvb_get_uint8(tvb, offset) & 0x03;
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
        hf_ieee80211_eht_eml_control_mcs_map_count,
        ett_eht_eml_control_mcs_map_count,
        eht_eml_control_mcs_map_count_headers, ENC_LITTLE_ENDIAN,
        BMT_NO_APPEND);
    offset++;

    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
      hf_ieee80211_eht_mcs_and_nss_le_80mhz,
      ett_eht_phy_mcs_nss_set,
      eht_le_80_mcs_map_hdrs,
      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 3;
    if (mcs_map_count >= 1) {
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
        hf_ieee80211_eht_mcs_and_nss_eq_160mhz,
        ett_eht_phy_mcs_nss_set,
        eht_160_mcs_map_hdrs,
        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      offset += 3;
    }
    if (mcs_map_count >= 2) {
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
        hf_ieee80211_eht_mcs_and_nss_eq_320mhz,
        ett_eht_phy_mcs_nss_set,
        eht_320_mcs_map_hdrs,
        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      offset += 3;
    }
  }

  if (mode & 0x04) { /* EMLSR Parameter Update */
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
      hf_ieee80211_eht_emlsr_para_update, ett_eht_emlsr_para_update,
      eht_emlsr_para_update_headers, ENC_LITTLE_ENDIAN,
      BMT_NO_APPEND);
    offset++;
  }

  return offset - start;
}

static int
add_tag_relay_capabilities(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static int * const ieee80211_tag_relay_capabilities[] = {
    &hf_ieee80211_tag_relay_support,
    &hf_ieee80211_tag_relay_use,
    &hf_ieee80211_tag_relay_permission,
    &hf_ieee80211_tag_AC_power,
    &hf_ieee80211_tag_relay_prefer,
    &hf_ieee80211_tag_duplex,
    &hf_ieee80211_tag_cooperation,
    NULL
  };

  if (tag_len < 2) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length must be 2");
    return 1;
  }

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_tag_relay_capabilities, ENC_NA);
  return tvb_captured_length(tvb);
}

#if 0
static unsigned
add_ff_relay_capable_sta_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_item *rcsi_item = proto_tree_add_item(tree, hf_ieee80211_ff_rcsi, tvb, offset, 3, ENC_LITTLE_ENDIAN);
  proto_tree *rcsi_tree = proto_item_add_subtree(rcsi_item, ett_rcsi_tree);
  proto_tree_add_item(rcsi_tree, hf_ieee80211_ff_rcsi_aid, tvb, offset, 1, ENC_NA);
  offset += 1;
  add_tag_relay_capabilities(pinfo, rcsi_item, 2, rcsi_tree, tvb, &offset);
  return 3;
}
#endif

#define NEXT_TBTT_PRESENT       0x01
#define COMPRESSED_SSID_PRESENT 0x02
#define ANO_PRESENT             0x04

static void
dissect_ieee80211_extension(uint16_t fcf, tvbuff_t *tvb, packet_info *pinfo,
                            proto_tree *tree, uint16_t flags)
{
  proto_item *ti;
  proto_tree *ext_tree;
  proto_tree *fixed_tree;
  proto_tree *tagged_tree;

  int offset = 0;
  int tagged_parameter_tree_len;

  ti = proto_tree_add_item(tree, proto_wlan_ext, tvb, offset, -1, ENC_NA);
  ext_tree = proto_item_add_subtree(ti, ett_80211_ext);

  switch (COMPOSE_FRAME_TYPE(fcf))
  {
    case EXTENSION_DMG_BEACON:
    {
      bool cc, dis;
      uint16_t bic_field;
      fixed_tree = get_fixed_parameter_tree(ext_tree, tvb, offset, 20, false);
      offset += add_ff_timestamp(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_sector_sweep(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_beacon_interval(fixed_tree, tvb, pinfo, offset);
      bic_field = tvb_get_letohs(tvb, offset);
      cc = (bic_field & 0x1);
      dis  = (bic_field & 0x2) >> 1;
      offset += add_ff_beacon_interval_ctrl(fixed_tree, tvb, pinfo, offset);
      offset += add_ff_dmg_params(fixed_tree, tvb, pinfo, offset);
      if(cc) {
        offset += add_ff_cc_field(fixed_tree, tvb, offset, dis);
      }
      tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);

      /*
       * The tagged params are optional here. See Table 8.33a of the 2012
       * version of the standard.
       */
      if (tagged_parameter_tree_len) {
        tagged_tree = get_tagged_parameter_tree(ext_tree, tvb, offset, tagged_parameter_tree_len);
        ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree, tagged_parameter_tree_len, EXTENSION_DMG_BEACON, NULL);
      }
      break;
    }
    case EXTENSION_S1G_BEACON:
    {
        int params_size = 5;

        if (flags & NEXT_TBTT_PRESENT) params_size += 3;
        if (flags & COMPRESSED_SSID_PRESENT) params_size += 4;
        if (flags & ANO_PRESENT) params_size += 1;

        fixed_tree = get_fixed_parameter_tree( ext_tree, tvb, offset, params_size, false);
        offset += add_ff_s1g_timestamp(fixed_tree, tvb, pinfo, offset);
        offset += add_ff_change_sequence(fixed_tree, tvb, pinfo, offset);
        if (flags & NEXT_TBTT_PRESENT)
          offset += add_ff_next_tbtt(fixed_tree, tvb, pinfo, offset);
        if (flags & COMPRESSED_SSID_PRESENT)
          offset += add_ff_compressed_ssid(fixed_tree, tvb, pinfo, offset);
        if (flags & ANO_PRESENT)
          offset += add_ff_access_network_options(fixed_tree, tvb, pinfo, offset);

        tagged_parameter_tree_len = tvb_reported_length_remaining(tvb, offset);

        if (tagged_parameter_tree_len) {
          tagged_tree = get_tagged_parameter_tree(ext_tree, tvb, offset, tagged_parameter_tree_len);
          ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tagged_tree, tagged_parameter_tree_len, EXTENSION_S1G_BEACON, NULL);
        }
    }
  }
}

static unsigned
add_ff_action_unprotected_dmg(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  unsigned  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_unprotected_dmg_action_code(tree, tvb, pinfo, offset);
  switch (code) {
    case UNPROTECTED_DMG_ANNOUNCE:
      offset += add_ff_timestamp(tree, tvb, pinfo, offset);
      offset += add_ff_beacon_interval(tree, tvb, pinfo, offset);
      break;
    case UNPROTECTED_DMG_BRP:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_BRP_request(tree, tvb, pinfo, offset);
      break;
  }
  return offset - start;
}

/* There is no easy way to skip all these subcarrier indices that must not
 * be displayed when showing compressed beamforming feedback matrices
 * Table 8-53g IEEE Std 802.11ac-2013 amendment.
 *
 * The irregular use of case statements in this function is to improve
 * readability in what is otherwise a large function that does very little.
 */
static inline int
vht_compressed_skip_scidx(uint8_t nchan_width, uint8_t ng, int scidx)
{
  switch(nchan_width) {
    /* 20 MHz */
    case 0:
      /* No Grouping */
      if (ng == 0)
        switch (scidx) {
          /* Pilot subcarriers */
          case -21: case -7: case 7: case 21:
          /* DC subcarrier */
          case 0:
            scidx++;
          default:
            break;
        }
      break;
    /* 40 MHz */
    case 1:
      /* No Grouping */
      if (ng == 0)
        switch (scidx) {
          /* Pilot subcarriers */
          case -53: case -25: case -11: case 11: case 25: case 53:
            scidx++;
            break;
          /* DC subcarriers */
          case -1: case 0: case 1:
            scidx = 2;
          default:
            break;
        }
      break;
    /* 80 MHz */
    case 2:
      /* No Grouping */
      if (ng == 0)
        switch (scidx) {
          /* Pilot subcarriers */
          case -103: case -75: case -39: case -11: case 11: case 39: case 75: case 103:
            scidx++;
            break;
          /* DC subcarriers, skip -1, 0, 1 */
          case -1:
            scidx = 2;
          default:
            break;
        }
      break;
    /* 160 MHz / 80+80 Mhz
     * Skip values here assume 160 MHz, as vht_mimo_control does not let us differentiate
     * between 160 MHz & 80-80MHz */
    case 3:
      switch (ng) {
        /* No Grouping */
        case 0:
          /* DC subcarriers, skip -5 to 5*/
          if (scidx == -5) {
            scidx = 6;
            break;
          }
          switch (scidx) {
            /* Pilot subcarriers */
            case -231: case -203: case -167: case -139: case -117: case -89: case -53: case -25:
            case 25: case 53: case 89: case 117: case 139: case 167: case 203: case 231:
              scidx++;
              break;
            /* Other subcarriers, skip -129 to -127, 127 to 129 */
            case -129:
              scidx = -126;
              break;
            case 127:
              scidx = 130;
              break;
            default:
              break;
          }
          break;
        /* Grouping of 2 */
        case 1:
          switch (scidx) {
            /* DC subcarriers */
            case -128: case -4: case -2: case 0: case 2: case 4: case 128:
              scidx++;
            default:
              break;
          }
          break;
        /* Grouping of 4 */
        case 2:
          if (scidx == -2 || scidx == 2)
            scidx++;
          break;
      }
      break;
    default:
      break;
  }

  return scidx;
}

static inline int vht_exclusive_skip_scidx(uint8_t nchan_width, uint8_t ng, int scidx)
{
  switch (nchan_width) {
    /* 20 MHz */
    case 0:
      switch (ng) {
        /* No Grouping */
        case 0:
          if (scidx == -2 || scidx == 1)
              scidx++;
          else
              scidx = scidx + 2;
          break;
        case 1:
          switch (scidx) {
            case -4: case 1:
              scidx = scidx + 3;
              break;
            case -1:
              scidx = 1;
              break;
            default:
              scidx = scidx + 4;
              break;
          }
          break;
        default:
          switch (scidx) {
            case -4: case 1:
              scidx = scidx + 3;
              break;
            case -1:
              scidx = 1;
              break;
            default:
              scidx = scidx + 8;
              break;
          }
          break;
      }
      break;
    /* 40 MHz */
    case 1:
    /* 80 MHz */
    case 2:
      switch (ng) {
        /* No Grouping */
        case 0:
          if (scidx == -2)
            scidx = 2;
          else
            scidx = scidx + 2;
          break;
        case 1:
          scidx = scidx + 4;
          break;
        default:
          if (scidx == -2)
            scidx = 2;
          else
            scidx = scidx + 8;
          break;
      }
      break;
    /* 160 MHz / 80+80 Mhz */
    case 3:
      switch (ng) {
        /* No Grouping */
        case 0:
          switch (scidx) {
            /* DC subcarriers, skip -4 to 4*/
            case -6:
              scidx = 6;
              break;
            /* Other subcarriers, skip -128, 128 */
            case -130:
              scidx = -126;
              break;
            case 126:
              scidx = 130;
              break;
            default:
              scidx = scidx + 2;
              break;
          }
          break;
        case 1:
          switch (scidx) {
            /* DC subcarriers, skip -4 to 4*/
            case -6:
              scidx = 6;
              break;
            default:
              scidx = scidx + 4;
              break;
          }
          break;
        default:
          switch (scidx) {
            case -6:
              scidx = 6;
              break;
            case -130:
              scidx = -126;
              break;
            case 126:
              scidx = 130;
              break;
            default:
              scidx = scidx + 8;
              break;
          }
        break;
      }
      break;
    default:
      break;
  }
  return scidx;
}

static int
dissect_he_feedback_matrix(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                           int offset, int bit_offset, int scidx,
                           int nr, int nc,
                           int phi_bits, int psi_bits,
                           int hf);

static unsigned
add_ff_vht_compressed_beamforming_report(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  uint32_t vht_mimo;
  uint8_t nc;
  uint8_t nr;
  uint8_t chan_width;
  uint8_t grouping;
  bool codebook_info;
  bool feedback_type;
  proto_item *vht_beam_item, *vht_excl_beam_item;
  proto_tree *vht_beam_tree, *subtree, *vht_excl_beam_tree;
  int i, len, pos, ns, scidx = 0;
  uint8_t phi, psi, carry;
  int j, ic, off_len = 0, sscidx = 0, xnsc;
  int bit_offset = 0;
  int start_offset = 0;
  /* Table 8-53g Subcarriers for which a Compressed Beamforming Feedback Matrix
   * subfield is sent back. IEEE Std 802.11ac-2013 amendment */
  static const int ns_arr[4][3] = { {  52,  30,  16 },
                                    { 108,  58,  30 },
                                    { 234, 122,  62 },
                                    { 468, 244, 124 }
                                  };

  /* Table 8-53j, no of Subcarriers for which the Delta SNR subfield is sent back to the beamformer.
   * IEEE Std 802.11ac-2013 amendment */
  static const int delta_ns_arr[4][3] = { {  30,  16,  10 },
                                          {  58,  30,  16 },
                                          { 122,  62,  32 },
                                          { 244, 124,  64 }
                                        };

  vht_mimo = tvb_get_letoh24(tvb, offset);
  proto_tree_add_bitmask(tree, tvb, offset, hf_ieee80211_ff_vht_mimo_cntrl,
                        ett_ff_vhtmimo_cntrl, hf_ieee80211_ff_vht_mimo_cntrl_fields, ENC_LITTLE_ENDIAN);
  offset += 3;

  /* Extract values for beamforming use */
  nc = (vht_mimo & 0x7) + 1;
  nr = ((vht_mimo & 0x38) >> 3) + 1;
  chan_width = (vht_mimo & 0xC0) >> 6;
  grouping = ((vht_mimo & 0x300) >> 8);
  codebook_info = (vht_mimo & 0x400) >> 10;
  feedback_type = (vht_mimo & 0x800) >> 11;

  if (feedback_type) {
    if (codebook_info) {
      psi = 7; phi = 9;
    } else {
      psi = 5; phi = 7;
    }
  } else {
    if (codebook_info) {
      psi = 4; phi = 6;
    } else {
      psi = 2; phi = 4;
    }
  }

  vht_beam_item = proto_tree_add_item(tree, hf_ieee80211_vht_compressed_beamforming_report, tvb,
                                  offset, -1, ENC_NA);
  vht_beam_tree = proto_item_add_subtree(vht_beam_item, ett_ff_vhtmimo_beamforming_report);

  subtree = proto_tree_add_subtree(vht_beam_tree, tvb, offset, nc,
                        ett_ff_vhtmimo_beamforming_report_snr, NULL, "Average Signal to Noise Ratio");

  for (i = 1; i <= nc; i++)
  {
    int8_t snr;
    char edge_sign;

    snr = tvb_get_int8(tvb, offset);

    switch(snr) {
      case -128:
        edge_sign = '<';
        break;
      case 127:
        edge_sign = '>';
        break;
      default:
        edge_sign = ' ';
        break;
    }

    proto_tree_add_int_format(subtree, hf_ieee80211_vht_compressed_beamforming_report_snr, tvb, offset, 1,
                               snr, "Stream %d - Signal to Noise Ratio: %c%3.2fdB", i, edge_sign,snr/4.0+22.0);

    offset += 1;
  }

  /* Table 8-53c Subfields of the VHT MIMO Control field (802.11ac-2013)
   * reserves value 3 of the Grouping subfield. */
  if (grouping == 3) {
    expert_add_info_format(pinfo, vht_beam_item, &ei_ieee80211_inv_val,
                           "Grouping subfield value 3 is reserved");
    return offset;
  }

  start_offset = offset;
  subtree = proto_tree_add_subtree(vht_beam_tree, tvb, offset, -1,
                        ett_ff_vhtmimo_beamforming_report_feedback_matrices,
                        NULL, "Feedback Matrices");

  ns = ns_arr[chan_width][grouping];
  switch(chan_width) {
    case 0:
      scidx = -28;
      break;
    case 1:
      scidx = -58;
      break;
    case 2:
      scidx = -122;
      break;
    case 3:
      /* This is -122 for 80+80MHz Channel Width but vht_mimo_control does not allow us
       * to differentiate between 160MHz and 80+80Mhz */
      scidx = -250;
      break;
  }

  bit_offset = offset * 8;
  for (i = 0; i < ns; i++) {

    scidx = vht_compressed_skip_scidx(chan_width, grouping, scidx);

    bit_offset = dissect_he_feedback_matrix(subtree, tvb, pinfo, offset,
                                    bit_offset, scidx++, nr, nc, phi, psi,
                                    hf_ieee80211_vht_compressed_beamform_scidx);
    offset = bit_offset / 8;
  }

  proto_item_set_len(subtree, offset - start_offset);

  if (feedback_type) {
    xnsc = delta_ns_arr[chan_width][grouping];
    if ((nc * xnsc *4) % 8)
        off_len = (nc * xnsc *4) / 8 + 1;
    else
        off_len = (nc * xnsc *4) / 8;
    switch(chan_width) {
      case 0:
        sscidx = -28;
        break;
      case 1:
        sscidx = -58;
        break;
      case 2:
        sscidx = -122;
        break;
      case 3:
        sscidx = -250;
        break;
    }
    vht_excl_beam_item = proto_tree_add_item(tree, hf_ieee80211_vht_mu_exclusive_beamforming_report, tvb, offset, off_len, ENC_NA);
    vht_excl_beam_tree = proto_item_add_subtree(vht_excl_beam_item, ett_ff_vhtmu_exclusive_beamforming_report_matrices);

    carry = 1;
    for (j = 1; j <= xnsc; j++) {
      for (ic = 1; ic <= nc; ic++) {
        if (carry % 2){
          pos = 0;
          len = 1;
        }
        else
        {
          pos = 1;
          len = 0;
        }
        proto_tree_add_none_format(vht_excl_beam_tree, hf_ieee80211_vht_mu_exclusive_beamforming_delta_snr, tvb,
                                      offset - pos, 1, "Delta SNR for space-time stream %d for subcarrier %d", ic, sscidx);
        offset += len;
        carry ++;
      }
      sscidx = vht_exclusive_skip_scidx(chan_width, grouping, sscidx);
    }
  }

  return offset;
}

static unsigned
add_ff_action_vht(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;
  uint8_t vht_action, field_val;
  uint64_t msa_value;
  uint64_t upa_value;
  int m, half, msa_index;
  proto_item *ti;
  proto_tree *ti_tree;
  proto_item *msa, *upa;
  proto_tree *msa_tree, *upa_tree;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  vht_action = tvb_get_uint8(tvb, offset);
  offset += add_ff_vht_action(tree, tvb, pinfo, offset);

  switch(vht_action){
    case VHT_ACT_VHT_COMPRESSED_BEAMFORMING:{
      offset = add_ff_vht_compressed_beamforming_report(tree, tvb, pinfo, offset);
      offset += tvb_reported_length_remaining(tvb, offset);
    }
    break;
    case VHT_ACT_GROUP_ID_MANAGEMENT:{
      ti = proto_tree_add_item(tree, hf_ieee80211_vht_group_id_management, tvb,
                          offset, -1, ENC_NA);
      ti_tree = proto_item_add_subtree(ti, ett_vht_grpidmgmt);

      msa_value = tvb_get_letoh64(tvb, offset);
      msa = proto_tree_add_item(ti_tree, hf_ieee80211_vht_membership_status_array, tvb,
                                offset, 8, ENC_NA);
      msa_tree = proto_item_add_subtree(msa, ett_vht_msa);
      for (m = 0; m < 64; m++) {
          if (msa_value & (INT64_C(1) << m))
              proto_tree_add_uint_format(msa_tree, hf_ieee80211_vht_membership_status_field,
                                               tvb, offset + (m/8), 1, 1, "Membership Status in Group ID %d: 1", m);
      }
      offset += 8;

      upa = proto_tree_add_item(ti_tree, hf_ieee80211_vht_user_position_array, tvb,
                                      offset, 16, ENC_NA);
      upa_tree = proto_item_add_subtree(upa, ett_vht_upa);

      for (half = 0, msa_index = 0; half < 2; half++) {
          upa_value = tvb_get_letoh64(tvb, offset);
          for (m = 0; m < 64; m += 2, msa_index++) {
              if (msa_value & (INT64_C(1) << msa_index)) {
                  field_val = (uint8_t)((upa_value >> m) & 0x3);
                  proto_tree_add_uint_format(upa_tree, hf_ieee80211_vht_user_position_field,
                      tvb, offset + (m / 8), 1, field_val, "User Position in Group ID %d: %u", msa_index, field_val);
              }
          }
          offset += 8;
      }

      offset += tvb_reported_length_remaining(tvb, offset);
    }
    break;
    case VHT_ACT_OPERATION_MODE_NOTIFICATION:{
      ti = proto_tree_add_item(tree, hf_ieee80211_vht_operation_mode_notification, tvb,
                          offset, -1, ENC_NA);
      expert_add_info(pinfo, ti, &ei_ieee80211_vht_action);
      offset += tvb_reported_length_remaining(tvb, offset);
    }
    break;
    default:
    break;
  }


  return offset - start;
}

static int * const s1g_sync_control_headers[] = {
  &hf_ieee80211_s1g_sync_control_uplink_sync_request,
  &hf_ieee80211_s1g_sync_control_time_slot_protection_request,
  &hf_ieee80211_s1g_sync_control_reserved,
  NULL
};

static int
add_ff_sync_control(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  int start = offset;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_sync_control,
                                    ett_s1g_sync_control_tree,
                                    s1g_sync_control_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
  offset += 1;

  return offset - start;
}

static int * const s1g_sector_id_index_headers[] = {
  &hf_ieee80211_s1g_sector_id_preferred_sector_id,
  &hf_ieee80211_s1g_sector_id_snr,
  &hf_ieee80211_s1g_sector_id_receive_sector_bitmap,
  NULL
};

static void
s1g_sector_id_index_snr_custom(char *result, uint16_t snr)
{
  switch (snr) {
  case 0:
    snprintf(result, ITEM_LABEL_LENGTH, "%s", "Less than or equal to -3dB");
    break;
  case 30:
    snprintf(result, ITEM_LABEL_LENGTH, "%s", "Greater than or equal to 27dB");
    break;
  case 31:
    snprintf(result, ITEM_LABEL_LENGTH, "%s", "No Feedback");
    break;
  default:
    snprintf(result, ITEM_LABEL_LENGTH, "%ddB", snr - 3);
  }
}

static int
add_ff_sector_id_index(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  int start = offset;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_sector_id_index,
                                    ett_s1g_sector_id_index,
                                    s1g_sector_id_index_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
  offset += 2;

  return offset - start;
}

static int * const s1g_twt_information_control_headers[] = {
  &hf_ieee80211_s1g_twt_flow_identifier,
  &hf_ieee80211_s1g_twt_response_required,
  &hf_ieee80211_s1g_twt_next_twt_request,
  &hf_ieee80211_s1g_twt_next_twt_subfield_size,
  &hf_ieee80211_s1g_twt_reserved,
  NULL
};

static int
add_ff_twt_information(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  int start = offset;
  uint8_t control = tvb_get_uint8(tvb, offset);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_twt_information_control,
                                    ett_s1g_twt_information_control,
                                    s1g_twt_information_control_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
  offset += 1;

  if (control & 0x60) {
    int len_bits = ((control >> 5) & 0x03);
    int len = 0;

    switch (len_bits) {
    case 0:
      len = 0; /* Should not happen! */
      break;
    case 1:
      len = 4;
      proto_tree_add_item(tree, hf_ieee80211_s1g_twt_next_twt_32, tvb, offset,
                          len, ENC_LITTLE_ENDIAN);
      break;
    case 2:
      len = 6;
      proto_tree_add_item(tree, hf_ieee80211_s1g_twt_next_twt_48, tvb, offset,
                          len, ENC_LITTLE_ENDIAN);
      break;
    case 3:
      len = 8;
      proto_tree_add_item(tree, hf_ieee80211_s1g_twt_next_twt_64, tvb, offset,
                          len, ENC_LITTLE_ENDIAN);
      break;
    }

    offset += len;
  }

  return offset - start;
}

static unsigned
add_ff_s1g_twt_setup(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  const uint8_t ids[] = { TAG_TWT };
  unsigned start = offset;

  offset += add_tagged_field(pinfo, tree, tvb, offset, 0, ids, G_N_ELEMENTS(ids), NULL);

  return offset - start;
}

static int * const ieee80211_twt_individual_flow[] = {
  &hf_ieee80211_twt_individual_flow_id,
  &hf_ieee80211_twt_individual_reserved,
  &hf_ieee80211_twt_neg_type,
  &hf_ieee80211_twt_bcast_teardown_all_twt,
  NULL,
};
static int * const ieee80211_twt_bcast_flow[] = {
  &hf_ieee80211_twt_bcast_id,
  &hf_ieee80211_twt_neg_type,
  &hf_ieee80211_twt_bcast_teardown_all_twt,
  NULL,
};
static int * const ieee80211_twt_neg_type2[] = {
  &hf_ieee80211_twt_neg_type2_reserved1,
  &hf_ieee80211_twt_neg_type,
  &hf_ieee80211_twt_neg_type2_reserved2,
  NULL,
};
static int * const ieee80211_twt_teardown_all[] = {
  &hf_ieee80211_twt_bcast_twt_id_reserved,
  &hf_ieee80211_twt_bcast_neg_type_reserved,
  &hf_ieee80211_twt_bcast_teardown_all_twt,
  NULL,
};

static unsigned
add_ff_s1g_twt_teardown(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t twt_flow_id = tvb_get_uint8(tvb, offset);

  // Bit 7 is means Teardown All TWT, and the other fields are reserved
  if (twt_flow_id & 0x80) {
    proto_tree_add_bitmask(tree, tvb, offset,
                           hf_ieee80211_twt_bcast_teardown_all,
                           ett_twt_tear_down_tree,
                           ieee80211_twt_teardown_all,
                           ENC_NA);
    return 1;
  }

  // Bits 5 and 6 are the negotiation type - See ieee80211.ax/D3.0 9.6.25.9
  switch ((twt_flow_id & 0x60) >> 5) {
    case 2:
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                        hf_ieee80211_twt_bcast_flow,
                                        ett_twt_tear_down_tree, ieee80211_twt_neg_type2,
                                        ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
      break;

    case 3:
      // According to 11ax, first 5 bits are the BCAST TWT flow ID
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                        hf_ieee80211_twt_bcast_flow,
                                        ett_twt_tear_down_tree, ieee80211_twt_bcast_flow,
                                        ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
    break;
    case 0:
    case 1:
      // According to 11ah / 11ax, first 3 bits are the UCAST TWT flow ID
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                        hf_ieee80211_twt_individual_flow,
                                        ett_twt_tear_down_tree, ieee80211_twt_individual_flow,
                                        ENC_LITTLE_ENDIAN, BMT_NO_FLAGS);
      break;
    default:
      proto_tree_add_expert(tree, pinfo, &ei_ieee80211_twt_tear_down_bad_neg_type,
                            tvb, offset, tvb_reported_length_remaining(tvb, offset));
  }

  // The TWT Flow ID size
  return 1;
}

static unsigned
add_ff_action_s1g(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;
  uint8_t s1g_action;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  s1g_action = tvb_get_uint8(tvb, offset);
  offset += add_ff_s1g_action(tree, tvb, pinfo, offset);

  switch(s1g_action) {
  case S1G_ACT_AID_SWITCH_REQUEST:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    break;
  case S1G_ACT_AID_SWITCH_RESPONSE:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    break;
  case S1G_ACT_SYNC_CONTROL:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_sync_control(tree, tvb, pinfo, offset);
    break;
  case S1G_ACT_STA_INFO_ANNOUNCE:
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
     break;
  case S1G_ACT_EDCA_PARAM_SET:
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    break;
  case S1G_ACT_EL_OPERATION:
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    break;
  case S1G_ACT_TWT_SETUP:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_s1g_twt_setup(tree, tvb, pinfo, offset);
    break;
  case S1G_ACT_TWT_TEARDOWN:
    offset += add_ff_s1g_twt_teardown(tree, tvb, pinfo, offset);
    break;
  case S1G_ACT_SECT_GROUP_ID_LIST:
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
    break;
  case S1G_ACT_SECT_ID_FEEDBACK:
    offset += add_ff_sector_id_index(tree, tvb, pinfo, offset);
    break;
  case S1G_ACT_TWT_INFORMATION:
    offset += add_ff_twt_information(tree, tvb, pinfo, offset);
    break;
  default:
    break;
  }

  return offset - start;
}

static unsigned
add_ff_action_protected_s1g(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;
  uint8_t s1g_action;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  s1g_action = tvb_get_uint8(tvb, offset);
  offset += add_ff_prot_s1g_action(tree, tvb, pinfo, offset);

  switch(s1g_action) {
    case PROT_S1G_ACT_REACH_ADDR_UPDATE:
    case PROT_S1G_ACT_RELAY_ACTIVATE_REQ:
    case PROT_S1G_ACT_RELAY_ACTIVATE_RESP:
    case PROT_S1G_ACT_HEADER_COMPRESSION:
      // TODO
      break;
    case PROT_S1G_ACT_TWT_SETUP:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_s1g_twt_setup(tree, tvb, pinfo, offset);
      break;
    case PROT_S1G_ACT_TWT_TEARDOWN:
      offset += add_ff_s1g_twt_teardown(tree, tvb, pinfo, offset);
      break;
    case PROT_S1G_ACT_TWT_INFORMATION:
      offset += add_ff_twt_information(tree, tvb, pinfo, offset);
      break;
    case PROT_S1G_ACT_AID_SWITCH_REQUEST:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
      break;
    case PROT_S1G_ACT_AID_SWITCH_RESPONSE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
      break;
    case PROT_S1G_ACT_SYNC_CONTROL:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_sync_control(tree, tvb, pinfo, offset);
      break;
    case PROT_S1G_ACT_STA_INFO_ANNOUNCE:
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
      break;
    case PROT_S1G_ACT_EDCA_PARAM_SET:
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
      break;
    case PROT_S1G_ACT_EL_OPERATION:
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
      break;
    case PROT_S1G_ACT_SECT_GROUP_ID_LIST:
      offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
      break;
    case PROT_S1G_ACT_SECT_ID_FEEDBACK:
      offset += add_ff_sector_id_index(tree, tvb, pinfo, offset);
      break;
    default:
      break;
  }

  return offset - start;
}

#define HE_COMPRESSED_BEAMFORMING_AND_CQI 0
#define HE_QUIET_TIME_PERIOD              1

static const range_string he_action_rvals[] = {
  { HE_COMPRESSED_BEAMFORMING_AND_CQI, HE_COMPRESSED_BEAMFORMING_AND_CQI,
        "HE Compressed Beamforming And CQI" },
  { HE_QUIET_TIME_PERIOD, HE_QUIET_TIME_PERIOD,
        "Quiet Time Period" },
  { HE_QUIET_TIME_PERIOD + 1, 255, "Reserved" },
  { 0, 0, NULL }
};

#define HE_BSS_COLOR_CHANGE_ANNOUNCEMENT  0

static const range_string protected_he_action_rvals[] = {
  { HE_BSS_COLOR_CHANGE_ANNOUNCEMENT, HE_BSS_COLOR_CHANGE_ANNOUNCEMENT,
        "HE BSS Color Change Announcement" },
  { HE_BSS_COLOR_CHANGE_ANNOUNCEMENT + 1, 255, "Reserved" },
  { 0, 0, NULL }
};

#define EHT_TID_LINK_MAP_REQ            0
#define EHT_TID_LINK_MAP_RESP           1
#define EHT_TID_LINK_MAP_TEAR_DOWN      2
#define EHT_EPCS_PRIO_ACCESS_REQ        3
#define EHT_EPCS_PRIO_ACCESS_RESP       4
#define EHT_EPCS_PRIO_ACCESS_TEAR_DOWN  5
#define EHT_EML_OP_MODE_NOTIFICATION    6
#define EHT_LINK_RECOMMENDATION         7
#define EHT_MULTI_LINK_OP_UPDATE_REQ    8
#define EHT_MULTI_LINK_OP_UPDATE_RESP   9
#define EHT_LINK_RECONFIG_NOTIFY       10
#define EHT_LINK_RECONFIG_REQ          11
#define EHT_LINK_RECONFIG_RESP         12

static const range_string protected_eht_action_rvals[] = {
  { EHT_TID_LINK_MAP_REQ, EHT_TID_LINK_MAP_REQ,
        "EHT TID-to-Link Mapping Request" },
  { EHT_TID_LINK_MAP_RESP, EHT_TID_LINK_MAP_RESP,
        "EHT TID-to-Link Mapping Response" },
  { EHT_TID_LINK_MAP_TEAR_DOWN, EHT_TID_LINK_MAP_TEAR_DOWN,
        "EHT TID-to-Link Mapping Teardown" },
  { EHT_EPCS_PRIO_ACCESS_REQ, EHT_EPCS_PRIO_ACCESS_REQ,
        "EHT EPCS Priority Access Request" },
  { EHT_EPCS_PRIO_ACCESS_RESP, EHT_EPCS_PRIO_ACCESS_RESP,
        "EHT EPCS Priority Access Response" },
  { EHT_EPCS_PRIO_ACCESS_TEAR_DOWN, EHT_EPCS_PRIO_ACCESS_TEAR_DOWN,
        "EHT EPCS Priority Access Teardown" },
  { EHT_EML_OP_MODE_NOTIFICATION, EHT_EML_OP_MODE_NOTIFICATION,
        "EHT EML Operating Mode Notification" },
  { EHT_LINK_RECOMMENDATION, EHT_LINK_RECOMMENDATION,
        "EHT Link Recommendation" },
  { EHT_MULTI_LINK_OP_UPDATE_REQ, EHT_MULTI_LINK_OP_UPDATE_REQ,
        "EHT Multi-Link Operation Update Request" },
  { EHT_MULTI_LINK_OP_UPDATE_RESP, EHT_MULTI_LINK_OP_UPDATE_RESP,
        "EHT Multi-Link Operation Update Response" },
  { EHT_LINK_RECONFIG_NOTIFY, EHT_LINK_RECONFIG_NOTIFY,
        "EHT Link Reconfiguration Notify" },
  { EHT_LINK_RECONFIG_REQ, EHT_LINK_RECONFIG_REQ,
        "EHT Link Reconfiguration Request" },
  { EHT_LINK_RECONFIG_RESP, EHT_LINK_RECONFIG_RESP,
        "EHT Link Reconfiguration Response" },
  { EHT_LINK_RECONFIG_RESP + 1, 255, "Reserved" },
  { 0, 0, NULL }
};

/*
 * This currently only works for SU, 20MHz, 40MHz and 80MHz and grouping 4 and 16.
 */
struct scidx_start_end {
  int start;
  int end;
};

#define N_SCIDX_20MHZ_NG4 9
static const struct scidx_start_end scidx_20MHz_Ng4[N_SCIDX_20MHZ_NG4] = {
  { -122, -96 },
  { -96, -68 },
  { -68, -40 },
  { -44, -16 },
  { -16, 16 },
  { 16, 44 },
  { 40, 68 },
  { 68, 96 },
  { 96, 122 }
};

#define N_SCIDX_20MHZ_NG16 9
static const struct scidx_start_end scidx_20MHz_Ng16[9] = {
  { -122, -84 },
  { -96, -64 },
  { -80, -32 },
  { -52, -4 },
  { -20, 20 },
  { 4, 52 },
  { 32, 80 },
  { 64, 96 },
  { 84, 122 }
};

#define N_SCIDX_40MHZ_NG4 18
static const struct scidx_start_end scidx_40MHz_Ng4[N_SCIDX_40MHZ_NG4] = {
  { -500 + 256, -472 + 256 },
  { -476 + 256, -448 + 256 },
  { -488 + 256, -420 + 256 },
  { -420 + 256, -392 + 256 },
  { -392 + 256, -364 + 256 },
  { -368 + 256, -340 + 256 },
  { -340 + 256, -312 + 256 },
  { -312 + 256, -284 + 256 },
  { -288 + 256, -260 + 256 },
  {  260 - 256,  288 - 256 },
  {  284 - 256,  312 - 256 },
  {  312 - 256,  340 - 256 },
  {  340 - 256,  368 - 256 },
  {  364 - 256,  392 - 256 },
  {  392 - 256,  420 - 256 },
  {  420 - 256,  448 - 256 },
  {  448 - 256,  476 - 256 },
  {  472 - 256,  500 - 256 }
};

#define N_SCIDX_40MHZ_NG16 18
static const struct scidx_start_end scidx_40MHz_Ng16[N_SCIDX_40MHZ_NG16] = {
  { -500 + 256, -468 + 256 },
  { -484 + 256, -436 + 256 },
  { -452 + 256, -420 + 256 },
  { -420 + 256, -388 + 256 },
  { -404 + 256, -356 + 256 },
  { -372 + 256, -340 + 256 },
  { -340 + 256, -308 + 256 },
  { -324 + 256, -276 + 256 },
  { -292 + 256, -260 + 245 },
  {  260 - 256,  292 - 256 },
  {  276 - 256,  324 - 256 },
  {  308 - 256,  340 - 256 },
  {  340 - 256,  322 - 256 },
  {  356 - 256,  404 - 256 },
  {  388 - 256,  420 - 256 },
  {  420 - 256,  452 - 256 },
  {  436 - 256,  484 - 256 },
  {  468 - 256,  500 - 256 }
};

#define N_SCIDX_80MHZ_NG4 37
static const struct scidx_start_end scidx_80MHz_Ng4[N_SCIDX_80MHZ_NG4] = {
  { -500, -472 },
  { -476, -448 },
  { -448, -420 },
  { -420, -392 },
  { -392, -364 },
  { -368, -340 },
  { -340, -312 },
  { -312, -284 },
  { -288, -260 },
  { -260, -232 },
  { -232, -204 },
  { -204, -176 },
  { -180, -152 },
  { -152, -124 },
  { -124,  -96 },
  { -100,  -72 },
  {  -72,  -44 },
  {  -44,  -16 },
  {  -16,   16 },
  {   16,   44 },
  {   44,   72 },
  {   72,  100 },
  {   96,  124 },
  {  124,  152 },
  {  152,  180 },
  {  176,  204 },
  {  204,  232 },
  {  232,  260 },
  {  260,  288 },
  {  284,  312 },
  {  312,  340 },
  {  340,  368 },
  {  364,  392 },
  {  392,  420 },
  {  420,  448 },
  {  448,  476 },
  {  472,  500 }
};

#define N_SCIDX_80MHZ_NG16 37
static const struct scidx_start_end scidx_80MHz_Ng16[N_SCIDX_80MHZ_NG16] = {
  { -500, -468 },
  { -484, -436 },
  { -452, -420 },
  { -420, -388 },
  { -404, -356 },
  { -372, -340 },
  { -340, -308 },
  { -324, -276 },
  { -292, -260 },
  { -260, -228 },
  { -244, -196 },
  { -212, -164 },
  { -180, -148 },
  { -164, -116 },
  { -132,  -84 },
  { -100,  -68 },
  {  -84,  -36 },
  {  -52,   -4 },
  {  -20,   20 },
  {    4,   52 },
  {   36,   84 },
  {   68,  100 },
  {   84,  132 },
  {  116,  164 },
  {  148,  180 },
  {  164,  212 },
  {  196,  244 },
  {  228,  260 },
  {  260,  292 },
  {  276,  324 },
  {  308,  340 },
  {  340,  372 },
  {  356,  404 },
  {  388,  420 },
  {  420,  452 },
  {  436,  484 },
  {  468,  500 },
};

#define SU_FEEDBACK (0)
#define MU_FEEDBACK (1)
#define CQI_FEEDBACK (2)
#define RESERVED_FEEDBACK (3)

#define BW_20MHz (0)
#define BW_40MHz (1)
#define BW_80MHz (2)
#define BW_160MHz (3)

#define SCIDX_END_SENTINAL (0x80000000)

static int
next_he_scidx(int scidx, int bw _U_, int grouping _U_, int feedback _U_,
              int ru_start_index, int ru_end_index)
{
  int incr = 4;

  /*
   * We need to check the correct bw value to determine if we have hit
   * the end of the range of SCIDXes.
   */
  switch (bw) {
  case BW_20MHz:
    if (grouping == 0) {
      if (ru_end_index >= N_SCIDX_20MHZ_NG4 ||
          scidx == scidx_20MHz_Ng4[ru_end_index].end)  /* we returned the max */
        return SCIDX_END_SENTINAL;
    } else {
      if (ru_end_index >= N_SCIDX_20MHZ_NG16 ||
          scidx == scidx_20MHz_Ng16[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    }
    break;
  case BW_40MHz:
    if (grouping == 0) {
      if (ru_end_index >= N_SCIDX_40MHZ_NG4 ||
          scidx == scidx_40MHz_Ng4[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    } else {
      if (ru_end_index >= N_SCIDX_40MHZ_NG16 ||
          scidx == scidx_40MHz_Ng16[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    }
    break;
  case BW_80MHz:
    if (grouping == 0) {
      if (ru_end_index >= N_SCIDX_80MHZ_NG4 ||
          scidx == scidx_80MHz_Ng4[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    } else {
      if (ru_end_index >= N_SCIDX_80MHZ_NG16 ||
          scidx == scidx_80MHz_Ng16[ru_end_index].end)
        return SCIDX_END_SENTINAL;
    }
    break;
  case BW_160MHz:
    return SCIDX_END_SENTINAL;
  }

  /*
   * Check if this is the first time though and figure out the starting
   * SCIDX.
   */
  if (scidx == (int)SCIDX_END_SENTINAL)
    switch (bw) {
    case BW_20MHz:
      if (grouping == 0) {
        if (ru_start_index >= N_SCIDX_20MHZ_NG4)
          return SCIDX_END_SENTINAL;
        else
          return scidx_20MHz_Ng4[ru_start_index].start;
      } else {
        if (ru_start_index >= N_SCIDX_20MHZ_NG16)
          return SCIDX_END_SENTINAL;
        else
          return scidx_20MHz_Ng16[ru_start_index].start;
      }
    case BW_40MHz:
      if (grouping == 0) {
        if (ru_start_index >= N_SCIDX_40MHZ_NG4)
          return SCIDX_END_SENTINAL;
        else
          return scidx_40MHz_Ng4[ru_start_index].start;
      } else {
        if (ru_start_index >= N_SCIDX_40MHZ_NG16)
          return SCIDX_END_SENTINAL;
        else
          return scidx_40MHz_Ng16[ru_start_index].start;
      }
    case BW_80MHz:
      if (grouping == 0) {
        if (ru_start_index >= N_SCIDX_80MHZ_NG4)
          return SCIDX_END_SENTINAL;
        else
          return scidx_80MHz_Ng4[ru_start_index].start;
      } else {
        if (ru_start_index >= N_SCIDX_80MHZ_NG16)
          return SCIDX_END_SENTINAL;
        else
          return scidx_80MHz_Ng16[ru_start_index].start;
      }
    case BW_160MHz:
      return SCIDX_END_SENTINAL;
  }

  /*
   * for BW_20MHz it is more complex, and the start and end sets have an
   * increment of 2, and around zero they go -4, -2, 2, 4 as well.
   */
  if (bw == BW_20MHz) {
    if (grouping == 0) {
      if (scidx == -122)
        return -120;
      if (scidx == -4)
        return -2;
      if (scidx == 2)
        return 4;
      if (scidx == 120)
        return 122;
    } else {
      if (scidx == -122)
        return -116;
      if (scidx == -4)
        return -2;
      if (scidx == -2)
        return 2;
      if (scidx == 2)
        return 4;
      if (scidx == 116)
        return 122;
    }
  }

  if (grouping == 1)
    incr = 16;

  scidx += incr;

  if (scidx == 0)  /* Not sure if this is needed */
    scidx += incr;

  return scidx;
}

/*
 * This might have a problem if there are not enough bits in the TVB.
 * Will only handle a limited number of bits.
 */
static uint16_t
he_get_bits(tvbuff_t *tvb, int bit_offset, int bit_len)
{
  uint32_t bits;
  int byte_offset = bit_offset / 8;
  int bit_start = bit_offset % 8;
  int bit_mask = (1 << bit_len) - 1;  /* Select that many bits */
  int remaining_length = tvb_reported_length_remaining(tvb, byte_offset);

  if (remaining_length >= 3)
    bits = tvb_get_letoh24(tvb, byte_offset);
  else if (remaining_length == 2)
    bits = tvb_get_letohs(tvb, byte_offset);
  else
    bits = tvb_get_uint8(tvb, byte_offset);

  bits = bits >> bit_start;

  return bits & bit_mask;
}

static int
dissect_he_feedback_matrix(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                           int offset, int bit_offset, int scidx,
                           int nr, int nc, int phi_bits, int psi_bits,
                           int hf)
{
  int ri, ci;
  int start_bit_offset = bit_offset;
  int start_offset = offset;
  wmem_strbuf_t *angles = wmem_strbuf_new(pinfo->pool, NULL);

  if (nc == nr)  /* If they are the same, reduce Nc by one */
    nc -= 1;

  wmem_strbuf_append_printf(angles, "%d", scidx);
  /* Reset to the start bit offset */
  bit_offset = start_bit_offset;

  for (ci = 1; ci <= nc; ci++) {
    for (ri = ci; ri < nr; ri++) {
      int angle = he_get_bits(tvb, bit_offset, phi_bits);
      wmem_strbuf_append_printf(angles, ", φ%d%d:%d", ri, ci, angle);
      bit_offset += phi_bits;
    }
    for (ri = ci + 1; ri <= nr; ri++) {
      int angle = he_get_bits(tvb, bit_offset, psi_bits);
      wmem_strbuf_append_printf(angles, ", ψ%d%d:%d", ri, ci, angle);
      bit_offset += psi_bits;
    }
  }

  /* Update this */
  proto_tree_add_string(tree, hf, tvb, offset,
                        ((start_bit_offset + 7) / 8) - start_offset,
                        wmem_strbuf_get_str(angles));

  return bit_offset;
}

static int * const he_mimo_control_headers[] = {
  &hf_ieee80211_he_mimo_control_nc_index,
  &hf_ieee80211_he_mimo_control_nr_index,
  &hf_ieee80211_he_mimo_control_bw,
  &hf_ieee80211_he_mimo_control_grouping,
  &hf_ieee80211_he_mimo_control_codebook_info,
  &hf_ieee80211_he_mimo_control_feedback_type,
  &hf_ieee80211_he_mimo_control_remaining_feedback_segs,
  &hf_ieee80211_he_mimo_control_first_feedback_seg,
  &hf_ieee80211_he_mimo_control_ru_start_index,
  &hf_ieee80211_he_mimo_control_ru_end_index,
  &hf_ieee80211_he_mimo_control_sounding_dialog_token_num,
  &hf_ieee80211_he_mimo_control_reserved,
  NULL
};

/*
 * Handle compressed beamforming matrices and CQI
 */
static unsigned
dissect_compressed_beamforming_and_cqi(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  int byte_count = 0;
  uint64_t mimo_cntl = tvb_get_letoh40(tvb, offset);
  int nc = 0, nr = 0, i;
  int bw, grouping, codebook, feedback, bit_offset, scidx;
  int phi_bits = 0, psi_bits = 0;
  proto_tree *snr_tree = NULL, *feedback_tree = NULL;
  int start_offset;
  int ru_start_index, ru_end_index;

  nc = (int)((mimo_cntl & 0x07) + 1);
  nr = (int)(((mimo_cntl >> 3) & 0x07) + 1);
  bw = (int)((mimo_cntl >> 6) & 0x03);
  grouping = (int)((mimo_cntl >> 8) & 0x01);
  codebook = (int)((mimo_cntl >> 9) & 0x01);
  feedback = (int)((mimo_cntl >> 10) & 0x03);
  ru_start_index = (int)((mimo_cntl >> 16) & 0x7F);
  ru_end_index = (int)((mimo_cntl >> 23) & 0x7F);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
    hf_ieee80211_he_mimo_control_field, ett_ff_he_mimo_control,
    he_mimo_control_headers, ENC_LITTLE_ENDIAN,
    BMT_NO_APPEND);
  offset += 5;

  snr_tree = proto_tree_add_subtree(tree, tvb, offset, nc,
                ett_ff_he_mimo_beamforming_report_snr, NULL,
                "Average Signal to Noise Ratio");

  for (i = 0; i < nc; i++) {
    int8_t snr = tvb_get_int8(tvb, offset);

    proto_tree_add_int_format(snr_tree,
        hf_ieee80211_he_compressed_beamforming_report_snr, tvb, offset, 1,
        snr, "Stream %d: %s%0.2fdB (0x%02x)", i, (snr == 127 ? ">=" :
                                                  (snr == -128 ? "<=" : "")),
                                             (float)((float)88 + snr)/4,
                                             (uint8_t)snr);
    offset++;
  }

  /*
   * The rest of the data consists of the compressed beamforming matrices, one
   * for each SCIDX per group. Each matrix consists of phi and psi angles
   * encoded using the number of bits specified using the codebook field.
   *
   * The matrices contain a number entries related to Nr -1 & Nc except when
   * Nr == Nc, and then it is Nr -1 x Nc -1, with Nr - 1 phi angles, Nc - 1
   * psi angles, Nr - 2 phi angles, Nc - 2 psi angles ...
   */
  if (feedback == 0) {  /* SU */
    if (codebook == 0) {
      psi_bits = 2; phi_bits = 4;
    } else {
      psi_bits = 4; phi_bits = 6;
    }
  } else if (feedback == 1) { /* MU */
    if (grouping == 1) {
      psi_bits = 9; phi_bits = 7;
    } else {
      if (codebook == 0) {
        psi_bits = 5; phi_bits = 7;
      } else {
        psi_bits = 7; phi_bits = 9;
      }
    }
  }  /* DO something about CQI etc. */

  feedback_tree = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_ff_he_mimo_feedback_matrices, NULL,
                        "Feedback Matrices");

  start_offset = offset;
  bit_offset = offset * 8;
  scidx = SCIDX_END_SENTINAL;
  while ((scidx = next_he_scidx(scidx, bw, grouping, feedback,
          ru_start_index, ru_end_index)) != (int)SCIDX_END_SENTINAL) {
    int prev_bit_offset = bit_offset;
    bit_offset = dissect_he_feedback_matrix(feedback_tree, tvb, pinfo, offset,
                        bit_offset, scidx, nr, nc, phi_bits, psi_bits,
                        hf_ieee80211_he_compressed_beamform_scidx);
    if (bit_offset <= prev_bit_offset) {
      expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
      break;
    }

    offset = bit_offset / 8;
  }

  offset = (bit_offset + 7) / 8;
  proto_item_set_len(feedback_tree, offset - start_offset);

  /* Sometimes the FCS is in the buffer as well ... */
  byte_count = tvb_reported_length_remaining(tvb, offset);
  if (byte_count > 0)
    offset += byte_count; /* Should fix the real problem */

  return offset;
}

enum quiet_time_period_control {
  QTP_CONTROL_SETUP = 0,
  QTP_CONTROL_REQUEST = 1,
  QTP_CONTROL_RESPONSE = 2,
};

static const range_string quiet_time_period_control_rvals[] = {
  { QTP_CONTROL_SETUP,    QTP_CONTROL_SETUP,    "Quiet Time Period Setup" },
  { QTP_CONTROL_REQUEST,  QTP_CONTROL_REQUEST,  "Quiet Time Period Request" },
  { QTP_CONTROL_RESPONSE, QTP_CONTROL_RESPONSE, "Quiet Time Period Response" },
  { QTP_CONTROL_RESPONSE +1 , 255, "Reserved" },
  { 0, 0, NULL }
};

static void
dissect_bss_color_change(tvbuff_t *tvb, packet_info *pinfo _U_,
proto_tree *tree, int offset, int len _U_);

static unsigned
dissect_quiet_time_period(tvbuff_t *tvb, packet_info *pinfo _U_,
proto_tree *tree, int offset, int len _U_)
{
  uint8_t control = tvb_get_uint8(tvb, offset);

  proto_tree_add_item(tree, hf_ieee80211_he_qtp_control, tvb, offset, 1, ENC_NA);
  offset += 1;

  switch (control)
  {
  case QTP_CONTROL_SETUP:
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_setup_quiet_period_duration, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_setup_srv_specific_identif, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    break;

  case QTP_CONTROL_REQUEST:
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_request_dialog_token, tvb, offset, 2,
                ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_request_quiet_period_offset, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_request_quiet_period_duration, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_request_quiet_period_interval, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_request_repetition_count, tvb, offset, 1,
                ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_request_srv_specific_identif, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    break;

  case QTP_CONTROL_RESPONSE:
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_response_dialog_token, tvb, offset, 2,
                ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_response_status_code, tvb, offset, 1,
                ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_response_quiet_period_offset, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_response_quiet_period_duration, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_response_quiet_period_interval, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_response_repetition_count, tvb, offset,
                1, ENC_NA);
    offset += 1;
    proto_tree_add_item(tree, hf_ieee80211_he_qtp_response_srv_specific_identif, tvb, offset,
                2, ENC_LITTLE_ENDIAN);
    offset += 2;
    break;

  default:
    /* Reserved */
    break;
  }

  return offset;
}

static unsigned
add_ff_action_he(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;
  uint8_t he_action;
  uint8_t length;
  //uint8_t elem_id, elem_id_extension;
  proto_item *item;
  proto_tree *subtree;
  unsigned int len = tvb_reported_length_remaining(tvb, offset);

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  item = proto_tree_add_item_ret_uint8(tree, hf_ieee80211_ff_he_action, tvb, offset, 1, ENC_NA, &he_action);
  offset += 1;

  subtree = proto_item_add_subtree(item, ett_ff_he_action);


  switch (he_action) {
  case HE_COMPRESSED_BEAMFORMING_AND_CQI:
    proto_tree_add_uint_format(subtree, hf_ieee80211_he_beamforming_report_len,
        tvb, offset, 0, len, "Total length: %u", len);
    offset = dissect_compressed_beamforming_and_cqi(subtree, tvb, pinfo, offset);
    break;

  case HE_QUIET_TIME_PERIOD:
    //elem_id = tvb_get_uint8(tvb, offset);
    length = tvb_get_uint8(tvb, offset + 1);
    //elem_id_extension = tvb_get_uint8(tvb, offset + 2);
    /* Should check following condition?
     * elem_id == TAG_ELEMENT_ID_EXTENSION && elem_id_extension == ETAG_QUIET_TIME_PERIOD
     */
    offset = dissect_quiet_time_period(tvb, pinfo, subtree, offset + 3, length);
    break;

  default:
    break;
  }
  return offset - start;
}

static unsigned
add_ff_action_protected_he(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;
  uint8_t protected_he_action;
  uint8_t length;
  //uint8_t elem_id, elem_id_extension;
  proto_item *item;
  proto_tree *subtree;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  protected_he_action = tvb_get_uint8(tvb, offset);

  item = proto_tree_add_item(tree, hf_ieee80211_ff_protected_he_action, tvb, offset,
        1, ENC_NA);
  offset += 1;

  subtree = proto_item_add_subtree(item, ett_ff_protected_he_action);


  switch (protected_he_action) {
  case HE_BSS_COLOR_CHANGE_ANNOUNCEMENT:
    //elem_id = tvb_get_uint8(tvb, offset);
    length = tvb_get_uint8(tvb, offset + 1);
    //elem_id_extension = tvb_get_uint8(tvb, offset + 2);
    /* Should check following condition?
     * elem_id == TAG_ELEMENT_ID_EXTENSION && elem_id_extension == ETAG_BSS_COLOR_CHANGE_ANNOUNCEMENT
     */
    dissect_bss_color_change(tvb, pinfo, subtree, offset + 3, length);
    offset += 5;
    break;

  default:
    break;
  }

  return offset - start;
}

static unsigned
add_ff_action_protected_ftm(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;
  uint8_t action;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  proto_tree_add_item_ret_uint8(tree, hf_ieee80211_ff_protected_ftm_action, tvb, offset, 1, ENC_NA, &action);
  offset += 1;

  col_set_str(pinfo->cinfo, COL_INFO, val_to_str_const(action, protected_ftm_action_vals, "Unknown"));

  switch (action) {
    case 1:
      offset += add_ff_ftm_request(tree, tvb, pinfo, offset);
      break;
    case 2:
      offset += add_ff_ftm(tree, tvb, pinfo, offset);
      break;
    case 3:
      offset += add_ff_lmr_report(tree, tvb, pinfo, offset);
      break;
    default:  /* reserved */
      break;
  }

  return offset - start;
}

static int * const eht_reconfig_link_id_hdrs[] = {
  &hf_ieee80211_eht_reconfig_link_id,
  &hf_ieee80211_eht_reconfig_link_id_reserved,
  NULL
};

static unsigned
add_ff_count(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_count, tvb, offset, 1,
                      ENC_LITTLE_ENDIAN);
  return 1;
}

static unsigned
add_ff_action_protected_eht(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  unsigned start = offset;
  uint8_t protected_eht_action;
  proto_item *item;
  int len = 0, count, i, id;
  uint16_t status;
  bool invalid = false;
  /* Default Extension Element is Multi-Link */
  uint8_t ext_ids[1] = {ETAG_MULTI_LINK};
  proto_tree *sub_tree;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  protected_eht_action = tvb_get_uint8(tvb, offset);

  item = proto_tree_add_item(tree, hf_ieee80211_ff_protected_eht_action, tvb,
                             offset, 1, ENC_NA);
  offset += 1;

  switch (protected_eht_action) {
  case EHT_TID_LINK_MAP_REQ:
    ext_ids[0] = ETAG_TID_TO_LINK_MAPPING;
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

    /*
     * Next contains one or two TID-To-Link mappings. We have to look into
     * the elements to verify them because we don't have the length of this
     * element and there might be other IEs after us.
     */
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    } else {
      invalid = true;
    }
    if (!invalid && (len >= 1) &&
        (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, NULL,
         0, false, ext_ids, G_N_ELEMENTS(ext_ids), false, NULL) > 0)) {
      offset += len + 2;
    } else {
      invalid = true;
    }
    if (invalid) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                             "Invalid TID_TO_LINK_MAPPING_REQUEST. "
                             "There should be one or two Tid-To-Link IEs "
                             "but none found");
      break;
    }

    len = 0;
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    }
    if ((len >= 1) &&
        (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, NULL,
         0, false, ext_ids, G_N_ELEMENTS(ext_ids), false, NULL) > 0)) {
      offset += len + 2;
    }
    break;
  case EHT_TID_LINK_MAP_RESP:
    ext_ids[0] = ETAG_TID_TO_LINK_MAPPING;
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);

    status = tvb_get_ntohs(tvb, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    if (status == 134) {
      /* There should be one or two TID-To-Link mappings. */
      if (tvb_captured_length_remaining(tvb, offset) >= 2) {
        len = tvb_get_uint8(tvb, offset + 1);
      } else {
        invalid = true;
      }
      if (!invalid && (len >= 1) &&
          (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, NULL,
           0, false, ext_ids, G_N_ELEMENTS(ext_ids), false, NULL) > 0)) {
        offset += len + 2;
      } else {
        invalid = true;
      }
      if (invalid) {
        expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                               "Invalid TID_TO_LINK_MAPPING_RESPONSE. "
                               "There should be one or two Tid-To-Link IEs "
                               "but none found");
        break;
      }

      len = 0;
      if (tvb_captured_length_remaining(tvb, offset) >= 2) {
        len = tvb_get_uint8(tvb, offset + 1);
      }
      if ((len >= 1) &&
          (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, NULL,
           0, false, ext_ids, G_N_ELEMENTS(ext_ids), false, NULL) > 0)) {
        offset += len + 2;
      }
    }
    break;
  case EHT_TID_LINK_MAP_TEAR_DOWN:
    /* Seems to be nothing to do here */
    break;
  case EHT_EPCS_PRIO_ACCESS_REQ:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      offset += add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0,
                NULL, 0, FALSE, ext_ids, G_N_ELEMENTS(ext_ids), FALSE, NULL);
    }
    break;
  case EHT_EPCS_PRIO_ACCESS_RESP:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      offset += add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0,
                NULL, 0, FALSE, ext_ids, G_N_ELEMENTS(ext_ids), FALSE, NULL);
    }
    break;
  case EHT_EPCS_PRIO_ACCESS_TEAR_DOWN:
    break;
  case EHT_EML_OP_MODE_NOTIFICATION:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += dissect_eht_eml_control_field(tree, tvb, pinfo, offset);
    break;
  case EHT_LINK_RECOMMENDATION:
    offset += add_ff_reason_code(tree, tvb, pinfo, offset);
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    } else {
      invalid = true;
    }
    ext_ids[0] = ETAG_AID_BITMAP;
    if (!invalid && (len >= 1) &&
        (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, NULL,
         0, false, ext_ids, G_N_ELEMENTS(ext_ids), false, NULL) > 0)) {
      offset += len + 2;
    } else {
      invalid = true;
    }
    if (invalid) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                             "Invalid LINK_RECOMMENDATION. "
                             "There should be AID_BITMAP element "
                             "but none found");
      break;
    }

    invalid = false;
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    } else {
      invalid = true;
    }
    ext_ids[0] = ETAG_MULTI_LINK_TRAFFIC;
    if (!invalid && (len >= 1) &&
        (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, NULL,
         0, false, ext_ids, G_N_ELEMENTS(ext_ids), false, NULL) > 0)) {
      offset += len + 2;
    } else {
      invalid = true;
    }
    if (invalid) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                             "Invalid LINK_RECOMMENDATION. "
                             "There should be MULTI_LINK_TRAFFIC element "
                             "but none found");
      break;
    }
    break;
  case EHT_MULTI_LINK_OP_UPDATE_REQ:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    } else {
      invalid = true;
    }
    if (!invalid && (len >= 1)) {
      offset += add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0,
                NULL, 0, FALSE, ext_ids, G_N_ELEMENTS(ext_ids), FALSE, NULL);
    } else {
      invalid = true;
    }
    if (invalid) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                             "Invalid Multi-Link Operation Update Request. "
                             "There should be Reconf Multi-Link element "
                             "but none found");
      break;
    }
    break;
  case EHT_MULTI_LINK_OP_UPDATE_RESP:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    offset += add_ff_status_code(tree, tvb, pinfo, offset);
    break;
  case EHT_LINK_RECONFIG_NOTIFY:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    } else {
      invalid = TRUE;
    }
    if (!invalid && (len >= 1)) {
      offset += add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0,
                NULL, 0, FALSE, ext_ids, G_N_ELEMENTS(ext_ids), FALSE, NULL);
    } else {
      invalid = TRUE;
    }
    if (invalid) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                             "Invalid Link Reconfiguration Notify. "
                             "There should be Reconf Multi-Link element "
                             "but none found");
      break;
    }
    break;
  case EHT_LINK_RECONFIG_REQ:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    } else {
      invalid = TRUE;
    }
    if (!invalid && (len >= 1)) {
      offset += add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0,
                NULL, 0, FALSE, ext_ids, G_N_ELEMENTS(ext_ids), FALSE, NULL);
    } else {
      invalid = TRUE;
    }
    if (invalid) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                             "Invalid Link Reconfiguration Req. "
                             "There should be Reconf Multi-Link element "
                             "but none found");
      break;
    }
    ext_ids[0] = ETAG_OCI;
    if (tvb_captured_length_remaining(tvb, offset) >= 2) {
      len = tvb_get_uint8(tvb, offset + 1);
    } else {
      /* OCI element field is optionally present */
      break;
    }
    if (!invalid && (len >= 1) &&
        (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, NULL,
         0, FALSE, ext_ids, G_N_ELEMENTS(ext_ids), FALSE, NULL) > 0)) {
      offset += len + 2;
    } else {
      invalid = TRUE;
    }
    if (invalid) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                             "Invalid Link Reconfiguration Req. "
                             "There should be OCI element but none found");
      break;
    }
    break;
  case EHT_LINK_RECONFIG_RESP:
    offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
    count = tvb_get_uint8(tvb, offset);
    offset += add_ff_count(tree, tvb, pinfo, offset);
    if (count == 0)
      break;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, count * 3,
                                      ett_eht_reconfig_status_list, NULL,
                                      "Reconfiguration Status List");
    for (i = 0; i < count; i++) {
      proto_tree_add_bitmask(sub_tree, tvb, offset,
                             hf_ieee80211_eht_reconfig_link_id_info,
                             ett_eht_multi_link_common_info_link_id,
                             eht_reconfig_link_id_hdrs, ENC_NA);
      offset += 1;
      proto_tree_add_item(sub_tree, hf_ieee80211_eht_reconfig_status_code, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
    }

    if (tvb_captured_length_remaining(tvb, offset) <= 0)
      break;

    /* Check next is Group Key Data */
    id = tvb_get_uint8(tvb, offset);
    if (id != 0xdd && id != 0xff) {
      sub_tree = proto_tree_add_subtree(tree, tvb, offset, id,
                                        ett_eht_group_key_data, NULL,
                                        "Group Key Data");
      proto_tree_add_item(sub_tree, hf_ieee80211_eht_group_key_data_length, tvb, offset, 1, ENC_NA);
      offset += 1;
      while (tvb_reported_length_remaining(tvb, offset)) {
        id = tvb_get_uint8(tvb, offset);
        /* Key Data should be MLO KDEs */
        if (id != 0xdd)
          break;
        offset += add_tagged_field(pinfo, sub_tree, tvb, offset, 0, NULL, 0, NULL);
      }
    }
    break;
  default:
    expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                           "Reserved Protected EHT Action %u",
                           protected_eht_action);

    break;
  }

  return offset - start;
}

/*
 * Structure for handling the EHT scidx info. Allows for a compact
 * representation.
 */
struct scidx_part {
  unsigned start_val;      /* What we start at                 */
  bool use_ng;      /* Should we use Ng or the inc next */
  unsigned inc;            /* The increment when not using Ng  */
  unsigned stop_val;       /* When we should stop              */
  bool last_ent;    /* This is the last one in the list */
};

struct scidx_ctx {
  uint8_t ru_index;
  const struct scidx_part *scidx_array;
  uint8_t ng;
  bool just_inited;
  unsigned last_val;
};

static const struct scidx_part ru_242_tone_1_20MHz_ng4[] = {
  { -122, false, 2, -120, false },
  { -120, true,  0,   -4, false },
  {   -2, false, 4,    2, false },
  {    4, true,  0,  120, false },
  {  120, false, 2,  122, true  }
};

static const struct scidx_part ru_242_tone_1_20MHz_ng16[] = {
  { -122, false,  6, -116, false },
  { -116, true,   0,   -4, false },
  {   -2, false,  4,    2, false },
  {    4, true,   0,  116, false },
  {  116, false,  6,  122, true  }
};

/* Here, there is one per RU index */
/*Start, UseNg, Inc,End, last */
static const struct scidx_part ru_242_tone_40MHz[] = {
  { -244, true, 0,   -4, true },
  {    4, true, 0,  244, true }
};

static const struct scidx_part ru_242_tone_80MHz[] = {
  { -500, true, 0, -260, true },
  { -252, true, 0,  -12, true },
  {   12, true, 0,  252, true },
  {  260, true, 0,  500, true }
};

static const struct scidx_part ru_242_tone_160MHz[] = {
  { -1012, true, 0, -772, true },
  {  -764, true, 0, -524, true },
  {  -500, true, 0, -260, true },
  {  -252, true, 0,  -12, true },
  {    12, true, 0,  252, true },
  {   260, true, 0,  500, true },
  {   524, true, 0,  764, true },
  {   772, true, 0, 1012, true }
};

static const struct scidx_part ru_242_tone_320MHz[] = {
  { -2036, true, 0, -1796, true },
  { -1788, true, 0, -1548, true },
  { -1524, true, 0, -1284, true },
  { -1276, true, 0, -1036, true },
  { -1012, true, 0,  -772, true },
  {  -764, true, 0,  -524, true },
  {  -500, true, 0,  -260, true },
  {  -252, true, 0,   -12, true },
  {    12, true, 0,   252, true },
  {   260, true, 0,   500, true },
  {   524, true, 0,   764, true },
  {   772, true, 0,  1012, true },
  {  1036, true, 0,  1276, true },
  {  1284, true, 0,  1524, true },
  {  1548, true, 0,  1788, true },
  {  1796, true, 0,  2036, true }
};

/* All these ru_96 tone sets for NG=4 go in pairs. */
static const struct scidx_part ru_996_tone_80MHz_ng4[] = {
  { -500, false, 4,  -4, false },
  {    4, false, 4, 500, true }
};

static const struct scidx_part ru_996_tone_80MHz_ng16[] = {
  { -500,  true, 0, -260, false },
  { -252,  true, 0,  -12, false },
  {   -4, false, 8,    4, false },
  {   12,  true, 0,  252, false },
  {  260,  true, 0,  500,  true }
};

static const struct scidx_part ru_996_tone_160MHz_ng4[] = {
  { -1012, true, 0, -516, false },
  {  -508, true, 0,  -12,  true },

  {    12, true, 0,  508, false },
  {   516, true, 0, 1012,  true }
};

static const struct scidx_part ru_996_tone_160MHz_ng16[] = {
  { -1012,  true, 0, -772, false },
  {  -764,  true, 0, -524, false },
  {  -516, false, 8, -508, false },
  {  -500,  true, 0, -260, false },
  {  -252,  true, 0, -12 ,  true },

  {    12,  true, 0,  252, false },
  {   260,  true, 0,  500, false },
  {   508, false, 8,  516, false },
  {   524,  true, 0,  764, false },
  {   772,  true, 0, 1012,  true }
};

static const struct scidx_part ru_996_tone_320MHz_ng4[] = {
  { -2036,  true, 0, -1540, false },
  { -1532,  true, 0, -1036,  true },

  { -1012,  true, 0,  -516, false },
  {  -508,  true, 0,   -12,  true },

  {    12,  true, 0,   508, false },
  {   516,  true, 0,  1012,  true },

  {  1036,  true, 0,  1532, false },
  {  1540,  true, 0,  2036,  true }
};

static const struct scidx_part ru_996_tone_320MHz_ng16[] = {
  { -2036,  true, 0, -1796, false },
  { -1788,  true, 0, -1548, false },
  { -1540, false, 8, -1532, false },
  { -1524,  true, 0, -1284, false },
  { -1276,  true, 0, -1036,  true },

  { -1012,  true, 0,  -772, false },
  {  -764,  true, 0,  -524, false },
  {  -516, false, 8,  -508, false },
  {  -500,  true, 0,  -260, false },
  {  -252,  true, 0,   -12,  true },

  {    12,  true, 0,   252, false },
  {   260,  true, 0,   500, false },
  {   508, false, 8,   516, false },
  {   524,  true, 0,   764, false },
  {   772,  true, 0,  1012,  true },

  {  1036,  true, 0,  1276, false },
  {  1284,  true, 0,  1524, false },
  {  1532, false, 8,  1540, false },
  {  1548,  true, 0,  1788, false },
  {  1796,  true, 0,  2036,  true }
};


static void
init_eht_scidx(struct scidx_ctx *ctx, uint8_t ru_index,
               const struct scidx_part *scidx_array, uint8_t ng)
{
  ctx->ru_index = ru_index;
  ctx->scidx_array = scidx_array;
  ctx->ng = ng;
  ctx->just_inited = true;
}

/* What about the special 20MHz ones? */
/* Figure out the next SCIDX */
static bool
next_eht_scidx(struct scidx_ctx *ctx, unsigned *scidx)
{
  if (ctx->just_inited) {
    ctx->last_val = ctx->scidx_array->start_val;
    ctx->just_inited = false;
    *scidx = ctx->last_val;
    return true;
  }

  /* Move to the next val ... but check if it is a short seg first */
  if (ctx->last_val == ctx->scidx_array->stop_val) {
    if (ctx->scidx_array->last_ent) {
        return false;
    } else {
        /* Pretend like we just started again */
        /* Also, note that the arrays need to be set up correctly */
        ctx->scidx_array++;
        if (ctx->last_val == ctx->scidx_array->start_val) {
          if (ctx->scidx_array->use_ng) {
            ctx->last_val += ctx->ng;
          } else {
            ctx->last_val += ctx->scidx_array->inc;
          }
        } else {
          ctx->last_val = ctx->scidx_array->start_val;
        }
        *scidx = ctx->last_val;
        return true;
    }
  }

  /*
   * If the increment is not ng, then handle that.
   */
  if (ctx->scidx_array->use_ng) {
    ctx->last_val += ctx->ng;
  } else {
    ctx->last_val += ctx->scidx_array->inc;
  }
  *scidx = ctx->last_val;

  return true;
}

static int
add_ff_eht_mu_exclusive_20MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
                                  int offset, uint8_t nc_index,
                                  uint8_t grouping, uint16_t partial_bw_info _U_)
{
  int start_offset = offset;
  unsigned scidx;
  struct scidx_ctx scidx_ctx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;
  unsigned ss = 0, get_snr = 1;
  int8_t snr = 0, value;

  if (ng == 4) {
    init_eht_scidx(&scidx_ctx, 1, &ru_242_tone_1_20MHz_ng4[0], ng);
  } else {
    init_eht_scidx(&scidx_ctx, 1, &ru_242_tone_1_20MHz_ng16[0], ng);
  }

  ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                &ruii,
                                                "RU Index %d for 20MHz, Partial BW "
                                                "not all one.", 1);
  while (next_eht_scidx(&scidx_ctx, &scidx)) {
    for (ss = 1; ss <= nc_index; ss++) {
      if (get_snr) {
        snr = tvb_get_int8(tvb, offset);
        offset += 1;
        value = snr & 0x0f;
        get_snr = 0;
      } else {
        value = snr >> 4;
        get_snr = 1;
      }
      proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                ss, scidx, value);
    }
  }
  proto_item_set_len(ruii, offset - start_offset);
  return offset - start_offset;
}

static int
add_ff_eht_mu_exclusive_40MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
                                  int offset, uint8_t nc_index,
                                  uint8_t grouping, uint16_t partial_bw_info)
{
  int start_offset = offset;
  int i = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;
  unsigned ss = 0, get_snr = 1;
  int8_t snr = 0, value;

  /* Add each of the RU index groups set */
  for (i = 1; i <= 2; i++) {
    int tree_offset = offset;
    if ((partial_bw_info & (1 << i)) == 0x0) {
      continue;  /* Only dissect those with the bit set */
    }

    init_eht_scidx(&scidx_ctx, i, &ru_242_tone_40MHz[i - 1], ng);
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "RU Index %d for 40MHz, Partial BW "
                                                  "not all one.", i);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      for (ss = 1; ss <= nc_index; ss++) {
        if (get_snr) {
          snr = tvb_get_int8(tvb, offset);
          offset += 1;
          value = snr & 0x0f;
          get_snr = 0;
        } else {
          value = snr >> 4;
          get_snr = 1;
        }
        proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                  tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                  ss, scidx, value);
      }
    }
    proto_item_set_len(ruii, offset - tree_offset);
  }

  return offset - start_offset;
}

static int
add_ff_eht_mu_exclusive_80MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
                                  int offset, uint8_t nc_index,
                                  uint8_t grouping, uint16_t partial_bw_info)
{
  int start_offset = offset;
  int i = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;
  unsigned ss = 0, get_snr = 1;
  int8_t snr = 0, value;

  if (partial_bw_info == 0x1E) { /* Uses 996-tone RU indices */
    if (ng == 4) {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_80MHz_ng4[0], ng);
    } else {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_80MHz_ng16[0], ng);
    }
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "996-tone RU Index %d for 80MHz", 1);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      for (ss = 1; ss <= nc_index; ss++) {
        if (get_snr) {
          snr = tvb_get_int8(tvb, offset);
          offset += 1;
          value = snr & 0x0f;
          get_snr = 0;
        } else {
          value = snr >> 4;
          get_snr = 1;
        }
        proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                  tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                  ss, scidx, value);
      }
    }
    proto_item_set_len(ruii, offset - start_offset);

    return offset - start_offset;
  }

  /* Add each of the RU index groups */
  for (i = 1; i <= 4; i++) {
    int tree_offset = offset;
    if ((partial_bw_info & (1 << i)) == 0x0) {
      continue;  /* Only dissect those with the bit set */
    }

    init_eht_scidx(&scidx_ctx, i, &ru_242_tone_80MHz[i - 1], ng);
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "RU Index %d for 80MHz, Partial BW "
                                                  "not all one.", i);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      for (ss = 1; ss <= nc_index; ss++) {
        if (get_snr) {
          snr = tvb_get_int8(tvb, offset);
          offset += 1;
          value = snr & 0x0f;
          get_snr = 0;
        } else {
          value = snr >> 4;
          get_snr = 1;
        }
        proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                  tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                  ss, scidx, value);
      }
    }
    proto_item_set_len(ruii, offset - tree_offset);
  }

  return offset - start_offset;
}

static int
add_ff_eht_mu_exclusive_160MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
                                   int offset, uint8_t nc_index,
                                   uint8_t grouping, uint16_t partial_bw_info)
{
  int start_offset = offset;
  int tree_offset;
  int i = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;
  unsigned ss = 0, get_snr = 1;
  int8_t snr = 0, value;

  /* Is the first lot a 996-tone RU? */
  if ((partial_bw_info & 0x1E) == 0x1E) { /* Uses 996-tone RU indices */
    tree_offset = offset;
    if (ng == 4) {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng4[0], ng);
    } else {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng16[0], ng);
    }
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "996-tone RU Index %d for 160MHz", 1);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      for (ss = 1; ss <= nc_index; ss++) {
        if (get_snr) {
          snr = tvb_get_int8(tvb, offset);
          offset += 1;
          value = snr & 0x0f;
          get_snr = 0;
        } else {
          value = snr >> 4;
          get_snr = 1;
        }
        proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                  tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                  ss, scidx, value);
      }
    }
    proto_item_set_len(ruii, offset - tree_offset);
  } else {
    /* Add each of the RU index groups for the lower 80MHz */
    for (i = 1; i <= 4; i++) {
      if ((partial_bw_info & (1 << i)) == 0x0) {
        continue;  /* Only dissect those with the bit set */
      }

      tree_offset = offset;
      init_eht_scidx(&scidx_ctx, i, &ru_242_tone_160MHz[i - 1], ng);
      ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                    ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                    &ruii,
                                                    "RU Index %d for 160MHz, Partial BW "
                                                    "not all one, lower 80MHz.", i);
      while (next_eht_scidx(&scidx_ctx, &scidx)) {
        for (ss = 1; ss <= nc_index; ss++) {
          if (get_snr) {
            snr = tvb_get_int8(tvb, offset);
            offset += 1;
            value = snr & 0x0f;
            get_snr = 0;
          } else {
            value = snr >> 4;
            get_snr = 1;
          }
          proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                    tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                    ss, scidx, value);
        }
      }
      proto_item_set_len(ruii, offset - tree_offset);
    }
  }

  /* Is the second lot a 996-tone RU? */
  get_snr = 1;
  if ((partial_bw_info & 0x1E0) == 0x1E0) { /* Uses 996-tone RU indices */
    tree_offset = offset;
    if (ng == 4) {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng4[2], ng);
    } else {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng16[5], ng);
    }
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "996-tone RU Index %d for 160MHz", 2);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      for (ss = 1; ss <= nc_index; ss++) {
        if (get_snr) {
          snr = tvb_get_int8(tvb, offset);
          offset += 1;
          value = snr & 0x0f;
          get_snr = 0;
        } else {
          value = snr >> 4;
          get_snr = 1;
        }
        proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                  tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                  ss, scidx, value);
      }
    }
    proto_item_set_len(ruii, offset - tree_offset);
  } else {
    /* Add each of the RU index groups for the lower 80MHz */
    for (i = 5; i <= 8; i++) {
      if ((partial_bw_info & (1 << i)) == 0x0) {
        continue;  /* Only dissect those with the bit set */
      }

      tree_offset = offset;
      init_eht_scidx(&scidx_ctx, i, &ru_242_tone_160MHz[i - 1], ng);
      ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                    ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                    &ruii,
                                                    "RU Index %d for 160MHz, Partial BW "
                                                    "not all one, upper 80MHz.", i);
      while (next_eht_scidx(&scidx_ctx, &scidx)) {
        for (ss = 1; ss <= nc_index; ss++) {
          if (get_snr) {
            snr = tvb_get_int8(tvb, offset);
            offset += 1;
            value = snr & 0x0f;
            get_snr = 0;
          } else {
            value = snr >> 4;
            get_snr = 1;
          }
          proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                    tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                    ss, scidx, value);
        }
      }
      proto_item_set_len(ruii, offset - tree_offset);
    }
  }

  return offset - start_offset;
}

static int
add_ff_eht_mu_exclusive_320MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo _U_,
                                   int offset, uint8_t nc_index,
                                   uint8_t grouping, uint16_t partial_bw_info)
{
  int start_offset = offset;
  int tree_offset = offset;
  int i = 0, j = 0, k = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;
  unsigned ss = 0, get_snr = 1;
  int8_t snr = 0, value;

  for ( i = 0; i < 4; i++) {
    if (((partial_bw_info >> (2*i+1)) & 0x03) == 0x03) { /* Uses 996-tone RU indices */
      if (ng == 4) {
        init_eht_scidx(&scidx_ctx, i+1, &ru_996_tone_320MHz_ng4[i * 2], ng);
      } else {
        init_eht_scidx(&scidx_ctx, i+1, &ru_996_tone_320MHz_ng16[i * 5], ng);
      }
      ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                    ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                                                    &ruii,
                                                    "996-tone RU Index %d for 320MHz",
                                                    i+1);
      tree_offset = offset;
      while (next_eht_scidx(&scidx_ctx, &scidx)) {
        for (ss = 1; ss <= nc_index; ss++) {
          if (get_snr) {
            snr = tvb_get_int8(tvb, offset);
            offset += 1;
            value = snr & 0x0f;
            get_snr = 0;
          } else {
            value = snr >> 4;
            get_snr = 1;
          }
          proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                    tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                    ss, scidx, value);
        }
      }
      proto_item_set_len(ruii, offset - tree_offset);
    } else {
      for (j = 2*i; j <= 2*i+1; j++) {
        if ((partial_bw_info & (1 << (j+1))) == 0x0) {
          continue;  /* Only dissect those with the bit set */
        }
        /* Each 484-tone RU contains two 242-tone RUs */
        for (k = 2*j;k <= 2*j+1;k++) {
          init_eht_scidx(&scidx_ctx, k+1, &ru_242_tone_320MHz[k], ng);
          ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                            ett_eht_mu_exclusive_beamforming_rpt_ru_index,
                            &ruii,
                            "RU Index %d for 320MHz, Partial BW "
                            "not all one.", k+1);
          tree_offset = offset;
          while (next_eht_scidx(&scidx_ctx, &scidx)) {
            for (ss = 1; ss <= nc_index; ss++) {
              if (get_snr) {
                snr = tvb_get_int8(tvb, offset);
                offset += 1;
                value = snr & 0x0f;
                get_snr = 0;
              } else {
                value = snr >> 4;
                get_snr = 1;
              }
              proto_tree_add_int_format(ru_index_tree, hf_ieee80211_eht_mu_exclusive_beamforming_report_delta_snr,
                                        tvb, offset, 1, value, "Delta SNR for Stream %d for subcarrier %u: %d dB",
                                        ss, scidx, value);
            }
          }
          proto_item_set_len(ruii, offset - tree_offset);
        }
      }
    }
  }

  return offset - start_offset;
}

static unsigned
add_ff_eht_mu_exclusive_beamforming_rpt(proto_tree *tree, tvbuff_t *tvb,
                                        packet_info *pinfo, int offset,
                                        uint64_t mimo_control)
{
  uint8_t nc_index = mimo_control & 0x0F;
  uint8_t bw = (mimo_control >> 8) & 0x07;
  uint8_t grouping = (mimo_control >> 11) & 0x01;
  uint16_t partial_bw_info = (mimo_control >> 21) & 0x01FF;
  proto_tree *exclusive_tree = NULL;
  proto_item *fti = NULL;
  unsigned start_offset = offset;
  unsigned tree_len = 0, byte_count = 0;

  exclusive_tree = proto_tree_add_subtree(tree, tvb, offset, -1,
                     ett_ff_eht_mimo_mu_exclusive_report, &fti,
                     "EHT MU Exclusive Beamforming Report");

  switch (bw) {
  case 0: /*  20 MHz */
    tree_len = add_ff_eht_mu_exclusive_20MHz_rpt(exclusive_tree, tvb, pinfo, offset,
                                                 nc_index, grouping, partial_bw_info);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 1: /*  40 MHz */
    tree_len = add_ff_eht_mu_exclusive_40MHz_rpt(exclusive_tree, tvb, pinfo, offset,
                                                 nc_index, grouping, partial_bw_info);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 2: /*  80 MHz */
    tree_len = add_ff_eht_mu_exclusive_80MHz_rpt(exclusive_tree, tvb, pinfo, offset,
                                                 nc_index, grouping, partial_bw_info);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 3: /* 160 MHz */
    tree_len = add_ff_eht_mu_exclusive_160MHz_rpt(exclusive_tree, tvb, pinfo, offset,
                                                  nc_index, grouping, partial_bw_info);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 4: /* 320 MHz */
    tree_len = add_ff_eht_mu_exclusive_320MHz_rpt(exclusive_tree, tvb, pinfo, offset,
                                                  nc_index, grouping, partial_bw_info);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  default:
    /* Add EI about invalid BW setting */
    break;
  }

  byte_count = tvb_reported_length_remaining(tvb, offset);
  if (byte_count > 0)
    offset += byte_count; /* Should fix the real problem. */

  return offset - start_offset;
}

static int
add_ff_eht_su_20MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                        int offset, uint8_t nc_index, uint8_t nr_index,
                        uint8_t grouping, uint16_t partial_bw_info _U_,
                        uint8_t phi_bits, uint8_t psi_bits)
{
  int start_offset = offset;
  unsigned scidx;
  struct scidx_ctx scidx_ctx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  unsigned bit_offset = offset * 8;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;

  if (ng == 4) {
    init_eht_scidx(&scidx_ctx, 1, &ru_242_tone_1_20MHz_ng4[0], ng);
  } else {
    init_eht_scidx(&scidx_ctx, 1, &ru_242_tone_1_20MHz_ng16[0], ng);
  }

  ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                ett_eht_beamforming_rpt_ru_index,
                                                &ruii,
                                                "RU Index %d for 20MHz, Partial BW "
                                                "not all one.", 1);
  while (next_eht_scidx(&scidx_ctx, &scidx)) {
    bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo, offset,
                                            bit_offset, scidx, nr_index + 1,
                                            nc_index + 1, phi_bits, psi_bits,
                                            hf_ieee80211_eht_compressed_beamform_scidx);
    offset = bit_offset / 8;
  }
  proto_item_set_len(ruii, offset - start_offset);

  return offset - start_offset;
}

static int
add_ff_eht_su_40MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                        int offset, uint8_t nc_index, uint8_t nr_index,
                        uint8_t grouping, uint16_t partial_bw_info,
                        uint8_t phi_bits, uint8_t psi_bits)
{
  int start_offset = offset;
  int i = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  unsigned bit_offset = offset * 8;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;

  /* Add each of the RU index groups set */
  for (i = 1; i <= 2; i++) {
    int tree_offset = offset;
    if ((partial_bw_info & (1 << i)) == 0x0) {
      continue;  /* Only dissect those with the bit set */
    }

    init_eht_scidx(&scidx_ctx, i, &ru_242_tone_40MHz[i - 1], ng);
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "RU Index %d for 40MHz, Partial BW "
                                                  "not all one.", i);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo, offset,
                                              bit_offset, scidx, nr_index + 1,
                                              nc_index + 1, phi_bits, psi_bits,
                                              hf_ieee80211_eht_compressed_beamform_scidx);
      offset = bit_offset / 8;
    }
    proto_item_set_len(ruii, offset - tree_offset);
  }

  return offset - start_offset;
}

static int
add_ff_eht_su_80MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                        int offset, uint8_t nc_index, uint8_t nr_index,
                        uint8_t grouping, uint16_t partial_bw_info,
                        uint8_t phi_bits, uint8_t psi_bits)
{
  int start_offset = offset;
  int i = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  unsigned bit_offset = offset * 8;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;

  if (partial_bw_info == 0x1E) { /* Uses 996-tone RU indices */
    if (ng == 4) {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_80MHz_ng4[0], ng);
    } else {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_80MHz_ng16[0], ng);
    }
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "996-tone RU Index %d for 80MHz", 1);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo, offset,
                                              bit_offset, scidx, nr_index + 1,
                                              nc_index + 1, phi_bits, psi_bits,
                                              hf_ieee80211_eht_compressed_beamform_scidx);
      offset = bit_offset / 8;
    }
    proto_item_set_len(ruii, offset - start_offset);

    return offset - start_offset;
  }

  /* Add each of the RU index groups */
  for (i = 1; i <= 4; i++) {
    int tree_offset = offset;
    if ((partial_bw_info & (1 << i)) == 0x0) {
      continue;  /* Only dissect those with the bit set */
    }

    init_eht_scidx(&scidx_ctx, i, &ru_242_tone_80MHz[i - 1], ng);
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "RU Index %d for 80MHz, Partial BW "
                                                  "not all one.", i);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo, offset,
                                              bit_offset, scidx, nr_index + 1,
                                              nc_index + 1, phi_bits, psi_bits,
                                              hf_ieee80211_eht_compressed_beamform_scidx);
      offset = bit_offset / 8;
    }
    proto_item_set_len(ruii, offset - tree_offset);
  }

  return offset - start_offset;
}

static int
add_ff_eht_su_160MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                         int offset, uint8_t nc_index, uint8_t nr_index,
                         uint8_t grouping, uint16_t partial_bw_info,
                         uint8_t phi_bits, uint8_t psi_bits)
{
  int start_offset = offset;
  int tree_offset;
  int i = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  unsigned bit_offset = offset * 8;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;

  /* Is the first lot a 996-tone RU? */
  if ((partial_bw_info & 0x1E) == 0x1E) { /* Uses 996-tone RU indices */
    tree_offset = offset;
    if (ng == 4) {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng4[0], ng);
    } else {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng16[0], ng);
    }
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "996-tone RU Index %d for 160MHz", 1);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo, offset,
                                              bit_offset, scidx, nr_index + 1,
                                              nc_index + 1, phi_bits, psi_bits,
                                              hf_ieee80211_eht_compressed_beamform_scidx);
      offset = bit_offset / 8;
    }
    proto_item_set_len(ruii, offset - tree_offset);
  } else {
    /* Add each of the RU index groups for the lower 80MHz */
    for (i = 1; i <= 4; i++) {
      if ((partial_bw_info & (1 << i)) == 0x0) {
        continue;  /* Only dissect those with the bit set */
      }

      tree_offset = offset;
      init_eht_scidx(&scidx_ctx, i, &ru_242_tone_160MHz[i - 1], ng);
      ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                    ett_eht_beamforming_rpt_ru_index,
                                                    &ruii,
                                                    "RU Index %d for 160MHz, Partial BW "
                                                    "not all one, lower 80MHz.", i);
      while (next_eht_scidx(&scidx_ctx, &scidx)) {
        bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo,
                                                offset, bit_offset, scidx,
                                                nr_index + 1, nc_index + 1,
                                                phi_bits, psi_bits,
                                                hf_ieee80211_eht_compressed_beamform_scidx);
        offset = bit_offset / 8;
      }
      proto_item_set_len(ruii, offset - tree_offset);
    }
  }

  /* Is the second lot a 996-tone RU? */
  if ((partial_bw_info & 0x1E0) == 0x1E0) { /* Uses 996-tone RU indices */
    tree_offset = offset;
    if (ng == 4) {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng4[2], ng);
    } else {
      init_eht_scidx(&scidx_ctx, i, &ru_996_tone_160MHz_ng16[5], ng);
    }
    ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                  ett_eht_beamforming_rpt_ru_index,
                                                  &ruii,
                                                  "996-tone RU Index %d for 160MHz", 2);
    while (next_eht_scidx(&scidx_ctx, &scidx)) {
      bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo, offset,
                                              bit_offset, scidx, nr_index + 1,
                                              nc_index + 1, phi_bits, psi_bits,
                                              hf_ieee80211_eht_compressed_beamform_scidx);
      offset = bit_offset / 8;
    }
    proto_item_set_len(ruii, offset - tree_offset);
  } else {
    /* Add each of the RU index groups for the lower 80MHz */
    for (i = 5; i <= 8; i++) {
      if ((partial_bw_info & (1 << i)) == 0x0) {
        continue;  /* Only dissect those with the bit set */
      }

      tree_offset = offset;
      init_eht_scidx(&scidx_ctx, i, &ru_242_tone_160MHz[i - 1], ng);
      ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                    ett_eht_beamforming_rpt_ru_index,
                                                    &ruii,
                                                    "RU Index %d for 160MHz, Partial BW "
                                                    "not all one, upper 80MHz.", i);
      while (next_eht_scidx(&scidx_ctx, &scidx)) {
        bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo,
                                                offset, bit_offset, scidx,
                                                nr_index + 1, nc_index + 1,
                                                phi_bits, psi_bits,
                                                hf_ieee80211_eht_compressed_beamform_scidx);
        offset = bit_offset / 8;
      }
      proto_item_set_len(ruii, offset - tree_offset);
    }
  }

  return offset - start_offset;
}

static int
add_ff_eht_su_320MHz_rpt(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                         int offset, uint8_t nc_index, uint8_t nr_index,
                         uint8_t grouping, uint16_t partial_bw_info,
                         uint8_t phi_bits, uint8_t psi_bits)
{
  int start_offset = offset;
  int tree_offset = offset;
  int i = 0, j = 0, k = 0;
  struct scidx_ctx scidx_ctx;
  unsigned scidx;
  uint8_t ng = grouping == 0 ? 4 : 16;
  unsigned bit_offset = offset * 8;
  proto_tree *ru_index_tree = NULL;
  proto_item *ruii = NULL;

  /* Resolution is 40 MHz */
  for (i = 0; i < 4; i++) {
    if (((partial_bw_info >> (2*i+1)) & 0x03) == 0x03) { /* Uses 996-tone RU indices */
      if (ng == 4) {
        init_eht_scidx(&scidx_ctx, i+1, &ru_996_tone_320MHz_ng4[i * 2], ng);
      } else {
        init_eht_scidx(&scidx_ctx, i+1, &ru_996_tone_320MHz_ng16[i * 5], ng);
      }
      ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                                                    ett_eht_beamforming_rpt_ru_index,
                                                    &ruii,
                                                    "996-tone RU Index %d for 320MHz",
                                                    i+1);
      tree_offset = offset;
      while (next_eht_scidx(&scidx_ctx, &scidx)) {
        bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo,
                                                offset, bit_offset, scidx,
                                                nr_index + 1, nc_index + 1,
                                                phi_bits, psi_bits,
                                                hf_ieee80211_eht_compressed_beamform_scidx);
        offset = bit_offset / 8;
      }
      proto_item_set_len(ruii, offset - tree_offset);
    } else {
      for (j = 2*i; j <= 2*i+1; j++) {
        if ((partial_bw_info & (1 << (j+1))) == 0x0) {
          continue;  /* Only dissect those with the bit set */
        }
        /* Each 484-tone RU contains two 242-tone RUs */
        for (k = 2*j;k <= 2*j+1;k++) {
          init_eht_scidx(&scidx_ctx, k+1, &ru_242_tone_320MHz[k], ng);
          ru_index_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1,
                            ett_eht_beamforming_rpt_ru_index,
                            &ruii,
                            "RU Index %d for 320MHz, Partial BW "
                            "not all one.", k+1);
          tree_offset = offset;
          while (next_eht_scidx(&scidx_ctx, &scidx)) {
            bit_offset = dissect_he_feedback_matrix(ru_index_tree, tvb, pinfo, offset,
                           bit_offset, scidx, nr_index + 1,
                           nc_index + 1, phi_bits, psi_bits,
                           hf_ieee80211_eht_compressed_beamform_scidx);
            offset = bit_offset / 8;
          }
          proto_item_set_len(ruii, offset - tree_offset);
        }
      }
    }
  }

  return offset - start_offset;
}

static unsigned
add_ff_eht_su_beamforming_rpt(proto_tree *tree, tvbuff_t *tvb,
                              packet_info *pinfo, int offset,
                              uint64_t mimo_control)
{
  uint8_t nc_index = mimo_control & 0x0F;
  uint8_t nr_index = (mimo_control >> 4) & 0x0F;
  uint8_t bw = (mimo_control >> 8) & 0x07;
  uint8_t grouping = (mimo_control >> 11) & 0x01;
  uint16_t partial_bw_info = (mimo_control >> 21) & 0x01FF;
  uint8_t codebook_info = (mimo_control >> 36) & 0x01;
  proto_tree *snr_tree = NULL, *feedback_tree = NULL;
  proto_item *fti = NULL;
  uint8_t i = 0, phi_bits, psi_bits;
  unsigned start_offset = offset;
  unsigned tree_len = 0, byte_count = 0;

  snr_tree = proto_tree_add_subtree(tree, tvb, offset, nc_index + 1,
               ett_ff_eht_mimo_beamforming_report_snr, NULL,
               "Average Signal to Noise Ratio");

  for (i = 0; i < nc_index + 1; i++) {
    int8_t snr = tvb_get_int8(tvb, offset);

    proto_tree_add_int_format(snr_tree,
      hf_ieee80211_eht_compressed_beamforming_report_snr, tvb, offset, 1,
      snr, "Stream %d: %s%0.2fdB (0x%02x)", i,
      (snr == 127 ? ">=" : (snr == -128 ? "<=" : "")),
      (float)((float)88 + snr)/4, (uint8_t)snr);
    offset++;
  }

  if (codebook_info == 0) {
    phi_bits = 4;
    psi_bits = 2;
  } else {
    phi_bits = 6;
    psi_bits = 4;
  }

  feedback_tree = proto_tree_add_subtree(tree, tvb, offset, -1,
                                         ett_eht_beamforming_feedback_tree,
                                         &fti, "Feedback Matrices");
  switch (bw) {
  case 0: /*  20 MHz */
    tree_len = add_ff_eht_su_20MHz_rpt(feedback_tree, tvb, pinfo, offset,
                                       nc_index, nr_index, grouping,
                                       partial_bw_info, phi_bits, psi_bits);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 1: /*  40 MHz */
    tree_len = add_ff_eht_su_40MHz_rpt(feedback_tree, tvb, pinfo, offset,
                                       nc_index, nr_index, grouping,
                                       partial_bw_info, phi_bits, psi_bits);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 2: /*  80 MHz */
    tree_len = add_ff_eht_su_80MHz_rpt(feedback_tree, tvb, pinfo, offset,
                                       nc_index, nr_index, grouping,
                                       partial_bw_info, phi_bits, psi_bits);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 3: /* 160 MHz */
    tree_len = add_ff_eht_su_160MHz_rpt(feedback_tree, tvb, pinfo, offset,
                                        nc_index, nr_index, grouping,
                                        partial_bw_info, phi_bits, psi_bits);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  case 4: /* 320 MHz */
    tree_len = add_ff_eht_su_320MHz_rpt(feedback_tree, tvb, pinfo, offset,
                                        nc_index, nr_index, grouping,
                                        partial_bw_info, phi_bits, psi_bits);
    proto_item_set_len(fti, tree_len);
    offset += tree_len;
    break;
  default:
    /* Add EI about invalid BW setting */
    break;
  }

  byte_count = tvb_reported_length_remaining(tvb, offset);
  if (byte_count > 0)
    offset += byte_count; /* Should fix the real problem. */

  return offset - start_offset;
}

static const val64_string eht_mimo_bw_vals[] = {
  { 0, "20 MHz" },
  { 1, "40 MHz" },
  { 2, "80 MHz" },
  { 3, "160 MHz" },
  { 4, "320 MHz" },
  { 5, "Reserved" },
  { 6, "Reserved" },
  { 7, "Reserved" },
  { 0, NULL }
};

static const val64_string eht_mimo_grouping_vals[] = {
  { 0, "Ng = 4" },
  { 1, "Ng = 16" },
  { 0, NULL }
};

static const val64_string eht_feedback_type_vals[] = {
  { 0, "SU" },
  { 1, "MU" },
  { 2, "CQI" },
  { 3, "Reserved" },
  { 0, NULL }
};

static int * const eht_mimo_ctrl_hdrs[] = {
  &hf_ieee80211_eht_mimo_ctrl_nc_index,
  &hf_ieee80211_eht_mimo_ctrl_nr_index,
  &hf_ieee80211_eht_mimo_ctrl_bw,
  &hf_ieee80211_eht_mimo_ctrl_grouping,
  &hf_ieee80211_eht_mimo_ctrl_feedback_type,
  &hf_ieee80211_eht_mimo_ctrl_reserved1,
  &hf_ieee80211_eht_mimo_ctrl_remaining_feedback_segments,
  &hf_ieee80211_eht_mimo_ctrl_first_feedback_segment,
  &hf_ieee80211_eht_mimo_ctrl_partial_bw_info,
  &hf_ieee80211_eht_mimo_ctrl_sounding_dialog_token_number,
  &hf_ieee80211_eht_mimo_ctrl_codebook_info,
  &hf_ieee80211_eht_mimo_ctrl_reserved2,
  NULL
};

static unsigned
add_ff_eht_mimo_control_etc(proto_tree *tree _U_, tvbuff_t *tvb _U_,
                            packet_info *pinfo _U_,
                            int offset _U_)
{
  unsigned start = offset;
  uint64_t mimo_control = tvb_get_uint40(tvb, offset, ENC_LITTLE_ENDIAN);
  uint8_t feedback_type = (mimo_control >> 12) & 0x03;
  uint8_t nc_index = mimo_control & 0x0F;
  uint8_t nr_index = (mimo_control >> 4) & 0x0F;
  proto_item *mci = NULL;

  mci = proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                          hf_ieee80211_eht_mimo_ctrl_field,
                                          ett_eht_mimo_ctrl, eht_mimo_ctrl_hdrs,
                                          ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  /*
   * Validate nc_index and nr_index and go no further if they exceed the
   * limits.
   *
   * 802.11be
   */
  if (nc_index > 7) {
    expert_add_info_format(pinfo, mci, &ei_ieee80211_eht_invalid_nc_nr,
                           "Nc indices (%d) > 7 are reserved", nc_index);
    return offset;
  }
  if (nr_index == 0 || nr_index > 7) {
    expert_add_info_format(pinfo, mci, &ei_ieee80211_eht_invalid_nc_nr,
                           "Nr indices (%d) 0 and > 7 are reserved", nr_index);
    return offset;
  }
  offset += 5;

  switch (feedback_type) {
  case SU_FEEDBACK:
    offset += add_ff_eht_su_beamforming_rpt(tree, tvb, pinfo, offset,
                                            mimo_control);
    break;
  case MU_FEEDBACK:
    offset += add_ff_eht_su_beamforming_rpt(tree, tvb, pinfo, offset,
                                            mimo_control);
    offset += add_ff_eht_mu_exclusive_beamforming_rpt(tree, tvb, pinfo, offset,
                                                      mimo_control);
    break;
  case CQI_FEEDBACK:
    /* TODO */
    break;
  default:
    break;
  }

  return offset - start;
}

static const range_string eht_action_rvals[] = {
  { 0, 0, "EHT Compressed Beamforming/CQI" },
  { 1, 255, "Reserved" },
  { 0, 0, NULL }
};

static unsigned
add_ff_action_eht(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo,
                  int offset)
{
  unsigned start = offset;
  uint8_t eht_action;
  proto_item *item = NULL;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);

  eht_action = tvb_get_uint8(tvb, offset);

  item = proto_tree_add_item(tree, hf_ieee80211_ff_eht_action, tvb, offset, 1,
                             ENC_NA);
  offset += 1;

  switch (eht_action) {
  case 0:
    offset += add_ff_eht_mimo_control_etc(tree, tvb, pinfo, offset);
    break;
  default:
    expert_add_info_format(pinfo, item, &ei_ieee80211_eht_invalid_action,
                           "Reserved EHT Action %u", eht_action);
  }

  return offset - start;
}

static unsigned
add_ff_action_fst(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  unsigned  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_fst_action_code(tree, tvb, pinfo, offset);
  switch (code) {
    case FST_SETUP_REQUEST:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_llt(tree, tvb, pinfo, offset);
      break;
    case FST_SETUP_RESPONSE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case FST_TEAR_DOWN:
      offset += add_ff_fsts_id(tree, tvb, pinfo, offset);
      break;
    case FST_ACK_REQUEST:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_fsts_id(tree, tvb, pinfo, offset);
      break;
    case FST_ACK_RESPONSE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_fsts_id(tree, tvb, pinfo, offset);
      break;
    case FST_ON_CHANNEL_TUNNEL_REQUEST:
      offset += add_ff_oct_mmpdu(tree, tvb, pinfo, offset);
      break;
  }
  return offset - start;
}

static int
add_ff_scs_descriptor_list(proto_tree *tree, tvbuff_t *tvb,
                           packet_info *pinfo _U_, int offset)
{
  unsigned start = offset;

  /*
   * This is could be a list, so it needs change.
   */
  offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
  return offset - start;
}

static int
add_ff_scs_response_count(proto_tree *tree, tvbuff_t *tvb,
                          packet_info *pinfo _U_, int offset)
{
  proto_tree_add_item(tree, hf_ieee80211_ff_scs_response_count, tvb, offset, 1,
                      ENC_NA);

  return 1;
}

static int
add_ff_scs_status_list(proto_tree *tree, tvbuff_t *tvb,
                       packet_info *pinfo _U_, int offset, uint8_t count)
{
  unsigned start = offset;

  while (count > 0) {
    if (tvb_reported_length_remaining(tvb, offset) >= 3) {
      proto_tree_add_item(tree, hf_ieee80211_ff_scs_scsid, tvb, offset, 1,
                          ENC_NA);
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_ff_scs_status, tvb, offset, 2,
                          ENC_LITTLE_ENDIAN);
      offset += 2;
    }
    count--;
  }

  return offset - start;
}

static int
add_ff_mscs_descriptor_elt(proto_tree *tree, tvbuff_t *tvb,
                           packet_info *pinfo _U_, int offset)
{
  unsigned start = offset;

  offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
  return offset - start;
}

static unsigned
add_ff_action_robust_av_streaming(proto_tree *tree, tvbuff_t *tvb,
                                  packet_info *pinfo, int offset)
{
  uint8_t code, count;
  unsigned  start = offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_robust_av_streaming_action_code(tree, tvb, pinfo, offset);

  switch (code) {
    case ROBUST_AV_STREAMING_SCS_REQUEST:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_scs_descriptor_list(tree, tvb, pinfo, offset);
      break;
    case ROBUST_AV_STREAMING_SCS_RESPONSE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      count = tvb_get_uint8(tvb, offset);
      offset += add_ff_scs_response_count(tree, tvb, pinfo, offset);
      offset += add_ff_scs_status_list(tree, tvb, pinfo, offset, count);
      break;
    case ROBUST_AV_STREAMING_MSCS_REQUEST:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_mscs_descriptor_elt(tree, tvb, pinfo, offset);
    break;
    case ROBUST_AV_STREAMING_MSCS_RESPONSE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      /* If there is any more data it is probably an mscs descriptor */
      if (tvb_reported_length_remaining(tvb, offset) > 0)
        offset += add_ff_mscs_descriptor_elt(tree, tvb, pinfo, offset);
    break;
  }
  return offset - start;
}

static unsigned
add_ff_action_dmg(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t code;
  unsigned  start = offset;
  int left_offset;

  offset += add_ff_category_code(tree, tvb, pinfo, offset);
  code    = tvb_get_uint8(tvb, offset);
  offset += add_ff_dmg_action_code(tree, tvb, pinfo, offset);
  switch (code) {
    case DMG_ACTION_PWR_SAVE_CONFIG_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_dmg_pwr_mgmt(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_PWR_SAVE_CONFIG_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_INFO_REQ:
      offset += add_ff_subject_address(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_INFO_RES:
      offset += add_ff_subject_address(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_HANDOVER_REQ:
      offset += add_ff_handover_reason(tree, tvb, pinfo, offset);
      offset += add_ff_handover_remaining_bi(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_HANDOVER_RES:
      offset += add_ff_handover_result(tree, tvb, pinfo, offset);
      offset += add_ff_handover_reject_reason(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_DTP_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_DTP_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RELAY_SEARCH_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_destination_reds_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RELAY_SEARCH_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_MUL_RELAY_CHANNEL_MEASURE_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      left_offset =
          tvb_reported_length_remaining(tvb, offset);
      while(left_offset > 0) {
        proto_tree_add_item(tree, hf_ieee80211_ff_peer_sta_aid, tvb, offset, 1, ENC_NA);
        proto_tree_add_item(tree, hf_ieee80211_ff_snr, tvb, offset+1, 1, ENC_NA);
        proto_tree_add_item(tree, hf_ieee80211_ff_internal_angle, tvb, offset+2, 1, ENC_NA);
        proto_tree_add_item(tree, hf_ieee80211_ff_recommend, tvb, offset+2, 1, ENC_NA);
        /* another reserved byte */
        offset += 4;
        left_offset -= 4;
      }
      break;
    case DMG_ACTION_RLS_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_destination_aid(tree, tvb, pinfo, offset);
      offset += add_ff_relay_aid(tree, tvb, pinfo, offset);
      offset += add_ff_source_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RLS_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RLS_ANNOUNCE:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      offset += add_ff_destination_aid(tree, tvb, pinfo, offset);
      offset += add_ff_relay_aid(tree, tvb, pinfo, offset);
      offset += add_ff_source_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RLS_TEARDOWN:
      offset += add_ff_destination_aid(tree, tvb, pinfo, offset);
      offset += add_ff_relay_aid(tree, tvb, pinfo, offset);
      offset += add_ff_source_aid(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_RELAY_ACK_REQ:
    case DMG_ACTION_RELAY_ACK_RES:
      break;
    case DMG_ACTION_TPA_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_timing_offset(tree, tvb, pinfo, offset);
      offset += add_ff_sampling_frequency_offset(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_TPA_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_TPA_REP:
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_ROC_REQ:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_relay_operation_type(tree, tvb, pinfo, offset);
      break;
    case DMG_ACTION_ROC_RES:
      offset += add_ff_dialog_token(tree, tvb, pinfo, offset);
      offset += add_ff_status_code(tree, tvb, pinfo, offset);
      break;
  }
  return offset - start;
}

unsigned
add_ff_action(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset,
              association_sanity_check_t *association_sanity_check)
{
  switch (tvb_get_uint8(tvb, offset) & 0x7f) {
  case CAT_SPECTRUM_MGMT: /* 0 */
    return add_ff_action_spectrum_mgmt(tree, tvb, pinfo, offset);
  case CAT_QOS: /* 1 */
    return add_ff_action_qos(tree, tvb, pinfo, offset);
  case CAT_DLS: /* 2 */
    return add_ff_action_dls(tree, tvb, pinfo, offset);
  case CAT_BLOCK_ACK: /* 3 */
    return add_ff_action_block_ack(tree, tvb, pinfo, offset);
  case CAT_PUBLIC: /* 4 */
    return add_ff_action_public(tree, tvb, pinfo, offset);
  case CAT_RADIO_MEASUREMENT: /* 5 */
    return add_ff_action_radio_measurement(tree, tvb, pinfo, offset);
  case CAT_FAST_BSS_TRANSITION: /* 6 */
    return add_ff_action_fast_bss_transition(tree, tvb, pinfo, offset);
  case CAT_HT: /* 7 */
    return add_ff_action_ht(tree, tvb, pinfo, offset);
  case CAT_SA_QUERY: /* 8 */
    return add_ff_action_sa_query(tree, tvb, pinfo, offset);
  case CAT_PUBLIC_PROTECTED: /* 9 */
    return add_ff_action_protected_public(tree, tvb, pinfo, offset);
  case CAT_WNM: /* 10 */
    return add_ff_action_wnm(tree, tvb, pinfo, offset);
  case CAT_UNPROTECTED_WNM: /* 11 */
    return add_ff_action_unprotected_wnm(tree, tvb, pinfo, offset);
  case CAT_TDLS: /* 12 */
    return add_ff_action_tdls(tree, tvb, pinfo, offset);
  case CAT_MESH: /* 13 */
    return add_ff_action_mesh(tree, tvb, pinfo, offset);
  case CAT_MULTIHOP: /* 14 */
    return add_ff_action_multihop(tree, tvb, pinfo, offset);
  case CAT_SELF_PROTECTED: /* 15 */
    return add_ff_action_self_protected(tree, tvb, pinfo, offset, association_sanity_check);
  case CAT_DMG: /* 16 */
    return add_ff_action_dmg(tree, tvb, pinfo, offset);
  case CAT_MGMT_NOTIFICATION:  /* Management notification frame - 17 */
    return add_ff_action_mgmt_notification(tree, tvb, pinfo, offset);
  case CAT_FAST_SESSION_TRANSFER: /* 18 */
    return add_ff_action_fst(tree, tvb, pinfo, offset);
  case CAT_ROBUST_AV_STREAMING: /* 19 */
    return add_ff_action_robust_av_streaming(tree, tvb, pinfo, offset);
  case CAT_UNPROTECTED_DMG: /* 20 */
    return add_ff_action_unprotected_dmg(tree, tvb, pinfo, offset);
  case CAT_VHT: /* 21 */
    return add_ff_action_vht(tree, tvb, pinfo, offset);
  case CAT_S1G: /* 22 */
    return add_ff_action_s1g(tree, tvb, pinfo, offset);
  case CAT_PROTECTED_S1G: /* 23 */
    return add_ff_action_protected_s1g(tree, tvb, pinfo, offset);
  case CAT_HE:
    return add_ff_action_he(tree, tvb, pinfo, offset);
  case CAT_PROTECTED_HE:
    return add_ff_action_protected_he(tree, tvb, pinfo, offset);
  case CAT_PROTECTED_EHT:
    return add_ff_action_protected_eht(tree, tvb, pinfo, offset);
  case CAT_PROTECTED_FTM:
    return add_ff_action_protected_ftm(tree, tvb, pinfo, offset);
  case CAT_EHT:
    return add_ff_action_eht(tree, tvb, pinfo, offset);
  case CAT_VENDOR_SPECIFIC_PROTECTED: /* Same as below for now */
  case CAT_VENDOR_SPECIFIC:  /* Vendor Specific Protected Category - 127 */
    return add_ff_action_vendor_specific(tree, tvb, pinfo, offset);
  default:
    add_ff_category_code(tree, tvb, pinfo, offset);
    return 1;
  }
}

static const value_string ieee80211_rsn_cipher_vals[] = {
  {0, "NONE"},
  {1, "WEP (40-bit)"},
  {2, "TKIP"},
  {3, "AES (OCB)"},
  {4, "AES (CCM)"},
  {5, "WEP (104-bit)"},
  {6, "BIP (128)"},
  {7, "Group addressed traffic not allowed"},
  {8, "GCMP (128)" },
  {9, "GCMP (256)" },
  {10, "CCMP (256)" },
  {11, "BIP (GMAC-128)" },
  {12, "BIP (GMAC-256)" },
  {13, "BIP (CMAC-256)" },
  {0, NULL}
};

#define AKMS_NONE                 0x000FAC00
#define AKMS_WPA                  0x000FAC01
#define AKMS_PSK                  0x000FAC02
#define AKMS_FT_IEEE802_1X        0x000FAC03
#define AKMS_FT_PSK               0x000FAC04
#define AKMS_WPA_SHA256           0x000FAC05
#define AKMS_PSK_SHA256           0x000FAC06
#define AKMS_TDLS                 0x000FAC07
#define AKMS_SAE                  0x000FAC08
#define AKMS_FT_SAE               0x000FAC09
#define AKMS_AP_PEER_KEY          0x000FAC0A
#define AKMS_WPA_SHA256_SUITEB    0x000FAC0B
#define AKMS_WPA_SHA384_SUITEB    0x000FAC0C
#define AKMS_FT_IEEE802_1X_SHA384 0x000FAC0D
#define AKMS_FILS_SHA256          0x000FAC0E
#define AKMS_FILS_SHA384          0x000FAC0F
#define AKMS_FT_FILS_SHA256       0x000FAC10
#define AKMS_FT_FILS_SHA384       0x000FAC11
#define AKMS_OWE                  0x000FAC12
#define AKMS_SAE_GROUP_DEPEND     0x000FAC18
#define AKMS_FT_SAE_GROUP_DEPEND  0x000FAC19

static const value_string ieee80211_rsn_keymgmt_vals[] = {
  {0, "NONE"},
  {1, "WPA"},
  {2, "PSK"},
  {3, "FT over IEEE 802.1X"},
  {4, "FT using PSK"},
  {5, "WPA (SHA256)"},
  {6, "PSK (SHA256)"},
  {7, "TDLS / TPK Handshake (SHA256)"},
  {8, "SAE (SHA256)" },
  {9, "FT using SAE (SHA256)" },
  {10, "APPeerKey (SHA256)" },
  {11, "WPA (SHA256-SuiteB)" },
  {12, "WPA (SHA384-SuiteB)" },
  {13, "FT over IEEE 802.1X (SHA384)" },
  {14, "FILS (SHA256 and AES-SIV-256)" },
  {15, "FILS (SHA384 and AES-SIV-512)" },
  {16, "FT over FILS (SHA256 and AES-SIV-256)" },
  {17, "FT over FILS (SHA384 and AES-SIV-512)" },
  {18, "Opportunistic Wireless Encryption"},
  {19, "FT using PSK (SHA384)"},
  {20, "PSK (SHA384)"},
  {21, "PASN"},
  {24, "SAE (GROUP-DEPEND)"},
  {25, "FT using SAE (GROUP-DEPEND)"},
  {0, NULL}
};

#define OUIBASELEN (MAXNAMELEN + 20)

static void
oui_base_custom(char *result, uint32_t oui)
{
  uint8_t      p_oui[3];
  const char *manuf_name;

  p_oui[0] = oui >> 16 & 0xFF;
  p_oui[1] = oui >> 8 & 0xFF;
  p_oui[2] = oui & 0xFF;

  static_assert(OUIBASELEN <= ITEM_LABEL_LENGTH, "Buffer size mismatch!");
  /* Attempt an OUI lookup. */
  manuf_name = uint_get_manuf_name_if_known(oui);
  if (manuf_name == NULL) {
    /* Could not find an OUI. */
    snprintf(result, OUIBASELEN, "%02x:%02x:%02x", p_oui[0], p_oui[1], p_oui[2]);
  }
  else {
   char name[MAXNAMELEN+2];
   snprintf(name, MAXNAMELEN+1, "%.*s", MAXNAMELEN, manuf_name);
   /* Found an address string. */
    snprintf(result, OUIBASELEN, "%02x:%02x:%02x (%s)", p_oui[0], p_oui[1], p_oui[2], name);
  }
}

static void
rsn_gcs_base_custom(char *result, uint32_t gcs)
{
  char oui_result[OUIBASELEN];
  char *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, gcs >> 8);
  tmp_str = val_to_str(NULL, gcs & 0xFF, ieee80211_rsn_cipher_vals, "Unknown %d");
  snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static void
rsn_pcs_base_custom(char *result, uint32_t pcs)
{
  char oui_result[OUIBASELEN];
  char *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, pcs >> 8);
  tmp_str = val_to_str(NULL, pcs & 0xFF, ieee80211_rsn_cipher_vals, "Unknown %d");
  snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);

}
static void
rsn_akms_base_custom(char *result, uint32_t akms)
{
  char oui_result[OUIBASELEN];
  char *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, akms >> 8);
  tmp_str = val_to_str(NULL, akms & 0xFF, ieee80211_rsn_keymgmt_vals, "Unknown %d");
  snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static char *
rsn_pcs_return(wmem_allocator_t *scope, uint32_t pcs)
{
  char *result;

  result = (char *)wmem_alloc(scope, SHORT_STR);
  result[0] = '\0';
  rsn_pcs_base_custom(result, pcs);

  return result;
}

static char *
rsn_akms_return(wmem_allocator_t *scope, uint32_t akms)
{
  char *result;

  result = (char *)wmem_alloc(scope, SHORT_STR);
  result[0] = '\0';
  rsn_akms_base_custom(result, akms);

  return result;
}

static void
rsn_gmcs_base_custom(char *result, uint32_t gmcs)
{
  char oui_result[OUIBASELEN];
  char *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, gmcs >> 8);
  tmp_str = val_to_str(NULL, gmcs & 0xFF, ieee80211_rsn_cipher_vals, "Unknown %d");
  snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static void
rsni_base_custom(char *result, uint8_t rsni)
{
  double temp_double;

  if (rsni < 255) {
    temp_double = (double)rsni - 20;
    snprintf(result, ITEM_LABEL_LENGTH, "%.1f dB", (temp_double / 2));
  } else
    snprintf(result, ITEM_LABEL_LENGTH, "%d (Measurement not available)", rsni);
}

static void
vht_tpe_custom(char *result, uint8_t txpwr)
{
  int8_t txpwr_db;

  txpwr_db = (int8_t)(txpwr);
  snprintf(result, ITEM_LABEL_LENGTH, "%3.1f dBm", (txpwr_db/2.0));
}

static void
tpe_psd_custom(char *result, uint8_t txpwr)
{
  int8_t txpwr_db;

  txpwr_db = (int8_t)(txpwr);
  if (txpwr_db == -128) {
    snprintf(result, ITEM_LABEL_LENGTH, "Channel cannot be used for transmission");
  } else if (txpwr_db == 127) {
    snprintf(result, ITEM_LABEL_LENGTH, "No maximum PSD is specified for channel");
  } else {
    snprintf(result, ITEM_LABEL_LENGTH, "%3.1f dBm/MHz", (txpwr_db/2.0));
  }
}

static void
channel_number_custom(char *result, uint8_t channel_number)
{
  switch(channel_number){
    case 0:
      snprintf(result, ITEM_LABEL_LENGTH, "%u (iterative measurements on all supported channels in the specified Operating Class)", channel_number);
    break;
    case 255:
      snprintf(result, ITEM_LABEL_LENGTH, "%u (iterative measurements on all supported channels listed in the AP Channel Report)", channel_number);
    break;
    default :
      snprintf(result, ITEM_LABEL_LENGTH, "%u (iterative measurements on that Channel Number)", channel_number);
    break;
  }
}

/* WPA / WME */
static const value_string ieee802111_wfa_ie_type_vals[] = {
  { 1, "WPA Information Element" },
  { 2, "WMM/WME" },
  { 4, "WPS" },
  { 17, "Network Cost" },
  { 18, "Tethering" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wpa_cipher_vals[] = {
  { 0, "NONE" },
  { 1, "WEP (40-bit)" },
  { 2, "TKIP" },
  { 3, "AES (OCB)" },
  { 4, "AES (CCM)" },
  { 5, "WEP (104-bit)" },
  { 6, "BIP" },
  { 7, "Group addressed traffic not allowed" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wpa_keymgmt_vals[] = {
  { 0, "NONE" },
  { 1, "WPA" },
  { 2, "PSK" },
  { 3, "FT over IEEE 802.1X" },
  { 4, "FT using PSK" },
  { 5, "WPA (SHA256)" },
  { 6, "PSK (SHA256)" },
  { 7, "TDLS / TPK Handshake" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_acs_vals[] = {
  { 0, "Best Effort" },
  { 1, "Background" },
  { 2, "Video" },
  { 3, "Voice" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_tspec_tsinfo_direction_vals[] = {
  { 0, "Uplink" },
  { 1, "Downlink" },
  { 2, "Direct link" },
  { 3, "Bidirectional link" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_tspec_tsinfo_psb_vals[] = {
  { 0, "Legacy" },
  { 1, "U-APSD" },
  { 0, NULL }
};

static const value_string ieee80211_wfa_ie_wme_tspec_tsinfo_up_vals[] = {
  { 0, "Best Effort" },
  { 1, "Background" },
  { 2, "Spare" },
  { 3, "Excellent Effort" },
  { 4, "Controlled Load" },
  { 5, "Video" },
  { 6, "Voice" },
  { 7, "Network Control" },
  { 0, NULL }
};

/* Cost Level https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nct/24b04427-4ed6-4d12-a73d-c89ea72c7a94 */
static const value_string ieee80211_wfa_ie_nc_cost_level_vals[] = {
  { 0x0, "Unknown / The connection cost is unknown" },
  { 0x01, "Unrestricted / The connection is unlimited and has unrestricted usage constraints" },
  { 0x02, "Fixed / Usage counts toward a fixed allotment of data which the user has already paid for (or agreed to pay for)" },
  { 0x04, "Variable / The connection cost is on a per-byte basis" },
  {0, NULL}
};

/* Cost Flags https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nct/b601a6a0-a4ff-4527-bf43-2eeee8c5796b */
static const value_string ieee80211_wfa_ie_nc_cost_flags_vals[] = {
  { 0x0, "Unknown / The usage is unknown or unrestricted" },
  { 0x01, "Over Data Limit / Usage has exceeded the data limit of the metered network; different network costs or conditions might apply" },
  { 0x02, "Congested / The network operator is experiencing or expecting heavy load" },
  { 0x04, "Roaming / The tethering connection is roaming outside the provider's home network or affiliates" },
  { 0x08, "Approaching Data Limit / Usage is near the data limit of the metered network; different network costs or conditions might apply once the limit is reached" },
  {0, NULL}
};


static const value_string ieee80211_wfa_ie_tethering_type_vals[] = {
  { 0x2B, "Broadcasted" },
  {0, NULL}
};


static const value_string ieee802111_wfa_ie_wme_qos_info_sta_max_sp_length_vals[] = {
  { 0, "WMM AP may deliver all buffered frames (MSDUs and MMPDUs)" },
  { 1, "WMM AP may deliver a maximum of 2 buffered frames (MSDUs and MMPDUs) per USP" },
  { 2, "WMM AP may deliver a maximum of 4 buffered frames (MSDUs and MMPDUs) per USP" },
  { 3, "WMM AP may deliver a maximum of 6 buffered frames (MSDUs and MMPDUs) per USP" },
  { 0, NULL}
};
static const true_false_string ieee802111_wfa_ie_wme_qos_info_sta_ac_tfs = {
  "WMM delivery and trigger enabled",
  "non-WMM PS"
};

/* az: Ranging Parameters */
static const val64_string ieee80211_ranging_status_vals[] = {
  { 0, "Reserved" },
  { 1, "Successful; measurement exchanges are about to begin" },
  { 2, "Request incapable; do not send same request again; FTM session ends" },
  { 3, "Request failed; do not send new request for Value seconds; FTM session ends" },
  { 0, NULL }
};

static const val64_string ieee80211_ranging_fmt_bw_vals[] = {
  { 0, "HE 20 MHz" },
  { 1, "HE 40 MHz" },
  { 2, "HE 80 MHz" },
  { 3, "HE 80+80 MHz" },
  { 4, "HE 160 MHz (two separate RF LOs)" },
  { 5, "HE 160 MHz (single RF LO)" },
  /* values 6-63 reserved */
  { 0, NULL }
};

static const val64_string ieee80211_ranging_ltf_total_vals[] = {
  { 0, "4 LTFs" },
  { 1, "8 LTFs" },
  { 2, "16 LTFs" },
  { 3, "No max LTFs specified" },
  { 0, NULL }
};

static void
wpa_mcs_base_custom(char *result, uint32_t mcs)
{
  char oui_result[OUIBASELEN];
  char *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, mcs >> 8);
  tmp_str = val_to_str(NULL, mcs & 0xFF, ieee80211_wfa_ie_wpa_cipher_vals, "Unknown %d");
  snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static void
wpa_ucs_base_custom(char *result, uint32_t ucs)
{
  char oui_result[OUIBASELEN];
  char *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, ucs >> 8);
  tmp_str = val_to_str(NULL, ucs & 0xFF, ieee80211_wfa_ie_wpa_cipher_vals, "Unknown %d");
  snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}
static void
wpa_akms_base_custom(char *result, uint32_t akms)
{
  char oui_result[OUIBASELEN];
  char *tmp_str;

  oui_result[0] = '\0';
  oui_base_custom(oui_result, akms >> 8);
  tmp_str = val_to_str(NULL, akms & 0xFF, ieee80211_wfa_ie_wpa_keymgmt_vals, "Unknown %d");
  snprintf(result, ITEM_LABEL_LENGTH, "%s %s", oui_result, tmp_str);
  wmem_free(NULL, tmp_str);
}

static char *
wpa_ucs_return(wmem_allocator_t *scope, uint32_t ucs)
{
  char *result;

  result = (char *)wmem_alloc(scope, SHORT_STR);
  result[0] = '\0';
  wpa_ucs_base_custom(result, ucs);

  return result;
}

static char *
wpa_akms_return(wmem_allocator_t *scope, uint32_t akms)
{
  char *result;

  result = (char *)wmem_alloc(scope, SHORT_STR);
  result[0] = '\0';
  wpa_akms_base_custom(result, akms);

  return result;
}

/* For each Field */
static const value_string ieee80211_wapi_suite_type[] = {
  {0, "Reserved"},
  {1, "WAI Certificate Authentication and Key Management"},
  {2, "WAI Preshared Key Authentication and Key Management"},
  {0, NULL},
};
/* For Summary Tag Information */
static const value_string ieee80211_wapi_suite_type_short[] = {
  {0, "Reserved"},
  {1, "WAI-CERT"},
  {2, "WAI-PSK"},
  {0, NULL},
};

static const value_string ieee80211_wapi_cipher_type[] = {
  {0, "Reserved"},
  {1, "WPI-SMS4"},
  {0, NULL},
};

static const value_string ieee802111_wfa_ie_wme_type[] = {
  { 0, "Information Element" },
  { 1, "Parameter Element" },
  { 2, "TSPEC Element" },
  { 0, NULL}
};

static const value_string ft_subelem_id_vals[] = {
  {0, "Reserved"},
  {1, "PMK-R1 key holder identifier (R1KH-ID)"},
  {2, "GTK subelement"},
  {3, "PMK-R0 key holder identifier (R0KH-ID)"},
  {4, "IGTK"},
  {5, "Operating Channel Information (OCI)"},
  {6, "BIGTK"},
  {7, "WIGTK"},
  {8, "MLO GTK"},
  {9, "MLO IGTK"},
  {10, "MLO BIGTK"},
  {0, NULL}
};

static int
dissect_wme_qos_info(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int ftype)
{
  proto_item *wme_qos_info_item;

  static int * const ieee80211_mgt_req[] = {
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_max_sp_length,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_be,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_bk,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vi,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_ac_vo,
    &hf_ieee80211_wfa_ie_wme_qos_info_sta_reserved,
    NULL
  };

  static int * const ieee80211_mgt_resp[] = {
    &hf_ieee80211_wfa_ie_wme_qos_info_ap_u_apsd,
    &hf_ieee80211_wfa_ie_wme_qos_info_ap_parameter_set_count,
    &hf_ieee80211_wfa_ie_wme_qos_info_ap_reserved,
    NULL
  };

  switch (ftype) {
    case MGT_ASSOC_REQ:
    case MGT_PROBE_REQ:
    case MGT_REASSOC_REQ:
    {
      /* To AP so decode as per WMM standard Figure 7 QoS Info field when sent from WMM STA*/
      proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_wfa_ie_wme_qos_info,
                                    ett_wme_qos_info, ieee80211_mgt_req,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      break;
    }
    case MGT_BEACON:
    case MGT_PROBE_RESP:
    case MGT_ASSOC_RESP:
    case MGT_REASSOC_RESP:
    case MGT_ACTION:
    {
      /* From AP so decode as per WMM standard Figure 6 QoS Info field when sent from WMM AP */
      proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_wfa_ie_wme_qos_info,
                                    ett_wme_qos_info, ieee80211_mgt_resp,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      break;
    }
    default:
        wme_qos_info_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_qos_info, tvb, offset, 1, ENC_NA);
        expert_add_info_format(pinfo, wme_qos_info_item, &ei_ieee80211_wfa_ie_wme_qos_info_bad_ftype, "Could not deduce direction to decode correctly, ftype %u", ftype);
      break;
    }

  offset += 1;
  return offset;
}

static int * const update_edca_info_headers[] = {
  &hf_ieee80211_s1g_update_edca_override,
  &hf_ieee80211_s1g_update_edca_ps_poll_aci,
  &hf_ieee80211_s1g_update_edca_raw_aci,
  &hf_ieee80211_s1g_update_edca_sta_type,
  &hf_ieee80211_s1g_update_edca_reserved,
  NULL
};

static const value_string sta_field_type_vals[] = {
  { 0, "Valid for both sensor and non-sensor STAs" },
  { 1, "Valid for sensor STAs" },
  { 2, "Valid for non-sensor STAs" },
  { 3, "Reserved" },
  { 0, NULL }
};

static int
decode_qos_parameter_set(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int ftype)
{
  int i;
  bool is_s1g = sta_is_s1g(pinfo);
  /* WME QoS Info Field */
  offset = dissect_wme_qos_info(tree, tvb, pinfo, offset, ftype);
  /* WME Reserved Field or EDCA Update */
  if (is_s1g) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                            hf_ieee80211_s1g_update_edca_info,
                            ett_update_edca_info, update_edca_info_headers,
                            ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  } else {
    proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_reserved, tvb, offset, 1, ENC_NA);
  }

  offset += 1;
  /* AC Parameters */
  for (i = 0; i < 4; i++)
  {
    proto_item *ac_item, *aci_aifsn_item, *ecw_item, *cw_item;
    proto_tree *ac_tree, *ecw_tree;
    uint8_t aci_aifsn, ecw, ecwmin, ecwmax;
    uint16_t cwmin, cwmax;
    static int * const ieee80211_wfa_ie_wme_be[] = {
        &hf_ieee80211_wfa_ie_wme_acp_aci_be,
        &hf_ieee80211_wfa_ie_wme_acp_acm_be,
        &hf_ieee80211_wfa_ie_wme_acp_aifsn_be,
        &hf_ieee80211_wfa_ie_wme_acp_reserved_be,
        NULL
    };

    static int * const ieee80211_wfa_ie_wme_bk[] = {
        &hf_ieee80211_wfa_ie_wme_acp_aci_bk,
        &hf_ieee80211_wfa_ie_wme_acp_acm_bk,
        &hf_ieee80211_wfa_ie_wme_acp_aifsn_bk,
        &hf_ieee80211_wfa_ie_wme_acp_reserved_bk,
        NULL
    };

    static int * const ieee80211_wfa_ie_wme_vi[] = {
        &hf_ieee80211_wfa_ie_wme_acp_aci_vi,
        &hf_ieee80211_wfa_ie_wme_acp_acm_vi,
        &hf_ieee80211_wfa_ie_wme_acp_aifsn_vi,
        &hf_ieee80211_wfa_ie_wme_acp_reserved_vi,
        NULL
    };

    static int * const ieee80211_wfa_ie_wme_vo[] = {
        &hf_ieee80211_wfa_ie_wme_acp_aci_vo,
        &hf_ieee80211_wfa_ie_wme_acp_acm_vo,
        &hf_ieee80211_wfa_ie_wme_acp_aifsn_vo,
        &hf_ieee80211_wfa_ie_wme_acp_reserved_vo,
        NULL
    };

    static int * const * ie_wme_hdrs[] = {
       ieee80211_wfa_ie_wme_be,
       ieee80211_wfa_ie_wme_bk,
       ieee80211_wfa_ie_wme_vi,
       ieee80211_wfa_ie_wme_vo
    };

    static int * const ecw_max_hf[] = {
      &hf_ieee80211_wfa_ie_wme_acp_ecw_max_be,
      &hf_ieee80211_wfa_ie_wme_acp_ecw_max_bk,
      &hf_ieee80211_wfa_ie_wme_acp_ecw_max_vi,
      &hf_ieee80211_wfa_ie_wme_acp_ecw_max_vo
    };

    static int * const ecw_min_hf[] = {
      &hf_ieee80211_wfa_ie_wme_acp_ecw_min_be,
      &hf_ieee80211_wfa_ie_wme_acp_ecw_min_bk,
      &hf_ieee80211_wfa_ie_wme_acp_ecw_min_vi,
      &hf_ieee80211_wfa_ie_wme_acp_ecw_min_vo
    };

    static int * const txop_limit_hf[] = {
      &hf_ieee80211_wfa_ie_wme_acp_txop_limit_be,
      &hf_ieee80211_wfa_ie_wme_acp_txop_limit_bk,
      &hf_ieee80211_wfa_ie_wme_acp_txop_limit_vi,
      &hf_ieee80211_wfa_ie_wme_acp_txop_limit_vo
    };

    ac_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_ac_parameters, tvb, offset, 4, ENC_NA);
    ac_tree = proto_item_add_subtree(ac_item, ett_wme_ac);

    /* ACI/AIFSN Field */
    aci_aifsn_item = proto_tree_add_bitmask_with_flags(ac_tree, tvb, offset, hf_ieee80211_wfa_ie_wme_acp_aci_aifsn,
                            ett_wme_aci_aifsn, ie_wme_hdrs[i],
                            ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    aci_aifsn = tvb_get_uint8(tvb, offset);
    /* 802.11-2012, 8.4.2.31 EDCA Parameter Set element */
    if (aci_aifsn < 2) {
       expert_add_info_format(pinfo, aci_aifsn_item, &ei_ieee80211_qos_bad_aifsn,
         "The minimum value for the AIFSN subfield is 2 (found %u).", aci_aifsn);
    }
    proto_item_append_text(ac_item, " ACI %u (%s), ACM %s, AIFSN %u",
      (aci_aifsn & 0x60) >> 5, try_val_to_str((aci_aifsn & 0x60) >> 5, ieee80211_wfa_ie_wme_acs_vals),
      (aci_aifsn & 0x10) ? "yes" : "no", aci_aifsn & 0x0f);
    offset += 1;

    /* ECWmin/ECWmax field */
    ecw_item = proto_tree_add_item(ac_tree, hf_ieee80211_wfa_ie_wme_acp_ecw, tvb, offset, 1, ENC_NA);
    ecw_tree = proto_item_add_subtree(ecw_item, ett_wme_ecw);
    ecw = tvb_get_uint8(tvb, offset);
    ecwmin = ecw & 0x0f;
    ecwmax = (ecw & 0xf0) >> 4;
    cwmin= (1 << ecwmin) - 1;
    cwmax= (1 << ecwmax) - 1;
    cw_item = proto_tree_add_item(ecw_tree, *ecw_max_hf[i], tvb, offset, 1, ENC_NA);
    proto_item_append_text(cw_item, " (CW Max: %u)", cwmax);
    cw_item = proto_tree_add_item(ecw_tree, *ecw_min_hf[i], tvb, offset, 1, ENC_NA);
    proto_item_append_text(cw_item, " (CW Min: %u)", cwmin);
    proto_item_append_text(ac_item, ", ECWmin/max %u/%u (CWmin/max %u/%u)", ecwmin, ecwmax, cwmin, cwmax);
    offset += 1;

    /* TXOP Limit */
    proto_tree_add_item(ac_tree, *txop_limit_hf[i], tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_item_append_text(ac_item, ", TXOP %u", tvb_get_letohs(tvb, offset));
    offset += 2;
  }

  return offset;
}

static int
dissect_vendor_ie_wpawme(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, uint32_t tag_len, int ftype)
{
  uint8_t type;

  proto_tree_add_item_ret_uint8(tree, hf_ieee80211_wfa_ie_type, tvb, offset, 1, ENC_NA, &type);
  proto_item_append_text(tree, ": %s", val_to_str(pinfo->pool, type, ieee802111_wfa_ie_type_vals, "Unknown %d"));
  offset += 1;

  switch (type) {
    case 1:   /* Wi-Fi Protected Access (WPA) */
    {
      proto_item *wpa_mcs_item, *wpa_ucs_item, *wpa_akms_item;
      proto_item *wpa_sub_ucs_item, *wpa_sub_akms_item;
      proto_tree *wpa_mcs_tree, *wpa_ucs_tree, *wpa_akms_tree;
      proto_tree *wpa_sub_ucs_tree, *wpa_sub_akms_tree;
      uint16_t ucs_count, akms_count;
      unsigned ii;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_version, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      /* Multicast Cipher Suite */
      wpa_mcs_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_mcs, tvb, offset, 4, ENC_BIG_ENDIAN);
      wpa_mcs_tree = proto_item_add_subtree(wpa_mcs_item, ett_wpa_mcs_tree);
      proto_tree_add_item(wpa_mcs_tree, hf_ieee80211_wfa_ie_wpa_mcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

      /* Check if OUI is 00:50:F2 (WFA) */
      if (tvb_get_ntoh24(tvb, offset) == OUI_WPAWME)
      {
        proto_tree_add_item(wpa_mcs_tree, hf_ieee80211_wfa_ie_wpa_mcs_wfa_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
        save_proto_data_value(pinfo, tvb_get_uint8(tvb, offset + 3), GROUP_CIPHER_KEY);
      } else {
        proto_tree_add_item(wpa_mcs_tree, hf_ieee80211_wfa_ie_wpa_mcs_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
      }
      offset += 4;

      /* Unicast Cipher Suites */
      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_ucs_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      ucs_count = tvb_get_letohs(tvb, offset);
      offset += 2;

      wpa_ucs_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_ucs_list, tvb, offset, ucs_count * 4, ENC_NA);
      wpa_ucs_tree = proto_item_add_subtree(wpa_ucs_item, ett_wpa_ucs_tree);
      for (ii = 0; ii < ucs_count; ii++)
      {
        wpa_sub_ucs_item = proto_tree_add_item(wpa_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs, tvb, offset, 4, ENC_BIG_ENDIAN);
        wpa_sub_ucs_tree = proto_item_add_subtree(wpa_sub_ucs_item, ett_wpa_sub_ucs_tree);
        proto_tree_add_item(wpa_sub_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

        /* Check if OUI is 00:50:F2 (WFA) */
        if (tvb_get_ntoh24(tvb, offset) == OUI_WPAWME)
        {
          proto_tree_add_item(wpa_sub_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs_wfa_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
          proto_item_append_text(wpa_ucs_item, " %s", wpa_ucs_return(pinfo->pool, tvb_get_ntohl(tvb, offset)));
          save_proto_data_value(pinfo, tvb_get_uint8(tvb, offset + 3), CIPHER_KEY);
        } else {
          proto_tree_add_item(wpa_sub_ucs_tree, hf_ieee80211_wfa_ie_wpa_ucs_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
        }
        offset += 4;
      }

      /* Authenticated Key Management Suites */
      proto_tree_add_item_ret_uint16(tree, hf_ieee80211_wfa_ie_wpa_akms_count, tvb, offset, 2, ENC_LITTLE_ENDIAN, &akms_count);
      offset += 2;

      wpa_akms_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wpa_akms_list, tvb, offset, akms_count * 4, ENC_NA);
      wpa_akms_tree = proto_item_add_subtree(wpa_akms_item, ett_wpa_akms_tree);
      for (ii = 0; ii < akms_count; ii++)
      {
        wpa_sub_akms_item = proto_tree_add_item(wpa_akms_tree, hf_ieee80211_wfa_ie_wpa_akms, tvb, offset, 4, ENC_BIG_ENDIAN);
        wpa_sub_akms_tree = proto_item_add_subtree(wpa_sub_akms_item, ett_wpa_sub_akms_tree);
        proto_tree_add_item(wpa_sub_akms_tree, hf_ieee80211_wfa_ie_wpa_akms_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

        /* Check if OUI is 00:50:F2 (WFA) */
        if (tvb_get_ntoh24(tvb, offset) == OUI_WPAWME)
        {
          proto_tree_add_item(wpa_sub_akms_tree, hf_ieee80211_wfa_ie_wpa_akms_wfa_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
          proto_item_append_text(wpa_akms_item, " %s", wpa_akms_return(pinfo->pool, tvb_get_ntohl(tvb, offset)));
          save_proto_data_value(pinfo, tvb_get_uint8(tvb, offset + 3), AKM_KEY);
        } else {
          proto_tree_add_item(wpa_sub_akms_tree, hf_ieee80211_wfa_ie_wpa_akms_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
        }
        offset += 4;
      }
      break;
    }
    case 2:   /* Wireless Multimedia Enhancements (WME) */
    {
      uint8_t subtype;

      proto_tree_add_item_ret_uint8(tree, hf_ieee80211_wfa_ie_wme_subtype, tvb, offset, 1, ENC_NA, &subtype);
      proto_item_append_text(tree, ": %s", val_to_str(pinfo->pool, subtype, ieee802111_wfa_ie_wme_type, "Unknown %d"));
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_version, tvb, offset, 1, ENC_NA);
      offset += 1;
      switch (subtype) {
        case 0: /* WME Information Element */
        {
          /* WME QoS Info Field */
          offset = dissect_wme_qos_info(tree, tvb, pinfo, offset, ftype);
          break;
        }
        case 1: /* WME Parameter Element */
        {
          offset = decode_qos_parameter_set(tree, tvb, pinfo, offset, ftype);
          break;
        }
        case 2:   /* WME TSPEC Element */
        {
            static int * const ieee80211_wfa_ie_wme_tspec_tsinfo[] = {
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_tid,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_direction,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_psb,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_up,
              &hf_ieee80211_wfa_ie_wme_tspec_tsinfo_reserved,
              NULL
            };

            proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_wfa_ie_wme_tspec_tsinfo,
                                    ett_tsinfo_tree, ieee80211_wfa_ie_wme_tspec_tsinfo,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset += 3;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_nor_msdu, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_max_msdu, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_min_srv, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_max_srv, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_inact_int, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_susp_int, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_srv_start, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_min_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_mean_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_peak_data, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_burst_size, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_delay_bound, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_min_phy, tvb, offset, 4, ENC_LITTLE_ENDIAN);
            offset += 4;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_surplus, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

            proto_tree_add_item(tree, hf_ieee80211_wfa_ie_wme_tspec_medium, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset += 2;

          break;
        }
        default:
          /* No default Action */
        break;
      } /* End switch (subtype) */
      break;
    }
    case 4: /* WPS: Wifi Protected Setup */
    {
      dissect_wps_tlvs(tree, tvb, offset, tag_len-1, pinfo, false);
    }
    break;
    case 17: /* Network Cost: https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nct/88f0cdf4-cdf2-4455-b849-4abf1e5c11ac */
    {
      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_nc_cost_level, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_nc_reserved, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_nc_cost_flags, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_nc_reserved, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
    }
    break;
    case 18: /* Tethering: https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nct/a097f5bb-6eca-44ad-9a02-20d46ad30d6d */
    {
      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_tethering_type, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_tethering_mac_length, tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      proto_tree_add_item(tree, hf_ieee80211_wfa_ie_tethering_mac, tvb, offset, 6, ENC_NA);
      offset += 6;

    }
    break;
    default:
      /* No default Action...*/
    break;
  } /* End switch (type) */

  return offset;
}

/*
 * Dissect a group data cipher suite which consists of an OUI and a one-byte
 * selector: IEEE802.11 2012 Figure 9-256.
 *
 * Accepts a two entry array of header fields so we can use this elsewhere.
 */
static int dissect_group_data_cipher_suite(tvbuff_t *tvb, packet_info *pinfo _U_,
    proto_tree *tree, int offset, int *hf_array, int ett_val, char *label)
{
  proto_tree *gdcs_tree = NULL;

  gdcs_tree = proto_tree_add_subtree(tree, tvb, offset, 4, ett_val, NULL,
                         label);
  proto_tree_add_item(gdcs_tree, hf_array[0], tvb, offset, 3, ENC_BIG_ENDIAN);
  offset += 3;
  proto_tree_add_item(gdcs_tree, hf_array[1], tvb, offset, 1, ENC_NA);
  offset += 1;

  return offset;
}

static int
dissect_rsn_ie(packet_info *pinfo, proto_tree *tree, tvbuff_t *tvb,
               int offset, uint32_t tag_len, association_sanity_check_t *association_sanity_check);

static int
dissect_wfa_rsn_override(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);

  if (tag_len > 0)
    dissect_rsn_ie(pinfo, tree, tvb, 0, tag_len, NULL);

  return tag_len;
}

static int
dissect_wfa_rsn_override_2(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);

  if (tag_len > 0)
    dissect_rsn_ie(pinfo, tree, tvb, 0, tag_len, NULL);

  return tag_len;
}

static int
dissect_rsnx_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int tag_len);

static int
dissect_wfa_rsnx_override(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);

  if (tag_len > 0)
    dissect_rsnx_ie(tvb, pinfo, tree, tag_len);

  return tag_len;
}

static int
dissect_wfa_rsn_selection(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);

  proto_tree_add_item(tree, hf_ieee80211_wfa_rsn_selection, tvb, 0,
                      1, ENC_NA);

  return tag_len;
}

static int
dissect_wfa_rsn_override_link_kde(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_wfa_rsn_or_link_kde_link_id, tvb, offset,
                      1, ENC_NA);
  offset++;
  ieee_80211_add_tagged_parameters(tvb, offset, pinfo, tree,
                                   tag_len - 1, -1, NULL);

  return tag_len;
}

static const range_string qos_mgmt_attributes[] = {
  { 0, 0, "Reserved" },
  { 1, 1, "Port Range" },
  { 2, 2, "DSCP Policy" },
  { 3, 3, "TCLAS" },
  { 4, 4, "Domain Name" },
  { 5, 255, "Reserved" },
  { 0, 0, NULL }
};

static int
ieee80211_frame_classifier(tvbuff_t *tvb, packet_info *pinfo _U_,
                           proto_tree *tree, int offset, int tag_len);

static int
dissect_qos_mgmt(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  uint8_t attr_id;
  uint8_t attr_len;
  uint8_t attr_num = 0;
  proto_tree *sub_tree = NULL;

  while (tvb_captured_length_remaining(tvb, offset)) {
    attr_id = tvb_get_uint8(tvb, offset);
    attr_len = tvb_get_uint8(tvb, offset + 1);
    proto_tree *attr = NULL;

    attr = proto_tree_add_subtree_format(tree, tvb, offset, attr_len + 2,
                                  ett_qos_mgmt_attributes, NULL,
                                  "QoS Management Attribute %d", attr_num++);
    proto_tree_add_item(attr, hf_ieee80211_qos_mgmt_attribute_id, tvb, offset,
                        1, ENC_NA);
    offset += 1;

    proto_tree_add_item(attr, hf_ieee80211_qos_mgmt_attribute_len, tvb, offset,
                        1, ENC_NA);
    offset += 1;

    switch (attr_id) {
    case 1:
      sub_tree = proto_tree_add_subtree(attr, tvb, offset, 1,
                          ett_qos_mgmt_dscp_policy_capabilities, NULL,
                          "Port Range");
      proto_tree_add_item(sub_tree, hf_ieee80211_qos_mgmt_start_port_range,
                          tvb, offset, 2, ENC_BIG_ENDIAN);
      offset += 2;

      proto_tree_add_item(sub_tree, hf_ieee80211_qos_mgmt_end_port_range, tvb,
                          offset, 2, ENC_BIG_ENDIAN);
      offset += 2;
      break;
    case 2:
      sub_tree = proto_tree_add_subtree(attr, tvb, offset, attr_len,
                          ett_qos_mgmt_dscp_policy, NULL,
                          "DSCP Policy");

      proto_tree_add_item(sub_tree, hf_ieee80211_qos_mgmt_dscp_pol_id, tvb,
                          offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(sub_tree, hf_ieee80211_qos_mgmt_dscp_pol_req_type,
                          tvb, offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(sub_tree, hf_ieee80211_qos_mgmt_dscp_pol_dscp, tvb,
                          offset, 1, ENC_NA);
      offset += 1;
      break;
    case 3:
      sub_tree = proto_tree_add_subtree(attr, tvb, offset, attr_len,
                          ett_qos_mgmt_tclas, NULL,
                          "TCLAS");

      ieee80211_frame_classifier(tvb, pinfo, sub_tree, offset, attr_len);
      offset += attr_len;
      break;
    case 4:
      sub_tree = proto_tree_add_subtree(attr, tvb, offset, attr_len,
                          ett_qos_mgmt_domain_name, NULL,
                          "Domain Name");
      proto_tree_add_item(sub_tree, hf_ieee80211_qos_mgmt_domain_name, tvb,
                          offset, attr_len, ENC_ASCII);
      offset += attr_len;
      break;
    default:
      sub_tree = proto_tree_add_subtree(attr, tvb, offset, attr_len,
                          ett_qos_mgmt_unknown_attribute, NULL,
                          "Unknown attribute");
      proto_tree_add_item(sub_tree, hf_ieee80211_qos_mgmt_unknown_attr, tvb,
                          offset, attr_len, ENC_NA);
      offset += attr_len;
      break;
    }
  }

  return offset;
}

static int * const wfa_capa_fields[] = {
  &hf_ieee80211_wfa_capa_mgmt_dscp_policy,
  &hf_ieee80211_wfa_capa_mgmt_unsol_dscp_policy,
  &hf_ieee80211_wfa_capa_mgmt_scs_traffic_desc,
  &hf_ieee80211_wfa_capa_5g_wifi_qos_mapping,
  &hf_ieee80211_wfa_capa_data_plane_stat_report,
  &hf_ieee80211_wfa_capa_radio_counters_stat_support,
  &hf_ieee80211_wfa_capa_control_plane_stat_report,
  &hf_ieee80211_wfa_capa_unsolicited_report_support,
  NULL
};

static int * const wfa_capa_supp_gene[] = {
  &hf_ieee80211_wfa_capa_attr_supp_generations_b0,
  &hf_ieee80211_wfa_capa_attr_supp_generations_b1,
  &hf_ieee80211_wfa_capa_attr_supp_generations_b2,
  &hf_ieee80211_wfa_capa_attr_supp_generations_b3,
  &hf_ieee80211_wfa_capa_attr_supp_generations_reserved,
  NULL
};

static int * const wfa_capa_cert_gene[] = {
  &hf_ieee80211_wfa_capa_attr_cert_generations_b0,
  &hf_ieee80211_wfa_capa_attr_cert_generations_b1,
  &hf_ieee80211_wfa_capa_attr_cert_generations_b2,
  &hf_ieee80211_wfa_capa_attr_cert_generations_b3,
  &hf_ieee80211_wfa_capa_attr_cert_generations_reserved,
  NULL
};

static const range_string wfa_capa_attr_id[] = {
  { 0, 0, "Reserved" },
  { 1, 1, "Generational Capabilities Indication attribute" },
  { 2, 220, "Reserved" },
  { 221, 221, "Vendor Specific attribute" },
  { 222, 225, "Reserved" },
  { 0, 0, NULL }
};

static int
dissect_wfa_capa(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  uint32_t attr_id = 0, attr_len = 0;
  unsigned offset = 0, tag_len = tvb_captured_length_remaining(tvb, offset);
  proto_tree *attr_tree = NULL;
  uint32_t capa_len = 0;

  if (tag_len < 2) {
    proto_item *tag_data;

    tag_data = proto_tree_add_item(tree, hf_ieee80211_wfa_capa_tag_data, tvb,
                                   offset, tag_len, ENC_NA);
    expert_add_info_format(pinfo, tag_data, &ei_ieee80211_tag_length,
                           "WFA Capabilities length %u too short, must be >= 2", tag_len);

    return tag_len;
  }

  /* There is at least two bytes, a length and the capabilities. */
  /* Capa length can be 0 */
  proto_tree_add_item_ret_uint(tree, hf_ieee80211_wfa_capa_tag_len, tvb, offset,
                               1, ENC_NA, &capa_len);
  offset += 1;

  if (capa_len > 0) {
    proto_tree_add_bitmask(tree, tvb, offset,
                           hf_ieee80211_wfa_capa_tag_capabilities,
                           ett_wfa_capa, wfa_capa_fields,
                           ENC_NA);
    offset += capa_len;
  }

  if (tag_len > offset)
    attr_tree = proto_tree_add_subtree(tree, tvb, offset, tag_len - offset,
                                       ett_wfa_capa_attributes, NULL,
                                       "Attributes");
  /* Deal with the attributes */
  while (tag_len > offset) {
    proto_tree_add_item_ret_uint(attr_tree, hf_ieee80211_wfa_capa_attr_id, tvb,
                                 offset, 1, ENC_NA, &attr_id);
    offset += 1;

    proto_tree_add_item_ret_uint(attr_tree, hf_ieee80211_wfa_capa_attr_len, tvb,
                                 offset, 1, ENC_NA, &attr_len);
    offset += 1;

    if (attr_id == 1) {/* Generational Capa indication */
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_capa_attr_supp_gene_length,
                          tvb, offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_bitmask_with_flags(attr_tree, tvb, offset,
                                        hf_ieee80211_wfa_capa_attr_supp_generations,
                                        ett_wfa_capa_supp_gene, wfa_capa_supp_gene,
                                        ENC_NA, BMT_NO_APPEND);
      offset += 1;

      if (attr_len > 2) {
        proto_tree_add_item(attr_tree, hf_ieee80211_wfa_capa_attr_cert_gene_length,
                            tvb, offset, 1, ENC_NA);
        offset += 1;

        proto_tree_add_bitmask_with_flags(attr_tree, tvb, offset,
                                          hf_ieee80211_wfa_capa_attr_cert_generations,
                                          ett_wfa_capa_cert_gene, wfa_capa_cert_gene,
                                          ENC_NA, BMT_NO_APPEND);
        offset += 1;
      }
    } else {
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_capa_attr_field, tvb,
                          offset, attr_len, ENC_NA);
      offset += attr_len;
    }
  }

  return offset;
}

/*
 * Handle the HS 2.0 rev 2 OSU Server-only authenticated layer 2 Encryption
 * Network element. This is almost the same format as the RSNE so maybe some
 * common code can be used.
 */
static int
dissect_hs20_osen(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  int hf_array[2] = { hf_ieee80211_group_data_cipher_suite_oui,
                       hf_ieee80211_group_data_cipher_suite_type };
  proto_tree *pwc_list = NULL;
  proto_item *pwcsi = NULL;
  uint16_t pwc_count = 0, pwc_index = 0;
  uint16_t akms_count = 0, akms_index = 0;
  static int * const osen_rsn_cap[] = {
    &hf_ieee80211_osen_rsn_cap_preauth,
    &hf_ieee80211_osen_rsn_cap_no_pairwise,
    &hf_ieee80211_osen_rsn_cap_ptksa_replay_counter,
    &hf_ieee80211_osen_rsn_cap_gtksa_replay_counter,
    &hf_ieee80211_osen_rsn_cap_mfpr,
    &hf_ieee80211_osen_rsn_cap_mfpc,
    &hf_ieee80211_osen_rsn_cap_jmr,
    &hf_ieee80211_osen_rsn_cap_peerkey,
    &hf_ieee80211_osen_rsn_spp_a_msdu_capable,
    &hf_ieee80211_osen_rsn_spp_a_msdu_required,
    &hf_ieee80211_osen_rsn_pbac,
    &hf_ieee80211_osen_extended_key_id_iaf,
    &hf_ieee80211_osen_reserved,
    NULL
  };
  uint16_t pmkid_count = 0, pmkid_index = 0;
  int gmcs_array[2] = { hf_ieee80211_osen_group_management_cipher_suite_oui,
                        hf_ieee80211_osen_group_management_cipher_suite_type };

  offset = dissect_group_data_cipher_suite(tvb, pinfo, tree, offset, hf_array,
                        ett_osen_group_data_cipher_suite,
                        "OSEN Group Data Cipher Suite");

  pwc_count = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_osen_pcs_count, tvb, offset,
                      2, ENC_LITTLE_ENDIAN);
  offset += 2;

  if (pwc_count > 0) {
    int start_offset = offset;
    pwc_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_osen_pairwise_cipher_suites, &pwcsi,
                        "OSEN Pairwise Cipher Suite List");

    while (pwc_count > 0) {
      if (tvb_reported_length_remaining(tvb, offset) >= 4) {
        int hf_array2[2] = { hf_ieee80211_osen_pairwise_cipher_suite_oui,
                            hf_ieee80211_osen_pairwise_cipher_suite_type };
        char label[128];

        snprintf(label, sizeof(label), "OSEN Pairwise Cipher Suite %d", pwc_index);
        offset = dissect_group_data_cipher_suite(tvb, pinfo, pwc_list,
                        offset, hf_array2, ett_osen_pairwise_cipher_suite,
                        label);
        pwc_index++;
        pwc_count--;
      } else {
        /* Insert the remaining? Expert Info? */
        offset += tvb_reported_length_remaining(tvb, offset);
        break;
      }
    }

    proto_item_set_len(pwcsi, offset - start_offset);
  }

  if (tvb_reported_length_remaining(tvb, offset) == 0) {
    return tvb_captured_length(tvb);
  }

  /* Now handle the AKM Suites */
  akms_count = tvb_get_letohs(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_osen_akm_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  if (akms_count > 0) {
    int start_offset = offset;
    proto_tree *akm_list = NULL;
    proto_item *akmcsi = NULL;

    akm_list = proto_tree_add_subtree(tree, tvb, offset, -1,
                        ett_osen_akm_cipher_suites, &akmcsi,
                        "OSEN AKM Cipher Suite List");

    while (akms_count > 0) {
      if (tvb_reported_length_remaining(tvb, offset) >= 4) {
        int hf_array3[2] = { hf_ieee80211_osen_akm_cipher_suite_oui,
                             hf_ieee80211_osen_akm_cipher_suite_type};
        char label[128];

        snprintf(label, sizeof(label), "OSEN AKM Cipher Suite %d", akms_index);
        offset = dissect_group_data_cipher_suite(tvb, pinfo, akm_list,
                          offset, hf_array3, ett_osen_akm_cipher_suite,
                          label);
        akms_index++;
        akms_count--;
      } else {
        /* Expert info? */
        offset += tvb_reported_length_remaining(tvb, offset);
        break;
      }
    }
    proto_item_set_len(akmcsi, offset - start_offset);
  }

  /* Any more? */
  if (tvb_reported_length_remaining(tvb, offset) == 0) {
    return tvb_captured_length(tvb);
  }

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_osen_rsn_cap_flags,
                                    ett_osen_rsn_cap_tree, osen_rsn_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* Any more? */
  if (tvb_reported_length_remaining(tvb, offset) == 0) {
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item_ret_uint16(tree, hf_ieee80211_osen_pmkid_count, tvb, offset, 2, ENC_LITTLE_ENDIAN, &pmkid_count);
  offset += 2;

  if (pmkid_count > 0) {
    proto_tree *pmkid_list = NULL;

    pmkid_list = proto_tree_add_subtree(tree, tvb, offset, pmkid_count * 16,
                                ett_osen_pmkid_list, NULL,
                                "OSEN PKMID List");

    while (pmkid_count > 0) {
      proto_tree *pmkid_tree = NULL;

      pmkid_tree = proto_tree_add_subtree_format(pmkid_list, tvb,offset, 16,
                                ett_osen_pmkid_tree, NULL,
                                "OSEN PKMID %d", pmkid_index);
      proto_tree_add_item(pmkid_tree, hf_ieee80211_osen_pmkid, tvb, offset, 16,
                          ENC_NA);
      offset += 16;
      pmkid_index++;
      pmkid_count--;
    }
  }

  offset = dissect_group_data_cipher_suite(tvb, pinfo, tree, offset, gmcs_array,
                        ett_osen_group_management_cipher_suite,
                        "OSEN Group Management Cipher Suite");

  return offset;
}

static const value_string hs20_indication_version_number_vals[] = {
  { 0, "1.x" },
  { 1, "2.x" },
  { 2, "3.x" },
  { 0, NULL }
};

static int
dissect_hs20_indication(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  static int * const ieee80211_hs20_indication[] = {
    &hf_ieee80211_hs20_indication_dgaf_disabled,
    &hf_ieee80211_hs20_indication_pps_mo_id_present,
    &hf_ieee80211_hs20_indication_anqp_domain_id_present,
    &hf_ieee80211_hs20_reserved,
    &hf_ieee80211_hs20_indication_version_number,
    NULL
  };
  int len = tvb_captured_length(tvb);
  int offset = 0;
  uint8_t indic = tvb_get_uint8(tvb, offset);

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_hs20_indication,
                              ENC_NA);
  offset++;

  if (len >= 3 && (indic & 0x02)) { /* Contains a PPS MO ID field ... display it. */
      proto_tree_add_item(tree, hf_ieee80211_hs20_indication_pps_mo_id, tvb, offset,
                          2, ENC_LITTLE_ENDIAN);
      offset += 2;
  }

  if ((len >= (offset + 2)) && (indic & 0x04)) {
     proto_tree_add_item(tree, hf_ieee80211_hs20_indication_anqp_domain_id, tvb, offset,
                         2, ENC_LITTLE_ENDIAN);
     offset += 2;
  }

  return offset;
}

enum ieee80211_wfa_60g_attr {
  /* 0 Reserved */
  WIFI_60G_ATTR_CAPABILITY = 1,
  /* 2 - 225 Reserved */
};

static const value_string ieee80211_wfa_60g_attr_ids[] = {
  { WIFI_60G_ATTR_CAPABILITY, "60GHz Capability" },
  { 0, NULL }
};

static int
dissect_wfa_60g_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void *data _U_)
{
  int end = tvb_reported_length(tvb);
  int offset = 0;
  uint8_t id;
  uint16_t len;
  proto_tree *wf60g_tree;
  proto_item *attrs;

  while (offset < end) {
    if (end - offset < 2) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_wfa_60g_attr_len_invalid, "Packet too short for Wi-Fi 60G attribute");
      break;
    }

    id = tvb_get_uint8(tvb, offset);
    len = tvb_get_ntohs(tvb, offset + 1);
    attrs = proto_tree_add_item(tree, hf_ieee80211_wfa_60g_attr, tvb, offset, 0, ENC_NA);
    proto_item_append_text(attrs, ": %s", val_to_str(pinfo->pool, id, ieee80211_wfa_60g_attr_ids,
                                             "Unknown attribute ID (%u)"));
    wf60g_tree = proto_item_add_subtree(attrs, ett_ieee80211_wfa_60g_attr);
    proto_tree_add_item(wf60g_tree, hf_ieee80211_wfa_60g_attr_id, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;
    proto_tree_add_item(wf60g_tree, hf_ieee80211_wfa_60g_attr_len, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;


    switch (id) {
    case WIFI_60G_ATTR_CAPABILITY:
      if (len - offset < 7) {
        expert_add_info_format(pinfo, tree, &ei_ieee80211_wfa_60g_attr_len_invalid, "Packet too short for 60G capability attribute");
        break;
      }

      proto_tree_add_item(wf60g_tree, hf_ieee80211_wfa_60g_attr_cap_sta_mac_addr, tvb, offset, 6, ENC_NA);
      offset += 6;
      proto_tree_add_item(wf60g_tree, hf_ieee80211_wfa_60g_attr_cap_recv_amsdu_frames, tvb, offset, 1, ENC_BIG_ENDIAN);
      proto_tree_add_item(wf60g_tree, hf_ieee80211_wfa_60g_attr_cap_reserved, tvb, offset, 1, ENC_BIG_ENDIAN);
      offset += 1;
      break;
    default:
      proto_tree_add_expert_format(tree, pinfo, &ei_ieee80211_wfa_60g_unknown_attribute, tvb,
                                         offset, len+2, "Unknown attribute ID (%u)", id);
    }

    offset += len;
  }
  return offset;
}

static int
dissect_owe_transition_mode(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  uint8_t ssid_len;

  int len = tvb_captured_length(tvb);
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_bssid, tvb, offset, 6, ENC_NA);
  offset  += 6;
  len -= 6;

  ssid_len = tvb_get_uint8(tvb, offset);

  proto_tree_add_uint(tree, hf_ieee80211_wfa_ie_owe_ssid_length, tvb, offset, 1, ssid_len);
  offset  += 1;
  len -= 1;

  if (len < ssid_len) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
    return offset;
  }

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_ssid, tvb, offset, ssid_len, ENC_ASCII);
  offset  += len;
  len -= len;

  if (len >= 2) {
    proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_band_info, tvb, offset, 1, ENC_NA);
    offset  += 1;

    proto_tree_add_item(tree, hf_ieee80211_wfa_ie_owe_channel_info, tvb, offset, 1, ENC_NA);
    offset  += 1;
  }

  return offset;
}

static int
dissect_transition_disable_kde(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_captured_length(tvb);
  int offset = 0;
  static int * const ieee80211_wfa_transition_disable_flags[] = {
    &hf_ieee80211_wfa_ie_transition_disable_wpa3_personal,
    &hf_ieee80211_wfa_ie_transition_disable_sae_pk,
    &hf_ieee80211_wfa_ie_transition_disable_wpa3_enterprise,
    &hf_ieee80211_wfa_ie_transition_disable_enhanced_open,
    &hf_ieee80211_wfa_ie_transition_disable_reserved_b4thru7,
    NULL
  };

  if (tag_len < 1) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
    return 0;
  }

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_wfa_ie_transition_disable_bitmap,
                                    ett_ieee80211_wfa_transition_disable_tree,
                                    ieee80211_wfa_transition_disable_flags,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset++;

  if (offset < tag_len)
    proto_tree_add_item(tree, hf_ieee80211_wfa_ie_transition_disable_reserved, tvb, offset, tag_len-offset, ENC_NA);
  offset = tag_len;

  return offset;
}

static int
dissect_mbo_oce(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int len = tvb_reported_length(tvb);
  int offset = 0;

  while (len >= 2) {
    proto_item *attr_item;
    proto_tree *attr_tree;
    uint8_t attr_id = tvb_get_uint8(tvb, offset);
    uint8_t attr_len = tvb_get_uint8(tvb, offset + 1);

    if (len < (attr_len + 2)) {
      expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
      return offset;
    }

    attr_item = proto_tree_add_item(tree, hf_ieee80211_wfa_ie_mbo_oce_attr, tvb, offset, attr_len + 2, ENC_NA);
    attr_tree = proto_item_add_subtree(attr_item, ett_mbo_oce_attr);
    proto_item_append_text(attr_item, " (%s)", val_to_str_const(attr_id, wfa_mbo_oce_attr_id_vals, "Unknown"));

    proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_oce_attr_id, tvb, offset, 1, ENC_NA);
    offset += 1;
    len -= 1;

    proto_tree_add_uint(attr_tree, hf_ieee80211_wfa_ie_mbo_oce_attr_len, tvb, offset, 1, attr_len);
    offset  += 1;
    len -= 1;

    switch (attr_id) {
    case MBO_AP_CAPABILITY_INDICATION:
    {
      proto_item *cap_item;
      proto_tree *cap_tree;

      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      cap_item = proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_ap_cap, tvb, offset, 1, ENC_NA);
      cap_tree = proto_item_add_subtree(cap_item, ett_mbo_ap_cap);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_mbo_ap_cap_cell, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_mbo_ap_cap_reserved, tvb, offset, 1, ENC_NA);
      break;
    }
    case MBO_NON_PREF_CHANNEL_REPORT:
      if (attr_len == 0)
        break;

      if (attr_len < 3) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_op_class, tvb, offset, 1, ENC_NA);
      offset += 1;
      len -= 1;
      attr_len -= 1;
      while (attr_len > 2) {
        proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_chan, tvb, offset, 1, ENC_NA);
        offset += 1;
        len -= 1;
        attr_len -= 1;
      }

      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_pref, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_reason, tvb, offset + 1, 1, ENC_NA);
      break;
    case MBO_CELLULAR_DATA_CAPABILITIES:
      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_cellular_cap, tvb, offset, 1, ENC_NA);
      break;
    case MBO_ASSOCIATION_DISALLOWED:
      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_assoc_disallow_reason, tvb, offset, 1, ENC_NA);
      break;
    case MBO_CELLULAR_DATA_PREFERENCE:
      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_cellular_pref, tvb, offset, 1, ENC_NA);
      break;
    case MBO_TRANSITION_REASON:
      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_transition_reason, tvb, offset, 1, ENC_NA);
      break;
    case MBO_TRANSITION_REJECTION_REASON:
      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_transition_rej_reason, tvb, offset, 1, ENC_NA);
      break;
    case MBO_ASSOCIATION_RETRY_DELAY:
    {
      if (attr_len != 2) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_mbo_assoc_retry_delay, tvb, offset,
                          2, ENC_LITTLE_ENDIAN);
      break;
    }
    case OCE_CAPABILITY_INDICATION:
    {
      proto_item *cap_item;
      proto_tree *cap_tree;

      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      cap_item = proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_oce_cap_ctrl, tvb, offset, 1, ENC_NA);
      cap_tree = proto_item_add_subtree(cap_item, ett_oce_cap);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_cap_release, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_cap_sta_cfon, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_cap_11b_only_ap, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_cap_hlp, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_cap_non_oce_ap, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_cap_reserved, tvb, offset, 1, ENC_NA);
      break;
    }
    case OCE_RSSI_ASSOCIATION_REJECTION:
    {
      if (attr_len != 2) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_oce_rssi_assoc_rej_delta, tvb,
                          offset, 1, ENC_NA);
      proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_oce_rssi_assoc_rej_delay, tvb,
                          offset + 1, 1, ENC_NA);
      break;
    }
    case OCE_REDUCED_WAN_METRICS:
    {
      proto_item *cap_item;
      proto_tree *cap_tree;
      uint8_t capacity;

      if (attr_len != 1) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      cap_item = proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_oce_wan_metrics_avail_cap,
                                     tvb, offset, 1, ENC_NA);
      cap_tree = proto_item_add_subtree(cap_item, ett_oce_metrics_cap);

      capacity = tvb_get_uint8(tvb, offset);
      cap_item = proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_wan_metrics_avail_cap_downlink,
                                     tvb, offset, 1, ENC_NA);
      proto_item_append_text(cap_item, " (%d kbit/s)", (1 << (capacity & 0xF)) * 100);
      cap_item = proto_tree_add_item(cap_tree, hf_ieee80211_wfa_ie_oce_wan_metrics_avail_cap_uplink,
                                     tvb, offset, 1, ENC_NA);
      capacity >>= 4;
      proto_item_append_text(cap_item, " (%d kbit/s)", (1 << (capacity & 0xF)) * 100);
      break;
    }
    case OCE_RNR_COMPLETENESS:
      while (attr_len >= 4) {
        proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_oce_rnr_completeness_short_ssid,
                            tvb, offset, 4, ENC_ASCII);
        offset += 4;
        attr_len -= 4;
        len -= 4;
      }
      break;
    case OCE_PROBE_SUPPR_BSSID:
      while (attr_len >= 6) {
        proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_oce_probe_suppr_bssid,
                            tvb, offset, 6, ENC_NA);
        offset += 6;
        attr_len -= 6;
        len -= 6;
      }
      break;
    case OCE_PROBE_SUPPR_SSID:
      if (attr_len < 4) {
        expert_add_info(pinfo, attr_tree, &ei_ieee80211_bad_length);
        return offset;
      }
      while (attr_len >= 4) {
        proto_tree_add_item(attr_tree, hf_ieee80211_wfa_ie_oce_probe_suppr_ssid,
                            tvb, offset, 4, ENC_ASCII);
        offset += 4;
        attr_len -= 4;
        len -= 4;
      }
      break;
    default:
      break;
    }

    offset += attr_len;
    len -= attr_len;
  }

  if (len != 0) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
  }

  return offset;
}

static int
dissect_wfa_wnm_non_pref_chan(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int len = tvb_reported_length(tvb);
  int offset = 0;

  if (len == 0)
    return 0;

  if (len < 3) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
    return 0;
  }

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_op_class, tvb, offset, 1, ENC_NA);
  offset ++;
  len --;
  while (len > 2) {
    proto_tree_add_item(tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_chan, tvb, offset, 1, ENC_NA);
    offset ++;
    len --;
  }

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_pref, tvb, offset, 1, ENC_NA);
  offset ++;
  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_mbo_non_pref_chan_reason, tvb, offset, 1, ENC_NA);
  offset ++;
  return offset;
}

static int
dissect_wfa_wnm_cell_cap(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int len = tvb_reported_length(tvb);

  if (len != 1) {
    expert_add_info(pinfo, tree, &ei_ieee80211_bad_length);
    return 0;
  }

  proto_tree_add_item(tree, hf_ieee80211_wfa_ie_mbo_cellular_cap, tvb, 0, 1, ENC_NA);
  return len;
}

static void
dissect_vendor_ie_wfa(packet_info *pinfo, proto_item *item, tvbuff_t *tag_tvb)
{
  int tag_len = tvb_reported_length(tag_tvb);
  int dissect;
  uint8_t subtype;
  int offset = 0;
  tvbuff_t *vendor_tvb;

  if (tag_len < 4)
    return;

  subtype = tvb_get_uint8(tag_tvb, 3);
  proto_item_append_text(item, ": %s", val_to_str_const(subtype, wfa_subtype_vals, "Unknown"));
  vendor_tvb = tvb_new_subset_length(tag_tvb, offset + 4, tag_len - 4);
  dissect = dissector_try_uint_with_data(wifi_alliance_ie_table, subtype, vendor_tvb, pinfo, item, false, NULL);
  if (dissect <= 0) {
      proto_tree_add_item(item, hf_ieee80211_tag_vendor_data, vendor_tvb, 0, tag_len - 4, ENC_NA);
  }
}

static const range_string kde_selectors_rvals[] = {
  { 0, 0, "Reserved" },
  { 1, 1, "GTK KDE" },
  { 2, 2, "Reserved" },
  { 3, 3, "MAC address KDE" },
  { 4, 4, "PMKID KDE" },
  { 5, 5, "Reserved" },
  { 6, 6, "Nonce KDE" },
  { 7, 7, "Lifetime KDE" },
  { 8, 8, "Error KDE" },
  { 9, 9, "IGTK KDE" },
  { 10, 10, "Key ID KDE" },
  { 11, 11, "Multi-band GTK KDE" },
  { 12, 12, "Multi-band Key ID KDE" },
  { 13, 13, "OCI KDE" },
  { 14, 14, "BIGTK KDE" },
  { 15, 15, "Reserved" },
  { 16, 16, "MLO GTK KDE" },
  { 17, 17, "MLO IGTK KDE" },
  { 18, 18, "MLO BIGTK KDE" },
  { 19, 19, "MLO LINK KDE" },
  { 20, 255, "Reserved" },
  { 0, 0, NULL }
};

static const true_false_string tfs_rsn_gtk_kde_tx = {
  "Temporal key used for both transmission and reception",
  "Temporal key used only for reception"
};

static int * const mlo_kde_link_hdrs[] = {
  &hf_ieee80211_rsn_ie_mlo_linkid,
  &hf_ieee80211_rsn_ie_mlo_rnse_present,
  &hf_ieee80211_rsn_ie_mlo_rnsxe_present,
  &hf_ieee80211_rsn_ie_mlo_reserved,
  NULL
};

static void
dissect_rsn_ie_mlo_link(proto_item *item, proto_tree *tree, tvbuff_t *tvb,
                        int offset, uint32_t tag_len _U_, packet_info *pinfo)
{
  uint8_t info = tvb_get_uint8(tvb, offset);
  int mlo_links = GPOINTER_TO_UINT(p_get_proto_data(pinfo->pool, pinfo, proto_wlan, MLO_LINK_COUNT));

  proto_tree_add_bitmask(tree, tvb, offset,
                         hf_ieee80211_rsn_ie_mlo_link_info,
                         ett_kde_mlo_link_info, mlo_kde_link_hdrs,
                         ENC_NA);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_mac_addr, tvb, offset, 6,
                      ENC_NA);
  if (mlo_links < DOT11DECRYPT_MAX_MLO_LINKS) {
    uint8_t link_id = info & 0x0f;
    save_proto_data_value(pinfo, link_id, MLO_LINK_ID_FIRST + mlo_links);
    save_proto_data(tvb, pinfo, offset, 6, MLO_LINK_MAC_FIRST + mlo_links);
    save_proto_data_value(pinfo, mlo_links + 1, MLO_LINK_COUNT);
  }
  offset += 6;
  if ((info & 0x10) == 0x10) { /* Add the RSNE if present */
    offset += add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
  }

  if ((info & 0x20) == 0x20) { /* Add the RSNXE if present */
    add_tagged_field(pinfo, tree, tvb, offset, 0, NULL, 0, NULL);
  }

  proto_item_append_text(item, ": MLO Link KDE");
}

static void
dissect_vendor_ie_rsn(proto_item * item, proto_tree * tree, tvbuff_t * tvb,
                      int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint8_t data_type = tvb_get_uint8(tvb, offset);
  int mlo_gtk_nb;
  uint8_t flags;
  proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_data_type, tvb,
                      offset, 1, ENC_NA);
  offset += 1;

  switch(data_type) {
    case 1:
    {
      /* IEEE 802.11i / Key Data Encapsulation / Data Type=1 - GTK.
       * This is only used within EAPOL-Key frame Key Data. */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_key_id, tvb,
                          offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_tx, tvb, offset,
                          1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_reserved1, tvb,
                          offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_reserved2, tvb,
                          offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_gtk, tvb, offset,
                          tag_len - 3, ENC_NA);
      proto_item_append_text(item, ": RSN GTK");
      save_proto_data(tvb, pinfo, offset, tag_len - 3, GTK_KEY);
      save_proto_data_value(pinfo, tag_len - 3, GTK_LEN_KEY);
      break;
    }
    case 3: /* MAC Address KDE */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mac_address_kde_mac, tvb,
                          offset, 6, ENC_NA);
      proto_item_append_text(item, ": MAC Address KDE");
      save_proto_data(tvb, pinfo, offset, 6, MLD_MAC_KEY);
      break;
    case 4:
    {
      /* IEEE 802.11i / Key Data Encapsulation / Data Type=4 - PMKID.
       * This is only used within EAPOL-Key frame Key Data. */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_pmkid, tvb, offset, 16, ENC_NA);
      proto_item_append_text(item, ": RSN PMKID");
      break;
    }
    case 6:
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_nonce, tvb, offset,
                          32, ENC_NA);
      proto_item_append_text(item, ": NONCE KDE");
      break;
    case 7: /* Lifetime KDE */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_gtk_kde_lifetime, tvb,
                          offset, 4, ENC_LITTLE_ENDIAN);
      proto_item_append_text(item, ": Lifetime KDE");
      break;
    case 8: /* Error KDE */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_error_kde_res, tvb, offset,
                          2, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_error_kde_error_type, tvb,
                          offset, 2, ENC_LITTLE_ENDIAN);
      proto_item_append_text(item, ": Error KDE");
      break;
    case 9: /* IGTK KDE */
    {
      /* IEEE 802.11i / Key Data Encapsulation / Data Type=9 - IGTK.
       * This is only used within EAPOL-Key frame Key Data. */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_igtk_kde_keyid, tvb,
                          offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_igtk_kde_ipn, tvb, offset,
                          6, ENC_LITTLE_ENDIAN);
      offset += 6;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_igtk_kde_igtk, tvb, offset,
                          tag_len - 9, ENC_NA);
      proto_item_append_text(item, ": RSN IGTK");
      break;
    }
    case 10:
    {
      /* IEEE 802.11 - 2016 / Key Data Encapsulation / Data Type=10 - KeyID
       * This is only used within EAPOL-Key frame Key Data when using Extended Key ID */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_ptk_keyid, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(item, ": RSN PTK");
      break;
    }
    case 13: /* OCI KDE */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_oci_operating_class, tvb,
                          offset, 1, ENC_NA);
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_oci_primary_channel_number,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_oci_frequency_segment_1,
                          tvb, offset, 1, ENC_NA);
      proto_item_append_text(item, ": OCI KDE");
      break;
    case 14: /* BIGTK KDE */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_bigtk_key_id, tvb, offset,
                          2, ENC_LITTLE_ENDIAN);
      offset += 2;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_bigtk_bipn, tvb, offset,
                          6, ENC_LITTLE_ENDIAN);
      offset += 6;
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_bigtk_bigtk, tvb, offset,
                          tag_len - 9, ENC_NA);
      proto_item_append_text(item, ": BIGTK KDE");
      break;
    case 16: /* MLO GTK KDE */
      mlo_gtk_nb = GPOINTER_TO_UINT(p_get_proto_data(pinfo->pool, pinfo, proto_wlan, MLO_GTK_COUNT));
      flags = tvb_get_uint8(tvb, offset);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_gtk_kde_key_id, tvb,
                          offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_gtk_kde_tx, tvb,
                          offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_gtk_kde_reserved, tvb,
                          offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_gtk_kde_linkid, tvb,
                          offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_gtk_kde_pn, tvb,
                          offset, 6, ENC_LITTLE_ENDIAN);
      offset += 6;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_gtk_kde_gtk, tvb,
                          offset, tag_len - 8, ENC_NA);
      if (mlo_gtk_nb < DOT11DECRYPT_MAX_MLO_LINKS) {
        uint8_t link_id = (flags & 0xf0) >> 4;
        save_proto_data_value(pinfo, link_id, MLO_GTK_LINK_ID_FIRST + mlo_gtk_nb);
        save_proto_data(tvb, pinfo, offset, tag_len - 8, MLO_GTK_KEY_FIRST + mlo_gtk_nb);
        save_proto_data_value(pinfo, tag_len - 8, MLO_GTK_KEY_LEN_FIRST + mlo_gtk_nb);
        save_proto_data_value(pinfo, mlo_gtk_nb + 1, MLO_GTK_COUNT);
      }

      proto_item_append_text(item, ": MLO GTK KDE");
      break;
    case 17: /* MLO IGTK KDE */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_igtk_kde_key_id, tvb,
                          offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_igtk_kde_ipn, tvb,
                          offset, 6, ENC_LITTLE_ENDIAN);
      offset += 6;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_igtk_kde_reserved, tvb,
                          offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_igtk_kde_linkid, tvb,
                          offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_igtk_kde_igtk, tvb,
                          offset, tag_len - 10, ENC_NA);

      proto_item_append_text(item, ": MLO IGTK KDE");
      break;
    case 18: /* MLO BIGTK KDE */
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_bigtk_kde_key_id, tvb,
                          offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_bigtk_kde_ipn, tvb,
                          offset, 6, ENC_LITTLE_ENDIAN);
      offset += 6;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_bigtk_kde_reserved, tvb,
                          offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_bigtk_kde_linkid, tvb,
                          offset, 1, ENC_NA);
      offset += 1;

      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_mlo_bigtk_kde_bigtk, tvb,
                          offset, tag_len - 10, ENC_NA);

      proto_item_append_text(item, ": MLO BIGTK KDE");
      break;
    case 19: /*
              * MLO Link KDE, contains  Link info, MAC Addr and possibly
              * RSNE and RSNXE
              */
      dissect_rsn_ie_mlo_link(item, tree, tvb, offset, tag_len, pinfo);
      break;
    default:
      proto_tree_add_item(tree, hf_ieee80211_rsn_ie_unknown, tvb, offset,
                          tag_len - 1, ENC_NA);
      proto_item_append_text(item, ": RSN UNKNOWN");
      break;
  }
}

typedef enum {
  MARVELL_IE_MESH = 4
} marvell_ie_type_t;

static void
dissect_vendor_ie_marvell(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len)
{
  uint8_t type;

  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_marvell_ie_type, tvb, offset, 1, ENC_LITTLE_ENDIAN, &type);
  offset += 1;

  switch (type) {
  case MARVELL_IE_MESH:
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_subtype, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_version, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_active_proto_id, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_active_metric_id, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_mesh_cap, tvb,
                         offset++, 1, ENC_LITTLE_ENDIAN);
    break;

  default:
    proto_tree_add_item(ietree, hf_ieee80211_marvell_ie_data, tvb, offset,
      tag_len - 1, ENC_NA);
    break;
  }
}

typedef enum {
  ATHEROS_IE_ADVCAP = 1,
  ATHEROS_IE_XR = 3
} atheros_ie_type_t;

typedef enum {
  ATHEROS_IE_ADVCAP_S = 1
} atheros_ie_advcap_subtype_t;

typedef enum {
  ATHEROS_IE_XR_S = 1
} atheros_ie_xr_subtype_t;

typedef enum {
  ATHEROS_IE_CAP_TURBOP = 0x01,
  ATHEROS_IE_CAP_COMP   = 0x02,
  ATHEROS_IE_CAP_FF     = 0x04,
  ATHEROS_IE_CAP_XR     = 0x08,
  ATHEROS_IE_CAP_AR     = 0x10,
  ATHEROS_IE_CAP_BURST  = 0x20,
  ATHEROS_IE_CAP_WME    = 0x40,
  ATHEROS_IE_CAP_BOOST  = 0x80
} atheros_ie_cap_t;

static const value_string atheros_ie_type_vals[] = {
  { ATHEROS_IE_ADVCAP, "Advanced Capability"},
  { ATHEROS_IE_XR,     "eXtended Range"},
  { 0,                 NULL }
};

static int * const ieee80211_atheros_ie_cap[] = {
  &hf_ieee80211_atheros_ie_cap_f_turbop,
  &hf_ieee80211_atheros_ie_cap_f_comp,
  &hf_ieee80211_atheros_ie_cap_f_ff,
  &hf_ieee80211_atheros_ie_cap_f_xr,
  &hf_ieee80211_atheros_ie_cap_f_ar,
  &hf_ieee80211_atheros_ie_cap_f_burst,
  &hf_ieee80211_atheros_ie_cap_f_wme,
  &hf_ieee80211_atheros_ie_cap_f_boost,
  NULL
};

static int * const ieee80211_extreme_mesh_ie_hello[] = {
  &hf_ieee80211_extreme_mesh_ie_hello_f_root,
  &hf_ieee80211_extreme_mesh_ie_hello_f_proxy,
  &hf_ieee80211_extreme_mesh_ie_hello_f_geo,
  &hf_ieee80211_extreme_mesh_ie_hello_f_path_pref,
  &hf_ieee80211_extreme_mesh_ie_hello_f_mobile,
  NULL
};

typedef enum {
  EXTREME_MESH_IE_SERVICES_ROOT      = 0x01,
  EXTREME_MESH_IE_SERVICES_PROXY     = 0x02,
  EXTREME_MESH_IE_SERVICES_GEO       = 0x04,
  EXTREME_MESH_IE_SERVICES_PATH_PREF = 0x08,
  EXTREME_MESH_IE_SERVICES_MOBILE    = 0x10,
} extreme_mesh_ie_services;

/* Mesh Fields found in Management Frames KJG */
static const value_string extreme_mesh_ie_type_vals[] = {
  {1, "Hello"},
  {2, "Mesh ID"},
  {3, "MPID"},
  {0, NULL}
};

static void
dissect_vendor_ie_extreme_mesh(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, unsigned tag_len,
                          packet_info *pinfo, proto_item *ti_len)
{
  uint8_t     type;

  if (tag_len <= 3) {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length %u too short, must be >= 6", tag_len+3);
    /* Add length of OUI to tag_length */
    return;
  }
  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_extreme_mesh_ie_type, tvb, offset, 1, ENC_LITTLE_ENDIAN, &type);
  proto_item_append_text(item, ": %s", val_to_str_const(type, extreme_mesh_ie_type_vals, "Unknown"));
  offset  += 1;
  tag_len -= 1;

  switch (type) {
    case 1:
      {
        proto_tree_add_bitmask_with_flags(ietree, tvb, offset, hf_ieee80211_extreme_mesh_ie_services,
        ett_extreme_mesh_services_tree, ieee80211_extreme_mesh_ie_hello, ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
        offset   += 1;

        proto_tree_add_item(ietree, hf_ieee80211_extreme_mesh_ie_htr, tvb, offset, 1, ENC_NA);
        offset   += 1;

        proto_tree_add_item(ietree, hf_ieee80211_extreme_mesh_ie_mtr, tvb, offset, 2, ENC_LITTLE_ENDIAN);
        offset   += 2;

        proto_tree_add_item(ietree, hf_ieee80211_extreme_mesh_ie_root, tvb, offset, 6, ENC_NA);
        offset   += 6;

        proto_tree_add_item(ietree, hf_ieee80211_extreme_mesh_ie_nh, tvb, offset, 6, ENC_NA);
      }
      break;
    case 2:
      {
         proto_tree_add_item(ietree, hf_ieee80211_extreme_mesh_ie_mesh_id, tvb, offset, tag_len, ENC_ASCII);
      }
      break;
    case 3:
      {
         proto_tree_add_item(ietree, hf_ieee80211_extreme_mesh_ie_mp_id, tvb, offset, 6, ENC_NA);
      }
      break;
    default:
      break;
  }
}

static void
dissect_vendor_ie_atheros(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, unsigned tag_len,
                          packet_info *pinfo, proto_item *ti_len)
{
  uint8_t     type;
  uint8_t     subtype;
  uint8_t     version;

  if (tag_len <= 3) {
        expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length %u too short, must be >= 6", tag_len+3); /* Add length of OUI to tag_length */
        return;
  }
  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_atheros_ie_type, tvb, offset, 1, ENC_NA, &type);
  proto_item_append_text(item, ": %s", val_to_str_const(type, atheros_ie_type_vals, "Unknown"));
  offset  += 1;
  tag_len -= 1;

  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_atheros_ie_subtype, tvb, offset, 1, ENC_NA, &subtype);
  offset  += 1;
  tag_len -= 1;

  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_atheros_ie_version, tvb, offset, 1, ENC_NA, &version);
  offset  += 1;
  tag_len -= 1;

  if (version == 0)
  {
    switch (type) {
      case ATHEROS_IE_ADVCAP:
      {
        switch (subtype) {
          case ATHEROS_IE_ADVCAP_S:
          {
            proto_tree_add_bitmask_with_flags(ietree, tvb, offset, hf_ieee80211_atheros_ie_advcap_cap,
                                    ett_ath_cap_tree, ieee80211_atheros_ie_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset   += 1;
            tag_len  -= 1;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_advcap_defkey, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset  += 2;
            tag_len -= 2;
            break;
          }
          default:
          /* No default Action */
          break;
        } /* End switch (subtype) */
        break;
      }
      case ATHEROS_IE_XR:
      {
        switch (subtype) {
          case ATHEROS_IE_XR_S:
          {
            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_info, tvb, offset, 1, ENC_NA);
            offset  += 1;
            tag_len -= 1;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_base_bssid, tvb, offset, 6, ENC_NA);
            offset  += 6;
            tag_len -= 6;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_xr_bssid, tvb, offset, 6, ENC_NA);
            offset  += 6;
            tag_len -= 6;

            proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_xr_xr_beacon, tvb, offset, 2, ENC_LITTLE_ENDIAN);
            offset  += 2;
            tag_len -= 2;

            proto_tree_add_bitmask_with_flags(ietree, tvb, offset, hf_ieee80211_atheros_ie_xr_base_cap,
                                    ett_ath_cap_tree, ieee80211_atheros_ie_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset   += 1;
            tag_len  -= 1;

            proto_tree_add_bitmask_with_flags(ietree, tvb, offset, hf_ieee80211_atheros_ie_xr_xr_cap,
                                    ett_ath_cap_tree, ieee80211_atheros_ie_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
            offset   += 1;
            tag_len  -= 1;
            break;
          }
          default:
          /* No default Action */
          break;
        } /* End switch (subtype) */
        break;
        default:
        /* No default Action */
        break;
      } /* End switch (type) */

    }
  }
  if (tag_len > 0) {
    proto_tree_add_item(ietree, hf_ieee80211_atheros_ie_data, tvb, offset, tag_len, ENC_NA);
  }
}

typedef enum {
  AIRONET_IE_DTPC = 0,
  AIRONET_IE_UNKNOWN1 = 1,
  AIRONET_IE_VERSION = 3,
  AIRONET_IE_QOS,
  AIRONET_IE_UNKNOWN11 = 11,
  AIRONET_IE_QBSS_V2 = 14,
  AIRONET_IE_CLIENT_MFP = 20,
  AIRONET_IE_APNAME_V2 = 47
} aironet_ie_type_t;

static const value_string aironet_ie_type_vals[] = {
  { AIRONET_IE_DTPC,      "DTPC"},
  { AIRONET_IE_UNKNOWN1,  "Unknown (1)"},
  { AIRONET_IE_VERSION,   "CCX version"},
  { AIRONET_IE_QOS,       "Qos"},
  { AIRONET_IE_UNKNOWN11, "Unknown (11)"},
  { AIRONET_IE_QBSS_V2,   "QBSS V2 - CCA"},
  { AIRONET_IE_CLIENT_MFP, "Client MFP"},
  { AIRONET_IE_APNAME_V2, "AP NAME v2"},
  { 0,                    NULL }
};

static const value_string aironet_mfp_vals[] = {
  { 0,      "Disabled"},
  { 1,      "Enabled"},
  { 0,      NULL }
};

static void
dissect_vendor_ie_aironet(proto_item *aironet_item, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len,packet_info *pinfo)
{
  uint8_t type, length;
  int i;
  bool dont_change = false; /* Don't change the IE item text to default */
  const uint8_t* apname;

  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_aironet_ie_type, tvb, offset, 1, ENC_LITTLE_ENDIAN, &type);
  offset += 1;

  switch (type) {
  case AIRONET_IE_DTPC:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_dtpc, tvb, offset, 1, ENC_NA);
    proto_item_append_text(aironet_item, ": Aironet DTPC Powerlevel %ddBm", tvb_get_uint8(tvb, offset));
    offset += 1;
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_dtpc_unknown, tvb, offset, 1, ENC_NA);
    dont_change = true;
    break;
  case AIRONET_IE_VERSION:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_version, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    proto_item_append_text(aironet_item, ": Aironet CCX version = %d", tvb_get_uint8(tvb, offset));
    dont_change = true;
    break;
  case AIRONET_IE_QOS:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_qos_reserved, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_qos_paramset, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    /* XXX: just copied over from WME. Maybe "Best Effort" and "Background"
     *  need to be swapped. Also, the "TXOP" may be TXOP - or not.
     */
    for (i = 0; i < 4; i++) {
      uint8_t byte1, byte2;
      uint16_t txop;
      byte1 = tvb_get_uint8(tvb, offset);
      byte2 = tvb_get_uint8(tvb, offset + 1);
      txop = tvb_get_letohs(tvb, offset + 2);
      proto_tree_add_bytes_format(ietree, hf_ieee80211_aironet_ie_qos_val, tvb, offset, 4, NULL,
          "CCX QoS Parameters: ACI %u (%s), Admission Control %sMandatory, AIFSN %u, ECWmin %u, ECWmax %u, TXOP %u",
        (byte1 & 0x60) >> 5, val_to_str(pinfo->pool, (byte1 & 0x60) >> 5, wme_acs, "(Unknown: %d)"),
        (byte1 & 0x10) ? "" : "not ", byte1 & 0x0f,
        byte2 & 0x0f, (byte2 & 0xf0) >> 4,
        txop);
      offset += 4;
    }
    break;
  case AIRONET_IE_QBSS_V2:
    /* Extract Values */
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_scount, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_cu, tvb, offset + 2, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_cal, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(ietree, hf_ieee80211_qbss2_gl, tvb, offset + 4, 1, ENC_LITTLE_ENDIAN);
    break;
  case AIRONET_IE_CLIENT_MFP:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_clientmfp, tvb, offset, 1, ENC_LITTLE_ENDIAN);
    proto_item_append_text(aironet_item, ": Aironet Client MFP %s",
      val_to_str_const(1 & tvb_get_uint8(tvb, offset), aironet_mfp_vals, "Unknown"));
    dont_change = true;
    break;
  case AIRONET_IE_APNAME_V2:
  /* Adds support for the new AP name v2 format;
this supports:
- 32 character long AP names (vs 16 in v1/ccx)
- is used on both WLC and Meraki
- does not require CCX to be enabled.
 */
    tag_len -= 1;
    length = tag_len;

    proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_cisco_ap_name_v2, tvb, offset, length, ENC_ASCII|ENC_NA, pinfo->pool,&apname);
    proto_item_append_text(ietree, ": AP name v2: %s", apname);
    // Set to true, so we don't append "Aironet type"
    dont_change = true;
  break;

  default:
    proto_tree_add_item(ietree, hf_ieee80211_aironet_ie_data, tvb, offset,
      tag_len - 1, ENC_NA);
    break;
  }
  if (!dont_change) {
    proto_item_append_text(aironet_item, ": Aironet %s (%d)",
      val_to_str_const(type, aironet_ie_type_vals, "Unknown"), type);
  }
}

/* info from lswifi https://github.com/joshschmelzle/lswifi/blob/v0.1.49/lswifi/elements.py#L1526 */
#define ARUBA_CAC 1
#define ARUBA_MESH 2
#define ARUBA_APNAME 3
#define ARUBA_ARM 4
#define ARUBA_SLB 5
#define ARUBA_SJ_LOOP_PROTECT 6
#define ARUBA_AUTO_MESH 7
#define ARUBA_LCI 8
#define ARUBA_GPS 9
#define ARUBA_AP_HEALTH 10

static const value_string ieee80211_vs_aruba_subtype_vals[] = {
  { ARUBA_CAC, "CAC"},
  { ARUBA_MESH, "Mesh"},
  { ARUBA_APNAME, "AP Name"},
  { ARUBA_ARM, "ARM"},
  { ARUBA_SLB, "SLB"},
  { ARUBA_SJ_LOOP_PROTECT, "SJ Loop Protect"},
  { ARUBA_AUTO_MESH, "Auto Mesh"},
  { ARUBA_LCI, "LCI"},
  { ARUBA_GPS, "GPS Ellipse"},
  { ARUBA_AP_HEALTH, "AP Health"},
  { 0,                 NULL }
};
static void
dissect_vendor_ie_aruba(proto_item *item, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint8_t type;
  const uint8_t* name;

  offset += 1; /* VS OUI Type */
  tag_len -= 1;

  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_vs_aruba_subtype, tvb, offset, 1, ENC_NA, &type);
  proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_aruba_subtype_vals, "Unknown"));
  offset += 1;
  tag_len -= 1;

  switch (type) {
  case ARUBA_APNAME:
    offset += 1;
    tag_len -= 1;

    proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_aruba_apname, tvb,
                         offset, tag_len, ENC_ASCII|ENC_NA, pinfo->pool, &name);
    proto_item_append_text(item, " (%s)", name);
    break;

  case ARUBA_GPS:
    {
      double latitude, longitude, major_axis, minor_axis, orientation, distance;
      offset += 1;
      tag_len -= 1;
      if (tag_len < 1) {
        expert_add_info_format(pinfo, ietree, &ei_ieee80211_bad_length,
                          "Malformed Aruba GPS data (insufficient data)");
        break;
      }

      uint32_t gps_ie_length = tvb_get_uint8(tvb, offset);
      offset += 1;
      tag_len -= 1;
      gps_ie_length -= 1;

      if (tag_len < gps_ie_length) {
        expert_add_info_format(pinfo, ietree, &ei_ieee80211_bad_length,
                      "Truncated Aruba GPS data (IE needs %u bytes but got %u)",
                      gps_ie_length, tag_len);
        break;
      }

      proto_tree_add_item(ietree, hf_ieee80211_vs_aruba_gps_subversion, tvb,
                        offset, 1, ENC_NA);
      offset += 1;
      tag_len -= 1;

      proto_tree_add_item(ietree, hf_ieee80211_vs_aruba_gps_hop, tvb,
                        offset, 1, ENC_NA);
      offset += 1;
      tag_len -= 1;

      latitude = tvb_get_ntohieee_double(tvb, offset);
      proto_tree_add_double_format_value(ietree, hf_ieee80211_vs_aruba_gps_latitude, tvb,
                        offset, 8, latitude, "%.6f", latitude);
      offset += 8;
      tag_len -= 8;

      longitude = tvb_get_ntohieee_double(tvb, offset);
      proto_tree_add_double_format_value(ietree, hf_ieee80211_vs_aruba_gps_longitude, tvb,
                        offset, 8, longitude, "%.6f", longitude);
      offset += 8;
      tag_len -= 8;

      major_axis = tvb_get_ntohieee_double(tvb, offset);
      proto_tree_add_double_format_value(ietree, hf_ieee80211_vs_aruba_gps_major_axis, tvb,
                        offset, 8, major_axis, "%.6f", major_axis);
      offset += 8;
      tag_len -= 8;

      minor_axis = tvb_get_ntohieee_double(tvb, offset);
      proto_tree_add_double_format_value(ietree, hf_ieee80211_vs_aruba_gps_minor_axis, tvb,
                        offset, 8, minor_axis, "%.6f", minor_axis);
      offset += 8;
      tag_len -= 8;

      orientation = tvb_get_ntohieee_double(tvb, offset);
      proto_tree_add_double_format_value(ietree, hf_ieee80211_vs_aruba_gps_orientation, tvb,
                        offset, 8, orientation, "%.6f", orientation);
      offset += 8;
      tag_len -= 8;

      distance = tvb_get_ntohieee_double(tvb, offset);
      proto_tree_add_double_format_value(ietree, hf_ieee80211_vs_aruba_gps_distance, tvb,
                        offset, 8, distance, "%.6f", distance);
      tag_len -= 8;

      if (tag_len > 0) {
        expert_add_info_format(pinfo, ietree, &ei_ieee80211_bad_length,
                        "Aruba GPS data contains additional %u bytes of unexpected data",
                        tag_len);
      }

      proto_item_append_text(item, " (coords: [%.6f, %.6f], ellipse: [%.2fm x %.2fm, %.2f°], distance: %.2fm)",
                        latitude, longitude, major_axis, minor_axis, orientation, distance);
    }
    break;

  default:
    proto_tree_add_item(ietree, hf_ieee80211_vs_aruba_data, tvb, offset,
      tag_len, ENC_NA);
    if (tag_len > 0)
      proto_item_append_text(item, " (Data: %s)", tvb_bytes_to_str(pinfo->pool, tvb, offset, tag_len));
    break;
  }
}

static void
dissect_vendor_ie_routerboard(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint8_t type, length;
  proto_item *subitem;
  proto_tree *subtree;

  offset += 1; /* VS OUI Type */
  tag_len -= 1;
  /* FIXME: Make sure we have at least 2 bytes left */
  proto_tree_add_item(ietree, hf_ieee80211_vs_routerboard_unknown, tvb, offset, 2, ENC_NA);

  offset += 2;
  tag_len -= 2;

  while (tag_len >= 2) {
    type = tvb_get_uint8(tvb, offset);
    length = tvb_get_uint8(tvb, offset+1);
    subitem = proto_tree_add_item(ietree, hf_ieee80211_vs_routerboard_subitem, tvb, offset, length+2, ENC_NA);
    subtree = proto_item_add_subtree(subitem, ett_routerboard);
    proto_item_set_text(subitem, "Sub IE (T/L: %d/%d)", type, length);

    proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subtype, tvb, offset, 1, ENC_NA);
    offset += 1;
    tag_len -= 1;

    proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_sublength, tvb, offset, 1, ENC_NA);
    offset += 1;
    tag_len -= 1;

    if (tag_len < length)
      length = tag_len;
    if (length == 0) {
      expert_add_info(pinfo, subitem, &ei_ieee80211_vs_routerboard_unexpected_len);
      break;
    }

    proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subdata, tvb, offset, length, ENC_NA);

    if(type == 1){
      if(length == 30){
        proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subtype1_prefix, tvb, offset, 10, ENC_NA);
        proto_tree_add_item(subtree, hf_ieee80211_vs_routerboard_subtype1_data, tvb, offset + 10, length - 10, ENC_ASCII);
      }else{
        expert_add_info(pinfo, subitem, &ei_ieee80211_vs_routerboard_unexpected_len);
      }
    }

    offset += length;
    tag_len -= length;
  }
}

#define AEROHIVE_HOSTNAME 33
static const value_string ieee80211_vs_aerohive_type_vals[] = {
  { AEROHIVE_HOSTNAME, "Host Name"},
  { 0,                 NULL }
};
static void
dissect_vendor_ie_aerohive(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint32_t type, length;
  const uint8_t* hostname;
  proto_item *ti_len;

  /* VS OUI Type */
  type = tvb_get_uint8(tvb, offset);
  offset += 1;
  tag_len -= 1;

  proto_tree_add_item(ietree, hf_ieee80211_vs_aerohive_version, tvb, offset, 1, ENC_NA);
  proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_aerohive_type_vals, "Unknown"));
  offset += 1;
  tag_len -= 1;

  switch(type){
    case AEROHIVE_HOSTNAME: /* Subtype (1 byte) + Host Name Length (1 byte) + Host Name */

      proto_tree_add_item(ietree, hf_ieee80211_vs_aerohive_subtype, tvb, offset, 1, ENC_NA);
      offset += 1;
      tag_len -= 1;

      ti_len = proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_aerohive_hostname_length, tvb, offset, 1, ENC_NA, &length);
      offset += 1;
      tag_len -= 1;

      if (tag_len < length) {
        expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length < Host Name Length");
        length = tag_len;
      }

      proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_aerohive_hostname, tvb, offset, length, ENC_ASCII|ENC_NA, pinfo->pool, &hostname);
      proto_item_append_text(item, " (%s)", hostname);

    break;

    default:
      proto_tree_add_item(ietree, hf_ieee80211_vs_aerohive_data, tvb, offset, tag_len, ENC_NA);
    break;
  }
}

#define MIST_APNAME 1
static const value_string ieee80211_vs_mist_type_vals[] = {
    { MIST_APNAME, "AP Name"},
    { 0,           NULL }
};
static void
dissect_vendor_ie_mist(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
    uint32_t type, length;
    const uint8_t* apname;

    /* VS OUI Type */
    type = tvb_get_uint8(tvb, offset);
    proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_mist_type_vals, "Unknown"));
    offset += 1;
    tag_len -= 1;

    switch(type){
        case MIST_APNAME:
            length = tag_len;
            proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_mist_ap_name, tvb, offset, length, ENC_ASCII|ENC_NA, pinfo->pool, &apname);
            proto_item_append_text(item, " (%s)", apname);
            break;

        default:
            proto_tree_add_item(ietree, hf_ieee80211_vs_mist_data, tvb, offset, tag_len, ENC_NA);
            break;
    }
}

#define UBIQUITI_APNAME 0x01
static const value_string ieee80211_vs_ubiquiti_type_vals[] = {
    { UBIQUITI_APNAME, "AP Name"},
    { 0,           NULL }
};
static void
dissect_vendor_ie_ubiquiti(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
    uint32_t type, length;
    const uint8_t* apname;

    /* VS OUI Type */
    proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_ubiquiti_type, tvb, offset, 1, ENC_NA, &type);
    proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_ubiquiti_type_vals, "Unknown"));
    offset += 1;
    tag_len -= 1;

    switch(type){
        case UBIQUITI_APNAME:
            length = tag_len;
            proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_ubiquiti_ap_name, tvb, offset, length, ENC_ASCII|ENC_NA, pinfo->pool, &apname);
            proto_item_append_text(item, " (%s)", apname);
            break;

        default:
            proto_tree_add_item(ietree, hf_ieee80211_vs_ubiquiti_data, tvb, offset, tag_len, ENC_NA);
            break;
    }
}

#define METER_APNAME 0
static const value_string ieee80211_vs_meter_type_vals[] = {
    { METER_APNAME, "AP Name"},
    { 0,           NULL }
};
static void
dissect_vendor_ie_meter(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
    uint32_t type, length;
    const uint8_t* apname;

    /* VS OUI Type */
    proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_meter_type, tvb, offset, 1, ENC_NA, &type);
    proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_meter_type_vals, "Unknown"));
    offset += 1;
    tag_len -= 1;

    switch(type){
        case METER_APNAME:
            length = tag_len;
            proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_meter_ap_name, tvb, offset, length, ENC_ASCII|ENC_NA, pinfo->pool, &apname);
            proto_item_append_text(item, " (%s)", apname);
            break;

        default:
            proto_tree_add_item(ietree, hf_ieee80211_vs_meter_data, tvb, offset, tag_len, ENC_NA);
            break;
    }
}

#define RUCKUS_APNAME 3
static const value_string ieee80211_vs_ruckus_type_vals[] = {
    { RUCKUS_APNAME, "AP Name"},
    { 0,           NULL }
};
static void
dissect_vendor_ie_ruckus(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
    uint32_t type, length;
    const uint8_t* apname;

    /* VS OUI Type */
    type = tvb_get_uint8(tvb, offset);
    proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_ruckus_type_vals, "Unknown"));
    offset += 1;
    tag_len -= 1;

    switch(type){
        case RUCKUS_APNAME:
          length = tag_len;
            proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_ruckus_ap_name, tvb, offset, length, ENC_ASCII|ENC_NA, pinfo->pool, &apname);
            proto_item_append_text(item, " (%s)", apname);
            break;

        default:
            proto_tree_add_item(ietree, hf_ieee80211_vs_ruckus_data, tvb, offset, tag_len, ENC_NA);
            break;
    }
}

#define ALCATEL_APNAME 1
static const value_string ieee80211_vs_alcatel_type_vals[] = {
    { ALCATEL_APNAME, "AP Name"},
    { 0, NULL }
};
static void
dissect_vendor_ie_alcatel(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
    uint32_t type, length;
    const uint8_t* apname;

    /* VS OUI Type */
    type = tvb_get_uint8(tvb, offset);
    proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_alcatel_type_vals, "Unknown"));
    offset += 1;
    tag_len -= 1;

    switch(type){
        case ALCATEL_APNAME:
            length = tag_len;
            proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_alcatel_ap_name, tvb, offset, length, ENC_ASCII|ENC_NA, pinfo->pool, &apname);
            proto_item_append_text(item, " (%s)", apname);
            break;

        default:
            proto_tree_add_item(ietree, hf_ieee80211_vs_alcatel_data, tvb, offset, tag_len, ENC_NA);
            break;
    }
}

enum vs_sgdsn_type {
  SGDSN_VERSION = 0x01,
  SGDSN_IDFR = 0x02,
  SGDSN_IDANSI = 0x03,
  SGDSN_LATITUDE = 0x04,
  SGDSN_LONGITUDE = 0x05,
  SGDSN_ALTITUDE_ABS = 0x06,
  SGDSN_ALTITUDE_REL = 0x07,
  SGDSN_LATITUDE_TAKEOFF = 0x08,
  SGDSN_LONGITUDE_TAKEOFF = 0x09,
  SGDSN_H_SPEED = 0x0a,
  SGDSN_HEADING = 0x0b,
};

static const value_string ieee80211_vs_sgdsn_type_vals[] = {
  { SGDSN_VERSION,  "Version"},
  { SGDSN_IDFR, "ID FR"},
  { SGDSN_IDANSI, "ID ANSI"},
  { SGDSN_LATITUDE, "Latitude"},
  { SGDSN_LONGITUDE, "Longitude"},
  { SGDSN_ALTITUDE_ABS, "Altitude AMSL"},
  { SGDSN_ALTITUDE_REL, "Altitude AGL"},
  { SGDSN_LATITUDE_TAKEOFF, "Latitude Takeoff"},
  { SGDSN_LONGITUDE_TAKEOFF, "Longitude Takeoff"},
  { SGDSN_H_SPEED, "Horizontal Speed"},
  { SGDSN_HEADING, "Heading"},
  { 0, NULL }
};

static void
dissect_vendor_ie_sgdsn(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, uint32_t tag_len,
                       packet_info *pinfo)
{
  // Technical specification defined in French law "NOR: ECOI1934044A"
  // https://www.legifrance.gouv.fr/eli/arrete/2019/12/27/ECOI1934044A/jo/texte

  uint8_t type = tvb_get_uint8(tvb, offset);
  offset += 1;
  tag_len -= 1;

  if (type == 1) {

    while (tag_len > 2) {

      uint8_t tlv_type = tvb_get_uint8(tvb, offset);
      uint8_t tlv_len = tvb_get_uint8(tvb, offset+1);

      if (tag_len < tlv_len) {
        break;
      }

      proto_item *item_tlv = proto_tree_add_item(ietree, hf_ieee80211_vs_sgdsn_tag, tvb, offset, tlv_len + 2, ENC_NA);
      proto_item *tree = proto_item_add_subtree(item_tlv, ett_sgdsn);

      proto_tree_add_item(tree, hf_ieee80211_vs_sgdsn_type, tvb, offset, 1, ENC_NA);
      proto_tree_add_item(tree, hf_ieee80211_vs_sgdsn_length, tvb, offset + 1,  1, ENC_NA);

      offset += 2;
      tag_len -= 2;

      proto_item_append_text(tree, ": %s", val_to_str_const(tlv_type, ieee80211_vs_sgdsn_type_vals, "Unknown"));

      switch(tlv_type) {
      case SGDSN_VERSION:
        if (tlv_len == 1) {
          uint32_t value;
          proto_tree_add_item_ret_uint(tree, hf_ieee80211_vs_sgdsn_version, tvb, offset, 1, ENC_NA, &value);
          proto_item_append_text(tree, ": %d", value);
        } else {
          expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Value length must be 1");
        }
        break;
      case SGDSN_IDFR:
        if (tlv_len == 30) {
          const uint8_t* string1;
          const uint8_t* string2;
          const uint8_t* string3;
          proto_tree_add_item_ret_string(tree, hf_ieee80211_vs_sgdsn_manufacturer, tvb, offset, 3, ENC_ASCII|ENC_NA, pinfo->pool, &string1);
          proto_tree_add_item_ret_string(tree, hf_ieee80211_vs_sgdsn_model, tvb, offset+3, 3, ENC_ASCII|ENC_NA, pinfo->pool, &string2);
          proto_tree_add_item_ret_string(tree, hf_ieee80211_vs_sgdsn_serialnumber, tvb, offset+6, tlv_len-6, ENC_ASCII|ENC_NA, pinfo->pool, &string3);
          proto_item_append_text(tree, ": %s %s %s", string1, string2, string3);
        } else {
          expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Value length must be 30");
        }
        break;
      case SGDSN_IDANSI:
        if (tlv_len >= 6 && tlv_len <= 20) {
          // ANSI/CTA-2063 Small UAS Serial Number.
          // Doc (free): https://shop.cta.tech/products/small-unmanned-aerial-systems-serial-numbers
          const uint8_t* icao_mfr_code;
          uint32_t sn_len;
          const uint8_t* serial_number;
          proto_tree_add_item_ret_string(tree, hf_ieee80211_vs_sgdsn_icaomfrcode, tvb, offset, 4, ENC_ASCII|ENC_NA, pinfo->pool, &icao_mfr_code);
          proto_tree_add_item_ret_uint(tree, hf_ieee80211_vs_sgdsn_serialnumber_len, tvb, offset+4, 1, ENC_NA, &sn_len);
          if(sn_len < 0x30 || (sn_len > 0x39 && sn_len < 0x41) || sn_len > 0x46) {
            expert_add_info_format(pinfo, tree, &ei_ieee80211_vs_sgdsn_serialnumber_invalid_len_val, "Serial Number Length must be '0' to '9', or 'A' to 'F'");
          } else if (sn_len != (uint32_t)tlv_len+(sn_len>0x39 ? 0x32 : 0x2A)) {
            // Check that sn_len equals tlv_len - 5 + ( 0x37 if sn_len is 'A' to 'F', 0x30 otherwise).
            // We suppressed the minus 5 in the check above to avoid a compilation warning
            expert_add_info_format(pinfo, tree, &ei_ieee80211_vs_sgdsn_serialnumber_unexpected_len_val, "Expected %d byte(s), got %d byte(s)", tlv_len-5, (sn_len>0x39?sn_len-0x37:sn_len-0x30));
          }
          proto_tree_add_item_ret_string(tree, hf_ieee80211_vs_sgdsn_serialnumber, tvb, offset+5, tlv_len-5, ENC_ASCII|ENC_NA, pinfo->pool, &serial_number);
          proto_item_append_text(tree, ": %s %s", icao_mfr_code, serial_number);
        } else {
          expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Value length must be between 6 and 20");
        }
        break;
      case SGDSN_LATITUDE_TAKEOFF:
      case SGDSN_LATITUDE:
      case SGDSN_LONGITUDE_TAKEOFF:
      case SGDSN_LONGITUDE:
        if (tlv_len == 4) {
          int32_t value;
          proto_tree_add_item_ret_int(tree, hf_ieee80211_vs_sgdsn_gpscoord, tvb, offset, 4, ENC_LITTLE_ENDIAN, &value);
          proto_item_append_text(tree, ": %.5f", value / 100000.0);
        } else {
          expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Value length must be 4");
        }
        break;
      case SGDSN_ALTITUDE_ABS:
      case SGDSN_ALTITUDE_REL:
        if (tlv_len == 2) {
          int32_t value;
          proto_tree_add_item_ret_int(tree, hf_ieee80211_vs_sgdsn_altitude, tvb, offset, 2, ENC_LITTLE_ENDIAN, &value);
          proto_item_append_text(tree, ": %d m", value);
        } else {
          expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Value length must be 4");
        }
        break;
      case SGDSN_H_SPEED:
        if (tlv_len == 1) {
          uint32_t value;
          proto_tree_add_item_ret_uint(tree, hf_ieee80211_vs_sgdsn_speed, tvb, offset, 1, ENC_NA, &value);
          proto_item_append_text(tree, ": %d m/s", value);
        } else {
          expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Value length must be 1");
        }
        break;
      case SGDSN_HEADING:
        if (tlv_len == 2) {
          uint32_t value;
          proto_tree_add_item_ret_uint(tree, hf_ieee80211_vs_sgdsn_heading, tvb, offset, 2, ENC_LITTLE_ENDIAN, &value);
          proto_item_append_text(tree, ": %d deg", value);
        } else {
          expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length, "Value length must be 2");
        }
        break;
      default:
        /* TODO: use/define a more appropriate ei item for this? */
        expert_add_info_format(pinfo, tree, &ei_ieee80211_extra_data, "Unknown type");
        break;
      }

      offset += tlv_len;
      tag_len -= tlv_len;
    }

    if (tag_len) {
      expert_add_info_format(pinfo, item, &ei_ieee80211_tag_length, "Remaining bytes, TLV structure error");
    }
  }
}

enum vs_nintendo_type {
  NINTENDO_SERVICES = 0x11,
  NINTENDO_CONSOLEID = 0xF0
};

static const value_string ieee80211_vs_nintendo_type_vals[] = {
  { NINTENDO_SERVICES,  "Services"},
  { NINTENDO_CONSOLEID, "ConsoleID"},
  { 0, NULL }
};

static proto_tree*
dissect_vendor_ie_nintendo_tlv(const int hfindex, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t sublen)
{
  proto_item *nintendo_item;
  proto_tree *nintendo_tree;

  nintendo_item = proto_tree_add_item(ietree, hfindex, tvb, offset, sublen, ENC_NA);
  nintendo_tree = proto_item_add_subtree(nintendo_item, ett_nintendo);

  proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_type, tvb, offset, 1, ENC_NA);
  proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_length, tvb, offset + 1,  1, ENC_NA);

  return nintendo_tree;
}

static void
dissect_vendor_ie_nintendo(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len)
{
  proto_tree *nintendo_tree;

  uint8_t     subtype;
  uint8_t     sublength;
  uint32_t    length = tag_len;

  /* Skip OUI type for now - the code is for type 1 (StreetPass) only */
  /* http://3dbrew.org/wiki/StreetPass */
  offset += 1;
  length -= 1;

  while(length > 0 && length < 256) { /* otherwise we are < 0 but on unsigned */
    subtype = tvb_get_uint8(tvb, offset);
    sublength = tvb_get_uint8(tvb, offset + 1);

    switch(subtype) {
    case NINTENDO_SERVICES:
      nintendo_tree = dissect_vendor_ie_nintendo_tlv(hf_ieee80211_vs_nintendo_servicelist, ietree, tvb, offset, sublength + 2);
      offset += 2;
      length -= 2;

      while (sublength > 4) {

        proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_service, tvb, offset, 5, ENC_NA);
        offset += 5;
        length -= 5;
        sublength -= 5;
      }
      break;
    case NINTENDO_CONSOLEID:
      nintendo_tree = dissect_vendor_ie_nintendo_tlv(hf_ieee80211_vs_nintendo_consoleid, ietree, tvb, offset, sublength + 2);
      offset += + 2;
      length -= + 2;

      proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_consoleid, tvb, offset, sublength, ENC_NA);
      offset += sublength;
      length -= sublength;
      break;
    default:
      nintendo_tree = dissect_vendor_ie_nintendo_tlv(hf_ieee80211_vs_nintendo_unknown, ietree, tvb, offset, sublength + 2);
      offset += + 2;
      length -= + 2;

      proto_tree_add_item(nintendo_tree, hf_ieee80211_vs_nintendo_unknown, tvb, offset, sublength, ENC_NA);
      offset += sublength;
      length -= sublength;
      break;
    }
  }
}

static void
dissect_vendor_ie_meru(proto_item *item _U_, proto_tree *ietree,
                       tvbuff_t *tvb, int offset, uint32_t tag_len,
                       packet_info *pinfo)
{
  uint32_t type, length;
  proto_item *subitem, *ti_len;
  proto_tree *subtree;

  while (tag_len >= 2) {
    subitem = proto_tree_add_item(ietree, hf_ieee80211_vs_meru_subitem, tvb, offset, 2, ENC_NA);
    subtree = proto_item_add_subtree(subitem, ett_meru);

    proto_tree_add_item_ret_uint(subtree, hf_ieee80211_vs_meru_subtype, tvb, offset, 1, ENC_NA, &type);
    offset += 1;
    tag_len -= 1;

    ti_len = proto_tree_add_item_ret_uint(subtree, hf_ieee80211_vs_meru_sublength, tvb, offset, 1, ENC_NA, &length);
    offset += 1;
    tag_len -= 1;

    if (tag_len < length) {
      expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length < Sub Length");
      length = tag_len;
    }

    proto_item_append_text(subitem, " (t=%d, l=%d)", type, length);
    proto_item_set_len(subitem, 2+length);

    proto_tree_add_item(subtree, hf_ieee80211_vs_meru_subdata, tvb, offset, length, ENC_NA);
    offset += length;
    tag_len -= length;

  }
}

static const value_string ieee80211_vs_extreme_subtype_vals[] = {
  { 1, "AP Name"},
  { 0, NULL }
};

static void
dissect_vendor_ie_extreme(proto_item *item _U_, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len,
                          packet_info *pinfo)
{
  uint32_t type, length;
  proto_item *ti_len;

  proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_extreme_subtype, tvb, offset, 1, ENC_NA, &type);
  offset += 1;
  tag_len -= 1;

  proto_tree_add_item(ietree, hf_ieee80211_vs_extreme_subdata, tvb, offset, tag_len, ENC_NA);

  switch(type){
    case 1: /* Unknown (7 bytes) + AP Name Length (1 byte) + AP Name */

      proto_tree_add_item(ietree, hf_ieee80211_vs_extreme_unknown, tvb, offset, 7, ENC_NA);
      offset += 7;
      tag_len -= 1;

      ti_len = proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_extreme_ap_length, tvb, offset, 1, ENC_NA, &length);
      offset += 1;
      tag_len -= 1;

      if (tag_len < length) {
        expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Tag length < AP Length");
        length = tag_len;
      }

    proto_tree_add_item(ietree, hf_ieee80211_vs_extreme_ap_name, tvb, offset, length, ENC_ASCII);

    break;
    default:
    /* Expert info ? */
    break;
  }
}

#define FORTINET_SYSTEM 10
static const value_string ieee80211_vs_fortinet_subtype_vals[] = {
  { FORTINET_SYSTEM, "SYSTEM"},
  { 0,                 NULL }
};

#define FORTINET_SYSTEM_APNAME 1
#define FORTINET_SYSTEM_APMODEL 2
#define FORTINET_SYSTEM_APSERIAL 3
static const value_string ieee80211_vs_fortinet_system_type_vals[] = {
  { FORTINET_SYSTEM_APNAME, "AP NAME"},
  { FORTINET_SYSTEM_APMODEL, "AP MODEL"},
  { FORTINET_SYSTEM_APSERIAL, "AP SERIAL"},
  { 0,                 NULL }
};

static void
dissect_vendor_ie_fortinet(proto_item *item, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint32_t type;


  proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_fortinet_subtype, tvb, offset, 2, ENC_LITTLE_ENDIAN, &type);
  proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_fortinet_subtype_vals, "Unknown"));
  offset += 2;
  tag_len -= 2;

  switch (type) {
  case FORTINET_SYSTEM:
    while (tag_len > 2) {
      uint32_t system_type, system_length;

      proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_fortinet_system_type, tvb, offset, 1, ENC_NA, &system_type);
      proto_item_append_text(item, " - %s:", val_to_str_const(system_type, ieee80211_vs_fortinet_system_type_vals, "Unknown"));
      offset += 1;
      tag_len -= 1;

      proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_fortinet_system_length, tvb, offset, 1, ENC_NA, &system_length);
      offset += 1;
      tag_len -= 1;

      switch (system_type) {
        case FORTINET_SYSTEM_APNAME:{
          const uint8_t* name;
          proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_fortinet_system_apname, tvb,
                               offset, system_length, ENC_ASCII|ENC_NA, pinfo->pool, &name);
          proto_item_append_text(item, " %s", name);
        }
        break;
        case FORTINET_SYSTEM_APMODEL:{
          const uint8_t* model;
          proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_fortinet_system_apmodel, tvb,
                               offset, system_length, ENC_ASCII|ENC_NA, pinfo->pool, &model);
          proto_item_append_text(item, " %s", model);
        }
        break;
        case FORTINET_SYSTEM_APSERIAL:{
          const uint8_t* serial;
          proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_fortinet_system_apserial, tvb,
                               offset, system_length, ENC_ASCII|ENC_NA, pinfo->pool, &serial);
          proto_item_append_text(item, " %s", serial);
        }
        break;
      }
      offset += system_length;
      tag_len -= system_length;
    }
    break;

  default:
    proto_tree_add_item(ietree, hf_ieee80211_vs_fortinet_data, tvb, offset,
      tag_len, ENC_NA);
    if (tag_len > 0)
      proto_item_append_text(item, " (Data: %s)", tvb_bytes_to_str(pinfo->pool, tvb, offset, tag_len));
    break;
  }
}

#define ARISTA_APNAME  6
static const value_string ieee80211_vs_arista_subtype_vals[] = {
  { ARISTA_APNAME, "AP Name"},
  { 0,                 NULL }
};
static void
dissect_vendor_ie_arista(proto_item *item, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint8_t type;
  const uint8_t* name;

  offset += 1; /* VS OUI Type */
  tag_len -= 1;

  proto_tree_add_item_ret_uint8(ietree, hf_ieee80211_vs_arista_subtype, tvb, offset, 1, ENC_NA, &type);
  proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_arista_subtype_vals, "Unknown"));
  offset += 1;
  tag_len -= 1;

  switch (type) {
  case ARISTA_APNAME:
    offset += 1;
    tag_len -= 1;

    proto_tree_add_item_ret_string(ietree, hf_ieee80211_vs_arista_apname, tvb,
                         offset, tag_len, ENC_ASCII|ENC_NA, pinfo->pool, &name);
    proto_item_append_text(item, " (%s)", name);
    break;

  default:
    proto_tree_add_item(ietree, hf_ieee80211_vs_arista_data, tvb, offset,
      tag_len, ENC_NA);
    if (tag_len > 0)
      proto_item_append_text(item, " (Data: %s)", tvb_bytes_to_str(pinfo->pool, tvb, offset, tag_len));
    break;
  }
}

#define WISUN_PTKID 1
#define WISUN_GTKL  2
#define WISUN_NR    3
#define WISUN_LGTKL 4
#define WISUN_LGTK  5

static const value_string ieee80211_vs_wisun_type_vals[] = {
  { WISUN_PTKID, "PTKID" },
  { WISUN_GTKL,  "GTKL"  },
  { WISUN_NR,    "NR"    },
  { WISUN_LGTKL, "LGTKL" },
  { WISUN_LGTK,  "LGTK"  },
  { 0,           NULL    }
};

#define WISUN_GTKL_GTK0 0x01
#define WISUN_GTKL_GTK1 0x02
#define WISUN_GTKL_GTK2 0x04
#define WISUN_GTKL_GTK3 0x08

#define WISUN_NR_BR     0
#define WISUN_NR_ROUTER 1
#define WISUN_NR_LFN    2

static const value_string ieee80211_vs_wisun_nr_vals[] = {
  { WISUN_NR_BR,     "Border Router" },
  { WISUN_NR_ROUTER, "Router"        },
  { WISUN_NR_LFN,    "LFN"           },
  { 0,               NULL            }
};

#define WISUN_LGTKL_LGTK0 0x01
#define WISUN_LGTKL_LGTK1 0x02
#define WISUN_LGTKL_LGTK2 0x04

static void
dissect_vendor_ie_wisun(proto_item *item, proto_tree *ietree,
                        tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint32_t type;

  proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_wisun_type, tvb, offset, 1, ENC_LITTLE_ENDIAN, &type);
  proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_wisun_type_vals, "Unknown"));
  offset += 1;
  tag_len -= 1;

  switch(type) {
  case WISUN_PTKID:
    proto_tree_add_item(ietree, hf_ieee80211_vs_wisun_ptkid, tvb, offset, 16, ENC_NA);
    break;
  case WISUN_GTKL: {
    static int * const wisun_gtkl[] = {
        &hf_ieee80211_vs_wisun_gtkl_gtk0,
        &hf_ieee80211_vs_wisun_gtkl_gtk1,
        &hf_ieee80211_vs_wisun_gtkl_gtk2,
        &hf_ieee80211_vs_wisun_gtkl_gtk3,
        NULL,
    };

    proto_tree_add_bitmask(ietree, tvb, offset, hf_ieee80211_vs_wisun_gtkl,
                           ett_wisun_gtkl, wisun_gtkl, ENC_LITTLE_ENDIAN);
    break;
  }
  case WISUN_NR:
    proto_tree_add_item(ietree, hf_ieee80211_vs_wisun_nr, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    break;
  case WISUN_LGTKL: {
    static int * const wisun_lgtkl[] = {
        &hf_ieee80211_vs_wisun_lgtkl_lgtk0,
        &hf_ieee80211_vs_wisun_lgtkl_lgtk1,
        &hf_ieee80211_vs_wisun_lgtkl_lgtk2,
        NULL,
    };

    proto_tree_add_bitmask(ietree, tvb, offset, hf_ieee80211_vs_wisun_lgtkl,
                           ett_wisun_lgtkl, wisun_lgtkl, ENC_LITTLE_ENDIAN);
    break;
  }
  case WISUN_LGTK:
    proto_tree_add_item(ietree, hf_ieee80211_vs_wisun_lgtk_key_id, tvb,
                        offset, 1, ENC_LITTLE_ENDIAN);
    offset += 2;
    tag_len -= 2;
    proto_tree_add_item(ietree, hf_ieee80211_vs_wisun_lgtk_lgtk, tvb,
                        offset, tag_len, ENC_NA);
    break;
  default:
    proto_tree_add_item(ietree, hf_ieee80211_vs_wisun_data, tvb, offset, tag_len, ENC_NA);
    if (tag_len > 0)
      proto_item_append_text(item, " (Data: %s)", tvb_bytes_to_str(pinfo->pool, tvb, offset, tag_len));
    break;
  }
}

static const value_string ieee80211_vs_apple_subtype_vals[] = {
  { 0,                 NULL }
};

static void
dissect_vendor_ie_apple(proto_item *item, proto_tree *ietree,
                          tvbuff_t *tvb, int offset, uint32_t tag_len, packet_info *pinfo)
{
  uint32_t subtype, type, length;

  proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_apple_type, tvb, offset, 1, ENC_LITTLE_ENDIAN, &type);
  offset += 1;
  tag_len -= 1;

  if(type == 6) { /* when type is 6, there is some subtype...*/
    proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_apple_subtype, tvb, offset, 2, ENC_LITTLE_ENDIAN, &subtype);
    proto_item_append_text(item, ": %s", val_to_str_const(type, ieee80211_vs_apple_subtype_vals, "Unknown"));
    offset += 2;
    tag_len -= 2;

    proto_tree_add_item_ret_uint(ietree, hf_ieee80211_vs_apple_length, tvb, offset, 1, ENC_NA, &length);
    offset += 1;
    tag_len -= 1;

    switch (subtype) {
    default:
      proto_tree_add_item(ietree, hf_ieee80211_vs_apple_data, tvb, offset, tag_len, ENC_NA);
      if (tag_len > 0)
        proto_item_append_text(item, " (Data: %s)", tvb_bytes_to_str(pinfo->pool, tvb, offset, tag_len));
      break;
    }
  } else{
    proto_tree_add_item(ietree, hf_ieee80211_vs_apple_data, tvb, offset, tag_len, ENC_NA);
    if (tag_len > 0)
      proto_item_append_text(item, " (Data: %s)", tvb_bytes_to_str(pinfo->pool, tvb, offset, tag_len));
  }

}

/* 802.11-2012 8.4.2.37 QoS Capability element */
static int
dissect_qos_capability(proto_tree *tree, tvbuff_t *tvb, packet_info *pinfo, int offset, int ftype)
{
  switch (ftype) {
    case MGT_ASSOC_REQ:
    case MGT_PROBE_REQ:
    case MGT_REASSOC_REQ:
    {
      /* To AP so decode Qos Info as STA */
      offset += add_ff_qos_info_sta(tree, tvb, pinfo, offset);
      break;
    }

    case MGT_BEACON:
    case MGT_PROBE_RESP:
    case MGT_ASSOC_RESP:
    case MGT_REASSOC_RESP:
    {
      /* From AP so decode QoS Info as AP */
      offset += add_ff_qos_info_ap(tree, tvb, pinfo, offset);
      break;
    }

    default:
      expert_add_info_format(pinfo, proto_tree_get_parent(tree), &ei_ieee80211_qos_info_bad_ftype,
                             "Could not deduce direction to decode correctly, ftype %u", ftype);
      break;
  }

  return offset;
}

static ieee80211_packet_data_t* get_or_create_packet_data(packet_info *pinfo) {
  ieee80211_packet_data_t *packet_data =
    (ieee80211_packet_data_t*)p_get_proto_data(pinfo->pool, pinfo, proto_wlan, PACKET_DATA_KEY);
  if (!packet_data) {
    packet_data = wmem_new(pinfo->pool, ieee80211_packet_data_t);
    p_add_proto_data(pinfo->pool, pinfo, proto_wlan, PACKET_DATA_KEY, packet_data);
    memset(packet_data, 0, sizeof(ieee80211_packet_data_t));
  }
  return packet_data;
}

/* See ieee80211_rsn_keymgmt_vals */
static bool is_ft_akm_suite(uint32_t akm_suite)
{
  switch (akm_suite) {
    case AKMS_FT_IEEE802_1X:
    case AKMS_FT_PSK:
    case AKMS_FT_SAE:
    case AKMS_FT_IEEE802_1X_SHA384:
    case AKMS_FT_FILS_SHA256:
    case AKMS_FT_FILS_SHA384:
    case AKMS_FT_SAE_GROUP_DEPEND:
      return true;
    default:
      return false;
  }
}

static void
save_proto_data(tvbuff_t *tvb, packet_info *pinfo, int offset, size_t size, int key)
{
  uint8_t *data;

  if (!enable_decryption) {
    return;
  }
  data = (uint8_t *)wmem_alloc(pinfo->pool, size);
  tvb_memcpy(tvb, data, offset, size);
  p_add_proto_data(pinfo->pool, pinfo, proto_wlan, key, data);
}

static void
save_proto_data_value(packet_info *pinfo, unsigned value, int key)
{
  if (!enable_decryption) {
    return;
  }
  p_add_proto_data(pinfo->pool, pinfo, proto_wlan, key, GUINT_TO_POINTER(value));
}

static void
save_tag_for_dot11decrypt(tvbuff_t *tvb, packet_info *pinfo, int offset)
{
  uint8_t tag_no;
  uint8_t tag_len;

  if (!enable_decryption) {
    return;
  }
  tag_no  = tvb_get_uint8(tvb, offset);
  tag_len = tvb_get_uint8(tvb, offset + 1);

  switch (tag_no) {
    case TAG_MOBILITY_DOMAIN:
      save_proto_data(tvb, pinfo, offset, tag_len + 2, MDE_TAG_KEY);
      break;
    case TAG_FAST_BSS_TRANSITION:
      save_proto_data(tvb, pinfo, offset, tag_len + 2, FTE_TAG_KEY);
      break;
    case TAG_RIC_DATA:
      save_proto_data(tvb, pinfo, offset, tag_len + 2, RDE_TAG_KEY);
      break;
    case TAG_RSN_IE:
      save_proto_data(tvb, pinfo, offset, tag_len + 2, RSNE_TAG_KEY);
      break;
    case TAG_RSNX:
      save_proto_data(tvb, pinfo, offset, tag_len + 2, RSNXE_TAG_KEY);
      break;
    default:
      break;
  }
}

static void
set_packet_data_last_akm_suite(ieee80211_packet_data_t *packet_data,
                               uint32_t last_akm_suite)
{
  packet_data->last_akm_suite_set = true;
  packet_data->last_akm_suite = last_akm_suite;
}

static void
set_conversation_last_akm_suite(ieee80211_conversation_data_t *conv,
                                uint32_t last_akm_suite)
{
  conv->last_akm_suite_set = true;
  conv->last_akm_suite = last_akm_suite;
}

/*
 * 7.3.2.25 RSNE information element. Common format with OSEN except the
 * version... should refactor
 */
static int
dissect_rsn_ie(packet_info *pinfo, proto_tree *tree, tvbuff_t *tvb,
               int offset, uint32_t tag_len, association_sanity_check_t *association_sanity_check)
{
  proto_item *rsn_gcs_item, *rsn_pcs_item, *rsn_akms_item, *rsn_pmkid_item, *rsn_gmcs_item;
  proto_item *rsn_sub_pcs_item, *rsn_sub_akms_item;
  proto_item *rsn_pcs_count, *rsn_akms_count, *rsn_pmkid_count;
  proto_tree *rsn_gcs_tree, *rsn_pcs_tree, *rsn_akms_tree, *rsn_pmkid_tree, *rsn_gmcs_tree;
  proto_tree *rsn_sub_pcs_tree, *rsn_sub_akms_tree;
  uint16_t    pcs_count, akms_count, pmkid_count;
  unsigned    ii;
  int         tag_end = offset + tag_len;
  static int * const ieee80211_rsn_cap[] = {
    &hf_ieee80211_rsn_cap_preauth,
    &hf_ieee80211_rsn_cap_no_pairwise,
    &hf_ieee80211_rsn_cap_ptksa_replay_counter,
    &hf_ieee80211_rsn_cap_gtksa_replay_counter,
    &hf_ieee80211_rsn_cap_mfpr,
    &hf_ieee80211_rsn_cap_mfpc,
    &hf_ieee80211_rsn_cap_jmr,
    &hf_ieee80211_rsn_cap_peerkey,
    &hf_ieee80211_rsn_cap_spp_amsdu_cap,
    &hf_ieee80211_rsn_cap_spp_amsdu_req,
    &hf_ieee80211_rsn_cap_pbac,
    &hf_ieee80211_rsn_cap_extended_key_id_iaf,
    &hf_ieee80211_rsn_cap_ocvc,
    NULL
  };

  proto_tree_add_item(tree, hf_ieee80211_rsn_version, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  if (offset >= tag_end)
    return offset;

  /* 7.3.2.25.1 Group Cipher suites */
  rsn_gcs_item = proto_tree_add_item(tree, hf_ieee80211_rsn_gcs, tvb, offset, 4, ENC_BIG_ENDIAN);
  rsn_gcs_tree = proto_item_add_subtree(rsn_gcs_item, ett_rsn_gcs_tree);
  proto_tree_add_item(rsn_gcs_tree, hf_ieee80211_rsn_gcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

    /* Check if OUI is 00:0F:AC (ieee80211) */
  if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
  {
    proto_tree_add_item(rsn_gcs_tree, hf_ieee80211_rsn_gcs_80211_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
    save_proto_data_value(pinfo, tvb_get_uint8(tvb, offset + 3), GROUP_CIPHER_KEY);
  } else {
    proto_tree_add_item(rsn_gcs_tree, hf_ieee80211_rsn_gcs_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  }
  offset += 4;

  if (offset >= tag_end)
    return offset;

  /* 7.3.2.25.2 Pairwise Cipher suites */
  rsn_pcs_count = proto_tree_add_item_ret_uint16(tree, hf_ieee80211_rsn_pcs_count, tvb, offset, 2, ENC_LITTLE_ENDIAN, &pcs_count);
  offset += 2;

  if (offset + (pcs_count * 4) > tag_end)
  {
    expert_add_info_format(pinfo, rsn_pcs_count, &ei_ieee80211_rsn_pcs_count,
        "Pairwise Cipher Suite Count too large, 4*%u > %d", pcs_count, tag_end - offset);
    pcs_count = (tag_end - offset) / 4;
  }

  rsn_pcs_item = proto_tree_add_item(tree, hf_ieee80211_rsn_pcs_list, tvb, offset, pcs_count * 4, ENC_NA);
  rsn_pcs_tree = proto_item_add_subtree(rsn_pcs_item, ett_rsn_pcs_tree);
  for (ii = 0; ii < pcs_count; ii++)
  {
    rsn_sub_pcs_item = proto_tree_add_item(rsn_pcs_tree, hf_ieee80211_rsn_pcs, tvb, offset, 4, ENC_BIG_ENDIAN);
    rsn_sub_pcs_tree = proto_item_add_subtree(rsn_sub_pcs_item, ett_rsn_sub_pcs_tree);
    proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

    /* Check if OUI is 00:0F:AC (ieee80211) */
    if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
    {
      proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_80211_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(rsn_pcs_item, " %s", rsn_pcs_return(pinfo->pool, tvb_get_ntohl(tvb, offset)));
      save_proto_data_value(pinfo, tvb_get_uint8(tvb, offset + 3), CIPHER_KEY);
    } else {
      proto_tree_add_item(rsn_sub_pcs_tree, hf_ieee80211_rsn_pcs_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
    }
    offset += 4;
  }

  if (offset >= tag_end)
  {
    return offset;
  }

  /* 7.3.2.25.2 AKM suites */
  rsn_akms_count = proto_tree_add_item_ret_uint16(tree, hf_ieee80211_rsn_akms_count, tvb, offset, 2, ENC_LITTLE_ENDIAN, &akms_count);
  offset += 2;

  if (offset + (akms_count * 4) > tag_end)
  {
    expert_add_info_format(pinfo, rsn_akms_count, &ei_ieee80211_rsn_pmkid_count,
        "Auth Key Management (AKM) Suite Count too large, 4*%u > %d", akms_count, tag_end - offset);
    akms_count = (tag_end - offset) / 4;
  }

  rsn_akms_item = proto_tree_add_item(tree, hf_ieee80211_rsn_akms_list, tvb, offset, akms_count * 4, ENC_NA);
  rsn_akms_tree = proto_item_add_subtree(rsn_akms_item, ett_rsn_akms_tree);

  ieee80211_packet_data_t *packet_data = get_or_create_packet_data(pinfo);

  for (ii = 0; ii < akms_count; ii++)
  {
    rsn_sub_akms_item = proto_tree_add_item(rsn_akms_tree, hf_ieee80211_rsn_akms, tvb, offset, 4, ENC_BIG_ENDIAN);
    rsn_sub_akms_tree = proto_item_add_subtree(rsn_sub_akms_item, ett_rsn_sub_akms_tree);
    proto_tree_add_item(rsn_sub_akms_tree, hf_ieee80211_rsn_akms_oui, tvb, offset, 3, ENC_BIG_ENDIAN);

    /* Check if OUI is 00:0F:AC (ieee80211) */
    if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
    {
      proto_tree_add_item(rsn_sub_akms_tree, hf_ieee80211_rsn_akms_80211_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
      proto_item_append_text(rsn_akms_item, " %s", rsn_akms_return(pinfo->pool, tvb_get_ntohl(tvb, offset)));
      save_proto_data_value(pinfo, tvb_get_uint8(tvb, offset + 3), AKM_KEY);

      set_packet_data_last_akm_suite(packet_data, tvb_get_ntohl(tvb, offset));
      if (association_sanity_check) {
        uint32_t akm_suite = tvb_get_ntohl(tvb, offset);
        association_sanity_check->last_akm_suite = akm_suite;

        if (is_ft_akm_suite(akm_suite)) {
          /* This is an FT AKM suite */
          association_sanity_check->has_ft_akm_suite = true;
          if (association_sanity_check->rsn_first_ft_akm_suite == NULL && rsn_sub_akms_tree != NULL) {
            association_sanity_check->rsn_first_ft_akm_suite = rsn_sub_akms_tree->last_child;
          }
        } else {
          /* This is a non-FT AKM suite */
          association_sanity_check->has_non_ft_akm_suite = true;
          if (association_sanity_check->rsn_first_non_ft_akm_suite == NULL && rsn_sub_akms_tree != NULL) {
            association_sanity_check->rsn_first_non_ft_akm_suite = rsn_sub_akms_tree->last_child;
          }
        }
      }
    } else {
      proto_tree_add_item(rsn_sub_akms_tree, hf_ieee80211_rsn_akms_type, tvb, offset+3, 1, ENC_LITTLE_ENDIAN);
    }
    offset += 4;
  }

  /* 7.3.2.25.3 RSN capabilities */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_rsn_cap,
                                    ett_rsn_cap_tree, ieee80211_rsn_cap,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  if (offset >= tag_end)
  {
    return offset;
  }
  /* 7.3.2.25.4 PMKID */
  rsn_pmkid_count = proto_tree_add_item_ret_uint16(tree, hf_ieee80211_rsn_pmkid_count, tvb, offset, 2, ENC_LITTLE_ENDIAN, &pmkid_count);
  offset += 2;

  if (offset + (pmkid_count * 16) > tag_end)
  {
    expert_add_info_format(pinfo, rsn_pmkid_count, &ei_ieee80211_pmkid_count_too_large,
        "PMKID Count too large, 16*%u > %d", pmkid_count, tag_end - offset);
    pmkid_count = (tag_end - offset) / 16;
  }

  rsn_pmkid_item = proto_tree_add_item(tree, hf_ieee80211_rsn_pmkid_list, tvb, offset, pmkid_count * 16, ENC_NA);
  rsn_pmkid_tree = proto_item_add_subtree(rsn_pmkid_item, ett_rsn_pmkid_tree);
  for (ii = 0; ii < pmkid_count; ii++)
  {
    proto_tree_add_item(rsn_pmkid_tree, hf_ieee80211_rsn_pmkid, tvb, offset, 16, ENC_NA);
    offset += 16;
  }

  if (offset >= tag_end)
  {
    return offset;
  }
  /* Group Management Cipher Suite (802.11w)*/
  rsn_gmcs_item = proto_tree_add_item(tree, hf_ieee80211_rsn_gmcs, tvb, offset, 4, ENC_BIG_ENDIAN);
  rsn_gmcs_tree = proto_item_add_subtree(rsn_gmcs_item, ett_rsn_gmcs_tree);
  proto_tree_add_item(rsn_gmcs_tree, hf_ieee80211_rsn_gmcs_oui, tvb, offset, 3, ENC_BIG_ENDIAN);
  /* Check if OUI is 00:0F:AC (ieee80211) */
  if (tvb_get_ntoh24(tvb, offset) == OUI_RSN)
  {
    proto_tree_add_item(rsn_gmcs_tree, hf_ieee80211_rsn_gmcs_80211_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  } else {
    proto_tree_add_item(rsn_gmcs_tree, hf_ieee80211_rsn_gmcs_type, tvb, offset + 3, 1, ENC_LITTLE_ENDIAN);
  }
  offset += 4;

  return offset;
}

/* 7.3.2.27 Extended Capabilities information element (127) */
static int
dissect_extended_capabilities_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *ti_ex_cap;
  static int * const ieee80211_tag_extended_capabilities_byte1[] = {
    &hf_ieee80211_tag_extended_capabilities_b0,
    &hf_ieee80211_tag_extended_capabilities_b1,
    &hf_ieee80211_tag_extended_capabilities_b2,
    &hf_ieee80211_tag_extended_capabilities_b3,
    &hf_ieee80211_tag_extended_capabilities_b4,
    &hf_ieee80211_tag_extended_capabilities_b5,
    &hf_ieee80211_tag_extended_capabilities_b6,
    &hf_ieee80211_tag_extended_capabilities_b7,
    NULL
  };
  static int * const ieee80211_tag_extended_capabilities_byte2[] = {
    &hf_ieee80211_tag_extended_capabilities_b8,
    &hf_ieee80211_tag_extended_capabilities_b9,
    &hf_ieee80211_tag_extended_capabilities_b10,
    &hf_ieee80211_tag_extended_capabilities_b11,
    &hf_ieee80211_tag_extended_capabilities_b12,
    &hf_ieee80211_tag_extended_capabilities_b13,
    &hf_ieee80211_tag_extended_capabilities_b14,
    &hf_ieee80211_tag_extended_capabilities_b15,
    NULL
  };
  static int * const ieee80211_tag_extended_capabilities_byte3[] = {
    &hf_ieee80211_tag_extended_capabilities_b16,
    &hf_ieee80211_tag_extended_capabilities_b17,
    &hf_ieee80211_tag_extended_capabilities_b18,
    &hf_ieee80211_tag_extended_capabilities_b19,
    &hf_ieee80211_tag_extended_capabilities_b20,
    &hf_ieee80211_tag_extended_capabilities_b21,
    &hf_ieee80211_tag_extended_capabilities_b22,
    &hf_ieee80211_tag_extended_capabilities_b23,
    NULL
  };
  static int * const ieee80211_tag_extended_capabilities_byte4[] = {
    &hf_ieee80211_tag_extended_capabilities_b24,
    &hf_ieee80211_tag_extended_capabilities_b25,
    &hf_ieee80211_tag_extended_capabilities_b26,
    &hf_ieee80211_tag_extended_capabilities_b27,
    &hf_ieee80211_tag_extended_capabilities_b28,
    &hf_ieee80211_tag_extended_capabilities_b29,
    &hf_ieee80211_tag_extended_capabilities_b30,
    &hf_ieee80211_tag_extended_capabilities_b31,
    NULL
  };
  static int * const ieee80211_tag_extended_capabilities_byte5[] = {
    &hf_ieee80211_tag_extended_capabilities_b32,
    &hf_ieee80211_tag_extended_capabilities_b33,
    &hf_ieee80211_tag_extended_capabilities_b34,
    &hf_ieee80211_tag_extended_capabilities_b35,
    &hf_ieee80211_tag_extended_capabilities_b36,
    &hf_ieee80211_tag_extended_capabilities_b37,
    &hf_ieee80211_tag_extended_capabilities_b38,
    &hf_ieee80211_tag_extended_capabilities_b39,
    NULL
  };
  static int * const ieee80211_tag_extended_capabilities_byte6[] = {
    &hf_ieee80211_tag_extended_capabilities_b40,
    &hf_ieee80211_tag_extended_capabilities_serv_int_granularity,
    &hf_ieee80211_tag_extended_capabilities_b44,
    &hf_ieee80211_tag_extended_capabilities_b45,
    &hf_ieee80211_tag_extended_capabilities_b46,
    &hf_ieee80211_tag_extended_capabilities_b47,
    NULL
  };
  static int * const ieee80211_tag_extended_capabilities_byte7[] = {
    &hf_ieee80211_tag_extended_capabilities_b48,
    &hf_ieee80211_tag_extended_capabilities_b49,
    &hf_ieee80211_tag_extended_capabilities_b50,
    &hf_ieee80211_tag_extended_capabilities_b51,
    &hf_ieee80211_tag_extended_capabilities_b52,
    &hf_ieee80211_tag_extended_capabilities_b53,
    &hf_ieee80211_tag_extended_capabilities_b54,
    &hf_ieee80211_tag_extended_capabilities_b55,
    NULL
  };

  static int * const ieee80211_tag_extended_capabilities_byte8[] = {
    &hf_ieee80211_tag_extended_capabilities_b56,
    &hf_ieee80211_tag_extended_capabilities_b57,
    &hf_ieee80211_tag_extended_capabilities_b58,
    &hf_ieee80211_tag_extended_capabilities_b59,
    &hf_ieee80211_tag_extended_capabilities_b60,
    &hf_ieee80211_tag_extended_capabilities_b61,
    &hf_ieee80211_tag_extended_capabilities_b62,
    &hf_ieee80211_tag_extended_capabilities_b63,
    NULL
  };

  static int * const ieee80211_tag_extended_capabilities_bytes89[] = {
    &hf_ieee80211_tag_extended_capabilities_b56_2,
    &hf_ieee80211_tag_extended_capabilities_b57_2,
    &hf_ieee80211_tag_extended_capabilities_b58_2,
    &hf_ieee80211_tag_extended_capabilities_b59_2,
    &hf_ieee80211_tag_extended_capabilities_b60_2,
    &hf_ieee80211_tag_extended_capabilities_b61_2,
    &hf_ieee80211_tag_extended_capabilities_b62_2,
    &hf_ieee80211_tag_extended_capabilities_max_num_msdus,
    &hf_ieee80211_tag_extended_capabilities_b65_2,
    &hf_ieee80211_tag_extended_capabilities_b66_2,
    &hf_ieee80211_tag_extended_capabilities_b67_2,
    &hf_ieee80211_tag_extended_capabilities_b68_2,
    &hf_ieee80211_tag_extended_capabilities_b69_2,
    &hf_ieee80211_tag_extended_capabilities_b70_2,
    &hf_ieee80211_tag_extended_capabilities_b71_2,
    NULL
  };

  static int * const ieee80211_tag_extended_capabilities_byte10[] = {
    &hf_ieee80211_tag_extended_capabilities_b72,
    &hf_ieee80211_tag_extended_capabilities_b73,
    &hf_ieee80211_tag_extended_capabilities_b74,
    &hf_ieee80211_tag_extended_capabilities_b75,
    &hf_ieee80211_tag_extended_capabilities_b76,
    &hf_ieee80211_tag_extended_capabilities_b77,
    &hf_ieee80211_tag_extended_capabilities_b78,
    &hf_ieee80211_tag_extended_capabilities_b79,
    NULL
  };

  static int * const ieee80211_tag_extended_capabilities_byte11[] = {
    &hf_ieee80211_tag_extended_capabilities_b80,
    &hf_ieee80211_tag_extended_capabilities_b81,
    &hf_ieee80211_tag_extended_capabilities_b82,
    &hf_ieee80211_tag_extended_capabilities_b83,
    &hf_ieee80211_tag_extended_capabilities_b84,
    &hf_ieee80211_tag_extended_capabilities_b85,
    &hf_ieee80211_tag_extended_capabilities_b86,
    &hf_ieee80211_tag_extended_capabilities_b87,
    NULL
  };

  static int * const ieee80211_tag_extended_capabilities_byte12[] = {
    &hf_ieee80211_tag_extended_capabilities_b88,
    &hf_ieee80211_tag_extended_capabilities_b89,
    &hf_ieee80211_tag_extended_capabilities_b90,
    &hf_ieee80211_tag_extended_capabilities_b91,
    &hf_ieee80211_tag_extended_capabilities_b92,
    &hf_ieee80211_tag_extended_capabilities_b93,
    &hf_ieee80211_tag_extended_capabilities_b94,
    &hf_ieee80211_tag_extended_capabilities_b95,
    NULL
  };

  static int * const ieee80211_tag_extended_capabilities_byte13[] = {
    &hf_ieee80211_tag_extended_capabilities_b96,
    &hf_ieee80211_tag_extended_capabilities_b97,
    &hf_ieee80211_tag_extended_capabilities_b98,
    &hf_ieee80211_tag_extended_capabilities_b99,
    &hf_ieee80211_tag_extended_capabilities_b100,
    &hf_ieee80211_tag_extended_capabilities_b101,
    &hf_ieee80211_tag_extended_capabilities_b102,
    &hf_ieee80211_tag_extended_capabilities_b103,
    NULL
  };

  static int * const ieee80211_tag_extended_capabilities_byte14[] = {
    &hf_ieee80211_tag_extended_capabilities_b104,
    &hf_ieee80211_tag_extended_capabilities_b105,
    &hf_ieee80211_tag_extended_capabilities_reserved2,
    NULL
  };

  if (tag_len < 1)
  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "Tag length %u too short, must be greater than 0", tag_len);
    return 1;
  }
  proto_item_append_text(field_data->item_tag, " (%u octet%s)", tag_len, plurality(tag_len, "", "s"));

  /* Extended Capability octet 1 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap1, ieee80211_tag_extended_capabilities_byte1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 1)");
  offset += 1;

  /* Extended Capability octet 2 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap2, ieee80211_tag_extended_capabilities_byte2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 2)");
  offset += 1;

  /* Extended Capability octet 3 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap3, ieee80211_tag_extended_capabilities_byte3,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 3)");
  offset += 1;

  /* Extended Capability octet 4 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap4, ieee80211_tag_extended_capabilities_byte4,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 4)");
  offset += 1;

  /* Extended Capability octet 5 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap5, ieee80211_tag_extended_capabilities_byte5,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 5)");
  offset += 1;

  /* Extended Capability octet 6 */
  if (offset >= tag_len) {
    return offset;
  }

  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap6, ieee80211_tag_extended_capabilities_byte6,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 6)");
  offset += 1;


  /* Extended Capability octet 7 */
  if (offset >= tag_len) {
    return offset;
  }
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap7, ieee80211_tag_extended_capabilities_byte7,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 7)");
  offset += 1;

  /* Extended Capability octet 8 & 9 since two bits cross the boundary */
  if (offset >= tag_len) {
    return offset;
  }

  /* If only the first of the two bytes is present, do the best we can */
  if (offset == tag_len - 1) {
    ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap8, ieee80211_tag_extended_capabilities_byte8,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    proto_item_append_text(ti_ex_cap, " (octet 8)");
    offset += 1;
  } else { /* Both bytes are there */
    ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities_2,
                                    ett_tag_ex_cap89, ieee80211_tag_extended_capabilities_bytes89,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    proto_item_append_text(ti_ex_cap, " (octets 8 & 9)");
    offset += 2;
  }

  if (offset >= tag_len) {
    return offset;
  }

  /* Extended Capability octet 10 */
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap10, ieee80211_tag_extended_capabilities_byte10,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 10)");
  offset += 1;

  if (offset >= tag_len) {
    return offset;
  }

  /* Extended Capability octet 11 */
#if 0
  /* Added for SAE support */
  sae_byte = tvb_get_uint8(tvb, offset);
  /*
   * If one of the SAE bits is set, assume we will see Password identifiers
   */
  if (sae_byte & 0x6) {
    bool sae_val = true;
    uint64_t *key = NULL;

    /* Must be for the source of the request */
    key = (uint64_t *)wmem_new(wmem_file_scope(), uint64_t);
    *key = *(uint64_t *)pinfo->src.data;
    wmem_map_insert(sae_prop_hash, key, GINT_TO_POINTER(sae_val));
  }
#endif
  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap11, ieee80211_tag_extended_capabilities_byte11,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 11)");
  offset += 1;

  if (offset >= tag_len) {
    return offset;
  }

  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap12,
                                    ieee80211_tag_extended_capabilities_byte12,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 12)");
  offset += 1;

  if (offset >= tag_len) {
    return offset;
  }

  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap13,
                                    ieee80211_tag_extended_capabilities_byte13,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 13)");
  offset += 1;

  if (offset >= tag_len) {
    return offset;
  }

  ti_ex_cap = proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_tag_extended_capabilities,
                                    ett_tag_ex_cap14,
                                    ieee80211_tag_extended_capabilities_byte14,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  proto_item_append_text(ti_ex_cap, " (octet 14)");
  offset += 1;

  return offset;
}

static int
dissect_vht_mcs_set(proto_tree *tree, tvbuff_t *tvb, int offset)
{
  proto_item *ti;
  proto_tree *mcs_tree;
  static int * const ieee80211_vht_mcsset_rx_max_mcs[] = {
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_1_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_2_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_3_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_4_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_5_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_6_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_7_ss,
    &hf_ieee80211_vht_mcsset_rx_max_mcs_for_8_ss,
    NULL
  };
  static int * const ieee80211_vht_mcsset_tx_max_mcs[] = {
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_1_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_2_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_3_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_4_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_5_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_6_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_7_ss,
    &hf_ieee80211_vht_mcsset_tx_max_mcs_for_8_ss,
    NULL
  };

  /* 8 byte Supported MCS set */
  ti = proto_tree_add_item(tree, hf_ieee80211_vht_mcsset, tvb, offset, 8, ENC_NA);

  mcs_tree = proto_item_add_subtree(ti, ett_vht_mcsset_tree);

  /* B0 - B15 */
  proto_tree_add_bitmask_with_flags(mcs_tree, tvb, offset, hf_ieee80211_vht_mcsset_rx_mcs_map,
                                    ett_vht_rx_mcsbit_tree, ieee80211_vht_mcsset_rx_max_mcs,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* B16 - B28 13 bits*/
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_rx_highest_long_gi, tvb, offset, 2, ENC_LITTLE_ENDIAN);

  /* B29 - B31 2 Max NSTS, total*/

  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_max_nsts_total, tvb, offset,
                                2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* B32 - B47 */
  proto_tree_add_bitmask_with_flags(mcs_tree, tvb, offset, hf_ieee80211_vht_mcsset_tx_mcs_map,
                                    ett_vht_tx_mcsbit_tree, ieee80211_vht_mcsset_tx_max_mcs,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;
  /* B48 - B60 13 bits */
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_tx_highest_long_gi, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  /* B61 */
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_extended_nss_bw_capable,
                                tvb, offset, 2, ENC_LITTLE_ENDIAN);

  /* B62 - B63 2 reserved bits*/
  proto_tree_add_item(mcs_tree, hf_ieee80211_vht_mcsset_reserved, tvb, offset, 2,
                                ENC_LITTLE_ENDIAN);offset += 2;

  return offset;
}

static int
dissect_vht_capability_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static int * const ieee80211_vht_caps[] = {
    &hf_ieee80211_vht_max_mpdu_length,
    &hf_ieee80211_vht_supported_chan_width_set,
    &hf_ieee80211_vht_rx_ldpc,
    &hf_ieee80211_vht_short_gi_for_80,
    &hf_ieee80211_vht_short_gi_for_160,
    &hf_ieee80211_vht_tx_stbc,
    /* End of first byte */
    &hf_ieee80211_vht_rx_stbc,
    &hf_ieee80211_vht_su_beamformer_cap,
    &hf_ieee80211_vht_su_beamformee_cap,
    &hf_ieee80211_vht_beamformer_antennas,
    /* End of second byte */
    &hf_ieee80211_vht_sounding_dimensions,
    &hf_ieee80211_vht_mu_beamformer_cap,
    &hf_ieee80211_vht_mu_beamformee_cap,
    &hf_ieee80211_vht_txop_ps,
    &hf_ieee80211_vht_var_htc_field,
    &hf_ieee80211_vht_max_ampdu,
    &hf_ieee80211_vht_link_adaptation_cap,
    &hf_ieee80211_vht_rx_pattern,
    &hf_ieee80211_vht_tx_pattern,
    &hf_ieee80211_vht_ext_nss_bw_support,
    NULL
  };

  if (tag_len != 12) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "VHT Capabilities IE length %u wrong, must be = 12", tag_len);
    return 1;
  }

  /* 4 byte VHT Capabilities  Info*/
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_vht_cap,
                                    ett_vht_cap_tree, ieee80211_vht_caps,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 4;

  /* 8 byte MCS set */
  offset = dissect_vht_mcs_set(tree, tvb, offset);

  return offset;
}

static int
dissect_vht_operation_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *op_item;
  proto_tree *op_tree;
  static int * const ieee80211_vht_op_max_basic_mcs[] = {
    &hf_ieee80211_vht_op_max_basic_mcs_for_1_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_2_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_3_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_4_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_5_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_6_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_7_ss,
    &hf_ieee80211_vht_op_max_basic_mcs_for_8_ss,
    NULL
  };

  if (tag_len != 5) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "VHT Operation IE length %u wrong, must be = 5", tag_len);
    return 1;
  }

  /* 3 byte VHT Operation Info*/
  op_item = proto_tree_add_item(tree, hf_ieee80211_vht_op, tvb, offset, 3, ENC_NA);
  op_tree = proto_item_add_subtree(op_item, ett_vht_op_tree);
  proto_tree_add_item(op_tree, hf_ieee80211_vht_op_channel_width, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(op_tree, hf_ieee80211_vht_op_channel_center0, tvb, offset+1, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(op_tree, hf_ieee80211_vht_op_channel_center1, tvb, offset+2, 1, ENC_LITTLE_ENDIAN);

  offset += 3;
  /* VHT Basic MCS Set */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_vht_op_basic_mcs_map,
                                    ett_vht_basic_mcsbit_tree, ieee80211_vht_op_max_basic_mcs,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  return offset;
}

static int
dissect_vht_tx_pwr_envelope(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_item *tx_pwr_item, *ti, *unit_ti;
  proto_tree *tx_pwr_info_tree;
  uint8_t opt_ie_cnt=0;
  uint8_t i;
  unsigned mtpi;

  if (tag_len < 2 || tag_len > 18) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "VHT TX PWR Envelope IE length %u wrong, must be >= 2 and <= 18", tag_len);
    return 1;
  }

  mtpi = (tvb_get_uint8(tvb, offset) >> 3) & 0x7;
  tx_pwr_item = proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_info, tvb, offset, 1, ENC_NA);
  tx_pwr_info_tree =  proto_item_add_subtree(tx_pwr_item, ett_vht_tpe_info_tree);

  ti = proto_tree_add_item(tx_pwr_info_tree, hf_ieee80211_vht_tpe_pwr_info_count, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  unit_ti = proto_tree_add_item(tx_pwr_info_tree, hf_ieee80211_vht_tpe_pwr_info_unit, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tx_pwr_info_tree, hf_ieee80211_vht_tpe_pwr_info_category, tvb, offset, 1, ENC_LITTLE_ENDIAN);

  opt_ie_cnt = tvb_get_uint8(tvb, offset) & 0x07;

  offset += 1;

  switch (mtpi) {

  case 1:
  case 3:
  case 5:
    /* Is it a power spectral density? */
    /* Handle the zero case */
    if (opt_ie_cnt == 0) {
      proto_tree_add_item(tree, hf_ieee80211_vht_tpe_any_bw_psd, tvb, offset,
                          1, ENC_NA);
      offset += 1;
      return offset;
    }
    switch (opt_ie_cnt) {
    case 0:
      opt_ie_cnt = 0;
      break;
    case 1:
      opt_ie_cnt = 1;
      break;
    case 2:
      opt_ie_cnt = 2;
      break;
    case 3:
      opt_ie_cnt = 4;
      break;
    case 4:
      opt_ie_cnt = 8;
      break;
    default:
      opt_ie_cnt = 1; /* Add an expert info here ... */
      break;
    }
    for (i = 0; i < opt_ie_cnt; i++) {
      proto_tree *psd_tree;
      psd_tree = proto_tree_add_subtree_format(tree, tvb, offset, 1,
                                               ett_tpe_psd, NULL,
                                               "20 MHz Channel #%u", i);
      proto_tree_add_item(psd_tree, hf_ieee80211_vht_tpe_psd,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
    }
    /* Extension Max Tx Power */
    if (offset < tag_len) {
        proto_tree *psd_tree;
        uint8_t j;
        uint8_t ext_cnt = tvb_get_uint8(tvb, offset) & 0x0f;

        proto_tree_add_item(tree, hf_ieee80211_vht_tpe_ext_count, tvb, offset, 1, ENC_NA);
        proto_tree_add_item(tree, hf_ieee80211_vht_tpe_ext_reserved, tvb, offset, 1, ENC_NA);
        offset += 1;
        for (j = 0; j < ext_cnt; j++) {
          psd_tree = proto_tree_add_subtree_format(tree, tvb, offset, 1,
                                                   ett_tpe_psd, NULL,
                                                   "20 MHz Channel #%u", i+j);
          proto_tree_add_item(psd_tree, hf_ieee80211_vht_tpe_psd,
                              tvb, offset, 1, ENC_NA);
          offset += 1;
        }
    }
    break;

  case 0:
  case 2:
  case 4:
    /* Power Constraint info is mandatory only for 20MHz, others are optional*/
    /* Power is expressed in terms of 0.5dBm from -64 to 63 and is encoded
     * as 8-bit 2's complement */
    for (i = 0; i <= opt_ie_cnt; i++) {
      switch(i) {
      case 0:
        proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_20, tvb, offset, 1, ENC_NA);
        offset += 1;
        break;
      case 1:
        proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_40, tvb, offset, 1, ENC_NA);
        offset += 1;
        break;
      case 2:
        proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_80, tvb, offset, 1, ENC_NA);
        offset += 1;
        break;
      case 3:
        proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_160, tvb, offset, 1, ENC_NA);
        offset += 1;
        break;
      default:
        expert_add_info(pinfo, ti, &ei_ieee80211_vht_tpe_pwr_info_count);
        offset += 1;
        break;
      }
    }
    /* Extension Max Tx Power */
    if (offset < tag_len) {
        proto_tree_add_item(tree, hf_ieee80211_vht_tpe_pwr_constr_320, tvb, offset, 1, ENC_NA);
        offset += 1;
    }
    break;
  default:
    /* Reserved in 802.11ax-2021. 802.11be? */
    expert_add_info(pinfo, unit_ti, &ei_ieee80211_vht_tpe_pwr_info_unit);
  }

  return offset;
}

static int
dissect_mobility_domain(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (field_data->sanity_check != NULL) {
    field_data->sanity_check->association_has_mobility_domain_element = true;
  }

  if (tag_len < 3) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "MDIE content length must be at least 3 bytes");
    return 1;
  }

  save_proto_data(tvb, pinfo, offset, 2, MDID_KEY);
  proto_tree_add_item(tree, hf_ieee80211_tag_mobility_domain_mdid,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_tree_add_bitmask_with_flags(tree, tvb, offset + 2,
                                    hf_ieee80211_tag_mobility_domain_ft_capab,
                                    ett_tag_mobility_domain_ft_capab_tree,
                                    ieee80211_tag_mobility_domain_ft_capab_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  return tvb_captured_length(tvb);
}

static uint16_t get_mic_len_owe(uint16_t group) {
  switch(group) {
    // FFC, len(p) <= 2048
    case 1:
    case 2:
    case 5:
    case 14:
    case 22:
    case 23:
    case 24:
    // ECC, len(p) <= 256
    case 19:
    case 25:
    case 26:
    case 27:
    case 28:
    case 31:
      // HMAC-SHA-256
      return 16;

    // FFC, 2048 < len(p) <= 3072
    case 15:
    // ECC, 256 < len(p) <= 384
    case 20:
    case 29:
      // HMAC-SHA-384
      return 24;

    // FCC, 3072 < len(p)
    case 16:
    case 17:
    case 18:
    // ECC, 384 < len(p)
    case 21:
    case 30:
    case 32:
      // HMAC-SHA-512
      return 32;

    default:
      return 16;
  }
}

static uint16_t get_mic_len(uint32_t akm_suite) {
  switch(akm_suite) {
    case AKMS_WPA_SHA384_SUITEB:
    case AKMS_FT_IEEE802_1X_SHA384:
    case AKMS_FT_FILS_SHA384:
      // HMAC-SHA-384
      return 24;

    case AKMS_FILS_SHA256:
    case AKMS_FILS_SHA384:
      // AES-SIV-256 and AES-SIV-512
      return 0;

    default:
      // HMAC-SHA-1-128, AES-128-CMAC, HMAC-SHA-256, AKMS_FT_FILS_SHA256
      return 16;
  }
}

static conversation_t *find_wlan_conversation_pinfo(packet_info *pinfo)
{
  /* HACK to avoid collision with conversation in EAP dissector */
  pinfo->srcport = GPOINTER_TO_UINT(
    p_get_proto_data(wmem_file_scope(), pinfo, proto_wlan, ASSOC_COUNTER_KEY));
  pinfo->destport = pinfo->srcport;
  return find_conversation_pinfo(pinfo, 0);
}

static bool determine_nonce_is_set(tvbuff_t *tvb) {
  int offset;

  for (offset = 12; offset < 12 + 32; offset++)
    if (tvb_get_uint8(tvb, offset))
      return true;
  return false;
}

static uint16_t determine_mic_len(packet_info *pinfo, bool assoc_frame,
                                 bool *defaulted, bool *group_depend) {
  uint16_t eapol_key_mic_len = 16; /* Default MIC length */
  conversation_t *conversation = find_wlan_conversation_pinfo(pinfo);
  ieee80211_conversation_data_t *conversation_data = NULL;
  ieee80211_packet_data_t *packet_data =
    (ieee80211_packet_data_t*)p_get_proto_data(pinfo->pool, pinfo, proto_wlan, PACKET_DATA_KEY);
  if (conversation) {
      conversation_data = (ieee80211_conversation_data_t*)conversation_get_proto_data(conversation, proto_wlan);
  }

  if (wlan_key_mic_len_enable) {
    /* 1st - Use user overridden MIC length setting */
    eapol_key_mic_len = wlan_key_mic_len;
  }
  else if (!assoc_frame && conversation_data &&
           conversation_data->discovered_key_mic_len) {
    /*
     * 2nd - Use the discovered key mic len.
     * We will only set the discovered key mic len if it was defaulted
     * in an earlier call to determine_mic_len, so it should be tested second.
     */
      eapol_key_mic_len = conversation_data->discovered_key_mic_len;
  }
  else if (!assoc_frame && conversation_data &&
           conversation_data->last_akm_suite_set) {
    /* 3rd - Use AKMS negotiated during association to determine MIC length */
    if (conversation_data->last_akm_suite == AKMS_OWE) {
      /* For OWE the length of MIC depends on the selected group */
      *group_depend = true;
      eapol_key_mic_len = get_mic_len_owe(conversation_data->owe_group);
    } else if (conversation_data->last_akm_suite == AKMS_SAE_GROUP_DEPEND ||
               conversation_data->last_akm_suite == AKMS_FT_SAE_GROUP_DEPEND) {
      *group_depend = true;
      *defaulted = true;
    }
    else {
      eapol_key_mic_len = get_mic_len(conversation_data->last_akm_suite);
    }
  }
  else if (packet_data && packet_data->last_akm_suite_set) {
    /* 3rd - Use AKMS from current packet to determine MIC length */
    if (packet_data->last_akm_suite == AKMS_OWE) {
      /* For OWE the length of MIC depends on the selected group */
      *group_depend = true;
      eapol_key_mic_len = get_mic_len_owe(packet_data->owe_group);
    } else if (packet_data->last_akm_suite == AKMS_SAE_GROUP_DEPEND ||
               packet_data->last_akm_suite == AKMS_FT_SAE_GROUP_DEPEND) {
      *group_depend = true;
      *defaulted = true;
    }
    else {
      eapol_key_mic_len = get_mic_len(packet_data->last_akm_suite);
    }
  } else {
    /*
     * We used the default so say so.
     */
    *defaulted = true;
  }
  return eapol_key_mic_len;
}

static int
dissect_fast_bss_transition(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  bool assoc_frame = field_data->sanity_check != NULL;
  int offset = 0;
  uint64_t mic_control;
  if (tag_len < 82) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "FTIE content length must be at least 82 bytes");
    return 1;
  }

  proto_tree_add_bitmask_with_flags_ret_uint64(tree, tvb, offset, hf_ieee80211_tag_ft_mic_control,
                                    ett_tag_ft_mic_control_tree,
                                    ieee80211_tag_ft_mic_control_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND, &mic_control);
  offset += 2;

  bool defaulted_mic_len = false, group_depended_mic_len = false;
  int mic_len = determine_mic_len(pinfo, assoc_frame, &defaulted_mic_len, &group_depended_mic_len);
  if (group_depended_mic_len) {
    /* IEEE 802.11-2024 9.4.2.46 FTE, Table 9-220 MIC Length subfield values */
    uint8_t mic_len_field = (mic_control & 0x0e) >> 1; /* B1 to B3 */
    if (mic_len_field == 0)
        mic_len = 16;
    else if (mic_len_field == 1)
      mic_len = 24;
    else if (mic_len_field == 2)
      mic_len = 32;
  }
  save_proto_data(tvb, pinfo, offset, mic_len, FTE_MIC_KEY);
  save_proto_data_value(pinfo, mic_len, FTE_MIC_LEN_KEY);
  proto_tree_add_item(tree, hf_ieee80211_tag_ft_mic,
                      tvb, offset, mic_len, ENC_NA);
  offset += mic_len;

  save_proto_data(tvb, pinfo, offset, 32, FTE_ANONCE_KEY);
  proto_tree_add_item(tree, hf_ieee80211_tag_ft_anonce,
                      tvb, offset, 32, ENC_NA);
  offset += 32;

  save_proto_data(tvb, pinfo, offset, 32, FTE_SNONCE_KEY);
  proto_tree_add_item(tree, hf_ieee80211_tag_ft_snonce,
                      tvb, offset, 32, ENC_NA);
  offset += 32;

  while (offset + 2 <= tag_len) {
    uint8_t id, len;
    int s_end;
    proto_item *ti;
    proto_tree *subtree;
    const char *subtree_name;
    proto_keydata_t *proto;

    id = tvb_get_uint8(tvb, offset);
    len = tvb_get_uint8(tvb, offset + 1);
    subtree_name = val_to_str_const(id, ft_subelem_id_vals, "Unknown");
    subtree = proto_tree_add_subtree_format(tree, tvb, offset, len + 2,
                                            ett_tag_ft_subelem_tree, NULL,
                                            "Subelement: %s", subtree_name);

    proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_id,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    ti = proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_len,
                             tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;

    if (offset + len > tag_len) {
      expert_add_info_format(pinfo, ti, &ei_ieee80211_tag_length,
                            "FTIE subelement length is too large for the FTIE content length");
      return offset;
    }

    s_end = offset + len;
    switch (id) {
    case 1:
      save_proto_data(tvb, pinfo, offset, len, FTE_R1KH_ID_KEY);
      save_proto_data_value(pinfo, len, FTE_R1KH_ID_LEN_KEY);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_r1kh_id,
                          tvb, offset, len, ENC_NA);
      break;
    case 2:
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_gtk_key_info,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_gtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      save_proto_data_value(pinfo, tvb_get_uint8(tvb, offset), GTK_SUBELEM_KEY_LEN_KEY);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_gtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_gtk_rsc,
                          tvb, offset, 8, ENC_NA);
      offset += 8;
      if (offset > s_end)
        break;
      save_proto_data_value(pinfo, s_end - offset, GTK_LEN_KEY);
      save_proto_data(tvb, pinfo, offset, s_end - offset, GTK_KEY);

      proto = (proto_keydata_t *)
        p_get_proto_data(wmem_file_scope(), pinfo, proto_wlan, DECRYPTED_GTK_KEY);
      if (proto) {
        unsigned keydata_len = proto->keydata_len;
        tvbuff_t *next_tvb = tvb_new_child_real_data(tvb, proto->keydata,
                                                    keydata_len, keydata_len);
        add_new_data_source(pinfo, next_tvb, "Decrypted GTK");
        proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_gtk_key,
                            next_tvb, 0, keydata_len, ENC_NA);
        add_ptk_analysis(tvb, subtree, &proto->used_key);
      } else {
        proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_gtk_key_encrypted,
                            tvb, offset, s_end - offset, ENC_NA);
      }
      break;
    case 3:
      save_proto_data(tvb, pinfo, offset, len, FTE_R0KH_ID_KEY);
      save_proto_data_value(pinfo, len, FTE_R0KH_ID_LEN_KEY);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_r0kh_id,
                          tvb, offset, len, ENC_NA);
      break;
    case 4:
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_igtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_igtk_ipn,
                          tvb, offset, 6, ENC_NA);
      offset += 6;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_igtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_igtk_key,
                          tvb, offset, 24, ENC_NA);
      break;
    case 5:
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_oci_op_class,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_oci_prim_chan_num,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_oci_freq_seg_1,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
      if (offset >= s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_oci_oct_op_class,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
      if (offset >= s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_oci_oct_prim_chan_num,
                          tvb, offset, 1, ENC_NA);
      offset += 1;
      if (offset >= s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_oci_oct_freq_seg_1,
                          tvb, offset, 1, ENC_NA);
      break;
    case 6:
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_bigtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_bigtk_bipn,
                          tvb, offset, 6, ENC_NA);
      offset += 6;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_bigtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_bigtk_key,
                          tvb, offset, s_end - offset, ENC_NA);
      break;
    case 8: /* MLO GTK */
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_gtk_key_info,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_gtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_gtk_link_id_info,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_gtk_link_id,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_gtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_gtk_rsc,
                          tvb, offset, 8, ENC_NA);
      offset += 8;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_gtk_key,
                          tvb, offset, s_end - offset, ENC_NA);
      break;
    case 9: /* MLO IGTK */
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_igtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_igtk_ipn,
                          tvb, offset, 6, ENC_NA);
      offset += 6;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_igtk_link_id_info,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_igtk_link_id,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_igtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_igtk_key,
                          tvb, offset, s_end - offset, ENC_NA);
      break;
    case 10: /* MLO BIGTK */
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_bigtk_key_id,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_bigtk_bipn,
                          tvb, offset, 6, ENC_NA);
      offset += 6;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_bigtk_link_id_info,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_bigtk_link_id,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_bigtk_key_length,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      if (offset > s_end)
        break;
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_mlo_bigtk_key,
                          tvb, offset, s_end - offset, ENC_NA);
      break;
    default:
      proto_tree_add_item(subtree, hf_ieee80211_tag_ft_subelem_data,
                          tvb, offset, len, ENC_NA);
      break;
    }
    offset = s_end;
  }

  return tvb_captured_length(tvb);
}

static int
dissect_mmie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  int mic_len = 8;

  if (!(tag_len == 16 || tag_len == 24)) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "MMIE content length must be 16 or 24 bytes");
    return 1;
  }

  if (tag_len == 24) {
    mic_len = 16;
  }
  proto_tree_add_item(tree, hf_ieee80211_tag_mmie_keyid, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_mmie_ipn, tvb, offset + 2, 6,
                      ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_mmie_mic, tvb, offset + 8, mic_len,
                      ENC_NA);
  return tvb_captured_length(tvb);
}

static int
ieee80211_tag_dmg_capabilities(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data);

static int
dissect_no_bssid_capability(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;

  static int * const ieee80211_tag_no_bssid_capability_dmg_bss_control[] = {
    &hf_ieee80211_tag_no_bssid_capability_dmg_bss_control_type,
    &hf_ieee80211_tag_no_bssid_capability_dmg_bss_control_reserved,
    NULL
  };

  add_ff_cap_info(tree, tvb, pinfo, offset);
  offset += 2;
  tag_len -= 2;

  /* On nontransmitted BSSID, there is only DMG Capability Info */
  if (tag_len) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_no_bssid_capability_dmg_bss_control,
                                      ett_tag_no_bssid_capability_dmg_bss_control_tree,
                                      ieee80211_tag_no_bssid_capability_dmg_bss_control,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    /* offset += 1; */

    ieee80211_tag_dmg_capabilities(tvb, pinfo, tree, data);
  }

  return tvb_captured_length(tvb);
}

static int
dissect_ssid_list(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;
  proto_tree *entry;
  bool first = true;

  while (offset + 1 <= tag_len) {
    uint8_t len = tvb_get_uint8(tvb, offset + 1);
    char *str;

    if (offset + 2 + len > tag_len)
      break;

    str = tvb_format_text(pinfo->pool, tvb, offset + 2, len);
    proto_item_append_text(tree, "%c %s", (first ? ':' : ','), str);
    first = false;
    entry = proto_tree_add_subtree_format(tree, tvb, offset, 2 + len, ett_ssid_list, NULL, "SSID: %s", str);
    proto_tree_add_item(entry, hf_ieee80211_tag_number, tvb, offset, 1,
                        ENC_LITTLE_ENDIAN);
    offset++;
    proto_tree_add_uint(entry, hf_ieee80211_tag_length, tvb, offset, 1, len);
    offset++;
    /* XXX: IEEE 802.11-2020 seems to say that these SSIDs are also affected
     * by the UTF-8 Encoding bit in the Extended Capabilities element
     * (though at least SSID List comes after Extended Capabilities).
     */
    proto_tree_add_item(entry, hf_ieee80211_tag_ssid, tvb, offset, len,
                        ENC_NA);
    offset += len;
  }

  return tvb_captured_length(tvb);
}

static int
dissect_multiple_bssid_index(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_tag_multiple_bssid_index_bssid_index, tvb, offset, 1, ENC_NA);
  offset += 1;
  tag_len -= 1;

  if (tag_len) {
    proto_tree_add_item(tree, hf_ieee80211_tag_multiple_bssid_index_dtim_period, tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(tree, hf_ieee80211_tag_multiple_bssid_index_dtim_count, tvb, offset, 1, ENC_NA);
    offset += 1;
  }

  return offset;
}

static int
dissect_link_identifier(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 18) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "Link Identifier content length must be at least "
                          "18 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_link_id_bssid, tvb,
                      offset, 6, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_link_id_init_sta, tvb,
                      offset + 6, 6, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_tag_link_id_resp_sta, tvb,
                      offset + 12, 6, ENC_NA);
  return tvb_captured_length(tvb);
}

static int
dissect_wakeup_schedule(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 18) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "Wakeup Schedule content length must be at least "
                          "18 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_offset, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_interval, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree,
                      hf_ieee80211_tag_wakeup_schedule_awake_window_slots, tvb,
                      offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_max_awake_dur,
                      tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(tree, hf_ieee80211_tag_wakeup_schedule_idle_count, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_channel_switch_timing(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 4) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                          "Channel Switch Timing content length must be at "
                          "least 4 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_channel_switch_timing_switch_time,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree,
                      hf_ieee80211_tag_channel_switch_timing_switch_timeout,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_pti_control(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;

  if (tag_len < 3) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "PTI Control content length must be at least 3 bytes");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_pti_control_tid, tvb,
                      offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_pti_control_sequence_control, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_pu_buffer_status(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static int * const ieee80211_pu_buffer_status[] = {
    &hf_ieee80211_tag_pu_buffer_status_ac_bk,
    &hf_ieee80211_tag_pu_buffer_status_ac_be,
    &hf_ieee80211_tag_pu_buffer_status_ac_vi,
    &hf_ieee80211_tag_pu_buffer_status_ac_vo,
    NULL
  };

  if (tag_len < 1) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length, "PU Buffer Status content length must be at least 1 byte");
    return tvb_captured_length(tvb);
  }

  proto_tree_add_bitmask_list(tree, tvb, offset, 1, ieee80211_pu_buffer_status, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_timeout_interval(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;
  proto_item *pi;

  pi = proto_tree_add_item(tree, hf_ieee80211_tag_timeout_int_type, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  if (tag_len < 5) {
    expert_add_info_format(pinfo, pi, &ei_ieee80211_tag_length,
                           "Timeout Interval content length must be at least "
                          "5 bytes");
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_timeout_int_value, tvb,
                      offset + 1, 4, ENC_LITTLE_ENDIAN);
  return tvb_captured_length(tvb);
}

static int
dissect_ric_data(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  proto_tree  *sub_tree;
  uint8_t      desc_cnt = 0;
  uint32_t     next_ie;
  int          offset_r = 0;
  const uint8_t ids[] = { TAG_RIC_DESCRIPTOR };

  if (tag_len != 4)  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "RIC Data Length must be 4 bytes");
    return 0;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_ric_data_id, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  desc_cnt = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_ric_data_desc_cnt, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_tag_ric_data_status_code, tvb,
                           offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* Our Design is such that all the Resource request IE's part of the RIC
   * must be in the sub tree of RIC for better readability
   * Even omnipeek does the same way.
   */
  sub_tree = proto_item_add_subtree(tree, ett_tag_ric_data_desc_ie);

  proto_item_append_text(field_data->item_tag, " :Resource Descriptor List");
  if (desc_cnt == 0) {
    proto_item_append_text(field_data->item_tag, " :0 (Weird?)");
  }

  while ( desc_cnt != 0 ) {

    next_ie = tvb_get_uint8(tvb, offset);
    proto_item_append_text(field_data->item_tag, " :(%d:%s)", desc_cnt, val_to_str_ext(pinfo->pool, next_ie, &tag_num_vals_ext, "Reserved (%d)"));
    /* Recursive call to avoid duplication of code*/
    offset_r = add_tagged_field(pinfo, sub_tree, tvb, offset, field_data->ftype, ids, G_N_ELEMENTS(ids), NULL);
    if (offset_r == 0 )/* should never happen, returns a min of 2*/
      break;
    /* This will ensure that the IE after RIC is processed
     * only once. This gives us a good looking RIC IE :-)
     */
    tag_len += offset_r;
    desc_cnt--;
  }

  return tvb_captured_length(tvb);
}

/* Overlapping BSS Scan Parameters (74) */
static int
dissect_overlap_bss_scan_par(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int offset = 0;
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int tag_len = tvb_reported_length(tvb);

  if (tag_len != 14)  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "OBSS Length must be 14 bytes");
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_spd, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_sad, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_cwtsi, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_sptpc, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_satpc, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_wctdf, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_tag_obss_sat, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  return offset;
}

/* RIC Descriptor (75) */
static int
dissect_ric_descriptor(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int offset = 0;
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int tag_len = tvb_reported_length(tvb);
  uint8_t      rsrc_type = 0;

  if (tag_len < 1)  {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "RIC Data Length must be at least 1 byte");
    return 1;
  }

  rsrc_type = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_ric_desc_rsrc_type, tvb,
                           offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  if (rsrc_type == 1) {
    /* Block ACK params
     * 802.11-2012: 8.4.2.53 RIC Descriptor element
     * Block Ack parameter set as defined in 8.4.1.14,
     * Block Ack timeout value as defined in 8.4.1.15, and
     * Block Ack starting sequence control as defined in 8.3.1.8
     */
    /* TODO: Still figuring out how to parse these ones,
     * need a sample capture with at least HEX Dump
     */
    proto_item_append_text(field_data->item_tag, " : Block ACK Params");
    proto_tree_add_item(tree, hf_ieee80211_tag_ric_desc_var_params, tvb,
                        offset, tag_len-1, ENC_NA);
    offset += tag_len -1;
  }else {
    /* 0, 2-255 are reserved*/
    proto_item_append_text(field_data->item_tag, " :Reserved (type != 1)");
  }

  return offset;
}

static int
dissect_ext_bss_load(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_mu_mimo_capable_sta_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_ss_underutilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_observable_sec_20mhz_utilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_observable_sec_40mhz_utilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_ext_bss_observable_sec_80mhz_utilization, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  return offset;
}

static int
dissect_wide_bw_channel_switch(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_wide_bw_new_channel_width, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_wide_bw_new_channel_center_freq_segment0, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_tree_add_item(tree, hf_ieee80211_wide_bw_new_channel_center_freq_segment1, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  return offset;
}

static int
dissect_channel_switch_wrapper(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int tag_len = tvb_reported_length(tvb);
  int offset = 0;
  int tmp_sublen;
  const uint8_t ids[] = { TAG_COUNTRY_INFO, TAG_WIDE_BW_CHANNEL_SWITCH,
    TAG_TX_PWR_ENVELOPE };
  const uint8_t ext_ids[] = {ETAG_BANDWIDTH_INDICATION};

  /*
  Decode three subelement in IE-196(Channel Switch Wrapper element):
        (1) New Country subelement
        (2) Wide Bandwidth Channel Switch subelement
        (3) New VHT Transmit Power Envelope subelement
        (4) Bandwidth Indication subelement
  */
  while (tag_len > 0){
    tmp_sublen = tvb_get_uint8(tvb, offset + 1);
    if (add_tagged_field_with_validation(pinfo, tree, tvb, offset, 0, ids,
        G_N_ELEMENTS(ids), false, ext_ids, G_N_ELEMENTS(ext_ids), false, NULL) == 0) {
      break;
    }
    tag_len -= (tmp_sublen + 2);
    offset += (tmp_sublen + 2);
  }
  return offset;
}

static int
dissect_operating_mode_notification(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  static int * const ieee80211_operat_mode_field[] = {
    &hf_ieee80211_operat_mode_field_channel_width,
    &hf_ieee80211_operat_mode_field_160_80plus80_bw,
    &hf_ieee80211_operat_mode_field_no_ldpc,
    &hf_ieee80211_operat_mode_field_rxnss,
    &hf_ieee80211_operat_mode_field_rxnsstype,
    NULL
  };

  /* Operating Mode field */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_operat_notification_mode,
                                    ett_mcsbit_tree, ieee80211_operat_mode_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;
  return offset;
}

static const range_string tbtt_info_length[] = {
  { 0, 0, "Reserved" },
  { 1, 1, "Neighbor AP TBTT Offset subfield" },
  { 2, 2, "Neighbor AP TBTT Offset subfield and the BSS Parameters subfield" },
  { 3, 4, "Reserved" },
  { 5, 5, "Neighbor AP TBTT Offset subfield and the Short-SSID subfield" },
  { 6, 6, "Neighbor AP TBTT offset subfield, the Short SSID subfield and the BSS Parameters subfield" },
  { 7, 7, "Neighbor AP TBTT Offset subfield and the BSSID subfield" },
  { 8, 8, "Neighbor AP TBTT Offset subfield, the BSSID subfield and the BSS Parameters subfield" },
  { 9, 9, "Neighbor AP TBTT Offset subfield, the BSSID subfield, the BSS Parameters subfield and the 20 MHz PSD subfield" },
  { 10, 10, "Reserved" },
  { 11, 11, "Neighbor AP TBTT Offset subfield, the BSSID subfield and the Short-SSID subfield" },
  { 12, 12, "Neighbor AP TBTT Offset subfield, the BSSID subfield, the Short SSID subfield and the BSS Parameters subfield" },
  { 13, 13, "Neighbor AP TBTT Offset subfield, the BSSID subfield, the Short SSID subfield, the BSS Parameters subfield and the 20 MHz PSD subfield" },
  { 14, 15, "Reserved" },
  { 16, 16, "Neighbor AP TBTT Offset subfield, the BSSID subfield, the Short SSID subfield, the BSS Parameters subfield, the 20 MHz PSD subfield and the MLD Parameters subfield" },
  { 17, 255, "First 16 octets contain Neighbor AP TBTT Offset, the BSSID, the Short SSID, the BSS Parameters, the 20 MHz PSD and the MLD Parameters subfield. The remaining octets are reserved"},
  { 0, 0, NULL }
};

static int * const bss_params_headers[] = {
  &hf_ieee80211_rnr_oct_recommended,
  &hf_ieee80211_rnr_same_ssid,
  &hf_ieee80211_rnr_multiple_bssid,
  &hf_ieee80211_rnr_transmitted_bssid,
  &hf_ieee80211_rnr_ess_with_colocated_ap,
  &hf_ieee80211_rnr_unsolicited_probe_responses,
  &hf_ieee80211_rnr_same_colocated_ap,
  &hf_ieee80211_rnr_same_reserved,
  NULL
};

static int * const mld_params_headers[] = {
  &hf_ieee80211_rnr_mld_id,
  &hf_ieee80211_rnr_mld_link_id,
  &hf_ieee80211_rnr_mld_bss_params_change_count,
  &hf_ieee80211_rnr_mld_all_updates_included,
  &hf_ieee80211_rnr_mld_disabled_link_indication,
  &hf_ieee80211_rnr_mld_reserved,
  NULL
};

static int
dissect_neighbor_ap_info(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree,
                         int offset)
{
  uint16_t tbtt_info_h;
  uint16_t tbtt_info_len;
  uint16_t tbtt_info_count;
  uint16_t len_remain = 0;
  int i = 0;
  bool display_tbtt_offset = false;
  bool display_bssid_subfield = false;
  bool display_short_bssid_subfield = false;
  bool display_bss_parameters_subfield = false;
  bool display_20mhz_psd_subfield = false;
  bool display_mld_params_subfield = false;
  bool display_reserved = false;

  proto_item *item;
  proto_tree *query;

  tbtt_info_h = tvb_get_uint16(tvb, offset, ENC_LITTLE_ENDIAN);
  tbtt_info_len = TBTT_INFO_LENGTH(tbtt_info_h);
  tbtt_info_count = TBTT_INFO_COUNT(tbtt_info_h);
  switch(tbtt_info_len) {
  case 0: case 3: case 4: case 10: case 14: case 15:
    /* Reserved cases */
    display_reserved = true;
    len_remain = tbtt_info_len;
    break;
  case 1:
    display_tbtt_offset = true;
    break;
  case 2:
    display_tbtt_offset = true;
    display_bss_parameters_subfield = true;
    break;
  case 5:
    display_tbtt_offset = true;
    display_short_bssid_subfield = true;
    break;
  case 6:
    display_tbtt_offset = true;
    display_short_bssid_subfield = true;
    display_bss_parameters_subfield = true;
    break;
  case 7:
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    break;
  case 8:
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    display_bss_parameters_subfield = true;
    break;
  case 9:
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    display_bss_parameters_subfield = true;
    display_20mhz_psd_subfield = true;
    break;
  case 11:
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    display_short_bssid_subfield = true;
    break;
  case 12:
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    display_short_bssid_subfield = true;
    display_bss_parameters_subfield = true;
    break;
  case 13:
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    display_short_bssid_subfield = true;
    display_bss_parameters_subfield = true;
    display_20mhz_psd_subfield = true;
    break;
  case 16: /* EHT stuff */
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    display_short_bssid_subfield = true;
    display_bss_parameters_subfield = true;
    display_20mhz_psd_subfield = true;
    display_mld_params_subfield = true;
    break;
  default: /* 17- 255: Same as 16 but the remaining bytes are reserved. */
    display_tbtt_offset = true;
    display_bssid_subfield = true;
    display_short_bssid_subfield = true;
    display_bss_parameters_subfield = true;
    display_20mhz_psd_subfield = true;
    display_mld_params_subfield = true;
    display_reserved = true;
    len_remain = tbtt_info_len - 16;
    break;
  }

  proto_tree_add_item(tree, hf_ieee80211_tbtt_info, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tbtt_filtered_nap, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  item = proto_tree_add_item(tree, hf_ieee80211_tbtt_info_count, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tbtt_info_length, tvb, offset, 2,
                      ENC_LITTLE_ENDIAN);
  offset+=2;

  proto_tree_add_item(tree, hf_ieee80211_tbtt_operating_class, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset+=1;
  proto_tree_add_item(tree, hf_ieee80211_tbtt_channel_number, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset+=1;

  for (i=0; i < tbtt_info_count + 1; i++) {
    query = proto_tree_add_subtree(tree, tvb, offset, tbtt_info_len,
                                  ett_tbtt_infos, &item, "TBTT Information");

    if (display_tbtt_offset) {
      proto_tree_add_item(query, hf_ieee80211_tbtt_offset, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset+=1;
    }
    if (display_bssid_subfield) {
      proto_tree_add_item(query, hf_ieee80211_tbtt_bssid, tvb, offset, 6, ENC_NA);
      offset+=6;
    }
    if (display_short_bssid_subfield) {
      proto_tree_add_item(query, hf_ieee80211_tbtt_short_ssid, tvb, offset, 4, ENC_LITTLE_ENDIAN);
      offset+=4;
    }
    if (display_bss_parameters_subfield) {

      proto_tree_add_bitmask_with_flags(query, tvb, offset,
                                        hf_ieee80211_rnr_bss_params,
                                        ett_rnr_bss_params_tree,
                                        bss_params_headers,
                                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      offset += 1;
    }
    if (display_20mhz_psd_subfield) {
      proto_tree_add_item(query, hf_ieee80211_rnr_20mhz_psd_subfield, tvb,
                          offset, 1, ENC_NA);
      offset += 1;
    }
    if (display_mld_params_subfield) {

      proto_tree_add_bitmask_with_flags(query, tvb, offset,
                                        hf_ieee80211_rnr_mld_params,
                                        ett_rnr_mld_params_tree,
                                        mld_params_headers,
                                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      offset += 3;
    }
    if (display_reserved) {
        proto_tree_add_item(query, hf_ieee80211_rnr_reserved_data, tvb, offset,
                            len_remain, ENC_NA);
    }

    offset += len_remain;
  }

  return offset;
}

static int
dissect_reduced_neighbor_report(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  while (tvb_reported_length_remaining(tvb, offset)) {
    int start_offset;
    proto_tree *neighbor_ap_info = NULL;
    proto_item *naii = NULL;

    start_offset = offset;
    neighbor_ap_info = proto_tree_add_subtree(tree, tvb, offset, -1,
                                              ett_neighbor_ap_info, &naii,
                                              "Neighbor AP Information");
    offset = dissect_neighbor_ap_info(tvb, pinfo, neighbor_ap_info, offset);

    proto_item_set_len(naii, offset - start_offset);
  }

  return tvb_reported_length(tvb);
}

static const value_string s1g_supported_channel_width_vals[] = {
  { 0, "STA supports 1MHz and 2MHz operation" },
  { 1, "STA supports 1MHz, 2MHz and 4MHz operation" },
  { 2, "STA supports 1MHz, 2MHz, 4MHz and 8MHz operation" },
  { 3, "STA supports 1MHz, 2MHz, 4MHz, 8MHz and 16MHz operation" },
  { 0, NULL }
};

static const value_string s1g_traveling_pilot_support_vals[] = {
  { 0, "Traveling Pilot Support not activated" },
  { 1, "Traveling Pilot Support activated for only one space-time stream" },
  { 2, "Reserved" },
  { 3, "Traveling Pilot Support activated for one and two space-time streams" },
  { 0, NULL }
};

static const value_string s1g_max_mpdu_length_vals[] = {
  { 0, "3895" },
  { 1, "7991" },
  { 0, NULL }
};

static const value_string s1g_min_mpdu_start_spacing_vals[] = {
  { 0, "No restriction" },
  { 1, "1/4 uS" },
  { 2, "1/2 uS" },
  { 3, "1 uS" },
  { 4, "2 uS" },
  { 5, "4 uS" },
  { 6, "8 uS" },
  { 7, "16 uS" },
  { 0, NULL }
};

static const value_string s1g_sta_type_support_vals[] = {
  { 0, "AP-Only. Supports sensor and non-sensor STAs." },
  { 1, "AP supports only sensor STAs. STA is a sensor STA." },
  { 2, "AP supports only non-sensor STAs. STA is a non-sensor STA" },
  { 3, "Reserved" },
  { 0, NULL }
};

static const value_string s1g_sectorized_beam_capable_vals[] = {
  { 0, "AP or non-AP: Not supported" },
  { 1, "AP: TXOP-based sectorization only. Non-AP: Both group and TXOP" },
  { 2, "AP: Group sectorization only. Non-AP: Reserved" },
  { 3, "AP: Both group and TXOP sectorization. Non-AP: Reserved" },
  { 0, NULL }
};

static const value_string s1g_vht_link_adaptation_vals[] = {
  { 0, "STA does not provide VHT MFB" },
  { 1, "Reserved" },
  { 2, "STA can only provide unsolicited VHT MFB" },
  { 3, "STA can provide unsolicited and solicited VHT MFB" },
  { 0, NULL }
};

#if 0
static const value_string s1g_mcs_map[] = {
  { 0, "Support for S1G-MCS 2 for n spatial streams" },
  { 1, "Support for S1G-MCS 7 for n spatial streamS" },
  { 2, "Support for S1G-MCS 9 for n spatial streams" },
  { 3, "n spatial streams not supported" },
  { 0, NULL }
};
#endif

static int * const ieee80211_s1g_cap_byte1[] = {
  &hf_ieee80211_s1g_cap_s1g_long_support,
  &hf_ieee80211_s1g_cap_short_gi_for_1_mhz,
  &hf_ieee80211_s1g_cap_short_gi_for_2_mhz,
  &hf_ieee80211_s1g_cap_short_gi_for_4_mhz,
  &hf_ieee80211_s1g_cap_short_gi_for_8_mhz,
  &hf_ieee80211_s1g_cap_short_gi_for_16_mhz,
  &hf_ieee80211_s1g_cap_supported_channel_width,
  NULL
};

static int * const ieee80211_s1g_cap_byte2[] = {
  &hf_ieee80211_s1g_cap_rx_ldpc,
  &hf_ieee80211_s1g_cap_tx_stbc,
  &hf_ieee80211_s1g_cap_rx_stbc,
  &hf_ieee80211_s1g_cap_su_beamformer_capable,
  &hf_ieee80211_s1g_cap_su_beamformee_capable,
  &hf_ieee80211_s1g_cap_beamformee_sts_capability, /* Needs global */
  NULL
};

static int * const ieee80211_s1g_cap_byte3[] = {
  &hf_ieee80211_s1g_cap_number_sounding_dimensions,
  &hf_ieee80211_s1g_cap_mu_beamformer_capable,
  &hf_ieee80211_s1g_cap_mu_beamformee_capable,
  &hf_ieee80211_s1g_cap_htc_vht_capable,
  &hf_ieee80211_s1g_cap_travelling_pilot_support,
  NULL
};

static int * const ieee80211_s1g_cap_byte4[] = {
  &hf_ieee80211_s1g_cap_rd_responder,
  &hf_ieee80211_s1g_cap_ht_delayed_block_ack,
  &hf_ieee80211_s1g_cap_maximum_mpdu_length,
  &hf_ieee80211_s1g_cap_maximum_a_mpdu_length_exp,
  &hf_ieee80211_s1g_cap_minimum_mpdu_start_spacing,
  NULL
};

static int * const ieee80211_s1g_cap_byte5[] = {
  &hf_ieee80211_s1g_cap_uplink_sync_capable,
  &hf_ieee80211_s1g_cap_dynamic_aid,
  &hf_ieee80211_s1g_cap_bat_support,
  &hf_ieee80211_s1g_cap_tim_ade_support,
  &hf_ieee80211_s1g_cap_non_tim_support,
  &hf_ieee80211_s1g_cap_group_aid_support,
  &hf_ieee80211_s1g_cap_sta_type_support,
  NULL
};

static int * const ieee80211_s1g_cap_byte6[] = {
  &hf_ieee80211_s1g_cap_centralized_authentication_control,
  &hf_ieee80211_s1g_cap_distributed_authentication_control,
  &hf_ieee80211_s1g_cap_a_msdu_support,
  &hf_ieee80211_s1g_cap_a_mpdu_support,
  &hf_ieee80211_s1g_cap_asymmetic_block_ack_support,
  &hf_ieee80211_s1g_cap_flow_control_support,
  &hf_ieee80211_s1g_cap_sectorized_beam_capable,
  NULL
};

static int * const ieee80211_s1g_cap_byte7[] = {
  &hf_ieee80211_s1g_cap_obss_mitigation_support,
  &hf_ieee80211_s1g_cap_fragment_ba_support,
  &hf_ieee80211_s1g_cap_ndp_ps_poll_supported,
  &hf_ieee80211_s1g_cap_raw_operation_support,
  &hf_ieee80211_s1g_cap_page_slicing_support,
  &hf_ieee80211_s1g_cap_txop_sharing_implicit_ack_support,
  &hf_ieee80211_s1g_cap_vht_link_adaptation_capable,
  NULL
};

static int * const ieee80211_s1g_cap_byte8[] = {
  &hf_ieee80211_s1g_cap_tack_support_as_ps_poll_response,
  &hf_ieee80211_s1g_cap_duplicate_1_mhz_support,
  &hf_ieee80211_s1g_cap_mcs_negotiation_support,
  &hf_ieee80211_s1g_cap_1_mhz_control_response_preamble_support,
  &hf_ieee80211_s1g_cap_ndp_beamforming_report_poll_support,
  &hf_ieee80211_s1g_cap_unsolicited_dynamic_aid,
  &hf_ieee80211_s1g_cap_sector_training_operation_supported,
  &hf_ieee80211_s1g_cap_temporary_ps_mode_switch,
  NULL,
};

static int * const ieee80211_s1g_cap_byte9[] = {
  &hf_ieee80211_s1g_cap_twt_grouping_support,
  &hf_ieee80211_s1g_cap_bdt_capable,
  &hf_ieee80211_s1g_cap_color,
  &hf_ieee80211_s1g_cap_twt_requester_support,
  &hf_ieee80211_s1g_cap_twt_responder_support,
  &hf_ieee80211_s1g_cap_pv1_frame_support,
  NULL
};

static int * const ieee80211_s1g_cap_byte10[] = {
  &hf_ieee80211_s1g_cap_link_adaptation_per_normal_control_response_capable,
  &hf_ieee80211_s1g_cap_reserved,
  NULL
};

static int * const ieee80211_s1g_mcs_and_nss_set[] = {
  &hf_ieee80211_s1g_rx_s1g_mcs_map,
  &hf_ieee80211_s1g_rx_highest_supported_long_gi_data_rate,
  &hf_ieee80211_s1g_tx_s1g_mcs_map,
  &hf_ieee80211_s1g_tx_highest_supported_long_gi_data_rate,
  &hf_ieee80211_s1g_rx_single_spatial_stream_map_for_1_mhz,
  &hf_ieee80211_s1g_tx_single_spatial_stream_map_for_1_mhz,
  &hf_ieee80211_s1g_mcs_and_nss_reserved,
  NULL
};

static int
dissect_s1g_capabilities(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree *s1g_cap_info = NULL;
  proto_tree *s1g_caps = NULL;
  proto_tree *sup_mcs_nss_set = NULL;

  s1g_cap_info = proto_tree_add_subtree(tree, tvb, offset, 15,
                                ett_ieee80211_s1g_capabilities_info,
                                NULL, "S1G Capabilities Information");

  s1g_caps = proto_tree_add_subtree(s1g_cap_info, tvb, offset, 10,
                                ett_ieee80211_s1g_capabilities,
                                NULL, "S1G Capabilities");

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte1,
                                    ett_s1g_cap_byte1,
                                    ieee80211_s1g_cap_byte1,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte2,
                                    ett_s1g_cap_byte2,
                                    ieee80211_s1g_cap_byte2,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte3,
                                    ett_s1g_cap_byte3,
                                    ieee80211_s1g_cap_byte3,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte4,
                                    ett_s1g_cap_byte4,
                                    ieee80211_s1g_cap_byte4,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte5,
                                    ett_s1g_cap_byte5,
                                    ieee80211_s1g_cap_byte5,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte6,
                                    ett_s1g_cap_byte6,
                                    ieee80211_s1g_cap_byte6,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte7,
                                    ett_s1g_cap_byte7,
                                    ieee80211_s1g_cap_byte7,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte8,
                                    ett_s1g_cap_byte8,
                                    ieee80211_s1g_cap_byte8,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte9,
                                    ett_s1g_cap_byte9,
                                    ieee80211_s1g_cap_byte9,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  proto_tree_add_bitmask_with_flags(s1g_caps, tvb, offset,
                                    hf_ieee80211_s1g_cap_byte10,
                                    ett_s1g_cap_byte10,
                                    ieee80211_s1g_cap_byte10,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  sup_mcs_nss_set = proto_tree_add_subtree(s1g_cap_info, tvb, offset, 5,
                                    ett_ieee80211_s1g_sup_mcs_and_nss_set,
                                    NULL, "Supported S1G-MCS and NSS Set");

  proto_tree_add_bitmask_with_flags(sup_mcs_nss_set, tvb, offset,
                                    hf_ieee80211_s1g_mcs_and_nss_set,
                                    ett_s1g_mcs_and_mcs_set,
                                    ieee80211_s1g_mcs_and_nss_set,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 5;

  return offset;
}

static int * const s1g_subchannel_selective_transmission_headers0[] = {
  &hf_ieee80211_s1g_sst_sounding_option,
  &hf_ieee80211_s1g_channel_activity_bitmap,
  &hf_ieee80211_s1g_ul_activity,
  &hf_ieee80211_s1g_dl_activity,
  &hf_ieee80211_s1g_max_trans_width,
  &hf_ieee80211_s1g_activity_start_time,
  NULL
};

static const value_string max_trans_width_vals[] = {
  { 0, "channel width unit" },
  { 1, "4MHz" },
  { 2, "8MHz" },
  { 3, "16MHz" },
  { 0, NULL }
};

static int
dissect_subchannel_selective_transmission(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  uint8_t control = tvb_get_uint8(tvb, offset);

  /* Different if sounding option is 0 or 1 */
  if ((control & 0x01) == 0x00) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                        hf_ieee80211_s1g_subchannel_selective_transmission,
                        ett_s1g_subchannel_selective_transmission,
                        s1g_subchannel_selective_transmission_headers0,
                        ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 4;
  } else {
    uint8_t control2 = tvb_get_uint8(tvb, offset + 1);
    proto_item *pi = NULL;
    proto_tree *chan_tree = NULL;

    chan_tree = proto_tree_add_subtree(tree, tvb, offset, -1,
                                       ett_s1g_subchannel_selective_transmission,
                                       &pi , "Channel Activity Schedule");
    proto_tree_add_item(chan_tree, hf_ieee80211_s1g_sst_sounding_option1, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(chan_tree, hf_ieee80211_s1g_channel_activity_bitmap1,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(chan_tree, hf_ieee80211_s1g_sounding_start_time_present,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(chan_tree, hf_ieee80211_s1g_channel_activity_reserved,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(chan_tree, hf_ieee80211_s1g_max_trans_width1, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    /* sounding_start_time_present */
    if (control2 & 0x02) {
      proto_tree_add_item(chan_tree, hf_ieee80211_s1g_sounding_start_time, tvb,
                          offset, 2, ENC_LITTLE_ENDIAN);
      offset += 2;
      proto_item_set_len(pi, 4);
    } else {
      proto_item_set_len(pi, 2);
    }
  }

  return offset;
}

static void
s1g_open_loop_link_margin_custom(char *result, uint8_t ollm_index)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%3.1f dB",
                                        (-128.0 + ollm_index * 0.5));
}

static int
dissect_s1g_open_loop_link_margin_index(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_open_loop_link_margin, tvb,
                      offset, 1, ENC_NA);
  offset += 1;

  return offset;
}

#define RAW_START_TIME_INDICATION       0x10
#define RAW_GROUP_INDICATION            0x20
#define RAW_CHANNEL_INDICATION_PRESENCE 0x40
#define RAW_PERIODIC_RAW_INDICATION     0x80

static const value_string s1g_raw_control_raw_type[] = {
  { 0, "Generic RAW" },
  { 1, "Sounding RAW" },
  { 2, "Simplex RAW" },
  { 3, "Triggering frame RAW" },
  { 0, NULL }
};

static uint8_t global_s1g_raw_type;

static void
s1g_raw_type_options_custom(char *result, uint8_t raw_type)
{
  switch (global_s1g_raw_type) {
  case 0x00:
    switch (raw_type) {
    case 0x00:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "None");
      break;
    case 0x01:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Paged STA");
      break;
    case 0x02:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "RA Frame");
      break;
    case 0x03:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Paged STA and RA Frame");
      break;
    }
    break;
  case 0x01:
    switch (raw_type) {
    case 0x00:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "SST sounding RAW");
      break;
    case 0x01:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "SST report RAW");
      break;
    case 0x02:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Sector sounding RAW");
      break;
    case 0x03:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Sector report RAW");
      break;
    }
    break;
  case 0x02:
    switch (raw_type) {
    case 0x00:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "AP PM RAW");
      break;
    case 0x01:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Non-TIM RAW");
      break;
    case 0x02:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Omni RAW");
      break;
    case 0x03:
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Reserved");
      break;
    }
    break;
  case 0x03:
    snprintf(result, ITEM_LABEL_LENGTH, "%s", "Reserved");
    break;
  }
}

static void s1g_raw_slot_duration_custom(char *result, uint16_t slot_def)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%u (%u uS)", slot_def, (500 + slot_def * 120));
}

static int * const s1g_raw_control_headers[] = {
  &hf_ieee80211_s1g_raw_type,
  &hf_ieee80211_s1g_raw_type_options,
  &hf_ieee80211_s1g_raw_start_time_indication,
  &hf_ieee80211_s1g_raw_raw_group_indication,
  &hf_ieee80211_s1g_raw_channel_indication_preference,
  &hf_ieee80211_s1g_raw_periodic_raw_indication,
  NULL
};

static int * const s1g_slot_def_8_bit[] = {
  &hf_ieee80211_s1g_slot_def_format_indication,
  &hf_ieee80211_s1g_slot_def_cross_slot_boundary,
  &hf_ieee80211_s1g_slot_def_slot_duration_count8,
  &hf_ieee80211_s1g_slot_def_num_slots6,
  NULL
};

static int * const s1g_slot_def_11_bit[] = {
  &hf_ieee80211_s1g_slot_def_format_indication,
  &hf_ieee80211_s1g_slot_def_cross_slot_boundary,
  &hf_ieee80211_s1g_slot_def_slot_duration_count11,
  &hf_ieee80211_s1g_slot_def_num_slots3,
  NULL
};

static int * const s1g_raw_group_fields[] = {
  &hf_ieee80211_s1g_raw_group_page_index,
  &hf_ieee80211_s1g_raw_group_start_aid,
  &hf_ieee80211_s1g_raw_group_end_aid,
  NULL
};

static int * const s1g_raw_channel_indication_fields[] = {
  &hf_ieee80211_s1g_raw_ci_channel_activity_bitmap,
  &hf_ieee80211_s1g_raw_ci_max_trans_width,
  &hf_ieee80211_s1g_raw_ci_ul_activity,
  &hf_ieee80211_s1g_raw_ci_dl_activity,
  &hf_ieee80211_s1g_raw_ci_reserved,
  NULL
};

static int
dissect_rps(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  proto_tree *raw_assignment_tree;
  proto_tree *assn_tree;
  proto_item *rati;
  int offset = 0;
  int idx = 0;
  uint8_t rps_len = tvb_reported_length_remaining(tvb, offset);

  raw_assignment_tree = proto_tree_add_subtree(tree, tvb, offset, rps_len,
                                    ett_s1g_raw_assignment, NULL,
                                    "RAW assignments");

  while (tvb_reported_length_remaining(tvb, offset) > 0) {

    assn_tree = proto_tree_add_subtree_format(raw_assignment_tree, tvb, offset, -1,
                                          ett_s1g_raw_assn_tree, &rati,
                                          "Assignment %d", idx);

    uint8_t raw_control = tvb_get_uint8(tvb, offset);
    uint8_t raw_slot_def = tvb_get_uint8(tvb, offset + 1);

    global_s1g_raw_type = raw_control & 0x03;

    proto_tree_add_bitmask_with_flags(assn_tree, tvb, offset,
                                      hf_ieee80211_s1g_raw_control,
                                      ett_s1g_raw_control,
                                      s1g_raw_control_headers,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 1;

    if (((raw_slot_def & 0x01) == 0x00) || (global_s1g_raw_type == 0x01)) {
      proto_tree_add_bitmask_with_flags(assn_tree, tvb, offset,
                                      hf_ieee80211_s1g_raw_slot_def,
                                      ett_s1g_raw_slot_def,
                                      s1g_slot_def_8_bit,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    } else {
      proto_tree_add_bitmask_with_flags(assn_tree, tvb, offset,
                                      hf_ieee80211_s1g_raw_slot_def,
                                      ett_s1g_raw_slot_def,
                                      s1g_slot_def_11_bit,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    }

    offset += 2;

    if (raw_control & RAW_START_TIME_INDICATION) {
      proto_tree_add_item(assn_tree, hf_ieee80211_s1g_raw_start_time, tvb, offset, 1,
                          ENC_NA);
      offset += 1;
    }

    if (raw_control & RAW_GROUP_INDICATION) {
      uint32_t raw_group = tvb_get_letoh24(tvb, offset);

      if (raw_group == 0) {
        proto_item *it = NULL;

        it = proto_tree_add_item(assn_tree, hf_ieee80211_s1g_raw_group_subfield, tvb,
                                offset, 3, ENC_LITTLE_ENDIAN);
        proto_item_append_text(it, ": All STAs allowed access within the RAW");
      } else {
        proto_tree_add_bitmask_with_flags(assn_tree, tvb, offset,
                                      hf_ieee80211_s1g_raw_group_subfield,
                                      ett_s1g_raw_group_subfield,
                                      s1g_raw_group_fields,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      }
      offset += 3;
    }

    if (raw_control & RAW_CHANNEL_INDICATION_PRESENCE) {
      proto_tree_add_bitmask_with_flags(assn_tree, tvb, offset,
                                      hf_ieee80211_s1g_raw_channel_indication,
                                      ett_s1g_raw_channel_indication,
                                      s1g_raw_channel_indication_fields,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      offset += 2;
    }

    if (raw_control & RAW_PERIODIC_RAW_INDICATION) {
      proto_tree_add_item(assn_tree, hf_ieee80211_s1g_raw_praw_periodicity, tvb,
                          offset, 1, ENC_NA);
      offset += 1;
      proto_tree_add_item(assn_tree, hf_ieee80211_s1g_raw_praw_validity, tvb,
                          offset, 1, ENC_NA);
      offset += 1;
      proto_tree_add_item(assn_tree, hf_ieee80211_s1g_raw_praw_start_offset, tvb,
                          offset, 1, ENC_NA);
      offset += 1;
    }

    idx += 1;
  }
  return offset;
}

static int * const s1g_page_slice_headers[] = {
  &hf_ieee80211_s1g_page_slice_page_index,
  &hf_ieee80211_s1g_page_slice_page_slice_length,
  &hf_ieee80211_s1g_page_slice_page_slice_count,
  &hf_ieee80211_s1g_page_slice_block_offset,
  &hf_ieee80211_s1g_page_slice_tim_offset,
  &hf_ieee80211_s1g_page_slice_reserved,
  NULL
};

static int
dissect_page_slice(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  int len = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_page_slice_page_period, tvb,
                      offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_page_slice_control,
                                    ett_s1g_page_slice_control,
                                    s1g_page_slice_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

  offset += 3;

  len = tvb_reported_length_remaining(tvb, offset);
  if (len > 0) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_page_slice_page_bitmap, tvb,
                        offset, len, ENC_NA);
    offset += len;
  }

  return offset;
}

#define AID_REQUEST_INTERVAL_PRESENT   0x01
#define PER_STA_ADDRESS_PRESENT        0x02
#define SERVICE_CHARACTERISTIC_PRESENT 0x04
#define GROUP_ADDRESS_PRESENT          0x20

static int * const s1g_aid_request_mode_headers[] = {
  &hf_ieee80211_s1g_aid_request_interval_present,
  &hf_ieee80211_s1g_aid_request_per_sta_address_present,
  &hf_ieee80211_s1g_aid_request_service_characteristic_present,
  &hf_ieee80211_s1g_aid_request_non_tim_mode_switch,
  &hf_ieee80211_s1g_aid_request_tim_mode_switch,
  &hf_ieee80211_s1g_aid_request_group_address_present,
  &hf_ieee80211_s1g_aid_request_reserved,
  NULL
};

static int * const s1g_aid_request_characteristic_headers[] = {
  &hf_ieee80211_s1g_aid_request_characteristic_sensor,
  &hf_ieee80211_s1g_aid_request_characteristic_offload,
  &hf_ieee80211_s1g_aid_request_characteristic_official_service,
  &hf_ieee80211_s1g_aid_request_characteristic_reserved,
  NULL
};

static int
dissect_aid_request(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  uint8_t mode = tvb_get_uint8(tvb, offset);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_aid_request_mode,
                                    ett_s1g_aid_request_mode,
                                    s1g_aid_request_mode_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  if (mode & AID_REQUEST_INTERVAL_PRESENT) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_aid_request_interval, tvb,
                        offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }

  if (mode & PER_STA_ADDRESS_PRESENT) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_aid_req_peer_sta_addr, tvb,
                        offset, 6, ENC_NA);
    offset += 6;
  }

  if (mode & SERVICE_CHARACTERISTIC_PRESENT) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                      hf_ieee80211_s1g_aid_request_characteristic,
                                      ett_s1g_aid_characteristic,
                                      s1g_aid_request_characteristic_headers,
                                      ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 1;
  }

  if (mode & GROUP_ADDRESS_PRESENT) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_aid_req_group_addr, tvb, offset,
                        6, ENC_NA);
    offset += 6;
  }

  return offset;
}

static int
dissect_aid_response(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_aid_rsp_aid_group_aid, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_s1g_aid_rsp_aid_switch_count,
                      tvb, offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_s1g_aid_rsp_aid_response_interval,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  return offset;
}

static int * const s1g_sector_op_control_headers[] = {
  &hf_ieee80211_s1g_sector_op_sectorization_type,
  &hf_ieee80211_s1g_sector_op_period,
  &hf_ieee80211_s1g_sector_op_omni,
  NULL
};

static int * const s1g_txop_sector_op_control_headers[] = {
  &hf_ieee80211_s1g_sector_op_sectorization_type_b16,
  &hf_ieee80211_s1g_sector_op_periodic_training_indicator,
  &hf_ieee80211_s1g_sector_op_training_period,
  &hf_ieee80211_s1g_sector_op_remaining_beacon_interval,
  &hf_ieee80211_s1g_sector_op_reserved_b16,
  NULL
};

static const true_false_string sectorization_type_tfs = {
  "Reserved",
  "Group Sectorization Operation"
};

static const true_false_string sectorization_omni_tfs = {
  "Omnidirectional",
  "Sectorized"
};

static int
dissect_s1g_sector_operation(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  int len = 0;
  uint8_t control = tvb_get_uint8(tvb, offset);

  if (control & 0x01) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_sector_op_control_16b,
                                    ett_s1g_sector_operation,
                                    s1g_txop_sector_op_control_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);

    offset += 2;
  } else {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_sector_op_control,
                                    ett_s1g_sector_operation,
                                    s1g_sector_op_control_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 1;
  }

  /* Break this out more */
  len = tvb_reported_length_remaining(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_s1g_sector_op_group_info, tvb,
                      offset, len, ENC_NA);
  offset += len;

  return offset;
}

static int
dissect_s1g_beacon_compatibility(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_beacon_compatibility_info, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_s1g_beacon_interval, tvb, offset,
                      2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_s1g_tsf_completion, tvb, offset,
                      4, ENC_LITTLE_ENDIAN);
  offset += 4;

  return offset;
}

static int
dissect_s1g_short_beacon_interval(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

   proto_tree_add_item(tree, hf_ieee80211_s1g_short_beacon_interval, tvb,
                       offset, 2, ENC_LITTLE_ENDIAN);
   offset += 2;

   return offset;
}

static int
dissect_s1g_change_sequence(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_change_sequence, tvb, offset, 1,
                      ENC_NA);
  offset += 1;

  return offset;
}

static int
dissect_authentication_control(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  uint16_t control_params = tvb_get_letohs(tvb, offset);
  proto_tree *auth_tree = NULL;

  if ((control_params & 0x0001) == 0) {
    auth_tree = proto_tree_add_subtree(tree, tvb, offset, 2,
                        ett_ieee80211_s1g_auth_control,
                        NULL,
                        "Centralized Authentication Control Parameters");
  } else {
    auth_tree = proto_tree_add_subtree(tree, tvb, offset, 3,
                        ett_ieee80211_s1g_auth_control,
                        NULL,
                        "Distributed Authentication Control Parameters");
  }

  if ((control_params & 0x0001) == 0) { /* This is all there should be here */
    proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_control_control,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_control_deferral,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_control_reserved,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);

    if ((control_params & 0x0002) == 0) { /* Deferral or not */
      proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_control_thresh,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
    } else {
      proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_control_thresh_tus,
                          tvb, offset, 2, ENC_LITTLE_ENDIAN);
    }
    offset += 2;
  } else {
    proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_control_control,
                        tvb, offset, 3, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_slot_duration, tvb,
                        offset, 3, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_max_trans_int, tvb,
                        offset, 3, ENC_LITTLE_ENDIAN);
    proto_tree_add_item(auth_tree, hf_ieee80211_s1g_auth_min_trans_int, tvb,
                        offset, 3, ENC_LITTLE_ENDIAN);
    offset += 3;
  }
  return offset;
}

static int
dissect_tsf_timer_accuracy(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_tsf_timer_accuracy, tvb, offset,
                      1, ENC_NA);
  offset += 1;

  return offset;
}

static int * const relay_control_headers[] = {
  &hf_ieee80211_s1g_relay_hierarchy_identifier,
  &hf_ieee80211_s1g_relay_no_more_relay_flag,
  NULL
};

static const range_string relay_hierarchy_rstrs[] = {
  { 0, 0,   "Root AP" },
  { 1, 1,   "S1G Relay AP" },
  { 2, 127, "Reserved" },
  { 0, 0,   NULL }
};

static const true_false_string no_more_relay_flag_tfs = {
  "AP does not accept any requests for relaying",
  "AP does accept requests for relaying"
};

static int
dissect_s1g_relay(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  uint8_t relay_control = tvb_get_uint8(tvb, offset);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_relay_control,
                                    ett_s1g_relay_control,
                                    relay_control_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;
  if ((relay_control & 0x7F) == 1) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_relay_control_rootap_bssid,
                        tvb, offset, 6, ENC_NA);
    offset += 6;
  }

  return offset;
}

static const true_false_string relay_activation_mode_tfs = {
  "Relay Activation Request",
  "Relay Activation Response"
};

static const true_false_string relay_direction_tfs = {
  "Sent by an AP",
  "Sent by a non-AP STA"
};

static unsigned relay_function_field;

static void
enable_relay_function_custom(char *result, uint8_t enable_relay_function)
{
  switch (relay_function_field & 0x03) {
  case 0x00: /* Relay Activation Mode == 0 && Direction == 0 */
    if (enable_relay_function)
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "non-AP STA activates its relay function");
    else
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "non-AP STA terminates its relay function");
    break;
  case 0x01: /* Relay Activation Mode == 1 && Direction == 0 */
    if (enable_relay_function)
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "non-AP STA requests to activate relay function");
    else
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "non-AP STA requests to terminate relay function");
    break;
  case 0x02: /* Relay Activation Mode == 0 && Direction == 1 */
    if (enable_relay_function)
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "non-AP STA is allowed to operate as a relay");
    else
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "non-AP STA must not operate as a relay");
    break;
  case 0x03: /* Relay Activation Mode == 1 && Direction == 1 */
    if (enable_relay_function)
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Non-AP STA can operate as a relay");
    else
      snprintf(result, ITEM_LABEL_LENGTH, "%s", "Non-AP STA must terminate relay function");
    break;
  }
}

static int
dissect_s1g_relay_activation(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree *relay_activ = NULL;

  relay_function_field = tvb_get_uint8(tvb, offset);

  relay_activ = proto_tree_add_subtree_format(tree, tvb, offset, 1,
                                       ett_s1g_relay_function,
                                       NULL, "Relay Activation: 0x%0x",
                                       relay_function_field);

  proto_tree_add_item(relay_activ,
                      hf_ieee80211_s1g_relay_function_activation_mode, tvb,
                      offset, 1, ENC_NA);
  proto_tree_add_item(relay_activ, hf_ieee80211_s1g_relay_function_direction,
                      tvb, offset, 1, ENC_NA);
  proto_tree_add_item(relay_activ,
                      hf_ieee80211_s1g_relay_function_enable_relay_function,
                      tvb, offset, 1, ENC_NA);
  proto_tree_add_item(relay_activ,
                      hf_ieee80211_s1g_relay_function_stas_present_indic,
                      tvb, offset, 1, ENC_NA);
  proto_tree_add_item(relay_activ, hf_ieee80211_s1g_relay_function_reserved,
                      tvb, offset, 1, ENC_NA);
  offset += 1;

  if (relay_function_field & 0x04) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_number_of_stas, tvb, offset,
                        1, ENC_NA);
    offset += 1;
  }

  return offset;
}

static const true_false_string reachable_address_add_remove_tfs = {
  "STA joining the relay",
  "STA leaving the relay"
};

static int
dissect_reachable_address(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree *reach_list = NULL;
  uint8_t addr_count = 0, addr_num = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_initiator_mac_address, tvb,
                      offset, 6, ENC_NA);
  offset += 6;

  addr_count = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_s1g_address_count, tvb, offset, 1,
                      ENC_NA);
  offset++;

  reach_list = proto_tree_add_subtree(tree, tvb, offset, 7 * addr_count,
                                      ett_ieee80211_s1g_addr_list,
                                      NULL, "Reachable Addresses");
  while (addr_count != 0) {
    proto_tree *reach_addr = NULL;

    reach_addr = proto_tree_add_subtree_format(reach_list, tvb, offset, 7,
                                        ett_ieee80211_s1g_reach_addr,
                                        NULL, "Reachable Address %u", addr_num);

    proto_tree_add_item(reach_addr, hf_ieee80211_s1g_reachable_add_remove,
                        tvb, offset, 1, ENC_NA);
    proto_tree_add_item(reach_addr, hf_ieee80211_s1g_reachable_relay_capable,
                        tvb, offset, 1, ENC_NA);
    proto_tree_add_item(reach_addr, hf_ieee80211_s1g_reachable_reserved,
                        tvb, offset, 1, ENC_NA);
    offset += 1;

    proto_tree_add_item(reach_addr, hf_ieee80211_s1g_reachable_mac_address,
                        tvb, offset, 6, ENC_NA);
    offset += 6;

    addr_num++;
    addr_count--;
  }
  return offset;
}

static int * const relay_discovery_control_headers[] = {
  &hf_ieee80211_s1g_min_data_rate_included,
  &hf_ieee80211_s1g_mean_data_rate_included,
  &hf_ieee80211_s1g_max_data_rate_included,
  &hf_ieee80211_s1g_delay_and_min_phy_rate,
  &hf_ieee80211_s1g_information_not_available,
  &hf_ieee80211_s1g_relay_discovery_reserved,
  NULL
};

static int
dissect_s1g_relay_discovery(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  uint8_t relay_discovery_control = tvb_get_uint8(tvb, offset);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_relay_discovery_control,
                                    ett_s1g_relay_discovery_control,
                                    relay_discovery_control_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  if (relay_discovery_control & 0x01) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_relay_control_ul_min, tvb,
                        offset, 1, ENC_NA);
    offset += 1;
  }

  if (relay_discovery_control & 0x02) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_relay_control_ul_mean, tvb,
                        offset, 1, ENC_NA);
    offset += 1;
  }

  if (relay_discovery_control & 0x04) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_relay_control_ul_max, tvb,
                        offset, 1, ENC_NA);
    offset += 1;
  }

  if (relay_discovery_control & 0x01) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_relay_control_dl_min, tvb,
                        offset, 1, ENC_NA);
    offset += 1;
  }

  if (relay_discovery_control & 0x02) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_relay_control_dl_mean, tvb,
                        offset, 1, ENC_NA);
    offset += 1;
  }

  if (relay_discovery_control & 0x04) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_relay_control_dl_max, tvb,
                        offset, 1, ENC_NA);
    offset += 1;
  }

  if (relay_discovery_control & 0x08) {

  }

  return offset;
}

static int
dissect_aid_announcement(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  int entry_no = 0;

  /*
   * There should be 8N bytes ...
   */
  while (tvb_reported_length_remaining(tvb, offset) >= 8) {
    proto_tree *aid_entry = NULL;

    aid_entry = proto_tree_add_subtree_format(tree, tvb, offset, 8,
                                ett_ieee80211_s1g_aid_entry,
                                NULL, "AID Entry %d", entry_no++);
    proto_tree_add_item(aid_entry, hf_ieee80211_s1g_aid_entry_mac_addr,
                        tvb, offset, 6, ENC_NA);
    offset += 6;

    proto_tree_add_item(aid_entry, hf_ieee80211_s1g_aid_entry_assoc_id,
                        tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
  }

  /* TODO: EI if bytes remaining. */

  return offset;
}

static int * const subfield_0[] = {
  &hf_ieee80211_pv1_probe_response_req_full_ssid,
  &hf_ieee80211_pv1_probe_response_req_next_tbtt,
  &hf_ieee80211_pv1_probe_response_req_access_network_option,
  &hf_ieee80211_pv1_probe_response_req_s1g_beacon_compatibility,
  &hf_ieee80211_pv1_probe_response_req_supported_rates,
  &hf_ieee80211_pv1_probe_response_req_s1g_capability,
  &hf_ieee80211_pv1_probe_response_req_s1g_operation,
  &hf_ieee80211_pv1_probe_response_req_rsn,
  NULL
};

static int
dissect_pv1_probe_response_option(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  /*
   * TODO: Check that the number of bytes matches what the probe response
   * group bitmap says should be there.
   */
  if (tvb_reported_length_remaining(tvb, offset) == 1) {
    proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_probe_resp_subfield_0,
                                    ett_s1g_probe_resp_subfield_0,
                                    subfield_0,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
    offset += 1;
  } else if (tvb_reported_length_remaining(tvb, offset) > 1) {
    uint8_t opt_bitmaps = tvb_get_uint8(tvb, offset);

    proto_tree_add_item(tree, hf_ieee80211_s1g_probe_response_group_bitmap,
                        tvb, offset, 1, ENC_NA);
    offset += 1;
    if (opt_bitmaps & 0x01) { /* Default Bitmap */
      proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_probe_resp_subfield_0,
                                    ett_s1g_probe_resp_subfield_0,
                                    subfield_0,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
      offset += 1;
    }
  }

  return offset;
}

static void
s1g_max_awake_duration_custom(char *result, uint16_t duration)
{
  if (duration == 0)
    snprintf(result, ITEM_LABEL_LENGTH, "%s", "No limit applies");
  else
    snprintf(result, ITEM_LABEL_LENGTH, "%d uS", (int)duration * 40);
}

static void
s1g_recovery_time_duration_custom(char *result, uint16_t duration)
{
  snprintf(result, ITEM_LABEL_LENGTH, "%d uS", (int)duration * 40);
}

static int
dissect_el_operation(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_el_op_max_awake_duration, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_item(tree, hf_ieee80211_s1g_el_op_recovery_time_duration, tvb,
                      offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  return offset;
}

static int
dissect_sectorized_group_id_list(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  int len = tvb_reported_length_remaining(tvb, offset);

  /* Break this out some more */
  proto_tree_add_item(tree, hf_ieee80211_s1g_sectorized_group_id_list, tvb,
                      offset, len, ENC_NA);
  offset += len;

  return offset;
}

static int * const channel_width_fields[] = {
  &hf_ieee80211_s1g_primary_channel_width,
  &hf_ieee80211_s1g_bss_operating_channel_width,
  &hf_ieee80211_s1g_primary_channel_location,
  &hf_ieee80211_s1g_reserved_b6,
  &hf_ieee80211_s1g_mcs10_use,
  NULL
};

static const value_string one_mhz_primary_channel_vals[] = {
  { 0, "1 MHz BSS operating channel width" },
  { 1, "2 MHz BSS operating channel width" },
  { 3, "4 MHz BSS operating channel width" },
  { 7, "8 MHz BSS operating channel width" },
  { 15, "16 MHz BSS operating channel width" },
  { 0, NULL },
};

static const value_string two_mhz_primary_channel_vals[] = {
  { 1, "2 MHz BSS operating channel width" },
  { 3, "4 MHz BSS operating channel width" },
  { 7, "8 MHz BSS operating channel width" },
  { 15, "16 MHz BSS operating channel width" },
  { 0, NULL },
};

static const value_string primary_channel_width_vals[] = {
  { 0, "2MHz BSS Primary Channel Width" },
  { 1, "1MHz BSS Primary Channel Width" },
  { 0, NULL }
};

static const value_string one_mhz_primary_channel_location_vals[] = {
  { 0, "Located on lower side of 2MHz primary channel" },
  { 1, "Located on upper side of 2MHz primary channel" },
  { 0, NULL }
};

static const value_string mcs10_use_vals[] = {
  { 0, "Use of MCS10 possible" },
  { 1, "Use of MCS10 not recommended" },
  { 0, NULL }
};

static int
dissect_s1g_operation(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  proto_tree *s1g_op_info = NULL;
  proto_item *cw_item = NULL;
  uint8_t chan_width = 0;

  s1g_op_info = proto_tree_add_subtree(tree, tvb, offset, 4,
                         ett_s1g_operation_info,
                         NULL, "S1G Operation Information");

  chan_width = tvb_get_uint8(tvb, offset);
  cw_item = proto_tree_add_bitmask_with_flags(s1g_op_info, tvb, offset,
                                    hf_ieee80211_s1g_channel_width,
                                    ett_s1g_channel_width,
                                    channel_width_fields,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  if (chan_width & 0x01) {
        proto_item_append_text(cw_item, ": %s",
                               val_to_str(pinfo->pool, (chan_width >> 1) & 0x0F,
                                          one_mhz_primary_channel_vals,
                                          "Invalid BSS Channel Width value"));
  } else {
        proto_item_append_text(cw_item, ": %s",
                               val_to_str(pinfo->pool, (chan_width >> 1) & 0x0F,
                                          two_mhz_primary_channel_vals,
                                          "Invalid BSS Channel Width value"));
  }
  offset += 1;

  proto_tree_add_item(s1g_op_info, hf_ieee80211_s1g_operating_class, tvb,
                      offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_item(s1g_op_info, hf_ieee80211_s1g_primary_channel_number, tvb,
                      offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_item(s1g_op_info, hf_ieee80211_s1g_channel_center_frequency,
                      tvb, offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_item(s1g_op_info, hf_ieee80211_s1g_basic_mcs_and_nss_set,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  return offset;
}

#define HEADER_COMP_STORE_A3    0x02
#define HEADER_COMP_STORE_A4    0x04
#define HEADER_COMP_CCMP_UPDATE 0x08

static int * const header_compression_control_headers[] = {
  &hf_ieee80211_s1g_header_comp_req_resp,
  &hf_ieee80211_s1g_header_comp_store_a3,
  &hf_ieee80211_s1g_header_comp_store_a4,
  &hf_ieee80211_s1g_header_comp_ccmp_update_present,
  &hf_ieee80211_s1g_header_comp_pv1_data_type_3_supported,
  &hf_ieee80211_s1g_header_comp_reserved,
  NULL
};

static int
dissect_header_compression(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;
  uint8_t control = tvb_get_uint8(tvb, offset);

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_s1g_header_comp_control,
                                    ett_s1g_header_comp_control,
                                    header_compression_control_headers,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;

  if (control & HEADER_COMP_STORE_A3) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_header_comp_a3, tvb,
                        offset, 6, ENC_NA);
    offset += 6;
  }

  if (control & HEADER_COMP_STORE_A4) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_header_comp_a4, tvb,
                        offset, 6, ENC_NA);
    offset += 6;
  }

  /* TODO: Break this out */
  if (control & HEADER_COMP_CCMP_UPDATE) {
    proto_tree_add_item(tree, hf_ieee80211_s1g_header_comp_ccmp_update, tvb,
                        offset, 5, ENC_NA);
    offset += 5;
  }

  return offset;
}

static const value_string sst_channel_unit_vals[] = {
  { 0, "Channel Width Unit is 2 MHz" },
  { 1, "Channel Width Unit is 1 MHz" },
  { 0, NULL }
};

static int
dissect_sst_operation(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_sst_enabled_channel_bitmap, tvb,
                      offset, 1, ENC_NA);
  offset += 1;

  proto_tree_add_item(tree, hf_ieee80211_s1g_sst_primary_channel_offset, tvb,
                      offset, 1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_s1g_sst_channel_unit, tvb, offset,
                      1, ENC_NA);
  proto_tree_add_item(tree, hf_ieee80211_s1g_sst_reserved, tvb, offset, 1,
                      ENC_NA);
  offset += 1;

  return offset;
}

static int
dissect_max_away_duration(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_s1g_max_away_duration, tvb, offset,
                      2, ENC_LITTLE_ENDIAN);
  offset += 2;

  return offset;
}

static int
dissect_mcs_set(proto_tree *tree, packet_info* pinfo, tvbuff_t *tvb, int offset, bool basic, bool vendorspecific)
{
  proto_item *ti;
  proto_tree *mcs_tree, *bit_tree;
  uint8_t rx_nss, tx_nss; /* 0-4 for HT and 0-8 for VHT*/
  uint32_t value_mcs_0_31, value_mcs_32_52, value_mcs_53_76;
  uint16_t tx_mcs_set;
  rx_nss = tx_nss = 8;
  /* 16 byte Supported MCS set */
  if (vendorspecific)
  {
    ti = proto_tree_add_item(tree, hf_ieee80211_mcsset_vs, tvb, offset, 16, ENC_NA);
  } else
  {
    ti = proto_tree_add_item(tree, hf_ieee80211_mcsset, tvb, offset, 16, ENC_NA);
  }
  proto_item_append_text(ti, ": %s", basic ? "Basic MCS Set" : "MCS Set");
  mcs_tree = proto_item_add_subtree(ti, ett_mcsset_tree);

  /* Rx MCS Bitmask */
  ti = proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_rx_bitmask, tvb, offset, 10, ENC_NA);
  bit_tree = proto_item_add_subtree(ti, ett_mcsbit_tree);

  /* Bits 0 - 31 */
  value_mcs_0_31 = tvb_get_letohl(tvb, offset);

  /* Handle all zeroes/ff's case..*/
  if (value_mcs_0_31 != 0x0)
  {
    if (!(value_mcs_0_31 & (0xffffff00))) {
      /*
       * At least one MCS from 0-7 is supported, but no MCS from 8-31 are
       * supported, so only 1 spatial stream is supported.
       */
      rx_nss = 0;
    } else if (!(value_mcs_0_31 & (0xffff0000))) {
      /*
       * At least one MCS from 8-15 is supported, but no MCS from 16-31 are
       * supported, so only 2 spatial streams are supported.
       */
      rx_nss = 1;
    } else if (!(value_mcs_0_31 & (0xff000000))) {
      /*
       * At least one MCS from 16-23 is supported, but no MCS from 24-31 are
       * supported, so only 3 spatial streams are supported.
       */
      rx_nss = 2;
    } else {
      /*
       * At least one MCS from 24-31 is supported, so 4 spatial streams
       * are supported.
       */
      rx_nss = 3;
    }
  }

  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_0to7, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_8to15, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_16to23, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_24to31, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  /* Should be we check UEQM Supported?*/
  /* Bits 32 - 52 */
  value_mcs_32_52 = tvb_get_letohl(tvb, offset);
  if (!(value_mcs_32_52 & (0x1ffffe))) {
    /*
     * MCS 33-52 aren't supported, so the number of spatial streams we get
     * from whichever MCSes from 0-31 that we support is the total number
     * of spatial streams we support.
     */
    ;
  } else if (!(value_mcs_32_52 & (0x1fff80))) {
    /*
     * At least one MCS from 33-38 is supported, but no MCS from 39-52 is
     * supported, so we have at least 2 spatial streams, but none of the
     * MCSs in that range give us any more.
     */
    rx_nss = MAX(1, rx_nss);
  } else {
    /*
     * At least one MCS from 39-52 is supported, so we have at least 3
     * spatial streams.
     */
    rx_nss = MAX(2, rx_nss);
  }

  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_32, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_33to38, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_39to52, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* Bits 53 - 76 */
  value_mcs_53_76 = tvb_get_letohl(tvb, offset);
  if ((value_mcs_53_76 & (0x1fffffe0))) {
    /*
     * At least one MCS from 53-76 is supported, so we have at least 4
     * spatial streams.
     */
    rx_nss = MAX(3, rx_nss);
  }

  proto_tree_add_item(bit_tree, hf_ieee80211_mcsset_rx_bitmask_53to76, tvb, offset, 4, ENC_LITTLE_ENDIAN);
  offset += 4;

  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_highest_data_rate, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  /* Follow table 8-126 from 802.11-2012 */
  tx_mcs_set = tvb_get_letohs(tvb, offset);

  if (!(tx_mcs_set & 0x0001) && !(tx_mcs_set & 0x0002))
  {
    /* TX MCS Set is not defined
     * so there is no interpretation for Max Tx Spatial Streams
     */
     tx_nss = 4; /* Not Defined*/
  }

  if ((tx_mcs_set & 0x0001) && !(tx_mcs_set & 0x0002))
  {
    /* TX MCS Set is defined to be equal to Rx MCS Set
     * So, get the Max Spatial Streams from Rx
     * MCS set
     */
     tx_nss = rx_nss;
  }
  proto_item_append_text(ti, ": %s", val_to_str(pinfo->pool, rx_nss, mcsset_tx_max_spatial_streams_flags, "Reserved:%d" ) );

  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_mcs_set_defined, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_rx_mcs_set_not_equal, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  ti = proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_max_spatial_streams, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  proto_item_append_text(ti, ", %s", val_to_str(pinfo->pool, tx_nss, mcsset_tx_max_spatial_streams_flags, "Reserved:%d" ) );
  proto_tree_add_item(mcs_tree, hf_ieee80211_mcsset_tx_unequal_modulation, tvb, offset, 1,
      ENC_LITTLE_ENDIAN);
  offset += 1;

  offset += 3;
  return offset;
}

/*  802.11n - HT Operation IE  */
static int
dissect_ht_operation_ie(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  static int * const ieee80211_ht_operation_info1_field[] = {
    &hf_ieee80211_ht_operation_info_secondary_channel_offset,
    &hf_ieee80211_ht_operation_info_sta_channel_width,
    &hf_ieee80211_ht_operation_info_rifs_mode,
    &hf_ieee80211_ht_operation_info_reserved_b4_b7,
    NULL
  };

  static int * const ieee80211_ht_operation_info2_field[] = {
    &hf_ieee80211_ht_operation_info_protection,
    &hf_ieee80211_ht_operation_info_non_greenfield_sta_present,
    &hf_ieee80211_ht_operation_info_reserved_b11,
    &hf_ieee80211_ht_operation_info_obss_non_ht_stas_present,
    &hf_ieee80211_ht_operation_info_channel_center_freq_seg_2,
    &hf_ieee80211_ht_operation_info_reserved_b21_b23,
    NULL
  };

  static int * const ieee80211_ht_operation_info3_field[] = {
    &hf_ieee80211_ht_operation_info_reserved_b24_b29,
    &hf_ieee80211_ht_operation_info_dual_beacon,
    &hf_ieee80211_ht_operation_info_dual_cts_protection,
    &hf_ieee80211_ht_operation_info_stbc_beacon,
    &hf_ieee80211_ht_operation_info_reserved_b33_b39,
    NULL
  };

  if (tag_len < 22) {
    expert_add_info_format(pinfo, field_data->item_tag_length, &ei_ieee80211_tag_length,
                           "HT Operation IE content length %u wrong, must be at least 22 bytes", tag_len);
    return 1;
  }

  proto_tree_add_item(tree, hf_ieee80211_ht_operation_primary_channel, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_operation_info_delimiter1,
                                    ett_ht_operation_info_delimiter1_tree, ieee80211_ht_operation_info1_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 1;


  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_operation_info_delimiter2,
                                    ett_ht_operation_info_delimiter2_tree, ieee80211_ht_operation_info2_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_ht_operation_info_delimiter3,
                                    ett_ht_operation_info_delimiter3_tree, ieee80211_ht_operation_info3_field,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* Basic HT-MCS Set present in Beacon, Probe Response, Mesh Peering Open and Mesh Peering
   * Confirm frames. Otherwise reserved.
   */
  if ((field_data->ftype && (field_data->ftype == MGT_BEACON || field_data->ftype == MGT_PROBE_RESP)) ||
      (field_data->sanity_check && field_data->sanity_check->ampe_frame &&
       (field_data->sanity_check->ampe_frame == SELFPROT_ACTION_MESH_PEERING_OPEN ||
        field_data->sanity_check->ampe_frame == SELFPROT_ACTION_MESH_PEERING_CONFIRM))) {
    offset = dissect_mcs_set(tree, pinfo, tvb, offset, true, false);
  } else {
    proto_tree_add_item(tree, hf_ieee80211_ht_operation_mcsset_reserved,
                        tvb, offset, 16, ENC_NA);
    offset += 16;
  }

  return offset;
}

static int
dissect_wapi_param_set(tvbuff_t *tvb, packet_info *pinfo,
                          proto_tree *tree, int offset, uint32_t tag_len, proto_item *ti_len,
                          proto_item *ti, int ftype)
{
  /* Parse the WAPI Parameter Set IE Here*/
  proto_item *item;
  proto_tree *subtree;
  uint16_t loop_cnt, version, akm_cnt  = 1, ucast_cnt = 1, bkid_cnt = 1;
  uint8_t akm_suite_type = 0, ucast_cipher_type = 0, mcast_cipher_type = 0;
  static int * const ieee80211_tag_wapi_param_set[] = {
    &hf_ieee80211_tag_wapi_param_set_capab_preauth,
    &hf_ieee80211_tag_wapi_param_set_capab_rsvd,
    NULL
  };

  proto_tree_add_item_ret_uint16(tree, hf_ieee80211_tag_wapi_param_set_version, tvb, offset, 2, ENC_LITTLE_ENDIAN, &version);
  offset += 2;

  /*MIN: 2 + (2+4)+ (2+4) + 4 + 2 + 0 (BKID CNT and LIST)  =20*/
  if (tag_len < 20) {
      expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                "tag_len is  %d, it's neither WAPI not BSS-AC-Access-Delay", tag_len);
    return offset;
  }

  if (version != 1) {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length,
                           "Version of WAPI protocol is %d, must be = 1", version);
    return offset;
  }

  /* AKM Suites: list can't be 0*/
  akm_cnt = tvb_get_letohs(tvb, offset);
  item = proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_akm_suite_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  if (akm_cnt != 0) {
    proto_item_append_text(ti, " : AKM Suite List:");
    for (loop_cnt = 0; loop_cnt < akm_cnt; loop_cnt++) {
      subtree = proto_item_add_subtree(item, ett_tag_wapi_param_set_akm_tree);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_akm_suite_oui, tvb, offset, 3, ENC_BIG_ENDIAN);
      offset += 3;
      akm_suite_type = tvb_get_uint8(tvb, offset);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_akm_suite_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_item_append_text(ti, " (%d,%s)", loop_cnt+1, val_to_str(pinfo->pool, akm_suite_type,
      ieee80211_wapi_suite_type_short, "Reserved: %d"));
    }
    proto_item_append_text(ti, " /");
  } else {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Number of AKM suites is 0, must be min 1");
    return offset;

  }
  /* Unicast Cipher Suites: list can't be 0*/
  ucast_cnt = tvb_get_letohs(tvb, offset);
  item = proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_count,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;
  if (ucast_cnt != 0) {
    proto_item_append_text(ti, " Unicast Cipher List:");
    for (loop_cnt = 0; loop_cnt < ucast_cnt; loop_cnt++) {
      subtree = proto_item_add_subtree(item, ett_tag_wapi_param_set_ucast_tree);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_oui, tvb, offset, 3, ENC_BIG_ENDIAN);
      offset += 3;
      ucast_cipher_type = tvb_get_uint8(tvb, offset);
      proto_tree_add_item(subtree, hf_ieee80211_tag_wapi_param_set_ucast_cipher_suite_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
      offset += 1;
      proto_item_append_text(ti, " (%d,%s)", loop_cnt+1, val_to_str(pinfo->pool, ucast_cipher_type, ieee80211_wapi_cipher_type, "Reserved: %d"));
    }
  proto_item_append_text(ti, " /");
  } else {
    expert_add_info_format(pinfo, ti_len, &ei_ieee80211_tag_length, "Number of Unicast Cipher suites is 0, must be min 1");
    return offset;

  }

  /* Multicast Cipher Suites*/
  proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_oui, tvb, offset, 3, ENC_BIG_ENDIAN);
  offset += 3;
  mcast_cipher_type = tvb_get_uint8(tvb, offset);
  proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_mcast_cipher_suite_type, tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset += 1;
  proto_item_append_text(ti, " Multicast Cipher: %s", val_to_str(pinfo->pool, mcast_cipher_type, ieee80211_wapi_cipher_type, "Reserved: %d"));

  /* WAPI capability */
  proto_tree_add_bitmask_with_flags(tree, tvb, offset, hf_ieee80211_tag_wapi_param_set_capab,
                                    ett_tag_wapi_param_set_preauth_tree, ieee80211_tag_wapi_param_set,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset += 2;

  /* BKID List: The list can be 0
   * Applicable only for assoc/re-assoc
   */
  if (ftype == MGT_ASSOC_REQ || ftype == MGT_REASSOC_REQ ) {
    bkid_cnt = tvb_get_letohs(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_bkid_count, tvb, offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
    if (bkid_cnt != 0) {
      for (loop_cnt = 0; loop_cnt < bkid_cnt; loop_cnt++) {
        proto_tree_add_item(tree, hf_ieee80211_tag_wapi_param_set_bkid_list, tvb, offset, 16, ENC_NA);
        offset += 16;
      }
    }
  }
  return offset;
}

static int * const ieee80211_bss_max_idle_options[] = {
  &hf_ieee80211_tag_bss_max_idle_options_protected,
  &hf_ieee80211_tag_bss_idle_options_reserved,
  NULL
};

static int
dissect_bss_max_idle_period(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, void* data _U_)
{
  int offset = 0;

  proto_tree_add_item(tree, hf_ieee80211_tag_bss_max_idle_period,
                      tvb, offset, 2, ENC_LITTLE_ENDIAN);
  offset += 2;

  proto_tree_add_bitmask_with_flags(tree, tvb, offset,
                                    hf_ieee80211_tag_bss_max_idle_options,
                                    ett_max_idle_period_options,
                                    ieee80211_bss_max_idle_options,
                                    ENC_LITTLE_ENDIAN, BMT_NO_APPEND);
  offset++;

  return offset;
}

enum tfs_request_subelem_id {
  TFS_REQ_SUBELEM_TFS = 1,
  TFS_REQ_SUBELEM_VENDOR_SPECIFIC = 221
};

static const value_string tfs_request_subelem_ids[] = {
  { TFS_REQ_SUBELEM_TFS, "TFS subelement" },
  { TFS_REQ_SUBELEM_VENDOR_SPECIFIC, "Vendor Specific subelement" },
  { 0, NULL }
};

static int
dissect_tfs_request(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  const uint8_t ids[] = {
    1, /* TFS Subelement */
    TAG_VENDOR_SPECIFIC_IE
  };

  proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_id,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;
  proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_ac_delete_after_match,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_ac_notify,
                      tvb, offset, 1, ENC_LITTLE_ENDIAN);
  offset++;
  if (offset + 1 >= tag_len) {
    expert_add_info_format(pinfo, tree, &ei_ieee80211_missing_data,
                           "No TFS Request subelements in TFS Request");
    return tvb_captured_length(tvb);
  }

  while (offset + 1 < tag_len) {
    uint8_t id, len;
    int s_offset, s_end;

    id = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_subelem_id,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    len = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_subelem_len,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (offset + len > tag_len) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length,
                             "Not enough data for TFS Request subelement");
      return tvb_captured_length(tvb);
    }
    switch (id) {
    case TFS_REQ_SUBELEM_TFS:
      s_offset = offset;
      s_end = offset + len;
      while (s_offset < s_end) {
        /* TODO 1 is interpreted as TAG_SUPP_RATES, fix this! */
        int tlen = add_tagged_field(pinfo, tree, tvb, s_offset, field_data->ftype, ids, G_N_ELEMENTS(ids), NULL);
        if (tlen==0)
          break;
        s_offset += tlen;
      }
      break;
    default:
      proto_tree_add_item(tree, hf_ieee80211_tag_tfs_request_subelem,
                          tvb, offset, len, ENC_NA);
      break;
    }
    offset += len;
  }

  if (offset < tag_len) {
    proto_tree_add_expert_format(tree, pinfo, &ei_ieee80211_extra_data,
                           tvb, offset, tag_len - offset, "Extra data after TFS Subelements");
  }

  return tvb_captured_length(tvb);
}

enum tfs_response_subelem_id {
  TFS_RESP_SUBELEM_TFS_STATUS = 1,
  TFS_RESP_SUBELEM_TFS = 2,
  TFS_RESP_SUBELEM_VENDOR_SPECIFIC = 221
};

static const value_string tfs_response_subelem_ids[] = {
  { TFS_RESP_SUBELEM_TFS_STATUS, "TFS Status subelement" },
  { TFS_RESP_SUBELEM_TFS, "TFS subelement" },
  { TFS_RESP_SUBELEM_VENDOR_SPECIFIC, "Vendor Specific subelement" },
  { 0, NULL }
};

static int
dissect_tfs_response(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
  int tag_len = tvb_reported_length(tvb);
  ieee80211_tagged_field_data_t* field_data = (ieee80211_tagged_field_data_t*)data;
  int offset = 0;
  const uint8_t ids[] = {
    1, /* TFS Status subelement*/
    2, /* TFS subelement */
    TAG_VENDOR_SPECIFIC_IE
  };

  while (offset + 3 <= tag_len) {
    uint8_t id, len;
    int s_offset, s_end;

    id = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_subelem_id,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    len = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_subelem_len,
                        tvb, offset, 1, ENC_LITTLE_ENDIAN);
    offset++;
    if (offset + len > tag_len) {
      expert_add_info_format(pinfo, tree, &ei_ieee80211_tag_length,
                             "Not enough data for TFS Request subelement");
      return tvb_captured_length(tvb);
    }
    switch (id) {
    case TFS_RESP_SUBELEM_TFS_STATUS:
      proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_status,
                          tvb, offset, 1, ENC_LITTLE_ENDIAN);
      proto_tree_add_item(tree, hf_ieee80211_tag_tfs_response_id,
                          tvb, offset + 1, 1, ENC_LITTLE_ENDIAN);
      break;
    case TFS_RESP_SUBELEM_TFS:
      s_offset = offset;
      s_end = offset + len;
      while (s_offset < s_end) {
        /* TODO Element IDs 1 and 2 are misinterpreted! */
        int tlen = add_tagged_field(pinfo, tree, tvb, s