/* packet-dis.c
 * Routines for Distributed Interactive Simulation packet
 * disassembly (IEEE-1278).
 * Copyright 2005, Scientific Research Corporation
 * Initial implementation by Jeremy Ouellette <jouellet@scires.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* TODO / NOTES:
 * Lots more PDUs to implement. Most of the basic field types are
 * complete, however, so declaring new PDUs should be fairly simple.
 *
 */

#include "config.h"

#include <wireshark.h>

#include <epan/packet.h>
#include <epan/epan.h>
#include <epan/tfs.h>
#include <epan/expert.h>

#include <wsutil/ws_padding_to.h>

#include "packet-link16.h"

#define DEFAULT_DIS_UDP_PORT 3000 /* Not IANA registered */

/* Encoding type the last 14 bits */
#define DIS_ENCODING_TYPE(word) ((word) & 0x3FFF)

// Global hash table to store previous EntityStatePDU EntityIdentifier
static wmem_map_t *packet_context_map = NULL;

// Structure to hold this EntityStatePDU entitykind and entitydomain
typedef struct {
    uint8_t entity_kind;
    uint8_t entity_domain;
} packet_context_t;
/* SISO-REF-010-2023 Version 34 draft d11 - 21 July 2024 XML generated Content Begin */
/*  	Reprinted with permission from SISO Inc.    */

/* SISO-REF-010 [UID 3] */
typedef enum
{
     DIS_VERSION_OTHER = 0,
     DIS_VERSION_DIS_PDU_VERSION_1_0_MAY_92 = 1,
     DIS_VERSION_IEEE_1278_1993 = 2,
     DIS_VERSION_DIS_APPLICATIONS_VERSION_2_0_THIRD_DRAFT_28_MAY_1993 = 3,
     DIS_VERSION_DIS_APPLICATION_PROTOCOLS_VERSION_2_0_FOURTH_DRAFT_REVISED_16_MARCH_1994 = 4,
     DIS_VERSION_IEEE_1278_1_1995 = 5,
     DIS_VERSION_IEEE_1278_1A_1998 = 6,
     DIS_VERSION_IEEE_1278_1_2012 = 7,
     DIS_VERSION_IEEE_1278_1_202X = 8
}  DIS_PDU_ProtocolVersion;

static const value_string DIS_PDU_ProtocolVersion_Strings[] = {
     { DIS_VERSION_OTHER, "Other" },
     { DIS_VERSION_DIS_PDU_VERSION_1_0_MAY_92, "DIS PDU version 1.0 (May 92)" },
     { DIS_VERSION_IEEE_1278_1993, "IEEE 1278-1993" },
     { DIS_VERSION_DIS_APPLICATIONS_VERSION_2_0_THIRD_DRAFT_28_MAY_1993, "DIS Applications Version 2.0 - Third Draft (28 May 1993)" },
     { DIS_VERSION_DIS_APPLICATION_PROTOCOLS_VERSION_2_0_FOURTH_DRAFT_REVISED_16_MARCH_1994, "DIS Application Protocols Version 2.0 - Fourth Draft (Revised) (16 March 1994)" },
     { DIS_VERSION_IEEE_1278_1_1995, "IEEE 1278.1-1995" },
     { DIS_VERSION_IEEE_1278_1A_1998, "IEEE 1278.1A-1998" },
     { DIS_VERSION_IEEE_1278_1_2012, "IEEE 1278.1-2012" },
     { DIS_VERSION_IEEE_1278_1_202X, "IEEE 1278.1-202X" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 4] */
typedef enum
{
     DIS_PDUTYPE_OTHER = 0,
     DIS_PDUTYPE_ENTITY_STATE = 1,
     DIS_PDUTYPE_FIRE = 2,
     DIS_PDUTYPE_DETONATION = 3,
     DIS_PDUTYPE_COLLISION = 4,
     DIS_PDUTYPE_SERVICE_REQUEST = 5,
     DIS_PDUTYPE_RESUPPLY_OFFER = 6,
     DIS_PDUTYPE_RESUPPLY_RECEIVED = 7,
     DIS_PDUTYPE_RESUPPLY_CANCEL = 8,
     DIS_PDUTYPE_REPAIR_COMPLETE = 9,
     DIS_PDUTYPE_REPAIR_RESPONSE = 10,
     DIS_PDUTYPE_CREATE_ENTITY = 11,
     DIS_PDUTYPE_REMOVE_ENTITY = 12,
     DIS_PDUTYPE_START_RESUME = 13,
     DIS_PDUTYPE_STOP_FREEZE = 14,
     DIS_PDUTYPE_ACKNOWLEDGE = 15,
     DIS_PDUTYPE_ACTION_REQUEST = 16,
     DIS_PDUTYPE_ACTION_RESPONSE = 17,
     DIS_PDUTYPE_DATA_QUERY = 18,
     DIS_PDUTYPE_SET_DATA = 19,
     DIS_PDUTYPE_DATA = 20,
     DIS_PDUTYPE_EVENT_REPORT = 21,
     DIS_PDUTYPE_COMMENT = 22,
     DIS_PDUTYPE_ELECTROMAGNETIC_EMISSION = 23,
     DIS_PDUTYPE_DESIGNATOR = 24,
     DIS_PDUTYPE_TRANSMITTER = 25,
     DIS_PDUTYPE_SIGNAL = 26,
     DIS_PDUTYPE_RECEIVER = 27,
     DIS_PDUTYPE_IFF = 28,
     DIS_PDUTYPE_UNDERWATER_ACOUSTIC = 29,
     DIS_PDUTYPE_SUPPLEMENTAL_EMISSION_ENTITY_STATE = 30,
     DIS_PDUTYPE_INTERCOM_SIGNAL = 31,
     DIS_PDUTYPE_INTERCOM_CONTROL = 32,
     DIS_PDUTYPE_AGGREGATE_STATE = 33,
     DIS_PDUTYPE_ISGROUPOF = 34,
     DIS_PDUTYPE_TRANSFER_OWNERSHIP = 35,
     DIS_PDUTYPE_ISPARTOF = 36,
     DIS_PDUTYPE_MINEFIELD_STATE = 37,
     DIS_PDUTYPE_MINEFIELD_QUERY = 38,
     DIS_PDUTYPE_MINEFIELD_DATA = 39,
     DIS_PDUTYPE_MINEFIELD_RESPONSE_NACK = 40,
     DIS_PDUTYPE_ENVIRONMENTAL_PROCESS = 41,
     DIS_PDUTYPE_GRIDDED_DATA = 42,
     DIS_PDUTYPE_POINT_OBJECT_STATE = 43,
     DIS_PDUTYPE_LINEAR_OBJECT_STATE = 44,
     DIS_PDUTYPE_AREAL_OBJECT_STATE = 45,
     DIS_PDUTYPE_TSPI = 46,
     DIS_PDUTYPE_APPEARANCE = 47,
     DIS_PDUTYPE_ARTICULATED_PARTS = 48,
     DIS_PDUTYPE_LE_FIRE = 49,
     DIS_PDUTYPE_LE_DETONATION = 50,
     DIS_PDUTYPE_CREATE_ENTITY_R = 51,
     DIS_PDUTYPE_REMOVE_ENTITY_R = 52,
     DIS_PDUTYPE_START_RESUME_R = 53,
     DIS_PDUTYPE_STOP_FREEZE_R = 54,
     DIS_PDUTYPE_ACKNOWLEDGE_R = 55,
     DIS_PDUTYPE_ACTION_REQUEST_R = 56,
     DIS_PDUTYPE_ACTION_RESPONSE_R = 57,
     DIS_PDUTYPE_DATA_QUERY_R = 58,
     DIS_PDUTYPE_SET_DATA_R = 59,
     DIS_PDUTYPE_DATA_R = 60,
     DIS_PDUTYPE_EVENT_REPORT_R = 61,
     DIS_PDUTYPE_COMMENT_R = 62,
     DIS_PDUTYPE_RECORD_R = 63,
     DIS_PDUTYPE_SET_RECORD_R = 64,
     DIS_PDUTYPE_RECORD_QUERY_R = 65,
     DIS_PDUTYPE_COLLISION_ELASTIC = 66,
     DIS_PDUTYPE_ENTITY_STATE_UPDATE = 67,
     DIS_PDUTYPE_DIRECTED_ENERGY_FIRE = 68,
     DIS_PDUTYPE_ENTITY_DAMAGE_STATUS = 69,
     DIS_PDUTYPE_INFORMATION_OPERATIONS_ACTION = 70,
     DIS_PDUTYPE_INFORMATION_OPERATIONS_REPORT = 71,
     DIS_PDUTYPE_ATTRIBUTE = 72,
     DIS_PDUTYPE_APPLICATION_CONTROL = 73,
     DIS_PDUTYPE_IFF_INTERACTIVE = 74,
     /* Manually inserted "Experimental" non-SISO definitions begin */
     DIS_PDUTYPE_ANNOUNCE_OBJECT                    = 129,
     DIS_PDUTYPE_DELETE_OBJECT                      = 130,
     DIS_PDUTYPE_DESCRIBE_APPLICATION               = 131,
     DIS_PDUTYPE_DESCRIBE_EVENT                     = 132,
     DIS_PDUTYPE_DESCRIBE_OBJECT                    = 133,
     DIS_PDUTYPE_REQUEST_EVENT                      = 134,
     DIS_PDUTYPE_REQUEST_OBJECT                     = 135,
     DIS_PDUTYPE_STEALTH_STATE                      = 201
    /* Manually inserted "Experimental" non-SISO definitions end */
}  DIS_PDU_Type;

static const value_string DIS_PDU_Type_Strings[] = {
     { DIS_PDUTYPE_OTHER, "Other" },
     { DIS_PDUTYPE_ENTITY_STATE, "Entity State" },
     { DIS_PDUTYPE_FIRE, "Fire" },
     { DIS_PDUTYPE_DETONATION, "Detonation" },
     { DIS_PDUTYPE_COLLISION, "Collision" },
     { DIS_PDUTYPE_SERVICE_REQUEST, "Service Request" },
     { DIS_PDUTYPE_RESUPPLY_OFFER, "Resupply Offer" },
     { DIS_PDUTYPE_RESUPPLY_RECEIVED, "Resupply Received" },
     { DIS_PDUTYPE_RESUPPLY_CANCEL, "Resupply Cancel" },
     { DIS_PDUTYPE_REPAIR_COMPLETE, "Repair Complete" },
     { DIS_PDUTYPE_REPAIR_RESPONSE, "Repair Response" },
     { DIS_PDUTYPE_CREATE_ENTITY, "Create Entity" },
     { DIS_PDUTYPE_REMOVE_ENTITY, "Remove Entity" },
     { DIS_PDUTYPE_START_RESUME, "Start/Resume" },
     { DIS_PDUTYPE_STOP_FREEZE, "Stop/Freeze" },
     { DIS_PDUTYPE_ACKNOWLEDGE, "Acknowledge" },
     { DIS_PDUTYPE_ACTION_REQUEST, "Action Request" },
     { DIS_PDUTYPE_ACTION_RESPONSE, "Action Response" },
     { DIS_PDUTYPE_DATA_QUERY, "Data Query" },
     { DIS_PDUTYPE_SET_DATA, "Set Data" },
     { DIS_PDUTYPE_DATA, "Data" },
     { DIS_PDUTYPE_EVENT_REPORT, "Event Report" },
     { DIS_PDUTYPE_COMMENT, "Comment" },
     { DIS_PDUTYPE_ELECTROMAGNETIC_EMISSION, "Electromagnetic Emission" },
     { DIS_PDUTYPE_DESIGNATOR, "Designator" },
     { DIS_PDUTYPE_TRANSMITTER, "Transmitter" },
     { DIS_PDUTYPE_SIGNAL, "Signal" },
     { DIS_PDUTYPE_RECEIVER, "Receiver" },
     { DIS_PDUTYPE_IFF, "IFF" },
     { DIS_PDUTYPE_UNDERWATER_ACOUSTIC, "Underwater Acoustic" },
     { DIS_PDUTYPE_SUPPLEMENTAL_EMISSION_ENTITY_STATE, "Supplemental Emission / Entity State" },
     { DIS_PDUTYPE_INTERCOM_SIGNAL, "Intercom Signal" },
     { DIS_PDUTYPE_INTERCOM_CONTROL, "Intercom Control" },
     { DIS_PDUTYPE_AGGREGATE_STATE, "Aggregate State" },
     { DIS_PDUTYPE_ISGROUPOF, "IsGroupOf" },
     { DIS_PDUTYPE_TRANSFER_OWNERSHIP, "Transfer Ownership" },
     { DIS_PDUTYPE_ISPARTOF, "IsPartOf" },
     { DIS_PDUTYPE_MINEFIELD_STATE, "Minefield State" },
     { DIS_PDUTYPE_MINEFIELD_QUERY, "Minefield Query" },
     { DIS_PDUTYPE_MINEFIELD_DATA, "Minefield Data" },
     { DIS_PDUTYPE_MINEFIELD_RESPONSE_NACK, "Minefield Response NACK" },
     { DIS_PDUTYPE_ENVIRONMENTAL_PROCESS, "Environmental Process" },
     { DIS_PDUTYPE_GRIDDED_DATA, "Gridded Data" },
     { DIS_PDUTYPE_POINT_OBJECT_STATE, "Point Object State" },
     { DIS_PDUTYPE_LINEAR_OBJECT_STATE, "Linear Object State" },
     { DIS_PDUTYPE_AREAL_OBJECT_STATE, "Areal Object State" },
     { DIS_PDUTYPE_TSPI, "TSPI" },
     { DIS_PDUTYPE_APPEARANCE, "Appearance" },
     { DIS_PDUTYPE_ARTICULATED_PARTS, "Articulated Parts" },
     { DIS_PDUTYPE_LE_FIRE, "LE Fire" },
     { DIS_PDUTYPE_LE_DETONATION, "LE Detonation" },
     { DIS_PDUTYPE_CREATE_ENTITY_R, "Create Entity-R" },
     { DIS_PDUTYPE_REMOVE_ENTITY_R, "Remove Entity-R" },
     { DIS_PDUTYPE_START_RESUME_R, "Start/Resume-R" },
     { DIS_PDUTYPE_STOP_FREEZE_R, "Stop/Freeze-R" },
     { DIS_PDUTYPE_ACKNOWLEDGE_R, "Acknowledge-R" },
     { DIS_PDUTYPE_ACTION_REQUEST_R, "Action Request-R" },
     { DIS_PDUTYPE_ACTION_RESPONSE_R, "Action Response-R" },
     { DIS_PDUTYPE_DATA_QUERY_R, "Data Query-R" },
     { DIS_PDUTYPE_SET_DATA_R, "Set Data-R" },
     { DIS_PDUTYPE_DATA_R, "Data-R" },
     { DIS_PDUTYPE_EVENT_REPORT_R, "Event Report-R" },
     { DIS_PDUTYPE_COMMENT_R, "Comment-R" },
     { DIS_PDUTYPE_RECORD_R, "Record-R" },
     { DIS_PDUTYPE_SET_RECORD_R, "Set Record-R" },
     { DIS_PDUTYPE_RECORD_QUERY_R, "Record Query-R" },
     { DIS_PDUTYPE_COLLISION_ELASTIC, "Collision-Elastic" },
     { DIS_PDUTYPE_ENTITY_STATE_UPDATE, "Entity State Update" },
     { DIS_PDUTYPE_DIRECTED_ENERGY_FIRE, "Directed Energy Fire" },
     { DIS_PDUTYPE_ENTITY_DAMAGE_STATUS, "Entity Damage Status" },
     { DIS_PDUTYPE_INFORMATION_OPERATIONS_ACTION, "Information Operations Action" },
     { DIS_PDUTYPE_INFORMATION_OPERATIONS_REPORT, "Information Operations Report" },
     { DIS_PDUTYPE_ATTRIBUTE, "Attribute" },
     { DIS_PDUTYPE_APPLICATION_CONTROL, "Application Control" },
     { DIS_PDUTYPE_IFF_INTERACTIVE, "IFF Interactive" },
     /* Manually inserted "Experimental" non-SISO definitions begin */
     { DIS_PDUTYPE_ANNOUNCE_OBJECT,                    "Announce Object" },
     { DIS_PDUTYPE_DELETE_OBJECT,                      "Delete Object" },
     { DIS_PDUTYPE_DESCRIBE_APPLICATION,               "Describe Application" },
     { DIS_PDUTYPE_DESCRIBE_EVENT,                     "Describe Event" },
     { DIS_PDUTYPE_DESCRIBE_OBJECT,                    "Describe Object" },
     { DIS_PDUTYPE_REQUEST_EVENT,                      "Request Event" },
     { DIS_PDUTYPE_REQUEST_OBJECT,                     "Request Object" },
     { DIS_PDUTYPE_STEALTH_STATE,                      "Stealth State" },
     /* Manually inserted "Experimental" non-SISO definitions end */
     { 0, NULL }
 };

static value_string_ext DIS_PDU_Type_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_Type_Strings);

/* SISO-REF-010 [UID 5] */
typedef enum
{
     DIS_PROTOCOLFAMILY_OTHER = 0,
     DIS_PROTOCOLFAMILY_ENTITY_INFORMATION_INTERACTION = 1,
     DIS_PROTOCOLFAMILY_WARFARE = 2,
     DIS_PROTOCOLFAMILY_LOGISTICS = 3,
     DIS_PROTOCOLFAMILY_RADIO_COMMUNICATIONS = 4,
     DIS_PROTOCOLFAMILY_SIMULATION_MANAGEMENT = 5,
     DIS_PROTOCOLFAMILY_DISTRIBUTED_EMISSION_REGENERATION = 6,
     DIS_PROTOCOLFAMILY_ENTITY_MANAGEMENT = 7,
     DIS_PROTOCOLFAMILY_MINEFIELD = 8,
     DIS_PROTOCOLFAMILY_SYNTHETIC_ENVIRONMENT = 9,
     DIS_PROTOCOLFAMILY_SIMULATION_MANAGEMENT_WITH_RELIABILITY = 10,
     DIS_PROTOCOLFAMILY_LIVE_ENTITY_LE_INFORMATION_INTERACTION = 11,
     DIS_PROTOCOLFAMILY_NON_REAL_TIME = 12,
     DIS_PROTOCOLFAMILY_INFORMATION_OPERATIONS = 13,
     /* Manually inserted "Experimental" non-SISO definitions begin */
     DIS_PROTOCOLFAMILY_EXPERIMENTAL_COMPUTER_GENERATED_FORCES = 129,
     DIS_PROTOCOLFAMILY_EXPERIMENTAL_VDIS                      = 130,
     DIS_PROTOCOLFAMILY_PERSISTENT_OBJECT                      = 140,
     DIS_PROTOCOLFAMILY_EXPERIMENTAL                           = 150
     /* Manually inserted "Experimental" non-SISO definitions end */
}  DIS_PDU_ProtocolFamily;

static const value_string DIS_PDU_ProtocolFamily_Strings[] = {
     { DIS_PROTOCOLFAMILY_OTHER, "Other" },
     { DIS_PROTOCOLFAMILY_ENTITY_INFORMATION_INTERACTION, "Entity Information/Interaction" },
     { DIS_PROTOCOLFAMILY_WARFARE, "Warfare" },
     { DIS_PROTOCOLFAMILY_LOGISTICS, "Logistics" },
     { DIS_PROTOCOLFAMILY_RADIO_COMMUNICATIONS, "Radio Communications" },
     { DIS_PROTOCOLFAMILY_SIMULATION_MANAGEMENT, "Simulation Management" },
     { DIS_PROTOCOLFAMILY_DISTRIBUTED_EMISSION_REGENERATION, "Distributed Emission Regeneration" },
     { DIS_PROTOCOLFAMILY_ENTITY_MANAGEMENT, "Entity Management" },
     { DIS_PROTOCOLFAMILY_MINEFIELD, "Minefield" },
     { DIS_PROTOCOLFAMILY_SYNTHETIC_ENVIRONMENT, "Synthetic Environment" },
     { DIS_PROTOCOLFAMILY_SIMULATION_MANAGEMENT_WITH_RELIABILITY, "Simulation Management with Reliability" },
     { DIS_PROTOCOLFAMILY_LIVE_ENTITY_LE_INFORMATION_INTERACTION, "Live Entity (LE) Information/Interaction" },
     { DIS_PROTOCOLFAMILY_NON_REAL_TIME, "Non-Real-Time" },
     { DIS_PROTOCOLFAMILY_INFORMATION_OPERATIONS, "Information Operations" },
     /* Manually inserted "Experimental" non-SISO definitions begin */
     { DIS_PROTOCOLFAMILY_EXPERIMENTAL_COMPUTER_GENERATED_FORCES, "Experimental - Computer Generated Forces" },
     { DIS_PROTOCOLFAMILY_EXPERIMENTAL_VDIS,                      "Experimental - V-DIS" },
     { DIS_PROTOCOLFAMILY_PERSISTENT_OBJECT,                      "Persistent object" },
     { DIS_PROTOCOLFAMILY_EXPERIMENTAL,                           "Experimental" },
     /* Manually inserted "Experimental" non-SISO definitions end */
     { 0, NULL }
 };

/* SISO-REF-010 [UID 6] */
typedef enum
{
     DIS_PDU_FORCE_ID_OTHER = 0,
     DIS_PDU_FORCE_ID_FRIENDLY = 1,
     DIS_PDU_FORCE_ID_OPPOSING = 2,
     DIS_PDU_FORCE_ID_NEUTRAL = 3,
     DIS_PDU_FORCE_ID_FRIENDLY_2 = 4,
     DIS_PDU_FORCE_ID_OPPOSING_2 = 5,
     DIS_PDU_FORCE_ID_NEUTRAL_2 = 6,
     DIS_PDU_FORCE_ID_FRIENDLY_3 = 7,
     DIS_PDU_FORCE_ID_OPPOSING_3 = 8,
     DIS_PDU_FORCE_ID_NEUTRAL_3 = 9,
     DIS_PDU_FORCE_ID_FRIENDLY_4 = 10,
     DIS_PDU_FORCE_ID_OPPOSING_4 = 11,
     DIS_PDU_FORCE_ID_NEUTRAL_4 = 12,
     DIS_PDU_FORCE_ID_FRIENDLY_5 = 13,
     DIS_PDU_FORCE_ID_OPPOSING_5 = 14,
     DIS_PDU_FORCE_ID_NEUTRAL_5 = 15,
     DIS_PDU_FORCE_ID_FRIENDLY_6 = 16,
     DIS_PDU_FORCE_ID_OPPOSING_6 = 17,
     DIS_PDU_FORCE_ID_NEUTRAL_6 = 18,
     DIS_PDU_FORCE_ID_FRIENDLY_7 = 19,
     DIS_PDU_FORCE_ID_OPPOSING_7 = 20,
     DIS_PDU_FORCE_ID_NEUTRAL_7 = 21,
     DIS_PDU_FORCE_ID_FRIENDLY_8 = 22,
     DIS_PDU_FORCE_ID_OPPOSING_8 = 23,
     DIS_PDU_FORCE_ID_NEUTRAL_8 = 24,
     DIS_PDU_FORCE_ID_FRIENDLY_9 = 25,
     DIS_PDU_FORCE_ID_OPPOSING_9 = 26,
     DIS_PDU_FORCE_ID_NEUTRAL_9 = 27,
     DIS_PDU_FORCE_ID_FRIENDLY_10 = 28,
     DIS_PDU_FORCE_ID_OPPOSING_10 = 29,
     DIS_PDU_FORCE_ID_NEUTRAL_10 = 30
}  DIS_PDU_Force_ID;

/* SISO-REF-010 [UID 7] */
typedef enum
{
     DIS_ENTITYKIND_OTHER = 0,
     DIS_ENTITYKIND_PLATFORM = 1,
     DIS_ENTITYKIND_MUNITION = 2,
     DIS_ENTITYKIND_LIFE_FORM = 3,
     DIS_ENTITYKIND_ENVIRONMENTAL = 4,
     DIS_ENTITYKIND_CULTURAL_FEATURE = 5,
     DIS_ENTITYKIND_SUPPLY = 6,
     DIS_ENTITYKIND_RADIO = 7,
     DIS_ENTITYKIND_EXPENDABLE = 8,
     DIS_ENTITYKIND_SENSOR_EMITTER = 9
}  DIS_PDU_EntityKind;

static const value_string DIS_PDU_EntityKind_Strings[] = {
     { DIS_ENTITYKIND_OTHER, "Other" },
     { DIS_ENTITYKIND_PLATFORM, "Platform" },
     { DIS_ENTITYKIND_MUNITION, "Munition" },
     { DIS_ENTITYKIND_LIFE_FORM, "Life form" },
     { DIS_ENTITYKIND_ENVIRONMENTAL, "Environmental" },
     { DIS_ENTITYKIND_CULTURAL_FEATURE, "Cultural feature" },
     { DIS_ENTITYKIND_SUPPLY, "Supply" },
     { DIS_ENTITYKIND_RADIO, "Radio" },
     { DIS_ENTITYKIND_EXPENDABLE, "Expendable" },
     { DIS_ENTITYKIND_SENSOR_EMITTER, "Sensor/Emitter" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 8] */
typedef enum
{
     DIS_DOMAIN_OTHER = 0,
     DIS_DOMAIN_LAND = 1,
     DIS_DOMAIN_AIR = 2,
     DIS_DOMAIN_SURFACE = 3,
     DIS_DOMAIN_SUBSURFACE = 4,
     DIS_DOMAIN_SPACE = 5
}  DIS_PDU_Domain;

static const value_string DIS_PDU_Domain_Strings[] = {
     { DIS_DOMAIN_OTHER, "Other" },
     { DIS_DOMAIN_LAND, "Land" },
     { DIS_DOMAIN_AIR, "Air" },
     { DIS_DOMAIN_SURFACE, "Surface" },
     { DIS_DOMAIN_SUBSURFACE, "Subsurface" },
     { DIS_DOMAIN_SPACE, "Space" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 9] */
typedef enum
{
     DIS_PDU_CATEGORY_LANDPLATFORM_OTHER = 0,
     DIS_PDU_CATEGORY_LANDPLATFORM_TANK = 1,
     DIS_PDU_CATEGORY_LANDPLATFORM_ARMORED_FIGHTING_VEHICLE = 2,
     DIS_PDU_CATEGORY_LANDPLATFORM_ARMORED_UTILITY_VEHICLE = 3,
     DIS_PDU_CATEGORY_LANDPLATFORM_SELF_PROPELLED_ARTILLERY = 4,
     DIS_PDU_CATEGORY_LANDPLATFORM_TOWED_ARTILLERY = 5,
     DIS_PDU_CATEGORY_LANDPLATFORM_SMALL_WHEELED_UTILITY_VEHICLE = 6,
     DIS_PDU_CATEGORY_LANDPLATFORM_LARGE_WHEELED_UTILITY_VEHICLE = 7,
     DIS_PDU_CATEGORY_LANDPLATFORM_SMALL_TRACKED_UTILITY_VEHICLE = 8,
     DIS_PDU_CATEGORY_LANDPLATFORM_LARGE_TRACKED_UTILITY_VEHICLE = 9,
     DIS_PDU_CATEGORY_LANDPLATFORM_MORTAR = 10,
     DIS_PDU_CATEGORY_LANDPLATFORM_MINE_PLOW = 11,
     DIS_PDU_CATEGORY_LANDPLATFORM_MINE_RAKE = 12,
     DIS_PDU_CATEGORY_LANDPLATFORM_MINE_ROLLER = 13,
     DIS_PDU_CATEGORY_LANDPLATFORM_CARGO_TRAILER = 14,
     DIS_PDU_CATEGORY_LANDPLATFORM_FUEL_TRAILER = 15,
     DIS_PDU_CATEGORY_LANDPLATFORM_GENERATOR_TRAILER = 16,
     DIS_PDU_CATEGORY_LANDPLATFORM_WATER_TRAILER = 17,
     DIS_PDU_CATEGORY_LANDPLATFORM_ENGINEER_EQUIPMENT = 18,
     DIS_PDU_CATEGORY_LANDPLATFORM_HEAVY_EQUIPMENT_TRANSPORT_TRAILER = 19,
     DIS_PDU_CATEGORY_LANDPLATFORM_MAINTENANCE_EQUIPMENT_TRAILER = 20,
     DIS_PDU_CATEGORY_LANDPLATFORM_LIMBER = 21,
     DIS_PDU_CATEGORY_LANDPLATFORM_CHEMICAL_DECONTAMINATION_TRAILER = 22,
     DIS_PDU_CATEGORY_LANDPLATFORM_WARNING_SYSTEM = 23,
     DIS_PDU_CATEGORY_LANDPLATFORM_TRAIN_ENGINE = 24,
     DIS_PDU_CATEGORY_LANDPLATFORM_TRAIN_CAR = 25,
     DIS_PDU_CATEGORY_LANDPLATFORM_TRAIN_CABOOSE = 26,
     DIS_PDU_CATEGORY_LANDPLATFORM_AIR_DEFENSE_MISSILE_DEFENSE_UNIT_EQUIPMENT = 28,
     DIS_PDU_CATEGORY_LANDPLATFORM_COMMAND_CONTROL_COMMUNICATIONS_AND_INTELLIGENCE_C3I_SYSTEM = 29,
     DIS_PDU_CATEGORY_LANDPLATFORM_OPERATIONS_FACILITY = 30,
     DIS_PDU_CATEGORY_LANDPLATFORM_INTELLIGENCE_FACILITY = 31,
     DIS_PDU_CATEGORY_LANDPLATFORM_SURVEILLANCE_FACILITY = 32,
     DIS_PDU_CATEGORY_LANDPLATFORM_COMMUNICATIONS_FACILITY = 33,
     DIS_PDU_CATEGORY_LANDPLATFORM_COMMAND_FACILITY = 34,
     DIS_PDU_CATEGORY_LANDPLATFORM_C4I_FACILITY = 35,
     DIS_PDU_CATEGORY_LANDPLATFORM_CONTROL_FACILITY = 36,
     DIS_PDU_CATEGORY_LANDPLATFORM_FIRE_CONTROL_FACILITY = 37,
     DIS_PDU_CATEGORY_LANDPLATFORM_MISSILE_DEFENSE_FACILITY = 38,
     DIS_PDU_CATEGORY_LANDPLATFORM_FIELD_COMMAND_POST = 39,
     DIS_PDU_CATEGORY_LANDPLATFORM_OBSERVATION_POST = 40,
     DIS_PDU_CATEGORY_LANDPLATFORM_MINE_FLAIL = 41,
     DIS_PDU_CATEGORY_LANDPLATFORM_UNMANNED = 50,
     DIS_PDU_CATEGORY_LANDPLATFORM_MOTORCYCLE = 80,
     DIS_PDU_CATEGORY_LANDPLATFORM_CAR = 81,
     DIS_PDU_CATEGORY_LANDPLATFORM_BUS = 82,
     DIS_PDU_CATEGORY_LANDPLATFORM_SINGLE_UNIT_CARGO_TRUCK = 83,
     DIS_PDU_CATEGORY_LANDPLATFORM_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK = 84,
     DIS_PDU_CATEGORY_LANDPLATFORM_MULTIPLE_UNIT_CARGO_TRUCK = 85,
     DIS_PDU_CATEGORY_LANDPLATFORM_MULTIPLE_UNIT_UTILITY_EMERGENCY_TRUCK = 86,
     DIS_PDU_CATEGORY_LANDPLATFORM_CONSTRUCTION_SPECIALTY_VEHICLE = 87,
     DIS_PDU_CATEGORY_LANDPLATFORM_FARM_SPECIALTY_VEHICLE = 88,
     DIS_PDU_CATEGORY_LANDPLATFORM_TRAILER = 89,
     DIS_PDU_CATEGORY_LANDPLATFORM_RECREATIONAL = 90,
     DIS_PDU_CATEGORY_LANDPLATFORM_NON_MOTORIZED = 91,
     DIS_PDU_CATEGORY_LANDPLATFORM_TRAINS = 92,
     DIS_PDU_CATEGORY_LANDPLATFORM_UTILITY_EMERGENCY_CAR = 93
}  DIS_PDU_Category_LandPlatform;

static const value_string DIS_PDU_Category_LandPlatform_Strings[] = {
     { DIS_PDU_CATEGORY_LANDPLATFORM_OTHER, "Other" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_TANK, "Tank" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_ARMORED_FIGHTING_VEHICLE, "Armored Fighting Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_ARMORED_UTILITY_VEHICLE, "Armored Utility Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_SELF_PROPELLED_ARTILLERY, "Self-Propelled Artillery" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_TOWED_ARTILLERY, "Towed Artillery" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_SMALL_WHEELED_UTILITY_VEHICLE, "Small Wheeled Utility Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_LARGE_WHEELED_UTILITY_VEHICLE, "Large Wheeled Utility Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_SMALL_TRACKED_UTILITY_VEHICLE, "Small Tracked Utility Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_LARGE_TRACKED_UTILITY_VEHICLE, "Large Tracked Utility Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MORTAR, "Mortar" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MINE_PLOW, "Mine Plow" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MINE_RAKE, "Mine Rake" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MINE_ROLLER, "Mine Roller" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_CARGO_TRAILER, "Cargo Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_FUEL_TRAILER, "Fuel Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_GENERATOR_TRAILER, "Generator Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_WATER_TRAILER, "Water Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_ENGINEER_EQUIPMENT, "Engineer Equipment" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_HEAVY_EQUIPMENT_TRANSPORT_TRAILER, "Heavy Equipment Transport Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MAINTENANCE_EQUIPMENT_TRAILER, "Maintenance Equipment Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_LIMBER, "Limber" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_CHEMICAL_DECONTAMINATION_TRAILER, "Chemical Decontamination Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_WARNING_SYSTEM, "Warning System" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_TRAIN_ENGINE, "Train - Engine" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_TRAIN_CAR, "Train - Car" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_TRAIN_CABOOSE, "Train - Caboose" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_AIR_DEFENSE_MISSILE_DEFENSE_UNIT_EQUIPMENT, "Air Defense / Missile Defense Unit Equipment" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_COMMAND_CONTROL_COMMUNICATIONS_AND_INTELLIGENCE_C3I_SYSTEM, "Command, Control, Communications, and Intelligence (C3I) System" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_OPERATIONS_FACILITY, "Operations Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_INTELLIGENCE_FACILITY, "Intelligence Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_SURVEILLANCE_FACILITY, "Surveillance Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_COMMUNICATIONS_FACILITY, "Communications Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_COMMAND_FACILITY, "Command Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_C4I_FACILITY, "C4I Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_CONTROL_FACILITY, "Control Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_FIRE_CONTROL_FACILITY, "Fire Control Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MISSILE_DEFENSE_FACILITY, "Missile Defense Facility" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_FIELD_COMMAND_POST, "Field Command Post" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_OBSERVATION_POST, "Observation Post" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MINE_FLAIL, "Mine Flail" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_UNMANNED, "Unmanned" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MOTORCYCLE, "Motorcycle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_CAR, "Car" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_BUS, "Bus" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_SINGLE_UNIT_CARGO_TRUCK, "Single Unit Cargo Truck" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK, "Single Unit Utility/Emergency Truck" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MULTIPLE_UNIT_CARGO_TRUCK, "Multiple Unit Cargo Truck" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_MULTIPLE_UNIT_UTILITY_EMERGENCY_TRUCK, "Multiple Unit Utility/Emergency Truck" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_CONSTRUCTION_SPECIALTY_VEHICLE, "Construction Specialty Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_FARM_SPECIALTY_VEHICLE, "Farm Specialty Vehicle" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_TRAILER, "Trailer" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_RECREATIONAL, "Recreational" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_NON_MOTORIZED, "Non-motorized" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_TRAINS, "Trains" },
     { DIS_PDU_CATEGORY_LANDPLATFORM_UTILITY_EMERGENCY_CAR, "Utility/Emergency Car" },
     { 0, NULL }
 };

static value_string_ext DIS_PDU_Category_LandPlatform_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_Category_LandPlatform_Strings);

/* SISO-REF-010 [UID 10] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CATEGORY_OTHER = 0,
     DIS_PDU_PLATFORM_AIR_CATEGORY_FIGHTER_AIR_DEFENSE = 1,
     DIS_PDU_PLATFORM_AIR_CATEGORY_ATTACK_STRIKE = 2,
     DIS_PDU_PLATFORM_AIR_CATEGORY_BOMBER = 3,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CARGO_TANKER = 4,
     DIS_PDU_PLATFORM_AIR_CATEGORY_ASW_PATROL_OBSERVATION = 5,
     DIS_PDU_PLATFORM_AIR_CATEGORY_ELECTRONIC_WARFARE_EW = 6,
     DIS_PDU_PLATFORM_AIR_CATEGORY_RECONNAISSANCE = 7,
     DIS_PDU_PLATFORM_AIR_CATEGORY_SURVEILLANCE_C2_AIRBORNE_EARLY_WARNING = 8,
     DIS_PDU_PLATFORM_AIR_CATEGORY_AIR_SEA_RESCUE_ASR = 9,
     DIS_PDU_PLATFORM_AIR_CATEGORY_ATTACK_HELICOPTER = 20,
     DIS_PDU_PLATFORM_AIR_CATEGORY_UTILITY_HELICOPTER = 21,
     DIS_PDU_PLATFORM_AIR_CATEGORY_ANTI_SUBMARINE_WARFARE_PATROL_HELICOPTER = 22,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CARGO_HELICOPTER = 23,
     DIS_PDU_PLATFORM_AIR_CATEGORY_OBSERVATION_HELICOPTER = 24,
     DIS_PDU_PLATFORM_AIR_CATEGORY_SPECIAL_OPERATIONS_HELICOPTER = 25,
     DIS_PDU_PLATFORM_AIR_CATEGORY_TRAINING_HELICOPTER = 26,
     DIS_PDU_PLATFORM_AIR_CATEGORY_TRAINER = 40,
     DIS_PDU_PLATFORM_AIR_CATEGORY_UNMANNED = 50,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_ULTRALIGHT_AIRCRAFT_NON_RIGID_WING = 80,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_ULTRALIGHT_AIRCRAFT_RIGID_WING = 81,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_GLIDER = 83,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_LIGHT_SPORT_UP_TO_1320_LBS_600_KG = 84,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_SMALL_UP_TO_12_500_LBS_5_670_KG = 85,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_MEDIUM_UP_TO_41_000_LBS_18_597_KG = 86,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_LARGE_UP_TO_255_000_LBS_115_666_KG = 87,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_HEAVY_ABOVE_255_000_LBS_115_666_KG = 88,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_HELICOPTER_SMALL_UP_TO_7_000_LBS_3_175_KG = 90,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_HELICOPTER_MEDIUM_UP_TO_20_000_LBS_9_072_KG = 91,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_HELICOPTER_LARGE_ABOVE_20_000_LBS_9_072_KG = 92,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_AUTOGYRO = 93,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_LIGHTER_THAN_AIR_BALLOON = 100,
     DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_LIGHTER_THAN_AIR_AIRSHIP = 101
}  DIS_PDU_Platform_Air_Category;

static const value_string DIS_PDU_Platform_Air_Category_Strings[] = {
     { DIS_PDU_PLATFORM_AIR_CATEGORY_OTHER, "Other" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_FIGHTER_AIR_DEFENSE, "Fighter/Air Defense" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_ATTACK_STRIKE, "Attack/Strike" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_BOMBER, "Bomber" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CARGO_TANKER, "Cargo/Tanker" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_ASW_PATROL_OBSERVATION, "ASW/Patrol/Observation" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_ELECTRONIC_WARFARE_EW, "Electronic Warfare (EW)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_RECONNAISSANCE, "Reconnaissance" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_SURVEILLANCE_C2_AIRBORNE_EARLY_WARNING, "Surveillance/C2 (Airborne Early Warning)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_AIR_SEA_RESCUE_ASR, "Air-Sea Rescue (ASR)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_ATTACK_HELICOPTER, "Attack Helicopter" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_UTILITY_HELICOPTER, "Utility Helicopter" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_ANTI_SUBMARINE_WARFARE_PATROL_HELICOPTER, "Anti-Submarine Warfare/Patrol Helicopter" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CARGO_HELICOPTER, "Cargo Helicopter" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_OBSERVATION_HELICOPTER, "Observation Helicopter" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_SPECIAL_OPERATIONS_HELICOPTER, "Special Operations Helicopter" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_TRAINING_HELICOPTER, "Training Helicopter" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_TRAINER, "Trainer" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_UNMANNED, "Unmanned" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_ULTRALIGHT_AIRCRAFT_NON_RIGID_WING, "Civilian Ultralight Aircraft, Non-rigid Wing" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_ULTRALIGHT_AIRCRAFT_RIGID_WING, "Civilian Ultralight Aircraft, Rigid Wing" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_GLIDER, "Civilian Fixed Wing Aircraft, Glider" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_LIGHT_SPORT_UP_TO_1320_LBS_600_KG, "Civilian Fixed Wing Aircraft, Light Sport (up to 1320 lbs / 600 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_SMALL_UP_TO_12_500_LBS_5_670_KG, "Civilian Fixed Wing Aircraft, Small (up to 12,500 lbs / 5,670 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_MEDIUM_UP_TO_41_000_LBS_18_597_KG, "Civilian Fixed Wing Aircraft, Medium (up to 41,000 lbs / 18,597 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_LARGE_UP_TO_255_000_LBS_115_666_KG, "Civilian Fixed Wing Aircraft, Large (up to 255,000 lbs / 115,666 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_FIXED_WING_AIRCRAFT_HEAVY_ABOVE_255_000_LBS_115_666_KG, "Civilian Fixed Wing Aircraft, Heavy (above 255,000 lbs / 115,666 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_HELICOPTER_SMALL_UP_TO_7_000_LBS_3_175_KG, "Civilian Helicopter, Small (up to 7,000 lbs / 3,175 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_HELICOPTER_MEDIUM_UP_TO_20_000_LBS_9_072_KG, "Civilian Helicopter, Medium (up to 20,000 lbs / 9,072 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_HELICOPTER_LARGE_ABOVE_20_000_LBS_9_072_KG, "Civilian Helicopter, Large (above 20,000 lbs / 9,072 kg)" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_AUTOGYRO, "Civilian Autogyro" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_LIGHTER_THAN_AIR_BALLOON, "Civilian Lighter than Air, Balloon" },
     { DIS_PDU_PLATFORM_AIR_CATEGORY_CIVILIAN_LIGHTER_THAN_AIR_AIRSHIP, "Civilian Lighter than Air, Airship" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 11] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_OTHER = 0,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_CARRIER = 1,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_COMMAND_SHIP_CRUISER = 2,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_GUIDED_MISSILE_CRUISER = 3,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_GUIDED_MISSILE_DESTROYER_DDG = 4,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_DESTROYER_DD = 5,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_GUIDED_MISSILE_FRIGATE_FFG = 6,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_LIGHT_PATROL_CRAFT = 7,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_MINE_COUNTERMEASURE_SHIP_CRAFT = 8,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_DOCK_LANDING_SHIP = 9,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_TANK_LANDING_SHIP = 10,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_LANDING_CRAFT = 11,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_LIGHT_CARRIER = 12,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_CRUISER_HELICOPTER_CARRIER = 13,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_HYDROFOIL = 14,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_AIR_CUSHION_SURFACE_EFFECT = 15,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_AUXILIARY = 16,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_AUXILIARY_MERCHANT_MARINE = 17,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_UTILITY = 18,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_UNMANNED_SURFACE_VEHICLE_USV = 19,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_LITTORAL_COMBAT_SHIPS_LCS = 20,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_SURVEILLANCE_SHIP = 21,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_FRIGATE_INCLUDING_CORVETTE = 50,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_BATTLESHIP = 51,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_HEAVY_CRUISER = 52,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_DESTROYER_TENDER = 53,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMPHIBIOUS_ASSAULT_SHIP = 54,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMPHIBIOUS_CARGO_SHIP = 55,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMPHIBIOUS_TRANSPORT_DOCK = 56,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMMUNITION_SHIP = 57,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_COMBAT_STORES_SHIP = 58,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_SURVEILLANCE_TOWED_ARRAY_SONAR_SYSTEM_SURTASS = 59,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_FAST_COMBAT_SUPPORT_SHIP = 60,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_COAST_GUARD_CUTTERS = 62,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_COAST_GUARD_BOATS = 63,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_FAST_ATTACK_CRAFT = 64,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_INFLATABLE_BOAT = 65,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_PASSENGER_VESSEL_GROUP_1_MERCHANT = 80,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_DRY_CARGO_SHIP_GROUP_2_MERCHANT = 81,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_TANKER_GROUP_3_MERCHANT = 82,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_OFFSHORE_SUPPORT_VESSEL = 83,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_PRIVATE_MOTORBOAT = 84,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_PRIVATE_SAILBOAT = 85,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_FISHING_VESSEL = 86,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_OTHER_VESSELS = 87,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_SEARCH_AND_RESCUE_VESSELS = 100,
     DIS_PDU_PLATFORM_SURFACE_CATEGORY_LIFE_SAVING_EQUIPMENT = 101
}  DIS_PDU_Platform_Surface_Category;

static const value_string DIS_PDU_Platform_Surface_Category_Strings[] = {
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_OTHER, "Other" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_CARRIER, "Carrier" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_COMMAND_SHIP_CRUISER, "Command Ship/Cruiser" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_GUIDED_MISSILE_CRUISER, "Guided Missile Cruiser" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_GUIDED_MISSILE_DESTROYER_DDG, "Guided Missile Destroyer (DDG)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_DESTROYER_DD, "Destroyer (DD)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_GUIDED_MISSILE_FRIGATE_FFG, "Guided Missile Frigate (FFG)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_LIGHT_PATROL_CRAFT, "Light/Patrol Craft" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_MINE_COUNTERMEASURE_SHIP_CRAFT, "Mine Countermeasure Ship/Craft" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_DOCK_LANDING_SHIP, "Dock Landing Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_TANK_LANDING_SHIP, "Tank Landing Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_LANDING_CRAFT, "Landing Craft" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_LIGHT_CARRIER, "Light Carrier" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_CRUISER_HELICOPTER_CARRIER, "Cruiser/Helicopter Carrier" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_HYDROFOIL, "Hydrofoil" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_AIR_CUSHION_SURFACE_EFFECT, "Air Cushion/Surface Effect" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_AUXILIARY, "Auxiliary" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_AUXILIARY_MERCHANT_MARINE, "Auxiliary, Merchant Marine" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_UTILITY, "Utility" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_UNMANNED_SURFACE_VEHICLE_USV, "Unmanned Surface Vehicle (USV)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_LITTORAL_COMBAT_SHIPS_LCS, "Littoral Combat Ships (LCS)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_SURVEILLANCE_SHIP, "Surveillance Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_FRIGATE_INCLUDING_CORVETTE, "Frigate (including Corvette)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_BATTLESHIP, "Battleship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_HEAVY_CRUISER, "Heavy Cruiser" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_DESTROYER_TENDER, "Destroyer Tender" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMPHIBIOUS_ASSAULT_SHIP, "Amphibious Assault Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMPHIBIOUS_CARGO_SHIP, "Amphibious Cargo Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMPHIBIOUS_TRANSPORT_DOCK, "Amphibious Transport Dock" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_AMMUNITION_SHIP, "Ammunition Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_COMBAT_STORES_SHIP, "Combat Stores Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_SURVEILLANCE_TOWED_ARRAY_SONAR_SYSTEM_SURTASS, "Surveillance Towed Array Sonar System (SURTASS)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_FAST_COMBAT_SUPPORT_SHIP, "Fast Combat Support Ship" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_COAST_GUARD_CUTTERS, "Coast Guard Cutters" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_COAST_GUARD_BOATS, "Coast Guard Boats" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_FAST_ATTACK_CRAFT, "Fast Attack Craft" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_INFLATABLE_BOAT, "Inflatable Boat" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_PASSENGER_VESSEL_GROUP_1_MERCHANT, "Passenger Vessel (Group 1 Merchant)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_DRY_CARGO_SHIP_GROUP_2_MERCHANT, "Dry Cargo Ship (Group 2 Merchant)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_TANKER_GROUP_3_MERCHANT, "Tanker (Group 3 Merchant)" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_OFFSHORE_SUPPORT_VESSEL, "Offshore Support Vessel" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_PRIVATE_MOTORBOAT, "Private Motorboat" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_PRIVATE_SAILBOAT, "Private Sailboat" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_FISHING_VESSEL, "Fishing Vessel" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_OTHER_VESSELS, "Other Vessels" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_SEARCH_AND_RESCUE_VESSELS, "Search and Rescue Vessels" },
     { DIS_PDU_PLATFORM_SURFACE_CATEGORY_LIFE_SAVING_EQUIPMENT, "Life-Saving Equipment" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 12] */
typedef enum
{
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_OTHER = 0,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSBN_NUCLEAR_BALLISTIC_MISSILE = 1,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSGN_NUCLEAR_GUIDED_MISSILE = 2,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSN_NUCLEAR_ATTACK_TORPEDO = 3,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSG_CONVENTIONAL_GUIDED_MISSILE = 4,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SS_CONVENTIONAL_ATTACK_TORPEDO_PATROL = 5,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSAN_NUCLEAR_AUXILIARY = 6,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSA_CONVENTIONAL_AUXILIARY = 7,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_UNMANNED_UNDERWATER_VEHICLE_UUV = 8,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSB_SUBMARINE_BALLISTIC_BALLISTIC_MISSILE_SUBMARINE = 9,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSC_COASTAL_SUBMARINE_OVER_150_TONS = 10,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSP_ATTACK_SUBMARINE_DIESEL_AIR_INDEPENDENT_PROPULSION = 11,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSM_MIDGET_SUBMARINE_UNDER_150_TONS = 12,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSNR_SPECIAL_ATTACK_SUBMARINE = 13,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SST_TRAINING_SUBMARINE = 14,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_AGSS_AUXILIARY_SUBMARINE = 15,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SEMI_SUBMERSIBLE_BOATS = 16,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_CIVILIAN_SUBMARINES = 80,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_CIVILIAN_SUBMERSIBLES = 81,
     DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_CIVILIAN_SEMI_SUBMERSIBLE_BOATS = 82
}  DIS_PDU_Platform_Subsurface_Category;

static const value_string DIS_PDU_Platform_Subsurface_Category_Strings[] = {
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_OTHER, "Other" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSBN_NUCLEAR_BALLISTIC_MISSILE, "SSBN (Nuclear Ballistic Missile)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSGN_NUCLEAR_GUIDED_MISSILE, "SSGN (Nuclear Guided Missile)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSN_NUCLEAR_ATTACK_TORPEDO, "SSN (Nuclear Attack - Torpedo)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSG_CONVENTIONAL_GUIDED_MISSILE, "SSG (Conventional Guided Missile)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SS_CONVENTIONAL_ATTACK_TORPEDO_PATROL, "SS (Conventional Attack - Torpedo, Patrol)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSAN_NUCLEAR_AUXILIARY, "SSAN (Nuclear Auxiliary)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSA_CONVENTIONAL_AUXILIARY, "SSA (Conventional Auxiliary)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_UNMANNED_UNDERWATER_VEHICLE_UUV, "Unmanned Underwater Vehicle (UUV)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSB_SUBMARINE_BALLISTIC_BALLISTIC_MISSILE_SUBMARINE, "SSB (Submarine Ballistic, Ballistic Missile Submarine)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSC_COASTAL_SUBMARINE_OVER_150_TONS, "SSC (Coastal Submarine, over 150 tons)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSP_ATTACK_SUBMARINE_DIESEL_AIR_INDEPENDENT_PROPULSION, "SSP (Attack Submarine - Diesel Air-Independent Propulsion)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSM_MIDGET_SUBMARINE_UNDER_150_TONS, "SSM (Midget Submarine, under 150 tons)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SSNR_SPECIAL_ATTACK_SUBMARINE, "SSNR (Special Attack Submarine)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SST_TRAINING_SUBMARINE, "SST (Training Submarine)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_AGSS_AUXILIARY_SUBMARINE, "AGSS (Auxiliary Submarine)" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_SEMI_SUBMERSIBLE_BOATS, "Semi-Submersible Boats" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_CIVILIAN_SUBMARINES, "Civilian Submarines" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_CIVILIAN_SUBMERSIBLES, "Civilian Submersibles" },
     { DIS_PDU_PLATFORM_SUBSURFACE_CATEGORY_CIVILIAN_SEMI_SUBMERSIBLE_BOATS, "Civilian Semi-Submersible Boats" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 13] */
typedef enum
{
     DIS_PDU_PLATFORM_SPACE_CATEGORY_OTHER = 0,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_MANNED_SPACECRAFT = 1,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_BOOSTER = 3,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_DEBRIS = 10,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_UNKNOWN_UNSPECIFIED_MISSION = 11,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_COMMUNICATION = 12,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_NAVIGATION = 13,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_SCIENCE_EXPERIMENTAL_DEMONSTRATION = 14,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_INERT_TARGET_REFLECTOR_CALIBRATION = 15,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_EARTH_OBSERVATION = 16,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_SPACE_SURVEILLANCE = 17,
     DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_ASTRONOMY = 18
}  DIS_PDU_Platform_Space_Category;

static const value_string DIS_PDU_Platform_Space_Category_Strings[] = {
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_OTHER, "Other" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_MANNED_SPACECRAFT, "Manned Spacecraft" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_BOOSTER, "Booster" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_DEBRIS, "Debris" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_UNKNOWN_UNSPECIFIED_MISSION, "Satellite - Unknown/Unspecified Mission" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_COMMUNICATION, "Satellite - Communication" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_NAVIGATION, "Satellite - Navigation" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_SCIENCE_EXPERIMENTAL_DEMONSTRATION, "Satellite - Science/Experimental/Demonstration" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_INERT_TARGET_REFLECTOR_CALIBRATION, "Satellite - Inert (Target/Reflector/Calibration)" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_EARTH_OBSERVATION, "Satellite - Earth Observation" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_SPACE_SURVEILLANCE, "Satellite - Space Surveillance" },
     { DIS_PDU_PLATFORM_SPACE_CATEGORY_SATELLITE_ASTRONOMY, "Satellite - Astronomy" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 14] */
typedef enum
{
     DIS_PDU_MUNITION_DOMAIN_OTHER = 0,
     DIS_PDU_MUNITION_DOMAIN_ANTI_AIR = 1,
     DIS_PDU_MUNITION_DOMAIN_ANTI_ARMOR = 2,
     DIS_PDU_MUNITION_DOMAIN_ANTI_GUIDED_WEAPON = 3,
     DIS_PDU_MUNITION_DOMAIN_ANTI_RADAR = 4,
     DIS_PDU_MUNITION_DOMAIN_ANTI_SATELLITE = 5,
     DIS_PDU_MUNITION_DOMAIN_ANTI_SHIP = 6,
     DIS_PDU_MUNITION_DOMAIN_ANTI_SUBMARINE = 7,
     DIS_PDU_MUNITION_DOMAIN_ANTI_PERSONNEL = 8,
     DIS_PDU_MUNITION_DOMAIN_BATTLEFIELD_SUPPORT = 9,
     DIS_PDU_MUNITION_DOMAIN_STRATEGIC = 10,
     DIS_PDU_MUNITION_DOMAIN_TACTICAL = 11,
     DIS_PDU_MUNITION_DOMAIN_DIRECTED_ENERGY_DE_WEAPON = 12
}  DIS_PDU_Munition_Domain;

/* SISO-REF-010 [UID 15] */
typedef enum
{
     DIS_PDU_MUNITION_CATEGORY_OTHER = 0,
     DIS_PDU_MUNITION_CATEGORY_GUIDED = 1,
     DIS_PDU_MUNITION_CATEGORY_BALLISTIC = 2,
     DIS_PDU_MUNITION_CATEGORY_FIXED = 3
}  DIS_PDU_Munition_Category;

/* SISO-REF-010 [UID 16] */
typedef enum
{
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_ASSAULT_MACHINE_PISTOL_KF_AMP = 1,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_AUTOMATIC_MODEL_1911A1_45 = 2,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_COMBAT_MASTER_MARK_VI_45_DETRONICS = 3,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_DE_COCKER_KP90DC_45 = 4,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_DE_COCKER_KP91DC_40 = 5,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_GENERAL_OFFICER_S_MODEL_15_45 = 6,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_NOVA_9_MM_LAFRANCE = 7,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_PERSONAL_DEFENSE_WEAPON_MP5K_PDW_9_MM = 8,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SILENCED_COLT_45_LAFRANCE = 9,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_5900_SERIES_9_MM_SMITH_WESSON_S_W = 10,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M9 = 11,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MODEL_1911A1_SPRINGFIELD_ARMORY = 12,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MODEL_2000_9_MM = 13,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_P_9_9_MM_SPRINGFIELD_ARMORY = 14,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_P_12_9_MM = 15,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_P_85_MARK_II_9_MM_RUGER = 16,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_ADVANCED_COMBAT_RIFLE_5_56_MM_AAI = 17,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_COMMANDO_ASSAULT_RIFLE_MODEL_733_5_56_MM_COLT = 18,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_INFANTRY_RIFLE_MINI_14_20_GB_5_56_MM_RUGER = 19,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MINI_14_5_56_MM_RUGER = 20,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MINI_THIRTY_7_62_MM_RUGER = 21,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SEMI_AUTOMATIC_MODEL_82A2_50_BARRETT = 22,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SNIPER_WEAPON_SYSTEM_M24_7_62_MM = 23,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SNIPING_RIFLE_M21_SPRINGFIELD_ARMORY = 24,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SNIPING_RIFLE_M40A1_7_62_MM = 25,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SNIPING_RIFLE_M600_7_62_MM = 26,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_AR_15_M16_5_56_MM = 27,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M1_30 = 28,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M14_7_62_MM_NATO = 29,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M14_M1A_M1A1_A1_SPRINGFIELD_ARMORY = 30,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M14K_ASSAULT_RIFLE_LAFRANCE = 31,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M16A2_ASSAULT_RIFLE_5_56_MM_COLT = 32,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M21_7_62_MM_U_S = 33,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M77_MARK_II_5_56_MM_RUGER = 34,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M77V_7_62_MM_RUGER = 35,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_S_16_7_62_X_36_MM_GRENDEL = 36,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SAR_8_7_62_MM = 37,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SAR_4800_7_62_MM = 38,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_ASSAULT_CARBINE_M16K_LAFRANCE = 39,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M4_MODEL_720_5_56_MM_COLT = 41,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M_900_9_MM_CALICO = 42,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_AC_556F_5_56_MM_RUGER = 43,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M3_45 = 44,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M11_COBRAY = 45,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M951_9_MM_CALICO = 46,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MP5_10_10_MM = 47,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_9_MM_COLT = 48,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_INGRAM = 49,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_EXTERNALLY_POWERED_EPG_7_62_MM_ARES = 50,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_GECAL_50 = 51,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_GENERAL_PURPOSE_M60_7_62_MM = 52,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_HEAVY_M2HB_QCB_50_RAMO = 53,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LIGHT_ASSAULT_M60E3_ENHANCED_7_62_MM = 54,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LIGHT_M16A2_5_56_MM_COLT = 55,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LIGHT_5_56_MM_ARES = 56,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LIGHTWEIGHT_M2_50_RAMO = 57,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LIGHTWEIGHT_ASSAULT_M60E3_7_62_MM = 58,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MINIGUN_M134_7_62_MM_GENERAL_ELECTRIC = 59,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MG_SYSTEM_MK19_MOD_3_40_MM = 60,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MG_SYSTEM_OR_KIT_M2HB_QCB_50_SACO_DEFENSE = 61,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M1919A4_30_CAL_BROWNING = 62,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_50_CAL_BROWNING = 63,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_COLORED_SMOKE_HAND_GRENADE_M18 = 64,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_COLORED_SMOKE_GRENADES_FEDERAL_LABORATORIES = 65,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_INFRARED_SMOKE_GRENADE_M76 = 66,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SMOKE_HAND_GRENADE_AN_M8_HC = 67,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_DELAY_FRAGMENTATION_HAND_GRENADE_M61 = 68,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_DELAY_FRAGMENTATION_HAND_GRENADE_M67 = 69,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_IMPACT_FRAGMENTATION_HAND_GRENADE_M57 = 70,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_IMPACT_FRAGMENTATION_HAND_GRENADE_M68 = 71,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_INCENDIARY_HAND_GRENADE_AN_M14_TH3 = 72,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LAUNCHER_I_M203_40_MM = 73,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LAUNCHER_M79_40_MM = 74,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MULTIPLE_GRENADE_LAUNCHER_MM_1_40_MM = 75,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MULTI_SHOT_PORTABLE_FLAME_WEAPON_M202A2_66_MM = 76,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_PORTABLE_ABC_M9_7 = 77,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_PORTABLE_M2A1_7 = 78,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_PORTABLE_M9E1_7 = 79,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_DRAGON_MEDIUM_ANTI_ARMOR_MISSILE_M47_FGM_77A = 80,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_JAVELIN_AAWS_M = 81,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LIGHT_ANTI_TANK_WEAPON_M72_LAW_II = 82,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_REDEYE_FIM_43_GENERAL_DYNAMICS = 83,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SABER_DUAL_PURPOSE_MISSILE_SYSTEM = 84,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_STINGER_FIM_92_GENERAL_DYNAMICS = 85,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_TOW_HEAVY_ANTI_TANK_WEAPON = 86,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_BEAR_TRAP_AP_DEVICE_PANCOR = 87,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_CHAIN_GUN_AUTOMATIC_WEAPON_EX_34_7_62_MM = 88,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_CLOSE_ASSAULT_WEAPON_SYSTEM_CAWS_AAI = 89,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_CAWS_OLIN_HECKLER_AND_KOCH = 90,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_CROSSFIRE_SAM_MODEL_88 = 91,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_DRAGON_AND_M16 = 92,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_FIRING_PORT_WEAPON_M231_5_56_MM_COLT = 93,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_FOXHOLE_DIGGER_EXPLOSIVE_KIT_EXFODA = 94,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_INFANTRY_SUPPORT_WEAPON_ASP_30_RM_30_MM = 95,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_JACKHAMMER_MK_3_A2_PANCOR = 96,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_LIGHT_ANTI_ARMOR_WEAPON_M136_AT4 = 97,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M26A2 = 98,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MASTER_KEY_S = 99,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MINIGUN_5_56_MM = 100,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MULTIPURPOSE_INDIVIDUAL_MUNITION_MPIM_MARQUARDT = 101,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MULTIPURPOSE_WEAPON_AT8 = 102,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_RECOILLESS_RIFLE_M40_M40A2_AND_M40A4_106_MM = 103,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_RECOILLESS_RIFLE_M67_90_MM = 104,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_REVOLVER_SP_101 = 105,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_REVOLVER_SUPER_REDHAWK_44_MAGNUM_RUGER = 106,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_RAW_ROCKET_140_MM_BRUNSWICK = 107,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_RIFLE_LAUNCHER_ANTI_ARMOR_MUNITION_RAAM_OLIN = 108,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_ROCKET_LAUNCHER_M_20_3_5_IN = 109,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_ROCKET_LAUNCHER_ENHANCED_M72_E_SERIES_HEAT_66_MM = 110,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SELECTIVE_FIRE_WEAPON_AC_556_5_56_MM_RUGER = 111,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SELECTIVE_FIRE_WEAPON_AC_556F_5_56_MM_RUGER = 112,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SHOTGUN_M870_MK_1_U_S_MARINE_CORPS_REMINGTON = 113,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SMAW_MK_193_83_MM_MCDONNELL_DOUGLAS = 114,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SMAW_D_DISPOSABLE_SMAW = 115,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_SQUAD_AUTOMATIC_WEAPON_SAW_M249_5_56_MM = 116,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_TACTICAL_SUPPORT_WEAPON_50_12_50_CAL_PEREGRINE = 117,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_TELESCOPED_AMMUNITION_REVOLVER_GUN_TARG_50_CAL_ARES = 118,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_ULTIMATE_OVER_UNDER_COMBINATION_CIENER = 119,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_M18A1_CLAYMORE_MINE = 120,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MORTAR_81_MM = 121,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_S_WEAPONS_MACHINEGUN_M240_7_62MM = 134
}  DIS_PDU_Life_Forms_Subcategory_U_S_Weapons;

/* SISO-REF-010 [UID 17] */
typedef enum
{
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_AUTOMATIC_APS_9_MM_STECHKIN = 201,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_PSM_5_45_MM = 202,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SELF_LOADING_PM_9_MM_MAKAROV = 203,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_TT_33_7_62_MM_TOKAREV = 204,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_ASSAULT_RIFLE_AK_AND_AKM_7_62_MM = 205,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_ASSAULT_RIFLE_AK_74_AND_AKS_74_5_45_MM = 206,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SELF_LOADING_RIFLE_SKS_7_62_MM_SIMONOV = 207,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SNIPER_RIFLE_SVD_7_62_MM_DRAGUNOV = 208,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_AKSU_74_5_45_MM = 209,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_PPS_43_7_62_MM = 210,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_PPSH_41_7_62_MM = 211,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_GENERAL_PURPOSE_PK_7_62_MM = 212,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_HEAVY_DSHK_38_AND_MODEL_38_46_12_7_MM_DEGTYAREV = 213,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_HEAVY_NSV_12_7_MM = 214,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_LIGHT_RPD_7_62_MM = 215,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_LIGHT_RPK_7_62_MM = 216,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_LIGHT_RPK_74_5_45_MM = 217,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_HAND_GRENADE_M75 = 218,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_HAND_GRENADE_RGD_5 = 219,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_AP_HAND_GRENADE_F1 = 220,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_AT_HAND_GRENADE_RKG_3 = 221,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_AT_HAND_GRENADE_RKG_3M = 222,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_AT_HAND_GRENADE_RKG_3T = 223,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_FRAGMENTATION_HAND_GRENADE_RGN = 224,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_FRAGMENTATION_HAND_GRENADE_RGO = 225,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SMOKE_HAND_GRENADE_RDG_1 = 226,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_PLAMYA_LAUNCHER_30_MM_AGS_17 = 227,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_RIFLE_MOUNTED_LAUNCHER_BG_15_40_MM = 228,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_LPO_50 = 229,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_ROKS_3 = 230,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_CART_MOUNTED_TPO_50 = 231,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_GIMLET_SA_16 = 232,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_GRAIL_SA_7 = 233,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_GREMLIN_SA_14 = 234,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SAGGER_AT_3_MCLOS = 235,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SAXHORN_AT_7 = 236,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SPIGOT_A_B_AT_14 = 237,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SA_18 = 238,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SA_19 = 239,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_GRAD_1P_MANPORTABLE_TRIPOD_ROCKET_LAUNCHER_122_MM_FOR_SPESNATZ_AND_OTHER_SPECIALISTS_AKA_9P132 = 240,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_LIGHT_ANTI_ARMOR_WEAPON_RPG_18 = 241,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_LIGHT_ANTI_TANK_WEAPON_RPG_22 = 242,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_MG_RPG = 243,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_PORTABLE_ROCKET_LAUNCHER_RPG_16 = 244,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_RECOILLESS_GUN_73_MM_SPG_9 = 245,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_VAT_ROCKET_LAUNCHER_RPG_7 = 246,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_MON_50_ANTI_PERSONNEL_MINE = 248,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_RPG_29_VAMPIR = 249,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_LASER_DESIGNATOR = 250,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_AT_4_SPIGOT = 251,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_SA_24_IGLA_S = 252,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_C_I_S_WEAPONS_TYPE_69_RPG = 253
}  DIS_PDU_Life_Forms_Subcategory_C_I_S_Weapons;

/* SISO-REF-010 [UID 18] */
typedef enum
{
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_LAW_80 = 1,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_BLOWPIPE = 2,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_JAVELIN = 3,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_51_MM_MORTAR = 4,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_SLR_7_62_MM_RIFLE = 5,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_STERLING_9_MM_SUBMACHINE_GUN = 6,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_L7A2_GENERAL_PURPOSE_MG = 7,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_L6_WOMBAT_RECOILLESS_RIFLE_ = 8,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_CARL_GUSTAV_89_MM_RECOILLESS_RIFLE = 9,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_SA80_INDIVIDUAL_LIGHT_SUPPORT_WEAPON = 10,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_TRIGAT = 11,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_U_K_WEAPONS_MILAN_AT_MISSILE = 12
}  DIS_PDU_Life_Forms_Subcategory_U_K_Weapons;

/* SISO-REF-010 [UID 19] */
typedef enum
{
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_ACL_STRIM = 1,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_MISTRAL_MISSILE = 2,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_MILAN_AT_MISSILE = 3,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_LRAC_F1_89_MM_AT_ROCKET_LAUNCHER = 4,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_FA_MAS_RIFLE = 5,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_AA_52_MACHINE_GUN = 6,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_58_MM_RIFLE_GRENADE = 7,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_FRENCH_WEAPONS_FR_F1_SNIPER_RIFLE = 8
}  DIS_PDU_Life_Forms_Subcategory_French_Weapons;

/* SISO-REF-010 [UID 20] */
typedef enum
{
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_G3_RIFLE = 1,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_G11_RIFLE = 2,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_P1_PISTOL = 3,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_MG3_MACHINE_GUN = 4,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_MILAN_MISSILE = 5,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_MP1_UZI_SUBMACHINE_GUN = 6,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_PANZERFAUST_3_LIGHT_ANTI_TANK_WEAPON = 7,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_DM19_HAND_GRENADE = 8,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_GERMAN_WEAPONS_DM29_HAND_GRENADE = 9
}  DIS_PDU_Life_Forms_Subcategory_German_Weapons;

/* SISO-REF-010 [UID 21] */
typedef enum
{
     DIS_PDU_ENVIRONMENTAL_SUBCATEGORY_OTHER = 0,
     DIS_PDU_ENVIRONMENTAL_SUBCATEGORY_VERY_SMALL = 20,
     DIS_PDU_ENVIRONMENTAL_SUBCATEGORY_SMALL = 40,
     DIS_PDU_ENVIRONMENTAL_SUBCATEGORY_MEDIUM = 60,
     DIS_PDU_ENVIRONMENTAL_SUBCATEGORY_LARGE = 80,
     DIS_PDU_ENVIRONMENTAL_SUBCATEGORY_VERY_LARGE = 100
}  DIS_PDU_Environmental_Subcategory;

/* SISO-REF-010 [UID 22] */
typedef enum
{
     DIS_PDU_RADIO_CATEGORY_OTHER = 0,
     DIS_PDU_RADIO_CATEGORY_VOICE_TRANSMISSION_RECEPTION = 1,
     DIS_PDU_RADIO_CATEGORY_DATA_LINK_TRANSMISSION_RECEPTION = 2,
     DIS_PDU_RADIO_CATEGORY_VOICE_AND_DATA_LINK_TRANSMISSION_RECEPTION = 3,
     DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_GLIDESLOPE_TRANSMITTER = 4,
     DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_LOCALIZER_TRANSMITTER = 5,
     DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_OUTER_MARKER_BEACON = 6,
     DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_MIDDLE_MARKER_BEACON = 7,
     DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_INNER_MARKER_BEACON = 8,
     DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_RECEIVER_PLATFORM_RADIO = 9,
     DIS_PDU_RADIO_CATEGORY_TACTICAL_AIR_NAVIGATION_TACAN_TRANSMITTER_GROUND_FIXED_EQUIPMENT = 10,
     DIS_PDU_RADIO_CATEGORY_TACTICAL_AIR_NAVIGATION_TACAN_RECEIVER_MOVING_PLATFORM_EQUIPMENT = 11,
     DIS_PDU_RADIO_CATEGORY_TACTICAL_AIR_NAVIGATION_TACAN_TRANSMITTER_RECEIVER_MOVING_PLATFORM_EQUIPMENT = 12,
     DIS_PDU_RADIO_CATEGORY_VARIABLE_OMNI_RANGING_VOR_TRANSMITTER_GROUND_FIXED_EQUIPMENT = 13,
     DIS_PDU_RADIO_CATEGORY_VARIABLE_OMNI_RANGING_VOR_WITH_DISTANCE_MEASURING_EQUIPMENT_DME_TRANSMITTER_GROUND_FIXED_EQUIPMENT = 14,
     DIS_PDU_RADIO_CATEGORY_COMBINED_VOR_ILS_RECEIVER_MOVING_PLATFORM_EQUIPMENT = 15,
     DIS_PDU_RADIO_CATEGORY_COMBINED_VOR_TACAN_VORTAC_TRANSMITTER = 16,
     DIS_PDU_RADIO_CATEGORY_NON_DIRECTIONAL_BEACON_NDB_TRANSMITTER = 17,
     DIS_PDU_RADIO_CATEGORY_NON_DIRECTIONAL_BEACON_NDB_RECEIVER = 18,
     DIS_PDU_RADIO_CATEGORY_NON_DIRECTIONAL_BEACON_NDB_WITH_DISTANCE_MEASURING_EQUIPMENT_DME_TRANSMITTER = 19,
     DIS_PDU_RADIO_CATEGORY_DISTANCE_MEASURING_EQUIPMENT_DME = 20,
     DIS_PDU_RADIO_CATEGORY_LINK_16_TERMINAL = 21,
     DIS_PDU_RADIO_CATEGORY_LINK_11_TERMINAL = 22,
     DIS_PDU_RADIO_CATEGORY_LINK_11B_TERMINAL = 23,
     DIS_PDU_RADIO_CATEGORY_EPLRS_SADL_TERMINAL = 24,
     DIS_PDU_RADIO_CATEGORY_F_22_INTRA_FLIGHT_DATA_LINK_IFDL = 25,
     DIS_PDU_RADIO_CATEGORY_F_35_MULTIFUNCTION_ADVANCED_DATA_LINK_MADL = 26,
     DIS_PDU_RADIO_CATEGORY_SINCGARS_TERMINAL = 27,
     DIS_PDU_RADIO_CATEGORY_L_BAND_SATCOM_TERMINAL = 28,
     DIS_PDU_RADIO_CATEGORY_IBS_TERMINAL = 29,
     DIS_PDU_RADIO_CATEGORY_GPS = 30,
     DIS_PDU_RADIO_CATEGORY_TACTICAL_VIDEO = 31,
     DIS_PDU_RADIO_CATEGORY_AIR_TO_AIR_MISSILE_DATALINK = 32,
     DIS_PDU_RADIO_CATEGORY_LINK_16_SURROGATE_FOR_NON_NATO_TDL_TERMINAL = 33,
     DIS_PDU_RADIO_CATEGORY_MQ_1_9_C_BAND_LOS_DATALINK = 34,
     DIS_PDU_RADIO_CATEGORY_MQ_1_9_KU_BAND_SATCOM_DATALINK = 35,
     DIS_PDU_RADIO_CATEGORY_AIR_TO_GROUND_WEAPON_DATALINK = 36,
     DIS_PDU_RADIO_CATEGORY_AUTOMATIC_IDENTIFICATION_SYSTEM_AIS = 37,
     DIS_PDU_RADIO_CATEGORY_JPALS_DATA_LINK = 38,
     DIS_PDU_RADIO_CATEGORY_COMBAT_SEARCH_AND_RESCUE_CSAR_RADIO = 40,
     DIS_PDU_RADIO_CATEGORY_COUNTER_UNMANNED_AIRCRAFT_SYSTEM_C_UAS_RADIO = 41,
     DIS_PDU_RADIO_CATEGORY_EMERGENCY_POSITION_INDICATING_RADIO_BEACONS_EPIRB = 42,
     DIS_PDU_RADIO_CATEGORY_ELECTRONIC_ATTACK_SYSTEMS = 50,
     DIS_PDU_RADIO_CATEGORY_TACTICAL_TARGETING_NETWORK_TECHNOLOGY_TTNT = 51
}  DIS_PDU_Radio_Category;

static const value_string DIS_PDU_Radio_Category_Strings[] = {
     { DIS_PDU_RADIO_CATEGORY_OTHER, "Other" },
     { DIS_PDU_RADIO_CATEGORY_VOICE_TRANSMISSION_RECEPTION, "Voice Transmission/Reception" },
     { DIS_PDU_RADIO_CATEGORY_DATA_LINK_TRANSMISSION_RECEPTION, "Data Link Transmission/Reception" },
     { DIS_PDU_RADIO_CATEGORY_VOICE_AND_DATA_LINK_TRANSMISSION_RECEPTION, "Voice and Data Link Transmission/Reception" },
     { DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_GLIDESLOPE_TRANSMITTER, "Instrumented Landing System (ILS) Glideslope Transmitter" },
     { DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_LOCALIZER_TRANSMITTER, "Instrumented Landing System (ILS) Localizer Transmitter" },
     { DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_OUTER_MARKER_BEACON, "Instrumented Landing System (ILS) Outer Marker Beacon" },
     { DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_MIDDLE_MARKER_BEACON, "Instrumented Landing System (ILS) Middle Marker Beacon" },
     { DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_INNER_MARKER_BEACON, "Instrumented Landing System (ILS) Inner Marker Beacon" },
     { DIS_PDU_RADIO_CATEGORY_INSTRUMENTED_LANDING_SYSTEM_ILS_RECEIVER_PLATFORM_RADIO, "Instrumented Landing System (ILS) Receiver (Platform Radio)" },
     { DIS_PDU_RADIO_CATEGORY_TACTICAL_AIR_NAVIGATION_TACAN_TRANSMITTER_GROUND_FIXED_EQUIPMENT, "Tactical Air Navigation (TACAN) Transmitter (Ground Fixed Equipment)" },
     { DIS_PDU_RADIO_CATEGORY_TACTICAL_AIR_NAVIGATION_TACAN_RECEIVER_MOVING_PLATFORM_EQUIPMENT, "Tactical Air Navigation (TACAN) Receiver (Moving Platform Equipment)" },
     { DIS_PDU_RADIO_CATEGORY_TACTICAL_AIR_NAVIGATION_TACAN_TRANSMITTER_RECEIVER_MOVING_PLATFORM_EQUIPMENT, "Tactical Air Navigation (TACAN) Transmitter/Receiver (Moving Platform Equipment)" },
     { DIS_PDU_RADIO_CATEGORY_VARIABLE_OMNI_RANGING_VOR_TRANSMITTER_GROUND_FIXED_EQUIPMENT, "Variable Omni-Ranging (VOR) Transmitter (Ground Fixed Equipment)" },
     { DIS_PDU_RADIO_CATEGORY_VARIABLE_OMNI_RANGING_VOR_WITH_DISTANCE_MEASURING_EQUIPMENT_DME_TRANSMITTER_GROUND_FIXED_EQUIPMENT, "Variable Omni-Ranging (VOR) with Distance Measuring Equipment (DME) Transmitter (Ground Fixed Equipment)" },
     { DIS_PDU_RADIO_CATEGORY_COMBINED_VOR_ILS_RECEIVER_MOVING_PLATFORM_EQUIPMENT, "Combined VOR/ILS Receiver (Moving Platform Equipment)" },
     { DIS_PDU_RADIO_CATEGORY_COMBINED_VOR_TACAN_VORTAC_TRANSMITTER, "Combined VOR & TACAN (VORTAC) Transmitter" },
     { DIS_PDU_RADIO_CATEGORY_NON_DIRECTIONAL_BEACON_NDB_TRANSMITTER, "Non-Directional Beacon (NDB) Transmitter" },
     { DIS_PDU_RADIO_CATEGORY_NON_DIRECTIONAL_BEACON_NDB_RECEIVER, "Non-Directional Beacon (NDB) Receiver" },
     { DIS_PDU_RADIO_CATEGORY_NON_DIRECTIONAL_BEACON_NDB_WITH_DISTANCE_MEASURING_EQUIPMENT_DME_TRANSMITTER, "Non-Directional Beacon (NDB) with Distance Measuring Equipment (DME) Transmitter" },
     { DIS_PDU_RADIO_CATEGORY_DISTANCE_MEASURING_EQUIPMENT_DME, "Distance Measuring Equipment (DME)" },
     { DIS_PDU_RADIO_CATEGORY_LINK_16_TERMINAL, "Link 16 Terminal" },
     { DIS_PDU_RADIO_CATEGORY_LINK_11_TERMINAL, "Link 11 Terminal" },
     { DIS_PDU_RADIO_CATEGORY_LINK_11B_TERMINAL, "Link 11B Terminal" },
     { DIS_PDU_RADIO_CATEGORY_EPLRS_SADL_TERMINAL, "EPLRS/SADL Terminal" },
     { DIS_PDU_RADIO_CATEGORY_F_22_INTRA_FLIGHT_DATA_LINK_IFDL, "F-22 Intra-Flight Data Link (IFDL)" },
     { DIS_PDU_RADIO_CATEGORY_F_35_MULTIFUNCTION_ADVANCED_DATA_LINK_MADL, "F-35 Multifunction Advanced Data Link (MADL)" },
     { DIS_PDU_RADIO_CATEGORY_SINCGARS_TERMINAL, "SINCGARS Terminal" },
     { DIS_PDU_RADIO_CATEGORY_L_BAND_SATCOM_TERMINAL, "L-Band SATCOM Terminal" },
     { DIS_PDU_RADIO_CATEGORY_IBS_TERMINAL, "IBS Terminal" },
     { DIS_PDU_RADIO_CATEGORY_GPS, "GPS" },
     { DIS_PDU_RADIO_CATEGORY_TACTICAL_VIDEO, "Tactical Video" },
     { DIS_PDU_RADIO_CATEGORY_AIR_TO_AIR_MISSILE_DATALINK, "Air-to-Air Missile Datalink" },
     { DIS_PDU_RADIO_CATEGORY_LINK_16_SURROGATE_FOR_NON_NATO_TDL_TERMINAL, "Link 16 Surrogate for Non-NATO TDL Terminal" },
     { DIS_PDU_RADIO_CATEGORY_MQ_1_9_C_BAND_LOS_DATALINK, "MQ-1/9 C-Band LOS Datalink" },
     { DIS_PDU_RADIO_CATEGORY_MQ_1_9_KU_BAND_SATCOM_DATALINK, "MQ-1/9 Ku-Band SATCOM Datalink" },
     { DIS_PDU_RADIO_CATEGORY_AIR_TO_GROUND_WEAPON_DATALINK, "Air-to-Ground Weapon Datalink" },
     { DIS_PDU_RADIO_CATEGORY_AUTOMATIC_IDENTIFICATION_SYSTEM_AIS, "Automatic Identification System (AIS)" },
     { DIS_PDU_RADIO_CATEGORY_JPALS_DATA_LINK, "JPALS Data Link" },
     { DIS_PDU_RADIO_CATEGORY_COMBAT_SEARCH_AND_RESCUE_CSAR_RADIO, "Combat Search and Rescue (CSAR) Radio" },
     { DIS_PDU_RADIO_CATEGORY_COUNTER_UNMANNED_AIRCRAFT_SYSTEM_C_UAS_RADIO, "Counter Unmanned Aircraft System (C-UAS) Radio" },
     { DIS_PDU_RADIO_CATEGORY_EMERGENCY_POSITION_INDICATING_RADIO_BEACONS_EPIRB, "Emergency Position-Indicating Radio Beacons (EPIRB)" },
     { DIS_PDU_RADIO_CATEGORY_ELECTRONIC_ATTACK_SYSTEMS, "Electronic Attack Systems" },
     { DIS_PDU_RADIO_CATEGORY_TACTICAL_TARGETING_NETWORK_TECHNOLOGY_TTNT, "Tactical Targeting Network Technology (TTNT)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 23] */
typedef enum
{
     DIS_PDU_RADIO_SUBCATEGORY_OTHER = 0,
     DIS_PDU_RADIO_SUBCATEGORY_JOINT_ELECTRONICS_TYPE_DESIGNATION_SYSTEM_JETDS_NON_SPECIFIC_SERIES = 1,
     DIS_PDU_RADIO_SUBCATEGORY_MANUFACTURER_DESIGNATION = 2,
     DIS_PDU_RADIO_SUBCATEGORY_NATIONAL_DESIGNATION = 3,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_1 = 11,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_2 = 12,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_3 = 13,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_4 = 14,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_1 = 15,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_2 = 16,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_3 = 17,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_4 = 18,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_1 = 19,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_2 = 20,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_3 = 21,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_4 = 22,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_1 = 23,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_2 = 24,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_3 = 25,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_4 = 26,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_1 = 27,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_2 = 28,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_3 = 29,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_4 = 30,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_1 = 31,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_2 = 32,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_3 = 33,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_4 = 34,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_1 = 35,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_2 = 36,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_3 = 37,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_4 = 38,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_1 = 39,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_2 = 40,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_3 = 41,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_4 = 42,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_1 = 43,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_2 = 44,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_3 = 45,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_4 = 46,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_1 = 47,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_2 = 48,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_3 = 49,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_4 = 50,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_1 = 51,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_2 = 52,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_3 = 53,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_4 = 54,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_1 = 55,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_2 = 56,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_3 = 57,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_4 = 58,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_1 = 59,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_2 = 60,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_3 = 61,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_4 = 62,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_1 = 63,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_2 = 64,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_3 = 65,
     DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_4 = 66
}  DIS_PDU_Radio_Subcategory;

static const value_string DIS_PDU_Radio_Subcategory_Strings[] = {
     { DIS_PDU_RADIO_SUBCATEGORY_OTHER, "Other" },
     { DIS_PDU_RADIO_SUBCATEGORY_JOINT_ELECTRONICS_TYPE_DESIGNATION_SYSTEM_JETDS_NON_SPECIFIC_SERIES, "Joint Electronics Type Designation System (JETDS) Non-specific Series" },
     { DIS_PDU_RADIO_SUBCATEGORY_MANUFACTURER_DESIGNATION, "Manufacturer Designation" },
     { DIS_PDU_RADIO_SUBCATEGORY_NATIONAL_DESIGNATION, "National Designation" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_1, "JETDS ARC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_2, "JETDS ARC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_3, "JETDS ARC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ARC_SET_4, "JETDS ARC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_1, "JETDS BRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_2, "JETDS BRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_3, "JETDS BRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_BRC_SET_4, "JETDS BRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_1, "JETDS CRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_2, "JETDS CRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_3, "JETDS CRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_CRC_SET_4, "JETDS CRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_1, "JETDS DRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_2, "JETDS DRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_3, "JETDS DRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_DRC_SET_4, "JETDS DRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_1, "JETDS FRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_2, "JETDS FRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_3, "JETDS FRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_FRC_SET_4, "JETDS FRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_1, "JETDS GRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_2, "JETDS GRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_3, "JETDS GRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_GRC_SET_4, "JETDS GRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_1, "JETDS KRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_2, "JETDS KRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_3, "JETDS KRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_KRC_SET_4, "JETDS KRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_1, "JETDS MRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_2, "JETDS MRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_3, "JETDS MRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_MRC_SET_4, "JETDS MRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_1, "JETDS PRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_2, "JETDS PRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_3, "JETDS PRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_PRC_SET_4, "JETDS PRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_1, "JETDS SRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_2, "JETDS SRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_3, "JETDS SRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_SRC_SET_4, "JETDS SRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_1, "JETDS TRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_2, "JETDS TRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_3, "JETDS TRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_TRC_SET_4, "JETDS TRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_1, "JETDS VRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_2, "JETDS VRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_3, "JETDS VRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_VRC_SET_4, "JETDS VRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_1, "JETDS WRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_2, "JETDS WRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_3, "JETDS WRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_WRC_SET_4, "JETDS WRC Set 4" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_1, "JETDS ZRC Set 1" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_2, "JETDS ZRC Set 2" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_3, "JETDS ZRC Set 3" },
     { DIS_PDU_RADIO_SUBCATEGORY_JETDS_ZRC_SET_4, "JETDS ZRC Set 4" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 24] was deprecated by SISO */

/* SISO-REF-010 [UID 25] */
typedef enum
{
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_OTHER = 0,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_CHAFF = 1,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_FLARE = 2,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_COMBINED_CHAFF_AND_FLARE = 3,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_ACTIVE_EMITTER = 4,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_PASSIVE_DECOY = 5,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_WINGED_DECOY = 6,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_SIGNAL_ILLUMINATION_FLARE = 7,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_SMOKE_GENERATOR = 8,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_COMBINED_FLARE_AND_SMOKE_GENERATOR = 12,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_SAR_NIGHT_LIGHT_SRNL = 13,
     DIS_PDU_EXPENDABLE_AIR_CATEGORY_SAR_BUOY = 14
}  DIS_PDU_Expendable_Air_Category;

/* SISO-REF-010 [UID 26] */
typedef enum
{
     DIS_PDU_EXPENDABLE_SURFACE_CATEGORY_OTHER = 0,
     DIS_PDU_EXPENDABLE_SURFACE_CATEGORY_FLARE = 2,
     DIS_PDU_EXPENDABLE_SURFACE_CATEGORY_ACTIVE_EMITTER = 4,
     DIS_PDU_EXPENDABLE_SURFACE_CATEGORY_PASSIVE_DECOY = 5,
     DIS_PDU_EXPENDABLE_SURFACE_CATEGORY_SMOKE_GENERATOR = 8,
     DIS_PDU_EXPENDABLE_SURFACE_CATEGORY_COMBINED_FLARE_AND_SMOKE_GENERATOR = 12,
     DIS_PDU_EXPENDABLE_SURFACE_CATEGORY_SAR_BUOY = 14
}  DIS_PDU_Expendable_Surface_Category;

/* SISO-REF-010 [UID 27] */
typedef enum
{
     DIS_PDU_EXPENDABLE_SUBSURFACE_CATEGORY_OTHER = 0,
     DIS_PDU_EXPENDABLE_SUBSURFACE_CATEGORY_ACTIVE_EMITTER = 4,
     DIS_PDU_EXPENDABLE_SUBSURFACE_CATEGORY_PASSIVE_DECOY = 5,
     DIS_PDU_EXPENDABLE_SUBSURFACE_CATEGORY_SIGNAL = 7,
     DIS_PDU_EXPENDABLE_SUBSURFACE_CATEGORY_NOISE_MAKER_DECOY = 9,
     DIS_PDU_EXPENDABLE_SUBSURFACE_CATEGORY_BUBBLE_MAKER_DECOY = 10,
     DIS_PDU_EXPENDABLE_SUBSURFACE_CATEGORY_MULTI_MODE_DECOY = 11
}  DIS_PDU_Expendable_Subsurface_Category;

/* SISO-REF-010 [UID 28] */
typedef enum
{
     DIS_PDU_SENSOR_EMITTER_CATEGORY_OTHER = 0,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_MULTI_SPECTRAL = 1,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_RF_ACTIVE = 2,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_RF_PASSIVE_INTERCEPT_AND_DF = 3,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_OPTICAL_DIRECT_VIEWING_WITH_OR_WITHOUT_OPTICS = 4,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_ELECTRO_OPTICAL = 5,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_SEISMIC = 6,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_CHEMICAL_POINT_DETECTOR = 7,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_CHEMICAL_STANDOFF = 8,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_THERMAL_TEMPERATURE_SENSING = 9,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_ACOUSTIC_ACTIVE = 10,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_ACOUSTIC_PASSIVE = 11,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_CONTACT_PRESSURE_PHYSICAL_HYDROSTATIC_BAROMETRIC = 12,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_ELECTRO_MAGNETIC_RADIATION_GAMMA_RADIATION = 13,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_PARTICLE_RADIATION_NEUTRONS_ALPHA_BETA_PARTICLES = 14,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_MAGNETIC = 15,
     DIS_PDU_SENSOR_EMITTER_CATEGORY_GRAVITATIONAL = 16
}  DIS_PDU_Sensor_Emitter_Category;

/* SISO-REF-010 [UID 29] */
typedef enum
{
     DIS_PDU_COUNTRY_OTHER = 0,
     DIS_PDU_COUNTRY_AFGHANISTAN_AFG = 1,
     DIS_PDU_COUNTRY_ALBANIA_ALB = 2,
     DIS_PDU_COUNTRY_ALGERIA_DZA = 3,
     DIS_PDU_COUNTRY_AMERICAN_SAMOA_ASM = 4,
     DIS_PDU_COUNTRY_ANDORRA_AND = 5,
     DIS_PDU_COUNTRY_ANGOLA_AGO = 6,
     DIS_PDU_COUNTRY_ANGUILLA_AIA = 7,
     DIS_PDU_COUNTRY_ANTARCTICA_ATA = 8,
     DIS_PDU_COUNTRY_ANTIGUA_AND_BARBUDA_ATG = 9,
     DIS_PDU_COUNTRY_ARGENTINA_ARG = 10,
     DIS_PDU_COUNTRY_ARUBA_ABW = 11,
     DIS_PDU_COUNTRY_AUSTRALIA_AUS = 13,
     DIS_PDU_COUNTRY_AUSTRIA_AUT = 14,
     DIS_PDU_COUNTRY_BAHAMAS_BHS = 15,
     DIS_PDU_COUNTRY_BAHRAIN_BHR = 16,
     DIS_PDU_COUNTRY_BANGLADESH_BGD = 18,
     DIS_PDU_COUNTRY_BARBADOS_BRB = 19,
     DIS_PDU_COUNTRY_BELGIUM_BEL = 21,
     DIS_PDU_COUNTRY_BELIZE_BLZ = 22,
     DIS_PDU_COUNTRY_BENIN_BEN = 23,
     DIS_PDU_COUNTRY_BERMUDA_BMU = 24,
     DIS_PDU_COUNTRY_BHUTAN_BTN = 25,
     DIS_PDU_COUNTRY_BOLIVIA_PLURINATIONAL_STATE_OF_BOL = 26,
     DIS_PDU_COUNTRY_BOTSWANA_BWA = 27,
     DIS_PDU_COUNTRY_BOUVET_ISLAND_BVT = 28,
     DIS_PDU_COUNTRY_BRAZIL_BRA = 29,
     DIS_PDU_COUNTRY_BRITISH_INDIAN_OCEAN_TERRITORY_IOT = 30,
     DIS_PDU_COUNTRY_VIRGIN_ISLANDS_BRITISH_VGB = 31,
     DIS_PDU_COUNTRY_BRUNEI_DARUSSALAM_BRN = 32,
     DIS_PDU_COUNTRY_BULGARIA_BGR = 33,
     DIS_PDU_COUNTRY_BURKINA_FASO_BFA = 34,
     DIS_PDU_COUNTRY_MYANMAR_MMR = 35,
     DIS_PDU_COUNTRY_BURUNDI_BDI = 36,
     DIS_PDU_COUNTRY_CAMBODIA_KHM = 37,
     DIS_PDU_COUNTRY_CAMEROON_CMR = 38,
     DIS_PDU_COUNTRY_CANADA_CAN = 39,
     DIS_PDU_COUNTRY_CABO_VERDE_CPV = 40,
     DIS_PDU_COUNTRY_CAYMAN_ISLANDS_CYM = 41,
     DIS_PDU_COUNTRY_CENTRAL_AFRICAN_REPUBLIC_CAF = 42,
     DIS_PDU_COUNTRY_CHAD_TCD = 43,
     DIS_PDU_COUNTRY_CHILE_CHL = 44,
     DIS_PDU_COUNTRY_CHINA_PEOPLE_S_REPUBLIC_OF_CHN = 45,
     DIS_PDU_COUNTRY_CHRISTMAS_ISLAND_CXR = 46,
     DIS_PDU_COUNTRY_COCOS_KEELING_ISLANDS_CCK = 47,
     DIS_PDU_COUNTRY_COLOMBIA_COL = 48,
     DIS_PDU_COUNTRY_COMOROS_COM = 49,
     DIS_PDU_COUNTRY_CONGO_COG = 50,
     DIS_PDU_COUNTRY_COOK_ISLANDS_COK = 51,
     DIS_PDU_COUNTRY_COSTA_RICA_CRI = 53,
     DIS_PDU_COUNTRY_CUBA_CUB = 54,
     DIS_PDU_COUNTRY_CYPRUS_CYP = 55,
     DIS_PDU_COUNTRY_DENMARK_DNK = 57,
     DIS_PDU_COUNTRY_DJIBOUTI_DJI = 58,
     DIS_PDU_COUNTRY_DOMINICA_DMA = 59,
     DIS_PDU_COUNTRY_DOMINICAN_REPUBLIC_DOM = 60,
     DIS_PDU_COUNTRY_ECUADOR_ECU = 61,
     DIS_PDU_COUNTRY_EGYPT_EGY = 62,
     DIS_PDU_COUNTRY_EL_SALVADOR_SLV = 63,
     DIS_PDU_COUNTRY_EQUATORIAL_GUINEA_GNQ = 64,
     DIS_PDU_COUNTRY_ETHIOPIA_ETH = 65,
     DIS_PDU_COUNTRY_FALKLAND_ISLANDS_MALVINAS_FLK = 67,
     DIS_PDU_COUNTRY_FAROE_ISLANDS_FRO = 68,
     DIS_PDU_COUNTRY_FIJI_FJI = 69,
     DIS_PDU_COUNTRY_FINLAND_FIN = 70,
     DIS_PDU_COUNTRY_FRANCE_FRA = 71,
     DIS_PDU_COUNTRY_FRENCH_GUIANA_GUF = 72,
     DIS_PDU_COUNTRY_FRENCH_POLYNESIA_PYF = 73,
     DIS_PDU_COUNTRY_FRENCH_SOUTHERN_TERRITORIES_ATF = 74,
     DIS_PDU_COUNTRY_GABON_GAB = 75,
     DIS_PDU_COUNTRY_GAMBIA_THE_GMB = 76,
     DIS_PDU_COUNTRY_GERMANY_DEU = 78,
     DIS_PDU_COUNTRY_GHANA_GHA = 79,
     DIS_PDU_COUNTRY_GIBRALTAR_GIB = 80,
     DIS_PDU_COUNTRY_GREECE_GRC = 82,
     DIS_PDU_COUNTRY_GREENLAND_GRL = 83,
     DIS_PDU_COUNTRY_GRENADA_GRD = 84,
     DIS_PDU_COUNTRY_GUADELOUPE_GLP = 85,
     DIS_PDU_COUNTRY_GUAM_GUM = 86,
     DIS_PDU_COUNTRY_GUATEMALA_GTM = 87,
     DIS_PDU_COUNTRY_GUERNSEY_GGY = 88,
     DIS_PDU_COUNTRY_GUINEA_GIN = 89,
     DIS_PDU_COUNTRY_GUINEA_BISSAU_GNB = 90,
     DIS_PDU_COUNTRY_GUYANA_GUY = 91,
     DIS_PDU_COUNTRY_HAITI_HTI = 92,
     DIS_PDU_COUNTRY_HEARD_ISLAND_AND_MCDONALD_ISLANDS_HMD = 93,
     DIS_PDU_COUNTRY_HONDURAS_HND = 94,
     DIS_PDU_COUNTRY_HONG_KONG_HKG = 95,
     DIS_PDU_COUNTRY_HUNGARY_HUN = 97,
     DIS_PDU_COUNTRY_ICELAND_ISL = 98,
     DIS_PDU_COUNTRY_INDIA_IND = 99,
     DIS_PDU_COUNTRY_INDONESIA_IDN = 100,
     DIS_PDU_COUNTRY_IRAN_ISLAMIC_REPUBLIC_OF_IRN = 101,
     DIS_PDU_COUNTRY_IRAQ_IRQ = 102,
     DIS_PDU_COUNTRY_IRELAND_IRL = 104,
     DIS_PDU_COUNTRY_ISRAEL_ISR = 105,
     DIS_PDU_COUNTRY_ITALY_ITA = 106,
     DIS_PDU_COUNTRY_COTE_D_IVOIRE_CIV = 107,
     DIS_PDU_COUNTRY_JAMAICA_JAM = 108,
     DIS_PDU_COUNTRY_JAPAN_JPN = 110,
     DIS_PDU_COUNTRY_JERSEY_JEY = 112,
     DIS_PDU_COUNTRY_JORDAN_JOR = 114,
     DIS_PDU_COUNTRY_KENYA_KEN = 116,
     DIS_PDU_COUNTRY_KIRIBATI_KIR = 118,
     DIS_PDU_COUNTRY_KOREA_DEMOCRATIC_PEOPLE_S_REPUBLIC_OF_PRK = 119,
     DIS_PDU_COUNTRY_KOREA_REPUBLIC_OF_KOR = 120,
     DIS_PDU_COUNTRY_KUWAIT_KWT = 121,
     DIS_PDU_COUNTRY_LAO_PEOPLE_S_DEMOCRATIC_REPUBLIC_LAO = 122,
     DIS_PDU_COUNTRY_LEBANON_LBN = 123,
     DIS_PDU_COUNTRY_LESOTHO_LSO = 124,
     DIS_PDU_COUNTRY_LIBERIA_LBR = 125,
     DIS_PDU_COUNTRY_LIBYA_LBY = 126,
     DIS_PDU_COUNTRY_LIECHTENSTEIN_LIE = 127,
     DIS_PDU_COUNTRY_LUXEMBOURG_LUX = 128,
     DIS_PDU_COUNTRY_MADAGASCAR_MDG = 129,
     DIS_PDU_COUNTRY_MACAO_MAC = 130,
     DIS_PDU_COUNTRY_MALAWI_MWI = 131,
     DIS_PDU_COUNTRY_MALAYSIA_MYS = 132,
     DIS_PDU_COUNTRY_MALDIVES_MDV = 133,
     DIS_PDU_COUNTRY_MALI_MLI = 134,
     DIS_PDU_COUNTRY_MALTA_MLT = 135,
     DIS_PDU_COUNTRY_ISLE_OF_MAN_IMN = 136,
     DIS_PDU_COUNTRY_MARSHALL_ISLANDS_MHL = 137,
     DIS_PDU_COUNTRY_MARTINIQUE_MTQ = 138,
     DIS_PDU_COUNTRY_MAURITANIA_MRT = 139,
     DIS_PDU_COUNTRY_MAURITIUS_MUS = 140,
     DIS_PDU_COUNTRY_MAYOTTE_MYT = 141,
     DIS_PDU_COUNTRY_MEXICO_MEX = 142,
     DIS_PDU_COUNTRY_MICRONESIA_FEDERATED_STATES_OF_FSM = 143,
     DIS_PDU_COUNTRY_MONACO_MCO = 144,
     DIS_PDU_COUNTRY_MONGOLIA_MNG = 145,
     DIS_PDU_COUNTRY_MONTSERRAT_MSR = 146,
     DIS_PDU_COUNTRY_MOROCCO_MAR = 147,
     DIS_PDU_COUNTRY_MOZAMBIQUE_MOZ = 148,
     DIS_PDU_COUNTRY_NAMIBIA_NAM = 149,
     DIS_PDU_COUNTRY_NAURU_NRU = 150,
     DIS_PDU_COUNTRY_NEPAL_NPL = 152,
     DIS_PDU_COUNTRY_NETHERLANDS_NLD = 153,
     DIS_PDU_COUNTRY_NEW_CALEDONIA_NCL = 155,
     DIS_PDU_COUNTRY_NEW_ZEALAND_NZL = 156,
     DIS_PDU_COUNTRY_NICARAGUA_NIC = 157,
     DIS_PDU_COUNTRY_NIGER_NER = 158,
     DIS_PDU_COUNTRY_NIGERIA_NGA = 159,
     DIS_PDU_COUNTRY_NIUE_NIU = 160,
     DIS_PDU_COUNTRY_NORFOLK_ISLAND_NFK = 161,
     DIS_PDU_COUNTRY_NORTHERN_MARIANA_ISLANDS_MNP = 162,
     DIS_PDU_COUNTRY_NORWAY_NOR = 163,
     DIS_PDU_COUNTRY_OMAN_OMN = 164,
     DIS_PDU_COUNTRY_PAKISTAN_PAK = 165,
     DIS_PDU_COUNTRY_PANAMA_PAN = 168,
     DIS_PDU_COUNTRY_PAPUA_NEW_GUINEA_PNG = 169,
     DIS_PDU_COUNTRY_PARAGUAY_PRY = 171,
     DIS_PDU_COUNTRY_PERU_PER = 172,
     DIS_PDU_COUNTRY_PHILIPPINES_PHL = 173,
     DIS_PDU_COUNTRY_PITCAIRN_PCN = 174,
     DIS_PDU_COUNTRY_POLAND_POL = 175,
     DIS_PDU_COUNTRY_PORTUGAL_PRT = 176,
     DIS_PDU_COUNTRY_PUERTO_RICO_PRI = 177,
     DIS_PDU_COUNTRY_QATAR_QAT = 178,
     DIS_PDU_COUNTRY_REUNION_REU = 179,
     DIS_PDU_COUNTRY_ROMANIA_ROU = 180,
     DIS_PDU_COUNTRY_RWANDA_RWA = 181,
     DIS_PDU_COUNTRY_SAINT_KITTS_AND_NEVIS_KNA = 182,
     DIS_PDU_COUNTRY_SAINT_HELENA_ASCENSION_AND_TRISTAN_DA_CUNHA_SHN = 183,
     DIS_PDU_COUNTRY_SAINT_LUCIA_LCA = 184,
     DIS_PDU_COUNTRY_SAINT_PIERRE_AND_MIQUELON_SPM = 185,
     DIS_PDU_COUNTRY_SAINT_VINCENT_AND_THE_GRENADINES_VCT = 186,
     DIS_PDU_COUNTRY_SAN_MARINO_SMR = 187,
     DIS_PDU_COUNTRY_SAO_TOME_AND_PRINCIPE_STP = 188,
     DIS_PDU_COUNTRY_SAUDI_ARABIA_SAU = 189,
     DIS_PDU_COUNTRY_SENEGAL_SEN = 190,
     DIS_PDU_COUNTRY_SEYCHELLES_SYC = 191,
     DIS_PDU_COUNTRY_SIERRA_LEONE_SLE = 192,
     DIS_PDU_COUNTRY_SINGAPORE_SGP = 193,
     DIS_PDU_COUNTRY_SOLOMON_ISLANDS_SLB = 194,
     DIS_PDU_COUNTRY_SOMALIA_SOM = 195,
     DIS_PDU_COUNTRY_SOUTH_GEORGIA_AND_THE_SOUTH_SANDWICH_ISLANDS_SGS = 196,
     DIS_PDU_COUNTRY_SOUTH_AFRICA_ZAF = 197,
     DIS_PDU_COUNTRY_SPAIN_ESP = 198,
     DIS_PDU_COUNTRY_SRI_LANKA_LKA = 200,
     DIS_PDU_COUNTRY_SUDAN_SDN = 201,
     DIS_PDU_COUNTRY_SURINAME_SUR = 202,
     DIS_PDU_COUNTRY_ESWATINI_SWZ = 204,
     DIS_PDU_COUNTRY_SWEDEN_SWE = 205,
     DIS_PDU_COUNTRY_SWITZERLAND_CHE = 206,
     DIS_PDU_COUNTRY_SYRIAN_ARAB_REPUBLIC_SYR = 207,
     DIS_PDU_COUNTRY_TAIWAN_PROVINCE_OF_CHINA_TWN = 208,
     DIS_PDU_COUNTRY_TANZANIA_UNITED_REPUBLIC_OF_TZA = 209,
     DIS_PDU_COUNTRY_THAILAND_THA = 210,
     DIS_PDU_COUNTRY_TOGO_TGO = 211,
     DIS_PDU_COUNTRY_TOKELAU_TKL = 212,
     DIS_PDU_COUNTRY_TONGA_TON = 213,
     DIS_PDU_COUNTRY_TRINIDAD_AND_TOBAGO_TTO = 214,
     DIS_PDU_COUNTRY_PALAU_PLW = 216,
     DIS_PDU_COUNTRY_TUNISIA_TUN = 217,
     DIS_PDU_COUNTRY_TURKEY_TUR = 218,
     DIS_PDU_COUNTRY_TURKS_AND_CAICOS_ISLANDS_TCA = 219,
     DIS_PDU_COUNTRY_TUVALU_TUV = 220,
     DIS_PDU_COUNTRY_UGANDA_UGA = 221,
     DIS_PDU_COUNTRY_RUSSIA_RUS = 222,
     DIS_PDU_COUNTRY_UNITED_ARAB_EMIRATES_ARE = 223,
     DIS_PDU_COUNTRY_UNITED_KINGDOM_OF_GREAT_BRITAIN_AND_NORTHERN_IRELAND_GBR = 224,
     DIS_PDU_COUNTRY_UNITED_STATES_OF_AMERICA_USA = 225,
     DIS_PDU_COUNTRY_URUGUAY_URY = 226,
     DIS_PDU_COUNTRY_VANUATU_VUT = 227,
     DIS_PDU_COUNTRY_HOLY_SEE_VAT = 228,
     DIS_PDU_COUNTRY_VENEZUELA_BOLIVARIAN_REPUBLIC_OF_VEN = 229,
     DIS_PDU_COUNTRY_VIET_NAM_VNM = 230,
     DIS_PDU_COUNTRY_VIRGIN_ISLANDS_U_S_VIR = 231,
     DIS_PDU_COUNTRY_WALLIS_AND_FUTUNA_WLF = 233,
     DIS_PDU_COUNTRY_WESTERN_SAHARA_ESH = 234,
     DIS_PDU_COUNTRY_SAMOA_WSM = 236,
     DIS_PDU_COUNTRY_YEMEN_YEM = 237,
     DIS_PDU_COUNTRY_ZAMBIA_ZMB = 242,
     DIS_PDU_COUNTRY_ZIMBABWE_ZWE = 243,
     DIS_PDU_COUNTRY_ARMENIA_ARM = 244,
     DIS_PDU_COUNTRY_AZERBAIJAN_AZE = 245,
     DIS_PDU_COUNTRY_BELARUS_BLR = 246,
     DIS_PDU_COUNTRY_BOSNIA_AND_HERZEGOVINA_BIH = 247,
     DIS_PDU_COUNTRY_CROATIA_HRV = 249,
     DIS_PDU_COUNTRY_ESTONIA_EST = 250,
     DIS_PDU_COUNTRY_GEORGIA_GEO = 251,
     DIS_PDU_COUNTRY_KAZAKHSTAN_KAZ = 252,
     DIS_PDU_COUNTRY_KYRGYZSTAN_KGZ = 253,
     DIS_PDU_COUNTRY_LATVIA_LVA = 254,
     DIS_PDU_COUNTRY_LITHUANIA_LTU = 255,
     DIS_PDU_COUNTRY_NORTH_MACEDONIA_MKD = 256,
     DIS_PDU_COUNTRY_MOLDOVA_REPUBLIC_OF_MDA = 258,
     DIS_PDU_COUNTRY_MONTENEGRO_MNE = 259,
     DIS_PDU_COUNTRY_SLOVENIA_SVN = 262,
     DIS_PDU_COUNTRY_TAJIKISTAN_TJK = 263,
     DIS_PDU_COUNTRY_TURKMENISTAN_TKM = 264,
     DIS_PDU_COUNTRY_UKRAINE_UKR = 265,
     DIS_PDU_COUNTRY_UZBEKISTAN_UZB = 266,
     DIS_PDU_COUNTRY_CZECH_REPUBLIC_CZE = 267,
     DIS_PDU_COUNTRY_SLOVAKIA_SVK = 268,
     DIS_PDU_COUNTRY_AALAND_ISLANDS_ALA = 269,
     DIS_PDU_COUNTRY_BONAIRE_SINT_EUSTATIUS_AND_SABA_BES = 270,
     DIS_PDU_COUNTRY_CONGO_DEMOCRATIC_REPUBLIC_OF_THE_COD = 271,
     DIS_PDU_COUNTRY_CURACAO_CUW = 272,
     DIS_PDU_COUNTRY_ERITREA_ERI = 273,
     DIS_PDU_COUNTRY_SAINT_BARTHELEMY_BLM = 274,
     DIS_PDU_COUNTRY_SAINT_MARTIN_FRENCH_PART_MAF = 275,
     DIS_PDU_COUNTRY_SERBIA_SRB = 276,
     DIS_PDU_COUNTRY_SINT_MAARTEN_DUTCH_PART_SXM = 277,
     DIS_PDU_COUNTRY_SOUTH_SUDAN_SSD = 278,
     DIS_PDU_COUNTRY_SVALBARD_AND_JAN_MAYEN_SJM = 279,
     DIS_PDU_COUNTRY_TIMOR_LESTE_TLS = 280,
     DIS_PDU_COUNTRY_UNITED_STATES_MINOR_OUTLYING_ISLANDS_UMI = 281,
     DIS_PDU_COUNTRY_PALESTINE_STATE_OF_PSE = 282
}  DIS_PDU_Country;

static const value_string DIS_PDU_Country_Strings[] = {
     { DIS_PDU_COUNTRY_OTHER, "Other" },
     { DIS_PDU_COUNTRY_AFGHANISTAN_AFG, "Afghanistan (AFG)" },
     { DIS_PDU_COUNTRY_ALBANIA_ALB, "Albania (ALB)" },
     { DIS_PDU_COUNTRY_ALGERIA_DZA, "Algeria (DZA)" },
     { DIS_PDU_COUNTRY_AMERICAN_SAMOA_ASM, "American Samoa (ASM)" },
     { DIS_PDU_COUNTRY_ANDORRA_AND, "Andorra (AND)" },
     { DIS_PDU_COUNTRY_ANGOLA_AGO, "Angola (AGO)" },
     { DIS_PDU_COUNTRY_ANGUILLA_AIA, "Anguilla (AIA)" },
     { DIS_PDU_COUNTRY_ANTARCTICA_ATA, "Antarctica (ATA)" },
     { DIS_PDU_COUNTRY_ANTIGUA_AND_BARBUDA_ATG, "Antigua and Barbuda (ATG)" },
     { DIS_PDU_COUNTRY_ARGENTINA_ARG, "Argentina (ARG)" },
     { DIS_PDU_COUNTRY_ARUBA_ABW, "Aruba (ABW)" },
     { DIS_PDU_COUNTRY_AUSTRALIA_AUS, "Australia (AUS)" },
     { DIS_PDU_COUNTRY_AUSTRIA_AUT, "Austria (AUT)" },
     { DIS_PDU_COUNTRY_BAHAMAS_BHS, "Bahamas (BHS)" },
     { DIS_PDU_COUNTRY_BAHRAIN_BHR, "Bahrain (BHR)" },
     { DIS_PDU_COUNTRY_BANGLADESH_BGD, "Bangladesh (BGD)" },
     { DIS_PDU_COUNTRY_BARBADOS_BRB, "Barbados (BRB)" },
     { DIS_PDU_COUNTRY_BELGIUM_BEL, "Belgium (BEL)" },
     { DIS_PDU_COUNTRY_BELIZE_BLZ, "Belize (BLZ)" },
     { DIS_PDU_COUNTRY_BENIN_BEN, "Benin (BEN)" },
     { DIS_PDU_COUNTRY_BERMUDA_BMU, "Bermuda (BMU)" },
     { DIS_PDU_COUNTRY_BHUTAN_BTN, "Bhutan (BTN)" },
     { DIS_PDU_COUNTRY_BOLIVIA_PLURINATIONAL_STATE_OF_BOL, "Bolivia (Plurinational State of) (BOL)" },
     { DIS_PDU_COUNTRY_BOTSWANA_BWA, "Botswana (BWA)" },
     { DIS_PDU_COUNTRY_BOUVET_ISLAND_BVT, "Bouvet Island (BVT)" },
     { DIS_PDU_COUNTRY_BRAZIL_BRA, "Brazil (BRA)" },
     { DIS_PDU_COUNTRY_BRITISH_INDIAN_OCEAN_TERRITORY_IOT, "British Indian Ocean Territory (IOT)" },
     { DIS_PDU_COUNTRY_VIRGIN_ISLANDS_BRITISH_VGB, "Virgin Islands (British) (VGB)" },
     { DIS_PDU_COUNTRY_BRUNEI_DARUSSALAM_BRN, "Brunei Darussalam (BRN)" },
     { DIS_PDU_COUNTRY_BULGARIA_BGR, "Bulgaria (BGR)" },
     { DIS_PDU_COUNTRY_BURKINA_FASO_BFA, "Burkina Faso (BFA)" },
     { DIS_PDU_COUNTRY_MYANMAR_MMR, "Myanmar (MMR)" },
     { DIS_PDU_COUNTRY_BURUNDI_BDI, "Burundi (BDI)" },
     { DIS_PDU_COUNTRY_CAMBODIA_KHM, "Cambodia (KHM)" },
     { DIS_PDU_COUNTRY_CAMEROON_CMR, "Cameroon (CMR)" },
     { DIS_PDU_COUNTRY_CANADA_CAN, "Canada (CAN)" },
     { DIS_PDU_COUNTRY_CABO_VERDE_CPV, "Cabo Verde (CPV)" },
     { DIS_PDU_COUNTRY_CAYMAN_ISLANDS_CYM, "Cayman Islands (CYM)" },
     { DIS_PDU_COUNTRY_CENTRAL_AFRICAN_REPUBLIC_CAF, "Central African Republic (CAF)" },
     { DIS_PDU_COUNTRY_CHAD_TCD, "Chad (TCD)" },
     { DIS_PDU_COUNTRY_CHILE_CHL, "Chile (CHL)" },
     { DIS_PDU_COUNTRY_CHINA_PEOPLE_S_REPUBLIC_OF_CHN, "China, People's Republic of (CHN)" },
     { DIS_PDU_COUNTRY_CHRISTMAS_ISLAND_CXR, "Christmas Island (CXR)" },
     { DIS_PDU_COUNTRY_COCOS_KEELING_ISLANDS_CCK, "Cocos (Keeling) Islands (CCK)" },
     { DIS_PDU_COUNTRY_COLOMBIA_COL, "Colombia (COL)" },
     { DIS_PDU_COUNTRY_COMOROS_COM, "Comoros (COM)" },
     { DIS_PDU_COUNTRY_CONGO_COG, "Congo (COG)" },
     { DIS_PDU_COUNTRY_COOK_ISLANDS_COK, "Cook Islands (COK)" },
     { DIS_PDU_COUNTRY_COSTA_RICA_CRI, "Costa Rica (CRI)" },
     { DIS_PDU_COUNTRY_CUBA_CUB, "Cuba (CUB)" },
     { DIS_PDU_COUNTRY_CYPRUS_CYP, "Cyprus (CYP)" },
     { DIS_PDU_COUNTRY_DENMARK_DNK, "Denmark (DNK)" },
     { DIS_PDU_COUNTRY_DJIBOUTI_DJI, "Djibouti (DJI)" },
     { DIS_PDU_COUNTRY_DOMINICA_DMA, "Dominica (DMA)" },
     { DIS_PDU_COUNTRY_DOMINICAN_REPUBLIC_DOM, "Dominican Republic (DOM)" },
     { DIS_PDU_COUNTRY_ECUADOR_ECU, "Ecuador (ECU)" },
     { DIS_PDU_COUNTRY_EGYPT_EGY, "Egypt (EGY)" },
     { DIS_PDU_COUNTRY_EL_SALVADOR_SLV, "El Salvador (SLV)" },
     { DIS_PDU_COUNTRY_EQUATORIAL_GUINEA_GNQ, "Equatorial Guinea (GNQ)" },
     { DIS_PDU_COUNTRY_ETHIOPIA_ETH, "Ethiopia (ETH)" },
     { DIS_PDU_COUNTRY_FALKLAND_ISLANDS_MALVINAS_FLK, "Falkland Islands (Malvinas) (FLK)" },
     { DIS_PDU_COUNTRY_FAROE_ISLANDS_FRO, "Faroe Islands (FRO)" },
     { DIS_PDU_COUNTRY_FIJI_FJI, "Fiji (FJI)" },
     { DIS_PDU_COUNTRY_FINLAND_FIN, "Finland (FIN)" },
     { DIS_PDU_COUNTRY_FRANCE_FRA, "France (FRA)" },
     { DIS_PDU_COUNTRY_FRENCH_GUIANA_GUF, "French Guiana (GUF)" },
     { DIS_PDU_COUNTRY_FRENCH_POLYNESIA_PYF, "French Polynesia (PYF)" },
     { DIS_PDU_COUNTRY_FRENCH_SOUTHERN_TERRITORIES_ATF, "French Southern Territories (ATF)" },
     { DIS_PDU_COUNTRY_GABON_GAB, "Gabon (GAB)" },
     { DIS_PDU_COUNTRY_GAMBIA_THE_GMB, "Gambia, The (GMB)" },
     { DIS_PDU_COUNTRY_GERMANY_DEU, "Germany (DEU)" },
     { DIS_PDU_COUNTRY_GHANA_GHA, "Ghana (GHA)" },
     { DIS_PDU_COUNTRY_GIBRALTAR_GIB, "Gibraltar (GIB)" },
     { DIS_PDU_COUNTRY_GREECE_GRC, "Greece (GRC)" },
     { DIS_PDU_COUNTRY_GREENLAND_GRL, "Greenland (GRL)" },
     { DIS_PDU_COUNTRY_GRENADA_GRD, "Grenada (GRD)" },
     { DIS_PDU_COUNTRY_GUADELOUPE_GLP, "Guadeloupe (GLP)" },
     { DIS_PDU_COUNTRY_GUAM_GUM, "Guam (GUM)" },
     { DIS_PDU_COUNTRY_GUATEMALA_GTM, "Guatemala (GTM)" },
     { DIS_PDU_COUNTRY_GUERNSEY_GGY, "Guernsey (GGY)" },
     { DIS_PDU_COUNTRY_GUINEA_GIN, "Guinea (GIN)" },
     { DIS_PDU_COUNTRY_GUINEA_BISSAU_GNB, "Guinea-Bissau (GNB)" },
     { DIS_PDU_COUNTRY_GUYANA_GUY, "Guyana (GUY)" },
     { DIS_PDU_COUNTRY_HAITI_HTI, "Haiti (HTI)" },
     { DIS_PDU_COUNTRY_HEARD_ISLAND_AND_MCDONALD_ISLANDS_HMD, "Heard Island and McDonald Islands (HMD)" },
     { DIS_PDU_COUNTRY_HONDURAS_HND, "Honduras (HND)" },
     { DIS_PDU_COUNTRY_HONG_KONG_HKG, "Hong Kong (HKG)" },
     { DIS_PDU_COUNTRY_HUNGARY_HUN, "Hungary (HUN)" },
     { DIS_PDU_COUNTRY_ICELAND_ISL, "Iceland (ISL)" },
     { DIS_PDU_COUNTRY_INDIA_IND, "India (IND)" },
     { DIS_PDU_COUNTRY_INDONESIA_IDN, "Indonesia (IDN)" },
     { DIS_PDU_COUNTRY_IRAN_ISLAMIC_REPUBLIC_OF_IRN, "Iran (Islamic Republic of) (IRN)" },
     { DIS_PDU_COUNTRY_IRAQ_IRQ, "Iraq (IRQ)" },
     { DIS_PDU_COUNTRY_IRELAND_IRL, "Ireland (IRL)" },
     { DIS_PDU_COUNTRY_ISRAEL_ISR, "Israel (ISR)" },
     { DIS_PDU_COUNTRY_ITALY_ITA, "Italy (ITA)" },
     { DIS_PDU_COUNTRY_COTE_D_IVOIRE_CIV, "Cote d'Ivoire (CIV)" },
     { DIS_PDU_COUNTRY_JAMAICA_JAM, "Jamaica (JAM)" },
     { DIS_PDU_COUNTRY_JAPAN_JPN, "Japan (JPN)" },
     { DIS_PDU_COUNTRY_JERSEY_JEY, "Jersey (JEY)" },
     { DIS_PDU_COUNTRY_JORDAN_JOR, "Jordan (JOR)" },
     { DIS_PDU_COUNTRY_KENYA_KEN, "Kenya (KEN)" },
     { DIS_PDU_COUNTRY_KIRIBATI_KIR, "Kiribati (KIR)" },
     { DIS_PDU_COUNTRY_KOREA_DEMOCRATIC_PEOPLE_S_REPUBLIC_OF_PRK, "Korea (Democratic People's Republic of) (PRK)" },
     { DIS_PDU_COUNTRY_KOREA_REPUBLIC_OF_KOR, "Korea (Republic of) (KOR)" },
     { DIS_PDU_COUNTRY_KUWAIT_KWT, "Kuwait (KWT)" },
     { DIS_PDU_COUNTRY_LAO_PEOPLE_S_DEMOCRATIC_REPUBLIC_LAO, "Lao People's Democratic Republic (LAO)" },
     { DIS_PDU_COUNTRY_LEBANON_LBN, "Lebanon (LBN)" },
     { DIS_PDU_COUNTRY_LESOTHO_LSO, "Lesotho (LSO)" },
     { DIS_PDU_COUNTRY_LIBERIA_LBR, "Liberia (LBR)" },
     { DIS_PDU_COUNTRY_LIBYA_LBY, "Libya (LBY)" },
     { DIS_PDU_COUNTRY_LIECHTENSTEIN_LIE, "Liechtenstein (LIE)" },
     { DIS_PDU_COUNTRY_LUXEMBOURG_LUX, "Luxembourg (LUX)" },
     { DIS_PDU_COUNTRY_MADAGASCAR_MDG, "Madagascar (MDG)" },
     { DIS_PDU_COUNTRY_MACAO_MAC, "Macao (MAC)" },
     { DIS_PDU_COUNTRY_MALAWI_MWI, "Malawi (MWI)" },
     { DIS_PDU_COUNTRY_MALAYSIA_MYS, "Malaysia (MYS)" },
     { DIS_PDU_COUNTRY_MALDIVES_MDV, "Maldives (MDV)" },
     { DIS_PDU_COUNTRY_MALI_MLI, "Mali (MLI)" },
     { DIS_PDU_COUNTRY_MALTA_MLT, "Malta (MLT)" },
     { DIS_PDU_COUNTRY_ISLE_OF_MAN_IMN, "Isle of Man (IMN)" },
     { DIS_PDU_COUNTRY_MARSHALL_ISLANDS_MHL, "Marshall Islands (MHL)" },
     { DIS_PDU_COUNTRY_MARTINIQUE_MTQ, "Martinique (MTQ)" },
     { DIS_PDU_COUNTRY_MAURITANIA_MRT, "Mauritania (MRT)" },
     { DIS_PDU_COUNTRY_MAURITIUS_MUS, "Mauritius (MUS)" },
     { DIS_PDU_COUNTRY_MAYOTTE_MYT, "Mayotte (MYT)" },
     { DIS_PDU_COUNTRY_MEXICO_MEX, "Mexico (MEX)" },
     { DIS_PDU_COUNTRY_MICRONESIA_FEDERATED_STATES_OF_FSM, "Micronesia (Federated States of) (FSM)" },
     { DIS_PDU_COUNTRY_MONACO_MCO, "Monaco (MCO)" },
     { DIS_PDU_COUNTRY_MONGOLIA_MNG, "Mongolia (MNG)" },
     { DIS_PDU_COUNTRY_MONTSERRAT_MSR, "Montserrat (MSR)" },
     { DIS_PDU_COUNTRY_MOROCCO_MAR, "Morocco (MAR)" },
     { DIS_PDU_COUNTRY_MOZAMBIQUE_MOZ, "Mozambique (MOZ)" },
     { DIS_PDU_COUNTRY_NAMIBIA_NAM, "Namibia (NAM)" },
     { DIS_PDU_COUNTRY_NAURU_NRU, "Nauru (NRU)" },
     { DIS_PDU_COUNTRY_NEPAL_NPL, "Nepal (NPL)" },
     { DIS_PDU_COUNTRY_NETHERLANDS_NLD, "Netherlands (NLD)" },
     { DIS_PDU_COUNTRY_NEW_CALEDONIA_NCL, "New Caledonia (NCL)" },
     { DIS_PDU_COUNTRY_NEW_ZEALAND_NZL, "New Zealand (NZL)" },
     { DIS_PDU_COUNTRY_NICARAGUA_NIC, "Nicaragua (NIC)" },
     { DIS_PDU_COUNTRY_NIGER_NER, "Niger (NER)" },
     { DIS_PDU_COUNTRY_NIGERIA_NGA, "Nigeria (NGA)" },
     { DIS_PDU_COUNTRY_NIUE_NIU, "Niue (NIU)" },
     { DIS_PDU_COUNTRY_NORFOLK_ISLAND_NFK, "Norfolk Island (NFK)" },
     { DIS_PDU_COUNTRY_NORTHERN_MARIANA_ISLANDS_MNP, "Northern Mariana Islands (MNP)" },
     { DIS_PDU_COUNTRY_NORWAY_NOR, "Norway (NOR)" },
     { DIS_PDU_COUNTRY_OMAN_OMN, "Oman (OMN)" },
     { DIS_PDU_COUNTRY_PAKISTAN_PAK, "Pakistan (PAK)" },
     { DIS_PDU_COUNTRY_PANAMA_PAN, "Panama (PAN)" },
     { DIS_PDU_COUNTRY_PAPUA_NEW_GUINEA_PNG, "Papua New Guinea (PNG)" },
     { DIS_PDU_COUNTRY_PARAGUAY_PRY, "Paraguay (PRY)" },
     { DIS_PDU_COUNTRY_PERU_PER, "Peru (PER)" },
     { DIS_PDU_COUNTRY_PHILIPPINES_PHL, "Philippines (PHL)" },
     { DIS_PDU_COUNTRY_PITCAIRN_PCN, "Pitcairn (PCN)" },
     { DIS_PDU_COUNTRY_POLAND_POL, "Poland (POL)" },
     { DIS_PDU_COUNTRY_PORTUGAL_PRT, "Portugal (PRT)" },
     { DIS_PDU_COUNTRY_PUERTO_RICO_PRI, "Puerto Rico (PRI)" },
     { DIS_PDU_COUNTRY_QATAR_QAT, "Qatar (QAT)" },
     { DIS_PDU_COUNTRY_REUNION_REU, "Reunion (REU)" },
     { DIS_PDU_COUNTRY_ROMANIA_ROU, "Romania (ROU)" },
     { DIS_PDU_COUNTRY_RWANDA_RWA, "Rwanda (RWA)" },
     { DIS_PDU_COUNTRY_SAINT_KITTS_AND_NEVIS_KNA, "Saint Kitts and Nevis (KNA)" },
     { DIS_PDU_COUNTRY_SAINT_HELENA_ASCENSION_AND_TRISTAN_DA_CUNHA_SHN, "Saint Helena, Ascension and Tristan da Cunha (SHN)" },
     { DIS_PDU_COUNTRY_SAINT_LUCIA_LCA, "Saint Lucia (LCA)" },
     { DIS_PDU_COUNTRY_SAINT_PIERRE_AND_MIQUELON_SPM, "Saint Pierre and Miquelon (SPM)" },
     { DIS_PDU_COUNTRY_SAINT_VINCENT_AND_THE_GRENADINES_VCT, "Saint Vincent and the Grenadines (VCT)" },
     { DIS_PDU_COUNTRY_SAN_MARINO_SMR, "San Marino (SMR)" },
     { DIS_PDU_COUNTRY_SAO_TOME_AND_PRINCIPE_STP, "Sao Tome and Principe (STP)" },
     { DIS_PDU_COUNTRY_SAUDI_ARABIA_SAU, "Saudi Arabia (SAU)" },
     { DIS_PDU_COUNTRY_SENEGAL_SEN, "Senegal (SEN)" },
     { DIS_PDU_COUNTRY_SEYCHELLES_SYC, "Seychelles (SYC)" },
     { DIS_PDU_COUNTRY_SIERRA_LEONE_SLE, "Sierra Leone (SLE)" },
     { DIS_PDU_COUNTRY_SINGAPORE_SGP, "Singapore (SGP)" },
     { DIS_PDU_COUNTRY_SOLOMON_ISLANDS_SLB, "Solomon Islands (SLB)" },
     { DIS_PDU_COUNTRY_SOMALIA_SOM, "Somalia (SOM)" },
     { DIS_PDU_COUNTRY_SOUTH_GEORGIA_AND_THE_SOUTH_SANDWICH_ISLANDS_SGS, "South Georgia and the South Sandwich Islands (SGS)" },
     { DIS_PDU_COUNTRY_SOUTH_AFRICA_ZAF, "South Africa (ZAF)" },
     { DIS_PDU_COUNTRY_SPAIN_ESP, "Spain (ESP)" },
     { DIS_PDU_COUNTRY_SRI_LANKA_LKA, "Sri Lanka (LKA)" },
     { DIS_PDU_COUNTRY_SUDAN_SDN, "Sudan (SDN)" },
     { DIS_PDU_COUNTRY_SURINAME_SUR, "Suriname (SUR)" },
     { DIS_PDU_COUNTRY_ESWATINI_SWZ, "Eswatini (SWZ)" },
     { DIS_PDU_COUNTRY_SWEDEN_SWE, "Sweden (SWE)" },
     { DIS_PDU_COUNTRY_SWITZERLAND_CHE, "Switzerland (CHE)" },
     { DIS_PDU_COUNTRY_SYRIAN_ARAB_REPUBLIC_SYR, "Syrian Arab Republic (SYR)" },
     { DIS_PDU_COUNTRY_TAIWAN_PROVINCE_OF_CHINA_TWN, "Taiwan, Province of China (TWN)" },
     { DIS_PDU_COUNTRY_TANZANIA_UNITED_REPUBLIC_OF_TZA, "Tanzania, United Republic of (TZA)" },
     { DIS_PDU_COUNTRY_THAILAND_THA, "Thailand (THA)" },
     { DIS_PDU_COUNTRY_TOGO_TGO, "Togo (TGO)" },
     { DIS_PDU_COUNTRY_TOKELAU_TKL, "Tokelau (TKL)" },
     { DIS_PDU_COUNTRY_TONGA_TON, "Tonga (TON)" },
     { DIS_PDU_COUNTRY_TRINIDAD_AND_TOBAGO_TTO, "Trinidad and Tobago (TTO)" },
     { DIS_PDU_COUNTRY_PALAU_PLW, "Palau (PLW)" },
     { DIS_PDU_COUNTRY_TUNISIA_TUN, "Tunisia (TUN)" },
     { DIS_PDU_COUNTRY_TURKEY_TUR, "Turkey (TUR)" },
     { DIS_PDU_COUNTRY_TURKS_AND_CAICOS_ISLANDS_TCA, "Turks and Caicos Islands (TCA)" },
     { DIS_PDU_COUNTRY_TUVALU_TUV, "Tuvalu (TUV)" },
     { DIS_PDU_COUNTRY_UGANDA_UGA, "Uganda (UGA)" },
     { DIS_PDU_COUNTRY_RUSSIA_RUS, "Russia (RUS)" },
     { DIS_PDU_COUNTRY_UNITED_ARAB_EMIRATES_ARE, "United Arab Emirates (ARE)" },
     { DIS_PDU_COUNTRY_UNITED_KINGDOM_OF_GREAT_BRITAIN_AND_NORTHERN_IRELAND_GBR, "United Kingdom of Great Britain and Northern Ireland (GBR)" },
     { DIS_PDU_COUNTRY_UNITED_STATES_OF_AMERICA_USA, "United States of America (USA)" },
     { DIS_PDU_COUNTRY_URUGUAY_URY, "Uruguay (URY)" },
     { DIS_PDU_COUNTRY_VANUATU_VUT, "Vanuatu (VUT)" },
     { DIS_PDU_COUNTRY_HOLY_SEE_VAT, "Holy See (VAT)" },
     { DIS_PDU_COUNTRY_VENEZUELA_BOLIVARIAN_REPUBLIC_OF_VEN, "Venezuela (Bolivarian Republic of) (VEN)" },
     { DIS_PDU_COUNTRY_VIET_NAM_VNM, "Viet Nam (VNM)" },
     { DIS_PDU_COUNTRY_VIRGIN_ISLANDS_U_S_VIR, "Virgin Islands (U.S.) (VIR)" },
     { DIS_PDU_COUNTRY_WALLIS_AND_FUTUNA_WLF, "Wallis and Futuna (WLF)" },
     { DIS_PDU_COUNTRY_WESTERN_SAHARA_ESH, "Western Sahara (ESH)" },
     { DIS_PDU_COUNTRY_SAMOA_WSM, "Samoa (WSM)" },
     { DIS_PDU_COUNTRY_YEMEN_YEM, "Yemen (YEM)" },
     { DIS_PDU_COUNTRY_ZAMBIA_ZMB, "Zambia (ZMB)" },
     { DIS_PDU_COUNTRY_ZIMBABWE_ZWE, "Zimbabwe (ZWE)" },
     { DIS_PDU_COUNTRY_ARMENIA_ARM, "Armenia (ARM)" },
     { DIS_PDU_COUNTRY_AZERBAIJAN_AZE, "Azerbaijan (AZE)" },
     { DIS_PDU_COUNTRY_BELARUS_BLR, "Belarus (BLR)" },
     { DIS_PDU_COUNTRY_BOSNIA_AND_HERZEGOVINA_BIH, "Bosnia and Herzegovina (BIH)" },
     { DIS_PDU_COUNTRY_CROATIA_HRV, "Croatia (HRV)" },
     { DIS_PDU_COUNTRY_ESTONIA_EST, "Estonia (EST)" },
     { DIS_PDU_COUNTRY_GEORGIA_GEO, "Georgia (GEO)" },
     { DIS_PDU_COUNTRY_KAZAKHSTAN_KAZ, "Kazakhstan (KAZ)" },
     { DIS_PDU_COUNTRY_KYRGYZSTAN_KGZ, "Kyrgyzstan (KGZ)" },
     { DIS_PDU_COUNTRY_LATVIA_LVA, "Latvia (LVA)" },
     { DIS_PDU_COUNTRY_LITHUANIA_LTU, "Lithuania (LTU)" },
     { DIS_PDU_COUNTRY_NORTH_MACEDONIA_MKD, "North Macedonia (MKD)" },
     { DIS_PDU_COUNTRY_MOLDOVA_REPUBLIC_OF_MDA, "Moldova (Republic of) (MDA)" },
     { DIS_PDU_COUNTRY_MONTENEGRO_MNE, "Montenegro (MNE)" },
     { DIS_PDU_COUNTRY_SLOVENIA_SVN, "Slovenia (SVN)" },
     { DIS_PDU_COUNTRY_TAJIKISTAN_TJK, "Tajikistan (TJK)" },
     { DIS_PDU_COUNTRY_TURKMENISTAN_TKM, "Turkmenistan (TKM)" },
     { DIS_PDU_COUNTRY_UKRAINE_UKR, "Ukraine (UKR)" },
     { DIS_PDU_COUNTRY_UZBEKISTAN_UZB, "Uzbekistan (UZB)" },
     { DIS_PDU_COUNTRY_CZECH_REPUBLIC_CZE, "Czech Republic (CZE)" },
     { DIS_PDU_COUNTRY_SLOVAKIA_SVK, "Slovakia (SVK)" },
     { DIS_PDU_COUNTRY_AALAND_ISLANDS_ALA, "Aaland Islands (ALA)" },
     { DIS_PDU_COUNTRY_BONAIRE_SINT_EUSTATIUS_AND_SABA_BES, "Bonaire, Sint Eustatius and Saba (BES)" },
     { DIS_PDU_COUNTRY_CONGO_DEMOCRATIC_REPUBLIC_OF_THE_COD, "Congo (Democratic Republic of the) (COD)" },
     { DIS_PDU_COUNTRY_CURACAO_CUW, "Curacao (CUW)" },
     { DIS_PDU_COUNTRY_ERITREA_ERI, "Eritrea (ERI)" },
     { DIS_PDU_COUNTRY_SAINT_BARTHELEMY_BLM, "Saint Barthelemy (BLM)" },
     { DIS_PDU_COUNTRY_SAINT_MARTIN_FRENCH_PART_MAF, "Saint Martin (French Part) (MAF)" },
     { DIS_PDU_COUNTRY_SERBIA_SRB, "Serbia (SRB)" },
     { DIS_PDU_COUNTRY_SINT_MAARTEN_DUTCH_PART_SXM, "Sint Maarten (Dutch part) (SXM)" },
     { DIS_PDU_COUNTRY_SOUTH_SUDAN_SSD, "South Sudan (SSD)" },
     { DIS_PDU_COUNTRY_SVALBARD_AND_JAN_MAYEN_SJM, "Svalbard and Jan Mayen (SJM)" },
     { DIS_PDU_COUNTRY_TIMOR_LESTE_TLS, "Timor-Leste (TLS)" },
     { DIS_PDU_COUNTRY_UNITED_STATES_MINOR_OUTLYING_ISLANDS_UMI, "United States Minor Outlying Islands (UMI)" },
     { DIS_PDU_COUNTRY_PALESTINE_STATE_OF_PSE, "Palestine, State of (PSE)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 44] */
typedef enum
{
     DIS_PDU_DEAD_RECKONING_ALGORITHM_OTHER = 0,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_STATIC_NON_MOVING_ENTITY = 1,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_FPW_CONSTANT_VELOCITY_LOW_ACCELERATION_LINEAR_MOTION_ENTITY = 2,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_RPW_CONSTANT_VELOCITY_LOW_ACCELERATION_LINEAR_MOTION_ENTITY_WITH_EXTRAPOLATION_OF_ORIENTATION = 3,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_RVW_HIGH_SPEED_OR_MANEUVERING_ENTITY_WITH_EXTRAPOLATION_OF_ORIENTATION = 4,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_FVW_HIGH_SPEED_OR_MANEUVERING_ENTITY = 5,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_FPB_SIMILAR_TO_FPW_EXCEPT_IN_BODY_COORDINATES = 6,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_RPB_SIMILAR_TO_RPW_EXCEPT_IN_BODY_COORDINATES = 7,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_RVB_SIMILAR_TO_RVW_EXCEPT_IN_BODY_COORDINATES = 8,
     DIS_PDU_DEAD_RECKONING_ALGORITHM_DRM_FVB_SIMILAR_TO_FVW_EXCEPT_IN_BODY_COORDINATES = 9
}  DIS_PDU_Dead_Reckoning_Algorithm;

/* SISO-REF-010 [UID 45] */
typedef enum
{
     DIS_PDU_ENTITY_MARKING_CHARACTER_SET_UNUSED = 0,
     DIS_PDU_ENTITY_MARKING_CHARACTER_SET_ASCII = 1,
     DIS_PDU_ENTITY_MARKING_CHARACTER_SET_U_S_ARMY_MARKING = 2,
     DIS_PDU_ENTITY_MARKING_CHARACTER_SET_DIGIT_CHEVRON = 3
}  DIS_PDU_Entity_Marking_Character_Set;

static const value_string DIS_PDU_Entity_Marking_Character_Set_Strings[] = {
     { DIS_PDU_ENTITY_MARKING_CHARACTER_SET_UNUSED, "Unused" },
     { DIS_PDU_ENTITY_MARKING_CHARACTER_SET_ASCII, "ASCII" },
     { DIS_PDU_ENTITY_MARKING_CHARACTER_SET_U_S_ARMY_MARKING, "U.S. Army Marking" },
     { DIS_PDU_ENTITY_MARKING_CHARACTER_SET_DIGIT_CHEVRON, "Digit Chevron" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 55] */
typedef enum
{
     DIS_PDU_ENTITY_CAPABILITIES_LAND_PLATFORM_ENTITY_CAPABILITIES = 0,
     DIS_PDU_ENTITY_CAPABILITIES_AIR_PLATFORM_ENTITY_CAPABILITIES = 1,
     DIS_PDU_ENTITY_CAPABILITIES_SURFACE_PLATFORM_ENTITY_CAPABILITIES = 2,
     DIS_PDU_ENTITY_CAPABILITIES_SUBSURFACE_PLATFORM_ENTITY_CAPABILITIES = 3,
     DIS_PDU_ENTITY_CAPABILITIES_SPACE_PLATFORM_ENTITY_CAPABILITIES = 4,
     DIS_PDU_ENTITY_CAPABILITIES_MUNITION_ENTITY_CAPABILITIES = 5,
     DIS_PDU_ENTITY_CAPABILITIES_LIFE_FORMS_ENTITY_CAPABILITIES = 6,
     DIS_PDU_ENTITY_CAPABILITIES_ENVIRONMENTAL_ENTITY_CAPABILITIES = 7,
     DIS_PDU_ENTITY_CAPABILITIES_CULTURAL_FEATURE_ENTITY_CAPABILITIES = 8,
     DIS_PDU_ENTITY_CAPABILITIES_SUPPLY_ENTITY_CAPABILITIES = 9,
     DIS_PDU_ENTITY_CAPABILITIES_RADIO_ENTITY_CAPABILITIES = 10,
     DIS_PDU_ENTITY_CAPABILITIES_EXPENDABLE_ENTITY_CAPABILITIES = 11,
     DIS_PDU_ENTITY_CAPABILITIES_SENSOR_EMITTER_ENTITY_CAPABILITIES = 12
}  DIS_PDU_Entity_Capabilities;

/* SISO-REF-010 [UID 56] */
typedef enum
{
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ARTICULATED_PART = 0,
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ATTACHED_PART = 1,
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_SEPARATION = 2,
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ENTITY_TYPE = 3,
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ENTITY_ASSOCIATION = 4,
     /* Manually inserted "Experimental" non-SISO definitions begin */
     DIS_PARAM_TYPE_DESIG_ANTENNA_LOCATION                        =  5,
     DIS_PARAM_TYPE_DESIG_EXTENDED_PLATFORM_APPEARANCE            = 20,
     DIS_PARAM_TYPE_DESIG_EXTENDED_LIFEFORM_APPEARANCE            = 21,
     DIS_PARAM_TYPE_DESIG_HIGH_FIDELITY_LIGHTS                    = 22,
     DIS_PARAM_TYPE_DESIG_CHEVRON_MARKING                         = 23,
     DIS_PARAM_TYPE_DESIG_HIGH_FIDELITY_THERMAL_SENSOR            = 24,
     DIS_PARAM_TYPE_DESIG_ENTITY_OFFSET                           = 25,
     DIS_PARAM_TYPE_DESIG_DEAD_RECKONING                          = 26,
     DIS_PARAM_TYPE_DESIG_ARMY_TASK_ORGANIZATION                  = 27,
     DIS_PARAM_TYPE_DESIG_HEAD_GAZING_WEAPON_AIMING               = 28,
     DIS_PARAM_TYPE_DESIG_LIFEFORM_ACTION_SEQUENCE                = 29,
     DIS_PARAM_TYPE_DESIG_LEGACY_EXTENDED_LIFEFORM_APPEARANCE     = 30,
     /* Manually inserted "Experimental" non-SISO definitions end */
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_MAX_FUEL_FLOW_RATE = 36,
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_VIRTUAL_BOOM_INFORMATION = 37,
     DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_AERIAL_REFUELING_LATCH_STATE = 38,
}  DIS_PDU_Variable_Parameter_Record_Type;

static const value_string DIS_PDU_Variable_Parameter_Record_Type_Strings[] = {
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ARTICULATED_PART, "Articulated Part" },
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ATTACHED_PART, "Attached Part" },
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_SEPARATION, "Separation" },
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ENTITY_TYPE, "Entity Type" },
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_ENTITY_ASSOCIATION, "Entity Association" },
     /* Manually inserted "Experimental" non-SISO definitions begin */
     { DIS_PARAM_TYPE_DESIG_ANTENNA_LOCATION,                        "Antenna Location" },
     { DIS_PARAM_TYPE_DESIG_EXTENDED_PLATFORM_APPEARANCE,            "Extended Platform Appearance" },
     { DIS_PARAM_TYPE_DESIG_EXTENDED_LIFEFORM_APPEARANCE,            "Extended Lifeform Appearance" },
     { DIS_PARAM_TYPE_DESIG_HIGH_FIDELITY_LIGHTS,                    "High Fidelity Lights" },
     { DIS_PARAM_TYPE_DESIG_CHEVRON_MARKING,                         "Chevron Marking" },
     { DIS_PARAM_TYPE_DESIG_HIGH_FIDELITY_THERMAL_SENSOR,            "High Fidelity Thermal Sensor" },
     { DIS_PARAM_TYPE_DESIG_ENTITY_OFFSET,                           "Entity Offset" },
     { DIS_PARAM_TYPE_DESIG_DEAD_RECKONING,                          "Dead Reckoning" },
     { DIS_PARAM_TYPE_DESIG_ARMY_TASK_ORGANIZATION,                  "Army Task Organization" },
     { DIS_PARAM_TYPE_DESIG_HEAD_GAZING_WEAPON_AIMING,               "Head Gazing Weapon Aiming" },
     { DIS_PARAM_TYPE_DESIG_LIFEFORM_ACTION_SEQUENCE,                "Lifeform Action Sequence" },
     { DIS_PARAM_TYPE_DESIG_LEGACY_EXTENDED_LIFEFORM_APPEARANCE,     "Legacy Extended Lifeform Appearance" },
     /* Manually inserted "Experimental" non-SISO definitions end */
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_MAX_FUEL_FLOW_RATE, "Max Fuel Flow Rate" },
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_VIRTUAL_BOOM_INFORMATION, "Virtual Boom Information" },
     { DIS_PDU_VARIABLE_PARAMETER_RECORD_TYPE_AERIAL_REFUELING_LATCH_STATE, "Aerial Refueling Latch State" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 57] */
typedef enum
{
     DIS_PDU_ATTACHED_PARTS_NOTHING_EMPTY = 0,
     DIS_PDU_ATTACHED_PARTS_M16A42_RIFLE = 896,
     DIS_PDU_ATTACHED_PARTS_M249_SAW = 897,
     DIS_PDU_ATTACHED_PARTS_M60_MACHINE_GUN = 898,
     DIS_PDU_ATTACHED_PARTS_M203_GRENADE_LAUNCHER = 899,
     DIS_PDU_ATTACHED_PARTS_M136_AT4 = 900,
     DIS_PDU_ATTACHED_PARTS_M47_DRAGON = 901,
     DIS_PDU_ATTACHED_PARTS_AAWS_M_JAVELIN = 902,
     DIS_PDU_ATTACHED_PARTS_M18A1_CLAYMORE_MINE = 903,
     DIS_PDU_ATTACHED_PARTS_MK19_GRENADE_LAUNCHER = 904,
     DIS_PDU_ATTACHED_PARTS_M2_MACHINE_GUN = 905
}  DIS_PDU_Attached_Parts;

/* SISO-REF-010 [UID 58] */
typedef enum
{
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_NOT_SPECIFIED = 0,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_POSITION = 1,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_POSITION_RATE = 2,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_EXTENSION = 3,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_EXTENSION_RATE = 4,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_X = 5,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_X_RATE = 6,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_Y = 7,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_Y_RATE = 8,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_Z = 9,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_Z_RATE = 10,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_AZIMUTH = 11,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_AZIMUTH_RATE = 12,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_ELEVATION = 13,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_ELEVATION_RATE = 14,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_ROTATION = 15,
     DIS_PDU_ARTICULATED_PARTS_TYPE_METRIC_ROTATION_RATE = 16
}  DIS_PDU_Articulated_Parts_Type_Metric;

/* SISO-REF-010 [UID 59] */
typedef enum
{
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_NOT_SPECIFIED = 0,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RUDDER = 1024,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_FLAP = 1056,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_FLAP = 1088,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_AILERON = 1120,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_AILERON = 1152,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_HELICOPTER_MAIN_ROTOR = 1184,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_HELICOPTER_TAIL_ROTOR = 1216,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_OTHER_AIRCRAFT_CONTROL_SURFACES_DEFINED_AS_NEEDED = 1248,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PROPELLER_NUMBER_1 = 1280,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PROPELLER_NUMBER_2 = 1312,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PROPELLER_NUMBER_3 = 1344,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PROPELLER_NUMBER_4 = 1376,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_STABILATOR_STABILATOR_NUMBER_1 = 1408,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_STABILATOR_STABILATOR_NUMBER_2 = 1440,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_RUDDERVATOR_RUDDERVATOR_NUMBER_1 = 1472,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_RUDDERVATOR_RUDDERVATOR_NUMBER_2 = 1504,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_LEADING_EDGE_FLAP_SLAT = 1536,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_LEADING_EDGE_FLAP_SLAT = 1568,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_ELEVATOR = 1600,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_ELEVATOR = 1632,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CANARD_LEFT = 1664,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CANARD_RIGHT = 1696,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ELEVON_INNER_LEFT = 1728,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ELEVON_INNER_RIGHT = 1760,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ELEVON_MIDDLE_LEFT = 1792,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ELEVON_MIDDLE_RIGHT = 1824,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ELEVON_OUTER_LEFT = 1856,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ELEVON_OUTER_RIGHT = 1888,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CANOPY_AIRCRAFT = 1920,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SPOILER_LEFT = 1952,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SPOILER_RIGHT = 1984,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PERISCOPE = 2048,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_GENERIC_ANTENNA = 2080,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SNORKEL = 2112,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_OTHER_EXTENDIBLE_PARTS_DEFINED_AS_NEEDED = 2144,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DIVE_PLANE_SAIL_LEFT = 2176,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DIVE_PLANE_SAIL_RIGHT = 2208,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DIVE_PLANE_BOW_LEFT = 2240,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DIVE_PLANE_BOW_RIGHT = 2272,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DIVE_PLANE_STERN_LEFT = 2304,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DIVE_PLANE_STERN_RIGHT = 2336,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_RUDDER = 2368,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_RUDDER = 2400,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_GEAR_DOOR_NOSE = 2976,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_GEAR_DOOR_LEFT_MAIN = 3008,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_GEAR_DOOR_RIGHT_MAIN = 3040,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_GEAR = 3072,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_TAIL_HOOK = 3104,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SPEED_BRAKE = 3136,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_DOOR_OF_PRIMARY_WEAPON_BAY = 3168,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_DOOR_OF_PRIMARY_WEAPON_BAY = 3200,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_TANK_OR_APC_HATCH = 3232,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_WINGSWEEP = 3264,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_BRIDGE_LAUNCHER = 3296,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_BRIDGE_SECTION_1 = 3328,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_BRIDGE_SECTION_2 = 3360,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_BRIDGE_SECTION_3 = 3392,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_BLADE_1 = 3424,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_BLADE_2 = 3456,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_BOOM = 3488,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_ARM = 3520,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_OTHER_FIXED_POSITION_PARTS_DEFINED_AS_NEEDED = 3552,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_GEAR_NOSE = 3584,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_GEAR_LEFT_MAIN = 3616,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_GEAR_RIGHT_MAIN = 3648,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DOORS_OF_LEFT_SIDE_WEAPON_BAY = 3680,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DOORS_OF_RIGHT_SIDE_WEAPON_BAY = 3712,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SPOT_SEARCH_LIGHT_1 = 3744,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SPOT_SEARCH_LIGHT_2 = 3776,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SPOT_SEARCH_LIGHT_3 = 3808,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SPOT_SEARCH_LIGHT_4 = 3840,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LANDING_LIGHT = 3872,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_1 = 4096,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_2 = 4128,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_3 = 4160,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_4 = 4192,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_5 = 4224,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_6 = 4256,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_7 = 4288,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_8 = 4320,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_9 = 4352,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_TURRET_NUMBER_10 = 4384,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_1 = 4416,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_2 = 4448,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_3 = 4480,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_4 = 4512,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_5 = 4544,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_6 = 4576,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_7 = 4608,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_8 = 4640,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_9 = 4672,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_NUMBER_10 = 4704,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_1 = 4736,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_2 = 4768,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_3 = 4800,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_4 = 4832,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_5 = 4864,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_6 = 4896,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_7 = 4928,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_8 = 4960,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_9 = 4992,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_LAUNCHER_10 = 5024,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_1 = 5056,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_2 = 5088,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_3 = 5120,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_4 = 5152,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_5 = 5184,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_6 = 5216,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_7 = 5248,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_8 = 5280,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_9 = 5312,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_10 = 5344,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_1 = 5376,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_2 = 5408,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_3 = 5440,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_4 = 5472,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_5 = 5504,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_6 = 5536,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_7 = 5568,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_8 = 5600,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_9 = 5632,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_RADAR_10 = 5664,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_1 = 5696,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_2 = 5728,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_3 = 5760,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_4 = 5792,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_5 = 5824,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_6 = 5856,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_7 = 5888,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_8 = 5920,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_9 = 5952,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_TURRET_NUMBER_10 = 5984,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_1 = 6016,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_2 = 6048,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_3 = 6080,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_4 = 6112,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_5 = 6144,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_6 = 6176,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_7 = 6208,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_8 = 6240,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_9 = 6272,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_NUMBER_10 = 6304,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_1 = 6336,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_2 = 6368,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_3 = 6400,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_4 = 6432,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_5 = 6464,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_6 = 6496,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_7 = 6528,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_8 = 6560,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_9 = 6592,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_LAUNCHER_10 = 6624,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_1 = 6656,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_2 = 6688,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_3 = 6720,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_4 = 6752,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_5 = 6784,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_6 = 6816,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_7 = 6848,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_8 = 6880,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_9 = 6912,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_10 = 6944,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_1 = 6976,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_2 = 7008,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_3 = 7040,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_4 = 7072,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_5 = 7104,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_6 = 7136,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_7 = 7168,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_8 = 7200,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_9 = 7232,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_RADAR_10 = 7264,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DECK_ELEVATOR_1 = 7296,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_DECK_ELEVATOR_2 = 7328,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CATAPULT_1 = 7360,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CATAPULT_2 = 7392,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_JET_BLAST_DEFLECTOR_1 = 7424,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_JET_BLAST_DEFLECTOR_2 = 7456,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ARRESTOR_WIRES_1 = 7488,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ARRESTOR_WIRES_2 = 7520,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ARRESTOR_WIRES_3 = 7552,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_WING_OR_ROTOR_FOLD = 7584,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_FUSELAGE_FOLD = 7616,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_MAIN_CARGO_DOOR = 7648,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_RAMP = 7680,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_AIR_TO_AIR_REFUELING_BOOM = 7712,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_AERIAL_REFUELING_RECEPTACLE_DOOR = 7744,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_AERIAL_REFUELING_RECEPTACLE_DOOR = 7776,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_AERIAL_REFUELING_RECEPTACLE_LATCH = 7808,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_1 = 7840,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_2 = 7872,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_3 = 7904,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_4 = 7936,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_5 = 7968,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_6 = 8000,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_7 = 8032,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_8 = 8064,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_9 = 8096,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CARGO_DOOR_10 = 8128,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CENTRE_REFUELLING_DROGUE = 8160,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PORT_REFUELLING_DROGUE = 8192,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_STARBOARD_REFUELLING_DROGUE = 8224,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_ENGINE_EXHAUST_MAST = 8256,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_1 = 8288,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_2 = 8320,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_3 = 8352,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_4 = 8384,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_5 = 8416,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_6 = 8448,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_7 = 8480,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_8 = 8512,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_9 = 8544,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SUBMARINE_MAST_10 = 8576,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_VECTORED_THRUST_NOZZLE = 8608,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_DOOR_OF_THE_LEFT_WEAPON_BAY = 8640,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_DOOR_OF_THE_LEFT_WEAPON_BAY = 8672,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_DOOR_OF_THE_RIGHT_WEAPON_BAY = 8704,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_DOOR_OF_THE_RIGHT_WEAPON_BAY = 8736,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_GUN_DOOR = 8768,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_COUNTERMEASURE_DOOR_LEFT = 8800,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_COUNTERMEASURE_DOOR_RIGHT = 8832,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_HOOK_DOOR_FORWARD = 8864,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_HOOK_DOOR_AFT = 8896,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LIFT_FAN_UPPER_DOOR = 8928,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LIFT_FAN_LOWER_DOOR_LEFT = 8960,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LIFT_FAN_LOWER_DOOR_RIGHT = 8992,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_REFUEL_PROBE_DOOR = 9024,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_LEFT_ENGINE_NACELLE = 9056,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_RIGHT_ENGINE_NACELLE = 9088,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_1ST_LEFT_WHEEL = 9120,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_1ST_RIGHT_WHEEL = 9152,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_2ND_LEFT_WHEEL = 9184,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_2ND_RIGHT_WHEEL = 9216,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_3RD_LEFT_WHEEL = 9248,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_3RD_RIGHT_WHEEL = 9280,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_4TH_LEFT_WHEEL = 9312,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_4TH_RIGHT_WHEEL = 9344,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_5TH_LEFT_WHEEL = 9376,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_5TH_RIGHT_WHEEL = 9408,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_6TH_LEFT_WHEEL = 9440,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_6TH_RIGHT_WHEEL = 9472,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_7TH_LEFT_WHEEL = 9504,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_7TH_RIGHT_WHEEL = 9536,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_8TH_LEFT_WHEEL = 9568,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_8TH_RIGHT_WHEEL = 9600,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_9TH_LEFT_WHEEL = 9632,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_9TH_RIGHT_WHEEL = 9664,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_10TH_LEFT_WHEEL = 9696,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_10TH_RIGHT_WHEEL = 9728,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_REFUELING_PROBE = 9760,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_STEERING_WHEEL = 9792,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CRANE_BODY = 9824,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CRANE_ARM_1 = 9856,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CRANE_ARM_2 = 9888,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CRANE_ARM_3 = 9920,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CRANE_BOOM = 9952,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_CRANE_HOOK = 9984,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_TRAILER = 10016,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ROLLER_LEFT = 10048,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_ROLLER_RIGHT = 10080,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_PRIMARY_GUN_RECOIL = 10112,
     DIS_PDU_ARTICULATED_PARTS_TYPE_CLASS_SECONDARY_GUN_RECOIL = 10144
}  DIS_PDU_Articulated_Parts_Type_Class;

/* SISO-REF-010 [UID 60] */
typedef enum
{
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_OTHER = 0000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CARGO_VARIABLE_SUBMUNITIONS = 0010,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_FUEL_AIR_EXPLOSIVE = 0020,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_GLASS_BEADS = 0030,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_1_UM = 0031,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_5_UM = 0032,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_10_UM = 0033,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HIGH_EXPLOSIVE_HE = 1000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_PLASTIC = 1100,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_INCENDIARY = 1200,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_FRAGMENTATION = 1300,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_ANTI_TANK = 1400,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_BOMBLETS = 1500,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_SHAPED_CHARGE = 1600,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_CONTINUOUS_ROD = 1610,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_TUNGSTEN_BALL = 1615,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_BLAST_FRAGMENTATION = 1620,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_STEERABLE_DARTS_WITH_HE = 1625,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_DARTS = 1630,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_FLECHETTES = 1635,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_DIRECTED_FRAGMENTATION = 1640,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_SEMI_ARMOR_PIERCING_SAP = 1645,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_SHAPED_CHARGE_FRAGMENTATION = 1650,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_SEMI_ARMOR_PIERCING_FRAGMENTATION = 1655,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_HOLLOW_CHARGE = 1660,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_DOUBLE_HOLLOW_CHARGE = 1665,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_GENERAL_PURPOSE = 1670,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_BLAST_PENETRATOR = 1675,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_ROD_PENETRATOR = 1680,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_ANTI_PERSONNEL = 1685,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_SHAPED_CHARGE_FRAGMENTATION_INCENDIARY = 1690,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HE_PENETRATOR_BLAST_FRAGMENTATION = 1695,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_SMOKE = 2000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_WP_WHITE_PHOSPHORUS = 2005,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_FOGO_FOG_OIL = 2010,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HC_HEXACHLOROETHANE = 2015,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_ILLUMINATION = 3000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_PRACTICE = 4000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BLANK = 4001,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DUMMY = 4002,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_KINETIC = 5000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_MINES = 6000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_NUCLEAR = 7000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_NUCLEAR_IMT = 7010,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CHEMICAL_GENERAL = 8000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CHEMICAL_BLISTER_AGENT = 8100,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HD_MUSTARD = 8110,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_THICKENED_HD_MUSTARD = 8115,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DUSTY_HD_MUSTARD = 8120,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_L_LEWISITE = 8125,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HN3_NITROGEN_MUSTARD = 8130,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HL_MUSTARD_LEWISITE = 8135,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CX_PHOSGENE_OXIME = 8140,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DMMP_PHOSPHATE_DIMETHYL_HYDROGEN = 8145,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DMHP_PHOSPHITE = 8150,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DMA_DIMETHYL_ACRYLATE = 8155,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DEM = 8160,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_PX_P_XLENE = 8165,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CHEMICAL_BLOOD_AGENT = 8200,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_AC_HCN = 8210,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CK_CNCI = 8215,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CG_PHOSGENE = 8220,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_CHEMICAL_NERVE_AGENT = 8300,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_VX = 8310,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_THICKENED_VX = 8315,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DUSTY_VX = 8320,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_GA_TABUN = 8325,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_THICKENED_GA_TABUN = 8330,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DUSTY_GA_TABUN = 8335,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_GB_SARIN = 8340,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_THICKENED_GB_SARIN = 8345,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DUSTY_GB_SARIN = 8350,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_GD_SOMAN = 8355,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_THICKENED_GD_SOMAN = 8360,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DUSTY_GD_SOMAN = 8365,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_GF = 8370,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_THICKENED_GF = 8375,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_DUSTY_GF = 8380,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_SVX_SOVIET_VX = 8385,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BIS = 8410,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_TCP = 8415,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_MS_METHYL_SALICYLATE = 8425,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_TEP = 8430,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_H2O_WATER = 8445,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_TO1_TOXIC_ORGANIC_1 = 8450,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_TO2_TOXIC_ORGANIC_2 = 8455,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_TO3_TOXIC_ORGANIC_3 = 8460,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_SULFUR_HEXAFLUORIDE = 8465,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_AA_ACETIC_ACID = 8470,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_HF_HYDROFLUORIC_ACID = 8475,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BIOLOGICAL = 9000,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BIOLOGICAL_VIRUS = 9100,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BIOLOGICAL_BACTERIA = 9200,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BIOLOGICAL_RICKETTSIA = 9300,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BIOLOGICAL_GENETICALLY_MODIFIED_MICRO_ORGANISMS = 9400,
     DIS_PDU_MUNITION_DESCRIPTOR_WARHEAD_BIOLOGICAL_TOXIN = 9500
}  DIS_PDU_Munition_Descriptor_Warhead;

/* SISO-REF-010 [UID 61] */
typedef enum
{
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_OTHER = 0000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_INTELLIGENT_INFLUENCE = 0010,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_SENSOR = 0020,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_SELF_DESTRUCT = 0030,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ULTRA_QUICK = 0040,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_BODY = 0050,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_DEEP_INTRUSION = 0060,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_MULTIFUNCTION = 0100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_POINT_DETONATION_PD = 0200,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_BASE_DETONATION_BD = 0300,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT = 1000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_INSTANT_IMPACT = 1100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_DELAYED = 1200,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_10_MS_DELAY = 1201,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_20_MS_DELAY = 1202,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_50_MS_DELAY = 1205,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_60_MS_DELAY = 1206,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_100_MS_DELAY = 1210,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_125_MS_DELAY = 1212,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_250_MS_DELAY = 1225,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_5_MS_DELAY = 1250,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_15_MS_DELAY = 1251,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_25_MS_DELAY = 1252,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_30_MS_DELAY = 1253,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_35_MS_DELAY = 1254,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_40_MS_DELAY = 1255,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_45_MS_DELAY = 1256,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_90_MS_DELAY = 1257,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_120_MS_DELAY = 1258,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_180_MS_DELAY = 1259,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_240_MS_DELAY = 1260,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_ELECTRONIC_OBLIQUE_CONTACT = 1300,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_GRAZE = 1400,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_CRUSH = 1500,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_HYDROSTATIC = 1600,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_MECHANICAL = 1700,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_CHEMICAL = 1800,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_PIEZOELECTRIC = 1900,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_POINT_INITIATING = 1910,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_POINT_INITIATING_BASE_DETONATING = 1920,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_BASE_DETONATING = 1930,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_BALLISTIC_CAP_AND_BASE = 1940,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_BASE = 1950,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_NOSE = 1960,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_FITTED_IN_STANDOFF_PROBE = 1970,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_CONTACT_NON_ALIGNED = 1980,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED = 2000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_PROGRAMMABLE = 2100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_BURNOUT = 2200,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_PYROTECHNIC = 2300,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_ELECTRONIC = 2400,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_BASE_DELAY = 2500,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_REINFORCED_NOSE_IMPACT_DELAY = 2600,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_SHORT_DELAY_IMPACT = 2700,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_NOSE_MOUNTED_VARIABLE_DELAY = 2800,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_LONG_DELAY_SIDE = 2900,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_SELECTABLE_DELAY = 2910,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_IMPACT = 2920,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TIMED_SEQUENCE = 2930,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY = 3000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY_ACTIVE_LASER = 3100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY_MAGNETIC_MAGPOLARITY = 3200,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY_ACTIVE_RADAR_DOPPLER_RADAR = 3300,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY_RADIO_FREQUENCY_RF = 3400,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY_PROGRAMMABLE = 3500,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY_PROGRAMMABLE_PREFRAGMENTED = 3600,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PROXIMITY_INFRARED = 3700,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_COMMAND = 4000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_COMMAND_ELECTRONIC_REMOTELY_SET = 4100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ALTITUDE = 5000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ALTITUDE_RADIO_ALTIMETER = 5100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ALTITUDE_AIR_BURST = 5200,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_DEPTH = 6000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ACOUSTIC = 7000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PRESSURE = 8000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PRESSURE_DELAY = 8010,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_INERT = 8100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_DUMMY = 8110,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PRACTICE = 8120,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PLUG_REPRESENTING = 8130,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_TRAINING = 8150,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PYROTECHNIC = 9000,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PYROTECHNIC_DELAY = 9010,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ELECTRO_OPTICAL = 9100,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ELECTROMECHANICAL = 9110,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ELECTROMECHANICAL_NOSE = 9120,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_STRIKERLESS = 9200,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_STRIKERLESS_NOSE_IMPACT = 9210,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_STRIKERLESS_COMPRESSION_IGNITION = 9220,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_COMPRESSION_IGNITION = 9300,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_COMPRESSION_IGNITION_STRIKERLESS_NOSE_IMPACT = 9310,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PERCUSSION = 9400,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_PERCUSSION_INSTANTANEOUS = 9410,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ELECTRONIC = 9500,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ELECTRONIC_INTERNALLY_MOUNTED = 9510,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ELECTRONIC_RANGE_SETTING = 9520,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_ELECTRONIC_PROGRAMMED = 9530,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_MECHANICAL = 9600,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_MECHANICAL_NOSE = 9610,
     DIS_PDU_MUNITION_DESCRIPTOR_FUSE_MECHANICAL_TAIL = 9620
}  DIS_PDU_Munition_Descriptor_Fuse;

/* SISO-REF-010 [UID 62] */
typedef enum
{
     DIS_PDU_DETONATION_RESULT_OTHER = 0,
     DIS_PDU_DETONATION_RESULT_ENTITY_IMPACT = 1,
     DIS_PDU_DETONATION_RESULT_ENTITY_PROXIMATE_DETONATION = 2,
     DIS_PDU_DETONATION_RESULT_GROUND_IMPACT = 3,
     DIS_PDU_DETONATION_RESULT_GROUND_PROXIMATE_DETONATION = 4,
     DIS_PDU_DETONATION_RESULT_DETONATION = 5,
     DIS_PDU_DETONATION_RESULT_NONE_OR_NO_DETONATION_DUD = 6,
     DIS_PDU_DETONATION_RESULT_HE_HIT_SMALL = 7,
     DIS_PDU_DETONATION_RESULT_HE_HIT_MEDIUM = 8,
     DIS_PDU_DETONATION_RESULT_HE_HIT_LARGE = 9,
     DIS_PDU_DETONATION_RESULT_ARMOR_PIERCING_HIT = 10,
     DIS_PDU_DETONATION_RESULT_DIRT_BLAST_SMALL = 11,
     DIS_PDU_DETONATION_RESULT_DIRT_BLAST_MEDIUM = 12,
     DIS_PDU_DETONATION_RESULT_DIRT_BLAST_LARGE = 13,
     DIS_PDU_DETONATION_RESULT_WATER_BLAST_SMALL = 14,
     DIS_PDU_DETONATION_RESULT_WATER_BLAST_MEDIUM = 15,
     DIS_PDU_DETONATION_RESULT_WATER_BLAST_LARGE = 16,
     DIS_PDU_DETONATION_RESULT_AIR_HIT = 17,
     DIS_PDU_DETONATION_RESULT_BUILDING_HIT_SMALL = 18,
     DIS_PDU_DETONATION_RESULT_BUILDING_HIT_MEDIUM = 19,
     DIS_PDU_DETONATION_RESULT_BUILDING_HIT_LARGE = 20,
     DIS_PDU_DETONATION_RESULT_MINE_CLEARING_LINE_CHARGE = 21,
     DIS_PDU_DETONATION_RESULT_ENVIRONMENT_OBJECT_IMPACT = 22,
     DIS_PDU_DETONATION_RESULT_ENVIRONMENT_OBJECT_PROXIMATE_DETONATION = 23,
     DIS_PDU_DETONATION_RESULT_WATER_IMPACT = 24,
     DIS_PDU_DETONATION_RESULT_AIR_BURST = 25,
     DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_1 = 26,
     DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_2 = 27,
     DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_3 = 28,
     DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_1_AFTER_FLY_OUT_FAILURE = 29,
     DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_2_AFTER_FLY_OUT_FAILURE = 30,
     DIS_PDU_DETONATION_RESULT_MISS_DUE_TO_FLY_OUT_FAILURE = 31,
     DIS_PDU_DETONATION_RESULT_MISS_DUE_TO_END_GAME_FAILURE = 32,
     DIS_PDU_DETONATION_RESULT_MISS_DUE_TO_FLY_OUT_AND_END_GAME_FAILURE = 33
}  DIS_PDU_Detonation_Result;

static const value_string DIS_PDU_DetonationResult_Strings[] = {
     { DIS_PDU_DETONATION_RESULT_OTHER, "Other" },
     { DIS_PDU_DETONATION_RESULT_ENTITY_IMPACT, "Entity Impact" },
     { DIS_PDU_DETONATION_RESULT_ENTITY_PROXIMATE_DETONATION, "Entity Proximate Detonation" },
     { DIS_PDU_DETONATION_RESULT_GROUND_IMPACT, "Ground Impact" },
     { DIS_PDU_DETONATION_RESULT_GROUND_PROXIMATE_DETONATION, "Ground Proximate Detonation" },
     { DIS_PDU_DETONATION_RESULT_DETONATION, "Detonation" },
     { DIS_PDU_DETONATION_RESULT_NONE_OR_NO_DETONATION_DUD, "None or No Detonation (Dud)" },
     { DIS_PDU_DETONATION_RESULT_HE_HIT_SMALL, "HE hit, small" },
     { DIS_PDU_DETONATION_RESULT_HE_HIT_MEDIUM, "HE hit, medium" },
     { DIS_PDU_DETONATION_RESULT_HE_HIT_LARGE, "HE hit, large" },
     { DIS_PDU_DETONATION_RESULT_ARMOR_PIERCING_HIT, "Armor-piercing hit" },
     { DIS_PDU_DETONATION_RESULT_DIRT_BLAST_SMALL, "Dirt blast, small" },
     { DIS_PDU_DETONATION_RESULT_DIRT_BLAST_MEDIUM, "Dirt blast, medium" },
     { DIS_PDU_DETONATION_RESULT_DIRT_BLAST_LARGE, "Dirt blast, large" },
     { DIS_PDU_DETONATION_RESULT_WATER_BLAST_SMALL, "Water blast, small" },
     { DIS_PDU_DETONATION_RESULT_WATER_BLAST_MEDIUM, "Water blast, medium" },
     { DIS_PDU_DETONATION_RESULT_WATER_BLAST_LARGE, "Water blast, large" },
     { DIS_PDU_DETONATION_RESULT_AIR_HIT, "Air hit" },
     { DIS_PDU_DETONATION_RESULT_BUILDING_HIT_SMALL, "Building hit, small" },
     { DIS_PDU_DETONATION_RESULT_BUILDING_HIT_MEDIUM, "Building hit, medium" },
     { DIS_PDU_DETONATION_RESULT_BUILDING_HIT_LARGE, "Building hit, large" },
     { DIS_PDU_DETONATION_RESULT_MINE_CLEARING_LINE_CHARGE, "Mine-clearing line charge" },
     { DIS_PDU_DETONATION_RESULT_ENVIRONMENT_OBJECT_IMPACT, "Environment object impact" },
     { DIS_PDU_DETONATION_RESULT_ENVIRONMENT_OBJECT_PROXIMATE_DETONATION, "Environment object proximate detonation" },
     { DIS_PDU_DETONATION_RESULT_WATER_IMPACT, "Water Impact" },
     { DIS_PDU_DETONATION_RESULT_AIR_BURST, "Air Burst" },
     { DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_1, "Kill with fragment type 1" },
     { DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_2, "Kill with fragment type 2" },
     { DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_3, "Kill with fragment type 3" },
     { DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_1_AFTER_FLY_OUT_FAILURE, "Kill with fragment type 1 after fly-out failure" },
     { DIS_PDU_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_2_AFTER_FLY_OUT_FAILURE, "Kill with fragment type 2 after fly-out failure" },
     { DIS_PDU_DETONATION_RESULT_MISS_DUE_TO_FLY_OUT_FAILURE, "Miss due to fly-out failure" },
     { DIS_PDU_DETONATION_RESULT_MISS_DUE_TO_END_GAME_FAILURE, "Miss due to end-game failure" },
     { DIS_PDU_DETONATION_RESULT_MISS_DUE_TO_FLY_OUT_AND_END_GAME_FAILURE, "Miss due to fly-out and end-game failure" },
     { 0, NULL }
 };

static value_string_ext DIS_PDU_DetonationResult_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_DetonationResult_Strings);

/* SISO-REF-010 [UID 63] */
typedef enum
{
     DIS_PDU_SERVICE_REQUEST_SERVICE_TYPE_REQUESTED_OTHER = 0,
     DIS_PDU_SERVICE_REQUEST_SERVICE_TYPE_REQUESTED_RESUPPLY = 1,
     DIS_PDU_SERVICE_REQUEST_SERVICE_TYPE_REQUESTED_REPAIR = 2,
     DIS_PDU_SERVICE_REQUEST_SERVICE_TYPE_REQUESTED_AERIAL_REFUELING_HIGH_FIDELITY = 3,
     DIS_PDU_SERVICE_REQUEST_SERVICE_TYPE_REQUESTED_AERIAL_REFUELING_LOW_FIDELITY = 4
}  DIS_PDU_Service_Request_Service_Type_Requested;

/* SISO-REF-010 [UID 64] */
typedef enum
{
     DIS_PDU_REPAIR_COMPLETE_REPAIR_GENERAL_REPAIR_CODES_NO_REPAIRS_PERFORMED = 0,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_GENERAL_REPAIR_CODES_ALL_REQUESTED_REPAIRS_PERFORMED = 1,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_MOTOR_ENGINE = 10,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_STARTER = 20,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_ALTERNATOR = 30,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_GENERATOR = 40,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_BATTERY = 50,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_ENGINE_COOLANT_LEAK = 60,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_FUEL_FILTER = 70,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_TRANSMISSION_OIL_LEAK = 80,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_ENGINE_OIL_LEAK = 90,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_PUMPS = 100,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_FILTERS = 110,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_TRANSMISSION = 120,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_BRAKES = 130,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_SUSPENSION_SYSTEM = 140,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_DRIVE_TRAIN_OIL_FILTER = 150,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HULL_AIRFRAME_BODY_HULL = 1000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HULL_AIRFRAME_BODY_AIRFRAME = 1010,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HULL_AIRFRAME_BODY_TRUCK_BODY = 1020,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HULL_AIRFRAME_BODY_TANK_BODY = 1030,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HULL_AIRFRAME_BODY_TRAILER_BODY = 1040,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HULL_AIRFRAME_BODY_TURRET = 1050,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_INTERFACES_WITH_ENVIRONMENT_PROPELLER = 1500,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_INTERFACES_WITH_ENVIRONMENT_FILTERS = 1520,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_INTERFACES_WITH_ENVIRONMENT_WHEELS = 1540,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_INTERFACES_WITH_ENVIRONMENT_TIRE = 1550,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_INTERFACES_WITH_ENVIRONMENT_TRACK = 1560,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_GUN_ELEVATION_DRIVE = 2000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_GUN_STABILIZATION_SYSTEM = 2010,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_GUNNER_S_PRIMARY_SIGHT_GPS = 2020,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_COMMANDER_S_EXTENSION_TO_THE_GPS = 2030,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_LOADING_MECHANISM = 2040,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_GUNNER_S_AUXILIARY_SIGHT = 2050,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_GUNNER_S_CONTROL_PANEL = 2060,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_GUNNER_S_CONTROL_ASSEMBLY_HANDLE_S = 2070,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_COMMANDER_S_CONTROL_HANDLES_ASSEMBLY = 2090,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_COMMANDER_S_WEAPON_STATION = 2100,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_COMMANDER_S_INDEPENDENT_THERMAL_VIEWER_CITV = 2110,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_WEAPONS_GENERAL_WEAPONS = 2120,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_FUEL_SYSTEMS_FUEL_TRANSFER_PUMP = 4000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_FUEL_SYSTEMS_FUEL_LINES = 4010,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_FUEL_SYSTEMS_GAUGES = 4020,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_FUEL_SYSTEMS_GENERAL_FUEL_SYSTEM = 4030,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ELECTRONIC_WARFARE_SYSTEMS = 4500,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_DETECTION_SYSTEMS = 4600,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_DETECTION_SYSTEMS_RADIO_FREQUENCY = 4610,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_DETECTION_SYSTEMS_MICROWAVE = 4620,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_DETECTION_SYSTEMS_INFRARED = 4630,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_DETECTION_SYSTEMS_LASER = 4640,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_RANGE_FINDERS = 4700,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_RANGE_ONLY_RADAR = 4710,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_LASER_RANGE_FINDER = 4720,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ELECTRONIC_SYSTEMS = 4800,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ELECTRONICS_SYSTEMS_RADIO_FREQUENCY = 4810,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ELECTRONICS_SYSTEMS_MICROWAVE = 4820,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ELECTRONICS_SYSTEMS_INFRARED = 4830,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ELECTRONICS_SYSTEMS_LASER = 4840,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_RADIOS = 5000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_COMMUNICATION_SYSTEMS = 5010,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_INTERCOMS = 5100,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ENCODERS = 5200,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_ENCRYPTION_DEVICES = 5250,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_DECODERS = 5300,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_DECRYPTION_DEVICES = 5350,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_COMPUTERS = 5500,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_NAVIGATION_AND_CONTROL_SYSTEMS = 6000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_ELECTRONICS_FIRE_CONTROL_SYSTEMS = 6500,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_LIFE_SUPPORT_SYSTEMS_AIR_SUPPLY = 8000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_LIFE_SUPPORT_SYSTEMS_FILTERS = 8010,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_LIFE_SUPPORT_SYSTEMS_WATER_SUPPLY = 8020,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_LIFE_SUPPORT_SYSTEMS_REFRIGERATION_SYSTEM = 8030,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_LIFE_SUPPORT_SYSTEMS_CHEMICAL_BIOLOGICAL_AND_RADIOLOGICAL_PROTECTION = 8040,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_LIFE_SUPPORT_SYSTEMS_WATER_WASH_DOWN_SYSTEMS = 8050,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_LIFE_SUPPORT_SYSTEMS_DECONTAMINATION_SYSTEMS = 8060,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HYDRAULIC_SYSTEMS_AND_ACTUATORS_WATER_SUPPLY = 9000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HYDRAULIC_SYSTEMS_AND_ACTUATORS_COOLING_SYSTEM = 9010,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HYDRAULIC_SYSTEMS_AND_ACTUATORS_WINCHES = 9020,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HYDRAULIC_SYSTEMS_AND_ACTUATORS_CATAPULTS = 9030,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HYDRAULIC_SYSTEMS_AND_ACTUATORS_CRANES = 9040,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_HYDRAULIC_SYSTEMS_AND_ACTUATORS_LAUNCHERS = 9050,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_AUXILIARY_CRAFT_LIFE_BOATS = 10000,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_AUXILIARY_CRAFT_LANDING_CRAFT = 10010,
     DIS_PDU_REPAIR_COMPLETE_REPAIR_AUXILIARY_CRAFT_EJECTION_SEATS = 10020
}  DIS_PDU_Repair_Complete_Repair;

/* SISO-REF-010 [UID 65] */
typedef enum
{
     DIS_PDU_REPAIR_RESPONSE_REPAIR_RESULT_OTHER = 0,
     DIS_PDU_REPAIR_RESPONSE_REPAIR_RESULT_REPAIR_ENDED = 1,
     DIS_PDU_REPAIR_RESPONSE_REPAIR_RESULT_INVALID_REPAIR = 2,
     DIS_PDU_REPAIR_RESPONSE_REPAIR_RESULT_REPAIR_INTERRUPTED = 3,
     DIS_PDU_REPAIR_RESPONSE_REPAIR_RESULT_SERVICE_CANCELED_BY_THE_SUPPLIER = 4
}  DIS_PDU_Repair_Response_Repair_Result;

/* SISO-REF-010 [UID 67] */
typedef enum
{
     DIS_PDU_STOP_FREEZE_REASON_OTHER = 0,
     DIS_PDU_STOP_FREEZE_REASON_RECESS = 1,
     DIS_PDU_STOP_FREEZE_REASON_TERMINATION = 2,
     DIS_PDU_STOP_FREEZE_REASON_SYSTEM_FAILURE = 3,
     DIS_PDU_STOP_FREEZE_REASON_SECURITY_VIOLATION = 4,
     DIS_PDU_STOP_FREEZE_REASON_ENTITY_RECONSTITUTION = 5,
     DIS_PDU_STOP_FREEZE_REASON_STOP_FOR_RESET = 6,
     DIS_PDU_STOP_FREEZE_REASON_STOP_FOR_RESTART = 7,
     DIS_PDU_STOP_FREEZE_REASON_ABORT_TRAINING_RETURN_TO_TACTICAL_OPERATIONS = 8
}  DIS_PDU_Stop_Freeze_Reason;

static const value_string DIS_PDU_Stop_Freeze_Reason_Strings[] = {
     { DIS_PDU_STOP_FREEZE_REASON_OTHER, "Other" },
     { DIS_PDU_STOP_FREEZE_REASON_RECESS, "Recess" },
     { DIS_PDU_STOP_FREEZE_REASON_TERMINATION, "Termination" },
     { DIS_PDU_STOP_FREEZE_REASON_SYSTEM_FAILURE, "System Failure" },
     { DIS_PDU_STOP_FREEZE_REASON_SECURITY_VIOLATION, "Security Violation" },
     { DIS_PDU_STOP_FREEZE_REASON_ENTITY_RECONSTITUTION, "Entity Reconstitution" },
     { DIS_PDU_STOP_FREEZE_REASON_STOP_FOR_RESET, "Stop for reset" },
     { DIS_PDU_STOP_FREEZE_REASON_STOP_FOR_RESTART, "Stop for restart" },
     { DIS_PDU_STOP_FREEZE_REASON_ABORT_TRAINING_RETURN_TO_TACTICAL_OPERATIONS, "Abort Training Return to Tactical Operations" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 69] */
typedef enum
{
     DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_CREATE_ENTITY = 1,
     DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_REMOVE_ENTITY = 2,
     DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_START_RESUME = 3,
     DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_STOP_FREEZE = 4,
     DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_TRANSFER_OWNERSHIP = 5
}  DIS_PDU_Acknowledge_Acknowledge_Flag;

static const value_string DIS_PDU_Acknowledge_Acknowledge_Flag_Strings[] = {
     { DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_CREATE_ENTITY, "Create Entity" },
     { DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_REMOVE_ENTITY, "Remove Entity" },
     { DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_START_RESUME, "Start/Resume" },
     { DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_STOP_FREEZE, "Stop/Freeze" },
     { DIS_PDU_ACKNOWLEDGE_ACKNOWLEDGE_FLAG_TRANSFER_OWNERSHIP, "Transfer Ownership" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 70] */
typedef enum
{
     DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_OTHER = 0,
     DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_ABLE_TO_COMPLY = 1,
     DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_UNABLE_TO_COMPLY = 2,
     DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_PENDING_OPERATOR_ACTION = 3
}  DIS_PDU_Acknowledge_Response_Flag;

static const value_string DIS_PDU_Acknowledge_Response_Flag_Strings[] = {
     { DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_OTHER, "Other" },
     { DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_ABLE_TO_COMPLY, "Able to comply" },
     { DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_UNABLE_TO_COMPLY, "Unable to comply" },
     { DIS_PDU_ACKNOWLEDGE_RESPONSE_FLAG_PENDING_OPERATOR_ACTION, "Pending Operator Action" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 71] */
typedef enum
{
     DIS_PDU_ACTION_REQUEST_ACTION_ID_OTHER = 0,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_LOCAL_STORAGE_OF_THE_REQUESTED_INFORMATION = 1,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_RAN_OUT_OF_AMMUNITION_ = 2,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_KILLED_IN_ACTION_ = 3,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_DAMAGE_ = 4,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_MOBILITY_DISABLED_ = 5,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_FIRE_DISABLED_ = 6,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_RAN_OUT_OF_FUEL_ = 7,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_RECALL_CHECKPOINT_DATA = 8,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_RECALL_INITIAL_PARAMETERS = 9,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_TETHER_LEAD = 10,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_TETHER_FOLLOW = 11,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_UNTETHER = 12,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_SERVICE_STATION_RESUPPLY = 13,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_TAILGATE_RESUPPLY = 14,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_HITCH_LEAD = 15,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_HITCH_FOLLOW = 16,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_UNHITCH = 17,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_MOUNT = 18,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_DISMOUNT = 19,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_START_DRC_DAILY_READINESS_CHECK = 20,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_STOP_DRC = 21,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_DATA_QUERY = 22,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_STATUS_REQUEST = 23,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SEND_OBJECT_STATE_DATA = 24,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_RECONSTITUTE = 25,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_LOCK_SITE_CONFIGURATION = 26,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_UNLOCK_SITE_CONFIGURATION = 27,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_UPDATE_SITE_CONFIGURATION = 28,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_QUERY_SITE_CONFIGURATION = 29,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_TETHERING_INFORMATION = 30,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_MOUNT_INTENT = 31,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_ACCEPT_SUBSCRIPTION = 33,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_UNSUBSCRIBE = 34,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_TELEPORT_ENTITY = 35,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CHANGE_AGGREGATE_STATE = 36,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_REQUEST_START_PDU = 37,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_WAKEUP_GET_READY_FOR_INITIALIZATION = 38,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIALIZE_INTERNAL_PARAMETERS = 39,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SEND_PLAN_DATA = 40,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SYNCHRONIZE_INTERNAL_CLOCKS = 41,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_RUN = 42,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SAVE_INTERNAL_PARAMETERS = 43,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SIMULATE_MALFUNCTION = 44,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_JOIN_EXERCISE = 45,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_RESIGN_EXERCISE = 46,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_TIME_ADVANCE = 47,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_COMMAND_FROM_SIMULATOR = 48,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_TACCSF_LOS_REQUEST_TYPE_1 = 100,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_TACCSF_LOS_REQUEST_TYPE_2 = 101,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_JOIN_TRANSACTION_JOIN_REQUEST_MESSAGE = 1001,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_SET_PLAYBACK_WINDOW_TRANSACTION_SET_PLAYBACK_WINDOW_REQUEST_MESSAGE = 1002,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_LOAD_MISSION_RECORDING_TRANSACTION_LOAD_MISSION_RECORDING_REQUEST_MESSAGE = 1003,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_CUE_TRANSACTION_CUE_REQUEST_MESSAGE = 1004,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_PLAY_TRANSACTION_PLAY_REQUEST_MESSAGE = 1005,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_STOP_TRANSACTION_STOP_REQUEST_MESSAGE = 1006,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_PAUSE_TRANSACTION_PAUSE_REQUEST_MESSAGE = 1007,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_RESERVED_FOR_DDCP = 1008,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_END_TRANSACTION_END_REQUEST_MESSAGE = 1009,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SLING_LOAD_CAPABILITY_REQUEST = 4300,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SLING_LOAD_ATTACH_REQUEST = 4301,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_SLING_LOAD_RELEASE_REQUEST = 4302,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_AIRMOUNT_MOUNT_REQUEST = 4303,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_AIRMOUNT_DISMOUNT_REQUEST = 4304,
     DIS_PDU_ACTION_REQUEST_ACTION_ID_AIRMOUNT_INFORMATION_REQUEST = 4305
}  DIS_PDU_Action_Request_Action_ID;

static const value_string DIS_PDU_Action_Request_Action_ID_Strings[] = {
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_OTHER, "Other" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_LOCAL_STORAGE_OF_THE_REQUESTED_INFORMATION, "Local storage of the requested information" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_RAN_OUT_OF_AMMUNITION_, "Inform SM of event ran out of ammunition" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_KILLED_IN_ACTION_, "Inform SM of event killed in action" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_DAMAGE_, "Inform SM of event damage" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_MOBILITY_DISABLED_, "Inform SM of event mobility disabled" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_FIRE_DISABLED_, "Inform SM of event fire disabled" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INFORM_SM_OF_EVENT_RAN_OUT_OF_FUEL_, "Inform SM of event ran out of fuel" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_RECALL_CHECKPOINT_DATA, "Recall checkpoint data" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_RECALL_INITIAL_PARAMETERS, "Recall initial parameters" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_TETHER_LEAD, "Initiate tether-lead" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_TETHER_FOLLOW, "Initiate tether-follow" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_UNTETHER, "Untether" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_SERVICE_STATION_RESUPPLY, "Initiate service station resupply" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_TAILGATE_RESUPPLY, "Initiate tailgate resupply" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_HITCH_LEAD, "Initiate hitch lead" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIATE_HITCH_FOLLOW, "Initiate hitch follow" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_UNHITCH, "Unhitch" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_MOUNT, "Mount" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_DISMOUNT, "Dismount" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_START_DRC_DAILY_READINESS_CHECK, "Start DRC (Daily Readiness Check)" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_STOP_DRC, "Stop DRC" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_DATA_QUERY, "Data Query" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_STATUS_REQUEST, "Status Request" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SEND_OBJECT_STATE_DATA, "Send Object State Data" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_RECONSTITUTE, "Reconstitute" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_LOCK_SITE_CONFIGURATION, "Lock Site Configuration" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_UNLOCK_SITE_CONFIGURATION, "Unlock Site Configuration" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_UPDATE_SITE_CONFIGURATION, "Update Site Configuration" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_QUERY_SITE_CONFIGURATION, "Query Site Configuration" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_TETHERING_INFORMATION, "Tethering Information" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_MOUNT_INTENT, "Mount Intent" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_ACCEPT_SUBSCRIPTION, "Accept Subscription" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_UNSUBSCRIBE, "Unsubscribe" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_TELEPORT_ENTITY, "Teleport entity" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CHANGE_AGGREGATE_STATE, "Change aggregate state" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_REQUEST_START_PDU, "Request Start PDU" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_WAKEUP_GET_READY_FOR_INITIALIZATION, "Wakeup get ready for initialization" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_INITIALIZE_INTERNAL_PARAMETERS, "Initialize internal parameters" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SEND_PLAN_DATA, "Send plan data" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SYNCHRONIZE_INTERNAL_CLOCKS, "Synchronize internal clocks" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_RUN, "Run" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SAVE_INTERNAL_PARAMETERS, "Save internal parameters" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SIMULATE_MALFUNCTION, "Simulate malfunction" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_JOIN_EXERCISE, "Join exercise" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_RESIGN_EXERCISE, "Resign exercise" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_TIME_ADVANCE, "Time advance" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_COMMAND_FROM_SIMULATOR, "Command from Simulator" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_TACCSF_LOS_REQUEST_TYPE_1, "TACCSF LOS Request-Type 1" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_TACCSF_LOS_REQUEST_TYPE_2, "TACCSF LOS Request-Type 2" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_JOIN_TRANSACTION_JOIN_REQUEST_MESSAGE, "CAF DMO DDCP Join Transaction Join Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_SET_PLAYBACK_WINDOW_TRANSACTION_SET_PLAYBACK_WINDOW_REQUEST_MESSAGE, "CAF DMO DDCP Set Playback Window Transaction Set Playback Window Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_LOAD_MISSION_RECORDING_TRANSACTION_LOAD_MISSION_RECORDING_REQUEST_MESSAGE, "CAF DMO DDCP Load Mission Recording Transaction Load Mission Recording Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_CUE_TRANSACTION_CUE_REQUEST_MESSAGE, "CAF DMO DDCP Cue Transaction Cue Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_PLAY_TRANSACTION_PLAY_REQUEST_MESSAGE, "CAF DMO DDCP Play Transaction Play Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_STOP_TRANSACTION_STOP_REQUEST_MESSAGE, "CAF DMO DDCP Stop Transaction Stop Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_PAUSE_TRANSACTION_PAUSE_REQUEST_MESSAGE, "CAF DMO DDCP Pause Transaction Pause Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_RESERVED_FOR_DDCP, "Reserved for DDCP" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_CAF_DMO_DDCP_END_TRANSACTION_END_REQUEST_MESSAGE, "CAF DMO DDCP End Transaction End Request Message" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SLING_LOAD_CAPABILITY_REQUEST, "Sling Load Capability Request" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SLING_LOAD_ATTACH_REQUEST, "Sling Load Attach Request" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_SLING_LOAD_RELEASE_REQUEST, "Sling Load Release Request" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_AIRMOUNT_MOUNT_REQUEST, "Airmount Mount Request" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_AIRMOUNT_DISMOUNT_REQUEST, "Airmount Dismount Request" },
     { DIS_PDU_ACTION_REQUEST_ACTION_ID_AIRMOUNT_INFORMATION_REQUEST, "Airmount Information Request" },
     { 0, NULL }
 };

static value_string_ext DIS_PDU_Action_Request_Action_ID_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_Action_Request_Action_ID_Strings);

/* SISO-REF-010 [UID 72] */
typedef enum
{
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_OTHER = 0,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_PENDING = 1,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_EXECUTING = 2,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_PARTIALLY_COMPLETE = 3,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_COMPLETE = 4,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_REQUEST_REJECTED = 5,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_RETRANSMIT_REQUEST_NOW = 6,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_RETRANSMIT_REQUEST_LATER = 7,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_INVALID_TIME_PARAMETERS = 8,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_SIMULATION_TIME_EXCEEDED = 9,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_REQUEST_DONE = 10,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_TACCSF_LOS_REPLY_TYPE_1 = 100,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_TACCSF_LOS_REPLY_TYPE_2 = 101,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_JOIN_EXERCISE_REQUEST_REJECTED = 201,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_OK = 1131,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_ERROR = 1132,
     DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_DELAY = 1133
}  DIS_PDU_Action_Response_Request_Status;

static const value_string DIS_PDU_Action_Response_Request_Status_Strings[] = {
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_OTHER, "Other" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_PENDING, "Pending" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_EXECUTING, "Executing" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_PARTIALLY_COMPLETE, "Partially Complete" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_COMPLETE, "Complete" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_REQUEST_REJECTED, "Request rejected" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_RETRANSMIT_REQUEST_NOW, "Retransmit request now" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_RETRANSMIT_REQUEST_LATER, "Retransmit request later" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_INVALID_TIME_PARAMETERS, "Invalid time parameters" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_SIMULATION_TIME_EXCEEDED, "Simulation time exceeded" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_REQUEST_DONE, "Request done" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_TACCSF_LOS_REPLY_TYPE_1, "TACCSF LOS Reply-Type 1" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_TACCSF_LOS_REPLY_TYPE_2, "TACCSF LOS Reply-Type 2" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_JOIN_EXERCISE_REQUEST_REJECTED, "Join Exercise Request Rejected" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_OK, "OK" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_ERROR, "Error" },
     { DIS_PDU_ACTION_RESPONSE_REQUEST_STATUS_DELAY, "Delay" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 73] */
typedef enum
{
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_OTHER = 0,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_RAN_OUT_OF_AMMUNITION = 2,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_KILLED_IN_ACTION_KIA = 3,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_DAMAGE = 4,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_MOBILITY_DISABLED = 5,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_FIRE_DISABLED = 6,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_RAN_OUT_OF_FUEL = 7,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_ENTITY_INITIALIZATION = 8,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_REQUEST_FOR_INDIRECT_FIRE_OR_CAS_MISSION = 9,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_INDIRECT_FIRE_OR_CAS_FIRE = 10,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_MINEFIELD_ENTRY = 11,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_MINEFIELD_DETONATION = 12,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_VEHICLE_MASTER_POWER_ON = 13,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_VEHICLE_MASTER_POWER_OFF = 14,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_AGGREGATE_STATE_CHANGE_REQUESTED = 15,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_PREVENT_COLLISION_DETONATION = 16,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_OWNERSHIP_REPORT = 17,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_RADAR_PERCEPTION = 18,
     DIS_PDU_EVENT_REPORT_EVENT_TYPE_DETECT = 19
}  DIS_PDU_Event_Report_Event_Type;

/* SISO-REF-010 [UID 74] */
typedef enum
{
     DIS_PDU_REQUIRED_RELIABILITY_SERVICE_ACKNOWLEDGED = 0,
     DIS_PDU_REQUIRED_RELIABILITY_SERVICE_UNACKNOWLEDGED = 1
}  DIS_PDU_Required_Reliability_Service;

static const value_string DIS_PDU_Required_Reliability_Service_Strings[] = {
     { DIS_PDU_REQUIRED_RELIABILITY_SERVICE_ACKNOWLEDGED, "Acknowledged" },
     { DIS_PDU_REQUIRED_RELIABILITY_SERVICE_UNACKNOWLEDGED, "Unacknowledged" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 75] */
typedef enum
{
     DIS_PDU_EMITTERNAME_1245_6X = 2,
     DIS_PDU_EMITTERNAME_1L117 = 3,
     DIS_PDU_EMITTERNAME_1L121E = 4,
     DIS_PDU_EMITTERNAME_1L250 = 5,
     DIS_PDU_EMITTERNAME_1L220_U = 6,
     DIS_PDU_EMITTERNAME_1L122_1E = 7,
     DIS_PDU_EMITTERNAME_1RL257 = 9,
     DIS_PDU_EMITTERNAME_1RL138 = 10,
     DIS_PDU_EMITTERNAME_1RL257_KRASUKHA_4_JAMMER = 11,
     DIS_PDU_EMITTERNAME_5N20 = 12,
     DIS_PDU_EMITTERNAME_5H62B = 13,
     DIS_PDU_EMITTERNAME_5P_10_PUMA = 14,
     DIS_PDU_EMITTERNAME_5P_10E_PUMA = 15,
     DIS_PDU_EMITTERNAME_5P_10_01_PUMA = 16,
     DIS_PDU_EMITTERNAME_5P_10_01E_PUMA = 17,
     DIS_PDU_EMITTERNAME_5P_10_02_PUMA = 18,
     DIS_PDU_EMITTERNAME_5P_10_02E_PUMA = 19,
     DIS_PDU_EMITTERNAME_5P_10_03_LASKA = 20,
     DIS_PDU_EMITTERNAME_5P_10_03E_LASKA = 21,
     DIS_PDU_EMITTERNAME_5P_10E_MOD = 22,
     DIS_PDU_EMITTERNAME_621A_3 = 25,
     DIS_PDU_EMITTERNAME_860F_1_AL_101 = 40,
     DIS_PDU_EMITTERNAME_9B_1103M2 = 42,
     DIS_PDU_EMITTERNAME_1226_DECCA_MIL = 45,
     DIS_PDU_EMITTERNAME_9B_1348 = 46,
     DIS_PDU_EMITTERNAME_3KM6 = 47,
     DIS_PDU_EMITTERNAME_9KR400 = 48,
     DIS_PDU_EMITTERNAME_50N6A = 49,
     DIS_PDU_EMITTERNAME_55G6_1 = 50,
     DIS_PDU_EMITTERNAME_59N6 = 55,
     DIS_PDU_EMITTERNAME_5N69 = 57,
     DIS_PDU_EMITTERNAME_67N6 = 60,
     DIS_PDU_EMITTERNAME_76T6 = 63,
     DIS_PDU_EMITTERNAME_77T6_ABM = 64,
     DIS_PDU_EMITTERNAME_80K6 = 65,
     DIS_PDU_EMITTERNAME_91N6A_M = 66,
     DIS_PDU_EMITTERNAME_96L6E = 70,
     DIS_PDU_EMITTERNAME_96L6_TSP = 75,
     DIS_PDU_EMITTERNAME_9C18M3 = 76,
     DIS_PDU_EMITTERNAME_9C36M = 77,
     DIS_PDU_EMITTERNAME_9GR400 = 80,
     DIS_PDU_EMITTERNAME_9_GR_400A = 81,
     DIS_PDU_EMITTERNAME_9GR600 = 90,
     DIS_PDU_EMITTERNAME_9GR606 = 91,
     DIS_PDU_EMITTERNAME_9_LV_100 = 125,
     DIS_PDU_EMITTERNAME_9LV_200_TA = 135,
     DIS_PDU_EMITTERNAME_9LV_200_TV = 180,
     DIS_PDU_EMITTERNAME_9LV_200_TT = 181,
     DIS_PDU_EMITTERNAME_9LV200_MK_III = 183,
     DIS_PDU_EMITTERNAME_9LV326 = 185,
     DIS_PDU_EMITTERNAME_9M96E2_SEEKER = 190,
     DIS_PDU_EMITTERNAME_9S15M2 = 195,
     DIS_PDU_EMITTERNAME_9S19M2 = 196,
     DIS_PDU_EMITTERNAME_9S19ME = 197,
     DIS_PDU_EMITTERNAME_9S32M = 198,
     DIS_PDU_EMITTERNAME_9S32ME = 199,
     DIS_PDU_EMITTERNAME_9S36E = 200,
     DIS_PDU_EMITTERNAME_9S112 = 215,
     DIS_PDU_EMITTERNAME_A310Z = 225,
     DIS_PDU_EMITTERNAME_A325A = 270,
     DIS_PDU_EMITTERNAME_A346Z = 315,
     DIS_PDU_EMITTERNAME_A353B = 360,
     DIS_PDU_EMITTERNAME_A372A = 405,
     DIS_PDU_EMITTERNAME_A372B = 450,
     DIS_PDU_EMITTERNAME_A372C = 495,
     DIS_PDU_EMITTERNAME_A377A = 540,
     DIS_PDU_EMITTERNAME_A377B = 585,
     DIS_PDU_EMITTERNAME_A380Z = 630,
     DIS_PDU_EMITTERNAME_A381Z = 675,
     DIS_PDU_EMITTERNAME_A398Z = 720,
     DIS_PDU_EMITTERNAME_A403Z = 765,
     DIS_PDU_EMITTERNAME_A409A = 810,
     DIS_PDU_EMITTERNAME_A418A = 855,
     DIS_PDU_EMITTERNAME_A419Z = 900,
     DIS_PDU_EMITTERNAME_A429Z = 945,
     DIS_PDU_EMITTERNAME_A432Z = 990,
     DIS_PDU_EMITTERNAME_A434Z = 1035,
     DIS_PDU_EMITTERNAME_AA_6C_ACRID_R_40 = 1070,
     DIS_PDU_EMITTERNAME_AA_7C_APEX_R_24R = 1073,
     DIS_PDU_EMITTERNAME_A401A = 1080,
     DIS_PDU_EMITTERNAME_AA_10A_R_27R = 1081,
     DIS_PDU_EMITTERNAME_AA_10C_ALAMO_C = 1082,
     DIS_PDU_EMITTERNAME_AA_13_ARROW_SEEKER = 1085,
     DIS_PDU_EMITTERNAME_AAM_4B_MH = 1090,
     DIS_PDU_EMITTERNAME_AA_300 = 1094,
     DIS_PDU_EMITTERNAME_AA_12_SEEKER = 1095,
     DIS_PDU_EMITTERNAME_AD4A = 1096,
     DIS_PDU_EMITTERNAME_ADES = 1097,
     DIS_PDU_EMITTERNAME_ADS_4_LRSR = 1098,
     DIS_PDU_EMITTERNAME_ACR_430 = 1099,
     DIS_PDU_EMITTERNAME_AGAVE = 1100,
     DIS_PDU_EMITTERNAME_ACSOPRI_E = 1101,
     DIS_PDU_EMITTERNAME_ABD_2000 = 1102,
     DIS_PDU_EMITTERNAME_ADAC_MK_1 = 1110,
     DIS_PDU_EMITTERNAME_ADAC_MK_2 = 1111,
     DIS_PDU_EMITTERNAME_ADAR = 1113,
     DIS_PDU_EMITTERNAME_ADOUR = 1115,
     DIS_PDU_EMITTERNAME_AGAT_9B_1348 = 1117,
     DIS_PDU_EMITTERNAME_ADROS_KT_01AV = 1118,
     DIS_PDU_EMITTERNAME_AGAT_9E420 = 1120,
     DIS_PDU_EMITTERNAME_AGM_158_JASSM_SAR = 1122,
     DIS_PDU_EMITTERNAME_AGM_88_HARM_MMW = 1123,
     DIS_PDU_EMITTERNAME_AGRION_15 = 1125,
     DIS_PDU_EMITTERNAME_AHV_7 = 1130,
     DIS_PDU_EMITTERNAME_AHV_17 = 1150,
     DIS_PDU_EMITTERNAME_AI_MK_23 = 1170,
     DIS_PDU_EMITTERNAME_AIDA_II = 1215,
     DIS_PDU_EMITTERNAME_AIM_120A = 1216,
     DIS_PDU_EMITTERNAME_AIM_7M_SPARROW = 1218,
     DIS_PDU_EMITTERNAME_1L271_AISTENOK = 1230,
     DIS_PDU_EMITTERNAME_ALA_51 = 1240,
     DIS_PDU_EMITTERNAME_ALBATROS_MK2 = 1260,
     DIS_PDU_EMITTERNAME_ALT_50 = 1263,
     DIS_PDU_EMITTERNAME_ALTAIR = 1264,
     DIS_PDU_EMITTERNAME_AM_APS_717 = 1265,
     DIS_PDU_EMITTERNAME_AMES_13_MK_1 = 1268,
     DIS_PDU_EMITTERNAME_WGU_16_B = 1270,
     DIS_PDU_EMITTERNAME_1L13_3_55G6_BOX_SPRING = 1280,
     DIS_PDU_EMITTERNAME_1L13_3_55G6_BOX_SPRING_B = 1282,
     DIS_PDU_EMITTERNAME_AMDR_3D = 1288,
     DIS_PDU_EMITTERNAME_ANA_SPS_502 = 1305,
     DIS_PDU_EMITTERNAME_ANEMONE = 1306,
     DIS_PDU_EMITTERNAME_ANRITSU_ELECTRIC_AR_30A = 1350,
     DIS_PDU_EMITTERNAME_ANTILOPE_V = 1395,
     DIS_PDU_EMITTERNAME_AN_AAQ_24 = 1397,
     DIS_PDU_EMITTERNAME_AN_ADM_160 = 1398,
     DIS_PDU_EMITTERNAME_AN_ALE_50 = 1400,
     DIS_PDU_EMITTERNAME_AN_ALQ_76 = 1410,
     DIS_PDU_EMITTERNAME_AN_ALQ_99 = 1440,
     DIS_PDU_EMITTERNAME_AN_ALQ_99_BAND_4 = 1441,
     DIS_PDU_EMITTERNAME_AN_ALQ_99_LBT = 1442,
     DIS_PDU_EMITTERNAME_AN_ALQ_100 = 1485,
     DIS_PDU_EMITTERNAME_AN_ALQ_101 = 1530,
     DIS_PDU_EMITTERNAME_AN_ALQ_119 = 1575,
     DIS_PDU_EMITTERNAME_AN_ALQ_122 = 1585,
     DIS_PDU_EMITTERNAME_AN_ALQ_126A = 1620,
     DIS_PDU_EMITTERNAME_AN_ALQ_128 = 1621,
     DIS_PDU_EMITTERNAME_AN_ALQ_126B = 1622,
     DIS_PDU_EMITTERNAME_AN_ALQ_131 = 1626,
     DIS_PDU_EMITTERNAME_AN_ALQ_131_BLK_II = 1627,
     DIS_PDU_EMITTERNAME_AN_ALQ_135C_D = 1628,
     DIS_PDU_EMITTERNAME_AN_ALQ_144A_V_3 = 1630,
     DIS_PDU_EMITTERNAME_AN_ALQ_153 = 1632,
     DIS_PDU_EMITTERNAME_AN_ALQ_157JAMMER = 1633,
     DIS_PDU_EMITTERNAME_AN_ALQ_155 = 1634,
     DIS_PDU_EMITTERNAME_AN_ALQ_156 = 1635,
     DIS_PDU_EMITTERNAME_AN_ALQ_161_A = 1636,
     DIS_PDU_EMITTERNAME_AN_ALQ_161 = 1637,
     DIS_PDU_EMITTERNAME_AN_ALQ_162 = 1638,
     DIS_PDU_EMITTERNAME_AN_ALQ_164 = 1639,
     DIS_PDU_EMITTERNAME_AN_ALQ_165 = 1640,
     DIS_PDU_EMITTERNAME_AN_ALQ_187_JAMMER = 1641,
     DIS_PDU_EMITTERNAME_AN_ALQ_167 = 1642,
     DIS_PDU_EMITTERNAME_AN_ALQ_172_V_1 = 1643,
     DIS_PDU_EMITTERNAME_AN_ALQ_172_V_2 = 1644,
     DIS_PDU_EMITTERNAME_AN_ALQ_172_V_3 = 1645,
     DIS_PDU_EMITTERNAME_AN_ALQ_176 = 1646,
     DIS_PDU_EMITTERNAME_AN_ALQ_178 = 1647,
     DIS_PDU_EMITTERNAME_AN_ALQ_184 = 1648,
     DIS_PDU_EMITTERNAME_AN_ALQ_184_V_9 = 1649,
     DIS_PDU_EMITTERNAME_AN_ALQ_188 = 1650,
     DIS_PDU_EMITTERNAME_AN_ALQ_214 = 1651,
     DIS_PDU_EMITTERNAME_AN_ALR_56 = 1652,
     DIS_PDU_EMITTERNAME_AN_ALQ_221 = 1653,
     DIS_PDU_EMITTERNAME_AN_ALR_69 = 1654,
     DIS_PDU_EMITTERNAME_AN_ALQ_211_V_SIRFC = 1655,
     DIS_PDU_EMITTERNAME_AN_ALT_16A = 1656,
     DIS_PDU_EMITTERNAME_AN_ALQ_173 = 1657,
     DIS_PDU_EMITTERNAME_AN_ALT_28 = 1658,
     DIS_PDU_EMITTERNAME_AN_ALR_66B_JAMMER = 1659,
     DIS_PDU_EMITTERNAME_AN_ALT_32A = 1660,
     DIS_PDU_EMITTERNAME_AN_ALQ_196 = 1661,
     DIS_PDU_EMITTERNAME_AN_ALQ_249_V_1_NEXT_GENERATION_JAMMER_MID_BAND_NGJ_MB = 1662,
     DIS_PDU_EMITTERNAME_AN_ALQ_240_V_1_JAMMER = 1663,
     DIS_PDU_EMITTERNAME_AN_APD_10 = 1665,
     DIS_PDU_EMITTERNAME_AN_ALQ_249_V_2_NEXT_GENERATION_JAMMER_LOW_BAND_NGJ_LB = 1666,
     DIS_PDU_EMITTERNAME_AN_ALQ_213 = 1670,
     DIS_PDU_EMITTERNAME_ALQ_214A_V_4_5_JAMMER = 1672,
     DIS_PDU_EMITTERNAME_AN_ALQ_218 = 1680,
     DIS_PDU_EMITTERNAME_AN_ALQ_250_EPAWSS = 1685,
     DIS_PDU_EMITTERNAME_AN_APG_50 = 1700,
     DIS_PDU_EMITTERNAME_AN_APG_53 = 1710,
     DIS_PDU_EMITTERNAME_AN_APG_59 = 1755,
     DIS_PDU_EMITTERNAME_AN_APG_63 = 1800,
     DIS_PDU_EMITTERNAME_AN_APG_63_V_1 = 1805,
     DIS_PDU_EMITTERNAME_AN_APG_63_V_2 = 1807,
     DIS_PDU_EMITTERNAME_AN_APG_63_V_3 = 1809,
     DIS_PDU_EMITTERNAME_AN_APG_65 = 1845,
     DIS_PDU_EMITTERNAME_AN_APG_66 = 1870,
     DIS_PDU_EMITTERNAME_AN_APG_66_V = 1871,
     DIS_PDU_EMITTERNAME_AN_APG_66_V_2 = 1872,
     DIS_PDU_EMITTERNAME_AN_APG_67 = 1880,
     DIS_PDU_EMITTERNAME_AN_APG_68 = 1890,
     DIS_PDU_EMITTERNAME_AN_APG_68_V_9 = 1895,
     DIS_PDU_EMITTERNAME_AN_APG_70 = 1935,
     DIS_PDU_EMITTERNAME_AN_APG_71 = 1940,
     DIS_PDU_EMITTERNAME_AN_APG_73 = 1945,
     DIS_PDU_EMITTERNAME_AN_APG_77 = 1960,
     DIS_PDU_EMITTERNAME_AN_APG_78 = 1970,
     DIS_PDU_EMITTERNAME_AN_APG_79 = 1971,
     DIS_PDU_EMITTERNAME_AN_APG_80 = 1972,
     DIS_PDU_EMITTERNAME_AN_APG_81 = 1974,
     DIS_PDU_EMITTERNAME_AN_APG_82_V_1 = 1975,
     DIS_PDU_EMITTERNAME_AN_APG_83 = 1976,
     DIS_PDU_EMITTERNAME_AN_APG_502 = 1980,
     DIS_PDU_EMITTERNAME_AN_APN_1 = 2025,
     DIS_PDU_EMITTERNAME_AN_APN_22 = 2070,
     DIS_PDU_EMITTERNAME_AN_APN_59 = 2115,
     DIS_PDU_EMITTERNAME_AN_APN_69 = 2160,
     DIS_PDU_EMITTERNAME_AN_APN_81 = 2205,
     DIS_PDU_EMITTERNAME_AN_APN_102 = 2220,
     DIS_PDU_EMITTERNAME_AN_APN_117 = 2250,
     DIS_PDU_EMITTERNAME_AN_APN_118 = 2295,
     DIS_PDU_EMITTERNAME_AN_APN_122 = 2320,
     DIS_PDU_EMITTERNAME_AN_APN_130 = 2340,
     DIS_PDU_EMITTERNAME_AN_APN_131 = 2385,
     DIS_PDU_EMITTERNAME_AN_APN_133 = 2430,
     DIS_PDU_EMITTERNAME_AN_APN_134 = 2475,
     DIS_PDU_EMITTERNAME_AN_APN_141_V = 2476,
     DIS_PDU_EMITTERNAME_AN_APN_147 = 2520,
     DIS_PDU_EMITTERNAME_AN_APN_150 = 2565,
     DIS_PDU_EMITTERNAME_AN_APN_153 = 2610,
     DIS_PDU_EMITTERNAME_AN_APN_154 = 2655,
     DIS_PDU_EMITTERNAME_AN_APN_155 = 2700,
     DIS_PDU_EMITTERNAME_AN_APN_159 = 2745,
     DIS_PDU_EMITTERNAME_AN_APN_177 = 2746,
     DIS_PDU_EMITTERNAME_AN_APN_179 = 2747,
     DIS_PDU_EMITTERNAME_AN_APN_169 = 2748,
     DIS_PDU_EMITTERNAME_AN_APN_182 = 2790,
     DIS_PDU_EMITTERNAME_AN_APN_187 = 2835,
     DIS_PDU_EMITTERNAME_AN_APN_190 = 2880,
     DIS_PDU_EMITTERNAME_AN_APN_194 = 2925,
     DIS_PDU_EMITTERNAME_AN_APN_195 = 2970,
     DIS_PDU_EMITTERNAME_AN_APN_198 = 3015,
     DIS_PDU_EMITTERNAME_AN_APN_200 = 3060,
     DIS_PDU_EMITTERNAME_AN_APN_202 = 3105,
     DIS_PDU_EMITTERNAME_AN_APN_205 = 3106,
     DIS_PDU_EMITTERNAME_AN_APN_209 = 3120,
     DIS_PDU_EMITTERNAME_AN_APN_209D = 3121,
     DIS_PDU_EMITTERNAME_AN_APN_209A = 3122,
     DIS_PDU_EMITTERNAME_AN_APN_215 = 3148,
     DIS_PDU_EMITTERNAME_AN_APN_217 = 3150,
     DIS_PDU_EMITTERNAME_AN_APN_218 = 3152,
     DIS_PDU_EMITTERNAME_AN_APN_224 = 3153,
     DIS_PDU_EMITTERNAME_AN_APN_227 = 3154,
     DIS_PDU_EMITTERNAME_AN_APN_230 = 3155,
     DIS_PDU_EMITTERNAME_AN_APN_232 = 3156,
     DIS_PDU_EMITTERNAME_AN_APN_237A = 3157,
     DIS_PDU_EMITTERNAME_AN_APN_234 = 3158,
     DIS_PDU_EMITTERNAME_AN_APN_235 = 3159,
     DIS_PDU_EMITTERNAME_AN_APN_238 = 3160,
     DIS_PDU_EMITTERNAME_AN_APN_222 = 3161,
     DIS_PDU_EMITTERNAME_AN_APN_239 = 3162,
     DIS_PDU_EMITTERNAME_AN_APN_241 = 3164,
     DIS_PDU_EMITTERNAME_AN_APN_242 = 3166,
     DIS_PDU_EMITTERNAME_AN_APN_243 = 3170,
     DIS_PDU_EMITTERNAME_AN_APN_506 = 3195,
     DIS_PDU_EMITTERNAME_AN_APQ_72 = 3240,
     DIS_PDU_EMITTERNAME_AN_APQ_99 = 3285,
     DIS_PDU_EMITTERNAME_AN_APQ_100 = 3330,
     DIS_PDU_EMITTERNAME_AN_APQ_102 = 3375,
     DIS_PDU_EMITTERNAME_AN_APQ_107 = 3376,
     DIS_PDU_EMITTERNAME_AN_APQ_109 = 3420,
     DIS_PDU_EMITTERNAME_AN_APQ_113 = 3465,
     DIS_PDU_EMITTERNAME_AN_APQ_120 = 3510,
     DIS_PDU_EMITTERNAME_AN_APQ_122 = 3512,
     DIS_PDU_EMITTERNAME_AN_APQ_126 = 3555,
     DIS_PDU_EMITTERNAME_AN_APQ_128 = 3600,
     DIS_PDU_EMITTERNAME_AN_APQ_129 = 3645,
     DIS_PDU_EMITTERNAME_AN_APQ_148 = 3690,
     DIS_PDU_EMITTERNAME_AN_APQ_150A = 3700,
     DIS_PDU_EMITTERNAME_AN_APQ_153 = 3735,
     DIS_PDU_EMITTERNAME_AN_APQ_155 = 3770,
     DIS_PDU_EMITTERNAME_AN_APQ_159 = 3780,
     DIS_PDU_EMITTERNAME_AN_APQ_164 = 3785,
     DIS_PDU_EMITTERNAME_AN_APQ_166 = 3788,
     DIS_PDU_EMITTERNAME_AN_APQ_170 = 3790,
     DIS_PDU_EMITTERNAME_AN_APQ_174 = 3791,
     DIS_PDU_EMITTERNAME_AN_APQ_180 = 3794,
     DIS_PDU_EMITTERNAME_AN_APQ_181 = 3795,
     DIS_PDU_EMITTERNAME_AN_APQ_186 = 3800,
     DIS_PDU_EMITTERNAME_AN_APS_15J = 3810,
     DIS_PDU_EMITTERNAME_AN_APS_16_V_2 = 3813,
     DIS_PDU_EMITTERNAME_AN_APS_31 = 3820,
     DIS_PDU_EMITTERNAME_AN_APS_42 = 3825,
     DIS_PDU_EMITTERNAME_AN_APS_80 = 3870,
     DIS_PDU_EMITTERNAME_AN_APS_88 = 3915,
     DIS_PDU_EMITTERNAME_AN_APS_88A = 3916,
     DIS_PDU_EMITTERNAME_AN_APS_94 = 3920,
     DIS_PDU_EMITTERNAME_AN_APS_96 = 3922,
     DIS_PDU_EMITTERNAME_AN_APS_113 = 3958,
     DIS_PDU_EMITTERNAME_AN_APS_115 = 3960,
     DIS_PDU_EMITTERNAME_AN_APS_116 = 4005,
     DIS_PDU_EMITTERNAME_AN_APS_120 = 4050,
     DIS_PDU_EMITTERNAME_AN_APS_121 = 4095,
     DIS_PDU_EMITTERNAME_AN_APS_124 = 4140,
     DIS_PDU_EMITTERNAME_AN_APS_125 = 4185,
     DIS_PDU_EMITTERNAME_AN_APS_127 = 4190,
     DIS_PDU_EMITTERNAME_AN_APS_128 = 4230,
     DIS_PDU_EMITTERNAME_AN_APS_130 = 4275,
     DIS_PDU_EMITTERNAME_AN_APS_133 = 4320,
     DIS_PDU_EMITTERNAME_AN_APS_134 = 4365,
     DIS_PDU_EMITTERNAME_AN_APS_137 = 4410,
     DIS_PDU_EMITTERNAME_AN_APS_137_V_5 = 4413,
     DIS_PDU_EMITTERNAME_AN_APS_137B = 4415,
     DIS_PDU_EMITTERNAME_AN_APS_137B_V_5 = 4420,
     DIS_PDU_EMITTERNAME_AN_APS_137D_V_5_ELTA = 4425,
     DIS_PDU_EMITTERNAME_AN_APS_138 = 4455,
     DIS_PDU_EMITTERNAME_AN_APS_139 = 4460,
     DIS_PDU_EMITTERNAME_AN_APS_143 = 4464,
     DIS_PDU_EMITTERNAME_AN_APS_143_V_1 = 4465,
     DIS_PDU_EMITTERNAME_AN_APS_143B = 4466,
     DIS_PDU_EMITTERNAME_AN_APS_143_V_3 = 4467,
     DIS_PDU_EMITTERNAME_AN_APS_143B_V_3 = 4468,
     DIS_PDU_EMITTERNAME_AN_APS_153 = 4475,
     DIS_PDU_EMITTERNAME_AN_APS_154 = 4476,
     DIS_PDU_EMITTERNAME_AN_APS_150 = 4480,
     DIS_PDU_EMITTERNAME_AN_APS_145 = 4482,
     DIS_PDU_EMITTERNAME_AN_APS_147 = 4485,
     DIS_PDU_EMITTERNAME_AN_APS_149 = 4486,
     DIS_PDU_EMITTERNAME_AN_APS_503 = 4489,
     DIS_PDU_EMITTERNAME_AN_APS_504 = 4490,
     DIS_PDU_EMITTERNAME_AN_APS_705 = 4491,
     DIS_PDU_EMITTERNAME_AN_APW_22 = 4500,
     DIS_PDU_EMITTERNAME_AN_APW_23 = 4545,
     DIS_PDU_EMITTERNAME_AN_APX_6 = 4590,
     DIS_PDU_EMITTERNAME_AN_APX_7 = 4635,
     DIS_PDU_EMITTERNAME_AN_APX_39 = 4680,
     DIS_PDU_EMITTERNAME_AN_APX_64_V = 4681,
     DIS_PDU_EMITTERNAME_AN_APX_72 = 4725,
     DIS_PDU_EMITTERNAME_AN_APX_76 = 4770,
     DIS_PDU_EMITTERNAME_AN_APX_78 = 4815,
     DIS_PDU_EMITTERNAME_AN_APX_100 = 4816,
     DIS_PDU_EMITTERNAME_AN_APX_101 = 4860,
     DIS_PDU_EMITTERNAME_AN_APX_113_AIFF = 4870,
     DIS_PDU_EMITTERNAME_AN_APY_1 = 4900,
     DIS_PDU_EMITTERNAME_AN_APY_2 = 4905,
     DIS_PDU_EMITTERNAME_AN_APY_3 = 4950,
     DIS_PDU_EMITTERNAME_AN_APY_7 = 4952,
     DIS_PDU_EMITTERNAME_AN_APY_8_LYNX_TM = 4953,
     DIS_PDU_EMITTERNAME_AN_APY_9 = 4954,
     DIS_PDU_EMITTERNAME_AN_APY_10 = 4955,
     DIS_PDU_EMITTERNAME_AN_ARN_21 = 4995,
     DIS_PDU_EMITTERNAME_AN_ARN_52 = 5040,
     DIS_PDU_EMITTERNAME_AN_ARN_84 = 5085,
     DIS_PDU_EMITTERNAME_AN_ARN_118 = 5130,
     DIS_PDU_EMITTERNAME_AN_ARN_153_V = 5131,
     DIS_PDU_EMITTERNAME_AN_ARN_153 = 5165,
     DIS_PDU_EMITTERNAME_AN_ARW_73 = 5175,
     DIS_PDU_EMITTERNAME_AN_ASB_1 = 5220,
     DIS_PDU_EMITTERNAME_AN_ASG_21 = 5265,
     DIS_PDU_EMITTERNAME_AN_ASN_137 = 5266,
     DIS_PDU_EMITTERNAME_AN_ASN_128 = 5270,
     DIS_PDU_EMITTERNAME_AN_ASQ_108 = 5280,
     DIS_PDU_EMITTERNAME_AN_ASQ_239 = 5285,
     DIS_PDU_EMITTERNAME_AN_AST_502 = 5290,
     DIS_PDU_EMITTERNAME_AN_AVQ_55 = 5300,
     DIS_PDU_EMITTERNAME_AN_AWG_9 = 5310,
     DIS_PDU_EMITTERNAME_AN_BRN_1 = 5320,
     DIS_PDU_EMITTERNAME_AN_BPS_5 = 5325,
     DIS_PDU_EMITTERNAME_AN_BPS_9 = 5355,
     DIS_PDU_EMITTERNAME_AN_BPS_15 = 5400,
     DIS_PDU_EMITTERNAME_AN_BPS_15_H = 5401,
     DIS_PDU_EMITTERNAME_AN_BPS_15J = 5402,
     DIS_PDU_EMITTERNAME_AN_BPS_16 = 5405,
     DIS_PDU_EMITTERNAME_AN_BPS_16_V_2 = 5406,
     DIS_PDU_EMITTERNAME_AN_CPN_4 = 5410,
     DIS_PDU_EMITTERNAME_AN_CPN_18 = 5415,
     DIS_PDU_EMITTERNAME_AN_CRM_30 = 5420,
     DIS_PDU_EMITTERNAME_AN_DPW_23 = 5430,
     DIS_PDU_EMITTERNAME_AN_DSQ_26_PHOENIX_MH = 5445,
     DIS_PDU_EMITTERNAME_AN_DSQ_28_HARPOON_MH = 5490,
     DIS_PDU_EMITTERNAME_AN_FPN_1 = 5491,
     DIS_PDU_EMITTERNAME_AN_FPN_28 = 5493,
     DIS_PDU_EMITTERNAME_AN_FPN_33 = 5494,
     DIS_PDU_EMITTERNAME_AN_FPN_40 = 5495,
     DIS_PDU_EMITTERNAME_AN_FPN_62 = 5500,
     DIS_PDU_EMITTERNAME_AN_FPN_66 = 5502,
     DIS_PDU_EMITTERNAME_AN_FPS_8 = 5503,
     DIS_PDU_EMITTERNAME_AN_FPN_67 = 5504,
     DIS_PDU_EMITTERNAME_AN_FPS_16 = 5505,
     DIS_PDU_EMITTERNAME_AN_FPS_5 = 5506,
     DIS_PDU_EMITTERNAME_AN_FPS_18 = 5507,
     DIS_PDU_EMITTERNAME_AN_FPS_89 = 5508,
     DIS_PDU_EMITTERNAME_AN_FPS_49 = 5509,
     DIS_PDU_EMITTERNAME_AN_FPS_117 = 5510,
     DIS_PDU_EMITTERNAME_AN_FPS_85 = 5511,
     DIS_PDU_EMITTERNAME_AN_FPS_88 = 5512,
     DIS_PDU_EMITTERNAME_AN_FPS_113 = 5513,
     DIS_PDU_EMITTERNAME_AN_FPS_115 = 5514,
     DIS_PDU_EMITTERNAME_AN_FPS_20R = 5515,
     DIS_PDU_EMITTERNAME_AN_FPS_132 = 5516,
     DIS_PDU_EMITTERNAME_AN_FPS_77 = 5520,
     DIS_PDU_EMITTERNAME_AN_FPS_41 = 5521,
     DIS_PDU_EMITTERNAME_AN_FPS_100A = 5522,
     DIS_PDU_EMITTERNAME_AN_FPS_103 = 5525,
     DIS_PDU_EMITTERNAME_AN_FPS_108 = 5526,
     DIS_PDU_EMITTERNAME_AN_GPN_12 = 5527,
     DIS_PDU_EMITTERNAME_AN_FPS_124_V = 5528,
     DIS_PDU_EMITTERNAME_AN_FPS_129 = 5529,
     DIS_PDU_EMITTERNAME_AN_GPX_6 = 5530,
     DIS_PDU_EMITTERNAME_AN_GPX_8 = 5535,
     DIS_PDU_EMITTERNAME_AN_GRN_12 = 5537,
     DIS_PDU_EMITTERNAME_AN_MPN_14K = 5538,
     DIS_PDU_EMITTERNAME_AN_MPN_14 = 5539,
     DIS_PDU_EMITTERNAME_AN_MPQ_10 = 5540,
     DIS_PDU_EMITTERNAME_AN_MPN_17 = 5541,
     DIS_PDU_EMITTERNAME_AN_MPQ_33_39_46_57_61_HPIR_ILL = 5545,
     DIS_PDU_EMITTERNAME_AN_MPQ_34_48_55_62_CWAR_TA = 5550,
     DIS_PDU_EMITTERNAME_AN_MPQ_49 = 5551,
     DIS_PDU_EMITTERNAME_AN_MPQ_35_50_PAR_TA = 5555,
     DIS_PDU_EMITTERNAME_AN_MPQ_50C = 5556,
     DIS_PDU_EMITTERNAME_AN_MPQ_37_51_ROR_TT = 5560,
     DIS_PDU_EMITTERNAME_AN_MPQ_43 = 5565,
     DIS_PDU_EMITTERNAME_AN_MPQ_50 = 5567,
     DIS_PDU_EMITTERNAME_AN_MPQ_53 = 5570,
     DIS_PDU_EMITTERNAME_AN_MPQ_63 = 5571,
     DIS_PDU_EMITTERNAME_AN_MPQ_64 = 5575,
     DIS_PDU_EMITTERNAME_AN_SLQ_32 = 5576,
     DIS_PDU_EMITTERNAME_AN_MPQ_65 = 5577,
     DIS_PDU_EMITTERNAME_AN_SLQ_32_V_4 = 5578,
     DIS_PDU_EMITTERNAME_AN_SLQ_32A = 5579,
     DIS_PDU_EMITTERNAME_AN_SPG_34 = 5580,
     DIS_PDU_EMITTERNAME_AN_MSQ_104 = 5582,
     DIS_PDU_EMITTERNAME_AN_MPS_36 = 5583,
     DIS_PDU_EMITTERNAME_AN_SLQ_503 = 5584,
     DIS_PDU_EMITTERNAME_AN_SPG_48_MK_25_MOD_3 = 5620,
     DIS_PDU_EMITTERNAME_AN_SPG_50 = 5625,
     DIS_PDU_EMITTERNAME_AN_SPG_51 = 5670,
     DIS_PDU_EMITTERNAME_AN_PPQ_2 = 5690,
     DIS_PDU_EMITTERNAME_AN_PPS_15 = 5700,
     DIS_PDU_EMITTERNAME_AN_PPS_5 = 5705,
     DIS_PDU_EMITTERNAME_AN_PPS_5D = 5710,
     DIS_PDU_EMITTERNAME_AN_SPG_51_CWI_TI = 5715,
     DIS_PDU_EMITTERNAME_AN_SPG_51_FC = 5760,
     DIS_PDU_EMITTERNAME_AN_SPG_51C_D = 5761,
     DIS_PDU_EMITTERNAME_AN_SPG_52 = 5805,
     DIS_PDU_EMITTERNAME_AN_SPG_53 = 5850,
     DIS_PDU_EMITTERNAME_AN_SPG_55B = 5895,
     DIS_PDU_EMITTERNAME_AN_SPG_60 = 5940,
     DIS_PDU_EMITTERNAME_AN_SPG_62 = 5985,
     DIS_PDU_EMITTERNAME_AN_SPG_503 = 5995,
     DIS_PDU_EMITTERNAME_AN_SPN_4 = 6015,
     DIS_PDU_EMITTERNAME_AN_SPN_11 = 6025,
     DIS_PDU_EMITTERNAME_AN_SPN_35 = 6030,
     DIS_PDU_EMITTERNAME_AN_SPN_41 = 6050,
     DIS_PDU_EMITTERNAME_AN_SPN_43 = 6075,
     DIS_PDU_EMITTERNAME_AN_SPN_43A = 6076,
     DIS_PDU_EMITTERNAME_AN_SPN_43C = 6078,
     DIS_PDU_EMITTERNAME_AN_SPN_46 = 6085,
     DIS_PDU_EMITTERNAME_AN_SPQ_2 = 6120,
     DIS_PDU_EMITTERNAME_AN_SPQ_5A = 6155,
     DIS_PDU_EMITTERNAME_AN_SPQ_9A = 6165,
     DIS_PDU_EMITTERNAME_AN_SPQ_9B = 6166,
     DIS_PDU_EMITTERNAME_AN_SPQ_34 = 6190,
     DIS_PDU_EMITTERNAME_AN_SPS_4 = 6210,
     DIS_PDU_EMITTERNAME_AN_SPS_5 = 6255,
     DIS_PDU_EMITTERNAME_AN_SPS_5C = 6300,
     DIS_PDU_EMITTERNAME_AN_SPS_6 = 6345,
     DIS_PDU_EMITTERNAME_AN_SPS_10 = 6390,
     DIS_PDU_EMITTERNAME_AN_SPS_21 = 6435,
     DIS_PDU_EMITTERNAME_AN_SPS_28 = 6480,
     DIS_PDU_EMITTERNAME_AN_SPS_37 = 6525,
     DIS_PDU_EMITTERNAME_AN_SPS_39A = 6570,
     DIS_PDU_EMITTERNAME_AN_SPS_40 = 6615,
     DIS_PDU_EMITTERNAME_AN_SPS_41 = 6660,
     DIS_PDU_EMITTERNAME_AN_SPS_48 = 6705,
     DIS_PDU_EMITTERNAME_AN_SPS_48C = 6750,
     DIS_PDU_EMITTERNAME_AN_SPS_48E = 6752,
     DIS_PDU_EMITTERNAME_AN_SPS_49 = 6795,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_1 = 6796,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_2 = 6797,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_3 = 6798,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_4 = 6799,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_5 = 6800,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_6 = 6801,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_7 = 6802,
     DIS_PDU_EMITTERNAME_AN_SPS_49_V_8 = 6803,
     DIS_PDU_EMITTERNAME_AN_SPS_49A_V_1 = 6804,
     DIS_PDU_EMITTERNAME_AN_SPS_52 = 6840,
     DIS_PDU_EMITTERNAME_AN_SPS_53 = 6885,
     DIS_PDU_EMITTERNAME_AN_SPS_55 = 6930,
     DIS_PDU_EMITTERNAME_AN_SPS_52C = 6945,
     DIS_PDU_EMITTERNAME_AN_SPS_55_CS = 6970,
     DIS_PDU_EMITTERNAME_AN_SPS_55_SS = 6975,
     DIS_PDU_EMITTERNAME_AN_SPS_58 = 7020,
     DIS_PDU_EMITTERNAME_AN_SPS_58C = 7025,
     DIS_PDU_EMITTERNAME_AN_SPS_59 = 7065,
     DIS_PDU_EMITTERNAME_AN_SPS_64 = 7110,
     DIS_PDU_EMITTERNAME_AN_SPS_64_V_9 = 7119,
     DIS_PDU_EMITTERNAME_SPS64_V_12 = 7120,
     DIS_PDU_EMITTERNAME_AN_SPS_65 = 7155,
     DIS_PDU_EMITTERNAME_AN_SPS_66 = 7175,
     DIS_PDU_EMITTERNAME_AN_SPS_67 = 7200,
     DIS_PDU_EMITTERNAME_AN_SPS_73_I = 7201,
     DIS_PDU_EMITTERNAME_AN_SPS_69 = 7210,
     DIS_PDU_EMITTERNAME_AN_SPS_73 = 7215,
     DIS_PDU_EMITTERNAME_AN_SPS_74 = 7216,
     DIS_PDU_EMITTERNAME_AN_SPS_88 = 7225,
     DIS_PDU_EMITTERNAME_AN_SPS_501 = 7226,
     DIS_PDU_EMITTERNAME_AN_SPS_505 = 7230,
     DIS_PDU_EMITTERNAME_AN_SPY_1 = 7245,
     DIS_PDU_EMITTERNAME_AN_SPY_1A = 7250,
     DIS_PDU_EMITTERNAME_AN_SPY_1B = 7252,
     DIS_PDU_EMITTERNAME_AN_SPY_1B_V = 7253,
     DIS_PDU_EMITTERNAME_AN_SPY_1D = 7260,
     DIS_PDU_EMITTERNAME_AN_SPY_1D_V = 7261,
     DIS_PDU_EMITTERNAME_AN_SPY_1F = 7265,
     DIS_PDU_EMITTERNAME_AN_SPY_3 = 7266,
     DIS_PDU_EMITTERNAME_AN_TPN_12 = 7267,
     DIS_PDU_EMITTERNAME_AN_SPY_4 = 7268,
     DIS_PDU_EMITTERNAME_AN_TLQ_32_ARM_DECOY = 7269,
     DIS_PDU_EMITTERNAME_AN_TPN_17 = 7270,
     DIS_PDU_EMITTERNAME_AN_TPN_8 = 7271,
     DIS_PDU_EMITTERNAME_AN_TPN_22 = 7272,
     DIS_PDU_EMITTERNAME_AN_TLQ_17A = 7273,
     DIS_PDU_EMITTERNAME_AN_TMS_1 = 7274,
     DIS_PDU_EMITTERNAME_AN_TPN_24 = 7275,
     DIS_PDU_EMITTERNAME_AN_TPN_25 = 7276,
     DIS_PDU_EMITTERNAME_AN_TMS_2 = 7277,
     DIS_PDU_EMITTERNAME_AN_TPN_19 = 7278,
     DIS_PDU_EMITTERNAME_AN_TPN_31 = 7279,
     DIS_PDU_EMITTERNAME_AN_TPQ_18 = 7280,
     DIS_PDU_EMITTERNAME_AN_SPY_6_V = 7281,
     DIS_PDU_EMITTERNAME_AN_TPQ_36 = 7295,
     DIS_PDU_EMITTERNAME_AN_TPQ_37 = 7300,
     DIS_PDU_EMITTERNAME_AN_TPQ_38_V8 = 7301,
     DIS_PDU_EMITTERNAME_AN_TPQ_39_V = 7302,
     DIS_PDU_EMITTERNAME_AN_TPQ_47 = 7303,
     DIS_PDU_EMITTERNAME_AN_TPS_43 = 7305,
     DIS_PDU_EMITTERNAME_AN_TPS_43E = 7310,
     DIS_PDU_EMITTERNAME_AN_TPQ_48 = 7311,
     DIS_PDU_EMITTERNAME_AN_TPQ_49 = 7312,
     DIS_PDU_EMITTERNAME_AN_TPQ_46A = 7313,
     DIS_PDU_EMITTERNAME_AN_TPS_34 = 7314,
     DIS_PDU_EMITTERNAME_AN_TPS_59 = 7315,
     DIS_PDU_EMITTERNAME_AN_TPS_44 = 7316,
     DIS_PDU_EMITTERNAME_AN_TPQ_50 = 7317,
     DIS_PDU_EMITTERNAME_AN_TPS_63 = 7320,
     DIS_PDU_EMITTERNAME_AN_TPS_65 = 7321,
     DIS_PDU_EMITTERNAME_AN_TPS_70_V_1 = 7322,
     DIS_PDU_EMITTERNAME_AN_TPS_63SS = 7323,
     DIS_PDU_EMITTERNAME_AN_TPS_73 = 7324,
     DIS_PDU_EMITTERNAME_AN_TPS_75 = 7325,
     DIS_PDU_EMITTERNAME_AN_TPS_77 = 7326,
     DIS_PDU_EMITTERNAME_AN_TPS_78 = 7327,
     DIS_PDU_EMITTERNAME_AN_TPS_79 = 7328,
     DIS_PDU_EMITTERNAME_AN_TPS_703 = 7329,
     DIS_PDU_EMITTERNAME_AN_TPX_46_V_7 = 7330,
     DIS_PDU_EMITTERNAME_AN_TPS_80 = 7331,
     DIS_PDU_EMITTERNAME_AN_TPY_2 = 7333,
     DIS_PDU_EMITTERNAME_AN_TSQ_288 = 7334,
     DIS_PDU_EMITTERNAME_AN_ULQ_6A = 7335,
     DIS_PDU_EMITTERNAME_AN_ULQ_19 = 7340,
     DIS_PDU_EMITTERNAME_AN_ULQ_21 = 7345,
     DIS_PDU_EMITTERNAME_AN_UPN_25 = 7380,
     DIS_PDU_EMITTERNAME_AN_UPS_1 = 7425,
     DIS_PDU_EMITTERNAME_AN_UPS_2 = 7426,
     DIS_PDU_EMITTERNAME_AN_UPS_3 = 7427,
     DIS_PDU_EMITTERNAME_AN_UPX_1 = 7470,
     DIS_PDU_EMITTERNAME_AN_UPX_5 = 7515,
     DIS_PDU_EMITTERNAME_AN_UPX_11 = 7560,
     DIS_PDU_EMITTERNAME_AN_UPX_12 = 7605,
     DIS_PDU_EMITTERNAME_AN_UPX_17 = 7650,
     DIS_PDU_EMITTERNAME_AN_UPX_23 = 7695,
     DIS_PDU_EMITTERNAME_AN_USQ_113_V_3 = 7700,
     DIS_PDU_EMITTERNAME_AN_VPS_2 = 7740,
     DIS_PDU_EMITTERNAME_AN_PLM_3 = 7750,
     DIS_PDU_EMITTERNAME_AN_PLM_3A = 7751,
     DIS_PDU_EMITTERNAME_AN_PLM_4 = 7752,
     DIS_PDU_EMITTERNAME_AN_ZPY1 = 7753,
     DIS_PDU_EMITTERNAME_AN_ZPY_2_MP_RTIP = 7754,
     DIS_PDU_EMITTERNAME_AN_ZPY_3 = 7755,
     DIS_PDU_EMITTERNAME_AN_ZPY_8 = 7760,
     DIS_PDU_EMITTERNAME_AORL_1AS = 7761,
     DIS_PDU_EMITTERNAME_AORL_85K_TK_MTA = 7762,
     DIS_PDU_EMITTERNAME_APAR = 7765,
     DIS_PDU_EMITTERNAME_APARNA = 7770,
     DIS_PDU_EMITTERNAME_APECS_II = 7780,
     DIS_PDU_EMITTERNAME_APELCO_AD_7_7 = 7785,
     DIS_PDU_EMITTERNAME_APG_71 = 7830,
     DIS_PDU_EMITTERNAME_APN_148 = 7875,
     DIS_PDU_EMITTERNAME_APN_227 = 7920,
     DIS_PDU_EMITTERNAME_APS_504_V3 = 8100,
     DIS_PDU_EMITTERNAME_AQUITAINE_II = 8102,
     DIS_PDU_EMITTERNAME_AR_1 = 8103,
     DIS_PDU_EMITTERNAME_AR_3D = 8105,
     DIS_PDU_EMITTERNAME_PLESSEY_AR_5 = 8112,
     DIS_PDU_EMITTERNAME_AR_15 = 8113,
     DIS_PDU_EMITTERNAME_AR_15_2 = 8114,
     DIS_PDU_EMITTERNAME_AR_320 = 8115,
     DIS_PDU_EMITTERNAME_AR_325 = 8118,
     DIS_PDU_EMITTERNAME_AR_327 = 8120,
     DIS_PDU_EMITTERNAME_ARBALET_52 = 8121,
     DIS_PDU_EMITTERNAME_ARBB_31 = 8122,
     DIS_PDU_EMITTERNAME_ARBB_33 = 8123,
     DIS_PDU_EMITTERNAME_ARIES = 8126,
     DIS_PDU_EMITTERNAME_ARIES_NAV = 8127,
     DIS_PDU_EMITTERNAME_ARIES_CS = 8128,
     DIS_PDU_EMITTERNAME_ARGS_14E = 8134,
     DIS_PDU_EMITTERNAME_ARGS_31 = 8135,
     DIS_PDU_EMITTERNAME_ARGUS = 8140,
     DIS_PDU_EMITTERNAME_AR_M31 = 8145,
     DIS_PDU_EMITTERNAME_ARECIBO = 8150,
     DIS_PDU_EMITTERNAME_ARED = 8160,
     DIS_PDU_EMITTERNAME_ARI_5954 = 8190,
     DIS_PDU_EMITTERNAME_ARI_5955 = 8235,
     DIS_PDU_EMITTERNAME_ARI_5979 = 8280,
     DIS_PDU_EMITTERNAME_ARGSN_31 = 8281,
     DIS_PDU_EMITTERNAME_ARGOS_10 = 8282,
     DIS_PDU_EMITTERNAME_ARGOS_800 = 8283,
     DIS_PDU_EMITTERNAME_ARI_5983 = 8284,
     DIS_PDU_EMITTERNAME_ARI_5991 = 8285,
     DIS_PDU_EMITTERNAME_ARI_5995 = 8290,
     DIS_PDU_EMITTERNAME_ARINC_564_BNDX_KING_RDR_1E = 8325,
     DIS_PDU_EMITTERNAME_ARINC_700_BNDX_KING_RDR_1E = 8370,
     DIS_PDU_EMITTERNAME_ARK_1 = 8375,
     DIS_PDU_EMITTERNAME_ARL_M_MTI = 8378,
     DIS_PDU_EMITTERNAME_ARMOR = 8379,
     DIS_PDU_EMITTERNAME_ARSR_3 = 8380,
     DIS_PDU_EMITTERNAME_ARS_400 = 8381,
     DIS_PDU_EMITTERNAME_ARSR_1 = 8382,
     DIS_PDU_EMITTERNAME_ARSR_4 = 8384,
     DIS_PDU_EMITTERNAME_ARSR_18 = 8390,
     DIS_PDU_EMITTERNAME_ARTHUR = 8395,
     DIS_PDU_EMITTERNAME_ARTHUR_MOD_B = 8400,
     DIS_PDU_EMITTERNAME_ARTHUR_MOD_C = 8405,
     DIS_PDU_EMITTERNAME_ARTISAN_3D = 8410,
     DIS_PDU_EMITTERNAME_AS_2_KIPPER = 8415,
     DIS_PDU_EMITTERNAME_AS_2_KIPPER_MH = 8460,
     DIS_PDU_EMITTERNAME_AS_3_YJ_83K_MMW_MH = 8470,
     DIS_PDU_EMITTERNAME_AS_34_KORMORAN_SEEKER = 8480,
     DIS_PDU_EMITTERNAME_AS_4_KITCHEN = 8505,
     DIS_PDU_EMITTERNAME_AS_4_KITCHEN_MH = 8550,
     DIS_PDU_EMITTERNAME_AS_5_KELT_MH = 8595,
     DIS_PDU_EMITTERNAME_AS_6_KINGFISH_MH = 8640,
     DIS_PDU_EMITTERNAME_AS_7_KERRY = 8685,
     DIS_PDU_EMITTERNAME_AS_7_KERRY_MG = 8730,
     DIS_PDU_EMITTERNAME_AS_15_KENT_ALTIMETER = 8735,
     DIS_PDU_EMITTERNAME_AS_17A_KRYPTON_SEEKER = 8736,
     DIS_PDU_EMITTERNAME_AS_17B_KRYPTON_SEEKER = 8737,
     DIS_PDU_EMITTERNAME_AS901 = 8750,
     DIS_PDU_EMITTERNAME_AS901A = 8751,
     DIS_PDU_EMITTERNAME_ASARS2 = 8755,
     DIS_PDU_EMITTERNAME_ASDE_KDD = 8756,
     DIS_PDU_EMITTERNAME_ASLESHA = 8757,
     DIS_PDU_EMITTERNAME_A_SMGCS = 8758,
     DIS_PDU_EMITTERNAME_ASMI_18X = 8759,
     DIS_PDU_EMITTERNAME_ASPIDE_AAM_SAM_ILL = 8760,
     DIS_PDU_EMITTERNAME_ASMI_3 = 8761,
     DIS_PDU_EMITTERNAME_ASELSAN_MAR = 8762,
     DIS_PDU_EMITTERNAME_ASR_2000 = 8771,
     DIS_PDU_EMITTERNAME_ASR_4 = 8772,
     DIS_PDU_EMITTERNAME_ASR_4D = 8773,
     DIS_PDU_EMITTERNAME_ASR_O = 8775,
     DIS_PDU_EMITTERNAME_ASR_12 = 8776,
     DIS_PDU_EMITTERNAME_ASR_22AL = 8778,
     DIS_PDU_EMITTERNAME_ASR_3 = 8779,
     DIS_PDU_EMITTERNAME_ASR_5 = 8780,
     DIS_PDU_EMITTERNAME_ASR_7 = 8782,
     DIS_PDU_EMITTERNAME_ASR_8 = 8785,
     DIS_PDU_EMITTERNAME_ASR_9 = 8790,
     DIS_PDU_EMITTERNAME_ASR_9000 = 8791,
     DIS_PDU_EMITTERNAME_ASTI = 8792,
     DIS_PDU_EMITTERNAME_ASR_11_DASR = 8793,
     DIS_PDU_EMITTERNAME_RAYTHEON_ASR_10SS = 8812,
     DIS_PDU_EMITTERNAME_ASR_23SS = 8816,
     DIS_PDU_EMITTERNAME_ARABEL = 8818,
     DIS_PDU_EMITTERNAME_ASTRE = 8819,
     DIS_PDU_EMITTERNAME_AT_2_SWATTER_MG = 8820,
     DIS_PDU_EMITTERNAME_9K114_SHTURM_MG_AT_6_SPIRAL_MG = 8824,
     DIS_PDU_EMITTERNAME_ASTOR = 8825,
     DIS_PDU_EMITTERNAME_ASTRA_RCI = 8826,
     DIS_PDU_EMITTERNAME_ATCR_22 = 8830,
     DIS_PDU_EMITTERNAME_ATCR_22_M = 8831,
     DIS_PDU_EMITTERNAME_ATCR_2T = 8832,
     DIS_PDU_EMITTERNAME_ATCR_33 = 8840,
     DIS_PDU_EMITTERNAME_ATCR_33_K_M = 8845,
     DIS_PDU_EMITTERNAME_ATCR_33S = 8846,
     DIS_PDU_EMITTERNAME_ATCR_3T = 8847,
     DIS_PDU_EMITTERNAME_ATCR_44 = 8848,
     DIS_PDU_EMITTERNAME_ATCR_44_K = 8849,
     DIS_PDU_EMITTERNAME_ARGOS_73 = 8850,
     DIS_PDU_EMITTERNAME_ATCR_44_M_S = 8851,
     DIS_PDU_EMITTERNAME_ATCR_4T = 8852,
     DIS_PDU_EMITTERNAME_ATLAS_ELEKTRONK_TRS_N = 8865,
     DIS_PDU_EMITTERNAME_ATLAS_8600X = 8866,
     DIS_PDU_EMITTERNAME_ATLAS_9600M = 8867,
     DIS_PDU_EMITTERNAME_ATLAS_9600X = 8868,
     DIS_PDU_EMITTERNAME_ATLAS_9600S = 8869,
     DIS_PDU_EMITTERNAME_ATLAS_9740_VTS = 8870,
     DIS_PDU_EMITTERNAME_ATLASS = 8871,
     DIS_PDU_EMITTERNAME_ATR_500C = 8880,
     DIS_PDU_EMITTERNAME_AVG_65 = 8910,
     DIS_PDU_EMITTERNAME_AVH_7 = 8955,
     DIS_PDU_EMITTERNAME_AVIA_CM = 8980,
     DIS_PDU_EMITTERNAME_AVIA_D = 8985,
     DIS_PDU_EMITTERNAME_AVIACONVERSIA = 8990,
     DIS_PDU_EMITTERNAME_AVIACONVERSIA_II = 8993,
     DIS_PDU_EMITTERNAME_AVIACONVERSIA_III = 8995,
     DIS_PDU_EMITTERNAME_AVQ_20 = 9000,
     DIS_PDU_EMITTERNAME_AVQ_21 = 9005,
     DIS_PDU_EMITTERNAME_AVQ30X = 9045,
     DIS_PDU_EMITTERNAME_AVQ_50_RCA = 9075,
     DIS_PDU_EMITTERNAME_AVQ_70 = 9090,
     DIS_PDU_EMITTERNAME_AWS_5 = 9135,
     DIS_PDU_EMITTERNAME_AWS_6 = 9180,
     DIS_PDU_EMITTERNAME_AWS_6B_300 = 9185,
     DIS_PDU_EMITTERNAME_B597Z = 9200,
     DIS_PDU_EMITTERNAME_B636Z = 9205,
     DIS_PDU_EMITTERNAME_BACK_BOARD = 9215,
     DIS_PDU_EMITTERNAME_BACK_NET_A_B = 9225,
     DIS_PDU_EMITTERNAME_BACK_TRAP = 9270,
     DIS_PDU_EMITTERNAME_BAE_SYSTEMS_RT_1805_APN = 9280,
     DIS_PDU_EMITTERNAME_BAES_DASS_2000_JAMMER = 9281,
     DIS_PDU_EMITTERNAME_BALANCE_BEAM = 9285,
     DIS_PDU_EMITTERNAME_BALTIKA_B = 9300,
     DIS_PDU_EMITTERNAME_BALTYK = 9310,
     DIS_PDU_EMITTERNAME_BALL_END = 9315,
     DIS_PDU_EMITTERNAME_BALL_GUN = 9360,
     DIS_PDU_EMITTERNAME_BALL_POINT = 9370,
     DIS_PDU_EMITTERNAME_BAND_STAND = 9405,
     DIS_PDU_EMITTERNAME_BAND_STAND_3 = 9406,
     DIS_PDU_EMITTERNAME_P_35_37_A_P_50_B_BAR_LOCK = 9450,
     DIS_PDU_EMITTERNAME_BARAX = 9475,
     DIS_PDU_EMITTERNAME_BASIR_110D = 9485,
     DIS_PDU_EMITTERNAME_BASS_TILT = 9495,
     DIS_PDU_EMITTERNAME_BADGER = 9505,
     DIS_PDU_EMITTERNAME_BARRACUDA_JAMMER = 9510,
     DIS_PDU_EMITTERNAME_BAVAR_373_TTR = 9511,
     DIS_PDU_EMITTERNAME_BAVAR_373_TAR = 9512,
     DIS_PDU_EMITTERNAME_BAVAR_373_TELAR_TER = 9520,
     DIS_PDU_EMITTERNAME_BAYKAL_COUNTERMEASURES_SUITE = 9530,
     DIS_PDU_EMITTERNAME_BEACON = 9540,
     DIS_PDU_EMITTERNAME_BEAN_STICKS = 9585,
     DIS_PDU_EMITTERNAME_BEE_HIND = 9630,
     DIS_PDU_EMITTERNAME_BELL_NIP_JAMMER = 9638,
     DIS_PDU_EMITTERNAME_BELL_PUSH_JAMMER = 9639,
     DIS_PDU_EMITTERNAME_BELL_CROWN_A = 9640,
     DIS_PDU_EMITTERNAME_BELL_CROWN_B = 9642,
     DIS_PDU_EMITTERNAME_BELL_SQUAT = 9643,
     DIS_PDU_EMITTERNAME_BIG_BACK = 9645,
     DIS_PDU_EMITTERNAME_BIG_BIRD_A_B_C = 9659,
     DIS_PDU_EMITTERNAME_BIG_BIRD_D = 9660,
     DIS_PDU_EMITTERNAME_BIG_BIRD_D_MOD = 9661,
     DIS_PDU_EMITTERNAME_BIG_BIRD_E_91N6E = 9662,
     DIS_PDU_EMITTERNAME_BIG_BULGE = 9675,
     DIS_PDU_EMITTERNAME_BIG_BULGE_A = 9720,
     DIS_PDU_EMITTERNAME_BIG_BULGE_B = 9765,
     DIS_PDU_EMITTERNAME_BIG_EYE = 9775,
     DIS_PDU_EMITTERNAME_SNAR_10_BIG_FRED = 9780,
     DIS_PDU_EMITTERNAME_BIG_HEAD_B = 9781,
     DIS_PDU_EMITTERNAME_BIG_MESH = 9810,
     DIS_PDU_EMITTERNAME_BIG_NET = 9855,
     DIS_PDU_EMITTERNAME_BIG_REST_B = 9875,
     DIS_PDU_EMITTERNAME_9S15MT_BILL_BOARD = 9885,
     DIS_PDU_EMITTERNAME_BILL_FOLD = 9900,
     DIS_PDU_EMITTERNAME_BLIGHTER_400 = 9903,
     DIS_PDU_EMITTERNAME_BLOWPIPE_MG = 9905,
     DIS_PDU_EMITTERNAME_BLR = 9920,
     DIS_PDU_EMITTERNAME_BLUE_FOX_SEA_HARRIER_FRS_MK_1_5 = 9930,
     DIS_PDU_EMITTERNAME_BLUE_KESTREL = 9933,
     DIS_PDU_EMITTERNAME_BLUE_VIXEN_SEA_HARRIER_F_A_MK_2 = 9935,
     DIS_PDU_EMITTERNAME_BLUE_SILK = 9945,
     DIS_PDU_EMITTERNAME_BLUE_PARROT = 9990,
     DIS_PDU_EMITTERNAME_BLUE_ORCHID = 10035,
     DIS_PDU_EMITTERNAME_BM_DJG_8715 = 10057,
     DIS_PDU_EMITTERNAME_BOAT_SAIL = 10080,
     DIS_PDU_EMITTERNAME_BOR_A_550 = 10090,
     DIS_PDU_EMITTERNAME_BOFORS_ELECTRONIC_9LV_331 = 10125,
     DIS_PDU_EMITTERNAME_BOFORS_ERICSSON_SEA_GIRAFFE_50_HC = 10170,
     DIS_PDU_EMITTERNAME_BOWL_MESH = 10215,
     DIS_PDU_EMITTERNAME_BOX_BRICK = 10260,
     DIS_PDU_EMITTERNAME_BOX_TAIL = 10305,
     DIS_PDU_EMITTERNAME_BM_KG300G_JAMMING_POD = 10308,
     DIS_PDU_EMITTERNAME_BM_KG600_JAMMING_POD = 10310,
     DIS_PDU_EMITTERNAME_BM_KG800_JAMMING_POD = 10312,
     DIS_PDU_EMITTERNAME_BM_KG_8601_8605_8606 = 10315,
     DIS_PDU_EMITTERNAME_BPS_11A = 10350,
     DIS_PDU_EMITTERNAME_BPS_14 = 10395,
     DIS_PDU_EMITTERNAME_BPS_15A = 10440,
     DIS_PDU_EMITTERNAME_BR_3440CA_X57 = 10450,
     DIS_PDU_EMITTERNAME_BR_15_TOKYO_KEIKI = 10485,
     DIS_PDU_EMITTERNAME_BRAHMOS = 10500,
     DIS_PDU_EMITTERNAME_BRIDGEMASTER = 10510,
     DIS_PDU_EMITTERNAME_BRIDGEMASTER_E_EPA = 10511,
     DIS_PDU_EMITTERNAME_BRIDGEMASTER_E_ATA_AND_ARPA = 10512,
     DIS_PDU_EMITTERNAME_BRIDGEMASTER_E_NAVAL = 10513,
     DIS_PDU_EMITTERNAME_BRIMSTONE_MMW_MH = 10520,
     DIS_PDU_EMITTERNAME_BREAD_BIN = 10530,
     DIS_PDU_EMITTERNAME_ASR_BRICK_BOARD = 10540,
     DIS_PDU_EMITTERNAME_BT_271 = 10575,
     DIS_PDU_EMITTERNAME_BU_304 = 10595,
     DIS_PDU_EMITTERNAME_BX_732 = 10620,
     DIS_PDU_EMITTERNAME_BUK_MB = 10630,
     DIS_PDU_EMITTERNAME_BURAN_D = 10642,
     DIS_PDU_EMITTERNAME_BUREVISNYK_1 = 10650,
     DIS_PDU_EMITTERNAME_BUZZ_STAND = 10665,
     DIS_PDU_EMITTERNAME_C_5A_MULTI_MODE_RADAR = 10710,
     DIS_PDU_EMITTERNAME_C_802_AL = 10711,
     DIS_PDU_EMITTERNAME_CAESAR = 10740,
     DIS_PDU_EMITTERNAME_CAIMAN = 10755,
     DIS_PDU_EMITTERNAME_CAKE_STAND = 10800,
     DIS_PDU_EMITTERNAME_CALYPSO_C61 = 10845,
     DIS_PDU_EMITTERNAME_CALYPSO_C63 = 10846,
     DIS_PDU_EMITTERNAME_CALYPSO_II = 10890,
     DIS_PDU_EMITTERNAME_CALYPSO_III = 10891,
     DIS_PDU_EMITTERNAME_CALYPSO_IV = 10892,
     DIS_PDU_EMITTERNAME_CARDION_COASTAL = 10895,
     DIS_PDU_EMITTERNAME_CASTOR_II = 10935,
     DIS_PDU_EMITTERNAME_CASTOR_2J_TT_CROTALE_NG = 10940,
     DIS_PDU_EMITTERNAME_CAT_HOUSE = 10980,
     DIS_PDU_EMITTERNAME_CDR_431 = 10985,
     DIS_PDU_EMITTERNAME_CEAFAR = 10987,
     DIS_PDU_EMITTERNAME_CEAMOUNT = 10988,
     DIS_PDU_EMITTERNAME_CEAFAR2_L = 10989,
     DIS_PDU_EMITTERNAME_CEROS_200 = 10990,
     DIS_PDU_EMITTERNAME_CEROS_200_CWI = 10991,
     DIS_PDU_EMITTERNAME_CEATAC = 10992,
     DIS_PDU_EMITTERNAME_CEAOPS = 10993,
     DIS_PDU_EMITTERNAME_CERBERUS_III = 10994,
     DIS_PDU_EMITTERNAME_CH_SS_N_6 = 10995,
     DIS_PDU_EMITTERNAME_CERBERUS_IV = 10996,
     DIS_PDU_EMITTERNAME_CHAIR_BACK_TT = 11000,
     DIS_PDU_EMITTERNAME_CHAIR_BACK_ILL = 11010,
     DIS_PDU_EMITTERNAME_LEMZ_96L6_CHEESE_BOARD = 11020,
     DIS_PDU_EMITTERNAME_CHEESE_BRICK = 11025,
     DIS_PDU_EMITTERNAME_CHEESE_CAKE = 11030,
     DIS_PDU_EMITTERNAME_CHROME_DOME = 11060,
     DIS_PDU_EMITTERNAME_LENINETZ_OBZOR_MS_CLAM_PIPE = 11070,
     DIS_PDU_EMITTERNAME_CLAMSHELL = 11115,
     DIS_PDU_EMITTERNAME_CLC_1 = 11117,
     DIS_PDU_EMITTERNAME_CLC_2 = 11118,
     DIS_PDU_EMITTERNAME_CLC_3 = 11119,
     DIS_PDU_EMITTERNAME_CLR_155 = 11120,
     DIS_PDU_EMITTERNAME_COAST_WATCHER_100 = 11123,
     DIS_PDU_EMITTERNAME_COASTAL_GIRAFFE = 11125,
     DIS_PDU_EMITTERNAME_COBRA = 11130,
     DIS_PDU_EMITTERNAME_COBRA_SHOE = 11133,
     DIS_PDU_EMITTERNAME_COLIBRI = 11137,
     DIS_PDU_EMITTERNAME_COLLINS_WXR_300 = 11155,
     DIS_PDU_EMITTERNAME_COLLINS_WXR_700X = 11160,
     DIS_PDU_EMITTERNAME_COLLINS_TWR_850_COLLINS_WXR_840 = 11165,
     DIS_PDU_EMITTERNAME_COLLINS_DN_101 = 11205,
     DIS_PDU_EMITTERNAME_COMET_1 = 11230,
     DIS_PDU_EMITTERNAME_CONDOR_MK_2 = 11235,
     DIS_PDU_EMITTERNAME_CONSILIUM_SELESMAR_RTM_25_XIM = 11240,
     DIS_PDU_EMITTERNAME_CONTRAVES_SEA_HUNTER_MK_4 = 11250,
     DIS_PDU_EMITTERNAME_CORN_CAN = 11260,
     DIS_PDU_EMITTERNAME_COSMO_SKYMED_1 = 11265,
     DIS_PDU_EMITTERNAME_CR_105_RMCA = 11270,
     DIS_PDU_EMITTERNAME_CREW_DUKE_2 = 11280,
     DIS_PDU_EMITTERNAME_CREW_DUKE_3 = 11290,
     DIS_PDU_EMITTERNAME_CROSS_BIRD = 11295,
     DIS_PDU_EMITTERNAME_CROSS_DOME = 11340,
     DIS_PDU_EMITTERNAME_CROSS_LEGS = 11385,
     DIS_PDU_EMITTERNAME_CROSS_OUT = 11430,
     DIS_PDU_EMITTERNAME_CROSS_SLOT = 11475,
     DIS_PDU_EMITTERNAME_CROSS_SWORD = 11520,
     DIS_PDU_EMITTERNAME_CROSS_UP = 11565,
     DIS_PDU_EMITTERNAME_CROSS_SWORD_FC = 11610,
     DIS_PDU_EMITTERNAME_CROTALE_ACQUISITION_TA_THD_5000 = 11655,
     DIS_PDU_EMITTERNAME_CROTALE_NG_TA_GRIFFON = 11660,
     DIS_PDU_EMITTERNAME_CROTALE_TT = 11665,
     DIS_PDU_EMITTERNAME_CROTALE_MGMISSILE_SYSTEM = 11700,
     DIS_PDU_EMITTERNAME_CS_10_TA = 11715,
     DIS_PDU_EMITTERNAME_CSA_20_TER = 11717,
     DIS_PDU_EMITTERNAME_CSA_20_TAR = 11718,
     DIS_PDU_EMITTERNAME_CSA_23_FDV_MG = 11719,
     DIS_PDU_EMITTERNAME_CSA_23_MISSILE_MH = 11720,
     DIS_PDU_EMITTERNAME_CSF_VARAN = 11725,
     DIS_PDU_EMITTERNAME_CSS_N_4_MH = 11735,
     DIS_PDU_EMITTERNAME_CSS_C_3C_CAS_1M1_M2_MH = 11745,
     DIS_PDU_EMITTERNAME_HY_2B_MH_CSS_C_3_SEERSUCKER_MH = 11748,
     DIS_PDU_EMITTERNAME_CSS_C_2B_HY_1A_MH = 11790,
     DIS_PDU_EMITTERNAME_CSS_N_4_SARDINE = 11800,
     DIS_PDU_EMITTERNAME_CSS_N_8_SACCADE = 11810,
     DIS_PDU_EMITTERNAME_CURL_STONE_B = 11825,
     DIS_PDU_EMITTERNAME_CWS_1 = 11830,
     DIS_PDU_EMITTERNAME_CWS_2 = 11835,
     DIS_PDU_EMITTERNAME_CWS_3 = 11840,
     DIS_PDU_EMITTERNAME_CYGNUS = 11860,
     DIS_PDU_EMITTERNAME_CYLINDER_HEAD = 11880,
     DIS_PDU_EMITTERNAME_CYMBELINE = 11902,
     DIS_PDU_EMITTERNAME_CYRANO_II = 11925,
     DIS_PDU_EMITTERNAME_CYRANO_IV = 11970,
     DIS_PDU_EMITTERNAME_CYRANO_IV_M = 11975,
     DIS_PDU_EMITTERNAME_DA_01_00 = 12010,
     DIS_PDU_EMITTERNAME_DA_05_00 = 12015,
     DIS_PDU_EMITTERNAME_DA_05_2 = 12016,
     DIS_PDU_EMITTERNAME_DA_08 = 12018,
     DIS_PDU_EMITTERNAME_DAWN = 12060,
     DIS_PDU_EMITTERNAME_DCR = 12090,
     DIS_PDU_EMITTERNAME_DEAD_DUCK = 12105,
     DIS_PDU_EMITTERNAME_DECCA_20_V90_9 = 12110,
     DIS_PDU_EMITTERNAME_DECCA_20_V90S = 12111,
     DIS_PDU_EMITTERNAME_DECCA_45 = 12150,
     DIS_PDU_EMITTERNAME_DECCA_50 = 12195,
     DIS_PDU_EMITTERNAME_DECCA_71 = 12196,
     DIS_PDU_EMITTERNAME_DECCA_72 = 12197,
     DIS_PDU_EMITTERNAME_DECCA_110 = 12240,
     DIS_PDU_EMITTERNAME_DECCA_170 = 12285,
     DIS_PDU_EMITTERNAME_DECCA_HF_2 = 12292,
     DIS_PDU_EMITTERNAME_DECCA_202 = 12330,
     DIS_PDU_EMITTERNAME_DECCA_D202 = 12375,
     DIS_PDU_EMITTERNAME_DECCA_303 = 12420,
     DIS_PDU_EMITTERNAME_DECCA_535 = 12430,
     DIS_PDU_EMITTERNAME_DECCA_626 = 12465,
     DIS_PDU_EMITTERNAME_DECCA_629 = 12510,
     DIS_PDU_EMITTERNAME_DECCA_914 = 12555,
     DIS_PDU_EMITTERNAME_DECCA_916 = 12600,
     DIS_PDU_EMITTERNAME_DECCA_926 = 12610,
     DIS_PDU_EMITTERNAME_DECCA_1070A = 12615,
     DIS_PDU_EMITTERNAME_DECCA_1008 = 12616,
     DIS_PDU_EMITTERNAME_DECCA_1226_COMMERCIAL = 12645,
     DIS_PDU_EMITTERNAME_DECCA_1290 = 12655,
     DIS_PDU_EMITTERNAME_DECCA_1626 = 12690,
     DIS_PDU_EMITTERNAME_DECCA_2070 = 12691,
     DIS_PDU_EMITTERNAME_DECCA_1630 = 12694,
     DIS_PDU_EMITTERNAME_DECCA_2459 = 12735,
     DIS_PDU_EMITTERNAME_DECCA_AWS_1 = 12780,
     DIS_PDU_EMITTERNAME_DECCA_AWS_2 = 12782,
     DIS_PDU_EMITTERNAME_DECCA_AWS_4 = 12785,
     DIS_PDU_EMITTERNAME_DECCA_AWS_4_2 = 12787,
     DIS_PDU_EMITTERNAME_DECCA_MAR = 12800,
     DIS_PDU_EMITTERNAME_DECCA_RM_326 = 12805,
     DIS_PDU_EMITTERNAME_DECCA_RM_416 = 12825,
     DIS_PDU_EMITTERNAME_DECCA_RM_970BT = 12850,
     DIS_PDU_EMITTERNAME_DECCA_RM_914 = 12870,
     DIS_PDU_EMITTERNAME_DF_21D_SEEKER = 12875,
     DIS_PDU_EMITTERNAME_DECCA_RM_1690 = 12915,
     DIS_PDU_EMITTERNAME_DECCA_1690 = 12916,
     DIS_PDU_EMITTERNAME_DECCA_SUPER_101_MK_3 = 12960,
     DIS_PDU_EMITTERNAME_DISS_1 = 13005,
     DIS_PDU_EMITTERNAME_DISS_7 = 13006,
     DIS_PDU_EMITTERNAME_DISS_013 = 13007,
     DIS_PDU_EMITTERNAME_DISS_15D = 13015,
     DIS_PDU_EMITTERNAME_DLD_100A = 13020,
     DIS_PDU_EMITTERNAME_RAPIER_TTDN_181_DN_181 = 13050,
     DIS_PDU_EMITTERNAME_RAPIER_2000_TT_BLINDFIRE_FSC_TT = 13055,
     DIS_PDU_EMITTERNAME_DOG_EAR = 13095,
     DIS_PDU_EMITTERNAME_DOG_HOUSE = 13140,
     DIS_PDU_EMITTERNAME_DM3 = 13141,
     DIS_PDU_EMITTERNAME_DM_3B = 13142,
     DIS_PDU_EMITTERNAME_DM_5 = 13143,
     DIS_PDU_EMITTERNAME_DON_2 = 13185,
     DIS_PDU_EMITTERNAME_DON_A_B_2_KAY = 13230,
     DIS_PDU_EMITTERNAME_DONETS = 13275,
     DIS_PDU_EMITTERNAME_DOPPLER_90_SERIES = 13280,
     DIS_PDU_EMITTERNAME_DOWN_BEAT = 13320,
     DIS_PDU_EMITTERNAME_DR_582 = 13360,
     DIS_PDU_EMITTERNAME_DRAA_2A = 13365,
     DIS_PDU_EMITTERNAME_DRAA_2B = 13410,
     DIS_PDU_EMITTERNAME_DRAA_9A = 13415,
     DIS_PDU_EMITTERNAME_DRAA_11A = 13420,
     DIS_PDU_EMITTERNAME_DRAC_37B = 13450,
     DIS_PDU_EMITTERNAME_DRAC_38 = 13452,
     DIS_PDU_EMITTERNAME_DRAC_39 = 13455,
     DIS_PDU_EMITTERNAME_DRAC_39A = 13456,
     DIS_PDU_EMITTERNAME_DRAC_43A = 13460,
     DIS_PDU_EMITTERNAME_DRAC_44A = 13465,
     DIS_PDU_EMITTERNAME_DRAGON_EYE = 13477,
     DIS_PDU_EMITTERNAME_DRAGON_EYE_2 = 13480,
     DIS_PDU_EMITTERNAME_DRAGON_EYE_3 = 13481,
     DIS_PDU_EMITTERNAME_DRAGON_EYE_4 = 13485,
     DIS_PDU_EMITTERNAME_DRBC_30B = 13500,
     DIS_PDU_EMITTERNAME_DRBC_31A = 13545,
     DIS_PDU_EMITTERNAME_DRBC_31D = 13546,
     DIS_PDU_EMITTERNAME_DRBC_32 = 13585,
     DIS_PDU_EMITTERNAME_DRBC_32A = 13590,
     DIS_PDU_EMITTERNAME_DRBC_32D = 13635,
     DIS_PDU_EMITTERNAME_DRBC_33A = 13680,
     DIS_PDU_EMITTERNAME_DRBI_10 = 13725,
     DIS_PDU_EMITTERNAME_DRBI_23 = 13770,
     DIS_PDU_EMITTERNAME_DRBJ_11B = 13815,
     DIS_PDU_EMITTERNAME_DRBN_30 = 13860,
     DIS_PDU_EMITTERNAME_DRBN_32 = 13905,
     DIS_PDU_EMITTERNAME_DRBN_34 = 13915,
     DIS_PDU_EMITTERNAME_DRBR_51 = 13950,
     DIS_PDU_EMITTERNAME_DRBV_20A = 13994,
     DIS_PDU_EMITTERNAME_DRBV_20B = 13995,
     DIS_PDU_EMITTERNAME_DRBV_21_MARS_05 = 14020,
     DIS_PDU_EMITTERNAME_DRBV_22 = 14040,
     DIS_PDU_EMITTERNAME_DRBV_23 = 14041,
     DIS_PDU_EMITTERNAME_DRBV_26C = 14085,
     DIS_PDU_EMITTERNAME_DRBV_26D = 14086,
     DIS_PDU_EMITTERNAME_DRBV_30 = 14130,
     DIS_PDU_EMITTERNAME_DRBV_31 = 14131,
     DIS_PDU_EMITTERNAME_DRBV_50 = 14175,
     DIS_PDU_EMITTERNAME_DRBV_51 = 14220,
     DIS_PDU_EMITTERNAME_DRBV_51A = 14265,
     DIS_PDU_EMITTERNAME_DRBV_51B = 14310,
     DIS_PDU_EMITTERNAME_DRBV_51C = 14355,
     DIS_PDU_EMITTERNAME_DROP_KICK = 14400,
     DIS_PDU_EMITTERNAME_DRUA_31 = 14445,
     DIS_PDU_EMITTERNAME_DRUM_TILT = 14490,
     DIS_PDU_EMITTERNAME_DRUM_TILT_A = 14535,
     DIS_PDU_EMITTERNAME_DRUM_TILT_B = 14545,
     DIS_PDU_EMITTERNAME_DRUN_30A = 14560,
     DIS_PDU_EMITTERNAME_DUMBO = 14580,
     DIS_PDU_EMITTERNAME_DWSR_92 = 14583,
     DIS_PDU_EMITTERNAME_DWSR_93S = 14585,
     DIS_PDU_EMITTERNAME_EAGLE = 14586,
     DIS_PDU_EMITTERNAME_EAGLE_MK_1 = 14587,
     DIS_PDU_EMITTERNAME_EAJP_JAMMING_POD = 14588,
     DIS_PDU_EMITTERNAME_EKCO_E390 = 14590,
     DIS_PDU_EMITTERNAME_ECR_90_CAPTOR = 14600,
     DIS_PDU_EMITTERNAME_ECR_90_JAMMER = 14601,
     DIS_PDU_EMITTERNAME_EGG_CUP_A_B = 14625,
     DIS_PDU_EMITTERNAME_EISCAT = 14640,
     DIS_PDU_EMITTERNAME_EKCO_E120 = 14660,
     DIS_PDU_EMITTERNAME_EKCO_190 = 14670,
     DIS_PDU_EMITTERNAME_EKRAN_1 = 14677,
     DIS_PDU_EMITTERNAME_EL_L_8222 = 14710,
     DIS_PDU_EMITTERNAME_EL_L_8240 = 14713,
     DIS_PDU_EMITTERNAME_EL_M_2001B = 14715,
     DIS_PDU_EMITTERNAME_EL_M_2022 = 14725,
     DIS_PDU_EMITTERNAME_EL_M_2032 = 14726,
     DIS_PDU_EMITTERNAME_EL_M_2052 = 14727,
     DIS_PDU_EMITTERNAME_EL_M_2055 = 14728,
     DIS_PDU_EMITTERNAME_EL_M_2060 = 14730,
     DIS_PDU_EMITTERNAME_EL_M_2075 = 14735,
     DIS_PDU_EMITTERNAME_EL_M_2022U_V_3 = 14736,
     DIS_PDU_EMITTERNAME_EL_M_2080 = 14737,
     DIS_PDU_EMITTERNAME_EL_M_2080S = 14738,
     DIS_PDU_EMITTERNAME_EL_M_2085 = 14739,
     DIS_PDU_EMITTERNAME_EL_M_2106 = 14740,
     DIS_PDU_EMITTERNAME_EL_M_2106NG = 14741,
     DIS_PDU_EMITTERNAME_EL_M_2125 = 14742,
     DIS_PDU_EMITTERNAME_EL_M_2129 = 14743,
     DIS_PDU_EMITTERNAME_EL_M_2150 = 14744,
     DIS_PDU_EMITTERNAME_EL_M_2083 = 14745,
     DIS_PDU_EMITTERNAME_EL_M_2084 = 14746,
     DIS_PDU_EMITTERNAME_EL_M_2160_V1 = 14747,
     DIS_PDU_EMITTERNAME_EL_M_2084_MMR = 14748,
     DIS_PDU_EMITTERNAME_EL_M_2112 = 14749,
     DIS_PDU_EMITTERNAME_EL_M_2200 = 14750,
     DIS_PDU_EMITTERNAME_EL_M_2133 = 14751,
     DIS_PDU_EMITTERNAME_EL_M_2205 = 14755,
     DIS_PDU_EMITTERNAME_EL_M_2207 = 14760,
     DIS_PDU_EMITTERNAME_EL_M_2215 = 14765,
     DIS_PDU_EMITTERNAME_EL_M_2216_V = 14770,
     DIS_PDU_EMITTERNAME_EL_M_2216XH = 14772,
     DIS_PDU_EMITTERNAME_EL_M_2218S = 14775,
     DIS_PDU_EMITTERNAME_ELT_361 = 14776,
     DIS_PDU_EMITTERNAME_EL_M_2258 = 14777,
     DIS_PDU_EMITTERNAME_ELT_553 = 14779,
     DIS_PDU_EMITTERNAME_ELT_558 = 14780,
     DIS_PDU_EMITTERNAME_ELT_572 = 14785,
     DIS_PDU_EMITTERNAME_ELT_715 = 14790,
     DIS_PDU_EMITTERNAME_ELTA_ELM_2022A = 14800,
     DIS_PDU_EMITTERNAME_ELTA_EL_M_2221_GM_STGR = 14805,
     DIS_PDU_EMITTERNAME_EL_M_2228S_3D = 14806,
     DIS_PDU_EMITTERNAME_EL_M_2705 = 14807,
     DIS_PDU_EMITTERNAME_EL_M_2226 = 14808,
     DIS_PDU_EMITTERNAME_EL_M_2228X = 14809,
     DIS_PDU_EMITTERNAME_ELTA_SIS = 14810,
     DIS_PDU_EMITTERNAME_EL_M_2238 = 14811,
     DIS_PDU_EMITTERNAME_EL_M_2248 = 14815,
     DIS_PDU_EMITTERNAME_EL_M_2288 = 14820,
     DIS_PDU_EMITTERNAME_EL_M_2311 = 14821,
     DIS_PDU_EMITTERNAME_ELM_2026 = 14822,
     DIS_PDU_EMITTERNAME_ELNA_4007 = 14830,
     DIS_PDU_EMITTERNAME_ELT_318 = 14831,
     DIS_PDU_EMITTERNAME_ELW_2085 = 14832,
     DIS_PDU_EMITTERNAME_ELT_521 = 14833,
     DIS_PDU_EMITTERNAME_ELW_2090 = 14835,
     DIS_PDU_EMITTERNAME_ENHANCED_METEOR_DETECTION_RADAR_EMDR = 14845,
     DIS_PDU_EMITTERNAME_EMD_2900 = 14850,
     DIS_PDU_EMITTERNAME_EMPAR = 14851,
     DIS_PDU_EMITTERNAME_END_TRAY = 14895,
     DIS_PDU_EMITTERNAME_EQ_36 = 14896,
     DIS_PDU_EMITTERNAME_ERICSSON_SLAR = 14897,
     DIS_PDU_EMITTERNAME_ERIEYE = 14898,
     DIS_PDU_EMITTERNAME_ES_3701_ECM_ESM_SUITE = 14899,
     DIS_PDU_EMITTERNAME_ESR_1 = 14900,
     DIS_PDU_EMITTERNAME_ESR_220 = 14901,
     DIS_PDU_EMITTERNAME_ESR380 = 14902,
     DIS_PDU_EMITTERNAME_ESTEREL = 14903,
     DIS_PDU_EMITTERNAME_ET_316 = 14905,
     DIS_PDU_EMITTERNAME_EXOCET_TYPE = 14935,
     DIS_PDU_EMITTERNAME_EXOCET_AL = 14936,
     DIS_PDU_EMITTERNAME_EXOCET_1 = 14940,
     DIS_PDU_EMITTERNAME_EXOCET_1_MH = 14985,
     DIS_PDU_EMITTERNAME_EXOCET_2 = 15030,
     DIS_PDU_EMITTERNAME_EYE_BOWL = 15075,
     DIS_PDU_EMITTERNAME_EYE_SHIELD = 15120,
     DIS_PDU_EMITTERNAME_F332Z = 15140,
     DIS_PDU_EMITTERNAME_FALCON_CLAW_TI = 15155,
     DIS_PDU_EMITTERNAME_FALCON_CLAW_TT = 15156,
     DIS_PDU_EMITTERNAME_FALCON = 15160,
     DIS_PDU_EMITTERNAME_FALCON_G = 15161,
     DIS_PDU_EMITTERNAME_FALCON_EYE = 15163,
     DIS_PDU_EMITTERNAME_FAN_SONG_A = 15165,
     DIS_PDU_EMITTERNAME_FAN_SONG_B_F_TA = 15200,
     DIS_PDU_EMITTERNAME_FAN_SONG_B_F_TT = 15210,
     DIS_PDU_EMITTERNAME_FAN_SONG_C_E_TA = 15220,
     DIS_PDU_EMITTERNAME_FAN_SONG_C_E_TT = 15230,
     DIS_PDU_EMITTERNAME_FAN_SONG_C_E_MG = 15240,
     DIS_PDU_EMITTERNAME_FAN_SONG_B_FF_MG = 15255,
     DIS_PDU_EMITTERNAME_FAN_TAIL = 15300,
     DIS_PDU_EMITTERNAME_FAR_2117 = 15301,
     DIS_PDU_EMITTERNAME_FAR_2827 = 15302,
     DIS_PDU_EMITTERNAME_FAR_2837S = 15303,
     DIS_PDU_EMITTERNAME_FAR_3000 = 15304,
     DIS_PDU_EMITTERNAME_FB_7_RADAR = 15305,
     DIS_PDU_EMITTERNAME_FCR_1401 = 15310,
     DIS_PDU_EMITTERNAME_FCS_2_12E = 15312,
     DIS_PDU_EMITTERNAME_FCS_2_12G = 15313,
     DIS_PDU_EMITTERNAME_FCS_2_21A = 15315,
     DIS_PDU_EMITTERNAME_FCS_2_21C = 15317,
     DIS_PDU_EMITTERNAME_FCS_2_22 = 15318,
     DIS_PDU_EMITTERNAME_FCS_2_31 = 15319,
     DIS_PDU_EMITTERNAME_FCS_3 = 15320,
     DIS_PDU_EMITTERNAME_FIN_CURVE = 15345,
     DIS_PDU_EMITTERNAME_FIRE_CAN = 15390,
     DIS_PDU_EMITTERNAME_FIRE_DISH = 15435,
     DIS_PDU_EMITTERNAME_FIRE_DOME_TA = 15470,
     DIS_PDU_EMITTERNAME_FIRE_DOME_TT = 15475,
     DIS_PDU_EMITTERNAME_FIRE_DOME_TI = 15480,
     DIS_PDU_EMITTERNAME_FIRE_IRON = 15525,
     DIS_PDU_EMITTERNAME_FIRE_WHEEL = 15570,
     DIS_PDU_EMITTERNAME_FISH_BOWL = 15615,
     DIS_PDU_EMITTERNAME_FK_3 = 15620,
     DIS_PDU_EMITTERNAME_FLAIR = 15650,
     DIS_PDU_EMITTERNAME_FLAP_LID = 15660,
     DIS_PDU_EMITTERNAME_30N6E_FLAP_LID_B = 15661,
     DIS_PDU_EMITTERNAME_FLAP_TRUCK = 15705,
     DIS_PDU_EMITTERNAME_FLAP_WHEEL = 15750,
     DIS_PDU_EMITTERNAME_FLASH_DANCE = 15795,
     DIS_PDU_EMITTERNAME_FLASH_DANCE_M = 15800,
     DIS_PDU_EMITTERNAME_P_15_FLAT_FACE_A_B_C_D = 15840,
     DIS_PDU_EMITTERNAME_35N6_FLAT_FACE_E = 15842,
     DIS_PDU_EMITTERNAME_FLAT_SCREEN = 15885,
     DIS_PDU_EMITTERNAME_FLAT_SPIN = 15930,
     DIS_PDU_EMITTERNAME_FLAT_TRACK_JAMMER = 15970,
     DIS_PDU_EMITTERNAME_FLAT_TWIN = 15975,
     DIS_PDU_EMITTERNAME_FL_400 = 15980,
     DIS_PDU_EMITTERNAME_FL_1800 = 15985,
     DIS_PDU_EMITTERNAME_FL_1800U = 15990,
     DIS_PDU_EMITTERNAME_FL_1800S = 16000,
     DIS_PDU_EMITTERNAME_FLEDERMAUS = 16020,
     DIS_PDU_EMITTERNAME_FLYCATCHER = 16030,
     DIS_PDU_EMITTERNAME_FLYCATCHER_MK_2 = 16035,
     DIS_PDU_EMITTERNAME_FLY_SCREEN = 16065,
     DIS_PDU_EMITTERNAME_FLY_SCREEN_A_B = 16110,
     DIS_PDU_EMITTERNAME_FLY_TRAP_B = 16155,
     DIS_PDU_EMITTERNAME_FM_90 = 16160,
     DIS_PDU_EMITTERNAME_FOG_LAMP_MG = 16200,
     DIS_PDU_EMITTERNAME_FOG_LAMP_TT = 16245,
     DIS_PDU_EMITTERNAME_FOIL_TWO = 16290,
     DIS_PDU_EMITTERNAME_FOOTBALL = 16300,
     DIS_PDU_EMITTERNAME_FOX_HUNTER = 16335,
     DIS_PDU_EMITTERNAME_FOX_FIRE_AL = 16380,
     DIS_PDU_EMITTERNAME_FOX_FIRE_ILL = 16390,
     DIS_PDU_EMITTERNAME_FR_151A = 16400,
     DIS_PDU_EMITTERNAME_FURUNO_FR_1500_FR_1600 = 16405,
     DIS_PDU_EMITTERNAME_FR_1505_DA = 16410,
     DIS_PDU_EMITTERNAME_FR_1510DS = 16412,
     DIS_PDU_EMITTERNAME_FR_2000 = 16420,
     DIS_PDU_EMITTERNAME_FURUNO_2855W = 16421,
     DIS_PDU_EMITTERNAME_FREGAT_MAE = 16422,
     DIS_PDU_EMITTERNAME_FREGAT_N1 = 16423,
     DIS_PDU_EMITTERNAME_FREGAT_N2 = 16424,
     DIS_PDU_EMITTERNAME_FRONT_DOME = 16425,
     DIS_PDU_EMITTERNAME_FREGAT_MAE_5 = 16426,
     DIS_PDU_EMITTERNAME_FRONT_DOOR = 16470,
     DIS_PDU_EMITTERNAME_FRONT_PIECE = 16515,
     DIS_PDU_EMITTERNAME_FURBY_MMW_MH = 16520,
     DIS_PDU_EMITTERNAME_FURKE = 16550,
     DIS_PDU_EMITTERNAME_FURKE_2_FURKE_E_POSITIV_ME1 = 16552,
     DIS_PDU_EMITTERNAME_FURKE_4 = 16554,
     DIS_PDU_EMITTERNAME_FURUNO = 16560,
     DIS_PDU_EMITTERNAME_FURUNO_1721 = 16561,
     DIS_PDU_EMITTERNAME_FURUNO_1934C = 16564,
     DIS_PDU_EMITTERNAME_FURUNO_1715 = 16565,
     DIS_PDU_EMITTERNAME_FURUNO_1730 = 16580,
     DIS_PDU_EMITTERNAME_FURUNO_1731_MARK_3 = 16581,
     DIS_PDU_EMITTERNAME_FURUNO_1832 = 16585,
     DIS_PDU_EMITTERNAME_FURUNO_1835 = 16587,
     DIS_PDU_EMITTERNAME_FURUNO_1932 = 16590,
     DIS_PDU_EMITTERNAME_FURUNO_1935 = 16596,
     DIS_PDU_EMITTERNAME_FURUNO_701 = 16605,
     DIS_PDU_EMITTERNAME_FURUNO_1940 = 16606,
     DIS_PDU_EMITTERNAME_FURUNO_711_2 = 16650,
     DIS_PDU_EMITTERNAME_FURUNO_FR_2130S = 16652,
     DIS_PDU_EMITTERNAME_FURUNO_FAR_2137S = 16654,
     DIS_PDU_EMITTERNAME_FURUNO_FAR_28X7 = 16655,
     DIS_PDU_EMITTERNAME_FURUNO_FAR_3230S = 16658,
     DIS_PDU_EMITTERNAME_FR_2110 = 16660,
     DIS_PDU_EMITTERNAME_FR_2115 = 16662,
     DIS_PDU_EMITTERNAME_FR_8062 = 16663,
     DIS_PDU_EMITTERNAME_FURUNO_2125 = 16670,
     DIS_PDU_EMITTERNAME_FURUNO_240 = 16690,
     DIS_PDU_EMITTERNAME_FURUNO_2400 = 16695,
     DIS_PDU_EMITTERNAME_FR_801D_FURUNO_FR_810D = 16725,
     DIS_PDU_EMITTERNAME_FURUNO_8051 = 16730,
     DIS_PDU_EMITTERNAME_FURUNO_DRS_2D = 16732,
     DIS_PDU_EMITTERNAME_FURUNO_DRS_4D = 16733,
     DIS_PDU_EMITTERNAME_FURUNO_DRS_4A = 16734,
     DIS_PDU_EMITTERNAME_G030A_APD_31 = 16735,
     DIS_PDU_EMITTERNAME_FURUNO_DRS6A_X_CLASS = 16736,
     DIS_PDU_EMITTERNAME_DRS4W = 16737,
     DIS_PDU_EMITTERNAME_GA_01_00 = 16740,
     DIS_PDU_EMITTERNAME_GABBIANO = 16750,
     DIS_PDU_EMITTERNAME_GAGE = 16785,
     DIS_PDU_EMITTERNAME_GAOFEN_3 = 16787,
     DIS_PDU_EMITTERNAME_GAOFEN_12 = 16789,
     DIS_PDU_EMITTERNAME_GAP_GATE = 16790,
     DIS_PDU_EMITTERNAME_GARDENIA = 16800,
     DIS_PDU_EMITTERNAME_GARMIN_GMR_1224 = 16815,
     DIS_PDU_EMITTERNAME_GARMIN_FANTOM_24 = 16820,
     DIS_PDU_EMITTERNAME_GARMIN_GWX_68_WEATHER_RADAR = 16825,
     DIS_PDU_EMITTERNAME_GARPIN = 16830,
     DIS_PDU_EMITTERNAME_GATE_GUARD = 16833,
     DIS_PDU_EMITTERNAME_GARPUN_BAL_E = 16835,
     DIS_PDU_EMITTERNAME_GAZETCHIK = 16837,
     DIS_PDU_EMITTERNAME_GBS1 = 16840,
     DIS_PDU_EMITTERNAME_GCA_2000 = 16850,
     DIS_PDU_EMITTERNAME_FURUNO_1832_1921_SERIES = 16858,
     DIS_PDU_EMITTERNAME_GEMINI_DB = 16870,
     DIS_PDU_EMITTERNAME_GEM_OLYMPUS = 16871,
     DIS_PDU_EMITTERNAME_GEM_SENTINEL = 16872,
     DIS_PDU_EMITTERNAME_GEM_BX_132 = 16875,
     DIS_PDU_EMITTERNAME_GEM_SC_2050X = 16876,
     DIS_PDU_EMITTERNAME_GEM_SEA_EAGLE_200_N = 16877,
     DIS_PDU_EMITTERNAME_GENERIC_INTERNAL_JAMMER = 16879,
     DIS_PDU_EMITTERNAME_MPDR_12_GEPARD_TA = 16880,
     DIS_PDU_EMITTERNAME_GEN_X = 16881,
     DIS_PDU_EMITTERNAME_GEPARD_TT = 16884,
     DIS_PDU_EMITTERNAME_GERAN_F = 16888,
     DIS_PDU_EMITTERNAME_GERFAUT = 16890,
     DIS_PDU_EMITTERNAME_GFE_L_1 = 16895,
     DIS_PDU_EMITTERNAME_GIRAFFE = 16900,
     DIS_PDU_EMITTERNAME_GIRAFFE_1X = 16903,
     DIS_PDU_EMITTERNAME_GIRAFFE_40 = 16905,
     DIS_PDU_EMITTERNAME_GIRAFFE_50_AT = 16908,
     DIS_PDU_EMITTERNAME_GIRAFFE_75 = 16912,
     DIS_PDU_EMITTERNAME_GIN_SLING_TA = 16915,
     DIS_PDU_EMITTERNAME_GIN_SLING_GIN_SLING_TT = 16920,
     DIS_PDU_EMITTERNAME_GIN_SLING_MG = 16925,
     DIS_PDU_EMITTERNAME_GOAL_KEEPER = 16930,
     DIS_PDU_EMITTERNAME_GOLDEN_DOME = 16935,
     DIS_PDU_EMITTERNAME_GOLDEN_HEART = 16940,
     DIS_PDU_EMITTERNAME_GOLDEN_ROCK = 16942,
     DIS_PDU_EMITTERNAME_GORY_MH = 16943,
     DIS_PDU_EMITTERNAME_GPN_22 = 16945,
     DIS_PDU_EMITTERNAME_GPSJ_10 = 16946,
     DIS_PDU_EMITTERNAME_GPSJ_25 = 16947,
     DIS_PDU_EMITTERNAME_GPSJ_40 = 16948,
     DIS_PDU_EMITTERNAME_GPSJ_50 = 16949,
     DIS_PDU_EMITTERNAME_GRN_9 = 16950,
     DIS_PDU_EMITTERNAME_GRAN_K = 16951,
     DIS_PDU_EMITTERNAME_GRACKLE_MH = 16953,
     DIS_PDU_EMITTERNAME_GRAVE_STONE = 16960,
     DIS_PDU_EMITTERNAME_GRAVES = 16963,
     DIS_PDU_EMITTERNAME_GREEN_STAIN = 16965,
     DIS_PDU_EMITTERNAME_GRID_BOW = 17010,
     DIS_PDU_EMITTERNAME_GRIFO_F = 17016,
     DIS_PDU_EMITTERNAME_9S32_GRILL_PAN_TT = 17025,
     DIS_PDU_EMITTERNAME_GRILL_SCREEN = 17027,
     DIS_PDU_EMITTERNAME_GROM_2 = 17029,
     DIS_PDU_EMITTERNAME_GROUND_MASTER_400 = 17030,
     DIS_PDU_EMITTERNAME_GT_4 = 17031,
     DIS_PDU_EMITTERNAME_GRS_440 = 17032,
     DIS_PDU_EMITTERNAME_GRUESOME_MH = 17034,
     DIS_PDU_EMITTERNAME_GUARDIAN = 17050,
     DIS_PDU_EMITTERNAME_GUARDSMAN = 17055,
     DIS_PDU_EMITTERNAME_RPK_2_GUN_DISH_ZSU_23_4 = 17070,
     DIS_PDU_EMITTERNAME_GUSHER_MH = 17072,
     DIS_PDU_EMITTERNAME_H_RJZ_726_4A_JAMMER = 17075,
     DIS_PDU_EMITTERNAME_H025_NO25E = 17079,
     DIS_PDU_EMITTERNAME_HADR = 17080,
     DIS_PDU_EMITTERNAME_HAIR_NET = 17100,
     DIS_PDU_EMITTERNAME_HALF_PLATE_A = 17145,
     DIS_PDU_EMITTERNAME_HALF_PLATE_B = 17190,
     DIS_PDU_EMITTERNAME_HARD = 17220,
     DIS_PDU_EMITTERNAME_HARPOON = 17225,
     DIS_PDU_EMITTERNAME_HAT_BOX = 17230,
     DIS_PDU_EMITTERNAME_HAWK_SCREECH = 17235,
     DIS_PDU_EMITTERNAME_HAY_POLE = 17250,
     DIS_PDU_EMITTERNAME_HAY_RICK = 17255,
     DIS_PDU_EMITTERNAME_HEAD_LIGHT_A = 17280,
     DIS_PDU_EMITTERNAME_HEAD_LIGHTS = 17325,
     DIS_PDU_EMITTERNAME_HEAD_LIGHTS_C = 17370,
     DIS_PDU_EMITTERNAME_HEAD_LIGHTS_MG_A = 17415,
     DIS_PDU_EMITTERNAME_HEAD_LIGHTS_MG_B = 17460,
     DIS_PDU_EMITTERNAME_HEAD_LIGHTS_TT = 17505,
     DIS_PDU_EMITTERNAME_HEAD_NET = 17550,
     DIS_PDU_EMITTERNAME_HEART_ACHE_B = 17572,
     DIS_PDU_EMITTERNAME_HELLFIRE_MMW_MH = 17590,
     DIS_PDU_EMITTERNAME_HEN_EGG = 17595,
     DIS_PDU_EMITTERNAME_HEN_HOUSE = 17640,
     DIS_PDU_EMITTERNAME_HEN_NEST = 17685,
     DIS_PDU_EMITTERNAME_HEN_ROOST = 17730,
     DIS_PDU_EMITTERNAME_HERAKLES = 17732,
     DIS_PDU_EMITTERNAME_HF_2_MG = 17735,
     DIS_PDU_EMITTERNAME_HGR_105 = 17745,
     DIS_PDU_EMITTERNAME_HERZ_9_TAR = 17750,
     DIS_PDU_EMITTERNAME_HERZ_9_TTR = 17751,
     DIS_PDU_EMITTERNAME_HERZ_9_MG = 17752,
     DIS_PDU_EMITTERNAME_HIGH_BRICK = 17775,
     DIS_PDU_EMITTERNAME_HIGH_FIX = 17820,
     DIS_PDU_EMITTERNAME_YLC_2V_HIGH_GUARD = 17842,
     DIS_PDU_EMITTERNAME_HIGH_LARK_TI = 17865,
     DIS_PDU_EMITTERNAME_HIGH_LARK_1 = 17910,
     DIS_PDU_EMITTERNAME_HIGH_LARK_2 = 17955,
     DIS_PDU_EMITTERNAME_HIGH_LARK_4 = 18000,
     DIS_PDU_EMITTERNAME_HIGH_LUNE = 18045,
     DIS_PDU_EMITTERNAME_HIGH_POLE_A_B = 18090,
     DIS_PDU_EMITTERNAME_HIGH_SCOOP = 18135,
     DIS_PDU_EMITTERNAME_9S19MT_HIGH_SCREEN = 18150,
     DIS_PDU_EMITTERNAME_HIGH_SIEVE = 18180,
     DIS_PDU_EMITTERNAME_HILL_BLOCK = 18185,
     DIS_PDU_EMITTERNAME_HIMALAYAS_COUNTERMEASURES_SUITE = 18189,
     DIS_PDU_EMITTERNAME_HG_9550 = 18190,
     DIS_PDU_EMITTERNAME_HJ_6374 = 18193,
     DIS_PDU_EMITTERNAME_HLD_RADAR_900 = 18194,
     DIS_PDU_EMITTERNAME_HLJQ_520 = 18195,
     DIS_PDU_EMITTERNAME_HN_503 = 18200,
     DIS_PDU_EMITTERNAME_HN_C03_M = 18201,
     DIS_PDU_EMITTERNAME_HOME_TALK = 18225,
     DIS_PDU_EMITTERNAME_HORN_SPOON = 18270,
     DIS_PDU_EMITTERNAME_HOT_BRICK = 18280,
     DIS_PDU_EMITTERNAME_HOT_FLASH = 18315,
     DIS_PDU_EMITTERNAME_HOT_FLASH_2 = 18316,
     DIS_PDU_EMITTERNAME_IHS_6 = 18318,
     DIS_PDU_EMITTERNAME_IRL144M_HOT_SHOT_TA = 18320,
     DIS_PDU_EMITTERNAME_IRL144M_HOT_SHOT_TT = 18325,
     DIS_PDU_EMITTERNAME_IRL144M_HOT_SHOT_MG = 18330,
     DIS_PDU_EMITTERNAME_HPS_106 = 18331,
     DIS_PDU_EMITTERNAME_HPS_104 = 18332,
     DIS_PDU_EMITTERNAME_HQ_9_MH = 18339,
     DIS_PDU_EMITTERNAME_HSR_1128 = 18340,
     DIS_PDU_EMITTERNAME_HQ_9A_TER = 18342,
     DIS_PDU_EMITTERNAME_HQ_9B_TER = 18344,
     DIS_PDU_EMITTERNAME_HT_233 = 18348,
     DIS_PDU_EMITTERNAME_HQ_61 = 18350,
     DIS_PDU_EMITTERNAME_HRJS = 18351,
     DIS_PDU_EMITTERNAME_I_DERBY_ER = 18352,
     DIS_PDU_EMITTERNAME_IBIS_80 = 18353,
     DIS_PDU_EMITTERNAME_IBIS_150 = 18355,
     DIS_PDU_EMITTERNAME_IBIS_200 = 18357,
     DIS_PDU_EMITTERNAME_HQ_16_TER = 18359,
     DIS_PDU_EMITTERNAME_IFF_MK_XII_AIMS_UPX_29 = 18360,
     DIS_PDU_EMITTERNAME_JANET = 18400,
     DIS_PDU_EMITTERNAME_IFF_MK_XV = 18405,
     DIS_PDU_EMITTERNAME_IFF_INT = 18406,
     DIS_PDU_EMITTERNAME_JACK_KNIFE = 18407,
     DIS_PDU_EMITTERNAME_IFF_TRSP = 18408,
     DIS_PDU_EMITTERNAME_J_MUSIC_ELBIT_SYSTEMS_JAMMER = 18409,
     DIS_PDU_EMITTERNAME_JAVELIN_MG = 18410,
     DIS_PDU_EMITTERNAME_IGLA_1_SLAR = 18411,
     DIS_PDU_EMITTERNAME_IR_SA_2B_TER = 18415,
     DIS_PDU_EMITTERNAME_IR_SA_4_TER = 18417,
     DIS_PDU_EMITTERNAME_IR_SA_5_TELAR_TER = 18419,
     DIS_PDU_EMITTERNAME_J_10B_PESA = 18420,
     DIS_PDU_EMITTERNAME_J_10C_AESA = 18421,
     DIS_PDU_EMITTERNAME_J_11D_AESA = 18422,
     DIS_PDU_EMITTERNAME_JL_10MP = 18443,
     DIS_PDU_EMITTERNAME_J_ALQ_8 = 18445,
     DIS_PDU_EMITTERNAME_J_FPS_7 = 18449,
     DIS_PDU_EMITTERNAME_JAY_BIRD = 18450,
     DIS_PDU_EMITTERNAME_J_FPS_3 = 18451,
     DIS_PDU_EMITTERNAME_JH_10_MIX_BOWL = 18452,
     DIS_PDU_EMITTERNAME_J_MPQ_P7 = 18453,
     DIS_PDU_EMITTERNAME_JL_7 = 18454,
     DIS_PDU_EMITTERNAME_JL_10B = 18455,
     DIS_PDU_EMITTERNAME_JMA_1576 = 18456,
     DIS_PDU_EMITTERNAME_JRC_JMA_9252_6CA = 18457,
     DIS_PDU_EMITTERNAME_JLP_40 = 18458,
     DIS_PDU_EMITTERNAME_JRC_JMR_9200_SERIES_X = 18459,
     DIS_PDU_EMITTERNAME_JRC_NMD_401 = 18460,
     DIS_PDU_EMITTERNAME_JRC_JRM_310_MK2 = 18461,
     DIS_PDU_EMITTERNAME_JMA_1596 = 18462,
     DIS_PDU_EMITTERNAME_JN_1104 = 18463,
     DIS_PDU_EMITTERNAME_JMA_7000 = 18464,
     DIS_PDU_EMITTERNAME_JRC_JMA_7700 = 18465,
     DIS_PDU_EMITTERNAME_JMA_5320 = 18466,
     DIS_PDU_EMITTERNAME_JRC_JMR_9210_6XC = 18467,
     DIS_PDU_EMITTERNAME_JERS_1 = 18468,
     DIS_PDU_EMITTERNAME_JINDALEE = 18469,
     DIS_PDU_EMITTERNAME_JRC_JMA_9900_SERIES = 18470,
     DIS_PDU_EMITTERNAME_JLP_40D = 18471,
     DIS_PDU_EMITTERNAME_JRC_JMA_5300_SERIES = 18475,
     DIS_PDU_EMITTERNAME_JUPITER = 18495,
     DIS_PDU_EMITTERNAME_JUPITER_II = 18540,
     DIS_PDU_EMITTERNAME_JY_8 = 18550,
     DIS_PDU_EMITTERNAME_JY_8A = 18551,
     DIS_PDU_EMITTERNAME_JY_9 = 18555,
     DIS_PDU_EMITTERNAME_JY_9_MODIFIED = 18556,
     DIS_PDU_EMITTERNAME_JY_11_EW = 18557,
     DIS_PDU_EMITTERNAME_JY_14 = 18560,
     DIS_PDU_EMITTERNAME_JY_14A = 18561,
     DIS_PDU_EMITTERNAME_JY_16 = 18565,
     DIS_PDU_EMITTERNAME_JY_24 = 18570,
     DIS_PDU_EMITTERNAME_J_APG_1 = 18571,
     DIS_PDU_EMITTERNAME_J_APG_2 = 18572,
     DIS_PDU_EMITTERNAME_JY_29 = 18575,
     DIS_PDU_EMITTERNAME_JYL_1 = 18578,
     DIS_PDU_EMITTERNAME_JYL_6 = 18580,
     DIS_PDU_EMITTERNAME_JYL_6A = 18582,
     DIS_PDU_EMITTERNAME_JZ_QF_612 = 18583,
     DIS_PDU_EMITTERNAME_K376Z = 18585,
     DIS_PDU_EMITTERNAME_K77M = 18586,
     DIS_PDU_EMITTERNAME_KAIGE = 18600,
     DIS_PDU_EMITTERNAME_KALKAN = 18610,
     DIS_PDU_EMITTERNAME_KBP_AFGANIT = 18611,
     DIS_PDU_EMITTERNAME_KALKAN_II = 18615,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_2A = 18630,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_14_9 = 18675,
     DIS_PDU_EMITTERNAME_KARPATY = 18700,
     DIS_PDU_EMITTERNAME_KASHTAN_3_JAMMING_SYSTEM = 18710,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_TYPE_1006 = 18720,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_TYPE_1007 = 18765,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_TYPE_1007_F_BAND = 18766,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_2007_F_BAND = 18767,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_2007_I_BAND = 18768,
     DIS_PDU_EMITTERNAME_KH_NUCLEUS_5000 = 18770,
     DIS_PDU_EMITTERNAME_KH_MANTA = 18774,
     DIS_PDU_EMITTERNAME_KH_NUCLEUS_2_6000 = 18775,
     DIS_PDU_EMITTERNAME_KH_NUCLEUS_3_5000 = 18776,
     DIS_PDU_EMITTERNAME_KH_NUCLEUS_3_6000A = 18777,
     DIS_PDU_EMITTERNAME_KH_FAMILY = 18780,
     DIS_PDU_EMITTERNAME_KELVIN_HUGHES_6000A = 18781,
     DIS_PDU_EMITTERNAME_KH_38MAE_MH = 18782,
     DIS_PDU_EMITTERNAME_KG8605A = 18784,
     DIS_PDU_EMITTERNAME_KH_902M = 18785,
     DIS_PDU_EMITTERNAME_KHOROM_K = 18786,
     DIS_PDU_EMITTERNAME_KHIBINY = 18787,
     DIS_PDU_EMITTERNAME_KG300E = 18789,
     DIS_PDU_EMITTERNAME_KH_SHARPEYE = 18790,
     DIS_PDU_EMITTERNAME_KH_SHARP_EYE_B = 18791,
     DIS_PDU_EMITTERNAME_KH_SHARP_EYE_C = 18792,
     DIS_PDU_EMITTERNAME_KH_1700 = 18795,
     DIS_PDU_EMITTERNAME_KING_PIN = 18797,
     DIS_PDU_EMITTERNAME_3RD_KHORDAD_TELAR_TIR = 18800,
     DIS_PDU_EMITTERNAME_3RD_KHORDAD_TAR = 18801,
     DIS_PDU_EMITTERNAME_15TH_KHORDAD_TER = 18803,
     DIS_PDU_EMITTERNAME_KG_300 = 18805,
     DIS_PDU_EMITTERNAME_KITE_SCREECH = 18810,
     DIS_PDU_EMITTERNAME_KITE_SCREECH_A = 18855,
     DIS_PDU_EMITTERNAME_KITE_SCREECH_B = 18900,
     DIS_PDU_EMITTERNAME_KLC_3B = 18930,
     DIS_PDU_EMITTERNAME_KJ_500_NANJING_RADAR = 18944,
     DIS_PDU_EMITTERNAME_KIVACH = 18945,
     DIS_PDU_EMITTERNAME_KJ_500_JAMMER = 18946,
     DIS_PDU_EMITTERNAME_KLC_1 = 18947,
     DIS_PDU_EMITTERNAME_KLJ_1 = 18948,
     DIS_PDU_EMITTERNAME_KLJ_3_TYPE_1473 = 18950,
     DIS_PDU_EMITTERNAME_KLJ_4 = 18951,
     DIS_PDU_EMITTERNAME_KLJ_4B = 18952,
     DIS_PDU_EMITTERNAME_KLJ_5 = 18955,
     DIS_PDU_EMITTERNAME_KLJ_7 = 18960,
     DIS_PDU_EMITTERNAME_KLJ_7B = 18961,
     DIS_PDU_EMITTERNAME_KLJ_7A = 18962,
     DIS_PDU_EMITTERNAME_KNIFE_REST = 18990,
     DIS_PDU_EMITTERNAME_P_10_KNIFE_REST_B = 19035,
     DIS_PDU_EMITTERNAME_KNIFE_REST_C = 19037,
     DIS_PDU_EMITTERNAME_KODEN_MD_3730 = 19039,
     DIS_PDU_EMITTERNAME_KJ_2000 = 19040,
     DIS_PDU_EMITTERNAME_KODEN_MDC_900 = 19041,
     DIS_PDU_EMITTERNAME_KOOPOL = 19042,
     DIS_PDU_EMITTERNAME_KOPYO_I = 19045,
     DIS_PDU_EMITTERNAME_KR_75 = 19050,
     DIS_PDU_EMITTERNAME_KRONOS = 19051,
     DIS_PDU_EMITTERNAME_KREDO_1E = 19052,
     DIS_PDU_EMITTERNAME_KRASUKHA_2 = 19053,
     DIS_PDU_EMITTERNAME_KRONOS_GRAND_NAVAL = 19054,
     DIS_PDU_EMITTERNAME_KRM_66E = 19060,
     DIS_PDU_EMITTERNAME_KRTZ_125_2M = 19065,
     DIS_PDU_EMITTERNAME_KSA_SRN = 19080,
     DIS_PDU_EMITTERNAME_KSA_TSR = 19125,
     DIS_PDU_EMITTERNAME_KS_1A_PHASED_ARRAY = 19127,
     DIS_PDU_EMITTERNAME_KS418 = 19129,
     DIS_PDU_EMITTERNAME_KS418E = 19130,
     DIS_PDU_EMITTERNAME_KZ100 = 19131,
     DIS_PDU_EMITTERNAME_KZ900 = 19132,
     DIS_PDU_EMITTERNAME_L175V = 19140,
     DIS_PDU_EMITTERNAME_L370_5_PRESIDENT_S_JAMMER = 19142,
     DIS_PDU_EMITTERNAME_L_415 = 19143,
     DIS_PDU_EMITTERNAME_L_88 = 19145,
     DIS_PDU_EMITTERNAME_LAADS = 19150,
     DIS_PDU_EMITTERNAME_LAND_FALL = 19170,
     DIS_PDU_EMITTERNAME_LAND_ROLL_MG = 19215,
     DIS_PDU_EMITTERNAME_LAND_ROLL_TA = 19260,
     DIS_PDU_EMITTERNAME_LAND_ROLL_TT = 19305,
     DIS_PDU_EMITTERNAME_LAZUR = 19306,
     DIS_PDU_EMITTERNAME_MODEL_791_A_LAST_STRAW = 19307,
     DIS_PDU_EMITTERNAME_LAP_3000 = 19309,
     DIS_PDU_EMITTERNAME_LC_150 = 19310,
     DIS_PDU_EMITTERNAME_LEER_3 = 19320,
     DIS_PDU_EMITTERNAME_LEG_DRIVE = 19330,
     DIS_PDU_EMITTERNAME_LENINETZ_V_004 = 19340,
     DIS_PDU_EMITTERNAME_LENINGRAF = 19350,
     DIS_PDU_EMITTERNAME_LIANA = 19370,
     DIS_PDU_EMITTERNAME_LIGHT_BULB = 19395,
     DIS_PDU_EMITTERNAME_LIRA_A10 = 19396,
     DIS_PDU_EMITTERNAME_LIROD_8 = 19397,
     DIS_PDU_EMITTERNAME_LIROD_MK_II = 19398,
     DIS_PDU_EMITTERNAME_LLX05K = 19399,
     DIS_PDU_EMITTERNAME_LMT_NRAI_6A = 19400,
     DIS_PDU_EMITTERNAME_LN_55 = 19440,
     DIS_PDU_EMITTERNAME_LN_66 = 19485,
     DIS_PDU_EMITTERNAME_LIMAN = 19500,
     DIS_PDU_EMITTERNAME_LIMAN_2 = 19505,
     DIS_PDU_EMITTERNAME_LOCKHEED_VIGILANCE = 19520,
     DIS_PDU_EMITTERNAME_LONG_BOW = 19530,
     DIS_PDU_EMITTERNAME_LONG_BRICK = 19575,
     DIS_PDU_EMITTERNAME_LONG_BULL = 19620,
     DIS_PDU_EMITTERNAME_LONG_EYE = 19665,
     DIS_PDU_EMITTERNAME_LONG_HEAD = 19710,
     DIS_PDU_EMITTERNAME_LONG_TALK = 19755,
     DIS_PDU_EMITTERNAME_LONG_TRACK = 19800,
     DIS_PDU_EMITTERNAME_LONG_TROUGH = 19845,
     DIS_PDU_EMITTERNAME_LOOK_TWO = 19890,
     DIS_PDU_EMITTERNAME_LOPAR = 19920,
     DIS_PDU_EMITTERNAME_LORAN = 19935,
     DIS_PDU_EMITTERNAME_LOW_BLOW_TA = 19950,
     DIS_PDU_EMITTERNAME_LOW_BLOW_TT = 19955,
     DIS_PDU_EMITTERNAME_LOW_BLOW_MG = 19960,
     DIS_PDU_EMITTERNAME_LOW_JACK_TT = 19970,
     DIS_PDU_EMITTERNAME_LOW_JACK_MG = 19971,
     DIS_PDU_EMITTERNAME_LOW_SIEVE = 19980,
     DIS_PDU_EMITTERNAME_LOW_TROUGH = 20025,
     DIS_PDU_EMITTERNAME_LR66 = 20029,
     DIS_PDU_EMITTERNAME_LRA_900 = 20030,
     DIS_PDU_EMITTERNAME_TRS_2050_LP_23 = 20040,
     DIS_PDU_EMITTERNAME_LW_01 = 20060,
     DIS_PDU_EMITTERNAME_M_1983_FCR = 20090,
     DIS_PDU_EMITTERNAME_M22_40 = 20115,
     DIS_PDU_EMITTERNAME_M44 = 20160,
     DIS_PDU_EMITTERNAME_M401Z = 20205,
     DIS_PDU_EMITTERNAME_M585Z = 20250,
     DIS_PDU_EMITTERNAME_M588Z = 20295,
     DIS_PDU_EMITTERNAME_MA_1_IFF_PORTION = 20340,
     DIS_PDU_EMITTERNAME_MAD_HACK = 20350,
     DIS_PDU_EMITTERNAME_MARELD = 20360,
     DIS_PDU_EMITTERNAME_MA_TYPE_909_ = 20385,
     DIS_PDU_EMITTERNAME_MARC_S_152 = 20420,
     DIS_PDU_EMITTERNAME_MARCONI_1810 = 20430,
     DIS_PDU_EMITTERNAME_MARCONI_CANADA_HC_75 = 20475,
     DIS_PDU_EMITTERNAME_MARCONI_S_713 = 20495,
     DIS_PDU_EMITTERNAME_MARCONI_S_1802 = 20520,
     DIS_PDU_EMITTERNAME_MARCONI_S247 = 20530,
     DIS_PDU_EMITTERNAME_MARCONI_S_810 = 20565,
     DIS_PDU_EMITTERNAME_MARCONI_SA_10 = 20585,
     DIS_PDU_EMITTERNAME_MARCONI_ST801 = 20589,
     DIS_PDU_EMITTERNAME_MARCONI_ST805 = 20590,
     DIS_PDU_EMITTERNAME_MARCONI_TYPE_967 = 20610,
     DIS_PDU_EMITTERNAME_MARCONI_TYPE_968 = 20655,
     DIS_PDU_EMITTERNAME_MARCONI_TYPE_992 = 20700,
     DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_1022 = 20745,
     DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_910 = 20790,
     DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_911 = 20835,
     DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_992R = 20880,
     DIS_PDU_EMITTERNAME_MARTELLO_743D = 20890,
     DIS_PDU_EMITTERNAME_MARTELLO_S_723A = 20895,
     DIS_PDU_EMITTERNAME_MASTER_A = 20897,
     DIS_PDU_EMITTERNAME_MBDA_FLAADS_M_SEA_CEPTOR_JAMMER = 20898,
     DIS_PDU_EMITTERNAME_MELCO_3 = 20915,
     DIS_PDU_EMITTERNAME_MELODI = 20917,
     DIS_PDU_EMITTERNAME_MERLIN = 20918,
     DIS_PDU_EMITTERNAME_MERAJ_4_ASCENSION = 20919,
     DIS_PDU_EMITTERNAME_NORTHROP_GRUMMAN_MESA = 20920,
     DIS_PDU_EMITTERNAME_MESH_BRICK = 20925,
     DIS_PDU_EMITTERNAME_METEOR_1500S = 20927,
     DIS_PDU_EMITTERNAME_METEOR_200 = 20929,
     DIS_PDU_EMITTERNAME_METEOR_50DX = 20930,
     DIS_PDU_EMITTERNAME_METEOR_300 = 20931,
     DIS_PDU_EMITTERNAME_METEOR_BVRAAM = 20933,
     DIS_PDU_EMITTERNAME_MFR = 20935,
     DIS_PDU_EMITTERNAME_MFSR_2100_45 = 20940,
     DIS_PDU_EMITTERNAME_MICA_MH = 20942,
     DIS_PDU_EMITTERNAME_MICA_RF = 20943,
     DIS_PDU_EMITTERNAME_MINERAL_ME = 20945,
     DIS_PDU_EMITTERNAME_MIRAGE_ILL = 20950,
     DIS_PDU_EMITTERNAME_MIYSIS_JAMMER = 20955,
     DIS_PDU_EMITTERNAME_MK_15_PHALANX_BLK_0 = 20969,
     DIS_PDU_EMITTERNAME_MK_15_CIWS = 20970,
     DIS_PDU_EMITTERNAME_MK_23 = 21015,
     DIS_PDU_EMITTERNAME_MK_23_TAS = 21060,
     DIS_PDU_EMITTERNAME_MK_25 = 21105,
     DIS_PDU_EMITTERNAME_MK_25_MOD_3 = 21110,
     DIS_PDU_EMITTERNAME_MK_25_MOD_7 = 21130,
     DIS_PDU_EMITTERNAME_MK_35_M2 = 21150,
     DIS_PDU_EMITTERNAME_MK_92 = 21195,
     DIS_PDU_EMITTERNAME_MK_92_CAS = 21240,
     DIS_PDU_EMITTERNAME_MK_92_STIR = 21285,
     DIS_PDU_EMITTERNAME_MK_95 = 21330,
     DIS_PDU_EMITTERNAME_MKS_818 = 21332,
     DIS_PDU_EMITTERNAME_MLA_1 = 21340,
     DIS_PDU_EMITTERNAME_MM_APQ_706 = 21359,
     DIS_PDU_EMITTERNAME_MM_950 = 21360,
     DIS_PDU_EMITTERNAME_MM_APS_705 = 21375,
     DIS_PDU_EMITTERNAME_MM_APS_784 = 21390,
     DIS_PDU_EMITTERNAME_MM_SPG_73_RTN_12X = 21419,
     DIS_PDU_EMITTERNAME_MM_SPG_74 = 21420,
     DIS_PDU_EMITTERNAME_MM_SPG_75 = 21465,
     DIS_PDU_EMITTERNAME_MM_SPN_703 = 21490,
     DIS_PDU_EMITTERNAME_MM_SPN_730 = 21492,
     DIS_PDU_EMITTERNAME_MM_SPN_753B = 21495,
     DIS_PDU_EMITTERNAME_MM_SPQ_3 = 21500,
     DIS_PDU_EMITTERNAME_MM_SPS_702 = 21510,
     DIS_PDU_EMITTERNAME_MM_SPS_768 = 21555,
     DIS_PDU_EMITTERNAME_MM_SPS_774 = 21600,
     DIS_PDU_EMITTERNAME_MM_SPS_791_RAN_30X = 21610,
     DIS_PDU_EMITTERNAME_MM_SPS_794_RAN_21S = 21615,
     DIS_PDU_EMITTERNAME_MM_SPS_798_RAN_40L = 21620,
     DIS_PDU_EMITTERNAME_MMSR = 21623,
     DIS_PDU_EMITTERNAME_MODEL_17C = 21625,
     DIS_PDU_EMITTERNAME_MOON_4 = 21645,
     DIS_PDU_EMITTERNAME_MOON_PIE = 21646,
     DIS_PDU_EMITTERNAME_MOON_CONE = 21647,
     DIS_PDU_EMITTERNAME_MOON_RACK = 21648,
     DIS_PDU_EMITTERNAME_MOON_FACE = 21649,
     DIS_PDU_EMITTERNAME_MMRS = 21650,
     DIS_PDU_EMITTERNAME_MOON_SHADE = 21651,
     DIS_PDU_EMITTERNAME_MODEL_360 = 21655,
     DIS_PDU_EMITTERNAME_MODEL_378 = 21660,
     DIS_PDU_EMITTERNAME_MODEL_970 = 21661,
     DIS_PDU_EMITTERNAME_MODEL_974 = 21665,
     DIS_PDU_EMITTERNAME_MONOLIT_B = 21672,
     DIS_PDU_EMITTERNAME_MONUMENT = 21675,
     DIS_PDU_EMITTERNAME_MOUSE = 21680,
     DIS_PDU_EMITTERNAME_MP_411_ESM = 21682,
     DIS_PDU_EMITTERNAME_MPDR_18_S = 21685,
     DIS_PDU_EMITTERNAME_MPDR_18_X = 21690,
     DIS_PDU_EMITTERNAME_MPDR_45_E = 21692,
     DIS_PDU_EMITTERNAME_MR_231_1 = 21693,
     DIS_PDU_EMITTERNAME_MR_231_3 = 21694,
     DIS_PDU_EMITTERNAME_MPR = 21695,
     DIS_PDU_EMITTERNAME_MR_231_4 = 21696,
     DIS_PDU_EMITTERNAME_MPS_1 = 21697,
     DIS_PDU_EMITTERNAME_MR_36B = 21698,
     DIS_PDU_EMITTERNAME_MR_231_MOD = 21699,
     DIS_PDU_EMITTERNAME_MR_1600 = 21700,
     DIS_PDU_EMITTERNAME_MRR = 21701,
     DIS_PDU_EMITTERNAME_MR35 = 21702,
     DIS_PDU_EMITTERNAME_MR36 = 21703,
     DIS_PDU_EMITTERNAME_MRL_1 = 21704,
     DIS_PDU_EMITTERNAME_MRL_4 = 21705,
     DIS_PDU_EMITTERNAME_MRL_5 = 21706,
     DIS_PDU_EMITTERNAME_MSAM = 21707,
     DIS_PDU_EMITTERNAME_MR_36A = 21708,
     DIS_PDU_EMITTERNAME_MSTAR = 21709,
     DIS_PDU_EMITTERNAME_MT_305X = 21710,
     DIS_PDU_EMITTERNAME_MR_10M1E = 21711,
     DIS_PDU_EMITTERNAME_MR_90 = 21712,
     DIS_PDU_EMITTERNAME_MRK_411 = 21715,
     DIS_PDU_EMITTERNAME_MR_320M_TOPAZ_V = 21716,
     DIS_PDU_EMITTERNAME_MSP_418K = 21720,
     DIS_PDU_EMITTERNAME_MUFF_COB = 21735,
     DIS_PDU_EMITTERNAME_MUSHROOM = 21780,
     DIS_PDU_EMITTERNAME_MUSHROOM_1 = 21825,
     DIS_PDU_EMITTERNAME_MUSHROOM_2 = 21870,
     DIS_PDU_EMITTERNAME_MUSHROOM_3 = 21871,
     DIS_PDU_EMITTERNAME_N_23 = 21872,
     DIS_PDU_EMITTERNAME_N_011M_BARS = 21873,
     DIS_PDU_EMITTERNAME_N_011M_BARS_B = 21874,
     DIS_PDU_EMITTERNAME_N_011M_BARS_C = 21875,
     DIS_PDU_EMITTERNAME_N_011M_BARS_R = 21876,
     DIS_PDU_EMITTERNAME_N035_IRBIS_E = 21877,
     DIS_PDU_EMITTERNAME_N036_BYELKA = 21878,
     DIS_PDU_EMITTERNAME_N_25 = 21879,
     DIS_PDU_EMITTERNAME_N920Z = 21880,
     DIS_PDU_EMITTERNAME_N001V = 21881,
     DIS_PDU_EMITTERNAME_N001VE = 21882,
     DIS_PDU_EMITTERNAME_N001VEP = 21883,
     DIS_PDU_EMITTERNAME_NACOS_RADARPILOT_PLATINUM = 21884,
     DIS_PDU_EMITTERNAME_NAMPO_B = 21885,
     DIS_PDU_EMITTERNAME_NAGIRA = 21886,
     DIS_PDU_EMITTERNAME_NANJING_B = 21890,
     DIS_PDU_EMITTERNAME_NANJING_C = 21895,
     DIS_PDU_EMITTERNAME_NAYADA = 21915,
     DIS_PDU_EMITTERNAME_NAYADA_5M = 21917,
     DIS_PDU_EMITTERNAME_NAYADA_5PV = 21918,
     DIS_PDU_EMITTERNAME_NEBO_M = 21919,
     DIS_PDU_EMITTERNAME_NEBO_SVU = 21920,
     DIS_PDU_EMITTERNAME_NEPTUN = 21960,
     DIS_PDU_EMITTERNAME_NETTUNO_4100 = 21965,
     DIS_PDU_EMITTERNAME_NIKE_HERCULES_MTR = 21970,
     DIS_PDU_EMITTERNAME_NIKE_TT = 21980,
     DIS_PDU_EMITTERNAME_NORTHROP_GRUMMAN_MFEW_JAMMER = 21981,
     DIS_PDU_EMITTERNAME_NORINCO_3D = 21982,
     DIS_PDU_EMITTERNAME_NJ_81E = 21983,
     DIS_PDU_EMITTERNAME_NORMANDIE = 21984,
     DIS_PDU_EMITTERNAME_NRJ_6A = 21985,
     DIS_PDU_EMITTERNAME_NOSTRADAMUS = 21986,
     DIS_PDU_EMITTERNAME_NPG_1240 = 21987,
     DIS_PDU_EMITTERNAME_NPG_1460 = 21988,
     DIS_PDU_EMITTERNAME_NPG_434 = 21989,
     DIS_PDU_EMITTERNAME_NPG_630 = 21990,
     DIS_PDU_EMITTERNAME_NPM_510 = 21991,
     DIS_PDU_EMITTERNAME_NUT_CAN = 21992,
     DIS_PDU_EMITTERNAME_NP_VEGA_LIANA = 21995,
     DIS_PDU_EMITTERNAME_NOVELLA_NV1_70 = 22000,
     DIS_PDU_EMITTERNAME_NOVELLA_P_38 = 22001,
     DIS_PDU_EMITTERNAME_NRBA_50 = 22005,
     DIS_PDU_EMITTERNAME_NRBA_51 = 22050,
     DIS_PDU_EMITTERNAME_NRBF_20A = 22095,
     DIS_PDU_EMITTERNAME_NRJ_5 = 22110,
     DIS_PDU_EMITTERNAME_NS_9005 = 22115,
     DIS_PDU_EMITTERNAME_NS_100_SERIES = 22125,
     DIS_PDU_EMITTERNAME_NUR_31 = 22127,
     DIS_PDU_EMITTERNAME_NWS_3 = 22130,
     DIS_PDU_EMITTERNAME_NYSA_B = 22140,
     DIS_PDU_EMITTERNAME_O524A = 22185,
     DIS_PDU_EMITTERNAME_O580B = 22230,
     DIS_PDU_EMITTERNAME_O625Z = 22275,
     DIS_PDU_EMITTERNAME_O626Z = 22320,
     DIS_PDU_EMITTERNAME_OCEAN_MASTER = 22335,
     DIS_PDU_EMITTERNAME_OCEAN_MASTER_400 = 22340,
     DIS_PDU_EMITTERNAME_ODD_GROUP = 22345,
     DIS_PDU_EMITTERNAME_ODD_LOT = 22365,
     DIS_PDU_EMITTERNAME_ODD_PAIR = 22410,
     DIS_PDU_EMITTERNAME_ODD_RODS = 22411,
     DIS_PDU_EMITTERNAME_OKA = 22455,
     DIS_PDU_EMITTERNAME_OFOGH = 22460,
     DIS_PDU_EMITTERNAME_OFOGH_3 = 22463,
     DIS_PDU_EMITTERNAME_OKEAN = 22500,
     DIS_PDU_EMITTERNAME_OKEAN_A = 22505,
     DIS_PDU_EMITTERNAME_OKINXE_12C = 22545,
     DIS_PDU_EMITTERNAME_OKO = 22560,
     DIS_PDU_EMITTERNAME_OMEGA = 22590,
     DIS_PDU_EMITTERNAME_OMERA_ORB32 = 22635,
     DIS_PDU_EMITTERNAME_OMUL = 22640,
     DIS_PDU_EMITTERNAME_ONE_EYE = 22680,
     DIS_PDU_EMITTERNAME_OP_28 = 22690,
     DIS_PDU_EMITTERNAME_OPRL_4 = 22695,
     DIS_PDU_EMITTERNAME_OPRM_71 = 22696,
     DIS_PDU_EMITTERNAME_OPS_9 = 22697,
     DIS_PDU_EMITTERNAME_OPS_11_B_C = 22700,
     DIS_PDU_EMITTERNAME_OPS_12 = 22701,
     DIS_PDU_EMITTERNAME_OPS_14B = 22705,
     DIS_PDU_EMITTERNAME_OPS_14C = 22706,
     DIS_PDU_EMITTERNAME_OPS_16B = 22725,
     DIS_PDU_EMITTERNAME_OPS_18 = 22730,
     DIS_PDU_EMITTERNAME_OPS_19 = 22732,
     DIS_PDU_EMITTERNAME_OPS_20 = 22735,
     DIS_PDU_EMITTERNAME_OPS_22 = 22736,
     DIS_PDU_EMITTERNAME_OPS_24 = 22737,
     DIS_PDU_EMITTERNAME_OPS_28 = 22740,
     DIS_PDU_EMITTERNAME_OPS_28C = 22745,
     DIS_PDU_EMITTERNAME_OPS_39 = 22750,
     DIS_PDU_EMITTERNAME_OPTIMA_3_2 = 22760,
     DIS_PDU_EMITTERNAME_OR_2 = 22770,
     DIS_PDU_EMITTERNAME_ORB_31D = 22800,
     DIS_PDU_EMITTERNAME_ORB_31S = 22810,
     DIS_PDU_EMITTERNAME_ORB_32 = 22815,
     DIS_PDU_EMITTERNAME_ORB_42 = 22830,
     DIS_PDU_EMITTERNAME_ORION_RTN_10X = 22860,
     DIS_PDU_EMITTERNAME_SURFACE_WAVE_OVER_THE_HORIZON = 22890,
     DIS_PDU_EMITTERNAME_OTOMAT_MK_1 = 22900,
     DIS_PDU_EMITTERNAME_OTOMAT_MK_II_TESEO = 22905,
     DIS_PDU_EMITTERNAME_OTOMAT_SERIES_AL = 22906,
     DIS_PDU_EMITTERNAME_OWL_SCREECH = 22950,
     DIS_PDU_EMITTERNAME_P360Z = 22955,
     DIS_PDU_EMITTERNAME_P_14 = 22956,
     DIS_PDU_EMITTERNAME_P_180U = 22957,
     DIS_PDU_EMITTERNAME_P_18_2 = 22959,
     DIS_PDU_EMITTERNAME_PA_1660 = 22960,
     DIS_PDU_EMITTERNAME_P_18M = 22961,
     DIS_PDU_EMITTERNAME_P_190U = 22962,
     DIS_PDU_EMITTERNAME_P_30 = 22963,
     DIS_PDU_EMITTERNAME_P_18_MOD = 22964,
     DIS_PDU_EMITTERNAME_P_35M = 22965,
     DIS_PDU_EMITTERNAME_PAGE = 22970,
     DIS_PDU_EMITTERNAME_PAINT_BOX = 22977,
     DIS_PDU_EMITTERNAME_PALM_FROND = 22995,
     DIS_PDU_EMITTERNAME_MODIFIED_PAINT_BOX = 22998,
     DIS_PDU_EMITTERNAME_PALM_FROND_AB = 23040,
     DIS_PDU_EMITTERNAME_PANDORA = 23041,
     DIS_PDU_EMITTERNAME_PALSAR_2 = 23042,
     DIS_PDU_EMITTERNAME_PANTSIR_SM_TAR = 23043,
     DIS_PDU_EMITTERNAME_PAR_2 = 23045,
     DIS_PDU_EMITTERNAME_PANTSIR_S1_2RL80_TAR = 23046,
     DIS_PDU_EMITTERNAME_PANTSIR_S1_1RS2_1_TT = 23047,
     DIS_PDU_EMITTERNAME_PAR_2000 = 23050,
     DIS_PDU_EMITTERNAME_PAR_2090C = 23053,
     DIS_PDU_EMITTERNAME_PAR_80 = 23055,
     DIS_PDU_EMITTERNAME_PAT_HAND_TT = 23085,
     DIS_PDU_EMITTERNAME_PAT_HAND_MG = 23095,
     DIS_PDU_EMITTERNAME_PATRIOT = 23100,
     DIS_PDU_EMITTERNAME_PATTY_CAKE = 23130,
     DIS_PDU_EMITTERNAME_PAWN_CAKE = 23175,
     DIS_PDU_EMITTERNAME_PBR_4_RUBIN = 23220,
     DIS_PDU_EMITTERNAME_PCS_514 = 23240,
     DIS_PDU_EMITTERNAME_PEA_STICKS = 23265,
     DIS_PDU_EMITTERNAME_PECHORA_SC = 23295,
     DIS_PDU_EMITTERNAME_PEEL_CONE = 23310,
     DIS_PDU_EMITTERNAME_PEEL_GROUP = 23355,
     DIS_PDU_EMITTERNAME_PEEL_GROUP_A = 23400,
     DIS_PDU_EMITTERNAME_PEEL_GROUP_B = 23445,
     DIS_PDU_EMITTERNAME_PEEL_GROUP_MG = 23450,
     DIS_PDU_EMITTERNAME_PEEL_PAIR = 23490,
     DIS_PDU_EMITTERNAME_PELENA = 23500,
     DIS_PDU_EMITTERNAME_PGZ07 = 23515,
     DIS_PDU_EMITTERNAME_PHALANX = 23525,
     DIS_PDU_EMITTERNAME_PHAZOTRON_GUKOL_4 = 23529,
     DIS_PDU_EMITTERNAME_PHAZOTRON_ZHUK_A_AE = 23530,
     DIS_PDU_EMITTERNAME_PHILIPS_9LV_200 = 23535,
     DIS_PDU_EMITTERNAME_PHILIPS_9LV_331 = 23580,
     DIS_PDU_EMITTERNAME_PHILIPS_LV_223 = 23625,
     DIS_PDU_EMITTERNAME_PHILIPS_SEA_GIRAFFE_50_HC = 23670,
     DIS_PDU_EMITTERNAME_PHIMAT_JAMMER = 23675,
     DIS_PDU_EMITTERNAME_PICOSAR = 23680,
     DIS_PDU_EMITTERNAME_PILOT_MK2 = 23685,
     DIS_PDU_EMITTERNAME_PIN_JIB = 23690,
     DIS_PDU_EMITTERNAME_PIN_TIP = 23695,
     DIS_PDU_EMITTERNAME_PL_11 = 23700,
     DIS_PDU_EMITTERNAME_PL_12 = 23701,
     DIS_PDU_EMITTERNAME_PL_15 = 23704,
     DIS_PDU_EMITTERNAME_PLANK_SHAD = 23710,
     DIS_PDU_EMITTERNAME_PLANK_SHAVE = 23715,
     DIS_PDU_EMITTERNAME_PLANK_SHAVE_A = 23760,
     DIS_PDU_EMITTERNAME_PLANK_SHAVE_B = 23805,
     DIS_PDU_EMITTERNAME_PLATE_STEER = 23850,
     DIS_PDU_EMITTERNAME_PLESSEY_AWS_1 = 23895,
     DIS_PDU_EMITTERNAME_PLESSEY_AWS_2 = 23925,
     DIS_PDU_EMITTERNAME_PLESSEY_AWS_4 = 23940,
     DIS_PDU_EMITTERNAME_PLESSEY_AWS_6 = 23985,
     DIS_PDU_EMITTERNAME_PLESSEY_RJ = 23990,
     DIS_PDU_EMITTERNAME_PLESSEY_TYPE_904 = 24020,
     DIS_PDU_EMITTERNAME_PLESSEY_TYPE_996 = 24030,
     DIS_PDU_EMITTERNAME_PLESSEY_AWS_9 = 24035,
     DIS_PDU_EMITTERNAME_PLINTH_NET = 24075,
     DIS_PDU_EMITTERNAME_PLUTO = 24095,
     DIS_PDU_EMITTERNAME_PNA_B_RUBIN_DOWN_BEAT = 24098,
     DIS_PDU_EMITTERNAME_POHJANPALO = 24100,
     DIS_PDU_EMITTERNAME_POLIMENT_K = 24110,
     DIS_PDU_EMITTERNAME_POLLUX = 24120,
     DIS_PDU_EMITTERNAME_POP_GROUP = 24165,
     DIS_PDU_EMITTERNAME_POP_GROUP_MG = 24210,
     DIS_PDU_EMITTERNAME_POP_GROUP_TA = 24255,
     DIS_PDU_EMITTERNAME_POP_GROUP_TT = 24300,
     DIS_PDU_EMITTERNAME_PORK_FIST = 24320,
     DIS_PDU_EMITTERNAME_PORK_TROUGH = 24345,
     DIS_PDU_EMITTERNAME_POZITIV_ME1_5P_26 = 24385,
     DIS_PDU_EMITTERNAME_POSITIVE_ME1_2 = 24386,
     DIS_PDU_EMITTERNAME_POZITIV_MK = 24387,
     DIS_PDU_EMITTERNAME_POST_BOW = 24390,
     DIS_PDU_EMITTERNAME_POST_LAMP = 24435,
     DIS_PDU_EMITTERNAME_POT_DRUM = 24480,
     DIS_PDU_EMITTERNAME_POT_HEAD = 24525,
     DIS_PDU_EMITTERNAME_POT_SHOT = 24535,
     DIS_PDU_EMITTERNAME_PRAETORIAN_COUNTERMEASURES_SUITE = 24540,
     DIS_PDU_EMITTERNAME_PRIMUS_30A = 24569,
     DIS_PDU_EMITTERNAME_PRIMUS_40_WXD = 24570,
     DIS_PDU_EMITTERNAME_PRIMUS_400 = 24614,
     DIS_PDU_EMITTERNAME_PRIMUS_300SL = 24615,
     DIS_PDU_EMITTERNAME_PRIMUS_500 = 24616,
     DIS_PDU_EMITTERNAME_PRIMUS_650 = 24617,
     DIS_PDU_EMITTERNAME_PRIMUS_700 = 24618,
     DIS_PDU_EMITTERNAME_PRIMUS_800 = 24619,
     DIS_PDU_EMITTERNAME_PRIMUS_3000 = 24620,
     DIS_PDU_EMITTERNAME_PRIMUS_870 = 24622,
     DIS_PDU_EMITTERNAME_PRORA = 24630,
     DIS_PDU_EMITTERNAME_PRS_2 = 24631,
     DIS_PDU_EMITTERNAME_PRS_3_ARGON_2 = 24633,
     DIS_PDU_EMITTERNAME_PRORA_PA_1660 = 24635,
     DIS_PDU_EMITTERNAME_PS_15 = 24640,
     DIS_PDU_EMITTERNAME_PS_05A = 24650,
     DIS_PDU_EMITTERNAME_PS_46_A = 24660,
     DIS_PDU_EMITTERNAME_PS_70_R = 24705,
     DIS_PDU_EMITTERNAME_PS_171_R = 24706,
     DIS_PDU_EMITTERNAME_PS_860 = 24707,
     DIS_PDU_EMITTERNAME_PS_870 = 24709,
     DIS_PDU_EMITTERNAME_PS_890 = 24710,
     DIS_PDU_EMITTERNAME_PSM_33 = 24720,
     DIS_PDU_EMITTERNAME_PUFF_BALL = 24750,
     DIS_PDU_EMITTERNAME_QUADRADAR_VI = 24755,
     DIS_PDU_EMITTERNAME_QW_1A = 24757,
     DIS_PDU_EMITTERNAME_PHAZOTRON_1RS2_1E = 24758,
     DIS_PDU_EMITTERNAME_PVS_200 = 24760,
     DIS_PDU_EMITTERNAME_PVS_2000 = 24761,
     DIS_PDU_EMITTERNAME_R_330ZH = 24768,
     DIS_PDU_EMITTERNAME_R_045 = 24769,
     DIS_PDU_EMITTERNAME_R_76 = 24770,
     DIS_PDU_EMITTERNAME_R_934B = 24771,
     DIS_PDU_EMITTERNAME_RA_20 = 24772,
     DIS_PDU_EMITTERNAME_RA723 = 24774,
     DIS_PDU_EMITTERNAME_R41XXX = 24775,
     DIS_PDU_EMITTERNAME_RAC_3D = 24776,
     DIS_PDU_EMITTERNAME_RAC_30 = 24780,
     DIS_PDU_EMITTERNAME_R_423AM = 24781,
     DIS_PDU_EMITTERNAME_RAAD_1_TER = 24785,
     DIS_PDU_EMITTERNAME_RAAD_2_TER = 24787,
     DIS_PDU_EMITTERNAME_RACAL_1229 = 24795,
     DIS_PDU_EMITTERNAME_DECCA_1230 = 24800,
     DIS_PDU_EMITTERNAME_RACAL_AC_2690_BT = 24840,
     DIS_PDU_EMITTERNAME_RACAL_DECCA_1216 = 24885,
     DIS_PDU_EMITTERNAME_RACAL_DECCA_20V90_9 = 24890,
     DIS_PDU_EMITTERNAME_RACAL_DECCA_360 = 24930,
     DIS_PDU_EMITTERNAME_RACAL_DECCA_AC_1290 = 24975,
     DIS_PDU_EMITTERNAME_RACAL_DECCA_TM_1229 = 25020,
     DIS_PDU_EMITTERNAME_RACAL_DECCA_TM_1626 = 25065,
     DIS_PDU_EMITTERNAME_RACAL_DRBN_34A = 25110,
     DIS_PDU_EMITTERNAME_RADA_MHR = 25150,
     DIS_PDU_EMITTERNAME_RADAR_24 = 25155,
     DIS_PDU_EMITTERNAME_RADARPILOT_1000 = 25170,
     DIS_PDU_EMITTERNAME_RADARPILOT_1100 = 25171,
     DIS_PDU_EMITTERNAME_RAJENDRA = 25180,
     DIS_PDU_EMITTERNAME_RAN_7S = 25200,
     DIS_PDU_EMITTERNAME_RAN_10S = 25205,
     DIS_PDU_EMITTERNAME_RAN_11_LX = 25245,
     DIS_PDU_EMITTERNAME_RANI = 25250,
     DIS_PDU_EMITTERNAME_RAPHAEL_TH = 25259,
     DIS_PDU_EMITTERNAME_RAPIER_TA = 25260,
     DIS_PDU_EMITTERNAME_RAPIER_2000_TA_DAGGER = 25265,
     DIS_PDU_EMITTERNAME_RAPIER_MG = 25270,
     DIS_PDU_EMITTERNAME_RASCAR_3400C = 25273,
     DIS_PDU_EMITTERNAME_RASHMI = 25275,
     DIS_PDU_EMITTERNAME_RASIT = 25276,
     DIS_PDU_EMITTERNAME_RASIT_3190B = 25277,
     DIS_PDU_EMITTERNAME_RAT_31_DL_M = 25278,
     DIS_PDU_EMITTERNAME_RAT_31_DL = 25279,
     DIS_PDU_EMITTERNAME_RAT_31S = 25280,
     DIS_PDU_EMITTERNAME_RAT_8_S = 25281,
     DIS_PDU_EMITTERNAME_RAT_31_SL = 25282,
     DIS_PDU_EMITTERNAME_RAVEN_ES_05 = 25283,
     DIS_PDU_EMITTERNAME_RATAC_LCT = 25285,
     DIS_PDU_EMITTERNAME_RAWL = 25286,
     DIS_PDU_EMITTERNAME_RATTLER = 25287,
     DIS_PDU_EMITTERNAME_RAWS = 25288,
     DIS_PDU_EMITTERNAME_RAWL_02 = 25289,
     DIS_PDU_EMITTERNAME_RAYTHEON_1220 = 25290,
     DIS_PDU_EMITTERNAME_RAWS_03 = 25291,
     DIS_PDU_EMITTERNAME_RAYTHEON_1210XX = 25292,
     DIS_PDU_EMITTERNAME_RAYTHEON_1302 = 25300,
     DIS_PDU_EMITTERNAME_RAYTHEON_1500 = 25335,
     DIS_PDU_EMITTERNAME_RAYTHEON_1645 = 25380,
     DIS_PDU_EMITTERNAME_RAYTHEON_1650 = 25425,
     DIS_PDU_EMITTERNAME_RAYTHEON_1900 = 25470,
     DIS_PDU_EMITTERNAME_RAYTHEON_2502 = 25515,
     DIS_PDU_EMITTERNAME_RAYTHEON_ANSCHUTZ_NAUTOSCAN_NX = 25530,
     DIS_PDU_EMITTERNAME_RAYTHEON_R41 = 25540,
     DIS_PDU_EMITTERNAME_RAYTHEON_RM_1025_6X = 25545,
     DIS_PDU_EMITTERNAME_RAYTHEON_SL_72 = 25550,
     DIS_PDU_EMITTERNAME_RAYTHEON_TM_1650_6X = 25560,
     DIS_PDU_EMITTERNAME_RAYTHEON_TM_1660_12S = 25605,
     DIS_PDU_EMITTERNAME_RAY_1220XR = 25630,
     DIS_PDU_EMITTERNAME_RAY_1401 = 25635,
     DIS_PDU_EMITTERNAME_RAY_2900 = 25650,
     DIS_PDU_EMITTERNAME_RAYMARINE_RD218 = 25694,
     DIS_PDU_EMITTERNAME_RAYPATH = 25695,
     DIS_PDU_EMITTERNAME_RAYTHEON_PATHFINDER_ST_MK2 = 25698,
     DIS_PDU_EMITTERNAME_RBE2 = 25735,
     DIS_PDU_EMITTERNAME_RBE2_AA = 25736,
     DIS_PDU_EMITTERNAME_RCT_180 = 25739,
     DIS_PDU_EMITTERNAME_RDM = 25740,
     DIS_PDU_EMITTERNAME_RDM_3 = 25745,
     DIS_PDU_EMITTERNAME_RDI = 25750,
     DIS_PDU_EMITTERNAME_RDY = 25760,
     DIS_PDU_EMITTERNAME_RDY_3 = 25762,
     DIS_PDU_EMITTERNAME_RDS_86 = 25770,
     DIS_PDU_EMITTERNAME_RDN_72 = 25785,
     DIS_PDU_EMITTERNAME_RDR_1A = 25830,
     DIS_PDU_EMITTERNAME_RDR_1E = 25835,
     DIS_PDU_EMITTERNAME_RDR_4A = 25840,
     DIS_PDU_EMITTERNAME_RDR_150 = 25845,
     DIS_PDU_EMITTERNAME_RDR_160XD = 25850,
     DIS_PDU_EMITTERNAME_RDR_230_HP = 25853,
     DIS_PDU_EMITTERNAME_RDR_1100 = 25855,
     DIS_PDU_EMITTERNAME_RDR_1150 = 25860,
     DIS_PDU_EMITTERNAME_RDR_1200 = 25875,
     DIS_PDU_EMITTERNAME_RDR_1400 = 25885,
     DIS_PDU_EMITTERNAME_RDR_1400_C = 25890,
     DIS_PDU_EMITTERNAME_RDR_4000 = 25892,
     DIS_PDU_EMITTERNAME_RDR_1500 = 25895,
     DIS_PDU_EMITTERNAME_RICE_CAKE = 25896,
     DIS_PDU_EMITTERNAME_RDR_1600 = 25897,
     DIS_PDU_EMITTERNAME_RDR_2000 = 25898,
     DIS_PDU_EMITTERNAME_RDR_1700B = 25899,
     DIS_PDU_EMITTERNAME_REMORA = 25900,
     DIS_PDU_EMITTERNAME_RICE_FIELD = 25901,
     DIS_PDU_EMITTERNAME_REC_1A = 25902,
     DIS_PDU_EMITTERNAME_REC_1B = 25903,
     DIS_PDU_EMITTERNAME_REC_1C = 25904,
     DIS_PDU_EMITTERNAME_RESOLVE_EAS = 25906,
     DIS_PDU_EMITTERNAME_RICE_CUP_C = 25907,
     DIS_PDU_EMITTERNAME_REL_6E = 25908,
     DIS_PDU_EMITTERNAME_REC_1 = 25909,
     DIS_PDU_EMITTERNAME_RICE_BOWL = 25910,
     DIS_PDU_EMITTERNAME_IMPROVED_REPORTER = 25911,
     DIS_PDU_EMITTERNAME_RICE_BUG = 25912,
     DIS_PDU_EMITTERNAME_RICE_CUP = 25915,
     DIS_PDU_EMITTERNAME_RICE_LAMP = 25920,
     DIS_PDU_EMITTERNAME_RICE_BIRD = 25921,
     DIS_PDU_EMITTERNAME_REVATHI = 25940,
     DIS_PDU_EMITTERNAME_REZONANS = 25950,
     DIS_PDU_EMITTERNAME_RGM_UGM_109B = 25955,
     DIS_PDU_EMITTERNAME_RGM_UGM_109E_HOMING_RADAR = 25958,
     DIS_PDU_EMITTERNAME_RICE_PAD = 25965,
     DIS_PDU_EMITTERNAME_RKL_526 = 25966,
     DIS_PDU_EMITTERNAME_RKZ_764 = 25967,
     DIS_PDU_EMITTERNAME_RKZ_766 = 25968,
     DIS_PDU_EMITTERNAME_RKL_165 = 25969,
     DIS_PDU_EMITTERNAME_RKL_609 = 25970,
     DIS_PDU_EMITTERNAME_RKL_800 = 25971,
     DIS_PDU_EMITTERNAME_RKZ_761 = 25972,
     DIS_PDU_EMITTERNAME_RKZ_2000 = 25973,
     DIS_PDU_EMITTERNAME_RIS_4C_A = 25974,
     DIS_PDU_EMITTERNAME_RL_2000 = 25975,
     DIS_PDU_EMITTERNAME_RL_41 = 25976,
     DIS_PDU_EMITTERNAME_RIR_778 = 25977,
     DIS_PDU_EMITTERNAME_RISAT = 25978,
     DIS_PDU_EMITTERNAME_RLM_S = 25979,
     DIS_PDU_EMITTERNAME_RIM_HAT_ESM_ECM_SUITE = 25980,
     DIS_PDU_EMITTERNAME_REL_86TM = 26008,
     DIS_PDU_EMITTERNAME_RICE_SCREEN = 26010,
     DIS_PDU_EMITTERNAME_DECCA_RM_1070A = 26011,
     DIS_PDU_EMITTERNAME_RM370BT = 26015,
     DIS_PDU_EMITTERNAME_ROCKWELL_COLLINS_FMR_200X = 26020,
     DIS_PDU_EMITTERNAME_RM_231_2 = 26040,
     DIS_PDU_EMITTERNAME_RM_231_3 = 26041,
     DIS_PDU_EMITTERNAME_RMT_0100A = 26043,
     DIS_PDU_EMITTERNAME_RN_222 = 26045,
     DIS_PDU_EMITTERNAME_ROLAND_2 = 26053,
     DIS_PDU_EMITTERNAME_ROLAND_BN = 26055,
     DIS_PDU_EMITTERNAME_ROLAND_MG = 26100,
     DIS_PDU_EMITTERNAME_ROLAND_TA = 26145,
     DIS_PDU_EMITTERNAME_ROLAND_TT = 26190,
     DIS_PDU_EMITTERNAME_ROTODOME = 26210,
     DIS_PDU_EMITTERNAME_ROUND_BALL = 26235,
     DIS_PDU_EMITTERNAME_RP_379D_TIRADA_D = 26236,
     DIS_PDU_EMITTERNAME_RP_3 = 26237,
     DIS_PDU_EMITTERNAME_RP_4G = 26238,
     DIS_PDU_EMITTERNAME_ROUND_HOUSE = 26280,
     DIS_PDU_EMITTERNAME_ROUND_HOUSE_B = 26325,
     DIS_PDU_EMITTERNAME_RPR_117 = 26326,
     DIS_PDU_EMITTERNAME_RS_02_50 = 26327,
     DIS_PDU_EMITTERNAME_RSR_210N = 26328,
     DIS_PDU_EMITTERNAME_RT_02_50 = 26330,
     DIS_PDU_EMITTERNAME_RTA_4100 = 26340,
     DIS_PDU_EMITTERNAME_RTN_1A = 26350,
     DIS_PDU_EMITTERNAME_RTN_25X = 26353,
     DIS_PDU_EMITTERNAME_RTS_6400 = 26354,
     DIS_PDU_EMITTERNAME_RUBY_RAKE = 26355,
     DIS_PDU_EMITTERNAME_RUM_SLING = 26360,
     DIS_PDU_EMITTERNAME_RUM_SLING_RO = 26361,
     DIS_PDU_EMITTERNAME_RUM_SLING_TT = 26362,
     DIS_PDU_EMITTERNAME_RV2 = 26370,
     DIS_PDU_EMITTERNAME_RV3 = 26415,
     DIS_PDU_EMITTERNAME_RV5 = 26460,
     DIS_PDU_EMITTERNAME_RV10 = 26505,
     DIS_PDU_EMITTERNAME_RV_15M = 26506,
     DIS_PDU_EMITTERNAME_RV17 = 26550,
     DIS_PDU_EMITTERNAME_RV18 = 26595,
     DIS_PDU_EMITTERNAME_RV_21 = 26596,
     DIS_PDU_EMITTERNAME_RV_21B = 26597,
     DIS_PDU_EMITTERNAME_RV_25 = 26600,
     DIS_PDU_EMITTERNAME_RV_377 = 26610,
     DIS_PDU_EMITTERNAME_RV_UM = 26640,
     DIS_PDU_EMITTERNAME_RWD_8 = 26650,
     DIS_PDU_EMITTERNAME_RXN_2_60 = 26660,
     DIS_PDU_EMITTERNAME_RYE_HOUSE = 26665,
     DIS_PDU_EMITTERNAME_S_1810CD = 26670,
     DIS_PDU_EMITTERNAME_SAHAB = 26672,
     DIS_PDU_EMITTERNAME_SALAMANDRE = 26673,
     DIS_PDU_EMITTERNAME_SAMYUNG_SMR_7200 = 26674,
     DIS_PDU_EMITTERNAME_S1850M = 26675,
     DIS_PDU_EMITTERNAME_S_511 = 26676,
     DIS_PDU_EMITTERNAME_S_512 = 26677,
     DIS_PDU_EMITTERNAME_S_600 = 26678,
     DIS_PDU_EMITTERNAME_S_604 = 26679,
     DIS_PDU_EMITTERNAME_S_763_LANZA_3D = 26680,
     DIS_PDU_EMITTERNAME_S_613 = 26681,
     DIS_PDU_EMITTERNAME_S_631 = 26682,
     DIS_PDU_EMITTERNAME_S_654 = 26683,
     DIS_PDU_EMITTERNAME_S_669 = 26684,
     DIS_PDU_EMITTERNAME_SA_2_GUIDELINE = 26685,
     DIS_PDU_EMITTERNAME_S_244 = 26686,
     DIS_PDU_EMITTERNAME_S_711 = 26687,
     DIS_PDU_EMITTERNAME_SA_3_GOA = 26730,
     DIS_PDU_EMITTERNAME_SA_8_GECKO_DT = 26775,
     DIS_PDU_EMITTERNAME_GLADIATOR_TELAR_TIR = 26795,
     DIS_PDU_EMITTERNAME_GIANT_TELAR_TIR = 26797,
     DIS_PDU_EMITTERNAME_SABER_M60 = 26799,
     DIS_PDU_EMITTERNAME_SAMOVAR = 26805,
     DIS_PDU_EMITTERNAME_SAMPSON = 26810,
     DIS_PDU_EMITTERNAME_SA_N_7_GADFLY_TI = 26820,
     DIS_PDU_EMITTERNAME_SA_N_11_CADS_1_UN = 26865,
     DIS_PDU_EMITTERNAME_SACCADE_MH = 26900,
     DIS_PDU_EMITTERNAME_SALT_POT_A_B = 26910,
     DIS_PDU_EMITTERNAME_SAP_14 = 26920,
     DIS_PDU_EMITTERNAME_SAP_518 = 26925,
     DIS_PDU_EMITTERNAME_SAP_518M = 26926,
     DIS_PDU_EMITTERNAME_SAND_BAR = 26930,
     DIS_PDU_EMITTERNAME_SA_26_PECHORA_2M_TTR = 26935,
     DIS_PDU_EMITTERNAME_SAR_ON_UAVS = 26945,
     DIS_PDU_EMITTERNAME_SATRAPE = 26950,
     DIS_PDU_EMITTERNAME_SATURNE_II = 26955,
     DIS_PDU_EMITTERNAME_SAYYAD_2_TER = 26957,
     DIS_PDU_EMITTERNAME_SCAN_CAN = 27000,
     DIS_PDU_EMITTERNAME_SCAN_FIX = 27045,
     DIS_PDU_EMITTERNAME_SCAN_ODD = 27090,
     DIS_PDU_EMITTERNAME_SCANTER_1002 = 27095,
     DIS_PDU_EMITTERNAME_SCANTER_2001 = 27100,
     DIS_PDU_EMITTERNAME_SCANTER_2002 = 27101,
     DIS_PDU_EMITTERNAME_SCANTER_2100 = 27102,
     DIS_PDU_EMITTERNAME_SCANTER_4002 = 27109,
     DIS_PDU_EMITTERNAME_SCANTER_4100 = 27110,
     DIS_PDU_EMITTERNAME_SCANTER_5102 = 27111,
     DIS_PDU_EMITTERNAME_SCANTER_5502 = 27113,
     DIS_PDU_EMITTERNAME_SCANTER_6000 = 27115,
     DIS_PDU_EMITTERNAME_SCANTER_6002 = 27116,
     DIS_PDU_EMITTERNAME_SCANTER_MIL_009 = 27125,
     DIS_PDU_EMITTERNAME_SCAN_THREE = 27135,
     DIS_PDU_EMITTERNAME_SCANTER_MIL_S = 27137,
     DIS_PDU_EMITTERNAME_SCANTER_SMR = 27139,
     DIS_PDU_EMITTERNAME_SCANTER_CSR = 27140,
     DIS_PDU_EMITTERNAME_SCORADS = 27141,
     DIS_PDU_EMITTERNAME_SCIMITAR = 27142,
     DIS_PDU_EMITTERNAME_STAR_2000 = 27143,
     DIS_PDU_EMITTERNAME_SCOREBOARD = 27150,
     DIS_PDU_EMITTERNAME_SCOOP_PAIR = 27175,
     DIS_PDU_EMITTERNAME_SCOUP_PLATE = 27180,
     DIS_PDU_EMITTERNAME_SCOUT = 27183,
     DIS_PDU_EMITTERNAME_SCR_584 = 27190,
     DIS_PDU_EMITTERNAME_SEA_ARCHER_2 = 27225,
     DIS_PDU_EMITTERNAME_SEA_BASED_X_BAND = 27230,
     DIS_PDU_EMITTERNAME_SEA_DRAGON = 27235,
     DIS_PDU_EMITTERNAME_SEA_EAGLE_TYPE_381 = 27239,
     DIS_PDU_EMITTERNAME_SEA_EAGLE_S_C_TYPE_382 = 27240,
     DIS_PDU_EMITTERNAME_SEA_FALCON = 27245,
     DIS_PDU_EMITTERNAME_SEA_GIRAFFE_AMB = 27248,
     DIS_PDU_EMITTERNAME_SEAGUARD = 27251,
     DIS_PDU_EMITTERNAME_SEA_HAWK_SHN_X12 = 27260,
     DIS_PDU_EMITTERNAME_SEA_HUNTER_4_MG = 27270,
     DIS_PDU_EMITTERNAME_SEA_HUNTER_4_TA = 27315,
     DIS_PDU_EMITTERNAME_SEA_HUNTER_4_TT = 27360,
     DIS_PDU_EMITTERNAME_SEA_GULL = 27405,
     DIS_PDU_EMITTERNAME_SEA_MASTER_400 = 27430,
     DIS_PDU_EMITTERNAME_SEA_NET = 27450,
     DIS_PDU_EMITTERNAME_SEA_SPRAY = 27495,
     DIS_PDU_EMITTERNAME_SEA_TIGER = 27540,
     DIS_PDU_EMITTERNAME_SEA_TIGER_M = 27550,
     DIS_PDU_EMITTERNAME_SEASTAR_SEA_WATCHER_100 = 27560,
     DIS_PDU_EMITTERNAME_SEARCHWATER = 27570,
     DIS_PDU_EMITTERNAME_SEARCHWATER_2000 = 27575,
     DIS_PDU_EMITTERNAME_SEASONDE = 27580,
     DIS_PDU_EMITTERNAME_SEASPRAY_7000E = 27582,
     DIS_PDU_EMITTERNAME_SEAVUE = 27583,
     DIS_PDU_EMITTERNAME_SEASPRAY_MK3 = 27584,
     DIS_PDU_EMITTERNAME_SELENIA_ORION_7 = 27585,
     DIS_PDU_EMITTERNAME_SELENIA_TYPE_912 = 27630,
     DIS_PDU_EMITTERNAME_SELENNIA_RAN_12_L_X = 27675,
     DIS_PDU_EMITTERNAME_SELENIA_RAN_20S = 27680,
     DIS_PDU_EMITTERNAME_SELENNIA_RTN_10X = 27720,
     DIS_PDU_EMITTERNAME_SELINIA_ARP_1645 = 27765,
     DIS_PDU_EMITTERNAME_SENTIR_M20 = 27770,
     DIS_PDU_EMITTERNAME_SERDAR = 27771,
     DIS_PDU_EMITTERNAME_SERHAT = 27773,
     DIS_PDU_EMITTERNAME_SERIES_10_COMPACT_SUBMARINE_RADAR_CSR = 27775,
     DIS_PDU_EMITTERNAME_SERIES_52 = 27780,
     DIS_PDU_EMITTERNAME_SERIES_320 = 27790,
     DIS_PDU_EMITTERNAME_SG = 27800,
     DIS_PDU_EMITTERNAME_SGJ_02 = 27802,
     DIS_PDU_EMITTERNAME_SGJ_03 = 27803,
     DIS_PDU_EMITTERNAME_SGR_102_00 = 27810,
     DIS_PDU_EMITTERNAME_SGR_103_02 = 27855,
     DIS_PDU_EMITTERNAME_SGR_104 = 27870,
     DIS_PDU_EMITTERNAME_SHAHED_129_SAR = 27873,
     DIS_PDU_EMITTERNAME_SHAHINE = 27875,
     DIS_PDU_EMITTERNAME_SHEET_BEND = 27900,
     DIS_PDU_EMITTERNAME_SHEET_CURVE = 27945,
     DIS_PDU_EMITTERNAME_SHIKRA = 27980,
     DIS_PDU_EMITTERNAME_SHIP_GLOBE = 27990,
     DIS_PDU_EMITTERNAME_SHIP_WHEEL = 28035,
     DIS_PDU_EMITTERNAME_SGR_114 = 28080,
     DIS_PDU_EMITTERNAME_SHORE_WALK_A = 28125,
     DIS_PDU_EMITTERNAME_SHORT_HORN = 28170,
     DIS_PDU_EMITTERNAME_SHOT_DOME = 28215,
     DIS_PDU_EMITTERNAME_SIDE_GLOBE_JN = 28260,
     DIS_PDU_EMITTERNAME_PRV_11_SIDE_NET = 28280,
     DIS_PDU_EMITTERNAME_SIDE_WALK_A = 28305,
     DIS_PDU_EMITTERNAME_SIGNAAL_BHARAT = 28340,
     DIS_PDU_EMITTERNAME_SIGNAAL_DA_02 = 28350,
     DIS_PDU_EMITTERNAME_SIGNAAL_DA_05 = 28395,
     DIS_PDU_EMITTERNAME_SIGNAAL_DA_08 = 28440,
     DIS_PDU_EMITTERNAME_SIGNAAL_DA_08_2LS = 28445,
     DIS_PDU_EMITTERNAME_SIGNAAL_LW_04 = 28480,
     DIS_PDU_EMITTERNAME_SIGNAAL_LW_08 = 28485,
     DIS_PDU_EMITTERNAME_SIGNAAL_LWOR = 28530,
     DIS_PDU_EMITTERNAME_SIGNAAL_M45 = 28575,
     DIS_PDU_EMITTERNAME_SIGNAAL_MW_08 = 28620,
     DIS_PDU_EMITTERNAME_SIGNAAL_SMART = 28665,
     DIS_PDU_EMITTERNAME_SIGNAAL_STING = 28710,
     DIS_PDU_EMITTERNAME_SIGNAAL_STIR = 28755,
     DIS_PDU_EMITTERNAME_SIGNAAL_STIR_1_8M = 28760,
     DIS_PDU_EMITTERNAME_SIGNAAL_STIR_24M = 28770,
     DIS_PDU_EMITTERNAME_SIGNAAL_WM_20_2 = 28800,
     DIS_PDU_EMITTERNAME_SIGNAAL_WM_25 = 28845,
     DIS_PDU_EMITTERNAME_SIGNAAL_WM_27 = 28890,
     DIS_PDU_EMITTERNAME_SIGNAAL_WM_28 = 28935,
     DIS_PDU_EMITTERNAME_SIGNAAL_ZW_01 = 28980,
     DIS_PDU_EMITTERNAME_SIGNAAL_ZW_06 = 29025,
     DIS_PDU_EMITTERNAME_SIGNAAL_ZW_07 = 29030,
     DIS_PDU_EMITTERNAME_SIGNAAL_ZW_0800 = 29035,
     DIS_PDU_EMITTERNAME_SIMRAD_3G = 29043,
     DIS_PDU_EMITTERNAME_SIMRAD_4G = 29045,
     DIS_PDU_EMITTERNAME_SIMRAD_CA54 = 29050,
     DIS_PDU_EMITTERNAME_SIMRAD_HALO_6 = 29060,
     DIS_PDU_EMITTERNAME_SKI_POLE = 29070,
     DIS_PDU_EMITTERNAME_SKIN_HEAD = 29115,
     DIS_PDU_EMITTERNAME_SKIP_SPIN = 29160,
     DIS_PDU_EMITTERNAME_SKYFENDER = 29172,
     DIS_PDU_EMITTERNAME_SKY_WAVE_OVER_THE_HORIZON = 29175,
     DIS_PDU_EMITTERNAME_SKYGUARD_B = 29180,
     DIS_PDU_EMITTERNAME_SKYGUARD_TA_UAR_1021 = 29185,
     DIS_PDU_EMITTERNAME_SKYGUARD_TT_UAR_1021 = 29190,
     DIS_PDU_EMITTERNAME_SKYGUARD_LR = 29191,
     DIS_PDU_EMITTERNAME_SKYMASTER = 29200,
     DIS_PDU_EMITTERNAME_SKY_WATCH = 29205,
     DIS_PDU_EMITTERNAME_SKY_RANGER = 29210,
     DIS_PDU_EMITTERNAME_SKYSHADOW = 29215,
     DIS_PDU_EMITTERNAME_SKYSHIELD_TA = 29220,
     DIS_PDU_EMITTERNAME_SL = 29250,
     DIS_PDU_EMITTERNAME_SL_ALQ_234 = 29270,
     DIS_PDU_EMITTERNAME_SLAP_SHOT = 29295,
     DIS_PDU_EMITTERNAME_SLAP_SHOT_G = 29297,
     DIS_PDU_EMITTERNAME_SLC_2 = 29300,
     DIS_PDU_EMITTERNAME_SLC_2E = 29301,
     DIS_PDU_EMITTERNAME_SLC_4 = 29305,
     DIS_PDU_EMITTERNAME_SLIM_NET = 29340,
     DIS_PDU_EMITTERNAME_SLOT_BACK_A = 29385,
     DIS_PDU_EMITTERNAME_SLOT_BACK_ILL = 29400,
     DIS_PDU_EMITTERNAME_SLOT_BACK_B = 29430,
     DIS_PDU_EMITTERNAME_SLOT_BACK_IV = 29431,
     DIS_PDU_EMITTERNAME_SLOT_BACK_B_TOPAZ = 29432,
     DIS_PDU_EMITTERNAME_SLOT_BACK_E = 29433,
     DIS_PDU_EMITTERNAME_SLOT_BACK_G = 29434,
     DIS_PDU_EMITTERNAME_SLOT_BACK_VI = 29435,
     DIS_PDU_EMITTERNAME_SLOT_REST = 29440,
     DIS_PDU_EMITTERNAME_SM_674A_UPM = 29450,
     DIS_PDU_EMITTERNAME_SMA_3_RM = 29475,
     DIS_PDU_EMITTERNAME_SMA_3_RM_20 = 29520,
     DIS_PDU_EMITTERNAME_SMA_3RM_20A_SMG = 29565,
     DIS_PDU_EMITTERNAME_SMA_BPS_704 = 29610,
     DIS_PDU_EMITTERNAME_SMA_SPIN_749_V_2 = 29655,
     DIS_PDU_EMITTERNAME_SMA_SPN_703 = 29700,
     DIS_PDU_EMITTERNAME_SMA_SPN_751 = 29745,
     DIS_PDU_EMITTERNAME_SMA_SPOS_748 = 29790,
     DIS_PDU_EMITTERNAME_SMA_SPQ_2 = 29835,
     DIS_PDU_EMITTERNAME_SMA_SPQ_2D = 29880,
     DIS_PDU_EMITTERNAME_SMA_SPQ_701 = 29925,
     DIS_PDU_EMITTERNAME_SMA_SPS_702_UPX = 29970,
     DIS_PDU_EMITTERNAME_SMA_ST_2_OTOMAT_II_MH = 30015,
     DIS_PDU_EMITTERNAME_SR_47A = 30016,
     DIS_PDU_EMITTERNAME_SMA_718_BEACON = 30060,
     DIS_PDU_EMITTERNAME_SMALL_FRED = 30065,
     DIS_PDU_EMITTERNAME_SMART_S = 30068,
     DIS_PDU_EMITTERNAME_SMART_S_MK2 = 30069,
     DIS_PDU_EMITTERNAME_SMART_L = 30070,
     DIS_PDU_EMITTERNAME_SM_932 = 30072,
     DIS_PDU_EMITTERNAME_SMOG_LAMP = 30075,
     DIS_PDU_EMITTERNAME_SNAP_SHOT = 30080,
     DIS_PDU_EMITTERNAME_SNOOP_DRIFT = 30105,
     DIS_PDU_EMITTERNAME_SNOOP_HALF = 30140,
     DIS_PDU_EMITTERNAME_SNOOP_HEAD = 30150,
     DIS_PDU_EMITTERNAME_SNOOP_PAIR = 30195,
     DIS_PDU_EMITTERNAME_SNOOP_PLATE = 30240,
     DIS_PDU_EMITTERNAME_SNOOP_PING = 30255,
     DIS_PDU_EMITTERNAME_SNOOP_SLAB = 30285,
     DIS_PDU_EMITTERNAME_SNOOP_TRAY = 30330,
     DIS_PDU_EMITTERNAME_SNOOP_TRAY_1 = 30375,
     DIS_PDU_EMITTERNAME_SNOOP_TRAY_2 = 30420,
     DIS_PDU_EMITTERNAME_SNOOP_TRAY_3 = 30421,
     DIS_PDU_EMITTERNAME_SNOOP_WATCH = 30465,
     DIS_PDU_EMITTERNAME_9S18M1_SNOW_DRIFT = 30470,
     DIS_PDU_EMITTERNAME_9S18M1E_SNOW_DRIFT = 30471,
     DIS_PDU_EMITTERNAME_SPB_7 = 30475,
     DIS_PDU_EMITTERNAME_SNOW_DROP = 30480,
     DIS_PDU_EMITTERNAME_SNW_10 = 30490,
     DIS_PDU_EMITTERNAME_SO_1 = 30510,
     DIS_PDU_EMITTERNAME_SO_12 = 30520,
     DIS_PDU_EMITTERNAME_SO_A_COMMUNIST = 30555,
     DIS_PDU_EMITTERNAME_SO_69 = 30580,
     DIS_PDU_EMITTERNAME_SOCK_EYE = 30600,
     DIS_PDU_EMITTERNAME_SOM_64 = 30645,
     DIS_PDU_EMITTERNAME_SOPKA_HILL = 30650,
     DIS_PDU_EMITTERNAME_SORBSIYA = 30660,
     DIS_PDU_EMITTERNAME_SORBTSIYA_L005 = 30661,
     DIS_PDU_EMITTERNAME_SORBTSIYA_L005S = 30662,
     DIS_PDU_EMITTERNAME_SPADA_SIR = 30665,
     DIS_PDU_EMITTERNAME_SPADA_TT = 30670,
     DIS_PDU_EMITTERNAME_SPARROW_AIM_RIM_7_ILL = 30690,
     DIS_PDU_EMITTERNAME_SPERRY_RASCAR = 30691,
     DIS_PDU_EMITTERNAME_SPECTRA = 30692,
     DIS_PDU_EMITTERNAME_SPEAR3_MMW = 30696,
     DIS_PDU_EMITTERNAME_SPERRY_M_3 = 30700,
     DIS_PDU_EMITTERNAME_SPERRY_VISIONMASTER_FT = 30701,
     DIS_PDU_EMITTERNAME_SPEXER_2000 = 30710,
     DIS_PDU_EMITTERNAME_SPG_53F = 30735,
     DIS_PDU_EMITTERNAME_SPG_70_RTN_10X = 30780,
     DIS_PDU_EMITTERNAME_SPG_74_RTN_20X = 30825,
     DIS_PDU_EMITTERNAME_SPG_75_RTN_30X = 30870,
     DIS_PDU_EMITTERNAME_SPG_76_RTN_30X = 30915,
     DIS_PDU_EMITTERNAME_SPIN_SCAN_A = 30960,
     DIS_PDU_EMITTERNAME_SPIN_SCAN_B = 31005,
     DIS_PDU_EMITTERNAME_SPIN_TROUGH = 31050,
     DIS_PDU_EMITTERNAME_SPINO_D_ADDA_WTR = 31070,
     DIS_PDU_EMITTERNAME_SPJ_40 = 31080,
     DIS_PDU_EMITTERNAME_SPLASH_DROP = 31095,
     DIS_PDU_EMITTERNAME_SPN_2 = 31096,
     DIS_PDU_EMITTERNAME_SPN_4 = 31097,
     DIS_PDU_EMITTERNAME_SPN_30 = 31100,
     DIS_PDU_EMITTERNAME_SPN_35A = 31140,
     DIS_PDU_EMITTERNAME_SPN_41 = 31185,
     DIS_PDU_EMITTERNAME_SPN_42 = 31230,
     DIS_PDU_EMITTERNAME_SPN_43A = 31275,
     DIS_PDU_EMITTERNAME_SPN_43B = 31320,
     DIS_PDU_EMITTERNAME_SPN_44 = 31365,
     DIS_PDU_EMITTERNAME_SPN_46 = 31410,
     DIS_PDU_EMITTERNAME_SPN_703 = 31455,
     DIS_PDU_EMITTERNAME_SPN_720_P_2080 = 31475,
     DIS_PDU_EMITTERNAME_SPN_728_V_1 = 31500,
     DIS_PDU_EMITTERNAME_SPN_748 = 31545,
     DIS_PDU_EMITTERNAME_SPN_750 = 31590,
     DIS_PDU_EMITTERNAME_SPO_8 = 31592,
     DIS_PDU_EMITTERNAME_SPN_753G = 31593,
     DIS_PDU_EMITTERNAME_SPONGE_CAKE = 31635,
     DIS_PDU_EMITTERNAME_P_12_SPOON_REST = 31680,
     DIS_PDU_EMITTERNAME_P_18_SPOON_REST_A = 31681,
     DIS_PDU_EMITTERNAME_P_18_SPOON_REST_B = 31682,
     DIS_PDU_EMITTERNAME_P_18_SPOON_REST_D = 31684,
     DIS_PDU_EMITTERNAME_P_18MH2 = 31685,
     DIS_PDU_EMITTERNAME_SPORK_REST = 31700,
     DIS_PDU_EMITTERNAME_SPQ_712_RAN_12_L_X = 31725,
     DIS_PDU_EMITTERNAME_SPR_2 = 31730,
     DIS_PDU_EMITTERNAME_SPR_51 = 31740,
     DIS_PDU_EMITTERNAME_SPS_5_FASOL = 31765,
     DIS_PDU_EMITTERNAME_SPS_6 = 31766,
     DIS_PDU_EMITTERNAME_SPS_6C = 31770,
     DIS_PDU_EMITTERNAME_SPS_10F = 31815,
     DIS_PDU_EMITTERNAME_SPS_12 = 31860,
     DIS_PDU_EMITTERNAME_SPS_22N_BUKET = 31870,
     DIS_PDU_EMITTERNAME_SPS_33N_BUKET = 31875,
     DIS_PDU_EMITTERNAME_SPS_44N_BUKET = 31880,
     DIS_PDU_EMITTERNAME_SPS_55N_BUKET = 31890,
     DIS_PDU_EMITTERNAME_SPS_62 = 31925,
     DIS_PDU_EMITTERNAME_SPS_100K = 31935,
     DIS_PDU_EMITTERNAME_SPS_141 = 31951,
     DIS_PDU_EMITTERNAME_SPS_142 = 31952,
     DIS_PDU_EMITTERNAME_SPS_143 = 31953,
     DIS_PDU_EMITTERNAME_SPS_151 = 31955,
     DIS_PDU_EMITTERNAME_SPS_152 = 31956,
     DIS_PDU_EMITTERNAME_SPS_153 = 31957,
     DIS_PDU_EMITTERNAME_SPS_160_GERAN = 31959,
     DIS_PDU_EMITTERNAME_SPS_161 = 31960,
     DIS_PDU_EMITTERNAME_SPS_95K = 31970,
     DIS_PDU_EMITTERNAME_SPS_171_JAMMER = 31971,
     DIS_PDU_EMITTERNAME_SPS_172_JAMMER = 31972,
     DIS_PDU_EMITTERNAME_SPS_768_RAN_EL = 31995,
     DIS_PDU_EMITTERNAME_SPS_540K = 32010,
     DIS_PDU_EMITTERNAME_SPS_550K_MF = 32020,
     DIS_PDU_EMITTERNAME_SPS_774_RAN_10S = 32040,
     DIS_PDU_EMITTERNAME_SPY_790 = 32085,
     DIS_PDU_EMITTERNAME_SQUARE_HEAD = 32130,
     DIS_PDU_EMITTERNAME_SQUARE_PAIR = 32175,
     DIS_PDU_EMITTERNAME_SQUARE_SLOT = 32220,
     DIS_PDU_EMITTERNAME_SQUARE_TIE = 32265,
     DIS_PDU_EMITTERNAME_SHMEL_SQUASH_DOME = 32310,
     DIS_PDU_EMITTERNAME_P_15M_SQUAT_EYE = 32330,
     DIS_PDU_EMITTERNAME_SQUINT_EYE = 32355,
     DIS_PDU_EMITTERNAME_SQUIRE = 32365,
     DIS_PDU_EMITTERNAME_SR2410C = 32373,
     DIS_PDU_EMITTERNAME_SR47B_G = 32375,
     DIS_PDU_EMITTERNAME_SRE_M5 = 32385,
     DIS_PDU_EMITTERNAME_SRN_6 = 32400,
     DIS_PDU_EMITTERNAME_SRN_15 = 32445,
     DIS_PDU_EMITTERNAME_SRN_206 = 32455,
     DIS_PDU_EMITTERNAME_SRN_745 = 32490,
     DIS_PDU_EMITTERNAME_SRO_1 = 32535,
     DIS_PDU_EMITTERNAME_SRO_2 = 32580,
     DIS_PDU_EMITTERNAME_SS_C_2B_SAMLET_MG = 32625,
     DIS_PDU_EMITTERNAME_SS_N_2A_B_CSSC = 32670,
     DIS_PDU_EMITTERNAME_SS_N_2A_B_CSSC_2A_3A2_MH = 32715,
     DIS_PDU_EMITTERNAME_SS_N_2C_SEEKER = 32760,
     DIS_PDU_EMITTERNAME_SS_N_2C_D_STYX = 32805,
     DIS_PDU_EMITTERNAME_SS_N_2C_D_STYX_C_D_MH = 32850,
     DIS_PDU_EMITTERNAME_SS_N_2C_STYX_AL = 32851,
     DIS_PDU_EMITTERNAME_SS_N_2D_STYX_AL = 32852,
     DIS_PDU_EMITTERNAME_SS_N_3_SSC_SS_C_18_BN = 32895,
     DIS_PDU_EMITTERNAME_SS_N_3B_SEPAL_AL = 32940,
     DIS_PDU_EMITTERNAME_SS_N_3B_SEPAL_MH = 32985,
     DIS_PDU_EMITTERNAME_SS_N_7_STARBRIGHT = 33025,
     DIS_PDU_EMITTERNAME_SS_N_9_SIREN = 33030,
     DIS_PDU_EMITTERNAME_SS_N_9_SIREN_AL = 33075,
     DIS_PDU_EMITTERNAME_SS_N_9_SIREN_MH = 33120,
     DIS_PDU_EMITTERNAME_SS_N_10A_FL_10_MMW_MH = 33125,
     DIS_PDU_EMITTERNAME_SS_N_11_NASR_1_MMW_MH = 33140,
     DIS_PDU_EMITTERNAME_SS_N_12_SANDBOX_AL = 33165,
     DIS_PDU_EMITTERNAME_SS_N_12_YJ_83J_MMW_MH = 33166,
     DIS_PDU_EMITTERNAME_SS_N_12_SANDBOX_MH = 33210,
     DIS_PDU_EMITTERNAME_SS_NX_13_SHREDDER = 33230,
     DIS_PDU_EMITTERNAME_SS_N_14B_SILEX_AL = 33231,
     DIS_PDU_EMITTERNAME_SS_N_19_SHIPWRECK = 33255,
     DIS_PDU_EMITTERNAME_SS_N_19_SHIPWRECK_AL = 33300,
     DIS_PDU_EMITTERNAME_SS_N_19_SHIPWRECK_MH = 33345,
     DIS_PDU_EMITTERNAME_SS_N_21_AL = 33390,
     DIS_PDU_EMITTERNAME_SS_N_22_SUNBURN = 33435,
     DIS_PDU_EMITTERNAME_SS_N_22_SUNBURN_MH = 33480,
     DIS_PDU_EMITTERNAME_SS_N_22_SUNBURN_AL = 33481,
     DIS_PDU_EMITTERNAME_SS_N_25_SWITCHBLADE_MH = 33483,
     DIS_PDU_EMITTERNAME_SS_N_26_STROBILE_MMW_MH = 33484,
     DIS_PDU_EMITTERNAME_SS_N_27_SIZZLER_MH = 33485,
     DIS_PDU_EMITTERNAME_SS_N_27A_SIZZLER_AL = 33486,
     DIS_PDU_EMITTERNAME_STING_EO_MK2 = 33505,
     DIS_PDU_EMITTERNAME_STIR_1_2_EO_MK2 = 33510,
     DIS_PDU_EMITTERNAME_STIR_2_4_HP_MK2 = 33511,
     DIS_PDU_EMITTERNAME_STONE_CAKE = 33525,
     DIS_PDU_EMITTERNAME_STR_41 = 33570,
     DIS_PDU_EMITTERNAME_ST_858 = 33580,
     DIS_PDU_EMITTERNAME_START_1M = 33582,
     DIS_PDU_EMITTERNAME_STENTOR = 33584,
     DIS_PDU_EMITTERNAME_STORM_SHADOW_AHR = 33585,
     DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH = 33586,
     DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH_TA = 33590,
     DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH_TT = 33595,
     DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH_ILL = 33600,
     DIS_PDU_EMITTERNAME_STRIKE_OUT = 33615,
     DIS_PDU_EMITTERNAME_STRUT_CURVE = 33660,
     DIS_PDU_EMITTERNAME_STRUT_PAIR = 33705,
     DIS_PDU_EMITTERNAME_STRUT_PAIR_1 = 33750,
     DIS_PDU_EMITTERNAME_STRUT_PAIR_2 = 33795,
     DIS_PDU_EMITTERNAME_SUN_VISOR = 33840,
     DIS_PDU_EMITTERNAME_SUPERDARN = 33850,
     DIS_PDU_EMITTERNAME_SUPERFLEDERMAUS = 33860,
     DIS_PDU_EMITTERNAME_SUPERSEARCHER = 33870,
     DIS_PDU_EMITTERNAME_SWIFT_ROD_1 = 33885,
     DIS_PDU_EMITTERNAME_SWIFT_ROD_2 = 33930,
     DIS_PDU_EMITTERNAME_SYMPHONY = 33933,
     DIS_PDU_EMITTERNAME_SYNAPSIS_MK2 = 33935,
     DIS_PDU_EMITTERNAME_SY80 = 33950,
     DIS_PDU_EMITTERNAME_T1166 = 33975,
     DIS_PDU_EMITTERNAME_T1171 = 34020,
     DIS_PDU_EMITTERNAME_T1202 = 34040,
     DIS_PDU_EMITTERNAME_T6004 = 34065,
     DIS_PDU_EMITTERNAME_T6031 = 34110,
     DIS_PDU_EMITTERNAME_T8067 = 34155,
     DIS_PDU_EMITTERNAME_T8068 = 34200,
     DIS_PDU_EMITTERNAME_T8124 = 34245,
     DIS_PDU_EMITTERNAME_T8408 = 34290,
     DIS_PDU_EMITTERNAME_T8911 = 34335,
     DIS_PDU_EMITTERNAME_T8937 = 34380,
     DIS_PDU_EMITTERNAME_T8944 = 34425,
     DIS_PDU_EMITTERNAME_T8987 = 34470,
     DIS_PDU_EMITTERNAME_TA_10K = 34480,
     DIS_PDU_EMITTERNAME_JY_11B_TALL_BOARD = 34500,
     DIS_PDU_EMITTERNAME_TACAN_SURF = 34505,
     DIS_PDU_EMITTERNAME_P_14_TALL_KING = 34515,
     DIS_PDU_EMITTERNAME_TALL_KING_B = 34516,
     DIS_PDU_EMITTERNAME_TALL_KING_C = 34517,
     DIS_PDU_EMITTERNAME_TALL_MIKE = 34560,
     DIS_PDU_EMITTERNAME_TALL_PATH = 34605,
     DIS_PDU_EMITTERNAME_TDR_94_MODE_S = 34607,
     DIS_PDU_EMITTERNAME_TEA_SPOON = 34610,
     DIS_PDU_EMITTERNAME_TEAM_PLAY = 34620,
     DIS_PDU_EMITTERNAME_TALISMAN = 34624,
     DIS_PDU_EMITTERNAME_TEAM_WORK = 34625,
     DIS_PDU_EMITTERNAME_T1135 = 34626,
     DIS_PDU_EMITTERNAME_TANCAN_SURF = 34627,
     DIS_PDU_EMITTERNAME_TECSAR = 34628,
     DIS_PDU_EMITTERNAME_TERRASAR_X = 34629,
     DIS_PDU_EMITTERNAME_TESAR = 34630,
     DIS_PDU_EMITTERNAME_THAAD_GBR = 34640,
     DIS_PDU_EMITTERNAME_THALES_RDY_2 = 34644,
     DIS_PDU_EMITTERNAME_THALES_NEDERLAND_SIGNAAL_APAR = 34645,
     DIS_PDU_EMITTERNAME_THALES_SCORPION_JAMMER = 34646,
     DIS_PDU_EMITTERNAME_THALES_VARIANT = 34647,
     DIS_PDU_EMITTERNAME_THALES_ICMS_JAMMER = 34648,
     DIS_PDU_EMITTERNAME_THALES_IMEWS_JAMMER = 34649,
     DIS_PDU_EMITTERNAME_THD_225 = 34650,
     DIS_PDU_EMITTERNAME_THD_1012 = 34655,
     DIS_PDU_EMITTERNAME_THD_1098 = 34660,
     DIS_PDU_EMITTERNAME_THD_1213 = 34665,
     DIS_PDU_EMITTERNAME_THD_1940_PICADOR = 34670,
     DIS_PDU_EMITTERNAME_THD_1955_PALMIER = 34680,
     DIS_PDU_EMITTERNAME_THD_5500 = 34695,
     DIS_PDU_EMITTERNAME_THIRD_OF_KHORDAD = 34700,
     DIS_PDU_EMITTERNAME_THIN_PATH = 34740,
     DIS_PDU_EMITTERNAME_PRV_9_THIN_SKIN = 34785,
     DIS_PDU_EMITTERNAME_PRV_16_THIN_SKIN_B = 34786,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_TA_10 = 34795,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_TH_D_1040_NEPTUNE = 34830,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_CALYPSO = 34875,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_CASTOR = 34920,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_CASTOR_II = 34965,
     DIS_PDU_EMITTERNAME_THOMSON_CSF_DOMINO_30 = 34966,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBC_32A = 35010,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBJ_11_D_E = 35055,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_15A = 35100,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_15C = 35145,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_22D = 35190,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_23B = 35235,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRUA_33 = 35280,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_MARS_DRBV_21A = 35325,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_SEA_TIGER = 35370,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_TRITON = 35415,
     DIS_PDU_EMITTERNAME_THOMPSON_CSF_VEGA_WITH_DRBC_32E = 35460,
     DIS_PDU_EMITTERNAME_THOMSON_ENR_EUROPEAN_NAVY_RADAR = 35470,
     DIS_PDU_EMITTERNAME_THOMSON_RDI = 35475,
     DIS_PDU_EMITTERNAME_TIER_II_PLUS = 35477,
     DIS_PDU_EMITTERNAME_TPS_755 = 35478,
     DIS_PDU_EMITTERNAME_TPS_830K = 35479,
     DIS_PDU_EMITTERNAME_TRS_2105_TIGER_G = 35480,
     DIS_PDU_EMITTERNAME_TR_23K = 35481,
     DIS_PDU_EMITTERNAME_TR_23MR = 35482,
     DIS_PDU_EMITTERNAME_TRAC_2100 = 35483,
     DIS_PDU_EMITTERNAME_TRAC_2300 = 35484,
     DIS_PDU_EMITTERNAME_HT_223_TIGER_PAW = 35485,
     DIS_PDU_EMITTERNAME_TRADEX = 35486,
     DIS_PDU_EMITTERNAME_TRAIL_XI = 35487,
     DIS_PDU_EMITTERNAME_TRD_1211 = 35488,
     DIS_PDU_EMITTERNAME_TRD_1235 = 35489,
     DIS_PDU_EMITTERNAME_TRS_2100_TIGER_S = 35490,
     DIS_PDU_EMITTERNAME_TRAC_NG = 35491,
     DIS_PDU_EMITTERNAME_TIE_RODS = 35505,
     DIS_PDU_EMITTERNAME_36D6_TIN_SHIELD = 35550,
     DIS_PDU_EMITTERNAME_TIN_TRAP = 35570,
     DIS_PDU_EMITTERNAME_TIRSPONDER = 35580,
     DIS_PDU_EMITTERNAME_TK_25E_5 = 35583,
     DIS_PDU_EMITTERNAME_TMK_MK2 = 35585,
     DIS_PDU_EMITTERNAME_TMX_MK2 = 35586,
     DIS_PDU_EMITTERNAME_TOAD_STOOL_1 = 35595,
     DIS_PDU_EMITTERNAME_TOAD_STOOL_2 = 35640,
     DIS_PDU_EMITTERNAME_TOAD_STOOL_3 = 35685,
     DIS_PDU_EMITTERNAME_TOAD_STOOL_4 = 35730,
     DIS_PDU_EMITTERNAME_TOAD_STOOL_5 = 35775,
     DIS_PDU_EMITTERNAME_TOKEN_B = 35785,
     DIS_PDU_EMITTERNAME_TOMB_STONE = 35800,
     DIS_PDU_EMITTERNAME_TONSON = 35810,
     DIS_PDU_EMITTERNAME_TOP_BOW = 35820,
     DIS_PDU_EMITTERNAME_TOP_DOME = 35865,
     DIS_PDU_EMITTERNAME_TOP_KNOT = 35910,
     DIS_PDU_EMITTERNAME_TOP_MESH = 35955,
     DIS_PDU_EMITTERNAME_TOP_PAIR = 36000,
     DIS_PDU_EMITTERNAME_TOP_PLATE = 36045,
     DIS_PDU_EMITTERNAME_TOP_PLATE_B = 36046,
     DIS_PDU_EMITTERNAME_TOP_SAIL = 36090,
     DIS_PDU_EMITTERNAME_TOP_STEER = 36135,
     DIS_PDU_EMITTERNAME_TOP_TROUGH = 36180,
     DIS_PDU_EMITTERNAME_TORNADO_GMR = 36200,
     DIS_PDU_EMITTERNAME_TORNADO_TFR = 36201,
     DIS_PDU_EMITTERNAME_SCRUM_HALF_TA = 36220,
     DIS_PDU_EMITTERNAME_SCRUM_HALF_TT = 36225,
     DIS_PDU_EMITTERNAME_TOR_M2_TER_SCRUM_HALF_C = 36226,
     DIS_PDU_EMITTERNAME_SCRUM_HALF_MG = 36230,
     DIS_PDU_EMITTERNAME_TRACK_DISH = 36270,
     DIS_PDU_EMITTERNAME_TR_47C = 36300,
     DIS_PDU_EMITTERNAME_TORSO_M = 36315,
     DIS_PDU_EMITTERNAME_TQN_2 = 36320,
     DIS_PDU_EMITTERNAME_TRAP_DOOR = 36360,
     DIS_PDU_EMITTERNAME_TRD_1500 = 36365,
     DIS_PDU_EMITTERNAME_TRICK_SHOT_TAR = 36370,
     DIS_PDU_EMITTERNAME_TRICK_SHOT_TER = 36371,
     DIS_PDU_EMITTERNAME_TRISPONDE = 36380,
     DIS_PDU_EMITTERNAME_TRML = 36381,
     DIS_PDU_EMITTERNAME_TRS_2215 = 36382,
     DIS_PDU_EMITTERNAME_TRML_3D = 36383,
     DIS_PDU_EMITTERNAME_TRM_S = 36384,
     DIS_PDU_EMITTERNAME_TRS_2056 = 36385,
     DIS_PDU_EMITTERNAME_TRS_3010 = 36386,
     DIS_PDU_EMITTERNAME_TRS_2060 = 36387,
     DIS_PDU_EMITTERNAME_TRS_2245 = 36388,
     DIS_PDU_EMITTERNAME_TRS_2310 = 36389,
     DIS_PDU_EMITTERNAME_TRITON_G = 36390,
     DIS_PDU_EMITTERNAME_TRS_22XX = 36391,
     DIS_PDU_EMITTERNAME_TRS_3030 = 36400,
     DIS_PDU_EMITTERNAME_TRS_3033 = 36405,
     DIS_PDU_EMITTERNAME_TRS_3203 = 36417,
     DIS_PDU_EMITTERNAME_TRS_3405 = 36420,
     DIS_PDU_EMITTERNAME_TRS_3410 = 36425,
     DIS_PDU_EMITTERNAME_TRS_3415 = 36430,
     DIS_PDU_EMITTERNAME_TRS_3D = 36440,
     DIS_PDU_EMITTERNAME_TRS_3D_16 = 36441,
     DIS_PDU_EMITTERNAME_TRS_3D_16_ES = 36442,
     DIS_PDU_EMITTERNAME_TRS_3D_32 = 36443,
     DIS_PDU_EMITTERNAME_TRS_4D = 36446,
     DIS_PDU_EMITTERNAME_TRS_C = 36447,
     DIS_PDU_EMITTERNAME_TRS_N = 36450,
     DIS_PDU_EMITTERNAME_TS_4478A = 36460,
     DIS_PDU_EMITTERNAME_TSE_5000 = 36495,
     DIS_PDU_EMITTERNAME_TSR_333 = 36540,
     DIS_PDU_EMITTERNAME_TSR_793 = 36550,
     DIS_PDU_EMITTERNAME_TUB_BRICK = 36563,
     DIS_PDU_EMITTERNAME_TUBE_ARM = 36585,
     DIS_PDU_EMITTERNAME_TW_1374 = 36590,
     DIS_PDU_EMITTERNAME_TW_1378 = 36595,
     DIS_PDU_EMITTERNAME_TW_1446 = 36600,
     DIS_PDU_EMITTERNAME_TWIN_EYES = 36630,
     DIS_PDU_EMITTERNAME_TWIN_PILL = 36675,
     DIS_PDU_EMITTERNAME_TWIN_SCAN = 36720,
     DIS_PDU_EMITTERNAME_TWIN_SCAN_RO = 36765,
     DIS_PDU_EMITTERNAME_TWO_SPOT = 36810,
     DIS_PDU_EMITTERNAME_TYPE_071_LPD = 36821,
     DIS_PDU_EMITTERNAME_TYPE_2_12_J_A = 36827,
     DIS_PDU_EMITTERNAME_TYPE_2_21_J_A = 36830,
     DIS_PDU_EMITTERNAME_TYPE_2_23 = 36835,
     DIS_PDU_EMITTERNAME_TYPE_80_ASM_1 = 36836,
     DIS_PDU_EMITTERNAME_TYPE_120 = 36838,
     DIS_PDU_EMITTERNAME_TYPE_208 = 36840,
     DIS_PDU_EMITTERNAME_TYPE_222 = 36843,
     DIS_PDU_EMITTERNAME_TYPE_226 = 36846,
     DIS_PDU_EMITTERNAME_TYPE_232H = 36850,
     DIS_PDU_EMITTERNAME_TYPE_245 = 36853,
     DIS_PDU_EMITTERNAME_TYPE_262 = 36855,
     DIS_PDU_EMITTERNAME_TYPE_275 = 36900,
     DIS_PDU_EMITTERNAME_TYPE_278 = 36905,
     DIS_PDU_EMITTERNAME_TYPE_293 = 36945,
     DIS_PDU_EMITTERNAME_TYPE_341 = 36946,
     DIS_PDU_EMITTERNAME_TYPE_313 = 36947,
     DIS_PDU_EMITTERNAME_TYPE_305A = 36948,
     DIS_PDU_EMITTERNAME_TYPE_334 = 36960,
     DIS_PDU_EMITTERNAME_TYPE_342 = 36985,
     DIS_PDU_EMITTERNAME_TYPE_343_SUN_VISOR_B = 36990,
     DIS_PDU_EMITTERNAME_TYPE_344 = 36992,
     DIS_PDU_EMITTERNAME_TYPE_345 = 37010,
     DIS_PDU_EMITTERNAME_TYPE_346 = 37011,
     DIS_PDU_EMITTERNAME_TYPE_349A = 37033,
     DIS_PDU_EMITTERNAME_TYPE_347B = 37035,
     DIS_PDU_EMITTERNAME_TYPE_347G = 37038,
     DIS_PDU_EMITTERNAME_TYPE_359 = 37039,
     DIS_PDU_EMITTERNAME_TYPE_352 = 37040,
     DIS_PDU_EMITTERNAME_TYPE_360 = 37041,
     DIS_PDU_EMITTERNAME_TYPE_362_ESR_1_SR_47B = 37043,
     DIS_PDU_EMITTERNAME_TYPE_354 = 37045,
     DIS_PDU_EMITTERNAME_TYPE_366 = 37047,
     DIS_PDU_EMITTERNAME_TYPE_363 = 37048,
     DIS_PDU_EMITTERNAME_TYPE_364_SEAGULL_C = 37049,
     DIS_PDU_EMITTERNAME_TYPE_404A_CH = 37050,
     DIS_PDU_EMITTERNAME_TYPE_405 = 37052,
     DIS_PDU_EMITTERNAME_TYPE_405J = 37053,
     DIS_PDU_EMITTERNAME_TYPE_408D_MOON_STOOL_B = 37058,
     DIS_PDU_EMITTERNAME_TYPE_517B = 37059,
     DIS_PDU_EMITTERNAME_TYPE_518_HAI_YING_GOD_EYE_REL_2 = 37060,
     DIS_PDU_EMITTERNAME_TYPE_589 = 37070,
     DIS_PDU_EMITTERNAME_TYPE_651 = 37073,
     DIS_PDU_EMITTERNAME_TYPE_702 = 37077,
     DIS_PDU_EMITTERNAME_TYPE_704 = 37078,
     DIS_PDU_EMITTERNAME_TYPE_753 = 37079,
     DIS_PDU_EMITTERNAME_TYPE_756 = 37080,
     DIS_PDU_EMITTERNAME_TYPE_713 = 37081,
     DIS_PDU_EMITTERNAME_TYPE_714 = 37082,
     DIS_PDU_EMITTERNAME_TYPE_702_D = 37083,
     DIS_PDU_EMITTERNAME_TYPE_760 = 37086,
     DIS_PDU_EMITTERNAME_TYPE_815 = 37090,
     DIS_PDU_EMITTERNAME_TYPE_793 = 37095,
     DIS_PDU_EMITTERNAME_TYPE_8A_813 = 37100,
     DIS_PDU_EMITTERNAME_TYPE_901M = 37105,
     DIS_PDU_EMITTERNAME_TYPE_902 = 37110,
     DIS_PDU_EMITTERNAME_TYPE_902B = 37124,
     DIS_PDU_EMITTERNAME_TYPE_903 = 37125,
     DIS_PDU_EMITTERNAME_TYPE_909_TI = 37170,
     DIS_PDU_EMITTERNAME_TYPE_909_TT = 37215,
     DIS_PDU_EMITTERNAME_TYPE_910 = 37260,
     DIS_PDU_EMITTERNAME_TYPE_931_CH = 37265,
     DIS_PDU_EMITTERNAME_TYPE_965 = 37305,
     DIS_PDU_EMITTERNAME_TYPE_967 = 37350,
     DIS_PDU_EMITTERNAME_TYPE_968 = 37395,
     DIS_PDU_EMITTERNAME_TYPE_974 = 37440,
     DIS_PDU_EMITTERNAME_TYPE_975 = 37485,
     DIS_PDU_EMITTERNAME_TYPE_978 = 37530,
     DIS_PDU_EMITTERNAME_TYPE_981 = 37534,
     DIS_PDU_EMITTERNAME_TYPE_981_3 = 37535,
     DIS_PDU_EMITTERNAME_TYPE_982 = 37540,
     DIS_PDU_EMITTERNAME_TYPE_984 = 37543,
     DIS_PDU_EMITTERNAME_TYPE_985 = 37544,
     DIS_PDU_EMITTERNAME_TYPE_992 = 37575,
     DIS_PDU_EMITTERNAME_TYPE_993 = 37620,
     DIS_PDU_EMITTERNAME_TYPE_994 = 37665,
     DIS_PDU_EMITTERNAME_TYPE_996 = 37670,
     DIS_PDU_EMITTERNAME_TYPE_997_ARTISAN = 37675,
     DIS_PDU_EMITTERNAME_TYPE_1006_1 = 37710,
     DIS_PDU_EMITTERNAME_TYPE_1006_2 = 37755,
     DIS_PDU_EMITTERNAME_TYPE_1022 = 37800,
     DIS_PDU_EMITTERNAME_TYPE_1047 = 37810,
     DIS_PDU_EMITTERNAME_TYPE_1048 = 37815,
     DIS_PDU_EMITTERNAME_TYPE_1474 = 37825,
     DIS_PDU_EMITTERNAME_TYPE_1493 = 37828,
     DIS_PDU_EMITTERNAME_ULTRA = 37840,
     DIS_PDU_EMITTERNAME_UK_MK_10 = 37845,
     DIS_PDU_EMITTERNAME_UPS_220C = 37850,
     DIS_PDU_EMITTERNAME_UPX_1_10 = 37890,
     DIS_PDU_EMITTERNAME_UPX_27 = 37935,
     DIS_PDU_EMITTERNAME_URN_20 = 37980,
     DIS_PDU_EMITTERNAME_UTES_A = 37985,
     DIS_PDU_EMITTERNAME_UTES_T = 37990,
     DIS_PDU_EMITTERNAME_URN_25 = 38025,
     DIS_PDU_EMITTERNAME_VIGILANT = 38035,
     DIS_PDU_EMITTERNAME_VITEBSK_L370_JAMMER = 38038,
     DIS_PDU_EMITTERNAME_VOLEX_III_IV = 38045,
     DIS_PDU_EMITTERNAME_VOLGA = 38046,
     DIS_PDU_EMITTERNAME_VORONEZH_DM = 38047,
     DIS_PDU_EMITTERNAME_VOSTOK = 38048,
     DIS_PDU_EMITTERNAME_VOSTOK_E = 38049,
     DIS_PDU_EMITTERNAME_VSR = 38050,
     DIS_PDU_EMITTERNAME_VOSTOK_3D = 38051,
     DIS_PDU_EMITTERNAME_VSTAR_PT = 38055,
     DIS_PDU_EMITTERNAME_W_160 = 38058,
     DIS_PDU_EMITTERNAME_W1028 = 38060,
     DIS_PDU_EMITTERNAME_W8818 = 38070,
     DIS_PDU_EMITTERNAME_W8838 = 38115,
     DIS_PDU_EMITTERNAME_W8852 = 38120,
     DIS_PDU_EMITTERNAME_WALL_BOARD = 38140,
     DIS_PDU_EMITTERNAME_WALL_RUST = 38150,
     DIS_PDU_EMITTERNAME_WAS_74S = 38160,
     DIS_PDU_EMITTERNAME_WASP_HEAD = 38205,
     DIS_PDU_EMITTERNAME_WATCHDOG = 38210,
     DIS_PDU_EMITTERNAME_WATCH_GUARD = 38250,
     DIS_PDU_EMITTERNAME_WATCHMAN = 38260,
     DIS_PDU_EMITTERNAME_WAVESTORM = 38270,
     DIS_PDU_EMITTERNAME_WATCHMAN_S = 38275,
     DIS_PDU_EMITTERNAME_WATCHMAN_T = 38276,
     DIS_PDU_EMITTERNAME_WEATHER_SCOUT_2 = 38280,
     DIS_PDU_EMITTERNAME_WESTERN_ELECTRIC_MK_10 = 38295,
     DIS_PDU_EMITTERNAME_WESTINGHOUSE_ADR_4_LRSR = 38320,
     DIS_PDU_EMITTERNAME_WESTINGHOUSE_ELECTRIC_SPG_50 = 38340,
     DIS_PDU_EMITTERNAME_WESTINGHOUSE_ELECTRIC_W_120 = 38385,
     DIS_PDU_EMITTERNAME_WESTINGHOUSE_SPS_29C = 38430,
     DIS_PDU_EMITTERNAME_WESTINGHOUSE_SPS_37 = 38475,
     DIS_PDU_EMITTERNAME_WET_EYE = 38520,
     DIS_PDU_EMITTERNAME_WET_EYE_2 = 38525,
     DIS_PDU_EMITTERNAME_WET_EYE_MOD = 38565,
     DIS_PDU_EMITTERNAME_WF44S = 38568,
     DIS_PDU_EMITTERNAME_WGU_41_B = 38570,
     DIS_PDU_EMITTERNAME_WGU_44_B = 38572,
     DIS_PDU_EMITTERNAME_WHIFF = 38610,
     DIS_PDU_EMITTERNAME_WHIFF_BRICK = 38655,
     DIS_PDU_EMITTERNAME_WHIFF_FIRE = 38700,
     DIS_PDU_EMITTERNAME_WHITE_HOUSE = 38715,
     DIS_PDU_EMITTERNAME_WIDE_MAT = 38730,
     DIS_PDU_EMITTERNAME_WINE_GLASS_JAMMER = 38735,
     DIS_PDU_EMITTERNAME_WILD_CARD = 38745,
     DIS_PDU_EMITTERNAME_WILDCAT = 38748,
     DIS_PDU_EMITTERNAME_WITCH_EIGHT = 38790,
     DIS_PDU_EMITTERNAME_WITCH_FIVE = 38835,
     DIS_PDU_EMITTERNAME_WLR = 38840,
     DIS_PDU_EMITTERNAME_WM2X_SERIES = 38880,
     DIS_PDU_EMITTERNAME_WM2X_SERIES_CAS = 38925,
     DIS_PDU_EMITTERNAME_WR_10X = 38930,
     DIS_PDU_EMITTERNAME_WR_2100 = 38935,
     DIS_PDU_EMITTERNAME_WSR_74C = 38950,
     DIS_PDU_EMITTERNAME_WSR_74S = 38955,
     DIS_PDU_EMITTERNAME_WSR_81 = 38957,
     DIS_PDU_EMITTERNAME_WXR_700C = 38960,
     DIS_PDU_EMITTERNAME_WXR_2100_MSTT = 38965,
     DIS_PDU_EMITTERNAME_WOOD_GAGE = 38970,
     DIS_PDU_EMITTERNAME_X_TAR25 = 38990,
     DIS_PDU_EMITTERNAME_X_TAR3D = 38995,
     DIS_PDU_EMITTERNAME_YAOGAN_3 = 39000,
     DIS_PDU_EMITTERNAME_YAOGAN_29 = 39014,
     DIS_PDU_EMITTERNAME_YARD_RAKE = 39015,
     DIS_PDU_EMITTERNAME_YH_96 = 39050,
     DIS_PDU_EMITTERNAME_YEW_LOOP = 39060,
     DIS_PDU_EMITTERNAME_YITIAN_ADS = 39061,
     DIS_PDU_EMITTERNAME_YD_3 = 39062,
     DIS_PDU_EMITTERNAME_YJ_12_MH = 39063,
     DIS_PDU_EMITTERNAME_YJ_62_MH = 39065,
     DIS_PDU_EMITTERNAME_YJ_82_MH = 39066,
     DIS_PDU_EMITTERNAME_YJ_83_MH = 39067,
     DIS_PDU_EMITTERNAME_YJ_63 = 39068,
     DIS_PDU_EMITTERNAME_YLC_2 = 39070,
     DIS_PDU_EMITTERNAME_YLC_2A = 39071,
     DIS_PDU_EMITTERNAME_YLC_4 = 39073,
     DIS_PDU_EMITTERNAME_YLC_6 = 39074,
     DIS_PDU_EMITTERNAME_YLC_6M = 39075,
     DIS_PDU_EMITTERNAME_YLC_8 = 39080,
     DIS_PDU_EMITTERNAME_YLC_8B = 39081,
     DIS_PDU_EMITTERNAME_YLC_18_TALL_PAGE = 39085,
     DIS_PDU_EMITTERNAME_YO_YO = 39105,
     DIS_PDU_EMITTERNAME_ZASLON_A = 39110,
     DIS_PDU_EMITTERNAME_ZASLON_MULTI_PURPOSE_X_AND_S_BAND = 39112,
     DIS_PDU_EMITTERNAME_ZOO_PARK_1 = 39125,
     DIS_PDU_EMITTERNAME_ZPS_6 = 39126,
     DIS_PDU_EMITTERNAME_ZOOPARK_3 = 39127,
     DIS_PDU_EMITTERNAME_ZOOPARK_1M = 39128,
     DIS_PDU_EMITTERNAME_ZD_12 = 39131,
     DIS_PDU_EMITTERNAME_ZW_06 = 39150,
     DIS_PDU_EMITTERNAME_AN_ALQ_136_V_1 = 39200,
     DIS_PDU_EMITTERNAME_AN_ALQ_136_V_2 = 39201,
     DIS_PDU_EMITTERNAME_AN_ALQ_136_V_3 = 39202,
     DIS_PDU_EMITTERNAME_AN_ALQ_136_V_4 = 39203,
     DIS_PDU_EMITTERNAME_AN_ALQ_136_V_5 = 39204,
     DIS_PDU_EMITTERNAME_AN_ALQ_162_V_2 = 39210,
     DIS_PDU_EMITTERNAME_AN_ALQ_162_V_3 = 39211,
     DIS_PDU_EMITTERNAME_AN_ALQ_162_V_4 = 39212,
     DIS_PDU_EMITTERNAME_ZHUK_M = 45300,
     DIS_PDU_EMITTERNAME_ZHUK_MAE = 45303,
     DIS_PDU_EMITTERNAME_ZHUK_ME = 45304,
     DIS_PDU_EMITTERNAME_ZHUK_MME = 45305,
     DIS_PDU_EMITTERNAME_ZHUK_MSE = 45307
}  DIS_PDU_EmitterName;

static const value_string DIS_PDU_EmitterName_Strings[] = {
     { DIS_PDU_EMITTERNAME_1245_6X, "1245/6X" },
     { DIS_PDU_EMITTERNAME_1L117, "1L117" },
     { DIS_PDU_EMITTERNAME_1L121E, "1L121E" },
     { DIS_PDU_EMITTERNAME_1L250, "1L250" },
     { DIS_PDU_EMITTERNAME_1L220_U, "1L220-U" },
     { DIS_PDU_EMITTERNAME_1L122_1E, "1L122-1E" },
     { DIS_PDU_EMITTERNAME_1RL257, "1RL257" },
     { DIS_PDU_EMITTERNAME_1RL138, "1RL138" },
     { DIS_PDU_EMITTERNAME_1RL257_KRASUKHA_4_JAMMER, "1RL257 (Krasukha-4) Jammer" },
     { DIS_PDU_EMITTERNAME_5N20, "5N20" },
     { DIS_PDU_EMITTERNAME_5H62B, "5H62B" },
     { DIS_PDU_EMITTERNAME_5P_10_PUMA, "5P-10 Puma" },
     { DIS_PDU_EMITTERNAME_5P_10E_PUMA, "5P-10E Puma" },
     { DIS_PDU_EMITTERNAME_5P_10_01_PUMA, "5P-10-01 Puma" },
     { DIS_PDU_EMITTERNAME_5P_10_01E_PUMA, "5P-10-01E Puma" },
     { DIS_PDU_EMITTERNAME_5P_10_02_PUMA, "5P-10-02 Puma" },
     { DIS_PDU_EMITTERNAME_5P_10_02E_PUMA, "5P-10-02E Puma" },
     { DIS_PDU_EMITTERNAME_5P_10_03_LASKA, "5P-10-03 Laska" },
     { DIS_PDU_EMITTERNAME_5P_10_03E_LASKA, "5P-10-03E Laska" },
     { DIS_PDU_EMITTERNAME_5P_10E_MOD, "5P-10E MOD" },
     { DIS_PDU_EMITTERNAME_621A_3, "621A-3" },
     { DIS_PDU_EMITTERNAME_860F_1_AL_101, " 860F-1 / AL-101" },
     { DIS_PDU_EMITTERNAME_9B_1103M2, "9B-1103M2" },
     { DIS_PDU_EMITTERNAME_1226_DECCA_MIL, "1226 DECCA MIL" },
     { DIS_PDU_EMITTERNAME_9B_1348, "9B-1348" },
     { DIS_PDU_EMITTERNAME_3KM6, "3KM6" },
     { DIS_PDU_EMITTERNAME_9KR400, "9KR400" },
     { DIS_PDU_EMITTERNAME_50N6A, "50N6A" },
     { DIS_PDU_EMITTERNAME_55G6_1, "55G6-1" },
     { DIS_PDU_EMITTERNAME_59N6, "59N6" },
     { DIS_PDU_EMITTERNAME_5N69, "5N69" },
     { DIS_PDU_EMITTERNAME_67N6, "67N6" },
     { DIS_PDU_EMITTERNAME_76T6, "76T6" },
     { DIS_PDU_EMITTERNAME_77T6_ABM, "77T6 ABM" },
     { DIS_PDU_EMITTERNAME_80K6, "80K6" },
     { DIS_PDU_EMITTERNAME_91N6A_M, "91N6A(M)" },
     { DIS_PDU_EMITTERNAME_96L6E, "96L6E" },
     { DIS_PDU_EMITTERNAME_96L6_TSP, "96L6-TsP" },
     { DIS_PDU_EMITTERNAME_9C18M3, "9C18M3" },
     { DIS_PDU_EMITTERNAME_9C36M, "9C36M" },
     { DIS_PDU_EMITTERNAME_9GR400, "9GR400" },
     { DIS_PDU_EMITTERNAME_9_GR_400A, "9 GR 400A" },
     { DIS_PDU_EMITTERNAME_9GR600, "9GR600" },
     { DIS_PDU_EMITTERNAME_9GR606, "9GR606" },
     { DIS_PDU_EMITTERNAME_9_LV_100, "9 LV 100" },
     { DIS_PDU_EMITTERNAME_9LV_200_TA, "9LV 200 TA" },
     { DIS_PDU_EMITTERNAME_9LV_200_TV, "9LV 200 TV" },
     { DIS_PDU_EMITTERNAME_9LV_200_TT, "9LV 200 TT" },
     { DIS_PDU_EMITTERNAME_9LV200_MK_III, "9LV200 MK III" },
     { DIS_PDU_EMITTERNAME_9LV326, "9LV326" },
     { DIS_PDU_EMITTERNAME_9M96E2_SEEKER, "9M96E2 Seeker" },
     { DIS_PDU_EMITTERNAME_9S15M2, "9S15M2" },
     { DIS_PDU_EMITTERNAME_9S19M2, "9S19M2" },
     { DIS_PDU_EMITTERNAME_9S19ME, "9S19ME" },
     { DIS_PDU_EMITTERNAME_9S32M, "9S32M" },
     { DIS_PDU_EMITTERNAME_9S32ME, "9S32ME" },
     { DIS_PDU_EMITTERNAME_9S36E, "9S36E" },
     { DIS_PDU_EMITTERNAME_9S112, "9S112" },
     { DIS_PDU_EMITTERNAME_A310Z, " A310Z" },
     { DIS_PDU_EMITTERNAME_A325A, " A325A" },
     { DIS_PDU_EMITTERNAME_A346Z, " A346Z" },
     { DIS_PDU_EMITTERNAME_A353B, " A353B" },
     { DIS_PDU_EMITTERNAME_A372A, " A372A" },
     { DIS_PDU_EMITTERNAME_A372B, " A372B" },
     { DIS_PDU_EMITTERNAME_A372C, " A372C" },
     { DIS_PDU_EMITTERNAME_A377A, " A377A" },
     { DIS_PDU_EMITTERNAME_A377B, " A377B" },
     { DIS_PDU_EMITTERNAME_A380Z, " A380Z" },
     { DIS_PDU_EMITTERNAME_A381Z, " A381Z" },
     { DIS_PDU_EMITTERNAME_A398Z, " A398Z" },
     { DIS_PDU_EMITTERNAME_A403Z, " A403Z" },
     { DIS_PDU_EMITTERNAME_A409A, " A409A" },
     { DIS_PDU_EMITTERNAME_A418A, " A418A" },
     { DIS_PDU_EMITTERNAME_A419Z, " A419Z" },
     { DIS_PDU_EMITTERNAME_A429Z, " A429Z" },
     { DIS_PDU_EMITTERNAME_A432Z, " A432Z" },
     { DIS_PDU_EMITTERNAME_A434Z, " A434Z" },
     { DIS_PDU_EMITTERNAME_AA_6C_ACRID_R_40, "AA-6C Acrid (R-40)" },
     { DIS_PDU_EMITTERNAME_AA_7C_APEX_R_24R, "AA-7C Apex (R-24R)" },
     { DIS_PDU_EMITTERNAME_A401A, " A401A" },
     { DIS_PDU_EMITTERNAME_AA_10A_R_27R, "AA-10A (R-27R)" },
     { DIS_PDU_EMITTERNAME_AA_10C_ALAMO_C, " AA-10C Alamo-C" },
     { DIS_PDU_EMITTERNAME_AA_13_ARROW_SEEKER, "AA-13 Arrow Seeker" },
     { DIS_PDU_EMITTERNAME_AAM_4B_MH, "AAM-4B MH" },
     { DIS_PDU_EMITTERNAME_AA_300, "AA-300" },
     { DIS_PDU_EMITTERNAME_AA_12_SEEKER, "AA-12 Seeker" },
     { DIS_PDU_EMITTERNAME_AD4A, " AD4A" },
     { DIS_PDU_EMITTERNAME_ADES, "ADES" },
     { DIS_PDU_EMITTERNAME_ADS_4_LRSR, "ADS-4 LRSR" },
     { DIS_PDU_EMITTERNAME_ACR_430, "ACR-430" },
     { DIS_PDU_EMITTERNAME_AGAVE, "Agave" },
     { DIS_PDU_EMITTERNAME_ACSOPRI_E, "ACSOPRI-E" },
     { DIS_PDU_EMITTERNAME_ABD_2000, "ABD 2000" },
     { DIS_PDU_EMITTERNAME_ADAC_MK_1, "ADAC MK 1" },
     { DIS_PDU_EMITTERNAME_ADAC_MK_2, "ADAC MK 2" },
     { DIS_PDU_EMITTERNAME_ADAR, "ADAR" },
     { DIS_PDU_EMITTERNAME_ADOUR, "ADOUR" },
     { DIS_PDU_EMITTERNAME_AGAT_9B_1348, "AGAT 9B-1348" },
     { DIS_PDU_EMITTERNAME_ADROS_KT_01AV, "Adros KT-01AV" },
     { DIS_PDU_EMITTERNAME_AGAT_9E420, "Agat 9E420" },
     { DIS_PDU_EMITTERNAME_AGM_158_JASSM_SAR, "AGM-158 JASSM SAR" },
     { DIS_PDU_EMITTERNAME_AGM_88_HARM_MMW, "AGM-88 HARM MMW" },
     { DIS_PDU_EMITTERNAME_AGRION_15, "AGRION 15" },
     { DIS_PDU_EMITTERNAME_AHV_7, " AHV-7" },
     { DIS_PDU_EMITTERNAME_AHV_17, "AHV-17" },
     { DIS_PDU_EMITTERNAME_AI_MK_23, "AI MK 23" },
     { DIS_PDU_EMITTERNAME_AIDA_II, "AIDA II" },
     { DIS_PDU_EMITTERNAME_AIM_120A, "AIM-120A" },
     { DIS_PDU_EMITTERNAME_AIM_7M_SPARROW, "AIM-7M Sparrow" },
     { DIS_PDU_EMITTERNAME_1L271_AISTENOK, "1L271 AISTENOK" },
     { DIS_PDU_EMITTERNAME_ALA_51, "ALA-51" },
     { DIS_PDU_EMITTERNAME_ALBATROS_MK2, "Albatros MK2" },
     { DIS_PDU_EMITTERNAME_ALT_50, "ALT-50" },
     { DIS_PDU_EMITTERNAME_ALTAIR, "ALTAIR" },
     { DIS_PDU_EMITTERNAME_AM_APS_717, "AM/APS-717" },
     { DIS_PDU_EMITTERNAME_AMES_13_MK_1, "AMES 13 MK 1" },
     { DIS_PDU_EMITTERNAME_WGU_16_B, "WGU-16/B" },
     { DIS_PDU_EMITTERNAME_1L13_3_55G6_BOX_SPRING, "1L13-3 (55G6) Box Spring" },
     { DIS_PDU_EMITTERNAME_1L13_3_55G6_BOX_SPRING_B, "1L13-3 (55G6) Box Spring B" },
     { DIS_PDU_EMITTERNAME_AMDR_3D, "AMDR 3D" },
     { DIS_PDU_EMITTERNAME_ANA_SPS_502, "ANA SPS 502" },
     { DIS_PDU_EMITTERNAME_ANEMONE, " Anemone" },
     { DIS_PDU_EMITTERNAME_ANRITSU_ELECTRIC_AR_30A, "ANRITSU Electric AR-30A" },
     { DIS_PDU_EMITTERNAME_ANTILOPE_V, "Antilope V" },
     { DIS_PDU_EMITTERNAME_AN_AAQ_24, "AN/AAQ-24" },
     { DIS_PDU_EMITTERNAME_AN_ADM_160, "AN/ADM-160" },
     { DIS_PDU_EMITTERNAME_AN_ALE_50, "AN/ALE-50" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_76, "AN/ALQ-76" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_99, "AN/ALQ-99" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_99_BAND_4, "AN/ALQ-99 Band 4" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_99_LBT, "AN/ALQ-99 LBT" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_100, "AN/ALQ-100" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_101, "AN/ALQ-101" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_119, "AN/ALQ-119" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_122, "AN/ALQ-122" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_126A, "AN/ALQ-126A" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_128, "AN/ALQ-128" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_126B, "AN/ALQ-126B" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_131, "AN/ALQ-131" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_131_BLK_II, "AN/ALQ-131 Blk II" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_135C_D, "AN/ALQ-135C/D" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_144A_V_3, "AN/ALQ-144A(V)3" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_153, "AN/ALQ-153" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_157JAMMER, "AN/ALQ-157Jammer" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_155, "AN/ALQ-155" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_156, "AN/ALQ-156" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_161_A, "AN/ALQ-161/A" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_161, "AN/ALQ-161" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_162, "AN/ALQ-162" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_164, "AN/ALQ-164" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_165, "AN/ALQ-165" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_187_JAMMER, "AN/ALQ-187 Jammer" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_167, "AN/ALQ-167" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_172_V_1, "AN/ALQ-172(V)1" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_172_V_2, "AN/ALQ-172(V)2" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_172_V_3, "AN/ALQ-172(V)3" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_176, "AN/ALQ-176" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_178, "AN/ALQ-178" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_184, "AN/ALQ-184" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_184_V_9, "AN/ALQ-184(V)9" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_188, "AN/ALQ-188" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_214, "AN/ALQ-214" },
     { DIS_PDU_EMITTERNAME_AN_ALR_56, "AN/ALR-56" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_221, "AN/ALQ-221" },
     { DIS_PDU_EMITTERNAME_AN_ALR_69, "AN/ALR-69" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_211_V_SIRFC, "AN/ALQ-211(V) SIRFC" },
     { DIS_PDU_EMITTERNAME_AN_ALT_16A, "AN/ALT-16A" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_173, "AN/ALQ-173" },
     { DIS_PDU_EMITTERNAME_AN_ALT_28, "AN/ALT-28" },
     { DIS_PDU_EMITTERNAME_AN_ALR_66B_JAMMER, "AN/ALR-66B Jammer" },
     { DIS_PDU_EMITTERNAME_AN_ALT_32A, "AN/ALT-32A" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_196, "AN/ALQ-196" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_249_V_1_NEXT_GENERATION_JAMMER_MID_BAND_NGJ_MB, "AN/ALQ-249(V)1 Next Generation Jammer-Mid Band (NGJ-MB)" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_240_V_1_JAMMER, "AN/ALQ-240(V)1 Jammer" },
     { DIS_PDU_EMITTERNAME_AN_APD_10, "AN/APD 10" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_249_V_2_NEXT_GENERATION_JAMMER_LOW_BAND_NGJ_LB, "AN/ALQ-249(V)2 Next Generation Jammer-Low Band (NGJ-LB)" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_213, "AN/ALQ-213" },
     { DIS_PDU_EMITTERNAME_ALQ_214A_V_4_5_JAMMER, "ALQ-214A(V)4/5 Jammer" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_218, "AN/ALQ-218" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_250_EPAWSS, "AN/ALQ-250 EPAWSS" },
     { DIS_PDU_EMITTERNAME_AN_APG_50, "AN/APG-50" },
     { DIS_PDU_EMITTERNAME_AN_APG_53, "AN/APG-53" },
     { DIS_PDU_EMITTERNAME_AN_APG_59, "AN/APG-59" },
     { DIS_PDU_EMITTERNAME_AN_APG_63, "AN/APG-63" },
     { DIS_PDU_EMITTERNAME_AN_APG_63_V_1, "AN/APG-63(V)1" },
     { DIS_PDU_EMITTERNAME_AN_APG_63_V_2, "AN/APG-63(V)2" },
     { DIS_PDU_EMITTERNAME_AN_APG_63_V_3, "AN/APG-63(V)3" },
     { DIS_PDU_EMITTERNAME_AN_APG_65, "AN/APG-65" },
     { DIS_PDU_EMITTERNAME_AN_APG_66, "AN/APG-66" },
     { DIS_PDU_EMITTERNAME_AN_APG_66_V, "AN/APG-66(V)" },
     { DIS_PDU_EMITTERNAME_AN_APG_66_V_2, "AN/APG-66(V)2" },
     { DIS_PDU_EMITTERNAME_AN_APG_67, "AN/APG-67" },
     { DIS_PDU_EMITTERNAME_AN_APG_68, "AN/APG-68" },
     { DIS_PDU_EMITTERNAME_AN_APG_68_V_9, "AN/APG-68(v)9" },
     { DIS_PDU_EMITTERNAME_AN_APG_70, "AN/APG-70" },
     { DIS_PDU_EMITTERNAME_AN_APG_71, "AN/APG-71" },
     { DIS_PDU_EMITTERNAME_AN_APG_73, "AN/APG-73" },
     { DIS_PDU_EMITTERNAME_AN_APG_77, "AN/APG-77" },
     { DIS_PDU_EMITTERNAME_AN_APG_78, "AN/APG-78" },
     { DIS_PDU_EMITTERNAME_AN_APG_79, "AN/APG-79" },
     { DIS_PDU_EMITTERNAME_AN_APG_80, "AN/APG-80" },
     { DIS_PDU_EMITTERNAME_AN_APG_81, "AN/APG-81" },
     { DIS_PDU_EMITTERNAME_AN_APG_82_V_1, "AN/APG-82(V)1" },
     { DIS_PDU_EMITTERNAME_AN_APG_83, "AN/APG-83" },
     { DIS_PDU_EMITTERNAME_AN_APG_502, "AN/APG-502" },
     { DIS_PDU_EMITTERNAME_AN_APN_1, "AN/APN-1" },
     { DIS_PDU_EMITTERNAME_AN_APN_22, "AN/APN-22" },
     { DIS_PDU_EMITTERNAME_AN_APN_59, "AN/APN-59" },
     { DIS_PDU_EMITTERNAME_AN_APN_69, "AN/APN-69" },
     { DIS_PDU_EMITTERNAME_AN_APN_81, "AN/APN-81" },
     { DIS_PDU_EMITTERNAME_AN_APN_102, "AN/APN-102" },
     { DIS_PDU_EMITTERNAME_AN_APN_117, "AN/APN-117" },
     { DIS_PDU_EMITTERNAME_AN_APN_118, "AN/APN-118" },
     { DIS_PDU_EMITTERNAME_AN_APN_122, "AN/APN-122" },
     { DIS_PDU_EMITTERNAME_AN_APN_130, "AN/APN-130" },
     { DIS_PDU_EMITTERNAME_AN_APN_131, "AN/APN-131" },
     { DIS_PDU_EMITTERNAME_AN_APN_133, "AN/APN-133" },
     { DIS_PDU_EMITTERNAME_AN_APN_134, "AN/APN-134" },
     { DIS_PDU_EMITTERNAME_AN_APN_141_V, "AN/APN-141(V)" },
     { DIS_PDU_EMITTERNAME_AN_APN_147, "AN/APN-147" },
     { DIS_PDU_EMITTERNAME_AN_APN_150, "AN/APN-150" },
     { DIS_PDU_EMITTERNAME_AN_APN_153, "AN/APN-153" },
     { DIS_PDU_EMITTERNAME_AN_APN_154, "AN/APN-154" },
     { DIS_PDU_EMITTERNAME_AN_APN_155, "AN/APN-155" },
     { DIS_PDU_EMITTERNAME_AN_APN_159, "AN/APN-159" },
     { DIS_PDU_EMITTERNAME_AN_APN_177, "AN/APN-177" },
     { DIS_PDU_EMITTERNAME_AN_APN_179, "AN/APN-179" },
     { DIS_PDU_EMITTERNAME_AN_APN_169, "AN/APN-169" },
     { DIS_PDU_EMITTERNAME_AN_APN_182, "AN/APN-182" },
     { DIS_PDU_EMITTERNAME_AN_APN_187, "AN/APN-187" },
     { DIS_PDU_EMITTERNAME_AN_APN_190, "AN/APN-190" },
     { DIS_PDU_EMITTERNAME_AN_APN_194, "AN/APN-194" },
     { DIS_PDU_EMITTERNAME_AN_APN_195, "AN/APN-195" },
     { DIS_PDU_EMITTERNAME_AN_APN_198, "AN/APN-198" },
     { DIS_PDU_EMITTERNAME_AN_APN_200, "AN/APN-200" },
     { DIS_PDU_EMITTERNAME_AN_APN_202, "AN/APN-202" },
     { DIS_PDU_EMITTERNAME_AN_APN_205, "AN/APN-205" },
     { DIS_PDU_EMITTERNAME_AN_APN_209, "AN/APN-209" },
     { DIS_PDU_EMITTERNAME_AN_APN_209D, "AN/APN-209D" },
     { DIS_PDU_EMITTERNAME_AN_APN_209A, "AN/APN-209A" },
     { DIS_PDU_EMITTERNAME_AN_APN_215, "AN/APN-215" },
     { DIS_PDU_EMITTERNAME_AN_APN_217, "AN/APN-217" },
     { DIS_PDU_EMITTERNAME_AN_APN_218, "AN/APN-218" },
     { DIS_PDU_EMITTERNAME_AN_APN_224, "AN/APN-224" },
     { DIS_PDU_EMITTERNAME_AN_APN_227, "AN/APN-227" },
     { DIS_PDU_EMITTERNAME_AN_APN_230, "AN/APN-230" },
     { DIS_PDU_EMITTERNAME_AN_APN_232, "AN/APN-232" },
     { DIS_PDU_EMITTERNAME_AN_APN_237A, "AN/APN-237A" },
     { DIS_PDU_EMITTERNAME_AN_APN_234, "AN/APN-234" },
     { DIS_PDU_EMITTERNAME_AN_APN_235, "AN/APN-235" },
     { DIS_PDU_EMITTERNAME_AN_APN_238, "AN/APN-238" },
     { DIS_PDU_EMITTERNAME_AN_APN_222, "AN/APN-222" },
     { DIS_PDU_EMITTERNAME_AN_APN_239, "AN/APN-239" },
     { DIS_PDU_EMITTERNAME_AN_APN_241, "AN/APN-241" },
     { DIS_PDU_EMITTERNAME_AN_APN_242, "AN/APN-242" },
     { DIS_PDU_EMITTERNAME_AN_APN_243, "AN/APN-243" },
     { DIS_PDU_EMITTERNAME_AN_APN_506, "AN/APN-506" },
     { DIS_PDU_EMITTERNAME_AN_APQ_72, "AN/APQ-72" },
     { DIS_PDU_EMITTERNAME_AN_APQ_99, "AN/APQ-99" },
     { DIS_PDU_EMITTERNAME_AN_APQ_100, "AN/APQ-100" },
     { DIS_PDU_EMITTERNAME_AN_APQ_102, "AN/APQ-102" },
     { DIS_PDU_EMITTERNAME_AN_APQ_107, "AN/APQ-107" },
     { DIS_PDU_EMITTERNAME_AN_APQ_109, "AN/APQ-109" },
     { DIS_PDU_EMITTERNAME_AN_APQ_113, "AN/APQ-113" },
     { DIS_PDU_EMITTERNAME_AN_APQ_120, "AN/APQ-120" },
     { DIS_PDU_EMITTERNAME_AN_APQ_122, "AN/APQ-122" },
     { DIS_PDU_EMITTERNAME_AN_APQ_126, "AN/APQ-126" },
     { DIS_PDU_EMITTERNAME_AN_APQ_128, "AN/APQ-128" },
     { DIS_PDU_EMITTERNAME_AN_APQ_129, "AN/APQ-129" },
     { DIS_PDU_EMITTERNAME_AN_APQ_148, "AN/APQ-148" },
     { DIS_PDU_EMITTERNAME_AN_APQ_150A, "AN/APQ-150A" },
     { DIS_PDU_EMITTERNAME_AN_APQ_153, "AN/APQ-153" },
     { DIS_PDU_EMITTERNAME_AN_APQ_155, "AN/APQ-155" },
     { DIS_PDU_EMITTERNAME_AN_APQ_159, "AN/APQ-159" },
     { DIS_PDU_EMITTERNAME_AN_APQ_164, "AN/APQ-164" },
     { DIS_PDU_EMITTERNAME_AN_APQ_166, "AN/APQ-166" },
     { DIS_PDU_EMITTERNAME_AN_APQ_170, "AN/APQ-170" },
     { DIS_PDU_EMITTERNAME_AN_APQ_174, "AN/APQ-174" },
     { DIS_PDU_EMITTERNAME_AN_APQ_180, "AN/APQ-180" },
     { DIS_PDU_EMITTERNAME_AN_APQ_181, "AN/APQ-181" },
     { DIS_PDU_EMITTERNAME_AN_APQ_186, "AN/APQ-186" },
     { DIS_PDU_EMITTERNAME_AN_APS_15J, "AN/APS-15J" },
     { DIS_PDU_EMITTERNAME_AN_APS_16_V_2, "AN/APS-16(V)2" },
     { DIS_PDU_EMITTERNAME_AN_APS_31, "AN/APS-31" },
     { DIS_PDU_EMITTERNAME_AN_APS_42, "AN/APS-42" },
     { DIS_PDU_EMITTERNAME_AN_APS_80, "AN/APS-80" },
     { DIS_PDU_EMITTERNAME_AN_APS_88, "AN/APS-88" },
     { DIS_PDU_EMITTERNAME_AN_APS_88A, "AN/APS-88A" },
     { DIS_PDU_EMITTERNAME_AN_APS_94, "AN/APS-94" },
     { DIS_PDU_EMITTERNAME_AN_APS_96, "AN/APS-96" },
     { DIS_PDU_EMITTERNAME_AN_APS_113, "AN/APS-113" },
     { DIS_PDU_EMITTERNAME_AN_APS_115, "AN/APS-115" },
     { DIS_PDU_EMITTERNAME_AN_APS_116, "AN/APS-116" },
     { DIS_PDU_EMITTERNAME_AN_APS_120, "AN/APS-120" },
     { DIS_PDU_EMITTERNAME_AN_APS_121, "AN/APS-121" },
     { DIS_PDU_EMITTERNAME_AN_APS_124, "AN/APS-124" },
     { DIS_PDU_EMITTERNAME_AN_APS_125, "AN/APS-125" },
     { DIS_PDU_EMITTERNAME_AN_APS_127, "AN/APS-127" },
     { DIS_PDU_EMITTERNAME_AN_APS_128, "AN/APS-128" },
     { DIS_PDU_EMITTERNAME_AN_APS_130, "AN/APS-130" },
     { DIS_PDU_EMITTERNAME_AN_APS_133, "AN/APS-133" },
     { DIS_PDU_EMITTERNAME_AN_APS_134, "AN/APS-134" },
     { DIS_PDU_EMITTERNAME_AN_APS_137, "AN/APS-137" },
     { DIS_PDU_EMITTERNAME_AN_APS_137_V_5, "AN/APS-137(V)5" },
     { DIS_PDU_EMITTERNAME_AN_APS_137B, "AN/APS-137B" },
     { DIS_PDU_EMITTERNAME_AN_APS_137B_V_5, "AN/APS-137B(V)5" },
     { DIS_PDU_EMITTERNAME_AN_APS_137D_V_5_ELTA, "AN/APS-137D(V)5 Elta" },
     { DIS_PDU_EMITTERNAME_AN_APS_138, "AN/APS-138" },
     { DIS_PDU_EMITTERNAME_AN_APS_139, "AN/APS-139" },
     { DIS_PDU_EMITTERNAME_AN_APS_143, "AN/APS-143" },
     { DIS_PDU_EMITTERNAME_AN_APS_143_V_1, "AN/APS-143 (V) 1" },
     { DIS_PDU_EMITTERNAME_AN_APS_143B, "AN/APS-143B" },
     { DIS_PDU_EMITTERNAME_AN_APS_143_V_3, "AN/APS-143(V)3" },
     { DIS_PDU_EMITTERNAME_AN_APS_143B_V_3, "AN/APS-143B(V)3" },
     { DIS_PDU_EMITTERNAME_AN_APS_153, "AN/APS-153" },
     { DIS_PDU_EMITTERNAME_AN_APS_154, "AN/APS-154" },
     { DIS_PDU_EMITTERNAME_AN_APS_150, "AN/APS-150" },
     { DIS_PDU_EMITTERNAME_AN_APS_145, "AN/APS-145" },
     { DIS_PDU_EMITTERNAME_AN_APS_147, "AN/APS-147" },
     { DIS_PDU_EMITTERNAME_AN_APS_149, "AN/APS-149" },
     { DIS_PDU_EMITTERNAME_AN_APS_503, "AN/APS-503" },
     { DIS_PDU_EMITTERNAME_AN_APS_504, "AN/APS-504" },
     { DIS_PDU_EMITTERNAME_AN_APS_705, "AN/APS-705" },
     { DIS_PDU_EMITTERNAME_AN_APW_22, "AN/APW-22" },
     { DIS_PDU_EMITTERNAME_AN_APW_23, "AN/APW-23" },
     { DIS_PDU_EMITTERNAME_AN_APX_6, "AN/APX-6" },
     { DIS_PDU_EMITTERNAME_AN_APX_7, "AN/APX-7" },
     { DIS_PDU_EMITTERNAME_AN_APX_39, "AN/APX-39" },
     { DIS_PDU_EMITTERNAME_AN_APX_64_V, "AN/APX-64(V)" },
     { DIS_PDU_EMITTERNAME_AN_APX_72, "AN/APX-72" },
     { DIS_PDU_EMITTERNAME_AN_APX_76, "AN/APX-76" },
     { DIS_PDU_EMITTERNAME_AN_APX_78, "AN/APX-78" },
     { DIS_PDU_EMITTERNAME_AN_APX_100, "AN/APX-100" },
     { DIS_PDU_EMITTERNAME_AN_APX_101, "AN/APX-101" },
     { DIS_PDU_EMITTERNAME_AN_APX_113_AIFF, "AN/APX-113 AIFF" },
     { DIS_PDU_EMITTERNAME_AN_APY_1, "AN/APY-1" },
     { DIS_PDU_EMITTERNAME_AN_APY_2, "AN/APY-2" },
     { DIS_PDU_EMITTERNAME_AN_APY_3, "AN/APY-3" },
     { DIS_PDU_EMITTERNAME_AN_APY_7, "AN/APY-7" },
     { DIS_PDU_EMITTERNAME_AN_APY_8_LYNX_TM, "AN/APY-8 LYNX(tm)" },
     { DIS_PDU_EMITTERNAME_AN_APY_9, "AN/APY-9" },
     { DIS_PDU_EMITTERNAME_AN_APY_10, "AN/APY-10" },
     { DIS_PDU_EMITTERNAME_AN_ARN_21, "AN/ARN-21" },
     { DIS_PDU_EMITTERNAME_AN_ARN_52, "AN/ARN-52" },
     { DIS_PDU_EMITTERNAME_AN_ARN_84, "AN/ARN-84" },
     { DIS_PDU_EMITTERNAME_AN_ARN_118, "AN/ARN-118" },
     { DIS_PDU_EMITTERNAME_AN_ARN_153_V, "AN/ARN-153(V)" },
     { DIS_PDU_EMITTERNAME_AN_ARN_153, "AN/ARN-153" },
     { DIS_PDU_EMITTERNAME_AN_ARW_73, "AN/ARW 73" },
     { DIS_PDU_EMITTERNAME_AN_ASB_1, "AN/ASB 1" },
     { DIS_PDU_EMITTERNAME_AN_ASG_21, "AN/ASG 21" },
     { DIS_PDU_EMITTERNAME_AN_ASN_137, "AN/ASN-137" },
     { DIS_PDU_EMITTERNAME_AN_ASN_128, "AN/ASN-128" },
     { DIS_PDU_EMITTERNAME_AN_ASQ_108, "AN/ASQ-108" },
     { DIS_PDU_EMITTERNAME_AN_ASQ_239, "AN/ASQ-239" },
     { DIS_PDU_EMITTERNAME_AN_AST_502, "AN/AST-502" },
     { DIS_PDU_EMITTERNAME_AN_AVQ_55, "AN/AVQ-55" },
     { DIS_PDU_EMITTERNAME_AN_AWG_9, "AN/AWG 9" },
     { DIS_PDU_EMITTERNAME_AN_BRN_1, "AN/BRN-1" },
     { DIS_PDU_EMITTERNAME_AN_BPS_5, "AN/BPS-5" },
     { DIS_PDU_EMITTERNAME_AN_BPS_9, "AN/BPS-9" },
     { DIS_PDU_EMITTERNAME_AN_BPS_15, "AN/BPS 15" },
     { DIS_PDU_EMITTERNAME_AN_BPS_15_H, "AN/BPS-15 H" },
     { DIS_PDU_EMITTERNAME_AN_BPS_15J, "AN/BPS-15J" },
     { DIS_PDU_EMITTERNAME_AN_BPS_16, "AN/BPS-16" },
     { DIS_PDU_EMITTERNAME_AN_BPS_16_V_2, "AN/BPS-16(V)2" },
     { DIS_PDU_EMITTERNAME_AN_CPN_4, "AN/CPN-4" },
     { DIS_PDU_EMITTERNAME_AN_CPN_18, "AN/CPN-18" },
     { DIS_PDU_EMITTERNAME_AN_CRM_30, "AN/CRM-30" },
     { DIS_PDU_EMITTERNAME_AN_DPW_23, "AN/DPW-23" },
     { DIS_PDU_EMITTERNAME_AN_DSQ_26_PHOENIX_MH, "AN/DSQ 26 Phoenix MH" },
     { DIS_PDU_EMITTERNAME_AN_DSQ_28_HARPOON_MH, "AN/DSQ 28 Harpoon MH" },
     { DIS_PDU_EMITTERNAME_AN_FPN_1, "AN/FPN-1" },
     { DIS_PDU_EMITTERNAME_AN_FPN_28, "AN/FPN-28" },
     { DIS_PDU_EMITTERNAME_AN_FPN_33, "AN/FPN-33" },
     { DIS_PDU_EMITTERNAME_AN_FPN_40, "AN/FPN-40" },
     { DIS_PDU_EMITTERNAME_AN_FPN_62, "AN/FPN-62" },
     { DIS_PDU_EMITTERNAME_AN_FPN_66, "AN/FPN-66" },
     { DIS_PDU_EMITTERNAME_AN_FPS_8, "AN/FPS-8" },
     { DIS_PDU_EMITTERNAME_AN_FPN_67, "AN/FPN-67" },
     { DIS_PDU_EMITTERNAME_AN_FPS_16, "AN/FPS-16" },
     { DIS_PDU_EMITTERNAME_AN_FPS_5, "AN/FPS-5" },
     { DIS_PDU_EMITTERNAME_AN_FPS_18, "AN/FPS-18" },
     { DIS_PDU_EMITTERNAME_AN_FPS_89, "AN/FPS-89" },
     { DIS_PDU_EMITTERNAME_AN_FPS_49, "AN/FPS-49" },
     { DIS_PDU_EMITTERNAME_AN_FPS_117, "AN/FPS-117" },
     { DIS_PDU_EMITTERNAME_AN_FPS_85, "AN/FPS-85" },
     { DIS_PDU_EMITTERNAME_AN_FPS_88, "AN/FPS-88" },
     { DIS_PDU_EMITTERNAME_AN_FPS_113, "AN/FPS-113" },
     { DIS_PDU_EMITTERNAME_AN_FPS_115, "AN/FPS-115" },
     { DIS_PDU_EMITTERNAME_AN_FPS_20R, "AN/FPS-20R" },
     { DIS_PDU_EMITTERNAME_AN_FPS_132, "AN/FPS-132" },
     { DIS_PDU_EMITTERNAME_AN_FPS_77, "AN/FPS-77" },
     { DIS_PDU_EMITTERNAME_AN_FPS_41, "AN/FPS-41" },
     { DIS_PDU_EMITTERNAME_AN_FPS_100A, "AN/FPS-100A" },
     { DIS_PDU_EMITTERNAME_AN_FPS_103, "AN/FPS-103" },
     { DIS_PDU_EMITTERNAME_AN_FPS_108, "AN/FPS-108" },
     { DIS_PDU_EMITTERNAME_AN_GPN_12, "AN/GPN-12" },
     { DIS_PDU_EMITTERNAME_AN_FPS_124_V, "AN/FPS-124(V)" },
     { DIS_PDU_EMITTERNAME_AN_FPS_129, "AN/FPS-129" },
     { DIS_PDU_EMITTERNAME_AN_GPX_6, "AN/GPX-6" },
     { DIS_PDU_EMITTERNAME_AN_GPX_8, "AN/GPX 8" },
     { DIS_PDU_EMITTERNAME_AN_GRN_12, "AN/GRN-12" },
     { DIS_PDU_EMITTERNAME_AN_MPN_14K, "AN/MPN-14K" },
     { DIS_PDU_EMITTERNAME_AN_MPN_14, "AN/MPN-14" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_10, "AN/MPQ-10" },
     { DIS_PDU_EMITTERNAME_AN_MPN_17, "AN/MPN-17" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_33_39_46_57_61_HPIR_ILL, "AN/MPQ-33/39/46/57/61 (HPIR) ILL" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_34_48_55_62_CWAR_TA, "AN/MPQ-34/48/55/62 (CWAR) TA" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_49, "AN/MPQ-49" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_35_50_PAR_TA, "AN/MPQ-35/50 (PAR) TA" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_50C, "AN/MPQ-50C" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_37_51_ROR_TT, "AN/MPQ-37/51 (ROR) TT" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_43, "AN/MPQ-43" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_50, "AN/MPQ-50" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_53, "AN/MPQ-53" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_63, "AN/MPQ-63" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_64, "AN/MPQ-64" },
     { DIS_PDU_EMITTERNAME_AN_SLQ_32, "AN/SLQ-32" },
     { DIS_PDU_EMITTERNAME_AN_MPQ_65, "AN/MPQ-65" },
     { DIS_PDU_EMITTERNAME_AN_SLQ_32_V_4, "AN/SLQ-32(V)4" },
     { DIS_PDU_EMITTERNAME_AN_SLQ_32A, "AN/SLQ-32A" },
     { DIS_PDU_EMITTERNAME_AN_SPG_34, "AN/SPG-34" },
     { DIS_PDU_EMITTERNAME_AN_MSQ_104, "AN/MSQ-104" },
     { DIS_PDU_EMITTERNAME_AN_MPS_36, "AN/MPS-36" },
     { DIS_PDU_EMITTERNAME_AN_SLQ_503, "AN/SLQ-503" },
     { DIS_PDU_EMITTERNAME_AN_SPG_48_MK_25_MOD_3, "AN/SPG-48/MK 25 MOD 3" },
     { DIS_PDU_EMITTERNAME_AN_SPG_50, "AN/SPG-50" },
     { DIS_PDU_EMITTERNAME_AN_SPG_51, "AN/SPG-51" },
     { DIS_PDU_EMITTERNAME_AN_PPQ_2, "AN/PPQ-2" },
     { DIS_PDU_EMITTERNAME_AN_PPS_15, "AN/PPS-15" },
     { DIS_PDU_EMITTERNAME_AN_PPS_5, "AN/PPS-5" },
     { DIS_PDU_EMITTERNAME_AN_PPS_5D, "AN/PPS-5D" },
     { DIS_PDU_EMITTERNAME_AN_SPG_51_CWI_TI, "AN/SPG-51 CWI TI" },
     { DIS_PDU_EMITTERNAME_AN_SPG_51_FC, "AN/SPG-51 FC" },
     { DIS_PDU_EMITTERNAME_AN_SPG_51C_D, "AN/SPG-51C/D" },
     { DIS_PDU_EMITTERNAME_AN_SPG_52, "AN/SPG-52" },
     { DIS_PDU_EMITTERNAME_AN_SPG_53, "AN/SPG-53" },
     { DIS_PDU_EMITTERNAME_AN_SPG_55B, "AN/SPG-55B" },
     { DIS_PDU_EMITTERNAME_AN_SPG_60, "AN/SPG-60" },
     { DIS_PDU_EMITTERNAME_AN_SPG_62, "AN/SPG-62" },
     { DIS_PDU_EMITTERNAME_AN_SPG_503, "AN/SPG-503" },
     { DIS_PDU_EMITTERNAME_AN_SPN_4, "AN/SPN-4" },
     { DIS_PDU_EMITTERNAME_AN_SPN_11, "AN/SPN-11" },
     { DIS_PDU_EMITTERNAME_AN_SPN_35, "AN/SPN-35" },
     { DIS_PDU_EMITTERNAME_AN_SPN_41, "AN/SPN-41" },
     { DIS_PDU_EMITTERNAME_AN_SPN_43, "AN/SPN-43" },
     { DIS_PDU_EMITTERNAME_AN_SPN_43A, "AN/SPN-43A" },
     { DIS_PDU_EMITTERNAME_AN_SPN_43C, "AN/SPN-43C" },
     { DIS_PDU_EMITTERNAME_AN_SPN_46, "AN/SPN-46" },
     { DIS_PDU_EMITTERNAME_AN_SPQ_2, "AN/SPQ-2" },
     { DIS_PDU_EMITTERNAME_AN_SPQ_5A, "AN/SPQ-5A" },
     { DIS_PDU_EMITTERNAME_AN_SPQ_9A, "AN/SPQ-9A" },
     { DIS_PDU_EMITTERNAME_AN_SPQ_9B, "AN/SPQ-9B" },
     { DIS_PDU_EMITTERNAME_AN_SPQ_34, "AN/SPQ-34" },
     { DIS_PDU_EMITTERNAME_AN_SPS_4, "AN/SPS-4" },
     { DIS_PDU_EMITTERNAME_AN_SPS_5, "AN/SPS-5" },
     { DIS_PDU_EMITTERNAME_AN_SPS_5C, "AN/SPS-5C" },
     { DIS_PDU_EMITTERNAME_AN_SPS_6, "AN/SPS-6" },
     { DIS_PDU_EMITTERNAME_AN_SPS_10, "AN/SPS-10" },
     { DIS_PDU_EMITTERNAME_AN_SPS_21, "AN/SPS-21" },
     { DIS_PDU_EMITTERNAME_AN_SPS_28, "AN/SPS-28" },
     { DIS_PDU_EMITTERNAME_AN_SPS_37, "AN/SPS-37" },
     { DIS_PDU_EMITTERNAME_AN_SPS_39A, "AN/SPS-39A" },
     { DIS_PDU_EMITTERNAME_AN_SPS_40, "AN/SPS-40" },
     { DIS_PDU_EMITTERNAME_AN_SPS_41, "AN/SPS-41" },
     { DIS_PDU_EMITTERNAME_AN_SPS_48, "AN/SPS-48" },
     { DIS_PDU_EMITTERNAME_AN_SPS_48C, "AN/SPS-48C" },
     { DIS_PDU_EMITTERNAME_AN_SPS_48E, "AN/SPS-48E" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49, "AN/SPS-49" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_1, "AN/SPS-49(V)1" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_2, "AN/SPS-49(V)2" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_3, "AN/SPS-49(V)3" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_4, "AN/SPS-49(V)4" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_5, "AN/SPS-49(V)5" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_6, "AN/SPS-49(V)6" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_7, "AN/SPS-49(V)7" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49_V_8, "AN/SPS-49(V)8" },
     { DIS_PDU_EMITTERNAME_AN_SPS_49A_V_1, "AN/SPS-49A(V)1" },
     { DIS_PDU_EMITTERNAME_AN_SPS_52, "AN/SPS-52" },
     { DIS_PDU_EMITTERNAME_AN_SPS_53, "AN/SPS-53" },
     { DIS_PDU_EMITTERNAME_AN_SPS_55, "AN/SPS-55" },
     { DIS_PDU_EMITTERNAME_AN_SPS_52C, "AN/SPS-52C" },
     { DIS_PDU_EMITTERNAME_AN_SPS_55_CS, "AN/SPS-55 CS" },
     { DIS_PDU_EMITTERNAME_AN_SPS_55_SS, "AN/SPS-55 SS" },
     { DIS_PDU_EMITTERNAME_AN_SPS_58, "AN/SPS-58" },
     { DIS_PDU_EMITTERNAME_AN_SPS_58C, "AN/SPS-58C" },
     { DIS_PDU_EMITTERNAME_AN_SPS_59, "AN/SPS-59" },
     { DIS_PDU_EMITTERNAME_AN_SPS_64, "AN/SPS-64" },
     { DIS_PDU_EMITTERNAME_AN_SPS_64_V_9, "AN/SPS-64(V)9" },
     { DIS_PDU_EMITTERNAME_SPS64_V_12, "SPS64(V)12" },
     { DIS_PDU_EMITTERNAME_AN_SPS_65, "AN/SPS-65" },
     { DIS_PDU_EMITTERNAME_AN_SPS_66, "AN/SPS-66" },
     { DIS_PDU_EMITTERNAME_AN_SPS_67, "AN/SPS-67" },
     { DIS_PDU_EMITTERNAME_AN_SPS_73_I, "AN/SPS-73(I)" },
     { DIS_PDU_EMITTERNAME_AN_SPS_69, "AN/SPS-69" },
     { DIS_PDU_EMITTERNAME_AN_SPS_73, "AN/SPS-73" },
     { DIS_PDU_EMITTERNAME_AN_SPS_74, "AN/SPS-74" },
     { DIS_PDU_EMITTERNAME_AN_SPS_88, "AN/SPS-88" },
     { DIS_PDU_EMITTERNAME_AN_SPS_501, "AN/SPS-501" },
     { DIS_PDU_EMITTERNAME_AN_SPS_505, "AN/SPS-505" },
     { DIS_PDU_EMITTERNAME_AN_SPY_1, "AN/SPY-1" },
     { DIS_PDU_EMITTERNAME_AN_SPY_1A, "AN/SPY-1A" },
     { DIS_PDU_EMITTERNAME_AN_SPY_1B, "AN/SPY-1B" },
     { DIS_PDU_EMITTERNAME_AN_SPY_1B_V, "AN/SPY-1B(V)" },
     { DIS_PDU_EMITTERNAME_AN_SPY_1D, "AN/SPY-1D" },
     { DIS_PDU_EMITTERNAME_AN_SPY_1D_V, "AN/SPY-1D(V)" },
     { DIS_PDU_EMITTERNAME_AN_SPY_1F, "AN/SPY-1F" },
     { DIS_PDU_EMITTERNAME_AN_SPY_3, "AN/SPY-3" },
     { DIS_PDU_EMITTERNAME_AN_TPN_12, "AN/TPN-12" },
     { DIS_PDU_EMITTERNAME_AN_SPY_4, "AN/SPY-4" },
     { DIS_PDU_EMITTERNAME_AN_TLQ_32_ARM_DECOY, "AN/TLQ-32 ARM Decoy" },
     { DIS_PDU_EMITTERNAME_AN_TPN_17, "AN/TPN-17" },
     { DIS_PDU_EMITTERNAME_AN_TPN_8, "AN/TPN-8" },
     { DIS_PDU_EMITTERNAME_AN_TPN_22, "AN/TPN-22" },
     { DIS_PDU_EMITTERNAME_AN_TLQ_17A, "AN/TLQ-17A" },
     { DIS_PDU_EMITTERNAME_AN_TMS_1, "AN/TMS-1" },
     { DIS_PDU_EMITTERNAME_AN_TPN_24, "AN/TPN-24" },
     { DIS_PDU_EMITTERNAME_AN_TPN_25, "AN/TPN-25" },
     { DIS_PDU_EMITTERNAME_AN_TMS_2, "AN/TMS-2" },
     { DIS_PDU_EMITTERNAME_AN_TPN_19, "AN/TPN-19" },
     { DIS_PDU_EMITTERNAME_AN_TPN_31, "AN/TPN-31" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_18, "AN/TPQ-18" },
     { DIS_PDU_EMITTERNAME_AN_SPY_6_V, "AN/SPY-6(V)" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_36, "AN/TPQ-36" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_37, "AN/TPQ-37" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_38_V8, "AN/TPQ-38 (V8)" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_39_V, "AN/TPQ-39(V)" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_47, "AN/TPQ-47" },
     { DIS_PDU_EMITTERNAME_AN_TPS_43, "AN/TPS-43" },
     { DIS_PDU_EMITTERNAME_AN_TPS_43E, "AN/TPS-43E" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_48, "AN/TPQ-48" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_49, "AN/TPQ-49" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_46A, "AN/TPQ-46A" },
     { DIS_PDU_EMITTERNAME_AN_TPS_34, "AN/TPS-34" },
     { DIS_PDU_EMITTERNAME_AN_TPS_59, "AN/TPS-59" },
     { DIS_PDU_EMITTERNAME_AN_TPS_44, "AN/TPS-44" },
     { DIS_PDU_EMITTERNAME_AN_TPQ_50, "AN/TPQ-50" },
     { DIS_PDU_EMITTERNAME_AN_TPS_63, "AN/TPS-63" },
     { DIS_PDU_EMITTERNAME_AN_TPS_65, "AN/TPS-65" },
     { DIS_PDU_EMITTERNAME_AN_TPS_70_V_1, "AN/TPS-70 (V) 1" },
     { DIS_PDU_EMITTERNAME_AN_TPS_63SS, "AN/TPS-63SS" },
     { DIS_PDU_EMITTERNAME_AN_TPS_73, "AN/TPS-73" },
     { DIS_PDU_EMITTERNAME_AN_TPS_75, "AN/TPS-75" },
     { DIS_PDU_EMITTERNAME_AN_TPS_77, "AN/TPS-77" },
     { DIS_PDU_EMITTERNAME_AN_TPS_78, "AN/TPS-78" },
     { DIS_PDU_EMITTERNAME_AN_TPS_79, "AN/TPS-79" },
     { DIS_PDU_EMITTERNAME_AN_TPS_703, "AN/TPS-703" },
     { DIS_PDU_EMITTERNAME_AN_TPX_46_V_7, "AN/TPX-46(V)7" },
     { DIS_PDU_EMITTERNAME_AN_TPS_80, "AN/TPS-80" },
     { DIS_PDU_EMITTERNAME_AN_TPY_2, "AN/TPY-2" },
     { DIS_PDU_EMITTERNAME_AN_TSQ_288, "AN/TSQ-288" },
     { DIS_PDU_EMITTERNAME_AN_ULQ_6A, "AN/ULQ-6A" },
     { DIS_PDU_EMITTERNAME_AN_ULQ_19, "AN/ULQ-19" },
     { DIS_PDU_EMITTERNAME_AN_ULQ_21, "AN/ULQ-21" },
     { DIS_PDU_EMITTERNAME_AN_UPN_25, "AN/UPN 25" },
     { DIS_PDU_EMITTERNAME_AN_UPS_1, "AN/UPS 1" },
     { DIS_PDU_EMITTERNAME_AN_UPS_2, "AN/UPS-2" },
     { DIS_PDU_EMITTERNAME_AN_UPS_3, "AN/UPS-3" },
     { DIS_PDU_EMITTERNAME_AN_UPX_1, "AN/UPX 1" },
     { DIS_PDU_EMITTERNAME_AN_UPX_5, "AN/UPX 5" },
     { DIS_PDU_EMITTERNAME_AN_UPX_11, "AN/UPX 11" },
     { DIS_PDU_EMITTERNAME_AN_UPX_12, "AN/UPX 12" },
     { DIS_PDU_EMITTERNAME_AN_UPX_17, "AN/UPX 17" },
     { DIS_PDU_EMITTERNAME_AN_UPX_23, "AN/UPX 23" },
     { DIS_PDU_EMITTERNAME_AN_USQ_113_V_3, "AN/USQ-113(V)3" },
     { DIS_PDU_EMITTERNAME_AN_VPS_2, "AN/VPS 2" },
     { DIS_PDU_EMITTERNAME_AN_PLM_3, "AN/PLM-3" },
     { DIS_PDU_EMITTERNAME_AN_PLM_3A, "AN/PLM-3A" },
     { DIS_PDU_EMITTERNAME_AN_PLM_4, "AN/PLM-4" },
     { DIS_PDU_EMITTERNAME_AN_ZPY1, "AN/ZPY1" },
     { DIS_PDU_EMITTERNAME_AN_ZPY_2_MP_RTIP, "AN/ZPY-2 MP-RTIP" },
     { DIS_PDU_EMITTERNAME_AN_ZPY_3, "AN/ZPY-3" },
     { DIS_PDU_EMITTERNAME_AN_ZPY_8, "AN/ZPY-8" },
     { DIS_PDU_EMITTERNAME_AORL_1AS, "AORL-1AS" },
     { DIS_PDU_EMITTERNAME_AORL_85K_TK_MTA, "AORL-85K/TK/MTA" },
     { DIS_PDU_EMITTERNAME_APAR, "APAR" },
     { DIS_PDU_EMITTERNAME_APARNA, "Aparna" },
     { DIS_PDU_EMITTERNAME_APECS_II, "APECS II" },
     { DIS_PDU_EMITTERNAME_APELCO_AD_7_7, " Apelco AD 7 7" },
     { DIS_PDU_EMITTERNAME_APG_71, "APG 71" },
     { DIS_PDU_EMITTERNAME_APN_148, "APN 148" },
     { DIS_PDU_EMITTERNAME_APN_227, "APN 227" },
     { DIS_PDU_EMITTERNAME_APS_504_V3, "APS 504 V3" },
     { DIS_PDU_EMITTERNAME_AQUITAINE_II, "AQUITAINE II" },
     { DIS_PDU_EMITTERNAME_AR_1, "AR-1" },
     { DIS_PDU_EMITTERNAME_AR_3D, "AR 3D" },
     { DIS_PDU_EMITTERNAME_PLESSEY_AR_5, " Plessey AR-5" },
     { DIS_PDU_EMITTERNAME_AR_15, " AR-15" },
     { DIS_PDU_EMITTERNAME_AR_15_2, "AR-15/2" },
     { DIS_PDU_EMITTERNAME_AR_320, "AR 320" },
     { DIS_PDU_EMITTERNAME_AR_325, "AR-325" },
     { DIS_PDU_EMITTERNAME_AR_327, "AR 327" },
     { DIS_PDU_EMITTERNAME_ARBALET_52, "Arbalet-52" },
     { DIS_PDU_EMITTERNAME_ARBB_31, "ARBB-31" },
     { DIS_PDU_EMITTERNAME_ARBB_33, " ARBB 33" },
     { DIS_PDU_EMITTERNAME_ARIES, " ARIES" },
     { DIS_PDU_EMITTERNAME_ARIES_NAV, "Aries-Nav" },
     { DIS_PDU_EMITTERNAME_ARIES_CS, "Aries-CS" },
     { DIS_PDU_EMITTERNAME_ARGS_14E, "ARGS-14E" },
     { DIS_PDU_EMITTERNAME_ARGS_31, "ARGS 31" },
     { DIS_PDU_EMITTERNAME_ARGUS, "ARGUS" },
     { DIS_PDU_EMITTERNAME_AR_M31, "AR M31" },
     { DIS_PDU_EMITTERNAME_ARECIBO, "ARECIBO" },
     { DIS_PDU_EMITTERNAME_ARED, "ARED" },
     { DIS_PDU_EMITTERNAME_ARI_5954, "ARI 5954" },
     { DIS_PDU_EMITTERNAME_ARI_5955, "ARI 5955" },
     { DIS_PDU_EMITTERNAME_ARI_5979, "ARI 5979" },
     { DIS_PDU_EMITTERNAME_ARGSN_31, "ARGSN-31" },
     { DIS_PDU_EMITTERNAME_ARGOS_10, "ARGOS-10" },
     { DIS_PDU_EMITTERNAME_ARGOS_800, "ARGOS-800" },
     { DIS_PDU_EMITTERNAME_ARI_5983, "ARI 5983" },
     { DIS_PDU_EMITTERNAME_ARI_5991, "ARI 5991" },
     { DIS_PDU_EMITTERNAME_ARI_5995, "ARI 5995" },
     { DIS_PDU_EMITTERNAME_ARINC_564_BNDX_KING_RDR_1E, " ARINC 564 BNDX/KING RDR 1E" },
     { DIS_PDU_EMITTERNAME_ARINC_700_BNDX_KING_RDR_1E, " ARINC 700 BNDX/KING RDR 1E" },
     { DIS_PDU_EMITTERNAME_ARK_1, "ARK-1" },
     { DIS_PDU_EMITTERNAME_ARL_M_MTI, " ARL-M MTI" },
     { DIS_PDU_EMITTERNAME_ARMOR, "ARMOR" },
     { DIS_PDU_EMITTERNAME_ARSR_3, "ARSR-3" },
     { DIS_PDU_EMITTERNAME_ARS_400, "ARS-400" },
     { DIS_PDU_EMITTERNAME_ARSR_1, "ARSR-1" },
     { DIS_PDU_EMITTERNAME_ARSR_4, "ARSR-4" },
     { DIS_PDU_EMITTERNAME_ARSR_18, "ARSR-18" },
     { DIS_PDU_EMITTERNAME_ARTHUR, " ARTHUR" },
     { DIS_PDU_EMITTERNAME_ARTHUR_MOD_B, "ARTHUR MOD B" },
     { DIS_PDU_EMITTERNAME_ARTHUR_MOD_C, "ARTHUR MOD C" },
     { DIS_PDU_EMITTERNAME_ARTISAN_3D, "ARTISAN 3D" },
     { DIS_PDU_EMITTERNAME_AS_2_KIPPER, " AS 2 Kipper" },
     { DIS_PDU_EMITTERNAME_AS_2_KIPPER_MH, " AS 2 Kipper MH" },
     { DIS_PDU_EMITTERNAME_AS_3_YJ_83K_MMW_MH, "AS 3 YJ-83K mmW MH" },
     { DIS_PDU_EMITTERNAME_AS_34_KORMORAN_SEEKER, "AS.34 Kormoran Seeker" },
     { DIS_PDU_EMITTERNAME_AS_4_KITCHEN, " AS 4 Kitchen" },
     { DIS_PDU_EMITTERNAME_AS_4_KITCHEN_MH, " AS 4 Kitchen MH" },
     { DIS_PDU_EMITTERNAME_AS_5_KELT_MH, " AS 5 Kelt MH" },
     { DIS_PDU_EMITTERNAME_AS_6_KINGFISH_MH, " AS 6 Kingfish MH" },
     { DIS_PDU_EMITTERNAME_AS_7_KERRY, " AS 7 Kerry" },
     { DIS_PDU_EMITTERNAME_AS_7_KERRY_MG, " AS 7 Kerry MG" },
     { DIS_PDU_EMITTERNAME_AS_15_KENT_ALTIMETER, " AS 15 KENT altimeter" },
     { DIS_PDU_EMITTERNAME_AS_17A_KRYPTON_SEEKER, " AS-17A Krypton Seeker" },
     { DIS_PDU_EMITTERNAME_AS_17B_KRYPTON_SEEKER, " AS-17B Krypton Seeker" },
     { DIS_PDU_EMITTERNAME_AS901, " AS901" },
     { DIS_PDU_EMITTERNAME_AS901A, "AS901A" },
     { DIS_PDU_EMITTERNAME_ASARS2, "ASARS2" },
     { DIS_PDU_EMITTERNAME_ASDE_KDD, "ASDE-KDD" },
     { DIS_PDU_EMITTERNAME_ASLESHA, "ASLESHA" },
     { DIS_PDU_EMITTERNAME_A_SMGCS, "A-SMGCS" },
     { DIS_PDU_EMITTERNAME_ASMI_18X, "ASMI-18X" },
     { DIS_PDU_EMITTERNAME_ASPIDE_AAM_SAM_ILL, "Aspide AAM/SAM ILL" },
     { DIS_PDU_EMITTERNAME_ASMI_3, "ASMI-3" },
     { DIS_PDU_EMITTERNAME_ASELSAN_MAR, "Aselsan MAR" },
     { DIS_PDU_EMITTERNAME_ASR_2000, "ASR-2000" },
     { DIS_PDU_EMITTERNAME_ASR_4, "ASR-4" },
     { DIS_PDU_EMITTERNAME_ASR_4D, "ASR-4D" },
     { DIS_PDU_EMITTERNAME_ASR_O, "ASR O" },
     { DIS_PDU_EMITTERNAME_ASR_12, "ASR-12" },
     { DIS_PDU_EMITTERNAME_ASR_22AL, "ASR-22AL" },
     { DIS_PDU_EMITTERNAME_ASR_3, "ASR-3" },
     { DIS_PDU_EMITTERNAME_ASR_5, "ASR-5" },
     { DIS_PDU_EMITTERNAME_ASR_7, "ASR-7" },
     { DIS_PDU_EMITTERNAME_ASR_8, "ASR-8" },
     { DIS_PDU_EMITTERNAME_ASR_9, "ASR-9" },
     { DIS_PDU_EMITTERNAME_ASR_9000, "ASR-9000" },
     { DIS_PDU_EMITTERNAME_ASTI, "ASTI" },
     { DIS_PDU_EMITTERNAME_ASR_11_DASR, "ASR-11/DASR" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_ASR_10SS, " Raytheon ASR-10SS" },
     { DIS_PDU_EMITTERNAME_ASR_23SS, "ASR-23SS" },
     { DIS_PDU_EMITTERNAME_ARABEL, "Arabel" },
     { DIS_PDU_EMITTERNAME_ASTRE, " ASTRE" },
     { DIS_PDU_EMITTERNAME_AT_2_SWATTER_MG, " AT 2 Swatter MG" },
     { DIS_PDU_EMITTERNAME_9K114_SHTURM_MG_AT_6_SPIRAL_MG, "9K114 Shturm MG AT-6 Spiral MG" },
     { DIS_PDU_EMITTERNAME_ASTOR, "ASTOR" },
     { DIS_PDU_EMITTERNAME_ASTRA_RCI, "ASTRA RCI" },
     { DIS_PDU_EMITTERNAME_ATCR_22, "ATCR-22" },
     { DIS_PDU_EMITTERNAME_ATCR_22_M, "ATCR-22 M" },
     { DIS_PDU_EMITTERNAME_ATCR_2T, "ATCR-2T" },
     { DIS_PDU_EMITTERNAME_ATCR_33, "ATCR-33" },
     { DIS_PDU_EMITTERNAME_ATCR_33_K_M, "ATCR 33 K/M" },
     { DIS_PDU_EMITTERNAME_ATCR_33S, "ATCR-33S" },
     { DIS_PDU_EMITTERNAME_ATCR_3T, "ATCR-3T" },
     { DIS_PDU_EMITTERNAME_ATCR_44, "ATCR-44" },
     { DIS_PDU_EMITTERNAME_ATCR_44_K, "ATCR-44 K" },
     { DIS_PDU_EMITTERNAME_ARGOS_73, " Argos 73" },
     { DIS_PDU_EMITTERNAME_ATCR_44_M_S, "ATCR-44 M/S" },
     { DIS_PDU_EMITTERNAME_ATCR_4T, "ATCR-4T" },
     { DIS_PDU_EMITTERNAME_ATLAS_ELEKTRONK_TRS_N, " Atlas Elektronk TRS N" },
     { DIS_PDU_EMITTERNAME_ATLAS_8600X, "ATLAS-8600X" },
     { DIS_PDU_EMITTERNAME_ATLAS_9600M, "Atlas-9600M" },
     { DIS_PDU_EMITTERNAME_ATLAS_9600X, "ATLAS-9600X" },
     { DIS_PDU_EMITTERNAME_ATLAS_9600S, "ATLAS-9600S" },
     { DIS_PDU_EMITTERNAME_ATLAS_9740_VTS, "ATLAS-9740 VTS" },
     { DIS_PDU_EMITTERNAME_ATLASS, "ATLASS" },
     { DIS_PDU_EMITTERNAME_ATR_500C, "ATR-500C" },
     { DIS_PDU_EMITTERNAME_AVG_65, "AVG 65" },
     { DIS_PDU_EMITTERNAME_AVH_7, "AVH 7" },
     { DIS_PDU_EMITTERNAME_AVIA_CM, "AVIA CM" },
     { DIS_PDU_EMITTERNAME_AVIA_D, "AVIA D" },
     { DIS_PDU_EMITTERNAME_AVIACONVERSIA, " Aviaconversia" },
     { DIS_PDU_EMITTERNAME_AVIACONVERSIA_II, " Aviaconversia II" },
     { DIS_PDU_EMITTERNAME_AVIACONVERSIA_III, " Aviaconversia III" },
     { DIS_PDU_EMITTERNAME_AVQ_20, "AVQ 20" },
     { DIS_PDU_EMITTERNAME_AVQ_21, "AVQ-21" },
     { DIS_PDU_EMITTERNAME_AVQ30X, "AVQ30X" },
     { DIS_PDU_EMITTERNAME_AVQ_50_RCA, "AVQ-50 (RCA)" },
     { DIS_PDU_EMITTERNAME_AVQ_70, "AVQ 70" },
     { DIS_PDU_EMITTERNAME_AWS_5, "AWS 5" },
     { DIS_PDU_EMITTERNAME_AWS_6, "AWS 6" },
     { DIS_PDU_EMITTERNAME_AWS_6B_300, "AWS-6B/300" },
     { DIS_PDU_EMITTERNAME_B597Z, "B597Z" },
     { DIS_PDU_EMITTERNAME_B636Z, "B636Z" },
     { DIS_PDU_EMITTERNAME_BACK_BOARD, " Back Board" },
     { DIS_PDU_EMITTERNAME_BACK_NET_A_B, " Back Net A B" },
     { DIS_PDU_EMITTERNAME_BACK_TRAP, " Back Trap" },
     { DIS_PDU_EMITTERNAME_BAE_SYSTEMS_RT_1805_APN, " BAE Systems RT-1805/APN" },
     { DIS_PDU_EMITTERNAME_BAES_DASS_2000_JAMMER, "BAES DASS-2000 Jammer" },
     { DIS_PDU_EMITTERNAME_BALANCE_BEAM, "Balance Beam" },
     { DIS_PDU_EMITTERNAME_BALTIKA_B, "BALTIKA-B" },
     { DIS_PDU_EMITTERNAME_BALTYK, "BALTYK" },
     { DIS_PDU_EMITTERNAME_BALL_END, " Ball End" },
     { DIS_PDU_EMITTERNAME_BALL_GUN, " Ball Gun" },
     { DIS_PDU_EMITTERNAME_BALL_POINT, " BALL POINT" },
     { DIS_PDU_EMITTERNAME_BAND_STAND, " Band Stand" },
     { DIS_PDU_EMITTERNAME_BAND_STAND_3, " Band Stand 3" },
     { DIS_PDU_EMITTERNAME_P_35_37_A_P_50_B_BAR_LOCK, "P-35/37 (A); P-50 (B) Bar Lock" },
     { DIS_PDU_EMITTERNAME_BARAX, "BARAX" },
     { DIS_PDU_EMITTERNAME_BASIR_110D, "BASIR-110D" },
     { DIS_PDU_EMITTERNAME_BASS_TILT, " Bass Tilt" },
     { DIS_PDU_EMITTERNAME_BADGER, "Badger" },
     { DIS_PDU_EMITTERNAME_BARRACUDA_JAMMER, "Barracuda Jammer" },
     { DIS_PDU_EMITTERNAME_BAVAR_373_TTR, "Bavar-373 TTR" },
     { DIS_PDU_EMITTERNAME_BAVAR_373_TAR, "Bavar-373 TAR" },
     { DIS_PDU_EMITTERNAME_BAVAR_373_TELAR_TER, "Bavar-373 TELAR TER" },
     { DIS_PDU_EMITTERNAME_BAYKAL_COUNTERMEASURES_SUITE, "Baykal Countermeasures Suite" },
     { DIS_PDU_EMITTERNAME_BEACON, " Beacon" },
     { DIS_PDU_EMITTERNAME_BEAN_STICKS, " Bean Sticks" },
     { DIS_PDU_EMITTERNAME_BEE_HIND, " Bee Hind" },
     { DIS_PDU_EMITTERNAME_BELL_NIP_JAMMER, "Bell Nip Jammer" },
     { DIS_PDU_EMITTERNAME_BELL_PUSH_JAMMER, "Bell Push Jammer" },
     { DIS_PDU_EMITTERNAME_BELL_CROWN_A, " Bell Crown A" },
     { DIS_PDU_EMITTERNAME_BELL_CROWN_B, " Bell Crown B" },
     { DIS_PDU_EMITTERNAME_BELL_SQUAT, " Bell Squat" },
     { DIS_PDU_EMITTERNAME_BIG_BACK, " BIG BACK" },
     { DIS_PDU_EMITTERNAME_BIG_BIRD_A_B_C, " Big Bird A/B/C" },
     { DIS_PDU_EMITTERNAME_BIG_BIRD_D, " Big Bird D" },
     { DIS_PDU_EMITTERNAME_BIG_BIRD_D_MOD, " Big Bird D Mod" },
     { DIS_PDU_EMITTERNAME_BIG_BIRD_E_91N6E, " Big Bird E 91N6E" },
     { DIS_PDU_EMITTERNAME_BIG_BULGE, " Big Bulge" },
     { DIS_PDU_EMITTERNAME_BIG_BULGE_A, " Big Bulge A" },
     { DIS_PDU_EMITTERNAME_BIG_BULGE_B, " Big Bulge B" },
     { DIS_PDU_EMITTERNAME_BIG_EYE, " BIG EYE" },
     { DIS_PDU_EMITTERNAME_SNAR_10_BIG_FRED, "SNAR-10 Big Fred" },
     { DIS_PDU_EMITTERNAME_BIG_HEAD_B, " BIG HEAD B" },
     { DIS_PDU_EMITTERNAME_BIG_MESH, " Big Mesh" },
     { DIS_PDU_EMITTERNAME_BIG_NET, " Big Net" },
     { DIS_PDU_EMITTERNAME_BIG_REST_B, " Big Rest B" },
     { DIS_PDU_EMITTERNAME_9S15MT_BILL_BOARD, "9S15MT Bill Board" },
     { DIS_PDU_EMITTERNAME_BILL_FOLD, " Bill Fold" },
     { DIS_PDU_EMITTERNAME_BLIGHTER_400, "BLIGHTER 400" },
     { DIS_PDU_EMITTERNAME_BLOWPIPE_MG, "Blowpipe MG" },
     { DIS_PDU_EMITTERNAME_BLR, "BLR" },
     { DIS_PDU_EMITTERNAME_BLUE_FOX_SEA_HARRIER_FRS_MK_1_5, "Blue Fox Sea Harrier FRS Mk 1/5" },
     { DIS_PDU_EMITTERNAME_BLUE_KESTREL, "Blue Kestrel" },
     { DIS_PDU_EMITTERNAME_BLUE_VIXEN_SEA_HARRIER_F_A_MK_2, "Blue Vixen Sea Harrier F/A Mk 2" },
     { DIS_PDU_EMITTERNAME_BLUE_SILK, "Blue Silk" },
     { DIS_PDU_EMITTERNAME_BLUE_PARROT, "Blue Parrot" },
     { DIS_PDU_EMITTERNAME_BLUE_ORCHID, "Blue Orchid" },
     { DIS_PDU_EMITTERNAME_BM_DJG_8715, "BM/DJG-8715" },
     { DIS_PDU_EMITTERNAME_BOAT_SAIL, " Boat Sail" },
     { DIS_PDU_EMITTERNAME_BOR_A_550, "BOR-A 550" },
     { DIS_PDU_EMITTERNAME_BOFORS_ELECTRONIC_9LV_331, " Bofors Electronic 9LV 331" },
     { DIS_PDU_EMITTERNAME_BOFORS_ERICSSON_SEA_GIRAFFE_50_HC, " Bofors Ericsson Sea Giraffe 50 HC" },
     { DIS_PDU_EMITTERNAME_BOWL_MESH, " Bowl Mesh" },
     { DIS_PDU_EMITTERNAME_BOX_BRICK, " Box Brick" },
     { DIS_PDU_EMITTERNAME_BOX_TAIL, " Box Tail" },
     { DIS_PDU_EMITTERNAME_BM_KG300G_JAMMING_POD, "BM/KG300G Jamming Pod" },
     { DIS_PDU_EMITTERNAME_BM_KG600_JAMMING_POD, "BM KG600 Jamming Pod" },
     { DIS_PDU_EMITTERNAME_BM_KG800_JAMMING_POD, "BM KG800 Jamming Pod" },
     { DIS_PDU_EMITTERNAME_BM_KG_8601_8605_8606, "BM/KG 8601/8605/8606" },
     { DIS_PDU_EMITTERNAME_BPS_11A, "BPS 11A" },
     { DIS_PDU_EMITTERNAME_BPS_14, "BPS 14" },
     { DIS_PDU_EMITTERNAME_BPS_15A, "BPS 15A" },
     { DIS_PDU_EMITTERNAME_BR_3440CA_X57, "BR-3440CA-X57" },
     { DIS_PDU_EMITTERNAME_BR_15_TOKYO_KEIKI, "BR-15 Tokyo KEIKI" },
     { DIS_PDU_EMITTERNAME_BRAHMOS, "BrahMos" },
     { DIS_PDU_EMITTERNAME_BRIDGEMASTER, " BridgeMaster" },
     { DIS_PDU_EMITTERNAME_BRIDGEMASTER_E_EPA, " BridgeMaster E EPA" },
     { DIS_PDU_EMITTERNAME_BRIDGEMASTER_E_ATA_AND_ARPA, " BridgeMaster E ATA and ARPA" },
     { DIS_PDU_EMITTERNAME_BRIDGEMASTER_E_NAVAL, " BridgeMaster E naval" },
     { DIS_PDU_EMITTERNAME_BRIMSTONE_MMW_MH, "Brimstone mmW MH" },
     { DIS_PDU_EMITTERNAME_BREAD_BIN, " Bread Bin" },
     { DIS_PDU_EMITTERNAME_ASR_BRICK_BOARD, "Asr Brick Board" },
     { DIS_PDU_EMITTERNAME_BT_271, "BT 271" },
     { DIS_PDU_EMITTERNAME_BU_304, "BU-304" },
     { DIS_PDU_EMITTERNAME_BX_732, "BX 732" },
     { DIS_PDU_EMITTERNAME_BUK_MB, "BUK-MB" },
     { DIS_PDU_EMITTERNAME_BURAN_D, "Buran-D" },
     { DIS_PDU_EMITTERNAME_BUREVISNYK_1, "BUREVISNYK-1" },
     { DIS_PDU_EMITTERNAME_BUZZ_STAND, " Buzz Stand" },
     { DIS_PDU_EMITTERNAME_C_5A_MULTI_MODE_RADAR, "C 5A Multi Mode Radar" },
     { DIS_PDU_EMITTERNAME_C_802_AL, "C-802 AL" },
     { DIS_PDU_EMITTERNAME_CAESAR, " CAESAR" },
     { DIS_PDU_EMITTERNAME_CAIMAN, "Caiman" },
     { DIS_PDU_EMITTERNAME_CAKE_STAND, " Cake Stand" },
     { DIS_PDU_EMITTERNAME_CALYPSO_C61, "Calypso C61" },
     { DIS_PDU_EMITTERNAME_CALYPSO_C63, "Calypso C63" },
     { DIS_PDU_EMITTERNAME_CALYPSO_II, "Calypso Ii" },
     { DIS_PDU_EMITTERNAME_CALYPSO_III, "Calypso III" },
     { DIS_PDU_EMITTERNAME_CALYPSO_IV, "Calypso IV" },
     { DIS_PDU_EMITTERNAME_CARDION_COASTAL, " Cardion Coastal" },
     { DIS_PDU_EMITTERNAME_CASTOR_II, "Castor Ii" },
     { DIS_PDU_EMITTERNAME_CASTOR_2J_TT_CROTALE_NG, "Castor 2J TT (Crotale NG)" },
     { DIS_PDU_EMITTERNAME_CAT_HOUSE, " Cat House" },
     { DIS_PDU_EMITTERNAME_CDR_431, "CDR-431" },
     { DIS_PDU_EMITTERNAME_CEAFAR, "CEAFAR" },
     { DIS_PDU_EMITTERNAME_CEAMOUNT, "CEAMOUNT" },
     { DIS_PDU_EMITTERNAME_CEAFAR2_L, "CEAFAR2-L" },
     { DIS_PDU_EMITTERNAME_CEROS_200, "CEROS 200" },
     { DIS_PDU_EMITTERNAME_CEROS_200_CWI, "CEROS 200 CWI" },
     { DIS_PDU_EMITTERNAME_CEATAC, "CEATAC" },
     { DIS_PDU_EMITTERNAME_CEAOPS, "CEAOPS" },
     { DIS_PDU_EMITTERNAME_CERBERUS_III, "Cerberus III" },
     { DIS_PDU_EMITTERNAME_CH_SS_N_6, "CH/SS-N-6" },
     { DIS_PDU_EMITTERNAME_CERBERUS_IV, "Cerberus IV" },
     { DIS_PDU_EMITTERNAME_CHAIR_BACK_TT, " Chair Back TT" },
     { DIS_PDU_EMITTERNAME_CHAIR_BACK_ILL, " Chair Back ILL" },
     { DIS_PDU_EMITTERNAME_LEMZ_96L6_CHEESE_BOARD, "LEMZ 96L6 Cheese Board" },
     { DIS_PDU_EMITTERNAME_CHEESE_BRICK, " Cheese Brick" },
     { DIS_PDU_EMITTERNAME_CHEESE_CAKE, " Cheese Cake" },
     { DIS_PDU_EMITTERNAME_CHROME_DOME, " Chrome Dome" },
     { DIS_PDU_EMITTERNAME_LENINETZ_OBZOR_MS_CLAM_PIPE, "Leninetz Obzor MS Clam Pipe" },
     { DIS_PDU_EMITTERNAME_CLAMSHELL, " Clamshell" },
     { DIS_PDU_EMITTERNAME_CLC_1, "CLC-1" },
     { DIS_PDU_EMITTERNAME_CLC_2, "CLC-2" },
     { DIS_PDU_EMITTERNAME_CLC_3, "CLC-3" },
     { DIS_PDU_EMITTERNAME_CLR_155, "CLR-155" },
     { DIS_PDU_EMITTERNAME_COAST_WATCHER_100, "COAST WATCHER 100" },
     { DIS_PDU_EMITTERNAME_COASTAL_GIRAFFE, "Coastal Giraffe" },
     { DIS_PDU_EMITTERNAME_COBRA, "COBRA" },
     { DIS_PDU_EMITTERNAME_COBRA_SHOE, "Cobra Shoe" },
     { DIS_PDU_EMITTERNAME_COLIBRI, "Colibri" },
     { DIS_PDU_EMITTERNAME_COLLINS_WXR_300, " Collins WXR-300" },
     { DIS_PDU_EMITTERNAME_COLLINS_WXR_700X, " Collins WXR-700X" },
     { DIS_PDU_EMITTERNAME_COLLINS_TWR_850_COLLINS_WXR_840, "Collins TWR-850 Collins WXR-840" },
     { DIS_PDU_EMITTERNAME_COLLINS_DN_101, " Collins DN 101" },
     { DIS_PDU_EMITTERNAME_COMET_1, " COMET-1" },
     { DIS_PDU_EMITTERNAME_CONDOR_MK_2, "CONDOR MK 2" },
     { DIS_PDU_EMITTERNAME_CONSILIUM_SELESMAR_RTM_25_XIM, " Consilium Selesmar RTM 25 XIM" },
     { DIS_PDU_EMITTERNAME_CONTRAVES_SEA_HUNTER_MK_4, " Contraves Sea Hunter MK 4" },
     { DIS_PDU_EMITTERNAME_CORN_CAN, " Corn Can" },
     { DIS_PDU_EMITTERNAME_COSMO_SKYMED_1, "COSMO SKYMED-1" },
     { DIS_PDU_EMITTERNAME_CR_105_RMCA, "CR-105 RMCA" },
     { DIS_PDU_EMITTERNAME_CREW_DUKE_2, "CREW Duke 2" },
     { DIS_PDU_EMITTERNAME_CREW_DUKE_3, "CREW Duke 3" },
     { DIS_PDU_EMITTERNAME_CROSS_BIRD, " Cross Bird" },
     { DIS_PDU_EMITTERNAME_CROSS_DOME, " Cross Dome" },
     { DIS_PDU_EMITTERNAME_CROSS_LEGS, " Cross Legs" },
     { DIS_PDU_EMITTERNAME_CROSS_OUT, " Cross Out" },
     { DIS_PDU_EMITTERNAME_CROSS_SLOT, " Cross Slot" },
     { DIS_PDU_EMITTERNAME_CROSS_SWORD, " Cross Sword" },
     { DIS_PDU_EMITTERNAME_CROSS_UP, " Cross Up" },
     { DIS_PDU_EMITTERNAME_CROSS_SWORD_FC, " Cross Sword FC" },
     { DIS_PDU_EMITTERNAME_CROTALE_ACQUISITION_TA_THD_5000, "Crotale Acquisition TA THD-5000" },
     { DIS_PDU_EMITTERNAME_CROTALE_NG_TA_GRIFFON, "Crotale NG TA Griffon" },
     { DIS_PDU_EMITTERNAME_CROTALE_TT, "Crotale TT" },
     { DIS_PDU_EMITTERNAME_CROTALE_MGMISSILE_SYSTEM, "Crotale MGMissile System" },
     { DIS_PDU_EMITTERNAME_CS_10_TA, "CS-10-TA" },
     { DIS_PDU_EMITTERNAME_CSA_20_TER, " CSA-20 TER" },
     { DIS_PDU_EMITTERNAME_CSA_20_TAR, " CSA-20 TAR" },
     { DIS_PDU_EMITTERNAME_CSA_23_FDV_MG, " CSA-23 FDV MG" },
     { DIS_PDU_EMITTERNAME_CSA_23_MISSILE_MH, " CSA-23 Missile MH" },
     { DIS_PDU_EMITTERNAME_CSF_VARAN, "CSF-Varan" },
     { DIS_PDU_EMITTERNAME_CSS_N_4_MH, "CSS-N-4 MH" },
     { DIS_PDU_EMITTERNAME_CSS_C_3C_CAS_1M1_M2_MH, "CSS C 3C CAS 1M1 M2 MH" },
     { DIS_PDU_EMITTERNAME_HY_2B_MH_CSS_C_3_SEERSUCKER_MH, "HY-2B MH CSS-C-3 Seersucker MH" },
     { DIS_PDU_EMITTERNAME_CSS_C_2B_HY_1A_MH, "CSS C 2B HY 1A MH" },
     { DIS_PDU_EMITTERNAME_CSS_N_4_SARDINE, "CSS-N-4 Sardine" },
     { DIS_PDU_EMITTERNAME_CSS_N_8_SACCADE, "CSS-N-8 Saccade" },
     { DIS_PDU_EMITTERNAME_CURL_STONE_B, " Curl Stone B" },
     { DIS_PDU_EMITTERNAME_CWS_1, "CWS-1" },
     { DIS_PDU_EMITTERNAME_CWS_2, "CWS 2" },
     { DIS_PDU_EMITTERNAME_CWS_3, "CWS-3" },
     { DIS_PDU_EMITTERNAME_CYGNUS, " Cygnus" },
     { DIS_PDU_EMITTERNAME_CYLINDER_HEAD, " Cylinder Head" },
     { DIS_PDU_EMITTERNAME_CYMBELINE, "Cymbeline" },
     { DIS_PDU_EMITTERNAME_CYRANO_II, "Cyrano II" },
     { DIS_PDU_EMITTERNAME_CYRANO_IV, "Cyrano IV" },
     { DIS_PDU_EMITTERNAME_CYRANO_IV_M, "Cyrano IV-M" },
     { DIS_PDU_EMITTERNAME_DA_01_00, "DA-01/00" },
     { DIS_PDU_EMITTERNAME_DA_05_00, "DA 05 00" },
     { DIS_PDU_EMITTERNAME_DA_05_2, "DA-05/2" },
     { DIS_PDU_EMITTERNAME_DA_08, "DA.08" },
     { DIS_PDU_EMITTERNAME_DAWN, " Dawn" },
     { DIS_PDU_EMITTERNAME_DCR, "DCR" },
     { DIS_PDU_EMITTERNAME_DEAD_DUCK, " Dead Duck" },
     { DIS_PDU_EMITTERNAME_DECCA_20_V90_9, " DECCA-20 V90/9" },
     { DIS_PDU_EMITTERNAME_DECCA_20_V90S, " DECCA-20 V90S" },
     { DIS_PDU_EMITTERNAME_DECCA_45, " DECCA 45" },
     { DIS_PDU_EMITTERNAME_DECCA_50, " DECCA 50" },
     { DIS_PDU_EMITTERNAME_DECCA_71, " DECCA 71" },
     { DIS_PDU_EMITTERNAME_DECCA_72, "Decca 72" },
     { DIS_PDU_EMITTERNAME_DECCA_110, " DECCA 110" },
     { DIS_PDU_EMITTERNAME_DECCA_170, " DECCA 170" },
     { DIS_PDU_EMITTERNAME_DECCA_HF_2, " DECCA HF 2" },
     { DIS_PDU_EMITTERNAME_DECCA_202, " DECCA 202" },
     { DIS_PDU_EMITTERNAME_DECCA_D202, " DECCA D202" },
     { DIS_PDU_EMITTERNAME_DECCA_303, " DECCA 303" },
     { DIS_PDU_EMITTERNAME_DECCA_535, " DECCA 535" },
     { DIS_PDU_EMITTERNAME_DECCA_626, " DECCA 626" },
     { DIS_PDU_EMITTERNAME_DECCA_629, " DECCA 629" },
     { DIS_PDU_EMITTERNAME_DECCA_914, " DECCA 914" },
     { DIS_PDU_EMITTERNAME_DECCA_916, " DECCA 916" },
     { DIS_PDU_EMITTERNAME_DECCA_926, " DECCA 926" },
     { DIS_PDU_EMITTERNAME_DECCA_1070A, " DECCA 1070A" },
     { DIS_PDU_EMITTERNAME_DECCA_1008, " Decca 1008" },
     { DIS_PDU_EMITTERNAME_DECCA_1226_COMMERCIAL, " DECCA 1226 Commercial" },
     { DIS_PDU_EMITTERNAME_DECCA_1290, " DECCA 1290" },
     { DIS_PDU_EMITTERNAME_DECCA_1626, " DECCA 1626" },
     { DIS_PDU_EMITTERNAME_DECCA_2070, " DECCA 2070" },
     { DIS_PDU_EMITTERNAME_DECCA_1630, " Decca 1630" },
     { DIS_PDU_EMITTERNAME_DECCA_2459, " DECCA 2459" },
     { DIS_PDU_EMITTERNAME_DECCA_AWS_1, " DECCA AWS 1" },
     { DIS_PDU_EMITTERNAME_DECCA_AWS_2, " DECCA AWS 2" },
     { DIS_PDU_EMITTERNAME_DECCA_AWS_4, " DECCA AWS 4" },
     { DIS_PDU_EMITTERNAME_DECCA_AWS_4_2, " DECCA AWS-4 (2)" },
     { DIS_PDU_EMITTERNAME_DECCA_MAR, " DECCA MAR" },
     { DIS_PDU_EMITTERNAME_DECCA_RM_326, " DECCA RM 326" },
     { DIS_PDU_EMITTERNAME_DECCA_RM_416, " DECCA RM 416" },
     { DIS_PDU_EMITTERNAME_DECCA_RM_970BT, "DECCA RM 970BT" },
     { DIS_PDU_EMITTERNAME_DECCA_RM_914, " DECCA RM 914" },
     { DIS_PDU_EMITTERNAME_DF_21D_SEEKER, "DF-21D Seeker" },
     { DIS_PDU_EMITTERNAME_DECCA_RM_1690, " DECCA RM 1690" },
     { DIS_PDU_EMITTERNAME_DECCA_1690, " DECCA 1690" },
     { DIS_PDU_EMITTERNAME_DECCA_SUPER_101_MK_3, " DECCA Super 101 MK 3" },
     { DIS_PDU_EMITTERNAME_DISS_1, "DISS 1" },
     { DIS_PDU_EMITTERNAME_DISS_7, "DISS-7" },
     { DIS_PDU_EMITTERNAME_DISS_013, "DISS-013" },
     { DIS_PDU_EMITTERNAME_DISS_15D, "DISS-15D" },
     { DIS_PDU_EMITTERNAME_DLD_100A, "DLD-100A" },
     { DIS_PDU_EMITTERNAME_RAPIER_TTDN_181_DN_181, "Rapier TTDN 181 DN 181" },
     { DIS_PDU_EMITTERNAME_RAPIER_2000_TT_BLINDFIRE_FSC_TT, "Rapier 2000 TT BLINDFIRE FSC TT" },
     { DIS_PDU_EMITTERNAME_DOG_EAR, " Dog Ear" },
     { DIS_PDU_EMITTERNAME_DOG_HOUSE, " Dog House" },
     { DIS_PDU_EMITTERNAME_DM3, "DM3" },
     { DIS_PDU_EMITTERNAME_DM_3B, "DM-3B" },
     { DIS_PDU_EMITTERNAME_DM_5, "DM-5" },
     { DIS_PDU_EMITTERNAME_DON_2, "Don 2" },
     { DIS_PDU_EMITTERNAME_DON_A_B_2_KAY, " Don A/B/2/Kay" },
     { DIS_PDU_EMITTERNAME_DONETS, " Donets" },
     { DIS_PDU_EMITTERNAME_DOPPLER_90_SERIES, " Doppler 90 Series" },
     { DIS_PDU_EMITTERNAME_DOWN_BEAT, " Down Beat" },
     { DIS_PDU_EMITTERNAME_DR_582, "DR-582" },
     { DIS_PDU_EMITTERNAME_DRAA_2A, "DRAA 2A" },
     { DIS_PDU_EMITTERNAME_DRAA_2B, "DRAA 2B" },
     { DIS_PDU_EMITTERNAME_DRAA_9A, "DRAA 9A" },
     { DIS_PDU_EMITTERNAME_DRAA_11A, "DRAA 11A" },
     { DIS_PDU_EMITTERNAME_DRAC_37B, "DRAC 37B" },
     { DIS_PDU_EMITTERNAME_DRAC_38, "DRAC 38" },
     { DIS_PDU_EMITTERNAME_DRAC_39, "DRAC 39" },
     { DIS_PDU_EMITTERNAME_DRAC_39A, "DRAC 39A" },
     { DIS_PDU_EMITTERNAME_DRAC_43A, "DRAC 43A" },
     { DIS_PDU_EMITTERNAME_DRAC_44A, "DRAC 44A" },
     { DIS_PDU_EMITTERNAME_DRAGON_EYE, " Dragon Eye" },
     { DIS_PDU_EMITTERNAME_DRAGON_EYE_2, " Dragon Eye 2" },
     { DIS_PDU_EMITTERNAME_DRAGON_EYE_3, " Dragon Eye 3" },
     { DIS_PDU_EMITTERNAME_DRAGON_EYE_4, " Dragon Eye 4" },
     { DIS_PDU_EMITTERNAME_DRBC_30B, "DRBC 30B" },
     { DIS_PDU_EMITTERNAME_DRBC_31A, "DRBC 31A" },
     { DIS_PDU_EMITTERNAME_DRBC_31D, "DRBC-31D" },
     { DIS_PDU_EMITTERNAME_DRBC_32, "DRBC-32" },
     { DIS_PDU_EMITTERNAME_DRBC_32A, "DRBC 32A" },
     { DIS_PDU_EMITTERNAME_DRBC_32D, "DRBC 32D" },
     { DIS_PDU_EMITTERNAME_DRBC_33A, "DRBC 33A" },
     { DIS_PDU_EMITTERNAME_DRBI_10, "DRBI 10" },
     { DIS_PDU_EMITTERNAME_DRBI_23, "DRBI 23" },
     { DIS_PDU_EMITTERNAME_DRBJ_11B, "DRBJ 11B" },
     { DIS_PDU_EMITTERNAME_DRBN_30, "DRBN 30" },
     { DIS_PDU_EMITTERNAME_DRBN_32, "DRBN 32" },
     { DIS_PDU_EMITTERNAME_DRBN_34, "DRBN 34" },
     { DIS_PDU_EMITTERNAME_DRBR_51, "DRBR 51" },
     { DIS_PDU_EMITTERNAME_DRBV_20A, "DRBV-20A" },
     { DIS_PDU_EMITTERNAME_DRBV_20B, "DRBV 20B" },
     { DIS_PDU_EMITTERNAME_DRBV_21_MARS_05, "DRBV-21 Mars 05" },
     { DIS_PDU_EMITTERNAME_DRBV_22, "DRBV 22" },
     { DIS_PDU_EMITTERNAME_DRBV_23, "DRBV-23" },
     { DIS_PDU_EMITTERNAME_DRBV_26C, "DRBV 26C" },
     { DIS_PDU_EMITTERNAME_DRBV_26D, "DRBV 26D" },
     { DIS_PDU_EMITTERNAME_DRBV_30, "DRBV 30" },
     { DIS_PDU_EMITTERNAME_DRBV_31, "DRBV-31" },
     { DIS_PDU_EMITTERNAME_DRBV_50, "DRBV 50" },
     { DIS_PDU_EMITTERNAME_DRBV_51, "DRBV 51" },
     { DIS_PDU_EMITTERNAME_DRBV_51A, "DRBV 51A" },
     { DIS_PDU_EMITTERNAME_DRBV_51B, "DRBV 51B" },
     { DIS_PDU_EMITTERNAME_DRBV_51C, "DRBV 51C" },
     { DIS_PDU_EMITTERNAME_DROP_KICK, "Drop Kick" },
     { DIS_PDU_EMITTERNAME_DRUA_31, "DRUA 31" },
     { DIS_PDU_EMITTERNAME_DRUM_TILT, " Drum Tilt" },
     { DIS_PDU_EMITTERNAME_DRUM_TILT_A, " Drum Tilt A" },
     { DIS_PDU_EMITTERNAME_DRUM_TILT_B, " Drum Tilt B" },
     { DIS_PDU_EMITTERNAME_DRUN_30A, "DRUN 30A" },
     { DIS_PDU_EMITTERNAME_DUMBO, " Dumbo" },
     { DIS_PDU_EMITTERNAME_DWSR_92, "DWSR-92" },
     { DIS_PDU_EMITTERNAME_DWSR_93S, "DWSR-93S" },
     { DIS_PDU_EMITTERNAME_EAGLE, "EAGLE" },
     { DIS_PDU_EMITTERNAME_EAGLE_MK_1, "EAGLE Mk 1" },
     { DIS_PDU_EMITTERNAME_EAJP_JAMMING_POD, "EAJP Jamming Pod" },
     { DIS_PDU_EMITTERNAME_EKCO_E390, "EKCO E390" },
     { DIS_PDU_EMITTERNAME_ECR_90_CAPTOR, "ECR-90 CAPTOR" },
     { DIS_PDU_EMITTERNAME_ECR_90_JAMMER, "ECR-90 Jammer" },
     { DIS_PDU_EMITTERNAME_EGG_CUP_A_B, " Egg Cup A/B" },
     { DIS_PDU_EMITTERNAME_EISCAT, "EISCAT" },
     { DIS_PDU_EMITTERNAME_EKCO_E120, "EKCO E120" },
     { DIS_PDU_EMITTERNAME_EKCO_190, "EKCO 190" },
     { DIS_PDU_EMITTERNAME_EKRAN_1, "Ekran-1" },
     { DIS_PDU_EMITTERNAME_EL_L_8222, "EL/L-8222" },
     { DIS_PDU_EMITTERNAME_EL_L_8240, " EL/L-8240" },
     { DIS_PDU_EMITTERNAME_EL_M_2001B, "EL M 2001B" },
     { DIS_PDU_EMITTERNAME_EL_M_2022, "EL/M-2022" },
     { DIS_PDU_EMITTERNAME_EL_M_2032, "EL/M-2032" },
     { DIS_PDU_EMITTERNAME_EL_M_2052, "EL/M-2052" },
     { DIS_PDU_EMITTERNAME_EL_M_2055, "EL/M-2055" },
     { DIS_PDU_EMITTERNAME_EL_M_2060, "EL/M-2060" },
     { DIS_PDU_EMITTERNAME_EL_M_2075, "EL/M-2075" },
     { DIS_PDU_EMITTERNAME_EL_M_2022U_V_3, "EL/M-2022U(V)3" },
     { DIS_PDU_EMITTERNAME_EL_M_2080, "EL/M-2080" },
     { DIS_PDU_EMITTERNAME_EL_M_2080S, "EL/M-2080S" },
     { DIS_PDU_EMITTERNAME_EL_M_2085, "EL/M-2085" },
     { DIS_PDU_EMITTERNAME_EL_M_2106, "EL/M-2106" },
     { DIS_PDU_EMITTERNAME_EL_M_2106NG, "EL/M-2106NG" },
     { DIS_PDU_EMITTERNAME_EL_M_2125, "EL/M-2125" },
     { DIS_PDU_EMITTERNAME_EL_M_2129, "EL/M-2129" },
     { DIS_PDU_EMITTERNAME_EL_M_2150, "EL/M-2150" },
     { DIS_PDU_EMITTERNAME_EL_M_2083, "EL/M-2083" },
     { DIS_PDU_EMITTERNAME_EL_M_2084, "EL/M-2084" },
     { DIS_PDU_EMITTERNAME_EL_M_2160_V1, "EL/M-2160-V1" },
     { DIS_PDU_EMITTERNAME_EL_M_2084_MMR, "EL/M-2084 MMR" },
     { DIS_PDU_EMITTERNAME_EL_M_2112, "EL/M-2112" },
     { DIS_PDU_EMITTERNAME_EL_M_2200, "EL/M-2200" },
     { DIS_PDU_EMITTERNAME_EL_M_2133, "EL/M-2133" },
     { DIS_PDU_EMITTERNAME_EL_M_2205, "EL/M-2205" },
     { DIS_PDU_EMITTERNAME_EL_M_2207, "EL M 2207" },
     { DIS_PDU_EMITTERNAME_EL_M_2215, "EL/M-2215" },
     { DIS_PDU_EMITTERNAME_EL_M_2216_V, " EL/M 2216(V)" },
     { DIS_PDU_EMITTERNAME_EL_M_2216XH, "EL/M-2216XH" },
     { DIS_PDU_EMITTERNAME_EL_M_2218S, "EL/M-2218S" },
     { DIS_PDU_EMITTERNAME_ELT_361, "ELT-361" },
     { DIS_PDU_EMITTERNAME_EL_M_2258, "EL/M-2258" },
     { DIS_PDU_EMITTERNAME_ELT_553, "ELT-553" },
     { DIS_PDU_EMITTERNAME_ELT_558, "ELT-558" },
     { DIS_PDU_EMITTERNAME_ELT_572, "ELT-572" },
     { DIS_PDU_EMITTERNAME_ELT_715, "ELT 715" },
     { DIS_PDU_EMITTERNAME_ELTA_ELM_2022A, "Elta ELM 2022A" },
     { DIS_PDU_EMITTERNAME_ELTA_EL_M_2221_GM_STGR, "ELTA EL/M 2221 GM STGR" },
     { DIS_PDU_EMITTERNAME_EL_M_2228S_3D, "EL/M-2228S/3D" },
     { DIS_PDU_EMITTERNAME_EL_M_2705, "EL/M-2705" },
     { DIS_PDU_EMITTERNAME_EL_M_2226, "EL/M-2226" },
     { DIS_PDU_EMITTERNAME_EL_M_2228X, "EL/M-2228X" },
     { DIS_PDU_EMITTERNAME_ELTA_SIS, "ELTA SIS" },
     { DIS_PDU_EMITTERNAME_EL_M_2238, "EL/M-2238" },
     { DIS_PDU_EMITTERNAME_EL_M_2248, "EL/M-2248" },
     { DIS_PDU_EMITTERNAME_EL_M_2288, "EL/M-2288" },
     { DIS_PDU_EMITTERNAME_EL_M_2311, "EL/M-2311" },
     { DIS_PDU_EMITTERNAME_ELM_2026, "ELM-2026" },
     { DIS_PDU_EMITTERNAME_ELNA_4007, " ELNA 4007" },
     { DIS_PDU_EMITTERNAME_ELT_318, "ELT/318" },
     { DIS_PDU_EMITTERNAME_ELW_2085, "ELW-2085" },
     { DIS_PDU_EMITTERNAME_ELT_521, "ELT/521" },
     { DIS_PDU_EMITTERNAME_ELW_2090, "ELW-2090" },
     { DIS_PDU_EMITTERNAME_ENHANCED_METEOR_DETECTION_RADAR_EMDR, " Enhanced Meteor Detection Radar EMDR" },
     { DIS_PDU_EMITTERNAME_EMD_2900, "EMD 2900" },
     { DIS_PDU_EMITTERNAME_EMPAR, "EMPAR" },
     { DIS_PDU_EMITTERNAME_END_TRAY, " End Tray" },
     { DIS_PDU_EMITTERNAME_EQ_36, "EQ-36" },
     { DIS_PDU_EMITTERNAME_ERICSSON_SLAR, "Ericsson SLAR" },
     { DIS_PDU_EMITTERNAME_ERIEYE, "Erieye" },
     { DIS_PDU_EMITTERNAME_ES_3701_ECM_ESM_SUITE, " ES-3701 ECM/ESM suite" },
     { DIS_PDU_EMITTERNAME_ESR_1, "ESR 1" },
     { DIS_PDU_EMITTERNAME_ESR_220, "ESR 220" },
     { DIS_PDU_EMITTERNAME_ESR380, "ESR380" },
     { DIS_PDU_EMITTERNAME_ESTEREL, "ESTEREL" },
     { DIS_PDU_EMITTERNAME_ET_316, "ET-316" },
     { DIS_PDU_EMITTERNAME_EXOCET_TYPE, "Exocet Type" },
     { DIS_PDU_EMITTERNAME_EXOCET_AL, "Exocet AL" },
     { DIS_PDU_EMITTERNAME_EXOCET_1, "Exocet 1" },
     { DIS_PDU_EMITTERNAME_EXOCET_1_MH, "Exocet 1 MH" },
     { DIS_PDU_EMITTERNAME_EXOCET_2, "Exocet 2" },
     { DIS_PDU_EMITTERNAME_EYE_BOWL, " Eye Bowl" },
     { DIS_PDU_EMITTERNAME_EYE_SHIELD, " Eye Shield" },
     { DIS_PDU_EMITTERNAME_F332Z, " F332Z" },
     { DIS_PDU_EMITTERNAME_FALCON_CLAW_TI, " Falcon Claw TI" },
     { DIS_PDU_EMITTERNAME_FALCON_CLAW_TT, " Falcon Claw TT" },
     { DIS_PDU_EMITTERNAME_FALCON, "FALCON" },
     { DIS_PDU_EMITTERNAME_FALCON_G, "FALCON-G" },
     { DIS_PDU_EMITTERNAME_FALCON_EYE, " Falcon Eye" },
     { DIS_PDU_EMITTERNAME_FAN_SONG_A, " Fan Song A" },
     { DIS_PDU_EMITTERNAME_FAN_SONG_B_F_TA, " Fan Song B/F TA" },
     { DIS_PDU_EMITTERNAME_FAN_SONG_B_F_TT, " Fan Song B/F TT" },
     { DIS_PDU_EMITTERNAME_FAN_SONG_C_E_TA, " Fan Song C/E TA" },
     { DIS_PDU_EMITTERNAME_FAN_SONG_C_E_TT, " Fan Song C/E TT" },
     { DIS_PDU_EMITTERNAME_FAN_SONG_C_E_MG, " Fan Song C/E MG" },
     { DIS_PDU_EMITTERNAME_FAN_SONG_B_FF_MG, " Fan Song B/FF MG" },
     { DIS_PDU_EMITTERNAME_FAN_TAIL, " Fan Tail" },
     { DIS_PDU_EMITTERNAME_FAR_2117, "FAR-2117" },
     { DIS_PDU_EMITTERNAME_FAR_2827, "FAR-2827" },
     { DIS_PDU_EMITTERNAME_FAR_2837S, "FAR-2837S" },
     { DIS_PDU_EMITTERNAME_FAR_3000, " FAR-3000" },
     { DIS_PDU_EMITTERNAME_FB_7_RADAR, "FB-7 Radar" },
     { DIS_PDU_EMITTERNAME_FCR_1401, "FCR-1401" },
     { DIS_PDU_EMITTERNAME_FCS_2_12E, "FCS-2-12E" },
     { DIS_PDU_EMITTERNAME_FCS_2_12G, "FCS-2-12G" },
     { DIS_PDU_EMITTERNAME_FCS_2_21A, "FCS-2-21A" },
     { DIS_PDU_EMITTERNAME_FCS_2_21C, "FCS-2-21C" },
     { DIS_PDU_EMITTERNAME_FCS_2_22, "FCS-2-22" },
     { DIS_PDU_EMITTERNAME_FCS_2_31, "FCS-2-31" },
     { DIS_PDU_EMITTERNAME_FCS_3, "FCS-3" },
     { DIS_PDU_EMITTERNAME_FIN_CURVE, " Fin Curve" },
     { DIS_PDU_EMITTERNAME_FIRE_CAN, " Fire Can" },
     { DIS_PDU_EMITTERNAME_FIRE_DISH, " Fire Dish" },
     { DIS_PDU_EMITTERNAME_FIRE_DOME_TA, " Fire Dome TA" },
     { DIS_PDU_EMITTERNAME_FIRE_DOME_TT, " Fire Dome TT" },
     { DIS_PDU_EMITTERNAME_FIRE_DOME_TI, " Fire Dome TI" },
     { DIS_PDU_EMITTERNAME_FIRE_IRON, " Fire Iron" },
     { DIS_PDU_EMITTERNAME_FIRE_WHEEL, " Fire Wheel" },
     { DIS_PDU_EMITTERNAME_FISH_BOWL, " Fish Bowl" },
     { DIS_PDU_EMITTERNAME_FK_3, "FK-3" },
     { DIS_PDU_EMITTERNAME_FLAIR, "FLAIR" },
     { DIS_PDU_EMITTERNAME_FLAP_LID, " Flap Lid" },
     { DIS_PDU_EMITTERNAME_30N6E_FLAP_LID_B, "30N6E Flap Lid B" },
     { DIS_PDU_EMITTERNAME_FLAP_TRUCK, " Flap Truck" },
     { DIS_PDU_EMITTERNAME_FLAP_WHEEL, " Flap Wheel" },
     { DIS_PDU_EMITTERNAME_FLASH_DANCE, " Flash Dance" },
     { DIS_PDU_EMITTERNAME_FLASH_DANCE_M, " Flash Dance M" },
     { DIS_PDU_EMITTERNAME_P_15_FLAT_FACE_A_B_C_D, "P-15 Flat Face A B C D" },
     { DIS_PDU_EMITTERNAME_35N6_FLAT_FACE_E, "35N6 Flat Face E" },
     { DIS_PDU_EMITTERNAME_FLAT_SCREEN, " Flat Screen" },
     { DIS_PDU_EMITTERNAME_FLAT_SPIN, " Flat Spin" },
     { DIS_PDU_EMITTERNAME_FLAT_TRACK_JAMMER, "Flat Track Jammer" },
     { DIS_PDU_EMITTERNAME_FLAT_TWIN, " Flat Twin" },
     { DIS_PDU_EMITTERNAME_FL_400, "FL-400" },
     { DIS_PDU_EMITTERNAME_FL_1800, "FL 1800" },
     { DIS_PDU_EMITTERNAME_FL_1800U, "FL 1800U" },
     { DIS_PDU_EMITTERNAME_FL_1800S, "FL 1800S" },
     { DIS_PDU_EMITTERNAME_FLEDERMAUS, "Fledermaus" },
     { DIS_PDU_EMITTERNAME_FLYCATCHER, " FLYCATCHER" },
     { DIS_PDU_EMITTERNAME_FLYCATCHER_MK_2, "FLYCATCHER MK 2" },
     { DIS_PDU_EMITTERNAME_FLY_SCREEN, " Fly Screen" },
     { DIS_PDU_EMITTERNAME_FLY_SCREEN_A_B, " Fly Screen A&B" },
     { DIS_PDU_EMITTERNAME_FLY_TRAP_B, " Fly Trap B" },
     { DIS_PDU_EMITTERNAME_FM_90, "FM-90" },
     { DIS_PDU_EMITTERNAME_FOG_LAMP_MG, " Fog Lamp MG" },
     { DIS_PDU_EMITTERNAME_FOG_LAMP_TT, " Fog Lamp TT" },
     { DIS_PDU_EMITTERNAME_FOIL_TWO, " Foil Two" },
     { DIS_PDU_EMITTERNAME_FOOTBALL, "FootBall" },
     { DIS_PDU_EMITTERNAME_FOX_HUNTER, "Fox Hunter" },
     { DIS_PDU_EMITTERNAME_FOX_FIRE_AL, " Fox Fire AL" },
     { DIS_PDU_EMITTERNAME_FOX_FIRE_ILL, " Fox Fire ILL" },
     { DIS_PDU_EMITTERNAME_FR_151A, "FR-151A" },
     { DIS_PDU_EMITTERNAME_FURUNO_FR_1500_FR_1600, " Furuno FR-1500/FR-1600" },
     { DIS_PDU_EMITTERNAME_FR_1505_DA, "FR-1505 DA" },
     { DIS_PDU_EMITTERNAME_FR_1510DS, "FR-1510DS" },
     { DIS_PDU_EMITTERNAME_FR_2000, "FR-2000" },
     { DIS_PDU_EMITTERNAME_FURUNO_2855W, "Furuno-2855W" },
     { DIS_PDU_EMITTERNAME_FREGAT_MAE, " Fregat MAE" },
     { DIS_PDU_EMITTERNAME_FREGAT_N1, " Fregat N1" },
     { DIS_PDU_EMITTERNAME_FREGAT_N2, " Fregat N2" },
     { DIS_PDU_EMITTERNAME_FRONT_DOME, " Front Dome" },
     { DIS_PDU_EMITTERNAME_FREGAT_MAE_5, "Fregat MAE-5" },
     { DIS_PDU_EMITTERNAME_FRONT_DOOR, " Front Door" },
     { DIS_PDU_EMITTERNAME_FRONT_PIECE, " Front Piece" },
     { DIS_PDU_EMITTERNAME_FURBY_MMW_MH, "Furby mmW MH" },
     { DIS_PDU_EMITTERNAME_FURKE, " Furke" },
     { DIS_PDU_EMITTERNAME_FURKE_2_FURKE_E_POSITIV_ME1, "Furke 2 (Furke-E, Positiv-ME1)" },
     { DIS_PDU_EMITTERNAME_FURKE_4, "Furke-4" },
     { DIS_PDU_EMITTERNAME_FURUNO, "Furuno" },
     { DIS_PDU_EMITTERNAME_FURUNO_1721, "Furuno 1721" },
     { DIS_PDU_EMITTERNAME_FURUNO_1934C, " Furuno 1934C" },
     { DIS_PDU_EMITTERNAME_FURUNO_1715, " Furuno 1715" },
     { DIS_PDU_EMITTERNAME_FURUNO_1730, "Furuno 1730" },
     { DIS_PDU_EMITTERNAME_FURUNO_1731_MARK_3, "Furuno 1731 Mark 3" },
     { DIS_PDU_EMITTERNAME_FURUNO_1832, "Furuno 1832" },
     { DIS_PDU_EMITTERNAME_FURUNO_1835, " Furuno 1835" },
     { DIS_PDU_EMITTERNAME_FURUNO_1932, "Furuno 1932" },
     { DIS_PDU_EMITTERNAME_FURUNO_1935, " Furuno 1935" },
     { DIS_PDU_EMITTERNAME_FURUNO_701, "Furuno 701" },
     { DIS_PDU_EMITTERNAME_FURUNO_1940, "Furuno 1940" },
     { DIS_PDU_EMITTERNAME_FURUNO_711_2, "Furuno 711 2" },
     { DIS_PDU_EMITTERNAME_FURUNO_FR_2130S, " Furuno FR-2130S" },
     { DIS_PDU_EMITTERNAME_FURUNO_FAR_2137S, "Furuno FAR-2137S" },
     { DIS_PDU_EMITTERNAME_FURUNO_FAR_28X7, "Furuno FAR-28X7" },
     { DIS_PDU_EMITTERNAME_FURUNO_FAR_3230S, " Furuno FAR-3230S" },
     { DIS_PDU_EMITTERNAME_FR_2110, "FR-2110" },
     { DIS_PDU_EMITTERNAME_FR_2115, "FR-2115" },
     { DIS_PDU_EMITTERNAME_FR_8062, "FR-8062" },
     { DIS_PDU_EMITTERNAME_FURUNO_2125, "Furuno 2125" },
     { DIS_PDU_EMITTERNAME_FURUNO_240, "Furuno 240" },
     { DIS_PDU_EMITTERNAME_FURUNO_2400, "Furuno 2400" },
     { DIS_PDU_EMITTERNAME_FR_801D_FURUNO_FR_810D, "FR-801D Furuno FR-810D" },
     { DIS_PDU_EMITTERNAME_FURUNO_8051, "Furuno 8051" },
     { DIS_PDU_EMITTERNAME_FURUNO_DRS_2D, " Furuno DRS-2D" },
     { DIS_PDU_EMITTERNAME_FURUNO_DRS_4D, " Furuno DRS-4D" },
     { DIS_PDU_EMITTERNAME_FURUNO_DRS_4A, " Furuno DRS-4A" },
     { DIS_PDU_EMITTERNAME_G030A_APD_31, "G030A(APD-31)" },
     { DIS_PDU_EMITTERNAME_FURUNO_DRS6A_X_CLASS, " Furuno DRS6A X-Class" },
     { DIS_PDU_EMITTERNAME_DRS4W, " DRS4W" },
     { DIS_PDU_EMITTERNAME_GA_01_00, "GA 01 00" },
     { DIS_PDU_EMITTERNAME_GABBIANO, "Gabbiano" },
     { DIS_PDU_EMITTERNAME_GAGE, " Gage" },
     { DIS_PDU_EMITTERNAME_GAOFEN_3, "Gaofen-3" },
     { DIS_PDU_EMITTERNAME_GAOFEN_12, "GAOFEN-12" },
     { DIS_PDU_EMITTERNAME_GAP_GATE, "GAP GATE" },
     { DIS_PDU_EMITTERNAME_GARDENIA, " Gardenia" },
     { DIS_PDU_EMITTERNAME_GARMIN_GMR_1224, " Garmin GMR 1224" },
     { DIS_PDU_EMITTERNAME_GARMIN_FANTOM_24, " Garmin Fantom 24" },
     { DIS_PDU_EMITTERNAME_GARMIN_GWX_68_WEATHER_RADAR, "Garmin GWX 68 Weather Radar" },
     { DIS_PDU_EMITTERNAME_GARPIN, " Garpin" },
     { DIS_PDU_EMITTERNAME_GATE_GUARD, " Gate Guard" },
     { DIS_PDU_EMITTERNAME_GARPUN_BAL_E, "Garpun-Bal-E" },
     { DIS_PDU_EMITTERNAME_GAZETCHIK, "Gazetchik" },
     { DIS_PDU_EMITTERNAME_GBS1, "GBS1" },
     { DIS_PDU_EMITTERNAME_GCA_2000, "GCA-2000" },
     { DIS_PDU_EMITTERNAME_FURUNO_1832_1921_SERIES, " Furuno 1832/1921 Series" },
     { DIS_PDU_EMITTERNAME_GEMINI_DB, " GEMINI-DB" },
     { DIS_PDU_EMITTERNAME_GEM_OLYMPUS, " GEM Olympus" },
     { DIS_PDU_EMITTERNAME_GEM_SENTINEL, " GEM Sentinel" },
     { DIS_PDU_EMITTERNAME_GEM_BX_132, "GEM BX 132" },
     { DIS_PDU_EMITTERNAME_GEM_SC_2050X, "GEM SC-2050X" },
     { DIS_PDU_EMITTERNAME_GEM_SEA_EAGLE_200_N, " GEM Sea Eagle 200 N" },
     { DIS_PDU_EMITTERNAME_GENERIC_INTERNAL_JAMMER, " Generic Internal Jammer" },
     { DIS_PDU_EMITTERNAME_MPDR_12_GEPARD_TA, "MPDR-12 Gepard TA" },
     { DIS_PDU_EMITTERNAME_GEN_X, "GEN-X" },
     { DIS_PDU_EMITTERNAME_GEPARD_TT, " Gepard TT" },
     { DIS_PDU_EMITTERNAME_GERAN_F, "GERAN-F" },
     { DIS_PDU_EMITTERNAME_GERFAUT, "GERFAUT" },
     { DIS_PDU_EMITTERNAME_GFE_L_1, "GFE(L)1" },
     { DIS_PDU_EMITTERNAME_GIRAFFE, "GIRAFFE" },
     { DIS_PDU_EMITTERNAME_GIRAFFE_1X, "GIRAFFE 1X" },
     { DIS_PDU_EMITTERNAME_GIRAFFE_40, "Giraffe-40" },
     { DIS_PDU_EMITTERNAME_GIRAFFE_50_AT, "Giraffe-50 AT" },
     { DIS_PDU_EMITTERNAME_GIRAFFE_75, "Giraffe 75" },
     { DIS_PDU_EMITTERNAME_GIN_SLING_TA, " Gin Sling TA" },
     { DIS_PDU_EMITTERNAME_GIN_SLING_GIN_SLING_TT, "Gin Sling Gin Sling TT" },
     { DIS_PDU_EMITTERNAME_GIN_SLING_MG, " Gin Sling MG" },
     { DIS_PDU_EMITTERNAME_GOAL_KEEPER, "Goal Keeper" },
     { DIS_PDU_EMITTERNAME_GOLDEN_DOME, " Golden Dome" },
     { DIS_PDU_EMITTERNAME_GOLDEN_HEART, " Golden Heart" },
     { DIS_PDU_EMITTERNAME_GOLDEN_ROCK, " Golden Rock" },
     { DIS_PDU_EMITTERNAME_GORY_MH, " Gory MH" },
     { DIS_PDU_EMITTERNAME_GPN_22, "GPN-22" },
     { DIS_PDU_EMITTERNAME_GPSJ_10, "GPSJ-10" },
     { DIS_PDU_EMITTERNAME_GPSJ_25, "GPSJ-25" },
     { DIS_PDU_EMITTERNAME_GPSJ_40, "GPSJ-40" },
     { DIS_PDU_EMITTERNAME_GPSJ_50, "GPSJ-50" },
     { DIS_PDU_EMITTERNAME_GRN_9, "GRN-9" },
     { DIS_PDU_EMITTERNAME_GRAN_K, "GRAN-K" },
     { DIS_PDU_EMITTERNAME_GRACKLE_MH, " Grackle MH" },
     { DIS_PDU_EMITTERNAME_GRAVE_STONE, " Grave Stone" },
     { DIS_PDU_EMITTERNAME_GRAVES, "GRAVES" },
     { DIS_PDU_EMITTERNAME_GREEN_STAIN, "Green Stain" },
     { DIS_PDU_EMITTERNAME_GRID_BOW, " Grid Bow" },
     { DIS_PDU_EMITTERNAME_GRIFO_F, "Grifo-F" },
     { DIS_PDU_EMITTERNAME_9S32_GRILL_PAN_TT, "9S32 Grill Pan TT" },
     { DIS_PDU_EMITTERNAME_GRILL_SCREEN, " GRILL SCREEN" },
     { DIS_PDU_EMITTERNAME_GROM_2, "Grom-2" },
     { DIS_PDU_EMITTERNAME_GROUND_MASTER_400, "GROUND MASTER 400" },
     { DIS_PDU_EMITTERNAME_GT_4, "GT-4" },
     { DIS_PDU_EMITTERNAME_GRS_440, "GRS 440" },
     { DIS_PDU_EMITTERNAME_GRUESOME_MH, " Gruesome MH" },
     { DIS_PDU_EMITTERNAME_GUARDIAN, "GUARDIAN" },
     { DIS_PDU_EMITTERNAME_GUARDSMAN, "Guardsman" },
     { DIS_PDU_EMITTERNAME_RPK_2_GUN_DISH_ZSU_23_4, "RPK-2 Gun Dish (ZSU-23/4)" },
     { DIS_PDU_EMITTERNAME_GUSHER_MH, " Gusher MH" },
     { DIS_PDU_EMITTERNAME_H_RJZ_726_4A_JAMMER, "H/RJZ-726-4A Jammer" },
     { DIS_PDU_EMITTERNAME_H025_NO25E, "H025 (NO25E)" },
     { DIS_PDU_EMITTERNAME_HADR, "HADR" },
     { DIS_PDU_EMITTERNAME_HAIR_NET, " Hair Net" },
     { DIS_PDU_EMITTERNAME_HALF_PLATE_A, " Half Plate A" },
     { DIS_PDU_EMITTERNAME_HALF_PLATE_B, " Half Plate B" },
     { DIS_PDU_EMITTERNAME_HARD, "HARD" },
     { DIS_PDU_EMITTERNAME_HARPOON, "Harpoon" },
     { DIS_PDU_EMITTERNAME_HAT_BOX, " Hat Box" },
     { DIS_PDU_EMITTERNAME_HAWK_SCREECH, " Hawk Screech" },
     { DIS_PDU_EMITTERNAME_HAY_POLE, " Hay Pole" },
     { DIS_PDU_EMITTERNAME_HAY_RICK, " Hay Rick" },
     { DIS_PDU_EMITTERNAME_HEAD_LIGHT_A, " Head Light A" },
     { DIS_PDU_EMITTERNAME_HEAD_LIGHTS, " Head Lights" },
     { DIS_PDU_EMITTERNAME_HEAD_LIGHTS_C, " Head Lights C" },
     { DIS_PDU_EMITTERNAME_HEAD_LIGHTS_MG_A, " Head Lights MG A" },
     { DIS_PDU_EMITTERNAME_HEAD_LIGHTS_MG_B, " Head Lights MG B" },
     { DIS_PDU_EMITTERNAME_HEAD_LIGHTS_TT, " Head Lights TT" },
     { DIS_PDU_EMITTERNAME_HEAD_NET, " Head Net" },
     { DIS_PDU_EMITTERNAME_HEART_ACHE_B, " Heart Ache B" },
     { DIS_PDU_EMITTERNAME_HELLFIRE_MMW_MH, "Hellfire mmW MH" },
     { DIS_PDU_EMITTERNAME_HEN_EGG, " Hen Egg" },
     { DIS_PDU_EMITTERNAME_HEN_HOUSE, " Hen House" },
     { DIS_PDU_EMITTERNAME_HEN_NEST, " Hen Nest" },
     { DIS_PDU_EMITTERNAME_HEN_ROOST, " Hen Roost" },
     { DIS_PDU_EMITTERNAME_HERAKLES, " Herakles" },
     { DIS_PDU_EMITTERNAME_HF_2_MG, "HF-2 MG" },
     { DIS_PDU_EMITTERNAME_HGR_105, "HGR-105" },
     { DIS_PDU_EMITTERNAME_HERZ_9_TAR, "Herz-9 TAR" },
     { DIS_PDU_EMITTERNAME_HERZ_9_TTR, "Herz-9 TTR" },
     { DIS_PDU_EMITTERNAME_HERZ_9_MG, "Herz-9 MG" },
     { DIS_PDU_EMITTERNAME_HIGH_BRICK, " High Brick" },
     { DIS_PDU_EMITTERNAME_HIGH_FIX, " High Fix" },
     { DIS_PDU_EMITTERNAME_YLC_2V_HIGH_GUARD, "YLC-2V High Guard" },
     { DIS_PDU_EMITTERNAME_HIGH_LARK_TI, " High Lark TI" },
     { DIS_PDU_EMITTERNAME_HIGH_LARK_1, " High Lark 1" },
     { DIS_PDU_EMITTERNAME_HIGH_LARK_2, " High Lark 2" },
     { DIS_PDU_EMITTERNAME_HIGH_LARK_4, " High Lark 4" },
     { DIS_PDU_EMITTERNAME_HIGH_LUNE, " High Lune" },
     { DIS_PDU_EMITTERNAME_HIGH_POLE_A_B, " High Pole A&B" },
     { DIS_PDU_EMITTERNAME_HIGH_SCOOP, " High Scoop" },
     { DIS_PDU_EMITTERNAME_9S19MT_HIGH_SCREEN, "9S19MT High Screen" },
     { DIS_PDU_EMITTERNAME_HIGH_SIEVE, " High Sieve" },
     { DIS_PDU_EMITTERNAME_HILL_BLOCK, " Hill Block" },
     { DIS_PDU_EMITTERNAME_HIMALAYAS_COUNTERMEASURES_SUITE, "Himalayas Countermeasures Suite" },
     { DIS_PDU_EMITTERNAME_HG_9550, " HG-9550" },
     { DIS_PDU_EMITTERNAME_HJ_6374, "HJ-6374" },
     { DIS_PDU_EMITTERNAME_HLD_RADAR_900, " HLD-RADAR 900" },
     { DIS_PDU_EMITTERNAME_HLJQ_520, "HLJQ-520" },
     { DIS_PDU_EMITTERNAME_HN_503, "HN-503" },
     { DIS_PDU_EMITTERNAME_HN_C03_M, "HN-C03-M" },
     { DIS_PDU_EMITTERNAME_HOME_TALK, " Home Talk" },
     { DIS_PDU_EMITTERNAME_HORN_SPOON, " Horn Spoon" },
     { DIS_PDU_EMITTERNAME_HOT_BRICK, " Hot Brick" },
     { DIS_PDU_EMITTERNAME_HOT_FLASH, " Hot Flash" },
     { DIS_PDU_EMITTERNAME_HOT_FLASH_2, " Hot Flash 2" },
     { DIS_PDU_EMITTERNAME_IHS_6, "IHS-6" },
     { DIS_PDU_EMITTERNAME_IRL144M_HOT_SHOT_TA, "IRL144M Hot Shot TA" },
     { DIS_PDU_EMITTERNAME_IRL144M_HOT_SHOT_TT, "IRL144M Hot Shot TT" },
     { DIS_PDU_EMITTERNAME_IRL144M_HOT_SHOT_MG, "IRL144M Hot Shot MG" },
     { DIS_PDU_EMITTERNAME_HPS_106, "HPS-106" },
     { DIS_PDU_EMITTERNAME_HPS_104, "HPS-104" },
     { DIS_PDU_EMITTERNAME_HQ_9_MH, "HQ-9 MH" },
     { DIS_PDU_EMITTERNAME_HSR_1128, " HSR-1128" },
     { DIS_PDU_EMITTERNAME_HQ_9A_TER, "HQ-9A TER" },
     { DIS_PDU_EMITTERNAME_HQ_9B_TER, "HQ-9B TER" },
     { DIS_PDU_EMITTERNAME_HT_233, "HT-233" },
     { DIS_PDU_EMITTERNAME_HQ_61, "HQ-61" },
     { DIS_PDU_EMITTERNAME_HRJS, "HRJS" },
     { DIS_PDU_EMITTERNAME_I_DERBY_ER, "I-Derby ER" },
     { DIS_PDU_EMITTERNAME_IBIS_80, "IBIS-80" },
     { DIS_PDU_EMITTERNAME_IBIS_150, "IBIS-150" },
     { DIS_PDU_EMITTERNAME_IBIS_200, "IBIS-200" },
     { DIS_PDU_EMITTERNAME_HQ_16_TER, "HQ-16 TER" },
     { DIS_PDU_EMITTERNAME_IFF_MK_XII_AIMS_UPX_29, "IFF MK XII AIMS UPX 29" },
     { DIS_PDU_EMITTERNAME_JANET, " Janet" },
     { DIS_PDU_EMITTERNAME_IFF_MK_XV, "IFF MK XV" },
     { DIS_PDU_EMITTERNAME_IFF_INT, "IFF INT" },
     { DIS_PDU_EMITTERNAME_JACK_KNIFE, " Jack Knife" },
     { DIS_PDU_EMITTERNAME_IFF_TRSP, "IFF TRSP" },
     { DIS_PDU_EMITTERNAME_J_MUSIC_ELBIT_SYSTEMS_JAMMER, "J-MUSIC Elbit Systems Jammer" },
     { DIS_PDU_EMITTERNAME_JAVELIN_MG, "Javelin MG" },
     { DIS_PDU_EMITTERNAME_IGLA_1_SLAR, "Igla-1 SLAR" },
     { DIS_PDU_EMITTERNAME_IR_SA_2B_TER, " IR-SA-2B TER" },
     { DIS_PDU_EMITTERNAME_IR_SA_4_TER, " IR-SA-4 TER" },
     { DIS_PDU_EMITTERNAME_IR_SA_5_TELAR_TER, " IR-SA-5 TELAR TER" },
     { DIS_PDU_EMITTERNAME_J_10B_PESA, "J-10B PESA" },
     { DIS_PDU_EMITTERNAME_J_10C_AESA, "J-10C AESA" },
     { DIS_PDU_EMITTERNAME_J_11D_AESA, "J-11D AESA" },
     { DIS_PDU_EMITTERNAME_JL_10MP, "JL-10MP" },
     { DIS_PDU_EMITTERNAME_J_ALQ_8, "J/ALQ-8" },
     { DIS_PDU_EMITTERNAME_J_FPS_7, "J/FPS-7" },
     { DIS_PDU_EMITTERNAME_JAY_BIRD, " Jay Bird" },
     { DIS_PDU_EMITTERNAME_J_FPS_3, "J/FPS-3" },
     { DIS_PDU_EMITTERNAME_JH_10_MIX_BOWL, "JH-10 Mix Bowl" },
     { DIS_PDU_EMITTERNAME_J_MPQ_P7, "J/MPQ-P7" },
     { DIS_PDU_EMITTERNAME_JL_7, "JL-7" },
     { DIS_PDU_EMITTERNAME_JL_10B, "JL-10B" },
     { DIS_PDU_EMITTERNAME_JMA_1576, "JMA 1576" },
     { DIS_PDU_EMITTERNAME_JRC_JMA_9252_6CA, "JRC JMA-9252-6CA" },
     { DIS_PDU_EMITTERNAME_JLP_40, "JLP-40" },
     { DIS_PDU_EMITTERNAME_JRC_JMR_9200_SERIES_X, "JRC JMR-9200 Series X" },
     { DIS_PDU_EMITTERNAME_JRC_NMD_401, "JRC-NMD-401" },
     { DIS_PDU_EMITTERNAME_JRC_JRM_310_MK2, "JRC JRM 310 MK2" },
     { DIS_PDU_EMITTERNAME_JMA_1596, "JMA 1596" },
     { DIS_PDU_EMITTERNAME_JN_1104, "JN-1104" },
     { DIS_PDU_EMITTERNAME_JMA_7000, "JMA 7000" },
     { DIS_PDU_EMITTERNAME_JRC_JMA_7700, "JRC JMA 7700" },
     { DIS_PDU_EMITTERNAME_JMA_5320, "JMA 5320" },
     { DIS_PDU_EMITTERNAME_JRC_JMR_9210_6XC, "JRC JMR-9210-6XC" },
     { DIS_PDU_EMITTERNAME_JERS_1, "JERS-1" },
     { DIS_PDU_EMITTERNAME_JINDALEE, "JINDALEE" },
     { DIS_PDU_EMITTERNAME_JRC_JMA_9900_SERIES, "JRC JMA-9900 series" },
     { DIS_PDU_EMITTERNAME_JLP_40D, "JLP-40D" },
     { DIS_PDU_EMITTERNAME_JRC_JMA_5300_SERIES, "JRC JMA-5300 series" },
     { DIS_PDU_EMITTERNAME_JUPITER, "Jupiter" },
     { DIS_PDU_EMITTERNAME_JUPITER_II, "Jupiter II" },
     { DIS_PDU_EMITTERNAME_JY_8, "JY-8" },
     { DIS_PDU_EMITTERNAME_JY_8A, "JY-8A" },
     { DIS_PDU_EMITTERNAME_JY_9, "JY-9" },
     { DIS_PDU_EMITTERNAME_JY_9_MODIFIED, "JY-9 Modified" },
     { DIS_PDU_EMITTERNAME_JY_11_EW, "JY-11 EW" },
     { DIS_PDU_EMITTERNAME_JY_14, "JY-14" },
     { DIS_PDU_EMITTERNAME_JY_14A, "JY-14A" },
     { DIS_PDU_EMITTERNAME_JY_16, "JY-16" },
     { DIS_PDU_EMITTERNAME_JY_24, "JY-24" },
     { DIS_PDU_EMITTERNAME_J_APG_1, "J/APG-1" },
     { DIS_PDU_EMITTERNAME_J_APG_2, "J/APG-2" },
     { DIS_PDU_EMITTERNAME_JY_29, "JY-29" },
     { DIS_PDU_EMITTERNAME_JYL_1, "JYL-1" },
     { DIS_PDU_EMITTERNAME_JYL_6, "JYL-6" },
     { DIS_PDU_EMITTERNAME_JYL_6A, "JYL-6A" },
     { DIS_PDU_EMITTERNAME_JZ_QF_612, "JZ/QF-612" },
     { DIS_PDU_EMITTERNAME_K376Z, " K376Z" },
     { DIS_PDU_EMITTERNAME_K77M, "K77M" },
     { DIS_PDU_EMITTERNAME_KAIGE, "Kaige" },
     { DIS_PDU_EMITTERNAME_KALKAN, "KALKAN" },
     { DIS_PDU_EMITTERNAME_KBP_AFGANIT, "KBP Afganit" },
     { DIS_PDU_EMITTERNAME_KALKAN_II, "KALKAN II" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_2A, " Kelvin Hughes 2A" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_14_9, " Kelvin Hughes 14/9" },
     { DIS_PDU_EMITTERNAME_KARPATY, " Karpaty" },
     { DIS_PDU_EMITTERNAME_KASHTAN_3_JAMMING_SYSTEM, "Kashtan-3 Jamming System" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_TYPE_1006, " Kelvin Hughes type 1006" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_TYPE_1007, " Kelvin Hughes type 1007" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_TYPE_1007_F_BAND, " Kelvin Hughes Type 1007 F Band" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_2007_F_BAND, " Kelvin Hughes 2007 F Band" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_2007_I_BAND, " Kelvin Hughes 2007 I Band" },
     { DIS_PDU_EMITTERNAME_KH_NUCLEUS_5000, " KH Nucleus 5000" },
     { DIS_PDU_EMITTERNAME_KH_MANTA, " KH MANTA" },
     { DIS_PDU_EMITTERNAME_KH_NUCLEUS_2_6000, " KH NUCLEUS 2 6000" },
     { DIS_PDU_EMITTERNAME_KH_NUCLEUS_3_5000, " KH NUCLEUS 3 5000" },
     { DIS_PDU_EMITTERNAME_KH_NUCLEUS_3_6000A, " KH NUCLEUS 3 6000A" },
     { DIS_PDU_EMITTERNAME_KH_FAMILY, "KH Family" },
     { DIS_PDU_EMITTERNAME_KELVIN_HUGHES_6000A, " Kelvin Hughes 6000A" },
     { DIS_PDU_EMITTERNAME_KH_38MAE_MH, "Kh-38MAE MH" },
     { DIS_PDU_EMITTERNAME_KG8605A, "KG8605A" },
     { DIS_PDU_EMITTERNAME_KH_902M, "KH-902M" },
     { DIS_PDU_EMITTERNAME_KHOROM_K, "KHOROM-K" },
     { DIS_PDU_EMITTERNAME_KHIBINY, "KHIBINY" },
     { DIS_PDU_EMITTERNAME_KG300E, "KG300E" },
     { DIS_PDU_EMITTERNAME_KH_SHARPEYE, " KH SharpEye" },
     { DIS_PDU_EMITTERNAME_KH_SHARP_EYE_B, " KH Sharp Eye B" },
     { DIS_PDU_EMITTERNAME_KH_SHARP_EYE_C, " KH Sharp Eye C" },
     { DIS_PDU_EMITTERNAME_KH_1700, "KH 1700" },
     { DIS_PDU_EMITTERNAME_KING_PIN, " King Pin" },
     { DIS_PDU_EMITTERNAME_3RD_KHORDAD_TELAR_TIR, "3rd Khordad TELAR TIR" },
     { DIS_PDU_EMITTERNAME_3RD_KHORDAD_TAR, "3rd Khordad TAR" },
     { DIS_PDU_EMITTERNAME_15TH_KHORDAD_TER, "15th Khordad TER" },
     { DIS_PDU_EMITTERNAME_KG_300, "KG-300" },
     { DIS_PDU_EMITTERNAME_KITE_SCREECH, " Kite Screech" },
     { DIS_PDU_EMITTERNAME_KITE_SCREECH_A, " Kite Screech A" },
     { DIS_PDU_EMITTERNAME_KITE_SCREECH_B, " Kite Screech B" },
     { DIS_PDU_EMITTERNAME_KLC_3B, "KLC-3B" },
     { DIS_PDU_EMITTERNAME_KJ_500_NANJING_RADAR, "KJ-500 Nanjing Radar" },
     { DIS_PDU_EMITTERNAME_KIVACH, " Kivach" },
     { DIS_PDU_EMITTERNAME_KJ_500_JAMMER, "KJ-500 Jammer" },
     { DIS_PDU_EMITTERNAME_KLC_1, "KLC-1" },
     { DIS_PDU_EMITTERNAME_KLJ_1, "KLJ-1" },
     { DIS_PDU_EMITTERNAME_KLJ_3_TYPE_1473, "KLJ-3 (Type 1473)" },
     { DIS_PDU_EMITTERNAME_KLJ_4, "KLJ-4" },
     { DIS_PDU_EMITTERNAME_KLJ_4B, "KLJ-4B" },
     { DIS_PDU_EMITTERNAME_KLJ_5, "KLJ-5" },
     { DIS_PDU_EMITTERNAME_KLJ_7, "KLJ-7" },
     { DIS_PDU_EMITTERNAME_KLJ_7B, "KLJ-7B" },
     { DIS_PDU_EMITTERNAME_KLJ_7A, "KLJ-7A" },
     { DIS_PDU_EMITTERNAME_KNIFE_REST, " Knife Rest" },
     { DIS_PDU_EMITTERNAME_P_10_KNIFE_REST_B, "P-10 Knife Rest B" },
     { DIS_PDU_EMITTERNAME_KNIFE_REST_C, " KNIFE REST C" },
     { DIS_PDU_EMITTERNAME_KODEN_MD_3730, " Koden MD 3730" },
     { DIS_PDU_EMITTERNAME_KJ_2000, "KJ-2000" },
     { DIS_PDU_EMITTERNAME_KODEN_MDC_900, " KODEN MDC-900" },
     { DIS_PDU_EMITTERNAME_KOOPOL, "Koopol" },
     { DIS_PDU_EMITTERNAME_KOPYO_I, "KOPYO-I" },
     { DIS_PDU_EMITTERNAME_KR_75, "KR-75" },
     { DIS_PDU_EMITTERNAME_KRONOS, "KRONOS" },
     { DIS_PDU_EMITTERNAME_KREDO_1E, "KREDO-1E" },
     { DIS_PDU_EMITTERNAME_KRASUKHA_2, "Krasukha-2" },
     { DIS_PDU_EMITTERNAME_KRONOS_GRAND_NAVAL, "KRONOS GRAND NAVAL" },
     { DIS_PDU_EMITTERNAME_KRM_66E, "KRM-66E" },
     { DIS_PDU_EMITTERNAME_KRTZ_125_2M, "KRTZ-125-2M" },
     { DIS_PDU_EMITTERNAME_KSA_SRN, "KSA SRN" },
     { DIS_PDU_EMITTERNAME_KSA_TSR, "KSA TSR" },
     { DIS_PDU_EMITTERNAME_KS_1A_PHASED_ARRAY, "KS-1A PHASED ARRAY" },
     { DIS_PDU_EMITTERNAME_KS418, "KS418" },
     { DIS_PDU_EMITTERNAME_KS418E, "KS418E" },
     { DIS_PDU_EMITTERNAME_KZ100, "KZ100" },
     { DIS_PDU_EMITTERNAME_KZ900, "KZ900" },
     { DIS_PDU_EMITTERNAME_L175V, "L175V" },
     { DIS_PDU_EMITTERNAME_L370_5_PRESIDENT_S_JAMMER, "L370-5 President-S Jammer" },
     { DIS_PDU_EMITTERNAME_L_415, "L-415" },
     { DIS_PDU_EMITTERNAME_L_88, "L-88" },
     { DIS_PDU_EMITTERNAME_LAADS, "LAADS" },
     { DIS_PDU_EMITTERNAME_LAND_FALL, " Land Fall" },
     { DIS_PDU_EMITTERNAME_LAND_ROLL_MG, " Land Roll MG" },
     { DIS_PDU_EMITTERNAME_LAND_ROLL_TA, " Land Roll TA" },
     { DIS_PDU_EMITTERNAME_LAND_ROLL_TT, " Land Roll TT" },
     { DIS_PDU_EMITTERNAME_LAZUR, "LAZUR" },
     { DIS_PDU_EMITTERNAME_MODEL_791_A_LAST_STRAW, "Model 791-A Last Straw" },
     { DIS_PDU_EMITTERNAME_LAP_3000, "LAP-3000" },
     { DIS_PDU_EMITTERNAME_LC_150, "LC-150" },
     { DIS_PDU_EMITTERNAME_LEER_3, "LEER-3" },
     { DIS_PDU_EMITTERNAME_LEG_DRIVE, " Leg Drive" },
     { DIS_PDU_EMITTERNAME_LENINETZ_V_004, "Leninetz V-004" },
     { DIS_PDU_EMITTERNAME_LENINGRAF, " Leningraf" },
     { DIS_PDU_EMITTERNAME_LIANA, "LIANA" },
     { DIS_PDU_EMITTERNAME_LIGHT_BULB, " Light Bulb" },
     { DIS_PDU_EMITTERNAME_LIRA_A10, "LIRA-A10" },
     { DIS_PDU_EMITTERNAME_LIROD_8, "LIROD 8" },
     { DIS_PDU_EMITTERNAME_LIROD_MK_II, "LIROD MK II" },
     { DIS_PDU_EMITTERNAME_LLX05K, "LLX05K" },
     { DIS_PDU_EMITTERNAME_LMT_NRAI_6A, "LMT NRAI-6A" },
     { DIS_PDU_EMITTERNAME_LN_55, "LN 55" },
     { DIS_PDU_EMITTERNAME_LN_66, " Ln 66" },
     { DIS_PDU_EMITTERNAME_LIMAN, " Liman" },
     { DIS_PDU_EMITTERNAME_LIMAN_2, " Liman 2" },
     { DIS_PDU_EMITTERNAME_LOCKHEED_VIGILANCE, "Lockheed Vigilance" },
     { DIS_PDU_EMITTERNAME_LONG_BOW, " Long Bow" },
     { DIS_PDU_EMITTERNAME_LONG_BRICK, " Long Brick" },
     { DIS_PDU_EMITTERNAME_LONG_BULL, " Long Bull" },
     { DIS_PDU_EMITTERNAME_LONG_EYE, " Long Eye" },
     { DIS_PDU_EMITTERNAME_LONG_HEAD, " Long Head" },
     { DIS_PDU_EMITTERNAME_LONG_TALK, " Long Talk" },
     { DIS_PDU_EMITTERNAME_LONG_TRACK, " Long Track" },
     { DIS_PDU_EMITTERNAME_LONG_TROUGH, " Long Trough" },
     { DIS_PDU_EMITTERNAME_LOOK_TWO, " Look Two" },
     { DIS_PDU_EMITTERNAME_LOPAR, "LOPAR" },
     { DIS_PDU_EMITTERNAME_LORAN, "LORAN" },
     { DIS_PDU_EMITTERNAME_LOW_BLOW_TA, " Low Blow TA" },
     { DIS_PDU_EMITTERNAME_LOW_BLOW_TT, " Low Blow TT" },
     { DIS_PDU_EMITTERNAME_LOW_BLOW_MG, " Low Blow MG" },
     { DIS_PDU_EMITTERNAME_LOW_JACK_TT, " Low Jack TT" },
     { DIS_PDU_EMITTERNAME_LOW_JACK_MG, " Low Jack MG" },
     { DIS_PDU_EMITTERNAME_LOW_SIEVE, " Low Sieve" },
     { DIS_PDU_EMITTERNAME_LOW_TROUGH, " Low Trough" },
     { DIS_PDU_EMITTERNAME_LR66, "LR66" },
     { DIS_PDU_EMITTERNAME_LRA_900, "LRA-900" },
     { DIS_PDU_EMITTERNAME_TRS_2050_LP_23, "TRS-2050 LP-23" },
     { DIS_PDU_EMITTERNAME_LW_01, "LW 01" },
     { DIS_PDU_EMITTERNAME_M_1983_FCR, " M-1983 FCR" },
     { DIS_PDU_EMITTERNAME_M22_40, "M22-40" },
     { DIS_PDU_EMITTERNAME_M44, "M44" },
     { DIS_PDU_EMITTERNAME_M401Z, " M401Z" },
     { DIS_PDU_EMITTERNAME_M585Z, " M585Z" },
     { DIS_PDU_EMITTERNAME_M588Z, " M588Z" },
     { DIS_PDU_EMITTERNAME_MA_1_IFF_PORTION, "MA 1 IFF Portion" },
     { DIS_PDU_EMITTERNAME_MAD_HACK, "MAD HACK" },
     { DIS_PDU_EMITTERNAME_MARELD, " MARELD" },
     { DIS_PDU_EMITTERNAME_MA_TYPE_909_, " MA Type 909#" },
     { DIS_PDU_EMITTERNAME_MARC_S_152, "MARC S-152" },
     { DIS_PDU_EMITTERNAME_MARCONI_1810, " Marconi 1810" },
     { DIS_PDU_EMITTERNAME_MARCONI_CANADA_HC_75, " Marconi Canada HC 75" },
     { DIS_PDU_EMITTERNAME_MARCONI_S_713, " Marconi S 713" },
     { DIS_PDU_EMITTERNAME_MARCONI_S_1802, " Marconi S 1802" },
     { DIS_PDU_EMITTERNAME_MARCONI_S247, " Marconi S247" },
     { DIS_PDU_EMITTERNAME_MARCONI_S_810, " Marconi S 810" },
     { DIS_PDU_EMITTERNAME_MARCONI_SA_10, " Marconi SA 10" },
     { DIS_PDU_EMITTERNAME_MARCONI_ST801, "MARCONI ST801" },
     { DIS_PDU_EMITTERNAME_MARCONI_ST805, " Marconi ST805" },
     { DIS_PDU_EMITTERNAME_MARCONI_TYPE_967, " Marconi type 967" },
     { DIS_PDU_EMITTERNAME_MARCONI_TYPE_968, " Marconi type 968" },
     { DIS_PDU_EMITTERNAME_MARCONI_TYPE_992, " Marconi type 992" },
     { DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_1022, " Marconi/signaal type 1022" },
     { DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_910, " Marconi/signaal type 910" },
     { DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_911, " Marconi/signaal type 911" },
     { DIS_PDU_EMITTERNAME_MARCONI_SIGNAAL_TYPE_992R, " Marconi/signaal type 992R" },
     { DIS_PDU_EMITTERNAME_MARTELLO_743D, "MARTELLO 743D" },
     { DIS_PDU_EMITTERNAME_MARTELLO_S_723A, "MARTELLO S-723A" },
     { DIS_PDU_EMITTERNAME_MASTER_A, "MASTER-A" },
     { DIS_PDU_EMITTERNAME_MBDA_FLAADS_M_SEA_CEPTOR_JAMMER, "MBDA FLAADS-M (Sea Ceptor) Jammer" },
     { DIS_PDU_EMITTERNAME_MELCO_3, "MELCO-3" },
     { DIS_PDU_EMITTERNAME_MELODI, "MELODI" },
     { DIS_PDU_EMITTERNAME_MERLIN, "MERLIN" },
     { DIS_PDU_EMITTERNAME_MERAJ_4_ASCENSION, "Meraj-4 (Ascension)" },
     { DIS_PDU_EMITTERNAME_NORTHROP_GRUMMAN_MESA, " Northrop Grumman MESA" },
     { DIS_PDU_EMITTERNAME_MESH_BRICK, " Mesh Brick" },
     { DIS_PDU_EMITTERNAME_METEOR_1500S, "METEOR 1500S" },
     { DIS_PDU_EMITTERNAME_METEOR_200, "METEOR 200" },
     { DIS_PDU_EMITTERNAME_METEOR_50DX, "METEOR 50DX" },
     { DIS_PDU_EMITTERNAME_METEOR_300, "METEOR 300" },
     { DIS_PDU_EMITTERNAME_METEOR_BVRAAM, "Meteor BVRAAM" },
     { DIS_PDU_EMITTERNAME_MFR, "MFR" },
     { DIS_PDU_EMITTERNAME_MFSR_2100_45, "MFSR 2100/45" },
     { DIS_PDU_EMITTERNAME_MICA_MH, "MICA MH" },
     { DIS_PDU_EMITTERNAME_MICA_RF, "MICA-RF" },
     { DIS_PDU_EMITTERNAME_MINERAL_ME, "Mineral-ME" },
     { DIS_PDU_EMITTERNAME_MIRAGE_ILL, "Mirage ILL" },
     { DIS_PDU_EMITTERNAME_MIYSIS_JAMMER, "Miysis Jammer" },
     { DIS_PDU_EMITTERNAME_MK_15_PHALANX_BLK_0, "MK 15 (Phalanx BLK 0)" },
     { DIS_PDU_EMITTERNAME_MK_15_CIWS, " MK 15 CIWS" },
     { DIS_PDU_EMITTERNAME_MK_23, "MK-23" },
     { DIS_PDU_EMITTERNAME_MK_23_TAS, "MK 23 TAS" },
     { DIS_PDU_EMITTERNAME_MK_25, "MK 25" },
     { DIS_PDU_EMITTERNAME_MK_25_MOD_3, "Mk-25 Mod-3" },
     { DIS_PDU_EMITTERNAME_MK_25_MOD_7, "Mk 25 Mod 7" },
     { DIS_PDU_EMITTERNAME_MK_35_M2, "MK-35 M2" },
     { DIS_PDU_EMITTERNAME_MK_92, "MK 92" },
     { DIS_PDU_EMITTERNAME_MK_92_CAS, "MK-92 CAS" },
     { DIS_PDU_EMITTERNAME_MK_92_STIR, "MK-92 STIR" },
     { DIS_PDU_EMITTERNAME_MK_95, "MK 95" },
     { DIS_PDU_EMITTERNAME_MKS_818, "MKS-818" },
     { DIS_PDU_EMITTERNAME_MLA_1, "MLA-1" },
     { DIS_PDU_EMITTERNAME_MM_APQ_706, "MM/APQ-706" },
     { DIS_PDU_EMITTERNAME_MM_950, "MM 950" },
     { DIS_PDU_EMITTERNAME_MM_APS_705, "MM APS 705" },
     { DIS_PDU_EMITTERNAME_MM_APS_784, "MM/APS-784" },
     { DIS_PDU_EMITTERNAME_MM_SPG_73_RTN_12X, "MM/SPG-73 (RTN-12X)" },
     { DIS_PDU_EMITTERNAME_MM_SPG_74, "MM SPG 74" },
     { DIS_PDU_EMITTERNAME_MM_SPG_75, "MM SPG 75" },
     { DIS_PDU_EMITTERNAME_MM_SPN_703, "MM SPN 703" },
     { DIS_PDU_EMITTERNAME_MM_SPN_730, "MM SPN 730" },
     { DIS_PDU_EMITTERNAME_MM_SPN_753B, "MM SPN-753B" },
     { DIS_PDU_EMITTERNAME_MM_SPQ_3, "MM/SPQ-3" },
     { DIS_PDU_EMITTERNAME_MM_SPS_702, "MM SPS 702" },
     { DIS_PDU_EMITTERNAME_MM_SPS_768, "MM SPS 768" },
     { DIS_PDU_EMITTERNAME_MM_SPS_774, "MM SPS 774" },
     { DIS_PDU_EMITTERNAME_MM_SPS_791_RAN_30X, "MM/SPS-791 (RAN-30X)" },
     { DIS_PDU_EMITTERNAME_MM_SPS_794_RAN_21S, "MM SPS-794 (RAN-21S)" },
     { DIS_PDU_EMITTERNAME_MM_SPS_798_RAN_40L, "MM/SPS-798 (RAN-40L)" },
     { DIS_PDU_EMITTERNAME_MMSR, "MMSR" },
     { DIS_PDU_EMITTERNAME_MODEL_17C, "Model-17C" },
     { DIS_PDU_EMITTERNAME_MOON_4, "Moon 4" },
     { DIS_PDU_EMITTERNAME_MOON_PIE, " Moon Pie" },
     { DIS_PDU_EMITTERNAME_MOON_CONE, "MOON CONE" },
     { DIS_PDU_EMITTERNAME_MOON_RACK, " Moon Rack" },
     { DIS_PDU_EMITTERNAME_MOON_FACE, "MOON FACE" },
     { DIS_PDU_EMITTERNAME_MMRS, " MMRS" },
     { DIS_PDU_EMITTERNAME_MOON_SHADE, " Moon Shade" },
     { DIS_PDU_EMITTERNAME_MODEL_360, "Model 360" },
     { DIS_PDU_EMITTERNAME_MODEL_378, "Model 378" },
     { DIS_PDU_EMITTERNAME_MODEL_970, "Model-970" },
     { DIS_PDU_EMITTERNAME_MODEL_974, "Model 974" },
     { DIS_PDU_EMITTERNAME_MONOLIT_B, "MONOLIT-B" },
     { DIS_PDU_EMITTERNAME_MONUMENT, " Monument" },
     { DIS_PDU_EMITTERNAME_MOUSE, " Mouse" },
     { DIS_PDU_EMITTERNAME_MP_411_ESM, "MP-411 ESM" },
     { DIS_PDU_EMITTERNAME_MPDR_18_S, "MPDR 18/S" },
     { DIS_PDU_EMITTERNAME_MPDR_18_X, "MPDR 18 X" },
     { DIS_PDU_EMITTERNAME_MPDR_45_E, "MPDR 45/E" },
     { DIS_PDU_EMITTERNAME_MR_231_1, "MR-231-1" },
     { DIS_PDU_EMITTERNAME_MR_231_3, "MR-231-3" },
     { DIS_PDU_EMITTERNAME_MPR, "MPR" },
     { DIS_PDU_EMITTERNAME_MR_231_4, " MR-231-4" },
     { DIS_PDU_EMITTERNAME_MPS_1, "MPS-1" },
     { DIS_PDU_EMITTERNAME_MR_36B, " MR-36B" },
     { DIS_PDU_EMITTERNAME_MR_231_MOD, " MR-231 MOD" },
     { DIS_PDU_EMITTERNAME_MR_1600, "MR-1600" },
     { DIS_PDU_EMITTERNAME_MRR, "MRR" },
     { DIS_PDU_EMITTERNAME_MR35, "MR35" },
     { DIS_PDU_EMITTERNAME_MR36, "MR36" },
     { DIS_PDU_EMITTERNAME_MRL_1, "MRL-1" },
     { DIS_PDU_EMITTERNAME_MRL_4, "MRL-4" },
     { DIS_PDU_EMITTERNAME_MRL_5, "MRL-5" },
     { DIS_PDU_EMITTERNAME_MSAM, "MSAM" },
     { DIS_PDU_EMITTERNAME_MR_36A, "MR-36A" },
     { DIS_PDU_EMITTERNAME_MSTAR, "MSTAR" },
     { DIS_PDU_EMITTERNAME_MT_305X, "MT-305X" },
     { DIS_PDU_EMITTERNAME_MR_10M1E, "MR-10M1E" },
     { DIS_PDU_EMITTERNAME_MR_90, "MR-90" },
     { DIS_PDU_EMITTERNAME_MRK_411, "MRK-411" },
     { DIS_PDU_EMITTERNAME_MR_320M_TOPAZ_V, "MR-320M Topaz-V" },
     { DIS_PDU_EMITTERNAME_MSP_418K, "MSP-418K" },
     { DIS_PDU_EMITTERNAME_MUFF_COB, " Muff Cob" },
     { DIS_PDU_EMITTERNAME_MUSHROOM, "Mushroom" },
     { DIS_PDU_EMITTERNAME_MUSHROOM_1, "Mushroom 1" },
     { DIS_PDU_EMITTERNAME_MUSHROOM_2, "Mushroom 2" },
     { DIS_PDU_EMITTERNAME_MUSHROOM_3, " Mushroom 3" },
     { DIS_PDU_EMITTERNAME_N_23, "N-23" },
     { DIS_PDU_EMITTERNAME_N_011M_BARS, "N-011M Bars" },
     { DIS_PDU_EMITTERNAME_N_011M_BARS_B, "N-011M Bars-B" },
     { DIS_PDU_EMITTERNAME_N_011M_BARS_C, "N-011M Bars-C" },
     { DIS_PDU_EMITTERNAME_N_011M_BARS_R, "N-011M Bars-R" },
     { DIS_PDU_EMITTERNAME_N035_IRBIS_E, "N035 Irbis-E" },
     { DIS_PDU_EMITTERNAME_N036_BYELKA, "N036 Byelka" },
     { DIS_PDU_EMITTERNAME_N_25, "N-25" },
     { DIS_PDU_EMITTERNAME_N920Z, "N920Z" },
     { DIS_PDU_EMITTERNAME_N001V, "N001V" },
     { DIS_PDU_EMITTERNAME_N001VE, "N001VE" },
     { DIS_PDU_EMITTERNAME_N001VEP, "N001VEP" },
     { DIS_PDU_EMITTERNAME_NACOS_RADARPILOT_PLATINUM, "NACOS RADARPILOT Platinum" },
     { DIS_PDU_EMITTERNAME_NAMPO_B, " Nampo B" },
     { DIS_PDU_EMITTERNAME_NAGIRA, "NAGIRA" },
     { DIS_PDU_EMITTERNAME_NANJING_B, " Nanjing B" },
     { DIS_PDU_EMITTERNAME_NANJING_C, " Nanjing C" },
     { DIS_PDU_EMITTERNAME_NAYADA, "Nayada" },
     { DIS_PDU_EMITTERNAME_NAYADA_5M, "NAYADA-5M" },
     { DIS_PDU_EMITTERNAME_NAYADA_5PV, "NAYADA-5PV" },
     { DIS_PDU_EMITTERNAME_NEBO_M, "NEBO-M" },
     { DIS_PDU_EMITTERNAME_NEBO_SVU, "Nebo-SVU" },
     { DIS_PDU_EMITTERNAME_NEPTUN, "Neptun" },
     { DIS_PDU_EMITTERNAME_NETTUNO_4100, "Nettuno 4100" },
     { DIS_PDU_EMITTERNAME_NIKE_HERCULES_MTR, "NIKE HERCULES MTR" },
     { DIS_PDU_EMITTERNAME_NIKE_TT, " NIKE TT" },
     { DIS_PDU_EMITTERNAME_NORTHROP_GRUMMAN_MFEW_JAMMER, "Northrop Grumman MFEW Jammer" },
     { DIS_PDU_EMITTERNAME_NORINCO_3D, "NORINCO 3D" },
     { DIS_PDU_EMITTERNAME_NJ_81E, "NJ-81E" },
     { DIS_PDU_EMITTERNAME_NORMANDIE, "Normandie" },
     { DIS_PDU_EMITTERNAME_NRJ_6A, "NRJ-6A" },
     { DIS_PDU_EMITTERNAME_NOSTRADAMUS, "NOSTRADAMUS" },
     { DIS_PDU_EMITTERNAME_NPG_1240, "NPG-1240" },
     { DIS_PDU_EMITTERNAME_NPG_1460, "NPG-1460" },
     { DIS_PDU_EMITTERNAME_NPG_434, "NPG-434" },
     { DIS_PDU_EMITTERNAME_NPG_630, "NPG-630" },
     { DIS_PDU_EMITTERNAME_NPM_510, "NPM-510" },
     { DIS_PDU_EMITTERNAME_NUT_CAN, " Nut Can" },
     { DIS_PDU_EMITTERNAME_NP_VEGA_LIANA, "NP Vega Liana" },
     { DIS_PDU_EMITTERNAME_NOVELLA_NV1_70, "Novella NV1.70" },
     { DIS_PDU_EMITTERNAME_NOVELLA_P_38, "Novella-P-38" },
     { DIS_PDU_EMITTERNAME_NRBA_50, "NRBA 50" },
     { DIS_PDU_EMITTERNAME_NRBA_51, "NRBA 51" },
     { DIS_PDU_EMITTERNAME_NRBF_20A, "NRBF 20A" },
     { DIS_PDU_EMITTERNAME_NRJ_5, "NRJ-5" },
     { DIS_PDU_EMITTERNAME_NS_9005, " NS-9005" },
     { DIS_PDU_EMITTERNAME_NS_100_SERIES, "NS-100 Series" },
     { DIS_PDU_EMITTERNAME_NUR_31, "NUR-31" },
     { DIS_PDU_EMITTERNAME_NWS_3, "NWS-3" },
     { DIS_PDU_EMITTERNAME_NYSA_B, " Nysa B" },
     { DIS_PDU_EMITTERNAME_O524A, " O524A" },
     { DIS_PDU_EMITTERNAME_O580B, " O580B" },
     { DIS_PDU_EMITTERNAME_O625Z, " O625Z" },
     { DIS_PDU_EMITTERNAME_O626Z, " O626Z" },
     { DIS_PDU_EMITTERNAME_OCEAN_MASTER, "Ocean Master" },
     { DIS_PDU_EMITTERNAME_OCEAN_MASTER_400, " Ocean Master 400" },
     { DIS_PDU_EMITTERNAME_ODD_GROUP, " Odd Group" },
     { DIS_PDU_EMITTERNAME_ODD_LOT, " Odd Lot" },
     { DIS_PDU_EMITTERNAME_ODD_PAIR, " Odd Pair" },
     { DIS_PDU_EMITTERNAME_ODD_RODS, " Odd Rods" },
     { DIS_PDU_EMITTERNAME_OKA, " Oka" },
     { DIS_PDU_EMITTERNAME_OFOGH, "OFOGH" },
     { DIS_PDU_EMITTERNAME_OFOGH_3, "OFOGH-3" },
     { DIS_PDU_EMITTERNAME_OKEAN, "OKEAN" },
     { DIS_PDU_EMITTERNAME_OKEAN_A, "OKEAN A" },
     { DIS_PDU_EMITTERNAME_OKINXE_12C, "OKINXE 12C" },
     { DIS_PDU_EMITTERNAME_OKO, "OKO" },
     { DIS_PDU_EMITTERNAME_OMEGA, "OMEGA" },
     { DIS_PDU_EMITTERNAME_OMERA_ORB32, "Omera ORB32" },
     { DIS_PDU_EMITTERNAME_OMUL, "OMUL" },
     { DIS_PDU_EMITTERNAME_ONE_EYE, " One Eye" },
     { DIS_PDU_EMITTERNAME_OP_28, "OP-28" },
     { DIS_PDU_EMITTERNAME_OPRL_4, "OPRL-4" },
     { DIS_PDU_EMITTERNAME_OPRM_71, "OPRM-71" },
     { DIS_PDU_EMITTERNAME_OPS_9, "OPS-9" },
     { DIS_PDU_EMITTERNAME_OPS_11_B_C, "OPS-11 B/C" },
     { DIS_PDU_EMITTERNAME_OPS_12, "OPS-12" },
     { DIS_PDU_EMITTERNAME_OPS_14B, "OPS-14B" },
     { DIS_PDU_EMITTERNAME_OPS_14C, "OPS-14C" },
     { DIS_PDU_EMITTERNAME_OPS_16B, "OPS-16B" },
     { DIS_PDU_EMITTERNAME_OPS_18, "OPS-18" },
     { DIS_PDU_EMITTERNAME_OPS_19, "OPS-19" },
     { DIS_PDU_EMITTERNAME_OPS_20, "OPS-20" },
     { DIS_PDU_EMITTERNAME_OPS_22, "OPS-22" },
     { DIS_PDU_EMITTERNAME_OPS_24, "OPS-24" },
     { DIS_PDU_EMITTERNAME_OPS_28, "OPS-28" },
     { DIS_PDU_EMITTERNAME_OPS_28C, "OPS-28C" },
     { DIS_PDU_EMITTERNAME_OPS_39, "OPS-39" },
     { DIS_PDU_EMITTERNAME_OPTIMA_3_2, "OPTIMA 3.2" },
     { DIS_PDU_EMITTERNAME_OR_2, " OR-2" },
     { DIS_PDU_EMITTERNAME_ORB_31D, "ORB-31D" },
     { DIS_PDU_EMITTERNAME_ORB_31S, "ORB-31S" },
     { DIS_PDU_EMITTERNAME_ORB_32, "ORB 32" },
     { DIS_PDU_EMITTERNAME_ORB_42, "ORB-42" },
     { DIS_PDU_EMITTERNAME_ORION_RTN_10X, "Orion Rtn 10X" },
     { DIS_PDU_EMITTERNAME_SURFACE_WAVE_OVER_THE_HORIZON, "Surface Wave (Over The Horizon)" },
     { DIS_PDU_EMITTERNAME_OTOMAT_MK_1, "Otomat MK 1" },
     { DIS_PDU_EMITTERNAME_OTOMAT_MK_II_TESEO, "Otomat MK II Teseo" },
     { DIS_PDU_EMITTERNAME_OTOMAT_SERIES_AL, "Otomat Series AL" },
     { DIS_PDU_EMITTERNAME_OWL_SCREECH, " Owl Screech" },
     { DIS_PDU_EMITTERNAME_P360Z, "P360Z" },
     { DIS_PDU_EMITTERNAME_P_14, "P-14" },
     { DIS_PDU_EMITTERNAME_P_180U, "P-180U" },
     { DIS_PDU_EMITTERNAME_P_18_2, "P-18-2" },
     { DIS_PDU_EMITTERNAME_PA_1660, "PA-1660" },
     { DIS_PDU_EMITTERNAME_P_18M, "P-18M" },
     { DIS_PDU_EMITTERNAME_P_190U, "P-190U" },
     { DIS_PDU_EMITTERNAME_P_30, "P-30" },
     { DIS_PDU_EMITTERNAME_P_18_MOD, "P-18 MOD" },
     { DIS_PDU_EMITTERNAME_P_35M, "P-35M" },
     { DIS_PDU_EMITTERNAME_PAGE, "PAGE" },
     { DIS_PDU_EMITTERNAME_PAINT_BOX, " Paint Box" },
     { DIS_PDU_EMITTERNAME_PALM_FROND, " Palm Frond" },
     { DIS_PDU_EMITTERNAME_MODIFIED_PAINT_BOX, " Modified Paint Box" },
     { DIS_PDU_EMITTERNAME_PALM_FROND_AB, " Palm Frond AB" },
     { DIS_PDU_EMITTERNAME_PANDORA, "Pandora" },
     { DIS_PDU_EMITTERNAME_PALSAR_2, "PALSAR-2" },
     { DIS_PDU_EMITTERNAME_PANTSIR_SM_TAR, "Pantsir-SM TAR" },
     { DIS_PDU_EMITTERNAME_PAR_2, "PAR-2" },
     { DIS_PDU_EMITTERNAME_PANTSIR_S1_2RL80_TAR, "Pantsir-S1 2RL80 TAR" },
     { DIS_PDU_EMITTERNAME_PANTSIR_S1_1RS2_1_TT, "Pantsir-S1 1RS2-1 TT" },
     { DIS_PDU_EMITTERNAME_PAR_2000, "PAR-2000" },
     { DIS_PDU_EMITTERNAME_PAR_2090C, "PAR-2090C" },
     { DIS_PDU_EMITTERNAME_PAR_80, "PAR-80" },
     { DIS_PDU_EMITTERNAME_PAT_HAND_TT, " Pat Hand TT" },
     { DIS_PDU_EMITTERNAME_PAT_HAND_MG, " Pat Hand MG" },
     { DIS_PDU_EMITTERNAME_PATRIOT, "PATRIOT" },
     { DIS_PDU_EMITTERNAME_PATTY_CAKE, " Patty Cake" },
     { DIS_PDU_EMITTERNAME_PAWN_CAKE, " Pawn Cake" },
     { DIS_PDU_EMITTERNAME_PBR_4_RUBIN, "PBR 4 Rubin" },
     { DIS_PDU_EMITTERNAME_PCS_514, "PCS 514" },
     { DIS_PDU_EMITTERNAME_PEA_STICKS, " Pea Sticks" },
     { DIS_PDU_EMITTERNAME_PECHORA_SC, "Pechora SC" },
     { DIS_PDU_EMITTERNAME_PEEL_CONE, " Peel Cone" },
     { DIS_PDU_EMITTERNAME_PEEL_GROUP, " Peel Group" },
     { DIS_PDU_EMITTERNAME_PEEL_GROUP_A, " Peel Group A" },
     { DIS_PDU_EMITTERNAME_PEEL_GROUP_B, " Peel Group B" },
     { DIS_PDU_EMITTERNAME_PEEL_GROUP_MG, " Peel Group MG" },
     { DIS_PDU_EMITTERNAME_PEEL_PAIR, " Peel Pair" },
     { DIS_PDU_EMITTERNAME_PELENA, " Pelena" },
     { DIS_PDU_EMITTERNAME_PGZ07, "PGZ07" },
     { DIS_PDU_EMITTERNAME_PHALANX, "Phalanx" },
     { DIS_PDU_EMITTERNAME_PHAZOTRON_GUKOL_4, "Phazotron Gukol-4" },
     { DIS_PDU_EMITTERNAME_PHAZOTRON_ZHUK_A_AE, "Phazotron Zhuk-A/AE" },
     { DIS_PDU_EMITTERNAME_PHILIPS_9LV_200, " Philips 9LV 200" },
     { DIS_PDU_EMITTERNAME_PHILIPS_9LV_331, " Philips 9LV 331" },
     { DIS_PDU_EMITTERNAME_PHILIPS_LV_223, " Philips LV 223" },
     { DIS_PDU_EMITTERNAME_PHILIPS_SEA_GIRAFFE_50_HC, " Philips Sea Giraffe 50 HC" },
     { DIS_PDU_EMITTERNAME_PHIMAT_JAMMER, "Phimat Jammer" },
     { DIS_PDU_EMITTERNAME_PICOSAR, "PICOSAR" },
     { DIS_PDU_EMITTERNAME_PILOT_MK2, " PILOT MK2" },
     { DIS_PDU_EMITTERNAME_PIN_JIB, " Pin Jib" },
     { DIS_PDU_EMITTERNAME_PIN_TIP, " Pin Tip" },
     { DIS_PDU_EMITTERNAME_PL_11, "PL-11" },
     { DIS_PDU_EMITTERNAME_PL_12, "PL-12" },
     { DIS_PDU_EMITTERNAME_PL_15, "PL-15" },
     { DIS_PDU_EMITTERNAME_PLANK_SHAD, " Plank Shad" },
     { DIS_PDU_EMITTERNAME_PLANK_SHAVE, " Plank Shave" },
     { DIS_PDU_EMITTERNAME_PLANK_SHAVE_A, " Plank Shave A" },
     { DIS_PDU_EMITTERNAME_PLANK_SHAVE_B, " Plank Shave B" },
     { DIS_PDU_EMITTERNAME_PLATE_STEER, " Plate Steer" },
     { DIS_PDU_EMITTERNAME_PLESSEY_AWS_1, " Plessey AWS 1" },
     { DIS_PDU_EMITTERNAME_PLESSEY_AWS_2, " Plessey AWS-2" },
     { DIS_PDU_EMITTERNAME_PLESSEY_AWS_4, " Plessey AWS 4" },
     { DIS_PDU_EMITTERNAME_PLESSEY_AWS_6, " Plessey AWS 6" },
     { DIS_PDU_EMITTERNAME_PLESSEY_RJ, " Plessey RJ" },
     { DIS_PDU_EMITTERNAME_PLESSEY_TYPE_904, " Plessey Type 904" },
     { DIS_PDU_EMITTERNAME_PLESSEY_TYPE_996, " Plessey type 996" },
     { DIS_PDU_EMITTERNAME_PLESSEY_AWS_9, "Plessey AWS 9" },
     { DIS_PDU_EMITTERNAME_PLINTH_NET, " Plinth Net" },
     { DIS_PDU_EMITTERNAME_PLUTO, " Pluto" },
     { DIS_PDU_EMITTERNAME_PNA_B_RUBIN_DOWN_BEAT, "PNA-B Rubin / Down Beat" },
     { DIS_PDU_EMITTERNAME_POHJANPALO, "POHJANPALO" },
     { DIS_PDU_EMITTERNAME_POLIMENT_K, "Poliment-K" },
     { DIS_PDU_EMITTERNAME_POLLUX, "POLLUX" },
     { DIS_PDU_EMITTERNAME_POP_GROUP, " Pop Group" },
     { DIS_PDU_EMITTERNAME_POP_GROUP_MG, " Pop Group MG" },
     { DIS_PDU_EMITTERNAME_POP_GROUP_TA, " Pop Group TA" },
     { DIS_PDU_EMITTERNAME_POP_GROUP_TT, " Pop Group TT" },
     { DIS_PDU_EMITTERNAME_PORK_FIST, " Pork Fist" },
     { DIS_PDU_EMITTERNAME_PORK_TROUGH, " Pork Trough" },
     { DIS_PDU_EMITTERNAME_POZITIV_ME1_5P_26, "Pozitiv-ME1 5P-26" },
     { DIS_PDU_EMITTERNAME_POSITIVE_ME1_2, "Positive-ME1.2" },
     { DIS_PDU_EMITTERNAME_POZITIV_MK, "Pozitiv-MK" },
     { DIS_PDU_EMITTERNAME_POST_BOW, " Post Bow" },
     { DIS_PDU_EMITTERNAME_POST_LAMP, " Post Lamp" },
     { DIS_PDU_EMITTERNAME_POT_DRUM, " Pot Drum" },
     { DIS_PDU_EMITTERNAME_POT_HEAD, " Pot Head" },
     { DIS_PDU_EMITTERNAME_POT_SHOT, " Pot Shot" },
     { DIS_PDU_EMITTERNAME_PRAETORIAN_COUNTERMEASURES_SUITE, "Praetorian Countermeasures Suite" },
     { DIS_PDU_EMITTERNAME_PRIMUS_30A, "PRIMUS 30A" },
     { DIS_PDU_EMITTERNAME_PRIMUS_40_WXD, "PRIMUS 40 WXD" },
     { DIS_PDU_EMITTERNAME_PRIMUS_400, "Primus 400" },
     { DIS_PDU_EMITTERNAME_PRIMUS_300SL, "PRIMUS 300SL" },
     { DIS_PDU_EMITTERNAME_PRIMUS_500, "Primus 500" },
     { DIS_PDU_EMITTERNAME_PRIMUS_650, "Primus 650" },
     { DIS_PDU_EMITTERNAME_PRIMUS_700, "Primus 700" },
     { DIS_PDU_EMITTERNAME_PRIMUS_800, "PRIMUS 800" },
     { DIS_PDU_EMITTERNAME_PRIMUS_3000, " Primus 3000" },
     { DIS_PDU_EMITTERNAME_PRIMUS_870, "Primus 870" },
     { DIS_PDU_EMITTERNAME_PRORA, "PRORA" },
     { DIS_PDU_EMITTERNAME_PRS_2, "PRS-2" },
     { DIS_PDU_EMITTERNAME_PRS_3_ARGON_2, "PRS-3 Argon-2" },
     { DIS_PDU_EMITTERNAME_PRORA_PA_1660, "PRORA PA-1660" },
     { DIS_PDU_EMITTERNAME_PS_15, "PS-15" },
     { DIS_PDU_EMITTERNAME_PS_05A, "PS-05A" },
     { DIS_PDU_EMITTERNAME_PS_46_A, "PS 46 A" },
     { DIS_PDU_EMITTERNAME_PS_70_R, "PS 70 R" },
     { DIS_PDU_EMITTERNAME_PS_171_R, "PS-171/R" },
     { DIS_PDU_EMITTERNAME_PS_860, "PS-860" },
     { DIS_PDU_EMITTERNAME_PS_870, "PS-870" },
     { DIS_PDU_EMITTERNAME_PS_890, "PS-890" },
     { DIS_PDU_EMITTERNAME_PSM_33, "PSM-33" },
     { DIS_PDU_EMITTERNAME_PUFF_BALL, " Puff Ball" },
     { DIS_PDU_EMITTERNAME_QUADRADAR_VI, "Quadradar VI" },
     { DIS_PDU_EMITTERNAME_QW_1A, "QW-1A" },
     { DIS_PDU_EMITTERNAME_PHAZOTRON_1RS2_1E, "Phazotron 1RS2-1E" },
     { DIS_PDU_EMITTERNAME_PVS_200, "PVS-200" },
     { DIS_PDU_EMITTERNAME_PVS_2000, "PVS 2000" },
     { DIS_PDU_EMITTERNAME_R_330ZH, "R-330ZH" },
     { DIS_PDU_EMITTERNAME_R_045, "R 045" },
     { DIS_PDU_EMITTERNAME_R_76, "R-76" },
     { DIS_PDU_EMITTERNAME_R_934B, "R-934B" },
     { DIS_PDU_EMITTERNAME_RA_20, "RA-20" },
     { DIS_PDU_EMITTERNAME_RA723, "RA723" },
     { DIS_PDU_EMITTERNAME_R41XXX, "R41XXX" },
     { DIS_PDU_EMITTERNAME_RAC_3D, "RAC-3D" },
     { DIS_PDU_EMITTERNAME_RAC_30, " RAC-30" },
     { DIS_PDU_EMITTERNAME_R_423AM, "R-423AM" },
     { DIS_PDU_EMITTERNAME_RAAD_1_TER, "Raad-1 TER" },
     { DIS_PDU_EMITTERNAME_RAAD_2_TER, "Raad-2 TER" },
     { DIS_PDU_EMITTERNAME_RACAL_1229, " Racal 1229" },
     { DIS_PDU_EMITTERNAME_DECCA_1230, "DECCA 1230" },
     { DIS_PDU_EMITTERNAME_RACAL_AC_2690_BT, " Racal AC 2690 BT" },
     { DIS_PDU_EMITTERNAME_RACAL_DECCA_1216, " Racal Decca 1216" },
     { DIS_PDU_EMITTERNAME_RACAL_DECCA_20V90_9, "Racal-DECCA 20V90/9" },
     { DIS_PDU_EMITTERNAME_RACAL_DECCA_360, " Racal Decca 360" },
     { DIS_PDU_EMITTERNAME_RACAL_DECCA_AC_1290, " Racal Decca AC 1290" },
     { DIS_PDU_EMITTERNAME_RACAL_DECCA_TM_1229, " Racal Decca TM 1229" },
     { DIS_PDU_EMITTERNAME_RACAL_DECCA_TM_1626, " Racal Decca TM 1626" },
     { DIS_PDU_EMITTERNAME_RACAL_DRBN_34A, " Racal DRBN 34A" },
     { DIS_PDU_EMITTERNAME_RADA_MHR, "RADA MHR" },
     { DIS_PDU_EMITTERNAME_RADAR_24, " Radar 24" },
     { DIS_PDU_EMITTERNAME_RADARPILOT_1000, " RADARPILOT 1000" },
     { DIS_PDU_EMITTERNAME_RADARPILOT_1100, " RADARPILOT 1100" },
     { DIS_PDU_EMITTERNAME_RAJENDRA, "RAJENDRA" },
     { DIS_PDU_EMITTERNAME_RAN_7S, "RAN 7S" },
     { DIS_PDU_EMITTERNAME_RAN_10S, "RAN 10S" },
     { DIS_PDU_EMITTERNAME_RAN_11_LX, "RAN 11 LX" },
     { DIS_PDU_EMITTERNAME_RANI, "Rani" },
     { DIS_PDU_EMITTERNAME_RAPHAEL_TH, "RAPHAEL-TH" },
     { DIS_PDU_EMITTERNAME_RAPIER_TA, "Rapier TA" },
     { DIS_PDU_EMITTERNAME_RAPIER_2000_TA_DAGGER, "Rapier 2000 TA Dagger" },
     { DIS_PDU_EMITTERNAME_RAPIER_MG, "Rapier MG" },
     { DIS_PDU_EMITTERNAME_RASCAR_3400C, "RASCAR 3400C" },
     { DIS_PDU_EMITTERNAME_RASHMI, "Rashmi" },
     { DIS_PDU_EMITTERNAME_RASIT, "Rasit" },
     { DIS_PDU_EMITTERNAME_RASIT_3190B, "Rasit 3190B" },
     { DIS_PDU_EMITTERNAME_RAT_31_DL_M, "RAT-31 DL/M" },
     { DIS_PDU_EMITTERNAME_RAT_31_DL, "RAT-31 DL" },
     { DIS_PDU_EMITTERNAME_RAT_31S, "RAT-31S" },
     { DIS_PDU_EMITTERNAME_RAT_8_S, "RAT-8 S" },
     { DIS_PDU_EMITTERNAME_RAT_31_SL, "RAT-31 SL" },
     { DIS_PDU_EMITTERNAME_RAVEN_ES_05, "Raven ES-05" },
     { DIS_PDU_EMITTERNAME_RATAC_LCT, "RATAC (LCT)" },
     { DIS_PDU_EMITTERNAME_RAWL, "RAWL" },
     { DIS_PDU_EMITTERNAME_RATTLER, " Rattler" },
     { DIS_PDU_EMITTERNAME_RAWS, "RAWS" },
     { DIS_PDU_EMITTERNAME_RAWL_02, "RAWL-02" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_1220, " Raytheon 1220" },
     { DIS_PDU_EMITTERNAME_RAWS_03, "RAWS-03" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_1210XX, " Raytheon 1210xx" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_1302, " Raytheon 1302" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_1500, " Raytheon 1500" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_1645, " Raytheon 1645" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_1650, " Raytheon 1650" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_1900, " Raytheon 1900" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_2502, " Raytheon 2502" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_ANSCHUTZ_NAUTOSCAN_NX, "Raytheon Anschutz NautoScan NX" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_R41, " Raytheon R41" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_RM_1025_6X, " Raytheon RM 1025 6X" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_SL_72, " Raytheon SL 72" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_TM_1650_6X, " Raytheon TM 1650/6X" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_TM_1660_12S, " Raytheon TM 1660/12S" },
     { DIS_PDU_EMITTERNAME_RAY_1220XR, "RAY-1220XR" },
     { DIS_PDU_EMITTERNAME_RAY_1401, " RAY-1401" },
     { DIS_PDU_EMITTERNAME_RAY_2900, " Ray 2900" },
     { DIS_PDU_EMITTERNAME_RAYMARINE_RD218, " Raymarine RD218" },
     { DIS_PDU_EMITTERNAME_RAYPATH, " Raypath" },
     { DIS_PDU_EMITTERNAME_RAYTHEON_PATHFINDER_ST_MK2, " Raytheon Pathfinder ST mk2" },
     { DIS_PDU_EMITTERNAME_RBE2, "RBE2" },
     { DIS_PDU_EMITTERNAME_RBE2_AA, "RBE2-AA" },
     { DIS_PDU_EMITTERNAME_RCT_180, "RCT-180" },
     { DIS_PDU_EMITTERNAME_RDM, "RDM" },
     { DIS_PDU_EMITTERNAME_RDM_3, "RDM-3" },
     { DIS_PDU_EMITTERNAME_RDI, "RDI" },
     { DIS_PDU_EMITTERNAME_RDY, "RDY" },
     { DIS_PDU_EMITTERNAME_RDY_3, "RDY-3" },
     { DIS_PDU_EMITTERNAME_RDS_86, "RDS-86" },
     { DIS_PDU_EMITTERNAME_RDN_72, "RDN 72" },
     { DIS_PDU_EMITTERNAME_RDR_1A, "RDR 1A" },
     { DIS_PDU_EMITTERNAME_RDR_1E, "RDR 1E" },
     { DIS_PDU_EMITTERNAME_RDR_4A, "RDR 4A" },
     { DIS_PDU_EMITTERNAME_RDR_150, "RDR-150" },
     { DIS_PDU_EMITTERNAME_RDR_160XD, "RDR-160XD" },
     { DIS_PDU_EMITTERNAME_RDR_230_HP, "RDR-230 HP" },
     { DIS_PDU_EMITTERNAME_RDR_1100, "RDR 1100" },
     { DIS_PDU_EMITTERNAME_RDR_1150, "RDR-1150" },
     { DIS_PDU_EMITTERNAME_RDR_1200, "RDR 1200" },
     { DIS_PDU_EMITTERNAME_RDR_1400, "RDR 1400" },
     { DIS_PDU_EMITTERNAME_RDR_1400_C, "RDR 1400 C" },
     { DIS_PDU_EMITTERNAME_RDR_4000, "RDR 4000" },
     { DIS_PDU_EMITTERNAME_RDR_1500, "RDR 1500" },
     { DIS_PDU_EMITTERNAME_RICE_CAKE, " Rice Cake" },
     { DIS_PDU_EMITTERNAME_RDR_1600, "RDR 1600" },
     { DIS_PDU_EMITTERNAME_RDR_2000, "RDR 2000" },
     { DIS_PDU_EMITTERNAME_RDR_1700B, "RDR 1700B" },
     { DIS_PDU_EMITTERNAME_REMORA, "Remora" },
     { DIS_PDU_EMITTERNAME_RICE_FIELD, "Rice Field" },
     { DIS_PDU_EMITTERNAME_REC_1A, "REC-1A" },
     { DIS_PDU_EMITTERNAME_REC_1B, "REC-1B" },
     { DIS_PDU_EMITTERNAME_REC_1C, "REC-1C" },
     { DIS_PDU_EMITTERNAME_RESOLVE_EAS, "Resolve EAS" },
     { DIS_PDU_EMITTERNAME_RICE_CUP_C, " Rice Cup C" },
     { DIS_PDU_EMITTERNAME_REL_6E, "REL-6E" },
     { DIS_PDU_EMITTERNAME_REC_1, "REC-1" },
     { DIS_PDU_EMITTERNAME_RICE_BOWL, " Rice Bowl" },
     { DIS_PDU_EMITTERNAME_IMPROVED_REPORTER, "Improved Reporter" },
     { DIS_PDU_EMITTERNAME_RICE_BUG, " Rice Bug" },
     { DIS_PDU_EMITTERNAME_RICE_CUP, " Rice Cup" },
     { DIS_PDU_EMITTERNAME_RICE_LAMP, " Rice Lamp" },
     { DIS_PDU_EMITTERNAME_RICE_BIRD, " Rice Bird" },
     { DIS_PDU_EMITTERNAME_REVATHI, "REVATHI" },
     { DIS_PDU_EMITTERNAME_REZONANS, "REZONANS" },
     { DIS_PDU_EMITTERNAME_RGM_UGM_109B, "RGM/UGM-109B" },
     { DIS_PDU_EMITTERNAME_RGM_UGM_109E_HOMING_RADAR, "RGM/UGM-109E Homing Radar" },
     { DIS_PDU_EMITTERNAME_RICE_PAD, " Rice Pad" },
     { DIS_PDU_EMITTERNAME_RKL_526, "RKL-526" },
     { DIS_PDU_EMITTERNAME_RKZ_764, "RKZ-764" },
     { DIS_PDU_EMITTERNAME_RKZ_766, "RKZ-766" },
     { DIS_PDU_EMITTERNAME_RKL_165, "RKL-165" },
     { DIS_PDU_EMITTERNAME_RKL_609, "RKL-609" },
     { DIS_PDU_EMITTERNAME_RKL_800, "RKL-800" },
     { DIS_PDU_EMITTERNAME_RKZ_761, "RKZ-761" },
     { DIS_PDU_EMITTERNAME_RKZ_2000, "RKZ-2000" },
     { DIS_PDU_EMITTERNAME_RIS_4C_A, "RIS-4C/A" },
     { DIS_PDU_EMITTERNAME_RL_2000, "RL-2000" },
     { DIS_PDU_EMITTERNAME_RL_41, "RL-41" },
     { DIS_PDU_EMITTERNAME_RIR_778, "RIR 778" },
     { DIS_PDU_EMITTERNAME_RISAT, "RISAT" },
     { DIS_PDU_EMITTERNAME_RLM_S, "RLM-S" },
     { DIS_PDU_EMITTERNAME_RIM_HAT_ESM_ECM_SUITE, "Rim Hat ESM/ECM Suite" },
     { DIS_PDU_EMITTERNAME_REL_86TM, " REL-86TM" },
     { DIS_PDU_EMITTERNAME_RICE_SCREEN, " Rice Screen" },
     { DIS_PDU_EMITTERNAME_DECCA_RM_1070A, " DECCA RM 1070A" },
     { DIS_PDU_EMITTERNAME_RM370BT, "RM370BT" },
     { DIS_PDU_EMITTERNAME_ROCKWELL_COLLINS_FMR_200X, " Rockwell Collins FMR-200X" },
     { DIS_PDU_EMITTERNAME_RM_231_2, " RM-231-2" },
     { DIS_PDU_EMITTERNAME_RM_231_3, " RM-231-3" },
     { DIS_PDU_EMITTERNAME_RMT_0100A, "RMT 0100A" },
     { DIS_PDU_EMITTERNAME_RN_222, "RN-222" },
     { DIS_PDU_EMITTERNAME_ROLAND_2, "ROLAND 2" },
     { DIS_PDU_EMITTERNAME_ROLAND_BN, "ROLAND BN" },
     { DIS_PDU_EMITTERNAME_ROLAND_MG, "ROLAND MG" },
     { DIS_PDU_EMITTERNAME_ROLAND_TA, "ROLAND TA" },
     { DIS_PDU_EMITTERNAME_ROLAND_TT, "ROLAND TT" },
     { DIS_PDU_EMITTERNAME_ROTODOME, "ROTODOME" },
     { DIS_PDU_EMITTERNAME_ROUND_BALL, " Round Ball" },
     { DIS_PDU_EMITTERNAME_RP_379D_TIRADA_D, "RP-379D Tirada D" },
     { DIS_PDU_EMITTERNAME_RP_3, "RP-3" },
     { DIS_PDU_EMITTERNAME_RP_4G, "RP-4G" },
     { DIS_PDU_EMITTERNAME_ROUND_HOUSE, " Round House" },
     { DIS_PDU_EMITTERNAME_ROUND_HOUSE_B, " Round House B" },
     { DIS_PDU_EMITTERNAME_RPR_117, "RPR-117" },
     { DIS_PDU_EMITTERNAME_RS_02_50, "RS-02/50" },
     { DIS_PDU_EMITTERNAME_RSR_210N, " RSR 210N" },
     { DIS_PDU_EMITTERNAME_RT_02_50, "RT-02/50" },
     { DIS_PDU_EMITTERNAME_RTA_4100, "RTA-4100" },
     { DIS_PDU_EMITTERNAME_RTN_1A, "RTN-1A" },
     { DIS_PDU_EMITTERNAME_RTN_25X, "RTN-25X" },
     { DIS_PDU_EMITTERNAME_RTS_6400, "RTS-6400" },
     { DIS_PDU_EMITTERNAME_RUBY_RAKE, " Ruby Rake" },
     { DIS_PDU_EMITTERNAME_RUM_SLING, " Rum Sling" },
     { DIS_PDU_EMITTERNAME_RUM_SLING_RO, " Rum Sling RO" },
     { DIS_PDU_EMITTERNAME_RUM_SLING_TT, " Rum Sling TT" },
     { DIS_PDU_EMITTERNAME_RV2, "RV2" },
     { DIS_PDU_EMITTERNAME_RV3, "RV3" },
     { DIS_PDU_EMITTERNAME_RV5, "RV5" },
     { DIS_PDU_EMITTERNAME_RV10, "RV10" },
     { DIS_PDU_EMITTERNAME_RV_15M, "RV-15M" },
     { DIS_PDU_EMITTERNAME_RV17, "RV17" },
     { DIS_PDU_EMITTERNAME_RV18, "RV18" },
     { DIS_PDU_EMITTERNAME_RV_21, "RV-21" },
     { DIS_PDU_EMITTERNAME_RV_21B, "RV-21B" },
     { DIS_PDU_EMITTERNAME_RV_25, "RV-25" },
     { DIS_PDU_EMITTERNAME_RV_377, "RV-377" },
     { DIS_PDU_EMITTERNAME_RV_UM, "RV UM" },
     { DIS_PDU_EMITTERNAME_RWD_8, "RWD-8" },
     { DIS_PDU_EMITTERNAME_RXN_2_60, " RXN 2-60" },
     { DIS_PDU_EMITTERNAME_RYE_HOUSE, " Rye House" },
     { DIS_PDU_EMITTERNAME_S_1810CD, "S-1810CD" },
     { DIS_PDU_EMITTERNAME_SAHAB, "Sahab" },
     { DIS_PDU_EMITTERNAME_SALAMANDRE, "Salamandre" },
     { DIS_PDU_EMITTERNAME_SAMYUNG_SMR_7200, " Samyung SMR-7200" },
     { DIS_PDU_EMITTERNAME_S1850M, "S1850M" },
     { DIS_PDU_EMITTERNAME_S_511, "S-511" },
     { DIS_PDU_EMITTERNAME_S_512, "S-512" },
     { DIS_PDU_EMITTERNAME_S_600, "S-600" },
     { DIS_PDU_EMITTERNAME_S_604, "S-604" },
     { DIS_PDU_EMITTERNAME_S_763_LANZA_3D, "S-763 LANZA 3D" },
     { DIS_PDU_EMITTERNAME_S_613, "S-613" },
     { DIS_PDU_EMITTERNAME_S_631, "S-631" },
     { DIS_PDU_EMITTERNAME_S_654, "S-654" },
     { DIS_PDU_EMITTERNAME_S_669, "S-669" },
     { DIS_PDU_EMITTERNAME_SA_2_GUIDELINE, " SA 2 Guideline" },
     { DIS_PDU_EMITTERNAME_S_244, "S-244" },
     { DIS_PDU_EMITTERNAME_S_711, "S-711" },
     { DIS_PDU_EMITTERNAME_SA_3_GOA, " SA 3 Goa" },
     { DIS_PDU_EMITTERNAME_SA_8_GECKO_DT, " SA 8 Gecko DT" },
     { DIS_PDU_EMITTERNAME_GLADIATOR_TELAR_TIR, " Gladiator TELAR TIR" },
     { DIS_PDU_EMITTERNAME_GIANT_TELAR_TIR, " Giant TELAR TIR" },
     { DIS_PDU_EMITTERNAME_SABER_M60, "SABER-M60" },
     { DIS_PDU_EMITTERNAME_SAMOVAR, "Samovar" },
     { DIS_PDU_EMITTERNAME_SAMPSON, "Sampson" },
     { DIS_PDU_EMITTERNAME_SA_N_7_GADFLY_TI, " SA N 7 Gadfly TI" },
     { DIS_PDU_EMITTERNAME_SA_N_11_CADS_1_UN, " SA N 11 Cads 1 UN" },
     { DIS_PDU_EMITTERNAME_SACCADE_MH, "Saccade MH" },
     { DIS_PDU_EMITTERNAME_SALT_POT_A_B, " Salt Pot A&B" },
     { DIS_PDU_EMITTERNAME_SAP_14, "SAP-14" },
     { DIS_PDU_EMITTERNAME_SAP_518, "SAP-518" },
     { DIS_PDU_EMITTERNAME_SAP_518M, "SAP-518M" },
     { DIS_PDU_EMITTERNAME_SAND_BAR, "Sand Bar" },
     { DIS_PDU_EMITTERNAME_SA_26_PECHORA_2M_TTR, " SA-26 Pechora-2M TTR" },
     { DIS_PDU_EMITTERNAME_SAR_ON_UAVS, "SAR (on UAVs)" },
     { DIS_PDU_EMITTERNAME_SATRAPE, "SATRAPE" },
     { DIS_PDU_EMITTERNAME_SATURNE_II, "SATURNE II" },
     { DIS_PDU_EMITTERNAME_SAYYAD_2_TER, "Sayyad-2 TER" },
     { DIS_PDU_EMITTERNAME_SCAN_CAN, " Scan Can" },
     { DIS_PDU_EMITTERNAME_SCAN_FIX, " Scan Fix" },
     { DIS_PDU_EMITTERNAME_SCAN_ODD, " Scan Odd" },
     { DIS_PDU_EMITTERNAME_SCANTER_1002, "SCANTER 1002" },
     { DIS_PDU_EMITTERNAME_SCANTER_2001, " SCANTER 2001" },
     { DIS_PDU_EMITTERNAME_SCANTER_2002, " SCANTER 2002" },
     { DIS_PDU_EMITTERNAME_SCANTER_2100, " SCANTER 2100" },
     { DIS_PDU_EMITTERNAME_SCANTER_4002, "SCANTER 4002" },
     { DIS_PDU_EMITTERNAME_SCANTER_4100, " SCANTER 4100" },
     { DIS_PDU_EMITTERNAME_SCANTER_5102, "SCANTER 5102" },
     { DIS_PDU_EMITTERNAME_SCANTER_5502, "SCANTER 5502" },
     { DIS_PDU_EMITTERNAME_SCANTER_6000, " SCANTER 6000" },
     { DIS_PDU_EMITTERNAME_SCANTER_6002, " SCANTER 6002" },
     { DIS_PDU_EMITTERNAME_SCANTER_MIL_009, " Scanter Mil 009" },
     { DIS_PDU_EMITTERNAME_SCAN_THREE, " Scan Three" },
     { DIS_PDU_EMITTERNAME_SCANTER_MIL_S, "SCANTER MIL S" },
     { DIS_PDU_EMITTERNAME_SCANTER_SMR, "Scanter SMR" },
     { DIS_PDU_EMITTERNAME_SCANTER_CSR, "SCANTER (CSR)" },
     { DIS_PDU_EMITTERNAME_SCORADS, "SCORADS" },
     { DIS_PDU_EMITTERNAME_SCIMITAR, "Scimitar" },
     { DIS_PDU_EMITTERNAME_STAR_2000, "STAR 2000" },
     { DIS_PDU_EMITTERNAME_SCOREBOARD, " SCOREBOARD" },
     { DIS_PDU_EMITTERNAME_SCOOP_PAIR, "Scoop Pair" },
     { DIS_PDU_EMITTERNAME_SCOUP_PLATE, " Scoup Plate" },
     { DIS_PDU_EMITTERNAME_SCOUT, " SCOUT" },
     { DIS_PDU_EMITTERNAME_SCR_584, "SCR-584" },
     { DIS_PDU_EMITTERNAME_SEA_ARCHER_2, "Sea Archer 2" },
     { DIS_PDU_EMITTERNAME_SEA_BASED_X_BAND, "Sea Based X-Band" },
     { DIS_PDU_EMITTERNAME_SEA_DRAGON, "Sea Dragon" },
     { DIS_PDU_EMITTERNAME_SEA_EAGLE_TYPE_381, "Sea Eagle (Type 381)" },
     { DIS_PDU_EMITTERNAME_SEA_EAGLE_S_C_TYPE_382, "Sea Eagle S/C (Type 382)" },
     { DIS_PDU_EMITTERNAME_SEA_FALCON, "SEA FALCON" },
     { DIS_PDU_EMITTERNAME_SEA_GIRAFFE_AMB, " Sea Giraffe AMB" },
     { DIS_PDU_EMITTERNAME_SEAGUARD, " Seaguard" },
     { DIS_PDU_EMITTERNAME_SEA_HAWK_SHN_X12, "Sea-Hawk SHN X12" },
     { DIS_PDU_EMITTERNAME_SEA_HUNTER_4_MG, "Sea Hunter 4 MG" },
     { DIS_PDU_EMITTERNAME_SEA_HUNTER_4_TA, "Sea Hunter 4 TA" },
     { DIS_PDU_EMITTERNAME_SEA_HUNTER_4_TT, "Sea Hunter 4 TT" },
     { DIS_PDU_EMITTERNAME_SEA_GULL, " Sea Gull" },
     { DIS_PDU_EMITTERNAME_SEA_MASTER_400, " Sea Master 400" },
     { DIS_PDU_EMITTERNAME_SEA_NET, " Sea Net" },
     { DIS_PDU_EMITTERNAME_SEA_SPRAY, "Sea Spray" },
     { DIS_PDU_EMITTERNAME_SEA_TIGER, "Sea Tiger" },
     { DIS_PDU_EMITTERNAME_SEA_TIGER_M, "Sea Tiger M" },
     { DIS_PDU_EMITTERNAME_SEASTAR_SEA_WATCHER_100, "Seastar Sea Watcher 100" },
     { DIS_PDU_EMITTERNAME_SEARCHWATER, "Searchwater" },
     { DIS_PDU_EMITTERNAME_SEARCHWATER_2000, "Searchwater 2000" },
     { DIS_PDU_EMITTERNAME_SEASONDE, "SEASONDE" },
     { DIS_PDU_EMITTERNAME_SEASPRAY_7000E, "SEASPRAY 7000E" },
     { DIS_PDU_EMITTERNAME_SEAVUE, "SeaVue" },
     { DIS_PDU_EMITTERNAME_SEASPRAY_MK3, " Seaspray Mk3" },
     { DIS_PDU_EMITTERNAME_SELENIA_ORION_7, " Selenia Orion 7" },
     { DIS_PDU_EMITTERNAME_SELENIA_TYPE_912, " Selenia type 912" },
     { DIS_PDU_EMITTERNAME_SELENNIA_RAN_12_L_X, " Selennia RAN 12 L/X" },
     { DIS_PDU_EMITTERNAME_SELENIA_RAN_20S, "Selenia RAN 20S" },
     { DIS_PDU_EMITTERNAME_SELENNIA_RTN_10X, " Selennia RTN 10X" },
     { DIS_PDU_EMITTERNAME_SELINIA_ARP_1645, " Selinia ARP 1645" },
     { DIS_PDU_EMITTERNAME_SENTIR_M20, "SENTIR-M20" },
     { DIS_PDU_EMITTERNAME_SERDAR, "SERDAR" },
     { DIS_PDU_EMITTERNAME_SERHAT, "SERHAT" },
     { DIS_PDU_EMITTERNAME_SERIES_10_COMPACT_SUBMARINE_RADAR_CSR, " Series 10 Compact Submarine Radar (CSR)" },
     { DIS_PDU_EMITTERNAME_SERIES_52, "SERIES 52" },
     { DIS_PDU_EMITTERNAME_SERIES_320, "SERIES 320" },
     { DIS_PDU_EMITTERNAME_SG, "SG" },
     { DIS_PDU_EMITTERNAME_SGJ_02, " SGJ-02" },
     { DIS_PDU_EMITTERNAME_SGJ_03, " SGJ-03" },
     { DIS_PDU_EMITTERNAME_SGR_102_00, "SGR 102 00" },
     { DIS_PDU_EMITTERNAME_SGR_103_02, "SGR 103/02" },
     { DIS_PDU_EMITTERNAME_SGR_104, "SGR-104" },
     { DIS_PDU_EMITTERNAME_SHAHED_129_SAR, "Shahed-129 SAR" },
     { DIS_PDU_EMITTERNAME_SHAHINE, "SHAHINE" },
     { DIS_PDU_EMITTERNAME_SHEET_BEND, " Sheet Bend" },
     { DIS_PDU_EMITTERNAME_SHEET_CURVE, " Sheet Curve" },
     { DIS_PDU_EMITTERNAME_SHIKRA, "SHIKRA" },
     { DIS_PDU_EMITTERNAME_SHIP_GLOBE, " Ship Globe" },
     { DIS_PDU_EMITTERNAME_SHIP_WHEEL, " Ship Wheel" },
     { DIS_PDU_EMITTERNAME_SGR_114, "SGR 114" },
     { DIS_PDU_EMITTERNAME_SHORE_WALK_A, " Shore Walk A" },
     { DIS_PDU_EMITTERNAME_SHORT_HORN, " Short Horn" },
     { DIS_PDU_EMITTERNAME_SHOT_DOME, " Shot Dome" },
     { DIS_PDU_EMITTERNAME_SIDE_GLOBE_JN, " Side Globe JN" },
     { DIS_PDU_EMITTERNAME_PRV_11_SIDE_NET, "PRV-11 Side Net" },
     { DIS_PDU_EMITTERNAME_SIDE_WALK_A, " Side Walk A" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_BHARAT, "Signaal - Bharat" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_DA_02, " Signaal DA 02" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_DA_05, " Signaal DA 05" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_DA_08, " Signaal DA 08" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_DA_08_2LS, " Signaal DA 08/2LS" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_LW_04, " Signaal LW 04" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_LW_08, " Signaal LW 08" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_LWOR, " Signaal LWOR" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_M45, " Signaal M45" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_MW_08, " Signaal MW 08" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_SMART, " Signaal SMART" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_STING, " Signaal STING" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_STIR, " Signaal STIR" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_STIR_1_8M, " Signaal STIR 1.8M" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_STIR_24M, " Signaal STIR 24M" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_WM_20_2, " Signaal WM 20/2" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_WM_25, " Signaal WM 25" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_WM_27, " Signaal WM 27" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_WM_28, " Signaal WM 28" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_ZW_01, " Signaal ZW 01" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_ZW_06, " Signaal ZW 06" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_ZW_07, " Signaal ZW 07" },
     { DIS_PDU_EMITTERNAME_SIGNAAL_ZW_0800, " Signaal ZW 0800" },
     { DIS_PDU_EMITTERNAME_SIMRAD_3G, "SIMRAD 3G" },
     { DIS_PDU_EMITTERNAME_SIMRAD_4G, "SIMRAD 4G" },
     { DIS_PDU_EMITTERNAME_SIMRAD_CA54, " Simrad CA54" },
     { DIS_PDU_EMITTERNAME_SIMRAD_HALO_6, " SIMRAD Halo-6" },
     { DIS_PDU_EMITTERNAME_SKI_POLE, " Ski Pole" },
     { DIS_PDU_EMITTERNAME_SKIN_HEAD, " Skin Head" },
     { DIS_PDU_EMITTERNAME_SKIP_SPIN, " Skip Spin" },
     { DIS_PDU_EMITTERNAME_SKYFENDER, "SKYFENDER" },
     { DIS_PDU_EMITTERNAME_SKY_WAVE_OVER_THE_HORIZON, "Sky Wave (Over The Horizon)" },
     { DIS_PDU_EMITTERNAME_SKYGUARD_B, "Skyguard B" },
     { DIS_PDU_EMITTERNAME_SKYGUARD_TA_UAR_1021, "SKYGUARD TA UAR-1021" },
     { DIS_PDU_EMITTERNAME_SKYGUARD_TT_UAR_1021, "SKYGUARD TT UAR-1021" },
     { DIS_PDU_EMITTERNAME_SKYGUARD_LR, "Skyguard LR" },
     { DIS_PDU_EMITTERNAME_SKYMASTER, "Skymaster" },
     { DIS_PDU_EMITTERNAME_SKY_WATCH, " Sky Watch" },
     { DIS_PDU_EMITTERNAME_SKY_RANGER, "Sky Ranger" },
     { DIS_PDU_EMITTERNAME_SKYSHADOW, " SKYSHADOW" },
     { DIS_PDU_EMITTERNAME_SKYSHIELD_TA, "SKYSHIELD TA" },
     { DIS_PDU_EMITTERNAME_SL, "SL" },
     { DIS_PDU_EMITTERNAME_SL_ALQ_234, "SL/ALQ-234" },
     { DIS_PDU_EMITTERNAME_SLAP_SHOT, " Slap Shot" },
     { DIS_PDU_EMITTERNAME_SLAP_SHOT_G, " Slap Shot G" },
     { DIS_PDU_EMITTERNAME_SLC_2, "SLC-2" },
     { DIS_PDU_EMITTERNAME_SLC_2E, "SLC-2E" },
     { DIS_PDU_EMITTERNAME_SLC_4, "SLC-4" },
     { DIS_PDU_EMITTERNAME_SLIM_NET, " Slim Net" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_A, " Slot Back A" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_ILL, " Slot Back ILL" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_B, " Slot Back B" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_IV, " Slot Back IV" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_B_TOPAZ, " Slot Back B Topaz" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_E, " Slot Back E" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_G, " Slot Back G" },
     { DIS_PDU_EMITTERNAME_SLOT_BACK_VI, " Slot Back VI" },
     { DIS_PDU_EMITTERNAME_SLOT_REST, " Slot Rest" },
     { DIS_PDU_EMITTERNAME_SM_674A_UPM, "SM-674A/UPM" },
     { DIS_PDU_EMITTERNAME_SMA_3_RM, " SMA 3 RM" },
     { DIS_PDU_EMITTERNAME_SMA_3_RM_20, " SMA 3 RM 20" },
     { DIS_PDU_EMITTERNAME_SMA_3RM_20A_SMG, " SMA 3RM 20A/SMG" },
     { DIS_PDU_EMITTERNAME_SMA_BPS_704, " SMA BPS 704" },
     { DIS_PDU_EMITTERNAME_SMA_SPIN_749_V_2, " SMA SPIN 749 (V) 2" },
     { DIS_PDU_EMITTERNAME_SMA_SPN_703, " SMA SPN 703" },
     { DIS_PDU_EMITTERNAME_SMA_SPN_751, " SMA SPN 751" },
     { DIS_PDU_EMITTERNAME_SMA_SPOS_748, " SMA SPOS 748" },
     { DIS_PDU_EMITTERNAME_SMA_SPQ_2, " SMA SPQ 2" },
     { DIS_PDU_EMITTERNAME_SMA_SPQ_2D, " SMA SPQ 2D" },
     { DIS_PDU_EMITTERNAME_SMA_SPQ_701, " SMA SPQ 701" },
     { DIS_PDU_EMITTERNAME_SMA_SPS_702_UPX, " SMA SPS 702 UPX" },
     { DIS_PDU_EMITTERNAME_SMA_ST_2_OTOMAT_II_MH, " SMA ST 2 OTOMAT II MH" },
     { DIS_PDU_EMITTERNAME_SR_47A, "SR-47A" },
     { DIS_PDU_EMITTERNAME_SMA_718_BEACON, " SMA 718 Beacon" },
     { DIS_PDU_EMITTERNAME_SMALL_FRED, " Small Fred" },
     { DIS_PDU_EMITTERNAME_SMART_S, "SMART-S" },
     { DIS_PDU_EMITTERNAME_SMART_S_MK2, "SMART-S Mk2" },
     { DIS_PDU_EMITTERNAME_SMART_L, "SMART-L" },
     { DIS_PDU_EMITTERNAME_SM_932, "SM-932" },
     { DIS_PDU_EMITTERNAME_SMOG_LAMP, " Smog Lamp" },
     { DIS_PDU_EMITTERNAME_SNAP_SHOT, " Snap Shot" },
     { DIS_PDU_EMITTERNAME_SNOOP_DRIFT, " Snoop Drift" },
     { DIS_PDU_EMITTERNAME_SNOOP_HALF, " Snoop Half" },
     { DIS_PDU_EMITTERNAME_SNOOP_HEAD, " Snoop Head" },
     { DIS_PDU_EMITTERNAME_SNOOP_PAIR, " Snoop Pair" },
     { DIS_PDU_EMITTERNAME_SNOOP_PLATE, " Snoop Plate" },
     { DIS_PDU_EMITTERNAME_SNOOP_PING, " Snoop Ping" },
     { DIS_PDU_EMITTERNAME_SNOOP_SLAB, " Snoop Slab" },
     { DIS_PDU_EMITTERNAME_SNOOP_TRAY, " Snoop Tray" },
     { DIS_PDU_EMITTERNAME_SNOOP_TRAY_1, " Snoop Tray 1" },
     { DIS_PDU_EMITTERNAME_SNOOP_TRAY_2, " Snoop Tray 2" },
     { DIS_PDU_EMITTERNAME_SNOOP_TRAY_3, " SNOOP TRAY 3" },
     { DIS_PDU_EMITTERNAME_SNOOP_WATCH, " Snoop Watch" },
     { DIS_PDU_EMITTERNAME_9S18M1_SNOW_DRIFT, "9S18M1 Snow Drift" },
     { DIS_PDU_EMITTERNAME_9S18M1E_SNOW_DRIFT, "9S18M1E Snow Drift" },
     { DIS_PDU_EMITTERNAME_SPB_7, "SPB-7" },
     { DIS_PDU_EMITTERNAME_SNOW_DROP, " Snow Drop" },
     { DIS_PDU_EMITTERNAME_SNW_10, "SNW-10" },
     { DIS_PDU_EMITTERNAME_SO_1, "SO-1" },
     { DIS_PDU_EMITTERNAME_SO_12, "SO-12" },
     { DIS_PDU_EMITTERNAME_SO_A_COMMUNIST, "SO A Communist" },
     { DIS_PDU_EMITTERNAME_SO_69, "SO-69" },
     { DIS_PDU_EMITTERNAME_SOCK_EYE, " Sock Eye" },
     { DIS_PDU_EMITTERNAME_SOM_64, "SOM 64" },
     { DIS_PDU_EMITTERNAME_SOPKA_HILL, "Sopka (Hill)" },
     { DIS_PDU_EMITTERNAME_SORBSIYA, " Sorbsiya" },
     { DIS_PDU_EMITTERNAME_SORBTSIYA_L005, "Sorbtsiya L005" },
     { DIS_PDU_EMITTERNAME_SORBTSIYA_L005S, "Sorbtsiya L005S" },
     { DIS_PDU_EMITTERNAME_SPADA_SIR, "SPADA SIR" },
     { DIS_PDU_EMITTERNAME_SPADA_TT, " SPADA TT" },
     { DIS_PDU_EMITTERNAME_SPARROW_AIM_RIM_7_ILL, "Sparrow (AIM/RIM-7) ILL" },
     { DIS_PDU_EMITTERNAME_SPERRY_RASCAR, "SPERRY RASCAR" },
     { DIS_PDU_EMITTERNAME_SPECTRA, "SPECTRA" },
     { DIS_PDU_EMITTERNAME_SPEAR3_MMW, "SPEAR3 MMW" },
     { DIS_PDU_EMITTERNAME_SPERRY_M_3, " Sperry M-3" },
     { DIS_PDU_EMITTERNAME_SPERRY_VISIONMASTER_FT, " SPERRY VISIONMASTER FT" },
     { DIS_PDU_EMITTERNAME_SPEXER_2000, "SPEXER 2000" },
     { DIS_PDU_EMITTERNAME_SPG_53F, "SPG 53F" },
     { DIS_PDU_EMITTERNAME_SPG_70_RTN_10X, "SPG 70 (RTN 10X)" },
     { DIS_PDU_EMITTERNAME_SPG_74_RTN_20X, "SPG 74 (RTN 20X)" },
     { DIS_PDU_EMITTERNAME_SPG_75_RTN_30X, "SPG 75 (RTN 30X)" },
     { DIS_PDU_EMITTERNAME_SPG_76_RTN_30X, "SPG 76 (RTN 30X)" },
     { DIS_PDU_EMITTERNAME_SPIN_SCAN_A, " Spin Scan A" },
     { DIS_PDU_EMITTERNAME_SPIN_SCAN_B, " Spin Scan B" },
     { DIS_PDU_EMITTERNAME_SPIN_TROUGH, " Spin Trough" },
     { DIS_PDU_EMITTERNAME_SPINO_D_ADDA_WTR, "SPINO D'ADDA WTR" },
     { DIS_PDU_EMITTERNAME_SPJ_40, "SPJ-40" },
     { DIS_PDU_EMITTERNAME_SPLASH_DROP, " Splash Drop" },
     { DIS_PDU_EMITTERNAME_SPN_2, "SPN-2" },
     { DIS_PDU_EMITTERNAME_SPN_4, "SPN-4" },
     { DIS_PDU_EMITTERNAME_SPN_30, "SPN-30" },
     { DIS_PDU_EMITTERNAME_SPN_35A, "SPN 35A" },
     { DIS_PDU_EMITTERNAME_SPN_41, "SPN 41" },
     { DIS_PDU_EMITTERNAME_SPN_42, "SPN 42" },
     { DIS_PDU_EMITTERNAME_SPN_43A, "SPN 43A" },
     { DIS_PDU_EMITTERNAME_SPN_43B, "SPN 43B" },
     { DIS_PDU_EMITTERNAME_SPN_44, "SPN 44" },
     { DIS_PDU_EMITTERNAME_SPN_46, "SPN 46" },
     { DIS_PDU_EMITTERNAME_SPN_703, "SPN 703" },
     { DIS_PDU_EMITTERNAME_SPN_720_P_2080, "SPN 720 P-2080" },
     { DIS_PDU_EMITTERNAME_SPN_728_V_1, "SPN 728 (V) 1" },
     { DIS_PDU_EMITTERNAME_SPN_748, "SPN 748" },
     { DIS_PDU_EMITTERNAME_SPN_750, "SPN 750" },
     { DIS_PDU_EMITTERNAME_SPO_8, "SPO-8" },
     { DIS_PDU_EMITTERNAME_SPN_753G, "SPN 753G" },
     { DIS_PDU_EMITTERNAME_SPONGE_CAKE, " Sponge Cake" },
     { DIS_PDU_EMITTERNAME_P_12_SPOON_REST, "P-12 Spoon Rest" },
     { DIS_PDU_EMITTERNAME_P_18_SPOON_REST_A, "P-18 Spoon Rest A" },
     { DIS_PDU_EMITTERNAME_P_18_SPOON_REST_B, "P-18 Spoon Rest B" },
     { DIS_PDU_EMITTERNAME_P_18_SPOON_REST_D, "P-18 Spoon Rest D" },
     { DIS_PDU_EMITTERNAME_P_18MH2, "P-18MH2" },
     { DIS_PDU_EMITTERNAME_SPORK_REST, " Spork Rest" },
     { DIS_PDU_EMITTERNAME_SPQ_712_RAN_12_L_X, "SPQ 712 (RAN 12 L/X)" },
     { DIS_PDU_EMITTERNAME_SPR_2, "SPR-2" },
     { DIS_PDU_EMITTERNAME_SPR_51, "SPR-51" },
     { DIS_PDU_EMITTERNAME_SPS_5_FASOL, "SPS-5 FASOL" },
     { DIS_PDU_EMITTERNAME_SPS_6, "SPS-6" },
     { DIS_PDU_EMITTERNAME_SPS_6C, "SPS 6C" },
     { DIS_PDU_EMITTERNAME_SPS_10F, "SPS 10F" },
     { DIS_PDU_EMITTERNAME_SPS_12, "SPS 12" },
     { DIS_PDU_EMITTERNAME_SPS_22N_BUKET, "SPS-22N BUKET" },
     { DIS_PDU_EMITTERNAME_SPS_33N_BUKET, "SPS-33N BUKET" },
     { DIS_PDU_EMITTERNAME_SPS_44N_BUKET, "SPS-44N BUKET" },
     { DIS_PDU_EMITTERNAME_SPS_55N_BUKET, "SPS-55N BUKET" },
     { DIS_PDU_EMITTERNAME_SPS_62, "SPS-62" },
     { DIS_PDU_EMITTERNAME_SPS_100K, "SPS-100K" },
     { DIS_PDU_EMITTERNAME_SPS_141, "SPS-141" },
     { DIS_PDU_EMITTERNAME_SPS_142, "SPS-142" },
     { DIS_PDU_EMITTERNAME_SPS_143, "SPS-143" },
     { DIS_PDU_EMITTERNAME_SPS_151, "SPS-151" },
     { DIS_PDU_EMITTERNAME_SPS_152, "SPS-152" },
     { DIS_PDU_EMITTERNAME_SPS_153, "SPS-153" },
     { DIS_PDU_EMITTERNAME_SPS_160_GERAN, "SPS-160 Geran" },
     { DIS_PDU_EMITTERNAME_SPS_161, "SPS-161" },
     { DIS_PDU_EMITTERNAME_SPS_95K, "SPS-95K" },
     { DIS_PDU_EMITTERNAME_SPS_171_JAMMER, "SPS-171 Jammer" },
     { DIS_PDU_EMITTERNAME_SPS_172_JAMMER, "SPS-172 Jammer" },
     { DIS_PDU_EMITTERNAME_SPS_768_RAN_EL, "SPS 768 (RAN EL)" },
     { DIS_PDU_EMITTERNAME_SPS_540K, "SPS-540K" },
     { DIS_PDU_EMITTERNAME_SPS_550K_MF, "SPS-550K MF" },
     { DIS_PDU_EMITTERNAME_SPS_774_RAN_10S, "SPS 774 (RAN 10S)" },
     { DIS_PDU_EMITTERNAME_SPY_790, "SPY 790" },
     { DIS_PDU_EMITTERNAME_SQUARE_HEAD, " Square Head" },
     { DIS_PDU_EMITTERNAME_SQUARE_PAIR, " Square Pair" },
     { DIS_PDU_EMITTERNAME_SQUARE_SLOT, " Square Slot" },
     { DIS_PDU_EMITTERNAME_SQUARE_TIE, " Square Tie" },
     { DIS_PDU_EMITTERNAME_SHMEL_SQUASH_DOME, "Shmel Squash Dome" },
     { DIS_PDU_EMITTERNAME_P_15M_SQUAT_EYE, "P-15M Squat Eye" },
     { DIS_PDU_EMITTERNAME_SQUINT_EYE, " Squint Eye" },
     { DIS_PDU_EMITTERNAME_SQUIRE, "SQUIRE" },
     { DIS_PDU_EMITTERNAME_SR2410C, "SR2410C" },
     { DIS_PDU_EMITTERNAME_SR47B_G, "SR47B-G" },
     { DIS_PDU_EMITTERNAME_SRE_M5, "SRE-M5" },
     { DIS_PDU_EMITTERNAME_SRN_6, "SRN 6" },
     { DIS_PDU_EMITTERNAME_SRN_15, "SRN 15" },
     { DIS_PDU_EMITTERNAME_SRN_206, "SRN 206" },
     { DIS_PDU_EMITTERNAME_SRN_745, "SRN 745" },
     { DIS_PDU_EMITTERNAME_SRO_1, "SRO 1" },
     { DIS_PDU_EMITTERNAME_SRO_2, "SRO 2" },
     { DIS_PDU_EMITTERNAME_SS_C_2B_SAMLET_MG, " SS C 2B Samlet MG" },
     { DIS_PDU_EMITTERNAME_SS_N_2A_B_CSSC, " SS N 2A B CSSC" },
     { DIS_PDU_EMITTERNAME_SS_N_2A_B_CSSC_2A_3A2_MH, " SS N 2A B CSSC 2A 3A2 MH" },
     { DIS_PDU_EMITTERNAME_SS_N_2C_SEEKER, " SS N 2C Seeker" },
     { DIS_PDU_EMITTERNAME_SS_N_2C_D_STYX, " SS N 2C D Styx" },
     { DIS_PDU_EMITTERNAME_SS_N_2C_D_STYX_C_D_MH, " SS N 2C D Styx C D MH" },
     { DIS_PDU_EMITTERNAME_SS_N_2C_STYX_AL, " SS-N-2C Styx AL" },
     { DIS_PDU_EMITTERNAME_SS_N_2D_STYX_AL, " SS-N-2D Styx AL" },
     { DIS_PDU_EMITTERNAME_SS_N_3_SSC_SS_C_18_BN, " SS N 3 SSC SS C 18 BN" },
     { DIS_PDU_EMITTERNAME_SS_N_3B_SEPAL_AL, " SS N 3B Sepal AL" },
     { DIS_PDU_EMITTERNAME_SS_N_3B_SEPAL_MH, " SS N 3B Sepal MH" },
     { DIS_PDU_EMITTERNAME_SS_N_7_STARBRIGHT, " SS-N-7 Starbright" },
     { DIS_PDU_EMITTERNAME_SS_N_9_SIREN, " SS N 9 Siren" },
     { DIS_PDU_EMITTERNAME_SS_N_9_SIREN_AL, " SS N 9 Siren AL" },
     { DIS_PDU_EMITTERNAME_SS_N_9_SIREN_MH, " SS N 9 Siren MH" },
     { DIS_PDU_EMITTERNAME_SS_N_10A_FL_10_MMW_MH, "SS-N-10A FL-10 mmW MH" },
     { DIS_PDU_EMITTERNAME_SS_N_11_NASR_1_MMW_MH, "SS-N-11 Nasr-1 mmW MH" },
     { DIS_PDU_EMITTERNAME_SS_N_12_SANDBOX_AL, " SS N 12 Sandbox AL" },
     { DIS_PDU_EMITTERNAME_SS_N_12_YJ_83J_MMW_MH, "SS-N-12 YJ-83J mmW MH" },
     { DIS_PDU_EMITTERNAME_SS_N_12_SANDBOX_MH, " SS N 12 Sandbox MH" },
     { DIS_PDU_EMITTERNAME_SS_NX_13_SHREDDER, " SS-NX-13 Shredder" },
     { DIS_PDU_EMITTERNAME_SS_N_14B_SILEX_AL, " SS-N-14B Silex AL" },
     { DIS_PDU_EMITTERNAME_SS_N_19_SHIPWRECK, " SS N 19 Shipwreck" },
     { DIS_PDU_EMITTERNAME_SS_N_19_SHIPWRECK_AL, " SS N 19 Shipwreck AL" },
     { DIS_PDU_EMITTERNAME_SS_N_19_SHIPWRECK_MH, " SS N 19 Shipwreck MH" },
     { DIS_PDU_EMITTERNAME_SS_N_21_AL, " SS N 21 AL" },
     { DIS_PDU_EMITTERNAME_SS_N_22_SUNBURN, " SS N 22 Sunburn" },
     { DIS_PDU_EMITTERNAME_SS_N_22_SUNBURN_MH, " SS N 22 Sunburn MH" },
     { DIS_PDU_EMITTERNAME_SS_N_22_SUNBURN_AL, " SS-N-22 Sunburn AL" },
     { DIS_PDU_EMITTERNAME_SS_N_25_SWITCHBLADE_MH, " SS-N-25 Switchblade MH" },
     { DIS_PDU_EMITTERNAME_SS_N_26_STROBILE_MMW_MH, "SS-N-26 Strobile MMW MH" },
     { DIS_PDU_EMITTERNAME_SS_N_27_SIZZLER_MH, " SS-N-27 Sizzler MH" },
     { DIS_PDU_EMITTERNAME_SS_N_27A_SIZZLER_AL, " SS-N-27A Sizzler AL" },
     { DIS_PDU_EMITTERNAME_STING_EO_MK2, " STING EO Mk2" },
     { DIS_PDU_EMITTERNAME_STIR_1_2_EO_MK2, " STIR 1.2 EO Mk2" },
     { DIS_PDU_EMITTERNAME_STIR_2_4_HP_MK2, " STIR 2.4 HP Mk2" },
     { DIS_PDU_EMITTERNAME_STONE_CAKE, " Stone Cake" },
     { DIS_PDU_EMITTERNAME_STR_41, "STR 41" },
     { DIS_PDU_EMITTERNAME_ST_858, "ST-858" },
     { DIS_PDU_EMITTERNAME_START_1M, "START-1M" },
     { DIS_PDU_EMITTERNAME_STENTOR, "STENTOR" },
     { DIS_PDU_EMITTERNAME_STORM_SHADOW_AHR, "Storm Shadow AHR" },
     { DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH, "STRAIGHT FLUSH" },
     { DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH_TA, " Straight Flush TA" },
     { DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH_TT, " Straight Flush TT" },
     { DIS_PDU_EMITTERNAME_STRAIGHT_FLUSH_ILL, " Straight Flush ILL" },
     { DIS_PDU_EMITTERNAME_STRIKE_OUT, " Strike Out" },
     { DIS_PDU_EMITTERNAME_STRUT_CURVE, " Strut Curve" },
     { DIS_PDU_EMITTERNAME_STRUT_PAIR, " Strut Pair" },
     { DIS_PDU_EMITTERNAME_STRUT_PAIR_1, " Strut Pair 1" },
     { DIS_PDU_EMITTERNAME_STRUT_PAIR_2, " Strut Pair 2" },
     { DIS_PDU_EMITTERNAME_SUN_VISOR, " Sun Visor" },
     { DIS_PDU_EMITTERNAME_SUPERDARN, "SUPERDARN" },
     { DIS_PDU_EMITTERNAME_SUPERFLEDERMAUS, "Superfledermaus" },
     { DIS_PDU_EMITTERNAME_SUPERSEARCHER, "Supersearcher" },
     { DIS_PDU_EMITTERNAME_SWIFT_ROD_1, " Swift Rod 1" },
     { DIS_PDU_EMITTERNAME_SWIFT_ROD_2, " Swift Rod 2" },
     { DIS_PDU_EMITTERNAME_SYMPHONY, "SYMPHONY" },
     { DIS_PDU_EMITTERNAME_SYNAPSIS_MK2, "SYNAPSIS Mk2" },
     { DIS_PDU_EMITTERNAME_SY80, "SY80" },
     { DIS_PDU_EMITTERNAME_T1166, " T1166" },
     { DIS_PDU_EMITTERNAME_T1171, " T1171" },
     { DIS_PDU_EMITTERNAME_T1202, " T1202" },
     { DIS_PDU_EMITTERNAME_T6004, " T6004" },
     { DIS_PDU_EMITTERNAME_T6031, " T6031" },
     { DIS_PDU_EMITTERNAME_T8067, " T8067" },
     { DIS_PDU_EMITTERNAME_T8068, " T8068" },
     { DIS_PDU_EMITTERNAME_T8124, " T8124" },
     { DIS_PDU_EMITTERNAME_T8408, " T8408" },
     { DIS_PDU_EMITTERNAME_T8911, " T8911" },
     { DIS_PDU_EMITTERNAME_T8937, " T8937" },
     { DIS_PDU_EMITTERNAME_T8944, " T8944" },
     { DIS_PDU_EMITTERNAME_T8987, " T8987" },
     { DIS_PDU_EMITTERNAME_TA_10K, "TA-10K" },
     { DIS_PDU_EMITTERNAME_JY_11B_TALL_BOARD, "JY-11B Tall Board" },
     { DIS_PDU_EMITTERNAME_TACAN_SURF, "TACAN/SURF" },
     { DIS_PDU_EMITTERNAME_P_14_TALL_KING, "P-14 Tall King" },
     { DIS_PDU_EMITTERNAME_TALL_KING_B, " TALL KING B" },
     { DIS_PDU_EMITTERNAME_TALL_KING_C, " TALL KING C" },
     { DIS_PDU_EMITTERNAME_TALL_MIKE, " Tall Mike" },
     { DIS_PDU_EMITTERNAME_TALL_PATH, " Tall Path" },
     { DIS_PDU_EMITTERNAME_TDR_94_MODE_S, "TDR-94 (MODE S)" },
     { DIS_PDU_EMITTERNAME_TEA_SPOON, " Tea Spoon" },
     { DIS_PDU_EMITTERNAME_TEAM_PLAY, " Team Play" },
     { DIS_PDU_EMITTERNAME_TALISMAN, "TALISMAN" },
     { DIS_PDU_EMITTERNAME_TEAM_WORK, " Team Work" },
     { DIS_PDU_EMITTERNAME_T1135, "T1135" },
     { DIS_PDU_EMITTERNAME_TANCAN_SURF, "TANCAN/SURF" },
     { DIS_PDU_EMITTERNAME_TECSAR, "TECSAR" },
     { DIS_PDU_EMITTERNAME_TERRASAR_X, "TERRASAR-X" },
     { DIS_PDU_EMITTERNAME_TESAR, "TESAR" },
     { DIS_PDU_EMITTERNAME_THAAD_GBR, "THAAD GBR" },
     { DIS_PDU_EMITTERNAME_THALES_RDY_2, "Thales RDY-2" },
     { DIS_PDU_EMITTERNAME_THALES_NEDERLAND_SIGNAAL_APAR, "Thales Nederland Signaal APAR" },
     { DIS_PDU_EMITTERNAME_THALES_SCORPION_JAMMER, "Thales Scorpion Jammer" },
     { DIS_PDU_EMITTERNAME_THALES_VARIANT, "Thales Variant" },
     { DIS_PDU_EMITTERNAME_THALES_ICMS_JAMMER, "Thales ICMS Jammer" },
     { DIS_PDU_EMITTERNAME_THALES_IMEWS_JAMMER, "Thales IMEWS Jammer" },
     { DIS_PDU_EMITTERNAME_THD_225, "THD 225" },
     { DIS_PDU_EMITTERNAME_THD_1012, "THD 1012" },
     { DIS_PDU_EMITTERNAME_THD_1098, "THD 1098" },
     { DIS_PDU_EMITTERNAME_THD_1213, "THD 1213" },
     { DIS_PDU_EMITTERNAME_THD_1940_PICADOR, "THD 1940 Picador" },
     { DIS_PDU_EMITTERNAME_THD_1955_PALMIER, "THD-1955 Palmier" },
     { DIS_PDU_EMITTERNAME_THD_5500, "THD 5500" },
     { DIS_PDU_EMITTERNAME_THIRD_OF_KHORDAD, "Third of Khordad" },
     { DIS_PDU_EMITTERNAME_THIN_PATH, " Thin Path" },
     { DIS_PDU_EMITTERNAME_PRV_9_THIN_SKIN, "PRV-9 Thin Skin" },
     { DIS_PDU_EMITTERNAME_PRV_16_THIN_SKIN_B, "PRV-16 Thin Skin B" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_TA_10, " Thompson CSF TA-10" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_TH_D_1040_NEPTUNE, " Thompson CSF TH D 1040 Neptune" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_CALYPSO, " Thompson CSF Calypso" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_CASTOR, " Thompson CSF CASTOR" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_CASTOR_II, " Thompson CSF Castor II" },
     { DIS_PDU_EMITTERNAME_THOMSON_CSF_DOMINO_30, "Thomson-CSF Domino 30" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBC_32A, " Thompson CSF DRBC 32A" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBJ_11_D_E, " Thompson CSF DRBJ 11 D/E" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_15A, " Thompson CSF DRBV 15A" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_15C, " Thompson CSF DRBV 15C" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_22D, " Thompson CSF DRBV 22D" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRBV_23B, " Thompson CSF DRBV 23B" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_DRUA_33, " Thompson CSF DRUA 33" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_MARS_DRBV_21A, " Thompson CSF Mars DRBV 21A" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_SEA_TIGER, " Thompson CSF Sea Tiger" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_TRITON, " Thompson CSF Triton" },
     { DIS_PDU_EMITTERNAME_THOMPSON_CSF_VEGA_WITH_DRBC_32E, " Thompson CSF Vega with DRBC 32E" },
     { DIS_PDU_EMITTERNAME_THOMSON_ENR_EUROPEAN_NAVY_RADAR, "Thomson ENR (European Navy Radar)" },
     { DIS_PDU_EMITTERNAME_THOMSON_RDI, "Thomson RDI" },
     { DIS_PDU_EMITTERNAME_TIER_II_PLUS, "Tier II Plus" },
     { DIS_PDU_EMITTERNAME_TPS_755, "TPS-755" },
     { DIS_PDU_EMITTERNAME_TPS_830K, "TPS-830K" },
     { DIS_PDU_EMITTERNAME_TRS_2105_TIGER_G, "TRS-2105 TIGER-G" },
     { DIS_PDU_EMITTERNAME_TR_23K, "TR-23K" },
     { DIS_PDU_EMITTERNAME_TR_23MR, "TR-23MR" },
     { DIS_PDU_EMITTERNAME_TRAC_2100, "TRAC-2100" },
     { DIS_PDU_EMITTERNAME_TRAC_2300, "TRAC-2300" },
     { DIS_PDU_EMITTERNAME_HT_223_TIGER_PAW, "HT-223 Tiger Paw" },
     { DIS_PDU_EMITTERNAME_TRADEX, "TRADEX" },
     { DIS_PDU_EMITTERNAME_TRAIL_XI, "TRAIL XI" },
     { DIS_PDU_EMITTERNAME_TRD_1211, "TRD-1211" },
     { DIS_PDU_EMITTERNAME_TRD_1235, "TRD-1235" },
     { DIS_PDU_EMITTERNAME_TRS_2100_TIGER_S, "TRS-2100 TIGER-S" },
     { DIS_PDU_EMITTERNAME_TRAC_NG, "TRAC NG" },
     { DIS_PDU_EMITTERNAME_TIE_RODS, " Tie Rods" },
     { DIS_PDU_EMITTERNAME_36D6_TIN_SHIELD, "36D6 Tin Shield" },
     { DIS_PDU_EMITTERNAME_TIN_TRAP, " Tin Trap" },
     { DIS_PDU_EMITTERNAME_TIRSPONDER, "TIRSPONDER" },
     { DIS_PDU_EMITTERNAME_TK_25E_5, "TK-25E-5" },
     { DIS_PDU_EMITTERNAME_TMK_MK2, "TMK Mk2" },
     { DIS_PDU_EMITTERNAME_TMX_MK2, "TMX Mk2" },
     { DIS_PDU_EMITTERNAME_TOAD_STOOL_1, " Toad Stool 1" },
     { DIS_PDU_EMITTERNAME_TOAD_STOOL_2, " Toad Stool 2" },
     { DIS_PDU_EMITTERNAME_TOAD_STOOL_3, " Toad Stool 3" },
     { DIS_PDU_EMITTERNAME_TOAD_STOOL_4, " Toad Stool 4" },
     { DIS_PDU_EMITTERNAME_TOAD_STOOL_5, " Toad Stool 5" },
     { DIS_PDU_EMITTERNAME_TOKEN_B, "Token B" },
     { DIS_PDU_EMITTERNAME_TOMB_STONE, " Tomb Stone" },
     { DIS_PDU_EMITTERNAME_TONSON, "Tonson" },
     { DIS_PDU_EMITTERNAME_TOP_BOW, " Top Bow" },
     { DIS_PDU_EMITTERNAME_TOP_DOME, " Top Dome" },
     { DIS_PDU_EMITTERNAME_TOP_KNOT, " Top Knot" },
     { DIS_PDU_EMITTERNAME_TOP_MESH, " Top Mesh" },
     { DIS_PDU_EMITTERNAME_TOP_PAIR, " Top Pair" },
     { DIS_PDU_EMITTERNAME_TOP_PLATE, " Top Plate" },
     { DIS_PDU_EMITTERNAME_TOP_PLATE_B, " Top Plate B" },
     { DIS_PDU_EMITTERNAME_TOP_SAIL, " Top Sail" },
     { DIS_PDU_EMITTERNAME_TOP_STEER, " Top Steer" },
     { DIS_PDU_EMITTERNAME_TOP_TROUGH, " Top Trough" },
     { DIS_PDU_EMITTERNAME_TORNADO_GMR, "Tornado GMR" },
     { DIS_PDU_EMITTERNAME_TORNADO_TFR, "Tornado TFR" },
     { DIS_PDU_EMITTERNAME_SCRUM_HALF_TA, " Scrum Half TA" },
     { DIS_PDU_EMITTERNAME_SCRUM_HALF_TT, " Scrum Half TT" },
     { DIS_PDU_EMITTERNAME_TOR_M2_TER_SCRUM_HALF_C, "TOR-M2 TER Scrum Half C" },
     { DIS_PDU_EMITTERNAME_SCRUM_HALF_MG, " Scrum Half MG" },
     { DIS_PDU_EMITTERNAME_TRACK_DISH, " Track Dish" },
     { DIS_PDU_EMITTERNAME_TR_47C, "TR-47C" },
     { DIS_PDU_EMITTERNAME_TORSO_M, "TORSO M" },
     { DIS_PDU_EMITTERNAME_TQN_2, "TQN-2" },
     { DIS_PDU_EMITTERNAME_TRAP_DOOR, " Trap Door" },
     { DIS_PDU_EMITTERNAME_TRD_1500, "TRD-1500" },
     { DIS_PDU_EMITTERNAME_TRICK_SHOT_TAR, " Trick Shot TAR" },
     { DIS_PDU_EMITTERNAME_TRICK_SHOT_TER, " Trick Shot TER" },
     { DIS_PDU_EMITTERNAME_TRISPONDE, "TRISPONDE" },
     { DIS_PDU_EMITTERNAME_TRML, "TRML" },
     { DIS_PDU_EMITTERNAME_TRS_2215, "TRS-2215" },
     { DIS_PDU_EMITTERNAME_TRML_3D, "TRML-3D" },
     { DIS_PDU_EMITTERNAME_TRM_S, "TRM-S" },
     { DIS_PDU_EMITTERNAME_TRS_2056, "TRS-2056" },
     { DIS_PDU_EMITTERNAME_TRS_3010, "TRS 3010" },
     { DIS_PDU_EMITTERNAME_TRS_2060, "TRS-2060" },
     { DIS_PDU_EMITTERNAME_TRS_2245, "TRS-2245" },
     { DIS_PDU_EMITTERNAME_TRS_2310, "TRS-2310" },
     { DIS_PDU_EMITTERNAME_TRITON_G, "Triton G" },
     { DIS_PDU_EMITTERNAME_TRS_22XX, "TRS-22XX" },
     { DIS_PDU_EMITTERNAME_TRS_3030, "TRS 3030" },
     { DIS_PDU_EMITTERNAME_TRS_3033, "TRS 3033" },
     { DIS_PDU_EMITTERNAME_TRS_3203, "TRS 3203" },
     { DIS_PDU_EMITTERNAME_TRS_3405, "TRS 3405" },
     { DIS_PDU_EMITTERNAME_TRS_3410, "TRS 3410" },
     { DIS_PDU_EMITTERNAME_TRS_3415, "TRS 3415" },
     { DIS_PDU_EMITTERNAME_TRS_3D, "TRS-3D" },
     { DIS_PDU_EMITTERNAME_TRS_3D_16, "TRS-3D/16" },
     { DIS_PDU_EMITTERNAME_TRS_3D_16_ES, "TRS-3D/16-ES" },
     { DIS_PDU_EMITTERNAME_TRS_3D_32, "TRS-3D/32" },
     { DIS_PDU_EMITTERNAME_TRS_4D, "TRS-4D" },
     { DIS_PDU_EMITTERNAME_TRS_C, "TRS-C" },
     { DIS_PDU_EMITTERNAME_TRS_N, "TRS-N" },
     { DIS_PDU_EMITTERNAME_TS_4478A, "TS-4478A" },
     { DIS_PDU_EMITTERNAME_TSE_5000, "TSE 5000" },
     { DIS_PDU_EMITTERNAME_TSR_333, "TSR 333" },
     { DIS_PDU_EMITTERNAME_TSR_793, "TSR 793" },
     { DIS_PDU_EMITTERNAME_TUB_BRICK, " Tub Brick" },
     { DIS_PDU_EMITTERNAME_TUBE_ARM, " Tube Arm" },
     { DIS_PDU_EMITTERNAME_TW_1374, "TW 1374" },
     { DIS_PDU_EMITTERNAME_TW_1378, "TW 1378" },
     { DIS_PDU_EMITTERNAME_TW_1446, "TW 1446" },
     { DIS_PDU_EMITTERNAME_TWIN_EYES, " Twin Eyes" },
     { DIS_PDU_EMITTERNAME_TWIN_PILL, " Twin Pill" },
     { DIS_PDU_EMITTERNAME_TWIN_SCAN, " Twin Scan" },
     { DIS_PDU_EMITTERNAME_TWIN_SCAN_RO, " Twin Scan Ro" },
     { DIS_PDU_EMITTERNAME_TWO_SPOT, " Two Spot" },
     { DIS_PDU_EMITTERNAME_TYPE_071_LPD, "Type 071 LPD" },
     { DIS_PDU_EMITTERNAME_TYPE_2_12_J_A, "Type 2-12 J/A" },
     { DIS_PDU_EMITTERNAME_TYPE_2_21_J_A, "Type 2-21 J/A" },
     { DIS_PDU_EMITTERNAME_TYPE_2_23, "Type 2-23" },
     { DIS_PDU_EMITTERNAME_TYPE_80_ASM_1, "Type 80/ASM-1" },
     { DIS_PDU_EMITTERNAME_TYPE_120, "Type 120" },
     { DIS_PDU_EMITTERNAME_TYPE_208, "Type 208" },
     { DIS_PDU_EMITTERNAME_TYPE_222, "Type 222" },
     { DIS_PDU_EMITTERNAME_TYPE_226, "Type 226" },
     { DIS_PDU_EMITTERNAME_TYPE_232H, "Type 232H" },
     { DIS_PDU_EMITTERNAME_TYPE_245, "TYPE 245" },
     { DIS_PDU_EMITTERNAME_TYPE_262, "TYPE 262" },
     { DIS_PDU_EMITTERNAME_TYPE_275, "TYPE 275" },
     { DIS_PDU_EMITTERNAME_TYPE_278, "TYPE 278" },
     { DIS_PDU_EMITTERNAME_TYPE_293, "TYPE 293" },
     { DIS_PDU_EMITTERNAME_TYPE_341, "Type 341" },
     { DIS_PDU_EMITTERNAME_TYPE_313, "TYPE 313" },
     { DIS_PDU_EMITTERNAME_TYPE_305A, "Type 305A" },
     { DIS_PDU_EMITTERNAME_TYPE_334, "Type 334" },
     { DIS_PDU_EMITTERNAME_TYPE_342, "Type 342" },
     { DIS_PDU_EMITTERNAME_TYPE_343_SUN_VISOR_B, "TYPE 343 SUN VISOR B" },
     { DIS_PDU_EMITTERNAME_TYPE_344, "Type 344" },
     { DIS_PDU_EMITTERNAME_TYPE_345, "Type 345" },
     { DIS_PDU_EMITTERNAME_TYPE_346, "Type 346" },
     { DIS_PDU_EMITTERNAME_TYPE_349A, "Type 349A" },
     { DIS_PDU_EMITTERNAME_TYPE_347B, "TYPE 347B" },
     { DIS_PDU_EMITTERNAME_TYPE_347G, "Type 347G" },
     { DIS_PDU_EMITTERNAME_TYPE_359, "Type 359" },
     { DIS_PDU_EMITTERNAME_TYPE_352, "Type 352" },
     { DIS_PDU_EMITTERNAME_TYPE_360, "Type 360" },
     { DIS_PDU_EMITTERNAME_TYPE_362_ESR_1_SR_47B, "Type 362 ESR-1 SR-47B" },
     { DIS_PDU_EMITTERNAME_TYPE_354, "Type 354" },
     { DIS_PDU_EMITTERNAME_TYPE_366, "Type 366" },
     { DIS_PDU_EMITTERNAME_TYPE_363, "Type 363" },
     { DIS_PDU_EMITTERNAME_TYPE_364_SEAGULL_C, "Type 364 Seagull-C" },
     { DIS_PDU_EMITTERNAME_TYPE_404A_CH, "Type-404A(CH)" },
     { DIS_PDU_EMITTERNAME_TYPE_405, "Type 405" },
     { DIS_PDU_EMITTERNAME_TYPE_405J, "TYPE 405J" },
     { DIS_PDU_EMITTERNAME_TYPE_408D_MOON_STOOL_B, "Type 408D Moon Stool B" },
     { DIS_PDU_EMITTERNAME_TYPE_517B, "Type 517B" },
     { DIS_PDU_EMITTERNAME_TYPE_518_HAI_YING_GOD_EYE_REL_2, "Type 518 (Hai Ying, God Eye, REL-2)" },
     { DIS_PDU_EMITTERNAME_TYPE_589, "Type 589" },
     { DIS_PDU_EMITTERNAME_TYPE_651, "TYPE 651" },
     { DIS_PDU_EMITTERNAME_TYPE_702, "Type 702" },
     { DIS_PDU_EMITTERNAME_TYPE_704, "Type 704" },
     { DIS_PDU_EMITTERNAME_TYPE_753, "Type 753" },
     { DIS_PDU_EMITTERNAME_TYPE_756, "Type 756" },
     { DIS_PDU_EMITTERNAME_TYPE_713, "TYPE 713" },
     { DIS_PDU_EMITTERNAME_TYPE_714, "TYPE 714" },
     { DIS_PDU_EMITTERNAME_TYPE_702_D, "TYPE 702-D" },
     { DIS_PDU_EMITTERNAME_TYPE_760, "Type 760" },
     { DIS_PDU_EMITTERNAME_TYPE_815, "Type 815" },
     { DIS_PDU_EMITTERNAME_TYPE_793, "Type 793" },
     { DIS_PDU_EMITTERNAME_TYPE_8A_813, "Type 8A-813" },
     { DIS_PDU_EMITTERNAME_TYPE_901M, "TYPE 901M" },
     { DIS_PDU_EMITTERNAME_TYPE_902, "TYPE 902" },
     { DIS_PDU_EMITTERNAME_TYPE_902B, "Type 902B" },
     { DIS_PDU_EMITTERNAME_TYPE_903, "TYPE 903" },
     { DIS_PDU_EMITTERNAME_TYPE_909_TI, "TYPE 909 TI" },
     { DIS_PDU_EMITTERNAME_TYPE_909_TT, "TYPE 909 TT" },
     { DIS_PDU_EMITTERNAME_TYPE_910, "TYPE 910" },
     { DIS_PDU_EMITTERNAME_TYPE_931_CH, "TYPE-931(CH)" },
     { DIS_PDU_EMITTERNAME_TYPE_965, "TYPE 965" },
     { DIS_PDU_EMITTERNAME_TYPE_967, "TYPE 967" },
     { DIS_PDU_EMITTERNAME_TYPE_968, "TYPE 968" },
     { DIS_PDU_EMITTERNAME_TYPE_974, "TYPE 974" },
     { DIS_PDU_EMITTERNAME_TYPE_975, "TYPE 975" },
     { DIS_PDU_EMITTERNAME_TYPE_978, "TYPE 978" },
     { DIS_PDU_EMITTERNAME_TYPE_981, "Type 981" },
     { DIS_PDU_EMITTERNAME_TYPE_981_3, "Type 981-3" },
     { DIS_PDU_EMITTERNAME_TYPE_982, "TYPE 982" },
     { DIS_PDU_EMITTERNAME_TYPE_984, "Type 984" },
     { DIS_PDU_EMITTERNAME_TYPE_985, "Type 985" },
     { DIS_PDU_EMITTERNAME_TYPE_992, "TYPE 992" },
     { DIS_PDU_EMITTERNAME_TYPE_993, "TYPE 993" },
     { DIS_PDU_EMITTERNAME_TYPE_994, "TYPE 994" },
     { DIS_PDU_EMITTERNAME_TYPE_996, "Type 996" },
     { DIS_PDU_EMITTERNAME_TYPE_997_ARTISAN, "Type 997 Artisan" },
     { DIS_PDU_EMITTERNAME_TYPE_1006_1, "TYPE 1006(1)" },
     { DIS_PDU_EMITTERNAME_TYPE_1006_2, "TYPE 1006(2)" },
     { DIS_PDU_EMITTERNAME_TYPE_1022, "TYPE 1022" },
     { DIS_PDU_EMITTERNAME_TYPE_1047, "Type 1047" },
     { DIS_PDU_EMITTERNAME_TYPE_1048, "Type 1048" },
     { DIS_PDU_EMITTERNAME_TYPE_1474, "Type 1474" },
     { DIS_PDU_EMITTERNAME_TYPE_1493, "Type 1493" },
     { DIS_PDU_EMITTERNAME_ULTRA, "ULTRA" },
     { DIS_PDU_EMITTERNAME_UK_MK_10, "UK MK 10" },
     { DIS_PDU_EMITTERNAME_UPS_220C, "UPS-220C" },
     { DIS_PDU_EMITTERNAME_UPX_1_10, "UPX 1 10" },
     { DIS_PDU_EMITTERNAME_UPX_27, "UPX 27" },
     { DIS_PDU_EMITTERNAME_URN_20, "URN 20" },
     { DIS_PDU_EMITTERNAME_UTES_A, "UTES-A" },
     { DIS_PDU_EMITTERNAME_UTES_T, "UTES-T" },
     { DIS_PDU_EMITTERNAME_URN_25, "URN 25" },
     { DIS_PDU_EMITTERNAME_VIGILANT, "VIGILANT" },
     { DIS_PDU_EMITTERNAME_VITEBSK_L370_JAMMER, "Vitebsk L370 Jammer" },
     { DIS_PDU_EMITTERNAME_VOLEX_III_IV, "VOLEX III/IV" },
     { DIS_PDU_EMITTERNAME_VOLGA, "VOLGA" },
     { DIS_PDU_EMITTERNAME_VORONEZH_DM, "VORONEZH-DM" },
     { DIS_PDU_EMITTERNAME_VOSTOK, "VOSTOK" },
     { DIS_PDU_EMITTERNAME_VOSTOK_E, "VOSTOK-E" },
     { DIS_PDU_EMITTERNAME_VSR, "VSR" },
     { DIS_PDU_EMITTERNAME_VOSTOK_3D, "VOSTOK-3D" },
     { DIS_PDU_EMITTERNAME_VSTAR_PT, "VSTAR-PT" },
     { DIS_PDU_EMITTERNAME_W_160, "W-160" },
     { DIS_PDU_EMITTERNAME_W1028, "W1028" },
     { DIS_PDU_EMITTERNAME_W8818, "W8818" },
     { DIS_PDU_EMITTERNAME_W8838, "W8838" },
     { DIS_PDU_EMITTERNAME_W8852, "W8852" },
     { DIS_PDU_EMITTERNAME_WALL_BOARD, " WALL BOARD" },
     { DIS_PDU_EMITTERNAME_WALL_RUST, " Wall Rust" },
     { DIS_PDU_EMITTERNAME_WAS_74S, "WAS-74S" },
     { DIS_PDU_EMITTERNAME_WASP_HEAD, " Wasp Head" },
     { DIS_PDU_EMITTERNAME_WATCHDOG, "WATCHDOG" },
     { DIS_PDU_EMITTERNAME_WATCH_GUARD, " Watch Guard" },
     { DIS_PDU_EMITTERNAME_WATCHMAN, "Watchman" },
     { DIS_PDU_EMITTERNAME_WAVESTORM, "WAVESTORM" },
     { DIS_PDU_EMITTERNAME_WATCHMAN_S, "WATCHMAN-S" },
     { DIS_PDU_EMITTERNAME_WATCHMAN_T, "WATCHMAN-T" },
     { DIS_PDU_EMITTERNAME_WEATHER_SCOUT_2, "WEATHER SCOUT 2" },
     { DIS_PDU_EMITTERNAME_WESTERN_ELECTRIC_MK_10, " Western Electric MK 10" },
     { DIS_PDU_EMITTERNAME_WESTINGHOUSE_ADR_4_LRSR, " Westinghouse ADR-4 LRSR" },
     { DIS_PDU_EMITTERNAME_WESTINGHOUSE_ELECTRIC_SPG_50, " Westinghouse Electric SPG 50" },
     { DIS_PDU_EMITTERNAME_WESTINGHOUSE_ELECTRIC_W_120, " Westinghouse Electric W 120" },
     { DIS_PDU_EMITTERNAME_WESTINGHOUSE_SPS_29C, " Westinghouse SPS 29C" },
     { DIS_PDU_EMITTERNAME_WESTINGHOUSE_SPS_37, " Westinghouse SPS 37" },
     { DIS_PDU_EMITTERNAME_WET_EYE, "Wet Eye" },
     { DIS_PDU_EMITTERNAME_WET_EYE_2, "Wet Eye 2" },
     { DIS_PDU_EMITTERNAME_WET_EYE_MOD, "Wet Eye Mod" },
     { DIS_PDU_EMITTERNAME_WF44S, "WF44S" },
     { DIS_PDU_EMITTERNAME_WGU_41_B, "WGU-41/B" },
     { DIS_PDU_EMITTERNAME_WGU_44_B, "WGU-44/B" },
     { DIS_PDU_EMITTERNAME_WHIFF, " Whiff" },
     { DIS_PDU_EMITTERNAME_WHIFF_BRICK, " Whiff Brick" },
     { DIS_PDU_EMITTERNAME_WHIFF_FIRE, " Whiff Fire" },
     { DIS_PDU_EMITTERNAME_WHITE_HOUSE, " WHITE HOUSE" },
     { DIS_PDU_EMITTERNAME_WIDE_MAT, " Wide Mat" },
     { DIS_PDU_EMITTERNAME_WINE_GLASS_JAMMER, "Wine Glass Jammer" },
     { DIS_PDU_EMITTERNAME_WILD_CARD, "Wild Card" },
     { DIS_PDU_EMITTERNAME_WILDCAT, "WILDCAT" },
     { DIS_PDU_EMITTERNAME_WITCH_EIGHT, " Witch Eight" },
     { DIS_PDU_EMITTERNAME_WITCH_FIVE, " Witch Five" },
     { DIS_PDU_EMITTERNAME_WLR, "WLR" },
     { DIS_PDU_EMITTERNAME_WM2X_SERIES, "WM2X Series" },
     { DIS_PDU_EMITTERNAME_WM2X_SERIES_CAS, "WM2X Series CAS" },
     { DIS_PDU_EMITTERNAME_WR_10X, "WR-10X" },
     { DIS_PDU_EMITTERNAME_WR_2100, "WR-2100" },
     { DIS_PDU_EMITTERNAME_WSR_74C, "WSR-74C" },
     { DIS_PDU_EMITTERNAME_WSR_74S, "WSR-74S" },
     { DIS_PDU_EMITTERNAME_WSR_81, "WSR-81" },
     { DIS_PDU_EMITTERNAME_WXR_700C, "WXR-700C" },
     { DIS_PDU_EMITTERNAME_WXR_2100_MSTT, "WXR-2100 MSTT" },
     { DIS_PDU_EMITTERNAME_WOOD_GAGE, " Wood Gage" },
     { DIS_PDU_EMITTERNAME_X_TAR25, "X-TAR25" },
     { DIS_PDU_EMITTERNAME_X_TAR3D, "X-TAR3D" },
     { DIS_PDU_EMITTERNAME_YAOGAN_3, "YAOGAN 3" },
     { DIS_PDU_EMITTERNAME_YAOGAN_29, "Yaogan-29" },
     { DIS_PDU_EMITTERNAME_YARD_RAKE, " Yard Rake" },
     { DIS_PDU_EMITTERNAME_YH_96, "YH-96" },
     { DIS_PDU_EMITTERNAME_YEW_LOOP, " Yew Loop" },
     { DIS_PDU_EMITTERNAME_YITIAN_ADS, "YITIAN ADS" },
     { DIS_PDU_EMITTERNAME_YD_3, "YD-3" },
     { DIS_PDU_EMITTERNAME_YJ_12_MH, "YJ-12 MH" },
     { DIS_PDU_EMITTERNAME_YJ_62_MH, "YJ-62 MH" },
     { DIS_PDU_EMITTERNAME_YJ_82_MH, "YJ-82 MH" },
     { DIS_PDU_EMITTERNAME_YJ_83_MH, "YJ-83 MH" },
     { DIS_PDU_EMITTERNAME_YJ_63, "YJ-63" },
     { DIS_PDU_EMITTERNAME_YLC_2, "YLC-2" },
     { DIS_PDU_EMITTERNAME_YLC_2A, "YLC-2A" },
     { DIS_PDU_EMITTERNAME_YLC_4, "YLC-4" },
     { DIS_PDU_EMITTERNAME_YLC_6, "YLC-6" },
     { DIS_PDU_EMITTERNAME_YLC_6M, "YLC-6M" },
     { DIS_PDU_EMITTERNAME_YLC_8, "YLC-8" },
     { DIS_PDU_EMITTERNAME_YLC_8B, "YLC-8B" },
     { DIS_PDU_EMITTERNAME_YLC_18_TALL_PAGE, "YLC-18 Tall Page" },
     { DIS_PDU_EMITTERNAME_YO_YO, " Yo-Yo" },
     { DIS_PDU_EMITTERNAME_ZASLON_A, "Zaslon-A" },
     { DIS_PDU_EMITTERNAME_ZASLON_MULTI_PURPOSE_X_AND_S_BAND, "Zaslon Multi-purpose (X- and S-band)" },
     { DIS_PDU_EMITTERNAME_ZOO_PARK_1, "Zoo Park 1" },
     { DIS_PDU_EMITTERNAME_ZPS_6, "ZPS-6" },
     { DIS_PDU_EMITTERNAME_ZOOPARK_3, "ZOOPARK-3" },
     { DIS_PDU_EMITTERNAME_ZOOPARK_1M, "ZOOPARK-1M" },
     { DIS_PDU_EMITTERNAME_ZD_12, "ZD-12" },
     { DIS_PDU_EMITTERNAME_ZW_06, "ZW-06" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_136_V_1, "AN/ALQ-136(V)1" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_136_V_2, "AN/ALQ-136(V)2" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_136_V_3, "AN/ALQ-136(V)3" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_136_V_4, "AN/ALQ-136(V)4" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_136_V_5, "AN/ALQ-136(V)5" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_162_V_2, "AN/ALQ-162(V)2" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_162_V_3, "AN/ALQ-162(V)3" },
     { DIS_PDU_EMITTERNAME_AN_ALQ_162_V_4, "AN/ALQ-162(V)4" },
     { DIS_PDU_EMITTERNAME_ZHUK_M, "Zhuk-M" },
     { DIS_PDU_EMITTERNAME_ZHUK_MAE, "ZHUK-MAE" },
     { DIS_PDU_EMITTERNAME_ZHUK_ME, "ZHUK-ME" },
     { DIS_PDU_EMITTERNAME_ZHUK_MME, "ZHUK-MME" },
     { DIS_PDU_EMITTERNAME_ZHUK_MSE, "Zhuk-MSE" },
     { 0, NULL }
 };

static value_string_ext DIS_PDU_EmitterName_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_EmitterName_Strings);

/* SISO-REF-010 [UID 76] */
typedef enum
{
     DIS_EMISSION_FUNCTION_OTHER = 0,
     DIS_EMISSION_FUNCTION_MULTI_FUNCTION = 1,
     DIS_EMISSION_FUNCTION_EARLY_WARNING_SURVEILLANCE = 2,
     DIS_EMISSION_FUNCTION_HEIGHT_FINDER = 3,
     DIS_EMISSION_FUNCTION_FIRE_CONTROL = 4,
     DIS_EMISSION_FUNCTION_ACQUISITION_DETECTION = 5,
     DIS_EMISSION_FUNCTION_TRACKER = 6,
     DIS_EMISSION_FUNCTION_GUIDANCE_ILLUMINATION = 7,
     DIS_EMISSION_FUNCTION_FIRING_POINT_LAUNCH_POINT_LOCATION = 8,
     DIS_EMISSION_FUNCTION_RANGE_ONLY = 9,
     DIS_EMISSION_FUNCTION_RADAR_ALTIMETER = 10,
     DIS_EMISSION_FUNCTION_IMAGING = 11,
     DIS_EMISSION_FUNCTION_MOTION_DETECTION = 12,
     DIS_EMISSION_FUNCTION_NAVIGATION = 13,
     DIS_EMISSION_FUNCTION_WEATHER_METEOROLOGICAL = 14,
     DIS_EMISSION_FUNCTION_INSTRUMENTATION = 15,
     DIS_EMISSION_FUNCTION_IDENTIFICATION_CLASSIFICATION_INCLUDING_IFF = 16,
     DIS_EMISSION_FUNCTION_AAA_ANTI_AIRCRAFT_ARTILLERY_FIRE_CONTROL = 17,
     DIS_EMISSION_FUNCTION_AIR_SEARCH_BOMB = 18,
     DIS_EMISSION_FUNCTION_AIR_INTERCEPT = 19,
     DIS_EMISSION_FUNCTION_ALTIMETER = 20,
     DIS_EMISSION_FUNCTION_AIR_MAPPING = 21,
     DIS_EMISSION_FUNCTION_AIR_TRAFFIC_CONTROL = 22,
     DIS_EMISSION_FUNCTION_BEACON = 23,
     DIS_EMISSION_FUNCTION_BATTLEFIELD_SURVEILLANCE = 24,
     DIS_EMISSION_FUNCTION_GROUND_CONTROL_APPROACH = 25,
     DIS_EMISSION_FUNCTION_GROUND_CONTROL_INTERCEPT = 26,
     DIS_EMISSION_FUNCTION_COASTAL_SURVEILLANCE = 27,
     DIS_EMISSION_FUNCTION_DECOY_MIMIC = 28,
     DIS_EMISSION_FUNCTION_DATA_TRANSMISSION = 29,
     DIS_EMISSION_FUNCTION_EARTH_SURVEILLANCE = 30,
     DIS_EMISSION_FUNCTION_GUN_LAY_BEACON = 31,
     DIS_EMISSION_FUNCTION_GROUND_MAPPING = 32,
     DIS_EMISSION_FUNCTION_HARBOR_SURVEILLANCE = 33,
     DIS_EMISSION_FUNCTION_ILS_INSTRUMENT_LANDING_SYSTEM = 35,
     DIS_EMISSION_FUNCTION_IONOSPHERIC_SOUND = 36,
     DIS_EMISSION_FUNCTION_INTERROGATOR = 37,
     DIS_EMISSION_FUNCTION_JAMMER = 42,
     DIS_EMISSION_FUNCTION_MISSILE_ACQUISITION = 47,
     DIS_EMISSION_FUNCTION_MISSILE_DOWNLINK = 48,
     DIS_EMISSION_FUNCTION_SPACE = 50,
     DIS_EMISSION_FUNCTION_SURFACE_SEARCH = 51,
     DIS_EMISSION_FUNCTION_SHELL_TRACKING = 52,
     DIS_EMISSION_FUNCTION_TELEVISION = 56,
     DIS_EMISSION_FUNCTION_UNKNOWN = 57,
     DIS_EMISSION_FUNCTION_VIDEO_REMOTING = 58,
     DIS_EMISSION_FUNCTION_EXPERIMENTAL_OR_TRAINING = 59,
     DIS_EMISSION_FUNCTION_MISSILE_GUIDANCE = 60,
     DIS_EMISSION_FUNCTION_MISSILE_HOMING = 61,
     DIS_EMISSION_FUNCTION_MISSILE_TRACKING = 62,
     DIS_EMISSION_FUNCTION_NAVIGATION_DISTANCE_MEASURING_EQUIPMENT = 71,
     DIS_EMISSION_FUNCTION_TERRAIN_FOLLOWING = 72,
     DIS_EMISSION_FUNCTION_WEATHER_AVOIDANCE = 73,
     DIS_EMISSION_FUNCTION_PROXIMITY_FUSE = 74,
     DIS_EMISSION_FUNCTION_RADIOSONDE = 76,
     DIS_EMISSION_FUNCTION_SONOBUOY = 77,
     DIS_EMISSION_FUNCTION_BATHYTHERMAL_SENSOR = 78,
     DIS_EMISSION_FUNCTION_TOWED_COUNTER_MEASURE = 79,
     DIS_EMISSION_FUNCTION_DIPPING_SONAR = 80,
     DIS_EMISSION_FUNCTION_TOWED_ACOUSTIC_SENSOR = 81,
     DIS_EMISSION_FUNCTION_WEAPON_NON_LETHAL = 96,
     DIS_EMISSION_FUNCTION_WEAPON_LETHAL = 97,
     DIS_EMISSION_FUNCTION_TEST_EQUIPMENT = 98,
     DIS_EMISSION_FUNCTION_ACQUISITION_TRACK = 99,
     DIS_EMISSION_FUNCTION_TRACK_GUIDANCE = 100,
     DIS_EMISSION_FUNCTION_GUIDANCE_ILLUMINATION_TRACK_ACQUISITION = 101,
     DIS_EMISSION_FUNCTION_SEARCH_ACQUISITION = 102,
     DIS_EMISSION_FUNCTION_DROPSONDE = 103
}  DIS_PDU_Emitter_System_Function;

static const value_string DIS_PDU_Emitter_System_Function_Strings[] = {
     { DIS_EMISSION_FUNCTION_OTHER, "Other" },
     { DIS_EMISSION_FUNCTION_MULTI_FUNCTION, "Multi-function" },
     { DIS_EMISSION_FUNCTION_EARLY_WARNING_SURVEILLANCE, "Early Warning/Surveillance" },
     { DIS_EMISSION_FUNCTION_HEIGHT_FINDER, "Height Finder" },
     { DIS_EMISSION_FUNCTION_FIRE_CONTROL, "Fire Control" },
     { DIS_EMISSION_FUNCTION_ACQUISITION_DETECTION, "Acquisition/Detection" },
     { DIS_EMISSION_FUNCTION_TRACKER, "Tracker" },
     { DIS_EMISSION_FUNCTION_GUIDANCE_ILLUMINATION, "Guidance/Illumination" },
     { DIS_EMISSION_FUNCTION_FIRING_POINT_LAUNCH_POINT_LOCATION, "Firing point/launch point location" },
     { DIS_EMISSION_FUNCTION_RANGE_ONLY, "Range-Only" },
     { DIS_EMISSION_FUNCTION_RADAR_ALTIMETER, "Radar Altimeter" },
     { DIS_EMISSION_FUNCTION_IMAGING, "Imaging" },
     { DIS_EMISSION_FUNCTION_MOTION_DETECTION, "Motion Detection" },
     { DIS_EMISSION_FUNCTION_NAVIGATION, "Navigation" },
     { DIS_EMISSION_FUNCTION_WEATHER_METEOROLOGICAL, "Weather / Meteorological" },
     { DIS_EMISSION_FUNCTION_INSTRUMENTATION, "Instrumentation" },
     { DIS_EMISSION_FUNCTION_IDENTIFICATION_CLASSIFICATION_INCLUDING_IFF, "Identification/Classification (including IFF)" },
     { DIS_EMISSION_FUNCTION_AAA_ANTI_AIRCRAFT_ARTILLERY_FIRE_CONTROL, "AAA (Anti-Aircraft Artillery) Fire Control" },
     { DIS_EMISSION_FUNCTION_AIR_SEARCH_BOMB, "Air Search/Bomb" },
     { DIS_EMISSION_FUNCTION_AIR_INTERCEPT, "Air Intercept" },
     { DIS_EMISSION_FUNCTION_ALTIMETER, "Altimeter" },
     { DIS_EMISSION_FUNCTION_AIR_MAPPING, "Air Mapping" },
     { DIS_EMISSION_FUNCTION_AIR_TRAFFIC_CONTROL, "Air Traffic Control" },
     { DIS_EMISSION_FUNCTION_BEACON, "Beacon" },
     { DIS_EMISSION_FUNCTION_BATTLEFIELD_SURVEILLANCE, "Battlefield Surveillance" },
     { DIS_EMISSION_FUNCTION_GROUND_CONTROL_APPROACH, "Ground Control Approach" },
     { DIS_EMISSION_FUNCTION_GROUND_CONTROL_INTERCEPT, "Ground Control Intercept" },
     { DIS_EMISSION_FUNCTION_COASTAL_SURVEILLANCE, "Coastal Surveillance" },
     { DIS_EMISSION_FUNCTION_DECOY_MIMIC, "Decoy/Mimic" },
     { DIS_EMISSION_FUNCTION_DATA_TRANSMISSION, "Data Transmission" },
     { DIS_EMISSION_FUNCTION_EARTH_SURVEILLANCE, "Earth Surveillance" },
     { DIS_EMISSION_FUNCTION_GUN_LAY_BEACON, "Gun Lay Beacon" },
     { DIS_EMISSION_FUNCTION_GROUND_MAPPING, "Ground Mapping" },
     { DIS_EMISSION_FUNCTION_HARBOR_SURVEILLANCE, "Harbor Surveillance" },
     { DIS_EMISSION_FUNCTION_ILS_INSTRUMENT_LANDING_SYSTEM, "ILS (Instrument Landing System)" },
     { DIS_EMISSION_FUNCTION_IONOSPHERIC_SOUND, "Ionospheric Sound" },
     { DIS_EMISSION_FUNCTION_INTERROGATOR, "Interrogator" },
     { DIS_EMISSION_FUNCTION_JAMMER, "Jammer" },
     { DIS_EMISSION_FUNCTION_MISSILE_ACQUISITION, "Missile Acquisition" },
     { DIS_EMISSION_FUNCTION_MISSILE_DOWNLINK, "Missile Downlink" },
     { DIS_EMISSION_FUNCTION_SPACE, "Space" },
     { DIS_EMISSION_FUNCTION_SURFACE_SEARCH, "Surface Search" },
     { DIS_EMISSION_FUNCTION_SHELL_TRACKING, "Shell Tracking" },
     { DIS_EMISSION_FUNCTION_TELEVISION, "Television" },
     { DIS_EMISSION_FUNCTION_UNKNOWN, "Unknown" },
     { DIS_EMISSION_FUNCTION_VIDEO_REMOTING, "Video Remoting" },
     { DIS_EMISSION_FUNCTION_EXPERIMENTAL_OR_TRAINING, "Experimental or Training" },
     { DIS_EMISSION_FUNCTION_MISSILE_GUIDANCE, "Missile Guidance" },
     { DIS_EMISSION_FUNCTION_MISSILE_HOMING, "Missile Homing" },
     { DIS_EMISSION_FUNCTION_MISSILE_TRACKING, "Missile Tracking" },
     { DIS_EMISSION_FUNCTION_NAVIGATION_DISTANCE_MEASURING_EQUIPMENT, "Navigation/Distance Measuring Equipment" },
     { DIS_EMISSION_FUNCTION_TERRAIN_FOLLOWING, "Terrain Following" },
     { DIS_EMISSION_FUNCTION_WEATHER_AVOIDANCE, "Weather Avoidance" },
     { DIS_EMISSION_FUNCTION_PROXIMITY_FUSE, "Proximity Fuse" },
     { DIS_EMISSION_FUNCTION_RADIOSONDE, "Radiosonde" },
     { DIS_EMISSION_FUNCTION_SONOBUOY, "Sonobuoy" },
     { DIS_EMISSION_FUNCTION_BATHYTHERMAL_SENSOR, "Bathythermal Sensor" },
     { DIS_EMISSION_FUNCTION_TOWED_COUNTER_MEASURE, "Towed Counter Measure" },
     { DIS_EMISSION_FUNCTION_DIPPING_SONAR, "Dipping Sonar" },
     { DIS_EMISSION_FUNCTION_TOWED_ACOUSTIC_SENSOR, "Towed Acoustic Sensor" },
     { DIS_EMISSION_FUNCTION_WEAPON_NON_LETHAL, "Weapon, non-lethal" },
     { DIS_EMISSION_FUNCTION_WEAPON_LETHAL, "Weapon, lethal" },
     { DIS_EMISSION_FUNCTION_TEST_EQUIPMENT, "Test Equipment" },
     { DIS_EMISSION_FUNCTION_ACQUISITION_TRACK, "Acquisition Track" },
     { DIS_EMISSION_FUNCTION_TRACK_GUIDANCE, "Track Guidance" },
     { DIS_EMISSION_FUNCTION_GUIDANCE_ILLUMINATION_TRACK_ACQUISITION, "Guidance Illumination Track Acquisition" },
     { DIS_EMISSION_FUNCTION_SEARCH_ACQUISITION, "Search Acquisition" },
     { DIS_EMISSION_FUNCTION_DROPSONDE, "Dropsonde" },
     { 0, NULL }
 };

static value_string_ext DIS_PDU_Emitter_System_Function_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_Emitter_System_Function_Strings);

/* SISO-REF-010 [UID 77] */
typedef enum
{
     DIS_PDU_ELECTROMAGNETIC_EMISSION_STATE_UPDATE_INDICATOR_HEARTBEAT_UPDATE = 0,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_STATE_UPDATE_INDICATOR_CHANGED_DATA_UPDATE = 1
}  DIS_PDU_Electromagnetic_Emission_State_Update_Indicator;

/* SISO-REF-010 [UID 78] */
typedef enum
{
     DIS_BEAM_FUNCTION_OTHER = 0,
     DIS_BEAM_FUNCTION_SEARCH = 1,
     DIS_BEAM_FUNCTION_HEIGHT_FINDING = 2,
     DIS_BEAM_FUNCTION_ACQUISITION = 3,
     DIS_BEAM_FUNCTION_TRACKING = 4,
     DIS_BEAM_FUNCTION_ACQUISITION_AND_TRACKING = 5,
     DIS_BEAM_FUNCTION_COMMAND_GUIDANCE = 6,
     DIS_BEAM_FUNCTION_ILLUMINATION = 7,
     DIS_BEAM_FUNCTION_RANGING = 8,
     DIS_BEAM_FUNCTION_MISSILE_BEACON = 9,
     DIS_BEAM_FUNCTION_MISSILE_FUSING = 10,
     DIS_BEAM_FUNCTION_ACTIVE_RADAR_MISSILE_SEEKER = 11,
     DIS_BEAM_FUNCTION_JAMMING = 12,
     DIS_BEAM_FUNCTION_IFF = 13,
     DIS_BEAM_FUNCTION_NAVIGATION_WEATHER = 14,
     DIS_BEAM_FUNCTION_METEOROLOGICAL = 15,
     DIS_BEAM_FUNCTION_DATA_TRANSMISSION = 16,
     DIS_BEAM_FUNCTION_NAVIGATIONAL_DIRECTIONAL_BEACON = 17,
     DIS_BEAM_FUNCTION_IR_JAMMER = 18,
     DIS_BEAM_FUNCTION_GROUND_MAPPING = 19,
     DIS_BEAM_FUNCTION_TIME_SHARED_SEARCH = 20,
     DIS_BEAM_FUNCTION_TIME_SHARED_ACQUISITION = 21,
     DIS_BEAM_FUNCTION_TIME_SHARED_TRACK = 22,
     DIS_BEAM_FUNCTION_TIME_SHARED_COMMAND_GUIDANCE = 23,
     DIS_BEAM_FUNCTION_TIME_SHARED_ILLUMINATION = 24,
     DIS_BEAM_FUNCTION_TIME_SHARED_JAMMING = 25
}  DIS_PDU_Electromagnetic_Emission_Beam_Function;

static const value_string DIS_PDU_Electromagnetic_Emission_Beam_Function_Strings[] = {
     { DIS_BEAM_FUNCTION_OTHER, "Other" },
     { DIS_BEAM_FUNCTION_SEARCH, "Search" },
     { DIS_BEAM_FUNCTION_HEIGHT_FINDING, "Height Finding" },
     { DIS_BEAM_FUNCTION_ACQUISITION, "Acquisition" },
     { DIS_BEAM_FUNCTION_TRACKING, "Tracking" },
     { DIS_BEAM_FUNCTION_ACQUISITION_AND_TRACKING, "Acquisition and tracking" },
     { DIS_BEAM_FUNCTION_COMMAND_GUIDANCE, "Command guidance" },
     { DIS_BEAM_FUNCTION_ILLUMINATION, "Illumination" },
     { DIS_BEAM_FUNCTION_RANGING, "Ranging" },
     { DIS_BEAM_FUNCTION_MISSILE_BEACON, "Missile beacon" },
     { DIS_BEAM_FUNCTION_MISSILE_FUSING, "Missile Fusing" },
     { DIS_BEAM_FUNCTION_ACTIVE_RADAR_MISSILE_SEEKER, "Active radar missile seeker" },
     { DIS_BEAM_FUNCTION_JAMMING, "Jamming" },
     { DIS_BEAM_FUNCTION_IFF, "IFF" },
     { DIS_BEAM_FUNCTION_NAVIGATION_WEATHER, "Navigation / Weather" },
     { DIS_BEAM_FUNCTION_METEOROLOGICAL, "Meteorological" },
     { DIS_BEAM_FUNCTION_DATA_TRANSMISSION, "Data transmission" },
     { DIS_BEAM_FUNCTION_NAVIGATIONAL_DIRECTIONAL_BEACON, "Navigational directional beacon" },
     { DIS_BEAM_FUNCTION_IR_JAMMER, "IR Jammer" },
     { DIS_BEAM_FUNCTION_GROUND_MAPPING, "Ground Mapping" },
     { DIS_BEAM_FUNCTION_TIME_SHARED_SEARCH, "Time-Shared Search" },
     { DIS_BEAM_FUNCTION_TIME_SHARED_ACQUISITION, "Time-Shared Acquisition" },
     { DIS_BEAM_FUNCTION_TIME_SHARED_TRACK, "Time-Shared Track" },
     { DIS_BEAM_FUNCTION_TIME_SHARED_COMMAND_GUIDANCE, "Time-Shared Command Guidance" },
     { DIS_BEAM_FUNCTION_TIME_SHARED_ILLUMINATION, "Time-Shared Illumination" },
     { DIS_BEAM_FUNCTION_TIME_SHARED_JAMMING, "Time-Shared Jamming" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 79] */
typedef enum
{
     DIS_PDU_HIGH_DENSITY_TRACK_JAM_NOT_SELECTED = 0,
     DIS_PDU_HIGH_DENSITY_TRACK_JAM_SELECTED = 1
}  DIS_PDU_High_Density_Track_Jam;

/* SISO-REF-010 [UID 80] */
typedef enum
{
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_NOT_SPECIFIED = 0,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_4 = 1000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_7 = 1100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_8 = 1200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_14_LANTIRN = 1300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_19 = 1400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_22A_SAFIRE = 1500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_22B_SAFIRE_LP = 1600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_22C_STAR_SAFIRE_I = 1700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_22D_BRITE_STAR = 1800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_24_V_DIRCM_NEMESIS = 1900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_25_LTS = 2000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_28_V_LITENING_II = 2100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_30 = 2200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_32 = 2300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_33_SNIPER = 2400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_37 = 2500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_38 = 2600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_40 = 2650,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_32 = 2700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_35V = 2800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_37 = 2900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_38 = 3000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_44_V = 3100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_46 = 3200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_49 = 3300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_51 = 3400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_52_MTS_A = 3500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_ALQ_10 = 3600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_ASQ_228 = 3700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AVQ_25 = 4400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AVQ_26 = 4500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_GVS_5 = 4600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_PED_1_LLDR = 4700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TADS_LRF_D = 4800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_MMS_LRF_D = 4900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AH_1_C_NITE = 5000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_MATES = 5100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TCV_115 = 5200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TIM = 5300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TMS_303 = 5400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TMY_303 = 5500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_ALRAD = 5600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_RFTDL = 5700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_VVLR = 5800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_P0705_HELL = 6000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_P0708_PULSE = 6100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_HELD = 6200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TYPE_105 = 6300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TYPE_118 = 6400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TYPE_121 = 6500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TYPE_126 = 6600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TYPE_629 = 6700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_CLDS = 6800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TAV_38 = 6900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TMV_630 = 7000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_ALTM_1020 = 7100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_ALATS = 7200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_DARK_STAR_LAMPS = 7300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_GLTD_II = 7400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_MBT_ELRF = 7500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_MARK_VII = 7600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_SIRE_V = 7700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_16B = 7800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_16D_AESOP = 7900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_21_STAR_SAFIRE_III = 8000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_22E_BRITE_STAR = 8100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAQ_36_STAR_SAFIRE_II = 8200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_38A_NITE_HAWK = 8300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_38B_NITE_HAWK = 8400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_44C_V = 8500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_AAS_53_CSP = 8600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_ASQ_28_ATFLIR = 8700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_DAS_1_MTS_B = 8800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_PAQ_1_LTD = 8900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_PAQ_3_MULE = 9000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_PEQ_1_SOFLAM = 9090,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_PEQ_3 = 9100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_PEQ_15_ATPIAL = 9140,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_PEQ_18_IZLID_1000P = 9150,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_TVQ_2_G_VLLD = 9200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_ZSQ_2_V_1_EOS = 9300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_AN_ZSQ_2_V_2_EOS = 9400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_CIRCM = 9500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_GUARDIAN = 9600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_IZLID_200P = 9700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_IZLID_1000P_W = 9800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_MMS = 9900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_M_TADS_PNVS_ARROWHEAD = 10000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_RBS_70 = 10100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_RBS_90 = 10200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TADS_PNVS = 10300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_COLIBRI = 10400,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_DAMOCLES = 10500,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_I_251_SHKVAL = 10600,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_KPS_53AV_EOTS = 10700,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_STAR_SAFIRE_380 = 10800,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_JANUS_T_EOS = 10900,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_LOTHAR_EOS = 11000,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_MK46_MOD_1_EOS = 11100,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_MTK_201ME_EOS = 11200,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_THALES_MIRADOR_MK2_EOS = 11300,
     DIS_PDU_DESIGNATOR_SYSTEM_NAME_TPN_1M_49_23_EOS = 11400
}  DIS_PDU_Designator_System_Name;

/* SISO-REF-010 [UID 82] */
typedef enum
{
     DIS_PDU_IFF_SYSTEM_TYPE_NOT_USED_INVALID_VALUE = 0,
     DIS_PDU_IFF_SYSTEM_TYPE_MARK_X_XII_ATCRBS_TRANSPONDER = 1,
     DIS_PDU_IFF_SYSTEM_TYPE_MARK_X_XII_ATCRBS_INTERROGATOR = 2,
     DIS_PDU_IFF_SYSTEM_TYPE_SOVIET_TRANSPONDER = 3,
     DIS_PDU_IFF_SYSTEM_TYPE_SOVIET_INTERROGATOR = 4,
     DIS_PDU_IFF_SYSTEM_TYPE_RRB_TRANSPONDER = 5,
     DIS_PDU_IFF_SYSTEM_TYPE_MARK_XIIA_INTERROGATOR = 6,
     DIS_PDU_IFF_SYSTEM_TYPE_MODE_5_INTERROGATOR = 7,
     DIS_PDU_IFF_SYSTEM_TYPE_MODE_S_INTERROGATOR = 8,
     DIS_PDU_IFF_SYSTEM_TYPE_MARK_XIIA_TRANSPONDER = 9,
     DIS_PDU_IFF_SYSTEM_TYPE_MODE_5_TRANSPONDER = 10,
     DIS_PDU_IFF_SYSTEM_TYPE_MODE_S_TRANSPONDER = 11,
     DIS_PDU_IFF_SYSTEM_TYPE_MARK_XIIA_COMBINED_INTERROGATOR_TRANSPONDER_CIT = 12,
     DIS_PDU_IFF_SYSTEM_TYPE_MARK_XII_COMBINED_INTERROGATOR_TRANSPONDER_CIT = 13,
     DIS_PDU_IFF_SYSTEM_TYPE_TCAS_ACAS_TRANSCEIVER = 14
}  DIS_PDU_IFF_System_Type;

static const value_string DIS_PDU_IFF_System_Type_Strings[] = {
     { DIS_PDU_IFF_SYSTEM_TYPE_NOT_USED_INVALID_VALUE, "Not Used (Invalid Value)" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MARK_X_XII_ATCRBS_TRANSPONDER, "Mark X/XII/ATCRBS Transponder" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MARK_X_XII_ATCRBS_INTERROGATOR, "Mark X/XII/ATCRBS Interrogator" },
     { DIS_PDU_IFF_SYSTEM_TYPE_SOVIET_TRANSPONDER, "Soviet Transponder" },
     { DIS_PDU_IFF_SYSTEM_TYPE_SOVIET_INTERROGATOR, "Soviet Interrogator" },
     { DIS_PDU_IFF_SYSTEM_TYPE_RRB_TRANSPONDER, "RRB Transponder" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MARK_XIIA_INTERROGATOR, "Mark XIIA Interrogator" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MODE_5_INTERROGATOR, "Mode 5 Interrogator" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MODE_S_INTERROGATOR, "Mode S Interrogator" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MARK_XIIA_TRANSPONDER, "Mark XIIA Transponder" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MODE_5_TRANSPONDER, "Mode 5 Transponder" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MODE_S_TRANSPONDER, "Mode S Transponder" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MARK_XIIA_COMBINED_INTERROGATOR_TRANSPONDER_CIT, "Mark XIIA Combined Interrogator/Transponder (CIT)" },
     { DIS_PDU_IFF_SYSTEM_TYPE_MARK_XII_COMBINED_INTERROGATOR_TRANSPONDER_CIT, "Mark XII Combined Interrogator/Transponder (CIT)" },
     { DIS_PDU_IFF_SYSTEM_TYPE_TCAS_ACAS_TRANSCEIVER, "TCAS/ACAS Transceiver" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 83] */
typedef enum
{
     DIS_PDU_IFF_SYSTEM_NAME_NOT_USED_INVALID_VALUE = 0,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X = 1,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XII = 2,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_ATCRBS = 3,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_SOVIET = 4,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MODE_S = 5,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_XII_ATCRBS = 6,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_XII_ATCRBS_MODE_S = 7,
     DIS_PDU_IFF_SYSTEM_NAME_ARI_5954_RRB = 8,
     DIS_PDU_IFF_SYSTEM_NAME_ARI_5983_RRB = 9,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_RRB = 10,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XIIA = 11,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MODE_5 = 12,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XIIA_COMBINED_INTERROGATOR_TRANSPONDER_CIT = 13,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XII_COMBINED_INTERROGATOR_TRANSPONDER_CIT = 14,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_TCAS_I_ACAS_I_TRANSCEIVER = 15,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_TCAS_II_ACAS_II_TRANSCEIVER = 16,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_A = 17,
     DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_SIF = 18
}  DIS_PDU_IFF_System_Name;

static const value_string DIS_PDU_IFF_System_Name_Strings[] = {
     { DIS_PDU_IFF_SYSTEM_NAME_NOT_USED_INVALID_VALUE, "Not Used (Invalid Value)" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X, "Generic Mark X" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XII, "Generic Mark XII" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_ATCRBS, "Generic ATCRBS" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_SOVIET, "Generic Soviet" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MODE_S, "Generic Mode S" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_XII_ATCRBS, "Generic Mark X/XII/ATCRBS" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_XII_ATCRBS_MODE_S, "Generic Mark X/XII/ATCRBS/Mode S" },
     { DIS_PDU_IFF_SYSTEM_NAME_ARI_5954_RRB, "ARI 5954 (RRB)" },
     { DIS_PDU_IFF_SYSTEM_NAME_ARI_5983_RRB, "ARI 5983 (RRB)" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_RRB, "Generic RRB" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XIIA, "Generic Mark XIIA" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MODE_5, "Generic Mode 5" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XIIA_COMBINED_INTERROGATOR_TRANSPONDER_CIT, "Generic Mark XIIA Combined Interrogator/Transponder (CIT)" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_XII_COMBINED_INTERROGATOR_TRANSPONDER_CIT, "Generic Mark XII Combined Interrogator/Transponder (CIT)" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_TCAS_I_ACAS_I_TRANSCEIVER, "Generic TCAS I/ACAS I Transceiver" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_TCAS_II_ACAS_II_TRANSCEIVER, "Generic TCAS II/ACAS II Transceiver" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_A, "Generic Mark X (A)" },
     { DIS_PDU_IFF_SYSTEM_NAME_GENERIC_MARK_X_SIF, "Generic Mark X (SIF)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 84] */
typedef enum
{
     DIS_PDU_IFF_SYSTEM_MODE_NO_STATEMENT = 0,
     DIS_PDU_IFF_SYSTEM_MODE_OFF = 1,
     DIS_PDU_IFF_SYSTEM_MODE_STANDBY = 2,
     DIS_PDU_IFF_SYSTEM_MODE_NORMAL = 3,
     DIS_PDU_IFF_SYSTEM_MODE_EMERGENCY = 4,
     DIS_PDU_IFF_SYSTEM_MODE_LOW_OR_LOW_SENSITIVITY = 5
}  DIS_PDU_IFF_System_Mode;

static const value_string DIS_PDU_IFF_System_Mode_Strings[] = {
     { DIS_PDU_IFF_SYSTEM_MODE_NO_STATEMENT, "No Statement" },
     { DIS_PDU_IFF_SYSTEM_MODE_OFF, "Off" },
     { DIS_PDU_IFF_SYSTEM_MODE_STANDBY, "Standby" },
     { DIS_PDU_IFF_SYSTEM_MODE_NORMAL, "Normal" },
     { DIS_PDU_IFF_SYSTEM_MODE_EMERGENCY, "Emergency" },
     { DIS_PDU_IFF_SYSTEM_MODE_LOW_OR_LOW_SENSITIVITY, "Low or Low Sensitivity" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 87] */
typedef enum
{
     DIS_PDU_IFF_LAYER_SPECIFIC_INFORMATION_NO_LAYER_SPECIFIC_INFORMATION_IS_PRESENT = 0
}  DIS_PDU_IFF_Layer_Specific_Information;

/* SISO-REF-010 [UID 96] */
typedef enum
{
     DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_NO_STATEMENT = 0,
     DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_VALID = 1,
     DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_INVALID = 2,
     DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_NO_RESPONSE = 3,
     DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_UNABLE_TO_VERIFY = 4
}  DIS_PDU_IFF_Alternate_Mode_4_Challenge_Reply;

static const value_string DIS_PDU_IFF_Alternate_Mode_4_Challenge_Reply_Strings[] = {
     { DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_NO_STATEMENT, "No Statement" },
     { DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_VALID, "Valid" },
     { DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_INVALID, "Invalid" },
     { DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_NO_RESPONSE, "No response" },
     { DIS_PDU_IFF_ALTERNATE_MODE_4_CHALLENGE_REPLY_UNABLE_TO_VERIFY, "Unable to Verify" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 97] */
typedef enum
{
     DIS_PDU_IFF_SYSTEM_TYPE_1_OPERATIONAL_PARAMETER_1_NO_OPERATIONAL_PARAMETER_1_DATA = 0
}  DIS_PDU_IFF_System_Type_1_Operational_Parameter_1;

/* SISO-REF-010 [UID 98] */
typedef enum
{
     DIS_PDU_IFF_SYSTEM_TYPE_1_OPERATIONAL_PARAMETER_2_NO_OPERATIONAL_PARAMETER_2_DATA = 0
}  DIS_PDU_IFF_System_Type_1_Operational_Parameter_2;

/* SISO-REF-010 [UID 99] */
typedef enum
{
     DIS_PDU_EXTENSION_RECORD_TYPES_ARTICULATED_PARTS = 2000,
     DIS_PDU_EXTENSION_RECORD_TYPES_ATTACHED_PARTS = 2001,
     DIS_PDU_EXTENSION_RECORD_TYPES_ENTITY_ASSOCIATION = 2002,
     DIS_PDU_EXTENSION_RECORD_TYPES_ENTITY_TYPE = 2003,
     DIS_PDU_EXTENSION_RECORD_TYPES_SEPARATION = 2004,
     DIS_PDU_EXTENSION_RECORD_TYPES_POINT_OBJECT = 2005,
     DIS_PDU_EXTENSION_RECORD_TYPES_LINEAR_OBJECT = 2006,
     DIS_PDU_EXTENSION_RECORD_TYPES_AREAL_OBJECT = 2007,
     DIS_PDU_EXTENSION_RECORD_TYPES_COMBINED_PARABOLIC_CIRCULAR_DRA = 2010,
     DIS_PDU_EXTENSION_RECORD_TYPES_MULTIPLE_STATIC_ENTITY = 2020,
     DIS_PDU_EXTENSION_RECORD_TYPES_MULTIPLE_MOVING_ENTITY = 2021,
     DIS_PDU_EXTENSION_RECORD_TYPES_MULTIPLE_ACCELERATING_ENTITY = 2022,
     DIS_PDU_EXTENSION_RECORD_TYPES_MULTIPLE_POINT_OBJECT = 2023,
     DIS_PDU_EXTENSION_RECORD_TYPES_IS_PART_OF = 2030,
     DIS_PDU_EXTENSION_RECORD_TYPES_COLLISION_ELASTIC = 2500,
     DIS_PDU_EXTENSION_RECORD_TYPES_HIGH_FIDELITY_HAVE_QUICK = 3000,
     DIS_PDU_EXTENSION_RECORD_TYPES_ANTENNA_LOCATION = 3001,
     DIS_PDU_EXTENSION_RECORD_TYPES_RELATIVE_ANTENNA_LOCATION = 3002,
     DIS_PDU_EXTENSION_RECORD_TYPES_CRYPTO_SYSTEM = 3003,
     DIS_PDU_EXTENSION_RECORD_TYPES_BEAM_ANTENNA_PATTERN = 3004,
     DIS_PDU_EXTENSION_RECORD_TYPES_ENCODED_AUDIO = 3100,
     DIS_PDU_EXTENSION_RECORD_TYPES_DATABASE_INDEX = 3101,
     DIS_PDU_EXTENSION_RECORD_TYPES_INTERCOM_ENTITY_DESTINATION = 3200,
     DIS_PDU_EXTENSION_RECORD_TYPES_INTERCOM_GROUP_DESTINATION = 3201,
     DIS_PDU_EXTENSION_RECORD_TYPES_INTERCOM_GROUP_ASSIGNMENT = 3202,
     DIS_PDU_EXTENSION_RECORD_TYPES_EE_BEAM = 3503,
     DIS_PDU_EXTENSION_RECORD_TYPES_JAMMER_PARAMETERS = 3504,
     DIS_PDU_EXTENSION_RECORD_TYPES_UNIVERSAL_SCAN_DATA = 3505,
     DIS_PDU_EXTENSION_RECORD_TYPES_PULSE_TABLE_DEFINITION = 3506,
     DIS_PDU_EXTENSION_RECORD_TYPES_BEAM_SHAPE_DEFINITION = 3507,
     DIS_PDU_EXTENSION_RECORD_TYPES_FUNDAMENTAL_PARAMETERS_LIST = 3508,
     DIS_PDU_EXTENSION_RECORD_TYPES_AZ_EL_LIST = 3509,
     DIS_PDU_EXTENSION_RECORD_TYPES_BEAM_ACTIVITY = 3510,
     DIS_PDU_EXTENSION_RECORD_TYPES_JAMMER_ACTIVATION = 3511,
     DIS_PDU_EXTENSION_RECORD_TYPES_JAMMER_FALSE_TARGETS = 3512,
     DIS_PDU_EXTENSION_RECORD_TYPES_JAMMER_TRIGGERED = 3513,
     DIS_PDU_EXTENSION_RECORD_TYPES_BLANKING_SECTORS = 3514,
     DIS_PDU_EXTENSION_RECORD_TYPES_ANGLE_DECEPTION_PARAMETERS = 3515,
     DIS_PDU_EXTENSION_RECORD_TYPES_RANGE_DECEPTION = 3516,
     DIS_PDU_EXTENSION_RECORD_TYPES_VELOCITY_DECEPTION = 3517,
     DIS_PDU_EXTENSION_RECORD_TYPES_FALSE_TARGETS_PARAMETERS = 3518,
     DIS_PDU_EXTENSION_RECORD_TYPES_DE_PRECISION_AIMPOINT = 4000,
     DIS_PDU_EXTENSION_RECORD_TYPES_DE_AREA_AIMPOINT_V2 = 4002,
     DIS_PDU_EXTENSION_RECORD_TYPES_DE_DAMAGE_DESCRIPTION = 4500,
     DIS_PDU_EXTENSION_RECORD_TYPES_IO_EFFECT = 5500,
     DIS_PDU_EXTENSION_RECORD_TYPES_IO_COMMUNICATIONS_NODE = 5501,
     DIS_PDU_EXTENSION_RECORD_TYPES_SUPPLY_QUANTITY = 6000,
     DIS_PDU_EXTENSION_RECORD_TYPES_LASER_BEAM = 6500,
     DIS_PDU_EXTENSION_RECORD_TYPES_UA_SHAFTS = 7000,
     DIS_PDU_EXTENSION_RECORD_TYPES_UA_ADDITIONAL_PASSIVE_ACTIVITY = 7001,
     DIS_PDU_EXTENSION_RECORD_TYPES_UA_SYSTEM = 7002,
     DIS_PDU_EXTENSION_RECORD_TYPES_CONE_1_GEOMETRY = 8001,
     DIS_PDU_EXTENSION_RECORD_TYPES_RECORD_TYPE_QUERY = 9000,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_STATE = 47100,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_MODE = 47150,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_HEALTH_STATUS_EXTENSION_RECORD = 47200,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_STATUS_TYPE_DESCRIPTION_EXTENSION_RECORD = 47210,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_STATUS_DESCRIPTION_EXTENSION_RECORD = 47220,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_HEALTH_STATUS_CAPABILITY_EXTENSION_RECORD = 47230,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_HEALTH_STATUS_HEARTBEAT_REQUEST_EXTENSION_RECORD = 47240,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_INITIALIZATION_DATA_EXTENSION_RECORD = 47300,
     DIS_PDU_EXTENSION_RECORD_TYPES_EXERCISE_ID_EXTENSION_RECORD = 47400,
     DIS_PDU_EXTENSION_RECORD_TYPES_APPLICATION_CAPABILITY_EXTENSION_RECORD = 47500,
     DIS_PDU_EXTENSION_RECORD_TYPES_DATA_QUERY_REQUEST_EXTENSION_RECORD = 47600,
     DIS_PDU_EXTENSION_RECORD_TYPES_ASSET_ID_EXTENSION_RECORD = 47700,
     DIS_PDU_EXTENSION_RECORD_TYPES_SET_CHECKPOINT_EXTENSION_RECORD = 47800,
     DIS_PDU_EXTENSION_RECORD_TYPES_RESTORE_CHECKPOINT_EXTENSION_RECORD = 47810
}  DIS_PDU_Extension_Record_Types;

/* SISO-REF-010 [UID 100] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_SMALL_DOG = 1,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_CHIHUAHUA = 2,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_MEDIUM_DOG = 10,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_AUSTRALIAN_CATTLE_DOG = 11,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_LARGE_DOG = 20,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_GERMAN_SHEPHERD = 21,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_VERY_LARGE_DOG = 30,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_GIANT_TURKISH_KANGAL = 31,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_SHEEP = 40,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_GOAT = 41,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_PIG = 50,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_COW = 60,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_OX = 61,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_OX_WITH_CART = 70,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_HORSE = 80,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_DONKEY = 81,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_MULE = 82,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_HORSE_WITH_RIDER = 90,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_HORSE_WITH_CARGO = 91,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_DONKEY_WITH_RIDER = 92,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_DONKEY_WITH_CARGO = 93,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_MULE_WITH_RIDER = 94,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_MULE_WITH_CARGO = 95,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_CAMEL = 100,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_DROMEDARY_CAMEL_ONE_HUMP = 101,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_BACTRIAN_CAMEL_TWO_HUMPS = 102,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_DROMEDARY_CAMEL_WITH_RIDER = 110,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_DROMEDARY_CAMEL_WITH_CARGO = 111,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_200_MAMMAL_RAT = 200
}  DIS_PDU_Subcategories_for_Land_Category_200_Mammal;

/* SISO-REF-010 [UID 101] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_NEW_ZEALAND_NORTHERN_TUATARA = 1,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_MONITOR = 3,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_GECKO = 8,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_IGUANA = 13,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_CHAMELEON = 17,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_NON_VENOMOUS_SNAKE = 30,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_BOA = 31,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_PYTHON = 35,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_BULLSNAKE = 39,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_KINGSNAKE = 43,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_VENOMOUS_SNAKE = 60,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_RATTLESNAKE = 61,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_COPPERHEAD = 62,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_COTTONMOUTH = 63,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_TAIPAN = 64,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_VIPER = 65,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_COBRA = 66,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_AUSTRALIAN_BROWN_SNAKE = 67,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_TORTOISE = 90,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_TURTLE = 100,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_AMERICAN_ALLIGATOR = 120,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_CROCODILE = 121,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_201_REPTILE_AUSTRALIAN_FRESHWATER_CROCODILE = 122
}  DIS_PDU_Subcategories_for_Land_Category_201_Reptile;

/* SISO-REF-010 [UID 102] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_202_AMPHIBIAN_FROG = 1,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_202_AMPHIBIAN_TOAD = 2,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_202_AMPHIBIAN_SALAMANDER = 170,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_202_AMPHIBIAN_CAECILIAN = 230
}  DIS_PDU_Subcategories_for_Land_Category_202_Amphibian;

/* SISO-REF-010 [UID 103] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_203_INSECT_BEETLE = 1,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_203_INSECT_MANTIS = 60,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_203_INSECT_COCKROACH = 70,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_203_INSECT_ARMY_ANT = 80,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_203_INSECT_FIRE_ANT = 81,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_203_INSECT_GRASSHOPPER = 90,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_203_INSECT_CENTIPEDE = 100
}  DIS_PDU_Subcategories_for_Land_Category_203_Insect;

/* SISO-REF-010 [UID 104] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_204_ARACHNID_SPIDER = 1,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_204_ARACHNID_TICK = 20,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_204_ARACHNID_SCORPION = 30,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_204_ARACHNID_HARVESTMEN = 40,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_204_ARACHNID_MITE = 50
}  DIS_PDU_Subcategories_for_Land_Category_204_Arachnid;

/* SISO-REF-010 [UID 105] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_205_MOLLUSK_SNAIL = 1,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_205_MOLLUSK_SLUG = 50
}  DIS_PDU_Subcategories_for_Land_Category_205_Mollusk;

/* SISO-REF-010 [UID 106] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_BROWN_FOUR_EYED_OPOSSUM = 1,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_BUSHY_TAILED_OPOSSUM = 2,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_TATE_S_SHREW_OPOSSUM = 90,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_GREATER_BILBY = 100,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_TASMANIAN_DEVIL = 110,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_BRUSH_TAILED_ROCK_WALLABY = 150,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_EASTERN_WALLAROO = 160,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_RED_KANGAROO = 170,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_QUEENSLAND_KOALA = 200,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_SOUTHERN_HAIRY_NOSED_WOMBAT = 205,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_BRUSHTAIL_POSSUM = 210,
     DIS_PDU_SUBCATEGORIES_FOR_LAND_CATEGORY_206_MARSUPIAL_SUGAR_GLIDER = 211
}  DIS_PDU_Subcategories_for_Land_Category_206_Marsupial;

/* SISO-REF-010 [UID 110] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_PENGUIN = 1,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_SEAGULL = 2,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_PELICAN = 3,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_ALBATROSS = 4,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_SWAN = 5,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_CORMORANT = 6,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_HERON = 7,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_CRANE = 8,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_OSPREY = 9,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_LOON = 10,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_STORK = 11,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_FLAMINGO = 12,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_DUCK = 13,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_OSTRICH = 20,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_EMU = 21,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_CHICKEN = 22,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_BLACK_BIRD = 30,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_STARLING = 31,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_BUDGERIGAR_PARAKEET = 32,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_CANADIAN_GOOSE = 40,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_CROW = 41,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_EAGLE = 50,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_VULTURE = 55,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_FALCON = 60,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_HAWK = 65,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_OWL = 70,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_200_BIRD_KITE = 80
}  DIS_PDU_Subcategories_for_Air_Category_200_Bird;

/* SISO-REF-010 [UID 111] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_MOTH = 1,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_BUTTERFLY = 2,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_FLY = 20,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_MOSQUITO = 30,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_WASP = 40,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_BEE = 50,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_BEETLE = 60,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_DRAGONFLY = 70,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_201_INSECT_LOCUST = 80
}  DIS_PDU_Subcategories_for_Air_Category_201_Insect;

/* SISO-REF-010 [UID 112] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_202_MAMMAL_BAT = 1,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_202_MAMMAL_FLYING_SQUIRREL = 10,
     DIS_PDU_SUBCATEGORIES_FOR_AIR_CATEGORY_202_MAMMAL_GLIDING_POSSUM = 20
}  DIS_PDU_Subcategories_for_Air_Category_202_Mammal;

/* SISO-REF-010 [UID 120] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_FORAGE_FISH_SMALL_SCHOOLING = 1,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_HERRING = 2,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_SARDINES = 3,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_KRILL = 4,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_SQUID = 5,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_MEDIUM_SCHOOLING_FISH = 30,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_HAKE = 31,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_COD = 32,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_HADDOCK = 33,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_MACKEREL = 34,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_LARGE_SCHOOLING_FISH = 60,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_TUNA = 61,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_SMALL_SHARK = 90,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_DOGFISH_SHARK = 91,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_MEDIUM_SHARK = 120,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_MAKO_SHARK = 121,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_HAMMERHEAD_SHARK = 122,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_LARGE_SHARK = 150,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_GREAT_WHITE_SHARK = 151,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_TIGER_SHARK = 152,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_BLUE_SHARK = 153,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_WHALE_SHARK = 154,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_SKATE = 180,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_STINGRAY = 181,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_EEL = 190,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_MARLIN = 200,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_200_FISH_SWORDFISH = 201
}  DIS_PDU_Subcategories_for_Subsurface_Category_200_Fish;

/* SISO-REF-010 [UID 121] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_WHALE = 1,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BEAKED_WHALE = 2,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BELUGA_WHALE = 3,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BLUE_WHALE = 4,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BOTTLENOSE_WHALE = 5,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_NORTHERN_BOTTLENOSE_WHALE = 6,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SOUTHERN_BOTTLENOSE_WHALE = 7,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BOWHEAD_WHALE = 8,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BRYDE_S_WHALE = 9,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_DWARF_SPERM_WHALE = 10,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_FINBACK_WHALE = 11,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_GRAY_WHALE = 12,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_HUMPBACK_WHALE = 13,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_LONG_FINNED_PILOT_WHALE = 14,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_MINKE_WHALE = 15,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_NORTHERN_MINKE_WHALE = 16,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SOUTHERN_MINKE_WHALE = 17,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_NARWHAL_WHALE = 18,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_ORCA_WHALE = 19,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_PYGMY_SPERM_WHALE = 20,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_RIGHT_WHALE = 21,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_NORTH_ATLANTIC_RIGHT_WHALE = 22,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_NORTH_PACIFIC_RIGHT_WHALE = 23,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SOUTHERN_RIGHT_WHALE = 24,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SEI_WHALE = 25,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SHORT_FINNED_PILOT_WHALE = 26,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SPERM_WHALE = 27,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_DOLPHIN = 50,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BOTTLENOSE_DOLPHIN = 51,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BOTTLENOSE_INDO_PACIFIC_DOLPHIN = 52,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BOTTLENOSE_BURRUNAN_DOLPHIN = 53,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_ATLANTIC_SPOTTED_DOLPHIN = 54,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_AUSTRALIAN_SNUBFIN_DOLPHIN = 55,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_CHILEAN_BLACK_DOLPHIN = 56,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_CHINESE_WHITE_DOLPHIN = 57,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_CLYMENE_DOLPHIN = 58,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_PORPOISE = 100,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_HARBOUR_PORPOISE = 101,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_CALIFORNIAN_PORPOISE = 102,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_DALL_S_PORPOISE = 103,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BURMEISTER_S_PORPOISE = 104,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SEAL = 120,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_BEARDED_SEAL = 121,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_HARBOR_SEAL = 122,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_FUR_SEAL = 123,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_WEDDELL_SEAL = 124,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_ELEPHANT_SEAL = 125,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SEA_LION = 130,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_AUSTRALIAN_SEA_LION = 131,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_CALIFORNIA_SEA_LION = 132,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_WALRUS = 140,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_ATLANTIC_WALRUS = 141,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_PACIFIC_WALRUS = 142,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_OTTER = 150,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SEA_OTTER = 151,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_MANATEE = 160,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_FLORIDA_MANATEE = 161,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_DUGONGS = 162,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_POLAR_BEAR = 200
}  DIS_PDU_Subcategories_for_Subsurface_Category_201_Mammal;

/* SISO-REF-010 [UID 122] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_SNAIL = 1,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_SLUG = 10,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_OCTOPUS = 20,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_SQUID = 30,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_CUTTLEFISH = 40,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_CLAM = 50,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_MUSCLE = 60,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_OYSTER = 70,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_202_MOLLUSK_SCALLOP = 80
}  DIS_PDU_Subcategories_for_Subsurface_Category_202_Mollusk;

/* SISO-REF-010 [UID 123] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_203_CRUSTACEAN_SHRIMP = 1,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_203_CRUSTACEAN_SNAPPING_SHRIMP = 2,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_203_CRUSTACEAN_CRAYFISH = 10,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_203_CRUSTACEAN_LOBSTER = 20,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_203_CRUSTACEAN_CRAB = 30
}  DIS_PDU_Subcategories_for_Subsurface_Category_203_Crustacean;

/* SISO-REF-010 [UID 124] */
typedef enum
{
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_204_INSECT_SEA_SKATER = 1,
     DIS_PDU_SUBCATEGORIES_FOR_SUBSURFACE_CATEGORY_204_INSECT_WATER_BEETLE = 2
}  DIS_PDU_Subcategories_for_Subsurface_Category_204_Insect;

/* SISO-REF-010 [UID 130] */
typedef enum
{
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_201_TO_249 = 201,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_250_TO_299 = 202,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_300_TO_399 = 203,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_400_TO_499 = 204,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_500_TO_999 = 205,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_1_000_TO_1_499 = 206,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_1_500_TO_1_999 = 207,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_2_000_TO_2_999 = 208,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_3_000_TO_4_999 = 210,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_5_000_TO_6_999 = 212,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_7_000_TO_9_999 = 214,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_10_000_TO_19_999 = 216,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_FROM_20_000_TO_50_000 = 218,
     DIS_PDU_ANIMAL_LIFEFORM_GROUP_SIZE_RANGE_ENUMERATION_FOR_ALL_DOMAINS_NUMBER_OF_ANIMALS_RANGE_GREATER_THAN_50_000 = 220
}  DIS_PDU_Animal_Lifeform_Group_Size_Range_Enumeration_for_all_Domains;

/* SISO-REF-010 [UID 131] */
typedef enum
{
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_LAND_AREA_SIZE_SMALL_AREA_2_000M2_4_000M2_5_TO_1_SQ_ACRE = 222,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_LAND_AREA_SIZE_SMALL_AREA_DENSE_2_000M2_4_000M2_5_TO1_SQ_ACRE = 223,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_LAND_AREA_SIZE_MEDIUM_AREA_20_000M2_41_000M2_10_TO_20_SQ_ACRES = 224,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_LAND_AREA_SIZE_MEDIUM_AREA_DENSE_20_000M2_41_000M2_10_TO_20_SQ_ACRES = 225,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_LAND_AREA_SIZE_LARGE_AREA_40_000M2_81_000M2_20_TO_40_SQ_ACRES = 226,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_LAND_AREA_SIZE_LARGE_AREA_DENSE_40_000M2_81_000M2_20_TO_40_SQ_ACRES = 227
}  DIS_PDU_Specific_Dimension_Enumerations_for_Land_Area_Size;

/* SISO-REF-010 [UID 132] */
typedef enum
{
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_AIR_AREA_SIZE_SMALL_FLOCK_SWARM_5KM_LONG_X_5M_WIDE_1KM_X_5M = 222,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_AIR_AREA_SIZE_SMALL_FLOCK_SWARM_DENSE_5KM_LONG_X_5M_WIDE_1KM_X_5M = 223,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_AIR_AREA_SIZE_MEDIUM_FLOCK_SWARM_1KM_LONG_X_10M_WIDE_2KM_X_10M = 224,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_AIR_AREA_SIZE_MEDIUM_FLOCK_SWARM_DENSE_1KM_LONG_X_10M_WIDE_2KM_X_10M = 225,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_AIR_AREA_SIZE_LARGE_FLOCK_SWARM_10KM_LONG_X_100M_WIDE_20KM_X_100M = 226,
     DIS_PDU_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_AIR_AREA_SIZE_LARGE_FLOCK_SWARM_DENSE_10KM_LONG_X_100M_WIDE_20KM_X_100M = 227
}  DIS_PDU_Specific_Dimension_Enumerations_for_Air_Area_Size;

/* SISO-REF-010 [UID 133] */
typedef enum
{
     DIS_PDU_ADD_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_SUBSURFACE_AREA_SIZE_SMALL_SCHOOL_5KM_LONG_X_250M_WIDE_X_83M_DEEP_1KM_X_5KM_X_166M = 222,
     DIS_PDU_ADD_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_SUBSURFACE_AREA_SIZE_SMALL_SCHOOL_DENSE_5KM_LONG_X_250M_WIDE_X_83M_DEEP_1KM_X_5KM_X_166M = 223,
     DIS_PDU_ADD_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_SUBSURFACE_AREA_SIZE_MEDIUM_SCHOOL_5KM_X_1KM_X_125M_1KM_X_2KM_X_500M = 224,
     DIS_PDU_ADD_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_SUBSURFACE_AREA_SIZE_MEDIUM_SCHOOL_DENSE_5KM_X_1KM_X_125M_1KM_X_2KM_X_500M = 225,
     DIS_PDU_ADD_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_SUBSURFACE_AREA_SIZE_LARGE_SCHOOL_5KM_X_2KM_X_500M_10KM_X_4KM_X_1KM = 226,
     DIS_PDU_ADD_SPECIFIC_DIMENSION_ENUMERATIONS_FOR_SUBSURFACE_AREA_SIZE_LARGE_SCHOOL_DENSE_5KM_X_2KM_X_500M_10KM_X_4KM_X_1KM = 227
}  DIS_PDU_Add_Specific_Dimension_Enumerations_for_Subsurface_Area_Size;

/* SISO-REF-010 [UID 134] */
typedef enum
{
     DIS_PDU_ADD_VARIANTS_FOR_LAND_CATEGORY_200_MAMMAL_ANIMAL_WITH_A_MALE_CHILD_RIDER = 1,
     DIS_PDU_ADD_VARIANTS_FOR_LAND_CATEGORY_200_MAMMAL_ANIMAL_WITH_A_FEMALE_CHILD_RIDER = 2,
     DIS_PDU_ADD_VARIANTS_FOR_LAND_CATEGORY_200_MAMMAL_ANIMAL_WITH_AN_ADULT_MALE_RIDER = 3,
     DIS_PDU_ADD_VARIANTS_FOR_LAND_CATEGORY_200_MAMMAL_ANIMAL_WITH_AN_ADULT_FEMALE_RIDER = 4,
     DIS_PDU_ADD_VARIANTS_FOR_LAND_CATEGORY_200_MAMMAL_ANIMAL_HARNESSED_TO_A_PLOW = 5,
     DIS_PDU_ADD_VARIANTS_FOR_LAND_CATEGORY_200_MAMMAL_ANIMAL_HARNESSED_TO_A_CART = 6
}  DIS_PDU_Add_Variants_for_Land_Category_200_Mammal;

/* SISO-REF-010 [UID 135] */
typedef enum
{
     DIS_PDU_VARIANTS_FOR_LAND_CATEGORIES_REPTILES_AMPHIBIANS_INSECTS_AND_ARACHNIDS_BLACK = 1,
     DIS_PDU_VARIANTS_FOR_LAND_CATEGORIES_REPTILES_AMPHIBIANS_INSECTS_AND_ARACHNIDS_GREEN = 2,
     DIS_PDU_VARIANTS_FOR_LAND_CATEGORIES_REPTILES_AMPHIBIANS_INSECTS_AND_ARACHNIDS_SPOTTED = 3,
     DIS_PDU_VARIANTS_FOR_LAND_CATEGORIES_REPTILES_AMPHIBIANS_INSECTS_AND_ARACHNIDS_RED = 4,
     DIS_PDU_VARIANTS_FOR_LAND_CATEGORIES_REPTILES_AMPHIBIANS_INSECTS_AND_ARACHNIDS_BROWN = 5
}  DIS_PDU_Variants_for_Land_Categories_Reptiles_Amphibians_Insects_and_Arachnids;

/* SISO-REF-010 [UID 136] */
typedef enum
{
     DIS_PDU_VARIANTS_FOR_AIR_CATEGORY_200_BIRD_BIRD_WITH_FISH = 1,
     DIS_PDU_VARIANTS_FOR_AIR_CATEGORY_200_BIRD_V_PATTERN_FLOCK_SHAPE = 2,
     DIS_PDU_VARIANTS_FOR_AIR_CATEGORY_200_BIRD_CIRCULAR_FLOCK_SHAPE = 3,
     DIS_PDU_VARIANTS_FOR_AIR_CATEGORY_200_BIRD_IRREGULAR_FLOCK_SHAPE = 4
}  DIS_PDU_Variants_for_Air_Category_200_Bird;

/* SISO-REF-010 [UID 137] */
typedef enum
{
     DIS_PDU_ADD_VARIANTS_FOR_AIR_CATEGORY_201_INSECT_VERTICAL_SHAPED_INSECT_SWARM = 1,
     DIS_PDU_ADD_VARIANTS_FOR_AIR_CATEGORY_201_INSECT_CIRCULAR_SHAPED_INSECT_SWARM = 2,
     DIS_PDU_ADD_VARIANTS_FOR_AIR_CATEGORY_201_INSECT_IRREGULAR_SHAPED_INSECT_SWARM = 3
}  DIS_PDU_Add_Variants_for_Air_Category_201_Insect;

/* SISO-REF-010 [UID 138] */
typedef enum
{
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_BLACK = 1,
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_GREEN = 2,
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_SPOTTED = 3,
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_RED = 4,
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_BROWN = 5,
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_BLUE = 6,
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_SILVER = 7,
     DIS_PDU_ADD_VARIANTS_FOR_SUBSURFACE_CATEGORIES_FISH_MOLLUSK_CRUSTACEAN_AND_INSECT_GREY = 8
}  DIS_PDU_Add_Variants_for_Subsurface_Categories_Fish_Mollusk_Crustacean_and_Insect;

/* SISO-REF-010 [UID 139] */
typedef enum
{
     DIS_PDU_VARIANTS_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SINGING = 1,
     DIS_PDU_VARIANTS_FOR_SUBSURFACE_CATEGORY_201_MAMMAL_SPOUTING = 2
}  DIS_PDU_Variants_for_Subsurface_Category_201_Mammal;

/* SISO-REF-010 [UID 143] */
typedef enum
{
     DIS_PDU_UA_STATE_CHANGE_UPDATE_INDICATOR_STATE_UPDATE = 0,
     DIS_PDU_UA_STATE_CHANGE_UPDATE_INDICATOR_CHANGED_DATA_UPDATE = 1
}  DIS_PDU_UA_State_Change_Update_Indicator;

/* SISO-REF-010 [UID 144] */
typedef enum
{
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_OTHER = 0,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_AN_BQQ_5 = 1,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_AN_SSQ_62 = 2,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_AN_SQS_23 = 3,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_AN_SQS_26 = 4,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_AN_SQS_53 = 5,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_ALFS = 6,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_LFA = 7,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_AN_AQS_901 = 8,
     DIS_PDU_UA_ACOUSTIC_SYSTEM_NAME_AN_AQS_902 = 9
}  DIS_PDU_UA_Acoustic_System_Name;

/* SISO-REF-010 [UID 145] */
typedef enum
{
     DIS_PDU_UA_ACOUSTIC_EMITTER_SYSTEM_FUNCTION_OTHER = 0,
     DIS_PDU_UA_ACOUSTIC_EMITTER_SYSTEM_FUNCTION_PLATFORM_SEARCH_DETECT_TRACK = 1,
     DIS_PDU_UA_ACOUSTIC_EMITTER_SYSTEM_FUNCTION_NAVIGATION = 2,
     DIS_PDU_UA_ACOUSTIC_EMITTER_SYSTEM_FUNCTION_MINE_HUNTING = 3,
     DIS_PDU_UA_ACOUSTIC_EMITTER_SYSTEM_FUNCTION_WEAPON_SEARCH_DETECT_TRACK_DETECT = 4
}  DIS_PDU_UA_Acoustic_Emitter_System_Function;

/* SISO-REF-010 [UID 146] */
typedef enum
{
     DIS_PDU_UA_ACTIVE_EMISSION_PARAMETER_INDEX_OTHER = 0
}  DIS_PDU_UA_Active_Emission_Parameter_Index;

/* SISO-REF-010 [UID 147] */
typedef enum
{
     DIS_PDU_UA_SCAN_PATTERN_SCAN_PATTERN_NOT_USED = 0,
     DIS_PDU_UA_SCAN_PATTERN_CONICAL = 1,
     DIS_PDU_UA_SCAN_PATTERN_HELICAL = 2,
     DIS_PDU_UA_SCAN_PATTERN_RASTER = 3,
     DIS_PDU_UA_SCAN_PATTERN_SECTOR_SEARCH = 4,
     DIS_PDU_UA_SCAN_PATTERN_CONTINUOUS_SEARCH = 5
}  DIS_PDU_UA_Scan_Pattern;

/* SISO-REF-010 [UID 148] */
typedef enum
{
     DIS_PDU_UA_PASSIVE_PARAMETER_INDEX_OTHER = 0
}  DIS_PDU_UA_Passive_Parameter_Index;

/* SISO-REF-010 [UID 150] */
typedef enum
{
     DIS_PDU_UA_ADDITIONAL_PASSIVE_ACTIVITY_PARAMETER_INDEX_OTHER = 0
}  DIS_PDU_UA_Additional_Passive_Activity_Parameter_Index;

/* SISO-REF-010 [UID 155] */
typedef enum
{
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_NO_STATEMENT = 0,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_AMPLITUDE = 1,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_AMPLITUDE_AND_ANGLE = 2,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_ANGLE = 3,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_COMBINATION = 4,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_PULSE = 5,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_UNMODULATED = 6,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_CARRIER_PHASE_SHIFT_MODULATION_CPSM = 7,
     DIS_PDU_TRANSMITTER_MAJOR_MODULATION_SATCOM = 8
}  DIS_PDU_Transmitter_Major_Modulation;

static const value_string DIS_PDU_Transmitter_Major_Modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_NO_STATEMENT, "No Statement" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_AMPLITUDE, "Amplitude" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_AMPLITUDE_AND_ANGLE, "Amplitude and Angle" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_ANGLE, "Angle" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_COMBINATION, "Combination" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_PULSE, "Pulse" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_UNMODULATED, "Unmodulated" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_CARRIER_PHASE_SHIFT_MODULATION_CPSM, "Carrier Phase Shift Modulation (CPSM)" },
     { DIS_PDU_TRANSMITTER_MAJOR_MODULATION_SATCOM, "SATCOM" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 156] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_OTHER = 0,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_AFSK_AUDIO_FREQUENCY_SHIFT_KEYING = 1,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_AM_AMPLITUDE_MODULATION = 2,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_CW_CONTINUOUS_WAVE_MODULATION = 3,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_DSB_DOUBLE_SIDEBAND = 4,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_ISB_INDEPENDENT_SIDEBAND = 5,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_LSB_SINGLE_BAND_SUPPRESSED_CARRIER_LOWER_SIDEBAND_MODE = 6,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_SSB_FULL_SINGLE_SIDEBAND_FULL_CARRIER = 7,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_SSB_REDUC_SINGLE_BAND_REDUCED_CARRIER = 8,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_USB_SINGLE_BAND_SUPPRESSED_CARRIER_UPPER_SIDEBAND_MODE = 9,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_VSB_VESTIGIAL_SIDEBAND = 10
}  DIS_PDU_Transmitter_Detail_Amplitude_Modulation;

static const value_string DIS_PDU_Transmitter_Detail_Amplitude_Modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_AFSK_AUDIO_FREQUENCY_SHIFT_KEYING, "AFSK (Audio Frequency Shift Keying)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_AM_AMPLITUDE_MODULATION, "AM (Amplitude Modulation)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_CW_CONTINUOUS_WAVE_MODULATION, "CW (Continuous Wave Modulation)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_DSB_DOUBLE_SIDEBAND, "DSB (Double Sideband)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_ISB_INDEPENDENT_SIDEBAND, "ISB (Independent Sideband)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_LSB_SINGLE_BAND_SUPPRESSED_CARRIER_LOWER_SIDEBAND_MODE, "LSB (Single Band Suppressed Carrier, Lower Sideband Mode)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_SSB_FULL_SINGLE_SIDEBAND_FULL_CARRIER, "SSB-Full (Single Sideband Full Carrier)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_SSB_REDUC_SINGLE_BAND_REDUCED_CARRIER, "SSB-Reduc (Single Band Reduced Carrier)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_USB_SINGLE_BAND_SUPPRESSED_CARRIER_UPPER_SIDEBAND_MODE, "USB (Single Band Suppressed Carrier, Upper Sideband Mode)" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_MODULATION_VSB_VESTIGIAL_SIDEBAND, "VSB (Vestigial Sideband)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 157] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_AND_ANGLE_MODULATION_OTHER = 0,
     DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_AND_ANGLE_MODULATION_AMPLITUDE_AND_ANGLE = 1
}  DIS_PDU_Transmitter_Detail_Amplitude_and_Angle_Modulation;

static const value_string DIS_PDU_Transmitter_Detail_Amplitude_and_Angle_Modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_AND_ANGLE_MODULATION_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_DETAIL_AMPLITUDE_AND_ANGLE_MODULATION_AMPLITUDE_AND_ANGLE, "Amplitude and Angle" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 158] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_OTHER = 0,
     DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_FM_FREQUENCY_MODULATION = 1,
     DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_FSK_FREQUENCY_SHIFT_KEYING = 2,
     DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_PM_PHASE_MODULATION = 3,
     DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_MSK_MINIMUM_SHIFT_KEYING = 4
}  DIS_PDU_Transmitter_Detail_Angle_modulation;

static const value_string DIS_PDU_Transmitter_Detail_Angle_modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_FM_FREQUENCY_MODULATION, "FM (Frequency Modulation)" },
     { DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_FSK_FREQUENCY_SHIFT_KEYING, "FSK (Frequency Shift Keying)" },
     { DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_PM_PHASE_MODULATION, "PM (Phase Modulation)" },
     { DIS_PDU_TRANSMITTER_DETAIL_ANGLE_MODULATION_MSK_MINIMUM_SHIFT_KEYING, "MSK (Minimum Shift Keying)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 159] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_COMBINATION_MODULATION_OTHER = 0,
     DIS_PDU_TRANSMITTER_DETAIL_COMBINATION_MODULATION_AMPLITUDE_ANGLE_PULSE = 1
}  DIS_PDU_Transmitter_Detail_Combination_Modulation;

static const value_string DIS_PDU_Transmitter_Detail_Combination_Modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_DETAIL_COMBINATION_MODULATION_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_DETAIL_COMBINATION_MODULATION_AMPLITUDE_ANGLE_PULSE, "Amplitude-Angle-Pulse" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 160] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_OTHER = 0,
     DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_PULSE = 1,
     DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_X_BAND_TACAN_PULSE = 2,
     DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_Y_BAND_TACAN_PULSE = 3
}  DIS_PDU_Transmitter_Detail_Pulse_Modulation;

static const value_string DIS_PDU_Transmitter_Detail_Pulse_Modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_PULSE, "Pulse" },
     { DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_X_BAND_TACAN_PULSE, "X Band TACAN Pulse" },
     { DIS_PDU_TRANSMITTER_DETAIL_PULSE_MODULATION_Y_BAND_TACAN_PULSE, "Y Band TACAN Pulse" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 161] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_UNMODULATED_MODULATION_OTHER = 0,
     DIS_PDU_TRANSMITTER_DETAIL_UNMODULATED_MODULATION_CONTINUOUS_WAVE_EMISSION_OF_AN_UNMODULATED_CARRIER = 1
}  DIS_PDU_Transmitter_Detail_Unmodulated_Modulation;

static const value_string DIS_PDU_Transmitter_Detail_Unmodulated_Modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_DETAIL_UNMODULATED_MODULATION_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_DETAIL_UNMODULATED_MODULATION_CONTINUOUS_WAVE_EMISSION_OF_AN_UNMODULATED_CARRIER, "Continuous Wave emission of an unmodulated carrier" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 162] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_CARRIER_PHASE_SHIFT_MODULATION_OTHER = 0
}  DIS_PDU_Transmitter_Detail_Carrier_Phase_Shift_Modulation;

static const value_string DIS_PDU_Transmitter_Detail_Carrier_Phase_Shift_Modulation_Strings[] = {
     { DIS_PDU_TRANSMITTER_DETAIL_CARRIER_PHASE_SHIFT_MODULATION_OTHER, "Other" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 163] */
typedef enum
{
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_OTHER = 0,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_GENERIC_RADIO_OR_SIMPLE_INTERCOM = 1,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_HAVE_QUICK_I = 2,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_HAVE_QUICK_II = 3,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_SATURN = 4,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_SINCGARS = 5,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_CCTT_SINCGARS = 6,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_EPLRS_ENHANCED_POSITION_LOCATION_REPORTING_SYSTEM = 7,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_JTIDS_MIDS = 8,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_LINK_11 = 9,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_LINK_11B = 10,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_L_BAND_SATCOM = 11,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_ENHANCED_SINCGARS_7_3 = 12,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_NAVIGATION_AID = 13,
     DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_MUOS = 14
}  DIS_PDU_Transmitter_Modulation_Type_System;

static const value_string DIS_PDU_Transmitter_Modulation_Type_System_Strings[] = {
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_GENERIC_RADIO_OR_SIMPLE_INTERCOM, "Generic Radio or Simple Intercom" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_HAVE_QUICK_I, "HAVE QUICK I" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_HAVE_QUICK_II, "HAVE QUICK II" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_SATURN, "SATURN" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_SINCGARS, "SINCGARS" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_CCTT_SINCGARS, "CCTT SINCGARS" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_EPLRS_ENHANCED_POSITION_LOCATION_REPORTING_SYSTEM, "EPLRS (Enhanced Position Location Reporting System)" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_JTIDS_MIDS, "JTIDS/MIDS" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_LINK_11, "Link 11" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_LINK_11B, "Link 11B" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_L_BAND_SATCOM, "L-Band SATCOM" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_ENHANCED_SINCGARS_7_3, "Enhanced SINCGARS 7.3" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_NAVIGATION_AID, "Navigation Aid" },
     { DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_MUOS, "MUOS" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 164] */
typedef enum
{
     DIS_PDU_TRANSMITTER_TRANSMIT_STATE_OFF = 0,
     DIS_PDU_TRANSMITTER_TRANSMIT_STATE_ON_BUT_NOT_TRANSMITTING = 1,
     DIS_PDU_TRANSMITTER_TRANSMIT_STATE_ON_AND_TRANSMITTING = 2
}  DIS_PDU_Transmitter_Transmit_State;

static const value_string DIS_PDU_Transmitter_Transmit_State_Strings[] = {
     { DIS_PDU_TRANSMITTER_TRANSMIT_STATE_OFF, "Off" },
     { DIS_PDU_TRANSMITTER_TRANSMIT_STATE_ON_BUT_NOT_TRANSMITTING, "On but not transmitting" },
     { DIS_PDU_TRANSMITTER_TRANSMIT_STATE_ON_AND_TRANSMITTING, "On and transmitting" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 165] */
typedef enum
{
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_OTHER = 0,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_PILOT = 1,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_COPILOT = 2,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_FIRST_OFFICER = 3,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_DRIVER = 4,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_LOADER = 5,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_GUNNER = 6,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_COMMANDER = 7,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_DIGITAL_DATA_DEVICE = 8,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_INTERCOM = 9,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_AUDIO_JAMMER = 10,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_DATA_JAMMER = 11,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_GPS_JAMMER = 12,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_GPS_MEACONER = 13,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_SATCOM_UPLINK_JAMMER = 14,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_CREW_OBSERVER_1 = 15,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_CREW_OBSERVER_2 = 16,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_AERIAL_REFUELING_OFFICER_ARO = 17,
     DIS_PDU_TRANSMITTER_INPUT_SOURCE_AERIAL_REFUELING_OFFICER_INSTRUCTOR_AROI = 18
}  DIS_PDU_Transmitter_Input_Source;

static const value_string DIS_PDU_Transmitter_Input_Source_Strings[] = {
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_OTHER, "Other" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_PILOT, "Pilot" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_COPILOT, "Copilot" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_FIRST_OFFICER, "First Officer" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_DRIVER, "Driver" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_LOADER, "Loader" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_GUNNER, "Gunner" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_COMMANDER, "Commander" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_DIGITAL_DATA_DEVICE, "Digital Data Device" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_INTERCOM, "Intercom" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_AUDIO_JAMMER, "Audio Jammer" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_DATA_JAMMER, "Data Jammer" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_GPS_JAMMER, "GPS Jammer" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_GPS_MEACONER, "GPS Meaconer" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_SATCOM_UPLINK_JAMMER, "SATCOM Uplink Jammer" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_CREW_OBSERVER_1, "Crew Observer 1" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_CREW_OBSERVER_2, "Crew Observer 2" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_AERIAL_REFUELING_OFFICER_ARO, "Aerial Refueling Officer (ARO)" },
     { DIS_PDU_TRANSMITTER_INPUT_SOURCE_AERIAL_REFUELING_OFFICER_INSTRUCTOR_AROI, "Aerial Refueling Officer Instructor (AROI)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 166] */
typedef enum
{
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_NO_ENCRYPTION_DEVICE = 0,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_28 = 1,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_58 = 2,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_NARROW_SPECTRUM_SECURE_VOICE_NSVE = 3,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_WIDE_SPECTRUM_SECURE_VOICE_WSVE = 4,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_SINCGARS_ICOM = 5,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_75 = 6,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_100 = 7,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_57 = 8,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KYV_5 = 9,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_LINK_11_KG_40A_P_NTDS = 10,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_LINK_11B_KG_40A_S = 11,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_LINK_11_KG_40AR = 12,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KGV_135A = 13,
     DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_TACTICAL_SECURE_VOICE_TSV = 14
}  DIS_PDU_Transmitter_Crypto_System;

static const value_string DIS_PDU_Transmitter_Crypto_System_Strings[] = {
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_NO_ENCRYPTION_DEVICE, "No Encryption Device" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_28, "KY-28" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_58, "KY-58" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_NARROW_SPECTRUM_SECURE_VOICE_NSVE, "Narrow Spectrum Secure Voice (NSVE)" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_WIDE_SPECTRUM_SECURE_VOICE_WSVE, "Wide Spectrum Secure Voice (WSVE)" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_SINCGARS_ICOM, "SINCGARS ICOM" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_75, "KY-75" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_100, "KY-100" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KY_57, "KY-57" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KYV_5, "KYV-5" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_LINK_11_KG_40A_P_NTDS, "Link 11 KG-40A-P (NTDS)" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_LINK_11B_KG_40A_S, "Link 11B KG-40A-S" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_LINK_11_KG_40AR, "Link 11 KG-40AR" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_KGV_135A, "KGV-135A" },
     { DIS_PDU_TRANSMITTER_CRYPTO_SYSTEM_TACTICAL_SECURE_VOICE_TSV, "Tactical Secure Voice (TSV)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 167] */
typedef enum
{
     DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_ISOTROPIC_SPHERICAL_RADIATION_PATTERN = 0,
     DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_BEAM = 1,
     DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_TRANSMITTER_RADIATION_VOLUME = 4,
     DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_BEAM_AND_TRANSMITTER_RADIATION_VOLUME = 5,
     DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_OMNIDIRECTIONAL_TOROIDAL_RADIATION_PATTERN = 6
}  DIS_PDU_Transmitter_Antenna_Pattern_Type;

static const value_string DIS_PDU_Transmitter_Antenna_Pattern_Type_Strings[] = {
     { DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_ISOTROPIC_SPHERICAL_RADIATION_PATTERN, "Isotropic (Spherical Radiation Pattern)" },
     { DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_BEAM, "Beam" },
     { DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_TRANSMITTER_RADIATION_VOLUME, "Transmitter Radiation Volume" },
     { DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_BEAM_AND_TRANSMITTER_RADIATION_VOLUME, "Beam and Transmitter Radiation Volume" },
     { DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_TYPE_OMNIDIRECTIONAL_TOROIDAL_RADIATION_PATTERN, "Omnidirectional (Toroidal Radiation Pattern)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 168] */
typedef enum
{
     DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_REFERENCE_SYSTEM_WORLD_COORDINATES = 1,
     DIS_PDU_TRANSMITTER_ANTENNA_PATTERN_REFERENCE_SYSTEM_ENTITY_COORDINATES = 2
}  DIS_PDU_Transmitter_Antenna_Pattern_Reference_System;

/* SISO-REF-010 [UID 170] */
typedef enum
{
     DIS_PDU_CCTT_SINCGARS_START_OF_MESSAGE_NOT_START_OF_MESSAGE = 0,
     DIS_PDU_CCTT_SINCGARS_START_OF_MESSAGE_START_OF_MESSAGE = 1
}  DIS_PDU_CCTT_SINCGARS_Start_of_Message;

static const value_string DIS_PDU_CCTT_SINCGARS_Start_of_Message_Strings[] = {
     { DIS_PDU_CCTT_SINCGARS_START_OF_MESSAGE_NOT_START_OF_MESSAGE, "Not start of message" },
     { DIS_PDU_CCTT_SINCGARS_START_OF_MESSAGE_START_OF_MESSAGE, "Start of Message" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 171] */
typedef enum
{
     DIS_PDU_CCTT_SINCGARS_CLEAR_CHANNEL_NOT_CLEAR_CHANNEL = 0,
     DIS_PDU_CCTT_SINCGARS_CLEAR_CHANNEL_CLEAR_CHANNEL = 1
}  DIS_PDU_CCTT_SINCGARS_Clear_Channel;

static const value_string DIS_PDU_CCTT_SINCGARS_Clear_Channel_Strings[] = {
     { DIS_PDU_CCTT_SINCGARS_CLEAR_CHANNEL_NOT_CLEAR_CHANNEL, "Not clear channel" },
     { DIS_PDU_CCTT_SINCGARS_CLEAR_CHANNEL_CLEAR_CHANNEL, "Clear channel" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 172] */
typedef enum
{
     DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_LOW_FIDELITY_LEVEL_0 = 0,
     DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_LOW_FIDELITY_LEVEL_1 = 1,
     DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_MEDIUM_FIDELITY_LEVEL_2 = 2,
     DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_MEDIUM_FIDELITY_LEVEL_3 = 3,
     DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_HIGH_FIDELITY_LEVEL_4 = 4
}  DIS_PDU_Time_Slot_Allocation_Level;

static const value_string DIS_PDU_Time_Slot_Allocation_Level_Strings[] = {
     { DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_LOW_FIDELITY_LEVEL_0, "Low Fidelity Level 0" },
     { DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_LOW_FIDELITY_LEVEL_1, "Low Fidelity Level 1" },
     { DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_MEDIUM_FIDELITY_LEVEL_2, "Medium Fidelity Level 2" },
     { DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_MEDIUM_FIDELITY_LEVEL_3, "Medium Fidelity Level 3" },
     { DIS_PDU_TIME_SLOT_ALLOCATION_LEVEL_HIGH_FIDELITY_LEVEL_4, "High Fidelity Level 4" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 173] */
typedef enum
{
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_PRIMARY_MODE_NTR = 1,
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_PRIMARY_MODE_JTIDS_UNIT_PARTICIPANT = 2
}  DIS_PDU_JTIDS_MIDS_Modulation_Parameters_Transmitting_Terminal_Primary_Mode;

static const value_string DIS_PDU_JTIDS_MIDS_Modulation_Parameters_Transmitting_Terminal_Primary_Mode_Strings[] = {
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_PRIMARY_MODE_NTR, "NTR" },
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_PRIMARY_MODE_JTIDS_UNIT_PARTICIPANT, "JTIDS Unit Participant" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 174] */
typedef enum
{
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_NONE = 0,
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_NET_POSITION_REFERENCE = 1,
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_PRIMARY_NAVIGATION_CONTROLLER = 2,
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_SECONDARY_NAVIGATION_CONTROLLER = 3
}  DIS_PDU_JTIDS_MIDS_Modulation_Parameters_Transmitting_Terminal_Secondary_Mode;

static const value_string DIS_PDU_JTIDS_MIDS_Modulation_Parameters_Transmitting_Terminal_Secondary_Mode_Strings[] = {
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_NONE, "None" },
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_NET_POSITION_REFERENCE, "Net Position Reference" },
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_PRIMARY_NAVIGATION_CONTROLLER, "Primary Navigation Controller" },
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_TRANSMITTING_TERMINAL_SECONDARY_MODE_SECONDARY_NAVIGATION_CONTROLLER, "Secondary Navigation Controller" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 175] */
typedef enum
{
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_NO_STATEMENT = 0,
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_INITIAL_NET_ENTRY = 1,
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_COARSE_SYNCHRONIZATION = 2,
     DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_FINE_SYNCHRONIZATION = 3
}  DIS_PDU_JTIDS_MIDS_Modulation_Parameters_Synchronization_State;

static const value_string DIS_PDU_JTIDS_MIDS_Modulation_Parameters_Synchronization_State_Strings[] = {
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_NO_STATEMENT, "No Statement" },
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_INITIAL_NET_ENTRY, "Initial Net Entry" },
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_COARSE_SYNCHRONIZATION, "Coarse Synchronization" },
     { DIS_PDU_JTIDS_MIDS_MODULATION_PARAMETERS_SYNCHRONIZATION_STATE_FINE_SYNCHRONIZATION, "Fine Synchronization" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 176] */
typedef enum
{
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_HEADER_MESSAGES = 0,
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_RTT_A_B = 1,
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_RTT_REPLY = 2,
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_VOICE_CVSD = 3,
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_VOICE_LPC10 = 4,
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_VOICE_LPC12 = 5,
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_LET = 6,
     DIS_PDU_MESSAGE_TYPE_IDENTIFIER_VMF = 7
}  DIS_PDU_Message_Type_Identifier;

static const value_string DIS_PDU_Message_Type_Identifier_Strings[] = {
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_HEADER_MESSAGES, "JTIDS Header/Messages" },
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_RTT_A_B, "RTT A/B" },
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_RTT_REPLY, "RTT Reply" },
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_VOICE_CVSD, "JTIDS Voice CVSD" },
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_VOICE_LPC10, "JTIDS Voice LPC10" },
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_VOICE_LPC12, "JTIDS Voice LPC12" },
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_LET, "JTIDS LET" },
     { DIS_PDU_MESSAGE_TYPE_IDENTIFIER_VMF, "VMF" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 177] */
typedef enum
{
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CCSIL_MARNIE_MITRE_ORG = 1,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_A2ATD_SINCGARS_ERF_WOODMAN_ORLANDO_LORAL_COM = 5,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_A2ATD_CAC2_WBEARD_ORLANDO_LORAL_COM = 6,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_BATTLE_COMMAND_GGAGNON_CAS_INC_COM = 20,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_AFIWC_IADS_TRACK_REPORT_RANDY_SCHUETZ_LACKLAND_AF_MIL = 30,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_AFIWC_IADS_COMM_C2_MESSAGE_RANDY_SCHUETZ_LACKLAND_AF_MIL = 31,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_AFIWC_IADS_GROUND_CONTROL_INTERCEPTOR_GCI_COMMAND_RANDY_SCHUETZ_LACKLAND_AF_MIL = 32,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_AFIWC_VOICE_TEXT_MESSAGE_RANDY_SCHUETZ_LACKLAND_AF_MIL = 35,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_MODSAF_TEXT_RADIO_RSCHAFFER_CAMB_LADS_LORAL_COM = 177,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CCTT_SINCGARS_ERF_LOCKOUT_JIMK_GREATWALL_CCTT_COM = 200,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CCTT_SINCGARS_ERF_HOPSET_JIMK_GREATWALL_CCTT_COM = 201,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CCTT_SINCGARS_OTAR_JIMK_GREATWALL_CCTT_COM = 202,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CCTT_SINCGARS_DATA_JIMK_GREATWALL_CCTT_COM = 203,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_MODSAF_FWA_FORWARD_AIR_CONTROLLER_DCOFFIN_CAMB_LADS_LORAL_COM = 546,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_MODSAF_THREAT_ADA_C3_DCOFFIN_CAMB_LADS_LORAL_COM = 832,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_F_16_MTC_AFAPD_PROTOCOL_ALBERT_J_LUDWIG_BOEING_COM = 1000,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_F_16_MTC_IDL_PROTOCOL_ALBERT_J_LUDWIG_BOEING_COM = 1100,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_MODSAF_ARTILLERY_FIRE_CONTROL_RSCHAFFER_CAMB_LADS_LORAL_COM = 4570,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_AGTS_GENDREAU_ESCMAIL_ORL_MMC_COM = 5361,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_GC3_KARL_SHEPHERD_GSC_GTE_COM = 6000,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_WNCP_DATA_KARL_SHEPHERD_GSC_GTE_COM = 6010,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_SPOKEN_TEXT_MESSAGE_BRETT_KAYLOR_GSC_GTE_COM = 6020,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_LONGBOW_IDM_MESSAGE_OBEAR_CATINET_COM = 6661,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_COMANCHE_IDM_MESSAGE_OBEAR_CATINET_COM = 6662,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_LONGBOW_AIRBORNE_TACFIRE_MESSAGE_OBEAR_CATINET_COM = 6663,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_LONGBOW_GROUND_TACFIRE_MESSAGE_OBEAR_CATINET_COM = 6664,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_LONGBOW_AFAPD_MESSAGE_OBEAR_CATINET_COM = 6665,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_LONGBOW_ERF_MESSAGE_JEFFREY_DAY_MW_BOEING_COM = 6666,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_VMF_IDM_PATRICK_MERLET_PARSONS_COM = 7000,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CSAR_RADIO_SURVIVOR_MESSAGE_MIKE_STROZ_NGC_COM = 7010,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CSAR_RADIO_INTERROGATOR_MESSAGE_MIKE_STROZ_NGC_COM = 7020,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_IMAGE_FILE_TRANSFER_MESSAGE_PATRICK_MERLET_PARSONS_COM = 7030,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_GEOTAG_DATA_MESSAGE_PATRICK_MERLET_PARSONS_COM = 7040,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_TACTICAL_VIDEO_REGENERATION_DATA_PATRICK_MERLET_PARSONS_COM = 7050,
     DIS_PDU_SIGNAL_USER_PROTOCOL_IDENTIFICATION_NUMBER_CALL_MESSAGE_CHRIS_KUBEK_ASTI_USA_COM = 8000
}  DIS_PDU_Signal_User_Protocol_Identification_Number;

/* SISO-REF-010 [UID 178] */
typedef enum
{
     DIS_PDU_SIGNAL_TDL_TYPE_OTHER = 0,
     DIS_PDU_SIGNAL_TDL_TYPE_PADIL = 1,
     DIS_PDU_SIGNAL_TDL_TYPE_NATO_LINK_1 = 2,
     DIS_PDU_SIGNAL_TDL_TYPE_ATDL_1 = 3,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_11B_TADIL_B = 4,
     DIS_PDU_SIGNAL_TDL_TYPE_SITUATIONAL_AWARENESS_DATA_LINK_SADL = 5,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_LEGACY_FORMAT_JTIDS_TADIL_J = 6,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_LEGACY_FORMAT_JTIDS_FDL_TADIL_J = 7,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_11_TADIL_A = 8,
     DIS_PDU_SIGNAL_TDL_TYPE_IJMS = 9,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_4A_TADIL_C = 10,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_4C = 11,
     DIS_PDU_SIGNAL_TDL_TYPE_TIBS = 12,
     DIS_PDU_SIGNAL_TDL_TYPE_ATL = 13,
     DIS_PDU_SIGNAL_TDL_TYPE_CONSTANT_SOURCE = 14,
     DIS_PDU_SIGNAL_TDL_TYPE_ABBREVIATED_COMMAND_AND_CONTROL = 15,
     DIS_PDU_SIGNAL_TDL_TYPE_MILSTAR = 16,
     DIS_PDU_SIGNAL_TDL_TYPE_ATHS = 17,
     DIS_PDU_SIGNAL_TDL_TYPE_OTHGOLD = 18,
     DIS_PDU_SIGNAL_TDL_TYPE_TACELINT = 19,
     DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AWW_13 = 20,
     DIS_PDU_SIGNAL_TDL_TYPE_ENHANCED_POSITION_LOCATION_REPORTING_SYSTEM_EPLRS = 22,
     DIS_PDU_SIGNAL_TDL_TYPE_POSITION_LOCATION_REPORTING_SYSTEM_PLRS = 23,
     DIS_PDU_SIGNAL_TDL_TYPE_SINCGARS = 24,
     DIS_PDU_SIGNAL_TDL_TYPE_HAVE_QUICK_I = 25,
     DIS_PDU_SIGNAL_TDL_TYPE_HAVE_QUICK_II = 26,
     DIS_PDU_SIGNAL_TDL_TYPE_SATURN = 27,
     DIS_PDU_SIGNAL_TDL_TYPE_INTRA_FLIGHT_DATA_LINK_1 = 28,
     DIS_PDU_SIGNAL_TDL_TYPE_INTRA_FLIGHT_DATA_LINK_2 = 29,
     DIS_PDU_SIGNAL_TDL_TYPE_IMPROVED_DATA_MODEM_IDM = 30,
     DIS_PDU_SIGNAL_TDL_TYPE_AIR_FORCE_APPLICATION_PROGRAM_DEVELOPMENT_AFAPD = 31,
     DIS_PDU_SIGNAL_TDL_TYPE_COOPERATIVE_ENGAGEMENT_CAPABILITY_CEC = 32,
     DIS_PDU_SIGNAL_TDL_TYPE_FORWARD_AREA_AIR_DEFENSE_FAAD_DATA_LINK_FDL = 33,
     DIS_PDU_SIGNAL_TDL_TYPE_GROUND_BASED_DATA_LINK_GBDL = 34,
     DIS_PDU_SIGNAL_TDL_TYPE_INTRA_VEHICULAR_INFO_SYSTEM_IVIS = 35,
     DIS_PDU_SIGNAL_TDL_TYPE_MARINE_TACTICAL_SYSTEM_MTS = 36,
     DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_FIRE_DIRECTION_SYSTEM_TACFIRE = 37,
     DIS_PDU_SIGNAL_TDL_TYPE_INTEGRATED_BROADCAST_SERVICE_IBS = 38,
     DIS_PDU_SIGNAL_TDL_TYPE_AIRBORNE_INFORMATION_TRANSFER_ABIT = 39,
     DIS_PDU_SIGNAL_TDL_TYPE_ADVANCED_TACTICAL_AIRBORNE_RECONNAISSANCE_SYSTEM_ATARS_DATA_LINK = 40,
     DIS_PDU_SIGNAL_TDL_TYPE_BATTLE_GROUP_PASSIVE_HORIZON_EXTENSION_SYSTEM_BGPHES_DATA_LINK = 41,
     DIS_PDU_SIGNAL_TDL_TYPE_COMMON_HIGH_BANDWIDTH_DATA_LINK_CHBDL = 42,
     DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_INTEROPERABLE_DATA_LINK_IDL = 43,
     DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_COMMON_SENSOR_SYSTEM_ONE_CSS1_DATA_LINK = 44,
     DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_COMMON_SENSOR_SYSTEM_TWO_CSS2_DATA_LINK = 45,
     DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_CSS2_MULTI_ROLE_DATA_LINK_MRDL = 46,
     DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_CSS2_DIRECT_AIR_TO_SATELLITE_RELAY_DASR_DATA_LINK = 47,
     DIS_PDU_SIGNAL_TDL_TYPE_LINE_OF_SIGHT_LOS_DATA_LINK_IMPLEMENTATION_LOS_TETHER = 48,
     DIS_PDU_SIGNAL_TDL_TYPE_LIGHTWEIGHT_CDL_LWCDL = 49,
     DIS_PDU_SIGNAL_TDL_TYPE_L_52M_SR_71 = 50,
     DIS_PDU_SIGNAL_TDL_TYPE_RIVET_REACH_RIVET_OWL_DATA_LINK = 51,
     DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_SPAN = 52,
     DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_SPUR = 53,
     DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_STRETCH = 54,
     DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_YEAR_INTEROPERABLE_DATA_LINK_IDL = 55,
     DIS_PDU_SIGNAL_TDL_TYPE_SPACE_CDL = 56,
     DIS_PDU_SIGNAL_TDL_TYPE_TR_1_MODE_MIST_AIRBORNE_DATA_LINK = 57,
     DIS_PDU_SIGNAL_TDL_TYPE_KU_BAND_SATCOM_DATA_LINK_IMPLEMENTATION_UAV = 58,
     DIS_PDU_SIGNAL_TDL_TYPE_MISSION_EQUIPMENT_CONTROL_DATA_LINK_MECDL = 59,
     DIS_PDU_SIGNAL_TDL_TYPE_RADAR_DATA_TRANSMITTING_SET_DATA_LINK = 60,
     DIS_PDU_SIGNAL_TDL_TYPE_SURVEILLANCE_AND_CONTROL_DATA_LINK_SCDL = 61,
     DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_UAV_VIDEO = 62,
     DIS_PDU_SIGNAL_TDL_TYPE_UHF_SATCOM_DATA_LINK_IMPLEMENTATION_UAV = 63,
     DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_COMMON_DATA_LINK_TCDL = 64,
     DIS_PDU_SIGNAL_TDL_TYPE_LOW_LEVEL_AIR_PICTURE_INTERFACE_LLAPI = 65,
     DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AGM_130 = 66,
     DIS_PDU_SIGNAL_TDL_TYPE_AUTOMATIC_IDENTIFICATION_SYSTEM_AIS = 67,
     DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AIM_120 = 68,
     DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AIM_9 = 69,
     DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_CAMM = 70,
     DIS_PDU_SIGNAL_TDL_TYPE_GC3 = 99,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_STANDARDIZED_FORMAT_JTIDS_MIDS_TADIL_J = 100,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_ENHANCED_DATA_RATE_EDR_JTIDS_MIDS_TADIL_J = 101,
     DIS_PDU_SIGNAL_TDL_TYPE_JTIDS_MIDS_NET_DATA_LOAD_TIMS_TOMS = 102,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_22 = 103,
     DIS_PDU_SIGNAL_TDL_TYPE_AFIWC_IADS_COMMUNICATIONS_LINKS = 104,
     DIS_PDU_SIGNAL_TDL_TYPE_F_22_INTRA_FLIGHT_DATA_LINK_IFDL = 105,
     DIS_PDU_SIGNAL_TDL_TYPE_L_BAND_SATCOM = 106,
     DIS_PDU_SIGNAL_TDL_TYPE_TSAF_COMMUNICATIONS_LINK = 107,
     DIS_PDU_SIGNAL_TDL_TYPE_ENHANCED_SINCGARS_7_3 = 108,
     DIS_PDU_SIGNAL_TDL_TYPE_F_35_MULTIFUNCTION_ADVANCED_DATA_LINK_MADL = 109,
     DIS_PDU_SIGNAL_TDL_TYPE_CURSOR_ON_TARGET = 110,
     DIS_PDU_SIGNAL_TDL_TYPE_ALL_PURPOSE_STRUCTURED_EUROCONTROL_SURVEILLANCE_INFORMATION_EXCHANGE_ASTERIX = 111,
     DIS_PDU_SIGNAL_TDL_TYPE_VARIABLE_MESSAGE_FORMAT_VMF_OVER_COMBAT_NET_RADIO_VMF_OVER_CNR = 112,
     DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_SURROGATE_FOR_NON_NATO_TDL = 113,
     DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_C_BAND_LOS_UPLINK = 114,
     DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_C_BAND_LOS_DOWNLINK = 115,
     DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_KU_BAND_SATCOM_UPLINK = 116,
     DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_KU_BAND_SATCOM_DOWNLINK = 117,
     DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATALINK_SDB_II = 118,
     DIS_PDU_SIGNAL_TDL_TYPE_JTAC_SA_UPLINK = 119,
     DIS_PDU_SIGNAL_TDL_TYPE_COMMON_INTERACTIVE_BROADCAST_CIB = 120,
     DIS_PDU_SIGNAL_TDL_TYPE_JOINT_RANGE_EXTENSION_APPLICATION_PROTOCOL_A_JREAP_A = 121,
     DIS_PDU_SIGNAL_TDL_TYPE_JPALS_DATA_LINK = 125,
     DIS_PDU_SIGNAL_TDL_TYPE_ONESAF_IADS_COMMUNICATIONS_LINK = 126,
     DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_TARGETING_NETWORK_TECHNOLOGY_TTNT_APPLICATION = 127
}  DIS_PDU_Signal_TDL_Type;

static const value_string DIS_PDU_Signal_TDL_Type_Strings[] = {
     { DIS_PDU_SIGNAL_TDL_TYPE_OTHER, "Other" },
     { DIS_PDU_SIGNAL_TDL_TYPE_PADIL, "PADIL" },
     { DIS_PDU_SIGNAL_TDL_TYPE_NATO_LINK_1, "NATO Link-1" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ATDL_1, "ATDL-1" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_11B_TADIL_B, "Link 11B (TADIL B)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SITUATIONAL_AWARENESS_DATA_LINK_SADL, "Situational Awareness Data Link (SADL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_LEGACY_FORMAT_JTIDS_TADIL_J, "Link 16 Legacy Format (JTIDS/TADIL-J)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_LEGACY_FORMAT_JTIDS_FDL_TADIL_J, "Link 16 Legacy Format (JTIDS/FDL/TADIL-J)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_11_TADIL_A, "Link 11 (TADIL A)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_IJMS, "IJMS" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_4A_TADIL_C, "Link 4A (TADIL C)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_4C, "Link 4C" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TIBS, "TIBS" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ATL, "ATL" },
     { DIS_PDU_SIGNAL_TDL_TYPE_CONSTANT_SOURCE, "Constant Source" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ABBREVIATED_COMMAND_AND_CONTROL, "Abbreviated Command and Control" },
     { DIS_PDU_SIGNAL_TDL_TYPE_MILSTAR, "MILSTAR" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ATHS, "ATHS" },
     { DIS_PDU_SIGNAL_TDL_TYPE_OTHGOLD, "OTHGOLD" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TACELINT, "TACELINT" },
     { DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AWW_13, "Weapons Data Link (AWW-13)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ENHANCED_POSITION_LOCATION_REPORTING_SYSTEM_EPLRS, "Enhanced Position Location Reporting System (EPLRS)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_POSITION_LOCATION_REPORTING_SYSTEM_PLRS, "Position Location Reporting System (PLRS)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SINCGARS, "SINCGARS" },
     { DIS_PDU_SIGNAL_TDL_TYPE_HAVE_QUICK_I, "HAVE QUICK I" },
     { DIS_PDU_SIGNAL_TDL_TYPE_HAVE_QUICK_II, "HAVE QUICK II" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SATURN, "SATURN" },
     { DIS_PDU_SIGNAL_TDL_TYPE_INTRA_FLIGHT_DATA_LINK_1, "Intra-Flight Data Link 1" },
     { DIS_PDU_SIGNAL_TDL_TYPE_INTRA_FLIGHT_DATA_LINK_2, "Intra-Flight Data Link 2" },
     { DIS_PDU_SIGNAL_TDL_TYPE_IMPROVED_DATA_MODEM_IDM, "Improved Data Modem (IDM)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_AIR_FORCE_APPLICATION_PROGRAM_DEVELOPMENT_AFAPD, "Air Force Application Program Development (AFAPD)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_COOPERATIVE_ENGAGEMENT_CAPABILITY_CEC, "Cooperative Engagement Capability (CEC)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_FORWARD_AREA_AIR_DEFENSE_FAAD_DATA_LINK_FDL, "Forward Area Air Defense (FAAD) Data Link (FDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_GROUND_BASED_DATA_LINK_GBDL, "Ground Based Data Link (GBDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_INTRA_VEHICULAR_INFO_SYSTEM_IVIS, "Intra Vehicular Info System (IVIS)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_MARINE_TACTICAL_SYSTEM_MTS, "Marine Tactical System (MTS)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_FIRE_DIRECTION_SYSTEM_TACFIRE, "Tactical Fire Direction System (TACFIRE)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_INTEGRATED_BROADCAST_SERVICE_IBS, "Integrated Broadcast Service (IBS)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_AIRBORNE_INFORMATION_TRANSFER_ABIT, "Airborne Information Transfer (ABIT)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ADVANCED_TACTICAL_AIRBORNE_RECONNAISSANCE_SYSTEM_ATARS_DATA_LINK, "Advanced Tactical Airborne Reconnaissance System (ATARS) Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_BATTLE_GROUP_PASSIVE_HORIZON_EXTENSION_SYSTEM_BGPHES_DATA_LINK, "Battle Group Passive Horizon Extension System (BGPHES) Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_COMMON_HIGH_BANDWIDTH_DATA_LINK_CHBDL, "Common High Bandwidth Data Link (CHBDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_INTEROPERABLE_DATA_LINK_IDL, "Guardrail Interoperable Data Link (IDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_COMMON_SENSOR_SYSTEM_ONE_CSS1_DATA_LINK, "Guardrail Common Sensor System One (CSS1) Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_COMMON_SENSOR_SYSTEM_TWO_CSS2_DATA_LINK, "Guardrail Common Sensor System Two (CSS2) Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_CSS2_MULTI_ROLE_DATA_LINK_MRDL, "Guardrail CSS2 Multi-Role Data Link (MRDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_GUARDRAIL_CSS2_DIRECT_AIR_TO_SATELLITE_RELAY_DASR_DATA_LINK, "Guardrail CSS2 Direct Air to Satellite Relay (DASR) Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINE_OF_SIGHT_LOS_DATA_LINK_IMPLEMENTATION_LOS_TETHER, "Line of Sight (LOS) Data Link Implementation (LOS tether)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LIGHTWEIGHT_CDL_LWCDL, "Lightweight CDL (LWCDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_L_52M_SR_71, "L-52M (SR-71)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_RIVET_REACH_RIVET_OWL_DATA_LINK, "Rivet Reach/Rivet Owl Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_SPAN, "Senior Span" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_SPUR, "Senior Spur" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_STRETCH, "Senior Stretch." },
     { DIS_PDU_SIGNAL_TDL_TYPE_SENIOR_YEAR_INTEROPERABLE_DATA_LINK_IDL, "Senior Year Interoperable Data Link (IDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SPACE_CDL, "Space CDL" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TR_1_MODE_MIST_AIRBORNE_DATA_LINK, "TR-1 mode MIST Airborne Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_KU_BAND_SATCOM_DATA_LINK_IMPLEMENTATION_UAV, "Ku-band SATCOM Data Link Implementation (UAV)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_MISSION_EQUIPMENT_CONTROL_DATA_LINK_MECDL, "Mission Equipment Control Data link (MECDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_RADAR_DATA_TRANSMITTING_SET_DATA_LINK, "Radar Data Transmitting Set Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_SURVEILLANCE_AND_CONTROL_DATA_LINK_SCDL, "Surveillance and Control Data Link (SCDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_UAV_VIDEO, "Tactical UAV Video" },
     { DIS_PDU_SIGNAL_TDL_TYPE_UHF_SATCOM_DATA_LINK_IMPLEMENTATION_UAV, "UHF SATCOM Data Link Implementation (UAV)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_COMMON_DATA_LINK_TCDL, "Tactical Common Data Link (TCDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LOW_LEVEL_AIR_PICTURE_INTERFACE_LLAPI, "Low Level Air Picture Interface (LLAPI)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AGM_130, "Weapons Data Link (AGM-130)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_AUTOMATIC_IDENTIFICATION_SYSTEM_AIS, "Automatic Identification System (AIS)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AIM_120, "Weapons Data Link (AIM-120)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_AIM_9, "Weapons Data Link (AIM-9)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATA_LINK_CAMM, "Weapons Data Link (CAMM)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_GC3, "GC3" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_STANDARDIZED_FORMAT_JTIDS_MIDS_TADIL_J, "Link 16 Standardized Format (JTIDS/MIDS/TADIL J)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_ENHANCED_DATA_RATE_EDR_JTIDS_MIDS_TADIL_J, "Link 16 Enhanced Data Rate (EDR JTIDS/MIDS/TADIL-J)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_JTIDS_MIDS_NET_DATA_LOAD_TIMS_TOMS, "JTIDS/MIDS Net Data Load (TIMS/TOMS)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_22, "Link 22" },
     { DIS_PDU_SIGNAL_TDL_TYPE_AFIWC_IADS_COMMUNICATIONS_LINKS, "AFIWC IADS Communications Links" },
     { DIS_PDU_SIGNAL_TDL_TYPE_F_22_INTRA_FLIGHT_DATA_LINK_IFDL, "F-22 Intra-Flight Data Link (IFDL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_L_BAND_SATCOM, "L-Band SATCOM" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TSAF_COMMUNICATIONS_LINK, "TSAF Communications Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ENHANCED_SINCGARS_7_3, "Enhanced SINCGARS 7.3" },
     { DIS_PDU_SIGNAL_TDL_TYPE_F_35_MULTIFUNCTION_ADVANCED_DATA_LINK_MADL, "F-35 Multifunction Advanced Data Link (MADL)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_CURSOR_ON_TARGET, "Cursor on Target" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ALL_PURPOSE_STRUCTURED_EUROCONTROL_SURVEILLANCE_INFORMATION_EXCHANGE_ASTERIX, "All Purpose Structured Eurocontrol Surveillance Information Exchange (ASTERIX)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_VARIABLE_MESSAGE_FORMAT_VMF_OVER_COMBAT_NET_RADIO_VMF_OVER_CNR, "Variable Message Format (VMF) over Combat Net Radio (VMF over CNR)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_SURROGATE_FOR_NON_NATO_TDL, "Link 16 Surrogate for Non-NATO TDL" },
     { DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_C_BAND_LOS_UPLINK, "MQ-1/9 C-Band LOS Uplink" },
     { DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_C_BAND_LOS_DOWNLINK, "MQ-1/9 C-Band LOS Downlink" },
     { DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_KU_BAND_SATCOM_UPLINK, "MQ-1/9 Ku-Band SATCOM Uplink" },
     { DIS_PDU_SIGNAL_TDL_TYPE_MQ_1_9_KU_BAND_SATCOM_DOWNLINK, "MQ-1/9 Ku-Band SATCOM Downlink" },
     { DIS_PDU_SIGNAL_TDL_TYPE_WEAPONS_DATALINK_SDB_II, "Weapons Datalink (SDB II)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_JTAC_SA_UPLINK, "JTAC SA Uplink" },
     { DIS_PDU_SIGNAL_TDL_TYPE_COMMON_INTERACTIVE_BROADCAST_CIB, "Common Interactive Broadcast (CIB)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_JOINT_RANGE_EXTENSION_APPLICATION_PROTOCOL_A_JREAP_A, "Joint Range Extension Application Protocol A (JREAP A)" },
     { DIS_PDU_SIGNAL_TDL_TYPE_JPALS_DATA_LINK, "JPALS Data Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_ONESAF_IADS_COMMUNICATIONS_LINK, "OneSAF IADS Communications Link" },
     { DIS_PDU_SIGNAL_TDL_TYPE_TACTICAL_TARGETING_NETWORK_TECHNOLOGY_TTNT_APPLICATION, "Tactical Targeting Network Technology (TTNT) Application" },
     { 0, NULL }
 };

static value_string_ext DIS_PDU_Signal_TDL_Type_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_Signal_TDL_Type_Strings);

/* SISO-REF-010 [UID 179] */
typedef enum
{
     DIS_PDU_RECEIVER_RECEIVER_STATE_OFF = 0,
     DIS_PDU_RECEIVER_RECEIVER_STATE_ON_BUT_NOT_RECEIVING = 1,
     DIS_PDU_RECEIVER_RECEIVER_STATE_ON_AND_RECEIVING = 2
}  DIS_PDU_Receiver_Receiver_State;

static const value_string DIS_PDU_Receiver_Receiver_State_Strings[] = {
     { DIS_PDU_RECEIVER_RECEIVER_STATE_OFF, "Off" },
     { DIS_PDU_RECEIVER_RECEIVER_STATE_ON_BUT_NOT_RECEIVING, "On but not receiving" },
     { DIS_PDU_RECEIVER_RECEIVER_STATE_ON_AND_RECEIVING, "On and receiving" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 180] */
typedef enum
{
     DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_RESERVED = 0,
     DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_STATUS = 1,
     DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_REQUEST_ACKNOWLEDGE_REQUIRED = 2,
     DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_REQUEST_NO_ACKNOWLEDGE = 3,
     DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_ACK_REQUEST_GRANTED = 4,
     DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_NACK_REQUEST_DENIED = 5
}  DIS_PDU_Intercom_Control_Control_Type;

static const value_string DIS_PDU_Intercom_Control_Control_Type_Strings[] = {
     { DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_RESERVED, "Reserved" },
     { DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_STATUS, "Status" },
     { DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_REQUEST_ACKNOWLEDGE_REQUIRED, "Request - Acknowledge Required" },
     { DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_REQUEST_NO_ACKNOWLEDGE, "Request - No Acknowledge" },
     { DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_ACK_REQUEST_GRANTED, "Ack - Request Granted" },
     { DIS_PDU_INTERCOM_CONTROL_CONTROL_TYPE_NACK_REQUEST_DENIED, "Nack - Request Denied" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 181] */
typedef enum
{
     DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_RESERVED = 0,
     DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_FDX = 1,
     DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_HDX_DESTINATION_IS_RECEIVE_ONLY = 2,
     DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_HDX_DESTINATION_IS_TRANSMIT_ONLY = 3,
     DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_HDX = 4
}  DIS_PDU_Intercom_Control_Communications_Type;

static const value_string DIS_PDU_Intercom_Control_Communications_Type_Strings[] = {
     { DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_RESERVED, "Reserved" },
     { DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_FDX, "Connection FDX" },
     { DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_HDX_DESTINATION_IS_RECEIVE_ONLY, "Connection HDX - Destination is Receive Only" },
     { DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_HDX_DESTINATION_IS_TRANSMIT_ONLY, "Connection HDX - Destination is Transmit Only" },
     { DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_TYPE_CONNECTION_HDX, "Connection HDX" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 182] */
typedef enum
{
     DIS_PDU_INTERCOM_CONTROL_COMMAND_NO_COMMAND = 0,
     DIS_PDU_INTERCOM_CONTROL_COMMAND_STATUS = 1,
     DIS_PDU_INTERCOM_CONTROL_COMMAND_CONNECT = 2,
     DIS_PDU_INTERCOM_CONTROL_COMMAND_DISCONNECT = 3,
     DIS_PDU_INTERCOM_CONTROL_COMMAND_RESET = 4,
     DIS_PDU_INTERCOM_CONTROL_COMMAND_ON = 5,
     DIS_PDU_INTERCOM_CONTROL_COMMAND_OFF = 6
}  DIS_PDU_Intercom_Control_Command;

/* SISO-REF-010 [UID 183] */
typedef enum
{
     DIS_PDU_INTERCOM_CONTROL_TRANSMIT_LINE_STATE_TRANSMIT_LINE_STATE_NOT_APPLICABLE = 0,
     DIS_PDU_INTERCOM_CONTROL_TRANSMIT_LINE_STATE_NOT_TRANSMITTING = 1,
     DIS_PDU_INTERCOM_CONTROL_TRANSMIT_LINE_STATE_TRANSMITTING = 2
}  DIS_PDU_Intercom_Control_Transmit_Line_State;

/* SISO-REF-010 [UID 184] */
typedef enum
{
     DIS_PDU_INTERCOM_CONTROL_DESTINATION_LINE_STATE_COMMAND_NONE = 0,
     DIS_PDU_INTERCOM_CONTROL_DESTINATION_LINE_STATE_COMMAND_SET_LINE_STATE_TRANSMITTING = 1,
     DIS_PDU_INTERCOM_CONTROL_DESTINATION_LINE_STATE_COMMAND_SET_LINE_STATE_NOT_TRANSMITTING = 2,
     DIS_PDU_INTERCOM_CONTROL_DESTINATION_LINE_STATE_COMMAND_RETURN_TO_LOCAL_LINE_STATE_CONTROL = 3
}  DIS_PDU_Intercom_Control_Destination_Line_State_Command;

/* SISO-REF-010 [UID 185] */
typedef enum
{
     DIS_PDU_INTERCOM_CONTROL_RECORD_TYPE_SPECIFIC_DESTINATION_RECORD = 1,
     DIS_PDU_INTERCOM_CONTROL_RECORD_TYPE_GROUP_DESTINATION_RECORD = 2,
     DIS_PDU_INTERCOM_CONTROL_RECORD_TYPE_GROUP_ASSIGNMENT_RECORD = 3
}  DIS_PDU_Intercom_Control_Record_Type;

/* SISO-REF-010 [UID 189] */
typedef enum
{
     DIS_PDU_COLLISION_TYPE_INELASTIC = 0,
     DIS_PDU_COLLISION_TYPE_ELASTIC = 1,
     DIS_PDU_COLLISION_TYPE_THE_BOOM_NOZZLE_IS_IN_PHYSICAL_CONTACT_WITH_THE_RECEPTACLE_AND_THE_BOOMS_SIGNAL_SYSTEM_IS_OPERATIVE = 2,
     DIS_PDU_COLLISION_TYPE_THE_BOOM_TRAINERS_SIGNAL_SYSTEM_HAS_SENT_A_DISCONNECT_SIGNAL_THAT_SHOULD_CAUSE_THE_RECEIVERS_REFUELING_RECEPTACLE_TO_UNLATCH_UNLESS_THERE_IS_A_MALFUNCTION = 3,
     DIS_PDU_COLLISION_TYPE_DISCONNECT_WITHOUT_A_SIGNAL_BEING_SENT_DISCONNECTED_BY_PHYSICAL_MEANS_BRUTE_FORCE_DISCONNECT_CONTROLLED_TENSION_DISCONNECT = 4,
     DIS_PDU_COLLISION_TYPE_THE_BOOM_NOZZLE_IS_IN_PHYSICAL_CONTACT_WITH_THE_RECEPTACLE_AND_THE_BOOMS_SIGNAL_SYSTEM_IS_INOPERATIVE = 5,
     DIS_PDU_COLLISION_TYPE_BOOM_SIMULATOR_HAS_CALCULATED_THAT_THE_RECEIVERS_LATCHES_HAVE_BEEN_DAMAGED = 6,
     DIS_PDU_COLLISION_TYPE_AR_RECEPTACLE_DOOR_1_DAMAGED = 7,
     DIS_PDU_COLLISION_TYPE_AR_RECEPTACLE_DOOR_2_DAMAGED = 8,
     DIS_PDU_COLLISION_TYPE_PILOTS_COCKPIT_WINDSHIELD_DAMAGED = 9,
     DIS_PDU_COLLISION_TYPE_COPILOTS_COCKPIT_WINDSHIELD_DAMAGED = 10,
     DIS_PDU_COLLISION_TYPE_PILOTS_LEFT_SIDE_WINDOW_DAMAGED_CALLED_PILOTS_CLEARVIEW_WINDOW_IN_C_17_FLIGHT_MANUAL = 11,
     DIS_PDU_COLLISION_TYPE_COPILOTS_RIGHT_SIDE_WINDOW_DAMAGED_CALLED_COPILOTS_CLEARVIEW_WINDOW_IN_C_17_FLIGHT_MANUAL = 12,
     DIS_PDU_COLLISION_TYPE_PILOTS_EYEBROW_WINDOW_DAMAGED = 13,
     DIS_PDU_COLLISION_TYPE_COPILOTS_EYEBROW_WINDOW_DAMAGED = 14,
     DIS_PDU_COLLISION_TYPE_MLS_GLIDE_SLOPE_1_ANTENNA_DAMAGED = 15,
     DIS_PDU_COLLISION_TYPE_MLS_GLIDE_SLOPE_2_ANTENNA_DAMAGED = 16,
     DIS_PDU_COLLISION_TYPE_ILS_GLIDE_1_SLOPE_ANTENNA_DAMAGED = 17,
     DIS_PDU_COLLISION_TYPE_ILS_GLIDE_2_SLOPE_ANTENNA_DAMAGED = 18,
     DIS_PDU_COLLISION_TYPE_SKE_OMNI_ANTENNA_DAMAGED = 19,
     DIS_PDU_COLLISION_TYPE_SKE_DIRECTIONAL_ANTENNA_DAMAGED = 20,
     DIS_PDU_COLLISION_TYPE_WEATHER_RADAR_ANTENNA_DAMAGED = 21,
     DIS_PDU_COLLISION_TYPE_COM_1_ARC_210_ANTENNA_DAMAGED = 22,
     DIS_PDU_COLLISION_TYPE_TACAN_ANTENNA_DAMAGED = 23,
     DIS_PDU_COLLISION_TYPE_MLS_ANTENNA_DAMAGED = 24,
     DIS_PDU_COLLISION_TYPE_AF_SATCOM_ANTENNA_DAMAGED = 25,
     DIS_PDU_COLLISION_TYPE_AERO_I_SATCOM_ANTENNA_DAMAGED = 26,
     DIS_PDU_COLLISION_TYPE_AERO_H_SATCOM_ANTENNA_DAMAGED = 28,
     DIS_PDU_COLLISION_TYPE_UHF_SATCOM_ANTENNA_DAMAGED = 29,
     DIS_PDU_COLLISION_TYPE_HMSA_ANTENNA_DAMAGED = 30,
     DIS_PDU_COLLISION_TYPE_IFF_ANTENNA_DAMAGED = 31,
     DIS_PDU_COLLISION_TYPE_LEFT_SIDE_LANDING_TAXI_LIGHT_DAMAGED = 32,
     DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_LANDING_TAXI_LIGHT_DAMAGED = 33,
     DIS_PDU_COLLISION_TYPE_LEFT_SIDE_RUNWAY_TURNOFF_LIGHT_DAMAGED = 34,
     DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_RUNWAY_TURNOFF_LIGHT_DAMAGED = 35,
     DIS_PDU_COLLISION_TYPE_LEFT_SIDE_FORMATION_LIGHT_DAMAGED = 36,
     DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_FORMATION_LIGHT_DAMAGED = 37,
     DIS_PDU_COLLISION_TYPE_LEFT_SIDE_NACELLE_SCANNING_LIGHT_DAMAGED = 38,
     DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_NACELLE_SCANNING_LIGHT_DAMAGED = 39,
     DIS_PDU_COLLISION_TYPE_COPILOT_PITOT_STATIC_PROBE_1_DAMAGED = 40,
     DIS_PDU_COLLISION_TYPE_COPILOT_PITOT_STATIC_PROBE_2_DAMAGED = 41,
     DIS_PDU_COLLISION_TYPE_PILOT_PITOT_STATIC_PROBE_1_DAMAGED = 42,
     DIS_PDU_COLLISION_TYPE_PILOT_PITOT_STATIC_PROBE_2_DAMAGED = 43,
     DIS_PDU_COLLISION_TYPE_TOTAL_AIR_TEMPERATURE_PROBE_1_DAMAGED = 44,
     DIS_PDU_COLLISION_TYPE_TOTAL_AIR_TEMPERATURE_PROBE_2_DAMAGED = 45,
     DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_1_DAMAGED = 46,
     DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_2_DAMAGED = 47,
     DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_3_DAMAGED = 48,
     DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_4_DAMAGED = 49,
     DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_5_DAMAGED = 50,
     DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_6_DAMAGED = 51,
     DIS_PDU_COLLISION_TYPE_LEFT_SIDE_SPOILER_DAMAGED = 52,
     DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_SPOILER_DAMAGED = 53,
     DIS_PDU_COLLISION_TYPE_UPPER_TCAS_ANTENNA_KC_135_R_T_DAMAGED = 54,
     DIS_PDU_COLLISION_TYPE_BOOM_NOZZLE_HAS_CLEARED_THE_RECEIVER_S_REFUELING_RECEPTACLE = 55
}  DIS_PDU_Collision_Type;

static const value_string DIS_PDU_Collision_Type_Strings[] = {
     { DIS_PDU_COLLISION_TYPE_INELASTIC, "Inelastic" },
     { DIS_PDU_COLLISION_TYPE_ELASTIC, "Elastic" },
     { DIS_PDU_COLLISION_TYPE_THE_BOOM_NOZZLE_IS_IN_PHYSICAL_CONTACT_WITH_THE_RECEPTACLE_AND_THE_BOOMS_SIGNAL_SYSTEM_IS_OPERATIVE, "The boom nozzle is in physical contact with the receptacle and the booms signal system is operative." },
     { DIS_PDU_COLLISION_TYPE_THE_BOOM_TRAINERS_SIGNAL_SYSTEM_HAS_SENT_A_DISCONNECT_SIGNAL_THAT_SHOULD_CAUSE_THE_RECEIVERS_REFUELING_RECEPTACLE_TO_UNLATCH_UNLESS_THERE_IS_A_MALFUNCTION, "The boom trainers signal system has sent a disconnect signal that should cause the receivers refueling receptacle to unlatch (unless there is a malfunction)." },
     { DIS_PDU_COLLISION_TYPE_DISCONNECT_WITHOUT_A_SIGNAL_BEING_SENT_DISCONNECTED_BY_PHYSICAL_MEANS_BRUTE_FORCE_DISCONNECT_CONTROLLED_TENSION_DISCONNECT, "Disconnect without a signal being sent - disconnected by physical means (brute force disconnect, controlled tension disconnect)." },
     { DIS_PDU_COLLISION_TYPE_THE_BOOM_NOZZLE_IS_IN_PHYSICAL_CONTACT_WITH_THE_RECEPTACLE_AND_THE_BOOMS_SIGNAL_SYSTEM_IS_INOPERATIVE, "The boom nozzle is in physical contact with the receptacle and the booms signal system is inoperative." },
     { DIS_PDU_COLLISION_TYPE_BOOM_SIMULATOR_HAS_CALCULATED_THAT_THE_RECEIVERS_LATCHES_HAVE_BEEN_DAMAGED, "Boom simulator has calculated that the receivers latches have been damaged." },
     { DIS_PDU_COLLISION_TYPE_AR_RECEPTACLE_DOOR_1_DAMAGED, "AR receptacle door #1 damaged." },
     { DIS_PDU_COLLISION_TYPE_AR_RECEPTACLE_DOOR_2_DAMAGED, "AR receptacle door #2 damaged." },
     { DIS_PDU_COLLISION_TYPE_PILOTS_COCKPIT_WINDSHIELD_DAMAGED, "Pilots Cockpit Windshield damaged" },
     { DIS_PDU_COLLISION_TYPE_COPILOTS_COCKPIT_WINDSHIELD_DAMAGED, "Copilots Cockpit Windshield damaged" },
     { DIS_PDU_COLLISION_TYPE_PILOTS_LEFT_SIDE_WINDOW_DAMAGED_CALLED_PILOTS_CLEARVIEW_WINDOW_IN_C_17_FLIGHT_MANUAL, "Pilots Left Side Window damaged (called Pilots Clearview window in C-17 flight manual)" },
     { DIS_PDU_COLLISION_TYPE_COPILOTS_RIGHT_SIDE_WINDOW_DAMAGED_CALLED_COPILOTS_CLEARVIEW_WINDOW_IN_C_17_FLIGHT_MANUAL, "Copilots Right Side Window damaged (called Copilots Clearview window in C-17 flight manual)" },
     { DIS_PDU_COLLISION_TYPE_PILOTS_EYEBROW_WINDOW_DAMAGED, "Pilots Eyebrow Window damaged" },
     { DIS_PDU_COLLISION_TYPE_COPILOTS_EYEBROW_WINDOW_DAMAGED, "Copilots Eyebrow Window damaged" },
     { DIS_PDU_COLLISION_TYPE_MLS_GLIDE_SLOPE_1_ANTENNA_DAMAGED, "MLS Glide Slope #1 Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_MLS_GLIDE_SLOPE_2_ANTENNA_DAMAGED, "MLS Glide Slope #2 Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_ILS_GLIDE_1_SLOPE_ANTENNA_DAMAGED, "ILS Glide #1 Slope Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_ILS_GLIDE_2_SLOPE_ANTENNA_DAMAGED, "ILS Glide #2 Slope Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_SKE_OMNI_ANTENNA_DAMAGED, "SKE OMNI Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_SKE_DIRECTIONAL_ANTENNA_DAMAGED, "SKE Directional Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_WEATHER_RADAR_ANTENNA_DAMAGED, "Weather Radar Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_COM_1_ARC_210_ANTENNA_DAMAGED, "Com #1 ARC 210 Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_TACAN_ANTENNA_DAMAGED, "TACAN Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_MLS_ANTENNA_DAMAGED, "MLS Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_AF_SATCOM_ANTENNA_DAMAGED, "AF SATCOM Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_AERO_I_SATCOM_ANTENNA_DAMAGED, "AERO-I SATCOM Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_AERO_H_SATCOM_ANTENNA_DAMAGED, "AERO-H SATCOM Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_UHF_SATCOM_ANTENNA_DAMAGED, "UHF SATCOM Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_HMSA_ANTENNA_DAMAGED, "HMSA Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_IFF_ANTENNA_DAMAGED, "IFF Antenna damaged" },
     { DIS_PDU_COLLISION_TYPE_LEFT_SIDE_LANDING_TAXI_LIGHT_DAMAGED, "Left Side Landing/Taxi Light damaged" },
     { DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_LANDING_TAXI_LIGHT_DAMAGED, "Right Side Landing/Taxi Light damaged" },
     { DIS_PDU_COLLISION_TYPE_LEFT_SIDE_RUNWAY_TURNOFF_LIGHT_DAMAGED, "Left Side Runway Turnoff Light damaged" },
     { DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_RUNWAY_TURNOFF_LIGHT_DAMAGED, "Right Side Runway Turnoff Light damaged" },
     { DIS_PDU_COLLISION_TYPE_LEFT_SIDE_FORMATION_LIGHT_DAMAGED, "Left Side Formation Light damaged" },
     { DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_FORMATION_LIGHT_DAMAGED, "Right Side Formation Light damaged" },
     { DIS_PDU_COLLISION_TYPE_LEFT_SIDE_NACELLE_SCANNING_LIGHT_DAMAGED, "Left Side Nacelle Scanning Light damaged" },
     { DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_NACELLE_SCANNING_LIGHT_DAMAGED, "Right Side Nacelle Scanning Light damaged" },
     { DIS_PDU_COLLISION_TYPE_COPILOT_PITOT_STATIC_PROBE_1_DAMAGED, "Copilot Pitot Static Probe #1 damaged" },
     { DIS_PDU_COLLISION_TYPE_COPILOT_PITOT_STATIC_PROBE_2_DAMAGED, "Copilot Pitot Static Probe #2 damaged" },
     { DIS_PDU_COLLISION_TYPE_PILOT_PITOT_STATIC_PROBE_1_DAMAGED, "Pilot Pitot Static Probe #1 damaged" },
     { DIS_PDU_COLLISION_TYPE_PILOT_PITOT_STATIC_PROBE_2_DAMAGED, "Pilot Pitot Static Probe #2 damaged" },
     { DIS_PDU_COLLISION_TYPE_TOTAL_AIR_TEMPERATURE_PROBE_1_DAMAGED, "Total Air Temperature Probe #1 damaged" },
     { DIS_PDU_COLLISION_TYPE_TOTAL_AIR_TEMPERATURE_PROBE_2_DAMAGED, "Total Air Temperature Probe #2 damaged" },
     { DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_1_DAMAGED, "Angle of Attack Sensor #1 damaged" },
     { DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_2_DAMAGED, "Angle of Attack Sensor #2 damaged" },
     { DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_3_DAMAGED, "Angle of Attack Sensor #3 damaged" },
     { DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_4_DAMAGED, "Angle of Attack Sensor #4 damaged" },
     { DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_5_DAMAGED, "Angle of Attack Sensor #5 damaged" },
     { DIS_PDU_COLLISION_TYPE_ANGLE_OF_ATTACK_SENSOR_6_DAMAGED, "Angle of Attack Sensor #6 damaged" },
     { DIS_PDU_COLLISION_TYPE_LEFT_SIDE_SPOILER_DAMAGED, "Left Side Spoiler damaged" },
     { DIS_PDU_COLLISION_TYPE_RIGHT_SIDE_SPOILER_DAMAGED, "Right Side Spoiler damaged" },
     { DIS_PDU_COLLISION_TYPE_UPPER_TCAS_ANTENNA_KC_135_R_T_DAMAGED, "Upper TCAS antenna (KC-135 R/T) damaged" },
     { DIS_PDU_COLLISION_TYPE_BOOM_NOZZLE_HAS_CLEARED_THE_RECEIVER_S_REFUELING_RECEPTACLE, "Boom nozzle has cleared the receiver's refueling receptacle" },
     { 0, NULL }
 };

static value_string_ext DIS_PDU_Collision_Type_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_Collision_Type_Strings);

/* SISO-REF-010 [UID 193] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OTHER = 0,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL = 1,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR = 2,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_RADAR = 3,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_MAGNETIC = 4,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_LASER = 5,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_SONAR = 6,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_PHYSICAL = 7,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_MULTISPECTRAL = 8
}  DIS_PDU_Minefield_Sensor_Types;

/* SISO-REF-010 [UID 194] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_UNAIDED_EYE_ACTIVELY_SEARCHING = 0,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_UNAIDED_EYE_NOT_ACTIVELY_SEARCHING = 1,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_BINOCULARS = 2,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_IMAGE_INTENSIFIER = 3,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_HMMWV_OCCUPANT_ACTIVELY_SEARCHING = 4,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_HMMWV_OCCUPANT_NOT_ACTIVELY_SEARCHING = 5,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_TRUCK_OCCUPANT_ACTIVELY_SEARCHING = 6,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_TRUCK_OCCUPANT_NOT_ACTIVELY_SEARCHING = 7,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_TRACKED_VEHICLE_OCCUPANT_CLOSED_HATCH_ACTIVELY_SEARCHING = 8,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_TRACKED_VEHICLE_OCCUPANT_CLOSED_HATCH_NOT_ACTIVELY_SEARCHING = 9,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_TRACKED_VEHICLE_OCCUPANT_OPEN_HATCH_ACTIVELY_SEARCHING = 10,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_OPTICAL_TRACKED_VEHICLE_OCCUPANT_OPEN_HATCH_NOT_ACTIVELY_SEARCHING = 11
}  DIS_PDU_Minefield_Sensor_Types_Optical;

/* SISO-REF-010 [UID 195] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_GENERIC_3_5 = 0,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_GENERIC_8_12 = 1,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_ASTAMIDS_I = 2,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_ASTAMIDS_II = 3,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_GSTAMIDS_3_5 = 4,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_GSTAMIDS_8_12 = 5,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_HSTAMIDS_3_5 = 6,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_HSTAMIDS_8_12 = 7,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_COBRA_3_5 = 8,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_FLIR_COBRA_8_12 = 9
}  DIS_PDU_Minefield_Sensor_Types_FLIR;

/* SISO-REF-010 [UID 196] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_RADAR_GENERIC = 0,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_RADAR_GENERIC_GPR = 1,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_RADAR_GSTAMIDS_I = 2,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_RADAR_GSTAMIDS_II = 3,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_RADAR_HSTAMIDS_I = 4,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_RADAR_HSTAMIDS_II = 5
}  DIS_PDU_Minefield_Sensor_Types_RADAR;

/* SISO-REF-010 [UID 197] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_MAGNETIC_GENERIC = 0,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_MAGNETIC_AN_PSS_11 = 1,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_MAGNETIC_AN_PSS_12 = 2,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_MAGNETIC_GSTAMIDS = 3
}  DIS_PDU_Minefield_Sensor_Types_Magnetic;

/* SISO-REF-010 [UID 198] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_LASER_GENERIC = 0,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_LASER_ASTAMIDS = 1
}  DIS_PDU_Minefield_Sensor_Types_Laser;

/* SISO-REF-010 [UID 199] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_SONAR_GENERIC = 0
}  DIS_PDU_Minefield_Sensor_Types_SONAR;

/* SISO-REF-010 [UID 200] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_PHYSICAL_GENERIC_PROBE = 0,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_PHYSICAL_PROBE_METAL_CONTENT = 1,
     DIS_PDU_MINEFIELD_SENSOR_TYPES_PHYSICAL_PROBE_NO_METAL_CONTENT = 2
}  DIS_PDU_Minefield_Sensor_Types_Physical;

/* SISO-REF-010 [UID 201] */
typedef enum
{
     DIS_PDU_MINEFIELD_SENSOR_TYPES_MULTISPECTRAL_GENERIC = 0
}  DIS_PDU_Minefield_Sensor_Types_Multispectral;

/* SISO-REF-010 [UID 204] */
typedef enum
{
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_OTHER = 0,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_AGGREGATED = 1,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_DISAGGREGATED = 2,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_FULLY_DISAGGREGATED = 3,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_PSEUDO_DISAGGREGATED = 4,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_PARTIALLY_DISAGGREGATED = 5
}  DIS_PDU_Aggregate_State_Aggregate_State;

static const value_string DIS_PDU_Aggregate_State_Aggregate_State_Strings[] = {
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_OTHER, "Other" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_AGGREGATED, "Aggregated" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_DISAGGREGATED, "Disaggregated" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_FULLY_DISAGGREGATED, "Fully disaggregated" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_PSEUDO_DISAGGREGATED, "Pseudo-disaggregated" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_STATE_PARTIALLY_DISAGGREGATED, "Partially-disaggregated" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 205] */
typedef enum
{
     DIS_PDU_AGGREGATE_STATE_FORMATION_OTHER = 0,
     DIS_PDU_AGGREGATE_STATE_FORMATION_ASSEMBLY = 1,
     DIS_PDU_AGGREGATE_STATE_FORMATION_VEE = 2,
     DIS_PDU_AGGREGATE_STATE_FORMATION_WEDGE = 3,
     DIS_PDU_AGGREGATE_STATE_FORMATION_LINE = 4,
     DIS_PDU_AGGREGATE_STATE_FORMATION_COLUMN = 5
}  DIS_PDU_Aggregate_State_Formation;

static const value_string DIS_PDU_Aggregate_State_Formation_Strings[] = {
     { DIS_PDU_AGGREGATE_STATE_FORMATION_OTHER, "Other" },
     { DIS_PDU_AGGREGATE_STATE_FORMATION_ASSEMBLY, "Assembly" },
     { DIS_PDU_AGGREGATE_STATE_FORMATION_VEE, "Vee" },
     { DIS_PDU_AGGREGATE_STATE_FORMATION_WEDGE, "Wedge" },
     { DIS_PDU_AGGREGATE_STATE_FORMATION_LINE, "Line" },
     { DIS_PDU_AGGREGATE_STATE_FORMATION_COLUMN, "Column" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 206] */
typedef enum
{
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_OTHER = 0,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_MILITARY_HIERARCHY = 1,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_COMMON_TYPE = 2,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_COMMON_MISSION = 3,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_SIMILAR_CAPABILITIES = 4,
     DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_COMMON_LOCATION = 5
}  DIS_PDU_Aggregate_State_Aggregate_Kind;

static const value_string DIS_PDU_Aggregate_State_Aggregate_Kind_Strings[] = {
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_OTHER, "Other" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_MILITARY_HIERARCHY, "Military Hierarchy" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_COMMON_TYPE, "Common Type" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_COMMON_MISSION, "Common Mission" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_SIMILAR_CAPABILITIES, "Similar Capabilities" },
     { DIS_PDU_AGGREGATE_STATE_AGGREGATE_KIND_COMMON_LOCATION, "Common Location" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 207] - deprecated */
/* Manually Inserted */
static const value_string aggregate_category_vals[] =
{
    { 0, "Other" },
    { 1, "Individual Vehicle" },
    { 2, "Element" },
    { 3, "Platoon" },
    { 4, "Battery" },
    { 5, "Company" },
    { 6, "Battalion" },
    { 7, "Regiment" },
    { 8, "Brigade" },
    { 9, "Division" },
    { 10, "Corps" },
    { 0, NULL }
};

/* SISO-REF-010 [UID 208] */
typedef enum
{
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_OTHER = 0,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CAVALRY_TROOP = 1,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMOR = 2,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_INFANTRY = 3,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_MECHANIZED_INFANTRY = 4,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CAVALRY = 5,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMORED_CAVALRY = 6,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARTILLERY = 7,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_SELF_PROPELLED_ARTILLERY = 8,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CLOSE_AIR_SUPPORT = 9,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ENGINEER = 10,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_AIR_DEFENSE_ARTILLERY = 11,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ANTI_TANK = 12,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMY_AVIATION_FIXED_WING = 13,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMY_AVIATION_ROTARY_WING = 14,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMY_ATTACK_HELICOPTER = 15,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_AIR_CAVALRY = 16,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMOR_HEAVY_TASK_FORCE = 17,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_MOTORIZED_RIFLE = 18,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_MECHANIZED_HEAVY_TASK_FORCE = 19,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_COMMAND_POST = 20,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CEWI = 21,
     DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_TANK_ONLY = 22
}  DIS_PDU_Aggregate_State_Subcategory;

static const value_string DIS_PDU_Aggregate_State_Subcategory_Strings[] = {
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_OTHER, "Other" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CAVALRY_TROOP, "Cavalry Troop" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMOR, "Armor" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_INFANTRY, "Infantry" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_MECHANIZED_INFANTRY, "Mechanized Infantry" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CAVALRY, "Cavalry" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMORED_CAVALRY, "Armored Cavalry" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARTILLERY, "Artillery" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_SELF_PROPELLED_ARTILLERY, "Self-Propelled Artillery" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CLOSE_AIR_SUPPORT, "Close Air Support" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ENGINEER, "Engineer" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_AIR_DEFENSE_ARTILLERY, "Air Defense Artillery" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ANTI_TANK, "Anti-Tank" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMY_AVIATION_FIXED_WING, "Army Aviation Fixed-wing" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMY_AVIATION_ROTARY_WING, "Army Aviation Rotary-wing" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMY_ATTACK_HELICOPTER, "Army Attack Helicopter" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_AIR_CAVALRY, "Air Cavalry" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_ARMOR_HEAVY_TASK_FORCE, "Armor Heavy Task Force" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_MOTORIZED_RIFLE, "Motorized Rifle" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_MECHANIZED_HEAVY_TASK_FORCE, "Mechanized Heavy Task Force" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_COMMAND_POST, "Command Post" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_CEWI, "CEWI" },
     { DIS_PDU_AGGREGATE_STATE_SUBCATEGORY_TANK_ONLY, "Tank only" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 209] */
typedef enum
{
     DIS_PDU_AGGREGATE_STATE_SPECIFIC_NO_HEADQUARTERS = 0,
     DIS_PDU_AGGREGATE_STATE_SPECIFIC_YES_AGGREGATE_UNIT_CONTAINS_A_HEADQUARTERS = 1
}  DIS_PDU_Aggregate_State_Specific;

static const value_string DIS_PDU_Aggregate_State_Specific_Strings[] = {
     { DIS_PDU_AGGREGATE_STATE_SPECIFIC_NO_HEADQUARTERS, "No headquarters" },
     { DIS_PDU_AGGREGATE_STATE_SPECIFIC_YES_AGGREGATE_UNIT_CONTAINS_A_HEADQUARTERS, "Yes aggregate unit contains a headquarters" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 210] */
typedef enum
{
     DIS_PDU_ISPARTOF_NATURE_OTHER = 0,
     DIS_PDU_ISPARTOF_NATURE_HOST_FIREABLE_MUNITION = 1,
     DIS_PDU_ISPARTOF_NATURE_MUNITION_CARRIED_AS_CARGO = 2,
     DIS_PDU_ISPARTOF_NATURE_FUEL_CARRIED_AS_CARGO = 3,
     DIS_PDU_ISPARTOF_NATURE_GUNMOUNT_ATTACHED_TO_HOST = 4,
     DIS_PDU_ISPARTOF_NATURE_COMPUTER_GENERATED_FORCES_CARRIED_AS_CARGO = 5,
     DIS_PDU_ISPARTOF_NATURE_VEHICLE_CARRIED_AS_CARGO = 6,
     DIS_PDU_ISPARTOF_NATURE_EMITTER_MOUNTED_ON_HOST = 7,
     DIS_PDU_ISPARTOF_NATURE_MOBILE_COMMAND_AND_CONTROL_ENTITY_CARRIED_ABOARD_HOST = 8,
     DIS_PDU_ISPARTOF_NATURE_ENTITY_STATIONED_AT_POSITION_WITH_RESPECT_TO_HOST = 9,
     DIS_PDU_ISPARTOF_NATURE_TEAM_MEMBER_IN_FORMATION_WITH = 10
}  DIS_PDU_IsPartOf_Nature;

/* SISO-REF-010 [UID 211] */
typedef enum
{
     DIS_PDU_ISPARTOF_POSITION_OTHER = 0,
     DIS_PDU_ISPARTOF_POSITION_ON_TOP_OF = 1,
     DIS_PDU_ISPARTOF_POSITION_INSIDE_OF = 2
}  DIS_PDU_IsPartOf_Position;

/* SISO-REF-010 [UID 212] */
typedef enum
{
     DIS_PDU_ISPARTOF_STATION_NAME_OTHER = 0,
     DIS_PDU_ISPARTOF_STATION_NAME_AIRCRAFT_WINGSTATION = 1,
     DIS_PDU_ISPARTOF_STATION_NAME_SHIP_S_FORWARD_GUNMOUNT_STARBOARD = 2,
     DIS_PDU_ISPARTOF_STATION_NAME_SHIP_S_FORWARD_GUNMOUNT_PORT = 3,
     DIS_PDU_ISPARTOF_STATION_NAME_SHIP_S_FORWARD_GUNMOUNT_CENTERLINE = 4,
     DIS_PDU_ISPARTOF_STATION_NAME_SHIP_S_AFT_GUNMOUNT_STARBOARD = 5,
     DIS_PDU_ISPARTOF_STATION_NAME_SHIP_S_AFT_GUNMOUNT_PORT = 6,
     DIS_PDU_ISPARTOF_STATION_NAME_SHIP_S_AFT_GUNMOUNT_CENTERLINE = 7,
     DIS_PDU_ISPARTOF_STATION_NAME_FORWARD_TORPEDO_TUBE = 8,
     DIS_PDU_ISPARTOF_STATION_NAME_AFT_TORPEDO_TUBE = 9,
     DIS_PDU_ISPARTOF_STATION_NAME_BOMB_BAY = 10,
     DIS_PDU_ISPARTOF_STATION_NAME_CARGO_BAY = 11,
     DIS_PDU_ISPARTOF_STATION_NAME_TRUCK_BED = 12,
     DIS_PDU_ISPARTOF_STATION_NAME_TRAILER_BED = 13,
     DIS_PDU_ISPARTOF_STATION_NAME_WELL_DECK = 14,
     DIS_PDU_ISPARTOF_STATION_NAME_ON_STATION_RANGE_AND_BEARING = 15,
     DIS_PDU_ISPARTOF_STATION_NAME_ON_STATION_XYZ = 16,
     DIS_PDU_ISPARTOF_STATION_NAME_AIR_TO_AIR_REFUELING_BOOM = 17,
     DIS_PDU_ISPARTOF_STATION_NAME_AERIAL_REFUELING_RECEPTACLE = 18,
     DIS_PDU_ISPARTOF_STATION_NAME_PORT_SIDE_REFUELING_DROGUE = 19,
     DIS_PDU_ISPARTOF_STATION_NAME_STARBOARD_SIDE_REFUELING_DROGUE = 20,
     DIS_PDU_ISPARTOF_STATION_NAME_CENTER_REFUELING_DROGUE = 21,
     DIS_PDU_ISPARTOF_STATION_NAME_AIR_REFUELING_PROBE = 22
}  DIS_PDU_IsPartOf_Station_Name;

/* SISO-REF-010 [UID 213] */
typedef enum
{
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_UNDEFINED = 0,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_BASIC_GROUND_COMBAT_VEHICLE = 1,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_ENHANCED_GROUND_COMBAT_VEHICLE = 2,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_BASIC_GROUND_COMBAT_SOLDIER = 3,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_ENHANCED_GROUND_COMBAT_SOLDIER = 4,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_BASIC_ROTOR_WING_AIRCRAFT = 5,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_ENHANCED_ROTOR_WING_AIRCRAFT = 6,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_BASIC_FIXED_WING_AIRCRAFT = 7,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_ENHANCED_FIXED_WING_AIRCRAFT = 8,
     DIS_PDU_ISGROUPOF_GROUPED_ENTITY_CATEGORY_GROUND_LOGISTICS_VEHICLE = 9
}  DIS_PDU_IsGroupOf_Grouped_Entity_Category;

/* SISO-REF-010 [UID 214] */
typedef enum
{
     DIS_PDU_ISGROUPOF_REST_STATUS_NOT_RESTED_HAS_NOT_SLEPT_IN_THE_LAST_THREE_DAYS = 0,
     DIS_PDU_ISGROUPOF_REST_STATUS_HAS_SLEPT_AN_AVERAGE_OF_1_HOUR_PER_DAY_IN_THE_LAST_THREE_DAYS = 1,
     DIS_PDU_ISGROUPOF_REST_STATUS_HAS_SLEPT_AN_AVERAGE_OF_2_HOURS_PER_DAY_IN_THE_LAST_THREE_DAYS = 2,
     DIS_PDU_ISGROUPOF_REST_STATUS_HAS_SLEPT_AN_AVERAGE_OF_3_HOURS_PER_DAY_IN_THE_LAST_THREE_DAYS = 3,
     DIS_PDU_ISGROUPOF_REST_STATUS_HAS_SLEPT_AN_AVERAGE_OF_4_HOURS_PER_DAY_IN_THE_LAST_THREE_DAYS = 4,
     DIS_PDU_ISGROUPOF_REST_STATUS_HAS_SLEPT_AN_AVERAGE_OF_5_HOURS_PER_DAY_IN_THE_LAST_THREE_DAYS = 5,
     DIS_PDU_ISGROUPOF_REST_STATUS_HAS_SLEPT_AN_AVERAGE_OF_6_HOURS_PER_DAY_IN_THE_LAST_THREE_DAYS = 6,
     DIS_PDU_ISGROUPOF_REST_STATUS_HAS_SLEPT_AN_AVERAGE_OF_7_HOURS_PER_DAY_IN_THE_LAST_THREE_DAYS = 7,
     DIS_PDU_ISGROUPOF_REST_STATUS_FULLY_RESTED_HAS_SLEPT_AN_AVERAGE_OF_8_HOURS_PER_DAY_IN_THE_LAST_THREE_DAYS = 8
}  DIS_PDU_IsGroupOf_Rest_Status;

/* SISO-REF-010 [UID 224] */
typedef enum
{
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_OTHER = 0,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_PUSH_TRANSFER_ENTITY = 1,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_AUTOMATIC_PULL_TRANSFER_ENTITY = 2,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_NOT_USED_3 = 3,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_PUSH_TRANSFER_ENVIRONMENTAL_PROCESS = 4,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_AUTOMATIC_PULL_TRANSFER_ENVIRONMENTAL_PROCESS = 5,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_NOT_USED_6 = 6,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_CANCEL_TRANSFER = 7,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_MANUAL_PULL_TRANSFER_ENTITY = 8,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_MANUAL_PULL_TRANSFER_ENVIRONMENTAL_PROCESS = 9,
     DIS_PDU_TRANSFER_CONTROL_TRANSFER_TYPE_REMOVE_ENTITY = 10
}  DIS_PDU_Transfer_Control_Transfer_Type;

/* SISO-REF-010 [UID 225] */
typedef enum
{
     DIS_PDU_OBJECT_KIND_OTHER = 0,
     DIS_PDU_OBJECT_KIND_OBSTACLE = 1,
     DIS_PDU_OBJECT_KIND_PREPARED_POSITION = 2,
     DIS_PDU_OBJECT_KIND_CULTURAL_FEATURE = 3,
     DIS_PDU_OBJECT_KIND_PASSAGEWAY = 4,
     DIS_PDU_OBJECT_KIND_TACTICAL_SMOKE = 5,
     DIS_PDU_OBJECT_KIND_OBSTACLE_MARKER = 6,
     DIS_PDU_OBJECT_KIND_OBSTACLE_BREACH = 7,
     DIS_PDU_OBJECT_KIND_ENVIRONMENTAL_OBJECT = 8
}  DIS_PDU_Object_Kind;

/* SISO-REF-010 [UID 244] */
typedef enum
{
     DIS_PDU_GRIDDED_DATA_COORDINATE_SYSTEM_RIGHT_HANDED_CARTESIAN_LOCAL_TOPOGRAPHIC_PROJECTION_EAST_NORTH_UP = 0,
     DIS_PDU_GRIDDED_DATA_COORDINATE_SYSTEM_LEFT_HANDED_CARTESIAN_LOCAL_TOPOGRAPHIC_PROJECTION_EAST_NORTH_DOWN = 1,
     DIS_PDU_GRIDDED_DATA_COORDINATE_SYSTEM_LATITUDE_LONGITUDE_HEIGHT = 2,
     DIS_PDU_GRIDDED_DATA_COORDINATE_SYSTEM_LATITUDE_LONGITUDE_DEPTH = 3
}  DIS_PDU_Gridded_Data_Coordinate_System;

/* SISO-REF-010 [UID 245] */
typedef enum
{
     DIS_PDU_GRIDDED_DATA_CONSTANT_GRID_CONSTANT_GRID = 0,
     DIS_PDU_GRIDDED_DATA_CONSTANT_GRID_UPDATED_GRID = 1
}  DIS_PDU_Gridded_Data_Constant_Grid;

/* SISO-REF-010 [UID 246] */
typedef enum
{
     DIS_PDU_GRIDDED_DATA_SAMPLE_TYPE_NOT_SPECIFIED = 0
}  DIS_PDU_Gridded_Data_Sample_Type;

/* SISO-REF-010 [UID 247] */
typedef enum
{
     DIS_PDU_GRIDDED_DATA_DATA_REPRESENTATION_TYPE_0 = 0,
     DIS_PDU_GRIDDED_DATA_DATA_REPRESENTATION_TYPE_1 = 1,
     DIS_PDU_GRIDDED_DATA_DATA_REPRESENTATION_TYPE_2 = 2
}  DIS_PDU_Gridded_Data_Data_Representation;

/* SISO-REF-010 [UID 248] */
typedef enum
{
     DIS_PDU_ENVIRONMENTAL_PROCESS_MODEL_TYPE_NO_STATEMENT = 0
}  DIS_PDU_Environmental_Process_Model_Type;

/* SISO-REF-010 [UID 250] */
typedef enum
{
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_GENERAL_REPAIR_CODES_COMBIC_STATE = 256,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_GENERAL_REPAIR_CODES_FLARE_STATE = 259,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_GENERAL_REPAIR_CODES_BIOLOGICAL_STATE = 4096,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_GENERAL_REPAIR_CODES_CHEMICAL_VAPOR_STATE = 4097,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_GENERAL_REPAIR_CODES_RADIOLOGICAL_STATE = 4098,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_GENERAL_REPAIR_CODES_CHEMICAL_LIQUID_STATE = 4099,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_GENERAL_REPAIR_CODES_OBSCURANT_STATE = 4100,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_BOUNDING_SPHERE_RECORD = 65536,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_UNIFORM_GEOMETRY_RECORD = 327680,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_POINT_RECORD_1 = 655360,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_LINE_RECORD_1 = 786432,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_SPHERE_RECORD_1 = 851968,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_ELLIPSOID_RECORD_1 = 1048576,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_CONE_RECORD_1 = 3145728,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_RECTANGULAR_VOLUME_RECORD_1 = 5242880,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_RECTANGULAR_VOLUME_RECORD_3 = 83886080,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_POINT_RECORD_2 = 167772160,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_LINE_RECORD_2 = 201326592,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_SPHERE_RECORD_2 = 218103808,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_ELLIPSOID_RECORD_2 = 268435456,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_CONE_RECORD_2 = 805306368,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_RECTANGULAR_VOLUME_RECORD_2 = 1342177280,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_GAUSSIAN_PLUME_RECORD = 1610612736,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_GAUSSIAN_PUFF_RECORD_1 = 1879048192,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_GAUSSIAN_PUFF_RECORD_2 = 1879048193,
     DIS_PDU_ENVIRONMENTAL_PROCESS_RECORD_TYPE_DRIVE_TRAIN_RECTANGULAR_VOLUME_RECORD_4 = 2147483648
}  DIS_PDU_Environmental_Process_Record_Type;

/* SISO-REF-010 [UID 270] */
typedef enum
{
     DIS_ENCODING_CLASS_ENCODED_AUDIO = 0,
     DIS_ENCODING_CLASS_RAW_BINARY_DATA = 1,
     DIS_ENCODING_CLASS_APPLICATION_SPECIFIC_DATA = 2,
     DIS_ENCODING_CLASS_DATABASE_INDEX = 3
}  DIS_PDU_Encoding_Class;

static const value_string DIS_PDU_Encoding_Class_Strings[] = {
     { DIS_ENCODING_CLASS_ENCODED_AUDIO, "Encoded audio" },
     { DIS_ENCODING_CLASS_RAW_BINARY_DATA, "Raw Binary Data" },
     { DIS_ENCODING_CLASS_APPLICATION_SPECIFIC_DATA, "Application-Specific Data" },
     { DIS_ENCODING_CLASS_DATABASE_INDEX, "Database index" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 271] */
typedef enum
{
     DIS_PDU_SIGNAL_ENCODING_TYPE_8_BIT_MU_LAW_ITU_T_G_711 = 1,
     DIS_PDU_SIGNAL_ENCODING_TYPE_CVSD_MIL_STD_188_113 = 2,
     DIS_PDU_SIGNAL_ENCODING_TYPE_ADPCM_ITU_T_G_726 = 3,
     DIS_PDU_SIGNAL_ENCODING_TYPE_16_BIT_LINEAR_PCM_2S_COMPLEMENT_BIG_ENDIAN = 4,
     DIS_PDU_SIGNAL_ENCODING_TYPE_8_BIT_LINEAR_PCM_UNSIGNED = 5,
     DIS_PDU_SIGNAL_ENCODING_TYPE_UNAVAILABLE_FOR_USE_7 = 7,
     DIS_PDU_SIGNAL_ENCODING_TYPE_GSM_FULL_RATE_ETSI_06_10 = 8,
     DIS_PDU_SIGNAL_ENCODING_TYPE_GSM_HALF_RATE_ETSI_06_20 = 9,
     DIS_PDU_SIGNAL_ENCODING_TYPE_SPEEX_NARROW_BAND = 10,
     DIS_PDU_SIGNAL_ENCODING_TYPE_OPUS = 11,
     DIS_PDU_SIGNAL_ENCODING_TYPE_LPC_10_FIPS_PUB_137 = 12,
     DIS_PDU_SIGNAL_ENCODING_TYPE_16_BIT_LINEAR_PCM_2S_COMPLEMENT_LITTLE_ENDIAN = 100,
     DIS_PDU_SIGNAL_ENCODING_TYPE_UNAVAILABLE_FOR_USE_255 = 255
}  DIS_PDU_Signal_Encoding_Type;

static const value_string DIS_PDU_Signal_Encoding_Type_Strings[] = {
     { DIS_PDU_SIGNAL_ENCODING_TYPE_8_BIT_MU_LAW_ITU_T_G_711, "8-bit mu-law (ITU-T G.711)" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_CVSD_MIL_STD_188_113, "CVSD (MIL-STD-188-113)" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_ADPCM_ITU_T_G_726, "ADPCM (ITU-T G.726)" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_16_BIT_LINEAR_PCM_2S_COMPLEMENT_BIG_ENDIAN, "16-bit Linear PCM 2s Complement, Big Endian" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_8_BIT_LINEAR_PCM_UNSIGNED, "8-bit Linear PCM, Unsigned" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_UNAVAILABLE_FOR_USE_7, "(unavailable for use)" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_GSM_FULL_RATE_ETSI_06_10, "GSM Full-Rate (ETSI 06.10)" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_GSM_HALF_RATE_ETSI_06_20, "GSM Half-Rate (ETSI 06.20)" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_SPEEX_NARROW_BAND, "Speex Narrow Band" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_OPUS, "Opus" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_LPC_10_FIPS_PUB_137, "LPC-10 (FIPS PUB 137)" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_16_BIT_LINEAR_PCM_2S_COMPLEMENT_LITTLE_ENDIAN, "16-bit Linear PCM 2s Complement, Little Endian" },
     { DIS_PDU_SIGNAL_ENCODING_TYPE_UNAVAILABLE_FOR_USE_255, "(unavailable for use)" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 272] */
typedef enum
{
     DIS_PDU_REPAIR_GROUPS_GENERAL_REPAIR_CODES = 0,
     DIS_PDU_REPAIR_GROUPS_DRIVE_TRAIN = 1,
     DIS_PDU_REPAIR_GROUPS_HULL_AIRFRAME_BODY = 2,
     DIS_PDU_REPAIR_GROUPS_INTERFACES_WITH_ENVIRONMENT = 3,
     DIS_PDU_REPAIR_GROUPS_WEAPONS = 4,
     DIS_PDU_REPAIR_GROUPS_FUEL_SYSTEMS = 5,
     DIS_PDU_REPAIR_GROUPS_ELECTRONICS = 6,
     DIS_PDU_REPAIR_GROUPS_LIFE_SUPPORT_SYSTEMS = 7,
     DIS_PDU_REPAIR_GROUPS_HYDRAULIC_SYSTEMS_AND_ACTUATORS = 8,
     DIS_PDU_REPAIR_GROUPS_AUXILIARY_CRAFT = 9
}  DIS_PDU_Repair_Groups;

/* SISO-REF-010 [UID 273] */
typedef enum
{
     DIS_PDU_ENVIRONMENT_RECORD_TYPE_GROUPS_STATE = 0,
     DIS_PDU_ENVIRONMENT_RECORD_TYPE_GROUPS_GEOMETRY = 1
}  DIS_PDU_Environment_Record_Type_Groups;

/* SISO-REF-010 [UID 274] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CIVILIAN_ULTRALIGHT_NON_RIGID_WING_AIRCRAFT_SUBCATEGORIES_HANG_GLIDER_UNPOWERED = 1,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_ULTRALIGHT_NON_RIGID_WING_AIRCRAFT_SUBCATEGORIES_HANG_GLIDER_POWERED = 2,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_ULTRALIGHT_NON_RIGID_WING_AIRCRAFT_SUBCATEGORIES_PARAGLIDER_UNPOWERED = 3,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_ULTRALIGHT_NON_RIGID_WING_AIRCRAFT_SUBCATEGORIES_PARAGLIDER_POWERED = 4,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_ULTRALIGHT_NON_RIGID_WING_AIRCRAFT_SUBCATEGORIES_POWERED_PARACHUTE = 5
}  DIS_PDU_Platform_Air_Civilian_Ultralight_Non_rigid_Wing_Aircraft_Subcategories;

/* SISO-REF-010 [UID 275] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CIVILIAN_ULTRALIGHT_RIGID_WING_AIRCRAFT_SUBCATEGORIES_WEIGHT_SHIFT_CONTROL = 1,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_ULTRALIGHT_RIGID_WING_AIRCRAFT_SUBCATEGORIES_CONTROL_SURFACE_ELEVATOR_RUDDER_AILERON_CONTROL = 2
}  DIS_PDU_Platform_Air_Civilian_Ultralight_Rigid_Wing_Aircraft_Subcategories;

/* SISO-REF-010 [UID 276] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CIVILIAN_GLIDER_SUBCATEGORIES_SAIL_PLANE = 1,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_GLIDER_SUBCATEGORIES_MOTOR_GLIDER = 2
}  DIS_PDU_Platform_Air_Civilian_Glider_Subcategories;

/* SISO-REF-010 [UID 277] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_SINGLE_PISTON_ENGINE = 11,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_TWIN_PISTON_ENGINE = 12,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_SINGLE_ENGINE_TURBOPROP = 21,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_TWIN_ENGINE_TURBOPROP = 22,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_FOUR_ENGINE_TURBOPROP = 24,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_TWIN_JET = 32,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_TRI_JET = 33,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_FIXED_WING_AIRCRAFT_SUBCATEGORIES_FOUR_ENGINE_JET = 34
}  DIS_PDU_Platform_Air_Civilian_Fixed_Wing_Aircraft_Subcategories;

/* SISO-REF-010 [UID 278] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CIVILIAN_HELICOPTER_SUBCATEGORIES_SINGLE_ROTOR_PISTON_ENGINE = 11,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_HELICOPTER_SUBCATEGORIES_SINGLE_ROTOR_TURBOSHAFT_ENGINE_CONVENTIONAL_TAIL_ROTOR = 12,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_HELICOPTER_SUBCATEGORIES_SINGLE_ROTOR_TURBOSHAFT_ENGINE_SHROUDED_TAIL_ROTOR = 13,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_HELICOPTER_SUBCATEGORIES_SINGLE_ROTOR_TURBOSHAFT_ENGINE_NO_TAIL_ROTOR = 14,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_HELICOPTER_SUBCATEGORIES_TANDEM_ROTOR = 21,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_HELICOPTER_SUBCATEGORIES_COAXIAL_ROTOR = 22,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_HELICOPTER_SUBCATEGORIES_INTERMESHING_ROTOR = 23
}  DIS_PDU_Platform_Air_Civilian_Helicopter_Subcategories;

/* SISO-REF-010 [UID 279] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_BALLOON_SUBCATEGORIES_GAS_FILLED_FREE = 1,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_BALLOON_SUBCATEGORIES_GAS_FILLED_TETHERED = 2,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_BALLOON_SUBCATEGORIES_HOT_AIR = 3,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_BALLOON_SUBCATEGORIES_ROZIERE_BALLOON = 4,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_BALLOON_SUBCATEGORIES_HELIKITE = 5
}  DIS_PDU_Platform_Air_Civilian_Lighter_than_Air_Balloon_Subcategories;

/* SISO-REF-010 [UID 280] */
typedef enum
{
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_AIRSHIP_SUBCATEGORIES_NON_RIGID_BLIMP = 1,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_AIRSHIP_SUBCATEGORIES_SEMI_RIGID = 2,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_AIRSHIP_SUBCATEGORIES_RIGID = 3,
     DIS_PDU_PLATFORM_AIR_CIVILIAN_LIGHTER_THAN_AIR_AIRSHIP_SUBCATEGORIES_HYBRID = 4
}  DIS_PDU_Platform_Air_Civilian_Lighter_than_Air_Airship_Subcategories;

/* SISO-REF-010 [UID 281] */
typedef enum
{
     DIS_PDU_APA_PARAMETER_INDEX_APA_STATUS_DESELECTED_OFF = 0,
     DIS_PDU_APA_PARAMETER_INDEX_APA_STATUS_APA_VALUE_CHANGE_ONLY = 1,
     DIS_PDU_APA_PARAMETER_INDEX_APA_STATUS_STATE_CHANGE = 2,
     DIS_PDU_APA_PARAMETER_INDEX_APA_STATUS_RECORD_ACTIVATION = 3
}  DIS_PDU_APA_Parameter_Index_APA_Status;

/* SISO-REF-010 [UID 282] */
typedef enum
{
     DIS_PDU_SEPARATION_VP_REASON_FOR_SEPARATION_NO_STATEMENT = 0,
     DIS_PDU_SEPARATION_VP_REASON_FOR_SEPARATION_ATTACHED_PART_SEPARATION = 1,
     DIS_PDU_SEPARATION_VP_REASON_FOR_SEPARATION_SUBMUNITION_SEPARATION = 2
}  DIS_PDU_Separation_VP_Reason_for_Separation;

/* SISO-REF-010 [UID 283] */
typedef enum
{
     DIS_PDU_SEPARATION_VP_PRE_ENTITY_INDICATOR_NO_STATEMENT = 0,
     DIS_PDU_SEPARATION_VP_PRE_ENTITY_INDICATOR_ENTITY_ID_EXISTED_PRIOR_TO_SEPARATION_WITHOUT_ENTITY_STATE_PDU = 1,
     DIS_PDU_SEPARATION_VP_PRE_ENTITY_INDICATOR_ENTITY_ID_EXISTED_PRIOR_TO_SEPARATION_WITH_ENTITY_STATE_PDU_ISSUED = 2,
     DIS_PDU_SEPARATION_VP_PRE_ENTITY_INDICATOR_ENTITY_INITIALLY_CREATED_AT_SEPARATION_EVENT = 3
}  DIS_PDU_Separation_VP_Pre_Entity_Indicator;

/* SISO-REF-010 [UID 285] */
typedef enum
{
     DIS_PDU_IO_ACTION_IO_WARFARE_TYPE_NO_STATEMENT = 0,
     DIS_PDU_IO_ACTION_IO_WARFARE_TYPE_ELECTRONIC_WARFARE_EW = 1,
     DIS_PDU_IO_ACTION_IO_WARFARE_TYPE_COMPUTER_NETWORK_OPERATIONS_CNO = 2,
     DIS_PDU_IO_ACTION_IO_WARFARE_TYPE_PSYCHOLOGICAL_OPERATIONS_PSYOPS = 3,
     DIS_PDU_IO_ACTION_IO_WARFARE_TYPE_MILITARY_DECEPTION_MILDEC = 4,
     DIS_PDU_IO_ACTION_IO_WARFARE_TYPE_OPERATIONS_SECURITY_OPSEC = 5,
     DIS_PDU_IO_ACTION_IO_WARFARE_TYPE_PHYSICAL_ATTACK = 6
}  DIS_PDU_IO_Action_IO_Warfare_Type;

/* SISO-REF-010 [UID 286] */
typedef enum
{
     DIS_PDU_IO_ACTION_IO_SIMULATION_SOURCE_NO_STATEMENT = 0
}  DIS_PDU_IO_Action_IO_Simulation_Source;

/* SISO-REF-010 [UID 287] */
typedef enum
{
     DIS_PDU_IO_ACTION_IO_ACTION_TYPE_NO_STATEMENT = 0,
     DIS_PDU_IO_ACTION_IO_ACTION_TYPE_IO_ATTACK_PROFILE_DATA_PARAMETRICS = 1,
     DIS_PDU_IO_ACTION_IO_ACTION_TYPE_IO_ATTACK_COMPUTED_EFFECTS = 2,
     DIS_PDU_IO_ACTION_IO_ACTION_TYPE_INTENT_BASED_EW = 3,
     DIS_PDU_IO_ACTION_IO_ACTION_TYPE_INTENT_BASED_EW_COMPUTED_EFFECTS = 4
}  DIS_PDU_IO_Action_IO_Action_Type;

/* SISO-REF-010 [UID 288] */
typedef enum
{
     DIS_PDU_IO_ACTION_IO_ACTION_PHASE_NO_STATEMENT = 0,
     DIS_PDU_IO_ACTION_IO_ACTION_PHASE_START_ATTACK_PROFILE = 1,
     DIS_PDU_IO_ACTION_IO_ACTION_PHASE_END_ATTACK_PROFILE = 2,
     DIS_PDU_IO_ACTION_IO_ACTION_PHASE_CONTINUE_ATTACK_PROFILE_WITH_CHANGES = 3,
     DIS_PDU_IO_ACTION_IO_ACTION_PHASE_START_ATTACK_EFFECTS = 4,
     DIS_PDU_IO_ACTION_IO_ACTION_PHASE_END_ATTACKED_EFFECTS = 5,
     DIS_PDU_IO_ACTION_IO_ACTION_PHASE_CONTINUE_ATTACK_EFFECTS_WITH_CHANGES = 6
}  DIS_PDU_IO_Action_IO_Action_Phase;

/* SISO-REF-010 [UID 289] */
typedef enum
{
     DIS_PDU_IO_REPORT_IO_REPORT_TYPE_NO_STATEMENT = 0,
     DIS_PDU_IO_REPORT_IO_REPORT_TYPE_INITIAL_REPORT = 1,
     DIS_PDU_IO_REPORT_IO_REPORT_TYPE_UPDATE_REPORT = 2,
     DIS_PDU_IO_REPORT_IO_REPORT_TYPE_FINAL_REPORT = 3
}  DIS_PDU_IO_Report_IO_Report_Type;

/* SISO-REF-010 [UID 290] */
typedef enum
{
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_NO_STATEMENT = 0,
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_EFFECT_ON_SENDER = 1,
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_EFFECT_ON_RECEIVER = 2,
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_EFFECT_ON_SENDER_AND_RECEIVER = 3,
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_EFFECT_ON_MESSAGE = 4,
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_EFFECT_ON_SENDER_AND_MESSAGE = 5,
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_EFFECT_ON_RECEIVER_AND_MESSAGE = 6,
     DIS_PDU_IO_EFFECTS_RECORD_IO_STATUS_EFFECT_ON_SENDER_RECEIVER_AND_MESSAGE = 7
}  DIS_PDU_IO_Effects_Record_IO_Status;

/* SISO-REF-010 [UID 291] */
typedef enum
{
     DIS_PDU_IO_EFFECTS_RECORD_IO_LINK_TYPE_NO_STATEMENT = 0,
     DIS_PDU_IO_EFFECTS_RECORD_IO_LINK_TYPE_LOGICAL_LINK = 1,
     DIS_PDU_IO_EFFECTS_RECORD_IO_LINK_TYPE_PHYSICAL_NODE = 2,
     DIS_PDU_IO_EFFECTS_RECORD_IO_LINK_TYPE_PHYSICAL_LINK = 3
}  DIS_PDU_IO_Effects_Record_IO_Link_Type;

/* SISO-REF-010 [UID 292] */
typedef enum
{
     DIS_PDU_IO_EFFECTS_RECORD_IO_EFFECT_NO_STATEMENT = 0,
     DIS_PDU_IO_EFFECTS_RECORD_IO_EFFECT_DENIAL = 1,
     DIS_PDU_IO_EFFECTS_RECORD_IO_EFFECT_DEGRADATION = 2,
     DIS_PDU_IO_EFFECTS_RECORD_IO_EFFECT_DISRUPTION = 3,
     DIS_PDU_IO_EFFECTS_RECORD_IO_EFFECT_TERMINATE_EFFECT = 255
}  DIS_PDU_IO_Effects_Record_IO_Effect;

/* SISO-REF-010 [UID 293] */
typedef enum
{
     DIS_PDU_IO_EFFECTS_RECORD_IO_PROCESS_NO_STATEMENT = 0
}  DIS_PDU_IO_Effects_Record_IO_Process;

/* SISO-REF-010 [UID 294] */
typedef enum
{
     DIS_PDU_IO_COMMS_NODE_RECORD_COMMS_NODE_TYPE_NO_STATEMENT = 0,
     DIS_PDU_IO_COMMS_NODE_RECORD_COMMS_NODE_TYPE_SENDER_NODE_ID = 1,
     DIS_PDU_IO_COMMS_NODE_RECORD_COMMS_NODE_TYPE_RECEIVER_NODE_ID = 2,
     DIS_PDU_IO_COMMS_NODE_RECORD_COMMS_NODE_TYPE_SENDER_RECEIVER_NODE_ID = 3
}  DIS_PDU_IO_Comms_Node_Record_Comms_Node_Type;

/* SISO-REF-010 [UID 295] */
typedef enum
{
     DIS_PDU_DIS_ATTRIBUTE_ACTION_CODE_NO_STATEMENT = 0
}  DIS_PDU_DIS_Attribute_Action_Code;

/* SISO-REF-010 [UID 296] */
typedef enum
{
     DIS_PDU_DR_PARAMETERS_TYPE_NONE = 0,
     DIS_PDU_DR_PARAMETERS_TYPE_LOCAL_EULER_ANGLES_YAW_PITCH_ROLL = 1,
     DIS_PDU_DR_PARAMETERS_TYPE_WORLD_ORIENTATION_QUATERNION = 2
}  DIS_PDU_DR_Parameters_Type;

/* SISO-REF-010 [UID 297] */
typedef enum
{
     DIS_PDU_HIGH_FIDELITY_HAVE_QUICK_TOD_TRANSMIT_INDICATOR_NO_TOD_IS_BEING_TRANSMITTED = 0,
     DIS_PDU_HIGH_FIDELITY_HAVE_QUICK_TOD_TRANSMIT_INDICATOR_TOD_TRANSMISSION_IN_PROGRESS = 1
}  DIS_PDU_High_Fidelity_HAVE_QUICK_TOD_Transmit_Indicator;

/* SISO-REF-010 [UID 298] */
typedef enum
{
     DIS_PDU_NET_ID_RECORD_MODE_A_HAVE_QUICK_I_OR_HAVE_QUICK_II_COMBAT = 1,
     DIS_PDU_NET_ID_RECORD_MODE_B_SATURN_COMBAT = 2,
     DIS_PDU_NET_ID_RECORD_MODE_T_TRAINING = 3
}  DIS_PDU_NET_ID_Record_Mode;

/* SISO-REF-010 [UID 299] */
typedef enum
{
     DIS_PDU_NET_ID_RECORD_FREQUENCY_TABLE_HQ_I_OPERATIONS = 0,
     DIS_PDU_NET_ID_RECORD_FREQUENCY_TABLE_HQII_NATO_EUROPE_AREA_OPERATIONS = 1,
     DIS_PDU_NET_ID_RECORD_FREQUENCY_TABLE_HQII_NON_NATO_EUROPE_AREA_OPERATIONS = 2,
     DIS_PDU_NET_ID_RECORD_FREQUENCY_TABLE_SATURN_OPERATIONS = 3
}  DIS_PDU_NET_ID_Record_Frequency_Table;

/* SISO-REF-010 [UID 300] */
typedef enum
{
     DIS_PDU_EE_ATTRIBUTE_STATE_INDICATOR_HEARTBEAT_UPDATE = 0,
     DIS_PDU_EE_ATTRIBUTE_STATE_INDICATOR_CHANGED_DATA = 1,
     DIS_PDU_EE_ATTRIBUTE_STATE_INDICATOR_HAS_CEASED = 2
}  DIS_PDU_EE_Attribute_State_Indicator;

/* SISO-REF-010 [UID 301] */
typedef enum
{
     DIS_PDU_DIS_PDU_STATUS_TRANSFERRED_ENTITY_INDICATOR_TEI_NO_DIFFERENCE = 0,
     DIS_PDU_DIS_PDU_STATUS_TRANSFERRED_ENTITY_INDICATOR_TEI_DIFFERENCE = 1
}  DIS_PDU_DIS_PDU_Status_Transferred_Entity_Indicator_TEI;

/* SISO-REF-010 [UID 302] */
typedef enum
{
     DIS_PDU_LVC_INDICATOR_NO_STATEMENT = 0,
     DIS_PDU_LVC_INDICATOR_LIVE = 1,
     DIS_PDU_LVC_INDICATOR_VIRTUAL = 2,
     DIS_PDU_LVC_INDICATOR_CONSTRUCTIVE = 3
}  DIS_PDU_LVC_Indicator;

static const value_string DIS_PDU_LVC_Indicator_Strings[] = {
     { DIS_PDU_LVC_INDICATOR_NO_STATEMENT, "No Statement" },
     { DIS_PDU_LVC_INDICATOR_LIVE, "Live" },
     { DIS_PDU_LVC_INDICATOR_VIRTUAL, "Virtual" },
     { DIS_PDU_LVC_INDICATOR_CONSTRUCTIVE, "Constructive" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 303] */
typedef enum
{
     DIS_PDU_DIS_PDU_STATUS_COUPLED_EXTENSION_INDICATOR_CEI_NOT_COUPLED = 0,
     DIS_PDU_DIS_PDU_STATUS_COUPLED_EXTENSION_INDICATOR_CEI_COUPLED = 1
}  DIS_PDU_DIS_PDU_Status_Coupled_Extension_Indicator_CEI;

static const value_string DIS_PDU_DIS_PDU_Status_Coupled_Extension_Indicator_CEI_Strings[] = {
     { DIS_PDU_DIS_PDU_STATUS_COUPLED_EXTENSION_INDICATOR_CEI_NOT_COUPLED, "Not Coupled" },
     { DIS_PDU_DIS_PDU_STATUS_COUPLED_EXTENSION_INDICATOR_CEI_COUPLED, "Coupled" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 304] */
typedef enum
{
     DIS_PDU_DIS_PDU_STATUS_FIRE_TYPE_INDICATOR_FTI_MUNITION = 0,
     DIS_PDU_DIS_PDU_STATUS_FIRE_TYPE_INDICATOR_FTI_EXPENDABLE = 1
}  DIS_PDU_DIS_PDU_Status_Fire_Type_Indicator_FTI;

static const value_string DIS_PDU_DIS_PDU_Status_Fire_Type_Indicator_FTI_Strings[] = {
     { DIS_PDU_DIS_PDU_STATUS_FIRE_TYPE_INDICATOR_FTI_MUNITION, "Munition" },
     { DIS_PDU_DIS_PDU_STATUS_FIRE_TYPE_INDICATOR_FTI_EXPENDABLE, "Expendable" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 305] */
typedef enum
{
     DIS_PDU_DIS_PDU_STATUS_DETONATION_TYPE_INDICATOR_DTI_MUNITION = 0,
     DIS_PDU_DIS_PDU_STATUS_DETONATION_TYPE_INDICATOR_DTI_EXPENDABLE = 1,
     DIS_PDU_DIS_PDU_STATUS_DETONATION_TYPE_INDICATOR_DTI_NON_MUNITION_EXPLOSION = 2
}  DIS_PDU_DIS_PDU_Status_Detonation_Type_Indicator_DTI;

static const value_string DIS_PDU_DIS_PDU_Status_Detonation_Type_Indicator_DTI_Strings[] = {
     { DIS_PDU_DIS_PDU_STATUS_DETONATION_TYPE_INDICATOR_DTI_MUNITION, "Munition" },
     { DIS_PDU_DIS_PDU_STATUS_DETONATION_TYPE_INDICATOR_DTI_EXPENDABLE, "Expendable" },
     { DIS_PDU_DIS_PDU_STATUS_DETONATION_TYPE_INDICATOR_DTI_NON_MUNITION_EXPLOSION, "Non-munition Explosion" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 306] */
typedef enum
{
     DIS_PDU_RADIO_ATTACHED_INDICATOR_NO_STATEMENT = 0,
     DIS_PDU_RADIO_ATTACHED_INDICATOR_UNATTACHED = 1,
     DIS_PDU_RADIO_ATTACHED_INDICATOR_ATTACHED = 2
}  DIS_PDU_Radio_Attached_Indicator;

static const value_string DIS_PDU_Radio_Attached_Indicator_Strings[] = {
     { DIS_PDU_RADIO_ATTACHED_INDICATOR_NO_STATEMENT, "No Statement" },
     { DIS_PDU_RADIO_ATTACHED_INDICATOR_UNATTACHED, "Unattached" },
     { DIS_PDU_RADIO_ATTACHED_INDICATOR_ATTACHED, "Attached" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 307] */
typedef enum
{
     DIS_PDU_DIS_PDU_STATUS_INTERCOM_ATTACHED_INDICATOR_IAI_NO_STATEMENT = 0,
     DIS_PDU_DIS_PDU_STATUS_INTERCOM_ATTACHED_INDICATOR_IAI_UNATTACHED = 1,
     DIS_PDU_DIS_PDU_STATUS_INTERCOM_ATTACHED_INDICATOR_IAI_ATTACHED = 2
}  DIS_PDU_DIS_PDU_Status_Intercom_Attached_Indicator_IAI;

static const value_string DIS_PDU_DIS_PDU_Status_Intercom_Attached_Indicator_IAI_Strings[] = {
     { DIS_PDU_DIS_PDU_STATUS_INTERCOM_ATTACHED_INDICATOR_IAI_NO_STATEMENT, "No Statement" },
     { DIS_PDU_DIS_PDU_STATUS_INTERCOM_ATTACHED_INDICATOR_IAI_UNATTACHED, "Unattached" },
     { DIS_PDU_DIS_PDU_STATUS_INTERCOM_ATTACHED_INDICATOR_IAI_ATTACHED, "Attached" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 308] */
typedef enum
{
     DIS_PDU_DIS_PDU_STATUS_IFF_SIMULATION_MODE_ISM_REGENERATION = 0,
     DIS_PDU_DIS_PDU_STATUS_IFF_SIMULATION_MODE_ISM_INTERACTIVE = 1
}  DIS_PDU_DIS_PDU_Status_IFF_Simulation_Mode_ISM;

static const value_string DIS_PDU_DIS_PDU_Status_IFF_Simulation_Mode_ISM_Strings[] = {
     { DIS_PDU_DIS_PDU_STATUS_IFF_SIMULATION_MODE_ISM_REGENERATION, "Regeneration" },
     { DIS_PDU_DIS_PDU_STATUS_IFF_SIMULATION_MODE_ISM_INTERACTIVE, "Interactive" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 309] */
typedef enum
{
     DIS_PDU_EXPLOSIVE_MATERIAL_GROUPS_GENERAL = 0,
     DIS_PDU_EXPLOSIVE_MATERIAL_GROUPS_LIQUID_AVIATION_MISSILE_FUELS = 1,
     DIS_PDU_EXPLOSIVE_MATERIAL_GROUPS_LIQUID_OTHER_FUELS = 2,
     DIS_PDU_EXPLOSIVE_MATERIAL_GROUPS_LIQUID_EXPLOSIVE_MATERIAL = 3,
     DIS_PDU_EXPLOSIVE_MATERIAL_GROUPS_SOLID = 4,
     DIS_PDU_EXPLOSIVE_MATERIAL_GROUPS_GASEOUS = 5,
     DIS_PDU_EXPLOSIVE_MATERIAL_GROUPS_DUST_MATERIAL = 6
}  DIS_PDU_Explosive_Material_Groups;

/* SISO-REF-010 [UID 310] */
typedef enum
{
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_GENERAL_REPAIR_CODES_NO_STATEMENT = 0,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_AVGAS_AVIATION_GAS = 10,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JET_FUEL_UNSPECIFIED = 11,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JP_4_F_40_JET_B = 12,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JP_5_F_44_JET_A = 13,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JP_7 = 14,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JP_8_F_34_JET_A_1 = 15,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JP_10_MISSILE_FUEL = 16,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JPTS = 17,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JET_A = 18,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JET_A_1 = 19,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JET_B = 20,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_DRIVE_TRAIN_JET_BIOFUEL = 21,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_GASOLINE_PETROL_UNSPECIFIED_OCTANE = 151,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_DIESEL_FUEL_UNSPECIFIED_GRADE = 152,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_ETHANOL = 153,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_E85_ETHANOL = 154,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_FUEL_OIL = 155,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_KEROSENE = 156,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_CRUDE_OIL_UNSPECIFIED = 157,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_LIGHT_CRUDE_OIL = 158,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_LIQUID_PETROLEUM_GAS_LPG = 159,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_RP_1_ROCKET_FUEL = 160,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_LH_2_ROCKET_FUEL = 161,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_LOX_ROCKET_FUEL = 162,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_ALCOHOL = 164,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_HULL_AIRFRAME_BODY_HYDROGEN_LIQUID = 166,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_INTERFACES_WITH_ENVIRONMENT_NITROGLYCERIN_NG = 301,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_INTERFACES_WITH_ENVIRONMENT_ANFO = 302,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_WEAPONS_DYNAMITE = 451,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_WEAPONS_TNT = 452,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_WEAPONS_RDX = 453,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_WEAPONS_PETN = 454,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_WEAPONS_HMX = 455,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_WEAPONS_C_4 = 456,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_WEAPONS_COMPOSITION_C_4 = 457,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_FUEL_SYSTEMS_NATURAL_GAS_NG = 601,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_FUEL_SYSTEMS_BUTANE = 602,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_FUEL_SYSTEMS_PROPANE = 603,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_FUEL_SYSTEMS_HELIUM = 604,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_FUEL_SYSTEMS_HYDROGEN_GASEOUS = 605,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_ELECTRONICS_DUST_UNSPECIFIED_TYPE = 801,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_ELECTRONICS_GRAIN_DUST = 802,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_ELECTRONICS_FLOUR_DUST = 803,
     DIS_PDU_EXPLOSIVE_MATERIAL_CATEGORIES_ELECTRONICS_SUGAR_DUST = 804
}  DIS_PDU_Explosive_Material_Categories;

/* SISO-REF-010 [UID 311] */
typedef enum
{
     DIS_PDU_DE_PRECISION_AIMPOINT_BEAM_SPOT_TYPE_OTHER = 0,
     DIS_PDU_DE_PRECISION_AIMPOINT_BEAM_SPOT_TYPE_GAUSSIAN = 1,
     DIS_PDU_DE_PRECISION_AIMPOINT_BEAM_SPOT_TYPE_TOP_HAT = 2
}  DIS_PDU_DE_Precision_Aimpoint_Beam_Spot_Type;

/* SISO-REF-010 [UID 312] */
typedef enum
{
     DIS_PDU_DE_FIRE_PULSE_SHAPE_OTHER = 0,
     DIS_PDU_DE_FIRE_PULSE_SHAPE_SQUARE_WAVE = 1,
     DIS_PDU_DE_FIRE_PULSE_SHAPE_CONTINUOUS_WAVE = 2,
     DIS_PDU_DE_FIRE_PULSE_SHAPE_GAUSSIAN = 3
}  DIS_PDU_DE_Fire_Pulse_Shape;

/* SISO-REF-010 [UID 314] */
typedef enum
{
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_ENTITY_CENTER_NO_SPECIFIC_COMPONENT = 0,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_ENTITY_STRUCTURE = 1,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_CONTROL_SYSTEM = 2,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_CONTROL_SURFACE = 3,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_ENGINE_PROPULSION_SYSTEM = 4,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_CREW_MEMBER = 5,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_FUSE = 6,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_ACQUISITION_SENSOR = 7,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_TRACKING_SENSOR = 8,
     DIS_PDU_ENTITY_DAMAGE_STATUS_COMPONENT_IDENTIFICATION_FUEL_TANK_SOLID_ROCKET_MOTOR = 9
}  DIS_PDU_Entity_Damage_Status_Component_Identification;

/* SISO-REF-010 [UID 315] */
typedef enum
{
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_DAMAGE_STATUS_NO_DAMAGE = 0,
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_DAMAGE_STATUS_MINOR_DAMAGE = 1,
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_DAMAGE_STATUS_MEDIUM_DAMAGE = 2,
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_DAMAGE_STATUS_MAJOR_DAMAGE = 3,
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_DAMAGE_STATUS_DESTROYED = 4
}  DIS_PDU_DE_Damage_Description_Component_Damage_Status;

/* SISO-REF-010 [UID 316] */
typedef enum
{
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_VISUAL_SMOKE_COLOR_NO_SMOKE = 0,
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_VISUAL_SMOKE_COLOR_WHITE = 1,
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_VISUAL_SMOKE_COLOR_GRAY = 2,
     DIS_PDU_DE_DAMAGE_DESCRIPTION_COMPONENT_VISUAL_SMOKE_COLOR_BLACK = 3
}  DIS_PDU_DE_Damage_Description_Component_Visual_Smoke_Color;

/* SISO-REF-010 [UID 318] */
typedef enum
{
     DIS_PDU_BEAM_STATUS_BEAM_STATE_ACTIVE = 0,
     DIS_PDU_BEAM_STATUS_BEAM_STATE_DEACTIVATED = 1
}  DIS_PDU_Beam_Status_Beam_State;

/* SISO-REF-010 [UID 319] */
typedef enum
{
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_STATUS_NOT_SPECIFIED = 0,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_STATUS_PHYSICAL_ASSOCIATION_GENERAL_OBJECT_1 = 1,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_STATUS_FUNCTIONAL_ASSOCIATION_GENERAL = 2,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_STATUS_ASSOCIATION_BROKEN = 3,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_STATUS_PHYSICAL_ASSOCIATION_OBJECT_2 = 4,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_STATUS_FUNCTIONAL_ASSOCIATION_OBJECT_1 = 5,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_STATUS_FUNCTIONAL_ASSOCIATION_OBJECT_2 = 6
}  DIS_PDU_Entity_Association_Association_Status;

/* SISO-REF-010 [UID 320] */
typedef enum
{
     DIS_PDU_ENTITY_VP_RECORD_CHANGE_INDICATOR_INITIAL_REPORT_OR_NO_CHANGE_SINCE_LAST_ISSUANCE = 0,
     DIS_PDU_ENTITY_VP_RECORD_CHANGE_INDICATOR_CHANGE_SINCE_LAST_ISSUANCE = 1
}  DIS_PDU_Entity_VP_Record_Change_Indicator;

static const value_string DIS_PDU_Entity_VP_Record_Change_Indicator_Strings[] = {
     { DIS_PDU_ENTITY_VP_RECORD_CHANGE_INDICATOR_INITIAL_REPORT_OR_NO_CHANGE_SINCE_LAST_ISSUANCE, "Initial Report or No Change Since Last Issuance" },
     { DIS_PDU_ENTITY_VP_RECORD_CHANGE_INDICATOR_CHANGE_SINCE_LAST_ISSUANCE, "Change Since Last Issuance" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 321] */
typedef enum
{
     DIS_PDU_ENTITY_ASSOCIATION_GROUP_MEMBER_TYPE_NOT_PART_OF_A_GROUP = 0,
     DIS_PDU_ENTITY_ASSOCIATION_GROUP_MEMBER_TYPE_GROUP_LEADER = 1,
     DIS_PDU_ENTITY_ASSOCIATION_GROUP_MEMBER_TYPE_GROUP_MEMBER = 2,
     DIS_PDU_ENTITY_ASSOCIATION_GROUP_MEMBER_TYPE_FORMATION_LEADER = 3,
     DIS_PDU_ENTITY_ASSOCIATION_GROUP_MEMBER_TYPE_FORMATION_MEMBER = 4,
     DIS_PDU_ENTITY_ASSOCIATION_GROUP_MEMBER_TYPE_CONVOY_LEADER = 5,
     DIS_PDU_ENTITY_ASSOCIATION_GROUP_MEMBER_TYPE_CONVOY_MEMBER = 6
}  DIS_PDU_Entity_Association_Group_Member_Type;

/* SISO-REF-010 [UID 322] */
typedef enum
{
     DIS_PDU_PHYSICAL_ASSOCIATION_TYPE_GROUPS_NOT_SPECIFIED = 0,
     DIS_PDU_PHYSICAL_ASSOCIATION_TYPE_GROUPS_TOWED_MOUNTED_SLING_LOAD = 1,
     DIS_PDU_PHYSICAL_ASSOCIATION_TYPE_GROUPS_RESTRAINED = 2,
     DIS_PDU_PHYSICAL_ASSOCIATION_TYPE_GROUPS_MISSION = 3,
     DIS_PDU_PHYSICAL_ASSOCIATION_TYPE_GROUPS_OTHER_CONNECTIONS = 4
}  DIS_PDU_Physical_Association_Type_Groups;

/* SISO-REF-010 [UID 323] */
typedef enum
{
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_GENERAL_REPAIR_CODES_NOT_SPECIFIED = 0,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_SINGLE_HOOK_NOT_SPECIFIED = 1,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_ON_LAND = 2,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_ON_WATER_SURFACE = 3,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_UNDERWATER = 4,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_MOUNTED_ATTACHED = 5,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_MOUNTED_UNATTACHED_AND_UNSUPPORTED = 6,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_MOUNTED_UNATTACHED_AND_SUPPORTED = 7,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_CENTER_HOOK = 8,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_FORWARD_HOOK = 9,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_AFT_HOOK = 10,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_TANDEM_HOOK_FORE_AND_AFT = 11,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_MISMANAGED_TANDEM_FORE_AND_CENTER = 12,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_MISMANAGED_TANDEM_CENTER_AND_AFT = 13,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_TOWED_IN_AIR_ALL_HOOKS = 14,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_DRIVE_TRAIN_HOISTED = 15,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_HULL_AIRFRAME_BODY_RESTRAINED_TO_A_LIFEFORM = 30,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_HULL_AIRFRAME_BODY_RESTRAINED_TO_A_PLATFORM = 31,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_HULL_AIRFRAME_BODY_RESTRAINED_TO_AN_OBJECT = 32,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_INTERFACES_WITH_ENVIRONMENT_REFUELING_OPERATION = 61,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_INTERFACES_WITH_ENVIRONMENT_SEARCH_AND_RESCUE_BASKET = 62,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_INTERFACES_WITH_ENVIRONMENT_SEARCH_AND_RESCUE_RESCUE_COLLAR = 63,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_INTERFACES_WITH_ENVIRONMENT_ENGAGEMENT_OBJECT_2_IS_BEING_ENGAGED = 64,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_INTERFACES_WITH_ENVIRONMENT_RETURN_TO_BASE_OBJECT_2_IS_THE_DESTINATION_OBJECT = 65,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_WEAPONS_LINE_BETWEEN_COMMUNICATION_TOWERS = 90,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_WEAPONS_LINE_BETWEEN_POWER_TOWERS = 91,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_WEAPONS_INDOORS = 92,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_ASSOCIATION_TYPE_WEAPONS_TOP_SURFACE = 93
}  DIS_PDU_Entity_Association_Physical_Association_Type;

/* SISO-REF-010 [UID 324] */
typedef enum
{
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_NOT_SPECIFIED = 0,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_ATTACHED_DIRECTLY_TO_SURFACE = 1,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_CABLE_WIRE = 2,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_ROPE = 3,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_CHAIN = 4,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_POWER_LINE = 5,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_TELEPHONE_LINE = 6,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_CABLE_LINE = 7,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_REFUELING_DROGUE = 8,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_REFUELING_BOOM = 9,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_HANDCUFFS = 10,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_IN_CONTACT_WITH = 11,
     DIS_PDU_ENTITY_ASSOCIATION_PHYSICAL_CONNECTION_TYPE_FAST_ROPE = 12
}  DIS_PDU_Entity_Association_Physical_Connection_Type;

/* SISO-REF-010 [UID 325] */
typedef enum
{
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_OTHER_ACTIVE_SENSORS_UNDEFINED = 0
}  DIS_PDU_Sensor_Record_Sensor_Type_Other_Active_Sensors;

/* SISO-REF-010 [UID 326] */
typedef enum
{
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_ALR_400 = 60000,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AAR_47 = 60001,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AAR_50 = 60002,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AAR_54 = 60003,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AAR_56 = 60004,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AAR_57 = 60005,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALQ_142 = 60006,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_45 = 60007,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_46 = 60008,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_56 = 60009,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_59 = 60010,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_64 = 60011,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_66 = 60012,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_67 = 60013,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_69 = 60014,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_73 = 60015,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_76 = 60016,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_91 = 60017,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_93 = 60018,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_94 = 60019,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_801 = 60020,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_APR_39 = 60021,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AYR_2 = 60022,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_ARI_18223 = 60023,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_BOW_21 = 60024,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_CHAPARRAL_IRST = 60025,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_FLANKER_IRST = 60026,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_FOXBAT_IRST = 60027,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_FOXHOUND_IRST = 60028,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_FULCRUM_IRST = 60029,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_HAVOC_IRST = 60030,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_HIND_IRST = 60031,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_KJ_200 = 60032,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_KJ_8602 = 60033,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_L_150_PASTEL = 60034,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SERVAL = 60035,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SHERLOC = 60036,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SHERLOCVF = 60037,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SIRENA_2 = 60038,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SIRENA_3 = 60039,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SIRENA_3M = 60040,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SKY_GUARDIAN = 60041,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SPO_15 = 60042,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SPS_200 = 60043,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_TARANG = 60044,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AAQ_29A = 60045,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_101KS_U_MAW = 60046,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_ABRAMS_2GF_FLIR = 60047,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_ABRAMS_3GF_FLIR = 60048,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_AAQ_13_LANTIRN_FLIR = 60049,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_74 = 60050,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_ALR_90 = 60051,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_AN_APR_48 = 60052,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_ELT_156X_V = 60053,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_101KS_V_FELON_IRST = 60054,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_TP_23ML_FLOGGER_IRST = 60055,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_GENERIC_FLIR = 60056,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_GENERIC_IRST = 60057,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_GENERIC_MAWS = 60058,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_GENERIC_RWR = 60059,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_L_136_MAK = 60060,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_LEONARDO_SASS_IRST = 60061,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_OSF_IRST = 60062,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_PIRATE_IRST = 60063,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_RECCELITE = 60064,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_THALES_NEDERLAND_SIRIUS_IRST = 60065,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_TORNADO_RWR = 60066,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_TOES_521_FLIR = 60067,
     DIS_PDU_SENSOR_RECORD_SENSOR_TYPE_PASSIVE_SENSORS_SAFRAN_VAMPIR_MB = 60068
}  DIS_PDU_Sensor_Record_Sensor_Type_Passive_Sensors;

/* SISO-REF-010 [UID 327] */
typedef enum
{
     DIS_PDU_MUNITION_EXPENDABLE_STATUS_OTHER = 0,
     DIS_PDU_MUNITION_EXPENDABLE_STATUS_READY = 1,
     DIS_PDU_MUNITION_EXPENDABLE_STATUS_INVENTORY = 2
}  DIS_PDU_Munition_Expendable_Status;

/* SISO-REF-010 [UID 328] */
typedef enum
{
     DIS_PDU_FUEL_MEASUREMENT_UNITS_OTHER = 0,
     DIS_PDU_FUEL_MEASUREMENT_UNITS_LITER = 1,
     DIS_PDU_FUEL_MEASUREMENT_UNITS_KILOGRAM = 2
}  DIS_PDU_Fuel_Measurement_Units;

/* SISO-REF-010 [UID 329] */
typedef enum
{
     DIS_PDU_FUEL_LOCATION_OTHER = 0
}  DIS_PDU_Fuel_Location;

/* SISO-REF-010 [UID 330] */
typedef enum
{
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_OTHER = 0,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_TARGET_ENTITY = 1,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_TARGET_LOCATION = 2,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_HOME_BASE_LOCATION = 3,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_CAP_POINT_LOCATION = 4,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_FLIGHT_LEADER = 5,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_FLIGHT_MEMBER = 6,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_IP_POINT = 7,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_RENDEZVOUS_POINT = 8,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_ON_STATION_LOCATION = 9,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_LANDING_ZONE_LOCATION = 10,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_DOWNED_PILOT = 11,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_TANKER_ENTITY_THAT_IS_CURRENTLY_REFUELING_THE_TRANSFERRED_ENTITY = 12,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_TANKER_ENTITY_THE_TRANSFERRED_ENTITY_IS_HEADED_TOWARDS_TO_REFUEL = 13,
     DIS_PDU_ENTITY_ASSOCIATION_ASSOCIATION_TYPE_ENTITY_HEADED_TOWARDS_TO_JOIN_UP_WITH = 14
}  DIS_PDU_Entity_Association_Association_Type;

/* SISO-REF-010 [UID 331] */
typedef enum
{
     DIS_PDU_SENSOR_ON_OFF_STATUS_OFF = 0,
     DIS_PDU_SENSOR_ON_OFF_STATUS_ON = 1
}  DIS_PDU_Sensor_On_Off_Status;

/* SISO-REF-010 [UID 332] */
typedef enum
{
     DIS_PDU_OWNERSHIP_STATUS_OTHER = 0,
     DIS_PDU_OWNERSHIP_STATUS_NEW_OWNER = 1,
     DIS_PDU_OWNERSHIP_STATUS_OWNERSHIP_QUERY_RESPONSE = 2,
     DIS_PDU_OWNERSHIP_STATUS_OWNERSHIP_CONFLICT = 3,
     DIS_PDU_OWNERSHIP_STATUS_LOCAL_ENTITY_CANCELLED_AUTO_RESOLVE_CONFLICT = 4,
     DIS_PDU_OWNERSHIP_STATUS_LOCAL_ENTITY_CANCELLED_MANUAL_RESOLVE_CONFLICT = 5,
     DIS_PDU_OWNERSHIP_STATUS_LOCAL_ENTITY_CANCELLED_REMOVE_ENTITY_TCR_RECEIVED = 6
}  DIS_PDU_Ownership_Status;

/* SISO-REF-010 [UID 333] */
typedef enum
{
     DIS_PDU_RECORD_R_EVENT_TYPE_OTHER = 0
}  DIS_PDU_Record_R_Event_Type;

/* SISO-REF-010 [UID 334] */
typedef enum
{
     DIS_PDU_RECORD_QUERY_R_EVENT_TYPE_PERIODIC = 0,
     DIS_PDU_RECORD_QUERY_R_EVENT_TYPE_INTERNAL_ENTITY_STATE_DATA = 1
}  DIS_PDU_Record_Query_R_Event_Type;

/* SISO-REF-010 [UID 335] */
typedef enum
{
     DIS_PDU_UA_PROPULSION_PLANT_CONFIGURATION_CONFIGURATION_OTHER = 0,
     DIS_PDU_UA_PROPULSION_PLANT_CONFIGURATION_CONFIGURATION_DIESEL_ELECTRIC = 1,
     DIS_PDU_UA_PROPULSION_PLANT_CONFIGURATION_CONFIGURATION_DIESEL = 2,
     DIS_PDU_UA_PROPULSION_PLANT_CONFIGURATION_CONFIGURATION_BATTERY = 3,
     DIS_PDU_UA_PROPULSION_PLANT_CONFIGURATION_CONFIGURATION_TURBINE_REDUCTION = 4,
     DIS_PDU_UA_PROPULSION_PLANT_CONFIGURATION_CONFIGURATION_STEAM = 6,
     DIS_PDU_UA_PROPULSION_PLANT_CONFIGURATION_CONFIGURATION_GAS_TURBINE = 7
}  DIS_PDU_UA_Propulsion_Plant_Configuration_Configuration;

/* SISO-REF-010 [UID 336] */
typedef enum
{
     DIS_PDU_MINEFIELD_STATE_PROTOCOL_MODE_HEARTBEAT_MODE = 0,
     DIS_PDU_MINEFIELD_STATE_PROTOCOL_MODE_QRP_MODE = 1
}  DIS_PDU_Minefield_State_Protocol_Mode;

/* SISO-REF-010 [UID 337] */
typedef enum
{
     DIS_PDU_TRANSPONDER_INTERROGATOR_INDICATOR_TRANSPONDER = 0,
     DIS_PDU_TRANSPONDER_INTERROGATOR_INDICATOR_INTERROGATOR = 1
}  DIS_PDU_Transponder_Interrogator_Indicator;

static const value_string DIS_PDU_Transponder_Interrogator_Indicator_Strings[] = {
     { DIS_PDU_TRANSPONDER_INTERROGATOR_INDICATOR_TRANSPONDER, "Transponder" },
     { DIS_PDU_TRANSPONDER_INTERROGATOR_INDICATOR_INTERROGATOR, "Interrogator" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 338] */
typedef enum
{
     DIS_PDU_IFF_SIMULATION_MODE_REGENERATION = 0,
     DIS_PDU_IFF_SIMULATION_MODE_INTERACTIVE = 1
}  DIS_PDU_IFF_Simulation_Mode;

static const value_string DIS_PDU_IFF_Simulation_Mode_Strings[] = {
     { DIS_PDU_IFF_SIMULATION_MODE_REGENERATION, "Regeneration" },
     { DIS_PDU_IFF_SIMULATION_MODE_INTERACTIVE, "Interactive" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 339] */
typedef enum
{
     DIS_PDU_IFF_APPLICABLE_MODES_NO_APPLICABLE_MODES_DATA = 0,
     DIS_PDU_IFF_APPLICABLE_MODES_ALL_MODES = 1
}  DIS_PDU_IFF_Applicable_Modes;

/* SISO-REF-010 [UID 340] */
typedef enum
{
     DIS_PDU_MODE_C_ALTITUDE_INDICATOR_POSITIVE_ALTITUDE_ABOVE_MSL = 0,
     DIS_PDU_MODE_C_ALTITUDE_INDICATOR_NEGATIVE_ALTITUDE_BELOW_MSL_USE_ALTERNATE_MODE_C_ALTITUDE = 1
}  DIS_PDU_Mode_C_Altitude_Indicator;

static const value_string DIS_PDU_Mode_C_Altitude_Indicator_Strings[] = {
     { DIS_PDU_MODE_C_ALTITUDE_INDICATOR_POSITIVE_ALTITUDE_ABOVE_MSL, "Positive Altitude Above MSL" },
     { DIS_PDU_MODE_C_ALTITUDE_INDICATOR_NEGATIVE_ALTITUDE_BELOW_MSL_USE_ALTERNATE_MODE_C_ALTITUDE, "Negative Altitude Below MSL/Use Alternate Mode C Altitude" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 341] */
typedef enum
{
     DIS_PDU_TCAS_ACAS_BASIC_ADVANCED_INDICATOR_BASIC = 0,
     DIS_PDU_TCAS_ACAS_BASIC_ADVANCED_INDICATOR_ADVANCED = 1
}  DIS_PDU_TCAS_ACAS_Basic_Advanced_Indicator;

static const value_string DIS_PDU_TCAS_ACAS_Basic_Advanced_Indicator_Strings[] = {
     { DIS_PDU_TCAS_ACAS_BASIC_ADVANCED_INDICATOR_BASIC, "Basic" },
     { DIS_PDU_TCAS_ACAS_BASIC_ADVANCED_INDICATOR_ADVANCED, "Advanced" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 342] */
typedef enum
{
     DIS_PDU_TCAS_ACAS_INDICATOR_TCAS = 0,
     DIS_PDU_TCAS_ACAS_INDICATOR_ACAS = 1
}  DIS_PDU_TCAS_ACAS_Indicator;

static const value_string DIS_PDU_TCAS_ACAS_Indicator_Strings[] = {
     { DIS_PDU_TCAS_ACAS_INDICATOR_TCAS, "TCAS" },
     { DIS_PDU_TCAS_ACAS_INDICATOR_ACAS, "ACAS" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 343] */
typedef enum
{
     DIS_PDU_TCAS_ACAS_SOFTWARE_VERSION_NO_STATEMENT = 0,
     DIS_PDU_TCAS_ACAS_SOFTWARE_VERSION_6_0_2 = 1,
     DIS_PDU_TCAS_ACAS_SOFTWARE_VERSION_7_0 = 2
}  DIS_PDU_TCAS_ACAS_Software_Version;

static const value_string DIS_PDU_TCAS_ACAS_Software_Version_Strings[] = {
     { DIS_PDU_TCAS_ACAS_SOFTWARE_VERSION_NO_STATEMENT, "No Statement" },
     { DIS_PDU_TCAS_ACAS_SOFTWARE_VERSION_6_0_2, "6.0.2" },
     { DIS_PDU_TCAS_ACAS_SOFTWARE_VERSION_7_0, "7.0" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 344] */
typedef enum
{
     DIS_PDU_TCAS_ACAS_TYPE_NO_STATEMENT = 0,
     DIS_PDU_TCAS_ACAS_TYPE_ACAS_I = 1,
     DIS_PDU_TCAS_ACAS_TYPE_ACAS_II = 2
}  DIS_PDU_TCAS_ACAS_Type;

static const value_string DIS_PDU_TCAS_ACAS_Type_Strings[] = {
     { DIS_PDU_TCAS_ACAS_TYPE_NO_STATEMENT, "No Statement" },
     { DIS_PDU_TCAS_ACAS_TYPE_ACAS_I, "ACAS I" },
     { DIS_PDU_TCAS_ACAS_TYPE_ACAS_II, "ACAS II" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 345] */
typedef enum
{
     DIS_PDU_TCAS_I_II_TYPE_TCAS_I_OR_NO_STATEMENT = 0,
     DIS_PDU_TCAS_I_II_TYPE_TCAS_II = 1
}  DIS_PDU_TCAS_I_II_Type;

static const value_string DIS_PDU_TCAS_I_II_Type_Strings[] = {
     { DIS_PDU_TCAS_I_II_TYPE_TCAS_I_OR_NO_STATEMENT, "TCAS I (or No Statement)" },
     { DIS_PDU_TCAS_I_II_TYPE_TCAS_II, "TCAS II" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 346] */
typedef enum
{
     DIS_PDU_MODE_5_IFF_MISSION_NO_STATEMENT = 0,
     DIS_PDU_MODE_5_IFF_MISSION_SURVEILLANCE_SHORAD = 1,
     DIS_PDU_MODE_5_IFF_MISSION_SHORAD_ASSOCIATED_WITH_A_WEAPONS_SYSTEM = 2,
     DIS_PDU_MODE_5_IFF_MISSION_WEAPON_SYSTEM = 3,
     DIS_PDU_MODE_5_IFF_MISSION_AIRBORNE_AND_SURFACE_SHIP_SURVEILLANCE_PLATFORMS = 4,
     DIS_PDU_MODE_5_IFF_MISSION_AIRBORNE_AND_SURFACE_SHIP_WEAPONS_PLATFORMS = 5,
     DIS_PDU_MODE_5_IFF_MISSION_GROUND_TO_GROUND = 6
}  DIS_PDU_Mode_5_IFF_Mission;

/* SISO-REF-010 [UID 347] */
typedef enum
{
     DIS_PDU_MODE_S_INTERROGATOR_STATUS_TRANSMIT_STATE_NO_STATEMENT = 0,
     DIS_PDU_MODE_S_INTERROGATOR_STATUS_TRANSMIT_STATE_ROLL_CALL = 1,
     DIS_PDU_MODE_S_INTERROGATOR_STATUS_TRANSMIT_STATE_ALL_CALL = 2,
     DIS_PDU_MODE_S_INTERROGATOR_STATUS_TRANSMIT_STATE_LOCKOUT_OVERRIDE = 3,
     DIS_PDU_MODE_S_INTERROGATOR_STATUS_TRANSMIT_STATE_TEMPORARY_LOCKOUT = 4,
     DIS_PDU_MODE_S_INTERROGATOR_STATUS_TRANSMIT_STATE_INTERMITTENT_LOCKOUT = 5
}  DIS_PDU_Mode_S_Interrogator_Status_Transmit_State;

/* SISO-REF-010 [UID 348] */
typedef enum
{
     DIS_PDU_MODE_S_INTERROGATOR_IDENTIFIER_IC_TYPE_II = 0,
     DIS_PDU_MODE_S_INTERROGATOR_IDENTIFIER_IC_TYPE_SI = 1
}  DIS_PDU_Mode_S_Interrogator_Identifier_IC_Type;

static const value_string DIS_PDU_Mode_S_Interrogator_Identifier_IC_Type_Strings[] = {
     { DIS_PDU_MODE_S_INTERROGATOR_IDENTIFIER_IC_TYPE_II, "II" },
     { DIS_PDU_MODE_S_INTERROGATOR_IDENTIFIER_IC_TYPE_SI, "SI" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 349] */
typedef enum
{
     DIS_PDU_ISLS_ANTENNA_TYPE_NO_STATEMENT = 0,
     DIS_PDU_ISLS_ANTENNA_TYPE_MONOPULSE_ANTENNA = 1
}  DIS_PDU_ISLS_Antenna_Type;

/* SISO-REF-010 [UID 350] */
typedef enum
{
     DIS_PDU_MODE_5_REPLY_NO_RESPONSE = 0,
     DIS_PDU_MODE_5_REPLY_VALID = 1,
     DIS_PDU_MODE_5_REPLY_INVALID = 2,
     DIS_PDU_MODE_5_REPLY_UNABLE_TO_VERIFY = 3
}  DIS_PDU_Mode_5_Reply;

/* SISO-REF-010 [UID 351] */
typedef enum
{
     DIS_PDU_ANTENNA_SELECTION_NO_STATEMENT = 0,
     DIS_PDU_ANTENNA_SELECTION_TOP = 1,
     DIS_PDU_ANTENNA_SELECTION_BOTTOM = 2,
     DIS_PDU_ANTENNA_SELECTION_DIVERSITY = 3
}  DIS_PDU_Antenna_Selection;

/* SISO-REF-010 [UID 352] */
typedef enum
{
     DIS_PDU_MODE_5_SQUITTER_TYPE_NOT_CAPABLE = 0,
     DIS_PDU_MODE_5_SQUITTER_TYPE_SHORT = 1,
     DIS_PDU_MODE_5_SQUITTER_TYPE_EXTENDED = 2
}  DIS_PDU_Mode_5_Squitter_Type;

/* SISO-REF-010 [UID 353] */
typedef enum
{
     DIS_PDU_LEVEL_2_SQUITTER_STATUS_DISABLED = 0,
     DIS_PDU_LEVEL_2_SQUITTER_STATUS_ENABLED = 1
}  DIS_PDU_Level_2_Squitter_Status;

/* SISO-REF-010 [UID 354] */
typedef enum
{
     DIS_PDU_MODE_S_SQUITTER_TYPE_NOT_CAPABLE = 0,
     DIS_PDU_MODE_S_SQUITTER_TYPE_ACQUISITION = 1,
     DIS_PDU_MODE_S_SQUITTER_TYPE_EXTENDED = 2,
     DIS_PDU_MODE_S_SQUITTER_TYPE_SHORT = 3
}  DIS_PDU_Mode_S_Squitter_Type;

/* SISO-REF-010 [UID 355] */
typedef enum
{
     DIS_PDU_MODE_S_SQUITTER_RECORD_SOURCE_LAYER_4_IFF_DATA_RECORDS = 0,
     DIS_PDU_MODE_S_SQUITTER_RECORD_SOURCE_LAYER_5_GICB_IFF_DATA_RECORDS = 1
}  DIS_PDU_Mode_S_Squitter_Record_Source;

/* SISO-REF-010 [UID 356] */
typedef enum
{
     DIS_PDU_AIRCRAFT_PRESENT_DOMAIN_NO_STATEMENT = 0,
     DIS_PDU_AIRCRAFT_PRESENT_DOMAIN_AIRBORNE = 1,
     DIS_PDU_AIRCRAFT_PRESENT_DOMAIN_ON_GROUND_SURFACE = 2
}  DIS_PDU_Aircraft_Present_Domain;

/* SISO-REF-010 [UID 357] */
typedef enum
{
     DIS_PDU_AIRCRAFT_IDENTIFICATION_TYPE_NO_STATEMENT = 0,
     DIS_PDU_AIRCRAFT_IDENTIFICATION_TYPE_FLIGHT_NUMBER = 1,
     DIS_PDU_AIRCRAFT_IDENTIFICATION_TYPE_TAIL_NUMBER = 2
}  DIS_PDU_Aircraft_Identification_Type;

/* SISO-REF-010 [UID 358] */
typedef enum
{
     DIS_PDU_CAPABILITY_REPORT_NO_COMMUNICATIONS_CAPABILITY_CA = 0,
     DIS_PDU_CAPABILITY_REPORT_RESERVED_1 = 1,
     DIS_PDU_CAPABILITY_REPORT_RESERVED_2 = 2,
     DIS_PDU_CAPABILITY_REPORT_RESERVED_3 = 3,
     DIS_PDU_CAPABILITY_REPORT_SIGNIFIES_AT_LEAST_COMM_A_AND_COMM_B_CAPABILITY_AND_ABILITY_TO_SET_CA_CODE_7_AND_ON_THE_GROUND = 4,
     DIS_PDU_CAPABILITY_REPORT_SIGNIFIES_AT_LEAST_COMM_A_AND_COMM_B_CAPABILITY_AND_ABILITY_TO_SET_CA_CODE_7_AND_AIRBORNE = 5,
     DIS_PDU_CAPABILITY_REPORT_SIGNIFIES_AT_LEAST_COMM_A_AND_COMM_B_CAPABILITY_AND_ABILITY_TO_SET_CA_CODE_7_AND_EITHER_AIRBORNE_OR_ON_THE_GROUND = 6,
     DIS_PDU_CAPABILITY_REPORT_SIGNIFIES_THE_DOWNLINK_REQUEST_DR_FIELD_IS_NOT_EQUAL_TO_0_AND_THE_FLIGHT_STATUS_FS_FIELD_EQUALS_2_3_4_OR_5_AND_EITHER_AIRBORNE_OR_ON_THE_GROUND = 7,
     DIS_PDU_CAPABILITY_REPORT_NO_STATEMENT = 255
}  DIS_PDU_Capability_Report;

/* SISO-REF-010 [UID 359] */
typedef enum
{
     DIS_PDU_NAVIGATION_SOURCE_NO_STATEMENT = 0,
     DIS_PDU_NAVIGATION_SOURCE_GPS = 1,
     DIS_PDU_NAVIGATION_SOURCE_INS = 2,
     DIS_PDU_NAVIGATION_SOURCE_INS_GPS = 3
}  DIS_PDU_Navigation_Source;

/* SISO-REF-010 [UID 360] */
typedef enum
{
     DIS_PDU_IFF_DATA_RECORD_AVAILABLE_COMPUTE_LOCALLY = 0,
     DIS_PDU_IFF_DATA_RECORD_AVAILABLE_IFF_DATA_RECORD_AVAILABLE = 1
}  DIS_PDU_IFF_Data_Record_Available;

/* SISO-REF-010 [UID 361] */
typedef enum
{
     DIS_PDU_MODE_5_S_ALTITUDE_RESOLUTION_100_FOOT = 0,
     DIS_PDU_MODE_5_S_ALTITUDE_RESOLUTION_25_FOOT = 1
}  DIS_PDU_Mode_5_S_Altitude_Resolution;

/* SISO-REF-010 [UID 362] */
typedef enum
{
     DIS_PDU_DELTA_MODE_5_S_ALTITUDE_POSITIVE_NEGATIVE_INDICATOR_POSITIVE = 0,
     DIS_PDU_DELTA_MODE_5_S_ALTITUDE_POSITIVE_NEGATIVE_INDICATOR_NEGATIVE = 1
}  DIS_PDU_Delta_Mode_5_S_Altitude_Positive_Negative_Indicator;

/* SISO-REF-010 [UID 363] */
typedef enum
{
     DIS_PDU_FORMAT_TYPE_NO_DATA = 0,
     DIS_PDU_FORMAT_TYPE_IDENTITY_FORMAT = 4,
     DIS_PDU_FORMAT_TYPE_SURFACE_FORMAT_5_METER_RNP = 5,
     DIS_PDU_FORMAT_TYPE_SURFACE_FORMAT_100_METER_RNP = 6,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_5_METER_RNP_25_FOOT_BAROMETRIC_ALTITUDE = 7,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_100_METER_RNP_25_FOOT_BAROMETRIC_ALTITUDE = 8,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_0_25_NMI_RNP_25_FOOT_BAROMETRIC_ALTITUDE = 9,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_1_0_NMI_RNP_25_FOOT_BAROMETRIC_ALTITUDE = 10,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_5_METER_RNP_100_FOOT_BAROMETRIC_ALTITUDE = 11,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_100_METER_RNP_100_FOOT_BAROMETRIC_ALTITUDE = 12,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_0_25_NMI_RNP_100_FOOT_BAROMETRIC_ALTITUDE = 13,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_1_0_NMI_RNP_100_FOOT_BAROMETRIC_ALTITUDE = 14,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_5_METER_RNP_GPS_HEIGHT = 15,
     DIS_PDU_FORMAT_TYPE_AIRBORNE_FORMAT_100_METER_RNP_GPS_HEIGHT = 16
}  DIS_PDU_Format_Type;

/* SISO-REF-010 [UID 364] */
typedef enum
{
     DIS_PDU_AIRCRAFT_ADDRESS_SOURCE_MODE_S_AIRCRAFT_ADDRESS_FIELD_VALUE = 0,
     DIS_PDU_AIRCRAFT_ADDRESS_SOURCE_GICB_IFF_DATA_RECORD_AVAILABLE = 1
}  DIS_PDU_Aircraft_Address_Source;

/* SISO-REF-010 [UID 365] */
typedef enum
{
     DIS_PDU_SURVEILLANCE_STATUS_NO_INFORMATION = 0,
     DIS_PDU_SURVEILLANCE_STATUS_EMERGENCY_LOSS_OF_COMMUNICATIONS = 1,
     DIS_PDU_SURVEILLANCE_STATUS_SPI = 2,
     DIS_PDU_SURVEILLANCE_STATUS_ATCRBS_CODE_CHANGE = 3
}  DIS_PDU_Surveillance_Status;

/* SISO-REF-010 [UID 366] */
typedef enum
{
     DIS_PDU_TURN_RATE_SOURCE_COMPUTE_LOCALLY = 0,
     DIS_PDU_TURN_RATE_SOURCE_LESS_THAN_1_DEGREE_TURN_OR_NOT_TURNING = 1,
     DIS_PDU_TURN_RATE_SOURCE_1_DEGREE_OR_GREATER_TURN_RATE = 2
}  DIS_PDU_Turn_Rate_Source;

/* SISO-REF-010 [UID 367] */
typedef enum
{
     DIS_PDU_TIME_TYPE_SOURCE_COMPUTE_LOCALLY = 0,
     DIS_PDU_TIME_TYPE_SOURCE_EVEN_SECOND = 1,
     DIS_PDU_TIME_TYPE_SOURCE_ODD_SECOND = 2
}  DIS_PDU_Time_Type_Source;

/* SISO-REF-010 [UID 368] */
typedef enum
{
     DIS_PDU_AIRCRAFT_TYPE_WAKE_NO_STATEMENT = 0
}  DIS_PDU_Aircraft_Type_Wake;

/* SISO-REF-010 [UID 369] */
typedef enum
{
     DIS_PDU_DATA_CATEGORY_NO_STATEMENT = 0,
     DIS_PDU_DATA_CATEGORY_FUNCTIONAL_DATA = 1,
     DIS_PDU_DATA_CATEGORY_TRANSPONDER_INTERROGATOR_DATA_LINK_MESSAGES = 2
}  DIS_PDU_Data_Category;

/* SISO-REF-010 [UID 370] */
typedef enum
{
     DIS_PDU_T_I_LINK_TYPE_NOT_USED = 0,
     DIS_PDU_T_I_LINK_TYPE_GROUND_INITIATED_COMMUNICATIONS_B_GICB = 1,
     DIS_PDU_T_I_LINK_TYPE_AUTOMATIC_DEPENDENT_SURVEILLANCE_ADS = 2,
     DIS_PDU_T_I_LINK_TYPE_GLOBAL_NAVIGATION_SATELLITE_SYSTEM_GNSS = 3,
     DIS_PDU_T_I_LINK_TYPE_DATA_LINK_INITIATION_CAPABILITY_DLIC = 4,
     DIS_PDU_T_I_LINK_TYPE_AIRCRAFT_COMMUNICATIONS_ADDRESSING_AND_REPORTING_SYSTEM_ACARS = 5,
     DIS_PDU_T_I_LINK_TYPE_ATC_COMMUNICATIONS_MANAGEMENT_ACM = 6,
     DIS_PDU_T_I_LINK_TYPE_VHF_DIGITAL_LINK_VDL = 7,
     DIS_PDU_T_I_LINK_TYPE_AERONAUTICAL_TELECOMMUNICATION_NETWORK_ATN = 8,
     DIS_PDU_T_I_LINK_TYPE_MODE_SELECT_S = 9,
     DIS_PDU_T_I_LINK_TYPE_AIRBORNE_COLLISION_AVOIDANCE_SYSTEMS_ACAS = 10,
     DIS_PDU_T_I_LINK_TYPE_TRAFFIC_COLLISION_AVOIDANCE_SYSTEMS_TCAS = 11,
     DIS_PDU_T_I_LINK_TYPE_AUTOMATIC_DEPENDENT_SURVEILLANCE_B_ADS_B = 12
}  DIS_PDU_T_I_Link_Type;

/* SISO-REF-010 [UID 371] */
typedef enum
{
     DIS_PDU_ANTENNA_STATUS_NO_STATEMENT = 0,
     DIS_PDU_ANTENNA_STATUS_NOT_ABLE_TO_EMIT = 1,
     DIS_PDU_ANTENNA_STATUS_ABLE_TO_EMIT = 2
}  DIS_PDU_Antenna_Status;

/* SISO-REF-010 [UID 372] */
typedef enum
{
     DIS_PDU_TRANSMISSION_INDICATOR_NO_STATEMENT = 0,
     DIS_PDU_TRANSMISSION_INDICATOR_ORIGINAL_INTERROGATION = 1,
     DIS_PDU_TRANSMISSION_INDICATOR_INTERROGATION_REPLY = 2,
     DIS_PDU_TRANSMISSION_INDICATOR_SQUITTER_TRANSMISSION = 3
}  DIS_PDU_Transmission_Indicator;

/* SISO-REF-010 [UID 373] */
typedef enum
{
     DIS_PDU_REPLY_AMPLIFICATION_NO_STATEMENT = 0,
     DIS_PDU_REPLY_AMPLIFICATION_COMPLETE = 1,
     DIS_PDU_REPLY_AMPLIFICATION_LIMTED = 2,
     DIS_PDU_REPLY_AMPLIFICATION_UNABLE_TO_RESPOND = 3
}  DIS_PDU_Reply_Amplification;

/* SISO-REF-010 [UID 374] */
typedef enum
{
     DIS_PDU_DE_FIRE_FLAGS_STATE_UPDATE_FLAG_UPDATE_DUE_TO_HEARTBEAT_TIMER = 0,
     DIS_PDU_DE_FIRE_FLAGS_STATE_UPDATE_FLAG_STATE_CHANGE = 1
}  DIS_PDU_DE_Fire_Flags_State_Update_Flag;

/* SISO-REF-010 [UID 375] */
typedef enum
{
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SMOKE_NO_SMOKE = 0,
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SMOKE_LIGHT_SMOKE = 1,
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SMOKE_MODERATE_SMOKE = 2,
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SMOKE_HEAVY_SMOKE = 3
}  DIS_PDU_Component_Visual_Damage_Status_Smoke;

/* SISO-REF-010 [UID 376] */
typedef enum
{
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SURFACE_DAMAGE_NORMAL_APPEARANCE = 0,
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SURFACE_DAMAGE_LIGHT_CHARRING = 1,
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SURFACE_DAMAGE_HEAVY_CHARRING = 2,
     DIS_PDU_COMPONENT_VISUAL_DAMAGE_STATUS_SURFACE_DAMAGE_ONE_OR_MORE_HOLES_BURNED_COMPLETELY_THROUGH_SURFACE = 3
}  DIS_PDU_Component_Visual_Damage_Status_Surface_Damage;

/* SISO-REF-010 [UID 377] */
typedef enum
{
     DIS_PDU_GRID_AXIS_DESCRIPTOR_AXIS_TYPE_REGULAR_AXIS = 0,
     DIS_PDU_GRID_AXIS_DESCRIPTOR_AXIS_TYPE_IRREGULAR_AXIS = 1
}  DIS_PDU_Grid_Axis_Descriptor_Axis_Type;

/* SISO-REF-010 [UID 378] */
typedef enum
{
     DIS_PDU_APPEARANCE_PAINT_SCHEME_UNIFORM_COLOR = 0,
     DIS_PDU_APPEARANCE_PAINT_SCHEME_CAMOUFLAGE = 1
}  DIS_PDU_Appearance_Paint_Scheme;

/* SISO-REF-010 [UID 379] */
typedef enum
{
     DIS_PDU_APPEARANCE_DAMAGE_NO_DAMAGE = 0,
     DIS_PDU_APPEARANCE_DAMAGE_SLIGHT_DAMAGE = 1,
     DIS_PDU_APPEARANCE_DAMAGE_MODERATE_DAMAGE = 2,
     DIS_PDU_APPEARANCE_DAMAGE_DESTROYED = 3
}  DIS_PDU_Appearance_Damage;

static const value_string DIS_PDU_Appearance_Damage_Strings[] = {
     { DIS_PDU_APPEARANCE_DAMAGE_NO_DAMAGE, "No Damage" },
     { DIS_PDU_APPEARANCE_DAMAGE_SLIGHT_DAMAGE, "Slight Damage" },
     { DIS_PDU_APPEARANCE_DAMAGE_MODERATE_DAMAGE, "Moderate Damage" },
     { DIS_PDU_APPEARANCE_DAMAGE_DESTROYED, "Destroyed" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 380] */
typedef enum
{
     DIS_PDU_MODE_5_MESSAGE_FORMATS_STATUS_CAPABILITY = 0,
     DIS_PDU_MODE_5_MESSAGE_FORMATS_STATUS_ACTIVE_INTERROGATION = 1
}  DIS_PDU_Mode_5_Message_Formats_Status;

/* SISO-REF-010 [UID 381] */
typedef enum
{
     DIS_PDU_APPEARANCE_TRAILING_EFFECTS_NONE = 0,
     DIS_PDU_APPEARANCE_TRAILING_EFFECTS_SMALL = 1,
     DIS_PDU_APPEARANCE_TRAILING_EFFECTS_MEDIUM = 2,
     DIS_PDU_APPEARANCE_TRAILING_EFFECTS_LARGE = 3
}  DIS_PDU_Appearance_Trailing_Effects;

static const value_string DIS_PDU_Appearance_Trailing_Effects_Strings[] = {
     { DIS_PDU_APPEARANCE_TRAILING_EFFECTS_NONE, "None" },
     { DIS_PDU_APPEARANCE_TRAILING_EFFECTS_SMALL, "Small" },
     { DIS_PDU_APPEARANCE_TRAILING_EFFECTS_MEDIUM, "Medium" },
     { DIS_PDU_APPEARANCE_TRAILING_EFFECTS_LARGE, "Large" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 382] */
typedef enum
{
     DIS_PDU_APPEARANCE_HATCH_NOT_APPLICABLE = 0,
     DIS_PDU_APPEARANCE_HATCH_CLOSED = 1,
     DIS_PDU_APPEARANCE_HATCH_POPPED = 2,
     DIS_PDU_APPEARANCE_HATCH_POPPED_AND_PERSON_IS_VISIBLE = 3,
     DIS_PDU_APPEARANCE_HATCH_OPEN = 4,
     DIS_PDU_APPEARANCE_HATCH_OPEN_AND_PERSON_IS_VISIBLE = 5
}  DIS_PDU_Appearance_Hatch;

/* SISO-REF-010 [UID 383] */
typedef enum
{
     DIS_PDU_APPEARANCE_LAUNCHER_OPERATIONAL_NOT_RAISED_NOT_OPERATIONAL = 0,
     DIS_PDU_APPEARANCE_LAUNCHER_OPERATIONAL_RAISED_OPERATIONAL = 1
}  DIS_PDU_Appearance_Launcher_Operational;

/* SISO-REF-010 [UID 384] */
typedef enum
{
     DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_DESERT_CAMOUFLAGE = 0,
     DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_WINTER_CAMOUFLAGE = 1,
     DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_FOREST_CAMOUFLAGE = 2,
     DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_OTHER = 3
}  DIS_PDU_Appearance_Camouflage_Type;

static const value_string DIS_PDU_Appearance_Camouflage_Type_Strings[] = {
     { DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_DESERT_CAMOUFLAGE, "Desert Camouflage" },
     { DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_WINTER_CAMOUFLAGE, "Winter Camouflage" },
     { DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_FOREST_CAMOUFLAGE, "Forest Camouflage" },
     { DIS_PDU_APPEARANCE_CAMOUFLAGE_TYPE_OTHER, "Other" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 385] */
typedef enum
{
     DIS_PDU_APPEARANCE_CONCEALED_POSITION_NOT_CONCEALED = 0,
     DIS_PDU_APPEARANCE_CONCEALED_POSITION_PREPARED_CONCEALED_POSITION = 1
}  DIS_PDU_Appearance_Concealed_Position;

/* SISO-REF-010 [UID 386] */
typedef enum
{
     DIS_PDU_APPEARANCE_ENTITY_OR_OBJECT_STATE_ACTIVE = 0,
     DIS_PDU_APPEARANCE_ENTITY_OR_OBJECT_STATE_DEACTIVATED = 1
}  DIS_PDU_Appearance_Entity_or_Object_State;

/* SISO-REF-010 [UID 387] */
typedef enum
{
     DIS_PDU_APPEARANCE_CANOPY_NOT_APPLICABLE = 0,
     DIS_PDU_APPEARANCE_CANOPY_SINGLE_CANOPY_SINGLE_TROOP_DOOR_CLOSED = 1,
     DIS_PDU_APPEARANCE_CANOPY_FRONT_AND_REAR_CANOPY_LEFT_AND_RIGHT_TROOP_DOOR_CLOSED = 2,
     DIS_PDU_APPEARANCE_CANOPY_FRONT_CANOPY_LEFT_TROOP_DOOR_OPEN = 3,
     DIS_PDU_APPEARANCE_CANOPY_SINGLE_CANOPY_SINGLE_TROOP_DOOR_OPEN = 4,
     DIS_PDU_APPEARANCE_CANOPY_REAR_CANOPY_RIGHT_TROOP_DOOR_OPEN = 5,
     DIS_PDU_APPEARANCE_CANOPY_FRONT_AND_REAR_CANOPY_LEFT_AND_RIGHT_TROOP_DOOR_OPEN = 6
}  DIS_PDU_Appearance_Canopy;

/* SISO-REF-010 [UID 388] */
typedef enum
{
     DIS_PDU_APPEARANCE_SUBSURFACE_HATCH_NOT_APPLICABLE = 0,
     DIS_PDU_APPEARANCE_SUBSURFACE_HATCH_HATCH_IS_CLOSED = 1,
     DIS_PDU_APPEARANCE_SUBSURFACE_HATCH_HATCH_IS_OPEN = 4
}  DIS_PDU_Appearance_Subsurface_Hatch;

/* SISO-REF-010 [UID 389] */
typedef enum
{
     DIS_PDU_DIS_PDU_STATUS_ACTIVE_INTERROGATION_INDICATOR_AII_NOT_ACTIVE = 0,
     DIS_PDU_DIS_PDU_STATUS_ACTIVE_INTERROGATION_INDICATOR_AII_ACTIVE = 1
}  DIS_PDU_DIS_PDU_Status_Active_Interrogation_Indicator_AII;

static const value_string DIS_PDU_DIS_PDU_Status_Active_Interrogation_Indicator_AII_Strings[] = {
     { DIS_PDU_DIS_PDU_STATUS_ACTIVE_INTERROGATION_INDICATOR_AII_NOT_ACTIVE, "Not Active" },
     { DIS_PDU_DIS_PDU_STATUS_ACTIVE_INTERROGATION_INDICATOR_AII_ACTIVE, "Active" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 390] */
typedef enum
{
     DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_NO_INJURY = 0,
     DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_SLIGHT_INJURY = 1,
     DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_MODERATE_INJURY = 2,
     DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_FATAL_INJURY = 3
}  DIS_PDU_Appearance_Lifeform_Health;

static const value_string DIS_PDU_Appearance_Lifeform_Health_Strings[] = {
     { DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_NO_INJURY, "No Injury" },
     { DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_SLIGHT_INJURY, "Slight Injury" },
     { DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_MODERATE_INJURY, "Moderate Injury" },
     { DIS_PDU_APPEARANCE_LIFEFORM_HEALTH_FATAL_INJURY, "Fatal Injury" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 391] */
typedef enum
{
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_NOT_SPECIFIED = 0,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_DETAINED = 1,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_SURRENDER = 2,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_FISTS = 3,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_VERBAL_ABUSE_LEVEL_1 = 4,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_VERBAL_ABUSE_LEVEL_2 = 5,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_VERBAL_ABUSE_LEVEL_3 = 6,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_PASSIVE_RESISTANCE_LEVEL_1 = 7,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_PASSIVE_RESISTANCE_LEVEL_2 = 8,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_PASSIVE_RESISTANCE_LEVEL_3 = 9,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_1 = 10,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_2 = 11,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_3 = 12,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_4 = 13,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_5 = 14,
     DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_6 = 15
}  DIS_PDU_Appearance_Life_Form_Compliance_Status;

static const value_string DIS_PDU_Appearance_Life_Form_Compliance_Status_Strings[] = {
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_NOT_SPECIFIED, "Not Specified" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_DETAINED, "Detained" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_SURRENDER, "Surrender" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_FISTS, "Using Fists" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_VERBAL_ABUSE_LEVEL_1, "Verbal Abuse Level 1" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_VERBAL_ABUSE_LEVEL_2, "Verbal Abuse Level 2" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_VERBAL_ABUSE_LEVEL_3, "Verbal Abuse Level 3" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_PASSIVE_RESISTANCE_LEVEL_1, "Passive Resistance Level 1" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_PASSIVE_RESISTANCE_LEVEL_2, "Passive Resistance Level 2" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_PASSIVE_RESISTANCE_LEVEL_3, "Passive Resistance Level 3" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_1, "Using Non-Lethal Weapon 1" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_2, "Using Non-Lethal Weapon 2" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_3, "Using Non-Lethal Weapon 3" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_4, "Using Non-Lethal Weapon 4" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_5, "Using Non-Lethal Weapon 5" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_COMPLIANCE_STATUS_USING_NON_LETHAL_WEAPON_6, "Using Non-Lethal Weapon 6" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 392] */
typedef enum
{
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_NOT_SPECIFIED = 0,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_UPRIGHT_STANDING_STILL = 1,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_UPRIGHT_WALKING = 2,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_UPRIGHT_RUNNING = 3,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_KNEELING = 4,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_PRONE = 5,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_CRAWLING = 6,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SWIMMING = 7,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_PARACHUTING = 8,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_JUMPING = 9,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SITTING = 10,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SQUATTING = 11,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_CROUCHING = 12,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_WADING = 13,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SURRENDER = 14,
     DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_DETAINED = 15
}  DIS_PDU_Appearance_Life_Form_Posture;

static const value_string DIS_PDU_Appearance_Life_Form_Posture_Strings[] = {
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_NOT_SPECIFIED, "Not Specified" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_UPRIGHT_STANDING_STILL, "Upright, Standing Still" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_UPRIGHT_WALKING, "Upright, Walking" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_UPRIGHT_RUNNING, "Upright, Running" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_KNEELING, "Kneeling" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_PRONE, "Prone" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_CRAWLING, "Crawling" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SWIMMING, "Swimming" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_PARACHUTING, "Parachuting" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_JUMPING, "Jumping" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SITTING, "Sitting" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SQUATTING, "Squatting" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_CROUCHING, "Crouching" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_WADING, "Wading" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_SURRENDER, "Surrender" },
     { DIS_PDU_APPEARANCE_LIFE_FORM_POSTURE_DETAINED, "Detained" },
     { 0, NULL }
 };

/* SISO-REF-010 [UID 393] */
typedef enum
{
     DIS_PDU_APPEARANCE_LIFE_FORM_WEAPON_IMPLEMENT_NOT_PRESENT = 0,
     DIS_PDU_APPEARANCE_LIFE_FORM_WEAPON_IMPLEMENT_STOWED = 1,
     DIS_PDU_APPEARANCE_LIFE_FORM_WEAPON_IMPLEMENT_DEPLOYED_ACTIVE = 2,
     DIS_PDU_APPEARANCE_LIFE_FORM_WEAPON_IMPLEMENT_FIRING_POSITION_IN_USE = 3
}  DIS_PDU_Appearance_Life_Form_Weapon_Implement;

/* SISO-REF-010 [UID 394] */
typedef enum
{
     DIS_PDU_APPEARANCE_CONCEALED_MOVEMENT_OPEN_MOVEMENT = 0,
     DIS_PDU_APPEARANCE_CONCEALED_MOVEMENT_RUSHES_BETWEEN_COVERED_POSITIONS = 1
}  DIS_PDU_Appearance_Concealed_Movement;

/* SISO-REF-010 [UID 395] */
typedef enum
{
     DIS_PDU_APPEARANCE_ENVIRONMENTAL_DENSITY_CLEAR = 0,
     DIS_PDU_APPEARANCE_ENVIRONMENTAL_DENSITY_HAZY = 1,
     DIS_PDU_APPEARANCE_ENVIRONMENTAL_DENSITY_DENSE = 2,
     DIS_PDU_APPEARANCE_ENVIRONMENTAL_DENSITY_VERY_DENSE = 3,
     DIS_PDU_APPEARANCE_ENVIRONMENTAL_DENSITY_OPAQUE = 4
}  DIS_PDU_Appearance_Environmental_Density;

/* SISO-REF-010 [UID 396] */
typedef enum
{
     DIS_PDU_MODE_5_PLATFORM_TYPE_GROUND_VEHICLE = 0,
     DIS_PDU_MODE_5_PLATFORM_TYPE_AIR_VEHICLE = 1
}  DIS_PDU_Mode_5_Platform_Type;

/* SISO-REF-010 [UID 397] */
typedef enum
{
     DIS_PDU_APPEARANCE_ANTI_COLLISION_DAY_NIGHT_DAY = 0,
     DIS_PDU_APPEARANCE_ANTI_COLLISION_DAY_NIGHT_NIGHT = 1
}  DIS_PDU_Appearance_Anti_Collision_Day_Night;

/* SISO-REF-010 [UID 398] */
typedef enum
{
     DIS_PDU_APPEARANCE_NAVIGATION_POSITION_BRIGHTNESS_DIM = 0,
     DIS_PDU_APPEARANCE_NAVIGATION_POSITION_BRIGHTNESS_BRIGHT = 1
}  DIS_PDU_Appearance_Navigation_Position_Brightness;


/* SISO-REF-010 [UID 399] */
typedef enum
{
     DIS_PDU_APPEARANCE_SUPPLY_DEPLOYED_NOT_APPLICABLE = 0,
     DIS_PDU_APPEARANCE_SUPPLY_DEPLOYED_STOWED = 1,
     DIS_PDU_APPEARANCE_SUPPLY_DEPLOYED_DEPLOYED = 2,
     DIS_PDU_APPEARANCE_SUPPLY_DEPLOYED_DEPLOYED_AND_ACTIVE = 3
}  DIS_PDU_Appearance_Supply_Deployed;

/* SISO-REF-010 [UID 400] */
typedef enum
{
     DIS_PDU_APPEARANCE_NVG_MODE_OVERT_LIGHTING = 0,
     DIS_PDU_APPEARANCE_NVG_MODE_COVERT_LIGHTING_NIGHT_VISION_GOGGLES = 1
}  DIS_PDU_Appearance_NVG_Mode;


/* SISO-REF-010 [UID 401] */
typedef enum
{
     DIS_PDU_PARACHUTE_NONE = 0,
     DIS_PDU_PARACHUTE_DEPLOYED = 1,
     DIS_PDU_PARACHUTE_COLLAPSED = 2,
     DIS_PDU_PARACHUTE_MALFUNCTION_STREAMER = 3
}  DIS_PDU_Parachute;

/* SISO-REF-010 [UID 402] */
typedef enum
{
     DIS_PDU_FLARE_SMOKE_COLOR_WHITE = 0,
     DIS_PDU_FLARE_SMOKE_COLOR_RED = 1,
     DIS_PDU_FLARE_SMOKE_COLOR_GREEN = 2,
     DIS_PDU_FLARE_SMOKE_COLOR_IR = 3
}  DIS_PDU_Flare_Smoke_Color;

/* SISO-REF-010 [UID 403] */
typedef enum
{
     DIS_PDU_FLARE_SMOKE_STATUS_NOT_IGNITED = 0,
     DIS_PDU_FLARE_SMOKE_STATUS_BURNING = 1,
     DIS_PDU_FLARE_SMOKE_STATUS_BURNED_OUT = 2
}  DIS_PDU_Flare_Smoke_Status;


/* SISO-REF-010 [UID 404] */
typedef enum
{
     DIS_PDU_SPOT_CHAFF_STATUS_NONE = 0,
     DIS_PDU_SPOT_CHAFF_STATUS_DEPLOYED = 1,
     DIS_PDU_SPOT_CHAFF_STATUS_MALFUNCTION = 2
}  DIS_PDU_Spot_Chaff_Status;

/* SISO-REF-010 [UID 405] */
typedef enum
{
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_DAMAGE_NO_DAMAGE = 0,
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_DAMAGE_DAMAGED = 1,
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_DAMAGE_DESTROYED = 2
}  DIS_PDU_Appearance_Object_General_Damage;

/* SISO-REF-010 [UID 406] */
typedef enum
{
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_PREDISTRIBUTED_OBJECT_CREATED_DURING_THE_EXERCISE = 0,
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_PREDISTRIBUTED_OBJECT_PREDISTRIBUTED_PRIOR_TO_EXERCISE_START = 1
}  DIS_PDU_Appearance_Object_General_Predistributed;


/* SISO-REF-010 [UID 407] */
typedef enum
{
     DIS_PDU_APPEARANCE_OBJECT_SPECIFIC_BREACH_STATE_NO_BREACHING = 0,
     DIS_PDU_APPEARANCE_OBJECT_SPECIFIC_BREACH_STATE_BREACHED = 1,
     DIS_PDU_APPEARANCE_OBJECT_SPECIFIC_BREACH_STATE_CLEARED = 2
}  DIS_PDU_Appearance_Object_Specific_Breach_State;

/* SISO-REF-010 [UID 408] */
typedef enum
{
     DIS_PDU_APPEARANCE_OBJECT_SPECIFIC_CHEMICAL_TYPE_OTHER = 0,
     DIS_PDU_APPEARANCE_OBJECT_SPECIFIC_CHEMICAL_TYPE_HYDROCHLORIC = 1,
     DIS_PDU_APPEARANCE_OBJECT_SPECIFIC_CHEMICAL_TYPE_WHITE_PHOSPHOROUS = 2,
     DIS_PDU_APPEARANCE_OBJECT_SPECIFIC_CHEMICAL_TYPE_RED_PHOSPHOROUS = 3
}  DIS_PDU_Appearance_Object_Specific_Chemical_Type;


/* SISO-REF-010 [UID 409] */
typedef enum
{
     DIS_PDU_APPEARANCE_LINEAR_OBJECT_TANK_DITCH_BREACH_NO_BREACHING = 0,
     DIS_PDU_APPEARANCE_LINEAR_OBJECT_TANK_DITCH_BREACH_SLIGHT_BREACHING = 1,
     DIS_PDU_APPEARANCE_LINEAR_OBJECT_TANK_DITCH_BREACH_MODERATE_BREACHED = 2,
     DIS_PDU_APPEARANCE_LINEAR_OBJECT_TANK_DITCH_BREACH_CLEARED = 3
}  DIS_PDU_Appearance_Linear_Object_Tank_Ditch_Breach;

/* SISO-REF-010 [UID 410] */
typedef enum
{
     DIS_PDU_APPEARANCE_LINEAR_OBJECT_LANE_MARKER_VISIBLE_LEFT_SIDE_IS_VISIBLE = 0,
     DIS_PDU_APPEARANCE_LINEAR_OBJECT_LANE_MARKER_VISIBLE_RIGHT_SIDE_IS_VISIBLE = 1,
     DIS_PDU_APPEARANCE_LINEAR_OBJECT_LANE_MARKER_VISIBLE_BOTH_SIDES_ARE_VISIBLE = 2
}  DIS_PDU_Appearance_Linear_Object_Lane_Marker_Visible;


/* SISO-REF-010 [UID 411] */
typedef enum
{
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_IED_PRESENT_NONE = 0,
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_IED_PRESENT_VISIBLE = 1,
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_IED_PRESENT_PARTIALLY_HIDDEN = 2,
     DIS_PDU_APPEARANCE_OBJECT_GENERAL_IED_PRESENT_COMPLETELY_HIDDEN = 3
}  DIS_PDU_Appearance_Object_General_IED_Present;

/* SISO-REF-010 [UID 412] */
typedef enum
{
     DIS_PDU_MODE_5_LEVEL_SELECTION_MODE_5_LEVEL_1 = 0,
     DIS_PDU_MODE_5_LEVEL_SELECTION_MODE_5_LEVEL_2 = 1
}  DIS_PDU_Mode_5_Level_Selection;


/* SISO-REF-010 [UID 413] */
typedef enum
{
     DIS_PDU_SUPPLY_FUEL_TYPE_OTHER = 0,
     DIS_PDU_SUPPLY_FUEL_TYPE_GASOLINE = 1,
     DIS_PDU_SUPPLY_FUEL_TYPE_DIESEL_FUEL_F_54_DF_2 = 2,
     DIS_PDU_SUPPLY_FUEL_TYPE_JP_4_F_40_JET_B = 3,
     DIS_PDU_SUPPLY_FUEL_TYPE_FUEL_OIL = 4,
     DIS_PDU_SUPPLY_FUEL_TYPE_JP_8_F_34_JET_A_1 = 5,
     DIS_PDU_SUPPLY_FUEL_TYPE_FOG_OIL = 6,
     DIS_PDU_SUPPLY_FUEL_TYPE_MULTI_SPECTRAL_FOG_OIL = 7,
     DIS_PDU_SUPPLY_FUEL_TYPE_JP_5_F_44_JET_A = 8,
     DIS_PDU_SUPPLY_FUEL_TYPE_JPTS = 9,
     DIS_PDU_SUPPLY_FUEL_TYPE_TS_1_RUSSIA_RUS = 10
}  DIS_PDU_Supply_Fuel_Type;

/* SISO-REF-010 [UID 414] */
typedef enum
{
     DIS_PDU_SENSOR_TYPE_SOURCE_OTHER_ACTIVE_SENSORS = 0,
     DIS_PDU_SENSOR_TYPE_SOURCE_ELECTROMAGNETIC = 1,
     DIS_PDU_SENSOR_TYPE_SOURCE_PASSIVE_SENSORS = 2,
     DIS_PDU_SENSOR_TYPE_SOURCE_MINEFIELD_SENSORS = 3,
     DIS_PDU_SENSOR_TYPE_SOURCE_UNDERWATER_ACOUSTICS = 4,
     DIS_PDU_SENSOR_TYPE_SOURCE_LASERS = 5
}  DIS_PDU_Sensor_Type_Source;


/* SISO-REF-010 [UID 415] */
typedef enum
{
     DIS_PDU_ATTACHED_PART_DETACHED_INDICATOR_ATTACHED = 0,
     DIS_PDU_ATTACHED_PART_DETACHED_INDICATOR_DETACHED = 1
}  DIS_PDU_Attached_Part_Detached_Indicator;

/* SISO-REF-010 [UID 416] */
typedef enum
{
     DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_CLASS_SIMULATED_COMMUNICATIONS_CHANNEL = 0,
     DIS_PDU_INTERCOM_CONTROL_COMMUNICATIONS_CLASS_SIMULATION_SUPPORT_COMMUNICATIONS_CHANNEL = 1
}  DIS_PDU_Intercom_Control_Communications_Class;


/* SISO-REF-010 [UID 417] */
typedef enum
{
     DIS_PDU_DIS_LIVE_ENTITY_SUBPROTOCOL_NUMBER_NO_SUBPROTOCOL = 0
}  DIS_PDU_DIS_Live_Entity_Subprotocol_Number;

/* SISO-REF-010 [UID 418] */
typedef enum
{
     DIS_PDU_MINEFIELD_APPEARANCE_MINEFIELD_TYPE_MIXED_ANTI_PERSONNEL_AND_ANTI_TANK_MINEFIELD = 0,
     DIS_PDU_MINEFIELD_APPEARANCE_MINEFIELD_TYPE_PURE_ANTI_PERSONNEL_MINEFIELD = 1,
     DIS_PDU_MINEFIELD_APPEARANCE_MINEFIELD_TYPE_PURE_ANTI_TANK_MINEFIELD = 2
}  DIS_PDU_Minefield_Appearance_Minefield_Type;


/* SISO-REF-010 [UID 419] */
typedef enum
{
     DIS_PDU_MINEFIELD_APPEARANCE_ACTIVE_STATUS_ACTIVE = 0,
     DIS_PDU_MINEFIELD_APPEARANCE_ACTIVE_STATUS_INACTIVE = 1
}  DIS_PDU_Minefield_Appearance_Active_Status;

/* SISO-REF-010 [UID 420] */
typedef enum
{
     DIS_PDU_MINEFIELD_APPEARANCE_LANE_MINEFIELD_HAS_ACTIVE_LANE = 0,
     DIS_PDU_MINEFIELD_APPEARANCE_LANE_MINEFIELD_HAS_AN_INACTIVE_LANE = 1
}  DIS_PDU_Minefield_Appearance_Lane;


/* SISO-REF-010 [UID 421] */
typedef enum
{
     DIS_PDU_MINEFIELD_APPEARANCE_STATE_ACTIVE = 0,
     DIS_PDU_MINEFIELD_APPEARANCE_STATE_DEACTIVATED = 1
}  DIS_PDU_Minefield_Appearance_State;

/* SISO-REF-010 [UID 422] */
typedef enum
{
     DIS_PDU_MINEFIELD_FUSING_FUSE_TYPE_NO_FUSE = 0,
     DIS_PDU_MINEFIELD_FUSING_FUSE_TYPE_OTHER = 1,
     DIS_PDU_MINEFIELD_FUSING_FUSE_TYPE_PRESSURE = 2,
     DIS_PDU_MINEFIELD_FUSING_FUSE_TYPE_MAGNETIC = 3,
     DIS_PDU_MINEFIELD_FUSING_FUSE_TYPE_TILT_ROD = 4,
     DIS_PDU_MINEFIELD_FUSING_FUSE_TYPE_COMMAND = 5,
     DIS_PDU_MINEFIELD_FUSING_FUSE_TYPE_TRIP_WIRE = 6
}  DIS_PDU_Minefield_Fusing_Fuse_Type;


/* SISO-REF-010 [UID 423] */
typedef enum
{
     DIS_PDU_MODE_5_LOCATION_ERRORS_NO_LOCATION_ERRORS = 0,
     DIS_PDU_MODE_5_LOCATION_ERRORS_IFF_DATA_RECORD_PRESENT = 1
}  DIS_PDU_Mode_5_Location_Errors;

/* SISO-REF-010 [UID 424] */
typedef enum
{
     DIS_PDU_MINEFIELD_PAINT_SCHEME_ALGAE_NONE = 0,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_ALGAE_LIGHT = 1,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_ALGAE_MODERATE = 2,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_ALGAE_HEAVY = 3
}  DIS_PDU_Minefield_Paint_Scheme_Algae;


/* SISO-REF-010 [UID 425] */
typedef enum
{
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_OTHER = 0,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_STANDARD = 1,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_CAMOUFLAGE_DESERT = 2,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_CAMOUFLAGE_JUNGLE = 3,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_CAMOUFLAGE_SNOW = 4,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_CAMOUFLAGE_GRAVEL = 5,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_CAMOUFLAGE_PAVEMENT = 6,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_CAMOUFLAGE_SAND = 7,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_NATURAL_WOOD = 8,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_CLEAR = 9,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_RED = 10,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_BLUE = 11,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_GREEN = 12,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_OLIVE = 13,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_WHITE = 14,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_TAN = 15,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_BLACK = 16,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_YELLOW = 17,
     DIS_PDU_MINEFIELD_PAINT_SCHEME_PAINT_SCHEME_BROWN = 18
}  DIS_PDU_Minefield_Paint_Scheme_Paint_Scheme;

/* SISO-REF-010 [UID 426] */
typedef enum
{
     DIS_PDU_COVER_SHROUD_STATUS_CLOSED = 0,
     DIS_PDU_COVER_SHROUD_STATUS_OPENING = 1,
     DIS_PDU_COVER_SHROUD_STATUS_COVER_SHROUD_BLOWN_DETACHED = 2,
     DIS_PDU_COVER_SHROUD_STATUS_OPEN_ATTACHED = 3
}  DIS_PDU_Cover_Shroud_Status;


/* SISO-REF-010 [UID 427] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_SCOOTER_SMALL = 1,
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_SPORT_STREET_MID_SIZE = 2,
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_CRUISER_LARGE = 3,
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_DIRT_BIKE = 4,
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_STANDARD = 5,
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_TOURING = 6,
     DIS_PDU_PLATFORM_LAND_MOTORCYCLE_SUBCATEGORIES_DUAL_PURPOSE = 7
}  DIS_PDU_Platform_Land_Motorcycle_Subcategories;

/* SISO-REF-010 [UID 428] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_GENERIC = 10,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_GENERIC_MINI_MICROCAR_TINY = 11,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_GENERIC_ECONOMY_COMPACT_SMALL = 12,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_GENERIC_INTERMEDIATE_STANDARD_MEDIUM = 13,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_GENERIC_FULL_PREMIUM_LUXURY_LARGE = 14,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_GENERIC_OVERSIZE = 15,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_2_DOOR_COUPE = 20,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_2_DOOR_COUPE_MINI_MICROCAR_TINY = 21,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_2_DOOR_COUPE_ECONOMY_COMPACT_SMALL = 22,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_2_DOOR_COUPE_INTERMEDIATE_STANDARD_MEDIUM = 23,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_3_DOOR_HATCHBACK = 30,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_3_DOOR_HATCHBACK_MINI_MICROCAR_TINY = 31,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_3_DOOR_HATCHBACK_ECONOMY_COMPACT_SMALL = 32,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_4_DOOR_SEDAN = 40,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_4_DOOR_SEDAN_MINI_MICROCAR_TINY = 41,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_4_DOOR_SEDAN_ECONOMY_COMPACT_SMALL = 42,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_4_DOOR_SEDAN_INTERMEDIATE_STANDARD_MEDIUM = 43,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_4_DOOR_SEDAN_FULL_PREMIUM_LUXURY_LARGE = 44,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_4_DOOR_SEDAN_OVERSIZE = 45,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_5_DOOR_HATCHBACK = 50,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_5_DOOR_HATCHBACK_MINI_MICROCAR_TINY = 51,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_5_DOOR_HATCHBACK_ECONOMY_COMPACT_SMALL = 52,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_5_DOOR_HATCHBACK_INTERMEDIATE_STANDARD_MEDIUM = 53,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_5_DOOR_HATCHBACK_FULL_PREMIUM_LUXURY_LARGE = 54,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_WAGON = 60,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_WAGON_ECONOMY_COMPACT_SMALL = 62,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_WAGON_INTERMEDIATE_STANDARD_MEDIUM = 63,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_WAGON_FULL_PREMIUM_LUXURY_LARGE = 64,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_MINIVAN = 70,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_LIMOUSINE = 80,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_LIMOUSINE_FULL_PREMIUM_LUXURY_LARGE = 84,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_LIMOUSINE_OVERSIZE = 85,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_SPORTS_HIGH_PERFORMANCE = 90,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_CONVERTIBLE = 100,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_CONVERTIBLE_MINI_MICROCAR_TINY = 101,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_CONVERTIBLE_ECONOMY_COMPACT_SMALL = 102,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_CONVERTIBLE_INTERMEDIATE_STANDARD_MEDIUM = 103,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_CONVERTIBLE_FULL_PREMIUM_LUXURY_LARGE = 104,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_SPORTS_UTILITY_VEHICLE_SUV = 110,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_SPORTS_UTILITY_VEHICLE_SUV_ECONOMY_COMPACT_SMALL = 112,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_SPORTS_UTILITY_VEHICLE_SUV_INTERMEDIATE_STANDARD_MEDIUM = 113,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_SPORTS_UTILITY_VEHICLE_SUV_FULL_PREMIUM_LUXURY_LARGE = 114,
     DIS_PDU_PLATFORM_LAND_CAR_SUBCATEGORIES_SPORTS_UTILITY_VEHICLE_SUV_OVERSIZE = 115
}  DIS_PDU_Platform_Land_Car_Subcategories;

/* SISO-REF-010 [UID 429] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_COMMUTER_FLAT_NOSE = 1,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_COMMUTER_SNOUT_NOSE = 2,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_SHUTTLE = 3,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_DOUBLE_DECKER = 4,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_GUIDED = 5,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_KNEELING = 6,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_MIDIBUS = 7,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_MINIBUS = 8,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_MINI_WHEELCHAIR = 9,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_MOTORCOACH = 10,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_PRISON_BUS = 11,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_SCHOOLBUS = 12,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_SCHOOL_WHEELCHAIR = 13,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_TOUR = 14,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_TRAM_PARKING_LOT = 15,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_TROLLEY = 16,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_AIRPORT_TRANSPORT = 17,
     DIS_PDU_PLATFORM_LAND_BUS_SUBCATEGORIES_ARTICULATED_MULTI_UNIT = 18
}  DIS_PDU_Platform_Land_Bus_Subcategories;

/* SISO-REF-010 [UID 430] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_PICKUP_TRUCK_MINI = 1,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_PICKUP_TRUCK_MID_SIZE = 2,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_PICKUP_TRUCK_FULL_SIZE = 3,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_PICKUP_TRUCK_CREW_CAB = 4,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_PICKUP_TRUCK_EXTENDED_CAB = 5,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_PICKUP_TRUCK_LONG_BED = 6,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_PICKUP_TRUCK_CAB_FORWARD = 7,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_CARGO_TRUCK = 10,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_CARGO_TRUCK_UP_TO_2_5_TON = 11,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_CARGO_TRUCK_UP_TO_5_TON = 12,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_CARGO_TRUCK_UP_TO_7_5_TON = 13,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_CARGO_TRUCK_UP_TO_10_TON = 14,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_CARGO_TRUCK_OVER_10_TON = 15,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_TANKER = 20,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_SEMI_TRAILER_CAB_W_O_TRAILER = 30,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_VAN = 70,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_VAN_EXTENDED = 71,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_VAN_COMPACT = 72,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_VAN_WHEELCHAIR = 73,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_VAN_DELIVERY = 74,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_DELIVERY_TRUCK = 100,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_DELIVERY_TRUCK_BOX = 101,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_DELIVERY_TRUCK_FLATBED = 102,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_DELIVERY_TRUCK_STAKE_BED = 103,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_MESS_TRUCK = 104,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_TRUCK_PALLETISED_LOAD_SYSTEM_PLS = 105,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_TRUCK_PETROLEUM_OIL_AND_LUBRICANTS_POL_PALLETISED_LOAD_SYSTEM_PLS = 106,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_TRUCK_PETROLEUM_OIL_AND_LUBRICANTS_POL_SURVEILLANCE = 107,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_REFRIGERATED_TRUCK_SMALL = 108,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_REFRIGERATED_TRUCK_MEDIUM = 109,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_REFRIGERATED_TRUCK_LARGE = 110
}  DIS_PDU_Platform_Land_Single_Unit_Cargo_Truck_Subcategories;

/* SISO-REF-010 [UID 431] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AMBULANCE_TRUCK = 1,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_FIRE_PARAMEDIC_TRUCK = 2,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AMBULANCE_ADVANCED_LIFE_SUPPORT = 3,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AMBULANCE_PICKUP_TRUCK = 4,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_FIRE_ENGINE = 10,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AERIAL_LADDER_FIRE_ENGINE = 11,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AIRPORT_FIRE_ENGINE = 12,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WILDLAND_FIRE_ENGINE = 13,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_FIRE_CHIEF = 14,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_POLICE_PADDY_WAGON = 20,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_POLICE_SWAT = 21,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_POLICE_BOMB_SQUAD = 22,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_POLICE_PICKUP_TRUCK = 23,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_HAZMAT = 30,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_NORMAL_HOOK_AND_CHAIN = 40,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_NORMAL_BOOM = 41,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_NORMAL_WHEEL_LIFT = 42,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_NORMAL_FLATBED = 43,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_NORMAL_INTEGRATED = 44,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_HEAVY_HOOK_AND_CHAIN = 45,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_HEAVY_BOOM = 46,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_HEAVY_WHEEL_LIFT = 47,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_HEAVY_FLATBED = 48,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WRECKER_HEAVY_INTEGRATED = 49,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_POSTAL_TRUCK = 60,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_STREET_SWEEPER = 70,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_STREET_SWEEPER_THREE_WHEELED = 71,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WASTE_COLLECTION_OTHER = 80,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WASTE_COLLECTION_FRONT_LOADER = 81,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WASTE_COLLECTION_REAR_LOADER = 82,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WASTE_COLLECTION_AUTOMATED_SIDE_LOADER = 83,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WASTE_COLLECTION_PNEUMATIC_COLLECTION = 84,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WASTE_COLLECTION_GRAPPLE = 85,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_UTILITY_TRUCK = 90,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_UTILITY_TRUCK_W_BOOM = 91,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AERIAL_WORK_PLATFORM_OTHER = 100,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AERIAL_WORK_PLATFORM_SCISSOR_LIFT = 101,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AERIAL_WORK_PLATFORM_TELESCOPING = 102,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_MAINTENANCE_TRUCK = 120,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_DECONTAMINATION_TRUCK = 121,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WATER_CANNON_TRUCK = 122,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_WATER_PURIFICATION_TRUCK = 123,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_SMOKE_GENERATOR_TRUCK = 124,
     DIS_PDU_PLATFORM_LAND_SINGLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_AUTO_RICKSHAW = 150
}  DIS_PDU_Platform_Land_Single_Unit_Utility_Emergency_Truck_Subcategories;

/* SISO-REF-010 [UID 432] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_MULTIPLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_MULTIPLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_TRACTOR_TRAILER = 1,
     DIS_PDU_PLATFORM_LAND_MULTIPLE_UNIT_CARGO_TRUCK_SUBCATEGORIES_TANKER = 2
}  DIS_PDU_Platform_Land_Multiple_Unit_Cargo_Truck_Subcategories;

/* SISO-REF-010 [UID 433] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_MULTIPLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_MULTIPLE_UNIT_UTILITY_EMERGENCY_TRUCK_SUBCATEGORIES_FIRE_ENGINE_HOOK_AND_LADDER = 1
}  DIS_PDU_Platform_Land_Multiple_Unit_Utility_Emergency_Truck_Subcategories;

/* SISO-REF-010 [UID 434] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TUG = 1,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_FORKLIFT = 2,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_LOADER = 3,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_LOADER_BACKHOE = 4,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CRANE_TRACTOR_MOUNTED = 5,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CRANE_WHEELED = 6,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_GRADER = 7,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_ROAD_ROLLER_OTHER = 8,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_ROAD_ROLLER_DOUBLE_DRUM_SMOOTH = 9,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_ROAD_ROLLER_SINGLE_DRUM_SMOOTH = 10,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_ROAD_ROLLER_DOUBLE_DRUM_SHEEPS = 11,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_ROAD_ROLLER_SINGLE_DRUM_SHEEPS = 12,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_ROAD_ROLLER_PNEUMATIC_TIRED = 13,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_EXCAVATOR_OTHER = 14,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_EXCAVATOR_DRAGLINE = 15,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_EXCAVATOR_LONG_REACH = 16,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_EXCAVATOR_MOBILE_TIRE = 17,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_MINI_EXCAVATOR = 18,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_EXCAVATOR_GIANT = 19,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_BULLDOZER_TRACTOR_MOUNTED = 20,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_BULLDOZER_TRACKED = 21,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_SCRAPER = 22,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_SKID_STEER = 23,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_DUMP_TRUCK_OTHER = 24,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_DUMP_TRUCK_ARTICULATED = 25,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_DUMP_TRUCK_TRANSFER = 26,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_DUMP_TRUCK_SUPER = 27,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_DUMP_TRUCK_OFF_ROAD = 28,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_PAVER = 29,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_DRILLING_MACHINE = 30,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CONCRETE_MIXER_OTHER = 31,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CONCRETE_MIXER_REAR_DISCHARGE = 32,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CONCRETE_MIXER_FRONT_DISCHARGE = 33,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CONCRETE_MIXER_SIX_AXLE = 34,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CONCRETE_MIXER_LONG_REACH_BOOM = 35,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CONCRETE_MIXER_VOLUMETRIC = 36,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRENCHER_CHAIN = 37,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRENCHER_ROCKWHEEL = 38,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_SNOWCAT = 39,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CRANE_TRACKED = 40,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CRANE_SHOVEL = 41,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_SWEEPER_ROTARY = 42,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_ROLLER_VIBRATORY_COMPACTOR = 43,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_FORK_LIFT_TRUCK = 44,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_FORK_LIFT_ROUGHT_TERRAIN = 45,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRANSLOADER = 46,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRUCK_WATER_CONSTRUCTION = 47,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRUCK_FUEL_DELIVERY = 48,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRUCK_SAWMILL = 49,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRUCK_LINE_MARKING_CONSTRUCTION = 50,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRACTOR_INDUSTRIAL = 51,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_COMPACTOR_HIGH_SPEED = 52,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRUCK_DRILLING = 53,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_TRUCK_DRILLING_SUPPORT = 54,
     DIS_PDU_PLATFORM_LAND_CONSTRUCTION_SPECIALTY_VEHICLE_SUBCATEGORIES_CRANE_CONSTRUCTION = 55
}  DIS_PDU_Platform_Land_Construction_Specialty_Vehicle_Subcategories;

/* SISO-REF-010 [UID 435] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_TRACTOR = 1,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_HARVESTER_REAPER = 2,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_SKIDDER = 3,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_FORWARDER = 4,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_LAWN_MOWER_OTHER = 5,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_LAWN_MOWER_RIDING = 6,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_LAWN_MOWER_STANDING = 7,
     DIS_PDU_PLATFORM_LAND_FARM_SPECIALTY_VEHICLE_SUBCATEGORIES_LAWN_MOWER_PUSH = 8
}  DIS_PDU_Platform_Land_Farm_Specialty_Vehicle_Subcategories;

/* SISO-REF-010 [UID 436] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_FLATBED = 1,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_CONTAINER = 2,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_CONTAINER_REFRIGERATED = 3,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_DOUBLE = 4,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_AUTO_TRANSPORT = 5,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_ARTICULATED = 6,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_TANKER = 7,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_TANKER_SMALL = 8,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_TANKER_LARGE = 9,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_TANKER_GASOLINE = 10,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_TANKER_MILK = 11,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_TANKER_WATER = 12,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_TANKER_SEPTIC = 13,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_BOAT = 14,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_BOAT_SMALL = 15,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_BOAT_LARGE = 16,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_RECREATIONAL = 17,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_RECREATIONAL_CONVENTIONAL = 18,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_RECREATIONAL_TRAVEL_EXPANDABLE = 19,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_RECREATIONAL_FIFTH_WHEEL_TRAVEL = 20,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_RECREATIONAL_FOLDING_CAMPING = 21,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_RECREATIONAL_TRUCK_CAMPER = 22,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_AEROSTAT_MOORING_PLATFORM = 23,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_HOUSEHOLD = 24,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_KITCHEN = 25,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_ULTRALIGHT_AIRCRAFT = 26,
     DIS_PDU_PLATFORM_LAND_TRAILER_SUBCATEGORIES_TRAILER_HEAVY_EQUIPMENT = 27
}  DIS_PDU_Platform_Land_Trailer_Subcategories;

/* SISO-REF-010 [UID 437] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_ATV_2X4 = 1,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_ATV_4X4 = 2,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_ATV_6X6 = 3,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_ATV_3_WHEELED = 4,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_TOY_OTHER = 5,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_TOY_CAR = 6,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_TOY_ATV = 7,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_GOLF_CART = 8,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_SNOWMOBILE = 9,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_RECREATIONAL_VEHICLE = 10,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_RECREATIONAL_VEHICLE_TYPE_A_MOTORHOME = 11,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_RECREATIONAL_VEHICLE_TYPE_B_MOTORHOME = 12,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_RECREATIONAL_VEHICLE_TYPE_C_MOTORHOME = 13,
     DIS_PDU_PLATFORM_LAND_RECREATIONAL_SUBCATEGORIES_CONVERSION_VAN = 14
}  DIS_PDU_Platform_Land_Recreational_Subcategories;

/* SISO-REF-010 [UID 438] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_UNICYCLE = 1,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_BICYCLE = 2,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_BICYCLE_MOUNTAIN = 3,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_BICYCLE_RACING = 4,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_TRICYCLE = 5,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_QUADRICYCLE = 6,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_RICKSHAW_TWO_PERSON = 7,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_RICKSHAW_ONE_PERSON = 8,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_TANDEM_BICYCLE = 9,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_CYCLE_TRAILER = 10,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_CYCLE_SIDECAR = 11,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_SLED = 12,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_SKIS = 13,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_SNOWBOARD = 14,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_SKATEBOARD = 15,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_SKATES = 16,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_SKATES_IN_LINE = 17,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_WAGON_CART = 18,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_DOLLY = 19,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_HANDTRUCK = 20,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_PUSH_CART = 21,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_WHEELBARROW = 22,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_KICK_SCOOTER = 23,
     DIS_PDU_PLATFORM_LAND_NON_MOTORIZED_SUBCATEGORIES_WHEELCHAIR = 24
}  DIS_PDU_Platform_Land_Non_motorized_Subcategories;

/* SISO-REF-010 [UID 439] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_ENGINE_LOCOMOTIVE = 1,
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_BOX_CAR = 2,
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_TANKER = 3,
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_FLATCAR = 4,
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_CABOOSE = 5,
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_PASSENGER_CAR = 6,
     DIS_PDU_PLATFORM_LAND_TRAINS_SUBCATEGORIES_HOPPER = 7
}  DIS_PDU_Platform_Land_Trains_Subcategories;

/* SISO-REF-010 [UID 440] */
typedef enum
{
     DIS_PDU_PLATFORM_LAND_UTILITY_EMERGENCY_CAR_SUBCATEGORIES_OTHER = 0,
     DIS_PDU_PLATFORM_LAND_UTILITY_EMERGENCY_CAR_SUBCATEGORIES_AMBULANCE_CAR = 1,
     DIS_PDU_PLATFORM_LAND_UTILITY_EMERGENCY_CAR_SUBCATEGORIES_POLICE_CAR = 2,
     DIS_PDU_PLATFORM_LAND_UTILITY_EMERGENCY_CAR_SUBCATEGORIES_POLICE_CHIEF = 3,
     DIS_PDU_PLATFORM_LAND_UTILITY_EMERGENCY_CAR_SUBCATEGORIES_HEARSE = 4,
     DIS_PDU_PLATFORM_LAND_UTILITY_EMERGENCY_CAR_SUBCATEGORIES_TAXI = 5
}  DIS_PDU_Platform_Land_Utility_Emergency_Car_Subcategories;

/* SISO-REF-010 [UID 441] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_PASSENGER_VESSEL_SUBCATEGORIES_CRUISE_SHIP = 1,
     DIS_PDU_PLATFORM_SURFACE_PASSENGER_VESSEL_SUBCATEGORIES_CRUISE_FERRY = 2,
     DIS_PDU_PLATFORM_SURFACE_PASSENGER_VESSEL_SUBCATEGORIES_HIGH_SPEED_FERRY = 3,
     DIS_PDU_PLATFORM_SURFACE_PASSENGER_VESSEL_SUBCATEGORIES_FERRY = 4,
     DIS_PDU_PLATFORM_SURFACE_PASSENGER_VESSEL_SUBCATEGORIES_OCEAN_LINER = 5
}  DIS_PDU_Platform_Surface_Passenger_Vessel_Subcategories;

/* SISO-REF-010 [UID 442] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_DRY_CARGO_SHIP_SUBCATEGORIES_COMMON_DRY_CARGO_SHIP = 1,
     DIS_PDU_PLATFORM_SURFACE_DRY_CARGO_SHIP_SUBCATEGORIES_DRY_BULK_CARGO_SHIP = 2,
     DIS_PDU_PLATFORM_SURFACE_DRY_CARGO_SHIP_SUBCATEGORIES_CONTAINER_SHIP = 3,
     DIS_PDU_PLATFORM_SURFACE_DRY_CARGO_SHIP_SUBCATEGORIES_REEFER_SHIP_REFRIGERATOR_SHIP = 4,
     DIS_PDU_PLATFORM_SURFACE_DRY_CARGO_SHIP_SUBCATEGORIES_RO_RO_SHIP_ROLL_ON_ROLL_OFF_SHIP = 5,
     DIS_PDU_PLATFORM_SURFACE_DRY_CARGO_SHIP_SUBCATEGORIES_BARGE = 6,
     DIS_PDU_PLATFORM_SURFACE_DRY_CARGO_SHIP_SUBCATEGORIES_HEAVY_LIFT_SHIP = 7
}  DIS_PDU_Platform_Surface_Dry_Cargo_Ship_Subcategories;

/* SISO-REF-010 [UID 443] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_LIQUID_PETROLEUM_GAS_LPG_TANKER = 1,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_CHEMICAL_TANKER = 2,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_LIQUID_NATURAL_GAS_LNG_TANKER = 3,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_COASTAL_TRADING_VESSEL_COASTER = 4,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_CRUDE_OIL_TANKER_UP_TO_159_999_DWT = 5,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_LIQUID_BULK_TANKER = 6,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_VERY_LARGE_CRUDE_CARRIER_160_000_319_999_DWT = 7,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_ULTRA_LARGE_CRUDE_CARRIER_320_000_549_999_DWT = 8,
     DIS_PDU_PLATFORM_SURFACE_TANKER_SUBCATEGORIES_CONDENSATE_STORAGE_TANKER = 9
}  DIS_PDU_Platform_Surface_Tanker_Subcategories;

/* SISO-REF-010 [UID 444] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_PLATFORM_SUPPLY_VESSEL = 1,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_TENDER_VESSEL = 2,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_TUGBOAT = 3,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_DIVE_SUPPORT_VESSEL = 4,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_FIREBOAT = 5,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_WELL_STIMULATION_VESSEL_WSV = 6,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_ANCHOR_HANDLING_TUG_SUPPLY_VESSEL_AHTS = 7,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_OFFSHORE_CONSTRUCTION_VESSEL_OCV = 8,
     DIS_PDU_PLATFORM_SURFACE_SUPPORT_VESSEL_SUBCATEGORIES_EMERGENCY_RESPONSE_AND_RESCUE_VESSEL_ERRV = 9
}  DIS_PDU_Platform_Surface_Support_Vessel_Subcategories;

/* SISO-REF-010 [UID 445] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_MOTORBOAT_SUBCATEGORIES_SMALL_MOTORBOAT_UP_TO_26FT_7_9M = 1,
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_MOTORBOAT_SUBCATEGORIES_MEDIUM_MOTORBOAT_UP_TO_39FT_11_9M = 2,
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_MOTORBOAT_SUBCATEGORIES_LARGE_MOTORBOAT_UP_TO_65FT_19_8M = 3,
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_MOTORBOAT_SUBCATEGORIES_VERY_LARGE_MOTORBOAT_GREATER_THAN_65FT_19_8M = 4
}  DIS_PDU_Platform_Surface_Private_Motorboat_Subcategories;

/* SISO-REF-010 [UID 446] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_SAILBOAT_SUBCATEGORIES_SMALL_SAILBOAT_UP_TO_26FT_7_9M = 1,
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_SAILBOAT_SUBCATEGORIES_MEDIUM_SAILBOAT_UP_TO_39FT_11_9M = 2,
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_SAILBOAT_SUBCATEGORIES_LARGE_SAILBOAT_UP_TO_65FT_19_8M = 3,
     DIS_PDU_PLATFORM_SURFACE_PRIVATE_SAILBOAT_SUBCATEGORIES_VERY_LARGE_SAILBOAT_GREATER_THAN_65FT_19_8M = 4
}  DIS_PDU_Platform_Surface_Private_Sailboat_Subcategories;

/* SISO-REF-010 [UID 447] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_FISHING_VESSEL_SUBCATEGORIES_SMALL_FISHING_VESSEL_UP_TO_26FT_7_9M = 1,
     DIS_PDU_PLATFORM_SURFACE_FISHING_VESSEL_SUBCATEGORIES_MEDIUM_FISHING_VESSEL_UP_TO_65FT_19_8M = 2,
     DIS_PDU_PLATFORM_SURFACE_FISHING_VESSEL_SUBCATEGORIES_LARGE_FISHING_VESSEL_GREATER_THAN_65FT_19_8M = 3,
     DIS_PDU_PLATFORM_SURFACE_FISHING_VESSEL_SUBCATEGORIES_FISH_PROCESSING_VESSEL = 4,
     DIS_PDU_PLATFORM_SURFACE_FISHING_VESSEL_SUBCATEGORIES_MASTED_FISHING_VESSEL = 5
}  DIS_PDU_Platform_Surface_Fishing_Vessel_Subcategories;

/* SISO-REF-010 [UID 448] */
typedef enum
{
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_GO_FAST_BOAT = 1,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_RESEARCH_VESSEL = 2,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_HYDROFOIL_VESSEL = 3,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_CABLE_LAYER_VESSEL = 4,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_DREDGER_VESSEL = 5,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_JUNK_DHOW_VESSEL = 6,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_CATAMARAN = 7,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_PONTOON = 8,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_PERSONAL_WATER_CRAFT = 9,
     DIS_PDU_PLATFORM_SURFACE_OTHER_VESSELS_SUBCATEGORIES_REFUGEE_RAFT = 10
}  DIS_PDU_Platform_Surface_Other_Vessels_Subcategories;

/* SISO-REF-010 [UID 449] */
typedef enum
{
     DIS_PDU_CRYPTO_KEY_ID_CRYPTO_MODE_BASEBAND = 0,
     DIS_PDU_CRYPTO_KEY_ID_CRYPTO_MODE_DIPHASE = 1
}  DIS_PDU_Crypto_Key_ID_Crypto_Mode;

/* SISO-REF-010 [UID 463] */
typedef enum
{
     DIS_PDU_COLOR_NOT_SPECIFIED = 0,
     DIS_PDU_COLOR_WHITE_VGA = 1,
     DIS_PDU_COLOR_RED_VGA = 2,
     DIS_PDU_COLOR_YELLOW_VGA = 3,
     DIS_PDU_COLOR_LIME_VGA = 4,
     DIS_PDU_COLOR_CYAN_VGA = 5,
     DIS_PDU_COLOR_BLUE_VGA = 6,
     DIS_PDU_COLOR_MAGENTA_VGA = 7,
     DIS_PDU_COLOR_GREY_VGA = 8,
     DIS_PDU_COLOR_SILVER_VGA = 9,
     DIS_PDU_COLOR_MAROON_VGA = 10,
     DIS_PDU_COLOR_OLIVE_VGA = 11,
     DIS_PDU_COLOR_GREEN_VGA = 12,
     DIS_PDU_COLOR_TEAL_VGA = 13,
     DIS_PDU_COLOR_NAVY_VGA = 14,
     DIS_PDU_COLOR_PURPLE_VGA = 15,
     DIS_PDU_COLOR_RESERVED_16 = 16,
     DIS_PDU_COLOR_RESERVED_17 = 17,
     DIS_PDU_COLOR_RESERVED_18 = 18,
     DIS_PDU_COLOR_RESERVED_19 = 19,
     DIS_PDU_COLOR_BLACK = 20,
     DIS_PDU_COLOR_NAVY = 21,
     DIS_PDU_COLOR_DARK_BLUE = 22,
     DIS_PDU_COLOR_MEDIUM_BLUE = 23,
     DIS_PDU_COLOR_BLUE = 24,
     DIS_PDU_COLOR_DARK_GREEN = 25,
     DIS_PDU_COLOR_GREEN = 26,
     DIS_PDU_COLOR_TEAL = 27,
     DIS_PDU_COLOR_DARK_CYAN = 28,
     DIS_PDU_COLOR_DEEP_SKY_BLUE = 29,
     DIS_PDU_COLOR_DARK_TURQUOISE = 30,
     DIS_PDU_COLOR_MEDIUM_SPRING_GREEN = 31,
     DIS_PDU_COLOR_LIME = 32,
     DIS_PDU_COLOR_SPRING_GREEN = 33,
     DIS_PDU_COLOR_CYAN = 34,
     DIS_PDU_COLOR_MIDNIGHT_BLUE = 35,
     DIS_PDU_COLOR_DODGER_BLUE = 36,
     DIS_PDU_COLOR_LIGHT_SEA_GREEN = 37,
     DIS_PDU_COLOR_FOREST_GREEN = 38,
     DIS_PDU_COLOR_SEA_GREEN = 39,
     DIS_PDU_COLOR_DARK_SLATE_GRAY = 40,
     DIS_PDU_COLOR_LIME_GREEN = 41,
     DIS_PDU_COLOR_MEDIUM_SEA_GREEN = 42,
     DIS_PDU_COLOR_TURQUOISE = 43,
     DIS_PDU_COLOR_ROYAL_BLUE = 44,
     DIS_PDU_COLOR_STEEL_BLUE = 45,
     DIS_PDU_COLOR_DARK_SLATE_BLUE = 46,
     DIS_PDU_COLOR_MEDIUM_TURQUOISE = 47,
     DIS_PDU_COLOR_INDIGO = 48,
     DIS_PDU_COLOR_DARK_OLIVE_GREEN = 49,
     DIS_PDU_COLOR_CADET_BLUE = 50,
     DIS_PDU_COLOR_CORNFLOWER_BLUE = 51,
     DIS_PDU_COLOR_MEDIUM_AQUAMARINE = 52,
     DIS_PDU_COLOR_DIM_GRAY = 53,
     DIS_PDU_COLOR_SLATE_BLUE = 54,
     DIS_PDU_COLOR_OLIVE_DRAB = 55,
     DIS_PDU_COLOR_SLATE_GRAY = 56,
     DIS_PDU_COLOR_LIGHT_SLATE_GRAY = 57,
     DIS_PDU_COLOR_MEDIUM_SLATE_BLUE = 58,
     DIS_PDU_COLOR_LAWN_GREEN = 59,
     DIS_PDU_COLOR_CHARTREUSE = 60,
     DIS_PDU_COLOR_AQUAMARINE = 61,
     DIS_PDU_COLOR_MAROON = 62,
     DIS_PDU_COLOR_PURPLE = 63,
     DIS_PDU_COLOR_OLIVE = 64,
     DIS_PDU_COLOR_GRAY = 65,
     DIS_PDU_COLOR_GREY = 66,
     DIS_PDU_COLOR_SKY_BLUE = 67,
     DIS_PDU_COLOR_LIGHT_SKY_BLUE = 68,
     DIS_PDU_COLOR_BLUE_VIOLET = 69,
     DIS_PDU_COLOR_DARK_RED = 70,
     DIS_PDU_COLOR_DARK_MAGENTA = 71,
     DIS_PDU_COLOR_SADDLE_BROWN = 72,
     DIS_PDU_COLOR_DARK_SEA_GREEN = 73,
     DIS_PDU_COLOR_LIGHT_GREEN = 74,
     DIS_PDU_COLOR_MEDIUM_PURPLE = 75,
     DIS_PDU_COLOR_DARK_VIOLET = 76,
     DIS_PDU_COLOR_PALE_GREEN = 77,
     DIS_PDU_COLOR_DARK_ORCHID = 78,
     DIS_PDU_COLOR_YELLOW_GREEN = 79,
     DIS_PDU_COLOR_SIENNA = 80,
     DIS_PDU_COLOR_BROWN = 81,
     DIS_PDU_COLOR_DARK_GRAY = 82,
     DIS_PDU_COLOR_LIGHT_BLUE = 83,
     DIS_PDU_COLOR_GREEN_YELLOW = 84,
     DIS_PDU_COLOR_PALE_TURQUOISE = 85,
     DIS_PDU_COLOR_LIGHT_STEEL_BLUE = 86,
     DIS_PDU_COLOR_POWDER_BLUE = 87,
     DIS_PDU_COLOR_FIRE_BRICK = 88,
     DIS_PDU_COLOR_DARK_GOLDEN_ROD = 89,
     DIS_PDU_COLOR_MEDIUM_ORCHID = 90,
     DIS_PDU_COLOR_ROSY_BROWN = 91,
     DIS_PDU_COLOR_DARK_KHAKI = 92,
     DIS_PDU_COLOR_SILVER = 93,
     DIS_PDU_COLOR_MEDIUM_VIOLET_RED = 94,
     DIS_PDU_COLOR_INDIAN_RED = 95,
     DIS_PDU_COLOR_PERU = 96,
     DIS_PDU_COLOR_CHOCOLATE = 97,
     DIS_PDU_COLOR_TAN = 98,
     DIS_PDU_COLOR_LIGHT_GRAY = 99,
     DIS_PDU_COLOR_PALE_VIOLET_RED = 100,
     DIS_PDU_COLOR_THISTLE = 101,
     DIS_PDU_COLOR_ORCHID = 102,
     DIS_PDU_COLOR_GOLDEN_ROD = 103,
     DIS_PDU_COLOR_CRIMSON = 104,
     DIS_PDU_COLOR_GAINSBORO = 105,
     DIS_PDU_COLOR_PLUM = 106,
     DIS_PDU_COLOR_BURLY_WOOD = 107,
     DIS_PDU_COLOR_LIGHT_CYAN = 108,
     DIS_PDU_COLOR_LAVENDER = 109,
     DIS_PDU_COLOR_DARK_SALMON = 110,
     DIS_PDU_COLOR_VIOLET = 111,
     DIS_PDU_COLOR_PALE_GOLDEN_ROD = 112,
     DIS_PDU_COLOR_LIGHT_CORAL = 113,
     DIS_PDU_COLOR_KHAKI = 114,
     DIS_PDU_COLOR_ALICE_BLUE = 115,
     DIS_PDU_COLOR_HONEY_DEW = 116,
     DIS_PDU_COLOR_AZURE = 117,
     DIS_PDU_COLOR_SANDY_BROWN = 118,
     DIS_PDU_COLOR_WHEAT = 119,
     DIS_PDU_COLOR_BEIGE = 120,
     DIS_PDU_COLOR_WHITE_SMOKE = 121,
     DIS_PDU_COLOR_MINT_CREAM = 122,
     DIS_PDU_COLOR_GHOST_WHITE = 123,
     DIS_PDU_COLOR_SALMON = 124,
     DIS_PDU_COLOR_ANTIQUE_WHITE = 125,
     DIS_PDU_COLOR_LINEN = 126,
     DIS_PDU_COLOR_LIGHT_GOLDEN_ROD_YELLOW = 127,
     DIS_PDU_COLOR_OLD_LACE = 128,
     DIS_PDU_COLOR_RED = 129,
     DIS_PDU_COLOR_FUCHSIA = 130,
     DIS_PDU_COLOR_MAGENTA = 131,
     DIS_PDU_COLOR_DEEP_PINK = 132,
     DIS_PDU_COLOR_ORANGE_RED = 133,
     DIS_PDU_COLOR_TOMATO = 134,
     DIS_PDU_COLOR_HOT_PINK = 135,
     DIS_PDU_COLOR_CORAL = 136,
     DIS_PDU_COLOR_DARK_ORANGE = 137,
     DIS_PDU_COLOR_LIGHT_SALMON = 138,
     DIS_PDU_COLOR_ORANGE = 139,
     DIS_PDU_COLOR_LIGHT_PINK = 140,
     DIS_PDU_COLOR_PINK = 141,
     DIS_PDU_COLOR_GOLD = 142,
     DIS_PDU_COLOR_PEACH_PUFF = 143,
     DIS_PDU_COLOR_NAVAJO_WHITE = 144,
     DIS_PDU_COLOR_MOCCASIN = 145,
     DIS_PDU_COLOR_BISQUE = 146,
     DIS_PDU_COLOR_MISTY_ROSE = 147,
     DIS_PDU_COLOR_BLANCHED_ALMOND = 148,
     DIS_PDU_COLOR_PAPAYA_WHIP = 149,
     DIS_PDU_COLOR_LAVENDER_BLUSH = 150,
     DIS_PDU_COLOR_SEA_SHELL = 151,
     DIS_PDU_COLOR_CORNSILK = 152,
     DIS_PDU_COLOR_LEMON_CHIFFON = 153,
     DIS_PDU_COLOR_FLORAL_WHITE = 154,
     DIS_PDU_COLOR_SNOW = 155,
     DIS_PDU_COLOR_YELLOW = 156,
     DIS_PDU_COLOR_LIGHT_YELLOW = 157,
     DIS_PDU_COLOR_IVORY = 158,
     DIS_PDU_COLOR_WHITE = 159
}  DIS_PDU_Color;

/* SISO-REF-010 [UID 464] */
typedef enum
{
     DIS_PDU_BUILDING_PAINT_SCHEME_DEFAULT = 0
}  DIS_PDU_Building_Paint_Scheme;

/* SISO-REF-010 [UID 465] */
typedef enum
{
     DIS_PDU_SEASON_SUMMER = 0,
     DIS_PDU_SEASON_WINTER = 1,
     DIS_PDU_SEASON_SPRING = 2,
     DIS_PDU_SEASON_AUTUMN = 3
}  DIS_PDU_Season;

/* SISO-REF-010 [UID 466] */
typedef enum
{
     DIS_PDU_MATERIAL_NOT_SPECIFIED = 0,
     DIS_PDU_MATERIAL_PLASTIC = 1,
     DIS_PDU_MATERIAL_RUBBER = 2,
     DIS_PDU_MATERIAL_ROAD = 3
}  DIS_PDU_Material;

/* SISO-REF-010 [UID 467] */
typedef enum
{
     DIS_PDU_LINK_11_11B_FIDELITY_LEVEL_FIDELITY_LEVEL_0 = 0,
     DIS_PDU_LINK_11_11B_FIDELITY_LEVEL_FIDELITY_LEVEL_1 = 1,
     DIS_PDU_LINK_11_11B_FIDELITY_LEVEL_FIDELITY_LEVEL_2 = 2
}  DIS_PDU_Link_11_11B_Fidelity_Level;

/* SISO-REF-010 [UID 468] */
typedef enum
{
     DIS_PDU_LINK_11_TERMINAL_MODE_NO_STATEMENT = 0,
     DIS_PDU_LINK_11_TERMINAL_MODE_NETWORK_CONTROL_STATION = 1,
     DIS_PDU_LINK_11_TERMINAL_MODE_PICKET = 2
}  DIS_PDU_Link_11_Terminal_Mode;

/* SISO-REF-010 [UID 470] */
typedef enum
{
     DIS_PDU_LINK_11_MODE_OF_OPERATION_NO_STATEMENT = 0,
     DIS_PDU_LINK_11_MODE_OF_OPERATION_NET_SYNC = 1,
     DIS_PDU_LINK_11_MODE_OF_OPERATION_NET_TEST = 2,
     DIS_PDU_LINK_11_MODE_OF_OPERATION_ROLL_CALL = 3,
     DIS_PDU_LINK_11_MODE_OF_OPERATION_SHORT_BROADCAST = 4,
     DIS_PDU_LINK_11_MODE_OF_OPERATION_BROADCAST = 5
}  DIS_PDU_Link_11_Mode_of_Operation;

/* SISO-REF-010 [UID 471] */
typedef enum
{
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_IRANIAN_WEAPONS_MISAGH_2 = 1,
     DIS_PDU_LIFE_FORMS_SUBCATEGORY_IRANIAN_WEAPONS_RBS_70 = 2
}  DIS_PDU_Life_Forms_Subcategory_Iranian_Weapons;

/* SISO-REF-010 [UID 472] */
typedef enum
{
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_CONVENTIONAL_ARMED_FORCES = 10,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_ARMY = 11,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_NAVAL_INFANTRY_MARINES = 12,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_AIR_FORCE = 13,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_NAVY = 14,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_COAST_GUARD = 15,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_UNITED_NATIONS = 16,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_SPECIAL_OPERATIONS_FORCES_SOF = 30,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_LAW_ENFORCEMENT = 50,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_NON_MILITARY_NATIONAL_GOVERNMENT_AGENCIES = 70,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_REGIONAL_LOCAL_FORCES = 90,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_IRREGULAR_FORCES = 100,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_TERRORIST_COMBATANT = 101,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_INSURGENT = 102,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_PARAMILITARY_FORCES = 110,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_HUMANITARIAN_ORGANIZATIONS = 120,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_CIVILIAN = 130,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_EMERGENCY_MEDICAL_TECHNICIAN_EMT = 131,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_FIREFIGHTER = 132,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_PRESS = 133,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_MAMMAL = 200,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_REPTILE = 201,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_AMPHIBIAN = 202,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_INSECT = 203,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_ARACHNID = 204,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_MOLLUSK = 205,
     DIS_PDU_LIFE_FORM_LAND_CATEGORIES_MARSUPIAL = 206
}  DIS_PDU_Life_Form_Land_Categories;

/* SISO-REF-010 [UID 473] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_NONE = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_WEAPON_NON_SPECIFIC = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_ASSAULT_RIFLES = 5,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_HIGH_POWER_RIFLES = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_SNIPER_RIFLES = 15,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_ANTI_MATERIEL_RIFLE_AMR = 17,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_SUB_MACHINE_GUNS = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_SHOT_GUNS = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_GRENADE_LAUNCHERS = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_MACHINE_GUNS = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_GRENADE_LAUNCHING_MACHINE_GUN = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_ANTI_TANK_ROCKETS = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_ANTI_TANK_MISSILES = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_ANTI_TANK_GUNS = 55,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_FLAME_ROCKETS = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_FLAME_THROWERS = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_ROCKET_LAUNCHERS = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_MORTARS = 75,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_HAND_GUNS = 80,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_MANPADS = 85,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_RECOILLESS_RIFLES = 90,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_DRONE_GUNS = 95,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_EQUIPMENT_NON_SPECIFIC = 150,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_SENSORS = 151,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_SIGNAL_SENSOR = 152,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_LASERS = 153,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_ANIMAL_COMPANION = 160,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_PERSONAL_ELECTRONICS = 171,
     DIS_PDU_LIFE_FORM_HUMAN_SUBCATEGORY_EQUIPMENT_CLASS_LOGISTICS_EQUIPMENT = 172
}  DIS_PDU_Life_Form_Human_Subcategory_Equipment_Class;

/* SISO-REF-010 [UID 474] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_4_5MM_INTERDYNAMICS_MKR = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_45MM_AK_74 = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_45MM_AKS_74 = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_45MM_AK_74M = 12,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_45MM_KBK_WZ_1988_TANTAL = 13,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_45MM_FORT_221 = 14,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_45MM_5_56MM_7_62MM_IPI_MALYUK = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AK_101 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_DIEMACO_C7 = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_COLT_CANADA_C8_CARBINE = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_GIAT_FAMAS_G2 = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_FN_FNC = 34,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_HK_G36 = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_IMI_GALIL = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_INSAS = 37,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_DAEWOO_K1 = 38,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_DAEWOO_K2 = 39,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_M16A1 = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_M16A2_A3_A4 = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_COLT_M4 = 42,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_COLT_M4_SPECIAL_OPERATIONS_PECULIAR_MODIFICATION_SOPMOD = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_RUGER_MINI_14 = 44,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_ENFIELD_SA_80A2 = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_PINDAD_SS1_V1 = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_PINDAD_SS1_V2 = 47,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_PINDAD_SS1_V3 = 48,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_STEYR_AUG_A1 = 49,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_T65 = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_T91 = 51,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_TAVOR_TAR_21 = 52,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_TYPE_CQ_M311 = 53,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_DAEWOO_K11 = 54,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_F88 = 55,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_F88_GLA = 56,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_F88_S_A1 = 57,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_F88_S_A2 = 58,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_F88_C = 59,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_F88_S_A1C = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_F88_S_A1_LTR = 61,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_AUSTEYR_EF88 = 62,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_BUSHMASTER_XM15 = 63,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_HK416 = 64,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_F90 = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_F90_G = 66,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_F90M = 67,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_F90M_G = 68,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_F90CQB = 69,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_MK17_SCAR_L = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_CZ_805_BREN = 71,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_FORT_224 = 72,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_FORT_227 = 73,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_ZBROYAR_UAR_15 = 74,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_56MM_ZBROYAR_Z_15 = 75,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_5_8MM_QBZ_95_TYPE_95 = 100,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_AK_103 = 110,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_AK_104 = 111,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_AK_47 = 112,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_AKM = 113,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_AKS_47 = 114,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_HK_G3A3 = 115,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_IMI_GALIL = 116,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_KLS = 117,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_SKS = 118,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_TYPE_56 = 119,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_TYPE_63_68 = 120,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_TYPE_81 = 121,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_MK17_SCAR_H = 122,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_FORT_228 = 123,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_FORT_229 = 124,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_ZBROYAR_Z_008_NON_SNIPER_VARIANT = 125,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_ZBROYAR_Z_10 = 126,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_MAYAK_MZ_10 = 127,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_7_62MM_STILETTO_SYSTEMS_STL_016_300_WINCHESTER = 128,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_6_8MM_M7_NEXT_GENERATION_SQUAD_WEAPON_NGSW = 150,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ASSAULT_RIFLES_8MM_LEBEL_M16 = 240
}  DIS_PDU_Life_Form_Human_Specific_Assault_Rifles;

/* SISO-REF-010 [UID 475] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_62MM_M14 = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_62MM_REMINGTON_700 = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_62MM_SIG_SAUER_SSG_2000 = 12,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_62MM_STONER_SR_25 = 13,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_62MM_MOSIN_NAGANT_MODEL_1891_30 = 14,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_62MM_HK417 = 15,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_62MM_HK417_16_RECCE = 16,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_65MM_BAR_M1918 = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HIGH_POWER_RIFLES_7_65MM_M1_GARAND = 51
}  DIS_PDU_Life_Form_Human_Specific_High_Power_Rifles;

/* SISO-REF-010 [UID 476] */
typedef enum
{
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_U_S_ARMY_USA = 11,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_U_S_MARINE_CORPS_USMC = 12,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_U_S_AIR_FORCE_USAF = 13,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_U_S_NAVY_USN = 14,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_U_S_COAST_GUARD_USCG = 15,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_SPECIAL_FORCES = 30,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_RANGERS = 31,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_ARSOF = 32,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_FORCE_RECONNAISSANCE = 33,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_NAVY_SEAL = 34,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_AFSOF = 35,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_DELTA_FORCE = 36,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_FEDERAL_BUREAU_OF_INVESTIGATION_FBI = 51,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_CENTRAL_INTELLIGENCE_AGENCY_CIA = 52,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_DEPARTMENT_OF_HOMELAND_SECURITY_DHS = 53,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_BUREAU_OF_ALCOHOL_TOBACCO_FIREARMS_AND_EXPLOSIVES_ATF = 54,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_U_S_SECRET_SERVICE_USSS = 55,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_U_S_MARSHAL = 56,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_LAW_ENFORCEMENT_NON_MILITARY = 70,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_STATE_POLICE_HIGHWAY_PATROL = 71,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_COUNTY_SHERIFF_POLICE = 72,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_MUNICIPAL_POLICE = 73,
     DIS_PDU_LIFE_FORM_CATEGORIES_U_S_RED_CROSS = 124
}  DIS_PDU_Life_Form_Categories_U_S;

/* SISO-REF-010 [UID 477] */
typedef enum
{
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_NOT_SPECIFIED_MALE = 0,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ASIAN_MALE = 1,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_PACIFIC_ISLANDER_MALE = 2,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_BLACK_MALE = 3,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_EAST_ASIAN_MALE = 4,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_HISPANIC_MALE = 5,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_WHITE_MALE = 6,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ARAB_MALE = 7,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_HOMOGENOUS_COUNTRY_CODE_MALE = 8,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_INDIGENOUS_COUNTRY_CODE_MALE = 9,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_INFANT_0_MONTHS_1_YEAR_MALE = 10,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_TODDLER_1_3_YEARS_MALE = 20,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_CHILD_3_10_YEARS_MALE = 30,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ADOLESCENT_10_12_YEARS_MALE = 40,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_TEENAGER_13_16_YEARS_MALE = 50,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_YOUNG_ADULT_17_25_YEARS_MALE = 60,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ADULT_25_55_YEARS_MALE = 70,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_SENIOR_ADULT_55_70_YEARS_MALE = 80,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ELDERLY_71_YEARS_MALE = 90,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_FEMALE = 100,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ASIAN_FEMALE = 101,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_PACIFIC_ISLANDER_FEMALE = 102,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_BLACK_FEMALE = 103,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_EAST_ASIAN_FEMALE = 104,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_HISPANIC_FEMALE = 105,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_WHITE_FEMALE = 106,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ARAB_FEMALE = 107,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_HOMOGENOUS_COUNTRY_CODE_FEMALE = 108,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_INDIGENOUS_COUNTRY_CODE_FEMALE = 109,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_INFANT_0_MONTHS_1_YEAR_FEMALE = 110,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_TODDLER_1_3_YEARS_FEMALE = 120,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_CHILD_3_10_YEARS_FEMALE = 130,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ADOLESCENT_10_12_YEARS_FEMALE = 140,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_TEENAGER_13_16_YEARS_FEMALE = 150,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_YOUNG_ADULT_17_25_YEARS_FEMALE = 160,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ADULT_25_55_YEARS_FEMALE = 170,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_SENIOR_ADULT_55_70_YEARS_FEMALE = 180,
     DIS_PDU_LIFE_FORM_EXTRA_PERSONAL_DATA_ELDERLY_71_YEARS_FEMALE = 190
}  DIS_PDU_Life_Form_Extra_Personal_Data;

/* SISO-REF-010 [UID 478] */
typedef enum
{
     DIS_PDU_LIFE_FORM_AIR_CATEGORIES_BIRD = 200,
     DIS_PDU_LIFE_FORM_AIR_CATEGORIES_INSECT = 201,
     DIS_PDU_LIFE_FORM_AIR_CATEGORIES_MAMMAL = 202
}  DIS_PDU_Life_Form_Air_Categories;

/* SISO-REF-010 [UID 479] */
typedef enum
{
     DIS_PDU_LIFE_FORM_SUBSURFACE_CATEGORIES_FISH = 200,
     DIS_PDU_LIFE_FORM_SUBSURFACE_CATEGORIES_MAMMAL = 201,
     DIS_PDU_LIFE_FORM_SUBSURFACE_CATEGORIES_MOLLUSK = 202,
     DIS_PDU_LIFE_FORM_SUBSURFACE_CATEGORIES_CRUSTACEAN = 203,
     DIS_PDU_LIFE_FORM_SUBSURFACE_CATEGORIES_INSECT = 204
}  DIS_PDU_Life_Form_Subsurface_Categories;

/* SISO-REF-010 [UID 481] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_5_8MM_QBU_88_TYPE_88 = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_C3 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_FR_F2 = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_AWM_F_G22 = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_G3_SG_1 = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_GALIL_SNIPER = 34,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_L96A1 = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_M14_DMR = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_M24_SNIPER_WEAPON_SYSTEM_SWS = 37,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_M40A1_A3 = 38,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_STEYR_SSG_69 = 39,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_SVD_DRAGUNOV = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_TYPE_79 = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_SR_25_MK11 = 42,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_AW_SR_98 = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_BLASER_R93 = 44,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_M2010_ENHANCED_SNIPER_RIFLE_ESR = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_M110_SEMI_AUTOMATIC_SNIPER_SYSTEM_SASS = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_L129A1 = 47,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_SAKO_TRG_22 = 48,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_COLT_CANADA_C20_DMR = 49,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_BREN_2_PPS = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_ZBROYAR_VPR_308_SNIPER_VERSION_OF_Z_008 = 51,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_ZBROYAR_UAR_10 = 52,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_62MM_FORT_301 = 53,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_7_7MM_TYPE_99 = 100,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_8_58MM_BLASER_R93_TACTICAL_2 = 105,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_8_6MM_SAKO_TRG_42 = 106,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_9MM_VSS_VINTOREZ = 110,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_STEYR_HS_50 = 170,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_M82A1A_SPECIAL_APPLICATIONS_SCOPED_RIFLE_SASR = 171,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_NSV = 172,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_OSV_96 = 173,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_RANGEMASTER_50 = 174,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_V94 = 175,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_M107 = 176,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_ZVI_FALCON_OP99 = 177,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_SNIPEX_RHINO_HUNTER = 178,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_12_7MM_SNIPEX_ALLIGATOR = 179,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_14_5MM_SNIPEX_T_REX = 190,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SNIPER_20MM_DENEL_NTW_20 = 200
}  DIS_PDU_Life_Form_Human_Specific_Sniper;

/* SISO-REF-010 [UID 482] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_5_45MM_AKS_74U_AKSU_74 = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_5_56MM_DAEWOO_K1A = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_DAEWOO_K7 = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_MAC_10 = 61,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_MADSEN_MK_II = 62,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_MINI_UZI = 63,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_MODEL_83_SKORPION_SMG = 64,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_MP5A2 = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_MP5_N = 66,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_STERLING_SMG = 67,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_TYPE_CF_05 = 68,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SUB_MACHINE_GUN_9MM_UZI = 69
}  DIS_PDU_Life_Form_Human_Specific_Sub_Machine_Gun;

/* SISO-REF-010 [UID 490] */
typedef enum
{
     DIS_PDU_RADAR_TRACK_STATUS_NOT_SPECIFIED = 0,
     DIS_PDU_RADAR_TRACK_STATUS_RADAR_TRACK_DETECTED = 1,
     DIS_PDU_RADAR_TRACK_STATUS_RADAR_TRACK_BROKEN = 2
}  DIS_PDU_Radar_Track_Status;

/* SISO-REF-010 [UID 495] */
typedef enum
{
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_CLAY_MASS = 0,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_CLAY_VOLUME_FRACTION = 1,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_SILT_MASS = 2,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_SILT_VOLUME_FRACTION = 3,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_SAND_MASS = 4,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_SAND_VOLUME_FRACTION = 5,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_GRAVEL_MASS = 6,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_GRAVEL_VOLUME_FRACTION = 7,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_WATER_MASS = 8,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_WATER_VOLUME_FRACTION = 9,
     DIS_PDU_SOIL_ATTRIBUTE_TYPE_WATER_PHASE = 128
}  DIS_PDU_Soil_Attribute_Type;

/* SISO-REF-010 [UID 500] */
typedef enum
{
     DIS_PDU_AUSTRALIAN_CATEGORY_OVERLAY_AUSTRALIAN_ARMY = 11,
     DIS_PDU_AUSTRALIAN_CATEGORY_OVERLAY_ROYAL_AUSTRALIAN_AIR_FORCE_RAAF = 13,
     DIS_PDU_AUSTRALIAN_CATEGORY_OVERLAY_ROYAL_AUSTRALIAN_NAVY_RAN = 14,
     DIS_PDU_AUSTRALIAN_CATEGORY_OVERLAY_AUSTRALIAN_SPECIAL_OPERATIONS_COMMAND_SOCOMD = 30,
     DIS_PDU_AUSTRALIAN_CATEGORY_OVERLAY_AUSTRALIAN_DEPARTMENT_OF_HOME_AFFAIRS = 51,
     DIS_PDU_AUSTRALIAN_CATEGORY_OVERLAY_AUSTRALIAN_FEDERAL_POLICE_AFP = 52
}  DIS_PDU_Australian_Category_Overlay;

/* SISO-REF-010 [UID 501] */
typedef enum
{
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_AFGHAN_NATIONAL_ARMY_ANA = 11,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_AFGHAN_AIR_FORCE_AAF = 13,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_NATIONAL_DIRECTORATE_OF_SECURITY_NDS = 51,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_AFGHAN_NATIONAL_POLICE_ANP = 52,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_AFGHAN_BORDER_POLICE_ABP = 53,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_AFGHAN_NATIONAL_CIVIL_ORDER_POLICE_ANCOP = 54,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_AFGHAN_PUBLIC_PROTECTION_FORCE_APPF = 55,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_NON_MILITARY_NATIONAL_GOVERNMENT_AGENCIES = 70,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_TERRORIST_COMBATANT = 101,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_HUMANITARIAN_ORGANIZATIONS = 120,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_RED_CRESCENT = 121,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_CIVILIAN = 130,
     DIS_PDU_LIFE_FORM_CATEGORIES_AFGHANISTAN_PRESS = 133
}  DIS_PDU_Life_Form_Categories_Afghanistan;

/* SISO-REF-010 [UID 505] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_EQUIPMENT_CLASS_SIGNAL_SMOKE = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_EQUIPMENT_CLASS_FLASH_LIGHT = 2,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_EQUIPMENT_CLASS_SIGNAL_MIRROR = 3,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_EQUIPMENT_CLASS_IR_STROBE = 4,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_EQUIPMENT_CLASS_IR_ILLUMINATOR = 5,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_EQUIPMENT_CLASS_SPOTLIGHT = 6
}  DIS_PDU_Life_Form_Human_Specific_Equipment_Class;

/* SISO-REF-010 [UID 508] */
typedef enum
{
     DIS_PDU_PLATFORM_SUBSURFACE_CIVILIAN_SEMI_SUBMERSIBLES_SUBCATEGORIES_NARCO_SUBMARINE = 1
}  DIS_PDU_Platform_Subsurface_Civilian_Semi_Submersibles_Subcategories;

/* SISO-REF-010 [UID 509] */
typedef enum
{
     DIS_PDU_LEAF_COVERAGE_NORMAL = 0,
     DIS_PDU_LEAF_COVERAGE_BARE = 1
}  DIS_PDU_Leaf_Coverage;

/* SISO-REF-010 [UID 510] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_MATERIEL_RIFLES_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_MATERIEL_RIFLES_12_7MM_AW50 = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_MATERIEL_RIFLES_12_7MM_AW50F = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_MATERIEL_RIFLES_12_7MM_TASKO_7ET3 = 12
}  DIS_PDU_Life_Form_Human_Specific_Anti_Materiel_Rifles;

/* SISO-REF-010 [UID 511] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_BROWNING_SUPERPOSED_O_U = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_BROWNING_CYNERGY = 21,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_BROWNING_AUTO_5 = 22,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BROWNING_CITORI_O_U_12_GAUGE = 23,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_16_8MM_BROWNING_CITORI_O_U_16_GAUGE = 24,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_BROWNING_CITORI_O_U_20_GAUGE = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_14MM_BROWNING_CITORI_O_U_28_GAUGE = 26,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_BROWNING_CITORI_O_U_410_BORE = 27,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BROWNING_DOUBLE_AUTOMATIC_12_GAUGE = 28,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_ITHACA_37_12_GAUGE = 29,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_16_8MM_ITHACA_37_16_GAUGE = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_ITHACA_37_20_GAUGE = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_14MM_ITHACA_37_28_GAUGE = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_19_7MM_ITHACA_MAG_10_SA_10_GAUGE = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_19_7MM_MARLIN_MODEL_55_10_GAUGE = 34,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_MARLIN_MODEL_55_12_GAUGE = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_16_8MM_MARLIN_MODEL_55_16_GAUGE = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_MARLIN_MODEL_55_20_GAUGE = 37,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_MOSSBERG_500_12_GAUGE = 38,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_MOSSBERG_500_20_GAUGE = 39,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_MOSSBERG_500_410_BORE = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_MOSSBERG_590_12_GAUGE = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_MOSSBERG_590_20_GAUGE = 42,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_MOSSBERG_590_410_BORE = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_MOSSBERG_930_SA_12_GAUGE = 44,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_REMINGTON_MODEL_11_SA = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_REMINGTON_MODEL_10_12_GAUGE = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_REMINGTON_MODEL_17_20_GAUGE = 47,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_REMINGTON_MODEL_31 = 48,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_REMINGTON_MODEL_11_48_SA = 49,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_870_12_GAUGE = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_16_8MM_REMINGTON_870_16_GAUGE = 51,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_REMINGTON_870_20_GAUGE = 52,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_14MM_REMINGTON_870_28_GAUGE = 53,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_REMINGTON_870_410_BORE = 54,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_REMINGTON_MODEL_58_SA = 55,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_878_SA_12_GAUGE = 56,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_MODEL_1100_SA_12_GAUGE = 57,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_16_8MM_REMINGTON_MODEL_1100_SA_16_GAUGE = 58,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_REMINGTON_MODEL_1100_SA_20_GAUGE = 59,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_14MM_REMINGTON_MODEL_1100_SA_28_GAUGE = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_REMINGTON_MODEL_1100_SA_410_BORE = 61,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_11_87_SA_12_GAUGE = 62,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_REMINGTON_11_87_SA_20_GAUGE = 63,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_19_7MM_REMINGTON_MODEL_SP_10_SA_10_GAUGE = 64,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_887_12_GAUGE = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_SPARTA_100_SXS_12_GAUGE = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_REMINGTON_SPARTA_100_SXS_20_GAUGE = 71,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_REMINGTON_SPARTA_100_SXS_410_BORE = 72,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_SPARTAN_310_O_U_12_GAUGE = 73,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_REMINGTON_SPARTAN_310_O_U_20_GAUGE = 74,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_14MM_REMINGTON_SPARTAN_310_O_U_28_GAUGE = 75,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_REMINGTON_SPARTAN_310_O_U_410_BORE = 76,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_REMINGTON_SPARTAN_453_SA_12_GAUGE = 77,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_WINCHESTER_MODEL_1200_12_GAUGE = 80,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_16_8MM_WINCHESTER_MODEL_1200_16_GAUGE = 81,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_WINCHESTER_MODEL_1200_20_GAUGE = 82,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_WINCHESTER_MODEL_1887_1901 = 83,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_WINCHESTER_MODEL_1897 = 84,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_WINCHESTER_MODEL_1912 = 85,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_WINCHESTER_MODEL_21_SXS = 86,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_WINCHESTER_MODEL_37_SXS = 87,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_HR_ULTRASLUG_SXS_12_GAUGE = 88,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_HR_ULTRASLUG_SXS_20_GAUGE = 89,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_CIENER_ULTIMATE_O_U_12_GAUGE = 90,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_COACH_GUN_SXS_DOUBLE_BARREL_12_GAUGE = 91,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_RUGER_GOLD_LABEL_SXS_12_GAUGE = 92,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_HIGH_STANDARD_MODEL_10_SA_12_GAUGE = 93,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_KEL_TEX_KSG_12_GAUGE = 94,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5_KAC_MASTERKEY_12_GAUGE = 95,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_M26_M_A_S_S_12_GAUGE = 96,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_SRM_ARMS_M1216_SA_12_GAUGE = 97,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_AA_12_FA_ATCHISSON_ASSAULT = 98,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_PANCOR_JACKHAMMER_FA_12_GAUGE = 99,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_USAS_12_FA_12_GAUGE = 110,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_MAUL_SA_12_GAUGE = 111,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_FN_SLP_SA_12_GAUGE = 112,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_FN_TPS_12_GAUGE = 113,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_ENARM_PENTAGUN_SA_12_GAUGE = 115,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_STEVENS_MODEL_520_620 = 116,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_STOEGER_COACH_GUN_SXS = 117,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_STOEGER_CONDOR_O_U = 118,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_ARMSCOR_MODEL_30_SA_12_GAUGE = 120,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_WEATHERBY_SA_08_SA = 121,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_FABARM_SDASS_TACTICAL_12_GAUGE = 122,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_MAG_7_12_GAUGE = 123,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_NEOSTEAD_12_GAUGE = 124,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_ARMSEL_STRIKER_SA_12_GAUGE = 125,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_PARKER_HALE_ROGUN_SA_12_GAUGE = 127,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_26MM_RGA_86_REVOLVER = 130,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_SJORGREN_SA_12_GAUGE = 131,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_AKDAL_MKA_1919_SA_12_GAUGE = 132,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_RETAY_MASAI_MARA_SA_12_GAUGE = 133,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_SAFIR_T_14_SA_12_GAUGE = 134,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BENELLI_M1_SUPER_90_SA_12_GAUGE = 150,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_BENELLI_M1_SUPER_90_SA_20_GAUGE = 151,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BENELLI_M3_SUPER_90_SA_12_GAUGE = 152,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_BENELLI_M3_SUPER_90_SA_20_GAUGE = 153,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BENELLI_M4_SUPER_90_SA_12_GAUGE = 154,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BENELLI_NOVA_12_GAUGE = 155,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_BENELLI_NOVE_20_GAUGE = 156,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BENELLI_RAFFAELLO_SA_12_GAUGE = 157,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BENELLI_SUPERNOVA_12_GAUGE = 158,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BENELLI_VINCI_SA_12_GAUGE = 159,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BERETTA_1201FP_SA_12_GAUGE = 160,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BERETTA_682_O_U_12_GAUGE = 161,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_BERETTA_682_O_U_20_GAUGE = 162,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_14MM_BERETTA_682_O_U_28_GAUGE = 163,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_BERETTA_682_O_U_410_BORE = 164,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BERETTA_A303_SA_12_GAUGE = 165,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BERETTA_AL391_SA_12_GAUGE = 166,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_BERETTA_AL391_SA_20_GAUGE = 167,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BERETTA_DT_10_O_U_12_GAUGE = 168,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_BERETTA_SILVER_PIGEON_O_U = 169,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BERETTA_XTREMA_2_SA_12_GAUGE = 170,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_FRANCHI_AL_48_SA_20_GAUGE = 171,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_14MM_FRANCHI_AL_48_SA_28_GAUGE = 172,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_FRANCHI_MOD_410_FA_410_BORE = 173,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_FRANCHI_SPAS_12_SA_12_GAUGE = 174,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_FRANCHI_SPAS_15_SA_12_GAUGE = 175,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_VALTRO_PM_5_PM_5_350_12_GAUGE = 176,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_BLAZER_F3_O_U = 180,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_HK_FABARM_FP6_12_GAUGE = 181,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_HK_CAWS_FA_12_GAUGE = 182,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BAIKAL_MP_153_SA_12_GAUGE = 200,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_BANDAYEVSKY_RB_12_12_GAUGE = 201,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_MOLOT_BEKAS_M_12_GAUGE = 202,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_16_8MM_MOLOT_BEKAS_M_16_GAUGE = 203,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_TOZ_194_12_GAUGE = 204,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_23MM_KS_23 = 205,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_MTS_255_REVOLER_12_GAUGE = 206,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_RMB_93_12_GAUGE = 207,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_SAIGA_12_SA_12_GAUGE = 208,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_15_6MM_SAIGA_12_SA_20_GAUGE = 209,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_10_4MM_SAIGA_12_SA_410_BORE = 210,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_VEPR_12_SA_12_GAUGE = 211,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_FORT_500_12_GAUGE = 212,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_SHOT_GUNS_18_5MM_NORINCO_HP9_1_12_GAUGE = 220
}  DIS_PDU_Life_Form_Human_Specific_Shot_Guns;

/* SISO-REF-010 [UID 512] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_OTHERS = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_60MM_M224 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_60MM_ANTOS = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_60MM_KBA_118 = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_60MM_KBA_48M = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_81MM_F2 = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_81MM_L16 = 51,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_81MM_M252 = 52,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_81MM_MX2_KM = 53,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_82MM_UPIK_82 = 54,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_120MM_SOLTAM_K6_M120 = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_120MM_M121 = 61,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MORTARS_120MM_M120_15_MOLOT = 62
}  DIS_PDU_Life_Form_Human_Specific_Mortars;

/* SISO-REF-010 [UID 513] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_5_45MM_PSM = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_MK3_SLP = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_BERETTA_92S_92FS_M9 = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_H_K_USP = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_STECHKIN_APS = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_MAKAROV_PM = 34,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_SMITH_AND_WESSON_SD_SIGMA = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_GLOCK_17 = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_SIG_SAUER_M17 = 37,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_SIG_PRO = 38,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_SMITH_AND_WESSON_SW1911 = 39,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_SMITH_AND_WESSON_5900_SERIES = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_45_CAL_M1911 = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_CZ_75 = 42,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_CZ_82 = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9_07MM_RUGER_GP_100 = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_10MM_GLOCK_20 = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_FORT_12 = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_FORT_14TP = 71,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_FORT_17 = 72,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_HAND_GUNS_9MM_FORT_21 = 73
}  DIS_PDU_Life_Form_Human_Specific_Hand_Guns;

/* SISO-REF-010 [UID 514] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_WEAPON_NON_SPECIFIC_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_WEAPON_NON_SPECIFIC_KNIFE = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_WEAPON_NON_SPECIFIC_MACHETE = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_WEAPON_NON_SPECIFIC_EXPLOSIVE_VEST = 100,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_WEAPON_NON_SPECIFIC_M18A1_CLAYMORE = 150
}  DIS_PDU_Life_Form_Human_Specific_Weapon_Non_specific;

/* SISO-REF-010 [UID 515] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_ARSENAL_UGGL_M1 = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_ARSENAL_MSGL = 2,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_VOG_ARSENAL_MSGL = 3,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_ARSENAL_UBGL_M16 = 4,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_ARSENAL_UBGL_M8 = 5,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_ARSENAL_UBGL_M7 = 6,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_30MM_BS_1_TISHINA = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_BTS_203 = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_INDUMIL_IMC_40 = 12,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_VOG_BG_15 = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_VOG_GP_25_KOSTOYOR = 21,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_VOG_GP_30_OBUVKA = 22,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_VOG_GP_34 = 23,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_VOG_RGM_40_KASTET = 24,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_VOG_RG_6 = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_M79 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_M203 = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X36MM_M320 = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_CIS_40_GL = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_EAGLE_GL = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_HK_AG36 = 37,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_HK_AG_C_GLM = 38,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_HK69A1 = 39,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_BERETTA_GLX_160 = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_ARDE_UBGL = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_XML148 = 42,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_CHINA_LAKE_GL = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X46MM_HAWK_MM_1 = 44,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_25X40MM_XM25_CDTE = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_37MM_MILKOR37_38_LL_STOPPER = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_MILKOR40_GL = 61,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40MM_MILKOR_MGL = 62,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X47MM_PALLAD_WZ1974 = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X47MM_PALLAD_WZ1983 = 66,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_UGL_200_CANISTER_RWGL_3 = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_20X30MM_ST_DAEWOO_K11 = 80,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_30MM_RGSH_30_GL = 85,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_35MM_TYPE_91_BREECHLOAD_GL = 90,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X53MM_CZW_40 = 95,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_40X53MM_UAG_40_AGL = 96,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_45MM_DP_64 = 100,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHERS_20X42MM_NEOPUP_PAW_20 = 105
}  DIS_PDU_Life_Form_Human_Specific_Grenade_Launchers;

/* SISO-REF-010 [UID 516] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_XM214_MICROGUN_SIX_PAK = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M134_XM196_MINIGUN = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_M249_FN_MINIMI_SAW_LMG = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_FN_MINIMI_MK3_LMG = 21,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_FN_MINIMI_7_62_MK3_GPMG = 22,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X63MM_M1941_JOHNSON = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X63MM_M1918_BAR = 26,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M1919A4_MK_21_MOD_0_BROWNING_MMG = 27,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X63MM_M1919A6_BROWNING_MMG = 28,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M37_BROWNING_MMG = 29,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_ARES_SHRIKE_5_56_LMG = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_LSAT_LMG = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_CMG_1_LMG = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_CMG_2_LMG = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_STONER_63A_LMG = 34,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_ULTIMAX_100_LMG = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X54MM_BERETTA_AS70_90_LMG = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_CETME_AMELI_LMG = 37,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_IMI_NEGEV_LMG = 38,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_INSAS_LMG = 39,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_AUG_LMG = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_AUG_HBAR_LMG = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_HK_MG4_LMG = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_HK23_GR_9_LMG = 44,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_M27_IAR_SAW = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_L86_LSW = 47,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_DAEWOO_K3_LMG = 48,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_VECTOR_MINI_SS_GPMG = 49,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M60_GPMG = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M60E3_GPMG = 51,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M60E4_GPMG = 52,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M60E6_GPMG = 53,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_MARK_48_GMPG = 55,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M240_FN_MAG_58_GPMG = 58,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M240E4_M240B_GPMG = 59,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M240E1_M240D_GPMG = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M240G_GPMG = 61,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M240E5_M240H_GPMG = 62,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_M240L_GPMG = 63,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X39MM_KK_62_LMG = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_VECTOR_SS_77_GPMG = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_SIG_MG_710_3_GPMG = 71,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_STERLING_7_62_GMPG = 72,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_SUMITOMO_TYPE_62_GPMG = 73,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_DAEWOO_K12_GPMG = 74,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_MG_51_GPMG = 75,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_RHEINMETALL_MG_3 = 76,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_RHEINMETALL_MG_3KWS = 77,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_MG5_HK121_GPMG = 80,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_HK21_GPMG = 81,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_AA_52_GPMP = 85,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_UKM_2000_GPMG = 86,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X54MM_UK_VZ_59_GPMG = 88,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_92X57MM_MG_42_GPMG = 89,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X99MM_M2A1_BROWNING_HMG = 100,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X99MM_M2HB_BROWNING_HMG = 101,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X99MM_M2HB_QCB_BROWNING_HMG = 102,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X99MM_M85C_HMG = 105,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X99MM_RHEINMETALL_RMG_50_HMG = 108,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X99MM_HK25_HMG = 110,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X99MM_CIS_50MG = 112,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_45X39MM_IP_2_LMG = 120,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_45X39MM_NIKONOV_LMG = 121,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_45X39MM_M74_RPK = 122,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X39MM_M43_RPK = 125,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X39MM_RPD_SAW = 126,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X39MM_ZASTAVA_M72 = 127,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X39MM_TYPE_81_LMG = 128,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X51MM_ZASTAVA_M77 = 135,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X54MM_PK_GPMG = 140,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X54MM_AEK_999_GPMP = 141,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X54MM_PECHENEG_GPMG = 142,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X54MM_ZASTAVA_M84 = 143,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X54MM_TYPE_67_GPMG = 144,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_7_62X54MM_TYPE_80_GPMG = 145,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_NSV_HMG = 150,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_KORD_HMG = 151,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_KPD_12_7_HMG = 152,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_ZASTAVA_M02_COYOTOE_HMG = 153,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_ZASTAVA_M87 = 154,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_TYPE_77_HMG = 155,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_W85_HMG = 156,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_12_7X108MM_TYPE_90_HMG = 157,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_8X42MM_QJY_88_LMG = 164,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_8X42MM_QBB_95_DBP87_LMG = 165,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_5_56X45MM_QBB_95_1_LMG = 166,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MACHINE_GUNS_6_8MM_M250_NEXT_GENERATION_SQUAD_WEAPON_NGSW = 170
}  DIS_PDU_Life_Form_Human_Specific_Machine_Guns;

/* SISO-REF-010 [UID 517] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X53MM_HK_GMG = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X53MM_MK47_STRIKER = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40MM_M75 = 26,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40MM_M129 = 27,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X46MM_XM_174 = 28,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X46MM_MK18_MOD_0 = 29,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X53MM_MK19 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X46MM_MK20_MOD_0 = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_30X29MM_RAG_30_SAG_30 = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_30X29MM_AGS_17_PLAMYA = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_30X29MM_AGS_30_ATLANT = 42,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40MM_VOG_AGS_40_BALKAN = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X53MM_SB_LAG_40 = 44,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X53MM_VEKTOR_Y3 = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X53MM_CIS_40 = 55,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X56MM_HOWA_TYPE_96 = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_40X53MM_DAEWOO_PRECISION_INDUSTRIES_K4 = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_25X59MM_XM307_ADVANCED_CREW_SERVED_WEAPON = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_GRENADE_LAUNCHING_MACHINE_GUN_35X32MM_QLZ87 = 80
}  DIS_PDU_Life_Form_Human_Specific_Grenade_Launching_Machine_Gun;

/* SISO-REF-010 [UID 518] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_82MM_B_300 = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_82MM_SHIPON = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_83MM_MK153_MOD_0_SMAW = 12,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72_LAW = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72A1_LAW = 21,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72A2_LAW = 22,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72A3_LAW = 23,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72A4_LAW = 24,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72A5_LAW = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72A6_LAW = 26,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72A7_LAW = 27,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72E8_LAW = 28,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72E9_LAW = 29,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72E10_LAW = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_66MM_M72AS_LAW = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_94MM_LAW_80 = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_M1_BAZOOKA = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_M1A1_BAZOOKA = 41,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_M9_BAZOOKA = 42,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_M9A1_BAZOOKA = 43,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_M20_SUPER_BAZOOKA = 44,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_M20A1_SUPER_BAZOOKA = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_M20B1_SUPER_BAZOOKA = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_M20A1B1_SUPER_BAZOOKA = 47,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_M25_THREE_SHOT_BAZOOKA = 48,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_INSTALAZA_M65 = 49,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_INSTALAZA_C90 = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_C90_CR_M3 = 51,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_C90_CR_AM_M3 = 52,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_C90_CR_BK_M3 = 53,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_C90_CR_IN_M3 = 54,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_PZF_3 = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_PZF_3_IT = 61,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_PZF_3_BUNKERFAUST = 62,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_44MM_PZF_44 = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_30MM_PANZERFAUST_30 = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_50MM_PANZERFAUST_60 = 71,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_PANZERFAUST_100 = 72,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_60MM_PANZERFAUST_150 = 73,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_88MM_PANZERSCHRECK_RPZB = 75,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_83MM_RL_83_BLINDICIDE = 80,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_100MM_RL_100_BLINDICIDE = 81,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_M79_OSA_WASP = 85,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_64MM_M80_ZOLJA_WASP = 86,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_67MM_ARMBURST_CROSSBOW = 90,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_40MM_TYPE_69_RPG = 93,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_PIAT = 95,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_40MM_RPG_2 = 100,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_64MM_RPG_18_MUKHA = 101,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_72_5MM_RPG_22_NETTO = 102,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_72_5MM_RPG_26_AGLEN = 103,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_105MM_RPG_29_VAMPIR = 104,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_105MM_RPG_30_KRYUK = 105,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_105MM_RPG_32_NASHSHAB = 106,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_105MM_RPG_27_TAVOLGA = 107,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_40MM_RPG_7 = 110,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_40MM_PSRL_1_RPG_7USA = 111,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_40MM_GS_777_PSRL_2 = 112,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_68MM_RPG_76_KOMAR_MOSQUITO = 120,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_120MM_SEP_DARD_120 = 125,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_58MM_WASP_58 = 128,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_73MM_LRAC_73_50 = 130,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_89MM_LRAC_89_F1_STRIM = 131,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_MATADOR_MAN_PORTABLE_ANTI_TANK_ANTI_DOOR = 135,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_MATADOR_MP = 136,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_MATADOR_WB = 137,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_90MM_MATADOR_AS = 138,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_78MM_MARA_ANTI_TANK_ROCKET_LAUNCHER = 140,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_ROCKETS_120MM_TYPE_98_PF98 = 145
}  DIS_PDU_Life_Form_Human_Specific_Anti_Tank_Rockets;

/* SISO-REF-010 [UID 519] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_TYPE_64_MAT_KAM_3 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_153MM_TYPE_79_JYU_MAT_KAM_9 = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_TYPE_87_CHU_MAT = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_140MM_TYPE_01_LMAT = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_140MM_M47_DRAGON = 58,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_140MM_SAEGHE_1_2 = 59,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_127MM_FGM_148_JAVELIN = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_139MM_FGM_172_SRAW = 63,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_139MM_FGM_172B_SRAW_MPV = 64,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_BGM_71_TOW = 68,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_OREV_TOW_II = 69,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_VICKERS_VIGILANT_CLEVITE = 75,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_110MM_BANTAM_RB_53 = 80,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_150MM_RBS_56_BILL_1 = 81,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_150MM_RBS_56_2_BILL_2 = 82,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_130MM_SPIKE_SR = 85,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_130MM_SPIKE_MR_CLU = 86,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_130MM_SPIKE_LR_CLU = 87,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_60MM_MOSQUITO = 95,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_160MM_SS_10 = 98,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_103MM_MILAN = 100,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_115MM_MILAN_2 = 101,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_115MM_MILAN_2T = 102,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_115MM_MILAN_3 = 103,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_115MM_MILAN_ER = 104,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_136MM_ERYX = 105,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_ENTAC = 107,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_RAAD = 110,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_I_RAAD_T = 111,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_TOOPHAN = 112,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_TOOPHAN_2 = 113,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_TOOPHAN_5 = 114,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_136MM_BUMBAR = 120,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_130MM_SHERSHEN_PK_2 = 125,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_SHERSHEN_Q_P_2B = 126,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_130MM_MECTRON_MSS_1_2 = 130,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8 = 140,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8A = 141,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8B = 142,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8C = 143,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8D = 144,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8E = 145,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8F = 146,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8FAE = 147,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8L = 148,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8H = 149,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_8S = 150,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_BAKTAR_SHIKAN = 151,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_HJ_11_AFT_11 = 152,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_HJ_9A = 153,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_135MM_HJ_12_RED_ARROW = 154,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_HJ_73_MCLOS = 155,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_HJ_73B_SACLOS = 156,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_HJ_73C_SACLOS_ERA = 157,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_AT_3_SAGGER_A_9M14_MALYUTKA = 170,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_AT_3B_SAGGER_B_9M14M_MALYUTKA_M = 171,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_AT_3C_SAGGER_C_9M14P_MALYUTKA_P = 172,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_AT_3D_SAGGER_D_9M14_2_MALYUTKA_2 = 173,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_SUSONG_PO = 174,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_AT_3C_POLK = 175,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_KUN_WU_1 = 176,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_125MM_MALIUTKA_M2T = 177,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_AT_4A_SPIGOT_A_9M111_FAGOT = 178,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_AT_4B_SPIGOT_B_9M111_2_FAGOT = 179,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_120MM_AT_4C_SPIGOT_C_9M111M_FAKTORIYA = 180,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_135MM_AT_5A_SPANDREL_9M113_KRONKURS = 181,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_135MM_AT_5B_SPANDREL_9M113M_KRONKURS_M = 182,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_135MM_TOSAN = 183,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_94MM_AT_7_SAXHORN_9K115_METIS = 184,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_130MM_AT_13_SAXHORN_2_9K115_2_METIS_M = 185,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_AT_14_SPRIGGAN_9M133_KORNET = 186,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_152MM_DEHLAVIE = 187,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ANTI_TANK_MISSILES_102MM_MATHOGO = 200
}  DIS_PDU_Life_Form_Human_Specific_Anti_Tank_Missiles;

/* SISO-REF-010 [UID 520] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_70MM_FIM_43_REDEYE = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_70MM_FIM_92_STINGER = 2,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_76MM_BLOWPIPE = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_76MM_STARBURST_JAVELIN_S_15 = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_130MM_STARSTREAK_HVM = 12,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_90MM_MISTRAL = 15,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_9K32M_STRELA_2_SA_7 = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_9K36_STRELA_3_SA_14 = 21,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_9K38_IGLA_SA_18 = 22,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_9K310_IGLA_M_SA_16 = 23,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_9K333_VERBA_SA_29 = 24,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_9K338_IGLA_S_SA_24_GRINCH = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_9K32M_STRELA_2M_SA_7B = 26,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_HN_5_HONG_YING_5 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_CSA_7_QW_1_VANGUARD = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_CSA_8_QW_2 = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_90MM_QW_3 = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_CSA_10_FN_6 = 34,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_CSA_11_QW_18 = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_72MM_CSA_14_FN_16 = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_71MM_MISAGH_1 = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_71MM_MISAGH_2 = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_80MM_TYPE_91_KIN_SAM = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_80MM_KP_SAM_SHUN_GUNG_CHIRON = 55,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_MAN_PORTABLE_AIR_DEFENSE_SYSTEM_106MM_RBS_70 = 60
}  DIS_PDU_Life_Form_Human_Specific_Man_Portable_Air_Defense_System;

/* SISO-REF-010 [UID 521] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_84MM_M136_AT_4_CS = 15,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_57MM_M18_RR = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_75MM_M20_RR = 21,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_120MM_M_28_DAVY_CROCKETT = 22,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_155MM_M_29_DAVY_CROCKETT = 23,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_106MM_M40_RECOILLESS_RIFLE = 24,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_82MM_M60_RR = 25,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_90MM_M67_RR = 26,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_84MM_M1_CARL_GUSTAV = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_84MM_M2_CARL_GUSTAV = 31,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_84MM_M3_CARL_GUSTAV = 32,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_84MM_M4_CARL_GUSTAV = 33,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_74MM_PANSARSKOTT_M_68_MINIMAN = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_84MM_ALAC = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_82MM_B_10_RR = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_107MM_B_11_RR = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_80MM_BREDA_FOLGORE = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_120MM_BAT_RR = 55,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_73MM_SPG_9_KOPYE = 60,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_88MM_RCL_3_45IN = 65,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_90MM_PVPJ_110 = 70,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_50MM_JAGDFAUST = 75,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_30MM_RHEINMETALL_RMK30 = 80,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_88MM_55_S_55_RAIKKA = 90,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_95MM_95_S_58_61 = 91,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_73MM_LG40 = 95,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_105MM_LG40 = 96,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_RECOILLESS_RIFLES_105MM_LG42 = 97
}  DIS_PDU_Life_Form_Human_Specific_Recoilless_Rifles;

/* SISO-REF-010 [UID 522] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_ROCKETS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_ROCKETS_66MM_M202_FLASH = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_ROCKETS_62MM_FHJ_84 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_ROCKETS_90MM_C90_CR_FIM_M3 = 40,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_ROCKETS_93MM_RPO_A_SHMEL = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_ROCKETS_93MM_RPO_Z_SHMEL = 51,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_ROCKETS_93MM_RPO_D_SHMEL = 52
}  DIS_PDU_Life_Form_Human_Specific_Flame_Rockets;

/* SISO-REF-010 [UID 523] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_HANDFLAMMPATRONE = 10,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_FMW_41 = 11,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_M1A1 = 20,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_M2A1_7 = 21,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_M9A1_7 = 22,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_LPO_50 = 30,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_K_PATTERN = 35,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_PORTABLE_NO_2_ACK_PACK = 36,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_MARSDEN = 37,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_HARVEY = 38,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_ROKS_2 = 45,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_ROKS_3 = 46,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_TYPE_93 = 50,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_FLAME_THROWERS_TYPE_100 = 51
}  DIS_PDU_Life_Form_Human_Specific_Flame_Throwers;

/* SISO-REF-010 [UID 524] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_DRONE_GUNS_OTHER = 0,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_DRONE_GUNS_DRONEGUN_TACTICAL = 15,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_DRONE_GUNS_DRONEGUN_MKII = 16
}  DIS_PDU_Life_Form_Human_Specific_Drone_Guns;

/* SISO-REF-010 [UID 525] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_LOGISTICS_EQ_CLASS_SLING_LOAD_PENDANT = 1
}  DIS_PDU_Life_Form_Human_Specific_Logistics_EQ_Class;

/* SISO-REF-010 [UID 526] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_PERSONAL_ELECTRONICS_CLASS_CELL_PHONE = 1
}  DIS_PDU_Life_Form_Human_Specific_Personal_Electronics_Class;

/* SISO-REF-010 [UID 527] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_LASERS_CLASS_GENERIC_LASER_DESIGNATOR = 1,
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_LASERS_CLASS_GENERIC_LASER_POINTER = 2
}  DIS_PDU_Life_Form_Human_Specific_Lasers_Class;

/* SISO-REF-010 [UID 528] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HUMAN_SPECIFIC_ROCKET_LAUNCHERS_90MM_ENFORCER = 1
}  DIS_PDU_Life_Form_Human_Specific_Rocket_Launchers;

/* SISO-REF-010 [UID 539] */
typedef enum
{
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_DEFAULT = 0,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_SOLID_COLOR_PRIMARY = 1,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_TWO_TONE_TOP_PRIMARY_BOTTOM_SECONDARY = 2,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_UNIFORMED_SOLID_COLOR_MILITARY = 3,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_UNIFORMED_SOLID_COLOR_CIVILIAN = 4,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_CAMOUFLAGE_DESERT = 5,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_CAMOUFLAGE_JUNGLE = 6,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_CAMOUFLAGE_SNOW = 7,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_CAMOUFLAGE_GRAVEL = 8,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_CAMOUFLAGE_PAVEMENT = 9,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_CAMOUFLAGE_SAND = 10,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_CAMOUFLAGE_FOREST = 11,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_NAKED = 12,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_TOPLESS_BUT_NOT_BOTTOMLESS = 13,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_BOTTOMLESS_BUT_NOT_TOPLESS = 14,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_INDIGENOUS_CLOTHING = 15,
     DIS_PDU_LIFE_FORM_CLOTHING_SCHEME_TRADITIONAL_CLOTHING = 16
}  DIS_PDU_Life_Form_Clothing_Scheme;

/* SISO-REF-010 [UID 540] */
typedef enum
{
     DIS_PDU_CLOTHING_DECAL_SCHEME_NONE = 0
}  DIS_PDU_Clothing_Decal_Scheme;

/* SISO-REF-010 [UID 541] */
typedef enum
{
     DIS_PDU_LIFE_FORM_HEAD_HAIR_NOT_SPECIFIED = 0,
     DIS_PDU_LIFE_FORM_HEAD_HAIR_BALD = 1,
     DIS_PDU_LIFE_FORM_HEAD_HAIR_SHORT = 2,
     DIS_PDU_LIFE_FORM_HEAD_HAIR_LONG = 3
}  DIS_PDU_Life_Form_Head_Hair;

/* SISO-REF-010 [UID 542] */
typedef enum
{
     DIS_PDU_LIFE_FORM_FACIAL_HAIR_NOT_SPECIFIED = 0,
     DIS_PDU_LIFE_FORM_FACIAL_HAIR_CLEAN_SHAVEN = 1,
     DIS_PDU_LIFE_FORM_FACIAL_HAIR_SHORT_BEARD = 2,
     DIS_PDU_LIFE_FORM_FACIAL_HAIR_LONG_BEARD = 3
}  DIS_PDU_Life_Form_Facial_Hair;

/* SISO-REF-010 [UID 543] */
typedef enum
{
     DIS_PDU_LIFE_FORM_UNIQUE_ID_NOT_SPECIFIED = 0
}  DIS_PDU_Life_Form_Unique_ID;

/* SISO-REF-010 [UID 546] */
typedef enum
{
     DIS_PDU_SPECIAL_LIGHTS_INTENSITY_OFF = 0,
     DIS_PDU_SPECIAL_LIGHTS_INTENSITY_INTENSITY_1 = 1,
     DIS_PDU_SPECIAL_LIGHTS_INTENSITY_INTENSITY_2 = 2,
     DIS_PDU_SPECIAL_LIGHTS_INTENSITY_INTENSITY_3 = 3,
     DIS_PDU_SPECIAL_LIGHTS_INTENSITY_INTENSITY_4 = 4,
     DIS_PDU_SPECIAL_LIGHTS_INTENSITY_INTENSITY_5 = 5
}  DIS_PDU_Special_Lights_Intensity;

/* SISO-REF-010 [UID 547] */
typedef enum
{
     DIS_PDU_IR_ANTI_COLLISION_FLASH_PATTERN_PATTERN_1 = 0,
     DIS_PDU_IR_ANTI_COLLISION_FLASH_PATTERN_PATTERN_2 = 1,
     DIS_PDU_IR_ANTI_COLLISION_FLASH_PATTERN_PATTERN_3 = 2,
     DIS_PDU_IR_ANTI_COLLISION_FLASH_PATTERN_PATTERN_4 = 3,
     DIS_PDU_IR_ANTI_COLLISION_FLASH_PATTERN_PATTERN_5 = 4
}  DIS_PDU_IR_Anti_Collision_Flash_Pattern;

/* SISO-REF-010 [UID 549] */
typedef enum
{
     DIS_PDU_DECK_STATUS_LIGHT_COLOR_OFF = 0,
     DIS_PDU_DECK_STATUS_LIGHT_COLOR_GREEN = 1,
     DIS_PDU_DECK_STATUS_LIGHT_COLOR_YELLOW = 2,
     DIS_PDU_DECK_STATUS_LIGHT_COLOR_RED = 3
}  DIS_PDU_Deck_Status_Light_Color;

/* SISO-REF-010 [UID 555] */
typedef enum
{
     DIS_PDU_SLING_LOAD_STATUS_NOT_PRESENT = 0,
     DIS_PDU_SLING_LOAD_STATUS_EMPTY = 1,
     DIS_PDU_SLING_LOAD_STATUS_LOADED = 2
}  DIS_PDU_Sling_Load_Status;

/* SISO-REF-010 [UID 556] */
typedef enum
{
     DIS_PDU_SLING_LOAD_DAMAGE_NO_DAMAGE = 0,
     DIS_PDU_SLING_LOAD_DAMAGE_SLING_DAMAGED = 1,
     DIS_PDU_SLING_LOAD_DAMAGE_LINE_DAMAGED = 2,
     DIS_PDU_SLING_LOAD_DAMAGE_SLING_OR_LINE_DESTROYED = 3
}  DIS_PDU_Sling_Load_Damage;

/* SISO-REF-010 [UID 557] */
typedef enum
{
     DIS_PDU_HOIST_STATUS_HOIST_NOT_PRESENT = 0,
     DIS_PDU_HOIST_STATUS_HOIST_UNAVAILABLE = 1,
     DIS_PDU_HOIST_STATUS_HOIST_AVAILABLE = 2
}  DIS_PDU_Hoist_Status;

/* SISO-REF-010 [UID 564] */
typedef enum
{
     DIS_PDU_LIFE_FORM_PARACHUTE_DOES_NOT_HAVE_A_PARACHUTE = 0,
     DIS_PDU_LIFE_FORM_PARACHUTE_HAS_A_PARACHUTE_BUT_NOT_DEPLOYED = 1,
     DIS_PDU_LIFE_FORM_PARACHUTE_HAS_A_PARACHUTE_DEPLOYED = 2
}  DIS_PDU_Life_Form_Parachute;

/* SISO-REF-010 [UID 565] */
typedef enum
{
     DIS_PDU_LIFE_FORM_LASER_NO_LASER_EQUIPMENT_IN_OPERATION = 0,
     DIS_PDU_LIFE_FORM_LASER_LASER_POINTER = 1,
     DIS_PDU_LIFE_FORM_LASER_LASER_DESIGNATOR = 2,
     DIS_PDU_LIFE_FORM_LASER_LASER_RANGE_FINDER_LRF = 3
}  DIS_PDU_Life_Form_Laser;

/* SISO-REF-010 [UID 567] */
typedef enum
{
     DIS_PDU_APPLICATION_STATE_OFFLINE_OR_UNKNOWN = 0,
     DIS_PDU_APPLICATION_STATE_AVAILABLE = 1,
     DIS_PDU_APPLICATION_STATE_ALLOCATED = 2,
     DIS_PDU_APPLICATION_STATE_ERROR = 3
}  DIS_PDU_Application_State;

/* SISO-REF-010 [UID 568] */
typedef enum
{
     DIS_PDU_EXERCISE_STATE_UNKNOWN = 0,
     DIS_PDU_EXERCISE_STATE_UNALLOCATED = 1,
     DIS_PDU_EXERCISE_STATE_ALLOCATED = 2,
     DIS_PDU_EXERCISE_STATE_INITIALIZED = 3,
     DIS_PDU_EXERCISE_STATE_PAUSED = 4,
     DIS_PDU_EXERCISE_STATE_EXECUTING = 5,
     DIS_PDU_EXERCISE_STATE_ERROR = 6
}  DIS_PDU_Exercise_State;

/* SISO-REF-010 [UID 569] */
typedef enum
{
     DIS_PDU_APPLICATION_GENERAL_STATUS_UNKNOWN = 0,
     DIS_PDU_APPLICATION_GENERAL_STATUS_FUNCTIONAL = 1,
     DIS_PDU_APPLICATION_GENERAL_STATUS_DEGRADED_BUT_FUNCTIONAL = 2,
     DIS_PDU_APPLICATION_GENERAL_STATUS_NOT_FUNCTIONAL = 3
}  DIS_PDU_Application_General_Status;

/* SISO-REF-010 [UID 570] */
typedef enum
{
     DIS_PDU_APPLICATION_SPECIFIC_STATUS_NOT_SPECIFIED = 0
}  DIS_PDU_Application_Specific_Status;

/* SISO-REF-010 [UID 571] */
typedef enum
{
     DIS_PDU_APPLICATION_TRANSITION_NOT_TRANSITIONING = 0,
     DIS_PDU_APPLICATION_TRANSITION_TRANSITIONING = 1
}  DIS_PDU_Application_Transition;

/* SISO-REF-010 [UID 572] */
typedef enum
{
     DIS_PDU_EXERCISE_TRANSITION_NOT_TRANSITIONING = 0,
     DIS_PDU_EXERCISE_TRANSITION_TRANSITIONING = 1
}  DIS_PDU_Exercise_Transition;

/* SISO-REF-010 [UID 573] */
typedef enum
{
     DIS_PDU_STATUS_TYPE_NOT_SPECIFIED = 0,
     DIS_PDU_STATUS_TYPE_USER = 10,
     DIS_PDU_STATUS_TYPE_SYSTEM = 11,
     DIS_PDU_STATUS_TYPE_I_O = 12,
     DIS_PDU_STATUS_TYPE_IDLE = 13,
     DIS_PDU_STATUS_TYPE_STEAL = 14,
     DIS_PDU_STATUS_TYPE_NICE = 15,
     DIS_PDU_STATUS_TYPE_MEMORY_FREE = 50,
     DIS_PDU_STATUS_TYPE_MEMORY_USED = 51,
     DIS_PDU_STATUS_TYPE_SWAP_FREE = 60,
     DIS_PDU_STATUS_TYPE_SWAP_USED = 61,
     DIS_PDU_STATUS_TYPE_SWAP_CACHED = 62,
     DIS_PDU_STATUS_TYPE_TRANSMITTED_PACKETS_SEC = 100,
     DIS_PDU_STATUS_TYPE_TRANSMITTED_BYTES_SEC = 101,
     DIS_PDU_STATUS_TYPE_RECEIVED_PACKETS_SEC = 110,
     DIS_PDU_STATUS_TYPE_RECEIVED_BYTES_SEC = 111,
     DIS_PDU_STATUS_TYPE_NICENESS_LEVEL = 150
}  DIS_PDU_Status_Type;

/* SISO-REF-010 [UID 574] */
typedef enum
{
     DIS_PDU_STEALTH_SPECTRUM_IMAGE_TYPE_OUT_THE_WINDOW_OTW = 0,
     DIS_PDU_STEALTH_SPECTRUM_IMAGE_TYPE_DAY_TV = 1,
     DIS_PDU_STEALTH_SPECTRUM_IMAGE_TYPE_THERMAL_MANUAL = 2,
     DIS_PDU_STEALTH_SPECTRUM_IMAGE_TYPE_THERMAL_AUTOMATIC = 3,
     DIS_PDU_STEALTH_SPECTRUM_IMAGE_TYPE_THERMAL_QUARTER_RESOLUTION = 4,
     DIS_PDU_STEALTH_SPECTRUM_IMAGE_TYPE_THERMAL_HALF_RESOLUTION = 5
}  DIS_PDU_Stealth_Spectrum_Image_Type;

/* SISO-REF-010 [UID 575] */
typedef enum
{
     DIS_PDU_STEALTH_STATE_SENSOR_MODE_OTHER = 0,
     DIS_PDU_STEALTH_STATE_SENSOR_MODE_OUT_THE_WINDOW_OTW = 1,
     DIS_PDU_STEALTH_STATE_SENSOR_MODE_INFRARED_IR = 2,
     DIS_PDU_STEALTH_STATE_SENSOR_MODE_NIGHT_VISION_GOGGLES_NVG = 3
}  DIS_PDU_Stealth_State_Sensor_Mode;

/* SISO-REF-010 [UID 576] */
typedef enum
{
     DIS_PDU_STEALTH_CONTROL_ATTACH_COMMAND_SPECIFIC_ENTITY = 0,
     DIS_PDU_STEALTH_CONTROL_ATTACH_COMMAND_ANY_ENTITY = 1,
     DIS_PDU_STEALTH_CONTROL_ATTACH_COMMAND_DETACH = 2
}  DIS_PDU_Stealth_Control_Attach_Command;

/* SISO-REF-010 [UID 577] */
typedef enum
{
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_FREE = 0,
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_HUG = 1,
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_TETHER = 2,
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_COMPASS = 3,
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_ORBIT = 4,
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_MIMIC = 5,
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_NO = 6,
     DIS_PDU_STEALTH_CONTROL_ATTACHMENT_MODE_GUNSIGHT = 7
}  DIS_PDU_Stealth_Control_Attachment_Mode;

/* SISO-REF-010 [UID 578] */
typedef enum
{
     DIS_PDU_GATEWAY_PDU_TRAFFIC_STATUS_NO_TRAFFIC = 0,
     DIS_PDU_GATEWAY_PDU_TRAFFIC_STATUS_TRAFFIC_CAN_ONLY_FLOW_TO_DESTINATION = 1,
     DIS_PDU_GATEWAY_PDU_TRAFFIC_STATUS_TRAFFIC_CAN_ONLY_FLOW_OUT_OF_DESTINATION = 2,
     DIS_PDU_GATEWAY_PDU_TRAFFIC_STATUS_BI_DIRECTIONAL_TRAFFIC = 3
}  DIS_PDU_Gateway_PDU_Traffic_Status;

/* SISO-REF-010 [UID 579] */
typedef enum
{
     DIS_PDU_GATEWAY_PROXIMITY_FILTER_SHAPE_SPHERICAL = 0,
     DIS_PDU_GATEWAY_PROXIMITY_FILTER_SHAPE_CYLINDRICAL = 1
}  DIS_PDU_Gateway_Proximity_Filter_Shape;

/* SISO-REF-010 [UID 580] */
typedef enum
{
     DIS_PDU_GATEWAY_FILTER_STATE_INACTIVE = 0,
     DIS_PDU_GATEWAY_FILTER_STATE_ACTIVE = 1
}  DIS_PDU_Gateway_Filter_State;

/* SISO-REF-010 [UID 581] */
typedef enum
{
     DIS_PDU_GATEWAY_FILTER_FUNCTION_INCLUSIVE = 0,
     DIS_PDU_GATEWAY_FILTER_FUNCTION_EXCLUSIVE_UNBLOCKABLE = 1
}  DIS_PDU_Gateway_Filter_Function;

/* SISO-REF-010 [UID 582] */
typedef enum
{
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_LINK_16_J_SERIES_MESSAGE = 0,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_BINARY_FILE = 1,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_VARIABLE_MESSAGE_FORMAT_VMF_K_SERIES_MESSAGE = 2,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_NATIONAL_IMAGERY_TRANSMISSION_FORMAT_SYSTEM_NITFS = 3,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_FORWARDED_MESSAGE_FWD_MSG = 4,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_UNITED_STATES_MESSAGE_TEXT_FORMAT_USMTF = 5,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_DOI_103 = 6,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_EXTENSIBLE_MARKUP_LANGUAGE_XML_MESSAGE_TEXT_FORMAT_MTF = 7,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_EXTENSIBLE_MARKUP_LANGUAGE_XML_VARIABLE_MESSAGE_FORMAT_VMF = 8,
     DIS_PDU_DIGITAL_MESSAGE_FORMAT_AFAPD = 16
}  DIS_PDU_Digital_Message_Format;

/* SISO-REF-010 [UID 583] */
typedef enum
{
     DIS_PDU_VMF_VERSION_TIDP_TE_R2 = 0,
     DIS_PDU_VMF_VERSION_TIDP_TE_R3 = 1,
     DIS_PDU_VMF_VERSION_TIDP_TE_R4 = 2,
     DIS_PDU_VMF_VERSION_TIDP_TE_R5 = 3,
     DIS_PDU_VMF_VERSION_TIDP_TE_R6 = 4,
     DIS_PDU_VMF_VERSION_TIDP_TE_R7 = 5,
     DIS_PDU_VMF_VERSION_6017 = 6,
     DIS_PDU_VMF_VERSION_6017A = 7,
     DIS_PDU_VMF_VERSION_6017B = 8,
     DIS_PDU_VMF_VERSION_6017C = 9,
     DIS_PDU_VMF_VERSION_DCX2_AV = 16
}  DIS_PDU_VMF_Version;

/* SISO-REF-010 [UID 584] */
typedef enum
{
     DIS_PDU_USMTF_VERSION_USMTF_1993 = 0,
     DIS_PDU_USMTF_VERSION_USMTF_1995 = 1,
     DIS_PDU_USMTF_VERSION_USMTF_1997 = 2,
     DIS_PDU_USMTF_VERSION_USMTF_1998 = 3,
     DIS_PDU_USMTF_VERSION_USMTF_1999 = 4,
     DIS_PDU_USMTF_VERSION_USMTF_2000 = 5,
     DIS_PDU_USMTF_VERSION_USMTF_2001 = 6,
     DIS_PDU_USMTF_VERSION_USMTF_2002 = 7,
     DIS_PDU_USMTF_VERSION_USMTF_2003 = 8,
     DIS_PDU_USMTF_VERSION_USMTF_2004 = 9
}  DIS_PDU_USMTF_Version;

/* SISO-REF-010 [UID 585] */
typedef enum
{
     DIS_PDU_AFAPD_VERSION_AFAPD_AV = 0,
     DIS_PDU_AFAPD_VERSION_AFAPD_BG = 1
}  DIS_PDU_AFAPD_Version;

/* SISO-REF-010 [UID 586] */
typedef enum
{
     DIS_PDU_DIGITAL_MESSAGE_PRECEDENCE_UNDEFINED = 0,
     DIS_PDU_DIGITAL_MESSAGE_PRECEDENCE_EMERGENCY = 2,
     DIS_PDU_DIGITAL_MESSAGE_PRECEDENCE_FLASH = 4,
     DIS_PDU_DIGITAL_MESSAGE_PRECEDENCE_IMMEDIATE = 5,
     DIS_PDU_DIGITAL_MESSAGE_PRECEDENCE_PRIORITY = 6,
     DIS_PDU_DIGITAL_MESSAGE_PRECEDENCE_ROUTINE = 7
}  DIS_PDU_Digital_Message_Precedence;

/* SISO-REF-010 [UID 587] */
typedef enum
{
     DIS_PDU_DIGITAL_MESSAGE_CLASSIFICATION_UNCLASSIFIED = 0,
     DIS_PDU_DIGITAL_MESSAGE_CLASSIFICATION_CONFIDENTIAL = 1,
     DIS_PDU_DIGITAL_MESSAGE_CLASSIFICATION_SECRET = 2,
     DIS_PDU_DIGITAL_MESSAGE_CLASSIFICATION_TOP_SECRET = 3
}  DIS_PDU_Digital_Message_Classification;

/* SISO-REF-010 [UID 588] */
typedef enum
{
     DIS_PDU_DIGITAL_MESSAGE_DESTINATION_TYPE_URN = 0,
     DIS_PDU_DIGITAL_MESSAGE_DESTINATION_TYPE_IPV4 = 1,
     DIS_PDU_DIGITAL_MESSAGE_DESTINATION_TYPE_SUBSCRIBER_ID = 2
}  DIS_PDU_Digital_Message_Destination_Type;

/* SISO-REF-010 [UID 589] */
typedef enum
{
     DIS_PDU_TRANSMITTER_DETAIL_SATCOM_MODULATION_OTHER = 0,
     DIS_PDU_TRANSMITTER_DETAIL_SATCOM_MODULATION_NO_DELAY = 1
}  DIS_PDU_Transmitter_Detail_SATCOM_Modulation;

/* SISO-REF-010 [UID 600] */
typedef enum
{
     DIS_PDU_SUPPLY_DOMAIN_CLASS_1_SUBSISTENCE = 1,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_2_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES = 2,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_3_PETROLEUM_OILS_LUBRICANTS = 3,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_4_CONSTRUCTION_MATERIALS = 4,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_6_PERSONNEL_DEMAND_ITEMS = 6,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_7_MAJOR_ITEMS = 7,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_8_MEDICAL_MATERIAL = 8,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_9_REPAIR_PARTS_AND_COMPONENTS = 9,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_10_MATERIAL_TO_SUPPORT_NON_MILITARY_PROGRAMS = 10,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_11_SUPPLIES_NON_DOCTRINAL = 11,
     DIS_PDU_SUPPLY_DOMAIN_CLASS_12_SLING_LOADS_NON_DOCTRINAL = 12
}  DIS_PDU_Supply_Domain;

/* SISO-REF-010 [UID 601] */
typedef enum
{
     DIS_PDU_CLASS_1_SUPPLY_CATEGORY_SUBSISTENCE_OTHER = 1,
     DIS_PDU_CLASS_1_SUPPLY_CATEGORY_SUBSISTENCE_A_NON_PERISHABLE = 2,
     DIS_PDU_CLASS_1_SUPPLY_CATEGORY_SUBSISTENCE_C_COMBAT_RATIONS = 3,
     DIS_PDU_CLASS_1_SUPPLY_CATEGORY_SUBSISTENCE_R_REFRIGERATED = 4,
     DIS_PDU_CLASS_1_SUPPLY_CATEGORY_SUBSISTENCE_S_OTHER_NON_REFRIGERATED = 5,
     DIS_PDU_CLASS_1_SUPPLY_CATEGORY_SUBSISTENCE_W_WATER = 6
}  DIS_PDU_Class_1_Supply_Category_Subsistence;

/* SISO-REF-010 [UID 602] */
typedef enum
{
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_OTHER = 1,
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_A_AIR = 2,
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_B_GROUND_SUPPORT_MATERIEL = 3,
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_E_GENERAL_SUPPLIES = 4,
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_F_CLOTHING = 5,
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_G_ELECTRONICS = 6,
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_M_WEAPONS_HUMANS = 7,
     DIS_PDU_CLASS_2_SUPPLY_CATEGORY_CLOTHING_INDIVIDUAL_EQUIPMENT_TOOLS_ADMIN_SUPPLIES_T_INDUSTRIAL_SUPPLIES = 8
}  DIS_PDU_Class_2_Supply_Category_Clothing_Individual_Equipment_Tools_Admin_Supplies;

/* SISO-REF-010 [UID 603] */
typedef enum
{
     DIS_PDU_CLASS_3_SUPPLY_CATEGORY_PETROLEUM_OILS_LUBRICANTS_OTHER = 1,
     DIS_PDU_CLASS_3_SUPPLY_CATEGORY_PETROLEUM_OILS_LUBRICANTS_A_POL_FOR_AIR_VEHICLES = 2,
     DIS_PDU_CLASS_3_SUPPLY_CATEGORY_PETROLEUM_OILS_LUBRICANTS_W_POL_FOR_LAND_VEHICLES = 3,
     DIS_PDU_CLASS_3_SUPPLY_CATEGORY_PETROLEUM_OILS_LUBRICANTS_P_PACKAGED_POL = 4
}  DIS_PDU_Class_3_Supply_Category_Petroleum_Oils_Lubricants;

/* SISO-REF-010 [UID 604] */
typedef enum
{
     DIS_PDU_CLASS_4_SUPPLY_CATEGORY_CONSTRUCTION_MATERIALS_OTHER = 1,
     DIS_PDU_CLASS_4_SUPPLY_CATEGORY_CONSTRUCTION_MATERIALS_A_CONSTRUCTION = 2,
     DIS_PDU_CLASS_4_SUPPLY_CATEGORY_CONSTRUCTION_MATERIALS_B_BARRIER = 3
}  DIS_PDU_Class_4_Supply_Category_Construction_Materials;

/* SISO-REF-010 [UID 606] */
typedef enum
{
     DIS_PDU_CLASS_6_SUPPLY_CATEGORY_PERSONNEL_DEMAND_ITEMS_OTHER = 1
}  DIS_PDU_Class_6_Supply_Category_Personnel_Demand_Items;

/* SISO-REF-010 [UID 607] */
typedef enum
{
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_OTHER = 1,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_A_AIR_NOT_USED_AS_DESCRIBED_IN_AIR_DOMAIN = 2,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_B_GROUND_SUPPORT_MATERIEL = 3,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_D_ADMIN_VEHICLES_NOT_USED_AS_DESCRIBED_IN_LAND_DOMAIN = 4,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_G_ELECTRONICS = 5,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_J_RACKS_ADAPTORS_PYLONS = 6,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_K_TACTICAL_VEHICLES_NOT_USED_AS_DESCRIBED_IN_LAND_DOMAIN = 7,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_L_MISSILES_NOT_USED_AS_DESCRIBED_IN_MUNITION_DOMAIN = 8,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_M_WEAPONS = 9,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_N_SPECIAL_WEAPONS = 10,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_X_AIRCRAFT_ENGINES = 11,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_DROP_TANK = 20,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_CONFORMAL_FUEL_TANK = 21,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_LUGGAGE_POD = 22,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_ECM_POD = 23,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_PARA_DROGUE = 24,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_TARGETING_POD = 25,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_FAIRING = 26,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_AIR_REFUELLING_POD = 27,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_HEAVY_AIRDROP = 28,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_CONTAINER_DELIVERY_SYSTEM_CDS_AIRDROP = 29,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_ROCKET_POD_LAUNCHER = 30,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_TACTICAL_POD = 31,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_RECCE_POD = 32,
     DIS_PDU_CLASS_7_SUPPLY_CATEGORY_MAJOR_ITEMS_FLIR_POD = 33
}  DIS_PDU_Class_7_Supply_Category_Major_Items;

/* SISO-REF-010 [UID 608] */
typedef enum
{
     DIS_PDU_CLASS_8_SUPPLY_CATEGORY_MEDICAL_MATERIAL_OTHER = 1,
     DIS_PDU_CLASS_8_SUPPLY_CATEGORY_MEDICAL_MATERIAL_A_MEDICAL_MATERIEL = 2,
     DIS_PDU_CLASS_8_SUPPLY_CATEGORY_MEDICAL_MATERIAL_B_BLOOD_FLUIDS = 3
}  DIS_PDU_Class_8_Supply_Category_Medical_Material;

/* SISO-REF-010 [UID 609] */
typedef enum
{
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_OTHER = 1,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_A_AIR = 2,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_B_GROUND_SUPPORT_MATERIEL = 3,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_D_ADMIN_VEHICLES = 4,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_G_ELECTRONICS = 5,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_K_TACTICAL_VEHICLES = 6,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_L_MISSILES = 7,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_M_WEAPONS = 8,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_N_SPECIAL_WEAPONS = 9,
     DIS_PDU_CLASS_9_SUPPLY_CATEGORY_REPAIR_PARTS_AND_COMPONENTS_X_AIRCRAFT_ENGINES = 10
}  DIS_PDU_Class_9_Supply_Category_Repair_Parts_and_Components;

/* SISO-REF-010 [UID 610] */
typedef enum
{
     DIS_PDU_CLASS_10_SUPPLY_CATEGORY_MATERIAL_TO_SUPPORT_NON_MILITARY_PROGRAMS_OTHER = 1
}  DIS_PDU_Class_10_Supply_Category_Material_to_Support_Non_Military_Programs;

/* SISO-REF-010 [UID 611] */
typedef enum
{
     DIS_PDU_CLASS_11_SUPPLY_CATEGORY_SUPPLIES_NON_DOCTRINAL_OTHER = 1,
     DIS_PDU_CLASS_11_SUPPLY_CATEGORY_SUPPLIES_NON_DOCTRINAL_PALLETS = 2,
     DIS_PDU_CLASS_11_SUPPLY_CATEGORY_SUPPLIES_NON_DOCTRINAL_FUEL_TANKS_DRUMS_AND_BLADDERS = 3,
     DIS_PDU_CLASS_11_SUPPLY_CATEGORY_SUPPLIES_NON_DOCTRINAL_CHESTS = 4,
     DIS_PDU_CLASS_11_SUPPLY_CATEGORY_SUPPLIES_NON_DOCTRINAL_BOXES = 5
}  DIS_PDU_Class_11_Supply_Category_Supplies_Non_Doctrinal;

/* SISO-REF-010 [UID 612] */
typedef enum
{
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_OTHER = 1,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_BLIVET = 2,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_CRATE = 3,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_WATER_BUCKET = 4,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_VEHICLES = 5,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_HOWITZER = 6,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_COLLAPSIBLE = 7,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_BLADDER = 8,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_PALLET_OF_CRATES = 9,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_HELICOPTERS = 10,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_HOIST = 11,
     DIS_PDU_CLASS_12_SUPPLY_CATEGORY_SLING_LOADS_NON_DOCTRINAL_SLING_LOAD_CONCRETE_BLOCK = 12
}  DIS_PDU_Class_12_Supply_Category_Sling_Loads_Non_Doctrinal;

/* SISO-REF-010 [UID 620] */
typedef enum
{
     DIS_PDU_PLATFORM_PAINT_SCHEME_DEFAULT = 0,
     DIS_PDU_PLATFORM_PAINT_SCHEME_SOLID_COLOR_PRIMARY = 1,
     DIS_PDU_PLATFORM_PAINT_SCHEME_TWO_TONE_TOP_PRIMARY_BOTTOM_SECONDARY = 2,
     DIS_PDU_PLATFORM_PAINT_SCHEME_TWO_TONE_BODY_PRIMARY_BED_SECONDARY = 3,
     DIS_PDU_PLATFORM_PAINT_SCHEME_TWO_TONE_BODY_PRIMARY_TRIM_SECONDARY = 4,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CAMOUFLAGE_DESERT = 5,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CAMOUFLAGE_JUNGLE = 6,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CAMOUFLAGE_SNOW = 7,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CAMOUFLAGE_GRAVEL = 8,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CAMOUFLAGE_PAVEMENT = 9,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CAMOUFLAGE_SAND = 10,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CAMOUFLAGE_FOREST = 11,
     DIS_PDU_PLATFORM_PAINT_SCHEME_NATURAL_WOOD = 12,
     DIS_PDU_PLATFORM_PAINT_SCHEME_CLEAR = 13,
     DIS_PDU_PLATFORM_PAINT_SCHEME_UNITED_NATIONS_UN = 14,
     DIS_PDU_PLATFORM_PAINT_SCHEME_TAXI = 15,
     DIS_PDU_PLATFORM_PAINT_SCHEME_NEWS = 16,
     DIS_PDU_PLATFORM_PAINT_SCHEME_JINGLE = 17
}  DIS_PDU_Platform_Paint_Scheme;

/* SISO-REF-010 [UID 621] */
typedef enum
{
     DIS_PDU_PLATFORM_DECAL_SCHEME_NONE = 0,
     DIS_PDU_PLATFORM_DECAL_SCHEME_UNITED_NATIONS_UN = 1
}  DIS_PDU_Platform_Decal_Scheme;

/* SISO-REF-010 [UID 622] */
typedef enum
{
     DIS_PDU_PLATFORM_EXTERIOR_CONDITION_NOT_SPECIFIED = 0,
     DIS_PDU_PLATFORM_EXTERIOR_CONDITION_SLIGHT = 1,
     DIS_PDU_PLATFORM_EXTERIOR_CONDITION_MODERATE = 2,
     DIS_PDU_PLATFORM_EXTERIOR_CONDITION_HEAVY = 3
}  DIS_PDU_Platform_Exterior_Condition;

/* SISO-REF-010 [UID 623] */
typedef enum
{
     DIS_PDU_PLATFORM_MATERIAL_NOT_SPECIFIED = 0,
     DIS_PDU_PLATFORM_MATERIAL_METALLIC_SHINY = 1,
     DIS_PDU_PLATFORM_MATERIAL_PLASTIC_MATTE = 2,
     DIS_PDU_PLATFORM_MATERIAL_ROUGH_COARSE = 3
}  DIS_PDU_Platform_Material;

/* SISO-REF-010 [UID 624] */
typedef enum
{
     DIS_PDU_STATUS_INVINCIBLE_NOT_INVINCIBLE = 0,
     DIS_PDU_STATUS_INVINCIBLE_INVINCIBLE = 1
}  DIS_PDU_Status_Invincible;

/* SISO-REF-010 [UID 625] */
typedef enum
{
     DIS_PDU_STATUS_PRESENT_DOMAIN_UNCHANGED = 0,
     DIS_PDU_STATUS_PRESENT_DOMAIN_LAND = 1,
     DIS_PDU_STATUS_PRESENT_DOMAIN_AIR = 2,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SURFACE = 3,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SUBSURFACE = 4,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SPACE = 5,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SUBTERRANEAN_LAND = 6,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SUBTERRANEAN_AIR = 7,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SUBTERRANEAN_SURFACE = 8,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SUBTERRANEAN_SUBSURFACE = 9,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SUBSURFACE_TERRAIN = 10,
     DIS_PDU_STATUS_PRESENT_DOMAIN_SUBSURFACE_CAVES = 11,
     DIS_PDU_STATUS_PRESENT_DOMAIN_INTERIOR_OTHER_ENTITY_OBJECT = 12,
     DIS_PDU_STATUS_PRESENT_DOMAIN_EXTERIOR_OTHER_ENTITY_OBJECT = 13,
     DIS_PDU_STATUS_PRESENT_DOMAIN_ADMINISTRATIVELY_INACTIVE = 14
}  DIS_PDU_Status_Present_Domain;

/* SISO-REF-010 [UID 626] */
typedef enum
{
     DIS_PDU_STATUS_DISGUISE_NOT_SPECIFIED = 0,
     DIS_PDU_STATUS_DISGUISE_NO_DISGUISE = 1,
     DIS_PDU_STATUS_DISGUISE_DISGUISED = 2,
     DIS_PDU_STATUS_DISGUISE_INDISTINGUISHABLE = 3
}  DIS_PDU_Status_Disguise;

/* SISO-REF-010 [UID 627] */
typedef enum
{
     DIS_PDU_APPLICATION_CONTROL_TYPE_OTHER = 0,
     DIS_PDU_APPLICATION_CONTROL_TYPE_SHUTDOWN = 1,
     DIS_PDU_APPLICATION_CONTROL_TYPE_DATA_QUERY = 2,
     DIS_PDU_APPLICATION_CONTROL_TYPE_DATA = 3,
     DIS_PDU_APPLICATION_CONTROL_TYPE_SET_DATA = 4,
     DIS_PDU_APPLICATION_CONTROL_TYPE_ADD_DATA = 5,
     DIS_PDU_APPLICATION_CONTROL_TYPE_REMOVE_DATA = 6,
     DIS_PDU_APPLICATION_CONTROL_TYPE_STATUS = 7
}  DIS_PDU_Application_Control_Type;

/* SISO-REF-010 [UID 628] */
typedef enum
{
     DIS_PDU_APPLICATION_TYPE_OTHER = 0,
     DIS_PDU_APPLICATION_TYPE_RESOURCE_MANAGER = 1,
     DIS_PDU_APPLICATION_TYPE_SIMULATION_MANAGER = 2,
     DIS_PDU_APPLICATION_TYPE_GATEWAY = 3,
     DIS_PDU_APPLICATION_TYPE_STEALTH = 4,
     DIS_PDU_APPLICATION_TYPE_TACTICAL_INTERNET_INTERFACE = 5
}  DIS_PDU_Application_Type;

/* SISO-REF-010 [UID 630] */
typedef enum
{
     DIS_PDU_ENTITY_OFFSET_TYPE_CARRIER_ORIGIN = 0,
     DIS_PDU_ENTITY_OFFSET_TYPE_STATION_LOCATION = 1,
     DIS_PDU_ENTITY_OFFSET_TYPE_POSITION_MODIFIED_BY_CARRIER = 2,
     DIS_PDU_ENTITY_OFFSET_TYPE_ORIENTATION_MODIFIED_BY_CARRIER = 3,
     DIS_PDU_ENTITY_OFFSET_TYPE_CENTER_OF_GRAVITY = 100
}  DIS_PDU_Entity_Offset_Type;

/* SISO-REF-010 [UID 633] */
typedef enum
{
     DIS_PDU_LIFE_SAVING_EQUIPMENT_LIFEBOAT = 1,
     DIS_PDU_LIFE_SAVING_EQUIPMENT_LIFERAFT = 2,
     DIS_PDU_LIFE_SAVING_EQUIPMENT_MOB_BOAT = 3,
     DIS_PDU_LIFE_SAVING_EQUIPMENT_LIFEBUOY = 4
}  DIS_PDU_Life_Saving_Equipment;

/* SISO-REF-010 [UID 650] */
typedef enum
{
     DIS_PDU_TIME_OF_DAY_MODE_SCENE = 0,
     DIS_PDU_TIME_OF_DAY_MODE_STATIC = 1,
     DIS_PDU_TIME_OF_DAY_MODE_CONTINUOUS = 2
}  DIS_PDU_Time_of_Day_Mode;

/* SISO-REF-010 [UID 651] */
typedef enum
{
     DIS_PDU_TIME_OF_DAY_SCENE_DAWN = 0,
     DIS_PDU_TIME_OF_DAY_SCENE_DAY = 1,
     DIS_PDU_TIME_OF_DAY_SCENE_DUSK = 2,
     DIS_PDU_TIME_OF_DAY_SCENE_NIGHT = 3
}  DIS_PDU_Time_of_Day_Scene;

/* SISO-REF-010 [UID 652] */
typedef enum
{
     DIS_PDU_NIGHT_TIME_ILLUMINATION_NONE = 0,
     DIS_PDU_NIGHT_TIME_ILLUMINATION_STARLIGHT = 1,
     DIS_PDU_NIGHT_TIME_ILLUMINATION_QUARTER_MOON = 2,
     DIS_PDU_NIGHT_TIME_ILLUMINATION_HALF_MOON = 3,
     DIS_PDU_NIGHT_TIME_ILLUMINATION_THREE_QUARTER_MOON = 4,
     DIS_PDU_NIGHT_TIME_ILLUMINATION_FULL_MOON = 5
}  DIS_PDU_Night_Time_Illumination;

/* SISO-REF-010 [UID 653] */
typedef enum
{
     DIS_PDU_PRECIPITATION_TYPE_NONE = 0,
     DIS_PDU_PRECIPITATION_TYPE_OTHER = 1,
     DIS_PDU_PRECIPITATION_TYPE_RAIN = 2,
     DIS_PDU_PRECIPITATION_TYPE_SNOW = 3,
     DIS_PDU_PRECIPITATION_TYPE_HAIL = 4,
     DIS_PDU_PRECIPITATION_TYPE_SLEET = 5,
     DIS_PDU_PRECIPITATION_TYPE_FREEZING_RAIN = 6
}  DIS_PDU_Precipitation_Type;

/* SISO-REF-010 [UID 654] */
typedef enum
{
     DIS_PDU_RAINSOAK_OFF = 0,
     DIS_PDU_RAINSOAK_ON = 1
}  DIS_PDU_Rainsoak;

/* SISO-REF-010 [UID 655] */
typedef enum
{
     DIS_PDU_PRECIPITATION_RATE_NO_PRECIPITATION = 0,
     DIS_PDU_PRECIPITATION_RATE_LIGHT_PRECIPITATION = 1,
     DIS_PDU_PRECIPITATION_RATE_MODERATE_PRECIPITATION = 2,
     DIS_PDU_PRECIPITATION_RATE_HEAVY_PRECIPITATION = 3
}  DIS_PDU_Precipitation_Rate;

/* SISO-REF-010 [UID 656] */
typedef enum
{
     DIS_PDU_CLOUD_STATUS_OFF = 0,
     DIS_PDU_CLOUD_STATUS_ON = 1
}  DIS_PDU_Cloud_Status;

/* SISO-REF-010 [UID 657] */
typedef enum
{
     DIS_PDU_CLOUD_TYPE_NONE = 0,
     DIS_PDU_CLOUD_TYPE_ALTOCUMULUS = 1,
     DIS_PDU_CLOUD_TYPE_ALTOSTRATUS = 2,
     DIS_PDU_CLOUD_TYPE_CIRROCUMULUS = 3,
     DIS_PDU_CLOUD_TYPE_CIRROSTRATUS = 4,
     DIS_PDU_CLOUD_TYPE_CIRRUS = 5,
     DIS_PDU_CLOUD_TYPE_CUMULONIMBUS = 6,
     DIS_PDU_CLOUD_TYPE_CUMULUS = 7,
     DIS_PDU_CLOUD_TYPE_NIMBOSTRATUS = 8,
     DIS_PDU_CLOUD_TYPE_STRATOCUMULUS = 9,
     DIS_PDU_CLOUD_TYPE_STRATUS = 10
}  DIS_PDU_Cloud_Type;

/* SISO-REF-010 [UID 658] */
typedef enum
{
     DIS_PDU_CLOUD_DENSITY_NOT_SPECIFIED = 0,
     DIS_PDU_CLOUD_DENSITY_CLEAR = 1,
     DIS_PDU_CLOUD_DENSITY_SCATTERED = 2,
     DIS_PDU_CLOUD_DENSITY_BROKEN = 3,
     DIS_PDU_CLOUD_DENSITY_OVERCAST = 4
}  DIS_PDU_Cloud_Density;

/* SISO-REF-010 [UID 659] */
typedef enum
{
     DIS_PDU_SCUD_FLAGS_OFF = 0,
     DIS_PDU_SCUD_FLAGS_BOTTOM = 1,
     DIS_PDU_SCUD_FLAGS_TOP_ON = 2,
     DIS_PDU_SCUD_FLAGS_TOP_AND_BOTTOM = 3
}  DIS_PDU_Scud_Flags;

/* SISO-REF-010 [UID 660] */
typedef enum
{
     DIS_PDU_GROUND_FOG_STATUS_OFF = 0,
     DIS_PDU_GROUND_FOG_STATUS_ON = 1
}  DIS_PDU_Ground_Fog_Status;

/* SISO-REF-010 [UID 661] */
typedef enum
{
     DIS_PDU_HAZE_STATUS_OFF = 0,
     DIS_PDU_HAZE_STATUS_ON = 1
}  DIS_PDU_Haze_Status;

/* SISO-REF-010 [UID 662] */
typedef enum
{
     DIS_PDU_HAZE_TYPE_NO_HAZE = 0,
     DIS_PDU_HAZE_TYPE_OTHER_HAZE = 1,
     DIS_PDU_HAZE_TYPE_RURUAL_HAZE = 2,
     DIS_PDU_HAZE_TYPE_MARITIME_HAZE = 3,
     DIS_PDU_HAZE_TYPE_URBAN_HAZE = 4,
     DIS_PDU_HAZE_TYPE_TROPOSPHERIC_HAZE = 5,
     DIS_PDU_HAZE_TYPE_ADVECTIVE_FOG_HAZE = 6,
     DIS_PDU_HAZE_TYPE_RADIATIVE_FOG_HAZE = 7,
     DIS_PDU_HAZE_TYPE_DESERT_HAZE = 8,
     DIS_PDU_HAZE_TYPE_DESERT_SUMMER_HAZE = 9,
     DIS_PDU_HAZE_TYPE_DESERT_WINTER_HAZE = 10,
     DIS_PDU_HAZE_TYPE_TEMPERATE_SUMMER_DAY = 11,
     DIS_PDU_HAZE_TYPE_TEMPERATE_SUMMER_NIGHT = 12,
     DIS_PDU_HAZE_TYPE_TEMPERATE_WINTER = 13,
     DIS_PDU_HAZE_TYPE_DUST_STORM_HAZE = 14,
     DIS_PDU_HAZE_TYPE_SNOW_HAZE = 15,
     DIS_PDU_HAZE_TYPE_BLOWING_SNOW_HAZE = 16,
     DIS_PDU_HAZE_TYPE_FOG_AND_SNOW_HAZE = 17
}  DIS_PDU_Haze_Type;

/* SISO-REF-010 [UID 663] */
typedef enum
{
     DIS_PDU_LIGHTNING_STATUS_OFF = 0,
     DIS_PDU_LIGHTNING_STATUS_INSTANTANEOUS_LIGHTNING = 1,
     DIS_PDU_LIGHTNING_STATUS_RANDOM_LIGHTNING = 2
}  DIS_PDU_Lightning_Status;

/* SISO-REF-010 [UID 664] */
typedef enum
{
     DIS_PDU_THUNDER_STATUS_OFF = 0,
     DIS_PDU_THUNDER_STATUS_INSTANTANEOUS_THUNDER = 1,
     DIS_PDU_THUNDER_STATUS_RANDOM_THUNDER = 2,
     DIS_PDU_THUNDER_STATUS_THUNDER_COINCIDENT_WITH_LIGHTNING = 3,
     DIS_PDU_THUNDER_STATUS_OVERCAST = 4
}  DIS_PDU_Thunder_Status;

/* SISO-REF-010 [UID 665] */
typedef enum
{
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_GROUND_FOG = 0,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_CLOUD_LAYER_1 = 1,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_CLOUD_LAYER_2 = 2,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_CLOUD_LAYER_3 = 3,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_RAIN = 4,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_SNOW = 5,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_SLEET = 6,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_HAIL = 7,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_SAND = 8,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_DUST = 9,
     DIS_PDU_WEATHER_STATE_LAYER_TYPE_HAZE = 10
}  DIS_PDU_Weather_State_Layer_Type;

/* SISO-REF-010 [UID 670] */
typedef enum
{
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_NOT_SPECIFIED = 0,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_UP = 1,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_DOWN = 2,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_LEFT = 3,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_RIGHT = 4,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_FORWARD = 5,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_BACKWARD = 6,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_SPECIFIED_GLOBAL_LOCATION = 7,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_SPECIFIED_ORIENTATION_VECTOR = 8,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_TRACKED_CENTER = 9,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_TRACKED_STATION = 10,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_OUT_STATION_NOT_TRACKED = 11,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_OUT_STATION_TRACKED_CENTER = 12,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_OUT_STATION_TRACKED_STATION = 13,
     DIS_PDU_HEAD_GAZING_WEAPON_AIMING_TYPE_STOWED_EYES_CLOSED = 14
}  DIS_PDU_Head_Gazing_Weapon_Aiming_Type;

/* SISO-REF-010 [UID 671] */
typedef enum
{
     DIS_PDU_ACTION_SEQUENCE_TYPE_NOT_SPECIFIED = 0,
     DIS_PDU_ACTION_SEQUENCE_TYPE_STANDING_UP_UNSPECIFIED = 1,
     DIS_PDU_ACTION_SEQUENCE_TYPE_STANDING_UP_USING_ONLY_HANDS = 2,
     DIS_PDU_ACTION_SEQUENCE_TYPE_STANDING_UP_USING_ONLY_KNEES = 3,
     DIS_PDU_ACTION_SEQUENCE_TYPE_STANDING_UP_USING_ONLY_ELBOWS = 4,
     DIS_PDU_ACTION_SEQUENCE_TYPE_KIP_UP = 99,
     DIS_PDU_ACTION_SEQUENCE_TYPE_FALLING_DOWN_UNSPECIFIED = 100,
     DIS_PDU_ACTION_SEQUENCE_TYPE_FALLING_DOWN_FACE_FIRST = 101,
     DIS_PDU_ACTION_SEQUENCE_TYPE_FALLING_DOWN_BACKWARD = 102,
     DIS_PDU_ACTION_SEQUENCE_TYPE_FALLING_DOWN_LEFT_SIDE = 103,
     DIS_PDU_ACTION_SEQUENCE_TYPE_FALLING_DOWN_RIGHT_SIDE = 104,
     DIS_PDU_ACTION_SEQUENCE_TYPE_FALLING_DOWN_CRUMPLING = 105,
     DIS_PDU_ACTION_SEQUENCE_TYPE_KNEELING_UNSPECIFIED = 200,
     DIS_PDU_ACTION_SEQUENCE_TYPE_KNEELING_BOTH_REAR_KNEES = 201,
     DIS_PDU_ACTION_SEQUENCE_TYPE_KNEELING_RIGHT_KNEE = 202,
     DIS_PDU_ACTION_SEQUENCE_TYPE_KNEELING_LEFT_KNEE = 203,
     DIS_PDU_ACTION_SEQUENCE_TYPE_KNEELING_ALL_KNEES_FOR_4_LEGGED_LIFEFORM = 204,
     DIS_PDU_ACTION_SEQUENCE_TYPE_CROUCHING_BENDING = 205,
     DIS_PDU_ACTION_SEQUENCE_TYPE_CROUCHING_SQUATTING = 206,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_UNSPECIFIED = 300,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_UNHEEDED = 301,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_CAREFUL = 302,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_TIP_TOE = 303,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_BACKWARDS = 304,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_UPSTAIRS = 305,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_DOWNSTAIRS = 306,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_LIMPING = 307,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_DRAGGING_INJURED_UNSPECIFIED_LEG = 308,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_DRAGGING_INJURED_RIGHT_LEG = 309,
     DIS_PDU_ACTION_SEQUENCE_TYPE_WALKING_DRAGGING_INJURED_LEFT_LEG = 310,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RUNNING_UNSPECIFIED = 400,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RUNNING_FULL = 401,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RUNNING_JOGGING = 402,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RUNNING_BACKWARDS = 403,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RUNNING_UPSTAIRS = 404,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RUNNING_DOWNSTAIRS = 405,
     DIS_PDU_ACTION_SEQUENCE_TYPE_CRAWLING_UNSPECIFIED = 500,
     DIS_PDU_ACTION_SEQUENCE_TYPE_CRAWLING_HANDS_AND_KNEES = 501,
     DIS_PDU_ACTION_SEQUENCE_TYPE_CRAWLING_ELBOWS = 502,
     DIS_PDU_ACTION_SEQUENCE_TYPE_CRAWLING_HANDS_DRAGGING_FEET = 503,
     DIS_PDU_ACTION_SEQUENCE_TYPE_VOMITING_UNSPECIFIED = 600,
     DIS_PDU_ACTION_SEQUENCE_TYPE_VOMITING_HANDS_AND_KNEES = 601,
     DIS_PDU_ACTION_SEQUENCE_TYPE_VOMITING_BENDING_OVER = 602,
     DIS_PDU_ACTION_SEQUENCE_TYPE_EATING_UNSPECIFIED = 700,
     DIS_PDU_ACTION_SEQUENCE_TYPE_DRINKING_UNSPECIFIED = 750,
     DIS_PDU_ACTION_SEQUENCE_TYPE_SLEEPING_UNSPECIFIED = 800,
     DIS_PDU_ACTION_SEQUENCE_TYPE_BREATHING_UNSPECIFIED = 900,
     DIS_PDU_ACTION_SEQUENCE_TYPE_BREATHING_HARD = 901,
     DIS_PDU_ACTION_SEQUENCE_TYPE_BREATHING_SLOW = 902,
     DIS_PDU_ACTION_SEQUENCE_TYPE_BREATHING_PANTING = 903,
     DIS_PDU_ACTION_SEQUENCE_TYPE_URINATING = 910,
     DIS_PDU_ACTION_SEQUENCE_TYPE_DEFECATING = 920,
     DIS_PDU_ACTION_SEQUENCE_TYPE_MOUNTING_A_VEHICLE = 1000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_DISMOUNTING_A_VEHICLE = 2000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_FIRING_USING_A_PORTABLE_WEAPON_INCLUDING_KNIVES = 3000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_THROWING_A_PORTABLE_WEAPON_OBJECT = 4000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_USING_CARRIED_EQUIPMENT = 5000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_USING_OTHER_EQUIPMENT_NOT_OPERATING_A_VEHICLE = 10000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_OPERATING_A_VEHICLE = 11000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RIDING_A_LIFEFORM = 12000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_MOUNTING_A_LIFEFORM = 13000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_DISMOUNTING_A_LIFEFORM = 14000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RELOADING_A_PORTABLE_WEAPON = 15000,
     DIS_PDU_ACTION_SEQUENCE_TYPE_RELOADING_A_NON_PORTABLE_WEAPON = 17000
}  DIS_PDU_Action_Sequence_Type;

/* SISO-REF-010 [UID 672] */
typedef enum
{
     DIS_PDU_ACTION_SEQUENCE_BODY_POSITION_TYPE_NOT_SPECIFIED = 0
}  DIS_PDU_Action_Sequence_Body_Position_Type;

/* SISO-REF-010 [UID 673] */
typedef enum
{
     DIS_PDU_ACTION_SEQUENCE_LOOP_MODE_ONE_SHOT = 0,
     DIS_PDU_ACTION_SEQUENCE_LOOP_MODE_CONTINUOUS = 1
}  DIS_PDU_Action_Sequence_Loop_Mode;

/* SISO-REF-010 [UID 674] */
typedef enum
{
     DIS_PDU_ACTION_SEQUENCE_ANIMATION_DIRECTION_FORWARD = 0,
     DIS_PDU_ACTION_SEQUENCE_ANIMATION_DIRECTION_REVERSE = 1
}  DIS_PDU_Action_Sequence_Animation_Direction;

/* SISO-REF-010 [UID 675] */
typedef enum
{
     DIS_PDU_ACTION_SEQUENCE_ANIMATION_STATE_STOP = 0,
     DIS_PDU_ACTION_SEQUENCE_ANIMATION_STATE_PAUSE = 1,
     DIS_PDU_ACTION_SEQUENCE_ANIMATION_STATE_START = 2,
     DIS_PDU_ACTION_SEQUENCE_ANIMATION_STATE_RESUME = 3
}  DIS_PDU_Action_Sequence_Animation_State;

/* SISO-REF-010 [UID 676] */
typedef enum
{
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_NOT_SPECIFIED = 0,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_UP = 1,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_DOWN = 2,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_LEFT = 3,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_RIGHT = 4,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_FORWARD = 5,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_BACKWARD = 6,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_STATION = 7,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_SPECIFIED_GLOBAL_LOCATION = 8,
     DIS_PDU_ACTION_SEQUENCE_ACTION_DIRECTION_SPECIFIED_ORIENTATION_VECTOR = 9
}  DIS_PDU_Action_Sequence_Action_Direction;

/* SISO-REF-010 [UID 677] */
typedef enum
{
     DIS_PDU_CYBER_ACKNOWLEDGE_RESPONSE_TYPE_ABLE_TO_COMPLY = 1,
     DIS_PDU_CYBER_ACKNOWLEDGE_RESPONSE_TYPE_UNABLE_TO_COMPLY = 2
}  DIS_PDU_Cyber_Acknowledge_Response_Type;

/* SISO-REF-010 [UID 678] */
typedef enum
{
     DIS_PDU_CYBER_ADMIN_TYPE_ADMINISTRATION = 1,
     DIS_PDU_CYBER_ADMIN_TYPE_ASSESSMENT = 2,
     DIS_PDU_CYBER_ADMIN_TYPE_COLLECTION = 3,
     DIS_PDU_CYBER_ADMIN_TYPE_CONFIGURATION = 4,
     DIS_PDU_CYBER_ADMIN_TYPE_EVALUATION = 5,
     DIS_PDU_CYBER_ADMIN_TYPE_FORENSICS = 6,
     DIS_PDU_CYBER_ADMIN_TYPE_INVESTIGATION = 7,
     DIS_PDU_CYBER_ADMIN_TYPE_OPERATIONS = 8,
     DIS_PDU_CYBER_ADMIN_TYPE_PROVISIONING = 9,
     DIS_PDU_CYBER_ADMIN_TYPE_TESTING = 10
}  DIS_PDU_Cyber_Admin_Type;

/* SISO-REF-010 [UID 679] */
typedef enum
{
     DIS_PDU_CYBER_CYBER_EVENT_PHASE_TYPE_START = 1,
     DIS_PDU_CYBER_CYBER_EVENT_PHASE_TYPE_SUSPEND = 2,
     DIS_PDU_CYBER_CYBER_EVENT_PHASE_TYPE_CONTINUE = 3,
     DIS_PDU_CYBER_CYBER_EVENT_PHASE_TYPE_CONTINUE_WITH_CHANGES = 4,
     DIS_PDU_CYBER_CYBER_EVENT_PHASE_TYPE_END = 5,
     DIS_PDU_CYBER_CYBER_EVENT_PHASE_TYPE_CANCEL = 6
}  DIS_PDU_Cyber_Cyber_Event_Phase_Type;

/* SISO-REF-010 [UID 680] */
typedef enum
{
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_ETHERNET = 1,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_WIFI = 2,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_ATM = 3,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_LOCALTALK = 4,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_PPP = 5,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_TOKEN_RING = 6,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_VLAN = 7,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_BLUETOOTH = 8,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_1553_BUS = 9,
     DIS_PDU_CYBER_DATA_LINK_PROTOCOL_TYPE_LLC = 10
}  DIS_PDU_Cyber_Data_Link_Protocol_Type;

/* SISO-REF-010 [UID 681] */
typedef enum
{
     DIS_PDU_CYBER_DATA_STATUS_TYPE_INTACT = 1,
     DIS_PDU_CYBER_DATA_STATUS_TYPE_COMPROMISED = 2,
     DIS_PDU_CYBER_DATA_STATUS_TYPE_CORRUPTED = 3,
     DIS_PDU_CYBER_DATA_STATUS_TYPE_MANIPULATED = 4,
     DIS_PDU_CYBER_DATA_STATUS_TYPE_NON_DECRYPTABLE = 5,
     DIS_PDU_CYBER_DATA_STATUS_TYPE_ERASED = 6
}  DIS_PDU_Cyber_Data_Status_Type;

/* SISO-REF-010 [UID 682] */
typedef enum
{
     DIS_PDU_CYBER_DATA_TYPE_FILE = 1,
     DIS_PDU_CYBER_DATA_TYPE_CODE = 2,
     DIS_PDU_CYBER_DATA_TYPE_CREDENTIALS = 3,
     DIS_PDU_CYBER_DATA_TYPE_COMMUNICATIONS = 4,
     DIS_PDU_CYBER_DATA_TYPE_SYSTEM_CONFIGURATION = 5
}  DIS_PDU_Cyber_Data_Type;

/* SISO-REF-010 [UID 683] */
typedef enum
{
     DIS_PDU_CYBER_DEVICE_TYPE_GENERIC = 1,
     DIS_PDU_CYBER_DEVICE_TYPE_NETWORKING = 2,
     DIS_PDU_CYBER_DEVICE_TYPE_COMPUTER_NODE = 3,
     DIS_PDU_CYBER_DEVICE_TYPE_PORTABLE_COMPUTER = 4,
     DIS_PDU_CYBER_DEVICE_TYPE_CONTROLLER = 5,
     DIS_PDU_CYBER_DEVICE_TYPE_STORAGE = 6,
     DIS_PDU_CYBER_DEVICE_TYPE_SENSOR = 7,
     DIS_PDU_CYBER_DEVICE_TYPE_PRINTER = 8,
     DIS_PDU_CYBER_DEVICE_TYPE_SCANNER = 9,
     DIS_PDU_CYBER_DEVICE_TYPE_COMMUNICATIONS = 10,
     DIS_PDU_CYBER_DEVICE_TYPE_HMI = 11,
     DIS_PDU_CYBER_DEVICE_TYPE_MONITORING = 12,
     DIS_PDU_CYBER_DEVICE_TYPE_IOT = 13,
     DIS_PDU_CYBER_DEVICE_TYPE_SECURITY = 14
}  DIS_PDU_Cyber_Device_Type;

/* SISO-REF-010 [UID 684] */
typedef enum
{
     DIS_PDU_CYBER_ENCRYPTION_TYPE_NOT_ENCRYPTED = 1,
     DIS_PDU_CYBER_ENCRYPTION_TYPE_DES = 2,
     DIS_PDU_CYBER_ENCRYPTION_TYPE_TRIPLE_DES = 3,
     DIS_PDU_CYBER_ENCRYPTION_TYPE_RSA = 4,
     DIS_PDU_CYBER_ENCRYPTION_TYPE_AES = 5,
     DIS_PDU_CYBER_ENCRYPTION_TYPE_TWOFISH = 6
}  DIS_PDU_Cyber_Encryption_Type;

/* SISO-REF-010 [UID 685] */
typedef enum
{
     DIS_PDU_CYBER_HARDWARE_DAMAGE_TYPE_BOOT_LOOP = 1,
     DIS_PDU_CYBER_HARDWARE_DAMAGE_TYPE_PHYSICAL_DESTRUCTION = 2,
     DIS_PDU_CYBER_HARDWARE_DAMAGE_TYPE_HARD_DRIVE_ERASED = 3
}  DIS_PDU_Cyber_Hardware_Damage_Type;

/* SISO-REF-010 [UID 686] */
typedef enum
{
     DIS_PDU_CYBER_HARDWARE_DEGRADE_TYPE_KEYBOARD = 1,
     DIS_PDU_CYBER_HARDWARE_DEGRADE_TYPE_MOUSE = 2,
     DIS_PDU_CYBER_HARDWARE_DEGRADE_TYPE_DISPLAY = 3,
     DIS_PDU_CYBER_HARDWARE_DEGRADE_TYPE_SOUND = 4,
     DIS_PDU_CYBER_HARDWARE_DEGRADE_TYPE_KERNEL_PANIC = 5,
     DIS_PDU_CYBER_HARDWARE_DEGRADE_TYPE_RANDOM_TEXT = 6,
     DIS_PDU_CYBER_HARDWARE_DEGRADE_TYPE_REBOOT = 7
}  DIS_PDU_Cyber_Hardware_Degrade_Type;

/* SISO-REF-010 [UID 687] */
typedef enum
{
     DIS_PDU_CYBER_LOAD_RATE_TYPE_UPLOAD = 1,
     DIS_PDU_CYBER_LOAD_RATE_TYPE_DOWNLOAD = 2
}  DIS_PDU_Cyber_Load_Rate_Type;

/* SISO-REF-010 [UID 688] */
typedef enum
{
     DIS_PDU_CYBER_MANIPULATION_TYPE_PACKET = 1,
     DIS_PDU_CYBER_MANIPULATION_TYPE_FILE = 2,
     DIS_PDU_CYBER_MANIPULATION_TYPE_DATABASE = 3
}  DIS_PDU_Cyber_Manipulation_Type;

/* SISO-REF-010 [UID 689] */
typedef enum
{
     DIS_PDU_CYBER_MESSAGE_TYPE_EMAIL = 1,
     DIS_PDU_CYBER_MESSAGE_TYPE_CHAT = 2,
     DIS_PDU_CYBER_MESSAGE_TYPE_TEXT = 3,
     DIS_PDU_CYBER_MESSAGE_TYPE_SOCIAL_MEDIA = 4
}  DIS_PDU_Cyber_Message_Type;

/* SISO-REF-010 [UID 690] */
typedef enum
{
     DIS_PDU_CYBER_NETWORK_PROTOCOL_TYPE_INTERNET_PROTOCOL = 1,
     DIS_PDU_CYBER_NETWORK_PROTOCOL_TYPE_NAT = 2,
     DIS_PDU_CYBER_NETWORK_PROTOCOL_TYPE_ICMP = 3,
     DIS_PDU_CYBER_NETWORK_PROTOCOL_TYPE_ARP = 4,
     DIS_PDU_CYBER_NETWORK_PROTOCOL_TYPE_RIP = 5,
     DIS_PDU_CYBER_NETWORK_PROTOCOL_TYPE_OSPF = 6,
     DIS_PDU_CYBER_NETWORK_PROTOCOL_TYPE_IPSEC = 7
}  DIS_PDU_Cyber_Network_Protocol_Type;

/* SISO-REF-010 [UID 691] */
typedef enum
{
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_MICROSOFT_DOS = 1,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_MICROSOFT_WINDOWS = 2,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_APPLE_MAC_OS = 3,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_DEC_VMS = 4,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_IBM_OS_2 = 5,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_ANDROID = 6,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_APPLE_IOS = 7,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_CISCO_IOS = 8,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_FIRMWARE = 9,
     DIS_PDU_CYBER_OPERATING_SYSTEM_TYPE_UNIX_LINUX = 10
}  DIS_PDU_Cyber_Operating_System_Type;

/* SISO-REF-010 [UID 692] */
typedef enum
{
     DIS_PDU_CYBER_PACKET_MANIPULATION_TYPE_DUPLICATION = 1,
     DIS_PDU_CYBER_PACKET_MANIPULATION_TYPE_CORRUPTION = 2,
     DIS_PDU_CYBER_PACKET_MANIPULATION_TYPE_REORDERING = 3,
     DIS_PDU_CYBER_PACKET_MANIPULATION_TYPE_DROPPED = 4
}  DIS_PDU_Cyber_Packet_Manipulation_Type;

/* SISO-REF-010 [UID 693] */
typedef enum
{
     DIS_PDU_CYBER_PHYSICAL_LAYER_TYPE_WIRED = 1,
     DIS_PDU_CYBER_PHYSICAL_LAYER_TYPE_WIRELESS = 2
}  DIS_PDU_Cyber_Physical_Layer_Type;

/* SISO-REF-010 [UID 694] */
typedef enum
{
     DIS_PDU_CYBER_RECON_TYPE_ACCOUNT_DISCOVERY = 1,
     DIS_PDU_CYBER_RECON_TYPE_AD_MALWARE = 2,
     DIS_PDU_CYBER_RECON_TYPE_ANTIVIRUS_TROJAN = 3,
     DIS_PDU_CYBER_RECON_TYPE_APPLICATION_WINDOW_DISCOVERY = 4,
     DIS_PDU_CYBER_RECON_TYPE_ARP_SCAN = 5,
     DIS_PDU_CYBER_RECON_TYPE_BANNER_GRABBING = 6,
     DIS_PDU_CYBER_RECON_TYPE_BOUNCE_SCAN = 7,
     DIS_PDU_CYBER_RECON_TYPE_BROWSER_BOOKMARK_DISCOVERY = 8,
     DIS_PDU_CYBER_RECON_TYPE_CLOUD_INFRASTRUCTURE_DISCOVERY = 9,
     DIS_PDU_CYBER_RECON_TYPE_CLOUD_SERVICE_DASHBOARD = 10,
     DIS_PDU_CYBER_RECON_TYPE_CLOUD_SERVICE_DISCOVERY = 11,
     DIS_PDU_CYBER_RECON_TYPE_COMPLIANCE = 12,
     DIS_PDU_CYBER_RECON_TYPE_CSRF = 13,
     DIS_PDU_CYBER_RECON_TYPE_DATABASE_INJECTION = 14,
     DIS_PDU_CYBER_RECON_TYPE_DATABASE_STRUCTURE = 15,
     DIS_PDU_CYBER_RECON_TYPE_DB_MANUFACTURE_VERSION = 16,
     DIS_PDU_CYBER_RECON_TYPE_DEVICE = 17,
     DIS_PDU_CYBER_RECON_TYPE_DNS = 18,
     DIS_PDU_CYBER_RECON_TYPE_DOMAIN = 19,
     DIS_PDU_CYBER_RECON_TYPE_DOMAIN_TRUST_DISCOVERY = 20,
     DIS_PDU_CYBER_RECON_TYPE_FILE_AND_DIRECTORY_DISCOVERY = 21,
     DIS_PDU_CYBER_RECON_TYPE_FIN_SCAN = 22,
     DIS_PDU_CYBER_RECON_TYPE_FTP = 23,
     DIS_PDU_CYBER_RECON_TYPE_HTTP = 24,
     DIS_PDU_CYBER_RECON_TYPE_IDLE_SCAN = 25,
     DIS_PDU_CYBER_RECON_TYPE_IGMP = 26,
     DIS_PDU_CYBER_RECON_TYPE_INPUT_VALIDATION = 27,
     DIS_PDU_CYBER_RECON_TYPE_IP = 28,
     DIS_PDU_CYBER_RECON_TYPE_LDAP_SCAN = 29,
     DIS_PDU_CYBER_RECON_TYPE_NET_BIOS_SCAN = 30,
     DIS_PDU_CYBER_RECON_TYPE_NETWORK_MAP = 31,
     DIS_PDU_CYBER_RECON_TYPE_NETWORK_SERVICE_SCANNING = 32,
     DIS_PDU_CYBER_RECON_TYPE_NETWORK_SHARE_DISCOVERY = 33,
     DIS_PDU_CYBER_RECON_TYPE_NETWORK_SNIFFING = 34,
     DIS_PDU_CYBER_RECON_TYPE_NTP = 35,
     DIS_PDU_CYBER_RECON_TYPE_NULL_SCAN = 36,
     DIS_PDU_CYBER_RECON_TYPE_OS_SCAN = 37,
     DIS_PDU_CYBER_RECON_TYPE_PASSWORD_POLICY_DISCOVERY = 38,
     DIS_PDU_CYBER_RECON_TYPE_PATCH_HISTORY = 39,
     DIS_PDU_CYBER_RECON_TYPE_PERIPHERAL_DEVICE_DISCOVERY = 40,
     DIS_PDU_CYBER_RECON_TYPE_PERMISSION_GROUPS_DISCOVERY = 41,
     DIS_PDU_CYBER_RECON_TYPE_PING = 42,
     DIS_PDU_CYBER_RECON_TYPE_PING_SCAN = 43,
     DIS_PDU_CYBER_RECON_TYPE_PORT_SCAN = 44,
     DIS_PDU_CYBER_RECON_TYPE_PORT_SWEEP = 45,
     DIS_PDU_CYBER_RECON_TYPE_PPP = 46,
     DIS_PDU_CYBER_RECON_TYPE_PROCESS_DISCOVERY = 47,
     DIS_PDU_CYBER_RECON_TYPE_QUERY_REGISTRY = 48,
     DIS_PDU_CYBER_RECON_TYPE_RARP = 49,
     DIS_PDU_CYBER_RECON_TYPE_REMOTE_SYSTEM_DISCOVERY = 50,
     DIS_PDU_CYBER_RECON_TYPE_ROOTKIT = 51,
     DIS_PDU_CYBER_RECON_TYPE_RPC_SCAN = 52,
     DIS_PDU_CYBER_RECON_TYPE_SERVICE = 53,
     DIS_PDU_CYBER_RECON_TYPE_SLIP = 54,
     DIS_PDU_CYBER_RECON_TYPE_SMTP = 55,
     DIS_PDU_CYBER_RECON_TYPE_SNMP_SWEEP = 56,
     DIS_PDU_CYBER_RECON_TYPE_SOFTWARE_DISCOVER = 57,
     DIS_PDU_CYBER_RECON_TYPE_SYN_SCAN = 58,
     DIS_PDU_CYBER_RECON_TYPE_SYSTEM_INFORMATION_DISCOVERY = 59,
     DIS_PDU_CYBER_RECON_TYPE_SYSTEM_NETWORK_CONFIGURATION_DISCOVERY = 60,
     DIS_PDU_CYBER_RECON_TYPE_SYSTEM_NETWORK_CONNECTIONS_DISCOVERY = 61,
     DIS_PDU_CYBER_RECON_TYPE_SYSTEM_OWNER_USER_DISCOVERY = 62,
     DIS_PDU_CYBER_RECON_TYPE_SYSTEM_SERVICE_DISCOVERY = 63,
     DIS_PDU_CYBER_RECON_TYPE_SYSTEM_TIME_DISCOVERY = 64,
     DIS_PDU_CYBER_RECON_TYPE_TCP_CONNECT = 65,
     DIS_PDU_CYBER_RECON_TYPE_TRACE_ROUTE = 66,
     DIS_PDU_CYBER_RECON_TYPE_UNIX_LINUX = 67,
     DIS_PDU_CYBER_RECON_TYPE_VIRTUALIZATION_SANDBOX_EVASION = 68,
     DIS_PDU_CYBER_RECON_TYPE_VULNERABILITY = 69,
     DIS_PDU_CYBER_RECON_TYPE_WEB_CRAWLER = 70,
     DIS_PDU_CYBER_RECON_TYPE_WINDOWS = 71,
     DIS_PDU_CYBER_RECON_TYPE_WIRELESS_ACTIVE = 72,
     DIS_PDU_CYBER_RECON_TYPE_WIRELESS_PASSIVE = 73,
     DIS_PDU_CYBER_RECON_TYPE_XMAS_SCAN = 74,
     DIS_PDU_CYBER_RECON_TYPE_XSS = 75
}  DIS_PDU_Cyber_Recon_Type;

/* SISO-REF-010 [UID 695] */
typedef enum
{
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_ADMINISTERS = 1,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_ADMINISTERED_BY = 2,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_COMPONENT_OF = 3,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_HAS_COMPONENT = 4,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_CONTAINED_IN = 5,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_CONTAINS = 6,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_PROVIDED_BY = 7,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_PROVIDES = 8,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_RESIDES_ON = 9,
     DIS_PDU_CYBER_RELATIONSHIP_TYPE_HAS_RESIDENT = 10
}  DIS_PDU_Cyber_Relationship_Type;

/* SISO-REF-010 [UID 696] */
typedef enum
{
     DIS_PDU_CYBER_SENSITIVITY_TYPE_UNCLASSIFIED = 1,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_CONFIDENTIAL = 2,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_FOUO = 3,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_SECRET = 4,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_SECRET_NOFORN = 5,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_TS = 6,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_TS_SCI = 7,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_NATO_RESTRICTED = 8,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_NATO_CONFIDENTIAL = 9,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_NATO_SECRET = 10,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_COSMIC_TOP_SECRET = 11,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_FVEY_PROPRIETARY = 12,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_PROPRIETARY = 13,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_PII = 14,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_HIPAA = 15,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_GDPR = 16,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_PUBLIC = 17,
     DIS_PDU_CYBER_SENSITIVITY_TYPE_CUI = 18
}  DIS_PDU_Cyber_Sensitivity_Type;

/* SISO-REF-010 [UID 697] */
typedef enum
{
     DIS_PDU_CYBER_SERVICE_TYPE_DNS = 1,
     DIS_PDU_CYBER_SERVICE_TYPE_EMAIL = 2,
     DIS_PDU_CYBER_SERVICE_TYPE_WEB = 3,
     DIS_PDU_CYBER_SERVICE_TYPE_DATABASE = 4,
     DIS_PDU_CYBER_SERVICE_TYPE_FILE = 5,
     DIS_PDU_CYBER_SERVICE_TYPE_CHAT = 6,
     DIS_PDU_CYBER_SERVICE_TYPE_FORUM = 7,
     DIS_PDU_CYBER_SERVICE_TYPE_SOCIAL_MEDIA = 8,
     DIS_PDU_CYBER_SERVICE_TYPE_CONTAINERIZATION = 9,
     DIS_PDU_CYBER_SERVICE_TYPE_VIRTUALIZATION = 10,
     DIS_PDU_CYBER_SERVICE_TYPE_NETWORK_TIME = 11
}  DIS_PDU_Cyber_Service_Type;

/* SISO-REF-010 [UID 698] */
typedef enum
{
     DIS_PDU_CYBER_SYSTEM_TYPE_GENERIC = 1,
     DIS_PDU_CYBER_SYSTEM_TYPE_SCADA = 2,
     DIS_PDU_CYBER_SYSTEM_TYPE_C2 = 3,
     DIS_PDU_CYBER_SYSTEM_TYPE_ICS = 4
}  DIS_PDU_Cyber_System_Type;

/* SISO-REF-010 [UID 700] */
typedef enum
{
     DIS_PDU_DEAD_RECKONING_TYPE_NOT_SPECIFIED = 0,
     DIS_PDU_DEAD_RECKONING_TYPE_LINEAR_VELOCITY = 1,
     DIS_PDU_DEAD_RECKONING_TYPE_LINEAR_ACCELERATION = 2,
     DIS_PDU_DEAD_RECKONING_TYPE_ANGULAR_VELOCITY = 3
}  DIS_PDU_Dead_Reckoning_Type;

/* SISO-REF-010 [UID 701] */
typedef enum
{
     DIS_PDU_SLING_LOAD_HOOK_TYPE_NOT_SPECIFIED = 0,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_SINGLE_HOOK = 1,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_FORWARD_HOOK = 2,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_CENTER_HOOK = 3,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_AFT_HOOK = 4,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_TANDEM_HOOK_FORE_AFT = 5,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_MISMANAGED_TANDEM_HOOK_FORE_CENTER = 6,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_MISMANAGED_TANDEM_HOOK_CENTER_AFT = 7,
     DIS_PDU_SLING_LOAD_HOOK_TYPE_ALL_HOOKS = 8
}  DIS_PDU_Sling_Load_Hook_Type;

/* SISO-REF-010 [UID 702] */
typedef enum
{
     DIS_PDU_COLLISION_ORIGIN_NO_STATEMENT = 0,
     DIS_PDU_COLLISION_ORIGIN_COLLISION_WITH_ATTACHED_PART = 1,
     DIS_PDU_COLLISION_ORIGIN_COLLISION_WITH_ARTICULATED_PART = 2,
     DIS_PDU_COLLISION_ORIGIN_COLLISION_WITH_MOUNTED_ENTITY = 3,
     DIS_PDU_COLLISION_ORIGIN_COLLISION_WITH_SLING_LOAD = 4,
     DIS_PDU_COLLISION_ORIGIN_COLLISION_WITH_TOWED_ENTITY = 5
}  DIS_PDU_Collision_Origin;

/* SISO-REF-010 [UID 715] */
typedef enum
{
     DIS_PDU_ISLAND_SUBCATEGORY_OTHER = 0,
     DIS_PDU_ISLAND_SUBCATEGORY_ISLANDS_1_000_2_499_KM2_386_965_SQ_MI = 1,
     DIS_PDU_ISLAND_SUBCATEGORY_ISLANDS_2_500_4_999_KM2_965_1_930_SQ_MI = 2,
     DIS_PDU_ISLAND_SUBCATEGORY_ISLANDS_5_000_9_999_KM2_1_931_3_861_SQ_MI = 3,
     DIS_PDU_ISLAND_SUBCATEGORY_ISLANDS_10_000_24_999_KM2_3_861_9_652_SQ_MI = 4,
     DIS_PDU_ISLAND_SUBCATEGORY_ISLANDS_25_000_99_999_KM2_9_653_38_610_SQ_MI = 5,
     DIS_PDU_ISLAND_SUBCATEGORY_ISLANDS_100_000_KM2_39_000_SQ_MI_AND_GREATER = 6
}  DIS_PDU_Island_Subcategory;

/* SISO-REF-010 [UID 716] */
typedef enum
{
     DIS_PDU_ELECTROMAGNETIC_EMISSION_EMITTER_STATUS_OFF = 0,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_EMITTER_STATUS_ON_NOT_EMITTING = 1,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_EMITTER_STATUS_ON_AND_EMITTING = 2,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_EMITTER_STATUS_TRIGGERED_AND_EMITTING = 3
}  DIS_PDU_Electromagnetic_Emission_Emitter_Status;

/* SISO-REF-010 [UID 717] */
typedef enum
{
     DIS_PDU_ELECTROMAGNETIC_EMISSIONS_BEAM_STABILIZATION_NOT_STABILIZED = 0,
     DIS_PDU_ELECTROMAGNETIC_EMISSIONS_BEAM_STABILIZATION_HORIZON_STABILIZED_WITH_NO_GIMBAL_LIMIT = 1,
     DIS_PDU_ELECTROMAGNETIC_EMISSIONS_BEAM_STABILIZATION_HORIZON_STABILIZED_WITH_GIMBAL_LIMIT = 2
}  DIS_PDU_Electromagnetic_Emissions_Beam_Stabilization;

/* SISO-REF-010 [UID 718] */
typedef enum
{
     DIS_PDU_ELECTROMAGNETIC_EMISSION_JAMMER_ARCHITECTURE_NONE = 0,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_JAMMER_ARCHITECTURE_OTHER = 1,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_JAMMER_ARCHITECTURE_TRANSPONDER = 10,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_JAMMER_ARCHITECTURE_ANALOG_REPEATER = 20,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_JAMMER_ARCHITECTURE_DIGITAL_RADIO_FREQUENCY_MEMORY_DRFM = 30,
     DIS_PDU_ELECTROMAGNETIC_EMISSION_JAMMER_ARCHITECTURE_DIRECT_DIGITAL_SYNTHESIS_DDS = 40
}  DIS_PDU_Electromagnetic_Emission_Jammer_Architecture;

/* SISO-REF-010 [UID 730] */
typedef enum
{
     DIS_PDU_LINK_11_MESSAGE_SUB_TYPE_NO_STATEMENT = 0,
     DIS_PDU_LINK_11_MESSAGE_SUB_TYPE_INTERROGATION = 1,
     DIS_PDU_LINK_11_MESSAGE_SUB_TYPE_DATA_START = 2,
     DIS_PDU_LINK_11_MESSAGE_SUB_TYPE_DATA = 3,
     DIS_PDU_LINK_11_MESSAGE_SUB_TYPE_DATA_STOP = 4
}  DIS_PDU_Link_11_Message_Sub_Type;

/* SISO-REF-010 [UID 731] */
typedef enum
{
     DIS_PDU_LINK_11_MESSAGE_TYPE_IDENTIFIER_NO_STATEMENT = 0,
     DIS_PDU_LINK_11_MESSAGE_TYPE_IDENTIFIER_NET_TEST = 1,
     DIS_PDU_LINK_11_MESSAGE_TYPE_IDENTIFIER_ROLL_CALL = 2,
     DIS_PDU_LINK_11_MESSAGE_TYPE_IDENTIFIER_PICKET_REPLY = 3,
     DIS_PDU_LINK_11_MESSAGE_TYPE_IDENTIFIER_SHORT_BROADCAST = 4,
     DIS_PDU_LINK_11_MESSAGE_TYPE_IDENTIFIER_BROADCAST = 5,
     DIS_PDU_LINK_11_MESSAGE_TYPE_IDENTIFIER_NET_SYNC = 6
}  DIS_PDU_Link_11_Message_Type_Identifier;

/* SISO-REF-010 [UID 732] */
typedef enum
{
     DIS_PDU_LINK_11_DATA_SIGNALLING_RATE_NO_STATEMENT = 0,
     DIS_PDU_LINK_11_DATA_SIGNALLING_RATE_1364_BPS = 1,
     DIS_PDU_LINK_11_DATA_SIGNALLING_RATE_2250_BPS = 2
}  DIS_PDU_Link_11_Data_Signalling_Rate;

/* SISO-REF-010 [UID 734] */
typedef enum
{
     DIS_PDU_LINK_11_SIGNAL_WAVEFORM_NO_STATEMENT_CLEW_FORMAT = 0,
     DIS_PDU_LINK_11_SIGNAL_WAVEFORM_CONVENTIONAL_LINK_ELEVEN_WAVEFORM_CLEW = 1,
     DIS_PDU_LINK_11_SIGNAL_WAVEFORM_SINGLE_TONE_LINK_ELEVEN_WAVEFORM_SLEW = 2
}  DIS_PDU_Link_11_Signal_Waveform;

/* SISO-REF-010 [UID 735] */
typedef enum
{
     DIS_PDU_LINK_11_11B_ENCRYPTION_FLAG_NO_ENCRYPTION_USED = 0,
     DIS_PDU_LINK_11_11B_ENCRYPTION_FLAG_ENCRYPTION_USED = 1
}  DIS_PDU_Link_11_11B_Encryption_Flag;

/* SISO-REF-010 [UID 736] */
typedef enum
{
     DIS_PDU_SISO_STD_002_VERSION_SISO_STD_002_2006 = 0,
     DIS_PDU_SISO_STD_002_VERSION_SISO_STD_002_2021 = 1
}  DIS_PDU_SISO_STD_002_Version;

/* SISO-REF-010 [UID 737] */
typedef enum
{
     DIS_PDU_LINK_11B_LINK_STATE_NO_STATEMENT = 0,
     DIS_PDU_LINK_11B_LINK_STATE_INACTIVE = 1,
     DIS_PDU_LINK_11B_LINK_STATE_READY = 2,
     DIS_PDU_LINK_11B_LINK_STATE_ACTIVE = 3,
     DIS_PDU_LINK_11B_LINK_STATE_OPERATIONAL = 4
}  DIS_PDU_Link_11B_Link_State;

/* SISO-REF-010 [UID 738] */
typedef enum
{
     DIS_PDU_LINK_11B_MODE_OF_OPERATION_NO_STATEMENT = 0,
     DIS_PDU_LINK_11B_MODE_OF_OPERATION_FULL_TRANSMISSION_OF_DATA = 1,
     DIS_PDU_LINK_11B_MODE_OF_OPERATION_LIMITED_TRANSMISSION_OF_DATA = 2,
     DIS_PDU_LINK_11B_MODE_OF_OPERATION_RECEIVE_ONLY = 3
}  DIS_PDU_Link_11B_Mode_of_Operation;

/* SISO-REF-010 [UID 739] */
typedef enum
{
     DIS_PDU_LINK_11B_MESSAGE_SUB_TYPE_NO_STATEMENT = 0,
     DIS_PDU_LINK_11B_MESSAGE_SUB_TYPE_TRANSMISSION_FRAME = 1,
     DIS_PDU_LINK_11B_MESSAGE_SUB_TYPE_STANDBY_SIGNAL = 2
}  DIS_PDU_Link_11B_Message_Sub_Type;

/* SISO-REF-010 [UID 740] */
typedef enum
{
     DIS_PDU_LINK_11B_DATA_SIGNALING_RATE_NO_STATEMENT = 0,
     DIS_PDU_LINK_11B_DATA_SIGNALING_RATE_NOT_USED_1 = 1,
     DIS_PDU_LINK_11B_DATA_SIGNALING_RATE_NOT_USED_2 = 2,
     DIS_PDU_LINK_11B_DATA_SIGNALING_RATE_1200_BPS = 3,
     DIS_PDU_LINK_11B_DATA_SIGNALING_RATE_2400_BPS = 4,
     DIS_PDU_LINK_11B_DATA_SIGNALING_RATE_600_BPS = 5
}  DIS_PDU_Link_11B_Data_Signaling_Rate;

/* SISO-REF-010 [UID 741] */
typedef enum
{
     DIS_PDU_LINK_11B_MODULATION_STANDARD_NO_STATEMENT = 0,
     DIS_PDU_LINK_11B_MODULATION_STANDARD_CCITT_V_23 = 1
}  DIS_PDU_Link_11B_Modulation_Standard;

/* SISO-REF-010 [UID 780] */
typedef enum
{
     DIS_PDU_CIGI_EXTENSION_PACKET_ID_IMAGE_CAPTURE_REQUEST_PACKET_ID = 4096,
     DIS_PDU_CIGI_EXTENSION_PACKET_ID_IMAGE_CAPTURE_RESPONSE_PACKET_ID = 4097,
     DIS_PDU_CIGI_EXTENSION_PACKET_ID_STATE_NOTIFICATION_REQUEST_PACKET_ID = 4098,
     DIS_PDU_CIGI_EXTENSION_PACKET_ID_STATE_NOTIFICATION_RESPONSE_PACKET_ID = 4099,
     DIS_PDU_CIGI_EXTENSION_PACKET_ID_GLOBALREFFRAMEDEF_PACKET_ID = 5000
}  DIS_PDU_CIGI_Extension_Packet_ID;

/* SISO-REF-010 [UID 790] */
typedef enum
{
     DIS_PDU_EE_BEAM_SHAPE_NOT_SPECIFIED = 0,
     DIS_PDU_EE_BEAM_SHAPE_CUSTOM_LOBES_ONLY = 1,
     DIS_PDU_EE_BEAM_SHAPE_OMNI_360_DEGREES_AT_GAIN_1_0 = 2,
     DIS_PDU_EE_BEAM_SHAPE_GAUSSIAN = 3,
     DIS_PDU_EE_BEAM_SHAPE_SINE_SINUSOIDAL = 4,
     DIS_PDU_EE_BEAM_SHAPE_SINC_SIN_X_X = 5,
     DIS_PDU_EE_BEAM_SHAPE_SINCNORMAL_SIN_X_PI_X_PI = 6,
     DIS_PDU_EE_BEAM_SHAPE_COSINE = 7,
     DIS_PDU_EE_BEAM_SHAPE_COSINE2_COS_SQUARED = 8,
     DIS_PDU_EE_BEAM_SHAPE_COSINEPEDESTAL_COS_ON_A_PEDASTAL = 9,
     DIS_PDU_EE_BEAM_SHAPE_COSECANT2_CSC_SQUARED = 10,
     DIS_PDU_EE_BEAM_SHAPE_SQUARE_CONSTANT_AMPLITUDE_FOR_ENTIRE_BEAMWIDTH = 11
}  DIS_PDU_EE_Beam_Shape;

/* SISO-REF-010 [UID 791] */
typedef enum
{
     DIS_PDU_LOBE_TYPE_AZIMUTH = 0,
     DIS_PDU_LOBE_TYPE_ELEVATION = 1
}  DIS_PDU_Lobe_Type;

/* SISO-REF-010 [UID 800] */
typedef enum
{
     DIS_PDU_LINK_16_VERSION_NO_STATEMENT = 0,
     DIS_PDU_LINK_16_VERSION_MIL_STD_6016C = 1,
     DIS_PDU_LINK_16_VERSION_MIL_STD_6016D = 2,
     DIS_PDU_LINK_16_VERSION_MIL_STD_6016E = 3,
     DIS_PDU_LINK_16_VERSION_MIL_STD_6016F = 4,
     DIS_PDU_LINK_16_VERSION_MIL_STD_6016F_C1 = 5,
     DIS_PDU_LINK_16_VERSION_STANAG_5516_ED_3 = 103,
     DIS_PDU_LINK_16_VERSION_STANAG_5516_ED_4 = 104,
     DIS_PDU_LINK_16_VERSION_STANAG_5516_ED_5 = 105,
     DIS_PDU_LINK_16_VERSION_STANAG_5516_ED_6 = 106,
     DIS_PDU_LINK_16_VERSION_STANAG_5516_ED_8 = 108
}  DIS_PDU_Link_16_Version;

/* SISO-REF-010 [UID 801] */
typedef enum
{
     DIS_PDU_AIRCRAFT_ID_SOURCE_MODE_S_AIRCRAFT_IDENTIFICATION_FIELD_VALUE = 0,
     DIS_PDU_AIRCRAFT_ID_SOURCE_GICB_IFF_DATA_RECORD_AVAILABLE = 1
}  DIS_PDU_Aircraft_ID_Source;

/* SISO-REF-010 [UID 802] */
typedef enum
{
     DIS_PDU_CLOTHING_IR_SIGNATURE_STANDARD_CLOTHING = 0,
     DIS_PDU_CLOTHING_IR_SIGNATURE_CAMOUFLAGE_NOT_JUST_PAINT_SCHEME = 1,
     DIS_PDU_CLOTHING_IR_SIGNATURE_THERMAL_BLANKET = 2,
     DIS_PDU_CLOTHING_IR_SIGNATURE_OTHER = 3
}  DIS_PDU_Clothing_IR_Signature;

/* SISO-REF-010 [UID 880] */
typedef enum
{
     DIS_PDU_LASER_BEAM_MOTION_PATTERN_NOT_SPECIFIED = 0,
     DIS_PDU_LASER_BEAM_MOTION_PATTERN_SPARKLE_NO_MOTION = 1,
     DIS_PDU_LASER_BEAM_MOTION_PATTERN_SNAKE = 2,
     DIS_PDU_LASER_BEAM_MOTION_PATTERN_ROPE = 3
}  DIS_PDU_Laser_Beam_Motion_Pattern;

/* SISO-REF-010 [UID 881] */
typedef enum
{
     DIS_PDU_LASER_BEAM_SHAPE_NOT_SPECIFIED = 0,
     DIS_PDU_LASER_BEAM_SHAPE_UNIFORM = 1,
     DIS_PDU_LASER_BEAM_SHAPE_GAUSSIAN = 2,
     DIS_PDU_LASER_BEAM_SHAPE_AIRY = 3
}  DIS_PDU_Laser_Beam_Shape;

/* SISO-REF-010 [UID 882] */
typedef enum
{
     DIS_PDU_LASER_BEAM_SPOT_TYPE_NOT_SPECIFIED = 0,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_TERRAIN_TERRAIN_MODEL = 1,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_TERRAIN_NO_TERRAIN_MODEL = 2,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_WATER_SURFACE = 3,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_FOLIAGE = 4,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_MAN_MADE_CULTURAL_FEATURE = 5,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_CLOUDS = 6,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_OBSCURANT_E_G_SMOKE = 7,
     DIS_PDU_LASER_BEAM_SPOT_TYPE_NON_INCIDENT_E_G_CLEAR_SKY = 8
}  DIS_PDU_Laser_Beam_Spot_Type;

/* SISO-REF-010 [UID 883] */
typedef enum
{
     DIS_PDU_LASER_BEAM_FUNCTION_NOT_SPECIFIED = 0,
     DIS_PDU_LASER_BEAM_FUNCTION_RANGE_FINDER = 1,
     DIS_PDU_LASER_BEAM_FUNCTION_DESIGNATOR = 2,
     DIS_PDU_LASER_BEAM_FUNCTION_BEAM_RIDER = 3,
     DIS_PDU_LASER_BEAM_FUNCTION_TARGET_SPOT_MARKER = 4,
     DIS_PDU_LASER_BEAM_FUNCTION_ILLUMINATOR_FLOOD = 5,
     DIS_PDU_LASER_BEAM_FUNCTION_JAMMER = 6
}  DIS_PDU_Laser_Beam_Function;

/* SISO-REF-010 [UID 884] */
typedef enum
{
     DIS_PDU_TARGET_SPOT_MARKER_FLASH_PATTERN_NO_FLASH_STEADY_ON = 0,
     DIS_PDU_TARGET_SPOT_MARKER_FLASH_PATTERN_FLASH_PATTERN_1 = 1,
     DIS_PDU_TARGET_SPOT_MARKER_FLASH_PATTERN_FLASH_PATTERN_2 = 2,
     DIS_PDU_TARGET_SPOT_MARKER_FLASH_PATTERN_FLASH_PATTERN_3 = 3,
     DIS_PDU_TARGET_SPOT_MARKER_FLASH_PATTERN_FLASH_PATTERN_4 = 4
}  DIS_PDU_Target_Spot_Marker_Flash_Pattern;

/* SISO-REF-010 [UID 885] */
typedef enum
{
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_NOT_SPECIFIED = 0,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RUDDER = 1,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_FLAP = 2,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_FLAP = 3,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_AILERON = 4,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_AILERON = 5,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_HELICOPTER_MAIN_ROTOR = 6,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_HELICOPTER_TAIL_ROTOR = 7,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_OTHER_AIRCRAFT_CONTROL_SURFACES_DEFINED_AS_NEEDED = 8,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PROPELLER_NUMBER_1 = 9,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PROPELLER_NUMBER_2 = 10,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PROPELLER_NUMBER_3 = 11,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PROPELLER_NUMBER_4 = 12,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_STABILATOR_STABILATOR_NUMBER_1 = 13,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_STABILATOR_STABILATOR_NUMBER_2 = 14,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_RUDDERVATOR_RUDDERVATOR_NUMBER_1 = 15,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_RUDDERVATOR_RUDDERVATOR_NUMBER_2 = 16,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_LEADING_EDGE_FLAP_SLAT = 17,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_LEADING_EDGE_FLAP_SLAT = 18,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_ELEVATOR = 19,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_ELEVATOR = 20,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CANARD_LEFT = 21,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CANARD_RIGHT = 22,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ELEVON_INNER_LEFT = 23,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ELEVON_INNER_RIGHT = 24,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ELEVON_MIDDLE_LEFT = 25,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ELEVON_MIDDLE_RIGHT = 26,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ELEVON_OUTER_LEFT = 27,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ELEVON_OUTER_RIGHT = 28,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CANOPY_AIRCRAFT = 29,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SPOILER_LEFT = 30,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SPOILER_RIGHT = 31,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PERISCOPE = 33,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_GENERIC_ANTENNA = 34,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SNORKEL = 35,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_OTHER_EXTENDIBLE_PARTS_DEFINED_AS_NEEDED = 36,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DIVE_PLANE_SAIL_LEFT = 37,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DIVE_PLANE_SAIL_RIGHT = 38,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DIVE_PLANE_BOW_LEFT = 39,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DIVE_PLANE_BOW_RIGHT = 40,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DIVE_PLANE_STERN_LEFT = 41,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DIVE_PLANE_STERN_RIGHT = 42,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_RUDDER = 43,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_RUDDER = 44,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_GEAR_DOOR_NOSE = 62,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_GEAR_DOOR_LEFT_MAIN = 63,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_GEAR_DOOR_RIGHT_MAIN = 64,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_GEAR = 65,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_TAIL_HOOK = 66,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SPEED_BRAKE = 67,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_DOOR_OF_PRIMARY_WEAPON_BAY = 68,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_DOOR_OF_PRIMARY_WEAPON_BAY = 69,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_TANK_OR_APC_HATCH = 70,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_WINGSWEEP = 71,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_BRIDGE_LAUNCHER = 72,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_BRIDGE_SECTION_1 = 73,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_BRIDGE_SECTION_2 = 74,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_BRIDGE_SECTION_3 = 75,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_BLADE_1 = 76,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_BLADE_2 = 77,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_BOOM = 78,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_ARM = 79,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_OTHER_FIXED_POSITION_PARTS_DEFINED_AS_NEEDED = 80,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_GEAR_NOSE = 81,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_GEAR_LEFT_MAIN = 82,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_GEAR_RIGHT_MAIN = 83,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DOORS_OF_LEFT_SIDE_WEAPON_BAY = 84,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DOORS_OF_RIGHT_SIDE_WEAPON_BAY = 85,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SPOT_SEARCH_LIGHT_1 = 86,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SPOT_SEARCH_LIGHT_2 = 87,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SPOT_SEARCH_LIGHT_3 = 88,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SPOT_SEARCH_LIGHT_4 = 89,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LANDING_LIGHT = 90,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_1 = 97,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_2 = 98,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_3 = 99,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_4 = 100,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_5 = 101,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_6 = 102,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_7 = 103,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_8 = 104,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_9 = 105,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_TURRET_NUMBER_10 = 106,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_1 = 107,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_2 = 108,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_3 = 109,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_4 = 110,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_5 = 111,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_6 = 112,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_7 = 113,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_8 = 114,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_9 = 115,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_NUMBER_10 = 116,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_1 = 117,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_2 = 118,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_3 = 119,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_4 = 120,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_5 = 121,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_6 = 122,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_7 = 123,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_8 = 124,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_9 = 125,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_LAUNCHER_10 = 126,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_1 = 127,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_2 = 128,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_3 = 129,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_4 = 130,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_5 = 131,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_6 = 132,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_7 = 133,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_8 = 134,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_9 = 135,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_DEFENSE_SYSTEMS_10 = 136,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_1 = 137,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_2 = 138,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_3 = 139,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_4 = 140,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_5 = 141,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_6 = 142,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_7 = 143,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_8 = 144,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_9 = 145,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_RADAR_10 = 146,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_1 = 147,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_2 = 148,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_3 = 149,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_4 = 150,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_5 = 151,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_6 = 152,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_7 = 153,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_8 = 154,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_9 = 155,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_TURRET_NUMBER_10 = 156,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_1 = 157,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_2 = 158,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_3 = 159,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_4 = 160,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_5 = 161,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_6 = 162,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_7 = 163,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_8 = 164,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_9 = 165,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_NUMBER_10 = 166,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_1 = 167,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_2 = 168,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_3 = 169,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_4 = 170,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_5 = 171,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_6 = 172,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_7 = 173,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_8 = 174,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_9 = 175,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_LAUNCHER_10 = 176,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_1 = 177,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_2 = 178,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_3 = 179,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_4 = 180,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_5 = 181,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_6 = 182,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_7 = 183,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_8 = 184,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_9 = 185,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_DEFENSE_SYSTEMS_10 = 186,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_1 = 187,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_2 = 188,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_3 = 189,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_4 = 190,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_5 = 191,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_6 = 192,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_7 = 193,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_8 = 194,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_9 = 195,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_RADAR_10 = 196,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DECK_ELEVATOR_1 = 197,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_DECK_ELEVATOR_2 = 198,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CATAPULT_1 = 199,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CATAPULT_2 = 200,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_JET_BLAST_DEFLECTOR_1 = 201,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_JET_BLAST_DEFLECTOR_2 = 202,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ARRESTOR_WIRES_1 = 203,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ARRESTOR_WIRES_2 = 204,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ARRESTOR_WIRES_3 = 205,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_WING_OR_ROTOR_FOLD = 206,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_FUSELAGE_FOLD = 207,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_MAIN_CARGO_DOOR = 208,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_RAMP = 209,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_AIR_TO_AIR_REFUELING_BOOM = 210,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_AERIAL_REFUELING_RECEPTACLE_DOOR = 211,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_AERIAL_REFUELING_RECEPTACLE_DOOR = 212,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_AERIAL_REFUELING_RECEPTACLE_LATCH = 213,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_1 = 214,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_2 = 215,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_3 = 216,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_4 = 217,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_5 = 218,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_6 = 219,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_7 = 220,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_8 = 221,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_9 = 222,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CARGO_DOOR_10 = 223,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CENTRE_REFUELLING_DROGUE = 224,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PORT_REFUELLING_DROGUE = 225,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_STARBOARD_REFUELLING_DROGUE = 226,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_ENGINE_EXHAUST_MAST = 227,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_1 = 228,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_2 = 229,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_3 = 230,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_4 = 231,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_5 = 232,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_6 = 233,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_7 = 234,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_8 = 235,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_9 = 236,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SUBMARINE_MAST_10 = 237,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_VECTORED_THRUST_NOZZLE = 238,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_DOOR_OF_THE_LEFT_WEAPON_BAY = 239,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_DOOR_OF_THE_LEFT_WEAPON_BAY = 240,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_DOOR_OF_THE_RIGHT_WEAPON_BAY = 241,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_DOOR_OF_THE_RIGHT_WEAPON_BAY = 242,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_GUN_DOOR = 243,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_COUNTERMEASURE_DOOR_LEFT = 244,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_COUNTERMEASURE_DOOR_RIGHT = 245,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_HOOK_DOOR_FORWARD = 246,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_HOOK_DOOR_AFT = 247,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LIFT_FAN_UPPER_DOOR = 248,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LIFT_FAN_LOWER_DOOR_LEFT = 249,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LIFT_FAN_LOWER_DOOR_RIGHT = 250,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_REFUEL_PROBE_DOOR = 251,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_LEFT_ENGINE_NACELLE = 252,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_RIGHT_ENGINE_NACELLE = 253,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_1ST_LEFT_WHEEL = 254,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_1ST_RIGHT_WHEEL = 255,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_2ND_LEFT_WHEEL = 256,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_2ND_RIGHT_WHEEL = 257,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_3RD_LEFT_WHEEL = 258,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_3RD_RIGHT_WHEEL = 259,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_4TH_LEFT_WHEEL = 260,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_4TH_RIGHT_WHEEL = 261,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_5TH_LEFT_WHEEL = 262,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_5TH_RIGHT_WHEEL = 263,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_6TH_LEFT_WHEEL = 264,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_6TH_RIGHT_WHEEL = 265,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_7TH_LEFT_WHEEL = 266,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_7TH_RIGHT_WHEEL = 267,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_8TH_LEFT_WHEEL = 268,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_8TH_RIGHT_WHEEL = 269,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_9TH_LEFT_WHEEL = 270,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_9TH_RIGHT_WHEEL = 271,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_10TH_LEFT_WHEEL = 272,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_10TH_RIGHT_WHEEL = 273,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_REFUELING_PROBE = 274,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_STEERING_WHEEL = 275,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CRANE_BODY = 276,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CRANE_ARM_1 = 277,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CRANE_ARM_2 = 278,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CRANE_ARM_3 = 279,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CRANE_BOOM = 280,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_CRANE_HOOK = 281,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_TRAILER = 282,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ROLLER_LEFT = 283,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_ROLLER_RIGHT = 284,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_PRIMARY_GUN_RECOIL = 285,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_CLASS_SECONDARY_GUN_RECOIL = 286
}  DIS_PDU_Articulated_Parts_Parameter_Type_Class;

/* SISO-REF-010 [UID 886] */
typedef enum
{
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_NOT_SPECIFIED = 0,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_POSITION = 1,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_EXTENSION = 2,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_X = 3,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_Y = 4,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_Z = 5,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_AZIMUTH = 6,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_ELEVATION = 7,
     DIS_PDU_ARTICULATED_PARTS_PARAMETER_TYPE_METRIC_ROTATION = 8
}  DIS_PDU_Articulated_Parts_Parameter_Type_Metric;

/* SISO-REF-010 [UID 889] */
typedef enum
{
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_1_DEFAULT_IS_SIDE_1_FRONT_SIDE_OR_NOT_APPLICABLE_IF_DAMAGE_AREAS_ARE_NOT_DEFINED = 0,
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_2_DEFAULT_IS_SIDE_2_RIGHT_SIDE = 1,
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_3_DEFAULT_IS_SIDE_3_BACK_SIDE = 2,
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_4_DEFAULT_IS_SIDE_4_LEFT_SIDE = 3,
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_5_DEFAULT_IS_CORNER_1_FRONT_RIGHT_CORNER = 4,
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_6_DEFAULT_IS_CORNER_2_BACK_RIGHT_CORNER = 5,
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_7_DEFAULT_IS_CORNER_3_BACK_LEFT_CORNER = 6,
     DIS_PDU_DAMAGE_AREA_DAMAGE_AREA_8_DEFAULT_IS_CORNER_4_FRONT_LEFT_CORNER = 7
}  DIS_PDU_Damage_Area;

/* SISO-REF-010-2023 Version 34 draft d11 - 21 July 2024 XML generated Content End */

/******************************************************************************
*
* Non-SISO
*
*******************************************************************************/

/* DIS Entity Information / Interaction PDUs
 */
static const true_false_string tfs_camouflage_uniform_color = { "Camouflage", "Uniform color" };
static const true_false_string tfs_mobility_kill = { "Mobility kill", "No mobility kill" };
static const true_false_string tfs_fire_power_kill = { "Fire-power kill", "No fire-power kill" };
static const true_false_string tfs_nvg_mode = { "Overt Lighting", "Covert Lighting (Night Vision Goggles)" };

static const value_string appearance_frozen_status_vals[] =
{
    { 0, "Not frozen" },
    { 1, "Frozen" },
    { 0, NULL }
};

static const value_string appearance_state_vals[] =
{
    { 0, "Active" },
    { 1, "Deactivated" },
    { 0, NULL }
};

static const value_string appearance_Weapon_1_vals[] =
{
    { 0, "No primary weapon present" },
    { 1, "Primary weapon is stowed" },
    { 2, "Primary weapon is deployed" },
    { 3, "Primary weapon is in firing position" },
    { 0, NULL }
};

static const value_string appearance_Weapon_2_vals[] =
{
    { 0, "No primary weapon present" },
    { 1, "Primary weapon is stowed" },
    { 2, "Primary weapon is deployed" },
    { 3, "Primary weapon is in firing position" },
    { 0, NULL }
};

static const value_string appearance_concealed_stationary_vals[] =
{
    { 0, "Not concealed" },
    { 1, "Entity in a prepared concealed position" },
    { 0, NULL }
};

static const value_string appearance_concealed_movement_vals[] =
{
    { 0, "Open movement" },
    { 1, "Rushes between covered positions" },
    { 0, NULL }
};

static const value_string appearance_smoke_entity_vals[] =
{
    { 0, "Not smoking" },
    { 1, "Smoke plume rising from the entity" },
    { 2, "Entity is emitting engine smoke" },
    { 3, "Entity is emitting engine smoke, and smoke plume is rising from the entity" },
    { 0, NULL }
};

static const value_string appearance_hatch_vals[] =
{
    { 0, "Not applicable" },
    { 1, "Primary hatch is closed" },
    { 2, "Primary hatch is popped" },
    { 3, "Primary hatch is popped and a person is visible under hatch" },
    { 4, "Primary hatch is open" },
    { 5, "Primary hatch is open and person is visible" },
    { 6, "Unused" },
    { 7, "Unused" },
    { 0, NULL }
};

static const value_string appearance_canopy_door_vals[] =
{
    { 0, "Not applicable" },
    { 1, "Single Canopy/Single Troop Door Closed" },
    { 2, "Front and Rear Canopy/Left and Right Troop Door Closed" },
    { 3, "Front Canopy/Left Troop Door Open" },
    { 4, "Single Canopy/Single Troop Door Open" },
    { 5, "Rear Canopy/Right Troop Door Open" },
    { 6, "Front and Rear Canopy/Left and Right Troop Door Open" },
    { 7, "Unused" },
    { 0, NULL }
};

static const value_string appearance_flaming_vals[] =
{
    { 0, "None" },
    { 1, "Flames present" },
    { 0, NULL }
};

static const value_string appearance_launcher_vals[] =
{
    { 0, "Not raised" },
    { 1, "Raised" },
    { 0, NULL }
};

static const value_string appearance_concealed_vals[] =
{
    { 0, "Not concealed" },
    { 1, "Entity in a prepared concealed position" },
    { 0, NULL }
};

static const value_string appearance_power_plant_status_vals[] =
{
    { 0, "Power plant off" },
    { 1, "Power plant on" },
    { 0, NULL }
};

static const value_string appearance_tent_vals[] =
{
    { 0, "Not extended" },
    { 1, "Extended" },
    { 0, NULL }
};

static const value_string appearance_surrentder_state_vals[] =
{
    { 0, "Not surrendered" },
    { 1, "Surrender" },
    { 0, NULL }
};

static const value_string appearance_masked_cloaked_vals[] =
{
    { 0, "Not Masked / Not Cloaked" },
    { 1, "Masked / Cloaked" },
    { 0, NULL }
};

static const value_string entity_dead_reckoning_algorithm_vals[] =
{
    { 0, "Other" },
    { 1, "Static (Entity does not move.)" },
    { 2, "DRM(F, P, W)" },
    { 3, "DRM(R, P, W)" },
    { 4, "DRM(R, V, W)" },
    { 5, "DRM(F, V, W)" },
    { 6, "DRM(F, P, B)" },
    { 7, "DRM(R, P, B)" },
    { 8, "DRM(R, V, B)" },
    { 9, "DRM(F, V, B)" },
    { 0, NULL }
};

static const value_string environmental_environment_status_last_vals[] =
{
    { 0, "Not Last" },
    { 1, "Last" },
    { 0, NULL }
};

static const value_string environmental_environment_status_on_vals[] =
{
    { 0, "Not Active" },
    { 1, "Active" },
    { 0, NULL }
};

static const value_string designator_code_name_vals[] =
{
    { 0, "Other" },
    { 1, "TBD" },
    { 0, NULL }
};

static const value_string designator_designator_code_vals[] =
{
    { 0, "Other" },
    { 1, "TBD" },
    { 0, NULL }
};

static const value_string dis_pdu_status_tei_vals[] = {
    { 0x0, "Entity owned by this simulation" },
    { 0x1, "Entity owned by different simulation" },
    { 0, NULL }
};

typedef enum
{
    DIS_PERSISTENT_OBJECT_TYPE_OTHER               = 0,
    DIS_PERSISTENT_OBJECT_TYPE_SIMULATOR_PRESENT   = 1,
    DIS_PERSISTENT_OBJECT_TYPE_DESCRIBE_OBJECT     = 2,
    DIS_PERSISTENT_OBJECT_TYPE_OBJECTS_PRESENT     = 3,
    DIS_PERSISTENT_OBJECT_TYPE_OBJECT_REQUEST      = 4,
    DIS_PERSISTENT_OBJECT_TYPE_DELETE_OBJECTS      = 5,
    DIS_PERSISTENT_OBJECT_TYPE_SET_WORLD_STATE     = 6,
    DIS_PERSISTENT_OBJECT_TYPE_NOMINATION          = 7
} DIS_PDU_PersistentObjectType;

static const value_string DIS_PDU_PersistentObjectType_Strings[] =
{
    { DIS_PERSISTENT_OBJECT_TYPE_OTHER,             "Other" },
    { DIS_PERSISTENT_OBJECT_TYPE_SIMULATOR_PRESENT, "Simulator Present" },
    { DIS_PERSISTENT_OBJECT_TYPE_DESCRIBE_OBJECT,   "Describe Object" },
    { DIS_PERSISTENT_OBJECT_TYPE_OBJECTS_PRESENT,   "Objects Present" },
    { DIS_PERSISTENT_OBJECT_TYPE_OBJECT_REQUEST,    "Object Request" },
    { DIS_PERSISTENT_OBJECT_TYPE_DELETE_OBJECTS,    "Delete Objects" },
    { DIS_PERSISTENT_OBJECT_TYPE_SET_WORLD_STATE,   "Set World State" },
    { DIS_PERSISTENT_OBJECT_TYPE_NOMINATION,        "Nomination" },
    { 0,                                            NULL }
};

typedef enum
{
    DIS_PO_OBJECT_CLASS_OTHER                      =  0,
    DIS_PO_OBJECT_CLASS_WORLD_STATE                =  1,
    DIS_PO_OBJECT_CLASS_OVERLAY                    =  2,
    DIS_PO_OBJECT_CLASS_POINT                      =  3,
    DIS_PO_OBJECT_CLASS_LINE                       =  4,
    DIS_PO_OBJECT_CLASS_SECTOR                     =  5,
    DIS_PO_OBJECT_CLASS_TEXT                       =  6,
    DIS_PO_OBJECT_CLASS_UNIT                       =  7,
    DIS_PO_OBJECT_CLASS_UNIT_DEFINITION            =  8,
    DIS_PO_OBJECT_CLASS_STEALTH_CONTROLLER         =  9,
    DIS_PO_OBJECT_CLASS_H_HOUR                     = 10,
    DIS_PO_OBJECT_CLASS_VARIABLE                   = 11,
    DIS_PO_OBJECT_CLASS_TASK                       = 12,
    DIS_PO_OBJECT_CLASS_TASK_STATE                 = 13,
    DIS_PO_OBJECT_CLASS_TASK_FRAME                 = 14,
    DIS_PO_OBJECT_CLASS_TASK_AUTHORIZATION         = 15,
    DIS_PO_OBJECT_CLASS_PARAMETRIC_INPUT           = 16,
    DIS_PO_OBJECT_CLASS_PARAMETRIC_INPUT_HOLDER    = 17,
    DIS_PO_OBJECT_CLASS_EXERCISE_INITIALIZER       = 18,
    DIS_PO_OBJECT_CLASS_FIRE_PARAMETERS            = 19,
    DIS_PO_OBJECT_CLASS_MINEFIELD                  = 20,
    DIS_PO_OBJECT_CLASS_SIMULATION_REQUEST         = 21,
    DIS_PO_OBJECT_CLASS_NET_SUBSCRIPTION           = 22,
    DIS_PO_OBJECT_CLASS_LINK                       = 23,
    DIS_PO_OBJECT_CLASS_MINEFIELD_PARENT           = 24,
    DIS_PO_OBJECT_CLASS_CHEMICAL                   = 25,
    DIS_PO_OBJECT_CLASS_AlertUser                  = 26,
    DIS_PO_OBJECT_CLASS_HAND_OFF                   = 27,
    DIS_PO_OBJECT_CLASS_CIRCUIT                    = 28,
    DIS_PO_OBJECT_CLASS_CARGO                      = 29,
    DIS_PO_OBJECT_CLASS_MCM_ROUTE                  = 30,
    DIS_PO_OBJECT_CLASS_MESSAGE                    = 31
} DIS_PDU_PO_ObjectClass;

static const value_string DIS_PDU_PO_ObjectClass_Strings[] =
{
    { DIS_PO_OBJECT_CLASS_OTHER,                    "Other" },
    { DIS_PO_OBJECT_CLASS_WORLD_STATE,              "World State" },
    { DIS_PO_OBJECT_CLASS_OVERLAY,                  "Overlay" },
    { DIS_PO_OBJECT_CLASS_POINT,                    "Point" },
    { DIS_PO_OBJECT_CLASS_LINE,                     "Line" },
    { DIS_PO_OBJECT_CLASS_SECTOR,                   "Sector" },
    { DIS_PO_OBJECT_CLASS_TEXT,                     "Text" },
    { DIS_PO_OBJECT_CLASS_UNIT,                     "Unit" },
    { DIS_PO_OBJECT_CLASS_UNIT_DEFINITION,          "Unit Definition" },
    { DIS_PO_OBJECT_CLASS_STEALTH_CONTROLLER,       "Stealth Controller" },
    { DIS_PO_OBJECT_CLASS_H_HOUR,                   "H Hour" },
    { DIS_PO_OBJECT_CLASS_VARIABLE,                 "Variable" },
    { DIS_PO_OBJECT_CLASS_TASK,                     "Task" },
    { DIS_PO_OBJECT_CLASS_TASK_STATE,               "Task State" },
    { DIS_PO_OBJECT_CLASS_TASK_FRAME,               "Task Frame" },
    { DIS_PO_OBJECT_CLASS_TASK_AUTHORIZATION,       "Task Authorization" },
    { DIS_PO_OBJECT_CLASS_PARAMETRIC_INPUT,         "Parametric Input" },
    { DIS_PO_OBJECT_CLASS_PARAMETRIC_INPUT_HOLDER,  "Parametric Input Holder" },
    { DIS_PO_OBJECT_CLASS_EXERCISE_INITIALIZER,     "Exercise Initializer" },
    { DIS_PO_OBJECT_CLASS_FIRE_PARAMETERS,          "Fire Parameters" },
    { DIS_PO_OBJECT_CLASS_MINEFIELD,                "Minefield" },
    { DIS_PO_OBJECT_CLASS_SIMULATION_REQUEST,       "Simulation Request" },
    { DIS_PO_OBJECT_CLASS_NET_SUBSCRIPTION,         "Net Subscription" },
    { DIS_PO_OBJECT_CLASS_LINK,                     "Link" },
    { DIS_PO_OBJECT_CLASS_MINEFIELD_PARENT,         "Minefield Parent" },
    { DIS_PO_OBJECT_CLASS_CHEMICAL,                 "Chemical" },
    { DIS_PO_OBJECT_CLASS_AlertUser,                "Alert User" },
    { DIS_PO_OBJECT_CLASS_HAND_OFF,                 "Hand Off" },
    { DIS_PO_OBJECT_CLASS_CIRCUIT,                  "Circuit" },
    { DIS_PO_OBJECT_CLASS_CARGO,                    "Cargo" },
    { DIS_PO_OBJECT_CLASS_MCM_ROUTE,                "Mcm Route" },
    { DIS_PO_OBJECT_CLASS_MESSAGE,                  "Message" },
    { 0,                                            NULL }
};

static value_string_ext DIS_PDU_PO_ObjectClass_Strings_Ext = VALUE_STRING_EXT_INIT(DIS_PDU_PO_ObjectClass_Strings);

typedef enum
{
    DIS_FROZEN_BEHAVIOR_RUN_INTERNAL_SIMULATION_CLOCK                = 0,
    DIS_FROZEN_BEHAVIOR_TRANSMIT_PDUS                                = 1,
    DIS_FROZEN_BEHAVIOR_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES          = 2,
    DIS_FROZEN_BEHAVIOR_CONTINUE_TRANSMIT_PDU                        = 3,
    DIS_FROZEN_BEHAVIOR_CEASE_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES    = 4,
    DIS_FROZEN_BEHAVIOR_CONTINUE_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES = 5
} DIS_PDU_FrozenBehavior;

static const value_string DIS_PDU_FrozenBehavior_Strings[] =
{
    { DIS_FROZEN_BEHAVIOR_RUN_INTERNAL_SIMULATION_CLOCK,                "Run Internal Simulation Clock" },
    { DIS_FROZEN_BEHAVIOR_TRANSMIT_PDUS,                                "Transmit PDUs" },
    { DIS_FROZEN_BEHAVIOR_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES,          "Update Sim Models Of Other Entities" },
    { DIS_FROZEN_BEHAVIOR_CONTINUE_TRANSMIT_PDU,                        "Continue Transmit PDU" },
    { DIS_FROZEN_BEHAVIOR_CEASE_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES,    "Cease Update Sim Models Of Other Entities" },
    { DIS_FROZEN_BEHAVIOR_CONTINUE_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES, "Continue Update Sim Models Of Other Entities" },
    { 0,                                                                NULL }
};

typedef enum
{
    DIS_CONTROL_ID_OTHER                                   =     0,
    DIS_CONTROL_ID_SHUTDOWN                                =     1,
    DIS_CONTROL_ID_DATA_QUERY                              =     2,
    DIS_CONTROL_ID_DATA                                    =     3,
    DIS_CONTROL_ID_SET_DATA                                =     4,
    DIS_CONTROL_ID_ADD_DATA                                =     5,
    DIS_CONTROL_ID_REMOVE_DATA                             =     6,
    DIS_CONTROL_ID_STATUS                                  =     7
} DIS_PDU_ControlId;

static const value_string DIS_PDU_ControlId_Strings[] =
{
    { DIS_CONTROL_ID_OTHER,                                  "Other" },
    { DIS_CONTROL_ID_SHUTDOWN,                               "Shutdown" },
    { DIS_CONTROL_ID_DATA_QUERY,                             "Data Query" },
    { DIS_CONTROL_ID_DATA,                                   "Data" },
    { DIS_CONTROL_ID_SET_DATA,                               "Set Data" },
    { DIS_CONTROL_ID_ADD_DATA,                               "Add Data" },
    { DIS_CONTROL_ID_REMOVE_DATA,                            "Remove Data" },
    { DIS_CONTROL_ID_STATUS,                                 "Status" },
    { 0,                                                     NULL }
};

typedef enum
{
    DIS_APPLICATION_GENERAL_STATUS_UNKNOWN                   = 1,
    DIS_APPLICATION_GENERAL_STATUS_FUNCTIONAL                = 2,
    DIS_APPLICATION_GENERAL_STATUS_DEGRADED_BUT_FUNCTIONAL   = 3,
    DIS_APPLICATION_GENERAL_STATUS_NOT_FUNCTIONAL            = 4
} DIS_PDU_ApplicationGeneralStatus;

static const value_string DIS_PDU_ApplicationGeneralStatus_Strings[] =
{
    { DIS_APPLICATION_GENERAL_STATUS_UNKNOWN,                  "Unknown" },
    { DIS_APPLICATION_GENERAL_STATUS_FUNCTIONAL,               "Functional" },
    { DIS_APPLICATION_GENERAL_STATUS_DEGRADED_BUT_FUNCTIONAL,  "Degraded But Functional" },
    { DIS_APPLICATION_GENERAL_STATUS_NOT_FUNCTIONAL,           "Not Functional" },
    { 0,                                                       NULL }
};

typedef enum
{
    DIS_APPLICATION_STATUS_TYPE_NOT_SPECIFIED             =     0,
    DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_USER            =    10,
    DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_SYSTEM          =    11,
    DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_IO              =    12,
    DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_IDLE            =    13,
    DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_STEAL           =    14,
    DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_NICE            =    15,
    DIS_APPLICATION_STATUS_TYPE_MEMORY_FREE               =    50,
    DIS_APPLICATION_STATUS_TYPE_MEMORY_USED               =    51,
    DIS_APPLICATION_STATUS_TYPE_SWAP_FREE                 =    60,
    DIS_APPLICATION_STATUS_TYPE_SWAP_USED                 =    61,
    DIS_APPLICATION_STATUS_TYPE_SWAP_CACHED               =    62,
    DIS_APPLICATION_STATUS_TYPE_TRANSMITTED_PACKETS_SEC   =   100,
    DIS_APPLICATION_STATUS_TYPE_TRANSMITTED_BYTES_SEC     =   101,
    DIS_APPLICATION_STATUS_TYPE_RECEIVED_PACKETS_SEC      =   110,
    DIS_APPLICATION_STATUS_TYPE_RECEIVED_BYTES_SEC        =   111,
    DIS_APPLICATION_STATUS_TYPE_NICE_LEVEL                =   150
} DIS_PDU_ApplicationStatusType;

static const value_string DIS_PDU_ApplicationStatusType_Strings[] =
{
    { DIS_APPLICATION_STATUS_TYPE_NOT_SPECIFIED,            "Not Specified" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_USER,           "Cpu Usage User" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_SYSTEM,         "Cpu Usage System" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_IO,             "Cpu Usage Io" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_IDLE,           "Cpu Usage Idle" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_STEAL,          "Cpu Usage Steal" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_NICE,           "Cpu Usage Nice" },
    { DIS_APPLICATION_STATUS_TYPE_MEMORY_FREE,              "Memory Free" },
    { DIS_APPLICATION_STATUS_TYPE_MEMORY_USED,              "Memory Used" },
    { DIS_APPLICATION_STATUS_TYPE_SWAP_FREE,                "Swap Free" },
    { DIS_APPLICATION_STATUS_TYPE_SWAP_USED,                "Swap Used" },
    { DIS_APPLICATION_STATUS_TYPE_SWAP_CACHED,              "Swap Cached" },
    { DIS_APPLICATION_STATUS_TYPE_TRANSMITTED_PACKETS_SEC,  "Transmitted Packets Sec" },
    { DIS_APPLICATION_STATUS_TYPE_TRANSMITTED_BYTES_SEC,    "Transmitted Bytes Sec" },
    { DIS_APPLICATION_STATUS_TYPE_RECEIVED_PACKETS_SEC,     "Received Packets Sec" },
    { DIS_APPLICATION_STATUS_TYPE_RECEIVED_BYTES_SEC,       "Received Bytes Sec" },
    { DIS_APPLICATION_STATUS_TYPE_NICE_LEVEL,               "Nice Level" },
    { 0,                                                    NULL }
};

typedef enum
{
    DIS_APPLICATION_TYPE_OTHER                                  =     0,
    DIS_APPLICATION_TYPE_RESOURCE_MANAGER                       =     1,
    DIS_APPLICATION_TYPE_SIMULATION_MANAGER                     =     2,
    DIS_APPLICATION_TYPE_GATEWAY                                =     3,
    DIS_APPLICATION_TYPE_STEALTH                                =     4,
    DIS_APPLICATION_TYPE_TACTICAL_INTERNET_INTERFACE            =     5
} DIS_PDU_ApplicationType;

static const value_string DIS_PDU_ApplicationType_Strings[] =
{
    { DIS_APPLICATION_TYPE_OTHER,                        "Other" },
    { DIS_APPLICATION_TYPE_RESOURCE_MANAGER,             "Resource Manager" },
    { DIS_APPLICATION_TYPE_SIMULATION_MANAGER,           "Simulation Manager" },
    { DIS_APPLICATION_TYPE_GATEWAY,                      "Gateway" },
    { DIS_APPLICATION_TYPE_STEALTH,                      "Stealth" },
    { DIS_APPLICATION_TYPE_TACTICAL_INTERNET_INTERFACE,  "Tactical Internet Interface" },
    { 0,                                                 NULL }
};

/* SISO-REF-010 [UID 24] deprecated */
static const value_string DIS_PDU_Nomenclature_Strings[] =
{
    {0,    "Other" },
    {1,    "AN/ARN-118" },
    {2,    "AN/ARN-139" },
    {3,    "Generic Ground Fixed Transmitter" },
    {4,    "Generic Ground Mobile Transmitter" },
    {0,    NULL }
};

static const value_string DIS_PDU_RadioTransmitState_Strings[] =
{
    {0,    "Off" },
    {1,    "On but not transmitting" },
    {2,    "On and transmitting" },
    {0,    NULL }
};

static const range_string DIS_PDU_Link16_CVLL_Strings[] = {
    { 0,   127, "Crypto Variable" },
    { 255, 255, "NO STATEMENT" },
    { 0,   0,   NULL }
};

static const value_string DIS_PDU_IffHeartbeat_Strings[] =
{
    {  0, "No Heartbeat" },
    {  1, "Heartbeat" },
    {  0, NULL }
};

static const value_string DIS_PDU_IffCapable_Strings[] =
{
    {  0, "Capable" },
    {  1, "Not capable" },
    {  0, NULL }
};

static const value_string DIS_PDU_IffOperational_Strings[] =
{
    {  0, "Operational" },
    {  1, "System failed" },
    {  0, NULL }
};

static const value_string DIS_PDU_IffMode4_Strings[] =
{
    { 4095, "No Pseudo-Crypto value. Use Alternate Mode 4 value" },
    {    0, NULL }
};

static const value_string DIS_PDU_IffModeC_Strings[] =
{
    { 2047, "No Mode C Altitude. Use Alternate Mode C value" },
    {    0, NULL }
};

static const value_string DIS_PDU_IffDamaged_Strings[] =
{
    {  0, "No damage" },
    {  1, "Damaged" },
    {  0, NULL }
};

static const value_string DIS_PDU_IffMalfunction_Strings[] =
{
    {  0, "No malfunction" },
    {  1, "Malfunction" },
    {  0, NULL }
};

/******************************************************************************
*
* FIELDS
*
*******************************************************************************/

/* DIS global */
static expert_module_t* expert_dis;
static expert_field ei_entityidentifier_not_yet_received;
static int proto_dis;
static int hf_dis_proto_ver;
static int hf_dis_exercise_id;
static int hf_dis_pdu_type;
static int hf_dis_proto_fam;
static int hf_dis_header_rel_ts;
static int hf_dis_pdu_length;
static int hf_dis_padding;
static int hf_dis_pdu_status;
static int hf_pdu_status_tei;
static int hf_pdu_status_lvc;
static int hf_pdu_status_cei;
static int hf_pdu_status_fti;
static int hf_pdu_status_dti;
static int hf_pdu_status_rai;
static int hf_pdu_status_iai;
static int hf_pdu_status_ism;
static int hf_pdu_status_aii;
static int hf_pdu_status_field;
static int hf_dis_event_type;
static int hf_dis_model_type;
static int hf_dis_po_ver;
static int hf_dis_po_pdu_type;
static int hf_dis_po_database_id;
static int hf_dis_po_length;
static int hf_dis_po_pdu_count;
static int hf_dis_entity_id_site;
static int hf_dis_entity_id_application;
static int hf_dis_entity_id_entity;
static int hf_dis_emitter_id;
static int hf_dis_beam_id;
static int hf_dis_num_art_params;
static int hf_dis_clocktime;
static int hf_dis_entityKind;
static int hf_dis_entityDomain;
static int hf_dis_category_land;
static int hf_dis_category_air;
static int hf_dis_category_surface;
static int hf_dis_category_subsurface;
static int hf_dis_category_space;
static int hf_dis_category;
static int hf_dis_country;
static int hf_dis_subcategory;
static int hf_dis_specific;
static int hf_dis_extra;
static int hf_dis_site;
static int hf_dis_request_id;
static int hf_dis_reason;
static int hf_dis_frozen_behavior;
static int hf_dis_acknowledge_flag;
static int hf_dis_response_flag;
static int hf_dis_application;
static int hf_dis_action_id;
static int hf_dis_request_status;
static int hf_dis_num_fixed_data;
static int hf_dis_num_variable_data;
static int hf_dis_datum_id;
static int hf_dis_fixed_datum_value;
static int hf_dis_datum_length;
static int hf_dis_variable_datum_value;
static int hf_dis_variable_datum_value_as_text;
static int hf_dis_time_interval8;
static int hf_dis_time_interval32;
static int hf_dis_num_fixed_datum_id;
static int hf_dis_num_variable_datum_id;
static int hf_dis_reliability;
static int hf_dis_control_id;
static int hf_dis_orig_app_type;
static int hf_dis_recv_app_type;
static int hf_dis_num_parts;
static int hf_dis_current_part;
static int hf_dis_num_variable_records;
static int hf_dis_variable_record_type;
static int hf_dis_variable_record_len;
static int hf_dis_event_number;
static int hf_dis_num_electromagnetic_emission_systems;
static int hf_dis_emitter_name;
static int hf_dis_emission_function;
static int hf_dis_em_data_length;
static int hf_dis_em_num_beams;
static int hf_dis_emitter_id_number;
static int hf_dis_em_location_x;
static int hf_dis_em_location_y;
static int hf_dis_em_location_z;
static int hf_dis_beam_function;
static int hf_dis_radio_id;
static int hf_dis_transmitter_radio_id;
static int hf_dis_ens;
static int hf_dis_ens_class;
static int hf_dis_ens_type;
static int hf_dis_ens_type_audio;
static int hf_dis_tdl_type;
static int hf_dis_sample_rate;
static int hf_dis_data_length;
static int hf_dis_num_of_samples;
static int hf_dis_signal_data;
static int hf_dis_radio_category;
static int hf_dis_nomenclature_version;
static int hf_dis_nomenclature;
static int hf_dis_radio_transmit_state;
static int hf_dis_radio_receive_state;
static int hf_dis_radio_input_source;
static int hf_dis_antenna_location_x;
static int hf_dis_antenna_location_y;
static int hf_dis_antenna_location_z;
static int hf_dis_rel_antenna_location_x;
static int hf_dis_rel_antenna_location_y;
static int hf_dis_rel_antenna_location_z;
static int hf_dis_antenna_pattern_type;
static int hf_dis_antenna_pattern_length;
static int hf_dis_transmit_frequency;
static int hf_dis_transmit_freq_bandwidth;
static int hf_dis_transmit_power;
static int hf_dis_receive_power;
static int hf_dis_spread_spectrum_usage;
static int hf_dis_frequency_hopping;
static int hf_dis_pseudo_noise_modulation;
static int hf_dis_time_hopping;
static int hf_dis_modulation_major;
static int hf_dis_modulation_amplitude;
static int hf_dis_modulation_amplitude_angle;
static int hf_dis_modulation_angle;
static int hf_dis_modulation_combination;
static int hf_dis_modulation_pulse;
static int hf_dis_modulation_unmodulated;
static int hf_dis_modulation_detail;
static int hf_dis_modulation_system;
static int hf_dis_crypto_system;
static int hf_dis_crypto_key;
static int hf_dis_encryption_mode;
static int hf_dis_key_identifier;
static int hf_dis_modulation_parameter_length;
static int hf_dis_mod_param_fh_net_id;
static int hf_dis_mod_param_fh_set_id;
static int hf_dis_mod_param_fh_lo_set_id;
static int hf_dis_mod_param_fh_msg_start;
static int hf_dis_mod_param_fh_reserved;
static int hf_dis_mod_param_fh_sync_time_offset;
static int hf_dis_mod_param_fh_security_key;
static int hf_dis_mod_param_fh_clear_channel;
static int hf_dis_mod_param_dump;
static int hf_dis_mod_param_ts_allocation_mode;
static int hf_dis_mod_param_transmitter_prim_mode;
static int hf_dis_mod_param_transmitter_second_mode;
static int hf_dis_mod_param_sync_state;
static int hf_dis_mod_param_network_sync_id;
static int hf_dis_force_id;

/* DIS aggregate */
static int hf_dis_aggregate_marking;
static int hf_dis_aggregate_number_of_aggregates;
static int hf_dis_aggregate_number_of_entities;
static int hf_dis_aggregate_number_of_silent_aggregates_types;
static int hf_dis_aggregate_number_of_silent_entity_types;
static int hf_dis_aggregate_number_of_variable_datum_records;
static int hf_dis_aggregate_state;
static int hf_dis_aggregate_formation;
static int hf_dis_aggregate_kind;
static int hf_dis_aggregate_domain;
static int hf_dis_aggregate_country;
static int hf_dis_aggregate_category;
static int hf_dis_aggregate_subcategory;
static int hf_dis_aggregate_specific;
static int hf_dis_aggregate_extra;
static int hf_dis_aggregate_dimensions_x;
static int hf_dis_aggregate_dimensions_y;
static int hf_dis_aggregate_dimensions_z;
static int hf_dis_aggregate_orientation_psi;
static int hf_dis_aggregate_orientation_theta;
static int hf_dis_aggregate_orientation_phi;
static int hf_dis_aggregate_center_of_mass_x;
static int hf_dis_aggregate_center_of_mass_y;
static int hf_dis_aggregate_center_of_mass_z;
static int hf_dis_aggregate_velocity_x;
static int hf_dis_aggregate_velocity_y;
static int hf_dis_aggregate_velocity_z;
static int hf_dis_aggregate_id_site;
static int hf_dis_aggregate_id_application;
static int hf_dis_aggregate_id_aggregate;

/* DIS environment */
static int hf_dis_environmental_number_of_environment_records;
static int hf_dis_environmental_sequence_number;
static int hf_dis_environment_status_last;
static int hf_dis_environment_status_on;
static int hf_dis_environment_kind;
static int hf_dis_environment_domain;
static int hf_dis_environment_class;
static int hf_dis_environment_category;
static int hf_dis_environment_subcategory;
static int hf_dis_environment_specific;
static int hf_dis_environment_extra;

/* DIS datums */
static int hf_dis_fixed_datum_value_as_uint;
static int hf_dis_fixed_datum_value_as_float;
static int hf_dis_fixed_datum_value_as_int;

/* Appearance */
static int hf_appearance_frozen_status;
static int hf_appearance_state;
static int hf_appearance_weapon_1;
static int hf_appearance_weapon_2;
static int hf_appearance_camouflage_type;
static int hf_appearance_concealed_stationary;
static int hf_appearance_concealed_movement;
static int hf_appearance_landform_paint_scheme;
static int hf_appearance_landform_mobility;
static int hf_appearance_landform_fire_power;
static int hf_appearance_landform_damage;
static int hf_appearance_landform_smoke_entity;
static int hf_appearance_landform_trailing_effects_entity;
static int hf_appearance_landform_hatch;
static int hf_appearance_landform_head_lights;
static int hf_appearance_landform_tail_lights;
static int hf_appearance_landform_brake_lights;
static int hf_appearance_landform_flaming;
static int hf_appearance_landform_launcher;
static int hf_appearance_landform_camouflage_type;
static int hf_appearance_landform_concealed;
static int hf_appearance_landform_frozen_status;
static int hf_appearance_landform_power_plant_status;
static int hf_appearance_landform_state;
static int hf_appearance_landform_tent;
static int hf_appearance_landform_ramp;
static int hf_appearance_landform_blackout_lights;
static int hf_appearance_landform_blackout_brake_lights;
static int hf_appearance_landform_spot_lights;
static int hf_appearance_landform_interior_lights;
static int hf_appearance_landform_surrender_state;
static int hf_appearance_landform_masked_cloaked;
static int hf_appearance_airform_paint_scheme;
static int hf_appearance_airform_propulsion_killed;
static int hf_appearance_airform_nvg_mode;
static int hf_appearance_airform_damage;
static int hf_appearance_airform_is_smoke_emanating;
static int hf_appearance_airform_is_engine_emitting_smoke;
static int hf_appearance_airform_trailing_effects;
static int hf_appearance_airform_canopy_troop_door;
static int hf_appearance_airform_landing_lights_on;
static int hf_appearance_airform_navigation_lights_on;
static int hf_appearance_airform_anti_collision_lights_on;
static int hf_appearance_airform_is_flaming;
static int hf_appearance_airform_afterburner_on;
static int hf_appearance_airform_lower_anti_collision_light_on;
static int hf_appearance_airform_upper_anti_collision_light_on;
static int hf_appearance_airform_anti_collision_light_day_night;
static int hf_appearance_airform_is_blinking;
static int hf_appearance_airform_is_frozen;
static int hf_appearance_airform_power_plant_on;
static int hf_appearance_airform_state;
static int hf_appearance_airform_formation_lights_on;
static int hf_appearance_airform_landing_gear_extended;
static int hf_appearance_airform_cargo_doors_opened;
static int hf_appearance_airform_navigation_position_brightness;
static int hf_appearance_airform_spot_search_light_1_on;
static int hf_appearance_airform_interior_lights_on;
static int hf_appearance_airform_reverse_thrust_engaged;
static int hf_appearance_airform_weight_on_wheels;
static int hf_appearance_lifeform_paint_scheme;
static int hf_appearance_lifeform_health;
static int hf_appearance_lifeform_compliance;
static int hf_appearance_lifeform_flash_lights;
static int hf_appearance_lifeform_state;

/* Entity */
static int hf_dis_entity_marking_character_set;
static int hf_dis_aggregate_marking_character_set;
static int hf_dis_entity_dead_reckoning_algorithm;
static int hf_dis_dead_reckoning_other_parameters;
static int hf_dis_entity_location_x_double;
static int hf_dis_entity_location_x_float;
static int hf_dis_entity_location_y_double;
static int hf_dis_entity_location_y_float;
static int hf_dis_entity_location_z_double;
static int hf_dis_entity_location_z_float;
static int hf_dis_entity_orientation_psi;
static int hf_dis_entity_orientation_theta;
static int hf_dis_entity_orientation_phi;
static int hf_dis_entity_linear_velocity_x;
static int hf_dis_entity_linear_velocity_y;
static int hf_dis_entity_linear_velocity_z;
static int hf_dis_entity_linear_acceleration_x;
static int hf_dis_entity_linear_acceleration_y;
static int hf_dis_entity_linear_acceleration_z;
static int hf_dis_entity_entity_angular_velocity_x;
static int hf_dis_entity_entity_angular_velocity_y;
static int hf_dis_entity_entity_angular_velocity_z;

/* Intercom */
static int hf_intercom_control_control_type;
static int hf_intercom_control_communications_channel_type;
static int hf_intercom_control_source_communications_device_id;
static int hf_intercom_control_source_line_id;
static int hf_intercom_control_transmit_priority;
static int hf_intercom_control_transmit_line_state;
static int hf_intercom_control_command;
static int hf_intercom_control_master_communications_device_id;
static int hf_intercom_control_master_channel_id;

/* TODO: put them in the best matched group */
static int hf_entity_appearance;
static int hf_dis_entity_marking;

/* Dis designator */
static int hf_dis_designator_code_name;
static int hf_dis_designator_designator_code;
static int hf_dis_designator_power;
static int hf_dis_designator_wavelength;
static int hf_dis_designator_spot_with_respect_to_designated_entity_x;
static int hf_dis_designator_spot_with_respect_to_designated_entity_y;
static int hf_dis_designator_spot_with_respect_to_designated_entity_z;
static int hf_dis_designator_spot_location_x;
static int hf_dis_designator_spot_location_y;
static int hf_dis_designator_spot_location_z;

/* Collision */
static int hf_dis_collision_type;
static int hf_dis_collision_mass;
static int hf_dis_collision_location_x;
static int hf_dis_collision_location_y;
static int hf_dis_collision_location_z;

/* More DIS global */
static int hf_dis_capabilities;
static int hf_dis_variable_parameter_type;
static int hf_dis_num_shafts;
static int hf_dis_num_apas;
static int hf_dis_state_update_indicator;
static int hf_dis_passive_parameter_index;
static int hf_dis_propulsion_plant_config;
static int hf_dis_shaft_rpm_current;
static int hf_dis_shaft_rpm_ordered;
static int hf_dis_shaft_rpm_change_rate;
static int hf_dis_num_ua_emitter_systems;
static int hf_dis_apas_parameter_index;
static int hf_dis_apas_value;
static int hf_dis_ua_emission_name;
static int hf_dis_ua_emission_function;
static int hf_dis_ua_emission_id_number;
static int hf_dis_ua_emitter_data_length;
static int hf_dis_ua_num_beams;
static int hf_dis_ua_location_x;
static int hf_dis_ua_location_y;
static int hf_dis_ua_location_z;
static int hf_dis_ua_beam_data_length;
static int hf_dis_ua_beam_id_number;
static int hf_dis_ua_beam_active_emission_parameter_index;
static int hf_dis_ua_beam_scan_pattern;
static int hf_dis_ua_beam_center_azimuth;
static int hf_dis_ua_beam_azimuthal_beamwidth;
static int hf_dis_ua_beam_center_de;
static int hf_dis_ua_beam_de_beamwidth;
static int hf_dis_em_beam_data_length;
static int hf_dis_em_beam_id_number;
static int hf_dis_em_beam_parameter_index;
static int hf_dis_em_fund_frequency;
static int hf_dis_em_fund_frequency_range;
static int hf_dis_em_fund_effective_radiated_power;
static int hf_dis_em_fund_pulse_repetition_freq;
static int hf_dis_em_fund_pulse_width;
static int hf_dis_em_fund_beam_azimuth_center;
static int hf_dis_em_fund_beam_azimuth_sweep;
static int hf_dis_em_fund_beam_elevation_center;
static int hf_dis_em_fund_beam_elevation_sweep;
static int hf_dis_em_fund_beem_sweep_sync;
static int hf_dis_track_jam_num_targ;
static int hf_dis_track_jam_high_density;
static int hf_dis_jamming_mode_seq;
static int hf_dis_warhead;
static int hf_dis_fuse;
static int hf_dis_quality;
static int hf_dis_rate;
static int hf_dis_fire_mission_index;
static int hf_dis_fire_location_x;
static int hf_dis_fire_location_y;
static int hf_dis_fire_location_z;
static int hf_dis_linear_velocity_x;
static int hf_dis_linear_velocity_y;
static int hf_dis_linear_velocity_z;
static int hf_dis_range;
static int hf_dis_detonation_location_x;
static int hf_dis_detonation_location_y;
static int hf_dis_detonation_location_z;
static int hf_dis_detonation_result;
static int hf_dis_simulator_type;
static int hf_dis_database_seq_num;
static int hf_dis_simulator_load;
static int hf_dis_simulation_load;
static int hf_dis_time;
static int hf_dis_packets_sent;
static int hf_dis_unit_database_version;
static int hf_dis_relative_battle_scheme;
static int hf_dis_terrain_name;
static int hf_dis_terrain_version;
static int hf_dis_host_name;
static int hf_dis_sequence_number;
static int hf_dis_persist_obj_class;
static int hf_dis_missing_from_world_state;
static int hf_dis_obj_count;
static int hf_dis_clock_rate;
static int hf_dis_sec_since_1970;
static int hf_dis_str_data;
static int hf_dis_record_data;
static int hf_dis_alignment_padding;
static int hf_dis_vp_change_indicator;
static int hf_dis_vp_association_status;
static int hf_dis_vp_association_type;
static int hf_dis_vp_own_station_location;
static int hf_dis_vp_phys_conn_type;
static int hf_dis_vp_group_member_type;
static int hf_dis_vp_group_number;
static int hf_dis_vp_offset_type;
static int hf_dis_vp_offset_x;
static int hf_dis_vp_offset_y;
static int hf_dis_vp_offset_z;
static int hf_dis_vp_attached_indicator;
static int hf_dis_vp_part_attached_to_id;
static int hf_dis_vp_artic_param_type;
static int hf_dis_vp_change;
static int hf_dis_vp_parameter_value;
static int hf_dis_vp_padding;
static int hf_dis_vr_exercise_id;
static int hf_dis_vr_exercise_file_path;
static int hf_dis_vr_exercise_file_name;
static int hf_dis_vr_application_role;
static int hf_dis_vr_num_records;
static int hf_dis_vr_status_type;
static int hf_dis_vr_general_status;
static int hf_dis_vr_specific_status;
static int hf_dis_vr_status_value_int;
static int hf_dis_vr_status_value_float;
static int hf_dis_signal_link16_npg;
static int hf_dis_signal_link16_tsec_cvll;
static int hf_dis_signal_link16_msec_cvll;
static int hf_dis_signal_link16_message_type;
static int hf_dis_signal_link16_ptt;
static int hf_dis_signal_link16_time_slot_type;
static int hf_dis_signal_link16_rti;
static int hf_dis_signal_link16_stn;
static int hf_dis_signal_link16_sdusn;
static int hf_dis_signal_link16_network_number;
static int hf_dis_signal_link16_time_slot_id;
static int hf_dis_iff_system_type;
static int hf_dis_iff_system_name;
static int hf_dis_iff_system_mode;
static int hf_dis_iff_change_options;
static int hf_dis_iff_change_indicator;
static int hf_dis_iff_alternate_mode_4;
static int hf_dis_iff_alternate_mode_c;
static int hf_dis_iff_heartbeat_indicator;
static int hf_dis_iff_transponder_interrogator_indicator;
static int hf_dis_iff_simulation_mode;
static int hf_dis_iff_interactive_capable;
static int hf_dis_iff_test_mode;
static int hf_dis_iff_system_designator;
static int hf_dis_iff_system_specific_data;
static int hf_dis_iff_system_status;
static int hf_dis_iff_system_status_system_onoff;
static int hf_dis_iff_system_status_parameter_1;
static int hf_dis_iff_system_status_parameter_2;
static int hf_dis_iff_system_status_parameter_3;
static int hf_dis_iff_system_status_parameter_4;
static int hf_dis_iff_system_status_parameter_5;
static int hf_dis_iff_system_status_parameter_6;
static int hf_dis_iff_system_status_operational;
static int hf_dis_iff_alternate_parameter_4;
static int hf_dis_iff_information_layers;
static int hf_dis_iff_information_layers_layer_1;
static int hf_dis_iff_information_layers_layer_2;
static int hf_dis_iff_information_layers_layer_3;
static int hf_dis_iff_information_layers_layer_4;
static int hf_dis_iff_information_layers_layer_5;
static int hf_dis_iff_information_layers_layer_6;
static int hf_dis_iff_information_layers_layer_7;
static int hf_dis_iff_modifier;
static int hf_dis_iff_modifier_emergency;
static int hf_dis_iff_modifier_ident;
static int hf_dis_iff_modifier_sti;
static int hf_dis_iff_modifier_unmanned_aircraft;
static int hf_dis_iff_parameter_1;
static int hf_dis_iff_parameter_2;
static int hf_dis_iff_parameter_3;
static int hf_dis_iff_parameter_4;
static int hf_dis_iff_parameter_5;
static int hf_dis_iff_parameter_6;
static int hf_dis_iff_mode_code_element_1;
static int hf_dis_iff_mode_code_element_2;
static int hf_dis_iff_mode_code_element_3;
static int hf_dis_iff_mode_code_element_4;
static int hf_dis_iff_rrb;
static int hf_dis_iff_rrb_rrb_code;
static int hf_dis_iff_rrb_power_reduction_indicator;
static int hf_dis_iff_rrb_radar_enhancement_indicator;
static int hf_dis_iff_mode_4;
static int hf_dis_iff_mode_s_interrogator_identifier;
static int hf_dis_iff_mode_s_interrogator_identifier_primary_ic_type;
static int hf_dis_iff_mode_s_interrogator_identifier_primary_ic_code;
static int hf_dis_iff_mode_s_interrogator_identifier_secondary_ic_type;
static int hf_dis_iff_mode_s_interrogator_identifier_secondary_ic_code;
static int hf_dis_iff_mode_c_altitude_indicator;
static int hf_dis_iff_mode_c_altitude;
static int hf_dis_iff_tcas_acas;
static int hf_dis_iff_tcas_acas_basic_advanced_indicator;
static int hf_dis_iff_tcas_acas_tcas_acas_indicator;
static int hf_dis_iff_tcas_acas_software_version;
static int hf_dis_iff_tcas_acas_tcas_acas_type;
static int hf_dis_iff_tcas_acas_tcas_type;
static int hf_dis_iff_mode_status;
static int hf_dis_iff_mode_damage;
static int hf_dis_iff_mode_malfunction;

static int ett_dis;
static int ett_dis_header;
static int ett_pdu_status;
static int ett_dis_po_header;
static int ett_dis_payload;
static int ett_entity;
static int ett_trackjam;
static int ett_dis_ens;
static int ett_radio_entity_type;
static int ett_entity_type;
static int ett_dis_crypto_key;
static int ett_antenna_location;
static int ett_rel_antenna_location;
static int ett_aggregate_dimensions;
static int ett_aggregate_orientation;
static int ett_aggregate_velocity;
static int ett_aggregate_id_list;
static int ett_entity_id_list;
static int ett_variable_datum;

static int ett_modulation_type;
static int ett_modulation_parameters;
static int ett_entity_linear_velocity;
static int ett_entity_location;
static int ett_entity_orientation;
static int ett_entity_marking_text;
static int ett_aggregate_marking_text;
static int ett_entity_dead_reckoning_parameters;
static int ett_entity_linear_acceleration;
static int ett_entity_angular_velocity;
static int ett_environmental_environment_status;
static int ett_environmental_environment_type;
static int ett_aggregate_type;
static int ett_aggregate_center_of_mass;
static int ett_designator_spot_location;
static int ett_designator_spot_with_respect_to_designated_entity;
static int ett_designator_entity_linear_acceleration;





static int ett_entity_appearance;
static int ett_variable_parameter;
static int ett_event_id;
static int ett_shafts;
static int ett_apas;
static int ett_underwater_acoustic_emission;
static int ett_acoustic_emitter_system;
static int ett_ua_location;
static int ett_ua_beams;
static int ett_ua_beam_data;
static int ett_emission_system;
static int ett_emitter_system;
static int ett_em_beam;
static int ett_emitter_location;
static int ett_em_fundamental_parameter_data;
static int ett_burst_descriptor;
static int ett_fire_location;
static int ett_linear_velocity;
static int ett_detonation_location;
static int ett_clock_time;
static int ett_fixed_datum;
static int ett_record;
static int ett_simulation_address;
static int ett_offset_vector;
static int ett_dis_signal_link16_network_header;
static int ett_dis_signal_link16_message_data;
static int ett_dis_signal_link16_jtids_header;
static int ett_iff_location;
static int ett_iff_system_id;
static int ett_iff_change_options;
static int ett_iff_fundamental_operational_data;
static int ett_iff_system_status;
static int ett_iff_information_layers;
static int ett_iff_modifier;
static int ett_iff_parameter_1;
static int ett_iff_rrb;
static int ett_iff_parameter_2;
static int ett_iff_parameter_3;
static int ett_iff_parameter_4;
static int ett_iff_mode_s_interrogator_identifier;
static int ett_iff_parameter_5;
static int ett_iff_parameter_6;

static dissector_handle_t link16_handle;

typedef struct dis_header
{
    uint8_t version;
    uint8_t pduType;
    uint8_t family;
}
dis_header_t;

/* Forward declarations */
static int parseField_Entity(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name, uint32_t* entityidentifier);
static int parseField_Aggregate(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name);
static int dissect_DIS_FIELDS_ENTITY_TYPE(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name, uint8_t* entityKind, uint8_t* entityDomain);
static int parseField_VariableParameter(tvbuff_t *tvb, proto_tree *tree, int offset, uint8_t paramType);
static int parseField_VariableRecord(tvbuff_t *tvb, proto_tree *tree, int offset, uint32_t variableRecordType, uint16_t record_length);

/* Composite types
 */
static int dissect_DIS_FIELDS_BURST_DESCRIPTOR(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree  *sub_tree;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 16, ett_burst_descriptor, NULL, "Burst Descriptor");

    offset = dissect_DIS_FIELDS_ENTITY_TYPE(tvb, sub_tree, offset, "Munition", NULL, NULL);

    proto_tree_add_item(sub_tree, hf_dis_warhead, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_fuse, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_quality, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_rate, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    return offset;
}

static int dissect_DIS_FIELDS_CLOCK_TIME(tvbuff_t *tvb, proto_tree *tree, int offset, const char* clock_name)
{
    proto_item  *ti;
    proto_tree  *sub_tree;
    /* some consts */
    static unsigned MSEC_PER_HOUR = 60 * 60 * 1000;
    static unsigned FSV = 0x7fffffff;
    uint32_t hour, uintVal;
    uint64_t ms;
    unsigned isAbsolute = 0;
    nstime_t tv;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 8, ett_clock_time, NULL, clock_name);

    hour = tvb_get_ntohl(tvb, offset);
    uintVal = tvb_get_ntohl(tvb, offset+4);

    /* determine absolute vis sim time */
    isAbsolute = uintVal & 1;

    /* convert TS to MS */
    ms = (uint64_t)(uintVal >> 1) * MSEC_PER_HOUR / FSV;

    tv.secs = (time_t)ms/1000;
    tv.nsecs = (int)(ms%1000)*1000000;

    /* add hour */
    tv.secs += (hour*3600);

    ti = proto_tree_add_time(sub_tree, hf_dis_clocktime, tvb, offset, 8, &tv);
    if (isAbsolute)
    {
        proto_item_append_text(ti, " (absolute)");
    }
    else
    {
        proto_item_append_text(ti, " (relative)");
    }

   return (offset+8);
}

static int dissect_DIS_FIELDS_ENTITY_TYPE_RECORD(tvbuff_t *tvb, proto_tree *tree, int offset, const char* name, int ett, int hfkind, int hfdomain, int hfcountry, int hfcategory, int hfsubcategory, int hfspecific, int hfextra, uint8_t* outentityKind, uint8_t* outentityDomain)
{
    uint8_t entityKind;
    uint8_t entityDomain;
    uint16_t entityCountry;
    uint8_t entityCategory;
    uint8_t entitySubcategory;
    uint8_t entitySpecific;
    uint8_t entityExtra;

    proto_tree  *sub_tree;
    int hf_cat = hfcategory;

    entityKind = tvb_get_uint8(tvb, offset);
    if (outentityKind) *outentityKind = entityKind;
    entityDomain = tvb_get_uint8(tvb, offset+1);
    if (outentityDomain) *outentityDomain = entityDomain;
    entityCountry = tvb_get_ntohs(tvb, offset+2);
    entityCategory = tvb_get_uint8(tvb, offset+4);
    entitySubcategory = tvb_get_uint8(tvb, offset+5);
    entitySpecific = tvb_get_uint8(tvb, offset+6);
    entityExtra = tvb_get_uint8(tvb, offset+7);

    sub_tree = proto_tree_add_subtree_format(tree, tvb, offset, 8, ett, NULL, "%s, (%u:%u:%u:%u:%u:%u:%u) ", name, entityKind, entityDomain, entityCountry, entityCategory, entitySubcategory, entitySpecific, entityExtra);

    proto_tree_add_uint(sub_tree, hfkind, tvb, offset, 1, entityKind);
    offset++;

    proto_tree_add_uint(sub_tree, hfdomain, tvb, offset, 1, entityDomain);
    offset++;

    proto_tree_add_uint(sub_tree, hfcountry, tvb, offset, 2, entityCountry);
    offset += 2;

    if (entityKind == DIS_ENTITYKIND_PLATFORM && hfcategory == hf_dis_category)
    {
        switch(entityDomain)
        {
        case DIS_DOMAIN_LAND:
            hf_cat = hf_dis_category_land;
            break;
        case DIS_DOMAIN_AIR:
            hf_cat = hf_dis_category_air;
            break;
        case DIS_DOMAIN_SURFACE:
            hf_cat = hf_dis_category_surface;
            break;
        case DIS_DOMAIN_SUBSURFACE:
            hf_cat = hf_dis_category_subsurface;
            break;
        case DIS_DOMAIN_SPACE:
            hf_cat = hf_dis_category_space;
            break;
        }
    }

    proto_tree_add_uint(sub_tree, hf_cat, tvb, offset, 1, entityCategory);
    offset++;

    proto_tree_add_uint(sub_tree, hfsubcategory, tvb, offset, 1, entitySubcategory);
    offset++;

    proto_tree_add_uint(sub_tree, hfspecific, tvb, offset, 1, entitySpecific);
    offset++;

    proto_tree_add_uint(sub_tree, hfextra, tvb, offset, 1, entityExtra);
    offset++;

    return offset;
}

static int dissect_DIS_FIELDS_ENTITY_TYPE(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name, uint8_t* outentityKind, uint8_t* outentityDomain)
{
    return dissect_DIS_FIELDS_ENTITY_TYPE_RECORD(tvb, tree, offset, entity_name, ett_entity_type, hf_dis_entityKind, hf_dis_entityDomain, hf_dis_country, hf_dis_category, hf_dis_subcategory, hf_dis_specific, hf_dis_extra, outentityKind, outentityDomain);
}

static int dissect_DIS_FIELDS_RADIO_ENTITY_TYPE(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name)
{
    proto_tree  *sub_tree;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 8, ett_radio_entity_type, NULL, entity_name);

    proto_tree_add_item(sub_tree, hf_dis_entityKind, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_entityDomain, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_country, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_radio_category, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_nomenclature_version, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_nomenclature, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    return offset;
}

static int dissect_DIS_FIELDS_AGGREGATE_TYPE(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name)
{
    return dissect_DIS_FIELDS_ENTITY_TYPE_RECORD(tvb, tree, offset, entity_name, ett_aggregate_type, hf_dis_aggregate_kind, hf_dis_aggregate_domain, hf_dis_aggregate_country, hf_dis_aggregate_category, hf_dis_aggregate_subcategory, hf_dis_aggregate_specific, hf_dis_aggregate_extra, NULL, NULL);
}

static int dissect_DIS_FIELDS_ENVIRONMENT_TYPE(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name)
{
    return dissect_DIS_FIELDS_ENTITY_TYPE_RECORD(tvb, tree, offset, entity_name, ett_environmental_environment_type, hf_dis_environment_kind, hf_dis_environment_domain, hf_dis_environment_class, hf_dis_environment_category, hf_dis_environment_subcategory, hf_dis_environment_specific, hf_dis_environment_extra, NULL, NULL);
}


static int dissect_DIS_FIELDS_MODULATION_TYPE(tvbuff_t *tvb, proto_tree *tree, int offset, uint16_t* systemModulation)
{
    proto_tree  *sub_tree;
    uint32_t majorModulation;
    int hf_mod_detail;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 8, ett_modulation_type, NULL, "Modulation Type");

    proto_tree_add_item(sub_tree, hf_dis_spread_spectrum_usage, tvb, offset,  2, ENC_BIG_ENDIAN);
    proto_tree_add_item(sub_tree, hf_dis_frequency_hopping, tvb, offset,  2, ENC_BIG_ENDIAN);
    proto_tree_add_item(sub_tree, hf_dis_pseudo_noise_modulation, tvb, offset,  2, ENC_BIG_ENDIAN);
    proto_tree_add_item(sub_tree, hf_dis_time_hopping, tvb, offset,  2, ENC_BIG_ENDIAN);
    offset += 2;

    majorModulation = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item(tree, hf_dis_modulation_major, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;


    switch (majorModulation) {
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_AMPLITUDE:
        hf_mod_detail = hf_dis_modulation_amplitude;
        break;
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_AMPLITUDE_AND_ANGLE:
        hf_mod_detail = hf_dis_modulation_amplitude_angle;
        break;
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_ANGLE:
        hf_mod_detail = hf_dis_modulation_angle;
        break;
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_COMBINATION:
        hf_mod_detail = hf_dis_modulation_combination;
        break;
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_PULSE:
        hf_mod_detail = hf_dis_modulation_pulse;
        break;
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_UNMODULATED:
        hf_mod_detail = hf_dis_modulation_unmodulated;
        break;
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_CARRIER_PHASE_SHIFT_MODULATION_CPSM: /* CPSM only has "other" defined */
    case DIS_PDU_TRANSMITTER_MAJOR_MODULATION_SATCOM:
    default:
        hf_mod_detail = hf_dis_modulation_detail;
        break;
    }

    proto_tree_add_item(tree, hf_mod_detail, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    *systemModulation = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item(tree, hf_dis_modulation_system, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    return offset;
}

static int dissect_DIS_FIELDS_EVENT_ID(tvbuff_t *tvb, proto_tree *tree, int offset, const char* event_name)
{
    proto_tree  *sub_tree;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 6, ett_event_id, NULL, event_name);

    proto_tree_add_item(sub_tree, hf_dis_site, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_application, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_event_number, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;


    return offset;
}

static int dissect_DIS_FIELDS_SIMULATION_ADDRESS(tvbuff_t *tvb, proto_tree *tree, int offset, const char* sim_name)
{
    proto_tree  *sub_tree;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 4, ett_simulation_address, NULL, sim_name);

    proto_tree_add_item(sub_tree, hf_dis_site, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_application, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    return offset;
}

static int dissect_DIS_FIELDS_MOD_PARAMS_CCTT_SINCGARS(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree  *sub_tree;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 16, ett_modulation_parameters, NULL, "Modulation Parameters");

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_net_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_set_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_lo_set_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_msg_start, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_reserved, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_sync_time_offset, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_security_key, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_fh_clear_channel, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_padding, tvb, offset, 1, ENC_NA);
    offset++;

    return offset;
}

static int dissect_DIS_FIELDS_MOD_PARAMS_JTIDS_MIDS(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree  *sub_tree;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 8, ett_modulation_parameters, NULL, "Modulation Parameters");

    proto_tree_add_item(sub_tree, hf_dis_mod_param_ts_allocation_mode, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_transmitter_prim_mode, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_transmitter_second_mode, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_sync_state, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_mod_param_network_sync_id, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    return offset;
}

static int parse_DIS_FIELDS_SIGNAL_LINK16_NETWORK_HEADER(tvbuff_t *tvb, proto_tree *tree,
                                                          int offset, uint8_t* messageType)
{
    proto_tree  *sub_tree;
    nstime_t tv;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 16, ett_dis_signal_link16_network_header, NULL, "Link 16 Network Header");

    proto_tree_add_item(sub_tree, hf_dis_signal_link16_npg, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_signal_link16_network_number, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_signal_link16_tsec_cvll, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_signal_link16_msec_cvll, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_signal_link16_message_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    if (messageType)
        *messageType = tvb_get_uint8(tvb, offset);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_signal_link16_time_slot_id, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    tv.secs = tvb_get_ntohl(tvb, offset);
    if (tv.secs == (time_t)0xFFFFFFFF)
    {
        tv.nsecs = 0;
        proto_tree_add_time_format_value(sub_tree, hf_dis_signal_link16_ptt, tvb, offset, 8, &tv, "NO STATEMENT");
    }
    else
    {
        proto_tree_add_item(sub_tree, hf_dis_signal_link16_ptt, tvb, offset, 8, ENC_TIME_NTP|ENC_BIG_ENDIAN);
    }
    offset += 8;

    return offset;
}

/* Parse Link 16 Message Data record (SISO-STD-002, Tables 5.2.5 through 5.2.12)
 */
static int parse_Link16_Message_Data(proto_tree *tree, tvbuff_t *tvb, int offset, packet_info *pinfo,
                                      uint32_t encodingScheme, uint8_t messageType)
{
    uint32_t cache, value, i;
    Link16State state;
    tvbuff_t *newtvb;

    static int * const jtids_message_header_fields[] = {
        &hf_dis_signal_link16_time_slot_type,
        &hf_dis_signal_link16_rti,
        &hf_dis_signal_link16_stn,
        NULL
    };

    switch (messageType) {
    case DIS_PDU_MESSAGE_TYPE_IDENTIFIER_JTIDS_HEADER_MESSAGES:
        proto_tree_add_bitmask_text(tree, tvb, offset, 4, "JTIDS Header", NULL, ett_dis_signal_link16_jtids_header, jtids_message_header_fields, ENC_BIG_ENDIAN, BMT_NO_APPEND);

        cache = tvb_get_ntohl(tvb, offset);
        value = (cache >> 4) & 0x7FFF;
        col_append_fstr(pinfo->cinfo, COL_INFO, ", STN=0%o, Link 16 Words:", value);

        value = (cache >> 19);
        offset += 4;
        cache = tvb_get_ntohl(tvb, offset);
        value |= (cache & 0x7) << 13;
        proto_tree_add_uint(tree, hf_dis_signal_link16_sdusn, tvb, offset - 4, 8, value);
        offset += 4;

        memset(&state, 0, sizeof(state));

        for (i = 0; i < (encodingScheme & 0x3FFF); i++) {
            uint8_t *word = (uint8_t *)wmem_alloc(pinfo->pool, 10);
            if (!(i & 1)) {
                word[0] = (cache >> 16) & 0xFF;
                word[1] = (cache >> 24) & 0xFF;
                cache = tvb_get_ntohl(tvb, offset);
                offset += 4;
                word[2] = cache & 0xFF;
                word[3] = (cache >> 8) & 0xFF;
                word[4] = (cache >> 16) & 0xFF;
                word[5] = (cache >> 24) & 0xFF;
                cache = tvb_get_ntohl(tvb, offset);
                offset += 4;
                word[6] = cache & 0xFF;
                word[7] = (cache >> 8) & 0xFF;
                word[8] = (cache >> 16) & 0xFF;
                word[9] = (cache >> 24) & 0xFF;
            } else {
                cache = tvb_get_ntohl(tvb, offset);
                offset += 4;
                word[0] = cache & 0xFF;
                word[1] = (cache >> 8) & 0xFF;
                word[2] = (cache >> 16) & 0xFF;
                word[3] = (cache >> 24) & 0xFF;
                cache = tvb_get_ntohl(tvb, offset);
                offset += 4;
                word[4] = cache & 0xFF;
                word[5] = (cache >> 8) & 0xFF;
                word[6] = (cache >> 16) & 0xFF;
                word[7] = (cache >> 24) & 0xFF;
                cache = tvb_get_ntohl(tvb, offset);
                offset += 4;
                word[8] = cache & 0xFF;
                word[9] = (cache >> 8) & 0xFF;
            }

            newtvb = tvb_new_child_real_data(tvb, word, 10, 10);
            add_new_data_source(pinfo, newtvb, "Link 16 Word");
            call_dissector_with_data(link16_handle, newtvb, pinfo, tree, &state);
        }
        break;
    }
    return offset;
}

/* Array records
 */
static int parseField_DIS_FIELDS_FIXED_DATUM(tvbuff_t *tvb, proto_tree *tree, int offset, const char* field_name, uint32_t num_items)
{
    proto_tree  *sub_tree;
    uint32_t i;


    for (i = 0; i < num_items; i++)
    {
        sub_tree = proto_tree_add_subtree(tree, tvb, offset, 8, ett_fixed_datum, NULL, field_name);

        proto_tree_add_item(sub_tree, hf_dis_datum_id, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        proto_tree_add_item(sub_tree, hf_dis_fixed_datum_value, tvb, offset, 4, ENC_NA);
        proto_tree_add_item(sub_tree, hf_dis_fixed_datum_value_as_int, tvb, offset, 4, ENC_BIG_ENDIAN);
        proto_tree_add_item(sub_tree, hf_dis_fixed_datum_value_as_float, tvb, offset, 4, ENC_BIG_ENDIAN);
        proto_tree_add_item(sub_tree, hf_dis_fixed_datum_value_as_uint, tvb, offset, 4, ENC_BIG_ENDIAN);

        offset += 4;
    }

    return offset;
}

static int parseField_DIS_FIELDS_VARIABLE_DATUM(tvbuff_t *tvb, proto_tree *tree, int offset, const char* field_name, uint32_t num_items)
{
    proto_item  *ti;
    proto_tree  *sub_tree;
    uint32_t i, data_length, lengthInBytes;
    unsigned char padding;

    for (i = 0; i < num_items; i++)
    {
        sub_tree = proto_tree_add_subtree(tree, tvb, offset, -1, ett_fixed_datum, &ti, field_name);

        proto_tree_add_item(sub_tree, hf_dis_datum_id, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        data_length = tvb_get_ntohl(tvb, offset);
        proto_tree_add_item(sub_tree, hf_dis_datum_length, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        lengthInBytes = data_length / 8;
        if (data_length % 8 > 0)
            lengthInBytes += 1;/* add another byte for the remaining bits */

        proto_tree_add_item(sub_tree, hf_dis_variable_datum_value, tvb, offset, lengthInBytes, ENC_NA);
        proto_tree_add_item(sub_tree, hf_dis_variable_datum_value_as_text, tvb, offset, lengthInBytes, ENC_ASCII);

        offset += lengthInBytes;

        padding = lengthInBytes % 8;
        if (padding > 0)
            padding = 8 - padding;

        offset += padding;

        proto_item_set_end(ti, tvb, offset);
    }

    return offset;
}

static int parseField_DIS_FIELDS_FIXED_DATUM_IDS(tvbuff_t *tvb, proto_tree *tree, int offset, const char* field_name, uint32_t num_items)
{
    proto_tree  *sub_tree;
    uint32_t i;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, num_items*4, ett_fixed_datum, NULL, field_name);

    for (i = 0; i < num_items; i++)
    {
        proto_tree_add_item(sub_tree, hf_dis_datum_id, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;
    }

    return offset;
}

static int parseField_DIS_FIELDS_VARIABLE_DATUM_IDS(tvbuff_t *tvb, proto_tree *tree, int offset, const char* field_name, uint32_t num_items)
{
    return parseField_DIS_FIELDS_FIXED_DATUM_IDS(tvb, tree, offset, field_name, num_items);
}

static int parseField_TRACK_JAM(tvbuff_t *tvb, proto_tree *tree, int offset, const char* entity_name)
{
    proto_tree  *sub_tree;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 8, ett_trackjam, NULL, entity_name);

    proto_tree_add_item(sub_tree, hf_dis_entity_id_site, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_entity_id_application, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_entity_id_entity, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_emitter_id, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(sub_tree, hf_dis_beam_id, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    return offset;
}

/* Array record contents - variable parameter records
 */
static int dissect_DIS_FIELDS_VP_ARTICULATED_PART(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree_add_item(tree, hf_dis_vp_change, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vp_part_attached_to_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_vp_artic_param_type, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(tree, hf_dis_vp_parameter_value, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(tree, hf_dis_vp_padding, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    return offset;
}

static int dissect_DIS_FIELDS_VP_ATTACHED_PART(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree_add_item(tree, hf_dis_vp_attached_indicator, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vp_part_attached_to_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_vp_artic_param_type, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    offset = dissect_DIS_FIELDS_ENTITY_TYPE(tvb, tree, offset, "Part Type", NULL, NULL);

    return offset;
}

static int dissect_DIS_FIELDS_VP_ENTITY_OFFSET(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree  *sub_tree;

    proto_tree_add_item(tree, hf_dis_vp_offset_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
    offset += 2;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_offset_vector, NULL, "Offset");

    proto_tree_add_item(sub_tree, hf_dis_vp_offset_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(sub_tree, hf_dis_vp_offset_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(sub_tree, hf_dis_vp_offset_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    return offset;
}

static int dissect_DIS_FIELDS_VP_ENTITY_ASSOCIATION(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree_add_item(tree, hf_dis_vp_change_indicator, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vp_association_status, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vp_association_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    offset = parseField_Entity(tvb, tree, offset, "Object Identifier", NULL);

    proto_tree_add_item(tree, hf_dis_vp_own_station_location, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_vp_phys_conn_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vp_group_member_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vp_group_number, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    return offset;
}

/* Variable Records
 */
static int dissect_DIS_FIELDS_VR_APPLICATION_HEALTH_STATUS(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 1, ENC_NA);
    offset++;

    proto_tree_add_item(tree, hf_dis_vr_status_type, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_vr_general_status, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vr_specific_status, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_vr_status_value_int, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_vr_status_value_float, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;

    return offset;
}

static int dissect_DIS_FIELDS_VR_APPLICATION_INITIALIZATION(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    proto_tree_add_item(tree, hf_dis_vr_exercise_id, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 1, ENC_NA);
    offset++;

    proto_tree_add_item(tree, hf_dis_vr_exercise_file_path, tvb, offset, 256, ENC_ASCII);
    offset += 256;

    proto_tree_add_item(tree, hf_dis_vr_exercise_file_name, tvb, offset, 128, ENC_ASCII);
    offset += 128;

    proto_tree_add_item(tree, hf_dis_vr_application_role, tvb, offset, 64, ENC_ASCII);
    offset += 64;

    return offset;
}

static int dissect_DIS_FIELDS_VR_DATA_QUERY(tvbuff_t *tvb, proto_tree *tree, int offset)
{
    uint32_t numFixed;

    numFixed = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item(tree, hf_dis_vr_num_records, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    offset = parseField_DIS_FIELDS_FIXED_DATUM_IDS(tvb, tree, offset, "Record", numFixed);

    return offset;
}

/******************************************************************************
*
* PDUS
*
*******************************************************************************/


static int dissect_DIS_PARSER_ENTITY_STATE_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    uint8_t entityKind;
    uint8_t entityDomain;
    static uint32_t entitySite;
    static uint32_t entityApplication;
    static uint32_t entityEntity;
    proto_item *ti;
    proto_tree *sub_tree;
    proto_tree *sub_tree2;
    uint8_t variableParameterType, numVariable, entity_marking_character_set;
    uint32_t i;
    const uint8_t *entity_marking_text;

    entitySite = tvb_get_ntohs(tvb, offset);
    entityApplication = tvb_get_ntohs(tvb, offset+2);
    entityEntity = tvb_get_ntohs(tvb, offset+4);

    uint32_t entityid =0;
    offset = parseField_Entity(tvb, tree, offset, "Entity ID", &entityid);

    proto_tree_add_item(tree, hf_dis_force_id, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    numVariable = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_dis_num_art_params, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    offset = dissect_DIS_FIELDS_ENTITY_TYPE(tvb, tree, offset, "Entity Type", &entityKind, &entityDomain);

    col_append_fstr( pinfo->cinfo, COL_INFO, ", %s, %s, (%u:%u:%u)",
                    val_to_str_const(entityKind, DIS_PDU_EntityKind_Strings, "Unknown Entity Kind"),
                    val_to_str_const(entityDomain, DIS_PDU_Domain_Strings, "Unknown Entity Domain"),
                    entitySite , entityApplication , entityEntity
                    );

    packet_context_t *context = wmem_map_lookup(packet_context_map, GUINT_TO_POINTER(entityid));
    if (context == NULL)
    {
        context = wmem_new0(wmem_file_scope(), packet_context_t);
        context->entity_kind = entityKind;
        context->entity_domain = entityDomain;

        wmem_map_insert(packet_context_map, GUINT_TO_POINTER(entityid), context);
    }

    offset = dissect_DIS_FIELDS_ENTITY_TYPE(tvb, tree, offset, "Alternative Entity Type", NULL, NULL);

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_entity_linear_velocity, NULL, "Entity Linear Velocity");
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_velocity_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_velocity_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_velocity_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 24, ett_entity_location, NULL, "Entity Location");
    proto_tree_add_item(sub_tree, hf_dis_entity_location_x_double, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_entity_location_y_double, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_entity_location_z_double, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_entity_orientation, NULL, "Entity Orientation");
    proto_tree_add_item(sub_tree, hf_dis_entity_orientation_psi, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_orientation_theta, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_orientation_phi, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    if ((entityKind == DIS_ENTITYKIND_PLATFORM) &&
        (entityDomain == DIS_DOMAIN_LAND))
    {
        static int * const entity_appearance_domain_land_bitmask[] =
        {
            &hf_appearance_landform_paint_scheme,
            &hf_appearance_landform_mobility,
            &hf_appearance_landform_fire_power,
            &hf_appearance_landform_damage,
            &hf_appearance_landform_smoke_entity,
            &hf_appearance_landform_trailing_effects_entity,
            &hf_appearance_landform_hatch,
            &hf_appearance_landform_head_lights,
            &hf_appearance_landform_tail_lights,
            &hf_appearance_landform_brake_lights,
            &hf_appearance_landform_flaming,
            &hf_appearance_landform_launcher,
            &hf_appearance_landform_camouflage_type,
            &hf_appearance_landform_concealed,
            &hf_appearance_landform_frozen_status,
            &hf_appearance_landform_power_plant_status,
            &hf_appearance_landform_state,
            &hf_appearance_landform_tent,
            &hf_appearance_landform_ramp,
            &hf_appearance_landform_blackout_lights,
            &hf_appearance_landform_blackout_brake_lights,
            &hf_appearance_landform_spot_lights,
            &hf_appearance_landform_interior_lights,
            &hf_appearance_landform_surrender_state,
            &hf_appearance_landform_masked_cloaked,
            NULL
        };

        proto_tree_add_bitmask(tree, tvb, offset, hf_entity_appearance, ett_entity_appearance, entity_appearance_domain_land_bitmask, ENC_BIG_ENDIAN);
    }
    else if ((entityKind == DIS_ENTITYKIND_PLATFORM) &&
        (entityDomain == DIS_DOMAIN_AIR))
    {
        static int * const entity_appearance_domain_air_bitmask[] =
        {
            &hf_appearance_airform_paint_scheme,
            &hf_appearance_airform_propulsion_killed,
            &hf_appearance_airform_nvg_mode,
            &hf_appearance_airform_damage,
            &hf_appearance_airform_is_smoke_emanating,
            &hf_appearance_airform_is_engine_emitting_smoke,
            &hf_appearance_airform_trailing_effects,
            &hf_appearance_airform_canopy_troop_door,
            &hf_appearance_airform_landing_lights_on,
            &hf_appearance_airform_navigation_lights_on,
            &hf_appearance_airform_anti_collision_lights_on,
            &hf_appearance_airform_is_flaming,
            &hf_appearance_airform_afterburner_on,
            &hf_appearance_airform_lower_anti_collision_light_on,
            &hf_appearance_airform_upper_anti_collision_light_on,
            &hf_appearance_airform_anti_collision_light_day_night,
            &hf_appearance_airform_is_blinking,
            &hf_appearance_airform_is_frozen,
            &hf_appearance_airform_power_plant_on,
            &hf_appearance_airform_state,
            &hf_appearance_airform_formation_lights_on,
            &hf_appearance_airform_landing_gear_extended,
            &hf_appearance_airform_cargo_doors_opened,
            &hf_appearance_airform_navigation_position_brightness,
            &hf_appearance_airform_spot_search_light_1_on,
            &hf_appearance_airform_interior_lights_on,
            &hf_appearance_airform_reverse_thrust_engaged,
            &hf_appearance_airform_weight_on_wheels,
            NULL
        };

        proto_tree_add_bitmask(tree, tvb, offset, hf_entity_appearance, ett_entity_appearance, entity_appearance_domain_air_bitmask, ENC_BIG_ENDIAN);
    }
    else if (entityKind == DIS_ENTITYKIND_LIFE_FORM)
    {
        static int * const entity_appearance_kind_life_form_bitmask[] =
        {
            &hf_appearance_lifeform_paint_scheme,
            &hf_appearance_lifeform_health,
            &hf_appearance_lifeform_compliance,
            &hf_appearance_lifeform_flash_lights,
            &hf_appearance_lifeform_state,
            &hf_appearance_frozen_status,
            &hf_appearance_state,
            &hf_appearance_weapon_1,
            &hf_appearance_weapon_2,
            &hf_appearance_camouflage_type,
            &hf_appearance_concealed_stationary,
            &hf_appearance_concealed_movement,
            NULL
        };

        proto_tree_add_bitmask(tree, tvb, offset, hf_entity_appearance, ett_entity_appearance, entity_appearance_kind_life_form_bitmask, ENC_BIG_ENDIAN);
    }
    else
    {
        proto_tree_add_item(tree, hf_entity_appearance, tvb, offset, 4, ENC_BIG_ENDIAN);
    }
    offset += 4;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 40, ett_entity_dead_reckoning_parameters, NULL, "Dead Reckoning Parameters");
    proto_tree_add_item(sub_tree, hf_dis_entity_dead_reckoning_algorithm, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;
    proto_tree_add_item(sub_tree, hf_dis_dead_reckoning_other_parameters, tvb, offset, 15, ENC_NA);
    offset += 15;

    sub_tree2 = proto_tree_add_subtree(sub_tree, tvb, offset, 12, ett_entity_linear_acceleration, NULL, "Entity Linear Acceleration");
    proto_tree_add_item(sub_tree2, hf_dis_entity_linear_acceleration_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree2, hf_dis_entity_linear_acceleration_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree2, hf_dis_entity_linear_acceleration_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    sub_tree2 = proto_tree_add_subtree(sub_tree, tvb, offset, 12, ett_entity_angular_velocity, NULL, "Entity Angular Velocity");
    proto_tree_add_item(sub_tree2, hf_dis_entity_entity_angular_velocity_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree2, hf_dis_entity_entity_angular_velocity_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree2, hf_dis_entity_entity_angular_velocity_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_entity_marking_text, NULL, "Entity Marking");

    entity_marking_character_set = tvb_get_uint8(tvb, offset);
    proto_tree_add_uint(sub_tree, hf_dis_entity_marking_character_set, tvb, offset, 1, entity_marking_character_set);
    offset += 1;
    switch (entity_marking_character_set)
    {
        case 0:/* Unused */
            break;/* Don't translate it, nothing to be translated */
        case 1:/* ASCII */
            proto_tree_add_item_ret_string(sub_tree, hf_dis_entity_marking, tvb, offset, 11, ENC_ASCII|ENC_NA, pinfo->pool, &entity_marking_text);
            col_append_fstr(pinfo->cinfo, COL_INFO, ", %s", entity_marking_text);
        case 2:/* Army Marking (CCTT) */
            /* TODO: Complete this */
            break;
        case 3:/* Digit Chevron */
            /* TODO: Complete this */
            break;
    }
    offset += 11;

    proto_tree_add_item(tree, hf_dis_capabilities, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    for (i = 0; i < numVariable; i++)
    {
        sub_tree = proto_tree_add_subtree(tree, tvb, offset, 1, ett_variable_parameter, &ti, "Variable Parameter");

        proto_tree_add_item(sub_tree, hf_dis_variable_parameter_type, tvb, offset, 1, ENC_BIG_ENDIAN);
        variableParameterType = tvb_get_uint8(tvb, offset);
        offset++;

        offset = parseField_VariableParameter(tvb, sub_tree, offset, variableParameterType);
        proto_item_set_end(ti, tvb, offset);
    }

    return offset;
}

static int dissect_DIS_PARSER_ENTITY_STATE_UPDATE_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    static uint32_t entitySite;
    static uint32_t entityApplication;
    static uint32_t entityEntity;
    proto_item *ti;
    proto_tree *sub_tree;
    uint8_t variableParameterType, numVariable;
    uint32_t i;

    entitySite = tvb_get_ntohs(tvb, offset);
    entityApplication = tvb_get_ntohs(tvb, offset+2);
    entityEntity = tvb_get_ntohs(tvb, offset+4);

    uint32_t entityid = 0;
    offset = parseField_Entity(tvb, tree, offset, "Entity ID", &entityid);

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 1, ENC_NA);
    offset++;

    numVariable = tvb_get_uint8(tvb, offset);

    proto_tree_add_item(tree, hf_dis_num_variable_records, tvb, offset, 1, ENC_BIG_ENDIAN); //number of variable parameter records
    offset++;

    col_append_fstr( pinfo->cinfo, COL_INFO, ", (%u:%u:%u)",
                    entitySite , entityApplication , entityEntity
                    );

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_entity_linear_velocity, NULL, "Entity Linear Velocity");
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_velocity_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_velocity_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_velocity_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 24, ett_entity_location, NULL, "Entity Location");
    proto_tree_add_item(sub_tree, hf_dis_entity_location_x_double, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_entity_location_y_double, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_entity_location_z_double, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_entity_orientation, NULL, "Entity Orientation");
    proto_tree_add_item(sub_tree, hf_dis_entity_orientation_psi, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_orientation_theta, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_orientation_phi, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    packet_context_t *context = wmem_map_lookup(packet_context_map, GUINT_TO_POINTER(entityid));
    if (context != NULL)
    {
        //look for the previous EntityStatePDU in order to get the EntityKind and EntityDomain
        if ((context->entity_kind == DIS_ENTITYKIND_PLATFORM) &&
            (context->entity_domain == DIS_DOMAIN_LAND))
        {
            static int * const entity_appearance_domain_land_bitmask[] =
            {
                &hf_appearance_landform_paint_scheme,
                &hf_appearance_landform_mobility,
                &hf_appearance_landform_fire_power,
                &hf_appearance_landform_damage,
                &hf_appearance_landform_smoke_entity,
                &hf_appearance_landform_trailing_effects_entity,
                &hf_appearance_landform_hatch,
                &hf_appearance_landform_head_lights,
                &hf_appearance_landform_tail_lights,
                &hf_appearance_landform_brake_lights,
                &hf_appearance_landform_flaming,
                &hf_appearance_landform_launcher,
                &hf_appearance_landform_camouflage_type,
                &hf_appearance_landform_concealed,
                &hf_appearance_landform_frozen_status,
                &hf_appearance_landform_power_plant_status,
                &hf_appearance_landform_state,
                &hf_appearance_landform_tent,
                &hf_appearance_landform_ramp,
                &hf_appearance_landform_blackout_lights,
                &hf_appearance_landform_blackout_brake_lights,
                &hf_appearance_landform_spot_lights,
                &hf_appearance_landform_interior_lights,
                &hf_appearance_landform_surrender_state,
                &hf_appearance_landform_masked_cloaked,
                NULL
            };

            proto_tree_add_bitmask(tree, tvb, offset, hf_entity_appearance, ett_entity_appearance, entity_appearance_domain_land_bitmask, ENC_BIG_ENDIAN);
        }
        else if ((context->entity_kind == DIS_ENTITYKIND_PLATFORM) &&
            (context->entity_domain == DIS_DOMAIN_AIR))
        {
            static int * const entity_appearance_domain_air_bitmask[] =
            {
                &hf_appearance_airform_paint_scheme,
                &hf_appearance_airform_propulsion_killed,
                &hf_appearance_airform_nvg_mode,
                &hf_appearance_airform_damage,
                &hf_appearance_airform_is_smoke_emanating,
                &hf_appearance_airform_is_engine_emitting_smoke,
                &hf_appearance_airform_trailing_effects,
                &hf_appearance_airform_canopy_troop_door,
                &hf_appearance_airform_landing_lights_on,
                &hf_appearance_airform_navigation_lights_on,
                &hf_appearance_airform_anti_collision_lights_on,
                &hf_appearance_airform_is_flaming,
                &hf_appearance_airform_afterburner_on,
                &hf_appearance_airform_lower_anti_collision_light_on,
                &hf_appearance_airform_upper_anti_collision_light_on,
                &hf_appearance_airform_anti_collision_light_day_night,
                &hf_appearance_airform_is_blinking,
                &hf_appearance_airform_is_frozen,
                &hf_appearance_airform_power_plant_on,
                &hf_appearance_airform_state,
                &hf_appearance_airform_formation_lights_on,
                &hf_appearance_airform_landing_gear_extended,
                &hf_appearance_airform_cargo_doors_opened,
                &hf_appearance_airform_navigation_position_brightness,
                &hf_appearance_airform_spot_search_light_1_on,
                &hf_appearance_airform_interior_lights_on,
                &hf_appearance_airform_reverse_thrust_engaged,
                &hf_appearance_airform_weight_on_wheels,
                NULL
            };

            proto_tree_add_bitmask(tree, tvb, offset, hf_entity_appearance, ett_entity_appearance, entity_appearance_domain_air_bitmask, ENC_BIG_ENDIAN);
        }
        else if (context->entity_kind == DIS_ENTITYKIND_LIFE_FORM)
        {
            static int * const entity_appearance_kind_life_form_bitmask[] =
            {
                &hf_appearance_lifeform_paint_scheme,
                &hf_appearance_lifeform_health,
                &hf_appearance_lifeform_compliance,
                &hf_appearance_lifeform_flash_lights,
                &hf_appearance_lifeform_state,
                &hf_appearance_frozen_status,
                &hf_appearance_state,
                &hf_appearance_weapon_1,
                &hf_appearance_weapon_2,
                &hf_appearance_camouflage_type,
                &hf_appearance_concealed_stationary,
                &hf_appearance_concealed_movement,
                NULL
            };

            proto_tree_add_bitmask(tree, tvb, offset, hf_entity_appearance, ett_entity_appearance, entity_appearance_kind_life_form_bitmask, ENC_BIG_ENDIAN);
        }
        else
        {
            proto_tree_add_item(tree, hf_entity_appearance, tvb, offset, 4, ENC_BIG_ENDIAN);
        }
    }
    else
    {
        sub_tree = proto_tree_add_item(tree, hf_entity_appearance, tvb, offset, 4, ENC_BIG_ENDIAN);
        expert_add_info(pinfo, sub_tree, &ei_entityidentifier_not_yet_received);
    }

    offset += 4;

    for (i = 0; i < numVariable; i++)
    {
        sub_tree = proto_tree_add_subtree(tree, tvb, offset, 1, ett_variable_parameter, &ti, "Variable Parameter");

        proto_tree_add_item(sub_tree, hf_dis_variable_parameter_type, tvb, offset, 1, ENC_BIG_ENDIAN);
        variableParameterType = tvb_get_uint8(tvb, offset);
        offset++;

        offset = parseField_VariableParameter(tvb, sub_tree, offset, variableParameterType);
        proto_item_set_end(ti, tvb, offset);
    }

    return offset;
}
/* DIS Collision PDUs
 */
static int dissect_DIS_PARSER_COLLISION_PDU(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{
    proto_tree *sub_tree;

    offset = parseField_Entity(tvb, tree, offset, "Issuing Entity ID", NULL);
    offset = parseField_Entity(tvb, tree, offset, "Colliding Entity ID", NULL);
    offset = dissect_DIS_FIELDS_EVENT_ID(tvb, tree, offset, "Event ID");

    /* 8 Bit Collision Type */
    proto_tree_add_item(tree, hf_dis_collision_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    /* 8 Bit Padding */
    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 1, ENC_NA);
    offset++;

    /* Velocity */
    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_linear_velocity, NULL, "Velocity");

    proto_tree_add_item(sub_tree, hf_dis_linear_velocity_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_linear_velocity_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_linear_velocity_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    /* Mass */
    proto_tree_add_item(tree, hf_dis_collision_mass, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    /* Location (with respect to entity) */
    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_ua_location, NULL, "Location (with respect to entity)"); /* TODO: ett_ua_location ==> ett_location? */

    proto_tree_add_item(sub_tree, hf_dis_collision_location_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_collision_location_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_collision_location_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    return offset;
}

/* DIS Distributed Emission Regeneration PDUs
 */
static int dissect_DIS_PARSER_ELECTROMAGNETIC_EMISSION_PDU(tvbuff_t *tvb, packet_info *pinfo _U_, proto_tree *tree, int offset)
{
    proto_item *emission_ti, *beam_ti;
    proto_tree *sub_tree, *sub_tree2, *fundamental_tree;
    uint8_t i, j, k, numVariable, numBeams, numTrackJamTargets;

    offset = parseField_Entity(tvb, tree, offset, "Emitting Entity ID", NULL);
    offset = dissect_DIS_FIELDS_EVENT_ID(tvb, tree, offset, "Event ID");

    proto_tree_add_item(tree, hf_dis_state_update_indicator, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    numVariable = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_dis_num_electromagnetic_emission_systems, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
    offset += 2;

    for (i = 0; i < numVariable; i++)
    {
        sub_tree = proto_tree_add_subtree(tree, tvb, offset, -1, ett_emission_system, &emission_ti, "Emission System");

        proto_tree_add_item(sub_tree, hf_dis_em_data_length, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        numBeams = tvb_get_uint8(tvb, offset);
        proto_tree_add_item(sub_tree, hf_dis_em_num_beams, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        proto_tree_add_item(sub_tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
        offset += 2;

        sub_tree2 = proto_tree_add_subtree(sub_tree, tvb, offset, 4, ett_emitter_system, NULL, "Emitter System");

        proto_tree_add_item(sub_tree2, hf_dis_emitter_name, tvb, offset, 2, ENC_BIG_ENDIAN);
        offset += 2;
        proto_tree_add_item(sub_tree2, hf_dis_emission_function, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;
        proto_tree_add_item(sub_tree2, hf_dis_emitter_id_number, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        sub_tree2 = proto_tree_add_subtree(sub_tree, tvb, offset, 12, ett_emitter_location, NULL, "Location");

        proto_tree_add_item(sub_tree2, hf_dis_em_location_x, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;
        proto_tree_add_item(sub_tree2, hf_dis_em_location_y, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;
        proto_tree_add_item(sub_tree2, hf_dis_em_location_z, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        for (j = 0; j < numBeams; j++)
        {
            sub_tree2 = proto_tree_add_subtree(sub_tree, tvb, offset, -1, ett_em_beam, &beam_ti, "Beam");

            proto_tree_add_item(sub_tree2, hf_dis_em_beam_data_length, tvb, offset, 1, ENC_BIG_ENDIAN);
            offset++;

            proto_tree_add_item(sub_tree2, hf_dis_em_beam_id_number, tvb, offset, 1, ENC_BIG_ENDIAN);
            offset++;

            proto_tree_add_item(sub_tree2, hf_dis_em_beam_parameter_index, tvb, offset, 2, ENC_BIG_ENDIAN);
            offset += 2;

            fundamental_tree = proto_tree_add_subtree(sub_tree2, tvb, offset, 40, ett_em_fundamental_parameter_data, NULL, "Fundamental Parameter Data");

            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_frequency, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_frequency_range, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_effective_radiated_power, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_pulse_repetition_freq, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_pulse_width, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_beam_azimuth_center, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_beam_azimuth_sweep, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_beam_elevation_center, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_beam_elevation_sweep, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;
            proto_tree_add_item(fundamental_tree, hf_dis_em_fund_beem_sweep_sync, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;

            proto_tree_add_item(sub_tree2, hf_dis_beam_function, tvb, offset, 1, ENC_BIG_ENDIAN);
            offset++;

            numTrackJamTargets = tvb_get_uint8(tvb, offset);
            proto_tree_add_item(sub_tree2, hf_dis_track_jam_num_targ, tvb, offset, 1, ENC_BIG_ENDIAN);
            offset++;

            proto_tree_add_item(sub_tree2, hf_dis_track_jam_high_density, tvb, offset, 1, ENC_BIG_ENDIAN);
            offset++;

            proto_tree_add_item(sub_tree2, hf_dis_padding, tvb, offset, 1, ENC_NA);
            offset++;

            proto_tree_add_item(sub_tree2, hf_dis_jamming_mode_seq, tvb, offset, 4, ENC_BIG_ENDIAN);
            offset += 4;

            for (k = 0; k < numTrackJamTargets; k++)
            {
                offset = parseField_TRACK_JAM(tvb, sub_tree2, offset, "Track/Jam Entity");
            }

            proto_item_set_end(beam_ti, tvb, offset);
        }

        proto_item_set_end(emission_ti, tvb, offset);
    }

    return offset;
}

/* DIS Underwater Acoustic PDUs
 */
static int dissect_DIS_PARSER_UNDERWATER_ACOUSTIC_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    proto_tree *sub_tree, *sub_tree2;
    uint8_t i, numShafts, numApas, numUAEmitter, numUABeams = 0;

    offset = parseField_Entity(tvb, tree, offset, "Emitting Entity ID", NULL);
    offset = dissect_DIS_FIELDS_EVENT_ID(tvb, tree, offset, "Event ID");

    proto_tree_add_item(tree, hf_dis_state_update_indicator, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 1, ENC_NA);
    offset++;

    proto_tree_add_item(tree, hf_dis_passive_parameter_index, tvb, offset, 2, ENC_BIG_ENDIAN); /* !! enum !! */
    offset += 2;

    proto_tree_add_item(tree, hf_dis_propulsion_plant_config, tvb, offset, 1, ENC_BIG_ENDIAN); /* !! enum !! */
    offset++;

    numShafts = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_dis_num_shafts, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    numApas = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_dis_num_apas, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    numUAEmitter = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_dis_num_ua_emitter_systems, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    col_append_fstr( pinfo->cinfo, COL_INFO, ", Shafts=%d, APA=%d, Acoustic Emitter=%d",
                      numShafts, numApas, numUAEmitter);

    for (i = 0; i < numShafts; i++)
    {
        sub_tree = proto_tree_add_subtree_format(tree, tvb, offset, 6, ett_shafts, NULL, "Shafts [%d of %d]", i+1, numShafts);

        proto_tree_add_item(sub_tree, hf_dis_shaft_rpm_current, tvb, offset, 2, ENC_BIG_ENDIAN);
        offset += 2;

        proto_tree_add_item(sub_tree, hf_dis_shaft_rpm_ordered, tvb, offset, 2, ENC_BIG_ENDIAN);
        offset += 2;

        proto_tree_add_item(sub_tree, hf_dis_shaft_rpm_change_rate, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;
    }

    for (i = 0; i < numApas; i++)
    {
        sub_tree = proto_tree_add_subtree_format(tree, tvb, offset, 4, ett_apas, NULL, "APAs [%d of %d]", i+1, numApas);

        proto_tree_add_item(sub_tree, hf_dis_apas_parameter_index, tvb, offset, 2, ENC_BIG_ENDIAN); /*FIXME enum*/
        offset += 2;

        proto_tree_add_item(sub_tree, hf_dis_apas_value, tvb, offset, 2, ENC_BIG_ENDIAN);
        offset += 2;
    }

    for (i = 0; i < numUAEmitter; i++)
    {
        sub_tree = proto_tree_add_subtree_format(tree, tvb, offset, 20, ett_underwater_acoustic_emission, NULL,
                            "Underwater Acoustic Emission System [%d of %d]", i+1, numUAEmitter);

        proto_tree_add_item(sub_tree, hf_dis_ua_emitter_data_length, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        numUABeams += tvb_get_uint8(tvb, offset);
        proto_tree_add_item(sub_tree, hf_dis_ua_num_beams, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        proto_tree_add_item(sub_tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
        offset += 2;

        sub_tree2 = proto_tree_add_subtree(sub_tree, tvb, offset, 4, ett_acoustic_emitter_system, NULL, "Acoustic Emitter System");

        proto_tree_add_item(sub_tree2, hf_dis_ua_emission_name, tvb, offset, 2, ENC_BIG_ENDIAN); /*FIXME enum*/
        offset += 2;
        proto_tree_add_item(sub_tree2, hf_dis_ua_emission_function, tvb, offset, 1, ENC_BIG_ENDIAN); /*FIXME enum*/
        offset++;
        proto_tree_add_item(sub_tree2, hf_dis_ua_emission_id_number, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        sub_tree2 = proto_tree_add_subtree(sub_tree, tvb, offset, 12, ett_ua_location, NULL, "Location (with respect to entity)");

        proto_tree_add_item(sub_tree2, hf_dis_ua_location_x, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        proto_tree_add_item(sub_tree2, hf_dis_ua_location_y, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        proto_tree_add_item(sub_tree2, hf_dis_ua_location_z, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;
    }

    for (i = 0; i < numUABeams; ++i)
    {
        sub_tree = proto_tree_add_subtree_format(tree, tvb, offset, 24, ett_ua_beams, NULL, "Beams [%d of %d]", i+1, numUABeams);

        proto_tree_add_item(sub_tree, hf_dis_ua_beam_data_length, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        proto_tree_add_item(sub_tree, hf_dis_ua_beam_id_number, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset++;

        proto_tree_add_item(sub_tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
        offset += 2;

        sub_tree2 = proto_tree_add_subtree_format(sub_tree, tvb, offset, 20, ett_ua_beam_data, NULL, "Fundamental Data Parameters");

        proto_tree_add_item(sub_tree2, hf_dis_ua_beam_active_emission_parameter_index, tvb, offset, 2, ENC_BIG_ENDIAN); /*FIXME enum!!!*/
        offset += 2;

        proto_tree_add_item(sub_tree2, hf_dis_ua_beam_scan_pattern, tvb, offset, 2, ENC_BIG_ENDIAN); /*FIXME enum!!!*/
        offset += 2;

        proto_tree_add_item(sub_tree2, hf_dis_ua_beam_center_azimuth, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        proto_tree_add_item(sub_tree2, hf_dis_ua_beam_azimuthal_beamwidth, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        proto_tree_add_item(sub_tree2, hf_dis_ua_beam_center_de, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;

        proto_tree_add_item(sub_tree2, hf_dis_ua_beam_de_beamwidth, tvb, offset, 4, ENC_BIG_ENDIAN);
        offset += 4;
    }

    return offset;
}

static int dissect_DIS_PARSER_IFF_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset, dis_header_t* header)
{
    proto_item *ti = NULL;
    proto_tree *sub_tree = NULL,*field_tree = NULL;
    uint16_t site = 0, application = 0, entity = 0;
    uint16_t systemType = 0;
    uint16_t mode1 = 0, mode1_element1 = 0, mode1_element2 = 0;
    uint16_t rrb = 0, rrb_code = 0;
    uint16_t mode2 = 0, mode2_element1 = 0, mode2_element2 = 0, mode2_element3 = 0, mode2_element4 = 0;
    uint16_t mode3 = 0, mode3_element1 = 0, mode3_element2 = 0, mode3_element3 = 0, mode3_element4 = 0;
    uint16_t mode4 = 0;
    uint16_t modeS = 0, modeS_primary_type = 0, modeS_primary_code = 0, modeS_secondary_type = 0, modeS_secondary_code = 0;
    uint16_t parameter_5 = 0;
    int16_t altitude = 0;
    uint16_t parameter_6 = 0, tcas_acas_indicator = 0, tcas_acas_type = 0, tcas_I_II_type = 0;

    site = tvb_get_ntohs(tvb, offset);
    application = tvb_get_ntohs(tvb, offset+2);
    entity = tvb_get_ntohs(tvb, offset+4);
    offset = parseField_Entity(tvb, tree, offset, "Emitting Entity ID", NULL);
    offset = dissect_DIS_FIELDS_EVENT_ID(tvb, tree, offset, "Event ID");

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_iff_location, NULL, "Location (with respect to entity)");

    proto_tree_add_item(sub_tree, hf_dis_ua_location_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(sub_tree, hf_dis_ua_location_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(sub_tree, hf_dis_ua_location_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 6, ett_iff_system_id, NULL, "System ID");

    systemType = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item(sub_tree, hf_dis_iff_system_type, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_iff_system_name, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(sub_tree, hf_dis_iff_system_mode, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    ti = proto_tree_add_item(sub_tree, hf_dis_iff_change_options, tvb, offset, 1, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_change_options);

    proto_tree_add_item(field_tree, hf_dis_iff_change_indicator, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_alternate_mode_4, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_alternate_mode_c, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_heartbeat_indicator, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_transponder_interrogator_indicator, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_simulation_mode, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_interactive_capable, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_test_mode, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    if (header->version < DIS_VERSION_IEEE_1278_1_2012)
    {
        proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
        offset += 2;
    }
    else
    {
        proto_tree_add_item(tree, hf_dis_iff_system_designator, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset += 1;
        proto_tree_add_item(tree, hf_dis_iff_system_specific_data, tvb, offset, 1, ENC_BIG_ENDIAN);
        offset += 1;
    }

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 16, ett_iff_fundamental_operational_data, NULL, "Fundamental Operational Data");

    ti = proto_tree_add_item(sub_tree, hf_dis_iff_system_status, tvb, offset, 1, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_system_status);

    proto_tree_add_item(field_tree, hf_dis_iff_system_status_system_onoff, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_system_status_parameter_1, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_system_status_parameter_2, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_system_status_parameter_3, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_system_status_parameter_4, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_system_status_parameter_5, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_system_status_parameter_6, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_system_status_operational, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(sub_tree, hf_dis_iff_alternate_parameter_4, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    ti = proto_tree_add_item(sub_tree, hf_dis_iff_information_layers, tvb, offset, 1, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_information_layers);
    proto_tree_add_item(field_tree, hf_dis_iff_information_layers_layer_1, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_information_layers_layer_2, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_information_layers_layer_3, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_information_layers_layer_4, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_information_layers_layer_5, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_information_layers_layer_6, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_information_layers_layer_7, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    ti = proto_tree_add_item(sub_tree, hf_dis_iff_modifier, tvb, offset, 1, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_modifier);
    proto_tree_add_item(field_tree, hf_dis_iff_modifier_emergency, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_modifier_ident, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_modifier_sti, tvb, offset, 1, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_modifier_unmanned_aircraft, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    if (systemType == DIS_PDU_IFF_SYSTEM_TYPE_RRB_TRANSPONDER)
    {
        rrb = tvb_get_ntohs(tvb, offset);
        rrb_code = rrb & 0x1f;
        ti = proto_tree_add_item(sub_tree, hf_dis_iff_rrb, tvb, offset, 2, ENC_BIG_ENDIAN);
        field_tree = proto_item_add_subtree(ti, ett_iff_rrb);
        proto_tree_add_item(field_tree, hf_dis_iff_rrb_rrb_code, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_rrb_power_reduction_indicator, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_rrb_radar_enhancement_indicator, tvb, offset, 2, ENC_BIG_ENDIAN);
    }
    else
    {
        mode1 = tvb_get_ntohs(tvb, offset) & 0x3f;
        mode1_element1 = (mode1) & 0x7;
        mode1_element2 = ((mode1) >> 3) & 0x7;
        ti = proto_tree_add_item(sub_tree, hf_dis_iff_parameter_1, tvb, offset, 2, ENC_BIG_ENDIAN);
        field_tree = proto_item_add_subtree(ti, ett_iff_parameter_1);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_1, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_2, tvb, offset, 2, ENC_BIG_ENDIAN);
    }
    proto_tree_add_item(field_tree, hf_dis_iff_mode_status, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_damage, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_malfunction, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    mode2 = tvb_get_ntohs(tvb, offset) & 0xfff;
    mode2_element1 = (mode2) & 0x7;
    mode2_element2 = ((mode2) >> 3) & 0x7;
    mode2_element3 = ((mode2) >> 6) & 0x7;
    mode2_element4 = ((mode2) >> 9) & 0x7;
    ti = proto_tree_add_item(sub_tree, hf_dis_iff_parameter_2, tvb, offset, 2, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_parameter_2);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_1, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_2, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_3, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_4, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_status, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_damage, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_malfunction, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    mode3 = tvb_get_ntohs(tvb, offset) & 0xfff;
    mode3_element1 = (mode3) & 0x7;
    mode3_element2 = ((mode3) >> 3) & 0x7;
    mode3_element4 = ((mode3) >> 9) & 0x7;
    mode3_element3 = ((mode3) >> 6) & 0x7;
    ti = proto_tree_add_item(sub_tree, hf_dis_iff_parameter_3, tvb, offset, 2, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_parameter_3);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_1, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_2, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_3, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_code_element_4, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_status, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_damage, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_malfunction, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    if (systemType == DIS_PDU_IFF_SYSTEM_TYPE_MODE_S_INTERROGATOR)
    {
        modeS = tvb_get_ntohs(tvb, offset) & 0xffff;
        modeS_primary_type = (modeS) & 0x1;
        modeS_primary_code = ((modeS) >> 1) & 0x7f;
        modeS_secondary_type = ((modeS) >> 8)& 0x1;
        modeS_secondary_code = ((modeS) >> 9) & 0x7f;
        ti = proto_tree_add_item(sub_tree, hf_dis_iff_mode_s_interrogator_identifier, tvb, offset, 2, ENC_BIG_ENDIAN);
        field_tree = proto_item_add_subtree(ti, ett_iff_mode_s_interrogator_identifier);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_s_interrogator_identifier_primary_ic_type, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_s_interrogator_identifier_primary_ic_code, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_s_interrogator_identifier_secondary_ic_type, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_s_interrogator_identifier_secondary_ic_code, tvb, offset, 2, ENC_BIG_ENDIAN);
    }
    else
    {
        mode4 = tvb_get_ntohs(tvb, offset) & 0xfff;
        ti = proto_tree_add_item(sub_tree, hf_dis_iff_parameter_4, tvb, offset, 2, ENC_BIG_ENDIAN);
        field_tree = proto_item_add_subtree(ti, ett_iff_parameter_4);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_4, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_status, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_damage, tvb, offset, 2, ENC_BIG_ENDIAN);
        proto_tree_add_item(field_tree, hf_dis_iff_mode_malfunction, tvb, offset, 2, ENC_BIG_ENDIAN);
    }
    offset += 2;

    ti = proto_tree_add_item(sub_tree, hf_dis_iff_parameter_5, tvb, offset, 2, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_parameter_5);
    parameter_5 = tvb_get_ntohs(tvb, offset);
    altitude = ((parameter_5 >> 1) & 0x7ff) * ((parameter_5 & 1) ? -1: 1);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_c_altitude_indicator, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_c_altitude, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_status, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_damage, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_malfunction, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    parameter_6 = tvb_get_ntohs(tvb, offset);
    tcas_acas_indicator = tvb_get_ntohs(tvb, offset) & 0x2;
    tcas_acas_type = (tvb_get_ntohs(tvb, offset) >> 9) & 0x7;
    tcas_I_II_type = (tvb_get_ntohs(tvb, offset) >> 12) & 0x1;
    ti = proto_tree_add_item(sub_tree, hf_dis_iff_parameter_6, tvb, offset, 2, ENC_BIG_ENDIAN);
    field_tree = proto_item_add_subtree(ti, ett_iff_parameter_6);
    proto_tree_add_item(field_tree, hf_dis_iff_tcas_acas_basic_advanced_indicator, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_tcas_acas_tcas_acas_indicator, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_tcas_acas_software_version, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_tcas_acas_tcas_acas_type, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_tcas_acas_tcas_type, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_status, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_damage, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(field_tree, hf_dis_iff_mode_malfunction, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    col_append_fstr(pinfo->cinfo, COL_INFO, ", %d-%d-%d", site, application, entity);

    if (mode1)
    {
        col_append_fstr(pinfo->cinfo, COL_INFO, ", 1=%o%o", mode1_element1, mode1_element2);
    }
    else if (rrb)
    {
        col_append_fstr(pinfo->cinfo, COL_INFO, ", RRB=%d", rrb_code);
    }

    if (mode2) col_append_fstr(pinfo->cinfo, COL_INFO, ", 2=%o%o%o%o", mode2_element1, mode2_element2, mode2_element3, mode2_element4);
    if (mode3) col_append_fstr(pinfo->cinfo, COL_INFO, ", 3=%o%o%o%o", mode3_element1, mode3_element2, mode3_element3, mode3_element4);

    if (systemType == DIS_PDU_IFF_SYSTEM_TYPE_MODE_S_INTERROGATOR)
    {
        if (modeS_primary_type == 0)
        {
            col_append_str(pinfo->cinfo, COL_INFO, ", P=II ");
        }
        else
        {
            col_append_str(pinfo->cinfo, COL_INFO, ", P=SI");
        }
        col_append_fstr(pinfo->cinfo, COL_INFO, " %d", modeS_primary_code);

        if (modeS_secondary_type == 0)
        {
            col_append_str(pinfo->cinfo, COL_INFO, ", S=II");
        }
        else
        {
            col_append_str(pinfo->cinfo, COL_INFO, ", S=SI");
        }
        col_append_fstr(pinfo->cinfo, COL_INFO, " %d", modeS_secondary_code);
    }
    else if (mode4)
    {
        col_append_fstr(pinfo->cinfo, COL_INFO, ", 4=%d", mode4);
    }

    if (altitude || (parameter_5 & 0x2000)) col_append_fstr(pinfo->cinfo, COL_INFO, ", C=FL%d", altitude);

    if (parameter_6)
    {
        if (tcas_acas_indicator == 0)
        {
            if (tcas_I_II_type == 0)
            {
                col_append_str(pinfo->cinfo, COL_INFO, ", TCAS I");
            }
            else
            {
                col_append_str(pinfo->cinfo, COL_INFO, ", TCAS II");
            }
        }
        else
        {
            if (tcas_acas_type == 1)
            {
                col_append_str(pinfo->cinfo, COL_INFO, ", ACAS I");
            }
            else if (tcas_acas_type == 2)
            {
                col_append_str(pinfo->cinfo, COL_INFO, ", ACAS II");
            }
        }
    }

    return offset;
}

/* DIS Radio Communications protocol (RCP) family PDUs
 */
static int dissect_DIS_PARSER_TRANSMITTER_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    proto_item* ti;
    proto_tree* sub_tree;
    uint32_t radioID, disRadioTransmitState, modulationParamLength;
    uint16_t systemModulation;

    offset = parseField_Entity(tvb, tree, offset, "Entity ID", NULL);

    proto_tree_add_item(tree, hf_dis_radio_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    radioID = tvb_get_ntohs(tvb, offset);
    col_append_fstr( pinfo->cinfo, COL_INFO, ", RadioID=%u", radioID);
    offset += 2;

    offset = dissect_DIS_FIELDS_RADIO_ENTITY_TYPE(tvb, tree, offset, "Radio Entity Type");

    disRadioTransmitState = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_dis_radio_transmit_state, tvb, offset, 1, ENC_BIG_ENDIAN);
    col_append_fstr( pinfo->cinfo, COL_INFO, ", Transmit State=%s", val_to_str_const(disRadioTransmitState, DIS_PDU_Transmitter_Transmit_State_Strings, "Unknown Transmit State"));
    offset++;

    proto_tree_add_item(tree, hf_dis_radio_input_source, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
    offset += 2;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 24, ett_antenna_location, NULL, "Antenna Location");

    proto_tree_add_item(sub_tree, hf_dis_antenna_location_x, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_antenna_location_y, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_antenna_location_z, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_rel_antenna_location, NULL, "Relative Antenna Location");

    proto_tree_add_item(sub_tree, hf_dis_rel_antenna_location_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_rel_antenna_location_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_rel_antenna_location_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(tree, hf_dis_antenna_pattern_type, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_antenna_pattern_length, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_transmit_frequency, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;

    proto_tree_add_item(tree, hf_dis_transmit_freq_bandwidth, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(tree, hf_dis_transmit_power, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    offset = dissect_DIS_FIELDS_MODULATION_TYPE(tvb, tree, offset, &systemModulation);

    proto_tree_add_item(tree, hf_dis_crypto_system, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    ti = proto_tree_add_item(tree, hf_dis_crypto_key, tvb, offset, 2, ENC_BIG_ENDIAN);
    sub_tree = proto_item_add_subtree(ti, ett_dis_crypto_key);
    proto_tree_add_item(sub_tree, hf_dis_encryption_mode, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(sub_tree, hf_dis_key_identifier, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    modulationParamLength = tvb_get_uint8(tvb, offset);
    proto_tree_add_item(tree, hf_dis_modulation_parameter_length, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset++;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 3, ENC_NA);
    offset += 3;

    /* need to check to see if mod parms length > 0 */
    /* could get here when there are antenna pattern parameter but no mod params */
    if (modulationParamLength > 0 ) { /* we do have a mod param */
        switch(systemModulation)
        {
        case DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_SINCGARS:
            offset = dissect_DIS_FIELDS_MOD_PARAMS_CCTT_SINCGARS(tvb, tree, offset);
            break;
        case DIS_PDU_TRANSMITTER_MODULATION_TYPE_SYSTEM_JTIDS_MIDS:
            offset = dissect_DIS_FIELDS_MOD_PARAMS_JTIDS_MIDS(tvb, tree, offset);
            break;
        default: /* just dump what is available */
            proto_tree_add_item(tree, hf_dis_mod_param_dump, tvb, offset, modulationParamLength, ENC_NA);
            offset += modulationParamLength;
            break;
        }
    } /* else, leave offset alone, and then check antenna pattern param field */

    /* need to finish decoding this PDU */
    return offset;
}

static int dissect_DIS_PARSER_DESIGNATOR_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    proto_tree* sub_tree;
    uint16_t code_name;

    offset = parseField_Entity(tvb, tree, offset, "Designating Entity ID", NULL);

    proto_tree_add_item_ret_uint16(tree, hf_dis_designator_code_name, tvb, offset, 2, ENC_BIG_ENDIAN, &code_name);
    col_append_fstr( pinfo->cinfo, COL_INFO, ", CodeName=%u", code_name);
    offset += 2;

    offset = parseField_Entity(tvb, tree, offset, "Designated Entity ID", NULL);

    proto_tree_add_item(tree, hf_dis_designator_designator_code, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_designator_power, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(tree, hf_dis_designator_wavelength, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_designator_spot_with_respect_to_designated_entity, NULL, "Designator Spot with Respect to Designated Entity");
    proto_tree_add_item(sub_tree, hf_dis_designator_spot_with_respect_to_designated_entity_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_designator_spot_with_respect_to_designated_entity_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_designator_spot_with_respect_to_designated_entity_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 24, ett_designator_spot_location, NULL, "Designator Spot Location");
    proto_tree_add_item(sub_tree, hf_dis_designator_spot_location_x, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_designator_spot_location_y, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;
    proto_tree_add_item(sub_tree, hf_dis_designator_spot_location_z, tvb, offset, 8, ENC_BIG_ENDIAN);
    offset += 8;

    proto_tree_add_item(tree, hf_dis_entity_dead_reckoning_algorithm, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 3, ENC_NA);
    offset += 3;

    sub_tree = proto_tree_add_subtree(tree, tvb, offset, 12, ett_designator_entity_linear_acceleration, NULL, "Entity Linear Acceleration");
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_acceleration_x, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_acceleration_y, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;
    proto_tree_add_item(sub_tree, hf_dis_entity_linear_acceleration_z, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    /* need to finish decoding this PDU */
    return offset;
}

static int dissect_DIS_PARSER_INTERCOM_CONTROL_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    int8_t source_line_id;
    proto_tree_add_item(tree, hf_intercom_control_control_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(tree, hf_intercom_control_communications_channel_type, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    offset = parseField_Entity(tvb, tree, offset, "Source Entity ID", NULL);


    proto_tree_add_item(tree, hf_intercom_control_source_communications_device_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_intercom_control_source_line_id, tvb, offset, 1, ENC_BIG_ENDIAN);
    source_line_id = tvb_get_uint8(tvb, offset);
    col_append_fstr( pinfo->cinfo, COL_INFO, ", SourceLineID=%u", source_line_id);
    offset += 1;

    proto_tree_add_item(tree, hf_intercom_control_transmit_priority, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(tree, hf_intercom_control_transmit_line_state, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    proto_tree_add_item(tree, hf_intercom_control_command, tvb, offset, 1, ENC_BIG_ENDIAN);
    offset += 1;

    offset = parseField_Entity(tvb, tree, offset, "Master Entity ID", NULL);

    proto_tree_add_item(tree, hf_intercom_control_master_communications_device_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_intercom_control_master_channel_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    return offset;
}

static int dissect_DIS_PARSER_SIGNAL_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    proto_item* ti;
    proto_tree* sub_tree;
    uint32_t radioID, encodingScheme, numSamples;
    uint16_t tdlType;
    uint8_t messageType;

    offset = parseField_Entity(tvb, tree, offset, "Entity ID", NULL);

    proto_tree_add_item(tree, hf_dis_radio_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    radioID = tvb_get_ntohs(tvb, offset);
    col_append_fstr( pinfo->cinfo, COL_INFO, ", RadioID=%u", radioID);
    offset += 2;

    encodingScheme = tvb_get_ntohs(tvb, offset);
    if ((encodingScheme & 0xC000) >> 14 == DIS_ENCODING_CLASS_ENCODED_AUDIO)
        col_append_fstr(pinfo->cinfo, COL_INFO,", Encoding Type=%s",
            val_to_str_const(DIS_ENCODING_TYPE(encodingScheme),
            DIS_PDU_Signal_Encoding_Type_Strings, "Unknown"));

    ti = proto_tree_add_item(tree, hf_dis_ens, tvb, offset, 2, ENC_BIG_ENDIAN);
    sub_tree = proto_item_add_subtree(ti, ett_dis_ens);

    proto_tree_add_item(sub_tree, hf_dis_ens_class, tvb, offset, 2, ENC_BIG_ENDIAN);
    proto_tree_add_item(sub_tree,
        ((encodingScheme >> 14) & 3) == DIS_ENCODING_CLASS_ENCODED_AUDIO ? hf_dis_ens_type_audio : hf_dis_ens_type,
        tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    tdlType = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item(tree, hf_dis_tdl_type, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_sample_rate, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    proto_tree_add_item(tree, hf_dis_data_length, tvb, offset, 2, ENC_BIG_ENDIAN);
    offset += 2;

    numSamples = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item(tree, hf_dis_num_of_samples, tvb, offset, 2, ENC_BIG_ENDIAN);
    if (numSamples)
        col_append_fstr(pinfo->cinfo, COL_INFO, ", Number of Samples=%u", numSamples);
    offset += 2;

    if (tdlType == DIS_PDU_SIGNAL_TDL_TYPE_LINK_16_STANDARDIZED_FORMAT_JTIDS_MIDS_TADIL_J) {
        offset = parse_DIS_FIELDS_SIGNAL_LINK16_NETWORK_HEADER(tvb, tree, offset, &messageType);

        sub_tree = proto_tree_add_subtree_format(tree, tvb, offset, -1, ett_dis_signal_link16_message_data, NULL,
            "Link 16 Message Data: %s", val_to_str_const(messageType, DIS_PDU_Message_Type_Identifier_Strings, ""));
        offset = parse_Link16_Message_Data(sub_tree, tvb, offset, pinfo, encodingScheme, messageType);
        proto_item_set_end(ti, tvb, offset);
    } else {
        proto_tree_add_item(tree, hf_dis_signal_data, tvb, offset, -1, ENC_NA );
        offset += tvb_reported_length_remaining(tvb, offset);
    }
    /* ****ck******* need to look for padding bytes */

    return offset;
}

static int dissect_DIS_PARSER_RECEIVER_PDU(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, int offset)
{
    uint32_t radioID, disRadioReceiveState;

    offset = parseField_Entity(tvb, tree, offset, "Entity ID", NULL);

    proto_tree_add_item(tree, hf_dis_radio_id, tvb, offset, 2, ENC_BIG_ENDIAN);
    radioID = tvb_get_ntohs(tvb, offset);
    col_append_fstr( pinfo->cinfo, COL_INFO, ", RadioID=%u", radioID);
    offset += 2;

    disRadioReceiveState = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item(tree, hf_dis_radio_receive_state, tvb, offset, 2, ENC_BIG_ENDIAN);
    col_append_fstr( pinfo->cinfo, COL_INFO, ", Receive State=%s", val_to_str_const(disRadioReceiveState, DIS_PDU_Receiver_Receiver_State_Strings, "Unknown Receive State"));
    offset += 2;

    proto_tree_add_item(tree, hf_dis_padding, tvb, offset, 2, ENC_NA);
    offset += 2;

    proto_tree_add_item(tree, hf_dis_receive_power, tvb, offset, 4, ENC_BIG_ENDIAN);
    offset += 4;

    offset = parseField_Entity(tvb, tree, offset, "Transm